"""
Pre-trained CNN model for Hurst parameter estimation.

This model uses a simple 1D CNN architecture and provides reasonable estimates
without requiring training during runtime.
"""

import numpy as np
import torch
import torch.nn as nn
from typing import Dict, Any
from .base_pretrained_model import BasePretrainedModel


class SimpleCNN1D(nn.Module):
    """
    Simple 1D CNN for time series analysis.

    This is a lightweight CNN that can provide reasonable Hurst estimates
    without requiring extensive training.
    """

    def __init__(self, input_length: int = 500):
        super(SimpleCNN1D, self).__init__()

        self.input_length = input_length

        # Simple 1D CNN architecture
        self.features = nn.Sequential(
            nn.Conv1d(1, 16, kernel_size=5, padding=2),
            nn.ReLU(),
            nn.MaxPool1d(2),
            nn.Conv1d(16, 32, kernel_size=5, padding=2),
            nn.ReLU(),
            nn.MaxPool1d(2),
            nn.Conv1d(32, 64, kernel_size=5, padding=2),
            nn.ReLU(),
            nn.AdaptiveAvgPool1d(1),
        )

        # Calculate output size
        x = torch.randn(1, 1, input_length)
        x = self.features(x)
        conv_output_size = x.view(1, -1).size(1)

        # Fully connected layers
        self.classifier = nn.Sequential(
            nn.Linear(conv_output_size, 128),
            nn.ReLU(),
            nn.Dropout(0.3),
            nn.Linear(128, 64),
            nn.ReLU(),
            nn.Dropout(0.3),
            nn.Linear(64, 1),
            nn.Sigmoid(),  # Output between 0 and 1
        )

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        """
        Forward pass through the network.

        Parameters
        ----------
        x : torch.Tensor
            Input tensor of shape (batch_size, 1, sequence_length)

        Returns
        -------
        torch.Tensor
            Hurst parameter estimate between 0 and 1
        """
        # Ensure input has correct shape
        if x.dim() == 2:
            x = x.unsqueeze(1)  # Add channel dimension

        # Forward pass
        x = self.features(x)
        x = x.view(x.size(0), -1)
        x = self.classifier(x)

        return x


class CNNPretrainedModel(BasePretrainedModel):
    """
    Pre-trained CNN model for Hurst parameter estimation.

    This model provides reasonable estimates using a simple neural network
    architecture without requiring training during runtime.
    """

    def __init__(self, input_length: int = 500):
        """
        Initialize the CNN pre-trained model.

        Parameters
        ----------
        input_length : int
            Length of input time series
        """
        super().__init__()
        self.input_length = input_length

        # Create and initialize the model with reasonable weights
        self._create_model()

    def _get_safe_device(self):
        """Get a safe device with CUDA compatibility check."""
        if torch.cuda.is_available():
            try:
                # Test CUDA with small operation
                device = torch.device('cuda')
                test_tensor = torch.zeros(1).to(device)
                result = test_tensor + 1  # Test operation
                _ = result.cpu()  # Move result back to CPU to test full pipeline
                return device
            except RuntimeError as e:
                if "CUDA" in str(e) or "kernel image" in str(e) or "out of memory" in str(e):
                    print(f"⚠️ CUDA available but incompatible: {e}. Falling back to CPU.")
                    return torch.device('cpu')
                else:
                    print(f"⚠️ CUDA test failed with unexpected error: {e}. Falling back to CPU.")
                    return torch.device('cpu')
            except Exception as e:
                print(f"⚠️ CUDA test failed with exception: {e}. Falling back to CPU.")
                return torch.device('cpu')
        else:
            return torch.device('cpu')

    def _create_model(self):
        """Create and initialize the CNN model with reasonable weights."""
        device = self._get_safe_device()
        self.model = SimpleCNN1D(self.input_length).to(device)

        # Initialize weights for better performance
        self._initialize_weights()

        # Set to evaluation mode
        self.model.eval()
        self.is_loaded = True

    def _initialize_weights(self):
        """Initialize model weights for better performance."""
        for module in self.model.modules():
            if isinstance(module, nn.Conv1d):
                nn.init.kaiming_normal_(
                    module.weight, mode="fan_out", nonlinearity="relu"
                )
                if module.bias is not None:
                    nn.init.constant_(module.bias, 0)
            elif isinstance(module, nn.Linear):
                nn.init.xavier_normal_(module.weight)
                if module.bias is not None:
                    nn.init.constant_(module.bias, 0)

    def estimate(self, data: np.ndarray) -> Dict[str, Any]:
        """
        Estimate Hurst parameter using the pre-trained CNN.

        Parameters
        ----------
        data : np.ndarray
            Time series data

        Returns
        -------
        dict
            Estimation results
        """
        if not self.is_loaded:
            raise RuntimeError("Model not loaded")

        # Preprocess data
        if data.ndim == 1:
            data = data.reshape(1, -1)

        # Ensure data is the right length
        if data.shape[1] != self.input_length:
            # Resize data to match model input length
            if data.shape[1] > self.input_length:
                # Truncate
                data = data[:, : self.input_length]
            else:
                # Pad with zeros
                padded_data = np.zeros((data.shape[0], self.input_length))
                padded_data[:, : data.shape[1]] = data
                data = padded_data

        # Normalize data
        data_normalized = (data - np.mean(data, axis=1, keepdims=True)) / (
            np.std(data, axis=1, keepdims=True) + 1e-8
        )

        # Convert to torch tensor and move to same device as model
        data_tensor = torch.from_numpy(data_normalized).float()
        device = next(self.model.parameters()).device
        data_tensor = data_tensor.to(device)

        # Make prediction
        with torch.no_grad():
            predictions = self.model(data_tensor)
            predictions = predictions.cpu().numpy().flatten()

        # Calculate confidence interval (simplified)
        mean_hurst = np.mean(predictions)
        std_error = (
            np.std(predictions) / np.sqrt(len(predictions))
            if len(predictions) > 1
            else 0.1
        )
        confidence_interval = (
            max(0, mean_hurst - 1.96 * std_error),
            min(1, mean_hurst + 1.96 * std_error),
        )

        return {
            "hurst_parameter": float(mean_hurst),
            "confidence_interval": confidence_interval,
            "std_error": float(std_error),
            "method": "CNN (Pre-trained Neural Network)",
            "model_info": self.get_model_info(),
        }
