from datetime import date
from io import BytesIO
from typing import Any, Dict

from django.contrib.auth import get_user_model
from django.template.loader import get_template

from wbreport.models import Report, ReportVersion

User = get_user_model()


class ReportMixin:
    HTML_TEMPLATE_FILE = ""

    @classmethod
    def has_view_permission(cls, report: "Report", user: "User") -> bool:
        """
        Return if the given report, user pair has the view permission
        Args:
            report: A Report object
            user: A User object

        Returns:
            True if the user has the permission to view the report
        Raises:
            NotImplementedError: If inheriting subclass does not explicitly defined this method
        """
        return False

    @classmethod
    def has_change_permission(cls, report: "Report", user: "User") -> bool:
        """
        Return if the given report, user pair has the change permission
        Args:
            report: A Report object
            user: A User object

        Returns:
            True if the user has the permission to change the report
        Raises:
            NotImplementedError: If inheriting subclass does not explicitly defined this method
        """
        return False

    @classmethod
    def has_delete_permission(cls, report: "Report", user: "User") -> bool:
        """
        Return if the given report, user pair has the delete permission
        Args:
            report: A Report object
            user: A User object

        Returns:
            True if the user has the permission to delete the report
        Raises:
            NotImplementedError: If inheriting subclass does not explicitly define this method
        """
        return False

    @classmethod
    def get_next_parameters(cls, parameters: Dict[str, Any]) -> Dict[str, Any]:
        """
        Return the temporal next set of parameters (e.g. next month)
        Args:
            parameters: The current parameters set

        Returns:
            The next parameter set
        Raises:
            NotImplementedError: If inheriting subclass does not explicitly define this method
        """
        raise NotImplementedError("Not Implemented")

    @classmethod
    def get_version_title(cls, report_title: str, parameters: Dict[str, Any]) -> str:
        """
        Derivate the version title based on its report's title and the given parameters set

        e.g. report_january_2021

        Args:
            report_title: The report title
            parameters: Current paramaters set

        Returns:
            The version title. Default to the report title
        """
        return report_title

    @classmethod
    def get_version_date(cls, parameters: Dict[str, Any]) -> date:
        """
        Extract the version date based on the current parameters set
        Args:
            parameters: Current parameters set

        Returns:
            The version date. Default to the value from the key "end" or None.
        """
        parameters = cls.parse_parameters(parameters)
        return parameters.get("end", None)

    @classmethod
    def get_context(cls, version: "ReportVersion") -> Dict[str, Any]:
        """
        Generate context as a dictionary for the passed object

        Args:
            version: The object where context needs to be generated against

        Returns:
            Dictionary of key,value pairs

        Raise:
            NotImplementedError: If inheriting subclass does not explicitly define this method
        """
        raise NotImplementedError

    @classmethod
    def generate_html(cls, context: Dict[str, Any]) -> str:
        """
        Generate a html given a predetermined context dictionary

        Args:
            context: The dictionary context

        Returns:
            A rendered template as string

        """
        template = get_template(cls.HTML_TEMPLATE_FILE)
        return template.render(context)

    @classmethod
    def generate_file(cls, context: Dict[str, Any]) -> BytesIO:
        """
        Generate a file given a predetermined context dictionary
        Args:
            context: The dictionary context

        Returns:
            The generated bytes stream

        Raises:
            NotImplementedError: If inheriting subclass does not explicitly define this method
        """
        raise NotImplementedError

    @classmethod
    def parse_parameters(cls, parameters: Dict[str, str]) -> Dict[str, Any]:
        """
        Parse the json parameters into a python dictionary (e.g. str to date)

        Args:
            parameters: parameters as json

        Returns:
            A python dictionary
        """
        raise parameters

    @classmethod
    def serialize_context(cls, context: Dict[str, Any]) -> Dict[str, Any]:
        if "uuid" in context:
            context["uuid"] = str(context["uuid"])
        return context

    @classmethod
    def deserialize_context(cls, context: Dict[str, Any]) -> Dict[str, Any]:
        """
        Method called before rendering the template. Defaults return the passed dictionary untouched.

        e.g. Can return the file url

        Args:
            context: The context as pickled dictionry

        Returns:
            The deserialized context
        """
        if report_logo_file_id := context.pop("report_logo_file_id", None):
            context["logo_file"] = Report.objects.get(id=report_logo_file_id).logo_file
        return context
