# Copyright (c) 2017 The University of Manchester
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from functools import reduce
import logging
import math
from typing import Any, Collection, Union, overload
from typing_extensions import TypeGuard
from spinn_utilities.log import FormatAdapter


logger = FormatAdapter(logging.getLogger(__name__))
FINISHED_FILENAME = "finished"


def is_singleton(value: Any) -> TypeGuard[Union[bool, int, float]]:
    """
    Tests whether the value is a singleton.

    Singleton types are strings and any other class that can not be
    iterated.

    Strings are considered singleton as only rarely will someone use a string
    to represent an iterable of characters.

    :returns: True if the value is a singleton, False otherwise.
    """
    return not hasattr(value, '__iter__') or isinstance(value, str)


def _lcm(a: int, b: int) -> int:
    return (a * b) // math.gcd(a, b)


# A sequence of integers..
@overload
def lcm(numbers: Collection[int] = (), /) -> int:
    ...


# ... or several integers. All in positional-only parameters.
@overload
def lcm(number: int, /, *numbers: int) -> int:
    ...


def lcm(*numbers) -> int:  # type: ignore[no-untyped-def]
    """
    Lowest common multiple of 0, 1 or more integers.

    GIGO: If any of the values are anything except positive `int` values
    this function will either produce incorrect results or raise an exception.

    :param numbers:
        The Positive integers to get the Lowest common multiple for.
        This can be zero, one or more int values or
        a singleton which is an iterator (possibly empty) of `int`\\s.
    :return: the  Lowest common multiple,
        or 1 if `numbers` is empty or an empty iterator
    :raises TypeError: If any value cannot be interpreted as an integer
    :raises ZeroDivisionError: May be raised if one of the values is zero
    """
    if len(numbers) == 1:
        try:
            return reduce(_lcm, iter(numbers[0]), 1)
        except TypeError:
            return numbers[0]
    return reduce(_lcm, numbers, 1)


# A sequence of integers..
@overload
def gcd(numbers: Collection[int] = (), /) -> int:
    ...


# ... or several integers. All in positional-only parameters.
@overload
def gcd(number: int, /, *numbers: int) -> int:
    ...


def gcd(*numbers) -> int:  # type: ignore[no-untyped-def]
    """
    Greatest common divisor of 1 or more integers.

    GIGO: If any of the values are anything except positive `int` values
    this function will either produce incorrect results or raise an exception.

    :param numbers: The positive integers to get the GCD for.
        This can be one or more `int` values or
        a singleton which is an iterator (*not* empty) of `int`\\s.
    :return: the GCD
    :raises TypeError: If any value cannot be interpreted as an integer or
        if no values are provided
    :raises ZeroDivisionError: May be raised if one of the values is zero
    """
    if len(numbers) == 1:
        try:
            return reduce(math.gcd, iter(numbers[0]))
        except TypeError:
            return numbers[0]
    return reduce(math.gcd, numbers)
