import os
import re
from enum import Enum, IntEnum
from functools import reduce
from pathlib import Path
from typing import Dict, List

from packaging.version import Version

from demisto_sdk.commands.common.StrEnum import StrEnum

# Note: Do NOT add imports of internal modules here, as it may cause circular imports.


PROJECT_DATA_DIR = Path.home() / ".demisto-sdk"
CACHE_DIR = PROJECT_DATA_DIR / "cache"
LOGS_DIR = PROJECT_DATA_DIR / "logs"
NEO4J_DIR = PROJECT_DATA_DIR / "neo4j"

LOG_FILE_NAME = "demisto_sdk_debug.log"

NEO4J_DEFAULT_VERSION = "5.22.0"

# Colors
RED = "\033[91m"
NO_COLOR = "\033[0m"

# --- Environment Variables ---
# General
ENV_DEMISTO_SDK_MARKETPLACE = "DEMISTO_SDK_MARKETPLACE"
DEMISTO_GIT_PRIMARY_BRANCH = os.getenv("DEMISTO_DEFAULT_BRANCH", "master")
DEMISTO_GIT_UPSTREAM = os.getenv("DEMISTO_DEFAULT_REMOTE", "origin")
DEMISTO_SDK_CI_SERVER_HOST = os.getenv("CI_SERVER_HOST", "gitlab.xdr.pan.local")
DEMISTO_SDK_OFFICIAL_CONTENT_PROJECT_ID = os.getenv(
    "CI_PROJECT_ID", "1061"
)  # Default value is the ID of the content repo on GitLab
ENV_SDK_WORKING_OFFLINE = "DEMISTO_SDK_OFFLINE_ENV"
SDK_OFFLINE_ERROR_MESSAGE = (
    "<red>An internet connection is required for this command. If connected to the "
    "internet, un-set the DEMISTO_SDK_OFFLINE_ENV environment variable.</red>"
)

DEFAULT_DOCKER_REGISTRY_URL = "docker.io"
DOCKER_REGISTRY_URL = os.getenv(
    "DEMISTO_SDK_CONTAINER_REGISTRY",
    os.getenv("DOCKER_IO", DEFAULT_DOCKER_REGISTRY_URL),
)


# Authentication
DEMISTO_BASE_URL = "DEMISTO_BASE_URL"
DEMISTO_USERNAME = "DEMISTO_USERNAME"
DEMISTO_PASSWORD = "DEMISTO_PASSWORD"  # guardrails-disable-line
DEMISTO_KEY = "DEMISTO_API_KEY"
AUTH_ID = "XSIAM_AUTH_ID"
XSIAM_TOKEN = "XSIAM_TOKEN"
XSIAM_COLLECTOR_TOKEN = "XSIAM_COLLECTOR_TOKEN"
DEMISTO_VERIFY_SSL = "DEMISTO_VERIFY_SSL"
PROJECT_ID = "PROJECT_ID"

# Logging
DEMISTO_SDK_LOG_FILE_PATH = "DEMISTO_SDK_LOG_FILE_PATH"
DEMISTO_SDK_LOG_NOTIFY_PATH = "DEMISTO_SDK_LOG_NOTIFY_PATH"
DEMISTO_SDK_LOG_FILE_SIZE = "DEMISTO_SDK_LOG_FILE_SIZE"
DEMISTO_SDK_LOG_FILE_COUNT = "DEMISTO_SDK_LOG_FILE_COUNT"
DEMISTO_SDK_LOG_NO_COLORS = "DEMISTO_SDK_LOG_NO_COLORS"
DEMISTO_SDK_LOGGING_SET = "DEMISTO_SDK_LOGGING_SET"
# Neo4j
DEMISTO_SDK_NEO4J_VERSION = "DEMISTO_SDK_NEO4J_VERSION"
DEMISTO_SDK_NEO4J_DATABASE_HTTP = "DEMISTO_SDK_NEO4J_DATABASE_HTTP"
DEMISTO_SDK_NEO4J_DATABASE_URL = "DEMISTO_SDK_NEO4J_DATABASE_URL"
DEMISTO_SDK_NEO4J_USERNAME = "DEMISTO_SDK_NEO4J_USERNAME"
DEMISTO_SDK_NEO4J_PASSWORD = "DEMISTO_SDK_NEO4J_PASSWORD"
# --- Environment Variables ---


CAN_START_WITH_DOT_SLASH = "(?:./)?"
NOT_TEST = "(?!Test)"

# NAMES OF ENTITY DIRECTORIES

INTEGRATIONS_DIR = "Integrations"
SCRIPTS_DIR = "Scripts"
PLAYBOOKS_DIR = "Playbooks"
TEST_PLAYBOOKS_DIR = "TestPlaybooks"
REPORTS_DIR = "Reports"
DASHBOARDS_DIR = "Dashboards"
WIDGETS_DIR = "Widgets"
INCIDENT_FIELDS_DIR = "IncidentFields"
INCIDENT_TYPES_DIR = "IncidentTypes"
INDICATOR_FIELDS_DIR = "IndicatorFields"
INDICATOR_TYPES_DIR = "IndicatorTypes"
GENERIC_FIELDS_DIR = "GenericFields"
GENERIC_TYPES_DIR = "GenericTypes"
GENERIC_MODULES_DIR = "GenericModules"
GENERIC_DEFINITIONS_DIR = "GenericDefinitions"
LAYOUTS_DIR = "Layouts"
CLASSIFIERS_DIR = "Classifiers"
MAPPERS_DIR = "Classifiers"
PACKS_DIR = "Packs"
TOOLS_DIR = "Tools"
RELEASE_NOTES_DIR = "ReleaseNotes"
TESTS_DIR = "Tests"
DOC_FILES_DIR = "doc_files"
DOCUMENTATION_DIR = "Documentation"
JOBS_DIR = "Jobs"
PRE_PROCESS_RULES_DIR = "PreProcessRules"
LISTS_DIR = "Lists"
PARSING_RULES_DIR = "ParsingRules"
MODELING_RULES_DIR = "ModelingRules"
SAMPLES_DIR = "Samples"
CORRELATION_RULES_DIR = "CorrelationRules"
XSIAM_DASHBOARDS_DIR = "XSIAMDashboards"
XSIAM_REPORTS_DIR = "XSIAMReports"
TRIGGER_DIR = "Triggers"
WIZARDS_DIR = "Wizards"
XDRC_TEMPLATE_DIR = "XDRCTemplates"
LAYOUT_RULES_DIR = "LayoutRules"
ASSETS_MODELING_RULES_DIR = "AssetsModelingRules"
CASE_LAYOUT_RULES_DIR = "CaseLayoutRules"
CASE_LAYOUTS_DIR = "CaseLayouts"
CASE_FIELDS_DIR = "CaseFields"
AGENTIX_ACTIONS_DIR = "AgentixActions"
AGENTIX_AGENTS_DIR = "AgentixAgents"

# NAMES OF ENTITIES

SCRIPT = "script"
AUTOMATION = "automation"
INTEGRATION = "integration"
PLAYBOOK = "playbook"
TEST_PLAYBOOK = "testplaybook"
LAYOUT = "layout"
LAYOUTS_CONTAINER = "layoutscontainer"
PRE_PROCESS_RULES = "pre-process-rules"
LISTS = "list"  # singular, as it is the prefix of the file
INCIDENT = "incident"  # prefix to identify any incident entity
INCIDENT_TYPE = "incidenttype"
INCIDENT_FIELD = "incidentfield"
INDICATOR_FIELD = "indicatorfield"
CLASSIFIER = "classifier"
DASHBOARD = "dashboard"
REPORT = "report"
INDICATOR_TYPE = "reputation"
OLD_INDICATOR_TYPE = "reputations"
WIDGET = "widget"
TOOL = "tools"
BETA_INTEGRATION = "betaintegration"
DOCUMENTATION = "doc"
MAPPER = "classifier-mapper"
OLD_REPUTATION = "reputations.json"
PACK_VERIFY_KEY = "content.pack.verify"
XSOAR_CONFIG_FILE = "xsoar_config.json"
DEMISTO_SDK_CONFIG_FILE = ".demisto-sdk-conf"
GENERIC_FIELD = "genericfield"
GENERIC_TYPE = "generictype"
GENERIC_MODULE = "genericmodule"
GENERIC_DEFINITION = "genericdefinition"
JOB = "job"
PARSING_RULE = "parsingrule"
MODELING_RULE = "modelingrule"
CORRELATION_RULE = "correlationrule"
XSIAM_DASHBOARD = "xsiamdashboard"
XSIAM_REPORT = "xsiamreport"
TRIGGER = "trigger"
WIZARD = "wizard"
XDRC_TEMPLATE = "xdrctemplate"
LAYOUT_RULE = "layoutrule"
MARKETPLACE_KEY_PACK_METADATA = "marketplaces"
EVENT_COLLECTOR = "EventCollector"
ASSETS_MODELING_RULE = "assetsmodelingrule"
CASE_LAYOUT_RULE = "caselayoutrule"
CASE_FIELD = "casefield"
CASE_LAYOUT = "caselayout"

# Marketplaces

DEMISTO_SDK_MARKETPLACE_XSOAR_DIST = "marketplace-dist"
DEMISTO_SDK_MARKETPLACE_XSIAM_DIST = "marketplace-v2-dist"
DEMISTO_SDK_MARKETPLACE_XPANSE_DIST = "xpanse-dist"
DEMISTO_SDK_MARKETPLACE_XSOAR_SAAS_DIST = "marketplace-saas-dist"
DEMISTO_SDK_MARKETPLACE_XSOAR_DIST_DEV = "marketplace-dist-dev"
DEMISTO_SDK_MARKETPLACE_PLATFORM_INTERNAL_PROD = "marketplace-cortex-content-dev"


# Server Types
XSOAR_SERVER_TYPE = "XSOAR"
XSIAM_SERVER_TYPE = "XSIAM"
XPANSE_SERVER_TYPE = "XPANSE"
XSOAR_SAAS_SERVER_TYPE = "XSOAR SAAS"

# Product Types
XSOAR_PRODUCT_TYPE = "XSOAR"
XSIAM_PRODUCT_TYPE = "XSIAM"
XPANSE_PRODUCT_TYPE = "XPANSE"


class FileType(StrEnum):
    INTEGRATION = "integration"
    SCRIPT = "script"
    TEST_SCRIPT = "testscript"
    PLAYBOOK = "playbook"
    TEST_PLAYBOOK = "testplaybook"
    BETA_INTEGRATION = "betaintegration"
    INCIDENT_FIELD = "incidentfield"
    INDICATOR_FIELD = "indicatorfield"
    REPUTATION = "reputation"
    LAYOUT = "layout"
    LAYOUTS_CONTAINER = "layoutscontainer"
    DASHBOARD = "dashboard"
    INCIDENT_TYPE = "incidenttype"
    MAPPER = "mapper"
    OLD_CLASSIFIER = "classifier_5_9_9"
    CLASSIFIER = "classifier"
    WIDGET = "widget"
    REPORT = "report"
    README = "readme"
    RELEASE_NOTES = "releasenotes"
    RELEASE_NOTES_CONFIG = "releasenotesconfig"
    DESCRIPTION = "description"
    CHANGELOG = "changelog"
    IMAGE = "image"
    AUTHOR_IMAGE = "author_image"
    DOC_IMAGE = "doc_image"
    PYTHON_FILE = "pythonfile"
    XIF_FILE = "xiffile"
    MODELING_RULE_SCHEMA = "modelingruleschema"
    JAVASCRIPT_FILE = "javascriptfile"
    POWERSHELL_FILE = "powershellfile"
    CONF_JSON = "confjson"
    METADATA = "metadata"
    WHITE_LIST = "whitelist"
    LANDING_PAGE_SECTIONS_JSON = "landingPage_sections.json"
    CONTRIBUTORS = "contributors"
    PACK = "pack"
    XSOAR_CONFIG = "xsoar_config"
    GENERIC_MODULE = "genericmodule"
    GENERIC_FIELD = "genericfield"
    GENERIC_TYPE = "generictype"
    GENERIC_DEFINITION = "genericdefinition"
    PRE_PROCESS_RULES = "pre-process-rule"
    LISTS = "list"
    JOB = "job"
    BUILD_CONFIG_FILE = "build-config-file"
    PARSING_RULE = "parsingrule"
    PARSING_RULE_XIF = "parsingrulexif"
    MODELING_RULE = "modelingrule"
    MODELING_RULE_TEST_DATA = "modelingruletestdata"
    MODELING_RULE_XIF = "modelingrulexif"
    CORRELATION_RULE = "correlationrule"
    XSIAM_DASHBOARD = "xsiamdashboard"
    XSIAM_DASHBOARD_IMAGE = "xsiamdashboardimage"
    XSIAM_REPORT = "xsiamreport"
    XSIAM_REPORT_IMAGE = "xsiamreportimage"
    TRIGGER = "trigger"
    WIZARD = "wizard"
    PACK_IGNORE = ".pack-ignore"
    SECRET_IGNORE = ".secrets-ignore"
    DOC_FILE = "doc_files"
    XDRC_TEMPLATE = "xdrctemplate"
    XDRC_TEMPLATE_YML = "xdrctemplateyml"
    INDICATOR_TYPE = "indicatortype"
    TOOL = "tools"
    PACK_METADATA = "packmetadata"
    LAYOUT_RULE = "layoutrule"
    PIPFILE = "pipfile"
    TXT = "txt"
    PIPFILE_LOCK = "pipfilelock"
    PYLINTRC = "pylintrc"
    LICENSE = "license"
    UNIFIED_YML = "unified_yml"
    INI = "ini"
    PEM = "pem"
    VULTURE_WHITELIST = "vulture_whitelist"
    ASSETS_MODELING_RULE_SCHEMA = "assetsmodelingruleschema"
    ASSETS_MODELING_RULE = "assetsmodelingrule"
    ASSETS_MODELING_RULE_XIF = "assetsmodelingrulexif"
    CASE_LAYOUT_RULE = "caselayoutrule"
    CASE_FIELD = "casefield"
    CASE_LAYOUT = "caselayout"
    VERSION_CONFIG = "version_config"
    AGENTIX_AGENT = "agentixagent"
    AGENTIX_ACTION = "agentixaction"


RN_HEADER_BY_FILE_TYPE = {
    FileType.PLAYBOOK: "Playbooks",
    FileType.INTEGRATION: "Integrations",
    FileType.BETA_INTEGRATION: "Integrations",
    FileType.SCRIPT: "Scripts",
    FileType.INCIDENT_FIELD: "Incident Fields",
    FileType.INDICATOR_FIELD: "Indicator Fields",
    FileType.REPUTATION: "Indicator Types",
    FileType.INCIDENT_TYPE: "Incident Types",
    FileType.CLASSIFIER: "Classifiers",
    FileType.OLD_CLASSIFIER: "Classifiers",
    FileType.LAYOUTS_CONTAINER: "Layouts",
    FileType.LAYOUT: "Layouts",
    FileType.REPORT: "Reports",
    FileType.WIDGET: "Widgets",
    FileType.DASHBOARD: "Dashboards",
    FileType.MAPPER: "Mappers",
    FileType.PRE_PROCESS_RULES: "PreProcess Rules",
    FileType.GENERIC_DEFINITION: "Objects",
    FileType.GENERIC_MODULE: "Modules",
    FileType.GENERIC_TYPE: "Object Types",
    FileType.GENERIC_FIELD: "Object Fields",
    FileType.LISTS: "Lists",
    FileType.JOB: "Jobs",
    FileType.PARSING_RULE: "Parsing Rules",
    FileType.MODELING_RULE: "Modeling Rules",
    FileType.MODELING_RULE_SCHEMA: "Modeling Rules Schema",
    FileType.CORRELATION_RULE: "Correlation Rules",
    FileType.XSIAM_DASHBOARD: "XSIAM Dashboards",
    FileType.XSIAM_REPORT: "XSIAM Reports",
    FileType.TRIGGER: "Triggers Recommendations",  # https://github.com/demisto/etc/issues/48153#issuecomment-1111988526
    FileType.WIZARD: "Wizards",
    FileType.XDRC_TEMPLATE: "XDRC Templates",
    FileType.LAYOUT_RULE: "Layout Rules",
    FileType.ASSETS_MODELING_RULE: "Assets Modeling Rules",
    FileType.CASE_LAYOUT_RULE: "Case Layout Rules",
    FileType.CASE_FIELD: "Case Fields",
    FileType.CASE_LAYOUT: "Case Layouts",
}

FILE_TYPE_BY_RN_HEADER = {
    header: file_type for file_type, header in RN_HEADER_BY_FILE_TYPE.items()
}

ENTITY_TYPE_TO_DIR = {
    FileType.INTEGRATION.value: INTEGRATIONS_DIR,
    FileType.PLAYBOOK.value: PLAYBOOKS_DIR,
    FileType.SCRIPT.value: SCRIPTS_DIR,
    AUTOMATION: SCRIPTS_DIR,
    FileType.LAYOUT.value: LAYOUTS_DIR,
    FileType.LAYOUTS_CONTAINER.value: LAYOUTS_DIR,
    FileType.INCIDENT_FIELD.value: INCIDENT_FIELDS_DIR,
    FileType.INCIDENT_TYPE.value: INCIDENT_TYPES_DIR,
    FileType.INDICATOR_FIELD.value: INDICATOR_FIELDS_DIR,
    FileType.INDICATOR_TYPE.value: INDICATOR_TYPES_DIR,
    FileType.CLASSIFIER.value: CLASSIFIERS_DIR,
    FileType.DASHBOARD.value: DASHBOARDS_DIR,
    FileType.REPUTATION.value: INDICATOR_TYPES_DIR,
    FileType.REPORT.value: REPORTS_DIR,
    FileType.WIDGET.value: WIDGETS_DIR,
    FileType.BETA_INTEGRATION.value: INTEGRATIONS_DIR,
    FileType.MAPPER.value: CLASSIFIERS_DIR,
    FileType.PRE_PROCESS_RULES.value: PRE_PROCESS_RULES_DIR,
    FileType.GENERIC_DEFINITION.value: GENERIC_DEFINITIONS_DIR,
    FileType.GENERIC_MODULE.value: GENERIC_MODULES_DIR,
    FileType.GENERIC_FIELD.value: GENERIC_FIELDS_DIR,
    FileType.GENERIC_TYPE.value: GENERIC_TYPES_DIR,
    FileType.LISTS.value: LISTS_DIR,
    FileType.JOB.value: JOBS_DIR,
    FileType.PARSING_RULE.value: PARSING_RULES_DIR,
    FileType.MODELING_RULE.value: MODELING_RULES_DIR,
    FileType.WIZARD.value: WIZARDS_DIR,
    FileType.XDRC_TEMPLATE.value: XDRC_TEMPLATE_DIR,
    FileType.CORRELATION_RULE.value: CORRELATION_RULES_DIR,
    FileType.XSIAM_DASHBOARD.value: XSIAM_DASHBOARDS_DIR,
    FileType.TRIGGER.value: TRIGGER_DIR,
    FileType.OLD_CLASSIFIER.value: CLASSIFIERS_DIR,
    FileType.LAYOUT_RULE.value: LAYOUT_RULES_DIR,
    FileType.ASSETS_MODELING_RULE.value: ASSETS_MODELING_RULES_DIR,
    FileType.CASE_FIELD.value: CASE_FIELDS_DIR,
    FileType.CASE_LAYOUT.value: CASE_LAYOUTS_DIR,
    FileType.CASE_LAYOUT_RULE.value: CASE_LAYOUT_RULES_DIR,
}

SIEM_ONLY_ENTITIES = [
    FileType.PARSING_RULE.value,
    FileType.MODELING_RULE.value,
    FileType.CORRELATION_RULE.value,
    FileType.XSIAM_DASHBOARD.value,
    FileType.XSIAM_REPORT.value,
    FileType.TRIGGER.value,
    FileType.XDRC_TEMPLATE.value,
    FileType.LAYOUT_RULE.value,
    FileType.ASSETS_MODELING_RULE,
    FileType.CASE_LAYOUT_RULE.value,
    FileType.CASE_FIELD.value,
    FileType.CASE_LAYOUT.value,
]

CONTENT_FILE_ENDINGS = ["py", "yml", "png", "json", "md"]

IGNORED_PACKS_IN_DEPENDENCY_CALC = [
    "NonSupported",
    "Base",
]  # Packs that are ignored when calculating dependencies
ALLOWED_EMPTY_PACKS = [
    "Cortex911"
]  # Packs that are allowed to be without content items in the id_set

CUSTOM_CONTENT_FILE_ENDINGS = ["yml", "json"]

CONTENT_ENTITIES_DIRS = [
    INTEGRATIONS_DIR,
    SCRIPTS_DIR,
    PLAYBOOKS_DIR,
    TEST_PLAYBOOKS_DIR,
    REPORTS_DIR,
    DASHBOARDS_DIR,
    WIDGETS_DIR,
    INCIDENT_FIELDS_DIR,
    INDICATOR_FIELDS_DIR,
    INDICATOR_TYPES_DIR,
    INCIDENT_TYPES_DIR,
    LAYOUTS_DIR,
    CLASSIFIERS_DIR,
    GENERIC_FIELDS_DIR,
    GENERIC_TYPES_DIR,
    GENERIC_MODULES_DIR,
    GENERIC_DEFINITIONS_DIR,
    PRE_PROCESS_RULES_DIR,
    LISTS_DIR,
    JOBS_DIR,
    WIZARDS_DIR,
    MODELING_RULES_DIR,
    XDRC_TEMPLATE_DIR,
    PARSING_RULES_DIR,
    LAYOUT_RULES_DIR,
    CORRELATION_RULES_DIR,
    XSIAM_DASHBOARDS_DIR,
    XSIAM_REPORTS_DIR,
    TRIGGER_DIR,
    ASSETS_MODELING_RULES_DIR,
    CASE_LAYOUT_RULES_DIR,
    CASE_FIELDS_DIR,
    CASE_LAYOUTS_DIR,
    AGENTIX_ACTIONS_DIR,
    AGENTIX_AGENTS_DIR,
]

CONTENT_ENTITY_UPLOAD_ORDER = [
    INTEGRATIONS_DIR,
    SCRIPTS_DIR,
    PLAYBOOKS_DIR,
    TEST_PLAYBOOKS_DIR,
    INCIDENT_TYPES_DIR,
    INCIDENT_FIELDS_DIR,
    INDICATOR_FIELDS_DIR,
    INDICATOR_TYPES_DIR,
    CLASSIFIERS_DIR,
    WIDGETS_DIR,
    LAYOUTS_DIR,
    DASHBOARDS_DIR,
    PRE_PROCESS_RULES_DIR,
    LISTS_DIR,
    JOBS_DIR,
    WIZARDS_DIR,
]

DEFAULT_IMAGE = "demisto_sdk/tests/test_files/default_image.png"

DEFAULT_IMAGE_PREFIX = "data:image/png;base64,"
DEFAULT_IMAGE_BASE64 = (
    "iVBORw0KGgoAAAANSUhEUgAAAFAAAABQCAMAAAC5zwKfAAACYVBMVEVHcEwAT4UAT4UAT4YAf/8A//8AT4UAf78AT4U"
    "AT4UAT4UAUYcAT4YAT4YAT48AXIsAT4UAT4UAUIUAUIUAT4UAT4UAVaoAW5EAUIYAWYwAT4UAT4UAT4UAUIgAT4YAUo"
    "UAUIYAUIUAT4YAVY0AUIUAT4UAUIUAUocAUYUAT4UAT4UAT4UAUIYAT4UAUIUAT4cAUYUAUIUAUIYAUocAT4UAUIUAT"
    "4YAUY4AUIUAUIYAT4UAVYgAT4UAT4UAT4YAVYUAT4UAT4UAT4YAT4cAT4UAT4UAUYYAZpkAWIUAT4UAT4gAbZEAT4UA"
    "UIYAT4UAUIUAT4cAUYgAT4UAZpkAT4UAT4UAT4UAVaoAUIUAT4UAWIkAT4UAU4kAUIUAUIUAU4gAT4UAT4UAT4UAVYg"
    "AUIUAT4YAVYkAUYUAT4UAU4cAUIYAUIUAT4gAUIYAVYsAT4YAUocAUYUAUIYAUYgAT4UAT4UAT4UAT4UAUYUAU4UAUY"
    "gAT4UAVY0AUIUAUIUAT4UAT4cAT4oAVY0AUYcAUIcAUIUAUIYAUIcAUYcAUIUAT4UAT4UAUIUAT4UAX58AT4UAUIUAU"
    "IYAT4UAUIYAUIgAT4UAT4UAUIUAT4UAUIUAT4YAT4UAUIYAT4YAUYkAT4UAUYYAUIUAT4UAT4YAT4YAT4YAT4cAUokA"
    "T4UAT4YAUIUAT4UAT4YAUIUAT4UAUIoAT4YAT4UAT4UAT4UAT4UAUIUAT4UAT4YAT4UAUYYAT4YAUYUAT4UAT4YAT4U"
    "AUoUAT4UAT4UAUIYAT4YAUIcAYokAT4UAT4UA65kA0ZYAu5PCXoiOAAAAx3RSTlMA+nO6AgG5BP799i9wShAL9/uVzN"
    "rxAw6JFLv08EmWKLyPmhI/x88+ccjz4WjtmU1F76VEoFbXGdKMrh71+K0qoZODIMuzSAoXni0H4HnjfnccQwXDjT0Gi"
    "/wa5zSCaSvBsWMPb9EnLMoxe3hHOSG+Ilh/S1BnzvJULjimCayy6UAwG1VPta91UVLNgJvZCNBcRuVsPIbb37BllNjC"
    "fTLsbrjukKejYCVtqb/5aqiXI9W0tnad4utdt2HEa1ro5EHWpBOBYg3JeEoS2QAAA5lJREFUGBmtwQN7Y0sABuAvbZK"
    "T1Ha3tt2ubdu2vXu517Zt27a+TH/VbXgmaTIz53nyvtDaV1+JdDrxHVvzkD43D5BsyUe6bKxmUP0qJNM2Y/Pxud9bMH"
    "d5DsNmlmGa/E8ZsvgumHqikFHzPUhgVTGipBxmun20LUCCw4zZAiPtjPMs4r3MmGvbYGA9E6yD7CwlN0FvPac5CckDl"
    "LRBK4dJPAxbDiXvQ+c9H5OZQMwW2lZDJ7eQyQ1vQsR+2j6ARnYnU6nKQ8gdtA1Co6mLqXX1AXBf72GUa6EbGmuotCvT"
    "u4tRBcOfQ+sATQ2cqoSBF2go6xiMtNNQA8zkH6GZ0zBU/mLFYEcBtbbCiVtrM6lxEA6NVFOpHk6d9lPpbjjVSKWCvXB"
    "oHzUyFyG1vuFzM3Yi3rfUqL5/E5Jzv8spz+chjpdao7VIag9D3kAcLw14szHd7h0MGfVAVkITvj/PI4H1OCNyITlPQ6"
    "7eDYjTzqirFmy9NDZnwRhsy0sZsw4xzX46kDVRiahHaPNleBD2+wDJSSGZpNK1v8sRstJP2StDFoDsXh+niIBEUOM/h"
    "NzLBDWtD/UwTAQkghr/IGgrFURAIqg2WoagzVQQAYmg2nUELaWKCEgEla56EFRMFRGQCCpdQtBlKomARFClA0GecSqJ"
    "gERQZSOCLlBNBCSCCucQZJVQTQQkggpnEHSFGiIgEQx76nhrDRPch5BiaoiARHCKv6gOgNW/n7LCOoT8e7GUSpNCMkm"
    "y5xmEeTJ8tBUh6q+K2XTA34yYPYx5qxK25Q0FNFYEmzXOqJ8RZ2eRi2Z8syDpY8RiNxIsmu+niSOQuR9liCsb0638ig"
    "a+RJwMhpxCUv1fUGsJ4jSt5ZRGpGBldFKjBPHOznjzmyGkNusHahyFQ1eyqPQZnHqQSv4n4VQVlTovwKGD1Mi89Bica"
    "KZWVsstFd35MLSUZoqXwcxLNJQBI699TENzYWDs4mya+hBadYOFjFp9YMlaKuVAw5rYwagb93gA1HYxtefKoeaeyRjf"
    "GYTkeZlK6TxofE2bFxHWCibn6oeG+zfatiOmgsn4foHOPEqehu1VJrEXWkOU5EKyhtPkQO9OSjZAdpIJDsOAVcOYccR"
    "bSJnvExjZzphuJGigzf8jzBz6gxG3u5HAs4JRrhGYGmthkK9xFaYpu41hWbkwVzbyTsdHb59AMtsyGVTahnRZ9hPJ13"
    "cjfQ4V89djSKcm71Ho/A9KDXs8/9v7cAAAAABJRU5ErkJggg=="
)
DEFAULT_DBOT_IMAGE_BASE64 = (
    "iVBORw0KGgoAAAANSUhEUgAAAEIAAABlCAYAAAD5/TVmAAAfJElEQVR4nNWceZxUxbX4v1X39jLdPQszI8uwCI"
    "iAiEuICyIxqHHFLT41MeLPZ4zRaDT5PWPM+vxEf3n56UtiTJTkPde4xaiJcU/QoA9QEYEgAUTWYWT2raf3vkvV"
    "+6N7hu6e7p4ehLzf73w+d+7tulV1zzl16tSpc06N4H8Ifrnq5LmNoWm/agwcvlBpu6s9uvFnu7bv/eWdl693Ku"
    "xCAupA4WMeqI5GA/euXFBvmOLJOv+hR0+qOR5XpZu6Ex/9+/hpgQ7gqQq7OWBMgH2MkKNsp9g3IsXu5dpgmPIE"
    "BHObB1awN7IGhYurLAlcdfQpDU9vXNE7rE2J36XKRg2DjNgf7qoR7qXaZJDWoNHYOjn0Umshx4zzFvZf7rlcGQ"
    "wfpJLlMuc6mJA3qgMRuTaVklu1zhRoIG0J1dsnf/dfz7YrRia8sM9SUGpwhpUPcuSAzrcyCAHI75+1oqerR17V"
    "2Wmu7+w06Wg3+7q6zR81f5h8IgevQig2JQ4YiAPZWQ6U0+gSUNIQnH/pRQu0I1cieKx2XvM1j393nSrTvpROGg"
    "0OuWV5/ZgMV0aVdDwSMmUQEJx73ue9SjPFSciZ2feNyc2zjv7hvQt7phzu6/jq4n9XWutCvEaL20hMy50e8mBJ"
    "RB48818/lEJ46l//w+7T2ppjF0opFpmGUaM1Xsu2vYYhldfjsTTaCtR5wtNm1q499jMT/hKq8fzZVXrvOUfeer"
    "CnbtGpMaKGLVM3t5zDZjep//zT/55umsaN776x97J3l7VO8nk8eDyZxUophe24SCnxmAYAWoNl20yaEeSCq2aF"
    "DUO8nHK496G7X1z/zMNvFVtOC3HMfTfSkj5U76BJxPKtPw0ZprwBuAUY+9rvd7Dtb/1U+X0AaK3RQM4fhNiHjv"
    "QqvnzbMRimBIi5micsl9u/uvj/drXs7Drg+BqD3yUjHTr7rIvULVc+2JZQTRXLNt09QxriceA6IAjQNDVIf6SH"
    "gf40ibhD2rJJW1b2nrks28bVDrWTExy32EtNvR9T+BBCeKXgOI9k8TmXL9y47e979rbu6RFl8ClGR265yMUZDv"
    "CqUVcf4oX37jgWwbPADABXp+lPtRBO7SHtxEglINLhIdZjkopKXFsgDfAEFKEGh5pxDlW1LtLQCGHgM0KM8U2l"
    "zj8ZQ3gB+lKuuObt19a9eMc3nzhgukOwb/7kzjdyymSJ58G6CmDM4Yfy7Ms3H+sxxAvAFKVd+tPN9CS2Y6tEzu"
    "eKDeIIGGrwGAHGVs2mzj8ZgRGxXX3Vv97y+xffeXlNLm4j0VBYd+h+oCRCLt/287FC8rqEuZYboy22gZjTndF+"
    "BwwE1d5xNAWPxWMEejSctWjGv6w/ED3LgudC7TqS+S0BXlhzh6mFuFfC3Jjdxa6BlcTsroqYMDo2aaJWB7sHVh"
    "C3exq15qFn199VP6ouioMcyWgZyZBRk2dNpHpM9UWm0BdFrQ72RtfgqGSZJvmwPyJpqQQfR98j4XQd3VDtufWG"
    "X3zlE++XKsGjrCm7fPvP66TgvYTdPbMlshpX2wXdlx5z1xa4lkApgZAa06sxPHrEdoNgCi+Tak6K+Y0xJ54+81"
    "tbRsC5UL/lvSvmmClmrpYmRotLlI7PbI2tL2ACZPfaaMCKSyJdHuLdHhJ9XpIDHpykiXYlOssIYSo8VQ6BeotA"
    "g0XtBItQo4Nh6qJD5miL9tj60PjQ/G8s/sap171y75uDr0azVQdQn0hZ/nHt3f4xteL9j6PvzY1aHRnSlSDaZd"
    "LX4ife4yUdNUlliZbCQFaonzWgXAWGQ/CQNKGxaYINNtXjLKoPsRE5EyHoGReu9cw55vNH39myv7TkeqhGvSaH"
    "qj0nDKT3zIxanQC4jmDrX2vo3VaLgSfPUjSNUr0UBwEYhgS8pLq9pLqr6dYahUPtoQPMOWsA05eZPgmnuy7lbj"
    "gPWDpaGgah3LwpC6dedAKC+IXdyQ+9g/O5fbOfrs21mMKLEAIpJE0TJuD3+ZgwYQJer5cJE8YTCARobGwY1mcw"
    "GKShoYG6ujpqaqqprx9DKBRi3Lix+Hw+hBAYwkP/zjp2vRccaqe1wlX9n//VOwv32wdbbJxKmdJ5cM9TX/dH1c"
    "7bY1bnxMGyPeuDWP2hIfGvravlttu+xcBAhGuv/TItLR9zw9euI5VOcc7ZZ/HOu6vz+ly06BROP/1UZhw2nYkT"
    "J3LCCcfRUF/PxRdfRE9PLx0dmeknpcCyHCYemcqdInVC8PCrD7bER0PHYL1iHKzI5+Cae+qj6dY5ud8SEhKpFB"
    "7TxGMaRAYiPPrbx2lubmZgYICdu3bz8CO/paO9g927m4d9eN26v7Fjx05s28a2Hfw+L4lkkm3bd9Da2obWGqU1"
    "tu0gVebLORqnXmvmAIM7skqkvKTPMve5cCueB12JrVNtNxHKRaV2go2rFKm0RTSeBMPg8su/wOEzZ3LllVcwaf"
    "Ikliy5nLlHHcn55y1GKZ25tEZrzbHHHs2555zNaact4sQTj+dznzudeZ/6FF/4wqVMmjSJWCJJLJ4klbaonWAj"
    "jfwBF4KjKyA+F4Zo3O9V4/7VC/8ZeCS3zE5L1j1TR9/HQ55ovF4vjmNjmia27eAxTVzlIqWBbduIQZNBgJSZVU"
    "XrzDZdCoHSGiklruugVIbwuiabT18SxhdyC9FaeuP8VTfuDz2jUS55U0Qpxmfs0kFeajw+xbyLwzSvC9K900u0"
    "y4NlWQBYVsbGsOzM3c0SNWSFa3BdF3eINoFLxn5wXRchINTgMGFOikM/ncAbGC75WjN+FPTkwWgYkfdl2xGmz6"
    "sp1EfeoGLmKVFmnCywEpKBdg/hVg8DHR7ivQZOWuLYZbYhAgxDY/oUwQaH2iabugk2NeMc/NVu1vIsDrYr9nvV"
    "2G+JSKSElWFEicqGxl/t4q92GTczBWRM6lTUIBWRmXtM4loCrQXS1Hj9Cl9IUVXrUlXrFh31cmBbIlUC55Hu+y"
    "8RkZjsqAlqDKPy/aPh0QTrHYIHYr9YCBoicVnow6s4GicL7oXPJSGRFJv6I/LAepeLqW5R/F1h1WhCEomLv+UU"
    "lVsRh7kbii2VFRHnOHprd7+xuqPHxLLF6PwvpeoWK9fF3w36fB1X0Bs2aOs2O1yHZTlViklBSQnJdd7qIs+5kF"
    "dn7Z+areM/P219Mi0WhKPGuERS4DgC0wRjpH2FKL9u9+/1EqgpPR6uKwhHJT1hg64+k2hctmnN9csf2Pp+z55Y"
    "btVcJ26ek7mQtkriGmXfXf/bRfWGKc8DLgUW+H267tAJjszTHZW6KrP1/vZ8HUeeHcFbpYa900Brp6kiMRkDNq"
    "D1C66jn3vlZxtbPv57XyGugzCSW2GIEfu1+8wFf8jDl3+9sEZIcWsoqH4wcayDUUTbuLZACJDmcM7EekxaN1Xx"
    "8YYq6qdYzD41RnCMs292aOjqM+gbMJ5zLHXT2uebu9a+0FzMcVsuIDUIeY7ekSzLUe9Or390UcDwykdCVeqypr"
    "EuZsGq4tqCPesC2ElJVW3GekpGDAbaTFJRg4apFlM+lcD0aYQAf42bZYKgs0fSHzXWa83i+69Y3lEB7qUidlDA"
    "rIMS6brxydNrQD/i9eiLxze6hIrYA4mwwUCbBzslMbyaQJ1D7XinqKSk0oKOHoNESq4XcOl9VyzfdaBxrkQiRh"
    "t6B+D6RxfVSI+8yxB8pTqkzMY6l3IGWDFwHEHvgCQcNXBdlmnNNUuXLN87ii4qxv+gRsOvvn+hDNR5nwcuEEAw"
    "oKgJKQJ+jWnofRoqe9c6Q3zKEkTjkkhcojJkNGulj1h65ZuFluMBg9z8iJH0gATUuKY6AsEg/T1JtMpsijweL+"
    "lUUnp9fqVcl2QqIQH++L3NasnSealMthTEEpJYQmJIME2NaYCUGgEolbEJHFfgusMWGWfDy61OIBCUpsej6sbU"
    "093ZIaVhgNZUBYIKIJGIy0DIq0I1Hpq3d5ZzQg9TqBVLxILTZ/OjpV86VsN16aTb2NueYteWCHs+itLblUKXYG"
    "P1NGt+3WHuJE+1izZcEBothnJGhoMGtERogVASN2EQbTFi/Vt8y3SJwaqt9zJ5RojD5tYyblIVVSEjIYR4cs+O"
    "7je+svhXFSn63OVzEIrFOHlt8+31HtNYqWFObgfJmEPn3iQ7N0do3hohGi506YPhgcAYQfV4CDSA4dMZGREFbn"
    "otEFqgHEGyH2IdgniPxk4P1y3+gMGUw6s5bG4NTVODhGo9iHzGhrXSnznziNu3UHwZzdt8lUovHMZFw5RzbJWe"
    "o7QNSKQwMIRJVchk6uxqps6uRrlN9HSk+Hh7jLbmBL2dSRJRF9tyiXZpol0Zd543KPAFJaZ/nxWqXHAssOKadE"
    "xndIMenHqSqpBJXaOXCYcGmDwjxPgpAUzPvvHTKBxlo7WLRiGEUWcI3+eATcO4WMTELmlQ/X7Vt80xjaF5oENS"
    "SAbSbSf0pLb9xFFW1kNtYgoffrOWgFlPlaceU2Q9U1kr0LYU/R02nS020X6bRNwmnXKwLBfXVrhKZ/QMmSQRaQ"
    "gMQ+D1Gfj8JlUBD8EaD2Mne2ho8uAP5NvuSjsknTAJp4+UE8ZWSZR20FohhUGtb/KjDb4Zj4uMVnYi4eT6fzrx"
    "JzGK2BRFGbHwjCO4/b7L7wVuGKwct3toja0vsTMVCCQ+M0TArMdv1uEzghjCixAGWgmSEYj3a5w0gytB3kZN5D"
    "wIAaYXArUQqBNIQ6NRuNrCUcks8f2knIGsBBRflhuqZqgG//QhvgEr+ntjZ1224O5h+d5FleUb2+7wJ51Ie3+q"
    "uU7pTBtXWaTcSAk2DN9KSGFgykzGi8eowpR+DLwo24OTMnFSEmXLTNwTQGoMU2P4XDx+F+m1UcLCUWlslcBRaR"
    "yVwtWV5qyDRwbwGoEsjpIaX5MTNA85/szZt2+gYPCLOmYc1x3fEd8UsNxYRRumYq+VdrHcOBZxGMI905kwBVST"
    "3USJbKnORsZ0ZgVKleo5r6uy5bZK5CSpQMLpl03BY6YAGyhQBUUZ0Z3cguXGS1NZMVaFkHXYDm4j9/3KPI3GqT"
    "Ean0YWlHboSnxYdHoXLYxY7XJ0KRwHMivmYILGcmNFcymKSkRPv6EaxwyLGewXKAXhqIHlsF/8Mk2oq1b7drH7"
    "kYaVCylLOhQxD4oyorNXquqgGvUmqRA00Nc5Bk/HUdS4gUwcpwwhWmRsrAxkKrrSpnfMhxxyaFvGsfgJUFJK0N"
    "5tOFQqEckOs6evRlsTxjreYu8r/rBjEOo8Dk96eOR7NODrCeGOex0ZSIxcuQxEYlL17fHGKOPFzoOuFbXz2z/y"
    "+ePJT5SWhFRePG4wr0wAUsqilxDFtz6G8mEW9DNasB1Be4vH7FpVc+ERRx49skScdsa5Eri2f3OV+fF4m8mTHI"
    "JVFe1bhkExsqprazj99FMzcySbhjw4W1LJFBs/2EhbW3tFfZcJluW9S1uC1g6T7o1VaJdLpk+fedeHmzfmxUCG"
    "McLvr5oKfC7db9K1NogmTm2tor7WxevJ+hBG5cUQeYh5PB4mTsqkVOzd28qkSRPzak+aNJGnf/cMsVgsJ6o6/I"
    "PlVIUmk8JkuxCOSsIDBr1bqoi3+ACmIDgTeCK3TS4jMk5MIS4G0YjWxHb7cBMS+4REVzgmGz2Glg11GaaMQHte"
    "8kIe0tkfWmveWfUuU6cdimmaGIbBUUfPxV/lp2liE9s+2lac2JFWDZHxe3R0G9iOwHVER//6QH10t8+b0+7aw2"
    "bMemrnjo8gJz9iENRZ514kgatyv5Ts9IR71gTOR3OF7Yg/JdPCKs+FHGKVxnGc/L2AGLwJUqkka957n3feeZeV"
    "K1eRTqeBbO5UDhMzKQEqr+9y37YsoSxbrNCaG52YPD66y7e2oN38WUfMnU0pE9s0zYXk+BuyzH9r08oda//wwO"
    "Y1wNPf/uOiD6CyhIy0bfPumvepMuqoq62htqYGf5U/i6/OSwkYjFwBpFJpenr7iESihCMRBiIDfHpKlDG1lXwV"
    "bJuIcvXnf/2/3gwDavEFlzwkYEEOL7xCiKura2pvjUYGIDfP8vBZcyRwJTlSkm34yEdbNw9xznZEs1IcLStYUD"
    "IuOEVfNExfOAzAQGxfJGrdBxvp7c2c8dQarvrnJVRRxa7mPazf8PchSRKjXLzStgj37IlF2DfiL2roA3LDzxd8"
    "5rOf+8mrL/2hD1BDnzh81hFNwLkFfTa7jvNGboGrxBbLrjxXshCUUkPLZDyRJBKNEYnGiMZiQ3sNpXXJrfVIml"
    "opsGyx7dkfvj80eLt3busD/lhQdYZALBr8McQIgTgThmWcPLN50wd5nmPX5e9pqzJGFNNrSmni8TiJRAKt8pfl"
    "/D1XqW+UVxKWLXBdsYmc+b9l80YF/A6w9p1ZERLBFYN1TICjjvm0BK4t6DMG+smPW3YXGhHrEylp1VariqzOQn"
    "I6elq49e4vAqDEWM44/VQgk2sVDGZ8BxnJyCW4crs6kRK4ircp8Lsq5b4jpbEJ9Lyc6uede/7Fk1596Y97TYBz"
    "zz5nXjqdPq6qqor+cJi2jk5c111r286Wgu/gOrotmRbNSjFzJD0xJN45ouGrdpl7bhg0tP/1SG6+6Wv5bbRmz5"
    "7CTOLKp2IiJRMU8Te89vLzqcUXXPJ7r9czb/LEiYypqyWVSnsj0egXJzZN/Kn5y1/ctyAUCj0//bBpZqAqwK7d"
    "u9m2bTubPvzo8WeefmyYO8hJu5GUx1ydtsTMKv/IHpvc1QAyTtpE2AAtSKdtOjo7h+qm02lefW0ZO3ftrojwQl"
    "CuIJkSW9C6aFx0+qFTXp0wftxddbW1fGresWitaWtt++HcI2ZvMoGfjJ8wbmwgECAWjTEQHqCutpaTTzy+7pmn"
    "HxvW2YNfXcGNT572eiQul1T53bIyIaQodLGT6DNZ9WBjFvG93PzNbw+9cxwH2x4eDhAVWrPxpMB2xFvP3b52UK"
    "/lue0PP2z6BZDJ0uvq7GLc+HFMaJpQ09fXf6cJLNi9czctzS3ZND6R/bg4Gfh5sQ9qxRvRuAwfMkbVS1laKgJB"
    "L7VjgnTFcv8jADhDylaRTI58yCUQ9FFXFwDK23LhmLSAlzp3RkplxnwaMh7z1r2ttLa2IRDYjj3PBPpcpcZqYN"
    "r0aQhgx46dAD2UCKK+8ZstHWfeOGdFLCEuqgmVZoQSac665HBWvthBuC9esl45qAr4OPmsQ3FqVmCVsezTliCZ"
    "lM3JAWtNTnFhIKcNoLa2lslTJrFj+04SiQRAmwn8BviB1lp2dXaRTCZxlcK27ScpEWLb9nYHZ9ww56H+iHFBdV"
    "DJErtntFbEqt/lgutOwisqNAsLwNFJ2hNrSVjlGdkfMXAVjz98w6pBsckdxOzKoR8A/dVwOOy1bXswGdYB7jGB"
    "u2Kx+HSvz7tERSLEE0l27m6ms7vbX/CtPOmwU+6KhDC2xJJybnWZfMi0E2VHOJvj9QndbKXAsgUDURlWrn6K4d"
    "MBskvpsjff6jukoYHpU6dgWRaWbRMeGFjm93iWmjd/8+uJ8y685H7TNL8kpZSO4+Jm8oDPgWFZakPMeO/ZXZGF"
    "Vx7+H919xr1Bv5bldMUQHCQfb3efgat4bqAr2VyiSibiLcRpPX193r5wGENKlNa4rvvCKy8+l5IAqVRqvW07Pe"
    "m0NcgEgNPOPPeCQqkY4vIHf/4YAU+k02JHX0T+j/mxYwlJNC4jwD1PfWt1SdE878JLARYD2cP5Dq7rOsAbkBWZ"
    "N/7yskX+6APMMA3PHMrAg9evDGu4szdsqFT6H/IfGPLAcTIpRUrz4Pt/2l1o/MmCqxGYX1Bn657mXc0UnPt8pa"
    "BSQAhx2qFTp1MKUlEbrXjOdVnW1mViO/84ZigF7T0Gli22oblrzbPDjDCVd2mOA5oK6ry8aeN6Re7uU2u9msyS"
    "mQvnT51+eNl87aVXLk9pzS1pS3S0dZk5xwwOHmgNnb0m0bhMALe8es/Gsv9HYcbMI6RGn0++/8VB69cGfwwxwr"
    "LSe4G1BX3MD4WqC7k4DJYuWb4FuCmeFKm9nQdXMpSCjh6TbB74XS0bel/eva5w/PJh1uw5fuC8guJdSquNgz+G"
    "Eg527dzGzFlzQsB5vpBi1skORyzwGdNP7p944U1Nm/o70z17t5Vey2eePG6rP+RxbEcsSiSl9Ps0ngP8f89sR9"
    "DW5SESkwp4zEo43/vdd9aUDI9PnhXixy8d3zTtuNT3m6YFzm6YLIj1u1iZMMWz772z8vlkMqEh/5SfnjlrTlug"
    "zr1x8TWHmGcsXMysqcfgeHuPjNvdXzrmlMa3XnmwpbXUR/++bK844rMT3vYFTMtxxSkDsUwujN+ri2bgloNCeV"
    "IKBqKS1k6TlCUV8Ggyat/80HUrcyM+w/LJ/8+fjp9imOJNr8+z+LNHLuGo2ccwda7J7h0dKhkR3/lgw9rBfM38"
    "jfSWTR+0zT0nsmbSxIko7dIe+4C+5A6Aeo2+/bq7jyhHknrsG++ovr3xnwJf05pwd5/B7lYP3WFjVNNlkBpXZZ"
    "LP97R5aO8xcVxhgb7bsdRND1+/MlbQrPCEgTQ84hvADKVsdvYvJ2H3cUh9E8ec5XSh9arc+sOwu3/1wpXAQoHA"
    "kD5M6SXlRFCKloGImPWDs1dWlOt4w5OnHi0Q9wKnAFIICPgVwSpNZtpopAQhMn5/rUFpcBxIW5J4UhBPDuVZAm"
    "xCc+vqZ3ctW5fJvx4J5C/f+cybhtSnmNKHKf2knQgI0Fp33Th/1bjcyoN5lrkdtwEY0sOcxvOJWd3sCr9Ff8To"
    "273Drjhd5eHrV228+r6Tz5emvAi4TWvmxJNSxrObTSkyTlmBzka9MozYR/iQPb4XuFcp/ejTt73X099WcfxT9f"
    "TLjrH1LkFPI7MazmZb7zLC6Y8BhvkrhkvEuwsXIXgJRMhjVJG2k4SjkmRa/GLiWOevwK7WHbGt/7ZkQ8VxwCvv"
    "OckMNfoXSCkuBRYAU4A6hkfaFBAhMxhr0foP6aT7xoPXrqiY+ruXzfcGq82FCOb1R+RE1xXfHFOj8Hl8KG2jtW"
    "tpzdduPGnVw2UZcdsjx8jGqbUXJNPix0oz22tq6fNqfD5tCfACYeC+dMK9819Oe3fkYE8BXPfIIr/hkeOBsULQ"
    "iBAhAVKjEyh6gC7l6o7lD3wY+2jVSAn4+fCrt0+ukYb4GbAE8EMmm9eyIZWWKE2L36vvchOpB7933hqLkTJv/d"
    "Uefrps/iUC/R8ISh1Fe1kprr3trNUdiWjlCV45sL9nRIa1mzN/DF//xZHTNTxORuLQSmSO7ezbDFpa892l39r6"
    "8y1vD7c7ijLi3hULAqZXbgeaNBCNSQZiEinhkDGZYHAW1iqlr7hpwdvbDgRB+9vuvtULjxPwJDATMi67rr6MlV"
    "sdVBxSr5CZDJSeVMKddstp7xauOMXzI0yvnAmMBegfMGjtMnuicXnHQERe3dJuPpNIycGV4zgpxV/vX71w0c+X"
    "nzTaZIr9PTE01O4HT31K3v/uwosE/IUsEyJx2dfaaX4/kRRHWpY4ozdsrO/uG0Ktxuc3phTrtBTyocGH/ohMob"
    "n21Z9t/NH9S5Y/Gh1wr+jsNa5LpsWgApsE/MFXZXxx/uKxlTCj1JHDUu+GHU0EuPSW6XLCtODXETxJNpQXT4qe"
    "7j7jn9p2xv5t6ZLlW39z9VtvaM3VkZgRzmlbGI+RpZAhd8a4LpscS728e32PAnjw2hXO3u2Jx7r7jOssWwyKWD"
    "2Ch6784czrb77vqJGYUcyDVO7dsDzxxiY/iy5tugzB7UAAIG2Jrp5+4wvP3vHBW89+/30A5dqKrSvaN2n0+pwz"
    "7EXzzouOjhAMGU1Ssuk3V7+Vpw2f/s57NG+NPxWJyRuVIgFsBX6ilX6jfVfJ/Ujht8pJxOBzbirg0LuethR7Nk"
    "efA04H7lZabIgl5BXP//TDt3JO+kmA5f/5oTIkO7IJGwpEqti3i2bna62HGGEaDFMsAL//7vvqsjvmPXHiZ4Nr"
    "TOnufeyO7Yl1r3eXm/elXOyFdQaVYd4pvML6d1/zgQNsWHztlI0nXdD0o2fuaUlsf6cT8pWpBJTHHKRHozXFLO"
    "Oi/34J0FaGe+D16pK2wjP/ul49k5GGSqDcMcTcOhTUKatUX3mgRb3yQEuuwTWsnS+HBq1VMXpUUUZohTOYk+Av"
    "w4j9hEqk5oBClW+IBoUWRRNOSynLIZ0Q8Kv9i8wMh4NCZCVQ5dPbyYbJhCyOR3FGZBRsDFjqNfVvDg56ZeGTJX"
    "gWQH974gngLGCNLmU7FcVCEtGaM1a/0rX+8Tv3x2j8xHBApedHl61zgLfufn3+GdFee7/6LjcyRQ2d/axfaqks"
    "VVbYxyd99/80/H+BZCGMBumi5yUq6G+0EliufaEE/sNgJBEv1WY0v0eLT959NAGIYYbQxIkTefXPf5ZVPr/0+r"
    "ym3+evkdJodJVbL6WsEUKEyOwFAhq8IuMsMQFPtgszeznsO/llZ58tMie7EkBCaRVDE9FKhTX0KNcNt7a1OR3t"
    "7eorX7lGdXd3V4x3MRCUMGFHgvnz5/PiSy+fCdxEJlt3fJboAxb+H6GbFBmX3jbg12MPaXyx4H0xBhRat0N3s0"
    "jliiBUXY1lWWs9Hs+PgZlCMAXNBIRoRFMH1JDZzgeyl5/MFthk+FY4FyyyEqEzxKbI2DQxMv7MMJnQZDvovVqz"
    "A1HUzK9kdzt0/2+exnQr4g2hrAAAAABJRU5ErkJggg=="
)
# structure regex of the from and to version
FROM_TO_VERSION_REGEX = re.compile(r"(?:\d{1,2}\.){2}\d{1,2}")
# file types regexes
PIPFILE_REGEX = r".*/Pipfile(\.lock)?"
TEST_DATA_REGEX = r".*test_data.*"
TEST_FILES_REGEX = r".*test_files.*"
DOCS_REGEX = r".*docs.*"
PNG_IMAGE_REGEX = r".*\.png$"
SVG_IMAGE_REGEX = r".*\.svg$"
DESCRIPTION_REGEX = r".*\.md"
SCHEMA_REGEX = "Tests/schemas/.*.yml"
# Checks if playbookID is a UUID format
INVALID_PLAYBOOK_ID = r"[\w\d]{8}-[\w\d]{4}-[\w\d]{4}-[\w\d]{4}-[\w\d]{12}"

# regex pattern used to convert incident/indicator fields to their CLI names
NON_LETTERS_OR_NUMBERS_PATTERN = re.compile(r"[^a-zA-Z0-9]")
INCORRECT_PACK_NAME_PATTERN = (
    "[^a-zA-Z]pack[^a-z]|^pack$|^pack[^a-z]|[^a-zA-Z]pack$|[^A-Z]PACK[^A-Z]|^PACK$|^PACK["
    "^A-Z]|[^A-Z]PACK$|[^A-Z]Pack[^a-z]|^Pack$|^Pack[^a-z]|[^A-Z]Pack$|[^a-zA-Z]playbook["
    "^a-z]|^playbook$|^playbook[^a-z]|[^a-zA-Z]playbook$|[^A-Z]PLAYBOOK["
    "^A-Z]|^PLAYBOOK$|^PLAYBOOK[^A-Z]|[^A-Z]PLAYBOOK$|[^A-Z]Playbook["
    "^a-z]|^Playbook$|^Playbook[^a-z]|[^A-Z]Playbook$|[^a-zA-Z]integration["
    "^a-z]|^integration$|^integration[^a-z]|[^a-zA-Z]integration$|[^A-Z]INTEGRATION["
    "^A-Z]|^INTEGRATION$|^INTEGRATION[^A-Z]|[^A-Z]INTEGRATION$|[^A-Z]Integration["
    "^a-z]|^Integration$|^Integration[^a-z]|[^A-Z]Integration$|[^a-zA-Z]script["
    "^a-z]|^script$|^script[^a-z]|[^a-zA-Z]script$|[^A-Z]SCRIPT[^A-Z]|^SCRIPT$|^SCRIPT["
    "^A-Z]|[^A-Z]SCRIPT$|[^A-Z]Script[^a-z]|^Script$|^Script[^a-z]|[^A-Z]Script$ "
)

PACKS_DIR_REGEX = rf"{CAN_START_WITH_DOT_SLASH}{PACKS_DIR}"
PACK_DIR_REGEX = rf"{PACKS_DIR_REGEX}\/([^\\\/]+)"

INTEGRATIONS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{INTEGRATIONS_DIR}"
INTEGRATION_PACKAGE_REGEX = rf"{INTEGRATIONS_DIR_REGEX}\/([^\\/]+)"

PACKS_INTEGRATION_PY_REGEX = rf"{INTEGRATION_PACKAGE_REGEX}\/\2\.py"
PACKS_INTEGRATION_TEST_PY_REGEX = rf"{INTEGRATION_PACKAGE_REGEX}/\2_test\.py"
PACKS_INTEGRATION_PS_REGEX = rf"{INTEGRATION_PACKAGE_REGEX}/\2.ps1$"
PACKS_INTEGRATION_PS_TEST_REGEX = rf"{INTEGRATION_PACKAGE_REGEX}/\2\.Tests\.ps1$"
PACKS_INTEGRATION_YML_REGEX = rf"{INTEGRATION_PACKAGE_REGEX}/\2\.yml"
PACKS_INTEGRATION_README_REGEX = rf"{INTEGRATION_PACKAGE_REGEX}/README.md$"

PACKS_INTEGRATION_NON_SPLIT_BASE_REGEX = (
    rf"{INTEGRATIONS_DIR_REGEX}/integration-([^\\/]+)"
)
PACKS_INTEGRATION_NON_SPLIT_YML_REGEX = (
    rf"{PACKS_INTEGRATION_NON_SPLIT_BASE_REGEX}\.yml$"
)
PACKS_INTEGRATION_NON_SPLIT_README_REGEX = (
    rf"{PACKS_INTEGRATION_NON_SPLIT_BASE_REGEX}_README.md$"
)

SCRIPTS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{SCRIPTS_DIR}"
SCRIPT_DIR_REGEX = rf"{SCRIPTS_DIR_REGEX}\/([^\\/]+)"
SCRIPT_TYPE_REGEX = ".*script-.*.yml"
PACKS_SCRIPT_PY_REGEX = rf"{SCRIPT_DIR_REGEX}/\2\.py"
PACKS_SCRIPT_TEST_PY_REGEX = rf"{SCRIPT_DIR_REGEX}/\2_test\.py"
PACKS_SCRIPT_PS_REGEX = rf"{SCRIPT_DIR_REGEX}/\2.ps1$"
PACKS_SCRIPT_TEST_PS_REGEX = rf"{SCRIPT_DIR_REGEX}/\2\.Tests\.ps1$"
PACKS_SCRIPT_YML_REGEX = rf"{SCRIPT_DIR_REGEX}\/\2\.yml"
PACKS_SCRIPT_README_REGEX = rf"{SCRIPT_DIR_REGEX}/README.md$"

PACKS_SCRIPT_NON_SPLIT_BASE_REGEX = rf"{SCRIPTS_DIR_REGEX}/script-([^\\/]+)"
PACKS_SCRIPT_TEST_PLAYBOOK = (
    rf"{PACK_DIR_REGEX}/{TEST_PLAYBOOKS_DIR}/script-([^\\/]+).yml$"
)
PACKS_SCRIPT_NON_SPLIT_YML_REGEX = rf"{PACKS_SCRIPT_NON_SPLIT_BASE_REGEX}\.yml$"
PACKS_SCRIPT_NON_SPLIT_README_REGEX = rf"{PACKS_SCRIPT_NON_SPLIT_BASE_REGEX}_README.md$"

PACKS_LAYOUTS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{LAYOUTS_DIR}"
PACKS_LAYOUT_JSON_REGEX = (
    rf"{PACKS_LAYOUTS_DIR_REGEX}\/(?!layoutscontainer)([^/]+)\.json"
)

PACKS_LAYOUTS_CONTAINER_JSON_REGEX = (
    rf"{PACKS_LAYOUTS_DIR_REGEX}\/layoutscontainer([^/]+)\.json"
)

PACK_LAYOUT_RULES_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{LAYOUT_RULES_DIR}"
PACK_LAYOUT_RULE_JSON_REGEX = rf"{PACK_LAYOUT_RULES_DIR_REGEX}\/([^/]+)\.json"

PACKS_PRE_PROCESS_RULES_DIR_REGEX = rf"{PACK_DIR_REGEX}/{PRE_PROCESS_RULES_DIR}"
PACKS_PRE_PROCESS_RULES_JSON_REGEX = (
    rf"{PACKS_PRE_PROCESS_RULES_DIR_REGEX}/(?:preprocessrule-)?([^/]+)\.json"
)

PACKS_WIDGETS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{WIDGETS_DIR}"
PACKS_WIDGET_JSON_REGEX = rf"{PACKS_WIDGETS_DIR_REGEX}\/([^/]+)\.json"

PACKS_DASHBOARDS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{DASHBOARDS_DIR}"
PACKS_DASHBOARD_JSON_REGEX = rf"{PACKS_DASHBOARDS_DIR_REGEX}\/([^/]+)\.json"

PACKS_REPORTS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{REPORTS_DIR}"
PACKS_REPORT_JSON_REGEX = rf"{PACKS_REPORTS_DIR_REGEX}\/([^/]+)\.json"

PACKS_INCIDENT_TYPES_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{INCIDENT_TYPES_DIR}"
PACKS_INCIDENT_TYPE_JSON_REGEX = rf"{PACKS_INCIDENT_TYPES_DIR_REGEX}\/([^/]+)\.json"

PACKS_INCIDENT_FIELDS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{INCIDENT_FIELDS_DIR}"
PACKS_INCIDENT_FIELD_JSON_REGEX = rf"{PACKS_INCIDENT_FIELDS_DIR_REGEX}\/([^/]+)\.json"

PACKS_INDICATOR_TYPES_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{INDICATOR_TYPES_DIR}"
PACKS_INDICATOR_TYPE_JSON_REGEX = rf"{PACKS_INDICATOR_TYPES_DIR_REGEX}\/([^/]+)\.json"

PACKS_INDICATOR_FIELDS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{INDICATOR_FIELDS_DIR}"
PACKS_INDICATOR_FIELD_JSON_REGEX = rf"{PACKS_INDICATOR_FIELDS_DIR_REGEX}\/([^/]+)\.json"

PACKS_GENERIC_TYPES_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{GENERIC_TYPES_DIR}\/([^\\\/]+)"
PACKS_GENERIC_TYPE_JSON_REGEX = rf"{PACKS_GENERIC_TYPES_DIR_REGEX}\/([^/]+)\.json"

PACKS_GENERIC_FIELDS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{GENERIC_FIELDS_DIR}\/([^\\\/]+)"
PACKS_GENERIC_FIELD_JSON_REGEX = rf"{PACKS_GENERIC_FIELDS_DIR_REGEX}\/([^/]+)\.json"

PACKS_GENERIC_MODULES_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{GENERIC_MODULES_DIR}"
PACKS_GENERIC_MODULE_JSON_REGEX = rf"{PACKS_GENERIC_MODULES_DIR_REGEX}\/([^/]+)\.json"

PACKS_GENERIC_DEFINITIONS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{GENERIC_DEFINITIONS_DIR}"
PACKS_GENERIC_DEFINITION_JSON_REGEX = (
    rf"{PACKS_GENERIC_DEFINITIONS_DIR_REGEX}\/([^/]+)\.json"
)

PACKS_CLASSIFIERS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{CLASSIFIERS_DIR}"

_PACKS_CLASSIFIER_BASE_REGEX = (
    rf"{PACKS_CLASSIFIERS_DIR_REGEX}\/*classifier-(?!mapper).*(?<!5_9_9)"
)
PACKS_CLASSIFIER_JSON_REGEX = rf"{_PACKS_CLASSIFIER_BASE_REGEX}\.json"

JOBS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{JOBS_DIR}"
JOB_JSON_REGEX = rf"{JOBS_DIR_REGEX}\/job-([^/]+)\.json"

WIZARD_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{WIZARDS_DIR}"
WIZARD_JSON_REGEX = rf"{WIZARD_DIR_REGEX}\/wizard-([^/]+)\.json"

XSIAM_DASHBOARD_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{XSIAM_DASHBOARDS_DIR}"
XSIAM_DASHBOARD_JSON_REGEX = rf"{XSIAM_DASHBOARD_DIR_REGEX}\/([^/]+)\.json"

XSIAM_REPORT_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{XSIAM_REPORTS_DIR}"
XSIAM_REPORT_JSON_REGEX = rf"{XSIAM_REPORT_DIR_REGEX}\/([^/]+)\.json"

TRIGGER_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{TRIGGER_DIR}"
TRIGGER_JSON_REGEX = rf"{TRIGGER_DIR_REGEX}\/([^/]+)\.json"

XDRC_TEMPLATE_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{XDRC_TEMPLATE_DIR}"
XDRC_TEMPLATE_PACKAGE_REGEX = rf"{XDRC_TEMPLATE_DIR_REGEX}\/([^\\/]+)"
XDRC_TEMPLATE_JSON_REGEX = rf"{XDRC_TEMPLATE_PACKAGE_REGEX}\/([^/]+)\.json"
XDRC_TEMPLATE_YML_REGEX = rf"{XDRC_TEMPLATE_PACKAGE_REGEX}\/([^/]+)\.yml"

CORRELATION_RULES_DIR_REGEX = rf"{PACK_DIR_REGEX}/{CORRELATION_RULES_DIR}"
CORRELATION_RULES_YML_REGEX = (
    rf"{CORRELATION_RULES_DIR_REGEX}/(?:correlationrule-)?([^/]+)\.yml"
)

PARSING_RULES_DIR_REGEX = rf"{PACK_DIR_REGEX}/{PARSING_RULES_DIR}"
PARSING_RULES_PACKAGE_REGEX = rf"{PARSING_RULES_DIR_REGEX}\/([^\\/]+)"
PARSING_RULE_YML_REGEX = rf"{PARSING_RULES_PACKAGE_REGEX}/(?:parsingrule-)?([^/]+)\.yml"
PARSING_RULE_RULES_REGEX = rf"{PARSING_RULES_PACKAGE_REGEX}\/([^/]+)\.xif"


# Modeling Rules
MODELING_RULE_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{MODELING_RULES_DIR}"
MODELING_RULE_PACKAGE_REGEX = rf"{MODELING_RULE_DIR_REGEX}\/([^\\/]+)"
MODELING_RULE_YML_REGEX = rf"{MODELING_RULE_PACKAGE_REGEX}\/([^/]+)\.yml"
MODELING_RULE_RULES_REGEX = rf"{MODELING_RULE_PACKAGE_REGEX}\/([^/]+)\.xif"
MODELING_RULE_SCHEMA_REGEX = rf"{MODELING_RULE_PACKAGE_REGEX}\/([^/]+)\.json"

RELATIVE_HREF_URL_REGEX = r'(<.*?href\s*=\s*"((?!(?:https?:\/\/)|#|(?:mailto:)).*?)")'
RELATIVE_MARKDOWN_URL_REGEX = (
    r"(?<![!])(\[.*?\])\(((?!(?:https?:\/\/)|#|(?:mailto:)).*?)\)"
)
URL_REGEX = r"(((http|https)\:\/\/)?[a-zA-Z0-9\.\/\?\:@\-_=#]+\.([a-zA-Z]){2,6}([a-zA-Z0-9\.\&\/\?\:@\-_=#])*)"

# old classifier structure
_PACKS_CLASSIFIER_BASE_5_9_9_REGEX = (
    rf"{PACKS_CLASSIFIERS_DIR_REGEX}\/*classifier-(?!mapper).*_5_9_9"
)
PACKS_CLASSIFIER_JSON_5_9_9_REGEX = rf"{_PACKS_CLASSIFIER_BASE_5_9_9_REGEX}\.json"

_PACKS_MAPPER_BASE_REGEX = rf"{PACKS_CLASSIFIERS_DIR_REGEX}\/classifier-(?=mapper).*"
PACKS_MAPPER_JSON_REGEX = rf"{_PACKS_MAPPER_BASE_REGEX}\.json"

PACKS_RELEASE_NOTES_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{RELEASE_NOTES_DIR}"

PLAYBOOKS_DIR_REGEX = rf"{PACK_DIR_REGEX}\/{PLAYBOOKS_DIR}"
PLAYBOOK_BASE_REGEX = rf"{PLAYBOOKS_DIR_REGEX}\/.*"
PLAYBOOK_YML_REGEX = rf"{PLAYBOOK_BASE_REGEX}\.yml"
PLAYBOOK_README_REGEX = rf"{PLAYBOOK_BASE_REGEX}_README\.md$"

TEST_SCRIPT_REGEX = rf"{CAN_START_WITH_DOT_SLASH}{TEST_PLAYBOOKS_DIR}.*script-.*\.yml$"
TEST_PLAYBOOK_YML_REGEX = (
    rf"{PACK_DIR_REGEX}/{TEST_PLAYBOOKS_DIR}\/(?!script-)([^.]+)\.yml"
)

PACKS_INDICATOR_TYPES_REPUTATIONS_REGEX = r"{}{}/([^/]+)/{}/reputations.json".format(
    CAN_START_WITH_DOT_SLASH, PACKS_DIR, INDICATOR_TYPES_DIR
)
PACKS_RELEASE_NOTES_REGEX = r"{}{}/([^/]+)/{}/([^/]+)\.md$".format(
    CAN_START_WITH_DOT_SLASH, PACKS_DIR, RELEASE_NOTES_DIR
)
PACKS_TOOLS_REGEX = (
    rf"{CAN_START_WITH_DOT_SLASH}{PACKS_DIR}/([^/]+)/{TOOLS_DIR}/([^.]+)\.zip"
)

PLAYBOOK_REGEX = r"playbook-.*\.yml$"

PLAYBOOK_REGEX_PATH = (
    rf"{CAN_START_WITH_DOT_SLASH}(?!Test){PLAYBOOKS_DIR}/{PLAYBOOK_REGEX}"
)

TEST_PLAYBOOK_REGEX = (
    rf"{CAN_START_WITH_DOT_SLASH}{TEST_PLAYBOOKS_DIR}/(?!script-).*\.yml$"
)
TEST_NOT_PLAYBOOK_REGEX = (
    rf"{CAN_START_WITH_DOT_SLASH}{TEST_PLAYBOOKS_DIR}/(?!playbook).*-.*\.yml$"
)

INDICATOR_TYPES_REPUTATIONS_REGEX = (
    rf"{CAN_START_WITH_DOT_SLASH}{INDICATOR_TYPES_DIR}.reputations\.json$"
)

INCIDENT_FIELD_FILE_NAME_REGEX = r"incidentfield-.*\.json"

LAYOUT_FILE_NAME__REGEX = r"layout.*\.json"

VALID_INDICATOR_TYPE_REGEX = "^[A-Za-z0-9_& ]*$"

# deprecated regex
DEPRECATED_DESC_REGEX = r"Deprecated\.\s*(.*?Use .*? instead\.*?)"
DEPRECATED_NO_REPLACE_DESC_REGEX = r"Deprecated\.\s*(.*?No available replacement\.*?)"
PACK_NAME_DEPRECATED_REGEX = r".* \(Deprecated\)"
DEPRECATED_COMMAND_REGEX = r"Command \*\*\*.*?\*\*\* is deprecated. Use .*? instead."

DEPRECATED_REGEXES: List[str] = [
    DEPRECATED_DESC_REGEX,
    DEPRECATED_NO_REPLACE_DESC_REGEX,
    PACK_NAME_DEPRECATED_REGEX,
    DEPRECATED_COMMAND_REGEX,
]

PACK_METADATA_NAME: str = "name"
PACK_METADATA_DESC: str = "description"
PACK_METADATA_SUPPORT: str = "support"
PACK_METADATA_MIN_VERSION: str = "serverMinVersion"
PACK_METADATA_CURR_VERSION: str = "currentVersion"
PACK_METADATA_AUTHOR: str = "author"
PACK_METADATA_URL: str = "url"
PACK_METADATA_EMAIL: str = "email"
PACK_METADATA_CATEGORIES: str = "categories"
PACK_METADATA_MODULES: str = "modules"
PACK_METADATA_TAGS: str = "tags"
PACK_METADATA_CREATED: str = "created"
PACK_METADATA_CERTIFICATION: str = "certification"
PACK_METADATA_USE_CASES: str = "useCases"
PACK_METADATA_KEYWORDS: str = "keywords"
PACK_METADATA_PRICE: str = "price"
PACK_METADATA_DEPENDENCIES: str = "dependencies"
PACK_METADATA_IRON_BANK_TAG: str = "Iron Bank"
PACK_METADATA_SERVER_MIN_VERSION: str = "serverMinVersion"
PACK_METADATA_EXCLUDED_DEPENDENCIES: str = "excludedDependencies"

ALLOWED_CERTIFICATION_VALUES = ["certified", "verified"]
USE_CASE_TAG: str = "Use Case"
MANDATORY_PACK_METADATA_FIELDS = (
    PACK_METADATA_NAME,
    PACK_METADATA_DESC,
    PACK_METADATA_SUPPORT,
    PACK_METADATA_CURR_VERSION,
    PACK_METADATA_AUTHOR,
    PACK_METADATA_URL,
    PACK_METADATA_CATEGORIES,
    PACK_METADATA_TAGS,
    PACK_METADATA_USE_CASES,
    PACK_METADATA_KEYWORDS,
    MARKETPLACE_KEY_PACK_METADATA,
)
PACK_METADATA_MANDATORY_FILLED_FIELDS = [
    PACK_METADATA_KEYWORDS,
    PACK_METADATA_TAGS,
    PACK_METADATA_CATEGORIES,
    PACK_METADATA_USE_CASES,
]

PACK_METADATA_REQUIRE_RN_FIELDS: set = {
    PACK_METADATA_SUPPORT,
    PACK_METADATA_DEPENDENCIES,
    PACK_METADATA_NAME,
    PACK_METADATA_PRICE,
    MARKETPLACE_KEY_PACK_METADATA,
    PACK_METADATA_SERVER_MIN_VERSION,
    PACK_METADATA_EXCLUDED_DEPENDENCIES,
}

API_MODULES_PACK = "ApiModules"
API_MODULE_FILE_SUFFIX = "ApiModule"
API_MODULE_PY_REGEX = r"{}{}/{}/{}/([^/]+)/([^.]+)\.py".format(
    CAN_START_WITH_DOT_SLASH, PACKS_DIR, API_MODULES_PACK, SCRIPTS_DIR
)
API_MODULE_YML_REGEX = r"{}{}/{}/{}/([^/]+)/([^.]+)\.yml".format(
    CAN_START_WITH_DOT_SLASH, PACKS_DIR, API_MODULES_PACK, SCRIPTS_DIR
)
API_MODULE_REGEXES = [API_MODULE_PY_REGEX, API_MODULE_YML_REGEX]

ID_IN_COMMONFIELDS = [  # entities in which 'id' key is under 'commonfields'
    "integration",
    "betaintegration",
    "script",
]
ID_IN_ROOT = [  # entities in which 'id' key is in the root
    "playbook",
    "dashboard",
    "incident_type",
    "layoutscontainer",
    "mapper",
    "pre_process_rule",
    "lists",
    JOB,
    WIZARD,
    "classifier",
]

INTEGRATION_PREFIX = "integration"
SCRIPT_PREFIX = "script"
PARSING_RULE_PREFIX = "parsingrule"
PARSING_RULE_ID_SUFFIX = "ParsingRule"
PARSING_RULE_NAME_SUFFIX = "Parsing Rule"
MODELING_RULE_PREFIX = "modelingrule"
MODELING_RULE_ID_SUFFIX = "ModelingRule"
MODELING_RULE_NAME_SUFFIX = "Modeling Rule"
ASSETS_MODELING_RULE_NAME_SUFFIX = "Asset Collection"
XDRC_TEMPLATE_PREFIX = "xdrctemplate"
LAYOUT_RULE_PREFIX = "layoutrule"
ASSETS_MODELING_RULE_ID_SUFFIX = "AssetsModelingRule"

# Pack Unique Files
PACKS_WHITELIST_FILE_NAME = ".secrets-ignore"
PACKS_PACK_IGNORE_FILE_NAME = ".pack-ignore"
PACKS_PACK_META_FILE_NAME = "pack_metadata.json"
PACKS_README_FILE_NAME = INTEGRATIONS_README_FILE_NAME = SCRIPTS_README_FILE_NAME = (
    "README.md"
)
PACKS_CONTRIBUTORS_FILE_NAME = "CONTRIBUTORS.json"
PACKS_VERSION_CONFIG_FILE_NAME = "version_config.json"
AUTHOR_IMAGE_FILE_NAME = "Author_image.png"
PACKS_FOLDER = "Packs"
PRIVATE_PACKS_FOLDER = "PrivatePacks"
GIT_IGNORE_FILE_NAME = ".gitignore"

CONF_JSON_FILE_NAME = "conf.json"
VERSION_CONFIG_FILE_NAME = "version_config.json"

PYTHON_TEST_REGEXES = [PACKS_SCRIPT_TEST_PY_REGEX, PACKS_INTEGRATION_TEST_PY_REGEX]

PYTHON_INTEGRATION_REGEXES = [PACKS_INTEGRATION_PY_REGEX]

PLAYBOOKS_REGEXES_LIST = [PLAYBOOK_REGEX_PATH, TEST_PLAYBOOK_REGEX]

PYTHON_SCRIPT_REGEXES = [PACKS_SCRIPT_PY_REGEX]

PYTHON_ALL_REGEXES: List[str] = sum(
    [PYTHON_SCRIPT_REGEXES, PYTHON_INTEGRATION_REGEXES, PYTHON_TEST_REGEXES], []
)

INTEGRATION_REGXES: List[str] = [PACKS_INTEGRATION_NON_SPLIT_YML_REGEX]

IMAGE_ALL_REGEXES: List[str] = [
    PNG_IMAGE_REGEX,
    SVG_IMAGE_REGEX,
]
YML_INTEGRATION_REGEXES: List[str] = [
    PACKS_INTEGRATION_YML_REGEX,
    PACKS_INTEGRATION_NON_SPLIT_YML_REGEX,
]

YML_ALL_INTEGRATION_REGEXES: List[str] = sum(
    [
        YML_INTEGRATION_REGEXES,
    ],
    [],
)

YML_SCRIPT_REGEXES: List[str] = [
    PACKS_SCRIPT_YML_REGEX,
    PACKS_SCRIPT_NON_SPLIT_YML_REGEX,
    PACKS_SCRIPT_TEST_PLAYBOOK,
]

YML_ALL_SCRIPTS_REGEXES: List[str] = sum([YML_SCRIPT_REGEXES], [])

YML_PLAYBOOKS_NO_TESTS_REGEXES: List[str] = [PLAYBOOK_YML_REGEX]

YML_TEST_PLAYBOOKS_REGEXES: List[str] = [TEST_PLAYBOOK_YML_REGEX]

YML_ALL_PLAYBOOKS_REGEX: List[str] = sum(
    [
        YML_PLAYBOOKS_NO_TESTS_REGEXES,
        YML_TEST_PLAYBOOKS_REGEXES,
    ],
    [],
)

YML_ALL_REGEXES: List[str] = sum(
    [
        YML_INTEGRATION_REGEXES,
        YML_SCRIPT_REGEXES,
        YML_PLAYBOOKS_NO_TESTS_REGEXES,
        YML_TEST_PLAYBOOKS_REGEXES,
    ],
    [],
)

JSON_INDICATOR_AND_INCIDENT_FIELDS = [
    PACKS_INCIDENT_FIELD_JSON_REGEX,
    PACKS_INDICATOR_FIELD_JSON_REGEX,
]

JSON_ALL_WIDGETS_REGEXES = [
    PACKS_WIDGET_JSON_REGEX,
]

JSON_ALL_DASHBOARDS_REGEXES = [
    PACKS_DASHBOARD_JSON_REGEX,
]

JSON_ALL_CLASSIFIER_REGEXES = [
    PACKS_CLASSIFIER_JSON_REGEX,
]

JSON_ALL_CLASSIFIER_REGEXES_5_9_9 = [
    PACKS_CLASSIFIER_JSON_5_9_9_REGEX,
]

JSON_ALL_MAPPER_REGEXES = [
    PACKS_MAPPER_JSON_REGEX,
]

JSON_ALL_LAYOUT_REGEXES = [
    PACKS_LAYOUT_JSON_REGEX,
]

JSON_ALL_LAYOUTS_CONTAINER_REGEXES = [
    PACKS_LAYOUTS_CONTAINER_JSON_REGEX,
]

JSON_ALL_LAYOUT_RULES_REGEXES = [PACK_LAYOUT_RULE_JSON_REGEX]

JSON_ALL_PRE_PROCESS_RULES_REGEXES = [
    PACKS_PRE_PROCESS_RULES_JSON_REGEX,
]

JSON_ALL_INCIDENT_FIELD_REGEXES = [
    PACKS_INCIDENT_FIELD_JSON_REGEX,
]

JSON_ALL_INCIDENT_TYPES_REGEXES = [
    PACKS_INCIDENT_TYPE_JSON_REGEX,
]

JSON_ALL_INDICATOR_FIELDS_REGEXES = [PACKS_INDICATOR_FIELD_JSON_REGEX]

JSON_ALL_INDICATOR_TYPES_REGEXES = [PACKS_INDICATOR_TYPE_JSON_REGEX]

JSON_ALL_GENERIC_FIELDS_REGEXES = [
    PACKS_GENERIC_FIELD_JSON_REGEX,
]

JSON_ALL_GENERIC_TYPES_REGEXES = [
    PACKS_GENERIC_TYPE_JSON_REGEX,
]

JSON_ALL_GENERIC_MODULES_REGEXES = [
    PACKS_GENERIC_MODULE_JSON_REGEX,
]

JSON_ALL_GENERIC_DEFINITIONS_REGEXES = [
    PACKS_GENERIC_DEFINITION_JSON_REGEX,
]

JSON_ALL_REPUTATIONS_INDICATOR_TYPES_REGEXES = [PACKS_INDICATOR_TYPES_REPUTATIONS_REGEX]

JSON_ALL_REPORTS_REGEXES = [PACKS_REPORT_JSON_REGEX]

JSON_ALL_JOB_REGEXES = [JOB_JSON_REGEX]

JSON_ALL_WIZARD_REGEXES = [
    WIZARD_JSON_REGEX,
]

CHECKED_TYPES_REGEXES = [
    # Playbooks
    PLAYBOOK_YML_REGEX,
    TEST_PLAYBOOK_YML_REGEX,
    # Integrations
    PACKS_INTEGRATION_YML_REGEX,
    PACKS_INTEGRATION_PY_REGEX,
    PACKS_INTEGRATION_PS_REGEX,
    PACKS_INTEGRATION_TEST_PY_REGEX,
    PACKS_INTEGRATION_README_REGEX,
    PACKS_INTEGRATION_NON_SPLIT_YML_REGEX,
    # Scripts yaml
    PACKS_SCRIPT_YML_REGEX,
    PACKS_SCRIPT_NON_SPLIT_YML_REGEX,
    PACKS_SCRIPT_PY_REGEX,
    PACKS_SCRIPT_PS_REGEX,
    PACKS_SCRIPT_TEST_PY_REGEX,
    PACKS_SCRIPT_README_REGEX,
    PACKS_SCRIPT_TEST_PLAYBOOK,
    # Modeling Rules
    MODELING_RULE_YML_REGEX,
    MODELING_RULE_RULES_REGEX,
    MODELING_RULE_SCHEMA_REGEX,
    # # Parsing Rules
    PARSING_RULE_YML_REGEX,
    PARSING_RULE_RULES_REGEX,
    PACKS_CLASSIFIER_JSON_REGEX,
    PACKS_CLASSIFIER_JSON_5_9_9_REGEX,
    PACKS_MAPPER_JSON_REGEX,
    PACKS_DASHBOARD_JSON_REGEX,
    PACKS_INCIDENT_TYPE_JSON_REGEX,
    PACKS_INCIDENT_FIELD_JSON_REGEX,
    PACKS_INDICATOR_FIELD_JSON_REGEX,
    PACKS_INDICATOR_TYPE_JSON_REGEX,
    PACKS_LAYOUT_JSON_REGEX,
    PACKS_LAYOUTS_CONTAINER_JSON_REGEX,
    PACKS_PRE_PROCESS_RULES_JSON_REGEX,
    PACKS_WIDGET_JSON_REGEX,
    PACKS_REPORT_JSON_REGEX,
    PACKS_RELEASE_NOTES_REGEX,
    PACKS_TOOLS_REGEX,
    JOB_JSON_REGEX,
    WIZARD_JSON_REGEX,
    # ReleaseNotes
    PACKS_RELEASE_NOTES_REGEX,
]

CHECKED_TYPES_NO_REGEX = [
    item.replace(CAN_START_WITH_DOT_SLASH, "").replace(NOT_TEST, "")
    for item in CHECKED_TYPES_REGEXES
]

PATHS_TO_VALIDATE: List[str] = sum([PYTHON_ALL_REGEXES, JSON_ALL_REPORTS_REGEXES], [])

PACKAGE_SCRIPTS_REGEXES = [PACKS_SCRIPT_PY_REGEX, PACKS_SCRIPT_YML_REGEX]

PACKAGE_SUPPORTING_DIRECTORIES = [
    INTEGRATIONS_DIR,
    SCRIPTS_DIR,
    MODELING_RULES_DIR,
    PARSING_RULES_DIR,
]

IGNORED_TYPES_REGEXES = [
    DESCRIPTION_REGEX,
    PNG_IMAGE_REGEX,
    SVG_IMAGE_REGEX,
    PIPFILE_REGEX,
    SCHEMA_REGEX,
]

IGNORED_PACK_NAMES = ["Legacy", "NonSupported", "ApiModules"]

PACK_IGNORE_TEST_FLAG = "auto-test"

PACKAGE_YML_FILE_REGEX = (
    r"(?:\./)?(?:Packs/[^/]+\/)?(?:Integrations|Scripts)\/([^\\/]+)/([^\\/]+)\.yml"
)

OLD_YML_FORMAT_FILE = [
    PACKS_INTEGRATION_NON_SPLIT_YML_REGEX,
    PACKS_SCRIPT_NON_SPLIT_YML_REGEX,
]

DIR_LIST_FOR_REGULAR_ENTETIES = [
    PLAYBOOKS_DIR,
    REPORTS_DIR,
    DASHBOARDS_DIR,
    WIDGETS_DIR,
    INCIDENT_TYPES_DIR,
    INCIDENT_FIELDS_DIR,
    LAYOUTS_DIR,
    PRE_PROCESS_RULES_DIR,
    CLASSIFIERS_DIR,
    INDICATOR_TYPES_DIR,
    INDICATOR_FIELDS_DIR,
    LISTS_DIR,
    JOBS_DIR,
    WIZARDS_DIR,
]
PACKS_DIRECTORIES = [
    SCRIPTS_DIR,
    INTEGRATIONS_DIR,
    DASHBOARDS_DIR,
    WIDGETS_DIR,
    INDICATOR_FIELDS_DIR,
    INDICATOR_TYPES_DIR,
    INCIDENT_FIELDS_DIR,
    INCIDENT_TYPES_DIR,
    REPORTS_DIR,
    PLAYBOOKS_DIR,
    JOBS_DIR,
    WIZARDS_DIR,
]
SPELLCHECK_FILE_TYPES = [
    PACKS_INTEGRATION_YML_REGEX,
    PACKS_SCRIPT_YML_REGEX,
    PLAYBOOK_YML_REGEX,
]

KNOWN_FILE_STATUSES = ["a", "m", "d", "r"] + [f"r{i:03}" for i in range(101)]

CODE_FILES_REGEX = [
    PACKS_INTEGRATION_PY_REGEX,
    PACKS_SCRIPT_PY_REGEX,
    PACKS_INTEGRATION_PS_REGEX,
    PACKS_SCRIPT_PS_REGEX,
]

SCRIPTS_REGEX_LIST = [
    PACKS_SCRIPT_YML_REGEX,
    PACKS_SCRIPT_PY_REGEX,
    PACKS_SCRIPT_PS_REGEX,
]

# All files that have related yml file
REQUIRED_YML_FILE_TYPES = [
    PACKS_INTEGRATION_PY_REGEX,
    PACKS_INTEGRATION_README_REGEX,
    PACKS_INTEGRATION_NON_SPLIT_README_REGEX,
    PACKS_SCRIPT_PY_REGEX,
    PACKS_SCRIPT_README_REGEX,
    PACKS_SCRIPT_NON_SPLIT_README_REGEX,
    PLAYBOOK_README_REGEX,
]

TYPE_PWSH = "powershell"
TYPE_PYTHON = "python"
TYPE_PYTHON3 = "python3"
TYPE_PYTHON2 = "python2"
TYPE_JS = "javascript"

TYPE_TO_EXTENSION = {
    TYPE_PYTHON: ".py",
    TYPE_PYTHON3: ".py",
    TYPE_PYTHON2: ".py",
    TYPE_JS: ".js",
    TYPE_PWSH: ".ps1",
}

TESTS_DIRECTORIES = [
    "testdata",
    "test_data",
    "data_test",
    "tests_data",
    "TestData",
]

DOCS_DIRECTORIES = [
    "doc_files",
    "doc_imgs",
]

TESTS_AND_DOC_DIRECTORIES = TESTS_DIRECTORIES + DOCS_DIRECTORIES

VALIDATION_USING_GIT_IGNORABLE_DATA = (
    "Pipfile",
    "Pipfile.lock",
    "command_examples.txt",
    "pack_metadata.json",
    "testdata",
    "test_data",
    "data_test",
    "tests_data",
    "doc_files",
    "doc_imgs",
    ".secrets-ignore",
    "version_config.json",  # TODO: remove and validate this file CIAC-12331
)

# A list of validation error codes that must always execute, regardless of ignore settings.
# This addresses unique cases where validation must run first, then filter the relevant results afterward.
ALWAYS_RUN_ON_ERROR_CODE = ["GR107"]


class GitStatuses(StrEnum):
    RENAMED = "R"
    MODIFIED = "M"
    ADDED = "A"
    DELETED = "D"


class ExecutionMode(StrEnum):
    ALL_FILES = "-a"
    USE_GIT = "-g"
    SPECIFIC_FILES = "-i"


FILE_TYPES_FOR_TESTING = [".py", ".js", ".yml", ".ps1"]

# python subtypes
PYTHON_SUBTYPES = {"python3", "python2"}


def urljoin(*args: str):
    """Gets arguments to join as url

    Args:
        *args: args to join

    Returns:
        Joined url

    Examples:
        >>> urljoin('https://www.example.com', 'suffix/', '/suffix2', 'suffix', 'file.json')
        'https://www.example.com/suffix/suffix2/suffix/file.json'
    """
    return reduce(
        lambda a, b: str(a).rstrip("/") + "/" + str(b).lstrip("/"), args
    ).rstrip("/")


OFFICIAL_CONTENT_ID_SET_PATH = f"https://storage.googleapis.com/{DEMISTO_SDK_MARKETPLACE_XSOAR_DIST}/content/id_set.json"

OFFICIAL_CONTENT_GRAPH_PATH = f"https://storage.googleapis.com/{DEMISTO_SDK_MARKETPLACE_XSOAR_DIST_DEV}/content_graph"

OFFICIAL_INDEX_JSON_PATH = f"https://storage.googleapis.com/{DEMISTO_SDK_MARKETPLACE_XSOAR_DIST}/content/packs/index.json"

# Run all test signal
RUN_ALL_TESTS_FORMAT = "Run all tests"

GOOGLE_CLOUD_STORAGE_PUBLIC_BASE_PATH = "https://storage.googleapis.com"


class PB_Status:
    NOT_SUPPORTED_VERSION = "Not supported version"
    COMPLETED = "completed"
    FAILED = "failed"
    IN_PROGRESS = "inprogress"
    FAILED_DOCKER_TEST = "failed_docker_test"
    CONFIGURATION_FAILED = "failed_configuration"


# change log regexes
UNRELEASE_HEADER = "## [Unreleased]\n"  # lgtm[py/regex/duplicate-in-character-class]
CONTENT_RELEASE_TAG_REGEX = r"^\d{2}\.\d{1,2}\.\d"
RELEASE_NOTES_REGEX = (
    re.escape(UNRELEASE_HEADER)
    + r"([\s\S]+?)## \[\d{2}\.\d{1,2}\.\d\] - \d{4}-\d{2}-\d{2}"
)
CONTENT_TYPE_SECTION_REGEX = re.compile(
    r"^#### ([\w ]+)$\n([\w\W]*?)(?=^#### )|^#### ([\w ]+)$\n([\w\W]*)", re.M
)
CONTENT_ITEM_SECTION_REGEX = re.compile(
    r"^##### (.+)$\n([\w\W]*?)(?=^##### )|^##### (.+)$\n([\w\W]*)|" r"^- (?:New: )?$",
    re.M,
)

# pack contributors template
CONTRIBUTORS_README_TEMPLATE = (
    "\n### Pack Contributors:\n\n---\n{contributors_names}\nContributions are welcome and "
    "appreciated. For more info, visit our [Contribution Guide](https://xsoar.pan.dev/docs"
    "/contributing/contributing)."
)

# Beta integration disclaimer
BETA_INTEGRATION_DISCLAIMER = (
    "Note: This is a beta Integration,"
    " which lets you implement and test pre-release software. "
    "Since the integration is beta, it might contain bugs. "
    "Updates to the integration during the beta phase might include "
    "non-backward compatible features. We appreciate your feedback on "
    "the quality and usability of the integration to help us identify issues, "
    "fix them, and continually improve."
)

# Integration categories according to the schema
INTEGRATION_CATEGORIES = [
    "Analytics & SIEM",
    "Utilities",
    "Messaging and Conferencing",
    "Endpoint",
    "Network Security",
    "Vulnerability Management",
    "Case Management",
    "Forensics & Malware Analysis",
    "IT Services",
    "Data Enrichment & Threat Intelligence",
    "Database",
    "Deception & Breach Simulation",
    "Email",
    "Identity and Access Management",
    "Cloud Services",
    "Authentication & Identity Management",
]
SCHEMA_TO_REGEX = {
    "integration": YML_INTEGRATION_REGEXES,
    "playbook": YML_ALL_PLAYBOOKS_REGEX,
    "script": YML_SCRIPT_REGEXES,
    "widget": JSON_ALL_WIDGETS_REGEXES,
    "dashboard": JSON_ALL_DASHBOARDS_REGEXES,
    "classifier_5_9_9": JSON_ALL_CLASSIFIER_REGEXES_5_9_9,
    "classifier": JSON_ALL_CLASSIFIER_REGEXES,
    "mapper": JSON_ALL_MAPPER_REGEXES,
    "layoutscontainer": JSON_ALL_LAYOUTS_CONTAINER_REGEXES,
    "layout": JSON_ALL_LAYOUT_REGEXES,
    "pre-process-rules": JSON_ALL_PRE_PROCESS_RULES_REGEXES,
    "incidentfield": JSON_ALL_INCIDENT_FIELD_REGEXES
    + JSON_ALL_INDICATOR_FIELDS_REGEXES,
    "incidenttype": JSON_ALL_INCIDENT_TYPES_REGEXES,
    "image": IMAGE_ALL_REGEXES,
    "reputation": JSON_ALL_INDICATOR_TYPES_REGEXES,
    "reputations": JSON_ALL_REPUTATIONS_INDICATOR_TYPES_REGEXES,
    "readme": [
        PACKS_INTEGRATION_README_REGEX,
        PACKS_INTEGRATION_NON_SPLIT_README_REGEX,
        PLAYBOOK_README_REGEX,
        PACKS_SCRIPT_README_REGEX,
        PACKS_SCRIPT_NON_SPLIT_README_REGEX,
    ],
    "report": [PACKS_REPORT_JSON_REGEX],
    "modelingrule": [MODELING_RULE_YML_REGEX],
    "modelingruleschema": [MODELING_RULE_SCHEMA_REGEX],
    "release-notes": [PACKS_RELEASE_NOTES_REGEX],
    "genericfield": JSON_ALL_GENERIC_FIELDS_REGEXES,
    "generictype": JSON_ALL_GENERIC_TYPES_REGEXES,
    "genericmodule": JSON_ALL_GENERIC_MODULES_REGEXES,
    "genericdefinition": JSON_ALL_GENERIC_DEFINITIONS_REGEXES,
    JOB: JSON_ALL_JOB_REGEXES,
    WIZARD: JSON_ALL_WIZARD_REGEXES,
    "correlationrule": [CORRELATION_RULES_YML_REGEX],
    "parsingrule": [PARSING_RULE_YML_REGEX],
    "xsiamdashboard": [XSIAM_DASHBOARD_JSON_REGEX],
    "xsiamreport": [XSIAM_REPORT_JSON_REGEX],
    "trigger": [TRIGGER_JSON_REGEX],
    "xdrctemplate": [XDRC_TEMPLATE_JSON_REGEX],
    LAYOUT_RULE: JSON_ALL_LAYOUT_RULES_REGEXES,
}
EXTERNAL_PR_REGEX = r"^pull/(\d+)$"

FILE_TYPES_PATHS_TO_VALIDATE = {"reports": JSON_ALL_REPORTS_REGEXES}
MDX_SERVER_DOCKER_IMAGE = "devdemisto/mdx_node_server:1.0.0.48264"
DEF_DOCKER = "demisto/python:1.3-alpine"
DEF_DOCKER_PWSH = "demisto/powershell:6.2.3.5563"

DIR_TO_PREFIX = {
    "Integrations": INTEGRATION_PREFIX,
    "Scripts": SCRIPT_PREFIX,
    "ModelingRules": MODELING_RULE_PREFIX,
    "ParsingRules": PARSING_RULE_PREFIX,
    "XDRCTemplates": XDRC_TEMPLATE_PREFIX,
}

ENTITY_NAME_SEPARATORS = [" ", "_", "-"]
FILE_EXIST_REASON = "File already exist"
FILE_NOT_IN_CC_REASON = "File does not exist in Demisto instance"

ACCEPTED_FILE_EXTENSIONS = [
    ".yml",
    ".json",
    ".md",
    ".py",
    ".js",
    ".ps1",
    ".png",
    "",
    ".lock",
]
ENDPOINT_COMMAND_NAME = "endpoint"
GET_MAPPING_FIELDS_COMMAND_NAME = "get-mapping-fields"
GET_MAPPING_FIELDS_COMMAND = {
    "description": "Retrieves a User Profile schema which holds all of the user fields in the application. Used for outgoing mapping through the Get Schema option.",
    "name": GET_MAPPING_FIELDS_COMMAND_NAME,
}

FEED_RELIABILITY = "feedReliability"
RELIABILITY = "reliability"

RELIABILITY_PARAMETER_NAMES = [
    "integration_reliability",  # First item in the list will be used in errors
    "integrationReliability",
    FEED_RELIABILITY,
    RELIABILITY,
]

RELIABILITY_PARAM = {
    "name": RELIABILITY,
    "display": "Source Reliability",
    "type": 15,
    "required": True,
    "options": [
        "A - Completely reliable",
        "B - Usually reliable",
        "C - Fairly reliable",
        "D - Not usually reliable",
        "E - Unreliable",
        "F - Reliability cannot be judged",
    ],
    "additionalinfo": "Reliability of the source providing the intelligence data",
}

COMMON_PARAMS_DISPLAY_NAME = {
    "insecure": "Trust any certificate (not secure)",
    "unsecure": "Trust any certificate (not secure)",
    "proxy": "Use system proxy settings",
}

REQUIRED_ALLOWED_PARAMS = ("insecure", "unsecure", "proxy", "isFetch")

REPUTATION_COMMAND_NAMES = {"file", "email", "domain", "url", "ip", "cve"}

BANG_COMMAND_NAMES = {"file", "email", "domain", "url", "ip", "cve", "endpoint"}

BANG_COMMAND_ARGS_MAPPING_DICT: Dict[str, dict] = {
    "file": {"default": ["file"], "required": True},
    "email": {"default": ["email"], "required": True},
    "domain": {"default": ["domain"], "required": True},
    "url": {"default": ["url"], "required": True},
    "ip": {"default": ["ip"], "required": True},
    "cve": {"default": ["cve"], "required": True},
    "endpoint": {"default": ["ip"], "required": False},
}

MANDATORY_REPUTATION_CONTEXT_NAMES = {
    "Certificate",
    "CVE",
    "Domain",
    "Email",
    "Endpoint",
    "File",
    "InfoFile",
    "IP",
    "URL",
}

ENDPOINT_FLEXIBLE_REQUIRED_ARGS = ["ip", "id", "hostname"]

GENERIC_COMMANDS_NAMES = BANG_COMMAND_NAMES.union(
    {"send-mail", "send-notification", "cve-latest", "cve-search"}
)

DBOT_SCORES_DICT: Dict[str, str] = {
    "DBotScore.Indicator": "The indicator that was tested.",
    "DBotScore.Type": "The indicator type.",
    "DBotScore.Vendor": "The vendor used to calculate the score.",
    "DBotScore.Score": "The actual score.",
}

IOC_OUTPUTS_DICT = {
    "domain": {"Domain.Name"},
    "file": {"File.MD5", "File.SHA1", "File.SHA256"},
    "ip": {"IP.Address"},
    "url": {"URL.Data"},
    "endpoint": {"Endpoint.Hostname", "Endpoint.IPAddress", "Endpoint.ID"},
}

# support types:
XSOAR_SUPPORT = "xsoar"
PARTNER_SUPPORT = "partner"
COMMUNITY_SUPPORT = "community"
DEVELOPER_SUPPORT = "developer"
PACK_SUPPORT_OPTIONS = [
    XSOAR_SUPPORT,
    PARTNER_SUPPORT,
    DEVELOPER_SUPPORT,
    COMMUNITY_SUPPORT,
]
CONTRIBUTORS_LIST = [PARTNER_SUPPORT, DEVELOPER_SUPPORT, COMMUNITY_SUPPORT]
SUPPORTED_CONTRIBUTORS_LIST = [DEVELOPER_SUPPORT, PARTNER_SUPPORT]
XSOAR_AUTHOR = "Cortex XSOAR"
PACK_INITIAL_VERSION = "1.0.0"
XSOAR_CONTEXT_STANDARD_URL = "https://xsoar.pan.dev/docs/integrations/context-standards"
XSOAR_CONTEXT_AND_OUTPUTS_URL = (
    "https://xsoar.pan.dev/docs/integrations/context-and-outputs"
)
XSOAR_SUPPORT_URL = "https://www.paloaltonetworks.com/cortex"
MARKETPLACE_LIVE_DISCUSSIONS = "https://live.paloaltonetworks.com/t5/cortex-xsoar-discussions/bd-p/Cortex_XSOAR_Discussions"
EXCLUDED_DISPLAY_NAME_WORDS = [PARTNER_SUPPORT, COMMUNITY_SUPPORT]
INCORRECT_PACK_NAME_WORDS = [
    "Pack",
    "Playbook",
    "Integration",
    "Script",
    PARTNER_SUPPORT,
    COMMUNITY_SUPPORT,
]
MARKETPLACES = ["xsoar", "marketplacev2"]
MODULES = ["compliance"]
SUPPORT_LEVEL_HEADER = "supportlevelheader"
CLASSIFICATION_TYPE = "classification"

# From Version constants
FILETYPE_TO_DEFAULT_FROMVERSION = {
    FileType.INTEGRATION: "4.5.0",
    FileType.SCRIPT: "4.5.0",
    FileType.PLAYBOOK: "4.5.0",
    FileType.WIZARD: "6.8.0",
    FileType.JOB: "6.8.0",
    FileType.PRE_PROCESS_RULES: "6.8.0",
    FileType.LISTS: "6.5.0",
    FileType.GENERIC_TYPE: "6.5.0",
    FileType.GENERIC_FIELD: "6.5.0",
    FileType.GENERIC_MODULE: "6.5.0",
    FileType.GENERIC_DEFINITION: "6.5.0",
    FileType.CORRELATION_RULE: "6.10.0",
    FileType.PARSING_RULE: "6.10.0",
    FileType.MODELING_RULE: "6.10.0",
    FileType.LAYOUT_RULE: "6.10.0",
    FileType.XSIAM_DASHBOARD: "6.10.0",
    FileType.ASSETS_MODELING_RULE: "6.2.1",
    FileType.CASE_LAYOUT_RULE: "8.7.0",
    FileType.CASE_FIELD: "8.7.0",
    FileType.CASE_LAYOUT: "8.7.0",
}

DEFAULT_PYTHON_VERSION = "3.10"
DEFAULT_PYTHON2_VERSION = "2.7"

# This constant below should always be two versions before the latest server version
GENERAL_DEFAULT_FROMVERSION = "6.10.0"
VERSION_5_5_0 = "5.5.0"
DEFAULT_CONTENT_ITEM_FROM_VERSION = "0.0.0"
DEFAULT_CONTENT_ITEM_TO_VERSION = "99.99.99"
MARKETPLACE_MIN_VERSION = "6.0.0"
MINIMUM_XSOAR_SAAS_VERSION = "8.0.0"

OLDEST_SUPPORTED_VERSION = "5.0.0"
OLDEST_INCIDENT_FIELD_SUPPORTED_VERSION = GENERAL_DEFAULT_FROMVERSION
LAYOUTS_CONTAINERS_OLDEST_SUPPORTED_VERSION = "6.0.0"
GENERIC_OBJECTS_OLDEST_SUPPORTED_VERSION = "6.5.0"

FEATURE_BRANCHES = ["v4.5.0"]
VERSION_REGEX = r"(\d{1,2}\.){2}\d{1,2}$"
DOC_FILE_IMAGE_REGEX = r"[\.\./]*doc_files/[a-zA-Z0-9_-]+\.png"
DOC_FILE_FULL_IMAGE_REGEX = r"!\[.*?\]\(((?!http).*?doc_files.*?)\)"
BASE_PACK = "Base"
NON_SUPPORTED_PACK = "NonSupported"
DEPRECATED_CONTENT_PACK = "DeprecatedContent"
IGNORED_DEPENDENCY_CALCULATION = {
    BASE_PACK,
    NON_SUPPORTED_PACK,
    DEPRECATED_CONTENT_PACK,
}
COMMON_TYPES_PACK = "CommonTypes"

FEED_REQUIRED_PARAMS = [
    {
        "name": "feed",
        "must_equal": {
            "defaultvalue": "true",
            "display": "Fetch indicators",
            "type": 8,
        },
        "must_contain": {},
        "must_be_one_of": {},
    },
    {
        "name": "feedReputation",
        "must_equal": {
            "display": "Indicator Reputation",
            "type": 18,
            "options": ["None", "Good", "Suspicious", "Bad"],
        },
        "must_contain": {
            "additionalinfo": "Indicators from this integration instance will be marked with this reputation"
        },
        "must_be_one_of": {},
    },
    {
        "name": FEED_RELIABILITY,
        "must_equal": {
            "display": "Source Reliability",
            "type": 15,
            "required": True,
            "options": [
                "A - Completely reliable",
                "B - Usually reliable",
                "C - Fairly reliable",
                "D - Not usually reliable",
                "E - Unreliable",
                "F - Reliability cannot be judged",
            ],
        },
        "must_contain": {
            "additionalinfo": "Reliability of the source providing the intelligence data"
        },
        "must_be_one_of": {},
    },
    {
        "name": "feedExpirationInterval",
        "must_equal": {"display": "", "type": 1},
        "must_contain": {},
        "must_be_one_of": {},
    },
    {
        "name": "feedFetchInterval",
        "must_equal": {"display": "Feed Fetch Interval", "type": 19},
        "must_contain": {},
        "must_be_one_of": {},
    },
    {
        "name": "feedBypassExclusionList",
        "must_equal": {
            "display": "Bypass exclusion list",
            "type": 8,
        },
        "must_contain": {
            "additionalinfo": "When selected, the exclusion list is ignored for indicators from this feed."
            " This means that if an indicator from this feed is on the exclusion list,"
            " the indicator might still be added to the system."
        },
        "must_be_one_of": {},
    },
    {
        "name": "feedTags",
        "must_equal": {"display": "Tags", "type": 0},
        "must_contain": {"additionalinfo": "Supports CSV values."},
        "must_be_one_of": {},
    },
    {
        "name": "tlp_color",
        "must_equal": {
            "display": "Traffic Light Protocol Color",
            "type": 15,
        },
        "must_contain": {
            "additionalinfo": "The Traffic Light Protocol (TLP) designation to apply to indicators fetched from the "
            "feed"
        },
        "must_be_one_of": {
            "options": [
                ["RED", "AMBER", "GREEN", "WHITE"],
                ["RED", "AMBER+STRICT", "AMBER", "GREEN", "CLEAR"],
            ]
        },
    },
]

ALLOWED_HIDDEN_PARAMS = {"longRunning", "feedIncremental", "feedReputation"}

INCIDENT_FETCH_REQUIRED_PARAMS = [
    {"display": "Incident type", "name": "incidentType", "type": 13},
    {"display": "Fetch incidents", "name": "isFetch", "type": 8},
]

ALERT_FETCH_REQUIRED_PARAMS = [
    {"display": "Alert type", "name": "incidentType", "type": 13},
    {"display": "Fetch alerts", "name": "isFetch", "type": 8},
]

MAX_FETCH_PARAM = {
    "name": "max_fetch",
    "type": 0,
    "defaultvalue": "50",
}

# for reference, the defaultvalue and display are not enforced.
FIRST_FETCH_PARAM = {
    "defaultvalue": "7 days",
    "display": "First fetch timestamp (<number> <time unit>, e.g., 12 hours, 7 days)",
    "name": "first_fetch",
    "required": False,
    "type": 0,
}

DOCS_COMMAND_SECTION_REGEX = r"(?:###\s{}).+?(?:(?=(?:\n###\s))|(?=(?:\n##\s))|\Z)"
# Ignore list for all 'run_all_validations_on_file' method
ALL_FILES_VALIDATION_IGNORE_WHITELIST = [
    "pack_metadata.json",  # this file is validated under 'validate_pack_unique_files' method
    "testdata",
    "test_data",
    "data_test",
    "testcommandsfunctions",
    "testhelperfunctions",
    "stixdecodetest",
    "testcommands",
    "setgridfield_test",
    "ipnetwork_test",
    "test-data",
    "testplaybook",
]
VALIDATED_PACK_ITEM_TYPES = [
    "Playbooks",
    "Integration",
    "Script",
    "IncidentFields",
    "IncidentTypes",
    "Classifiers",
    "Layouts",
    "PreProcessRules",
    "Lists",
    "Jobs",
    "Wizards",
]

VALID_SENTENCE_SUFFIX = [".", "!", "?", ".)", ".'", '."', "\n}", "\n]"]

FIRST_FETCH = "first_fetch"

MAX_FETCH = "max_fetch"
DEFAULT_MAX_FETCH = 10

SKIP_RELEASE_NOTES_FOR_TYPES = (
    FileType.RELEASE_NOTES,
    FileType.README,
    FileType.TEST_PLAYBOOK,
    FileType.TEST_SCRIPT,
    FileType.DOC_IMAGE,
    FileType.AUTHOR_IMAGE,
    None,
    FileType.RELEASE_NOTES_CONFIG,
    FileType.CONTRIBUTORS,
    FileType.PACK_IGNORE,
)

LAYOUT_AND_MAPPER_BUILT_IN_FIELDS = [
    "indicatortype",
    "source",
    "comment",
    "aggregatedreliability",
    "detectedips",
    "detectedhosts",
    "modified",
    "expiration",
    "timestamp",
    "shortdesc",
    "short_description",
    "description",
    "Tags",
    "blocked",
]

INTEGRATION_FIELDS_NOT_ALLOWED_TO_CHANGE = [
    "feed",
    "isfetch",
    "longRunning",
    "longRunningPort",
    "ismappable",
    "isremotesyncin",
    "isremotesyncout",
]

CONTEXT_OUTPUT_README_TABLE_HEADER = "| **Path** | **Type** | **Description** |"

ARGUMENT_FIELDS_TO_CHECK = ["defaultValue", "required", "isArray"]

PARAM_FIELDS_TO_CHECK = ["defaultvalue", "type", "required"]

INTEGRATION_ARGUMENT_TYPES = {
    "0": "ShortText",
    "4": "Encrypted",
    "8": "Boolean",
    "9": "Authentication",
    "12": "LongText",
    "15": "SingleSelect",
    "16": "MultiSelect",
}

BUILD_IN_COMMANDS = [
    "getIncidents",
    "DeleteContext",
    "isWhitelisted",
    "excludeIndicators",
    "deleteIndicators",
    "extractIndicators",
]


class Auto(StrEnum):
    PREDEFINED = "PREDEFINED"


class ContentItems(Enum):
    # the format is defined in issue #19786, may change in the future
    SCRIPTS = "automation"
    PLAYBOOKS = "playbook"
    INTEGRATIONS = "integration"
    INCIDENT_FIELDS = "incidentfield"
    INCIDENT_TYPES = "incidenttype"
    DASHBOARDS = "dashboard"
    INDICATOR_FIELDS = "indicatorfield"
    REPORTS = "report"
    INDICATOR_TYPES = "reputation"
    LAYOUTS = "layoutscontainer"
    CLASSIFIERS = "classifier"
    WIDGETS = "widget"
    GENERIC_MODULES = "genericmodule"
    GENERIC_DEFINITIONS = "genericdefinition"
    GENERIC_FIELDS = "genericfield"
    GENERIC_TYPES = "generictype"
    PRE_PROCESS_RULES = "pre-process-rule"
    LISTS = "list"
    JOB = "job"
    PARSING_RULES = "parsingrule"
    MODELING_RULES = "modelingrule"
    CORRELATION_RULES = "correlationrule"
    XSIAM_DASHBOARDS = "xsiamdashboard"
    XSIAM_REPORTS = "xsiamreport"
    TRIGGERS = ("trigger",)
    WIZARDS = ("wizard",)
    XDRC_TEMPLATE = "xdrctemplate"
    LAYOUT_RULES = "layoutrule"
    ASSETS_MODELING_RULES = "assetsmodelingrule"
    CASE_LAYOUT_RULES = "caselayoutrule"
    CASE_FIELDS = "casefield"
    CASE_LAYOUTS = "caselayout"


CONTENT_ITEMS_DISPLAY_FOLDERS = {
    SCRIPTS_DIR,
    DASHBOARDS_DIR,
    INCIDENT_FIELDS_DIR,
    INCIDENT_TYPES_DIR,
    INTEGRATIONS_DIR,
    PLAYBOOKS_DIR,
    INDICATOR_FIELDS_DIR,
    REPORTS_DIR,
    INDICATOR_TYPES_DIR,
    LAYOUTS_DIR,
    PRE_PROCESS_RULES_DIR,
    CLASSIFIERS_DIR,
    WIDGETS_DIR,
    JOBS_DIR,
    LISTS_DIR,
    PARSING_RULES_DIR,
    MODELING_RULES_DIR,
    CORRELATION_RULES_DIR,
    XSIAM_DASHBOARDS_DIR,
    XSIAM_REPORTS_DIR,
    TRIGGER_DIR,
    WIZARDS_DIR,
    XDRC_TEMPLATE_DIR,
    LAYOUT_RULES_DIR,
    ASSETS_MODELING_RULES_DIR,
    CASE_LAYOUT_RULES_DIR,
    CASE_FIELDS_DIR,
    CASE_LAYOUTS_DIR,
}


class PathLevel(Enum):
    PACK = ("Pack",)
    CONTENT_ENTITY_DIR = ("ContentDir",)
    PACKAGE = ("Package",)
    FILE = "File"
    CONTENT_GENERIC_ENTITY_DIR = "ContentGenericDir"


class DemistoException(Exception):
    pass


UUID_REGEX = r"([\w]{8}-[\w]{4}-[\w]{4}-[\w]{4}-[\w]{8,12})"


class IronBankDockers:
    API_LINK = "https://repo1.dso.mil/api/v4/projects/dsop%2Fopensource%2Fpalo-alto-networks%2Fdemisto%2F"


class MarketplaceVersions(StrEnum):
    XSOAR = "xsoar"
    MarketplaceV2 = "marketplacev2"
    XPANSE = "xpanse"
    XSOAR_SAAS = "xsoar_saas"
    XSOAR_ON_PREM = "xsoar_on_prem"
    PLATFORM = "platform"


MarketplaceVersionToMarketplaceName: Dict[str, str] = {
    MarketplaceVersions.XSOAR.value: DEMISTO_SDK_MARKETPLACE_XSOAR_DIST,
    MarketplaceVersions.MarketplaceV2.value: DEMISTO_SDK_MARKETPLACE_XSIAM_DIST,
    MarketplaceVersions.XPANSE.value: DEMISTO_SDK_MARKETPLACE_XPANSE_DIST,
    MarketplaceVersions.XSOAR_SAAS.value: DEMISTO_SDK_MARKETPLACE_XSOAR_SAAS_DIST,
    MarketplaceVersions.PLATFORM.value: DEMISTO_SDK_MARKETPLACE_PLATFORM_INTERNAL_PROD,
}

MARKETPLACE_TO_CORE_PACKS_FILE: Dict[MarketplaceVersions, str] = {
    MarketplaceVersions.XSOAR: "Config/core_packs_list.json",
    MarketplaceVersions.XSOAR_SAAS: "Config/core_packs_list.json",
    MarketplaceVersions.XSOAR_ON_PREM: "Config/core_packs_list.json",
    MarketplaceVersions.MarketplaceV2: "Config/core_packs_mpv2_list.json",
    MarketplaceVersions.XPANSE: "Config/core_packs_xpanse_list.json",
    MarketplaceVersions.PLATFORM: "Config/core_packs_platform_list.json",
}


class PlatformSupportedModules(StrEnum):
    C1 = "C1"
    C3 = "C3"
    XO = "X0"
    X1 = "X1"
    X3 = "X3"
    X5 = "X5"
    ENT_PLUS = "ENT_PLUS"
    AGENTIX = "agentix"
    ASM = "asm"
    EXPOSURE_MANAGEMENT = "exposure_management"
    AGENTIX_XSIAM = "agentix_xsiam"


INDICATOR_FIELD_TYPE_TO_MIN_VERSION = {
    "html": Version("6.1.0"),
    "grid": Version("5.5.0"),
}


class IdSetKeys(Enum):
    SCRIPTS = "scripts"
    INTEGRATIONS = "integrations"
    PLAYBOOKS = "playbooks"
    TEST_PLAYBOOKS = "TestPlaybooks"
    CLASSIFIERS = "Classifiers"
    INCIDENT_FIELDS = "IncidentFields"
    INCIDENT_TYPES = "IncidentTypes"
    INDICATOR_FIELDS = "IndicatorFields"
    INDICATOR_TYPES = "IndicatorTypes"
    LISTS = "Lists"
    JOBS = "Jobs"
    MAPPERS = "Mappers"
    PACKS = "Packs"
    GENERIC_TYPES = "GenericTypes"
    GENERIC_FIELDS = "GenericFields"
    GENERIC_MODULES = "GenericModules"
    GENERIC_DEFINITIONS = "GenericDefinitions"
    LAYOUTS = "Layouts"
    REPORTS = "Reports"
    WIDGETS = "Widgets"
    DASHBOARDS = "Dashboards"
    PARSING_RULES = "ParsingRules"
    MODELING_RULES = "ModelingRules"
    CORRELATION_RULES = "CorrelationRules"
    XSIAM_DASHBOARDS = "XSIAMDashboards"
    XSIAM_REPORTS = "XSIAMReports"
    TRIGGERS = "Triggers"


FileTypeToIDSetKeys = {
    FileType.INTEGRATION: IdSetKeys.INTEGRATIONS.value,
    FileType.BETA_INTEGRATION: IdSetKeys.INTEGRATIONS.value,
    FileType.PLAYBOOK: IdSetKeys.PLAYBOOKS.value,
    FileType.SCRIPT: IdSetKeys.SCRIPTS.value,
    FileType.TEST_SCRIPT: IdSetKeys.SCRIPTS.value,
    FileType.TEST_PLAYBOOK: IdSetKeys.PLAYBOOKS.value,
    FileType.DASHBOARD: IdSetKeys.DASHBOARDS.value,
    FileType.WIDGET: IdSetKeys.WIDGETS.value,
    FileType.REPORT: IdSetKeys.REPORTS.value,
    FileType.OLD_CLASSIFIER: IdSetKeys.CLASSIFIERS.value,
    FileType.CLASSIFIER: IdSetKeys.CLASSIFIERS.value,
    FileType.MAPPER: IdSetKeys.MAPPERS.value,
    FileType.LAYOUT: IdSetKeys.LAYOUTS.value,
    FileType.LAYOUTS_CONTAINER: IdSetKeys.LAYOUTS.value,
    FileType.LISTS: IdSetKeys.LISTS.value,
    FileType.REPUTATION: IdSetKeys.INDICATOR_TYPES.value,
    FileType.INDICATOR_FIELD: IdSetKeys.INDICATOR_FIELDS.value,
    FileType.INCIDENT_FIELD: IdSetKeys.INCIDENT_FIELDS.value,
    FileType.INCIDENT_TYPE: IdSetKeys.INCIDENT_TYPES.value,
    FileType.GENERIC_TYPE: IdSetKeys.GENERIC_TYPES.value,
    FileType.GENERIC_FIELD: IdSetKeys.GENERIC_FIELDS.value,
    FileType.GENERIC_MODULE: IdSetKeys.GENERIC_MODULES.value,
    FileType.GENERIC_DEFINITION: IdSetKeys.GENERIC_DEFINITIONS.value,
    FileType.JOB: IdSetKeys.JOBS.value,
    FileType.PARSING_RULE: IdSetKeys.PARSING_RULES.value,
    FileType.MODELING_RULE: IdSetKeys.MODELING_RULES.value,
    FileType.CORRELATION_RULE: IdSetKeys.CORRELATION_RULES.value,
    FileType.XSIAM_DASHBOARD: IdSetKeys.XSIAM_DASHBOARDS.value,
    FileType.XSIAM_REPORT: IdSetKeys.XSIAM_REPORTS.value,
    FileType.TRIGGER: IdSetKeys.TRIGGERS.value,
}

FileType_ALLOWED_TO_DELETE = {
    FileType.WHITE_LIST,
    FileType.DOC_IMAGE,
    FileType.TEST_PLAYBOOK,
}


class ParameterType(Enum):
    STRING = 0  # holds a string argument.
    NUMBER = 1  # holds an int argument.
    ENCRYPTED = 4  # holds a password argument.
    BOOLEAN = 8  # holds a boolean argument (checkbox).
    AUTH = 9  # holds an auth struct (credentials type).
    DOWNLOAD_LINK = 11
    TEXT_AREA = 12
    INCIDENT_TYPE = 13
    TEXT_AREA_ENCRYPTED = 14
    SINGLE_SELECT = 15
    MULTI_SELECT = 16
    EXPIRATION_FIELD = 17
    FEED_REPUDATION = 18
    INTERVAL = 19
    BOLD_TITLE = 20
    DAY_DROPDOWN = 21
    RESULT_LINK = 22  # A placeholder; the front-end will fill this with dynamically generated text.
    ENGINE_PLACEHOLDER = 23  # A signal to the front-end; the 'engine' dropdown's position in the UI will match the order of this parameter.


class IncidentFieldType(
    StrEnum
):  # For more info please see https://xsoar.pan.dev/docs/incidents/incident-fields#field-types
    SHORT_TEXT = "shortText"
    LONG_TEXT = "longText"
    NUMBER = "number"
    BOOLEAN = "boolean"
    SINGLE_SELECT = "singleSelect"
    MULTI_SELECT = "multiSelect"
    DATE = "date"
    MARKDOWN = "markdown"
    HTML = "html"
    URL = "url"
    USER = "user"
    ROLE = "role"
    ATTACHMENTS = "attachments"
    SLA = "sla"
    GRID = "grid"


NO_TESTS_DEPRECATED = "No tests (deprecated)"
NATIVE_IMAGE_FILE_NAME = "docker_native_image_config.json"
TESTS_REQUIRE_NETWORK_PACK_IGNORE = "tests_require_network"
NATIVE_IMAGE_DOCKER_NAME = "demisto/py3-native"


# SKIP PREPARE CONSTANTS
SKIP_PREPARE_SCRIPT_NAME = "script-name-incident-to-alert"


TABLE_INCIDENT_TO_ALERT = {
    "incident": "alert",
    "incidents": "alerts",
    "Incident": "Alert",
    "Incidents": "Alerts",
    "INCIDENT": "ALERT",
    "INCIDENTS": "ALERTS",
}


FORMATTING_SCRIPT = "indicator-format"

DOCKERFILES_INFO_REPO = "demisto/dockerfiles-info"
DOCKERFILES_INFO_REPO_PRIMARY_BRANCH = "master"
DOCKERFILES_REPO = "demisto/dockerfiles"
CONTENT_REPO = "demisto/content"

TEST_COVERAGE_DEFAULT_URL = f"https://storage.googleapis.com/{DEMISTO_SDK_MARKETPLACE_XSOAR_DIST_DEV}/code-coverage-reports/coverage-min.json"

URL_IMAGE_LINK_REGEX = r"(\!\[.*?\])\((?P<url>https://[a-zA-Z_/\.0-9\- :%]*?)\)((].*)?)"

HTML_IMAGE_LINK_REGEX = r'(<img.*?src\s*=\s*"(https://.*?)")'

MARKETPLACE_LIST_PATTERN = r"[A-Z_]+(?:,[A-Z_]+)*"
TAG_CONTENT_PATTERN = r"(?:.|\s)*?"

MARKETPLACE_TAG_MAPPING = {
    MarketplaceVersions.PLATFORM.value: ["PLATFORM", "XSIAM"],
    MarketplaceVersions.MarketplaceV2.value: ["XSIAM", "XSIAM_ONLY"],
    MarketplaceVersions.XPANSE.value: ["XPANSE"],
    MarketplaceVersions.XSOAR.value: [
        "XSOAR",
        "XSOAR_ON_PREM",
    ],  # If uploading to XSOAR, keep XSOAR and XSOAR_ON_PREM tags
    MarketplaceVersions.XSOAR_SAAS.value: [
        "XSOAR_SAAS",
        "XSOAR",
    ],  # If uploading to XSOAR_SAAS, keep XSOAR_SAAS and XSOAR tags
    MarketplaceVersions.XSOAR_ON_PREM.value: [
        "XSOAR_ON_PREM"
    ],  # If uploading to XSOAR_ON_PREM, keep XSOAR_ON_PREM tags
}

VALID_MARKETPLACE_TAGS = {
    "XSOAR",
    "XSOAR_SAAS",
    "XSOAR_ON_PREM",
    "XPANSE",
    "XSIAM",
    "XSIAM_ONLY",
    "PLATFORM",
}

MARKDOWN_IMAGES_ARTIFACT_FILE_NAME = "markdown_images.json"
MARKDOWN_RELATIVE_PATH_IMAGES_ARTIFACT_FILE_NAME = "markdown_relatve_path_images.json"
SERVER_API_TO_STORAGE = "api/marketplace/file?name=content/packs"

STRING_TO_BOOL_MAP = {
    # should only have lowercase keys, see string_to_bool
    "t": True,
    "y": True,
    "1": True,
    "yes": True,
    "true": True,
    "n": False,
    "f": False,
    "0": False,
    "no": False,
    "false": False,
}

SCHEMA_FILE_VALID_ATTRIBUTES_TYPE = {"string", "int", "float", "datetime", "boolean"}


#  date formats:
ISO_TIMESTAMP_FORMAT = "%Y-%m-%dT%H:%M:%SZ"


class ImagesFolderNames(StrEnum):
    README_IMAGES = "readme_images"
    INTEGRATION_DESCRIPTION_IMAGES = "integration_description_images"


class InvestigationPlaybookState(StrEnum):
    NEW = "new"  # indicates that playbook not executed yet
    IN_PROGRESS = "inprogress"  # indicates that playbook in progress
    PAUSED = "paused"  # indicates that playbook paused
    COMPLETED = "completed"  # indicates that playbook completed
    FAILED = "failed"  # indicates that playbook failed
    WAITING = "waiting"  # indicates that playbook currently stopped and waiting for user input on manual task


class XsoarIncidentState(IntEnum):
    NEW = 0
    IN_PROGRESS = 1
    CLOSED = 2
    ACKNOWLEDGED = 3


class XsiamAlertState(IntEnum):
    NEW = 0
    UNDER_INVESTIGATION = 1
    RESOLVED = 2


IncidentState = XsoarIncidentState  # To avoid breaking imports in other repos where SDK is installed


class PlaybookTaskType(StrEnum):
    REGULAR = "regular"
    PLAYBOOK = "playbook"
    CONDITION = "condition"
    START = "start"
    TITLE = "title"
    SECTION = "section"
    STANDARD = "standard"
    COLLECTION = "collection"


# Used to format the writing of the yml/json file
DEFAULT_JSON_INDENT = 4
DEFAULT_YAML_INDENT = 0


PACK_DEFAULT_MARKETPLACES: List = [
    MarketplaceVersions.XSOAR.value,
    MarketplaceVersions.MarketplaceV2.value,
]

INVALID_IMAGE_PATH_REGEX = (
    r"(\!\[.*?\]|src\=)(\(|\")(https://github.com/demisto/content/blob/.*?)(\)|\")"
)

# Test types:
TEST_PLAYBOOKS = "TestPlaybooks"
TEST_MODELING_RULES = "TestModelingRules"
TEST_USE_CASES = "TestUseCases"

PB_RELEASE_NOTES_FORMAT = {
    "This playbook addresses the following alerts:": 5,
    "Playbook Stages:": 5,
    "Requirements:": 5,
    "Triage:": 6,
    "Early Containment:": 6,
    "Investigation:": 6,
    "Containment:": 6,
}

INCIDENT_COMMANDS: list[str] = [
    "xsoar-search-incidents",
    "xsoar-get-incident",
]

MIRRORING_COMMANDS: list[str] = [
    "get-mapping-fields",
    "get-remote-data",
    "get-modified-remote-data",
    "update-remote-system",
]
