import os
import ast


def normalize_name(txt_name):
    """
    Normalize a file name by removing the `.txt` suffix if present.

    Parameters
    ----------
    txt_name : str
        The input file name.

    Returns
    -------
    str
        The normalized file name without the `.txt` extension.
    """
    if txt_name.endswith('.txt'):
        return txt_name[:-4]
    return txt_name


def read_file_lines(filename):
    """
    Read all lines from a file, evaluating literals when possible.

    Each line is read and passed through `ast.literal_eval`. If parsing fails
    due to malformed content, the raw line string is returned instead.

    Parameters
    ----------
    filename : str
        Path to the text file.

    Returns
    -------
    list
        A list of parsed lines (possibly containing lists, numbers, or strings).
    """
    with open(filename, 'r', encoding='utf-8') as f:
        result = []
        for line in f.read().splitlines():
            try:
                result.append(ast.literal_eval(line))
            except (ValueError, SyntaxError):
                result.append(line)
        return result


def is_2d_list(lst):
    """
    Check if the given list is a 2D list (a list of lists).

    Parameters
    ----------
    lst : list
        The list to check.

    Returns
    -------
    bool
        True if `lst` is a list of lists, False otherwise.
    """
    if not lst:
        return False
    return all(isinstance(item, list) for item in lst)


def normalize_to_list(item):
    """
    Normalize a given item to a list representation.

    Rules
    -----
    - str, int, float, bool, None → [item]
    - tuple, set, frozenset, range → list(item)
    - dict → list of values
    - list → unchanged
    - other types → raise TypeError

    Parameters
    ----------
    item : any
        The input object to normalize.

    Returns
    -------
    list
        The normalized list.

    Raises
    ------
    TypeError
        If the input type is not supported.
    """
    if isinstance(item, list):
        return item
    elif isinstance(item, (tuple, set, frozenset, range)):
        return list(item)
    elif isinstance(item, dict):
        return list(item.values())
    elif isinstance(item, (str, int, float, bool, type(None))):
        return [item]
    else:
        raise TypeError(
            f"Unsupported type: {type(item).__name__}. Expected a list-like structure."
        )


def validate_lines(lines, reference=None):
    """
    Validate that all rows in `lines` share the same structure and data types.

    Parameters
    ----------
    lines : list
        A list representing rows of data (1D or 2D).
    reference : list, optional
        A reference structure defining expected shape and types. If None,
        the first element of `lines` is used as the reference.

    Raises
    ------
    ValueError
        If row lengths or structures differ.
    TypeError
        If data types are inconsistent between rows or columns.
    """
    if not lines:
        raise ValueError("Input 'lines' cannot be empty.")
    if reference is not None and not reference:
        raise ValueError("Reference structure cannot be empty.")

    reference_structure = lines if reference is None else reference

    if is_2d_list(reference_structure):
        reference_row = reference_structure[0]
        expected_col_count = len(reference_row)
        expected_types = [type(item) for item in reference_row]

        for line_num, line in enumerate(lines):
            if not isinstance(line, list):
                raise ValueError(
                    f"\n\n🪖 𝗪𝗜𝗧𝗛𝗢𝗣𝗘𝗡 ▸ ❌ Structure mismatch detected!\n\n"
                    f"Line {line_num}: Expected a list (row) but found {type(line).__name__}\n"
                    f"Expected structure (from reference): {reference_row}\n"
                    f"    Actual structure (this line)   : {line}"
                )

            if len(line) != expected_col_count:
                raise ValueError(
                    f"\n\n🪖 𝗪𝗜𝗧𝗛𝗢𝗣𝗘𝗡 ▸ ❌ Column count mismatch detected!\n\n"
                    f"Line {line_num}: Expected {expected_col_count} columns, found {len(line)}\n"
                    f"Expected structure (from reference): {reference_row}\n"
                    f"    Actual structure (this line)   : {line}"
                )

            for col_idx, item in enumerate(line):
                expected_type = expected_types[col_idx]
                actual_type = type(item)
                if actual_type is not expected_type:
                    raise TypeError(
                        f"\n\n🪖 𝗪𝗜𝗧𝗛𝗢𝗣𝗘𝗡 ▸ ❌ Data type mismatch detected!\n\n"
                        f"Line {line_num}, Column {col_idx + 1}\n"
                        f" Expected  : {expected_type.__name__} "
                        f"(from reference value: {reference_row[col_idx]!r})\n"
                        f"   Found   : {actual_type.__name__} (from value: {item!r})\n"
                        f"Comparison : {expected_type.__name__} ≠ {actual_type.__name__}"
                    )

    else:
        expected_type = type(reference_structure[0])

        for row_idx, line in enumerate(lines):
            if isinstance(line, list):
                raise ValueError(
                    f"\n\n🪖 𝗪𝗜𝗧𝗛𝗢𝗣𝗘𝗡 ▸ ❌ Data structure mismatch detected!\n\n"
                    f"             Found Error           : in Row {row_idx}.\n"
                    f"Expected structure (from reference): 1D list (scalar elements)\n"
                    f"    Current structure (this line)  : {line}"
                )

            actual_type = type(line)
            if actual_type is not expected_type:
                raise TypeError(
                    f"\n\n🪖 𝗪𝗜𝗧𝗛𝗢𝗣𝗘𝗡 ▸ ❌ Data type mismatch detected!\n\n"
                    f"Found Error : in Row {row_idx}.\n"
                    f" Expected   : {expected_type.__name__}\n"
                    f"   Found    : {actual_type.__name__} (from value: {line})\n"
                    f" Comparison : {expected_type.__name__} ≠ {actual_type.__name__}"
                )


def smart_select(data, index):
    """
    Select elements from a 1D or 2D list based on the given index or indexes.

    Parameters:
    ----------
    data : list
        A list of values (1D) or a list of lists/tuples (2D).
    index : int, list, tuple
        - If data is 1D:
            - int: returns the element at that index.
            - list/tuple of ints: returns the elements at those indices.
        - If data is 2D:
            - int: selects that column index from each row.
            - list/tuple of ints: selects those column indices from each row.

    Returns:
    -------
    list or element
        - A single element if 1D with int index.
        - A list of selected elements otherwise.

    Raises:
    ------
    TypeError:
        If index is not an int, list, or tuple.
    IndexError:
        If any index is out of range for the given data structure.
    """

    # Validate index type
    if not isinstance(index, (int, list, tuple)):
        raise TypeError("Index must be an int, list, or tuple. Please confirm the index given.")

    is_2d = isinstance(data[0], (list, tuple))

    try:
        if is_2d:
            if isinstance(index, (list, tuple)):
                # Multiple elements from each row
                return [[row[i] for i in index] for row in data]
            else:
                # Single element from each row
                return [row[index] for row in data]
        else:
            if isinstance(index, (list, tuple)):
                return [data[i] for i in index]
            else:
                return data[index]
    except IndexError:
        raise IndexError("One or more indexes are out of range. Please confirm the index given in the parameter.")


def r_parameter_validation(txt_name, index, set_new, display, validate):
    """Basic parameter validation for the r() function."""
    if not isinstance(txt_name, str):
        raise TypeError("Parameter 'txt_name' must be a string.")

    # Validate index type
    if not (index is None or isinstance(index, (int, list, tuple))):
        raise TypeError("Index must be an int, list, tuple, or None.")

    if set_new is not None and not isinstance(set_new, list):
        raise TypeError("Parameter 'set_new' must be a list or None.")

    if not isinstance(display, bool):
        raise TypeError("Parameter 'display' must be a boolean (True/False).")

    if not isinstance(validate, bool):
        raise TypeError("Parameter 'validate' must be a boolean (True/False).")


def r(txt_name, index=None, set_new=None, display=True, validate=True):
    """
    Read data from the main and backup files.

    The function attempts to read `<name>.txt` first, then `<name>_backup.txt`.
    If both fail, it returns a default structure.

    Parameters
    ----------
    txt_name : str
        Base file name (with or without `.txt` extension).
    set_new : list, optional
        Default value to return if file is not found.
    display : bool, default True
        Whether to print a notice when the file does not exist.
    validate : bool, default True
        Whether to validate the file structure after reading.

    Returns
    -------
    list
        The contents of the file as a list.
    """
    r_parameter_validation(txt_name, index, set_new, display, validate)
    txt_name = normalize_name(txt_name)
    if set_new is None:
        set_new = []

    try:
        filenames = [
            f"{txt_name}.txt",
            f"{txt_name}_backup.txt"
        ]
        for filename in filenames:
            lines = read_file_lines(filename)
            if lines:
                if validate:
                    validate_lines(lines)
                break

        if index is not None:
            return smart_select(lines, index) #select function using specified index/indexes
        return lines if lines else set_new

    except FileNotFoundError:
        if display:
            print('🪖 𝗪𝗜𝗧𝗛𝗢𝗣𝗘𝗡 ▸ Reading As A New File 🟡.')
        return set_new


def validate_data_structure(write_list, is2d=None):
    """
    Ensure the structure of data being written matches the declared format.

    Parameters
    ----------
    write_list : list
        The list of data to validate (1D or 2D).
    is2d : bool or None
        Indicates whether the file is expected to store 2D data.

    Raises
    ------
    ValueError
        If 1D/2D structure conflicts with the declared mode.
    """
    is_2d = is_2d_list(write_list)

    if is_2d is False and is2d is None:
        raise TypeError(
            "🟡 1D data detected. On first write or after a reset/empty (None or []), setting parameter [ is2d=False ].\n"
            "               This prevents incorrect structure locking for a 2D file created inside a loop.\n"
            "               This action/parameter is only needed when a 1D list is passed at creation or reset.\n"
            "               Auto-correct only works after you've initialized the file (That is after proper first a write or append.)"
        )

    elif is_2d is False and is2d is True:
        raise TypeError(
            "⚠️ 'is2d=True' was set, but the data appears 1D. \n"
            "                If you're trying to create a 2D structure, wrap your list in an extra bracket like [[...]]."
        )

    elif is_2d and is2d is False:
        raise TypeError("⚠️ is2d parameter and new item Data-Type mismatch.")


def write_all_files(write_list, txt_name):
    """
    Write the given list to both the main and backup files.

    Parameters
    ----------
    write_list : list
        The data to write.
    txt_name : str
        The base file name (without extension).
    """
    if write_list is None:
        lines = ['None']
    else:
        lines = [str(x) for x in write_list]

    write_str = '\n'.join(lines)
    filenames = [
        f"{txt_name}.txt",
        f"{txt_name}_backup.txt"
    ]

    for filename in filenames:
        with open(filename, 'w', encoding='utf-8') as f:
            f.write(write_str)


def process_write_append(write_list, txt_name, is2d, validate):
    """
    Common logic for both write (`w`) and append (`a`) operations.

    Parameters
    ----------
    write_list : list
        The list of data to process.
    txt_name : str
        The base file name.
    is2d : bool or None
        Declared structure type.
    validate : bool
        Whether to perform validation.

    Returns
    -------
    list
        The current file content read before writing/appending.
    """
    if is2d not in (None, True, False):
        raise TypeError("Parameter 'is2d' must be either None, True, or False.")

    if write_list:
        write_list = normalize_to_list(write_list)

    txt_name = normalize_name(txt_name)
    read_file = r(txt_name, set_new=[], display=False, validate=False)

    if not read_file:
        validate_data_structure(write_list, is2d=is2d)

    if write_list and validate:
        if is_2d_list(read_file) and not is_2d_list(write_list):
            write_list = [write_list]
        if read_file:
            validate_lines(write_list, reference=read_file)
        else:
            validate_lines(write_list)

    return read_file


def wa_parameter_validation(txt_name, data_list, is2d, validate):
    """Shared parameter validation for w() and a()."""
    if not isinstance(txt_name, str):
        raise TypeError("Parameter 'txt_name' must be a string.")

    valid_data_types = (list, tuple, set, dict, range, str, int, float, bool, type(None))
    if not isinstance(data_list, valid_data_types):
        raise TypeError(
            f"Parameter 'write_list' or 'append_list' must be list-like or scalar "
            f"(not {type(data_list).__name__})."
        )

    if is2d not in (None, True, False):
        raise TypeError("Parameter 'is2d' must be None, True, or False.")

    if not isinstance(validate, bool):
        raise TypeError("Parameter 'validate' must be a boolean (True/False).")


def w(txt_name, write_list, is2d=None, validate=True):
    """
    Write new data to both the main and backup files, overwriting any existing content.

    Parameters
    ----------
    txt_name : str
        The base file name.
    write_list : list
        The data to write.
    is2d : bool or None, optional
        Whether the file should be treated as 2D.
    validate : bool, default True
        Whether to perform validation before writing.
    """
    wa_parameter_validation(txt_name, write_list, is2d, validate)
    process_write_append(write_list, txt_name, is2d, validate)
    write_all_files(write_list, txt_name)


def a(txt_name, append_list, is2d=None, validate=True):
    """
    Append data to both the main and backup files.

    Parameters
    ----------
    txt_name : str
        The base file name.
    append_list : list
        The data to append.
    is2d : bool or None, optional
        Whether the file should be treated as 2D.
    validate : bool, default True
        Whether to perform validation before appending.

    Returns
    -------
    list
        The combined list after appending.
    """
    wa_parameter_validation(txt_name, append_list, is2d, validate)
    read_file = process_write_append(append_list, txt_name, is2d, validate)
    combined_list = read_file + append_list
    write_all_files(combined_list, txt_name)
    return combined_list
