"""
Bytecode Optimizer for fr
Performs peephole optimizations on generated bytecode
"""

from typing import List, Dict, Tuple

class BytecodeOptimizer:
    """Optimizes bytecode instruction sequences"""

    def __init__(self):
        self.labels: Dict[str, int] = {}  # Label -> line number

    def optimize(self, bytecode: str) -> str:
        """Apply all optimization passes to bytecode"""
        lines = bytecode.strip().split('\n')

        # Pass 1: Combine common patterns (does INC_LOCAL conversion)
        lines = self.combine_patterns(lines)

        # Pass 2: Optimize variable copies (LOAD+STORE -> COPY_LOCAL_REF)
        lines = self.optimize_variable_copies(lines)

        # Pass 3: Remove redundant LOAD/STORE pairs
        lines = self.remove_redundant_loads(lines)

        # Pass 4: Remove dead stores
        lines = self.remove_dead_stores(lines)

        return '\n'.join(lines)

    def optimize_variable_copies(self, lines: List[str]) -> List[str]:
        """
        Convert LOAD x; STORE y patterns to COPY_LOCAL_REF x y when safe.
        Safe when: y is not modified before next use (simple heuristic: y is reassigned before next load)
        """
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()

            # Pattern: LOAD x, STORE y
            if (i + 1 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('STORE ')):

                src_var = line.split()[1]
                dst_var = lines[i+1].strip().split()[1]

                # Don't optimize if it's the same variable (handled by other pass)
                if src_var != dst_var:
                    # Check if this is safe: look ahead to see if dst is used before being reassigned
                    # For simplicity, use COPY_LOCAL_REF for common safe pattern: shifting values
                    # (like a=b, b=c in fibonacci)
                    
                    # Simple heuristic: if the next few instructions don't modify src,
                    # and dst is only used after being reassigned, it's safe
                    is_safe = self._is_copy_ref_safe(lines, i+2, src_var, dst_var)
                    
                    indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                    if is_safe:
                        result.append(f"{indent}COPY_LOCAL_REF {src_var} {dst_var}")
                    else:
                        result.append(f"{indent}COPY_LOCAL {src_var} {dst_var}")
                    i += 2
                    continue

            result.append(lines[i])
            i += 1

        return result

    def _is_copy_ref_safe(self, lines: List[str], start_idx: int, src_var: str, dst_var: str, lookahead: int = 10) -> bool:
        """
        Determine if using COPY_LOCAL_REF is safe.
        Safe if dst is not modified (via ADD/SUB/etc) before being reassigned.
        """
        for i in range(start_idx, min(start_idx + lookahead, len(lines))):
            line = lines[i].strip()
            
            # If we hit a label or control flow, be conservative
            if line.startswith('LABEL ') or line.startswith('JUMP'):
                return False
            
            # If dst is stored to again, it's safe (old alias is replaced)
            if line.startswith(f'STORE {dst_var}') or line.startswith(f'COPY_LOCAL_REF ') and dst_var in line:
                return True
            
            # If dst is loaded for an operation, we need to copy (not safe for ref)
            if line.startswith(f'LOAD {dst_var}'):
                # Check if next instruction modifies it
                if i + 1 < len(lines):
                    next_line = lines[i+1].strip()
                    # If it's used in arithmetic, we need deep copy
                    if any(next_line.startswith(op) for op in ['ADD', 'SUB', 'MUL', 'DIV', 'MOD']):
                        return False
            
            # If src is modified, the alias becomes invalid
            if line.startswith(f'STORE {src_var}'):
                return False
        
        # Default: be conservative
        return False

    def remove_redundant_loads(self, lines: List[str]) -> List[str]:
        """Remove STORE followed by LOAD of same variable when safe"""
        result = []
        i = 0
        while i < len(lines):
            line = lines[i].strip()

            # Check for STORE N followed by LOAD N
            if i + 1 < len(lines) and line.startswith('STORE '):
                next_line = lines[i + 1].strip()
                if next_line.startswith('LOAD '):
                    var1 = line.split()[1]
                    var2 = next_line.split()[1]
                    if var1 == var2:
                        # Only optimize if the LOAD is immediately consumed by an operation
                        # Check what follows the LOAD
                        if i + 2 < len(lines):
                            following_line = lines[i + 2].strip()
                            # Safe to optimize if followed by an arithmetic/comparison operation
                            safe_ops = ['ADD_', 'SUB_', 'MUL_', 'DIV_', 'MOD_', 'CMP_', 'AND', 'OR', 'NOT']
                            is_safe = any(following_line.startswith(op) for op in safe_ops)
                            
                            if is_safe:
                                # Keep STORE, add DUP before it, skip LOAD
                                result.append(lines[i].replace('STORE', 'DUP\n  STORE'))
                                i += 2
                                continue
                        # If not safe or no following line, keep both STORE and LOAD

            result.append(lines[i])
            i += 1

        return result

    def combine_patterns(self, lines: List[str]) -> List[str]:
        """Combine common instruction patterns into optimized forms"""
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()

            # Pattern: LOAD N, CONST_I64 1, ADD_I64, STORE N -> INC_LOCAL N
            if (i + 3 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip() == 'CONST_I64 1' and
                lines[i+2].strip() == 'ADD_I64' and
                lines[i+3].strip().startswith('STORE ')):

                load_var = line.split()[1]
                store_var = lines[i+3].strip().split()[1]

                if load_var == store_var:
                    # Replace with optimized INC instruction
                    indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                    result.append(f"{indent}INC_LOCAL {load_var}")
                    i += 4
                    continue

            # Pattern: LOAD N, CONST_I64 1, SUB_I64, STORE N -> DEC_LOCAL N
            if (i + 3 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip() == 'CONST_I64 1' and
                lines[i+2].strip() == 'SUB_I64' and
                lines[i+3].strip().startswith('STORE ')):

                load_var = line.split()[1]
                store_var = lines[i+3].strip().split()[1]

                if load_var == store_var:
                    indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                    result.append(f"{indent}DEC_LOCAL {load_var}")
                    i += 4
                    continue

            # Pattern: CONST_I64 N, ADD_I64 -> ADD_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'ADD_I64'):

                const_val = line.split()[1]
                indent = '  ' if not lines[i].startswith('CONST') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}ADD_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, SUB_I64 -> SUB_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'SUB_I64'):

                const_val = line.split()[1]
                indent = '  ' if not lines[i].startswith('CONST') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}SUB_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, MUL_I64 -> MUL_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'MUL_I64'):

                const_val = line.split()[1]
                indent = '  ' if not lines[i].startswith('CONST') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}MUL_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, DIV_I64 -> DIV_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'DIV_I64'):

                const_val = line.split()[1]
                indent = '  ' if not lines[i].startswith('CONST') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}DIV_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: LOAD N, LOAD N -> LOAD N, DUP
            if (i + 1 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ')):

                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]

                if var1 == var2:
                    result.append(lines[i])
                    indent = '  ' if not lines[i+1].startswith('LOAD') else lines[i+1][:len(lines[i+1]) - len(lines[i+1].lstrip())]
                    result.append(f"{indent}DUP")
                    i += 2
                    continue

            # NEW: Pattern: LOAD a, LOAD b, ADD_I64 -> LOAD2_ADD_I64 a b (before multi-LOAD combining!)
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'ADD_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_ADD_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, SUB_I64 -> LOAD2_SUB_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'SUB_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_SUB_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, MUL_I64 -> LOAD2_MUL_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'MUL_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_MUL_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_LT -> LOAD2_CMP_LT a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_LT'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_CMP_LT {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_GT -> LOAD2_CMP_GT a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_GT'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_CMP_GT {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_LE -> LOAD2_CMP_LE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_LE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_CMP_LE {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_GE -> LOAD2_CMP_GE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_GE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_CMP_GE {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_EQ -> LOAD2_CMP_EQ a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_EQ'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_CMP_EQ {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_NE -> LOAD2_CMP_NE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_NE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                result.append(f"{indent}LOAD2_CMP_NE {var1} {var2}")
                i += 3
                continue

            # Pattern: Multiple consecutive LOAD instructions -> LOAD with multiple args
            # Look for sequences of LOAD instructions
            # BUT: Don't combine 2 LOADs if followed by fusable operation (let LOAD2+OP handle it)
            if line.startswith('LOAD '):
                loads = [line.strip().split()[1]]
                j = i + 1
                while j < len(lines) and lines[j].strip().startswith('LOAD '):
                    loads.append(lines[j].strip().split()[1])
                    j += 1
                
                # Check if this is exactly 2 LOADs followed by a fusable operation
                fusable_ops = {'ADD_I64', 'SUB_I64', 'MUL_I64', 'CMP_LT', 'CMP_GT', 'CMP_LE', 'CMP_GE', 'CMP_EQ', 'CMP_NE'}
                if len(loads) == 2 and j < len(lines) and lines[j].strip() in fusable_ops:
                    # Don't combine - let it fall through to LOAD2+OP patterns below
                    pass
                elif len(loads) >= 2:
                    # Combine multiple LOADs
                    indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                    result.append(f"{indent}LOAD {' '.join(loads)}")
                    i = j
                    continue

            # Pattern: LOAD a b (multi-LOAD), OP -> LOAD2_OP a b (must run AFTER multi-LOAD combining)
            # Check if this is a multi-arg LOAD followed by fused-capable operation
            stripped_line = line.strip()
            if stripped_line.startswith('LOAD ') and len(stripped_line.split()) == 3:  # LOAD with exactly 2 args
                parts = stripped_line.split()
                var1, var2 = parts[1], parts[2]
                indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                
                # Check what operation follows
                if i + 1 < len(lines):
                    next_op = lines[i+1].strip()
                    
                    if next_op == 'ADD_I64':
                        result.append(f"{indent}LOAD2_ADD_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'SUB_I64':
                        result.append(f"{indent}LOAD2_SUB_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'MUL_I64':
                        result.append(f"{indent}LOAD2_MUL_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_LT':
                        result.append(f"{indent}LOAD2_CMP_LT {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_GT':
                        result.append(f"{indent}LOAD2_CMP_GT {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_LE':
                        result.append(f"{indent}LOAD2_CMP_LE {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_GE':
                        result.append(f"{indent}LOAD2_CMP_GE {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_EQ':
                        result.append(f"{indent}LOAD2_CMP_EQ {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_NE':
                        result.append(f"{indent}LOAD2_CMP_NE {var1} {var2}")
                        i += 2
                        continue

            # Pattern: Multiple LOAD src, STORE dst pairs -> FUSED_LOAD_STORE
            # Look for sequences of LOAD/STORE pairs
            if line.startswith('LOAD ') and i + 1 < len(lines) and lines[i+1].strip().startswith('STORE '):
                # Found first pair, look for more consecutive pairs
                pairs = []
                j = i
                while (j + 1 < len(lines) and 
                       lines[j].strip().startswith('LOAD ') and
                       lines[j+1].strip().startswith('STORE ')):
                    src = lines[j].strip().split()[1]
                    dst = lines[j+1].strip().split()[1]
                    pairs.append((src, dst))
                    j += 2
                
                # If we found 2 or more pairs, use FUSED_LOAD_STORE
                if len(pairs) >= 2:
                    indent = '  ' if not lines[i].startswith('LOAD') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                    # Build the fused instruction
                    args = ' '.join(f"{src} {dst}" for src, dst in pairs)
                    result.append(f"{indent}FUSED_LOAD_STORE {args}")
                    i = j
                    continue

            # Pattern: Multiple STORE dst, LOAD src pairs -> FUSED_STORE_LOAD
            # Look for sequences of STORE/LOAD pairs
            if line.startswith('STORE ') and i + 1 < len(lines) and lines[i+1].strip().startswith('LOAD '):
                # Found first pair, look for more consecutive pairs
                pairs = []
                j = i
                while (j + 1 < len(lines) and 
                       lines[j].strip().startswith('STORE ') and
                       lines[j+1].strip().startswith('LOAD ')):
                    dst = lines[j].strip().split()[1]
                    src = lines[j+1].strip().split()[1]
                    pairs.append((dst, src))
                    j += 2
                
                # If we found 2 or more pairs, use FUSED_STORE_LOAD
                if len(pairs) >= 2:
                    indent = '  ' if not lines[i].startswith('STORE') else lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                    # Build the fused instruction
                    args = ' '.join(f"{dst} {src}" for dst, src in pairs)
                    result.append(f"{indent}FUSED_STORE_LOAD {args}")
                    i = j
                    continue

            # Pattern: LOAD a, LOAD b, SWAP -> LOAD b, LOAD a (reduce SWAP usage)
            # This is actually worse, so skip it

            # Pattern: DUP, LOAD N -> LOAD N, DUP (may enable other optimizations)
            # Skip for now as it's context-dependent

            result.append(lines[i])
            i += 1

        return result

    def remove_dead_stores(self, lines: List[str]) -> List[str]:
        """Remove stores to variables that are never loaded (basic pass)"""
        # Build label map first
        label_map = {}
        for i, line in enumerate(lines):
            if line.strip().startswith('LABEL '):
                label_name = line.strip().split()[1]
                label_map[label_name] = i

        # For now, just return as-is (dead store elimination is complex with control flow)
        # TODO: Implement proper liveness analysis
        return lines

