from __future__ import annotations

from dataclasses import replace

import numpy as np

from .._checks import check_shape
from ..qarrays.layout import dense
from ..qarrays.qarray import QArray, QArrayLike
from ..qarrays.utils import asqarray
from .general import dag
from .operators import eye

__all__ = [
    'vectorize',
    'sdissipator',
    'slindbladian',
    'spost',
    'spre',
    'sprepost',
    'unvectorize',
]


def vectorize(x: QArrayLike) -> QArray:
    r"""Returns the vectorized version of an operator.

    The vectorized column vector $\kett{A}$ (shape $n^2\times 1$) is obtained by
    stacking the columns of the matrix $A$ (shape $n\times n$) on top of one another:
    $$
        A = \begin{pmatrix} a & b \\\\ c & d \end{pmatrix}
        \to
        \kett{A} = \begin{pmatrix} a \\\\ c \\\\ b \\\\ d \end{pmatrix}.
    $$

    Args:
        x (qarray-like of shape (..., n, n)): Operator.

    Returns:
        (qarray of shape (..., n^2, 1)): Vectorized operator.

    Examples:
        >>> A = jnp.array([[1 + 1j, 2 + 2j], [3 + 3j, 4 + 4j]])
        >>> A
        Array([[1.+1.j, 2.+2.j],
               [3.+3.j, 4.+4.j]], dtype=complex64)
        >>> dq.vectorize(A)
        QArray: shape=(4, 1), dims=(2,), dtype=complex64, layout=dense, vectorized=True
        [[1.+1.j]
         [3.+3.j]
         [2.+2.j]
         [4.+4.j]]
    """
    x = asqarray(x, layout=dense)
    check_shape(x, 'x', '(..., n, n)')
    bshape = x.shape[:-2]
    x = x.mT._reshape_unchecked(*bshape, -1, 1)
    return replace(x, vectorized=True)


def unvectorize(x: QArrayLike) -> QArray:
    r"""Returns the operator version of a vectorized operator.

    The matrix $A$ (shape $n\times n$) is obtained by stacking horizontally next to
    each other each group of $n$ elements of the vectorized column vector $\kett{A}$
    (shape $n^2\times 1$):
    $$
        \kett{A} = \begin{pmatrix} a \\\\ b \\\\ c \\\\ d \end{pmatrix}
        \to
        A = \begin{pmatrix} a & c \\\\ b & d \end{pmatrix}.
    $$

    Args:
        x (qarray-like of shape (..., n^2, 1)): Vectorized operator.

    Returns:
        (qarray of shape (..., n, n)): Operator.

    Examples:
        >>> Avec = jnp.array([[1 + 1j], [2 + 2j], [3 + 3j], [4 + 4j]])
        >>> Avec
        Array([[1.+1.j],
               [2.+2.j],
               [3.+3.j],
               [4.+4.j]], dtype=complex64)
        >>> dq.unvectorize(Avec)
        QArray: shape=(2, 2), dims=(2,), dtype=complex64, layout=dense
        [[1.+1.j 3.+3.j]
         [2.+2.j 4.+4.j]]
    """
    x = asqarray(x, layout=dense)
    check_shape(x, 'x', '(..., n^2, 1)')
    bshape = x.shape[:-2]
    n = round(np.sqrt(x.shape[-2]))
    x = replace(x, dims=(n,))
    x = x._reshape_unchecked(*bshape, n, n).mT
    return replace(x, vectorized=False)


def spre(x: QArrayLike) -> QArray:
    r"""Returns the superoperator formed from pre-multiplication by an operator.

    Pre-multiplication by matrix $A$ is defined by the superoperator
    $I_n \otimes A$ in vectorized form:
    $$
        AX \to (I_n \otimes A) \kett{X}.
    $$

    Args:
        x (qarray-like of shape (..., n, n)): Operator.

    Returns:
        (qarray of shape (..., n^2, n^2)): Pre-multiplication superoperator.

    Examples:
        >>> dq.spre(dq.destroy(3)).shape
        (9, 9)
    """
    x = asqarray(x)
    check_shape(x, 'x', '(..., n, n)')
    n = x.shape[-1]
    xpre = eye(n) & x
    return replace(xpre, dims=x.dims, vectorized=True)


def spost(x: QArrayLike) -> QArray:
    r"""Returns the superoperator formed from post-multiplication by an operator.

    Post-multiplication by matrix $A$ is defined by the superoperator
    $A^\mathrm{T} \otimes I_n$ in vectorized form:
    $$
        XA \to (A^\mathrm{T} \otimes I_n) \kett{X}.
    $$

    Args:
        x (qarray-like of shape (..., n, n)): Operator.

    Returns:
        (qarray of shape (..., n^2, n^2)): Post-multiplication superoperator.

    Examples:
        >>> dq.spost(dq.destroy(3)).shape
        (9, 9)
    """
    x = asqarray(x)
    check_shape(x, 'x', '(..., n, n)')
    n = x.shape[-1]
    xpost = x.mT & eye(n)
    return replace(xpost, dims=x.dims, vectorized=True)


def sprepost(x: QArrayLike, y: QArrayLike) -> QArray:
    r"""Returns the superoperator formed from pre- and post-multiplication by operators.

    Pre-multiplication by matrix $A$ and post-multiplication by matrix $B$ is defined
    by the superoperator $B^\mathrm{T} \otimes A$ in vectorized form:
    $$
        AXB \to (B^\mathrm{T} \otimes A) \kett{X}.
    $$

    Args:
        x (qarray-like of shape (..., n, n)): Operator for pre-multiplication.
        y (qarray-like of shape (..., n, n)): Operator for post-multiplication.

    Returns:
        (Qarray of shape (..., n^2, n^2)): Pre- and post-multiplication superoperator.

    Examples:
        >>> dq.sprepost(dq.destroy(3), dq.create(3)).shape
        (9, 9)
    """
    x = asqarray(x)
    y = asqarray(y)
    check_shape(x, 'x', '(..., n, n)')
    check_shape(y, 'y', '(..., n, n)')
    xyprepost = y.mT & x
    return replace(xyprepost, dims=x.dims, vectorized=True)


def sdissipator(L: QArrayLike) -> QArray:
    r"""Returns the Lindblad dissipation superoperator (in matrix form).

    The dissipation superoperator $\mathcal{D}[L]$ is defined by:
    $$
        \mathcal{D}[L] (\rho) = L\rho L^\dag - \frac{1}{2}L^\dag L \rho
        - \frac{1}{2}\rho L^\dag L.
    $$

    The vectorized form of this superoperator is:
    $$
        L^\* \otimes L
        - \frac{1}{2} (I_n \otimes L^\dag L)
        - \frac{1}{2} (L^\mathrm{T} L^\* \otimes I_n).
    $$

    Args:
        L (qarray-like of shape (..., n, n)): Jump operator.

    Returns:
        (qarray of shape (..., n^2, n^2)): Dissipation superoperator.

    See also:
        - [`dq.dissipator()`][dynamiqs.dissipator]: applies the dissipation
            superoperator to a state using only $n\times n$ matrix multiplications.
    """
    L = asqarray(L)
    check_shape(L, 'L', '(..., n, n)')
    Ldag = dag(L)
    LdagL = Ldag @ L
    return sprepost(L, Ldag) - 0.5 * spre(LdagL) - 0.5 * spost(LdagL)


def slindbladian(H: QArrayLike, jump_ops: list[QArrayLike]) -> QArray:
    r"""Returns the Lindbladian superoperator (in matrix form).

    The Lindbladian superoperator $\mathcal{L}$ is defined by:
    $$
        \mathcal{L} (\rho) = -i[H,\rho] + \sum_{k=1}^N \mathcal{D}[L_k] (\rho),
    $$

    where $H$ is the system Hamiltonian, $\{L_k\}$ is a set of $N$ jump operators
    (arbitrary operators) and $\mathcal{D}[L]$ is the Lindblad dissipation superoperator
    (see [`dq.sdissipator()`][dynamiqs.sdissipator]).

    The vectorized form of this superoperator is:
    $$
        -i (I_n \otimes H) + i (H^\mathrm{T} \otimes I_n) + \sum_{k=1}^N \left(
            L_k^\* \otimes L_k
            - \frac{1}{2} (I_n \otimes L_k^\dag L_k)
            - \frac{1}{2} (L_k^\mathrm{T} L_k^\* \otimes I_n)
        \right).
    $$

    Note:
        This superoperator is also sometimes called *Liouvillian*.

    Args:
        H (qarray-like of shape (..., n, n)): Hamiltonian.
        jump_ops (list of qarray-like, each of shape (..., n, n)): List of jump
            operators.

    Returns:
        (qarray of shape (..., n^2, n^2)): Lindbladian superoperator.

    See also:
        - [`dq.lindbladian()`][dynamiqs.lindbladian]: applies the Lindbladian
            superoperator to a state using only $n\times n$ matrix multiplications.
    """
    H = asqarray(H)
    jump_ops = [asqarray(L) for L in jump_ops]

    # === check H shape
    check_shape(H, 'H', '(..., n, n)')

    # === check jump_ops shape
    for i, L in enumerate(jump_ops):
        check_shape(L, f'jump_ops[{i}]', '(..., n, n)')

    return -1j * spre(H) + 1j * spost(H) + sum([sdissipator(L) for L in jump_ops])
