import difflib

# --- AST NODES (Same as before) ---
class NumNode:
    def __init__(self, token): 
        self.value = token.value
        self.lineno = token.lineno
    def __repr__(self): return f"Num({self.value})"

class StringNode:
    def __init__(self, token): 
        self.value = token.value
        self.lineno = token.lineno
    def __repr__(self): return f"Str({self.value})"

class VarAccessNode:
    def __init__(self, token): 
        self.name = token.value
        self.lineno = token.lineno
    def __repr__(self): return f"Var({self.name})"

class BinOpNode:
    def __init__(self, left, op_token, right):
        self.left = left
        self.op_token = op_token
        self.right = right
        self.lineno = op_token.lineno
    def __repr__(self): return f"BinOp({self.left}, {self.op_token.type}, {self.right})"

class AssignNode:
    def __init__(self, name_token, value_node):
        self.name = name_token.value
        self.value = value_node
        self.lineno = name_token.lineno
    def __repr__(self): return f"Assign({self.name}, {self.value})"

class PrintNode:
    def __init__(self, value_node, lineno_token):
        self.value = value_node
        self.lineno = lineno_token.lineno
    def __repr__(self): return f"Print({self.value})"

class IfNode:
    def __init__(self, cases, else_case, lineno):
        self.cases = cases
        self.else_case = else_case
        self.lineno = lineno
    def __repr__(self): return f"If(Cases: {self.cases}, Else: {self.else_case})"

class WhileNode:
    def __init__(self, condition_node, statements, lineno):
        self.condition_node = condition_node
        self.statements = statements
        self.lineno = lineno
    def __repr__(self): return f"While(Condition: {self.condition_node})"


# --- PARSER ---
class Parser:
    def __init__(self, tokens):
        self.tokens = tokens
        self.idx = -1
        self.current_token = None
        self.advance()

        # Define valid keywords for "Did you mean?" suggestions
        self.VALID_KEYWORDS = [
            'IPAKITA', 'IKABIL', 'NO', 'BAYAT', 'NALPAS', 'NO KET DI', 'NO KUMA'
        ]

    def advance(self):
        self.idx += 1
        self.current_token = self.tokens[self.idx] if self.idx < len(self.tokens) else None

    def error(self, message):
        """Raises a formatted syntax error with line number."""
        if self.current_token:
            line = self.current_token.lineno
        elif self.tokens:
            line = self.tokens[-1].lineno
        else:
            line = 1
        raise Exception(f"Syntax Error on Line {line}: {message}")

    def skip_newlines(self):
        while self.current_token and self.current_token.type == 'NEWLINE':
            self.advance()

    def parse(self):
        statements = []
        self.skip_newlines()
        
        while self.current_token:
            statements.append(self.parse_statement())
            
            if self.current_token:
                if self.current_token.type != 'NEWLINE':
                    if self.current_token.type not in ('NALPAS', 'NO_KET_DI', 'NO_KUMA'):
                        self.error(f"Expected newline, but got {self.current_token.type}")
                else:
                    self.skip_newlines()

        return statements

    def parse_statement(self):
        """Parse a single statement."""
        if self.current_token.type == 'LET':
            return self.parse_let_statement()
        elif self.current_token.type == 'PRINT':
            return self.parse_print_statement()
        elif self.current_token.type == 'NO':
            return self.parse_if_statement()
        elif self.current_token.type == 'BAYAT':
            return self.parse_while_statement()
        else:
            # --- NEW: Enhanced Error Handling with Suggestions ---
            unexpected_val = self.current_token.value
            
            # If the unexpected token is an identifier (like 'IPAKIA'), check for typos
            if self.current_token.type == 'ID':
                # Find closest match in keywords
                matches = difflib.get_close_matches(unexpected_val, self.VALID_KEYWORDS, n=1, cutoff=0.6)
                if matches:
                    self.error(f"Unknown command '{unexpected_val}'. Did you mean '{matches[0]}'?")
                else:
                    self.error(f"Unknown command '{unexpected_val}'")
            else:
                self.error(f"Unexpected token: {self.current_token.type}")

    def parse_let_statement(self):
        self.advance() 
        if self.current_token.type != 'ID':
            self.error("Expected identifier after 'IKABIL'")
        name_token = self.current_token
        self.advance()
        if self.current_token.type != 'EQUALS':
            self.error("Expected '=' in assignment")
        self.advance()
        value_node = self.parse_comparison()
        return AssignNode(name_token, value_node)

    def parse_print_statement(self):
        print_token = self.current_token
        self.advance()
        value_node = self.parse_comparison()
        return PrintNode(value_node, print_token)

    def parse_if_statement(self):
        if_token = self.current_token
        cases = []
        else_case = None

        self.advance() # Skip 'NO'
        condition = self.parse_comparison()
        
        if not self.current_token or self.current_token.type != 'NEWLINE':
            self.error("Expected newline after 'NO' condition")
        self.skip_newlines()

        statements = self.parse_statements_until(['NO_KET_DI', 'NO_KUMA', 'NALPAS'])
        cases.append((condition, statements))

        while self.current_token and self.current_token.type == 'NO_KET_DI':
            self.advance() 
            condition = self.parse_comparison()
            if not self.current_token or self.current_token.type != 'NEWLINE':
                self.error("Expected newline after 'NO KET DI'")
            self.skip_newlines()
            statements = self.parse_statements_until(['NO_KET_DI', 'NO_KUMA', 'NALPAS'])
            cases.append((condition, statements))

        if self.current_token and self.current_token.type == 'NO_KUMA':
            self.advance()
            if not self.current_token or self.current_token.type != 'NEWLINE':
                self.error("Expected newline after 'NO KUMA'")
            self.skip_newlines()
            else_case = self.parse_statements_until(['NALPAS'])

        if not self.current_token or self.current_token.type != 'NALPAS':
            self.error("Expected 'NALPAS' to end 'NO' block")
        self.advance()

        return IfNode(cases, else_case, if_token.lineno)

    def parse_while_statement(self):
        while_token = self.current_token
        self.advance() 
        condition = self.parse_comparison()

        if not self.current_token or self.current_token.type != 'NEWLINE':
            self.error("Expected newline after 'BAYAT' condition")
        self.skip_newlines()

        statements = self.parse_statements_until(['NALPAS'])

        if not self.current_token or self.current_token.type != 'NALPAS':
            self.error("Expected 'NALPAS' to end 'BAYAT' block")
        self.advance()

        return WhileNode(condition, statements, while_token.lineno)

    def parse_statements_until(self, stop_tokens):
        statements = []
        if not self.current_token or self.current_token.type in stop_tokens:
            return statements
            
        while self.current_token and self.current_token.type not in stop_tokens:
            statements.append(self.parse_statement())
            
            if self.current_token and self.current_token.type not in stop_tokens:
                if self.current_token.type != 'NEWLINE':
                    self.error("Expected newline after statement in block")
                self.skip_newlines()
                if not self.current_token:
                    self.error(f"Unexpected end of file, expected {stop_tokens}")
        return statements

    def parse_comparison(self):
        left = self.parse_expression()
        while self.current_token and self.current_token.type in ('EQUALTO', 'NOT_EQUAL', 'LESS', 'LESS_EQ', 'GREATER', 'GREATER_EQ'):
            op_token = self.current_token
            self.advance()
            right = self.parse_expression()
            left = BinOpNode(left, op_token, right)
        return left

    def parse_expression(self):
        left = self.parse_term()
        while self.current_token and self.current_token.type in ('PLUS', 'MINUS'):
            op_token = self.current_token
            self.advance()
            right = self.parse_term()
            left = BinOpNode(left, op_token, right)
        return left
        
    def parse_term(self):
        left = self.parse_atom()
        while self.current_token and self.current_token.type in ('MUL', 'DIV', 'MODULO'):
            op_token = self.current_token
            self.advance()
            right = self.parse_atom()
            left = BinOpNode(left, op_token, right)
        return left

    def parse_atom(self):
        token = self.current_token
        if token is None: self.error("Unexpected end of input")

        if token.type == 'NUMBER':
            self.advance()
            return NumNode(token)
        elif token.type == 'STRING':
            self.advance()
            return StringNode(token)
        elif token.type == 'ID':
            self.advance()
            return VarAccessNode(token)
        elif token.type == 'LPAREN':
            self.advance()
            node = self.parse_comparison()
            if not self.current_token or self.current_token.type != 'RPAREN':
                self.error("Expected ')'")
            self.advance()
            return node
        else:
            self.error(f"Unexpected token: {token.type}")