#!/usr/bin/env python3
import typer
import requests
from bs4 import BeautifulSoup
import sys
import time
import threading
import os

app = typer.Typer(no_args_is_help=False)

CURRENCY_NAMES = {
    "دلار آمریکا": "US Dollar",
    "دلار استرالیا": "AUD",
    "دلار کانادا": "CAD",
    "دلار نیوزیلند": "NZD",
    "دلار سنگاپور": "SGD",
    "دلار هنگ کنگ": "HKD",
    "یورو": "Euro",
    "درهم امارات": "AED",
    "پوند انگلیس": "GBP",
    "فرانک سوئیس": "CHF",
    "لیر ترکیه": "TRY",
    "ین ژاپن": "JPY",
    "یوان چین": "CNY",
    "کرون سوئد": "SEK",
    "کرون نروژ": "NOK",
    "روبل روسیه": "RUB",
    "دلار استانبول": "Istanbul Dollar",  # ✅ اضافه شد
}

GOLD_NAMES = {
    "گرم طلای 18 عیار": "Talaye 18 eyar",
    "آبشده": "Mesghal",
    "سکه امامی": "Sekke Emami",
    "انس طلا": "Ounce gold",
}

LOADING = False

def clear_screen():
    os.system('cls' if os.name == 'nt' else 'clear')

def show_header():
    typer.echo("=" * 50)
    typer.echo("           ARZ PLUS CLI")
    typer.echo("=" * 50)
    typer.echo()

def persian_to_latin_digits(text: str) -> str:
    persian_digits = "۰۱۲۳۴۵۶۷۸۹"
    arabic_digits = "٠١٢٣٤٥٦٧٨٩"
    latin_digits = "0123456789"
    trans_table = str.maketrans(persian_digits + arabic_digits, latin_digits * 2)
    return text.translate(trans_table).replace(",", "").strip()

def loading_animation():
    chars = "/—\\|"
    idx = 0
    while LOADING:
        sys.stdout.write("\rFetching data... " + chars[idx % len(chars)])
        sys.stdout.flush()
        time.sleep(0.15)
        idx += 1
    sys.stdout.write("\r" + " " * 30 + "\r")
    sys.stdout.flush()

def start_loading():
    global LOADING
    LOADING = True
    t = threading.Thread(target=loading_animation)
    t.daemon = True
    t.start()

def stop_loading():
    global LOADING
    LOADING = False
    time.sleep(0.1)

def fetch_dollar():
    start_loading()
    url = "https://alanchand.com"
    try:
        response = requests.get(url, timeout=10)
        response.encoding = 'utf-8'
        if response.status_code != 200:
            stop_loading()
            typer.echo("❌ Failed to fetch data from website.")
            return

        soup = BeautifulSoup(response.text, 'html.parser')
        table = soup.find('table', class_='CurrencyTbl table table-striped xt-center mb-0 w-100 bg-white rounded-3 shadow-lg')
        if not table:
            stop_loading()
            typer.echo("❌ Currency table not found.")
            return

        stop_loading()
        typer.echo("Live Currency Prices:")
        typer.echo("-" * 40)
        for row in table.find_all('tr'):
            cols = row.find_all('td')
            if cols:
                name_fa = cols[0].text.strip()
                name_en = CURRENCY_NAMES.get(name_fa, name_fa)
                sell_price = persian_to_latin_digits(cols[2].text.strip())
                typer.echo(f"{name_en}: {sell_price} Toman")
    except Exception as e:
        stop_loading()
        typer.echo(f"⚠️ Error: {e}")

def fetch_crypto():
    start_loading()
    url = "https://alanchand.com/crypto-price"
    try:
        response = requests.get(url, timeout=10)
        response.encoding = 'utf-8'
        if response.status_code != 200:
            stop_loading()
            typer.echo("❌ Failed to fetch data.")
            return

        soup = BeautifulSoup(response.text, 'html.parser')
        table = soup.find('table', class_='CurrencyTbl cryptoTbl table table-striped xt-center mb-0 w-100 bg-white rounded-3 shadow-lg')
        if not table:
            stop_loading()
            typer.echo("❌ Crypto table not found.")
            return

        stop_loading()
        typer.echo("Live Crypto Prices:")
        typer.echo("-" * 40)
        for row in table.find_all('tr'):
            cols = row.find_all('td')
            if cols:
                name = cols[0].find('span', class_='faCurr').text.strip()
                price_tmn = persian_to_latin_digits(cols[1].find('span', class_='tmn').text.strip())
                price_usd = persian_to_latin_digits(cols[1].find('span', class_='faSymbol').text.strip())
                typer.echo(f"{name}: {price_tmn} Toman ({price_usd} USD)")
    except Exception as e:
        stop_loading()
        typer.echo(f"⚠️ Error: {e}")

def fetch_gold():
    start_loading()
    url = "https://alanchand.com"
    try:
        response = requests.get(url, timeout=10)
        response.encoding = 'utf-8'
        soup = BeautifulSoup(response.text, 'html.parser')

        def find_price(keyword):
            link = soup.find('a', string=lambda text: text and keyword in text)
            if link:
                parent = link.find_parent('div', class_='d-flex justify-content-between flex-column w-100 align-items-center')
                if parent:
                    span = parent.find('span', class_='priceSymbol')
                    if span:
                        return span.text.strip()
            return None

        stop_loading()
        typer.echo("Live Gold & Coin Prices:")
        typer.echo("-" * 40)

        found = False
        for keyword, label in GOLD_NAMES.items():
            price = find_price(keyword)
            if price:
                clean_price = persian_to_latin_digits(price)
                unit = "USD" if "انس" in keyword else "Toman"
                typer.echo(f"{label}: {clean_price} {unit}")
                found = True

        if not found:
            typer.echo("❌ No prices found.")
    except Exception as e:
        stop_loading()
        typer.echo(f"⚠️ Error: {e}")

@app.command()
def dollar():
    clear_screen()
    show_header()
    fetch_dollar()

@app.command()
def crypto():
    clear_screen()
    show_header()
    fetch_crypto()

@app.command()
def gold():
    clear_screen()
    show_header()
    fetch_gold()

@app.callback(invoke_without_command=True)
def main(ctx: typer.Context):
    if ctx.invoked_subcommand is None:
        while True:
            clear_screen()
            show_header()
            typer.echo("Select an option:")
            typer.echo("  [1] dollar   → Live fiat currency prices")
            typer.echo("  [2] crypto   → Live cryptocurrency prices")
            typer.echo("  [3] gold     → Live gold & coin prices")
            typer.echo("  [q] quit     → Exit")
            typer.echo()
            choice = input("Enter your choice (1/2/3/q): ").strip().lower()

            if choice == "1":
                clear_screen()
                show_header()
                fetch_dollar()
                input("\nPress Enter to return to menu...")
            elif choice == "2":
                clear_screen()
                show_header()
                fetch_crypto()
                input("\nPress Enter to return to menu...")
            elif choice == "3":
                clear_screen()
                show_header()
                fetch_gold()
                input("\nPress Enter to return to menu...")
            elif choice == "q":
                typer.echo("Goodbye!")
                break
            else:
                typer.echo("Invalid choice. Try again.")
                time.sleep(1)

if __name__ == "__main__":
    app()