import requests
import json
import time

now = time.time()

class Update:
    def __init__(self, message_data, chat_id, client):
        self.text = message_data.get("text", "")
        self.message_id = message_data.get("message_id", None)
        self.sender_id = message_data.get("sender_id", None)
        self.chat_id = chat_id
        self.raw = message_data
        self._client = client

    def reply(self, text: str):
        url = f"https://botapi.rubika.ir/v3/{self._client.token}/sendMessage"
        data = {
            "chat_id": self.chat_id,
            "text": text,
            "reply_to_message_id": self.message_id
        }

        resp = requests.post(url, json=data)
        jsn = resp.json()
        if resp.status_code == 200:
            if jsn.get("status") != "OK":
                return resp.text
            return jsn
        return resp.text

class Client:
	
	def __init__(self, token: str):
	    self.token = token
	    jsn = self._request("getMe", {})
	    if isinstance(jsn, dict) and jsn.get("status") != "INVALID_ACCESS":
	        self.username = jsn["data"]["bot"]["username"]
	        print(f"You have successfully logged in to the @{self.username} bot.\n")
	        print("This library was created by the channel @roboka_library")
	    else:
	        raise ValueError(f"❌ Invalid token or connection issue: {jsn}")
			
	def _request(self, method:str, data: dict, headers: dict = None):
	    url = f"https://botapi.rubika.ir/v3/{self.token}/{method}"
	    response = requests.post(url, json=data, headers=headers)
	    try:
	        jsn = response.json()
	        if response.status_code == 200 and jsn.get("status") == "OK":
	            return jsn
	        return response.text
	    except ValueError:
	        return response.text
			
	def send_text(self, chat_id: str, text: str, message_id: str = None):
	    data = {
	        "chat_id": chat_id,
	        "text": text
	    }
	    if message_id is not None:
	        data["reply_to_message_id"] = message_id
	
	    return self._request("sendMessage", data)
			
	def get_me(self):
	    jsn = self._request("getMe", {})
	    if isinstance(jsn, dict) and jsn.get("status") != "INVALID_ACCESS":
	        return jsn
	    return jsn
	
	def on_message(self, func=None, filter=None):
		if func is None:
			def decorator(f):
				self._on_message_handler = f
				self._on_message_filter = filter
				return f
			return decorator
		else:
			self._on_message_handler = func
			self._on_message_filter = filter
			return func
	
	def run(self, limit=100):
		url = f'https://botapi.rubika.ir/v3/{self.token}/getUpdates'
		offset_id = None
		last_message_id = None
		while True:
			try:
				data = {"limit": limit}
				if offset_id:
					data["offset_id"] = offset_id
				response = requests.post(url, json=data, timeout=5)
				response.raise_for_status()
				result = response.json()
				updates = result["data"]["updates"]
				offset_id = result["data"].get("next_offset_id")
				if updates:
					last_update = updates[-1]
					new_msg = last_update.get("new_message", {})
					chat_id = last_update.get("chat_id")
					message_id = new_msg.get("message_id")
					tim = new_msg["time"]
					if int(tim) >= now:
						if message_id != last_message_id:
							last_message_id = message_id
							if chat_id.startswith("b"):
								chat_type = "private"
							elif chat_id.startswith("g"):
								chat_type = "group"
							elif chat_id.startswith("c"):
								chat_type = "channel"
							else:
								chat_type = None
							if hasattr(self, "_on_message_filter") and self._on_message_filter and self._on_message_filter != chat_type:
								continue
							update_obj = Update(new_msg, chat_id, client=self)
							if self._on_message_handler:
								self._on_message_handler(update_obj)
			except requests.exceptions.RequestException as e:
				print("❗ خطای ارتباط:", e)
				continue
			except (KeyError, ValueError):
				continue
			except Exception as e:
				print("❗ خطای ناشناخته:", e)
				continue
			time.sleep(0.1)
	        
	def create_keypad(self, chat_id: str, text: str, rows: list, message_id: str = None):
	    data = {
	        "chat_id": chat_id,
	        "text": text,
	        "chat_keypad_type": "New",
	        "chat_keypad": {
	            "rows": rows,
	            "resize_keyboard": True,
	            "on_time_keyboard": False
	        }
	    }
	    if message_id is not None:
	        data["reply_to_message_id"] = message_id
	
	    return self._request("sendMessage", data)

	def send_poll(self, chat_id:str, question:str, options:list):
		data = {
		    "chat_id": chat_id,
		    "question": question,
		    "options": options
		}

		return self._request("sendPoll", data)
	
	def send_location(self, chat_id: str, latitude: float, longitude: float, message_id: str = None):
	    data = {
	        "chat_id": chat_id,
	        "latitude": latitude,
	        "longitude": longitude
	    }
	
	    if message_id is not None:
	        data["reply_to_message_id"] = message_id
	
	    return self._request("sendLocation", data)

	def edit_text(self, chat_id: str, text: str, message_id: int):
	    data = {
	        "chat_id": chat_id,
	        "message_id": message_id,
	        "text": text
	    }
	    return self._request("editMessageText", data)

	def send_contact(self, chat_id: str, first_name: str, last_name: str, phone_number: str, message_id: int = None):
	    data = {
	        "chat_id": chat_id,
	        "first_name": first_name,
	        "last_name": last_name,
	        "phone_number": phone_number
	    }
	
	    if message_id is not None:
	        data["reply_to_message_id"] = message_id
	
	    return self._request("sendContact", data)

	def get_chat(self, chat_id: str):
	    data = {
	        "chat_id": chat_id
	    }
	    return self._request("getChat", data)

	def forward_message(self, from_chat_id: str, message_id: int, to_chat_id: str):
	    data = {
	        "from_chat_id": from_chat_id,
	        "message_id": message_id,
	        "to_chat_id": to_chat_id
	    }
	    return self._request("forwardMessage", data)

	def delete_message(self, chat_id: str, message_id: int):
	    data = {
	        "chat_id": chat_id,
	        "message_id": message_id
	    }
	    return self._request("deleteMessage", data)

	def create_inline_keypad(self, chat_id: str, text: str, rows: list, message_id: str = None):
	    data = {
	        "chat_id": chat_id,
	        "text": text,
	        "inline_keypad": {
	            "rows": rows
	        }
	    }
	
	    if message_id is not None:
	        data["reply_to_message_id"] = message_id
	
	    return self._request("sendMessage", data)

	def get_upload_url(self, type: str):
	    data = {
	        "type": type
	    }
	    response = self._request("requestSendFile", data)
	    return response["data"]["upload_url"]

	def get_file_id(self, url: str, file_name: str, file_path: str) -> str:
	    with open(file_path, "rb") as f:
	        files = {
	            "file": (file_name, f, "application/octet-stream")
	        }
	        response = requests.post(url, files=files, verify=False)
	        response.raise_for_status()
	
	    return response.json()["data"]["file_id"]
	
	def send_file(self, chat_id: str, text: str, file_name: str, file_path: str, file_type: str, message_id: str = None):
	    upload_url = self.get_upload_url(file_type)
	    file_id = self.get_file_id(upload_url, file_name, file_path)
	
	    data = {
	        "chat_id": chat_id,
	        "file_id": file_id,
	        "text": text
	    }
	
	    if message_id is not None:
	        data["reply_to_message_id"] = message_id
	
	    return self._request("sendFile", data)

	def send_image(self, chat_id: str, text: str, file_name: str, file_path: str, message_id: str = None):
	    upload_url = self.get_upload_url("Image")
	    file_id = self.get_file_id(upload_url, file_name, file_path)
	
	    data = {
	        "chat_id": chat_id,
	        "file_id": file_id,
	        "text": text
	    }
	
	    if message_id is not None:
	        data["reply_to_message_id"] = message_id
	
	    return self._request("sendFile", data)

	def send_video(self, chat_id: str, text: str, file_name: str, file_path: str, message_id: str = None):
	    upload_url = self.get_upload_url("Video")
	    file_id = self.get_file_id(upload_url, file_name, file_path)
	
	    data = {
	        "chat_id": chat_id,
	        "file_id": file_id,
	        "text": text
	    }
	
	    if message_id is not None:
	        data["reply_to_message_id"] = message_id
	
	    return self._request("sendFile", data)

	def get_updates(self, limit: str, offset_id: str = None):
	    data = {
	        "limit": limit
	    }
	
	    if offset_id is not None:
	        data["offset_id"] = offset_id
	
	    return self._request("getUpdates", data)

	def edit_keypad(self, chat_id: str, rows: list):
	    data = {
	        "chat_id": chat_id,
	        "chat_keypad_type": "New",
	        "chat_keypad": {
	            "rows": rows,
	            "resize_keyboard": True,
	            "on_time_keyboard": False
	        }
	    }
	
	    return self._request("editChatKeypad", data)

	def set_commands(self, commands: list):
	    data = {
	        "bot_commands": commands
	    }
	
	    return self._request("setCommands", data)

	def set_webhook(self, url: str, type: str):
	    data = {
	        "url": url,
	        "type": type
	    }
	
	    return self._request("updateBotEndpoints", data)