from __future__ import annotations

import click
import cloup

from fluxion import __version__
from fluxion._config import cli_ctx_settings, console
from fluxion.cli.cfg.group import cfg
from fluxion.cli.checkhealth.commands import checkhealth
from fluxion.cli.default_group import DefaultGroup
from fluxion.cli.init.commands import init
from fluxion.cli.plugins.commands import plugins
from fluxion.cli.render.commands import render
from fluxion.constants import EPILOG


def show_splash(ctx: click.Context, param: click.Option, value: str | None) -> None:
    """When giving a value by console, show an initial message with the fluxion
    version before executing any other command: ``fluxion Community vA.B.C``.

    Parameters
    ----------
    ctx
        The Click context.
    param
        A Click option.
    value
        A string value given by console, or None.
    """
    if value:
        console.print(f"fluxion Community [green]v{__version__}[/green]\n")


def print_version_and_exit(
    ctx: click.Context, param: click.Option, value: str | None
) -> None:
    """Same as :func:`show_splash`, but also exit when giving a value by
    console.

    Parameters
    ----------
    ctx
        The Click context.
    param
        A Click option.
    value
        A string value given by console, or None.
    """
    show_splash(ctx, param, value)
    if value:
        ctx.exit()


@cloup.group(
    context_settings=cli_ctx_settings,
    cls=DefaultGroup,
    default="render",
    no_args_is_help=True,
    help="Animation engine for explanatory math videos.",
    epilog="See 'fluxion <command>' to read about a specific subcommand.\n\n"
    "Note: the subcommand 'fluxion render' is called if no other subcommand "
    "is specified. Run 'fluxion render --help' if you would like to know what the "
    f"'-ql' or '-p' flags do, for example.\n\n{EPILOG}",
)
@cloup.option(
    "--version",
    is_flag=True,
    help="Show version and exit.",
    callback=print_version_and_exit,
    is_eager=True,
    expose_value=False,
)
@click.option(
    "--show-splash/--hide-splash",
    is_flag=True,
    default=True,
    help="Print splash message with version information.",
    callback=show_splash,
    is_eager=True,
    expose_value=False,
)
@cloup.pass_context
def main(ctx: click.Context) -> None:
    """The entry point for fluxion.

    Parameters
    ----------
    ctx
        The Click context.
    """
    pass


main.add_command(checkhealth)
main.add_command(cfg)
main.add_command(plugins)
main.add_command(init)
main.add_command(render)

if __name__ == "__main__":
    main()
