from fluxion import *
import numpy as np

class CosmicWaveHD(ThreeDScene):
    def construct(self):
        # --- Background & Camera ---
        self.renderer.background_color = BLACK  # Updated way to set background
        self.set_camera_orientation(phi=75 * DEGREES, theta=30 * DEGREES, distance=15)

        # --- Title Sequence ---
        title = Text("The Fluxion Engine", font_size=72, gradient=(BLUE, PURPLE)).to_edge(UP)
        subtitle = Text(
            "High-Definition 3D Surface Simulation",
            font_size=36,
            gradient=(YELLOW, ORANGE)
        ).next_to(title, DOWN)
        self.add_fixed_in_frame_mobjects(title, subtitle)
        self.play(Write(title), FadeIn(subtitle))
        self.wait(1)

        # --- 3D Axes ---
        axes = ThreeDAxes(
            x_range=[-5, 5],
            y_range=[-5, 5],
            z_range=[-3, 3],
            x_length=10,
            y_length=10,
            z_length=6
        )
        self.play(Create(axes))

        # --- High-Resolution 3D Surface ---
        def cosmic_surface(u, v):
            r = np.sqrt(u**2 + v**2)
            z = np.sin(4*r + 2*np.sin(u)) * np.exp(-0.2*r) + 0.2*np.sin(6*v)
            return np.array([u, v, z])

        surface = Surface(
            cosmic_surface,
            u_range=[-4, 4],
            v_range=[-4, 4],
            resolution=(64, 64),
            fill_opacity=0.9,
            stroke_color=WHITE,
            stroke_width=0.3
        )

        surface.set_fill_by_value(
            lambda p: color_gradient([PURPLE, BLUE, CYAN, GREEN, YELLOW], (p[2]+1)/2),
            axis=2
        )

        self.play(Create(surface), run_time=4)
        self.begin_ambient_camera_rotation(rate=0.3)

        # --- Dynamic "Surfer" Sphere ---
        tracker = ValueTracker(0)
        sphere = Sphere(radius=0.15, color=YELLOW)

        def update_sphere(mob):
            t = tracker.get_value()
            x = 3 * np.cos(t)
            y = 3 * np.sin(t)
            r = np.sqrt(x**2 + y**2)
            z = np.sin(4*r + 2*np.sin(x)) * np.exp(-0.2*r) + 0.2*np.sin(6*y)
            mob.move_to(axes.c2p(x, y, z))

        sphere.add_updater(update_sphere)
        self.add(sphere)

        self.play(
            tracker.animate.set_value(4 * TAU),
            run_time=8,
            rate_func=smooth
        )

        # --- Floating Particles ---
        num_particles = 80
        particles = VGroup(*[
            Sphere(radius=0.05, color=WHITE).move_to([
                np.random.uniform(-5,5),
                np.random.uniform(-5,5),
                np.random.uniform(-3,3)
            ])
            for _ in range(num_particles)
        ])

        for p in particles:
            p.add_updater(lambda m: m.shift([0,0,-0.002]) 
                          if m.get_center()[2] > -3 
                          else m.move_to([
                              np.random.uniform(-5,5),
                              np.random.uniform(-5,5),
                              3
                          ]))
        self.add(particles)

        # --- Camera Zoom & Rotate for Finale ---
        self.move_camera(phi=85*DEGREES, theta=-45*DEGREES, distance=18, run_time=3)
        self.wait(1.5)

        # --- Finale Text ---
        finale = Text("Simulation Complete", font_size=64, gradient=(ORANGE, PINK)).to_edge(DOWN)
        self.add_fixed_in_frame_mobjects(finale)
        self.play(Write(finale))

        # --- Fade Out Everything ---
        self.play(
            FadeOut(title),
            FadeOut(subtitle),
            FadeOut(axes),
            FadeOut(surface),
            FadeOut(sphere),
            FadeOut(particles)
        )
        self.wait()
