"""
SimaCode Tool System

This module provides a comprehensive tool framework for the SimaCode AI assistant,
enabling secure and controlled execution of various operations including file
operations, system commands, and custom tools.

The tool system is built around a plugin architecture with:
- Base tool abstractions
- Input validation and output formatting
- Permission-based access control
- Tool registration and discovery
- Execution monitoring and logging
"""

from .base import Tool, ToolResult, ToolInput, ToolRegistry, ToolResultType, execute_tool
from .bash import BashTool
from .file_read import FileReadTool
from .file_write import FileWriteTool
# Lazy import for UniversalOCRTool to avoid slow startup
# from .universal_ocr import UniversalOCRTool
# EmailSendTool has been migrated to MCP server: tools/mcp_smtp_send_email.py
# from .email_send import EmailSendTool
from .smc_content_coder import MCPContentExtraction, ContentForwardURL

def get_universal_ocr_tool():
    """Lazy import of UniversalOCRTool to avoid slow startup"""
    from .universal_ocr import UniversalOCRTool
    return UniversalOCRTool

def initialize_tools_with_session_manager(session_manager=None, config=None):
    """
    Initialize or re-register tools with SessionManager dependency.

    Args:
        session_manager: SessionManager instance to inject into tools
        config: Config instance to determine which tools to enable
    """
    # Clear existing tools to re-register with session manager
    ToolRegistry.clear()

    # Get tools configuration
    tools_config = config.tools if config else None

    # Register tools based on configuration
    tools = []

    # Register BashTool if enabled
    if tools_config is None or tools_config.bash:
        tools.append(BashTool(session_manager=session_manager))

    # Register FileReadTool if enabled
    if tools_config is None or tools_config.file_read:
        tools.append(FileReadTool(session_manager=session_manager))

    # Register FileWriteTool if enabled
    if tools_config is None or tools_config.file_write:
        tools.append(FileWriteTool(session_manager=session_manager))

    # Register MCPContentExtraction if enabled
    if tools_config is None or tools_config.mcp_content_extraction:
        tools.append(MCPContentExtraction(session_manager=session_manager))

    # Register ContentForwardURL if enabled
    if tools_config is None or tools_config.content_forward_url:
        tools.append(ContentForwardURL(session_manager=session_manager))

    # Try to register UniversalOCRTool if enabled and available (using lazy import)
    if tools_config is None or tools_config.universal_ocr:
        try:
            UniversalOCRTool = get_universal_ocr_tool()
            ocr_tool = UniversalOCRTool(session_manager=session_manager)
            tools.append(ocr_tool)
        except Exception:
            # OCR tool may have additional dependencies
            pass

    # Register all tools
    for tool in tools:
        ToolRegistry.register(tool)

    return tools


__all__ = [
    "Tool",
    "ToolResult",
    "ToolInput",
    "ToolRegistry",
    "ToolResultType",
    "execute_tool",
    "initialize_tools_with_session_manager",
    "get_universal_ocr_tool",
    "BashTool",
    "FileReadTool",
    "FileWriteTool",
    # "UniversalOCRTool",  # Use get_universal_ocr_tool() for lazy import
    # "EmailSendTool",  # Migrated to MCP server: tools/mcp_smtp_send_email.py
    "MCPContentExtraction",
    "ContentForwardURL",
]