from abc import ABC, abstractmethod
from typing import Optional, Any, Dict


from oldaplib.src.helpers.context import DEFAULT_CONTEXT
from oldaplib.src.userdataclass import UserData
from oldaplib.src.xsd.iri import Iri
from oldaplib.src.xsd.xsd_qname import Xsd_QName
from oldaplib.src.xsd.xsd_ncname import Xsd_NCName
from oldaplib.src.enums.sparql_result_format import SparqlResultFormat


class IConnection(ABC):
    """
    Defines an abstract base class for a connection interface.

    This class serves as a blueprint for defining connection objects
    that interact with a specific API or data source. Implementations
    of this class must provide concrete definitions for a variety
    of methods that perform operations such as querying, updating, and
    transaction management. It also provides a set of properties for
    managing user data, context, and authentication tokens.

    :ivar context_name: Getter for the name of the context associated with the
                        connection.
    :type context_name: str
    :ivar token: Authentication token used for API interactions. Its value may
                 be set or generated by the specific implementation.
    :type token: str
    :ivar userdata: UserData object containing user-related information,
                    or ``None`` if no user data is set.
    :type userdata: UserData | None
    """
    _context_name: str
    _userdata: UserData | None
    _token: str | None
    _transaction_url: Optional[str]

    @abstractmethod
    def __init__(self, context_name: Optional[str] = DEFAULT_CONTEXT):
        """
        Represents an abstract class with initialization for a context-specific
        configuration. This class is designed to set up and handle a context
        name and other related internal data.

        :param context_name: The name of the context to be initialized. Defaults
            to `DEFAULT_CONTEXT`.
        :type context_name: Optional[str]
        """
        self._context_name = context_name
        self._userdata = None
        self._token = None
        self._transaction_url = None

    @property
    def userdata(self) -> UserData | None:
        return self._userdata

    @property
    def userid(self) -> Xsd_NCName:
        return self._userdata.userId

    @property
    def userIri(self) -> Iri:
        return self._userdata.userIri

    @property
    def login(self) -> bool:
        return self._userIri is not None

    @property
    def context_name(self) -> str:
        """Getter for the context name"""
        return self._context_name

    @property
    def token(self) -> str:
        return self._token

    @abstractmethod
    def clear_graph(self, graph_iri: Xsd_QName) -> None:
        pass

    @abstractmethod
    def clear_repo(self) -> None:
        pass

    @abstractmethod
    def upload_turtle(self, filename: str, graphname: Optional[str] = None) -> None:
        pass

    @abstractmethod
    def query(self, query: str, format: SparqlResultFormat = SparqlResultFormat.JSON) -> Any:
        pass

    @abstractmethod
    def update_query(self, query: str) -> Dict[str, str]:
        pass

    @abstractmethod
    def transaction_start(self) -> None:
        pass

    @abstractmethod
    def transaction_query(self, query: str, result_format: SparqlResultFormat = SparqlResultFormat.JSON) -> Any:
        pass

    @abstractmethod
    def transaction_update(self, query: str) -> None:
        pass

    @abstractmethod
    def transaction_commit(self) -> None:
        pass

    @abstractmethod
    def transaction_abort(self) -> None:
        pass

    @abstractmethod
    def in_transaction(self) -> bool:
        pass

