# src/veriskgo/sqs.py
import json
import boto3
from typing import Optional, Dict, Any
from .config import get_cfg
from botocore.exceptions import BotoCoreError, ClientError


class _VeriskGoSQS:
    """Auto-initialized SQS handler with self-healing."""

    def __init__(self):
        self.client = None
        self.queue_url = None
        self.sqs_enabled = False
        self._init_once = False

        self._auto_initialize()

    # ---------------------------------------------------------
    # AUTO INITIALIZATION
    # ---------------------------------------------------------
    def _auto_initialize(self):
        """Automatically initialize SQS client on import."""
        if self._init_once:
            return

        cfg = get_cfg()
        self.queue_url = cfg.get("aws_sqs_url")

        if not self.queue_url:
            print("[veriskgo] No SQS URL found → SQS disabled.")
            return

        try:
            session = boto3.Session(
                profile_name=cfg.get("aws_profile"),
                region_name=cfg.get("aws_region", "us-east-1")
            )

            self.client = session.client("sqs")

            # Connectivity check
            self.client.get_queue_attributes(
                QueueUrl=self.queue_url,
                AttributeNames=["QueueArn"]
            )

            self.sqs_enabled = True
            print(f"[veriskgo] SQS connected → {self.queue_url}")

        except Exception as e:
            print("[veriskgo] ERROR connecting to SQS:", e)
            self.sqs_enabled = False

        self._init_once = True

    # ---------------------------------------------------------
    # PUBLIC SEND METHOD
    # ---------------------------------------------------------
    def send(self, message: Optional[Dict[str, Any]]) -> bool:
        """Send a message to SQS with auto-retry initialization."""

        if not message:
            print("[veriskgo] Empty message, skipping SQS send.")
            return False

        if not self.sqs_enabled or self.client is None:
            print("[veriskgo] SQS not ready → retrying initialization…")
            self._auto_initialize()

        if not self.sqs_enabled or self.client is None:
            print("[veriskgo] SQS unavailable → skipping send.")
            return False

        try:
            # Tell Pylance that client cannot be None here
            self.client.send_message(        # type: ignore[union-attr]
                QueueUrl=self.queue_url,
                MessageBody=json.dumps(message)
            )
            print("[veriskgo] Trace bundle sent to SQS.")
            return True

        except Exception as e:
            print(f"[veriskgo] ERROR sending to SQS: {e}")
            return False

# ---------------------------------------------------------
# SINGLETON INSTANCE — used automatically
# ---------------------------------------------------------
_sqs_instance = _VeriskGoSQS()

# Backward compatible old API:
def init_sqs() -> bool:
    """Backwards compatibility — simply return the state."""
    return _sqs_instance.sqs_enabled


def send_to_sqs(trace_bundle: Optional[Dict[str, Any]]) -> bool:
    """Main public method."""
    return _sqs_instance.send(trace_bundle)
