# -*- coding: utf-8 -*-


# Import necessary libraries
import numpy as np
import pandas as pd
from sklearn.model_selection import train_test_split, cross_val_score
from sklearn.metrics import accuracy_score, roc_auc_score
from sklearn.preprocessing import StandardScaler
from sklearn.pipeline import Pipeline
from sklearn.decomposition import PCA
from sklearn.cross_decomposition import PLSRegression
from sklearn.model_selection import permutation_test_score
import matplotlib.pyplot as plt
from sklearn.metrics import r2_score
from sklearn.utils import shuffle
import plotly.express as px
import plotly.graph_objects as go
from plotly.subplots import make_subplots
import plotly.offline as pyo
from .cross_validation import CrossValidation
#import plotting
from sklearn.linear_model import LinearRegression
from sklearn.metrics import mean_squared_error

import warnings
warnings.filterwarnings('ignore')

from .scaler import Scaler

import os

__auther__ = "aeiwz"


class opls_da:
    
    
        
    # Import necessary libraries
    import numpy as np
    import pandas as pd
    from sklearn.model_selection import train_test_split, cross_val_score
    from sklearn.metrics import accuracy_score, roc_auc_score
    from sklearn.preprocessing import StandardScaler
    from sklearn.pipeline import Pipeline
    from sklearn.decomposition import PCA
    from sklearn.cross_decomposition import PLSRegression
    from sklearn.model_selection import permutation_test_score
    import matplotlib.pyplot as plt
    from sklearn.metrics import r2_score
    from sklearn.utils import shuffle
    import plotly.express as px
    import plotly.graph_objects as go
    from plotly.subplots import make_subplots
    import plotly.offline as pyo
    from .cross_validation import CrossValidation
    #import plotting
    from sklearn.linear_model import LinearRegression
    from sklearn.metrics import mean_squared_error

    import warnings
    warnings.filterwarnings('ignore')


    import os




    '''
    OPLS-DA model
    
    Parameters
    ----------
    X : array-like, shape (n_samples, n_features)
        Training data, where n_samples is the number of samples and n_features is the number of features.
    y : array-like, shape (n_samples,)
        Target data, where n_samples is the number of samples.
    n_components : int, default=2
        Number of components to keep.
    scale : str, default='par'
        Method of scaling. 'par' for pareto scaling, 'mc' for mean centering, 'uv' for unitvarian scaling.
    cv : int, default=5
        Number of cross-validation folds.
    n_permutations : int, default=1000
        Number of permutations for permutation test.
    random_state : int, default=42
        Random state for permutation test.
    kfold : int, default=3
        Number of cross-validation folds.


    Examples:
    ----------
    
        ## Import package into python


        from metbit import opls_da, pca
        import pandas as pd
        import numpy as np


        ## Load dataset
        For example dataset are generated by random


        data = pd.DataFrame(np.random.rand(500, 50000))
        class_ = pd.Series(np.random.choice(['A', 'B'], 500), name='Group')

        datasets = pd.concat([class_, data], axis=1)


        # Assign X and target
        X = datasets.iloc[:, 2:]
        y = datasets['Group']
        time = datasets['Time point']
        features_name = list(X.columns.astype(float))

        ## Perform OPLS-DA model


        opls_da_mod = opls_da(X=X, y=y,features_name=features_name, n_components=2, scaling_method='pareto', kfold=3, estimator='opls', random_state=42):
                
        opls_da.fit()

        opls.permutation_test(n_permutataion=1000,cv=3, n_jobs=-1, verbose=10)

        opls_da.vip_scores()


        ## Isualiseation of OPLs-DA model



        opls_da_model.plot_oplsda_scores()

        opls_da_model.vip_plot()

        opls_da_model.plot_hist()

        opls_da_model.plot_s_scores()

        opls_da_model.plot_loading()


        

    '''
    

        
    
    def __init__(self, X, y, features_name=None, n_components=2, scaling_method='pareto', kfold=3, estimator='opls', random_state=42, auto_ncomp=True):
        

        self.auto_ncomp = auto_ncomp
        #check X and y must be dataframe or array
        if not isinstance(X, (pd.DataFrame, np.ndarray)):
            raise ValueError('X must be a dataframe or array')
        if not isinstance(y, (pd.Series, np.ndarray, list)):
            raise ValueError('y must be a series or array or list')
        if X.shape[0] != y.shape[0]:
            raise ValueError('X and y must have the same number of samples')
        if not isinstance(n_components, int):
            raise ValueError('n_components must be an integer')
        if not isinstance(scaling_method, str):
            raise ValueError('scaling method must be a string')
        if not isinstance(kfold, int):
            raise ValueError('kfold must be an integer')
        if not isinstance(estimator, str):
            raise ValueError('estimator must be a string')
        if not isinstance(random_state, int):
            raise ValueError('random_state must be an integer')
        if features_name is not None:
            if not isinstance(features_name, (pd.Series, np.ndarray, list)):
                raise ValueError('features_name must be a series, list or 1D array')
            if len(features_name) != X.shape[1]:
                raise ValueError('features_name must have the same number of features as X')
            



        self.features_name = features_name
        if features_name is None:
            if isinstance(X, pd.DataFrame):
                self.features_name = X.columns
            else:
                self.features_name = np.arange(X.shape[1])
        else:
            self.features_name = features_name
            
        if isinstance(X, pd.DataFrame):
            self.X = X.values
        else:
            self.X = X 

        #fill nan with 0
        self.X = np.nan_to_num(self.X)
                     
        self.X = X
        self.y = y
        self.features_name = features_name
        self.n_components = n_components
        self.scaling_method = scaling_method
        self.kfold = kfold
        self.estimator = estimator
        self.random_state = random_state
        self.auto_ncomp = auto_ncomp



        
    def fit(self):
        
        import time
        
        T1 = time.time()
        
        X = self.X
        y = self.y
        features_name = self.features_name
        n_components = self.n_components
        scaling_method = self.scaling_method

        random_state = self.random_state
        kfold = self.kfold
        estimator = self.estimator
        auto_ncomp = self.auto_ncomp
        
        if scaling_method == 'pareto':
            Scale_power = 0.5
        elif scaling_method == 'mean':
            Scale_power = 0
        elif scaling_method == 'uv':
            Scale_power = 1
        elif scaling_method == 'minmax':
            Scale_power = 0
            
        self.scaling_method = scaling_method
            
            
        # Create a pipeline with data preprocessing and OPLS-DA model
        pipeline = Pipeline([
                                ('scale', Scaler(scale_power=Scale_power)),
                                ('oplsda', PLSRegression(n_components=n_components)),
                                ('opls', CrossValidation(kfold=kfold, estimator=estimator, scaler=scaling_method))
                            ])

        self.pipeline = pipeline

        oplsda = pipeline.named_steps['oplsda']
        cv_model = pipeline.named_steps['opls']
        cv_model.fit(X.values, y)

        if auto_ncomp == False:
            cv_model.reset_optimal_num_component(n_components)

        else:
            pass

        oplsda.fit(X, pd.Categorical(y).codes)
        
        if self.estimator == 'opls':
            s_df_scores_ = pd.DataFrame({'correlation': cv_model.correlation,'covariance': cv_model.covariance}, index=features_name)
            df_opls_scores = pd.DataFrame({'t_scores': cv_model.scores, 't_ortho': cv_model.orthogonal_score, 't_pred': cv_model.predictive_score, 'Group': y})
            self.s_df_scores_ = s_df_scores_
            self.df_opls_scores = df_opls_scores        
            self.oplsda = oplsda

            R2Xcorr = cv_model.R2Xcorr
            R2y = cv_model.R2y
            q2 = cv_model.q2

            self.R2Xcorr = R2Xcorr
            self.R2y = R2y
            self.q2 = q2
        else:
            pass


        self.cv_model = cv_model

        T2 = time.time()
        
        duration = T2 - T1

        if self.estimator == 'opls':
    
            summary_model_oplsda = f'''
            Comparison of {y.unique()[0]} and {y.unique()[1]}
            Sample size: {y.unique()[0]}: {y.value_counts()[0]}
                         {y.unique()[1]}: {y.value_counts()[1]}
            Number of features: {X.shape[1]}
            Number of components: {n_components}
            Method of scaling: {self.scaling_method}
            OPLS-DA model is fitted in {duration} seconds
            R2Xcorr: {R2Xcorr}
            R2y: {R2y}
            Q2: {q2}        
            '''
            summary_model = summary_model_oplsda
        else:
            summary_model_plsda = f'''
            Comparison of {y.unique()[0]} and {y.unique()[1]}
            Sample size: {y.unique()[0]}: {y.value_counts()[0]}
                         {y.unique()[1]}: {y.value_counts()[1]}
            Number of features: {X.shape[1]}
            Number of components: {n_components}
            Method of scaling: {self.scaling_method}
            OPLS-DA model is fitted in {duration} seconds
            R2Xcorr: -
            R2y: - 
            Q2: -
            '''
            summary_model = summary_model_plsda

        return print(summary_model)


    def get_oplsda_scores(self):
            
        '''
        Get OPLS-DA scores
        '''
        df_opls_scores = self.df_opls_scores
        return df_opls_scores
    
    def get_s_scores(self):
                
        '''
        Get S scores
        '''
        s_df_scores_ = self.s_df_scores_
        return s_df_scores_

    def get_oplsda_model(self):
        
        '''
        Get OPLS-DA model
        '''
        oplsda = self.oplsda
        return oplsda

    def get_cv_model(self):

        '''
        Get cross-validation model
        '''
        cv_model = self.cv_model
        return cv_model
    


    def permutation_test(self, n_permutations=500, cv=3, n_jobs=-1, verbose=10):


        from sklearn.pipeline import Pipeline
        import time
        
        T1 = time.time()
        
        self.cv = cv
        self.n_permutations = n_permutations
        oplsda = self.oplsda
        X = self.X
        y = self.y
        pipeline = self.pipeline
        randomstate = self.random_state
        self.n_jobs = n_jobs
        self.verbose = verbose
        
        oplsda.fit(X, pd.Categorical(y).codes)

        # Permutation test to assess the significance of the model
        acc_score, permutation_scores, p_value = permutation_test_score(
                                                                        pipeline.named_steps['oplsda'], 
                                                                        X, pd.Categorical(y).codes, cv=3, 
                                                                        n_permutations=n_permutations, n_jobs=n_jobs, 
                                                                        random_state=randomstate, verbose=verbose)

        T2 = time.time()
        duration = T2 - T1
        if duration > 60:
            duration = duration/60
            unit = 'minutes'
        else:
            duration = duration
            unit = 'seconds'
            
            
        
        self.acc_score = acc_score
        self.permutation_scores = permutation_scores
        self.p_value = p_value
        
        return print(f'Permutation test is performed in {duration} {unit}')
        
    def get_permutation_scores(self):
        
        '''
        Get permutation scores
        '''
        permutation_scores = self.permutation_scores
        return permutation_scores
    
    def vip_scores(self, model=None, features_name = None):
        
        '''
        Get VIP score

        Parameters
        ----------
        model : object, default=None
            OPLS-DA model.
        features_name : array-like, shape (n_features,), default=None
            Name of features.

        '''
        
        if model is None:   
            model = self.oplsda
        else:
            model = model
            
        
        features_name = self.features_name



        t = model.x_scores_
        w = model.x_weights_
        q = model.y_loadings_
        p, h = w.shape
        vips = np.zeros((p,))
        s = np.diag(t.T @ t @ q.T @ q).reshape(h, -1)
        total_s = np.sum(s)
        for i in range(p):
            weight = np.array([ (w[i,j] / np.linalg.norm(w[:,j]))**2 for j in range(h) ])
            vips[i] = np.sqrt(p*(s.T @ weight)/total_s)
       
        if features_name is not None:
            vips = pd.DataFrame(vips, columns = ['VIP'])
            vips['Features'] = features_name
        else:
            vips = pd.DataFrame(vips, columns = ['VIP'])
            vips['Features'] = vips.index

        self.vips = vips

        return

    def get_vip_scores(self, filter_=False, threshold=1):
            
        '''
        Get VIP score

        Parameters
        ----------
        filter_ : bool, default=False
            If True, filter VIP score based on threshold.
        threshold : int, default=1
            Threshold of VIP score.

        '''

        vips = self.vips
        if filter_ == True:
            vips = vips[vips['VIP'] >= threshold]

        else:
            vips = vips

        return vips



    def vip_plot(self, x_range = 9, threshold = 2, marker_size = 12, 
                 fig_width = 1000, fig_height = 500, 
                 filter_ = False, vip_transform = True,
                 font_size = 20, title_font_size = 20,
                 xaxis_direction = 'reversed'):

        '''
        Plot VIP score

        Parameters
        ----------
        x_range : int, default=9
            Range of x-axis.
        threshold : int, default=2
            Threshold of VIP score.
        size : int, default=12
            Size of the marker.
        width : int, default=1000
            Width of the figure.
        height : int, default=500
            Height of the figure.
        filter_ : bool, default=False
            If True, filter VIP score based on threshold.

        '''

        s_df_scores_ = self.s_df_scores_

        
        # add scatter plot of VIP score
        import plotly.express as px
        vips = self.vips

        oplsda = self.oplsda
        coef_ = oplsda.coef_

        if vip_transform == True:
            vips['VIP'] = vips['VIP'] * np.sign(np.ravel(coef_))
            vips['threshold'] = np.where(vips['VIP'] >= threshold, f"High in {self.y.unique()[1]}", 
                                np.where(vips['VIP'] <= -threshold, f"High in {self.y.unique()[0]}", 
                                "Under cut off"))
        else:
            #add threshold column to define cutoff for VIP score if >= treschold then 1 else 0
            vips['threshold'] = np.where(vips['VIP'] >= threshold, "Pass", "Under cut off")

        if filter_ == True:
            vips = vips[vips['VIP'] >= threshold]
        
        vips['Correlation'] = s_df_scores_['correlation']
        vips['Covariance'] = s_df_scores_['covariance']

        fig = px.scatter(vips, x='Features', y='VIP',
                         color='threshold',
                         color_discrete_map={
                             'Pass':'#FF7961',
                             'Under cut off':'#ECECEC',
                             f'High in {self.y.unique()[1]}':'#B80F0A',
                             f'High in {self.y.unique()[0]}':'#03045E'}, 
                         height=fig_height, width=fig_width, 
                         title='VIP score',
                         hover_data={'Features':True, 'VIP':True, 'threshold':True})
        
              

        fig.update_traces(marker=dict(size=marker_size))
        fig.update_xaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_yaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_xaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_yaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')

        # tickformat of y-axis to 2 decimal places and x-axis to 3 decimal places
        fig.update_yaxes(tickformat=".2f")
        fig.update_xaxes(tickformat=".3f")

        
        fig.update_layout(
            title={
                'y':0.95,
                'x':0.5,
                'xanchor': 'center',
                'yanchor': 'top'},
            font=dict(size=title_font_size))

        # reverse the x-axis
        fig.update_xaxes(autorange=xaxis_direction)
        

        if vip_transform == True:
                fig.add_shape(type="line",
                                x0=0, y0=threshold, x1=x_range, y1=threshold,
                                line=dict(color="red",width=2))

                fig.add_shape(type="line",
                                x0=0, y0=-threshold, x1=x_range, y1=-threshold,
                                line=dict(color="blue",width=2))

        else:
            fig.add_shape(type="line",
                            x0=0, y0=threshold, x1=x_range, y1=threshold,
                            line=dict(color="red",width=2))


        
                    
        fig.update_layout(showlegend=False)
        
        return fig





    def plot_oplsda_scores(self,x_ = 't_scores', y_ = 't_ortho', color_ = None, color_dict = None, 
                           symbol_ = None, symbol_dict = None, 
                           fig_height = 900, fig_width = 1300,
                           marker_size = 35, marker_opacity = 0.7, 
                           font_size = 20, title_font_size = 21,
                           legend_name = ['Group', 'Time point'],
                           individual_ellipse = True):

        '''
        Plot OPLS-DA scores plot

        Parameters
        ----------
        color_ : array-like, shape (n_samples,), default=None
            color_ of the group. If None, color_ will be based on the group in y.
        color_dict : dict, default=None
            Dictionary of color_ for the group. If None, color_ will be based on the group in y.
        symbol_ : array-like, shape (n_samples,), default=None
            symbol_ of the group. If None, symbol_ will be based on the group in y.
        symbol_dict : dict, default=None
            Dictionary of symbol_ for the group. If None, symbol_ will be based on the group in y.
        fig_height : int, default=900
            Height of the figure.
        fig_width : int, default=1300
            Width of the figure.
        marker_size : int, default=35
            Size of the marker.
        marker_opacity : float, default=0.7
            Opacity of the marker.

        '''
        if color_ is not None:
            if len(color_) != len(self.y):
                raise ValueError('color_ must have the same number of samples as y')
            else:
                color_ = color_
            
        
        #Visualise
        #check symbol_ dimension must be equal to y
        if symbol_ is not None:
            if len(symbol_) != len(self.y):
                raise ValueError('symbol_ must have the same number of samples as y')

        #check symbol_dict must be a dictionary
        if symbol_dict is not None:
            if not isinstance(symbol_dict, dict):
                raise ValueError('symbol_dict must be a dictionary')
        else:
            symbol_dict = None

        df_opls_scores = self.df_opls_scores

        if color_ is not None:
            df_opls_scores['Group'] = color_
        

        if symbol_ is not None:
            df_opls_scores['symbol'] = symbol_
            
        df_opls_scores['Index'] = df_opls_scores.index
        

        #If user not input color_dict then get unique of label and create color_dict
        if color_dict is not None:
            color_dict = color_dict
        else:
            import plotly.colors as plotly_colour
            name_color_set = ['Plotly', 'D3', 'G10', 'T10', 'Alphabet', 'Dark24', 'Light24', 'Set1', 'Pastel1', 
                        'Dark2', 'Set2', 'Pastel2', 'Set3', 'Antique', 'Safe', 'Bold', 'Pastel', 
                        'Vivid', 'Prism']

            palette = []
            for name in name_color_set:
                palette += getattr(plotly_colour.qualitative, name) # This is a list of colors


            color_dict = {label: palette[i] for i, label in enumerate(df_opls_scores['Group'].unique())}
            

        # Create a new color dictionary to match with unique labels
        color_dict_2 = {}
        for group in df_opls_scores['Group'].unique():
            if group in color_dict:
                color_dict_2[group] = color_dict[group]
            else:
                # Assign a default color if the label is not found in the original color_dict

                import plotly.colors as plotly_colour
                name_color_set = ['Plotly', 'D3', 'G10', 'T10', 'Alphabet', 'Dark24', 'Light24', 
                                    'Set1', 'Pastel1', 'Dark2', 'Set2', 'Pastel2', 'Set3', 
                                    'Antique', 'Safe', 'Bold', 'Pastel', 'Vivid', 'Prism']

                palette = []
                for name in name_color_set:
                    palette += getattr(plotly_colour.qualitative, name) # This is a list of colors

                color_dict_2[group] = palette[len(color_dict_2) % len(palette)]


        
        from .pca_ellipse import confidence_ellipse
        fig = px.scatter(df_opls_scores, x=x_, y=y_, symbol=symbol_,     
                        symbol_map=symbol_dict,
                        color='Group', 
                        color_discrete_map=color_dict_2, 
                        title=f'<b>OPLS-DA Scores Plot<b> ({self.scaling_method} scaling)', 
                        height=fig_height, width=fig_width,
                        labels={
                            't_pred': 't<sub>predict</sub>',
                            't_ortho': 't<sub>orthogonal</sub>',
                            't_scores': 't<sub>scores</sub>',
                            'Group': legend_name[0], 'symbol': legend_name[1]},
                        hover_data={'t_scores':True, 't_ortho':True, 't_pred':True, 'Group':True, 'Index':True}
        )

        
        
        fig.update_traces(marker=dict(size=marker_size, 
                            opacity=marker_opacity, 
                            line=dict(width=2, color='DarkSlateGrey')))


        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=0,
                                y=1.04+0.05,
                                showarrow=False,
                                text='<b>R<sup>2</sup>X: {}%<b>'.format(np.round(self.R2Xcorr*100, decimals=2)),
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")


        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=0,
                                y=1.0+0.05,
                                showarrow=False,
                                text='<b>R<sup>2</sup>Y: {}%<b>'.format(np.round(self.R2y*100, decimals=2)),
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")


        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=0,
                                y=1.08+0.05,
                                showarrow=False,
                                text='<b>Q<sup>2</sup>: {}%<b>'.format(np.round(self.q2*100, decimals=2)),
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")

                        
        if individual_ellipse == True:
            for circle_ in df_opls_scores['Group'].unique():
                
                fig.add_shape(type='path',
                    path=confidence_ellipse(df_opls_scores[df_opls_scores['Group']==circle_][x_], df_opls_scores[df_opls_scores['Group']==circle_][y_]),
                    line=dict(color=color_dict_2[circle_], width=2))
        else:
            fig.add_shape(type='path',
                    path=confidence_ellipse(df_opls_scores[x_], df_opls_scores[y_]))



        fig.update_yaxes(tickformat=",.0")
        fig.update_xaxes(tickformat=",.0")

        fig.update_xaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_yaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_xaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_yaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_layout(
            title={
                'y':0.95,
                'x':0.5,
                'xanchor': 'center',
                'yanchor': 'top'},
            font=dict(size=title_font_size))
        fig.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')

        return fig



    def plot_hist(self, nbins_=50, fig_height=500, fig_width=1000, font_size=14, title_font_size=20):

        '''
        Plot histogram of permutation scores

        Parameters
        ----------
        nbins_ : int, default=50
            Number of bins for histogram.
        fig_height : int, default=500
            Height of the figure.
        fig_width : int, default=1000
            Width of the figure.

        '''


        #check permutation model must be fitted
        if self.permutation_scores is None:
            raise ValueError('Permutation test must be performed first')

        permutation_scores = self.permutation_scores


        #Histrogram
        #Plot histogram of permutation scores
        fig = px.histogram(permutation_scores, nbins=nbins_, height=fig_height, width=fig_width, 
                        title='<b>Permutation scores<b>',
                        labels={'value': 'Accuracy score', 
                                'count': 'Frequency'})



        fig.add_shape(type='line', yref='paper', y0=0, y1=1, 
                        xref='x', x0=self.acc_score, x1=self.acc_score, 
                        line=dict(dash='dash', color='red', width=3))



        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=0,
                                y=1.25,
                                #y=1.18,
                                showarrow=False,
                                text='Number of permutation: {}'.format(self.n_permutations),
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")

        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=0,
                                y=1.18,
                                showarrow=False,
                                text='Accuracy score: {}'.format(np.round(self.acc_score, decimals=3)),
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")
        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=0,
                                y=1.11,
                                showarrow=False,
                                text='<i>p-value</i>: {}'.format(np.round(self.p_value, decimals=6)),
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")


        
        fig.update_layout(showlegend=False)

        fig.update_layout(title_x=0.5)

        return fig



    def plot_s_scores(self, fig_height=900, fig_width=2000, range_color_=[-0.05,0.05], 
                      color_continuous_scale_='jet', marker_size=14, font_size=20, title_font_size=20):

        '''
        Plot S-plot

        Parameters
        ----------
        fig_height : int, default=900
            Height of the figure.
        fig_width : int, default=2000
            Width of the figure.
        range_color: list, default=[-0.05,0.05]
            Range of color_ for the plot.
        color_continuous_scale_ : str, default='jet'
            color_ scale for the plot.

        '''


        s_df_scores_ = self.s_df_scores_
        
        s_df_scores_['Features'] = self.features_name


        fig = px.scatter(s_df_scores_, x='covariance', y='correlation', 
                         color='covariance', range_color=range_color_,
                         color_continuous_scale=color_continuous_scale_,
                         height=fig_height, width=fig_width,
                         hover_data={'covariance':True, 'correlation':True, 'Features':True})
        
        fig.update_layout(title='<b>S-plot</b>', xaxis_title='Covariance', yaxis_title='Correlation')

        
        #add line of axis and set color_ to black and line width to 2 pixel
        fig.update_xaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_yaxes(showline=True, linewidth=2, linecolor='black')

        #Add tick width to 2 pixel
        fig.update_xaxes(tickwidth=2)
        fig.update_yaxes(tickwidth=2)
        fig.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
        fig.update_yaxes(tickformat=",.0")
        #fig.update_xaxes(tickformat=",.0")
        fig.update_xaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_yaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_xaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_yaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_layout(
            title={
                'y':0.95,
                'x':0.5,
                'xanchor': 'center',
                'yanchor': 'top'},
            font=dict(size=title_font_size))

        fig.update_traces(marker=dict(size=marker_size))

        return fig


    
    def plot_loading(self, fig_height=900, fig_width=2000, range_color_=[-0.05,0.05], color_continuous_scale_='jet', 
                     marker_size=5, font_size=20, title_font_size=20,
                     xaxis_direction='reversed', xaxis_title='𝛿<sub>H</sub> in ppm'):

        '''
        Plot loading plot

        Parameters
        ----------
        fig_height : int, default=900
            Height of the figure.
        fig_width : int, default=2000
            Width of the figure.
        range_color: list, default=[-0.05,0.05]
            Range of color_ for the plot.
        color_continuous_scale_ : str, default='jet'
            color_ scale for the plot.

        '''

        s_df_scores_ = self.s_df_scores_

        def median_corr(X):
            X_corr = np.median(X, axis=0)
            X_corr = X_corr * np.sign(s_df_scores_['correlation'])
            return X_corr

        X2 = median_corr(self.X)

        s_df_scores_ = self.s_df_scores_
        features_name = self.features_name
        
        s_df_scores_['Features'] = features_name

        fig = px.scatter(s_df_scores_, x=features_name, y=X2, 
                            color='covariance', 
                            color_continuous_scale=color_continuous_scale_, 
                            range_color=range_color_, 
                            height=fig_height, width=fig_width,
                            hover_data={'covariance':True, 'correlation':True, 'Features':True})

        fig.update_traces(marker=dict(size=marker_size))
        fig.update_xaxes(autorange=xaxis_direction)
        fig.update_layout(title='<b>Loading spectra</b>', xaxis_title=xaxis_title, yaxis_title='Correlation')
        fig.update_xaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_yaxes(showline=True, linewidth=2, linecolor='black')

        fig.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
        fig.update_yaxes(tickformat=",.0")
        #fig.update_xaxes(tickformat=",.0")
        fig.update_layout(
            title={
                'y':0.95,
                'x':0.5,
                'xanchor': 'center',
                'yanchor': 'top'},
            font=dict(size=font_size))

        return fig




class pca:

    '''
    
    PCA model
    
    Parameters
    ----------
    X : array-like, shape (n_samples, n_features)
        Training data, where n_samples is the number of samples and n_features is the number of features.
    label : array-like, shape (n_samples,)
        Target data, where n_samples is the number of samples.
    features_name : array-like, shape (n_features,), default=None
        Name of features.
    n_components : int, default=2
        Number of components to keep.
    scale : str, default='pareto'
        Method of scaling. 'pareto' for pareto scaling, 'mean' for mean centering, 'uv' for unitvarian scaling.
    random_state : int, default=42
        Random state for permutation test.
    test_size : float, default=0.3
        Size of test set.

    Examples:
    ----------
    import pandas as pd
    import numpy as np
    from metbit import pca

    # Create a dataset
    data = pd.DataFrame(np.random.rand(500, 50000))
    class_ = pd.Series(np.random.choice(['A', 'B', 'C'], 500), name='Group')
    time = pd.Series(np.random.choice(['1-wk', '2-wk', '3-wk', '4-wk'], 500), name='Time point')


    # Assign X and target
    X = datasets.iloc[:, 2:]
    y = datasets['Group']
    time = datasets['Time point']
    features_name = list(X.columns.astype(float))

    ## Perform PCA model


    pca_mod = pca(X = X, label = y, features_name=features_name, n_components=2, scaling_method='pareto', random_state=42, test_size=0.3)
    pca_mod.fit()


    # Visualisation of PCA model
    pca_mod.plot_observe_variance()

    pca_mod.plot_cumulative_observed()

    shape_ = {'1-wk': 'circle', '2-wk': 'square', '3-wk': 'diamond', '4-wk': 'cross'}

    pca_mod.plot_pca_scores(symbol=time, symbol_dict=shape_)

    pca_mod.plot_loading_()

    pca_mod.plot_pca_trajectory(time_=time, time_order={'1-wk': 0, '2-wk': 1, '3-wk': 2, '4-wk': 3}, color_dict={'A': '#636EFA', 'B': '#EF553B', 'C': '#00CC96'}, symbol_dict=shape_)

    '''

    import numpy as np

    from sklearn import preprocessing
    import pandas as pd
    import matplotlib.pyplot as plt


    from .scaler import Scaler

    # Use to obtain same values as in the text


    import os
    import plotly.express as px
    import plotly.graph_objects as go

    from sklearn import decomposition
    from sklearn.preprocessing import scale
    from .pca_ellipse import confidence_ellipse

    from sklearn.model_selection import train_test_split
    from sklearn.preprocessing import StandardScaler
    from sklearn.decomposition import PCA
    from sklearn.metrics import r2_score
    from lingress import unipair


    def __init__(self, X: pd.DataFrame, label: list = None, features_name: list = None, n_components=2, scaling_method='pareto', random_state=42, test_size=0.3):




        if features_name is not None:
            if not isinstance(features_name, (pd.Series, np.ndarray, list)):
                raise ValueError('features_name must be a series, list or 1D array')
            if len(features_name) != X.shape[1]:
                raise ValueError('features_name must have the same number of features as X')
        
        if label is None:
            label = ["data" for x in range(X.shape[0])]

        if not isinstance(X, (pd.DataFrame, np.ndarray)):
            raise ValueError('X must be a dataframe or array')

        if not isinstance(n_components, int):
            raise ValueError('n_components must be an integer')

        if not isinstance(scaling_method, str):
            raise ValueError('scaling method must be a string')

        if not isinstance(random_state, int):
            raise ValueError('random_state must be an integer')

        if not isinstance(label, (pd.Series, np.ndarray, list)):
            raise ValueError('label must be a series, list or array')
        if len(label) != X.shape[0]:
            raise ValueError('X and label must have the same number of samples')

        self.features_name = features_name
        if features_name is None:
            if isinstance(X, pd.DataFrame):
                self.features_name = X.columns
            else:
                self.features_name = np.arange(X.shape[1])
        else:
            self.features_name = features_name

        
        # Check missing values in X
        if isinstance(X, pd.DataFrame):
            if X.isnull().sum().sum() > 0:
                raise ValueError('X contains missing values')
        else:
            if np.isnan(X).sum().sum() > 0:
                raise ValueError('X contains missing values')

        self.X = X
        self.label = label
        self.n_components = n_components
        self.scaling_method = scaling_method
        self.random_state = random_state


        self.test_size = test_size



    
    def fit(self):

        import numpy as np

        from sklearn import preprocessing
        import pandas as pd
        import matplotlib.pyplot as plt


        from .scaler import Scaler

        # Use to obtain same values as in the text


        import os
        import plotly.express as px
        import plotly.graph_objects as go

        from sklearn import decomposition
        from sklearn.preprocessing import scale
        from .pca_ellipse import confidence_ellipse

        from sklearn.model_selection import train_test_split
        from sklearn.preprocessing import StandardScaler
        from sklearn.decomposition import PCA
        from sklearn.metrics import r2_score
        from lingress import unipair

        test_size=self.test_size

        X = self.X
        label = self.label
        n_components = self.n_components
        scaling_method = self.scaling_method
        random_state = self.random_state
        features_name = self.features_name
        Y = pd.Categorical(label).codes
    
        
        if isinstance(features_name, list):
            features_name = list(features_name)
        
        if not isinstance(X, pd.DataFrame):
            X = pd.DataFrame(X, columns=features_name)
        
        if not isinstance(label, pd.Series):
            label = pd.Series(label)

        if scaling_method == 'pareto':
            Scale_power = 0.5
        elif scaling_method == 'mean':
            Scale_power = 0
        elif scaling_method == 'uv':
            Scale_power = 1
        elif scaling_method == 'minmax':
            Scale_power = 0
            
            
        model_scaler = Scaler(scale_power=Scale_power)
        model_scaler.fit(X)
        model_X = model_scaler.transform(X)



        pca_model = decomposition.PCA(n_components=n_components)
        pca_model.fit(model_X)

        self.scores_ = pca_model.transform(model_X)
        self.loadings_ = pca_model.components_.T
        

        #Create dataframe for scores depending on the number of components
        for i in range(n_components):
            if i == 0:
                df_scores_ = pd.DataFrame(self.scores_[:,i], columns=['PC{}'.format(i+1)])
            else:
                df_scores_['PC{}'.format(i+1)] = self.scores_[:,i]
        df_scores_.index = label.index

        df_scores_['Group'] = label

        self.df_scores_ = df_scores_

        #Create dataframe for loadings depending on the number of components
        for i in range(n_components):
            if i == 0:
                df_loadings_ = pd.DataFrame(self.loadings_[:,i], index=features_name, columns=['PC{}'.format(i+1)])
            else:
                df_loadings_['PC{}'.format(i+1)] = self.loadings_[:,i]

        df_loadings_['Features'] = features_name

        self.df_loadings_ = df_loadings_

        explained_variance_ = pca_model.explained_variance_ratio_
        explained_variance_ = np.insert(explained_variance_, 0, 0)
        cumulative_variance_ = np.cumsum(explained_variance_)


        r2_index = ['']
        for i in range(n_components):
            r2_index.append('PC{}'.format(i+1))

        df_explained_variance_ = pd.DataFrame(r2_index, columns=['PC'])
        df_explained_variance_['Explained variance'] = explained_variance_
        df_explained_variance_['Cumulative variance'] = cumulative_variance_



        X_train, X_test, y_train, y_test = train_test_split(X, Y, test_size=test_size, random_state=random_state)
        X_test = model_scaler.transform(X_test)
        X_test_pca = pca_model.transform(X_test)

            # Inverse transform the test set from the PCA space
        X_test_reconstructed = pca_model.inverse_transform(X_test_pca)


        # Calculate Q2 score for the test set
        q2_test = r2_score(X_test, X_test_reconstructed)
        

        self.q2_test = q2_test
        self.explained_variance_ = explained_variance_  
        self.cumulative_variance_ = cumulative_variance_
        self.df_explained_variance_ = df_explained_variance_
        self.pca_model = pca_model
        self.model_scaler = model_scaler
        self.X_test = X_test
        self.y_test = y_test
        self.X_test_reconstructed = X_test_reconstructed
        self.X_test_pca = X_test_pca
        self.df_scores_ = df_scores_
        self.df_loadings_ = df_loadings_

        return pca_model

    def get_explained_variance(self):
        df_explained_variance_ = self.df_explained_variance_
        return df_explained_variance_

    def get_scores(self):
        df_scores_ = self.df_scores_
        return df_scores_

    def get_loadings(self):
        df_loadings_ = self.df_loadings_
        return df_loadings_

    def get_q2_test(self):
        q2_test = self.q2_test
        return q2_test

    def plot_observe_variance(self, fig_height=600, fig_width=800, font_size=15):

        '''
        Visualise explained variance plot

        Returns
        -------
        fig : plotly.graph_objects.Figure
            Explained variance plot.

        '''

        scaling_method = self.scaling_method
        
        df_explained_variance_ = self.df_explained_variance_
        
        fig = px.bar(df_explained_variance_, 
                x='PC', y=df_explained_variance_['Explained variance'],
                text=np.round(df_explained_variance_['Explained variance'], decimals=3),
                width=fig_width, height=fig_height,
                title='Explained Variance ({} scaling)'.format(scaling_method))
        fig.update_layout(
            title={
                'y':0.95,
                'x':0.5,
                'xanchor': 'center',
                'yanchor': 'top'},
            font=dict(size=font_size))
        fig.update_traces(texttemplate='%{text:.3f}', textposition='outside')
        return fig


    def plot_cumulative_observed(self, fig_height=600, fig_width=800, font_size=15, marker_size=10):

        '''
        Visualise cumulative variance plot
        
        Returns
        -------
        fig : plotly.graph_objects.Figure
            Cumulative variance plot.

        '''
        
        df_explained_variance_ = self.df_explained_variance_

        fig = go.Figure()

        fig.add_trace(
            go.Scatter(
                x=df_explained_variance_['PC'],
                y=df_explained_variance_['Cumulative variance'],
                marker=dict(size=marker_size, color="LightSeaGreen"),
                name='R<sup>2</sup>X (Cum)'
            ))

        fig.add_trace(
            go.Bar(
                x=df_explained_variance_['PC'],
                y=df_explained_variance_['Explained variance'],
                marker=dict(color="RoyalBlue"),
                name='R<sup>2</sup>X',
                text=np.round(df_explained_variance_['Explained variance'], decimals=3)
            ))
        fig.update_layout(width=fig_width, height=fig_height,
                        title='Explained Variance and Cumulative Variance')
        fig.update_layout(
            title={
                'y':0.95,
                'x':0.5,
                'xanchor': 'center',
                'yanchor': 'top'},
            font=dict(size=font_size))

        return fig



    def plot_pca_scores(self, pc=['PC1', 'PC2'], 
                        color_=None, color_dict=None, 
                        symbol_=None, symbol_dict=None, 
                        fig_height=900, fig_width=1300,
                        marker_size=35, marker_opacity=0.7,
                        font_size=20, title_font_size=21,
                        individual_ellipse=True,
                        legend_name=['Group', 'Time point']):

        '''
        Visualise PCA scores plot

        Parameters
        ----------
        pc : list, default=['PC1', 'PC2']
            List of principal components to plot.
        color: array-like, shape (n_samples,), default=None
            Target data, where n_samples is the number of samples.
        color_dict : dict, default=None
            Dictionary of color_ mapping.
        symbol_ : array-like, shape (n_samples,), default=None
            Target data, where n_samples is the number of samples.
        symbol_dict : dict, default=None
            Dictionary of symbol_ mapping.
        fig_height : int, default=900
            Height of figure.
        fig_width : int, default=1300
            Width of figure.
        marker_size : int, default=35
            Size of marker.
        marker_opacity : float, default=0.7
            Opacity of marker.
        text_ : array-like, shape (n_samples,), default=None
            Text to display on each point.


        Returns
        -------
        fig : plotly.graph_objects.Figure
            PCA scores plot.

        '''

        from .pca_ellipse import confidence_ellipse

        scaling_method = self.scaling_method
        df_scores_ = self.df_scores_
        r2 = self.df_explained_variance_
        q2_test = self.q2_test
        
        if color_ is not None:
            if len(color_) != len(self.label):
                raise ValueError('color_ must have the same number of samples as y')
            else:
                color_ = color_
                
                
        if symbol_ is not None:
            if len(symbol_) != len(self.label):
                raise ValueError('symbol_ must have the same number of samples as y')

        if color_ is not None:
            df_scores_['Group'] = color_
        else:
            pass

        #check symbol_ dimension must be equal to y
        if symbol_ is not None:
            if len(symbol_) != len(self.label):
                raise ValueError('symbol_ must have the same number of samples as y')

        #check symbol_dict must be a dictionary
        if symbol_dict is not None:
            if not isinstance(symbol_dict, dict):
                raise ValueError('symbol_dict must be a dictionary')
        else:
            symbol_dict = None


        # pc must be a list of 2
        if not isinstance(pc, list):
            raise ValueError("pc must be a list of string \n Example: pc=['PC1', 'PC2']")
        if len(pc) != 2:
            raise ValueError('pc must be a list of 2')
        # pc must be match with columns of df_scores_
        if pc[0] not in self.df_scores_.columns:
            raise ValueError("pc must be in df_scores_ columns \n Example: pc=['PC1', 'PC2']")
        if pc[1] not in self.df_scores_.columns:
            raise ValueError("pc must be in df_scores_ columns \n Example: pc=[\'PC1\', \'PC2\']")

        

        
        r2 = self.df_explained_variance_
        q2_test = self.q2_test
        df_scores_['Index'] = df_scores_.index


        #If user not input color_dict then get unique of label and create color_dict
        if color_dict is not None:
            color_dict = color_dict
        else:
            
            import plotly.colors as plotly_colour

            name_color_set = ['Plotly', 'D3', 'G10', 'T10', 'Alphabet', 'Dark24', 'Light24', 'Set1', 'Pastel1', 
                                'Dark2', 'Set2', 'Pastel2', 'Set3', 'Antique', 'Safe', 'Bold', 'Pastel', 
                                'Vivid', 'Prism']

            palette = []
            for name in name_color_set:
                palette += getattr(plotly_colour.qualitative, name) # This is a list of colors

            color_dict = {i: palette[i] for i in range(len(df_scores_['Group'].unique()))}
            

        #new color_dict to match with unique label
        group_unique = df_scores_['Group'].unique()
        #change key of color_dict to match with unique label
        color_dict_2 = {group_unique[i]: list(color_dict.values())[i] for i in range(len(group_unique))}
        

        fig = px.scatter(df_scores_, x=pc[0], y=pc[1], color='Group',
                        symbol=symbol_, 
                        color_discrete_map=color_dict_2, 
                        symbol_map=symbol_dict, 
                        title=f'<b>PCA Scores Plot<b> {self.scaling_method} scaling', 
                        height=fig_height, width=fig_width,
                        labels={'color': legend_name[0], 'symbol': legend_name[1],
                                'Group': legend_name[0],
                                pc[0]: "{} R<sup>2</sup>X: {}%".format(pc[0], np.round(r2.loc[r2.loc[r2['PC']==pc[0]].index, 'Explained variance'].values[0]*100, decimals=2)),
                                pc[1]: "{} R<sup>2</sup>X: {}%".format(pc[1], np.round(r2.loc[r2.loc[r2['PC']==pc[1]].index, 'Explained variance'].values[0]*100, decimals=2))},
                        hover_data={'Group':True, 'Index':True, pc[0]:True, pc[1]:True})
        

        fig.update_traces(marker=dict(size=marker_size, 
                            opacity=marker_opacity, 
                            line=dict(width=2, color='DarkSlateGrey')))


        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=1.0,
                                y=0.05,
                                showarrow=False,
                                text=f"<b>R<sup>2</sup>X (Cum): {np.round(r2.loc[r2.loc[r2['PC']==pc[1]].index, 'Cumulative variance'].values[0]*100, decimals=2)}%<b>",
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")

        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=1.0,
                                y=0.01,
                                showarrow=False,
                                text=f"<b>Q<sup>2</sup>X (Cum): {np.round(q2_test*100, decimals=2)}%<b>",
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")

        if individual_ellipse == True:
            for circle_ in df_scores_['Group'].unique():
                
                fig.add_shape(type='path',
                    path=confidence_ellipse(df_scores_[df_scores_['Group']==circle_][pc[0]], df_scores_[df_scores_['Group']==circle_][pc[1]]),
                    line=dict(color=color_dict_2[circle_], width=2))
        else:
            fig.add_shape(type='path',
                    path=confidence_ellipse(df_scores_[pc[0]], df_scores_[pc[1]]))



        fig.update_xaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_yaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_xaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_yaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_layout(
            title={
                'y':0.95,
                'x':0.5,
                'xanchor': 'center',
                'yanchor': 'top'},
            font=dict(size=title_font_size))
        fig.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')

        return fig



    def plot_loading_(self, pc=['PC1', 'PC2'], fig_height=600, fig_width=1800, 
                      font_size=20, title_font_size=20, marker_size=1, 
                      x_axis_title='𝛿<sub>H</sub> in ppm', xaxis_direction = "reversed"):

        '''
        Visualise PCA loadings

        Parameters
        ----------
        pc : list, default=['PC1', 'PC2']
            Principle component to plot.
        fig_height : int, default=600
            Height of figure.
        fig_width : int, default=1800
            Width of figure.

        Returns
        -------
        fig : plotly.graph_objects.Figure
            Plotly figure.

        ----------
        '''
        pc = pc
        df_loadings_ = self.df_loadings_

        loadings_label = self.features_name
        df_loadings_['Features'] = loadings_label


        fig = px.scatter(df_loadings_, x='Features', y=pc,
                                height=fig_height, width=fig_width,
                                title='Loadings plot')

        fig.update_xaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_yaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_layout(title={'y':0.95,
                                'x':0.5,
                                'xanchor': 'center',
                                'yanchor': 'top'},
                        font=dict(size=title_font_size))
        
        fig.update_layout(xaxis = dict(autorange=xaxis_direction))
                
        fig.update_traces(marker=dict(size=marker_size))
        fig.update_layout(xaxis_title=x_axis_title)
        
        
        return fig






    def plot_pca_trajectory(self, time_, time_order, stat_ = ['mean', 'sem'], pc=['PC1', 'PC2'],
                            color_dict = None, 
                            symbol_dict = None, 
                            fig_height=900, fig_width=1300,
                            marker_size=35, marker_opacity=0.7,
                            title_font_size=20, font_size=20,
                            legend_name=['Group', 'Time point']):

        '''

        Visualise PCA trajectory

        Parameters
        ----------
        time_ : array-like, shape (n_samples,)
            Time point of samples.
        time_order : dictionary
            Order of time point.
        stat_ : list, default=['mean', 'sem']  
            Statistic to calculate. First element is mean or median, second element is sem or std.
        pc : list, default=['PC1', 'PC2']
            Principle component to plot.
        color_dict : dictionary, default=None
            Dictionary of color_ for each group.
        symbol_dict : dictionary, default=None
            Dictionary of symbol_ for each time point.
        fig_height : int, default=900
            Height of figure.
        fig_width : int, default=1300
            Width of figure.
        marker_size : int, default=35
            Size of marker.
        marker_opacity : float, default=0.7
            Opacity of marker.

        Returns
        -------
        fig : plotly.graph_objects.Figure
            Plotly figure.

        '''


        from .pca_ellipse import confidence_ellipse


        #check time_order must be a dictionary
        if not isinstance(time_order, dict):
            raise ValueError("time_order must be a dictionary \n Example: time_order = {'Day 1': 0, 'Day 2': 1, 'Day 3': 2}")




        #check time are not missing
        if time_ is None:
            raise ValueError('time_ must be provided fot time trajectory analysis')

        #check time_ must be pandas series, list or array
        if not isinstance(time_, (pd.Series, np.ndarray, list)):
            raise ValueError('time_ must be a series, list or array')
        if len(time_) != len(self.label):
            raise ValueError('time_ must have the same number of samples as group')

        


        df_scores_ = self.df_scores_
        r2 = self.df_explained_variance_
        q2_test = self.q2_test
        df_scores_['Time point'] = time_
        



        #check stat_[0] must be mean or median
        if stat_[0] not in ['mean', 'median']:
            raise ValueError('stat_[0] must be mean or median')
        #check stat_[1] must be sem, std
        if stat_[1] not in ['sem','std']:
            raise ValueError('stat_[1] must be sem or std')

        if stat_[0] == 'mean':
            df_scores_point = df_scores_.groupby(['Group', 'Time point']).mean()
        if stat_[0] == 'median':
            df_scores_point = df_scores_.groupby(['Group', 'Time point']).median()

        if stat_[1] == 'sem':
            err_df = df_scores_.groupby(['Group', 'Time point']).sem()
        if stat_[1] == 'std':
            err_df = df_scores_.groupby(['Group', 'Time point']).std()


        df_scores_point.reset_index(inplace=True)
        df_scores_point['Time order'] = df_scores_point['Time point'].map(time_order)
        df_scores_point.sort_values(by=['Group', 'Time order'], inplace=True)

        err_df.reset_index(inplace=True)
        err_df['Time order'] = err_df['Time point'].map(time_order)
        err_df.sort_values(by=['Group', 'Time order'], inplace=True)



        #check color_dict must be a dictionary
        if color_dict is not None:
            if not isinstance(color_dict, dict):
                raise ValueError('color_dict must be a dictionary')
        else:
            color_dict = None


        if symbol_dict is not None:
            if not isinstance(symbol_dict, dict):
                raise ValueError('symbol_dict must be a dictionary')
        else:
            symbol_dict = None
        


        #If user not input color_dict then get unique of label and create color_dict
        if color_dict is not None:
            color_dict = color_dict
        else:

            import plotly.colors as plotly_colour
            name_color_set = ['Plotly', 'D3', 'G10', 'T10', 'Alphabet', 'Dark24', 'Light24', 'Set1', 'Pastel1', 
                                'Dark2', 'Set2', 'Pastel2', 'Set3', 'Antique', 'Safe', 'Bold', 'Pastel', 
                                'Vivid', 'Prism']

            palette = []
            for name in name_color_set:
                palette += getattr(plotly_colour.qualitative, name) # This is a list of colors


            color_dict = {i: palette[i] for i in range(len(df_scores_point['Group'].unique()))}
            

        #new color_dict to match with unique label
        group_unique = df_scores_point['Group'].unique()
        #change key of color_dict to match with unique label
        color_dict_2 = {group_unique[i]: list(color_dict.values())[i] for i in range(len(group_unique))}
        

        fig = px.line(df_scores_point, x=pc[0], y=pc[1], line_group='Time point', 
                        error_x=err_df[pc[0]], error_y=err_df[pc[1]],
                        color='Group', color_discrete_map=color_dict_2,
                        symbol='Time point', symbol_map=symbol_dict,
                        title='<b>Principle component analysis ({})<b>'.format(self.scaling_method), 
                        height=fig_height, width=fig_width,
                        labels={
                            pc[0]: "{} R<sup>2</sup>X: {} %".format(pc[0], np.round(r2.loc[r2.loc[r2['PC']==pc[0]].index, 'Explained variance'].values[0]*100, decimals=2)),
                            pc[1]: "{} R<sup>2</sup>X: {} %".format(pc[1], np.round(r2.loc[r2.loc[r2['PC']==pc[1]].index, 'Explained variance'].values[0]*100, decimals=2)),
                            'Group': legend_name[0],
                            'Time point': legend_name[1],
                            })


        



        for connect_line in range(len(group_unique)):
            # create a new trace for the connecting line
            fig.add_trace(go.Scatter(
                x=df_scores_point.loc[list(df_scores_point.loc[df_scores_point['Group'] == df_scores_point['Group'].unique()[connect_line]].index), pc[0]], # x-coordinates of the line
                y=df_scores_point.loc[list(df_scores_point.loc[df_scores_point['Group'] == df_scores_point['Group'].unique()[connect_line]].index), pc[1]], # y-coordinates of the line
                mode='lines', # specify the trace type as lines
                line=dict(color=color_dict_2[group_unique[connect_line]], width=2), # set the color_ and width of the line
                showlegend=False # hide the trace from the legend
            ))



        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=1.0,
                                y=0.05,
                                showarrow=False,
                                text=f"<b>R<sup>2</sup>X (Cum): {np.round(r2.loc[r2.loc[r2['PC']==pc[1]].index, 'Cumulative variance'].values[0]*100, decimals=2)}%<b>",
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")

        fig.add_annotation(dict(font=dict(color="black",size=font_size),
                                #x=x_loc,
                                x=1.0,
                                y=0.01,
                                showarrow=False,
                                text=f"<b>Q<sup>2</sup>X (Cum): {np.round(q2_test*100, decimals=2)}%<b>",
                                textangle=0,
                                xref="paper",
                                yref="paper"),
                                # set alignment of text to left side of entry
                                align="left")

        fig.add_shape(type='path',
                path=confidence_ellipse(df_scores_point[pc[0]], df_scores_point[pc[1]]))


                    #update axis as scitifics
        fig.update_xaxes(tickformat=".1e")
        fig.update_yaxes(tickformat=".1e")



        fig.update_xaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_yaxes(zeroline=True, zerolinewidth=2, zerolinecolor='Black')
        fig.update_xaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_yaxes(showline=True, linewidth=2, linecolor='black')
        fig.update_layout(
            title={
                'y':0.95,
                'x':0.5,
                'xanchor': 'center',
                'yanchor': 'top'},
            font=dict(size=title_font_size))


        fig.update_traces(marker=dict(size=marker_size, 
                                      opacity=marker_opacity, 
                                      line=dict(width=2, 
                                                color='DarkSlateGrey')))

        fig.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
        
        
        return fig
    

    def plot_3d_pca(self, pc=['PC1', 'PC2', 'PC3'], color_=None, color_dict=None, 
                    symbol_=None, symbol_dict=None, fig_height=900, fig_width=1300,
                    marker_size=35, marker_opacity=0.7, font_size=20, title_font_size=20,
                    legend_name=['Group', 'Time point']):
        import plotly.express as px
        '''
        Visualise 3D PCA scores plot

        Parameters
        ----------
        pc : list, default=['PC1', 'PC2', 'PC3']
            List of principal components to plot.
        color: array-like, shape (n_samples,), default=None
            Target data, where n_samples is the number of samples.
        color_dict : dict, default=None
            Dictionary of color_ mapping.
        symbol_ : array-like, shape (n_samples,), default=None
            Target data, where n_samples is the number of samples.
        symbol_dict : dict, default=None
            Dictionary of symbol_ mapping.
        fig_height : int, default=900
            Height of figure.
        fig_width : int, default=1300
            Width of figure.
        marker_size : int, default=35
            Size of marker.
        marker_opacity : float, default=0.7
            Opacity of marker.
        text_ : array-like, shape (n_samples,), default=None
            Text to display on each point.


        Returns
        -------
        fig : plotly.graph_objects.Figure
            PCA scores plot.

        '''



        scaling_method = self.scaling_method
        df_scores_ = self.df_scores_
        r2 = self.df_explained_variance_
        q2_test = self.q2_test
        
        if color_ is not None:
            if len(color_) != len(self.label):
                raise ValueError('color_ must have the same number of samples as y')
            else:
                color_ = color_
                
                
        if symbol_ is not None:
            if len(symbol_) != len(self.label):
                raise ValueError('symbol_ must have the same number of samples as y')

        if color_ is not None:
            df_scores_['Group'] = color_
        else:
            pass

        #check symbol_ dimension must be equal to y
        if symbol_ is not None:
            if len(symbol_) != len(self.label):
                raise ValueError('symbol_ must have the same number of samples as y')

        #check symbol_dict must be a dictionary
        if symbol_dict is not None:
            if not isinstance(symbol_dict, dict):
                raise ValueError('symbol_dict must be a dictionary')
        else:
            symbol_dict = None


        # pc must be a list of 3
        if not isinstance(pc, list):
            raise ValueError("pc must be a list of string \n Example: pc=['PC1', 'PC2', 'PC3']")
        if len(pc) != 3:
            raise ValueError('pc must be a list of 3')
        # pc must be match with columns of df_scores_
        if pc[0] not in self.df_scores_.columns:
            raise ValueError("pc must be in df_scores_ columns \n Example: pc=['PC1', 'PC2', 'PC3']")
        if pc[1] not in self.df_scores_.columns:
            raise ValueError("pc must be in df_scores_ columns \n Example: pc=[\'PC1\', \'PC2\', \'PC3\']")
        if pc[2] not in self.df_scores_.columns:
            raise ValueError("pc must be in df_scores_ columns \n Example: pc=[\'PC1\', \'PC2\', \'PC3\']")

        r2 = self.df_explained_variance_
        q2_test = self.q2_test
        df_scores_['Index'] = df_scores_.index


        #If user not input color_dict then get unique of label and create color_dict
        if color_dict is not None:
            color_dict = color_dict
        else:

            import plotly.colors as plotly_colour
            name_color_set = ['Plotly', 'D3', 'G10', 'T10', 'Alphabet', 'Dark24', 'Light24', 'Set1', 'Pastel1', 
                                'Dark2', 'Set2', 'Pastel2', 'Set3', 'Antique', 'Safe', 'Bold', 'Pastel', 
                                'Vivid', 'Prism']

            palette = []
            for name in name_color_set:
                palette += getattr(plotly_colour.qualitative, name) # This is a list of colors


            color_dict = {i: palette[i] for i in range(len(df_scores_['Group'].unique()))}

        #new color_dict to match with unique label
        group_unique = df_scores_['Group'].unique()
        #change key of color_dict to match with unique label
        color_dict_2 = {group_unique[i]: list(color_dict.values())[i] for i in range(len(group_unique))}


        fig = px.scatter_3d(df_scores_, x=pc[0], y=pc[1], z=pc[2], color='Group', symbol=symbol_, 
                            color_discrete_map=color_dict_2, symbol_map=symbol_dict, 
                            title=f'<b>PCA Scores Plot<b> {self.scaling_method} scaling', 
                            height=fig_height, width=fig_width,
                            labels={'color': legend_name[0], 'symbol': legend_name[1],
                                    'Group': legend_name[0],
                                    pc[0]: "{} R<sup>2</sup>X: {}%".format(pc[0], np.round(r2.loc[r2.loc[r2['PC']==pc[0]].index, 'Explained variance'].values[0]*100, decimals=2)),
                                    pc[1]: "{} R<sup>2</sup>X: {}%".format(pc[1], np.round(r2.loc[r2.loc[r2['PC']==pc[1]].index, 'Explained variance'].values[0]*100, decimals=2)),
                                    pc[2]: "{} R<sup>2</sup>X: {}%".format(pc[2], np.round(r2.loc[r2.loc[r2['PC']==pc[2]].index, 'Explained variance'].values[0]*100, decimals=2))},
                            hover_data={'Group':True, 'Index':True, pc[0]:True, pc[1]:True, pc[2]:True})

        fig.update_traces(marker=dict(size=marker_size,
                            opacity=marker_opacity,
                            line=dict(width=2, color='DarkSlateGrey')))

        fig.add_annotation(dict(font=dict(color="black",size=font_size), x=1.0, y=0.05, showarrow=False,
                                text=f"<b>R<sup>2</sup>X (Cum): {np.round(r2.loc[r2.loc[r2['PC']==pc[2]].index, 'Cumulative variance'].values[0]*100, decimals=2)}%<b>",
                                textangle=0, xref="paper", yref="paper"), align="left")

        fig.add_annotation(dict(font=dict(color="black",size=font_size), x=1.0, y=0.01, showarrow=False, 
                                text=f"<b>Q<sup>2</sup>X (Cum): {np.round(q2_test*100, decimals=2)}%<b>",
                                textangle=0, xref="paper", yref="paper"), align="left")

        fig.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')

        return fig

        

        