# -*- coding: utf-8 -*-
# *******************************************************
#   ____                     _               _
#  / ___|___  _ __ ___   ___| |_   _ __ ___ | |
# | |   / _ \| '_ ` _ \ / _ \ __| | '_ ` _ \| |
# | |__| (_) | | | | | |  __/ |_ _| | | | | | |
#  \____\___/|_| |_| |_|\___|\__(_)_| |_| |_|_|
#
#  Sign up for free at https://www.comet.com
#  Copyright (C) 2015-2024 Comet ML INC
#  This source code is licensed under the MIT license.
# *******************************************************
import logging
from typing import Any, Dict

from comet_ml.semantic_version import SemanticVersion

from xgboost.callback import TrainingCallback

from ..messages import ParameterMessage
from ._base import (
    _log_xgboost_model_graph,
    _log_xgboost_model_metrics,
    _log_xgboost_parameters,
    _log_xgboost_step,
)

LOGGER = logging.getLogger(__name__)


def _adjust_metrics_to_comet(eval_log):
    # type: (Dict[Any, Any]) -> Dict[Any, Any]
    """
    For metrics with more than 1 value, return the average of the values.
    """
    comet_metrics = {}

    for context, metric in eval_log.items():
        per_context_metrics = {}
        for metric_name, log in metric.items():
            per_context_metrics[metric_name] = log[-1]

        comet_metrics[context] = per_context_metrics

    return comet_metrics


def get_xgboost_rabit_parameters():
    try:
        from xgboost import __version__ as xgboost_version

        xgboost_v2 = SemanticVersion(2, 0, 0)

        if xgboost_v2.compare(xgboost_version) == 1:
            # old xgboost versions
            import xgboost.rabit as rabit
        else:
            import xgboost.collective as rabit

        return {"rank": rabit.get_rank(), "world_size": rabit.get_world_size()}
    except Exception:
        LOGGER.warning("Failed to log XGBoost rabit parameters", exc_info=True)
        return None


class XGBoostCometCallback(TrainingCallback):
    def __init__(self, experiment):
        super(XGBoostCometCallback, self).__init__()
        self.experiment = experiment
        self.iterations = []

    def after_iteration(self, model, epoch, evals_log):
        _log_xgboost_step(self.experiment, epoch)

        _log_xgboost_parameters(self.experiment, model)

        _log_xgboost_model_metrics(self.experiment, _adjust_metrics_to_comet(evals_log))

        _log_xgboost_model_graph(self.experiment, model)

        self.iterations.append(epoch)

        if self.experiment.auto_param_logging:
            rabit_parameters = get_xgboost_rabit_parameters()
            if rabit_parameters is not None:
                self.experiment._log_parameters(
                    rabit_parameters,
                    source=ParameterMessage.source_autologger,
                    flatten_nested=False,
                )

    def after_training(self, model):
        if self.experiment.auto_param_logging:
            self.experiment._log_parameter(
                "begin_iteration",
                self.iterations[0],
                source=ParameterMessage.source_autologger,
            )
            self.experiment._log_parameter(
                "end_iteration",
                self.iterations[-1],
                source=ParameterMessage.source_autologger,
            )
        return super().after_training(model)
