import os

import pytest

from artl_mcp.tools import (
    clean_text,
    extract_doi_from_url,
    extract_paper_info,
    get_abstract_from_pubmed_id,
    # DOIFetcher-based tools
    get_doi_metadata,
    get_full_text_from_doi,
    get_full_text_info,
    get_unpaywall_info,
)
from artl_mcp.utils.email_manager import EmailManager


# Test data from test_aurelian.py
def get_test_email():
    """Get a valid test email address from environment/local config."""
    em = EmailManager()
    email = em.get_email()
    if not email:
        pytest.skip(
            "No valid email address found for testing. "
            "Set ARTL_EMAIL_ADDR or add to local/.env"
        )
    return email


DOI_VALUE = "10.1099/ijsem.0.005153"
FULL_TEXT_DOI = "10.1128/msystems.00045-18"
PDF_URL = "https://ceur-ws.org/Vol-1747/IT201_ICBO2016.pdf"
DOI_URL = "https://doi.org/10.7717/peerj.16290"
DOI_PORTION = "10.7717/peerj.16290"
PMID_OF_DOI = "37933257"
PMCID = "PMC10625763"
PMID_FOR_ABSTRACT = "31653696"

# Expected text content
EXPECTED_TEXT_MAGELLANIC = "Magellanic"
EXPECTED_IN_ABSTRACT = "deglycase"
EXPECTED_BIOSPHERE = "biosphere"
EXPECTED_MICROBIOME = "microbiome"


@pytest.mark.external_api
@pytest.mark.slow
def test_get_abstract_from_pubmed_id():
    """Test abstract retrieval from PubMed ID."""
    result = get_abstract_from_pubmed_id(PMID_FOR_ABSTRACT)
    assert result is not None
    assert isinstance(result, dict)
    assert "content" in result
    assert "saved_to" in result
    assert "truncated" in result
    assert isinstance(result["content"], str)
    # Any string result is valid - function should handle unavailable abstracts
    assert len(result["content"]) >= 0  # Could be empty string if no abstract available


@pytest.mark.external_api
@pytest.mark.slow
def test_get_unpaywall_info():
    """Test Unpaywall information retrieval."""
    test_email = get_test_email()
    result = get_unpaywall_info(DOI_VALUE, test_email, strict=True)
    # Unpaywall may not have all DOIs, so we test more flexibly
    if result is not None:
        assert isinstance(result, dict)
        # If successful, should have genre field
        if "genre" in result:
            assert result["genre"] == "journal-article"


@pytest.mark.external_api
@pytest.mark.slow
def test_get_full_text_from_doi():
    """Test full text retrieval from DOI - now actually tests content."""
    test_email = get_test_email()
    result = get_full_text_from_doi(FULL_TEXT_DOI, test_email)
    # Test that we actually get meaningful full text content
    if result is not None:
        assert isinstance(result, dict)
        assert "content" in result
        assert "saved_to" in result
        assert "truncated" in result
        assert isinstance(result["content"], str)
        assert len(result["content"]) > 100  # Should have substantial content
        # Test for expected content that should be in the full text
        assert EXPECTED_MICROBIOME in result["content"].lower()
    else:
        pytest.skip("Full text not available for test DOI")


@pytest.mark.external_api
@pytest.mark.slow
def test_get_full_text_info():
    """Test full text information retrieval."""
    test_email = get_test_email()
    result = get_full_text_info(FULL_TEXT_DOI, test_email)
    # Test more flexibly since full text may not be available
    if result is not None:
        assert isinstance(result, dict)
        assert "success" in result
        assert "info" in result


def test_clean_text():
    """Test text cleaning functionality."""
    test_email = get_test_email()
    input_text = "   xxx   xxx   "
    expected_output = "xxx xxx"
    result = clean_text(input_text, test_email)
    assert isinstance(result, dict)
    assert "content" in result
    assert "saved_to" in result
    assert "truncated" in result
    assert result["content"] == expected_output


@pytest.mark.external_api
@pytest.mark.slow
@pytest.mark.skipif(
    os.environ.get("CI") == "true", reason="Skip flaky network test in CI"
)
def test_extract_doi_from_url():
    """Test DOI extraction from URL."""
    result = extract_doi_from_url(DOI_URL)
    assert result == DOI_PORTION


def test_get_doi_metadata_invalid_doi():
    """Test DOI metadata with invalid DOI."""
    result = get_doi_metadata("invalid-doi")
    assert result is None


def test_get_unpaywall_info_invalid_doi():
    """Test Unpaywall with invalid DOI."""
    test_email = get_test_email()
    result = get_unpaywall_info("invalid-doi", test_email)
    assert result is None


@pytest.mark.external_api
@pytest.mark.slow
def test_get_unpaywall_info_strict_false():
    """Test Unpaywall with strict=False."""
    test_email = get_test_email()
    result = get_unpaywall_info(DOI_VALUE, test_email, strict=False)
    # Unpaywall may not have all DOIs, test more flexibly
    if result is not None:
        assert isinstance(result, dict)


def test_clean_text_various_inputs():
    """Test text cleaning with various inputs."""
    test_cases = [
        ("  hello  world  ", "hello world"),
        ("single", "single"),
        ("", ""),
        ("  ", ""),
    ]

    test_email = get_test_email()
    for input_text, _expected in test_cases:
        result = clean_text(input_text, test_email)
        # The exact cleaning behavior depends on DOIFetcher implementation
        # Just ensure it returns a dict with proper structure
        assert isinstance(result, dict)
        assert "content" in result
        assert "saved_to" in result
        assert "truncated" in result
        assert isinstance(result["content"], str)


# Tests for extract_paper_info - core data processing function
def test_extract_paper_info_complete_data():
    """Test extract_paper_info with complete typical CrossRef data."""
    work_item = {
        "title": ["Machine Learning in Scientific Research: A Comprehensive Review"],
        "author": [
            {"given": "Alice", "family": "Johnson"},
            {"given": "Bob", "family": "Smith"},
            {"given": "Carol", "family": "Davis"},
        ],
        "container-title": ["Nature Machine Intelligence"],
        "published-print": {"date-parts": [[2023, 6, 15]]},
        "published-online": {"date-parts": [[2023, 5, 20]]},
        "DOI": "10.1038/s42256-023-00123-4",
        "URL": "https://www.nature.com/articles/s42256-023-00123-4",
        "abstract": "This comprehensive review examines machine learning...",
        "is-referenced-by-count": 142,
        "type": "journal-article",
        "publisher": "Springer Nature",
    }

    result = extract_paper_info(work_item)

    # Verify all fields are extracted correctly
    assert (
        result["title"]
        == "Machine Learning in Scientific Research: A Comprehensive Review"
    )
    assert result["authors"] == ["Alice Johnson", "Bob Smith", "Carol Davis"]
    assert result["journal"] == "Nature Machine Intelligence"
    assert result["published_date"] == {"date-parts": [[2023, 6, 15]]}
    assert result["doi"] == "10.1038/s42256-023-00123-4"
    assert result["url"] == "https://www.nature.com/articles/s42256-023-00123-4"
    assert (
        result["abstract"] == "This comprehensive review examines machine learning..."
    )
    assert result["citation_count"] == 142
    assert result["type"] == "journal-article"
    assert result["publisher"] == "Springer Nature"


def test_extract_paper_info_minimal_data():
    """Test extract_paper_info with minimal data (empty/missing fields)."""
    work_item = {"DOI": "10.1234/minimal-doi"}

    result = extract_paper_info(work_item)

    # Verify defaults are used for missing fields
    assert result["title"] == ""
    assert result["authors"] == []
    assert result["journal"] == ""
    assert result["published_date"] == {}
    assert result["doi"] == "10.1234/minimal-doi"
    assert result["url"] == ""
    assert result["abstract"] == ""
    assert result["citation_count"] == 0
    assert result["type"] == ""
    assert result["publisher"] == ""


def test_extract_paper_info_partial_authors():
    """Test extract_paper_info with incomplete author data."""
    work_item = {
        "title": ["Test Article"],
        "author": [
            {"given": "Alice", "family": "Johnson"},
            {"family": "Smith"},  # Missing given name
            {"given": "Carol"},  # Missing family name
            {},  # Empty author object
        ],
        "DOI": "10.1234/test-authors",
    }

    result = extract_paper_info(work_item)

    # Verify author handling with missing data
    assert result["authors"] == ["Alice Johnson", " Smith", "Carol ", " "]
    assert result["title"] == "Test Article"
    assert result["doi"] == "10.1234/test-authors"


def test_extract_paper_info_array_fields():
    """Test extract_paper_info handles array fields correctly."""
    work_item = {
        "title": ["First Title", "Alternative Title"],  # Multiple titles
        "container-title": [
            "Primary Journal",
            "Alternative Journal",
        ],  # Multiple journals
        "DOI": "10.1234/array-test",
    }

    result = extract_paper_info(work_item)

    # Should take first element from arrays
    assert result["title"] == "First Title"
    assert result["journal"] == "Primary Journal"
    assert result["doi"] == "10.1234/array-test"


def test_extract_paper_info_date_fallback():
    """Test extract_paper_info date fallback from print to online."""
    work_item_print_only = {
        "published-print": {"date-parts": [[2023, 3, 10]]},
        "DOI": "10.1234/print-only",
    }

    result = extract_paper_info(work_item_print_only)
    assert result["published_date"] == {"date-parts": [[2023, 3, 10]]}

    work_item_online_only = {
        "published-online": {"date-parts": [[2023, 2, 5]]},
        "DOI": "10.1234/online-only",
    }

    result = extract_paper_info(work_item_online_only)
    assert result["published_date"] == {"date-parts": [[2023, 2, 5]]}

    # Prefer print over online when both exist
    work_item_both = {
        "published-print": {"date-parts": [[2023, 4, 15]]},
        "published-online": {"date-parts": [[2023, 3, 1]]},
        "DOI": "10.1234/both-dates",
    }

    result = extract_paper_info(work_item_both)
    assert result["published_date"] == {"date-parts": [[2023, 4, 15]]}


def test_extract_paper_info_empty_arrays():
    """Test extract_paper_info with empty arrays."""
    work_item = {
        "title": [],
        "author": [],
        "container-title": [],
        "DOI": "10.1234/empty-arrays",
    }

    result = extract_paper_info(work_item)

    assert result["title"] == ""
    assert result["authors"] == []
    assert result["journal"] == ""
    assert result["doi"] == "10.1234/empty-arrays"


def test_extract_paper_info_exception_handling():
    """Test extract_paper_info handles exceptions gracefully."""
    # Pass something that will cause an exception during processing
    invalid_work_item = {
        "title": "not_a_list",  # This should be a list
        "author": "not_a_list_either",  # This should be a list too
    }

    result = extract_paper_info(invalid_work_item)

    # Should return empty dict on exception
    assert result == {}


def test_extract_paper_info_none_values():
    """Test extract_paper_info with None values."""
    work_item = {
        "title": None,
        "author": None,
        "container-title": None,
        "DOI": "10.1234/none-values",
    }

    result = extract_paper_info(work_item)

    # Function returns empty dict on exception with None values
    assert result == {}


# Error handling tests for meaningful edge cases
def test_get_doi_metadata_json_decode_error():
    """Test get_doi_metadata handles malformed JSON responses."""
    from unittest.mock import Mock, patch

    # Mock a response that raises JSONDecodeError
    mock_response = Mock()
    mock_response.raise_for_status.return_value = None
    mock_response.json.side_effect = ValueError("Invalid JSON")

    with patch("requests.get", return_value=mock_response):
        # This should raise the exception as the code re-raises unexpected errors
        with pytest.raises(ValueError, match="Invalid JSON"):
            get_doi_metadata("10.1234/test")


def test_extract_doi_from_url_edge_cases():
    """Test URL DOI extraction with various edge cases."""
    # These test real scenarios that could occur
    test_cases = [
        ("", None),  # Empty string
        ("not_a_url", None),  # Plain text
        ("https://example.com", None),  # URL without DOI
        ("https://doi.org/", None),  # DOI URL without actual DOI
        ("dx.doi.org/10.1234/test", "10.1234/test"),  # Without protocol
        (
            "http://dx.doi.org/10.1234/test?param=value",
            "10.1234/test",
        ),  # With query params
        ("https://doi.org/10.1234/test#fragment", "10.1234/test"),  # With fragment
    ]

    for url, expected in test_cases:
        result = extract_doi_from_url(url)
        assert result == expected, f"Failed for URL: {url}"


@pytest.mark.external_api
def test_get_abstract_from_pubmed_id_invalid_input():
    """Test abstract retrieval with invalid PMID inputs."""
    # Test with clearly invalid PMIDs
    invalid_pmids = [
        "",  # Empty string
        "not_a_number",  # Non-numeric
        "0",  # Zero (invalid PMID)
        "-123",  # Negative number
        "999999999999999999",  # Unreasonably large number
    ]

    for pmid in invalid_pmids:
        result = get_abstract_from_pubmed_id(pmid)
        # Function may return None for invalid PMIDs or dict with content
        if result is not None:
            assert isinstance(result, dict)
            assert "content" in result
            assert "saved_to" in result
            assert "truncated" in result
            assert isinstance(result["content"], str)
        # The function gracefully handles invalid inputs


def test_clean_text_edge_cases():
    """Test text cleaning with edge cases."""
    # Test valid string inputs
    valid_cases = [
        "",  # Empty string
        "   ",  # Only whitespace
        "normal text",  # Normal case
    ]

    test_email = get_test_email()
    for text_input in valid_cases:
        result = clean_text(text_input, test_email)
        assert isinstance(result, dict)
        assert "content" in result
        assert "saved_to" in result
        assert "truncated" in result
        assert isinstance(result["content"], str)

    # Test None input separately - this actually returns None
    result = clean_text(None, test_email)
    assert result is None


def test_get_full_text_info_invalid_doi():
    """Test full text info with invalid DOI format."""
    invalid_dois = [
        "",  # Empty
        "not-a-doi",  # Invalid format
        "10.1234",  # Incomplete DOI
        None,  # None input
    ]

    test_email = get_test_email()
    for doi in invalid_dois:
        try:
            result = get_full_text_info(doi, test_email)
            # Should return None for invalid DOIs
            assert result is None, f"Should return None for invalid DOI: {doi}"
        except Exception:
            # Some invalid inputs might raise exceptions, which is acceptable
            pass
