"""
Pydantic model for team information in user context.

Contains the TeamInfo model that represents team data available to a user
within a specific organization context.
"""

from typing import List, Optional

from pydantic import BaseModel, Field


class TeamInfo(BaseModel):
    """
    Information about a team the user has access to in the current organization.

    This model represents a single team membership, including the user's role
    and permissions within that team. Used as part of the StytchContext to
    provide stateless team access validation.
    """

    id: str = Field(
        ...,
        description="MongoDB ObjectId of the team (as string)",
        json_schema_extra={"example": "507f1f77bcf86cd799439011"},
    )
    name: str = Field(
        ...,
        description="Display name of the team",
        json_schema_extra={"example": "Marketing Team"},
    )
    organization_id: str = Field(
        ...,
        description="MongoDB ObjectId of the parent organization (as string)",
        json_schema_extra={"example": "507f1f77bcf86cd799439012"},
    )
    role: Optional[str] = Field(
        None,
        description="User's role within this team (e.g., 'admin', 'member', 'viewer')",
        json_schema_extra={"example": "admin"},
    )
    permissions: List[str] = Field(
        default_factory=list,
        description="Team-scoped permissions the user has (e.g., ['read', 'write', 'delete'])",
        json_schema_extra={"example": ["read", "write", "delete"]},
    )

    model_config = {
        # Allow both snake_case and camelCase for API compatibility
        "populate_by_name": True,
        "json_schema_extra": {
            "example": {
                "id": "507f1f77bcf86cd799439011",
                "name": "Marketing Team",
                "organization_id": "507f1f77bcf86cd799439012",
                "role": "admin",
                "permissions": ["read", "write", "delete"],
            }
        },
    }

    def __str__(self) -> str:
        """String representation for logging."""
        return f"TeamInfo(id={self.id}, name={self.name}, role={self.role})"

    def __repr__(self) -> str:
        """Detailed representation for debugging."""
        return (
            f"TeamInfo(id={self.id!r}, name={self.name!r}, "
            f"organization_id={self.organization_id!r}, "
            f"role={self.role!r}, permissions={self.permissions!r})"
        )
