import subprocess
import sys
import re

def get_latest_version(pkg: str) -> str:
    """nimble dump <pkg> 로 최신 버전 가져오기"""
    try:
        result = subprocess.run(
            ["nimble", "dump", pkg],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True
        )
        match = re.search(r'version\s*=\s*"([^"]+)"', result.stdout)
        return match.group(1) if match else "0.1.0"
    except:
        return "0.1.0"


def add_to_nimble(nimble_file: str, pkg: str, version: str):
    with open(nimble_file, "r") as f:
        content = f.read()

    # 이미 있는지 체크
    if f'requires "{pkg}' in content:
        print(f'[skip] "{pkg}" is already in {nimble_file}')
        return

    # 마지막 requires 아래에 넣기
    insert_text = f'requires "{pkg} >= {version}"'
    if "requires" in content:
        content = content.rstrip() + "\n" + insert_text + "\n"
    else:
        content = content.rstrip() + "\n\n" + insert_text + "\n"

    with open(nimble_file, "w") as f:
        f.write(content)

    print(f"[added] {insert_text}")


def nimble_add(pkg: str, nimble_file: str):
    print(f"[1/3] Installing package: {pkg}")
    subprocess.run(["nimble", "install", pkg, "-y"], check=True)

    print(f"[2/3] Fetching version...")
    version = get_latest_version(pkg)

    print(f"[3/3] Updating {nimble_file}")
    add_to_nimble(nimble_file, pkg, version)

    print("[done] Completed!")


def main():
    """Entry point for the CLI command."""
    if len(sys.argv) < 3:
        print("Usage: nimble_easy <package_name> <nimble_file>")
        sys.exit(1)

    pkg = sys.argv[1]
    nimble_file = sys.argv[2]
    nimble_add(pkg, nimble_file)


if __name__ == "__main__":
    main()
