# Changelog

All notable changes to this project will be documented in this file.

The format is based on [Keep a Changelog](https://keepachangelog.com/en/1.0.0/),
and this project adheres to [Semantic Versioning](https://semver.org/spec/v2.0.0.html).

## [Unreleased]

### Added
- Future enhancements and features will be listed here

## [0.4.0] - 2025-01-15

### 🚀 Enhanced SQL Method Integration (Phase 0.2)

### Added
- 🔗 **Fluent SQL API** with method chaining support:
  - `select()`, `where()`, `group_by()`, `order_by()` methods on ParquetFrame
  - Complete SQL query building with method chaining: `pf.select().where().group_by().execute()`
  - SQLBuilder class for complex query construction
- ⚡ **Query Performance Optimization**:
  - Query result caching with configurable cache size and automatic management
  - Execution profiling with timing, memory usage, and metadata tracking
  - QueryResult dataclass with convenience properties (`rows`, `columns`, `cached`, `dataframe`)
- 🔧 **Query Utilities and Builder Patterns**:
  - `parameterize_query()` function for safe parameter substitution
  - `sql_with_params()` method for parameterized queries with {param} syntax
  - `build_join_query()` utility for programmatic SQL construction
- 🤝 **Enhanced JOIN Operations**:
  - Convenience JOIN methods: `left_join()`, `right_join()`, `inner_join()`, `full_join()`
  - Proper table aliasing and JOIN syntax handling
  - Support for complex multi-table JOINs with chaining
- 📊 **Direct DataFrame Access**:
  - Added `pandas_df` property for direct access to underlying pandas DataFrame
  - Improved type annotations with proper TYPE_CHECKING imports

### Enhanced
- 🗃️ **Multi-Format SQL Integration** - Verified SQL queries work seamlessly across CSV, JSON, ORC, and Parquet formats
- 🎯 **Backward Compatibility** - All existing `sql()` method functionality preserved
- 🛡️ **Error Handling** - Enhanced parameter validation and missing parameter detection
- 🧪 **Comprehensive Testing** - 27 new tests covering fluent API, profiling, caching, and utilities

### Examples

**Fluent SQL API:**
```python
result = (pf.select("name", "age", "salary")
         .where("age > 25")
         .group_by("department")
         .having("COUNT(*) > 1")
         .order_by("salary", "DESC")
         .limit(10)
         .execute())
```

**Parameterized Queries:**
```python
result = pf.sql_with_params(
    "SELECT * FROM df WHERE age > {min_age} AND salary < {max_salary}",
    min_age=25, max_salary=100000
)
```

**Enhanced JOINs:**
```python
result = (pf.select("df.name", "dept.name")
         .left_join(departments, "df.dept_id = dept.id", "dept")
         .where("dept.budget > 500000")
         .execute())
```

**Query Profiling:**
```python
result = (pf.select("COUNT(*) as total")
         .profile(True)
         .cache(True)
         .execute())

print(f"Executed in {result.execution_time:.3f}s")
print(f"Cached: {result.cached}")
```

### Technical Details
- Implemented SQLBuilder class with full SQL clause support
- Added query result caching with SHA256-based cache keys
- Enhanced type safety with proper forward references
- Memory-efficient caching with automatic size management
- Thread-safe implementation using immutable cache keys
- Comprehensive error handling for SQL syntax and parameter validation

## [0.2.3.2] - 2025-09-27

### 🐛 Additional AI Interactive Mode Fixes

### Fixed
- 🔄 **Async Event Loop Conflict** - Fixed "asyncio.run() cannot be called from a running event loop" error in AI confirm dialogs
- 🎯 **AI Query Execution** - Resolved blocking issue preventing AI-generated queries from executing properly
- 📋 **Interactive Workflow** - Enabled complete AI query confirmation and execution workflow in interactive mode

### Technical Details
- Used `asyncio.to_thread()` to run `confirm()` function in separate thread to avoid event loop conflicts
- Fixed async context handling in interactive AI command processing
- Maintained user confirmation functionality while resolving async execution issues

## [0.2.3.1] - 2025-01-27

### 🐛 Critical AI Interactive Mode Hotfix

### Fixed
- 🤖 **AI Interactive Command Bug** - Fixed `confirm()` function call in interactive mode that used unsupported `default=True` parameter
- 🔍 **Model Availability Check** - Improved Ollama model parsing to handle different response formats and `:latest` tags
- 📦 **AI Dependencies** - Enhanced error handling for missing Python `ollama` package in AI functionality
- 🎯 **Interactive Experience** - Resolved crashes when using `\ai` commands in ParquetFrame interactive CLI
- 📝 **Save-Script Command** - Fixed "Invalid value NaN (not a number)" errors in `\save-script` command
- 🗄️ **History Export** - Resolved JSON serialization issues with DataFrame NaN values in session export

### Technical Details
- Removed unsupported `default` parameter from `prompt_toolkit.shortcuts.confirm()` calls
- Enhanced model availability detection with better JSON parsing and tag normalization
- Added clearer error messages and user guidance for AI setup requirements
- Verified AI query execution and response generation in interactive mode
- Fixed DataFrame to dict conversion by replacing NaN values with None using `pandas.where()`
- Improved history manager's export functionality to handle missing/null values properly

## [0.2.3] - 2025-09-26

### 🛠️ CI/CD Fixes and Test Stability Release

### Fixed
- 🐛 **Windows CI Compatibility** - Skip interactive tests on Windows CI to handle NoConsoleScreenBufferError
- 🧪 **Schema Mismatch Handling** - Added union_by_name=True to DuckDB read_parquet for mismatched schemas
- 🔍 **LLM Agent Tests** - Fixed test mocking of OLLAMA_AVAILABLE flag for proper dependency handling
- ⚠️ **Factory Validation** - Improved DataContextFactory parameter validation for None handling
- 🔡 **Encoding Issues** - Fixed Unicode encoding problems in CI workflows by removing emojis
- 🎯 **Test Coverage** - Maintained 55%+ test coverage across the codebase

### Enhanced
- 🧩 **Optional Dependency Handling** - Improved installation and validation of bioframe, SQLAlchemy
- 📝 **Error Messages** - Enhanced clarity of error messages for missing dependencies
- ⚡ **Test Reliability** - Ensured consistent test behavior across all platforms
- 🔄 **CI Workflow** - Optimized CI process with explicit dependency verification

### Tests
- ✅ **Cross-platform Testing** - Ensured tests run consistently on macOS, Linux, and Windows
- 🛡️ **Edge Case Handling** - Improved robustness for different CI environments
- 🧠 **Dependency Checking** - Better skip mechanisms for tests that require optional packages

## [0.2.2] - 2025-01-26

### 🚀 Enhanced Features & Documentation Release

### Added
- 🗃️ **SQL Support via DuckDB** with `.sql()` method and `pframe sql` CLI command
- 🧬 **BioFrame Integration** with `.bio` accessor supporting cluster, overlap, merge, complement, closest
- 🤖 **AI-Powered Data Exploration** with natural language to SQL conversion using local LLM (Ollama)
- 📊 **Performance Benchmarking Suite** with comprehensive analysis and CLI integration
- 🔄 **YAML Workflow Engine** for declarative data processing pipelines
- 🗄️ **DataContext Framework** for unified access to parquet files and databases
- 📈 **Workflow Visualization** and history tracking capabilities
- ➕ **Optional Extras**: `[sql]`, `[bio]`, `[ai]`, and `[all]` for easy installation of feature sets

### Enhanced
- 🧠 **Intelligent Backend Switching** with memory pressure analysis and file characteristic detection
- 🎨 **Rich CLI Experience** with enhanced interactive mode and comprehensive help
- 🔍 **Advanced Error Handling** with detailed exception hierarchy and user-friendly messages
- 📚 **Comprehensive Documentation** with architecture guides, AI features documentation, and examples
- 🧪 **Expanded Test Suite** with 334 passing tests across multiple categories (54% coverage)
- ⚡ **Performance Optimizations** showing 7-90% speed improvements over direct pandas usage

### Changed
- 📋 **CLI Updated** to include SQL commands, interactive SQL mode, and AI-powered queries
- 🔧 **Architecture Refactored** with dependency injection and factory patterns
- 📖 **Documentation Structure** enhanced with detailed guides and API references

### Fixed
- 🛠️ **CI/CD Pipeline** improvements and cross-platform compatibility
- 🐛 **Test Stability** across different Python versions and operating systems
- 🔍 **Memory Management** with intelligent threshold adjustment

### Tests
- ✅ **Comprehensive Test Coverage** for SQL, bioframe, AI, and workflow functionality
- 🧪 **Integration Tests** for end-to-end workflows and real-world scenarios
- 🔄 **Performance Tests** with benchmarking validation
- 🤖 **AI Integration Tests** with mock-based LLM testing

## [0.2.1] - 2025-01-25

### 🎉 First PyPI Release
- ✅ **Successfully Published to PyPI**: Package now available via `pip install parquetframe`
- 🔒 **Trusted Publishing Configured**: Secure automated releases without API tokens
- 📦 **GitHub Releases**: Automatic release creation with downloadable artifacts
- 🚀 **Full CI/CD Pipeline**: Comprehensive testing, building, and publishing automation

### Improved
- 📦 **Release Pipeline** - Enhanced GitHub Actions workflow with trusted PyPI publishing
- 🔧 **Package Metadata** - Updated classifiers and keywords for better PyPI discovery
- 📚 **Documentation** - Added comprehensive release process documentation

### Fixed
- 🛠️ Fixed PyPI trusted publishing configuration in release workflow
- 📋 Updated package status to Beta (Development Status :: 4)

### Enhanced
- 🖥️ **Complete CLI Interface** with three main commands (`info`, `run`, `interactive`)
- 🎨 **Rich Terminal Output** with beautiful tables and color formatting
- 🐍 **Interactive Python REPL** mode with full ParquetFrame integration
- 📝 **Automatic Script Generation** from CLI sessions for reproducibility
- 🔍 **Advanced Data Exploration** with query filters, column selection, and previews
- 📊 **Statistical Operations** directly from command line (describe, info, sampling)
- ⚙️ **Backend Control** with force pandas/Dask options in CLI
- 📁 **File Metadata Display** with schema information and recommendations
- 🔄 **Session History Tracking** with persistent readline support
- 🎯 **Batch Data Processing** with output file generation

### Enhanced
- ✨ **ParquetFrame Core** with indexing support (`__getitem__`, `__len__`)
- 🔧 **Attribute Delegation** with session history recording
- 📋 **CI/CD Pipeline** with dedicated CLI testing jobs
- 📖 **Documentation** with comprehensive CLI usage examples
- 🧪 **Test Coverage** expanded to include CLI functionality

### CLI Commands
- `pframe info <file>` - Display file information and schema
- `pframe run <file> [options]` - Batch data processing with extensive options
- `pframe interactive [file]` - Start interactive Python session with ParquetFrame

### CLI Options
- Data filtering with `--query` pandas/Dask expressions
- Column selection with `--columns` for focused analysis
- Preview options: `--head`, `--tail`, `--sample` for data exploration
- Statistical analysis: `--describe`, `--info` for data profiling
- Output control: `--output`, `--save-script` for results and reproducibility
- Backend control: `--force-pandas`, `--force-dask`, `--threshold`

## [0.1.1] - 2024-09-24

### Fixed
- 🐛 **Critical Test Suite Stability** - Resolved 29 failing tests, bringing test suite to 100% passing (203 tests)
- 🔧 **Dependency Issues** - Added missing `psutil` dependency for memory monitoring and system resource detection
- ⚠️ **pandas Deprecation** - Replaced deprecated `pd.np` with direct `numpy` imports throughout codebase
- 📅 **DateTime Compatibility** - Updated deprecated pandas frequency 'H' to 'h' for pandas 2.0+ compatibility
- 🔄 **Backend Switching Logic** - Fixed explicit `islazy` parameter override handling to ensure manual control works correctly
- 🗂️ **Directory Creation** - Enhanced `save()` method to automatically create parent directories when saving files
- 🔍 **Parameter Validation** - Added proper validation for `islazy` and `npartitions` parameters with clear error messages
- 📊 **Data Type Preservation** - Improved pandas/Dask dtype consistency to prevent conversion issues
- 🌐 **URL Path Support** - Enhanced path handling to support remote files and URLs
- 🖥️ **CLI Output** - Fixed CLI row limiting (head/tail/sample) operations to work correctly before saving
- ⚖️ **Memory Estimation** - Updated unrealistic memory threshold tests to use practical values
- 🔗 **Method Chaining** - Updated tests to handle pandas operations that return pandas objects vs ParquetFrame objects
- 📈 **Benchmark Tests** - Fixed division-by-zero errors in benchmark summary calculations
- 🎯 **Edge Case Handling** - Improved handling of negative parameters, invalid types, and boundary conditions

### Improved
- 📊 **Test Coverage** - Increased from 21% to 65% with comprehensive test improvements
- ⚡ **Test Suite Performance** - All 203 tests now pass reliably with consistent results
- 🛡️ **Error Handling** - Enhanced validation and error messages throughout the codebase
- 📝 **Code Quality** - Fixed various edge cases and improved robustness of core functionality

### Technical Details
- Fixed `psutil` import issues in benchmarking module
- Resolved pandas `pd.np` deprecation across multiple modules
- Enhanced `ParquetFrame.save()` with automatic directory creation
- Improved `islazy` parameter validation and override logic
- Fixed CLI test assertions to match actual output messages
- Added proper handling for URL-based file paths
- Resolved memory estimation test threshold issues
- Fixed benchmark module mock expectations and verbose flag handling
- Improved test data generation to avoid pandas errors with mismatched array lengths

## [0.1.0] - 2024-09-24

### Added
- 🎉 **Initial release of ParquetFrame**
- ✨ **Automatic pandas/Dask backend selection** based on file size (default 10MB threshold)
- 📁 **Smart file extension handling** for parquet files (`.parquet`, `.pqt`)
- 🔄 **Seamless conversion** between pandas and Dask DataFrames (`to_pandas()`, `to_dask()`)
- ⚡ **Full API compatibility** with pandas and Dask operations through transparent delegation
- 🎯 **Zero configuration** - works out of the box with sensible defaults
- 🧪 **Comprehensive test suite** with 95%+ coverage (410+ tests)
- 📚 **Complete documentation** with MkDocs, API reference, and examples
- 🔧 **Modern development tooling** (ruff, black, mypy, pre-commit hooks)
- 🚀 **CI/CD pipeline** with GitHub Actions for testing and PyPI publishing
- 📦 **Professional packaging** with hatchling build backend

### Features
- `ParquetFrame` class with automatic backend selection
- Convenience functions: `read()`, `create_empty()`
- Property-based backend switching with `islazy` setter
- Method chaining support for data pipeline workflows
- Comprehensive error handling and validation
- Support for all pandas/Dask parquet reading options
- Flexible file path handling (Path objects, relative/absolute paths)
- Memory-efficient processing for large datasets

### Testing
- Unit tests for all core functionality
- Integration tests for backend switching logic
- I/O format tests for compression and data types
- Edge case and error handling tests
- Platform-specific and performance tests
- Test fixtures for various DataFrame scenarios

### Documentation
- Complete user guide with installation, quickstart, and usage examples
- API reference with automatic docstring generation
- Real-world examples for common use cases
- Performance optimization tips
- Contributing guidelines and development setup

[0.1.0]: https://github.com/leechristophermurray/parquetframe/releases/tag/v0.1.0
