from .pdu_channel_config import PduChannelConfig
from .hako_binary import offset_map
from .hako_binary import binary_writer
from .hako_binary import binary_reader

class PduConvertor:
    """
    PduConvertor handles conversion between JSON dictionaries and binary PDU data.

    It uses offset definitions (usually generated by hakoniwa tools) to convert
    structured JSON data to raw binary representations and vice versa.

    Args:
        offset_path (str): Path to the offset files used for binary layout.
        pdu_channel_config (PduChannelConfig): PDU channel definition configuration.
    """    
    def __init__(self, offset_path: str, pdu_channel_config: PduChannelConfig):
        """
        Initialize the PduConvertor.

        Args:
            offset_path (str): Directory path to the offset map files (.offset).
            pdu_channel_config (PduChannelConfig): Configuration object describing PDU layout.
        """
        self.pdu_channel_config = pdu_channel_config
        self.offmap = offset_map.create_offmap(offset_path)

    def create_empty_pdu_json(self, robot_name: str, pdu_name: str) -> dict:
        """
        Create a blank JSON dictionary based on the offset definition and PDU size.

        This is typically used to generate an initial data structure before writing to PDU.

        Args:
            robot_name (str): The name of the robot.
            pdu_name (str): The name of the PDU.

        Returns:
            dict: A JSON-like dictionary representing a zero-initialized PDU structure.

        Raises:
            ValueError: If the PDU size or type is not defined.
        """        
        pdu_size = self.pdu_channel_config.get_pdu_size(robot_name, pdu_name)
        if pdu_size < 0:
            raise ValueError(f"PDU size for {robot_name}/{pdu_name} is not defined.")
        pdu_type = self.pdu_channel_config.get_pdu_type(robot_name, pdu_name)
        if pdu_type is None:
            raise ValueError(f"PDU type for {robot_name}/{pdu_name} is not defined.")
        binary_data = bytearray(pdu_size)
        value = binary_reader.binary_read(self.offmap, pdu_type, binary_data)
        return value

    def convert_json_to_binary(self, robot_name: str, pdu_name: str, json_data: dict) -> bytearray:
        """
        Convert JSON data into binary PDU representation.

        Args:
            robot_name (str): The name of the robot.
            pdu_name (str): The name of the PDU.
            json_data (dict): A dictionary containing the data to serialize.

        Returns:
            bytearray: The binary representation of the PDU.

        Raises:
            ValueError: If the PDU size or type is not defined.
        """
        pdu_size = self.pdu_channel_config.get_pdu_size(robot_name, pdu_name)
        if pdu_size < 0:
            raise ValueError(f"PDU size for {robot_name}/{pdu_name} is not defined.")
        pdu_type = self.pdu_channel_config.get_pdu_type(robot_name, pdu_name)
        if pdu_type is None:
            raise ValueError(f"PDU type for {robot_name}/{pdu_name} is not defined.")
        
        binary_data = bytearray(pdu_size)
        binary_writer.binary_write(self.offmap, binary_data, json_data, pdu_type)
        return binary_data
    
    def convert_binary_to_json(self, robot_name: str, pdu_name: str, binary_data: bytearray) -> dict:
        """
        Convert binary PDU data into a JSON dictionary representation.

        Args:
            robot_name (str): The name of the robot.
            pdu_name (str): The name of the PDU.
            binary_data (bytearray): The binary data received or read from the channel.

        Returns:
            dict: A dictionary representing the deserialized PDU content.

        Raises:
            ValueError: If the PDU type is not defined.
        """
        #print(f"[DEBUG] Converting binary data for {robot_name}/{pdu_name} of size {len(binary_data)}")
        pdu_type = self.pdu_channel_config.get_pdu_type(robot_name, pdu_name)
        #print(f"[DEBUG] PDU type for {robot_name}/{pdu_name}: {pdu_type}")
        if pdu_type is None:
            raise ValueError(f"PDU type for {robot_name}/{pdu_name} is not defined.")
        
        json_data = binary_reader.binary_read(self.offmap, pdu_type, binary_data)
        return json_data
    