"""European map implemented in SVG."""
# Author(s): Davide.De-Marchi@ec.europa.eu
# Copyright © European Union 2022-2023
# 
# Licensed under the EUPL, Version 1.2 or as soon they will be approved by 
# the European Commission subsequent versions of the EUPL (the "Licence");
# 
# You may not use this work except in compliance with the Licence.
# 
# You may obtain a copy of the Licence at:
# https://joinup.ec.europa.eu/collection/eupl/eupl-text-eupl-12

# Unless required by applicable law or agreed to in writing, software
# distributed under the Licence is distributed on an "AS IS"
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied.
# 
# See the Licence for the specific language governing permissions and
# limitations under the Licence.
import pandas as pd
import re

try:
    from . import colors
    from .vuetify import fontsettings
except:
    import colors
    from vuetify import fontsettings


# List of all the codes
country_codes = ['LV','AL','CH','AT','HR','ES','IS','RO','IT','VA','HU','SE','NL','CZ','DE','FR','ME','BE','SI','LU','IE','BA','MC','BG','PL','LI','GB','RS','SM','DK','IM',
                 'EE','SK','EL','LT','NO','PT','AD','MK','MT','GI','FI','XK','CY']
"""
List of 2 characters strings: List of all the codes for the countries managed by the svgMap module, following the EUROSTAT country codes.

See `EUROSTAT country codes <https://ec.europa.eu/eurostat/statistics-explained/index.php?title=Glossary:Country_codes for the EUROSTAT codes>`_
"""

country_name = {
    'LV': 'Latvia',
    'AL': 'Albania',
    'CH': 'Switzerland',
    'AT': 'Austria',
    'HR': 'Croatia',
    'ES': 'Spain',
    'IS': 'Iceland',
    'RO': 'Romania',
    'IT': 'Italy',
    'VA': 'Vatican',
    'HU': 'Hungary',
    'SE': 'Sweden',
    'NL': 'Netherlands',
    'CZ': 'Czechia',
    'DE': 'Germany',
    'FR': 'France',
    'ME': 'Montenegro',
    'BE': 'Belgium',
    'SI': 'Slovenia',
    'LU': 'Luxembourg',
    'IE': 'Ireland',
    'BA': 'Bosnia and Herzegovina',
    'MC': 'Monaco',
    'BG': 'Bulgaria',
    'PL': 'Poland',
    'LI': 'Liechtenstein',
    'GB': 'United Kingdom',
    'RS': 'Republic of Serbia',
    'SM': 'San Marino',
    'DK': 'Denmark',
    'IM': 'Isle of Man',
    'EE': 'Estonia',
    'SK': 'Slovakia',
    'EL': 'Greece',
    'LT': 'Lithuania',
    'NO': 'Norway',
    'PT': 'Portugal',
    'AD': 'Andorra',
    'MK': 'North Macedonia',
    'MT': 'Malta',
    'GI': 'Gibraltar',
    'FI': 'Finland',
    'XK': 'Kosovo',
    'CY': 'Cyprus'
}
"""
dict with key=EUROSTAT code of countries, value=name of the country.
"""



###########################################################################################################################################################################
# Utility: Returns a modified SVG string by substituting the original width with the width passed as argument
###########################################################################################################################################################################
def svgChangeWidth(svgstring, width='500px'):
    return re.sub('width="\w*"', 'width="%s"'%width, svgstring, 1)


###########################################################################################################################################################################
# Utility: Returns a modified SVG string by substituting the original width with the width passed as argument
###########################################################################################################################################################################
def svgChangeHeight(svgstring, height='600px'):
    return re.sub('height="\w*"', 'height="%s"'%height, svgstring, 1)


###########################################################################################################################################################################
# Return an SVG for the Map of Europe
###########################################################################################################################################################################
def svgMapEurope(df,                          # Pandas dataframe indexed on iso2_code and containing 'value' and 'label' columns
                 code_column=None,            # Name of the column containing the code of the country (None = the country code is in the index of the dataframe)
                 value_column='value',        # Name of the column containing the value
                 label_column='label',        # Name of the column containing the label
                 codes_selected=[],           # codes of the countries selected
                 width='400px',               # width of the drawing
                 height='600px',              # height of the drawing
                 colorlist=['#0d0887', '#46039f', '#7201a8', '#9c179e', '#bd3786', '#d8576b', '#ed7953', '#fb9f3a', '#fdca26', '#f0f921'],   # default color scale
                 stdevnumber=2.0,             # Number of stddev to calculate (minvalue,maxvalue) range
                 fill='#f1f1f1',              # fill color for countries
                 stroke='#232323',            # stroke color for countries border
                 stroke_selected='#00ffff',   # stroke color for border of selected country
                 stroke_width=3.0,            # border width for countries polygons
                 onhoverfill='yellow',        # Color for highlighting countries on hover
                 decimals=2,                  # Number of decimals for the legend number display
                 minallowed_value=None,       # Minimum value allowed
                 maxallowed_value=None,       # Maximum value allowed
                 hoveronempty=False,          # If True highlights polygon on hover even if no value present in input df for the polygon
                 legendtitle='',              # Title to add to the legend (top)
                 legendunits='',              # Units of measure to add to the legend (bottom)
                 bordercolor='black',         # Color for lines and rects
                 textcolor='black',           # Color for texts
                 dark=False):                 # Dark mode
    """
    Static map of European countries with color legend obtained by joining with a Pandas DataFrame.
    
    Parameters
    ----------
    df : Pandas DataFrame
        Pandas DataFrame to use for assigning values to the countries. It has to contain at least a column with numeric values.
    code_column : str, optional
        Name of the column of the Pandas DataFrame containing the unique code of the countries in the EUROSTAT Country Codes standard (see https://ec.europa.eu/eurostat/statistics-explained/index.php?title=Glossary:Country_codes). This column is used to perform the join with the internal attribute of the countries vector dataset that contains the country code. If the code_column is None, the code is taken from the index of the DataFrame, (default is None)
    value_column : str, optional
        Name of the column of the Pandas DataFrame containing the values to be assigned to the countries using the join on the EUROSTAT Country Codes (default is 'value')
    codes_selected : list of strings, optional
        List of codes of countries to display as selected (default is [])
    width : str, optional
        Width of the map (default is '400px')
    height : str, optional
        Height of the map (default is '600px')
    colorlist : list of colors, optional
        List of colors to assign to the country polygons (default is the Plotly px.colors.sequential.Plasma, see `Plotly sequential color scales <https://plotly.com/python/builtin-colorscales/#builtin-sequential-color-scales>`_ and `Plotly qualitative color sequences <https://plotly.com/python/discrete-color/#color-sequences-in-plotly-express>`_ )
    stdevnumber : float, optional
        The correspondance between the values assigned to country polygons and the colors list is done by calculating a range of values [min,max] to linearly map the values to the colors. This range is defined by calculating the mean and standard deviation of the country values and applying this formula [mean - stdevnumber*stddev, mean + stdevnumber*stddev]. Default is 2.0
    fill : str, optional
        Fill color to use for the  countries that are not joined (default is '#f1f1f1')
    stroke : str, optional
        Color to use for the border of countries (default is '#232323')
    stroke_selected : str, optional
        Color to use for the border of the selected countries (default is '#00ffff')
    stroke_width: float, optional
        Width of the border of the country polygons in pixels (default is 3.0)
    onhoverfill : str, optional
        Color for highlighting countries on hover (default is 'yellow')
    decimals : int, optional
        Number of decimals for the legend numbers display (default is 2)
    minallowed_value : float, optional
        Minimum value allowed, to force the calculation of the [min,max] range to map the values to the colors
    maxallowed_value : float, optional
        Maximum value allowed, to force the calculation of the [min,max] range to map the values to the colors
    hoveronempty : bool, optional
        If True highlights polygon on hover even if no value present in input df for the polygon (default is False)
    legendtitle : str, optional
        Title to add on top of the legend (default is '')
    legendunits : str, optional
        Units of measure to add to the bottom of the legend (default is '')
    bordercolor : str, optional
        Color for lines and rects of the legend (default is 'black')
    textcolor : str, optional
        Color for texts of the legend (default is 'black')
    dark : bool, optional
        If True, the bordercolor and textcolor are set to white (default is False)
        
    Returns
    -------
        a string containing SVG text to display the map of European countries
        
    Example
    -------
    Map of European countries joined with random values in [0,100]::
    
        import numpy as np
        import pandas as pd
        import plotly.express as px
        from vois import svgMap

        countries = svgMap.country_codes

        # Generate random values and create a dictionary: key=countrycode, value=random in [0.0,100.0]
        d = dict(zip(countries, list(np.random.uniform(size=len(countries),low=0.0,high=100.0))))

        # Create a pandas dataframe from the dictionary
        df = pd.DataFrame(d.items(), columns=['iso2code', 'value'])

        svg = svgMap.svgMapEurope(df,
                                  code_column='iso2code',
                                  width='650px',
                                  height='800px',
                                  stdevnumber=1.5, 
                                  colorlist=px.colors.sequential.Viridis,
                                  stroke_width=4.0,
                                  stroke_selected='red',
                                  onhoverfill='#f8bd1a',
                                  codes_selected=['IT'],
                                  legendtitle='Legent title',
                                  legendunits='KTOE per 100K inhabit.')
        display(HTML(svg))

    .. figure:: figures/svgMap.png
       :scale: 100 %
       :alt: svgMap example

       Example of a static map of European countries created as an SVG drawing
       
    """

    if df.shape[0] <= 0:
        minvalue = 1.0
        maxvalue = 2.0
    else:
        mean = df[value_column].mean()
        if df.shape[0] <= 1:
            minvalue = mean
            maxvalue = mean
        else:
            stddev = df[value_column].std()
            valuemin = df[value_column].min()
            valuemax = df[value_column].max()

            minvalue = mean - stdevnumber*stddev
            maxvalue = mean + stdevnumber*stddev

            if minvalue < valuemin: minvalue = valuemin
            if maxvalue > valuemax: maxvalue = valuemax

        if not minallowed_value is None:
            if minvalue < minallowed_value: minvalue = minallowed_value
        if not maxallowed_value is None:
            if maxvalue > maxallowed_value: maxvalue = maxallowed_value
        
    #print(mean,stddev, minvalue, maxvalue)

    if minvalue >= maxvalue: maxvalue = minvalue + 1
    ci = colors.colorInterpolator(colorlist,minvalue,maxvalue)

    if dark:
        if bordercolor=='black': bordercolor='white'
        if textcolor  =='black': textcolor  ='white'
    
    svg = '''
<svg
   xmlns:svg="http://www.w3.org/2000/svg"
   xmlns="http://www.w3.org/2000/svg"
   width="%s"
   height="%s"
   viewBox="0 0 3000 2480"
   version="1.1">

  <style type="text/css">
     @import url('%s');
      
     svg .country:hover path   { fill: %s; }

     svg .label {display: none; font-family: %s; fill: %s; }
     svg .country:hover .label { display: block; }

     svg .bar   {display: none; stroke: %s; stroke-width: 20; }
     svg .barselected  {stroke: %s; stroke-width: 20; }
     svg .country:hover .bar   { display: block; }
''' % (width, height, fontsettings.font_url, onhoverfill, fontsettings.font_name, textcolor, onhoverfill, stroke_selected)

   
    # Labels and colors indexed by iso2_code of countries
    labels     = {}    # Label for the polygons
    polycolors = {}    # Fill color of the polygon
    polyclass  = {}    # class to assign to the polygon ("country" or "")
    polybary   = {}    # Y coordinate of the bar on the legend highlighted when hover on a country
    polydash   = {}    # stroke-dasharray of the bar on the legend highlighted when hover on a country
    
    for c in country_codes:
        labels[c]     = country_name[c]
        polycolors[c] = fill
        polybary[c]   = -1000
        polydash[c]   = '1,0'   # continuous line
        if hoveronempty:
            polyclass[c] = 'country'
        else:
            polyclass[c] = ''
            
 
    # Positioning of the legend
    x1 = 2400
    w  = 200
    x2 = x1 + w
    
    y1 = 100
    h = 2200
    if len(legendunits) > 0:
        h -= 25
    y2 = y1 + h
    
    # Set colors for all the countries
    for index, row in df.iterrows():
        try:
            if code_column is None: code = index
            else:                   code = row[code_column]
            name  = country_name[code]
            value = row[value_column]
            polycolors[code] = ci.GetColor(value)
            polyclass[code]  = 'country'

            if label_column in df:
                v = row[label_column]
            else:
                v = str(value)
            labels[code] += ': ' + '{:.{prec}f}'.format(float(v), prec=decimals)
                
            y = y2 - (y2-y1) * (value - minvalue) / (maxvalue - minvalue)

            if y < y1:
                y = y1
                polydash[code] = '10,10'
                
            if y > y2:
                y = y2
                polydash[code] = '10,10'
                
            polybary[code] = y
        except:
            pass
        
    #print(minvalue,maxvalue, y1, y2)
        
    # Add color for every polygon
    #for c in country_codes:
    #    svg += 'svg #%s { fill: %s; }\n' % (c, polycolors[c])
        
    svg += '</style>'

    
    # Legend on the right
    if len(legendtitle) > 0:
        svg += '<text x="%d" y="%d" text-anchor="middle" font-size="54" font-family="%s" font-weight="bold" fill="%s">%s</text>' % (x1+w/2.0, y1-45, fontsettings.font_name, textcolor, legendtitle)
        
    if len(legendunits) > 0:
        svg += '<text x="%d" y="%d" text-anchor="middle" font-size="44" font-family="%s" font-weight="bold" fill="%s">%s</text>' % (x1+w/2.0, y2+110, fontsettings.font_name, textcolor, legendunits)
        
    svg += '<rect x="%d" y="%d" width="%d" height="%d" style="fill:none; stroke-width:10; stroke:%s;" />' % (x1, y1, w, h, bordercolor)
    
    y = y2
    for i in range(h):
        value = maxvalue - (y - y1) * (maxvalue - minvalue) / (y2 - y1)
        svg += '<line x1="%d" y1="%d" x2="%d" y2="%d" style="stroke:%s; stroke-width:2" />' % ( x1,y,x2,y, ci.GetColor(value) )
        y -= 1
        
    svg += '<line x1="%d" y1="%d" x2="%d" y2="%d" style="stroke:%s; stroke-width:4" />' % ( x2,y2,x2+30,y2, bordercolor )
    svg += '<line x1="%d" y1="%d" x2="%d" y2="%d" style="stroke:%s; stroke-width:4" />' % ( x2,y1-1,x2+30,y1-1, bordercolor )
    
    valmin = '{:.{prec}f}'.format(minvalue, prec=decimals)
    svg += '<text x="%d" y="%d" font-size="50" font-family="%s" fill="%s">%s</text>' % (x2+35, y2+18, fontsettings.font_name, textcolor, valmin)
    
    valmax = '{:.{prec}f}'.format(maxvalue, prec=decimals)
    svg += '<text x="%d" y="%d" font-size="50" font-family="%s" fill="%s">%s</text>' % (x2+35, y1+24, fontsettings.font_name, textcolor, valmax)
    
    valmed = '{:.{prec}f}'.format((minvalue+maxvalue)/2.0, prec=decimals)
    if valmed != valmin and valmed != valmax:
        y = (y1+y2)/2.0
        svg += '<line x1="%d" y1="%f" x2="%d" y2="%f" style="stroke:%s; stroke-width:4" />' % ( x2,y,x2+30,y, bordercolor )
        svg += '<text x="%d" y="%d" font-size="50" font-family="%s" fill="%s">%s</text>' % (x2+35, y+18, fontsettings.font_name, textcolor, valmed)
        
    val = '{:.{prec}f}'.format(minvalue + 3.0*(maxvalue-minvalue)/4.0, prec=decimals)
    if val != valmed and val != valmax:
        y = y1+(y2-y1)/4.0
        svg += '<line x1="%d" y1="%f" x2="%d" y2="%f" style="stroke:%s; stroke-width:4" />' % ( x2,y,x2+30,y, bordercolor )
        svg += '<text x="%d" y="%d" font-size="50" font-family="%s" fill="%s">%s</text>' % (x2+35, y+18, fontsettings.font_name, textcolor, val)
        
    val = '{:.{prec}f}'.format(minvalue + (maxvalue-minvalue)/4.0, prec=decimals)
    if val != valmin and val != valmed:
        y = y1+3.0*(y2-y1)/4.0
        svg += '<line x1="%d" y1="%f" x2="%d" y2="%f" style="stroke:%s; stroke-width:4" />' % ( x2,y,x2+30,y, bordercolor )
        svg += '<text x="%d" y="%d" font-size="50" font-family="%s" fill="%s">%s</text>' % (x2+35, y+18, fontsettings.font_name, textcolor, val)
    
    
    # Add horizontal lines in the legend for the selected countries
    for code in codes_selected:
        if code in country_codes:
            if polybary[code] >= y1 and polybary[code] <= y2:
                svg += '<text x="%d" y="%d" text-anchor="end" font-size="54" font-family="%s" font-weight="bold" fill="%s">%s</text>' % (x1-15.0, polybary[code]+18, fontsettings.font_name, bordercolor, code)
                svg += '<line class="barselected" stroke-dasharray="%s" x1="%d" y1="%d" x2="%d" y2="%d"/>' % (polydash[code], x1, polybary[code], x2, polybary[code])
    

    x1 -= 15.0
    x2 += 15.0
    
    svg_selected = ''
    stroke_replace = 'stroke="%s" stroke-width="%f"' % (stroke_selected, stroke_width*3.5)
    
    # Main group, setting color and border for all the polygons
    svg += '''
  <g fill="%s"
     stroke="%s"
     stroke-width="%f"
     stroke-linecap="square"
     stroke-linejoin="bevel"
     transform="matrix(0.999873,0,0,0.999873,0,0)"
     id="g1147">
''' % (fill, stroke, stroke_width)

   
    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1540.48,926.884 L1540.67,926.793 L1540.86,926.633 L1541.18,926.542 L1542.64,926.965 L1545.15,928.926 L1546.62,929.467 L1547.82,929.467 L1548.19,929.619 L1548.57,930.015 L1548.7,930.339 L1548.72,930.4 L1548.84,930.801 L1549.13,931.247 L1550.02,931.999 L1550.84,932.283 L1551.63,932.229 L1552.43,931.978 L1554.03,930.982 L1554.36,930.774 L1554.86,930.678 L1555.55,931.421 L1556.14,932.594 L1556.84,933.721 L1557.84,934.329 L1564.81,933.387 L1565.24,933.492 L1565.55,933.735 L1565.8,934.293 L1565.77,934.685 L1565.67,935.091 L1565.71,935.674 L1566.1,936.911 L1566.56,937.6 L1570.88,939.802 L1572.27,940.817 L1576.57,945.184 L1578,946.249 L1579.59,946.909 L1580.05,946.993 L1582.91,947.51 L1582.92,947.511 L1583.6,947.92 L1585.07,949.236 L1585.85,949.784 L1586.33,949.575 L1586.68,949.066 L1587.06,948.699 L1587.39,948.708 L1588.1,949.006 L1588.44,949.038 L1588.74,948.804 L1588.8,948.394 L1588.81,947.917 L1588.92,947.473 L1589.67,946.382 L1590.58,945.491 L1591.58,944.871 L1594.15,944.401 L1594.68,944.089 L1595.21,943.326 L1595.39,942.387 L1595.49,941.402 L1595.81,940.506 L1596.34,940.078 L1596.82,940.292 L1597.29,940.637 L1597.82,940.609 L1598.36,940.341 L1598.93,940.216 L1600.15,940.33 L1601.07,940.757 L1602.73,942.321 L1603.59,942.939 L1604.46,943.129 L1607.12,942.652 L1612.62,943.268 L1613.52,942.302 L1619.21,940.819 L1619.61,943.081 L1620.08,944.497 L1620.05,945.832 L1620.21,946.925 L1621.25,947.615 L1625.09,948.376 L1627.36,949.916 L1627.93,950.305 L1631.79,951.539 L1632.54,951.93 L1633.18,952.695 L1633.75,954.064 L1634.48,957.516 L1634.74,959.445 L1634.77,960.828 L1634.02,962.036 L1631.35,964.312 L1630.98,965.387 L1631.72,966.381 L1632.7,966.617 L1633.63,967.029 L1634.21,968.544 L1634.22,969.249 L1634.03,970.699 L1634,971.424 L1634.07,972.061 L1634.32,973.294 L1634.38,974.011 L1633.43,979.322 L1633.31,981.629 L1633.88,981.64 L1634.47,981.651 L1636.81,979.401 L1638.11,978.65 L1639.69,978.699 L1640.56,979.124 L1642.25,980.286 L1643.14,980.68 L1643.54,981.501 L1643.29,983.05 L1642.87,984.691 L1642.78,985.8 L1643.63,986.7 L1647.51,988.198 L1648.13,989.144 L1648.88,991.872 L1649.4,993.113 L1650.11,993.986 L1650.87,994.491 L1653.89,995.111 L1654.59,995.41 L1654.98,996.073 L1654.89,996.671 L1654.25,997.732 L1654.24,998.501 L1654.5,999.059 L1654.91,999.53 L1657.25,1001.32 L1657.64,1001.75 L1657.93,1002.38 L1658.37,1004.5 L1658.84,1005.86 L1659.41,1007.03 L1661.87,1010.83 L1662.12,1011.45 L1662.3,1012.36 L1662.23,1012.89 L1662.06,1013.36 L1661.96,1014.07 L1662.04,1015.58 L1662.23,1017.07 L1662.27,1018.5 L1661.9,1019.85 L1660.39,1019.31 L1659.1,1020.24 L1658.63,1020.81 L1657.89,1021.72 L1656.63,1022.8 L1655.32,1023.46 L1654.95,1023.79 L1654.57,1024.48 L1654.41,1025.23 L1654.31,1026.05 L1654.11,1026.96 L1652.27,1029.33 L1651.5,1030.71 L1651.73,1032.2 L1651.15,1034.47 L1648.35,1037.57 L1648.1,1040.51 L1648.6,1043.47 L1648.64,1044.84 L1648.58,1045.85 L1647.66,1046.22 L1643.27,1046.89 L1642.06,1046.83 L1640.83,1046.45 L1639.73,1045.67 L1639.24,1046.67 L1638.04,1048.63 L1637.45,1049.08 L1636.38,1048.78 L1633.8,1047.48 L1632.96,1047.25 L1631.51,1047.36 L1627.23,1049.01 L1627.18,1049.06 L1626.56,1049.71 L1625.19,1052.62 L1624.12,1056.71 L1623.63,1057.68 L1621.24,1059.76 L1620.47,1060.01 L1618.47,1059.7 L1617.53,1059.82 L1616.8,1060.48 L1616.38,1061.94 L1614.39,1062.24 L1612.32,1062.16 L1607.01,1060.95 L1604.47,1059.8 L1601.89,1056.94 L1600.66,1055.25 L1599.48,1054.1 L1595.39,1051.91 L1592.72,1049.99 L1590.5,1049.14 L1586.95,1047.77 L1583.69,1045.41 L1582.43,1044.5 L1580.16,1043.57 L1579.65,1043.24 L1579.2,1042.67 L1578.1,1040.75 L1577.73,1040.27 L1577.27,1040.18 L1575.28,1040.79 L1574.67,1040.8 L1572.78,1040.35 L1570.56,1041.28 L1566.82,1041.45 L1558.34,1041.79 L1556.91,1041.15 L1555.52,1039.51 L1554.48,1037.66 L1554.02,1037.14 L1552.61,1036.12 L1552.14,1035.56 L1551.51,1034.5 L1549.83,1030.97 L1548.38,1028.58 L1547.55,1027.68 L1546.78,1027.59 L1546.41,1028.13 L1546.42,1028.81 L1546.53,1029.5 L1546.43,1030.08 L1546.17,1030.2 L1545.35,1030.06 L1545.01,1030.15 L1542.24,1032.16 L1541.12,1032.97 L1540.01,1034.31 L1538.05,1038.65 L1537.55,1039.46 L1535.83,1041.01 L1534.78,1041.76 L1533.59,1041.69 L1531.09,1040.53 L1529.91,1040.32 L1528.89,1040.72 L1525.23,1043.71 L1524.36,1044.06 L1524.31,1044.08 L1523.24,1044.09 L1521.96,1043.67 L1518.25,1041.77 L1514.25,1041.3 L1513.78,1041.25 L1512.65,1041.44 L1510.41,1042.44 L1509.31,1042.48 L1509.23,1042.28 L1509.19,1041.89 L1509.06,1041.44 L1508.68,1041.07 L1508.38,1041.02 L1507.48,1041.36 L1505.09,1041.7 L1504.05,1042.23 L1502.25,1043.85 L1501,1043.99 L1497.18,1043.15 L1494.69,1042.6 L1493.9,1042.6 L1490.13,1044.27 L1489.58,1044.79 L1490,1045.37 L1488.64,1047.55 L1487,1048.28 L1485.21,1047.35 L1483.43,1044.53 L1482.26,1042.91 L1481.04,1042.49 L1479.87,1042.93 L1479,1043.82 L1477.91,1044.92 L1473.45,1047.8 L1472.95,1047.97 L1472.32,1047.83 L1470.51,1046.49 L1469.51,1046.16 L1469.42,1046.13 L1467.14,1046.01 L1457.07,1048.13 L1456.35,1047.97 L1454.91,1047.23 L1454.2,1047.01 L1452.64,1047.17 L1449.17,1049.61 L1448.61,1050 L1439.77,1056.15 L1436.75,1056.78 L1435.59,1057.65 L1431.41,1062.13 L1430.9,1062.39 L1429.36,1062.72 L1428.39,1063.44 L1427.29,1064.72 L1426.28,1066.26 L1425.63,1067.76 L1425.37,1069.84 L1425.39,1071.59 L1425.07,1072.85 L1423.74,1073.45 L1421.71,1073.79 L1420.45,1074.32 L1420.45,1074.32 L1420.45,1074.3 L1420.37,1074.02 L1419.67,1071.78 L1418.86,1069.82 L1416.7,1066.16 L1416.12,1064.88 L1415.76,1063.57 L1415.77,1063.18 L1415.89,1061.91 L1415.86,1061.19 L1415.76,1060.53 L1415.63,1059.98 L1414.88,1057.6 L1414.6,1056.37 L1414.48,1055.17 L1414.62,1054.18 L1414.95,1053.27 L1415.08,1052.75 L1415.15,1052.16 L1415.12,1051.47 L1414.93,1050.25 L1415,1049.59 L1414.58,1048.59 L1414.31,1047.48 L1414.42,1046.88 L1415.12,1047.37 L1415.44,1047.97 L1415.88,1049.57 L1416.3,1050.25 L1415.95,1051.18 L1416.23,1051.37 L1416.5,1051.49 L1416.64,1052.37 L1416.71,1052.79 L1416.35,1053.67 L1417.31,1054.67 L1417.63,1054.81 L1417.89,1054.62 L1418.08,1054.22 L1418.09,1053.72 L1417.5,1052.48 L1417.25,1051.39 L1417.14,1050.33 L1417.17,1049.65 L1416.94,1049.25 L1416.62,1048.18 L1416.36,1047.61 L1416.1,1047.18 L1415.89,1046.92 L1415.2,1046.49 L1414.8,1046.36 L1414.48,1046.37 L1414.21,1046.27 L1413.9,1045.82 L1413.65,1045.16 L1413.7,1044.87 L1413.84,1044.57 L1413.78,1041.98 L1414.43,1038.23 L1414.52,1036.16 L1413.9,1030.82 L1413.57,1029.61 L1413.15,1028.55 L1412.94,1027.51 L1413.18,1026.36 L1415.66,1024.07 L1415.89,1023.45 L1417.91,1021.73 L1418.41,1020.98 L1419.33,1018.87 L1419.61,1018.42 L1419.99,1018.11 L1420.79,1017.08 L1421.7,1015.92 L1422.26,1014.61 L1422.61,1012.74 L1422.65,1010.46 L1422.27,1007.94 L1422.06,1007.3 L1421.42,1005.7 L1421.38,1005.02 L1421.55,1004.1 L1421.54,1003.55 L1420.79,999.531 L1420.65,998.231 L1420.69,997.094 L1420.91,995.931 L1421.3,995.007 L1421.82,994.575 L1422.19,993.998 L1422.88,990.459 L1424.73,986.906 L1427.16,979.101 L1427.97,977.793 L1429.23,976.799 L1435.26,975.352 L1436.62,974.653 L1442.45,970.009 L1451.04,963.103 L1452.36,962.414 L1453.1,962.219 L1453.45,962.023 L1453.81,961.699 L1454.21,961.444 L1454.7,961.462 L1455.07,961.613 L1455.08,961.685 L1454.94,961.865 L1454.87,962.337 L1454.98,964.719 L1455.33,967.277 L1455.69,968.501 L1456.18,969.347 L1458.52,971.754 L1467.1,976.86 L1470.03,979.412 L1473.17,981.278 L1475.09,981.959 L1476.27,982.014 L1476.73,982.128 L1476.92,982.592 L1477.05,983.198 L1477.32,983.782 L1478.35,985.159 L1478.55,985.435 L1479.08,986.377 L1479.57,987.542 L1480.25,989.658 L1480.75,990.733 L1481.34,991.129 L1481.75,991.643 L1483.79,997.168 L1484.32,998.057 L1484.87,998.353 L1485.5,998.474 L1487.46,1000.04 L1490.45,1000.65 L1493.43,1000.65 L1493.91,1000.92 L1496.02,1003.03 L1496.37,1003.23 L1496.63,1003.37 L1498.3,1003.57 L1500.46,1003.38 L1505.9,1000.85 L1507.73,999.46 L1508.51,998.862 L1509.57,997.502 L1509.99,996.499 L1510.25,995.589 L1510.58,994.885 L1511.8,994.261 L1512.91,993.156 L1515.72,990.35 L1515.82,989.415 L1516.32,988.575 L1516.97,987.937 L1517.5,987.6 L1517.37,986.939 L1517.61,986.434 L1518.28,985.695 L1519.95,982.67 L1520.3,980.81 L1520.05,978.295 L1519.34,975.475 L1516.93,967.964 L1516.39,965.697 L1515.95,964.377 L1515.47,963.844 L1515.12,963.149 L1514.92,961.508 L1514.83,959.165 L1513.68,955.634 L1513.33,954.858 L1511.38,952.724 L1510.67,951.664 L1510.44,950.555 L1509.48,947.741 L1509.08,946.305 L1509.01,945.507 L1509.09,944.716 L1509.28,943.985 L1510.82,944.281 L1511.74,944.041 L1512.41,943.454 L1512.64,942.759 L1512.87,941.189 L1513.18,940.63 L1514.32,939.675 L1515.69,937.74 L1516.31,937.46 L1518.64,937.197 L1520.57,936.562 L1520.99,936.05 L1521.22,934.229 L1521.56,933.493 L1522.63,933.543 L1523.6,933.911 L1524.47,933.874 L1524.75,933.482 L1525.29,932.709 L1525.96,932.033 L1528.37,931.135 L1529.07,930.382 L1530.57,927.959 L1530.92,927.394 L1531.53,926.788 L1532.24,926.397 L1533.74,926.027 L1534.71,926.413 L1535.44,927.576 L1536.08,929.021 L1536.77,930.255 L1537.34,930.555 L1537.85,930.444 L1538.76,929.678 L1539.21,928.683 L1538.77,927.754 L1537.49,926.294 L1537.29,925.225 L1537.61,924.728 L1538.22,924.706 L1538.91,925.057 L1539.22,925.43 L1539.86,926.454 L1540.15,926.769 L1540.48,926.884"
         id="path995" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['LV'],polycolors['LV'],labels['LV'], x1, polybary['LV'], x2, polybary['LV'] )
    if 'LV' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1484.89,1918.13 L1484.8,1918.08 L1484.63,1918.45 L1484.38,1919.22 L1484.35,1919.67 L1484.38,1920.62 L1484.46,1921.03 L1484.69,1921.58 L1485.71,1923.4 L1487.36,1924.37 L1488.19,1924.66 L1489.04,1924.65 L1489.46,1924.46 L1489.74,1924.33 L1491.35,1923.07 L1491.73,1922.61 L1491.73,1922.6 L1491.73,1922.59 L1492.78,1921.65 L1495.02,1921.42 L1496.15,1920.93 L1497.53,1918.51 L1498.42,1917.67 L1499.69,1918.19 L1500.77,1919.1 L1503.22,1920.05 L1504.11,1920.93 L1504.12,1920.93 L1504.12,1920.93 L1504.13,1920.94 L1504.13,1920.94 L1505.82,1923.88 L1506.11,1924.2 L1506.81,1924.71 L1507.09,1925.15 L1507.14,1925.67 L1506.76,1926.46 L1506.75,1926.86 L1507.02,1927.22 L1507.81,1927.73 L1508.13,1928.07 L1508.3,1928.54 L1508.32,1929.38 L1508.44,1929.81 L1508.97,1930.77 L1509.4,1931.14 L1509.96,1931.15 L1513.09,1930.64 L1513.82,1930.66 L1520.05,1934.03 L1520.92,1934.75 L1521.34,1935.1 L1522.38,1936.18 L1524.62,1939.71 L1525.33,1941.29 L1525.59,1942.36 L1525.86,1944.34 L1526.3,1945.45 L1528.25,1949.05 L1528.62,1950.25 L1528.96,1951.03 L1529.06,1951.81 L1528.72,1953.01 L1528.41,1953.55 L1528.1,1953.85 L1527.87,1954.23 L1527.74,1955 L1527.81,1955.45 L1528.15,1956.18 L1528.18,1956.64 L1527.09,1957.52 L1527.18,1958.56 L1527.21,1958.9 L1527.79,1960.4 L1527.98,1961.27 L1528.07,1961.66 L1527.84,1962.25 L1526.89,1963.46 L1526.52,1964.12 L1526.25,1965 L1526.2,1965.67 L1527.15,1970.08 L1527.55,1971.37 L1528.91,1974.1 L1528.97,1974.51 L1528.99,1974.64 L1528.82,1975.31 L1528.48,1975.77 L1527.9,1976.24 L1527.26,1976.62 L1525.05,1977.43 L1525.34,1978.28 L1525.42,1979.97 L1525.87,1980.84 L1526.42,1981 L1526.82,1981.32 L1527.45,1982.21 L1527.56,1982.37 L1527.78,1982.83 L1528,1983.82 L1528.14,1984.14 L1528.51,1984.38 L1529.3,1984.39 L1529.68,1984.58 L1530.34,1985.28 L1530.65,1985.87 L1530.72,1986 L1530.82,1986.84 L1530.63,1987.92 L1530.29,1988.82 L1529.78,1989.67 L1529.16,1990.18 L1528.46,1990.06 L1528.45,1991.3 L1528.45,1991.42 L1528.94,1993.25 L1530.17,1996.47 L1530.96,1997.95 L1533.01,2000.15 L1533.94,2001.48 L1534.35,2002.84 L1534.47,2003.94 L1534.91,2004.73 L1535.92,2005.04 L1536.31,2005.16 L1536.96,2005.14 L1537.55,2005.03 L1537.68,2005.03 L1538.1,2005.03 L1538.66,2005.35 L1539.27,2006.27 L1540.71,2010.18 L1542.08,2013.6 L1542.95,2014.93 L1543.64,2015.37 L1544.18,2015.71 L1545.41,2015.8 L1546.19,2015.35 L1547.58,2013.82 L1548.47,2013.46 L1548.51,2013.45 L1551.04,2013.44 L1553.47,2013.8 L1554.36,2014.44 L1554.9,2015.54 L1555.14,2017.18 L1555.62,2020.09 L1555.51,2021.44 L1554.82,2022.52 L1555.33,2022.83 L1555.57,2023.27 L1555.73,2023.75 L1556,2024.18 L1557.51,2025.26 L1558.97,2026.59 L1559.57,2027.39 L1560.02,2028.42 L1560.25,2029.62 L1560.23,2030.78 L1560.02,2033 L1560.03,2033.72 L1560.08,2034.24 L1560.07,2034.73 L1559.91,2035.39 L1559.63,2035.78 L1559.23,2036.1 L1558.84,2036.51 L1558.6,2037.19 L1558.01,2037.44 L1557.56,2037.86 L1557.35,2038.45 L1557.48,2039.21 L1557.39,2039.59 L1557.28,2039.92 L1556.91,2040.63 L1555.83,2041.6 L1554.74,2041.49 L1553.59,2041.14 L1552.33,2041.35 L1551.75,2041.86 L1551.57,2042.41 L1551.46,2043.03 L1551.15,2043.72 L1550.76,2044.01 L1549.81,2044.39 L1549.44,2044.88 L1549.44,2045.47 L1549.87,2047.12 L1549.98,2047.76 L1549.91,2048.55 L1549.77,2049.09 L1549.38,2050.13 L1548.88,2052.05 L1548.78,2052.41 L1548.45,2052.8 L1547.97,2053.14 L1547.5,2053.57 L1547.2,2054.23 L1547.17,2055.66 L1547.43,2056.88 L1547.58,2058.15 L1547.24,2059.7 L1547.01,2059.94 L1546.33,2060.26 L1546.16,2060.63 L1546.18,2061.01 L1546.43,2061.67 L1546.48,2061.98 L1546.86,2063.09 L1546.82,2063.38 L1546.37,2065.49 L1546.22,2065.95 L1545.9,2066.24 L1544.79,2066.97 L1543.11,2068.07 L1541.93,2068.35 L1539.36,2068.33 L1538.52,2068.68 L1537.82,2069.21 L1536.22,2069.25 L1535.36,2069.78 L1534.82,2070.6 L1534.4,2071.65 L1534.14,2072.81 L1534.07,2073.94 L1533.3,2074.22 L1530.93,2074.62 L1530.37,2074.88 L1530.67,2075.33 L1531.86,2078.97 L1532.15,2079.31 L1533.43,2080.27 L1536.09,2083.6 L1536.42,2084.64 L1536.18,2085.93 L1535.46,2086.9 L1534.63,2086.92 L1533.72,2086.53 L1531.85,2086.09 L1531.78,2086.07 L1531.3,2086.06 L1530.92,2086.28 L1530.64,2086.79 L1530.79,2087.03 L1531.1,2087.25 L1531.32,2087.71 L1531.29,2088.04 L1531.08,2088.82 L1531.05,2089.02 L1531.19,2089.21 L1531.6,2089.6 L1531.7,2089.77 L1531.96,2090.28 L1532.25,2090.52 L1532.43,2090.83 L1532.37,2091.52 L1531.81,2092.41 L1530.98,2093.04 L1530.34,2093.84 L1530.37,2095.21 L1529.91,2095.01 L1529.48,2094.94 L1529.07,2094.99 L1528.7,2095.17 L1528.37,2095.64 L1528.38,2096.12 L1528.45,2096.57 L1528.32,2096.92 L1527.59,2097.21 L1526.55,2096.89 L1525.03,2095.85 L1522.69,2094.99 L1520.7,2094.64 L1519.03,2094.32 L1518.31,2094.91 L1518.31,2094.91 L1518.17,2094.98 L1518.17,2095.35 L1517.84,2095.39 L1517.47,2094.52 L1517.51,2093.48 L1518.04,2091.51 L1517.18,2091.28 L1516.91,2091.09 L1517.45,2089.41 L1517.96,2086.21 L1517.97,2085.45 L1517.5,2084.22 L1516.96,2083.93 L1516.21,2083.97 L1515.1,2083.73 L1515.02,2083.58 L1514.97,2083.33 L1514.89,2083.08 L1514.67,2082.89 L1513.97,2082.59 L1513.47,2082.48 L1512.65,2082.55 L1512.25,2082.35 L1512.21,2081.97 L1512.81,2080.99 L1513.22,2080.46 L1513.67,2080.06 L1512.74,2079.38 L1509.98,2075.38 L1509.58,2074.66 L1509.17,2074.05 L1508.75,2073.82 L1507.21,2073.97 L1506.44,2073.93 L1505.89,2073.72 L1506.22,2073.29 L1504.82,2072.58 L1504.67,2073.01 L1504.62,2073.2 L1504.6,2073.51 L1504.38,2072.97 L1504.12,2072.73 L1503.8,2072.58 L1503.43,2072.33 L1503.34,2072.09 L1502.91,2071.41 L1502.45,2070.85 L1502.29,2071 L1501.76,2070.58 L1498.49,2069.97 L1495.67,2068.74 L1493.98,2067.57 L1491.18,2067.01 L1489.47,2066.02 L1484.36,2061.55 L1483.73,2060.76 L1483.44,2060.27 L1482.48,2057.59 L1482.05,2056.88 L1480.2,2055.44 L1479.52,2054.73 L1479.41,2054.49 L1479.4,2054.42 L1479.13,2054.35 L1479.08,2053.93 L1480.27,2053.12 L1481.04,2052.95 L1481.84,2053.37 L1483.77,2055.22 L1484.3,2055.85 L1485.09,2057.23 L1485.46,2058.1 L1485.66,2058.93 L1486.14,2059.14 L1487.11,2058.78 L1488.05,2058.18 L1488.45,2057.7 L1488.12,2052.06 L1487.88,2051.16 L1486.58,2050.61 L1484.76,2049.17 L1483.25,2047.48 L1482.86,2046.18 L1483.23,2046.53 L1483.42,2046.97 L1483.6,2047.49 L1483.97,2047.33 L1484.33,2047.29 L1485.37,2048.21 L1485.92,2047.82 L1486.1,2046.62 L1486.02,2045.13 L1485.3,2044.01 L1483.97,2043.53 L1482.86,2043.99 L1482.81,2045.72 L1482.36,2044.81 L1478.52,2040.39 L1478.2,2039.88 L1478.35,2039.11 L1478.93,2038.42 L1479.63,2037.76 L1480.15,2037.06 L1480.46,2036.04 L1480.53,2034.99 L1480.16,2031.31 L1480.2,2030.28 L1480.55,2029.56 L1481.39,2029.2 L1481.86,2028.91 L1481.68,2028.4 L1481.26,2027.86 L1481.05,2027.5 L1481.2,2026.97 L1481.48,2026.43 L1481.75,2026.02 L1481.87,2025.89 L1481.42,2025.23 L1480.6,2024.4 L1480.04,2023.65 L1480.4,2023.26 L1481.09,2022.9 L1481.86,2021.46 L1482.57,2020.8 L1481.99,2021.61 L1482.56,2023.01 L1482.31,2024.3 L1482.47,2025.61 L1482.82,2025.56 L1482.86,2024.66 L1482.88,2024.85 L1482.97,2025.05 L1483.11,2025.52 L1483.64,2024.61 L1485.97,2023.1 L1486.66,2022.04 L1486.56,2020.51 L1485.92,2019.36 L1485.17,2018.32 L1484.75,2017.14 L1484.77,2018.75 L1484.35,2020.25 L1483.35,2022.25 L1483.3,2021.98 L1482.99,2021.65 L1483.44,2021.07 L1483.88,2020.13 L1484.09,2018.96 L1483.83,2017.69 L1483.56,2017.41 L1482.76,2017.19 L1482.45,2017 L1482.04,2016.35 L1482.03,2016.02 L1482.19,2015.7 L1482.16,2010.94 L1481.8,2009.98 L1481.49,2009.59 L1481.23,2009.13 L1481.22,2008.72 L1482.19,2008.14 L1482.27,2007.41 L1482.17,2006.58 L1482.17,2005.86 L1483.72,2002.55 L1483.92,2001.32 L1483.62,2000.09 L1483,1999.07 L1482.22,1998.4 L1481.43,1998.2 L1480.4,1998.33 L1479.52,1998.28 L1478.86,1997.81 L1478.5,1996.66 L1478.21,1994.11 L1477.99,1993.58 L1477.22,1993.03 L1477.01,1992.33 L1477.72,1993.07 L1478.76,1992.88 L1479.42,1991.93 L1479.04,1990.36 L1480.08,1989.6 L1481.12,1988.51 L1481.88,1987.12 L1482.06,1985.43 L1481.82,1984.42 L1481.45,1984.07 L1480.93,1983.98 L1480.24,1983.75 L1479.49,1983.19 L1478.6,1982.01 L1478,1981.42 L1482.13,1981.45 L1483.41,1980.88 L1483.8,1980.7 L1484.47,1978.46 L1484.79,1978.42 L1484.9,1979.09 L1485.11,1979.43 L1485.45,1979.45 L1485.88,1979.17 L1485.32,1977.56 L1485.04,1977.16 L1484.69,1977.61 L1484.42,1977.14 L1482.9,1976.08 L1483.66,1974.29 L1483.35,1972.36 L1482.86,1970.32 L1482.74,1969.21 L1483.62,1969.39 L1483.96,1968.53 L1483.85,1967.22 L1483.33,1966.09 L1483.03,1966.15 L1477.2,1964.57 L1476.48,1964.21 L1475.59,1963.95 L1472.56,1965.35 L1472.44,1964.7 L1472.29,1963.08 L1471.84,1962.47 L1471.36,1962.11 L1471.22,1961.87 L1471.22,1961.61 L1471.15,1961.2 L1471.17,1960.89 L1471.31,1960.54 L1471.37,1960.09 L1470.91,1958.7 L1471.12,1958.44 L1471.5,1958.35 L1471.75,1958.08 L1471.87,1957.01 L1471.48,1956.62 L1471.47,1956.61 L1470.93,1955.73 L1470.63,1954.57 L1470.54,1953.34 L1470.64,1952.25 L1471.27,1950.29 L1471.09,1949.71 L1470.96,1949.59 L1470.17,1948.84 L1467.3,1947.21 L1466.48,1946.5 L1466.47,1946.49 L1465.9,1945.56 L1465.96,1944.89 L1467.16,1943.25 L1470.78,1935.83 L1470.79,1935.51 L1470.66,1934.66 L1470.7,1934.2 L1471.01,1933.14 L1471.21,1932.75 L1473.25,1929.73 L1473.74,1928.64 L1474.58,1927.93 L1475.23,1926.97 L1475.74,1925.85 L1476.16,1924.63 L1476.16,1924.62 L1476.31,1923.66 L1476.8,1923.11 L1477.41,1922.65 L1477.91,1921.93 L1478.07,1921.15 L1478.14,1919.5 L1478.3,1918.61 L1478.92,1917.26 L1479.95,1915.7 L1480.36,1915.22 L1481.1,1914.36 L1482.1,1913.72 L1483.21,1914.13 L1484.08,1915.37 L1484.66,1916.9 L1484.78,1917.51 L1484.89,1918.13"
         id="path997" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['AL'],polycolors['AL'],labels['AL'], x1, polybary['AL'], x2, polybary['AL'] )
    if 'AL' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M976.528,1623.69 L977.057,1623.36 L977.3,1622 L977.69,1621.61 L978.232,1621.8 L978.626,1622.42 L978.945,1623.15 L979.25,1623.65 L979.684,1623.76 L980.152,1623.67 L980.565,1623.76 L980.833,1624.39 L980.716,1624.63 L980.139,1625.46 L979.993,1625.87 L980.163,1626.38 L980.492,1626.8 L980.588,1627.28 L980.664,1627.69 L982.895,1627.7 L982.557,1627.3 L982.956,1626.08 L984.09,1626.15 L985.456,1626.98 L986.532,1628.03 L985.752,1628.2 L985.75,1628.63 L986.338,1629.24 L986.369,1629.27 L987.583,1630.2 L988.614,1630.49 L990.272,1630.36 L991.816,1629.89 L992.498,1629.17 L993.302,1628.86 L998.007,1629.45 L1000.34,1629.48 L1000.91,1630.38 L1002.49,1630.39 L1004.13,1630.79 L1015.68,1638.1 L1015.91,1639.2 L1015.99,1639.58 L1017.23,1641.47 L1018.8,1642.2 L1020.01,1643.27 L1019.97,1645.92 L1019.55,1646.87 L1017.89,1648.94 L1017.69,1649.5 L1017.47,1650.6 L1017.3,1651.03 L1015.84,1652.76 L1014.46,1655.07 L1013.74,1656.26 L1012.99,1658.36 L1012.88,1660.46 L1013.2,1661.5 L1013.66,1662.4 L1014.01,1663.4 L1014.02,1664.74 L1013.61,1665.57 L1012.95,1666.23 L1012.45,1666.9 L1012.49,1667.48 L1013.45,1667.77 L1016.06,1668.22 L1016.94,1667.95 L1017.74,1668.17 L1019.98,1667.9 L1020.7,1668.01 L1028.77,1670.58 L1028.7,1671.29 L1028.88,1671.45 L1029.15,1671.43 L1029.31,1671.62 L1029.31,1671.99 L1029.13,1672.58 L1029.02,1674.05 L1028.89,1674.72 L1028.96,1675.31 L1029.5,1676.08 L1030.56,1676.9 L1035.15,1678.38 L1036.81,1679.95 L1037.77,1680.51 L1039.63,1681.1 L1040.23,1681.12 L1040.52,1681.13 L1041.63,1680.81 L1043.5,1679.86 L1043.93,1679.22 L1044.05,1678.74 L1044.08,1678.22 L1044.28,1677.42 L1044.93,1676.33 L1045.63,1676.19 L1046.46,1676.41 L1047.53,1676.36 L1047.55,1675.19 L1048.29,1673.75 L1049.36,1672.51 L1050.6,1671.79 L1050.84,1671.75 L1051.08,1671.79 L1051.3,1671.94 L1051.31,1671.94 L1051.31,1671.95 L1051.75,1672.42 L1052.65,1673.85 L1054.12,1675 L1054.5,1675.45 L1054.74,1676.5 L1054.22,1678.63 L1054.33,1679.96 L1054.11,1681.97 L1053.96,1682.54 L1053.71,1682.93 L1052.78,1684.07 L1052.87,1684.99 L1053.18,1685.9 L1053.29,1686.76 L1052.77,1687.55 L1052.05,1688.02 L1051.88,1688.6 L1051.92,1689.31 L1051.83,1690.19 L1051.42,1690.93 L1050.95,1691.38 L1050.75,1691.97 L1051.12,1693.17 L1051.9,1694.07 L1053.74,1694.25 L1054.64,1695 L1054.98,1696.2 L1054.96,1697.82 L1054.62,1699.37 L1054.36,1699.8 L1054.02,1700.37 L1053.24,1700.52 L1050.15,1699.73 L1048.64,1699.87 L1048.09,1699.79 L1047.6,1699.56 L1047.33,1699.28 L1047.1,1698.96 L1046.76,1698.67 L1044.98,1698.06 L1044.79,1697.38 L1045.02,1696.1 L1044.93,1695.4 L1044.24,1694.84 L1043.13,1694.85 L1039.05,1696.03 L1038.64,1696.26 L1038.45,1696.72 L1037.92,1698.76 L1037.57,1699.24 L1036.64,1700.17 L1036.27,1700.72 L1036.21,1701.17 L1036.2,1702.54 L1036.01,1703.21 L1036.07,1703.78 L1036.17,1704.24 L1036.38,1704.59 L1036.76,1704.84 L1035.99,1706.14 L1036.67,1706.98 L1037.94,1707.47 L1039.88,1707.85 L1040.63,1708.14 L1040.95,1708.84 L1040.62,1710.2 L1040.31,1710.62 L1039.4,1711.43 L1039.07,1712.04 L1038.86,1712.89 L1038.83,1713.46 L1039.01,1713.98 L1039.4,1714.68 L1041.19,1716.49 L1041.75,1717.61 L1041.18,1718.8 L1039.97,1719.56 L1038.14,1720.26 L1036.71,1720.23 L1036.67,1718.82 L1036.23,1717.76 L1034.52,1716.25 L1033.88,1715.39 L1033.6,1714.02 L1033.58,1712.78 L1033.29,1711.77 L1032.21,1711.06 L1031.32,1710.82 L1030.48,1710.76 L1028.59,1711.08 L1025.7,1712.55 L1024.8,1712.84 L1024.24,1712.72 L1023.19,1712.08 L1022.72,1712.07 L1022.25,1712.6 L1022.16,1713.31 L1022.23,1714.02 L1022.21,1714.51 L1021.54,1715.42 L1020.72,1715.75 L1015.75,1715.67 L1014.75,1715.29 L1013.83,1714.66 L1013.28,1713.9 L1012.43,1711.64 L1012.05,1711.32 L1011.1,1710.79 L1010.79,1710.48 L1010.7,1710.13 L1010.77,1709.18 L1010.56,1703.18 L1010.42,1702.79 L1010.08,1702.86 L1009.38,1703.37 L1009.08,1703.77 L1008.93,1704.21 L1008.71,1704.58 L1008.24,1704.77 L1007.92,1704.62 L1006.82,1703.56 L1006.8,1702.79 L1005.94,1702.56 L1003.84,1702.79 L1003,1703.55 L1002.24,1705.05 L1001.88,1706.57 L1002.29,1707.42 L1002.86,1707.83 L1002.94,1708.45 L1002.82,1709.24 L1002.82,1710.13 L1003.38,1712.35 L1003.43,1713.16 L1003.15,1714.53 L1001.86,1717.16 L1001.19,1719.39 L1000.79,1720.02 L1000.28,1720.48 L999.755,1720.72 L999.279,1721.06 L999.03,1721.68 L998.821,1722.4 L998.496,1722.99 L998.472,1723.04 L995.288,1725.13 L994.469,1726.32 L994.289,1727.13 L994.366,1728.54 L994.217,1729.3 L993.864,1729.88 L993.447,1730.12 L992.488,1730.41 L991.36,1731.25 L991.162,1731.96 L991.909,1734.14 L990.458,1735.45 L990.362,1735.6 L990.328,1735.76 L990.357,1735.92 L990.45,1736.08 L990.913,1736.56 L991.259,1737.7 L991.649,1738.29 L992.07,1738.54 L993.02,1738.74 L993.428,1739.01 L993.912,1740.05 L993.732,1741.11 L992.617,1743.21 L991.198,1744.9 L989.886,1744.64 L988.461,1743.98 L986.722,1744.48 L986.898,1743.52 L987.171,1742.78 L987.358,1742.01 L987.301,1740.93 L987.068,1740.11 L986.714,1739.29 L985.975,1737.93 L985.561,1736.93 L985.282,1736.53 L984.992,1736.29 L982.51,1734.92 L981.846,1734.67 L981.098,1734.61 L981.173,1734.44 L981.345,1734.15 L982.138,1732.4 L983.41,1730.91 L984.076,1729.46 L982.991,1727.98 L982.337,1727.74 L981,1727.75 L980.304,1727.64 L979.975,1727.41 L979.665,1726.82 L979.528,1726.77 L979.308,1726.68 L979.036,1726.81 L978.351,1727.39 L978.038,1727.56 L977.273,1727.51 L975.265,1726.29 L974.429,1725.98 L974.01,1725.76 L971.339,1721.66 L970.129,1720.37 L968.942,1719.74 L967.815,1719.27 L967.029,1718.59 L966.574,1717.58 L966.435,1716.05 L966.615,1714.45 L967.384,1711.23 L967.568,1709.43 L967.584,1707.55 L967.433,1706.14 L966.856,1705.26 L965.618,1705.02 L965.043,1705.13 L963.198,1705.49 L962.009,1706.1 L961.066,1707.05 L960.687,1707.83 L960.846,1708.1 L961.128,1708.33 L961.132,1708.96 L960.86,1709.53 L960.419,1710.03 L959.913,1710.4 L958.673,1710.99 L956.476,1713.74 L955.565,1714.34 L953.673,1714.72 L952.725,1715.13 L951.857,1715.96 L951.407,1716.56 L951.21,1717.08 L951.357,1717.33 L952.349,1718.23 L953.572,1720.74 L953.638,1723.03 L952.645,1725.03 L950.702,1726.63 L950.226,1726.77 L949.324,1726.84 L948.879,1727.17 L948.552,1727.8 L948.44,1728.52 L948.417,1729.21 L948.142,1730.99 L948.058,1731.11 L947.61,1732.03 L947.575,1732.16 L947.012,1732.85 L946.694,1733.1 L946.272,1733.22 L943.161,1733.83 L941.981,1734.61 L940.97,1736.41 L940.913,1736.51 L940.76,1737.14 L940.787,1737.47 L940.677,1737.68 L940.123,1737.96 L939.871,1737.94 L939.097,1737.68 L937.896,1737.66 L935.82,1736.84 L935.39,1736.87 L935.056,1737.02 L934.688,1737.09 L934.156,1736.89 L934.098,1736.73 L933.313,1735.51 L932.677,1734.89 L932.004,1734.47 L927.594,1733.22 L926.852,1733.57 L926.405,1734.27 L925.908,1734.87 L924.989,1734.96 L923.672,1735.48 L921.029,1737.27 L919.614,1737.71 L916.346,1737.24 L915.766,1737.42 L914.512,1738.13 L911.777,1739.13 L910.45,1739.33 L909.023,1739.3 L907.692,1738.97 L906.7,1738.33 L904.825,1736.06 L904.542,1735.55 L904.332,1734.91 L904.069,1733.76 L903.619,1732.44 L903.487,1731.77 L903.281,1731.62 L900.463,1728.18 L899.491,1727.7 L899.12,1727.83 L898.797,1728.14 L898.454,1728.38 L898.042,1728.3 L897.672,1728 L897.653,1727.8 L897.753,1727.54 L897.755,1727.06 L897.82,1727 L897.833,1726.35 L897.782,1725.73 L897.658,1725.78 L897.869,1725.47 L898.237,1725.06 L898.562,1724.59 L898.626,1724.12 L897.987,1723.45 L894.559,1722.54 L894.226,1721.48 L894.714,1719.36 L895.561,1717.15 L897.222,1714.24 L896.306,1712.5 L894.838,1710.81 L894.07,1709.35 L894.333,1708.64 L895.529,1707.35 L895.881,1706.32 L895.858,1705.71 L895.836,1705.13 L895.46,1704.5 L895.458,1704.5 L894.821,1704.15 L888.455,1702.2 L885.579,1701.97 L882.787,1702.39 L878.986,1704.74 L877.566,1705.02 L876.142,1705.02 L874.79,1705.39 L873.328,1706.53 L872.015,1708.09 L871.042,1709.28 L870.768,1710.12 L870.916,1710.75 L871.263,1711.84 L871.273,1712.09 L871.639,1713.14 L871.81,1713.41 L872.251,1713.67 L872.993,1713.38 L873.291,1713.52 L873.446,1714.97 L872.268,1716.1 L869.403,1717.79 L867.058,1720.27 L865.612,1720.91 L864.169,1720.18 L862.185,1720.15 L860.191,1720.51 L859.112,1721.08 L859.762,1719.76 L860.127,1719.15 L860.289,1718.62 L859.572,1717.62 L859.164,1716.74 L859.364,1716 L860.457,1715.38 L863.122,1714.25 L863.195,1714.23 L863.267,1714.23 L863.347,1714.24 L863.682,1714.37 L863.94,1714.42 L864.192,1714.4 L865.162,1714.15 L865.371,1713.75 L865.36,1713.17 L865.416,1712.5 L865.814,1710.75 L865.982,1710.27 L866.683,1708.89 L867.539,1707.21 L867.516,1706.52 L867.036,1705.55 L866.436,1704.83 L864.377,1703.42 L864.177,1703.28 L864.696,1702.83 L864.798,1702.43 L864.73,1702 L864.754,1701.52 L864.733,1701.33 L864.643,1701.07 L864.565,1700.77 L864.581,1700.45 L864.77,1700.08 L865.288,1699.57 L866.921,1697.08 L868.541,1695.24 L867.557,1694.02 L867.465,1693.19 L868.082,1692.47 L874.13,1687.47 L877.291,1685.94 L877.734,1685.6 L878.927,1684.38 L880.388,1683.7 L880.856,1683.38 L881.368,1682.8 L881.557,1682.29 L881.588,1681.24 L881.285,1680.87 L881.073,1680.55 L880.969,1680.19 L881.046,1679.89 L881.814,1677.9 L882.136,1677.37 L882.228,1677.17 L882.403,1676.79 L882.514,1675.93 L882.408,1675.23 L881.887,1674.12 L881.737,1673.54 L882.475,1671.39 L884.605,1670.28 L889.272,1669.05 L892.22,1667.01 L893.257,1665.65 L892.795,1664.47 L893.411,1663.5 L893.83,1663.13 L894.919,1662.79 L895.094,1662.4 L895.258,1662.3 L895.884,1662.01 L895.877,1660.94 L897.174,1660.55 L899.978,1658.17 L900.076,1658.09 L900.941,1656.79 L902.231,1655.57 L905.207,1653.57 L905.097,1652.01 L905.421,1650.74 L906.253,1649.95 L906.642,1649.92 L906.874,1649.84 L907.526,1649.03 L907.98,1648.79 L908.42,1648.69 L908.829,1648.45 L909.185,1647.77 L908.766,1647.12 L908.152,1646.52 L907.54,1645.98 L906.744,1646.32 L901.672,1646.61 L901.915,1645.82 L902.482,1644.87 L903.142,1644.07 L904.26,1643.48 L904.374,1642.3 L905.497,1642.1 L906.201,1641.73 L906.854,1641.22 L907.194,1640.71 L907.018,1639.95 L906.614,1639.07 L906.536,1638.37 L907.334,1638.17 L907.722,1637.89 L908.107,1637.8 L908.488,1637.92 L908.862,1638.23 L909.955,1638.43 L912.08,1638.15 L913.07,1638.39 L913.61,1638.73 L913.688,1638.75 L914.171,1638.85 L915.328,1638.77 L914.491,1640.52 L914.694,1641.54 L915.594,1642.13 L916.825,1642.57 L916.995,1642.72 L917.106,1642.94 L917.268,1643.18 L917.592,1643.33 L917.879,1643.28 L919.509,1642.64 L920.64,1642.45 L921.819,1642.54 L923.582,1642.68 L923.998,1642.53 L924.778,1642.25 L925.408,1641.48 L925.652,1641.18 L925.808,1640.6 L925.75,1640.25 L925.553,1639.95 L925.418,1639.79 L925.217,1639.4 L925.225,1639.02 L925.725,1638.89 L926.375,1639.14 L926.561,1639.27 L926.915,1639.51 L927.46,1639.61 L928.127,1639.05 L928.205,1638.94 L928.268,1638.77 L928.278,1638.61 L928.272,1638.58 L928.239,1638.45 L927.951,1638.01 L927.867,1637.76 L927.931,1637.56 L928.18,1637.44 L928.615,1637.55 L928.96,1637.45 L929.156,1637.1 L929.163,1636.47 L929.018,1636.47 L928.332,1635.91 L928.232,1635.86 L929.89,1634.6 L930.128,1634.41 L931.161,1633.88 L932.666,1633.37 L932.69,1633.36 L934.867,1632.79 L935.828,1632.72 L935.224,1634.26 L934.76,1634.68 L933.764,1634.64 L933.654,1634.64 L935.201,1635.52 L935.806,1635.86 L936.734,1636.02 L938.603,1635.69 L940.282,1635.39 L941.092,1634.96 L941.789,1634.18 L942.593,1633 L943.179,1633.32 L945.903,1633.56 L946.153,1633.84 L946.273,1634.43 L946.34,1635.02 L946.446,1635.29 L951.954,1635.44 L953.868,1635.06 L954.286,1634.79 L954.481,1634.52 L954.66,1634.21 L955.029,1633.81 L955.381,1633.55 L956.311,1633.08 L957.09,1632.87 L957.386,1632.57 L957.603,1632.27 L957.825,1632.13 L960.116,1631.8 L960.881,1631.82 L961.97,1632.18 L962.458,1632.24 L962.68,1632.52 L962.894,1633.13 L963.177,1633.74 L963.587,1634.02 L965.183,1634.48 L967.89,1634.55 L968.019,1634.54 L969.198,1634.36 L969.256,1634.07 L969.772,1633.01 L971.083,1632.18 L972.353,1632.08 L972.993,1632.7 L973.479,1633.55 L973.565,1633.66 L973.936,1634.14 L974.509,1633.96 L974.604,1633.8 L974.793,1633.48 L974.85,1633.01 L974.836,1632.57 L974.897,1632.14 L974.883,1631.93 L974.787,1631.68 L974.723,1631.4 L974.825,1631.05 L974.981,1630.93 L975.205,1630.85 L975.389,1630.84 L975.426,1630.94 L975.445,1631.33 L975.718,1631.47 L975.985,1630 L975.602,1629.95 L975.408,1629.87 L974.93,1629.66 L974.342,1629.55 L972.283,1629.86 L971.643,1630.16 L971.049,1630.57 L970.418,1630.88 L969.666,1630.9 L968.803,1630.38 L967.736,1629.53 L967.527,1629.5 L966.869,1629.24 L967.154,1628.57 L967.06,1628.03 L966.891,1627.57 L966.949,1627.11 L967.373,1626.65 L968.454,1626.08 L968.912,1625.67 L969.256,1624.43 L969.462,1623.98 L970.05,1623.15 L970.385,1622.96 L970.855,1622.99 L973.135,1622.58 L973.777,1622.27 L973.404,1621.28 L974.082,1620.91 L975.131,1620.98 L975.904,1621.35 L975.961,1621.81 L975.985,1622.61 L976.12,1623.38 L976.528,1623.69"
         id="path999" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['CH'],polycolors['CH'],labels['CH'], x1, polybary['CH'], x2, polybary['CH'] )
    if 'CH' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1247.38,1543.04 L1250.46,1541.97 L1250.66,1541.82 L1251.18,1541.1 L1251.3,1540.76 L1251.34,1539.91 L1251.49,1539.65 L1251.8,1539.62 L1252.03,1539.86 L1252.22,1540.16 L1252.43,1540.29 L1254.36,1540.17 L1255.29,1540.36 L1257.34,1541.23 L1259.21,1541.71 L1260.13,1542.15 L1262.29,1543.76 L1263.25,1544.05 L1265.77,1544.82 L1269.1,1546.37 L1270.04,1546.52 L1270.99,1546.45 L1271.66,1546.17 L1273.07,1545.29 L1273.42,1545.17 L1274.68,1545.07 L1274.93,1545.23 L1275.15,1545.98 L1275.38,1546.29 L1275.67,1546.38 L1276.46,1546.38 L1276.94,1546.67 L1277.26,1546.79 L1277.6,1546.74 L1277.22,1547.33 L1277.73,1547.16 L1278.2,1547.15 L1278.52,1547.4 L1278.62,1548.06 L1279.3,1547.82 L1279.54,1548.07 L1279.57,1548.5 L1279.66,1548.8 L1284.04,1551.48 L1286.3,1552.25 L1290.09,1551.7 L1295.92,1552.07 L1295.97,1552.02 L1296.81,1551.82 L1297.39,1552.22 L1297.63,1552.27 L1298.26,1552.02 L1298.68,1551.56 L1300.46,1547.8 L1301.13,1547.26 L1302.34,1547.33 L1302.76,1547.27 L1303.48,1546.89 L1303.85,1546.81 L1304.24,1546.9 L1304.98,1547.29 L1307.5,1547.78 L1308.27,1547.81 L1309.09,1548.05 L1309.48,1548.77 L1309.71,1549.66 L1310.07,1550.37 L1310.65,1550.7 L1311.31,1550.75 L1312.87,1550.55 L1313.52,1550.7 L1315.1,1551.56 L1315.88,1551.77 L1316.67,1551.6 L1318.2,1550.89 L1318.88,1550.88 L1319.98,1552.16 L1320.93,1556.23 L1322.51,1557.75 L1323.16,1560.63 L1323.02,1561.43 L1322.96,1561.76 L1322.34,1562.54 L1321.71,1563.15 L1321.46,1563.77 L1321.31,1564.62 L1320.37,1566.28 L1320.03,1567.16 L1319.94,1568.07 L1319.88,1571.43 L1319.8,1571.83 L1319.72,1572.02 L1319.67,1572.31 L1319.71,1573 L1319.87,1573.36 L1320.21,1573.53 L1321.05,1573.54 L1321.3,1573.65 L1321.77,1573.97 L1322.24,1574.41 L1322.47,1574.88 L1322.41,1575.17 L1322.28,1575.55 L1322.23,1575.87 L1322.42,1575.98 L1322.54,1576.1 L1322.71,1576.4 L1322.88,1576.75 L1322.96,1577.02 L1323.13,1577.38 L1323.46,1577.55 L1323.86,1577.69 L1324.36,1578.07 L1324.62,1578.13 L1324.8,1578.33 L1324.86,1578.6 L1324.87,1579.32 L1324.91,1579.61 L1325.77,1581.79 L1326.07,1582.89 L1326.2,1584.24 L1326.59,1585.19 L1327.74,1586.24 L1328.35,1586.54 L1329.03,1586.57 L1329.49,1586.81 L1330.22,1587.89 L1330.45,1588.23 L1331.05,1588.76 L1330.67,1589.33 L1330.52,1589.98 L1330.39,1591.16 L1330.56,1591.25 L1331.1,1591.62 L1330.97,1592.66 L1331.98,1593.09 L1333.37,1593.44 L1334.43,1594.23 L1332.02,1596.68 L1332.51,1597.15 L1332.53,1597.53 L1332.36,1597.94 L1332.29,1598.49 L1332.22,1598.6 L1331.86,1598.9 L1331.76,1599.09 L1331.83,1599.36 L1332.05,1599.46 L1332.27,1599.51 L1332.37,1599.63 L1332.3,1600.75 L1332.14,1601.59 L1331.78,1602.26 L1331.14,1602.87 L1329.72,1603.33 L1329.22,1603.66 L1329.3,1604.33 L1329.59,1604.59 L1330.51,1604.91 L1330.86,1605.13 L1331.37,1606.25 L1331.68,1606.62 L1331.12,1607.38 L1331.27,1608.45 L1331.67,1609.68 L1331.94,1611.71 L1332.2,1612.32 L1332.6,1612.76 L1333.11,1613.01 L1332.28,1613.49 L1329.26,1614.2 L1326,1615.34 L1324.38,1615.2 L1323.62,1613.63 L1323.11,1614.15 L1322.4,1615.54 L1321.98,1616.04 L1321.62,1616.16 L1319.22,1616.01 L1318.74,1615.77 L1318.23,1615.33 L1317.85,1614.71 L1317.63,1614.07 L1317.36,1613.53 L1316.8,1613.21 L1313.32,1612.2 L1311.53,1612.14 L1310.05,1612.97 L1309.85,1613.6 L1310,1614.41 L1309.81,1614.97 L1309.49,1615.34 L1307.89,1616.38 L1307.42,1616.87 L1307.16,1617.07 L1306.96,1617.05 L1306.74,1616.87 L1306.47,1616.58 L1306.18,1616.92 L1305.63,1618.01 L1305.3,1618.51 L1306.1,1618.89 L1308.54,1619.63 L1309.68,1619.27 L1312.55,1620.16 L1313.92,1619.8 L1314.88,1620.14 L1315.7,1621.06 L1316.16,1622.28 L1316.03,1623.51 L1316.78,1623.85 L1317.42,1624.39 L1317.82,1625.14 L1317.85,1626.1 L1317.47,1626.94 L1316.29,1627.57 L1315.85,1628.13 L1316.24,1630.63 L1315.7,1631.14 L1314.22,1632.52 L1309.85,1635 L1309.27,1634.22 L1308.67,1633.88 L1308.16,1634.07 L1307.8,1634.89 L1308.1,1637.27 L1308.03,1637.46 L1307.83,1637.59 L1307.66,1637.77 L1307.66,1638.15 L1307.8,1638.37 L1308,1638.46 L1308.19,1638.52 L1308.32,1638.63 L1309.83,1641.22 L1310.09,1642.22 L1309.88,1643.08 L1309.33,1643.69 L1308.07,1644.51 L1308.07,1644.52 L1308.07,1644.52 L1308.3,1645.59 L1307.76,1647.02 L1308,1648.03 L1308.27,1648.23 L1308.61,1648.17 L1308.96,1648.17 L1309.28,1648.55 L1309.34,1649.1 L1309.09,1650.19 L1309.1,1650.53 L1309.7,1650.86 L1311.08,1650.05 L1311.79,1650.05 L1312.37,1650.76 L1312.23,1651.51 L1311.37,1652.88 L1310.52,1653.51 L1310.32,1654.45 L1310.7,1655.24 L1311.58,1655.45 L1312.11,1655.66 L1312.28,1655.97 L1312.14,1656.33 L1311.69,1656.68 L1309.85,1657.62 L1309.36,1658.15 L1310.59,1658.19 L1311.21,1658.34 L1311.87,1658.85 L1312.12,1659.51 L1311.33,1659.77 L1310.22,1659.88 L1309.52,1660.07 L1309.52,1660.08 L1309.35,1660.27 L1309.15,1660.34 L1308.92,1660.29 L1308.68,1660.14 L1307.9,1659.65 L1306.99,1659.62 L1305.27,1659.98 L1303.65,1659.8 L1303.08,1659.92 L1302.73,1660.65 L1302.62,1661.61 L1302.3,1661.92 L1301.88,1662.05 L1301.48,1662.46 L1301.4,1662.85 L1301.46,1663.19 L1301.47,1663.57 L1301.22,1664.05 L1300.95,1664.3 L1300.09,1664.76 L1299.06,1665.64 L1298.61,1666.2 L1297.23,1668.44 L1296.73,1669.01 L1296.07,1669.39 L1294.39,1670.57 L1293.54,1671.18 L1293.39,1671.23 L1291.67,1671.8 L1291.43,1671.97 L1291.05,1672.44 L1291.1,1672.57 L1291.3,1672.69 L1291.37,1673.13 L1291.23,1675.09 L1291.23,1675.28 L1291.25,1676.19 L1291.37,1677.28 L1291.98,1678.76 L1292.93,1679.27 L1293.49,1680.2 L1293.7,1681.62 L1293.7,1681.63 L1293.69,1681.62 L1292.81,1680.68 L1292.32,1680.39 L1290.55,1680.22 L1287.53,1678.98 L1286.32,1678.84 L1285.13,1679.04 L1283.54,1679.83 L1282.32,1680.44 L1281.19,1680.59 L1277.86,1680.36 L1277.14,1679.99 L1277.08,1680.94 L1277.14,1681.74 L1276.92,1682.3 L1276.71,1682.49 L1276.5,1682.67 L1275.96,1682.87 L1275.3,1682.74 L1274.39,1682.82 L1273.48,1683.13 L1272.86,1683.68 L1272.51,1684.6 L1272.49,1684.72 L1272.42,1685.31 L1272.29,1685.68 L1272.19,1685.95 L1271.43,1686.64 L1270.16,1686.96 L1269.15,1686.42 L1268.93,1686.26 L1268.12,1685.65 L1266.82,1685.27 L1264.45,1685.56 L1258.98,1686.24 L1257.58,1686.19 L1254.68,1686.08 L1253.82,1686.05 L1252.79,1686.01 L1250.37,1686.97 L1248.91,1689.36 L1248.01,1688.23 L1247.39,1688.15 L1246.8,1688.57 L1245.91,1689.23 L1245.75,1689.26 L1245.02,1689.4 L1244.41,1689.37 L1243.89,1689.63 L1243.23,1690.73 L1242.81,1691.81 L1242.54,1692.84 L1242.28,1693.82 L1241.89,1694.88 L1241.59,1695.7 L1241.37,1695.93 L1240.44,1696.42 L1239.35,1696.68 L1238.95,1696.43 L1238.25,1696.8 L1238.01,1697.11 L1237.82,1697.55 L1237.35,1698.19 L1237.09,1698.98 L1236.77,1698.94 L1236.51,1698.91 L1236.32,1698.97 L1235.51,1699.93 L1235.08,1700.25 L1234.65,1700.4 L1233.72,1700.55 L1233.31,1700.74 L1232.72,1701.7 L1232.43,1702.93 L1232.25,1703.48 L1232.09,1703.98 L1231.36,1704.35 L1230.75,1703.79 L1230.2,1702.75 L1229.57,1701.98 L1228.1,1702.42 L1227.37,1702.35 L1226.76,1702.11 L1226,1701.82 L1225.74,1701.54 L1225.58,1701.2 L1225.36,1700.92 L1224.89,1700.84 L1223.46,1701.25 L1218.27,1701.4 L1214.28,1701.51 L1214.19,1701.49 L1213.72,1701.39 L1211.18,1699.44 L1210.53,1699.15 L1209.82,1698.98 L1209.06,1699.01 L1208.31,1699.18 L1207.62,1699.35 L1206.91,1699.3 L1202.86,1697.64 L1201.55,1697.49 L1198.77,1698.1 L1198.19,1698.14 L1195.3,1697.59 L1194.65,1697.57 L1193.98,1697.73 L1193.33,1697.7 L1188.03,1696.26 L1186.19,1696.28 L1185.85,1696.06 L1185.21,1695.4 L1184.91,1695.3 L1182.29,1695.61 L1180.39,1695.37 L1176.03,1696.5 L1174.29,1696.49 L1173.37,1696.16 L1170.56,1694.6 L1166.99,1693.93 L1156.88,1693.61 L1154.37,1692.11 L1153.28,1691.78 L1152.9,1691.67 L1152.57,1691.73 L1151.89,1692.02 L1151.48,1692.05 L1151.08,1691.89 L1150.27,1691.33 L1149.93,1691.19 L1149.85,1691.16 L1147.72,1691.02 L1145.23,1691.43 L1144.59,1691.39 L1143.87,1691.08 L1142.51,1690.22 L1141.2,1690.03 L1140.16,1689.89 L1138.39,1689.23 L1136.86,1687.96 L1135.98,1685.99 L1135.56,1684.63 L1134.84,1684.2 L1133.94,1684.1 L1133.03,1683.78 L1132.61,1683.51 L1132.36,1683.27 L1132.24,1682.85 L1132.23,1682.04 L1132.32,1681.32 L1132.47,1680.88 L1132.58,1680.42 L1132.54,1679.66 L1132.11,1678.3 L1131.41,1677.86 L1129.61,1677.85 L1129.02,1677.65 L1128.75,1677.35 L1128.5,1676.97 L1128,1676.49 L1127.51,1676.26 L1126.5,1676.11 L1126.04,1675.94 L1126.65,1675.3 L1126.64,1674.73 L1126.33,1674.14 L1126.03,1673.46 L1125.73,1672.02 L1125.52,1671.31 L1125.22,1670.7 L1125.64,1669.6 L1128.19,1668.08 L1129.07,1666.82 L1129.06,1665.97 L1129.05,1665.17 L1128.03,1664.86 L1126.01,1665.32 L1125.3,1665.48 L1121.03,1667.85 L1117.98,1668.08 L1116.11,1668.78 L1114.32,1669.81 L1112.87,1671.01 L1110.94,1671.37 L1110.47,1671.69 L1109.64,1672.39 L1109.15,1672.52 L1108.34,1672.23 L1106.94,1671.24 L1106.12,1671.17 L1105.8,1671.23 L1105.47,1671.2 L1105.42,1671.19 L1105.4,1671.19 L1103.19,1670.79 L1102.17,1670.91 L1100.94,1671.3 L1100.93,1671.3 L1100.92,1671.3 L1100.92,1671.3 L1100.52,1671.53 L1100.12,1671.61 L1099.73,1671.55 L1099.37,1671.33 L1099.13,1671.04 L1098.88,1670.81 L1098.6,1670.65 L1097.84,1670.48 L1097.42,1670.59 L1097.04,1670.89 L1096.72,1671.38 L1096.72,1671.39 L1095.29,1672.82 L1093.98,1672.78 L1092.64,1672.15 L1091.1,1671.85 L1088.11,1672.4 L1085.16,1673.41 L1084.4,1673.88 L1081.67,1676.68 L1081.33,1677.45 L1080.9,1679.65 L1080.12,1681.59 L1080.09,1681.83 L1080.15,1682.48 L1080.08,1682.79 L1079.89,1683 L1079.42,1683.22 L1079.25,1683.39 L1078.31,1685.05 L1077.71,1685.69 L1077.1,1685.78 L1076.37,1685.55 L1074.87,1685.42 L1074.13,1685.53 L1072.7,1686.03 L1072.28,1686.07 L1071.84,1685.89 L1071.1,1685.27 L1070.71,1685.08 L1070.27,1685.08 L1069.45,1685.32 L1069.03,1685.3 L1067.79,1684.62 L1067.29,1684.46 L1066.39,1684.75 L1065.92,1684.74 L1065.66,1684.19 L1065.86,1683.88 L1066.81,1683.1 L1067.01,1682.67 L1066.59,1682.04 L1063.29,1680.11 L1062.62,1679.94 L1061.87,1680.03 L1057.51,1681.26 L1055.73,1681.08 L1054.33,1679.96 L1054.22,1678.63 L1054.74,1676.5 L1054.5,1675.45 L1054.12,1675 L1052.65,1673.85 L1051.75,1672.42 L1051.31,1671.95 L1051.31,1671.94 L1051.3,1671.94 L1051.08,1671.79 L1050.84,1671.75 L1050.6,1671.79 L1049.36,1672.51 L1048.29,1673.75 L1047.55,1675.19 L1047.53,1676.36 L1046.46,1676.41 L1045.63,1676.19 L1044.93,1676.33 L1044.28,1677.42 L1044.08,1678.22 L1044.05,1678.74 L1043.93,1679.22 L1043.5,1679.86 L1041.63,1680.81 L1040.52,1681.13 L1040.23,1681.12 L1039.63,1681.1 L1037.77,1680.51 L1036.81,1679.95 L1035.15,1678.38 L1030.56,1676.9 L1029.5,1676.08 L1028.96,1675.31 L1028.89,1674.72 L1029.02,1674.05 L1029.13,1672.58 L1029.31,1671.99 L1029.31,1671.62 L1029.15,1671.43 L1028.88,1671.45 L1028.7,1671.29 L1028.77,1670.58 L1020.7,1668.01 L1019.98,1667.9 L1017.74,1668.17 L1016.94,1667.95 L1018.13,1666.46 L1018.44,1664.84 L1018.01,1663.25 L1017,1661.83 L1015.74,1661.07 L1015.74,1660.53 L1016.17,1659.59 L1016.21,1659.14 L1015.85,1658.28 L1015.45,1657.73 L1015.26,1657.18 L1015.55,1656.31 L1014.84,1655.64 L1014.46,1655.07 L1015.84,1652.76 L1017.3,1651.03 L1017.47,1650.6 L1017.69,1649.5 L1017.89,1648.94 L1019.55,1646.87 L1019.97,1645.92 L1020.01,1643.27 L1018.8,1642.2 L1017.23,1641.47 L1015.99,1639.58 L1015.91,1639.2 L1015.68,1638.1 L1015.78,1638.16 L1015.96,1638.21 L1018.43,1638.91 L1021.15,1638.87 L1022.31,1638.33 L1022.93,1637.37 L1023.45,1636.24 L1024.38,1635.17 L1025,1634.85 L1025.63,1634.78 L1026.23,1635 L1026.72,1635.54 L1026.86,1636.28 L1026.77,1637.04 L1026.93,1637.7 L1027.8,1638.14 L1028.14,1638.12 L1028.66,1637.79 L1028.87,1637.73 L1030.15,1638.2 L1031.41,1638.3 L1032.05,1638.18 L1032.56,1637.77 L1032.68,1638.8 L1033.12,1639.39 L1033.65,1639.92 L1034.04,1640.76 L1034.04,1640.95 L1034.05,1641.14 L1034.11,1641.5 L1034.58,1641.78 L1034.95,1641.62 L1035.33,1641.29 L1035.81,1641.07 L1037.9,1644.12 L1038.27,1644.85 L1038.08,1645.51 L1037.11,1646.26 L1037.58,1646.8 L1037.69,1647.45 L1037.73,1648.12 L1037.96,1648.72 L1038.38,1649.12 L1038.66,1649.11 L1038.99,1648.86 L1039.54,1648.6 L1039.75,1648.62 L1040.19,1648.89 L1040.41,1648.89 L1040.59,1648.75 L1041.1,1648.19 L1042.96,1647.85 L1043.75,1648.27 L1043.72,1649.4 L1042.96,1651.71 L1043.05,1652.12 L1043.08,1652.49 L1042.97,1652.91 L1042.77,1653.11 L1042.2,1653.28 L1042.03,1653.4 L1041.64,1654.11 L1041.47,1654.5 L1041.66,1654.61 L1042.3,1654.53 L1045.04,1654.19 L1045.97,1653.82 L1047.87,1652.65 L1048.7,1651.89 L1049.46,1650.86 L1050.4,1648.95 L1050.65,1648.56 L1051.09,1648.26 L1051.98,1647.95 L1052.35,1647.7 L1053.05,1646.76 L1053.66,1645.49 L1054.04,1644.08 L1054.02,1642.74 L1053.84,1641.97 L1053.61,1641.42 L1053.24,1641.02 L1052.64,1640.7 L1053.06,1638.21 L1053.11,1637.63 L1052.83,1636.93 L1052.45,1636.24 L1052.39,1635.7 L1053.02,1635.45 L1053.73,1635.32 L1054.05,1635.19 L1054.2,1635.31 L1054.4,1635.89 L1054.34,1636.13 L1054.14,1636.43 L1053.99,1636.83 L1054.06,1637.37 L1054.29,1637.65 L1054.64,1637.9 L1055.3,1638.2 L1057.1,1638.46 L1057.6,1638.37 L1058.14,1637.93 L1058.99,1636.73 L1059.57,1636.32 L1060.6,1636.32 L1066.2,1638.38 L1066.38,1638.58 L1066.52,1638.9 L1066.73,1639.19 L1067.1,1639.3 L1068.35,1639.13 L1070.05,1638.33 L1070.65,1638.16 L1071.24,1638.19 L1071.69,1638.44 L1072.67,1639.16 L1072.32,1639.59 L1070.96,1640.54 L1071.26,1641.03 L1073.45,1642.02 L1075.6,1644.27 L1075.73,1644.55 L1075.75,1644.97 L1075.66,1645.4 L1075.53,1645.69 L1075.42,1645.7 L1075.87,1646.54 L1076.47,1646.88 L1080.9,1646.88 L1081.72,1646.62 L1084.47,1644.66 L1085.54,1644.35 L1086.45,1644.7 L1086.38,1646.4 L1087.4,1646.5 L1088.32,1646.06 L1088.92,1645.4 L1089.47,1644.64 L1090.28,1643.86 L1091.06,1643.6 L1092.9,1643.52 L1093.59,1643.21 L1093.94,1642.59 L1093.77,1642.12 L1093.3,1641.83 L1092.78,1641.7 L1094.73,1639.36 L1095.69,1639.12 L1096.74,1639.24 L1097.7,1639.53 L1099.69,1639.16 L1100.6,1638.77 L1101.35,1637.9 L1101.55,1637.32 L1101.86,1635.86 L1102.13,1635.2 L1103.29,1634.15 L1103.43,1633.96 L1104.22,1633.97 L1106.07,1634.3 L1109.52,1634.23 L1111.91,1634.66 L1112.35,1634.51 L1112.74,1633.44 L1113.21,1633.14 L1116.75,1632.33 L1126.47,1632.48 L1126.81,1632.4 L1127.97,1630.82 L1128.11,1630.41 L1128.14,1629.83 L1128.02,1628.16 L1127.89,1627.82 L1127.67,1627.69 L1127.32,1627.35 L1127,1626.95 L1126.85,1626.61 L1126.76,1626.1 L1127.42,1625.81 L1128.41,1624.94 L1128.77,1624.7 L1129.37,1624.41 L1129.48,1624.38 L1129.35,1624.67 L1129.27,1625.34 L1129.13,1625.84 L1128.87,1626.22 L1128.75,1626.66 L1129,1627.35 L1129.43,1627.71 L1129.88,1627.62 L1130.81,1627.1 L1131.73,1626.94 L1134.18,1627.38 L1134.9,1627.25 L1136.53,1626.56 L1137.23,1626.67 L1137.44,1627.02 L1137.92,1628.34 L1138.17,1628.84 L1138.52,1629.24 L1139.81,1630.21 L1140.42,1630.48 L1141.29,1630.47 L1142.17,1630.27 L1142.82,1629.95 L1143.21,1629.58 L1144.65,1628.21 L1145.44,1627.78 L1146.93,1627.36 L1148.44,1627.31 L1150.83,1627.83 L1151.55,1627.71 L1151.15,1628.82 L1151.85,1629.61 L1152.97,1630.29 L1153.84,1631.07 L1152.7,1631.68 L1152.24,1633.15 L1152.52,1634.68 L1153.6,1635.48 L1154.25,1635.57 L1154.66,1635.7 L1155.03,1635.97 L1157.04,1638.03 L1159.18,1639.51 L1159.65,1639.68 L1160,1639.54 L1160.34,1639.34 L1160.79,1639.31 L1161.2,1639.51 L1161.45,1639.75 L1161.72,1639.9 L1162.17,1639.86 L1162.46,1639.59 L1163.61,1638.11 L1163.57,1637.59 L1163.45,1637.22 L1163.3,1636.9 L1163.19,1636.56 L1163.14,1636.18 L1163.1,1635.09 L1163.2,1634.92 L1163.56,1633.89 L1163.6,1633.8 L1163.51,1633.84 L1163.42,1632.43 L1163.54,1632.46 L1164.18,1631.54 L1164.2,1631.37 L1164.65,1630.11 L1164.75,1629.97 L1164.81,1629.21 L1164.81,1628.45 L1164.66,1627.66 L1164.29,1626.76 L1163.41,1625.39 L1162.86,1624.78 L1162.32,1624.41 L1161.67,1624.33 L1160.62,1624.84 L1159.99,1624.98 L1157.62,1624.63 L1156.9,1623.96 L1157.64,1622.72 L1157.65,1622.71 L1157.65,1622.71 L1157.9,1622.26 L1157.96,1621.84 L1157.96,1621.44 L1158.03,1621.04 L1158.24,1620.54 L1160.79,1616.08 L1160.58,1615.89 L1159.62,1614.57 L1158.07,1611.32 L1157.33,1610.66 L1156.99,1610.49 L1156.11,1609.65 L1155.08,1609.08 L1154.72,1608.6 L1154.47,1607.96 L1154.45,1607.87 L1154.32,1607.2 L1153.64,1605.83 L1150.57,1602.86 L1150.16,1602.25 L1149.76,1601.52 L1149.52,1600.7 L1149.57,1599.85 L1149.81,1599.4 L1150.07,1599.29 L1150.35,1599.3 L1150.6,1599.17 L1151.16,1598.21 L1151.32,1598 L1152.94,1596.77 L1153.46,1596.13 L1154.18,1594.89 L1154.55,1594.46 L1155.17,1594.1 L1157.46,1593.56 L1157.4,1593.41 L1158.03,1592.86 L1159.35,1592.06 L1160.84,1590.5 L1161.49,1590 L1165.75,1588.1 L1171.45,1586.85 L1172.82,1586 L1176.73,1582.26 L1177.27,1581.26 L1177.44,1579.93 L1177.51,1579.6 L1177.69,1579.27 L1177.85,1578.83 L1177.91,1578.18 L1177.75,1576.42 L1177.8,1575.83 L1177.97,1575.25 L1178.48,1574.04 L1178.56,1573.43 L1178.45,1572.88 L1178.03,1572.3 L1177.82,1571.18 L1177.68,1570.9 L1177.64,1570.68 L1177.85,1570.23 L1178.21,1569.87 L1179.54,1569.3 L1180.92,1569.05 L1185.27,1570.04 L1186.01,1570.39 L1186.71,1570.86 L1187.39,1571.8 L1189.13,1572.48 L1189.23,1572.58 L1189.51,1570.89 L1189.83,1570.16 L1189.97,1570.19 L1190.83,1570 L1191.15,1569.84 L1191.48,1569.49 L1191.66,1569.18 L1192.29,1567.6 L1192.51,1566.77 L1192.59,1565.87 L1192.57,1564.81 L1192.25,1562.81 L1192.32,1562.06 L1192.84,1561.5 L1192.31,1561.28 L1192.1,1561.1 L1191.61,1560.26 L1191.4,1560.35 L1191.47,1559.75 L1192.57,1557.08 L1194.25,1557.44 L1195.06,1557.83 L1195.84,1558.38 L1196.79,1559.08 L1199.65,1560.46 L1200.03,1560.83 L1200.77,1561.85 L1201.07,1562.35 L1201.29,1562.54 L1201.58,1562.64 L1201.84,1562.8 L1201.96,1563.17 L1201.91,1563.57 L1201.72,1563.64 L1201.52,1563.61 L1201.4,1563.7 L1201.15,1564.3 L1200.92,1564.4 L1200.86,1564.59 L1201.09,1565.43 L1201.29,1565.76 L1202.05,1566.61 L1202.42,1566.9 L1203.86,1567.42 L1209.76,1567.74 L1213.96,1568.96 L1214.35,1568.89 L1214.67,1568.74 L1215.47,1568.02 L1217.57,1566.97 L1218.18,1566.26 L1218.97,1563.74 L1219.55,1563.28 L1220.61,1564.39 L1222.31,1565.16 L1222.8,1565.24 L1223.42,1565.07 L1224.62,1564.39 L1225.25,1564.3 L1225.53,1564.52 L1225.61,1564.92 L1225.64,1565.33 L1225.79,1565.59 L1226.11,1565.6 L1226.73,1565.34 L1227,1565.38 L1227.49,1565.94 L1227.77,1566.52 L1228.12,1566.9 L1228.8,1566.9 L1229.4,1566.22 L1229.58,1566.02 L1229.69,1564.39 L1229.55,1562.48 L1229.59,1560.78 L1230.29,1559.47 L1232.35,1557.42 L1232.48,1556.21 L1232.85,1555.45 L1232.96,1554.68 L1233.17,1554.07 L1233.78,1553.79 L1235.95,1553.95 L1238.14,1554.68 L1238.97,1554.55 L1239.37,1553.42 L1239.27,1553.32 L1239.06,1552.93 L1238.85,1552.44 L1238.74,1552.06 L1238.78,1551.62 L1239.03,1550.81 L1239.64,1546.84 L1239.53,1545.12 L1239.24,1543.22 L1239.14,1541.44 L1239.61,1540.09 L1239.55,1539.75 L1239.53,1539.46 L1239.58,1539.25 L1239.7,1539.14 L1240.6,1538.96 L1242.93,1539.59 L1246.14,1539.62 L1246.5,1540.48 L1246.74,1541.32 L1246.78,1542.17 L1246.57,1543.08 L1247.38,1543.04"
         id="path1001" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['AT'],polycolors['AT'],labels['AT'], x1, polybary['AT'], x2, polybary['AT'] )
    if 'AT' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1309.5,1688.46 L1310.06,1688.49 L1310.6,1688.39 L1311.09,1688.39 L1311.49,1688.6 L1312.28,1689.31 L1312.66,1689.54 L1313.04,1689.55 L1313.96,1689.35 L1314.42,1689.38 L1314.87,1689.59 L1315.98,1690.33 L1316.24,1690.51 L1318.2,1691.5 L1318.82,1692.03 L1319.09,1692.01 L1319.23,1691.43 L1319.48,1691.2 L1319.82,1691.23 L1320.23,1691.44 L1321.39,1691.56 L1322.37,1692.12 L1323.24,1693.01 L1324.04,1694.14 L1324.94,1695.12 L1326.32,1695.81 L1326.95,1696.87 L1327.24,1696.84 L1327.48,1695.93 L1327.81,1696.1 L1328.99,1696.25 L1329.19,1696.32 L1329.4,1696.45 L1329.6,1696.62 L1329.81,1696.84 L1330.11,1697.07 L1330.21,1696.85 L1330.28,1696.56 L1330.51,1696.54 L1331.11,1697.04 L1331.54,1697.52 L1331.83,1697.85 L1332.34,1698.82 L1332.28,1699.81 L1332.69,1700.72 L1333.23,1701.92 L1333.55,1702.26 L1333.89,1702.5 L1335.68,1704.24 L1336.92,1706.13 L1337.38,1706.65 L1339.85,1708.3 L1340.25,1708.74 L1340.53,1709.39 L1341.08,1709.88 L1343.32,1711.25 L1343.99,1711.51 L1347.52,1711.29 L1348.03,1711.53 L1349.97,1713.62 L1350.27,1714.18 L1351,1716.12 L1351.16,1716.74 L1351.56,1717.75 L1352.26,1717.93 L1352.97,1717.96 L1353.42,1718.47 L1353.1,1718.56 L1352.58,1718.86 L1352.29,1718.96 L1352.33,1719.39 L1352.99,1719.55 L1353.08,1719.61 L1353.55,1719.95 L1354.01,1720.01 L1354.41,1719.18 L1354.76,1720.22 L1354.74,1720.78 L1354.9,1720.95 L1355.76,1720.8 L1356.76,1720.3 L1357.23,1720.45 L1357.61,1721.44 L1357.83,1721.23 L1357.98,1721.02 L1358.08,1720.8 L1358.42,1720.54 L1362.14,1720.34 L1364.02,1720.53 L1365.69,1721.09 L1367.95,1723.05 L1368.33,1723.6 L1368.43,1724.1 L1368.75,1725.01 L1368.87,1725.71 L1368.96,1726.03 L1369.08,1726.44 L1369.46,1726.64 L1370.45,1726.6 L1375.91,1727.68 L1377.27,1728.46 L1378.28,1729.85 L1378.31,1730.1 L1378.82,1729.94 L1379.02,1729.47 L1379.18,1728.96 L1379.56,1728.69 L1380.28,1728.61 L1383.28,1728.29 L1387.99,1729.06 L1388.49,1729.01 L1388.94,1728.78 L1389.64,1728.15 L1390.01,1727.98 L1393.59,1727.59 L1394.42,1727.76 L1395.86,1728.61 L1396.85,1728.51 L1397.69,1729.24 L1398.47,1729.44 L1399.19,1729.06 L1399.85,1728.58 L1400.55,1728.52 L1401.37,1729.37 L1401.39,1729.37 L1401.68,1729.42 L1401.98,1729.42 L1402.28,1729.35 L1402.59,1729.23 L1403.34,1728.49 L1404.65,1726.45 L1405.31,1725.89 L1406.33,1726.23 L1407.42,1725.68 L1409.12,1723.86 L1409.57,1723.16 L1410.31,1721.2 L1410.87,1720.4 L1410.94,1720.38 L1411.01,1719.98 L1411.06,1719.22 L1411.18,1718.89 L1412.05,1717.8 L1412.85,1717.53 L1415.08,1718.02 L1416.89,1718.72 L1417.43,1718.72 L1417.85,1718.42 L1417.99,1717.96 L1418.12,1717.37 L1418.48,1716.66 L1419.1,1717.1 L1419.3,1717.07 L1419.55,1717.03 L1420.23,1716.43 L1421.07,1716.07 L1421.53,1715.74 L1421.9,1715.19 L1422.53,1715.07 L1422.89,1716 L1422.2,1716.58 L1421.9,1716.94 L1421.56,1717.45 L1422.53,1718.05 L1423.05,1718.5 L1423.23,1718.97 L1422.97,1719.41 L1422.43,1719.6 L1421.07,1719.89 L1421.14,1721.6 L1421.09,1721.98 L1420.96,1722.26 L1420.91,1722.53 L1421.12,1722.9 L1421.86,1723.5 L1422.23,1723.99 L1423.67,1725.4 L1423.98,1726.31 L1424.38,1728.18 L1424.83,1728.93 L1425.49,1729.33 L1426.28,1729.67 L1426.98,1730.15 L1427.33,1731.01 L1427.15,1731.57 L1426.15,1733.38 L1425.71,1733.79 L1425.24,1734.38 L1425.2,1735.57 L1425.28,1736.74 L1425.19,1737.29 L1425.59,1737.54 L1426.62,1738.85 L1427.1,1739.21 L1427.9,1739.1 L1428.41,1738.59 L1428.79,1738.01 L1429.5,1737.51 L1429.85,1737.2 L1430.22,1737.04 L1430.65,1737.29 L1430.98,1737.75 L1431.15,1738.05 L1431.38,1738.3 L1431.86,1738.63 L1432.64,1738.88 L1433.28,1738.94 L1433.88,1739.19 L1434.47,1740.03 L1433.35,1741.42 L1430.72,1741.36 L1430.35,1741.35 L1429.95,1742.96 L1430.37,1744.07 L1430.82,1744.45 L1431.77,1745.24 L1432.11,1745.94 L1431.8,1746.61 L1429.62,1747.96 L1430.29,1748.87 L1431.04,1749.6 L1431.86,1750.06 L1432.75,1750.16 L1434.56,1749.93 L1435.02,1750.13 L1435.24,1750.7 L1435.36,1751.4 L1435.54,1751.96 L1436.83,1753.17 L1438.56,1753.78 L1447.64,1754.96 L1448.44,1756.04 L1449.33,1756.32 L1449.94,1757.51 L1450.01,1758.98 L1449.57,1759.55 L1449.44,1759.71 L1445.67,1760.36 L1445.44,1760.23 L1445.02,1759.74 L1444.8,1759.65 L1444.5,1759.83 L1444.37,1760.12 L1444.27,1760.42 L1444.07,1760.6 L1442.27,1761.07 L1441.32,1761.13 L1440.44,1760.92 L1439.91,1760.49 L1438.94,1759.41 L1438.39,1759.19 L1437.45,1759.54 L1437.86,1760.41 L1438.57,1761.52 L1438.51,1762.54 L1437.58,1762.85 L1435.09,1762.91 L1434.49,1763.58 L1434.62,1764.13 L1435.02,1764.57 L1435.51,1764.95 L1435.88,1765.31 L1436.31,1766.04 L1436.51,1766.63 L1436.85,1768.1 L1437.43,1769.88 L1437.55,1770.53 L1437.53,1771.12 L1437.4,1771.9 L1437.35,1772.67 L1437.51,1773.29 L1438.04,1773.67 L1438.9,1773.22 L1439.35,1773.61 L1439.64,1774.54 L1439.38,1775.02 L1438.9,1775.3 L1438.51,1775.61 L1437.96,1776.87 L1437.61,1777.45 L1437.12,1777.83 L1436.47,1777.83 L1435.47,1776.97 L1434.85,1776.85 L1433.73,1777.66 L1434,1778.89 L1435.16,1780.59 L1434.7,1780.79 L1429.62,1781.54 L1428.2,1781.25 L1427.42,1780.86 L1427.42,1780.86 L1426.26,1780.28 L1424.56,1778.84 L1423.89,1777.11 L1424.14,1776.47 L1425.01,1775.62 L1425.08,1775 L1424.98,1774.79 L1424.73,1774.51 L1424.61,1774.35 L1424.56,1774.17 L1424.53,1773.84 L1424.49,1773.71 L1424.32,1773.34 L1424.17,1773.1 L1423.97,1772.95 L1423.65,1772.87 L1422.41,1772.9 L1421.86,1772.73 L1421.26,1772.16 L1419.25,1768.99 L1419,1768.74 L1418.72,1768.61 L1418.42,1768.67 L1418.22,1768.94 L1418.14,1769.32 L1418.09,1769.7 L1417.98,1770.02 L1417.59,1770.29 L1417.22,1770.17 L1416.87,1769.79 L1416.56,1769.32 L1416.17,1768.54 L1415.94,1768.28 L1415.59,1768.17 L1415.31,1768.26 L1414.74,1768.7 L1414.45,1768.85 L1414.1,1768.88 L1413.37,1768.79 L1413.01,1768.8 L1412.39,1769.15 L1412.4,1769.27 L1412.43,1769.7 L1412.73,1770.43 L1412.91,1771.3 L1411.84,1771.38 L1410.64,1771.09 L1410.21,1770.88 L1409.47,1770.51 L1408.77,1769.91 L1408.77,1769.91 L1407.67,1768.95 L1406.84,1768.4 L1405.95,1768.11 L1402.82,1768.2 L1402.13,1767.98 L1399.98,1766.77 L1398.88,1766.46 L1397.82,1766.81 L1397.4,1767.47 L1397.38,1768.25 L1397.43,1769.04 L1397.2,1769.72 L1396.48,1770.29 L1395.56,1770.59 L1395.12,1770.63 L1395.12,1770.63 L1394.6,1770.68 L1393.73,1770.6 L1392.7,1770.25 L1390.04,1768.32 L1389.3,1768.01 L1388.62,1768.09 L1387.99,1768.51 L1385.64,1772.01 L1384.98,1772.77 L1384.63,1773.02 L1383.44,1773.38 L1382.2,1774.06 L1381.76,1774.16 L1380.84,1773.89 L1379.97,1773.25 L1377.18,1770.32 L1375.88,1769.27 L1374.5,1768.67 L1373.03,1768.73 L1372.13,1769.33 L1370.67,1771.22 L1369.81,1771.88 L1369.34,1772 L1368.36,1772 L1366.18,1772.51 L1365.86,1772.66 L1366.05,1772.24 L1366.33,1770.9 L1365.33,1771.56 L1364.81,1771.71 L1364.24,1771.57 L1364.52,1771.15 L1364.62,1770.79 L1364.54,1770.45 L1364.25,1770.06 L1363.88,1769.87 L1363.7,1770.13 L1363.62,1770.53 L1363.55,1770.75 L1362.13,1771.36 L1361.33,1771.53 L1360.69,1771.25 L1360.25,1770.94 L1358.71,1770.14 L1357.86,1769.88 L1356.04,1768.92 L1355.37,1770.85 L1355.2,1771.14 L1353.12,1771.5 L1352.67,1771.82 L1349.8,1770.81 L1348.6,1770.39 L1346.59,1769.15 L1345.57,1767.36 L1345.32,1767.61 L1344.66,1768.12 L1344.41,1768.37 L1344.55,1767.83 L1344.57,1767.45 L1344.46,1767.09 L1344.26,1766.63 L1343.91,1767.46 L1343.41,1767.19 L1342.82,1766.57 L1342.15,1766.4 L1342.37,1766.8 L1342.52,1767.22 L1341.62,1767.41 L1341.34,1767.09 L1341.32,1766.55 L1341.21,1766.06 L1340.74,1764.79 L1340.34,1764.46 L1339.23,1765.59 L1339.12,1765.89 L1338.81,1766.44 L1338.39,1767 L1338.06,1767.25 L1337.87,1767.97 L1337.47,1768.31 L1336.99,1768.57 L1336.59,1769.04 L1336.41,1769.81 L1336.49,1770.35 L1336.46,1770.81 L1335.95,1771.35 L1334.84,1771.6 L1325.84,1769.77 L1323.27,1769.91 L1321.32,1771.51 L1320.14,1774.83 L1319.62,1775.83 L1319.17,1776.22 L1318.04,1776.68 L1317.65,1777 L1317.37,1777.61 L1317.41,1777.96 L1317.53,1778.36 L1317.5,1779.09 L1317.34,1779.62 L1316.58,1781.37 L1316.24,1783.19 L1316.24,1783.19 L1316.07,1784.09 L1315.01,1784.47 L1313.35,1784.24 L1311.61,1783.65 L1310.3,1782.98 L1305.94,1779.28 L1305.09,1778.33 L1303.96,1775.97 L1303.19,1774.91 L1302.61,1774.53 L1301.63,1774.4 L1301.22,1774.15 L1301.23,1773.86 L1301.36,1773.39 L1301.34,1772.84 L1300.91,1772.32 L1300.36,1772.1 L1299.87,1772.11 L1298.88,1772.39 L1294.65,1772.61 L1292.3,1773.19 L1291.38,1774.17 L1290.96,1774.62 L1290.56,1776.02 L1290.56,1776.5 L1290.99,1780.18 L1290.87,1781.16 L1290.03,1783.93 L1290.1,1785.09 L1290.77,1786.94 L1290.68,1788.05 L1289.72,1790.05 L1289.56,1790.71 L1289.45,1791.17 L1289.71,1792.73 L1290.41,1793.93 L1291.21,1794.99 L1291.71,1796.15 L1291.58,1797.26 L1290.83,1797.68 L1289.94,1798.04 L1289.35,1798.94 L1289.49,1799.98 L1290.06,1801.02 L1291.32,1802.79 L1291.59,1803.09 L1292.69,1803.73 L1293.07,1804.28 L1293.41,1804.89 L1293.83,1805.27 L1294.47,1805.12 L1294.89,1804.94 L1295.68,1804.82 L1296.05,1804.66 L1296.35,1804.24 L1296.55,1803.68 L1296.84,1803.27 L1297.39,1803.26 L1297.72,1803.59 L1297.93,1804.16 L1298.1,1804.8 L1298.29,1805.32 L1298.66,1805.65 L1299.03,1805.52 L1299.38,1805.28 L1299.72,1805.28 L1300.25,1805.92 L1300.7,1807.45 L1301.36,1808.27 L1301.84,1808.53 L1302.41,1808.67 L1303.51,1808.74 L1304.07,1808.96 L1304.47,1809.51 L1304.75,1810.27 L1304.98,1811.1 L1304.91,1811.16 L1304.56,1811.38 L1304.42,1812.22 L1303.76,1813.22 L1303.48,1814.25 L1304.48,1815.15 L1305.41,1815.2 L1306.97,1814.34 L1307.93,1814.42 L1308.5,1815.06 L1308.88,1816.13 L1309.21,1817.33 L1310.55,1820.3 L1310.94,1821.46 L1311.03,1822.41 L1310.79,1822.88 L1310.43,1823.16 L1310.16,1823.44 L1310.17,1823.93 L1310.32,1823.72 L1310.82,1823.7 L1311.65,1823.84 L1312.63,1824.36 L1313.15,1824.8 L1313.32,1825.36 L1313.33,1825.41 L1313.29,1826.46 L1313.18,1826.99 L1312.8,1828.04 L1312.73,1828.73 L1312.84,1829.67 L1313.42,1831.58 L1314.47,1834.18 L1315.29,1835.19 L1317.4,1837.12 L1318.14,1838.19 L1319.33,1840.54 L1320.08,1841.58 L1321.01,1842.37 L1321.55,1842.72 L1322.01,1842.94 L1322.5,1842.94 L1323.67,1842.76 L1324.13,1842.9 L1324.79,1843.8 L1325.07,1844.7 L1325.5,1845.4 L1327.53,1845.94 L1328.37,1846.48 L1329.13,1847.24 L1329.68,1847.93 L1331.12,1849.7 L1333.51,1851.77 L1335.15,1853.67 L1337.16,1855.23 L1337.57,1856.1 L1337.83,1857.13 L1338.32,1858.42 L1339.06,1859.39 L1344.42,1862.92 L1352.22,1869.57 L1354.64,1871.92 L1356.2,1873.08 L1357.29,1873.88 L1360.06,1875.11 L1364.51,1875.37 L1366,1876.18 L1366.87,1877.71 L1366.5,1883.01 L1367.79,1886.01 L1369.8,1888.52 L1371.82,1890.06 L1372.79,1890.56 L1373.32,1890.95 L1373.68,1891.58 L1374.18,1892.81 L1374.82,1893.95 L1375.59,1894.71 L1376.02,1894.93 L1382.63,1898.4 L1383.17,1898.93 L1384.72,1900.42 L1386.43,1903.56 L1386.43,1903.56 L1386.44,1903.57 L1386.45,1903.57 L1386.77,1905.28 L1385.57,1906.36 L1383.15,1907.13 L1383.15,1907.12 L1383.12,1907.11 L1381.65,1906.45 L1380.99,1906.81 L1380.08,1906.38 L1379.22,1905.64 L1378.69,1905.04 L1377.95,1903.95 L1377.71,1903.2 L1378.04,1902.61 L1379.03,1901.97 L1378.37,1901.79 L1377.65,1902.14 L1376.86,1902.7 L1375.98,1903.13 L1376.29,1902.93 L1377.12,1902.16 L1376.68,1902.2 L1376.22,1902.17 L1375.82,1901.96 L1375.49,1901.47 L1375.79,1901.01 L1375.45,1900.46 L1375.37,1900.19 L1374.46,1901.14 L1372.79,1899.9 L1369.99,1896.83 L1363.12,1894.08 L1361.44,1892.96 L1360.65,1892.63 L1360.28,1892.39 L1359.93,1891.6 L1359.56,1891.3 L1359.08,1891.13 L1358.61,1891.09 L1358.3,1890.96 L1357.95,1890.63 L1357.42,1889.92 L1357.31,1889.66 L1357.08,1888.87 L1356.99,1888.64 L1356.78,1888.55 L1356.19,1888.44 L1355.98,1888.35 L1351.21,1883.2 L1349.87,1882.46 L1349.36,1882.05 L1349,1881.67 L1348.61,1881.41 L1342.71,1881.1 L1339.95,1880.17 L1336.81,1879.78 L1332.43,1877.15 L1332.35,1877 L1332.29,1876.71 L1332.15,1876.45 L1331.89,1876.34 L1331.62,1876.44 L1331.24,1876.76 L1331.01,1876.85 L1329.26,1876.57 L1325.85,1876.87 L1325.81,1876.44 L1326.72,1876.35 L1327.73,1876.01 L1328.68,1875.49 L1329.43,1874.85 L1328.64,1874.43 L1327.58,1874.28 L1323.16,1874.57 L1322.64,1874.78 L1321.84,1875.38 L1321.53,1875.52 L1320.71,1875.72 L1318.87,1876.47 L1313.35,1877.05 L1313.38,1877.48 L1314.91,1877.75 L1315.59,1878.15 L1316,1879.01 L1313.2,1879.24 L1313.36,1879.9 L1312.69,1879.89 L1311.17,1879.41 L1309.96,1879.44 L1309.43,1879.22 L1308.74,1878.71 L1308.46,1878.73 L1308.49,1879.16 L1307.91,1878.82 L1306.9,1878.91 L1306.54,1878.46 L1306.51,1878.57 L1306.53,1878.75 L1306.48,1878.9 L1306.26,1878.91 L1306.29,1878.21 L1306.52,1878.08 L1306.88,1878.06 L1307.28,1877.76 L1307.4,1877.36 L1307.25,1877.07 L1307.14,1876.76 L1307.33,1876.26 L1306.6,1876.41 L1306.34,1876.53 L1306.1,1876.78 L1305.91,1876.19 L1305.51,1875.91 L1304.45,1875.64 L1305.2,1875.49 L1305.27,1875.06 L1304.85,1874.71 L1304.06,1874.77 L1304.03,1874.39 L1304.23,1874.25 L1304.44,1874.03 L1304.63,1873.87 L1304.16,1873.57 L1303.96,1873.5 L1304.26,1873.09 L1303.9,1872.69 L1304.2,1872.2 L1303.84,1871.3 L1304.3,1870.92 L1305.97,1870.82 L1305.93,1870.35 L1303.86,1870.08 L1303.11,1870.15 L1303.63,1869.76 L1304.08,1869.76 L1304.46,1869.93 L1304.81,1870.02 L1305.28,1869.8 L1305.18,1869.4 L1304.75,1869 L1304.25,1868.78 L1304.86,1868.49 L1305.12,1868.01 L1304.99,1867.47 L1304.43,1867.02 L1304.55,1868.18 L1304.17,1868.28 L1303,1867.44 L1302.33,1866.93 L1302.02,1866.96 L1301.59,1866.92 L1301.31,1866.84 L1301.01,1866.55 L1300.31,1865.62 L1299.98,1865.27 L1298.07,1864.39 L1297.52,1863.93 L1296.54,1863.66 L1293.6,1863.67 L1292.93,1863.02 L1292.35,1861.65 L1291.25,1861.24 L1290.07,1861 L1289.3,1860.11 L1290.7,1860.61 L1291.42,1860.74 L1292.14,1860.75 L1292.11,1860.29 L1291.83,1860.19 L1291.65,1860.22 L1291.56,1860.29 L1291.52,1860.34 L1290.22,1859.38 L1287.19,1858.14 L1285.89,1857.17 L1285.81,1857.11 L1284.8,1855.76 L1284.18,1855.14 L1283.5,1854.92 L1282.81,1854.87 L1282.4,1854.77 L1282.05,1854.64 L1281.53,1854.28 L1281.26,1853.9 L1281.04,1853.48 L1280.69,1852.99 L1278.03,1850.2 L1276.74,1848.52 L1275.85,1847.73 L1274.92,1847.43 L1272.87,1845.25 L1272.89,1845.05 L1270.72,1842.99 L1270.19,1842.73 L1269.76,1842.42 L1269.55,1841.99 L1269.68,1841.35 L1268.94,1840.89 L1266.43,1838.63 L1267.07,1837.47 L1266.61,1836.12 L1265.6,1834.99 L1264.61,1834.45 L1265.1,1833.76 L1265.9,1833.23 L1266.67,1833.3 L1267.08,1834.47 L1267.65,1835.14 L1268.57,1834.61 L1268.95,1833.62 L1267.9,1832.9 L1268.23,1832.79 L1268.28,1832.67 L1268.29,1832.66 L1268.51,1832.86 L1268.84,1832.84 L1268.55,1832.1 L1268.4,1831.85 L1270.71,1833.17 L1271.73,1834.32 L1272.26,1834.7 L1272.97,1834.72 L1272.22,1832.6 L1272.6,1832.45 L1272.79,1832.26 L1272.9,1832.01 L1273.1,1831.68 L1272.49,1831.53 L1271.96,1831.09 L1271.49,1830.46 L1271.1,1829.73 L1271.99,1829.98 L1273.32,1831.03 L1274.62,1831.41 L1277.96,1833.01 L1278.69,1833.23 L1282.28,1833.51 L1283.25,1833.35 L1283.35,1832.65 L1282.7,1832.27 L1279.73,1832.06 L1279.49,1831.89 L1278.02,1830.24 L1277.64,1830.01 L1273.1,1828.22 L1272.43,1827.62 L1265.8,1821.71 L1263.51,1818.92 L1262.17,1817.62 L1260.87,1817.12 L1260.67,1816.83 L1259.14,1815.76 L1258.78,1815.75 L1258.48,1815.22 L1253.58,1808.56 L1252.67,1806.53 L1253.13,1805.72 L1253.01,1804.8 L1252.66,1803.73 L1252.42,1802.47 L1252.1,1799.04 L1252.14,1797.94 L1252.85,1793.95 L1253.21,1792.97 L1253.44,1791.77 L1253.02,1790.38 L1250.09,1784.63 L1249.86,1783.57 L1249.37,1782.86 L1248.66,1782.38 L1245.86,1781.1 L1240.86,1777.71 L1238.42,1774.87 L1237.79,1774.57 L1237,1773.84 L1236.63,1773.01 L1237.19,1772.39 L1236.66,1771.9 L1234.98,1770.81 L1235.83,1771.7 L1235.97,1771.8 L1235.85,1772.3 L1235.48,1772.59 L1235.03,1772.61 L1234.63,1772.33 L1233.98,1771.68 L1229.07,1769.55 L1227.35,1769.27 L1226.54,1768.88 L1225.82,1768.78 L1225.05,1769.45 L1224.22,1770.96 L1223.5,1772.9 L1223.02,1775.08 L1222.78,1779.42 L1222.31,1781.44 L1222.29,1781.54 L1222.23,1781.77 L1221.28,1783.27 L1219.93,1782.83 L1220.68,1783.66 L1220.63,1784.6 L1220.12,1785.51 L1219.45,1786.27 L1219.15,1785.82 L1219,1786.36 L1219.02,1786.79 L1219.26,1787.95 L1219.08,1787.85 L1218.77,1787.72 L1218.61,1787.6 L1218.63,1787.99 L1219.36,1788.75 L1219.73,1790.07 L1219.7,1791.52 L1219.21,1792.69 L1218.72,1793.1 L1218.36,1793.07 L1218.01,1792.88 L1217.52,1792.78 L1217,1793.01 L1216.51,1793.52 L1216.13,1794.12 L1215.94,1794.61 L1215.64,1794.63 L1215.7,1793.41 L1216.02,1792.83 L1216.19,1792.3 L1215.76,1791.17 L1214.78,1790.37 L1214.62,1789.91 L1214.67,1789.46 L1214.6,1789.16 L1214.09,1789.13 L1214.29,1789.85 L1214.16,1790.97 L1214.38,1791.46 L1214.86,1791.74 L1215.28,1791.76 L1215.62,1791.89 L1215.83,1792.48 L1215.37,1792.98 L1214.82,1794.24 L1214.55,1794.57 L1214.35,1794.27 L1214.1,1794.29 L1214.3,1794.71 L1214.4,1795.23 L1214.38,1795.81 L1214.21,1796.41 L1213.85,1796.32 L1213.66,1796.36 L1213.68,1796.54 L1213.93,1796.85 L1213.95,1797.28 L1213.72,1797.31 L1213.67,1797.37 L1213.69,1797.51 L1213.65,1797.76 L1213.2,1797.53 L1212.23,1797.86 L1211.52,1797.88 L1211.9,1798.14 L1212.14,1798.41 L1212.49,1799.06 L1212.19,1799.08 L1212.21,1799.47 L1212.51,1799.45 L1212.53,1799.92 L1212.23,1800.15 L1212.04,1800.43 L1211.69,1801.24 L1211.8,1801.7 L1211.89,1801.83 L1211.89,1801.99 L1211.76,1802.52 L1212.23,1802.47 L1212.51,1802.57 L1212.73,1802.85 L1213.02,1803.3 L1211.41,1803.46 L1210.63,1803.38 L1209.95,1803 L1209.88,1803.26 L1209.88,1803.33 L1209.67,1803.48 L1209.35,1802.5 L1208.94,1802.4 L1208.5,1802.54 L1208.07,1802.29 L1207.78,1802.73 L1208.26,1802.91 L1208.75,1803.26 L1209,1803.77 L1208.81,1804.41 L1209.26,1805.41 L1209.48,1805.66 L1208.83,1806.06 L1208.64,1805.47 L1208.6,1804.46 L1208.43,1803.54 L1208.1,1803.31 L1206.75,1802.84 L1206.26,1802.81 L1206.35,1802.59 L1206.44,1802.15 L1206.53,1801.94 L1206.38,1801.92 L1206.06,1801.97 L1205.92,1801.97 L1206.2,1801.53 L1205.87,1801.08 L1205.39,1801.22 L1204.68,1801.02 L1203.93,1800.56 L1203.36,1799.93 L1203.67,1799.52 L1204.16,1800.17 L1204.82,1800.32 L1205.44,1799.93 L1205.77,1798.99 L1204.58,1799.34 L1204.12,1799.25 L1203.63,1798.67 L1203.61,1798.29 L1203.79,1798.1 L1203.98,1797.68 L1204.17,1797.37 L1203.64,1796.8 L1202.51,1794.43 L1200.59,1792.58 L1198.15,1789.49 L1197.79,1789.21 L1197.2,1788.61 L1196.42,1788.07 L1195.5,1787.95 L1195.68,1787.54 L1195.89,1786.91 L1196.05,1786.64 L1194.9,1785.18 L1196.01,1784.37 L1199.88,1783.9 L1199,1783.53 L1197.97,1783.44 L1195.91,1783.62 L1194.82,1784 L1194.32,1784 L1194.09,1783.5 L1194.14,1782.84 L1194.24,1782.29 L1194.16,1781.88 L1193.7,1781.64 L1193.88,1781.2 L1193.9,1780.89 L1193.72,1780.62 L1193.34,1780.34 L1193.32,1779.95 L1193.67,1779.7 L1193.83,1779.31 L1193.8,1778.8 L1193.54,1778.2 L1193.8,1777.57 L1193.75,1776.93 L1193.42,1776.4 L1192.84,1776.09 L1192.97,1775.71 L1193.2,1775.38 L1193.53,1775.17 L1194,1775.15 L1193.98,1774.76 L1193.53,1774.57 L1193.16,1774.31 L1192.9,1773.96 L1192.72,1773.5 L1193.17,1773.4 L1193.55,1773.17 L1194.19,1772.58 L1193.66,1772.34 L1192.47,1772.35 L1192.06,1772.29 L1191.55,1771.82 L1191.1,1771.1 L1191.04,1770.44 L1191.66,1770.14 L1191.64,1769.75 L1191.1,1769.22 L1189.62,1765.54 L1189.7,1765.42 L1189.84,1765.27 L1189.92,1765.1 L1189.7,1764.84 L1189.68,1764.79 L1189.78,1764.69 L1189.88,1764.21 L1189.1,1762.59 L1188.92,1761.68 L1189.13,1760.84 L1189.4,1760.82 L1189.85,1761.15 L1191.89,1762.02 L1192.82,1762.1 L1194.64,1763.88 L1196,1763.75 L1200.36,1763.33 L1202.5,1764.54 L1203.12,1765.1 L1203.82,1765.29 L1206.19,1765.5 L1206.64,1765.13 L1207,1764.51 L1207.6,1763.85 L1209.62,1762.87 L1210.14,1762.07 L1209.16,1761 L1209.12,1760.33 L1209.25,1759.88 L1209.55,1759.66 L1210.24,1759.77 L1211.42,1759.96 L1212.06,1760.25 L1212.68,1760.81 L1212.68,1760.81 L1213.82,1761.56 L1215,1761.89 L1216.05,1761.89 L1216.07,1761.89 L1216.18,1761.89 L1217.3,1761.62 L1219.34,1760.52 L1220.45,1760.13 L1221.42,1760.32 L1223.21,1760.98 L1225.27,1761.25 L1227.29,1760.95 L1228.92,1759.89 L1229.69,1759.08 L1231.34,1757.72 L1231.86,1756.63 L1232.04,1755.74 L1232.14,1754.32 L1232.16,1754.05 L1232.39,1753.24 L1232.76,1752.61 L1233.79,1751.31 L1234.22,1750.61 L1234.69,1749.32 L1235.01,1748.92 L1235.23,1748.81 L1235.74,1748.56 L1236.23,1748.8 L1236.38,1749.75 L1236.4,1750.9 L1236.49,1751.74 L1236.94,1752.5 L1237.54,1753.06 L1239.12,1754.16 L1239.44,1754.27 L1239.72,1754.42 L1239.74,1754.42 L1240.01,1754.75 L1240.13,1755.23 L1240.04,1756.34 L1240.07,1756.66 L1240.99,1757.35 L1245.17,1758.88 L1245.17,1758.88 L1245.18,1758.89 L1245.18,1758.89 L1245.98,1760.59 L1247.84,1760.85 L1249.67,1760.06 L1250.43,1758.55 L1250.5,1757.21 L1251.3,1757.13 L1252.34,1757.71 L1253.14,1758.37 L1253.79,1758.41 L1254.19,1758.44 L1254.69,1758.63 L1255.16,1759 L1257.3,1758.91 L1257.76,1759.26 L1261.16,1761.77 L1263.15,1761.91 L1264.93,1761.11 L1267.32,1760.03 L1268.75,1759.96 L1269.23,1759.78 L1269.59,1759.4 L1270.3,1758.24 L1270.69,1757.84 L1268.38,1756.52 L1267.68,1755.5 L1267.24,1754.18 L1266.86,1752.57 L1266.84,1752.52 L1266.55,1751.84 L1266.2,1751.19 L1266.12,1750.68 L1266.64,1750.36 L1266.93,1750.08 L1267.05,1749.58 L1267.28,1749.1 L1268.53,1748.6 L1269.06,1748.27 L1269.65,1748.01 L1270.56,1747.95 L1270.28,1747.42 L1269.48,1746.29 L1269.39,1746.2 L1268.63,1745.44 L1268.49,1745.33 L1268.33,1745.42 L1267.81,1745.6 L1267.62,1745.79 L1267.58,1746.04 L1267.41,1746.18 L1266.83,1746.01 L1266.42,1745.75 L1266.17,1745.46 L1265.68,1744.66 L1265.23,1744.27 L1264.9,1744.14 L1264.79,1743.88 L1264.83,1743.76 L1265.01,1743.14 L1265.35,1742.68 L1267.05,1741.57 L1268.21,1741.09 L1272.39,1739.36 L1272.42,1739.34 L1272.88,1738.9 L1272.78,1738.31 L1272.56,1737.67 L1272.73,1737.07 L1273.19,1736.81 L1273.68,1736.83 L1274.19,1736.96 L1274.7,1737.01 L1275.12,1736.82 L1275.88,1736.09 L1276.29,1735.85 L1277.44,1735.96 L1279.1,1736.11 L1280.79,1735.93 L1281.62,1735.62 L1282.48,1735.08 L1282.87,1734.42 L1282.78,1733.53 L1282.18,1732.32 L1281.93,1731.54 L1282.05,1730.78 L1282.3,1730.02 L1282.47,1729.22 L1282.46,1728.18 L1282.25,1726.28 L1282.1,1724.99 L1282.1,1724.98 L1282.35,1723.7 L1282.8,1722.9 L1282.94,1722.23 L1282.69,1721.91 L1282.26,1721.35 L1281.7,1721 L1280.45,1720.59 L1279.89,1720.31 L1279.52,1719.97 L1278.6,1719.13 L1277.89,1717.77 L1277.78,1716.21 L1278.28,1714.39 L1278.97,1712.8 L1279.11,1712.56 L1279.63,1711.75 L1280.53,1711.2 L1281.25,1711.2 L1284.38,1711.19 L1285.17,1710.59 L1285.93,1709.04 L1286.36,1708.66 L1286.5,1708.53 L1287.13,1708.17 L1287.81,1707.94 L1289.76,1707.72 L1289.82,1707.72 L1289.95,1707.7 L1291.4,1706.75 L1292.64,1705.93 L1294.06,1705.56 L1294.55,1705.38 L1294.75,1705.3 L1295.62,1704.78 L1296.42,1704.11 L1296.85,1703.55 L1296.95,1703.43 L1297.2,1702.55 L1297.14,1701.74 L1296.97,1700.85 L1296.9,1699.74 L1298.25,1699.91 L1298.5,1699.94 L1299,1699.8 L1299.38,1699.43 L1299.67,1698.92 L1300,1698.51 L1300.53,1698.36 L1300.96,1698.53 L1302.07,1699.43 L1302.72,1699.74 L1303.46,1699.85 L1303.82,1699.8 L1305.33,1699.3 L1306.32,1699.23 L1306.5,1699.02 L1306.48,1698.87 L1306.38,1698.33 L1306.15,1698.06 L1305.38,1697.63 L1305.09,1697.34 L1304.93,1696.83 L1304.93,1695.82 L1304.86,1695.32 L1304.21,1693.64 L1303.98,1692.81 L1303.97,1692.42 L1303.95,1691.89 L1304.97,1690.54 L1305.09,1690.38 L1306.4,1689.74 L1307.02,1689.27 L1307.83,1688.92 L1308.23,1688.36 L1308.38,1688.15 L1308.72,1688.57 L1308.74,1688.59 L1308.97,1688.1 L1309.5,1688.46"
         id="path1003" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['HR'],polycolors['HR'],labels['HR'], x1, polybary['HR'], x2, polybary['HR'] )
    if 'HR' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M239.597,1780.97 L239.74,1781.28 L240.093,1781.27 L240.522,1781.19 L240.629,1781.04 L240.671,1780.74 L240.808,1780.47 L241.182,1780.43 L241.48,1780.46 L242.313,1780.27 L242.276,1781.13 L242.88,1783.52 L242.534,1784.45 L242.957,1784.39 L243.683,1783.75 L244.238,1782.97 L244.725,1783.3 L245.187,1783.08 L245.719,1782.7 L246.406,1782.61 L248.094,1783.22 L248.576,1783.53 L251.747,1786.48 L253.383,1787.59 L255.059,1793.23 L256.923,1794.76 L256.716,1794.97 L256.58,1795.15 L256.416,1795.29 L256.115,1795.4 L256.015,1795.86 L256.544,1796.27 L256.939,1796.19 L257.348,1795.94 L257.926,1795.83 L258.573,1795.98 L259.412,1796.5 L261.032,1797.16 L263.756,1797.68 L264.588,1797.39 L265.279,1798 L266.014,1798.31 L266.459,1798.83 L266.261,1800.05 L265.812,1800.65 L265.183,1801.26 L264.768,1802.06 L264.938,1803.21 L264.971,1803.22 L265.216,1803.28 L265.68,1802.34 L266.042,1801.35 L266.478,1800.58 L267.169,1800.26 L267.877,1798.92 L270.539,1798.94 L274.91,1799.78 L275.537,1800.15 L278.233,1800.54 L279.94,1801.5 L281.059,1801.89 L283.775,1801.14 L285.981,1801.88 L288.071,1803.21 L289.503,1804.45 L290.039,1804.21 L290.57,1804.2 L291.689,1804.47 L291.67,1804.16 L291.702,1803.45 L291.679,1803.15 L292.264,1803.42 L293.178,1804.26 L293.777,1804.54 L296.155,1804.59 L298.438,1805.1 L299.109,1805.12 L300.381,1804.83 L301.497,1804.79 L301.834,1804.53 L302.229,1804.29 L302.854,1804.33 L304.043,1805.7 L304.577,1806.02 L307.942,1807.16 L308.705,1806.93 L309.334,1807.43 L310.06,1807.28 L310.831,1806.9 L311.594,1806.73 L316.745,1807.85 L317.089,1807.34 L316.884,1806.82 L316.795,1806.3 L317.476,1805.78 L317.43,1805.56 L317.398,1805.09 L317.345,1804.88 L318.773,1805.19 L319.346,1805.02 L319.696,1804.32 L320.114,1803.81 L320.808,1804.01 L322.105,1805.03 L322.891,1806.07 L324.746,1809.5 L325.314,1810.96 L325.619,1811.02 L326.024,1810.2 L326.308,1810.53 L326.616,1811.71 L328.845,1812.58 L337.87,1814.09 L339.149,1814.74 L339.122,1815.62 L338.361,1816.48 L337.449,1817.05 L337.673,1817.54 L338.184,1817.35 L339.183,1817.23 L339.683,1817.09 L340.087,1816.75 L340.295,1816.37 L340.598,1816.1 L341.283,1816.12 L343.12,1816.69 L343.776,1817.09 L346.569,1819.91 L347.978,1820.78 L351.783,1821.75 L352.794,1822.27 L353.119,1823.15 L353.721,1822.91 L356.655,1824.5 L357.039,1824.45 L357.796,1824.01 L358.329,1824.02 L358.783,1824.25 L359.69,1825 L362.735,1826.31 L366.052,1828.76 L366.896,1829.22 L376.68,1832.08 L376.862,1832.13 L377.835,1833.52 L378.938,1832.78 L380.246,1832.78 L381.419,1833.37 L382.134,1834.36 L382.416,1834.42 L383.026,1833.74 L383.486,1833.6 L385.524,1834.25 L391.189,1834.73 L392.599,1834 L395.587,1833.65 L396.679,1833.69 L398.682,1834.07 L398.969,1833.95 L399.06,1833.67 L399.185,1833.4 L399.572,1833.33 L399.87,1833.5 L400.819,1834.48 L401.092,1834.23 L401.306,1834.19 L401.523,1834.23 L401.807,1834.23 L401.628,1834.08 L401.28,1833.66 L401.348,1833.27 L403.093,1832.44 L403.927,1832.24 L404.624,1832.59 L405.246,1832.35 L405.364,1832.44 L405.539,1832.76 L406.981,1832.2 L408.376,1832.16 L409.658,1832.68 L410.76,1833.74 L408.66,1834.26 L407.745,1834.65 L407.03,1835.21 L407.399,1835.86 L407.575,1836.07 L407.792,1836.26 L407.631,1836.28 L407.51,1836.25 L407.429,1836.3 L407.392,1836.58 L407.884,1836.69 L408.302,1836.7 L409.016,1836.49 L409.383,1836.24 L410.128,1835.5 L410.334,1835.36 L410.703,1835.57 L411.017,1836.25 L411.403,1836.46 L411.638,1836.25 L411.631,1835.67 L411.469,1835.05 L411.223,1834.7 L411.29,1834.31 L411.9,1834.52 L412.144,1834.66 L412.416,1834.92 L412.546,1834.62 L412.606,1834.64 L412.816,1834.6 L412.562,1834.08 L413.09,1833.95 L414.428,1834.1 L414.993,1833.88 L415.514,1833.52 L416.049,1833.36 L419.665,1832.91 L420.772,1833.2 L420.505,1834.24 L421.267,1834.43 L421.791,1834.99 L422.191,1835.6 L422.567,1835.92 L425.554,1837.02 L425.746,1837.28 L425.836,1837.62 L425.823,1838.01 L425.666,1838.54 L425.472,1838.59 L425.203,1838.47 L424.823,1838.5 L424.665,1838.3 L424.289,1837.95 L423.887,1837.75 L423.637,1838.04 L423.424,1838.37 L423.005,1838.64 L422.222,1838.93 L423.086,1839.26 L423.476,1840.05 L423.851,1840.39 L424.673,1839.38 L425.036,1840.19 L425.652,1840.36 L427.053,1840.2 L427.68,1840.44 L429.693,1841.58 L432.381,1842.46 L434.29,1843.59 L434.856,1843.81 L435.283,1844.16 L436.123,1845.53 L437.23,1846.11 L438.055,1846.54 L438.897,1846.8 L440.055,1846.19 L441.329,1847.33 L443.243,1849.66 L443.319,1849.2 L443.031,1847.62 L443.003,1846.83 L442.813,1846.34 L442.854,1846.09 L443.125,1845.84 L443.757,1845.75 L444.379,1845.21 L446.529,1844.35 L447.28,1844.27 L447.355,1843.81 L447.052,1843.76 L447.114,1843.38 L447.931,1843.07 L449.065,1843.14 L451.339,1843.64 L453.494,1843.8 L454.5,1843.58 L455.324,1843.43 L455.875,1843.66 L456.7,1844.53 L457.445,1845.57 L457.727,1846.3 L458.001,1847.93 L457.968,1849.26 L458.188,1849.05 L458.717,1847.98 L458.801,1847.69 L458.845,1847.32 L458.944,1846.98 L459.203,1846.76 L459.747,1846.76 L460.257,1847 L460.726,1847.3 L462.087,1847.79 L464.517,1849.36 L466.382,1850.09 L467.331,1850.66 L467.982,1852.12 L472.025,1854.92 L473.758,1855.77 L475.637,1856.29 L477.634,1856.39 L479.196,1856.22 L479.726,1856.26 L479.96,1856.49 L480.207,1856.98 L480.545,1857.49 L481.067,1857.77 L481.197,1857.72 L481.365,1857.61 L481.604,1857.5 L481.915,1857.48 L482.11,1857.65 L482.167,1857.96 L482.257,1858.26 L482.554,1858.45 L484.016,1857.63 L484.607,1857.53 L485.57,1857.54 L488.943,1856.89 L490.072,1856.39 L490.623,1856.24 L491.283,1856.37 L491.755,1856.89 L491.986,1857.55 L492.296,1857.9 L492.998,1857.55 L492.774,1857.32 L492.452,1856.82 L492.181,1856.52 L494.143,1855.99 L496.063,1854.72 L497.821,1854.1 L498.512,1854.79 L498.298,1855.49 L498.072,1856.78 L497.972,1857.54 L499.172,1858.37 L499.62,1859.23 L499.594,1860.45 L499.676,1860.74 L499.748,1860.99 L500.248,1860.98 L500.816,1860.81 L501.805,1860.83 L502.779,1860.59 L503.184,1860.7 L503.98,1861.24 L504.379,1861.63 L504.329,1861.92 L504.148,1862.28 L504.143,1862.83 L504.317,1863.94 L504.5,1864.26 L505.014,1864.59 L505.491,1864.74 L506.071,1864.76 L506.605,1864.55 L506.941,1864.05 L507.452,1863.05 L508.186,1862.75 L509.077,1862.92 L510.421,1863.56 L512.162,1864.84 L512.677,1864.82 L513.203,1864.72 L513.743,1864.81 L514.849,1866.35 L514.698,1868.64 L513.854,1871.02 L512.884,1872.87 L512.2,1873.55 L510.045,1875.27 L509.597,1875.74 L509.973,1877.17 L510.662,1878.22 L511.582,1878.95 L512.65,1879.42 L513.878,1879.72 L514.685,1879.58 L515.191,1878.81 L515.532,1877.2 L516.022,1876.01 L516.928,1875.25 L517.999,1875.02 L518.984,1875.45 L517.919,1876.12 L517.437,1876.87 L517.523,1877.73 L518.164,1878.73 L518.944,1879.38 L521.509,1880.24 L522.947,1881.34 L524.258,1882.77 L525.984,1882.87 L526.357,1883.08 L527.041,1883.67 L527.342,1884.01 L528.887,1884.44 L529.611,1884.77 L530.076,1885.5 L531.496,1886.92 L533.157,1887.74 L536.74,1888.57 L540.333,1888.45 L541.494,1889.04 L541.929,1890.85 L542.052,1891.36 L541.97,1891.79 L541.779,1892.06 L541.685,1892.38 L541.863,1892.98 L542.266,1893.5 L542.706,1893.62 L543.159,1893.62 L543.611,1893.76 L545.044,1894.95 L545.764,1895.75 L547.219,1898.37 L547.25,1898.64 L547.127,1898.9 L547.087,1899.13 L547.37,1899.31 L548.11,1899.42 L548.303,1899.51 L548.517,1899.92 L548.4,1900.31 L548.345,1900.68 L548.747,1901.03 L549.113,1901.03 L549.947,1900.61 L550.365,1900.53 L550.64,1899.07 L551.798,1899.27 L554.109,1900.76 L555.518,1900.85 L556.707,1900.34 L559.125,1898.92 L559.469,1899.38 L559.844,1899.29 L560.221,1898.86 L560.548,1898.29 L561.165,1899.04 L561.261,1899.12 L563.706,1901.09 L564.102,1902.32 L565.307,1902.63 L566.676,1902.45 L567.541,1902.21 L567.303,1903.27 L567.434,1904.38 L567.87,1905.37 L568.545,1906.09 L568.782,1906.47 L568.786,1906.92 L568.863,1907.33 L569.318,1907.61 L569.879,1907.86 L570.159,1908.18 L570.333,1908.52 L570.563,1908.83 L572.28,1909.79 L573.883,1909.85 L577.576,1909.03 L580.434,1908.98 L581.364,1908.52 L582.12,1908.5 L584.478,1909.45 L585.165,1909.91 L585.936,1911.77 L586.547,1912.71 L587.24,1912.7 L588.744,1911.06 L589.653,1910.35 L590.52,1910.16 L591.656,1911.4 L591.733,1911.61 L591.686,1911.87 L591.689,1912.13 L591.887,1912.37 L592.782,1912.66 L593.681,1912.73 L595.483,1912.5 L596.205,1912.91 L601.936,1913.47 L603.535,1913.89 L604.162,1913.75 L604.527,1913.27 L604.636,1913.13 L604.8,1912.14 L604.597,1911.17 L604.128,1910.41 L603.507,1910.1 L603.859,1909.51 L604.06,1909.27 L604.432,1909.04 L604.007,1908.28 L604.853,1906.63 L604.908,1905.29 L605.229,1904.43 L606.331,1904.14 L607.594,1904.21 L608.388,1904.39 L609.894,1905.21 L610.672,1905.48 L611.571,1905.5 L612.34,1905.74 L614.182,1907.15 L615.479,1907.84 L615.841,1908.19 L616.488,1909 L616.956,1909.27 L617.417,1909.09 L617.887,1908.8 L618.373,1908.7 L618.744,1908.9 L619.431,1909.61 L619.758,1909.86 L623.633,1910.59 L624.556,1911 L625.285,1911.82 L625.636,1912.77 L625.875,1913.77 L626.25,1914.68 L626.973,1915.34 L627.473,1915.47 L629.058,1915.16 L634.706,1915.83 L635.75,1916.25 L635.456,1917.11 L635.661,1917.38 L636.09,1917.46 L636.452,1917.71 L637.107,1919.26 L637.859,1920.57 L638.405,1922.79 L638.978,1923.73 L638.734,1924.08 L638.41,1924.67 L638.291,1925.25 L638.648,1925.56 L638.618,1925.79 L637.717,1927.02 L637.49,1927.71 L638.541,1927.69 L639.273,1928.5 L639.326,1929.49 L638.348,1930.04 L638.079,1930.08 L637.261,1930.34 L638.369,1932.56 L638.286,1933.34 L638.773,1933.79 L641.526,1934.47 L641.907,1934.47 L642.472,1934.23 L642.8,1933.91 L643.037,1933.58 L643.325,1933.32 L646.27,1933.27 L647.811,1932.81 L648.484,1931.2 L648.671,1931.02 L648.816,1930.97 L648.939,1931.03 L649.038,1931.24 L649.565,1931.5 L650.123,1931.55 L650.662,1931.37 L651.153,1930.91 L651.282,1931.01 L651.359,1931.15 L651.376,1931.31 L651.333,1931.49 L652.144,1931.94 L654.966,1932.65 L655.892,1933.15 L658.3,1935.2 L659.356,1935.57 L660.214,1935.71 L660.898,1936.1 L661.423,1937.21 L661.797,1939.59 L662.14,1940.6 L662.969,1941.48 L663.501,1941.91 L664.116,1942.07 L665.333,1942.07 L666.111,1941.95 L666.68,1941.67 L667.539,1940.92 L668.724,1940.2 L668.979,1939.86 L669.417,1938.98 L669.668,1938.68 L670.437,1938.41 L672.59,1938.91 L675.529,1938.32 L676.994,1938.31 L678.213,1939.17 L678.723,1939.68 L681.29,1941.21 L682.698,1941.67 L683.43,1941.78 L683.675,1942.82 L684.31,1943.86 L685.138,1944.65 L685.972,1944.96 L686.86,1945.59 L687.36,1945.81 L687.844,1945.84 L688.183,1945.57 L688.663,1944.61 L689.052,1944.32 L689.947,1944.38 L692.007,1945.31 L693.008,1945.6 L694.252,1945.66 L694.258,1945.38 L693.743,1944.69 L693.431,1943.46 L693.847,1942.96 L695.935,1941.62 L696.79,1941.28 L697.741,1941.3 L699.526,1941.75 L700.504,1941.69 L701.286,1941.3 L702.801,1940.08 L703.579,1939.71 L704.331,1939.7 L705.121,1939.81 L705.905,1939.75 L707.526,1938.72 L708.97,1939.43 L709.802,1938.99 L710.285,1939.14 L711.632,1939.3 L711.93,1939.42 L712.167,1940.13 L712.48,1940.74 L712.888,1941.26 L713.411,1941.66 L714.336,1942.05 L715.504,1942.29 L716.717,1942.34 L718.741,1942.18 L718.566,1943.01 L718.058,1944.3 L718.175,1945.29 L718.085,1946.35 L718.605,1947.29 L719.333,1947.98 L720.38,1947.89 L721.465,1947.95 L722.003,1948.87 L722.319,1948.9 L722.856,1948.47 L723.453,1948.64 L724.503,1949.48 L723.226,1951.55 L723.005,1952.16 L722.772,1953.08 L722.38,1953.5 L721.932,1953.81 L721.54,1954.42 L720.834,1953.99 L719.087,1954.2 L717.682,1953.45 L717.339,1952.8 L716.464,1952.9 L715.385,1953.87 L714.74,1955.15 L714.488,1957.34 L714.314,1958.85 L714.499,1959.99 L715.179,1960.86 L716.7,1961.97 L717.492,1963.24 L718.396,1964.86 L718.262,1965.62 L717.6,1966.17 L717.347,1966.97 L717.53,1967.96 L717.672,1969.27 L717.679,1969.69 L717.837,1970.04 L718.648,1970.84 L718.848,1971.33 L718.584,1972.52 L717.264,1974.41 L716.547,1975.95 L715.716,1976.69 L714.235,1978.42 L712.691,1978.14 L710.798,1979.28 L710.205,1980.95 L708.8,1981.58 L704.342,1984.45 L703.385,1985.73 L701.852,1985.94 L700.722,1986.13 L699.473,1986.32 L698.949,1986.42 L698.717,1986.53 L697.944,1987.12 L697.775,1987.38 L697.553,1987.81 L697.102,1987.87 L696.556,1987.8 L695.795,1989.19 L689.645,1991.12 L679.769,1994.52 L678.714,1995.43 L675.841,1997 L672.203,1998.13 L670.727,1999.02 L669.301,2000.61 L666.644,2004.35 L665.119,2006.07 L663.444,2007.34 L659.968,2008.79 L658.006,2009.13 L654.76,2009.11 L651.518,2009.59 L650.503,2010.07 L650.031,2010.15 L648.43,2009.9 L638.953,2011.67 L634.918,2011.52 L622.635,2014.58 L619.248,2014.91 L618.852,2015.16 L617.865,2016.54 L617.634,2017 L617.358,2017.33 L616.83,2017.51 L615.78,2016.7 L615.445,2016.53 L613.365,2016.34 L611.347,2016.47 L609.457,2017.07 L608.102,2017.66 L602.195,2022.27 L599.233,2025.37 L593.517,2029.96 L592.397,2031.66 L593.577,2032.54 L594.871,2033.24 L595.451,2033.28 L595.717,2033 L595.591,2032.53 L595.013,2031.98 L595.799,2032.14 L596.403,2032.56 L596.888,2033.14 L597.862,2034.68 L598.974,2035.84 L599.567,2035.85 L600.455,2036.28 L600.25,2037.56 L599.429,2038.77 L598.891,2039.39 L595.745,2039.91 L594.304,2040.9 L591.555,2044.25 L590.007,2045.21 L588.567,2045.54 L587.021,2045.57 L586.112,2043.66 L588.43,2043.65 L589.414,2044.53 L591.091,2043.76 L593.122,2041.29 L592.183,2041.02 L590.735,2040.97 L586.683,2041.75 L585.682,2042.25 L584.794,2043 L583.959,2044 L582.071,2047.4 L581.091,2049.17 L580.104,2050.24 L579.181,2051.03 L578.42,2051.97 L577.927,2053.52 L576.552,2054.61 L572.492,2060.57 L571.861,2061.15 L570.46,2062.03 L569.75,2062.61 L567.855,2064.69 L564.739,2066.68 L561.161,2072.03 L559.681,2073.24 L557.07,2074.43 L555.627,2076.06 L553.022,2080.17 L551.564,2081.87 L548.161,2084.79 L542.213,2092.3 L541.877,2092.92 L541.583,2094.39 L541.129,2095.38 L540.509,2096.09 L539.207,2097.19 L538.435,2098.01 L537.719,2099.02 L537.334,2100.02 L536.921,2100.75 L535.148,2102.72 L534.501,2103.75 L533.878,2105.91 L533.647,2109 L533.65,2111.67 L534.078,2115.87 L536.296,2122.86 L536.7,2123.92 L537.164,2125.08 L536.516,2125.41 L535.755,2125.87 L535.892,2127.03 L536.093,2129.33 L536.652,2131.48 L537.638,2133.9 L538.599,2136.54 L540.031,2139.93 L540.638,2140.93 L542.281,2142.59 L543.882,2144.72 L544.819,2145.67 L546.102,2146.17 L548.297,2146.68 L550.55,2147.66 L552.504,2149.1 L553.831,2150.99 L553.609,2151.06 L553.076,2151.36 L553.628,2151.7 L553.965,2152.15 L555.162,2153.38 L555.657,2153.31 L555.878,2154.11 L555.79,2155.08 L555.401,2155.51 L554.482,2155.45 L553.633,2155.54 L552.841,2155.84 L552.075,2156.4 L551.351,2157.62 L551.238,2157.77 L551.021,2157.96 L550.074,2157.89 L549.705,2158.01 L548.984,2158.56 L548.38,2158.79 L547.811,2159.13 L547.214,2160.04 L547.474,2160.96 L546.519,2160.6 L545.826,2160.67 L545.143,2160.89 L544.228,2160.93 L542.772,2160.76 L542.059,2160.9 L541.302,2161.36 L540.604,2162.3 L540.482,2163.19 L540.451,2164.05 L540.032,2164.89 L539.187,2165.57 L538.431,2165.76 L536.4,2165.49 L534.129,2165.69 L532.843,2166 L531.153,2166.9 L526.248,2168.21 L525.444,2168.63 L525.05,2169.3 L524.633,2169.5 L522.605,2171.24 L521.958,2172.08 L521.61,2173.26 L521.539,2173.9 L521.326,2174.13 L519.74,2174 L519.013,2174.11 L518.442,2174.54 L518.141,2175.45 L516.416,2175.11 L515.672,2177.49 L515.695,2183.23 L515.045,2183.87 L514.164,2183.96 L512.414,2183.59 L511.273,2183.71 L510.295,2184.3 L509.477,2185.17 L508.823,2186.12 L508.401,2186.94 L508.07,2187.88 L507.595,2190.46 L506.863,2194.49 L506.62,2195.61 L506.114,2196.11 L504.745,2196.42 L503.69,2197.27 L503.116,2198.02 L500.514,2202.64 L500.283,2203.63 L500.281,2203.71 L500.192,2206.19 L499.885,2206.77 L498.749,2204.93 L497.131,2208.01 L496.414,2208.45 L495.274,2208.62 L494.757,2209.4 L494.783,2210.57 L496.034,2213.7 L496.446,2214.37 L497.908,2215.63 L498.973,2216.33 L499.944,2216.59 L500.453,2215.9 L501.3,2216.56 L501.544,2217.32 L501.251,2218.02 L500.469,2218.47 L500.009,2218.85 L499.545,2219.04 L499.098,2218.98 L498.699,2218.61 L498.191,2218.84 L496.768,2219.14 L495.929,2219.21 L494.427,2219.75 L492.573,2219.49 L491.922,2219.53 L491.191,2219.75 L489.873,2220.39 L489.491,2220.01 L487.643,2218.78 L486.054,2218.23 L482.921,2217.71 L482.026,2217.68 L481.215,2217.85 L480.597,2218.24 L480.256,2218.79 L480.334,2219.41 L480.744,2220.55 L479.237,2219.96 L477.804,2219.39 L477.581,2218.65 L477.396,2218.07 L476.372,2217.75 L475.001,2216.95 L474.349,2216.66 L473.51,2217.3 L473.236,2217.85 L471.851,2217.59 L470.201,2217.12 L468.642,2217.65 L467.785,2218.05 L465.296,2219.87 L464.401,2220.26 L463.39,2220.32 L463.11,2220.4 L462.678,2220.63 L462.264,2220.93 L462.066,2221.19 L461.899,2221.47 L461.1,2223.42 L460.994,2223.81 L460.523,2224.08 L460.245,2223.95 L460.008,2223.7 L459.674,2223.61 L459.194,2223.83 L458.099,2224.52 L457.618,2224.62 L456.868,2224.65 L454.194,2225.35 L452.966,2225.86 L452.654,2226.56 L452.244,2226.99 L451.366,2226.73 L450.832,2227.21 L450.407,2228.24 L449.58,2229.4 L447.828,2231.26 L444.515,2233.71 L443.009,2235.24 L441.788,2237.6 L441.125,2239.68 L440.293,2241.26 L439.789,2242.92 L437.786,2246.45 L436.497,2247.99 L436.349,2248.74 L436.413,2249.31 L436.531,2249.81 L436.548,2250.36 L436.389,2250.89 L436.101,2251.17 L435.699,2251.26 L434.57,2251.17 L434.246,2251.26 L433.541,2251.77 L431.626,2252.69 L430.818,2253.41 L429.952,2255.86 L429.264,2256.51 L428.399,2256.96 L427.458,2257.71 L427.199,2258.15 L426.779,2259.26 L426.446,2259.73 L425.948,2259.94 L425.373,2260 L424.877,2260.11 L424.2,2261.2 L423.335,2261.67 L422.279,2261.91 L421.289,2261.9 L419.498,2261.01 L418.298,2259.38 L417.316,2257.51 L416.186,2255.9 L414.569,2254.52 L413.588,2253.96 L412.486,2253.62 L411.534,2253.66 L409.793,2254.24 L408.744,2254.3 L408.301,2254.1 L407.036,2253.3 L406.455,2253.07 L402.4,2253.1 L401.53,2253.43 L400.198,2255.61 L398.775,2258.18 L397.308,2259.28 L397.039,2259.41 L396.39,2259.89 L395.038,2260.22 L393.31,2260.18 L392.463,2259.7 L391.378,2259.96 L390.195,2258.87 L389.33,2257.92 L388.473,2257.36 L388.005,2257.42 L387.467,2257.91 L386.988,2257.97 L386.507,2257.59 L385.414,2255.56 L384.171,2254.34 L383.222,2254.1 L380.372,2254.53 L376.067,2252.99 L372.019,2252.62 L368.57,2251.37 L363.411,2251.09 L362.598,2251.45 L361.8,2252 L360.665,2252.56 L358.677,2253.37 L355.571,2252.78 L353.901,2250.91 L352.875,2250.89 L352.131,2251.14 L350.447,2249.95 L349.871,2248.84 L348.3,2248.63 L346.627,2248.45 L345.819,2248.97 L343.212,2248.8 L339.803,2247.02 L338.15,2246.12 L335.534,2246.18 L331.812,2246.54 L327.263,2244.37 L326.064,2244.42 L324.382,2245.23 L319.961,2245.06 L318.553,2244.98 L309.802,2242.6 L308.91,2242.69 L308.173,2243.13 L306.381,2245.26 L306.011,2245.72 L305.213,2246.31 L304.533,2247.1 L304.157,2248.05 L302.278,2250.2 L299.074,2250.02 L297.269,2252.4 L295.345,2253.19 L294.828,2253.38 L293.71,2253.59 L291.513,2253.66 L289.458,2253.2 L283.804,2251.35 L282.701,2251.08 L282.2,2251.14 L281.56,2251.46 L277.621,2253.2 L274.168,2252.82 L271.619,2253.78 L270.968,2253.86 L269.108,2253.72 L268.261,2253.84 L267.552,2254.09 L266.748,2254.81 L264.956,2256.93 L264.814,2257.5 L263.584,2259.56 L262.976,2260.38 L262.679,2260.61 L262.255,2260.87 L261.798,2261.06 L260.921,2261.21 L260.893,2261.68 L261.005,2262.28 L260.295,2262.98 L258.128,2265.66 L257.78,2266.24 L257.038,2269.04 L256.058,2268.84 L255.887,2267.41 L255.193,2266.69 L254.092,2266.03 L253.131,2265.8 L252.862,2266.37 L252.31,2266.79 L251.95,2267.52 L251.837,2268.4 L252.024,2269.26 L251.694,2269.66 L251.661,2270.63 L251.831,2271.28 L251.888,2271.81 L251.609,2272.46 L251.196,2272.8 L250.793,2273.06 L249.67,2273.26 L244.346,2274 L243.542,2274.32 L241.921,2274.46 L241.358,2272.43 L239.472,2270.82 L238.713,2269.96 L237.447,2270.15 L236.473,2269.59 L235.596,2268.78 L233.59,2267.73 L232.957,2266.88 L232.171,2265.1 L229.5,2261.28 L228.654,2260.45 L227.566,2260.24 L225.266,2260.19 L222.757,2258.86 L221.791,2256.9 L221.192,2254.65 L220.412,2252.58 L217.961,2250.56 L217.802,2249.61 L217.865,2248.57 L217.744,2247.39 L217.494,2246.25 L216.162,2243.95 L215.494,2241.01 L215.093,2239.03 L214.381,2236.66 L213.471,2235.53 L213.14,2234.95 L213.805,2234.8 L214.199,2234.97 L215.505,2236.75 L215.824,2237.39 L215.472,2237.85 L215.73,2238.52 L216.321,2239.53 L216.963,2239.52 L217.402,2239.27 L218.088,2238.57 L219.376,2237.6 L219.031,2237.48 L217.961,2237.59 L217.114,2237.28 L216.89,2236.26 L217.56,2234.47 L217.342,2233.21 L215.375,2230.46 L214.899,2230.08 L211.585,2229.12 L210.735,2228.7 L210.146,2227.77 L210.004,2225.21 L209.702,2223.93 L209.233,2222.64 L209.139,2221.43 L209.574,2220.39 L210.694,2219.59 L211.681,2219.33 L213.654,2219.08 L214.415,2218.67 L215.2,2217.36 L215.265,2214.73 L215.816,2213.33 L217.758,2212.27 L221.697,2213.43 L223.444,2211.91 L222.377,2212.34 L221.017,2212.58 L219.761,2212.34 L219,2211.35 L216.759,2211.7 L215.7,2212.2 L215.074,2213.39 L214.654,2217.14 L214.236,2217.76 L212.478,2217.41 L211.432,2215.3 L210.616,2210.9 L209.88,2208.75 L208.762,2206.97 L198.399,2195.67 L197.231,2194.68 L196.153,2194.13 L195.387,2193.58 L193.695,2191.45 L192.794,2190.77 L191.929,2190.07 L191.319,2188.51 L191.306,2186.85 L191.795,2186.16 L192.481,2185.79 L194.681,2183.63 L195.455,2182.68 L192.418,2184.9 L190.873,2185.55 L189.336,2185.21 L189.26,2185.59 L190,2187.47 L190.309,2188.65 L190.257,2189.33 L189.596,2189.26 L185.721,2186.19 L184.243,2185.32 L182.444,2184.25 L181.041,2184.54 L182.393,2185.03 L182.983,2185.51 L171.072,2183.08 L170.262,2183.24 L169.112,2183.9 L168.184,2183.8 L167.418,2183.27 L166.732,2182.64 L166.55,2179.79 L166.595,2178.36 L167.111,2177.52 L167.04,2175.41 L167.668,2170.49 L167.218,2168.4 L167.468,2167.74 L167.603,2166.37 L167.621,2164.81 L167.512,2163.59 L167.32,2163.16 L166.627,2162.28 L166.334,2161.8 L166.061,2161.08 L166.034,2161.01 L166.004,2160.63 L166.047,2160.21 L165.965,2159.31 L165.773,2159.28 L165.714,2159.19 L165.818,2158.55 L166.025,2158.34 L166.665,2157.87 L166.785,2157.58 L166.945,2157.33 L167.045,2157.04 L166.996,2156.64 L167.519,2156.44 L169.085,2155.37 L169.52,2154.96 L169.966,2154.11 L170.999,2151.28 L172.035,2149.6 L173.149,2148.55 L178.375,2145.86 L180.196,2143.8 L181.159,2141.85 L182.804,2139.25 L183.28,2138.23 L183.385,2137.61 L183.357,2136.99 L183.431,2136.5 L183.837,2136.21 L185.437,2136.25 L186.029,2136.02 L186.646,2136.22 L187.222,2136.55 L187.804,2136.76 L188.467,2136.58 L189.139,2136.55 L189.549,2136.4 L190.185,2135.85 L190.923,2134.69 L191.44,2134.32 L192.259,2134.6 L194.766,2136.34 L196.022,2136.48 L196.719,2135.09 L197.328,2134.43 L198.123,2132.68 L199.399,2131.04 L200.462,2129.15 L201.024,2128.34 L201.863,2126.71 L201.737,2126.57 L201.175,2125.95 L199.807,2125.76 L198.473,2125.85 L198.336,2125.81 L198.095,2125.66 L197.335,2126.35 L196.054,2126.61 L194.783,2126.58 L194.053,2126.4 L193.524,2125.64 L193.142,2125.09 L192.213,2119.7 L192.017,2119.37 L191.321,2118.57 L191.061,2118.15 L190.93,2117.72 L190.786,2116.84 L190.656,2116.44 L186.919,2108.74 L185.764,2107.43 L185.175,2106.94 L186.016,2106.38 L186.62,2105.81 L186.819,2105.11 L186.432,2104.19 L187.052,2103.58 L188.556,2100.71 L191.163,2098.28 L191.598,2097.49 L191.547,2096.99 L191.309,2096.71 L191.03,2096.46 L190.874,2096.02 L190.901,2095.52 L191.007,2094.89 L191.392,2093.53 L192.153,2091.52 L192.404,2091.12 L192.98,2090.4 L193.112,2090.19 L195.603,2088.94 L196.405,2088.54 L197.184,2088.42 L198.119,2087.94 L198.924,2087.33 L199.309,2086.81 L199.849,2086.53 L202.152,2086.55 L203.002,2086.35 L203.705,2085.88 L204.389,2085.54 L204.831,2085.38 L204.953,2084.5 L205.39,2083.6 L204.975,2082.26 L206.178,2081.64 L207.548,2080.4 L209.781,2077.66 L210.374,2077.16 L210.764,2076.6 L210.801,2076.03 L210.345,2075.48 L210.888,2074.51 L210.833,2073.25 L210.39,2072 L209.777,2071.09 L208.836,2070.08 L207.665,2069.46 L206.383,2069.31 L205.102,2069.74 L204.482,2069.83 L203.783,2069.54 L203.159,2069.02 L202.772,2068.41 L202.812,2067.89 L203.579,2066.82 L203.808,2066.22 L203.468,2064.92 L202.459,2064.15 L201.213,2063.49 L200.173,2062.56 L199.71,2061.45 L199.6,2060.15 L199.936,2058.88 L200.808,2057.92 L199.06,2054.49 L198.317,2053.57 L198.118,2052.77 L198.166,2051.42 L198.424,2050.07 L198.852,2049.27 L199.408,2048.48 L199.947,2047.14 L200.225,2045.84 L200.003,2045.15 L198.13,2044.16 L197.228,2043.44 L196.633,2042.68 L196.466,2041.24 L196.17,2040.62 L194.751,2039.42 L194.121,2038.47 L193.117,2036.27 L192.434,2035.29 L191.81,2030.7 L191.633,2029.76 L191.6,2029.59 L201.659,2032.46 L202.337,2032.43 L203.182,2032.28 L204.099,2032.3 L205.965,2032.8 L206.825,2032.73 L207.715,2033.59 L209.169,2034 L216.283,2034.73 L217.032,2034.6 L217.846,2033.42 L217.887,2032.96 L217.835,2032.51 L217.967,2031.95 L218.302,2031.42 L219.071,2030.55 L219.369,2030.01 L219.401,2029.49 L219.315,2028.88 L219.386,2028.21 L219.897,2027.46 L220.328,2027.23 L221.498,2027.03 L222,2026.87 L223.549,2025.36 L223.831,2024.81 L224.078,2023.8 L224.27,2023.28 L224.343,2022.97 L224.302,2022.32 L224.369,2021.97 L224.564,2021.73 L225.092,2021.49 L225.284,2021.28 L225.869,2019.84 L226.165,2019.33 L226.215,2018.65 L226.569,2018.05 L227.654,2016.89 L226.47,2013.06 L225.7,2011.51 L225.272,2010.36 L224.865,2009.79 L224.401,2009.49 L223.196,2009.02 L222.664,2008.73 L222.281,2007.77 L222.378,2005.61 L222.177,2004.63 L222.724,2004.01 L223.478,2002.7 L224.016,2002.23 L224.936,2002.05 L227.224,2001.19 L227.833,2001.22 L228.157,2001.24 L229.054,2001.53 L229.973,2001.68 L230.956,2001.28 L231.169,2000.94 L231.297,2000.46 L231.436,1999.95 L231.705,1999.51 L232.065,1999.25 L234.388,1998.22 L234.888,1997.88 L236.009,1996.6 L234.743,1994.77 L234.026,1992.75 L234.165,1990.7 L235.467,1988.75 L236.874,1987.45 L237.157,1986.75 L237.117,1985.63 L236.801,1984.69 L236.393,1984.02 L236.12,1983.31 L236.245,1982.22 L236.72,1981.33 L238.206,1979.6 L238.747,1978.63 L238.903,1977.89 L239.034,1976.78 L239.084,1974.65 L238.909,1972.52 L238.988,1971.63 L239.234,1970.45 L239.445,1969.87 L239.641,1969.56 L239.76,1969.23 L239.739,1968.56 L239.766,1968.14 L240.151,1966.61 L240.958,1966.54 L241.309,1965.81 L241.247,1964.95 L240.816,1964.48 L240.161,1964.14 L239.886,1963.54 L239.766,1962.17 L239.286,1959.46 L238.864,1958.22 L237.738,1956.34 L237.429,1955.55 L237.494,1954.76 L238.156,1954.07 L238.577,1952.96 L238.599,1952.96 L239.652,1952.68 L242.439,1953.59 L243.727,1953.74 L244.487,1953.25 L246.876,1950.95 L247.488,1949.93 L247.686,1948.39 L247.95,1948 L248.707,1947.37 L250.623,1946.34 L250.987,1945.9 L251.521,1945.06 L253.668,1942.96 L254.324,1942.48 L255.169,1942.53 L258.19,1943.15 L258.668,1942.95 L261.607,1942.37 L263.352,1941.53 L265.141,1940.34 L267.064,1938.31 L267.264,1938.29 L267.806,1938.53 L268.003,1938.52 L268.183,1938.2 L268.335,1937.74 L268.381,1937.36 L268.249,1937.3 L268.941,1936.95 L269.63,1936.85 L271.087,1937.05 L271.198,1936.53 L273.517,1934.1 L273.621,1933.7 L273.58,1933.26 L273.644,1932.77 L274.061,1932.22 L275.503,1931.22 L278.342,1928.48 L277.996,1927.77 L277.539,1926.73 L276.018,1924.26 L274.235,1922.47 L272.167,1921.15 L268.612,1919.56 L267.987,1919.4 L267.614,1919.6 L267.235,1919.91 L266.519,1920.2 L265.762,1920.4 L265.259,1920.43 L264.074,1919.65 L263.646,1918.53 L263.654,1917.28 L263.77,1916.12 L263.678,1915.5 L263.482,1914.9 L263.453,1914.34 L263.841,1913.82 L264.272,1913.39 L264.583,1912.86 L266.727,1908.11 L267.489,1907.08 L266.393,1906.42 L265.606,1906.12 L265.334,1905.61 L266.252,1903.05 L266.277,1902 L265.816,1901.12 L264.805,1900.4 L263.928,1901.57 L262.282,1901.71 L259.013,1901.11 L257.737,1900.39 L256.617,1897.44 L255.281,1896.87 L254.611,1898.63 L253.391,1899.05 L250.514,1898.64 L249.296,1898.09 L248.372,1896.96 L247.808,1896.49 L247.447,1896.19 L246.218,1896.72 L244.453,1897.06 L243.163,1896.07 L242.037,1894.55 L240.772,1893.29 L240.096,1893.24 L239.202,1893.51 L238.393,1893.88 L237.982,1894.16 L237.7,1894.99 L237.569,1897.04 L237.257,1897.98 L236.578,1898.61 L236.021,1899.13 L234.327,1899.72 L230.856,1900.08 L230.007,1900.02 L228.366,1899.58 L227.654,1899.54 L226.949,1899.81 L225.577,1900.82 L224.864,1901.2 L224.883,1900.18 L224.677,1899.02 L224.271,1898.07 L223.688,1897.69 L223.256,1897.54 L222.675,1896.89 L222.326,1896.8 L221.924,1897.02 L221.754,1897.36 L221.614,1897.76 L221.281,1898.15 L220.484,1898.48 L219.483,1898.59 L218.484,1898.5 L217.698,1898.23 L217.057,1897.64 L217.155,1897.06 L218.279,1895.31 L215.859,1894.62 L215.15,1894.04 L214.094,1892.83 L213.424,1892.47 L212.002,1892.83 L209.035,1892.83 L207.824,1892.46 L207.312,1892.5 L207.024,1892.79 L206.778,1893.14 L206.384,1893.43 L204.943,1893.07 L205.029,1891.24 L205.274,1889.56 L204.304,1889.69 L203.934,1890.34 L203.485,1891.3 L202.881,1891.99 L202.032,1891.8 L201.33,1891.5 L200.618,1891.65 L199.929,1892.1 L199.318,1892.73 L197.477,1893.92 L197.386,1893.92 L195.306,1894 L191.455,1892.8 L191.495,1892.68 L192.003,1891.27 L192.111,1890.44 L191.878,1889.88 L190.4,1888.84 L190.136,1887.62 L190.462,1886.5 L191.143,1885.51 L193.482,1883.34 L195.099,1882.37 L196.607,1881.83 L197.464,1881.4 L198.121,1880.75 L198.464,1879.72 L198.303,1878.72 L197.768,1877.88 L196.985,1877.29 L196.026,1877.15 L195.138,1877.33 L194.409,1877.35 L193.939,1876.76 L194.013,1875.73 L194.749,1873.14 L194.552,1872.11 L194.281,1871.43 L192.706,1872.07 L191.8,1872.2 L191.019,1872.05 L190.659,1872.04 L190.199,1872.2 L188.797,1872.97 L187.945,1873.19 L180.715,1872.57 L179.791,1872.69 L177.988,1873.37 L177.079,1873.5 L175.417,1873.13 L174.592,1873.09 L173.689,1873.48 L173.154,1873.99 L172.367,1875.18 L171.881,1875.7 L170.812,1876.09 L169.567,1876.15 L168.424,1876.41 L167.83,1876.66 L167.092,1877.52 L166.188,1878.29 L165.196,1878.77 L164.202,1878.81 L163.337,1879.05 L162.408,1879.75 L161.539,1880.21 L160.852,1879.75 L161.165,1878.93 L163.074,1866.87 L163.464,1865.85 L164.055,1865.55 L165.847,1865.98 L166.733,1865.95 L167.2,1865.48 L167.105,1864.95 L166.347,1864.31 L166.248,1863.87 L166.522,1863.72 L167.84,1863.4 L168.014,1863.17 L168.259,1862.63 L168.63,1862.12 L169.165,1861.98 L169.412,1861.87 L170.506,1861 L173.293,1859.78 L173.94,1859.68 L174.175,1859.6 L175.932,1858.61 L176.814,1858.48 L177.874,1858.08 L178.243,1857.84 L178.518,1857.38 L178.955,1855.51 L179.002,1855.1 L178.847,1855 L178.446,1855.03 L177.613,1855.51 L176.856,1856.53 L175.972,1857.45 L174.747,1857.64 L173.785,1857.63 L171.089,1858.14 L170.464,1858.55 L169.954,1858.54 L167.633,1857.77 L166.705,1857.82 L167.06,1857.1 L167.838,1855.88 L168.044,1855.04 L168.325,1855.11 L168.726,1856.23 L169.095,1855.84 L169.592,1853.88 L169.621,1853.36 L169.586,1853.06 L169.777,1852.95 L170.455,1852.98 L171.977,1853.77 L172.237,1853.68 L172.752,1852.97 L174.92,1851.39 L175.852,1850.88 L177.559,1850.28 L178.353,1849.81 L178.884,1849.01 L178.427,1849.11 L178.069,1848.99 L177.663,1848.7 L177.26,1848.64 L177.102,1848.67 L176.976,1848.74 L175.844,1849.1 L174.521,1849.71 L173.807,1849.92 L172.771,1849.93 L171.321,1849.7 L169.906,1849.14 L168.993,1848.19 L168.788,1847.41 L168.861,1846.2 L168.667,1845.47 L168.166,1844.99 L166.686,1844.32 L166.422,1844 L166.903,1843.5 L167.846,1843.19 L168.865,1843.05 L169.562,1843.06 L170.064,1843.39 L170.309,1843.99 L170.158,1844.57 L169.477,1844.79 L169.808,1845.16 L170.373,1845.58 L170.999,1845.86 L171.509,1845.8 L171.841,1845.21 L171.891,1844.46 L172.052,1843.8 L172.711,1843.47 L172.661,1841.22 L172.85,1840.16 L173.223,1839.29 L173.805,1838.67 L174.465,1838.33 L175.096,1838.31 L175.853,1837.69 L178.378,1834.38 L179.049,1833.11 L178.771,1833.04 L178.234,1833.93 L177.57,1834.77 L176.802,1835.33 L175.947,1835.4 L175.184,1834.86 L174.823,1833.96 L174.592,1832.89 L174.12,1832.23 L173.145,1833.33 L172.857,1834.06 L173.18,1834.69 L171.874,1835.84 L171.266,1836.06 L170.747,1835.39 L171.14,1834.81 L170.442,1834.96 L168.695,1835.74 L169.141,1836.67 L168.516,1836.96 L166.803,1837.01 L166.597,1837.29 L165.908,1838.58 L165.5,1838.62 L165.067,1838.87 L164.235,1839.47 L163.755,1838.54 L163.925,1837.4 L163.954,1836.33 L163.058,1835.63 L162.543,1834.8 L163.126,1833.39 L164.14,1832 L164.924,1831.23 L165.07,1830.49 L165.43,1829.62 L165.993,1828.49 L166.675,1827.89 L166.991,1827.52 L167.439,1827.18 L168.009,1827.12 L168.15,1827.03 L169.614,1826.62 L171.147,1825.3 L171.971,1824.85 L173.099,1824.87 L173.509,1824.52 L174.023,1823.64 L174.135,1822.63 L173.333,1821.89 L172.815,1823.23 L171.921,1823.91 L170.792,1824.09 L169.58,1823.96 L167.401,1823.03 L166.432,1822.98 L166.333,1824.04 L165.04,1824.88 L164.407,1825.03 L163.603,1824.65 L163.142,1823.97 L163.036,1823.11 L163.033,1822.08 L162.871,1820.88 L163.422,1821.04 L163.893,1820.92 L164.307,1820.58 L164.718,1820.04 L164.526,1819.24 L164.014,1818.66 L163.254,1818.1 L163.321,1817.61 L163.286,1816.74 L163.322,1816.37 L163.47,1816.15 L163.988,1815.79 L164.19,1815.45 L164.381,1814.92 L164.438,1814.52 L164.189,1814.2 L163.445,1813.93 L163.096,1813.57 L162.677,1812.27 L162.187,1811.87 L162.096,1812.25 L162.197,1812.97 L161.854,1813.26 L161.41,1813.26 L161.208,1813.14 L161.216,1812.28 L160.795,1812.1 L160.213,1812.29 L159.738,1812.56 L159.281,1812.99 L158.539,1814.09 L158.11,1814.55 L157.934,1814.37 L157.842,1812.84 L157.531,1812.37 L157.622,1812 L158.057,1811.49 L159.595,1809.18 L159.722,1808.82 L159.936,1808.36 L160.193,1807.91 L160.448,1807.62 L160.748,1807.13 L160.563,1806.66 L160.027,1806 L160.053,1804.47 L160.507,1804.24 L161.096,1804.37 L161.53,1803.91 L162.039,1802.86 L162.685,1802.48 L163.292,1802.24 L163.675,1801.58 L164.183,1802.33 L164.724,1802.64 L165.346,1802.6 L166.121,1802.23 L166.437,1801.97 L166.629,1801.7 L166.874,1801.45 L167.353,1801.23 L167.757,1801.14 L168.105,1801.12 L168.926,1801.24 L169.035,1800.79 L168.198,1800.96 L167.151,1800.93 L166.449,1800.68 L166.626,1800.16 L165.037,1800.51 L164.431,1800.27 L164.283,1799.1 L166.877,1797.62 L168.486,1797.22 L169.965,1798.38 L170.73,1798.57 L171.562,1798.62 L172.186,1798.52 L172.863,1798.17 L173.512,1797.97 L174.165,1797.26 L174.614,1796.41 L174.945,1796.5 L175.578,1797.43 L176.769,1797.7 L178.052,1797.53 L178.956,1797.14 L178.01,1796.95 L177.577,1796.46 L177.244,1794.93 L177.045,1794.9 L176.576,1794.46 L176.303,1793.99 L176.666,1793.9 L177.113,1793.97 L179.087,1793.63 L181.268,1792.84 L182.743,1792.83 L183.178,1792.48 L183.599,1792.27 L184.125,1792.26 L184.619,1792.73 L185.666,1794.52 L186.338,1795.04 L187.104,1795.37 L188.678,1796.38 L189.469,1796.72 L190.805,1796.56 L192.578,1796.07 L194.355,1795.98 L195.702,1797.04 L198.281,1796.51 L198.743,1796.22 L199.015,1795.72 L199.486,1795.34 L200.009,1795.13 L200.956,1795.13 L201.204,1794.94 L201.377,1794.68 L201.687,1794.55 L204.254,1794.51 L204.225,1795.35 L204.482,1796.05 L205.255,1797.17 L205.98,1796.62 L206.266,1796.1 L206.486,1794.63 L207.076,1793.95 L207.927,1794.32 L209.017,1795.47 L209.599,1795.85 L209.755,1796.46 L209.76,1797.2 L209.895,1797.93 L210.675,1799.2 L210.99,1799.91 L211.09,1800.91 L211.658,1800.02 L211.854,1799.07 L211.967,1797.13 L212.209,1796.2 L212.549,1795.7 L213.128,1795.39 L214.082,1795.01 L213.312,1794.37 L212.532,1794.07 L211.011,1793.81 L210.294,1793.36 L209.505,1791.94 L208.849,1791.46 L209.828,1791 L214.41,1791.53 L214.737,1791.5 L215.126,1791.3 L215.417,1790.94 L215.452,1790.47 L215.247,1790.37 L214.123,1790.58 L211.337,1790.37 L211.549,1790.19 L211.694,1790.12 L212.042,1790.06 L211.91,1789.86 L211.54,1789.5 L210.831,1790.16 L209.874,1790.31 L207.907,1789.91 L208.467,1789.06 L208.929,1787.82 L209.365,1786.96 L209.82,1787.3 L210.589,1786.62 L210.309,1786.21 L210.07,1785.62 L209.992,1784.96 L210.171,1784.31 L210.344,1784.35 L211.077,1784.68 L211.403,1784.62 L212.408,1785.04 L213.828,1784.68 L221.377,1781.34 L221.383,1782.05 L221.67,1782.5 L222.146,1782.67 L222.715,1782.55 L222.245,1782.47 L222.114,1782.41 L222.202,1782.02 L222.416,1781.96 L222.674,1781.81 L222.907,1781.72 L222.22,1780.82 L222.246,1780.14 L222.608,1779.53 L222.931,1778.87 L223.325,1778.41 L224.013,1778.41 L225.481,1778.8 L226.876,1778.75 L228.289,1778.3 L229.612,1777.56 L230.745,1776.57 L231.919,1777.06 L232.237,1778.07 L232.323,1779.22 L232.798,1780.19 L232.711,1780.57 L232.491,1780.36 L232.08,1780.13 L231.871,1779.97 L231.442,1780.41 L231.114,1780.9 L230.972,1781.42 L231.133,1781.94 L230.596,1782.19 L230.47,1782.09 L230.204,1781.71 L229.859,1781.93 L229.722,1781.99 L229.498,1782.02 L229.403,1782.44 L231.583,1783.1 L232.089,1782.75 L231.585,1781.22 L232.177,1780.98 L232.717,1781.08 L233.715,1781.74 L233.662,1781.51 L233.651,1781.04 L233.611,1780.79 L234.386,1780.25 L235.337,1779.77 L236.336,1779.47 L238.187,1779.44 L239.269,1779.02 L240.215,1778.33 L240.725,1777.48 L240.918,1777.31 L241.193,1777.57 L241.398,1778.1 L241.386,1778.77 L241.117,1779.08 L239.819,1780.1 L239.597,1780.97"
         id="path1005" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['ES'],polycolors['ES'],labels['ES'], x1, polybary['ES'], x2, polybary['ES'] )
    if 'ES' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M397.18,329.187 L398.28,331.928 L399.187,332.813 L398.647,333.219 L398.182,333.312 L397.726,333.55 L397.198,334.38 L397.443,334.458 L397.515,334.506 L397.36,334.866 L396.699,335.108 L396.754,335.32 L396.787,335.871 L396.833,336.088 L396.337,336.5 L396.616,336.711 L397.543,336.831 L398.383,337.416 L398.883,337.364 L398.726,337.61 L398.389,338.232 L398.231,338.494 L400.452,339.374 L400.806,339.696 L401.014,340.246 L401.007,340.855 L400.731,341.371 L400.577,341.732 L400.366,343.5 L398.609,344.95 L397.74,345.477 L396.783,345.782 L396.75,346.734 L397.165,346.528 L397.374,346.82 L397.55,347.292 L398.176,347.936 L398.084,348.379 L397.714,348.966 L397.188,349.684 L397.417,350.143 L399.101,350.899 L400.09,350.841 L400.334,351.028 L400.341,351.336 L400.406,351.829 L400.509,351.864 L400.692,351.815 L401.024,351.653 L400.81,352.304 L400.743,352.872 L400.83,353.316 L401.363,353.881 L401.731,354.949 L402.239,355.848 L402.362,356.828 L402.62,357.505 L403.384,357.225 L403.233,357.059 L403.205,356.916 L403.228,356.662 L403.77,356.517 L404.345,355.999 L404.919,355.158 L405.462,354.05 L405.858,352.843 L406.1,351.981 L406.491,351.478 L407.343,351.341 L410.233,352.458 L412.126,352.511 L412.442,352.111 L412.688,351.583 L413.005,351.181 L415.118,350.729 L415.668,350.133 L416.2,349.305 L416.885,348.745 L418.19,348.144 L419,347.935 L421.163,348.319 L421.651,348.688 L422.235,349.804 L422.682,350.269 L427.464,351.188 L425.854,352.014 L420.904,352.356 L418.462,353.081 L416.076,352.805 L415.082,353.141 L414.093,353.939 L414.578,354.38 L414.739,354.784 L414.668,355.288 L414.74,355.448 L414.824,356.68 L414.816,356.978 L414.85,357.359 L414.459,358.005 L413.764,358.872 L413.39,359.236 L410.227,360.563 L409.749,360.603 L408.253,360.135 L407.397,360.457 L406.424,361.587 L406.913,362.439 L407.204,362.737 L408.57,363.368 L408.684,363.936 L408.36,365.09 L408.038,365.173 L407.416,365.458 L407.046,365.549 L407.382,366.114 L410.018,367.229 L411.266,368.543 L412.382,368.859 L415.066,367.811 L416.339,368.121 L416.753,368.606 L417.001,369.323 L416.978,370.327 L416.569,371.697 L416.443,372.44 L416.683,372.798 L416.985,373.002 L417.053,373.274 L416.837,373.964 L416.544,375.356 L416.32,376.004 L416.151,376.217 L415.719,376.416 L415.51,376.617 L415.387,376.879 L415.165,377.529 L415.041,377.773 L413.908,379.002 L413.66,379.366 L413.284,379.509 L410.945,380.867 L408.64,381.525 L407.814,382.026 L407.443,383.06 L406.631,383.714 L404.875,384.591 L405.61,384.815 L406.459,384.539 L408.177,383.496 L407.32,384.55 L406.097,385.596 L405.387,386.538 L406.06,387.294 L407.025,387.371 L409.355,386.985 L415.085,387.511 L417.355,387.178 L418.44,387.288 L418.875,388.416 L416.849,390.483 L416.011,390.866 L415.871,391.388 L415.852,391.943 L415.873,392.18 L415.926,392.369 L416.303,393.001 L417.42,394.599 L417.645,395.331 L416.17,395.25 L415.243,395.546 L414.05,396.836 L412.204,397.86 L411.212,398.926 L408.48,400.458 L407.815,401.055 L407.18,401.826 L406.657,402.755 L407.151,402.412 L408.181,401.331 L408.546,401.163 L408.887,401.11 L410.035,400.579 L410.441,400.287 L411.263,399.502 L416.067,396.097 L416.742,395.915 L417.323,396.323 L418.677,398.459 L418.517,397.762 L418.129,396.754 L418.117,396.007 L422.177,401.155 L422.814,401.51 L425.748,401.8 L426.174,402.03 L426.347,402.42 L426.254,403.186 L426.053,403.796 L425.758,404.155 L425.397,404.15 L425.253,404.517 L426.029,404.765 L426.32,405.003 L426.411,405.436 L426.278,405.854 L426.007,406.368 L425.621,407.018 L425.62,407.508 L425.705,407.909 L425.988,408.476 L427.777,407.202 L428.583,406.961 L429.246,407.409 L429.278,407.701 L429.163,408.086 L429.021,408.438 L428.978,408.637 L429.112,408.737 L429.506,408.753 L429.628,408.87 L430.115,410.145 L430.165,410.473 L429.705,411.784 L429.043,412.474 L428.376,412.875 L427.912,413.315 L427.842,413.679 L427.692,414.897 L427.504,415.409 L427.253,415.58 L426.176,415.887 L426.288,416.335 L426.11,416.78 L425.814,417.291 L425.57,417.954 L424.507,418.261 L422.404,417.119 L421.222,417.302 L421.623,418.045 L423.27,419.411 L422.977,420.402 L422.161,420.859 L421.24,420.933 L420.012,420.501 L419.429,420.478 L418.141,420.756 L415.938,420.398 L415.143,420.583 L415.312,421.271 L419.782,421.224 L420.926,421.762 L423.545,421.99 L424.203,422.481 L425.109,423.735 L425.474,424.648 L425.469,425.339 L425.466,425.711 L424.701,425.965 L423.084,427.203 L422.412,427.36 L418.977,427.035 L415.678,425.844 L414.104,425.69 L412.59,425.139 L413.11,426.023 L416.218,426.456 L419.847,428.262 L421.718,428.436 L422.646,428.767 L422.817,429.054 L422.535,429.443 L421.903,429.953 L421.695,430.301 L421.614,430.576 L421.526,430.806 L421.299,431.023 L420.194,431.123 L419.328,430.908 L418.427,430.946 L418.257,431.385 L420.329,431.666 L420.621,432.233 L420.482,432.594 L420.221,432.574 L419.939,432.661 L419.637,432.846 L419.315,433.13 L420.085,433.406 L420.281,433.615 L420.372,433.949 L420.414,434.263 L420.474,434.418 L420.791,434.327 L421.497,433.666 L421.774,433.558 L421.913,433.347 L423.175,432.188 L424.649,431.348 L424.388,432.234 L423.9,433.478 L423.361,434.648 L422.946,435.299 L422.779,435.737 L423.111,436.238 L423.034,436.844 L422.596,437.497 L421.857,438.148 L421.176,438.432 L420.003,438.368 L419.342,438.616 L419.479,439.575 L419.494,440.209 L419.331,440.439 L418.869,440.492 L417.764,441.068 L417.31,441.126 L415.684,440.046 L414.03,439.77 L413.631,439.308 L413.606,438.675 L413.714,437.789 L413.734,436.969 L413.467,436.508 L413.115,436.231 L412.516,435.361 L411.385,434.453 L410.539,432.598 L409.897,432.016 L409.802,432.717 L409.875,433.283 L410.055,433.708 L410.269,433.982 L408.985,434.66 L403.861,433.01 L403.688,433.448 L411.213,437.063 L411.378,437.384 L411.549,437.955 L411.633,438.633 L411.539,439.258 L411.63,439.648 L412.375,441.139 L414.483,443.257 L414.653,444.281 L413.298,444.731 L412.147,444.105 L410.379,442.248 L407.728,441.784 L406.457,440.567 L405.942,440.639 L406.998,442.236 L410.246,444.362 L411.339,446.288 L411.406,446.704 L411.436,447.327 L411.324,447.858 L410.969,447.981 L410.488,448.023 L409.545,448.628 L409.143,448.696 L406.929,446.725 L405.927,447.114 L408.205,449.21 L408.489,449.816 L408.261,450.452 L407.458,450.419 L404.647,449.265 L403.979,448.652 L403.715,448.875 L403.429,449.002 L402.796,449.137 L402.625,449.574 L403.021,450.814 L403.176,451.609 L403.245,452.544 L402.777,452.549 L401.218,453.181 L395.176,452.341 L394.393,451.725 L394.021,451.222 L393.844,450.709 L394.033,448.159 L393.654,447.266 L393.518,445.666 L393.27,445.222 L391.484,444.058 L391.597,444.486 L391.818,444.702 L392.417,444.905 L392.652,445.304 L392.377,445.899 L391.948,446.5 L391.721,446.902 L391.911,447.538 L392.899,448.699 L393.294,448.9 L393.135,449.3 L392.804,450.019 L392.781,451.076 L392.936,452.048 L393.134,452.506 L394.234,453.414 L394.5,454.09 L393.965,454.647 L393.979,455.111 L390.335,452.135 L388.638,452.678 L388.848,453.33 L389.645,454.227 L389.905,454.979 L389.886,455.727 L389.636,456.273 L389.19,456.63 L388.582,456.782 L385.231,455.537 L384.771,455.807 L384.769,456.703 L385.006,457.546 L385.271,457.636 L385.289,458.332 L385.679,458.92 L386.156,459.138 L386.428,458.739 L387.577,458.839 L390.06,457.069 L391.067,456.786 L391.114,457.13 L390.469,457.641 L387.045,459.607 L386.451,460.166 L386.148,460.677 L385.594,461.857 L385.306,462.283 L383.901,463.44 L383.574,463.931 L383.458,464.304 L383.405,464.519 L383.3,464.951 L383.184,465.162 L382.985,465.298 L382.803,465.348 L382.62,465.477 L382.425,465.823 L382.267,466.055 L381.627,466.861 L381.447,467.018 L378.704,466.333 L376.623,466.307 L375.85,466.348 L377.295,465.888 L379.818,466.078 L381.154,466.034 L380.682,465.355 L378.054,463.856 L377.818,463.874 L377.686,464.109 L377.421,464.908 L377.259,465.039 L377.023,465.096 L376.263,465.443 L375.906,465.449 L375.727,465.562 L374.81,466.376 L374.396,466.571 L372.633,466.946 L371.815,467.336 L371.013,467.847 L370.4,468.438 L370.808,467.819 L371.008,467.568 L371.229,467.37 L371.39,466.97 L371.219,466.905 L370.81,467.423 L370.215,467.784 L369.639,467.986 L369.283,468.015 L369.113,468.168 L369.015,468.621 L369.036,469.12 L369.219,469.408 L369.792,469.554 L370.085,469.519 L370.424,469.365 L370.216,470.235 L370.126,471.094 L369.562,470.871 L368.419,471.571 L367.818,471.639 L366.691,471.067 L366.273,470.628 L365.955,470.011 L365.551,468.73 L365.243,468.111 L364.796,467.729 L364.501,467.809 L363.715,468.392 L363.378,468.533 L363.097,468.349 L362.901,468.01 L362.637,467.816 L362.155,468.065 L362.307,468.62 L361.918,468.804 L361.586,468.792 L360.965,468.527 L362.121,467.124 L362.268,466.763 L361.36,466.678 L360.977,466.496 L360.976,466.017 L361.23,465.074 L361.198,464.363 L360.943,464.029 L360.522,464.208 L360.357,464.609 L360.449,464.971 L360.473,465.321 L360.408,465.766 L360.237,466.401 L359.991,466.034 L359.382,465.487 L359.169,465.072 L359.317,464.711 L359.898,463.99 L360.031,462.62 L359.86,461.251 L359.524,460.552 L359.243,459.775 L358.413,458.992 L357.847,458.558 L358.129,459.08 L358.84,459.785 L359.254,460.241 L359.256,460.993 L359.129,462.599 L359.15,463.174 L358.805,463.409 L358.551,463.418 L358.381,463.221 L358.288,462.841 L358.108,463.277 L358.426,463.944 L358.146,464.876 L357.516,465.815 L356.781,466.496 L356.028,466.585 L354.779,465.653 L354.026,465.851 L354.252,466.302 L354.588,466.572 L355.418,466.894 L354.558,467.179 L352.694,467.142 L351.183,466.717 L349.417,466.7 L348.526,466.988 L348.527,466.65 L348.477,466.324 L348.532,466.018 L347.409,466.741 L344.403,467.679 L346.81,467.788 L347.158,468.34 L346.01,468.723 L338.606,468.164 L337.335,468.638 L338.528,469.11 L338.377,469.467 L336.174,470.231 L335.551,470.202 L335.24,470.258 L333.931,470.947 L333.141,471.155 L332.778,471.341 L332.373,471.76 L332.346,471.094 L332.135,470.894 L331.78,471.026 L331.319,471.339 L331.15,471.736 L331.494,471.874 L331.341,472.236 L329.362,472.697 L328.743,472.631 L329.255,471.866 L327.439,472.363 L326.439,472.902 L325.63,473.7 L326.036,473.891 L326.5,473.715 L326.893,473.491 L325.661,474.573 L325.105,474.928 L322.598,475.784 L320.55,477.228 L319.574,477.337 L319.966,477.291 L320.406,477.139 L320.861,476.8 L321.296,476.178 L316.08,477.827 L312.236,477.435 L311.792,477.076 L311.569,476.407 L311.611,476.838 L311.714,477.172 L311.877,477.413 L312.109,477.558 L311.511,477.655 L311.053,477.592 L310.873,478.237 L310.508,479.035 L310.075,479.665 L309.672,479.825 L309.627,479.59 L309.752,479.175 L310.097,478.383 L310.112,478.173 L309.84,477.562 L309.463,475.843 L309.314,475.482 L308.97,475.171 L308.69,475.253 L308.399,475.728 L308.018,476.58 L307.633,477.688 L307.29,478.433 L306.98,478.719 L306.727,478.379 L306.898,477.844 L307.671,476.436 L307.816,475.879 L307.928,475.128 L307.905,474.682 L307.654,475.029 L306.986,476.579 L306.627,477.282 L306.212,477.895 L306.323,478.563 L306.029,478.908 L305.514,478.99 L304.968,478.86 L305.828,477.669 L306.098,477.105 L306.164,476.557 L306.491,476.097 L306.658,475.899 L306.787,475.839 L307.091,475.772 L307.25,475.446 L307.288,475.125 L307.22,475.084 L308.043,473.591 L308.44,472.76 L308.528,472.357 L308.542,472.091 L308.462,471.535 L308.285,471.156 L308.007,471.423 L307.281,472.893 L307.056,473.523 L307.045,474.124 L306.466,474.33 L306.006,474.098 L305.548,474.009 L304.968,474.62 L305.017,475.341 L304.858,476.062 L304.757,476.751 L304.982,477.387 L304.231,477.849 L303.191,477.68 L301.395,476.874 L301.065,476.581 L300.456,475.51 L300.003,475.613 L299.484,475.865 L299.231,475.743 L299.575,474.717 L295.567,475.378 L291.557,475.098 L288.936,474.672 L289.402,474.375 L289.791,473.936 L290.056,473.314 L290.172,472.677 L289.615,473.418 L289.061,473.848 L288.625,473.859 L288.429,473.307 L288.346,473.651 L288.283,473.8 L288.248,473.917 L288.265,474.17 L288.072,474.599 L283.7,475.557 L284.518,474.405 L285.495,473.974 L286.452,473.998 L287.205,474.232 L287.398,473.802 L286.119,472.208 L285.829,471.263 L285.347,471.986 L285.228,472.31 L285.119,472.833 L284.26,473.015 L283.697,472.621 L283.198,472.053 L282.524,471.726 L282.848,472.376 L282.686,472.735 L282.054,472.448 L281.712,472.441 L281.298,472.653 L280.991,472.236 L280.375,472.383 L279.714,472.861 L279.271,473.429 L278.672,474.389 L278.008,474.944 L277.471,474.937 L277.245,474.202 L276.692,474.552 L276.341,475.008 L276.286,475.472 L276.623,475.815 L277.002,475.83 L277.753,475.506 L278.195,475.55 L280.096,476.636 L280.888,476.703 L278.823,478.382 L277.892,478.689 L277.391,478.021 L277.524,477.952 L277.553,477.964 L277.58,477.939 L277.727,477.737 L276.687,477.29 L276.481,477.843 L275.798,477.463 L275.257,477.616 L275.603,477.717 L275.752,477.98 L275.655,478.412 L275.271,479.033 L277.215,479.736 L277.722,479.573 L276.207,481.248 L274.48,482.637 L272.718,483.437 L269.556,483.519 L267.3,484.074 L265.535,483.758 L265.24,482.222 L265.756,481.941 L266.594,481.985 L267.013,481.791 L267.638,481.319 L268.087,481.177 L268.924,481.464 L269.104,481.071 L268.387,480.744 L266.826,480.58 L266.143,480.258 L264.912,478.409 L264.448,478.063 L264.405,478.564 L264.174,478.57 L264.048,478.638 L263.878,478.8 L263.961,479.257 L264.053,480.476 L264.133,480.755 L264.212,481.227 L263.96,482.091 L263.659,482.931 L263.577,483.339 L264.612,483.873 L265.212,484.134 L265.598,484.402 L259.754,482.866 L255.887,483.013 L249.836,482.124 L248.563,481.495 L244.411,477.789 L242.086,477.44 L241.453,476.893 L241.75,476.984 L241.906,476.948 L242.143,476.772 L241.334,476.159 L241.003,475.7 L240.798,475.175 L239.933,475.683 L236.876,473.401 L236.625,473.109 L235.795,471.487 L233.926,470.121 L231.956,466.749 L228.451,463.776 L228.007,462.685 L228.378,462.777 L228.679,462.647 L228.79,462.355 L228.592,461.959 L228.295,461.797 L227.703,461.949 L227.344,461.907 L225.146,460.363 L225.08,460.682 L227.337,462.376 L220.938,458.831 L220.853,457.927 L220.716,457.931 L220.61,457.984 L220.498,458.059 L220.351,458.127 L220.351,457.823 L220.305,457.733 L220.168,457.606 L219.953,458.467 L218.011,457.602 L217.189,456.931 L215.005,451.888 L214.865,451.245 L215.053,451.737 L215.387,452.065 L215.847,452.163 L216.413,451.978 L216.124,451.764 L215.687,451.232 L215.397,451.018 L215.956,450.196 L216.131,449.971 L215.335,450.435 L214.785,450.514 L214.398,450.185 L214.083,449.437 L214.26,449.084 L214.935,449.841 L215.034,448.818 L214.833,448.098 L214.544,447.464 L214.379,446.707 L214.458,445.562 L214.584,444.777 L214.466,444.293 L213.824,444.047 L213.647,444.4 L213.884,446.066 L213.918,447.235 L213.791,448.339 L212.906,446.023 L212.654,444.654 L212.848,443.102 L212.991,443.053 L213.398,442.841 L213.349,442.486 L213.459,441.431 L214.565,440.947 L215.586,440.852 L217.383,441.387 L217.983,442.129 L218.615,444.271 L219.381,444.768 L219.24,444.025 L219.162,442.682 L219.007,442.158 L217.681,440.13 L219.14,439.066 L219.53,438.575 L219.835,437.735 L219.663,437.397 L219.343,437.493 L219.199,437.981 L219.022,438.333 L218.453,438.739 L216.668,439.648 L215.564,440.128 L214.84,440.402 L214.134,440.319 L213.827,440.446 L213.529,440.973 L213.224,441.627 L212.882,442.158 L212.414,442.62 L212.235,442.68 L211.451,441.491 L210.848,439.998 L210.207,439.213 L209.181,436.983 L210.793,437.758 L211.01,438.045 L211.309,438.915 L211.66,439.135 L211.877,438.83 L212.256,438.883 L213.055,439.323 L213.25,438.936 L212.559,438.606 L212.755,438.219 L212.847,438.249 L213.013,438.36 L213.107,438.388 L213.287,438.033 L211.298,437.646 L210.728,437.246 L210.908,436.892 L212.316,436.176 L213.433,434.581 L213.831,432.959 L213.073,432.177 L212.896,432.527 L213.44,433.154 L212.966,433.896 L212.036,434.585 L210.472,435.407 L209.646,435.587 L208.905,435.418 L208.433,434.701 L208.226,435.755 L208.196,436.027 L207.752,434.461 L206.299,431.872 L203.766,429.143 L203.108,427.832 L202.659,427.26 L202.303,426.517 L202.253,425.395 L202.621,425.027 L202.873,424.453 L203.296,423.014 L205.142,424.072 L206.321,424.337 L207.373,424.041 L205.531,423.143 L204.713,421.898 L204.335,421.593 L202.901,421.318 L202.518,421.426 L202.118,421.663 L200.917,422.122 L200.538,422.186 L200.356,422.536 L201.311,423.712 L201.664,424.558 L201.534,425.566 L201.17,425.031 L200.15,424.315 L198.943,423.894 L198.104,423.846 L197.405,424.238 L197.22,425.375 L197.095,425.333 L196.966,425.402 L196.829,425.507 L196.67,425.586 L188.968,423.021 L188.423,421.979 L188.069,420.68 L187.6,419.747 L186.853,419.22 L181.996,418.682 L180.971,418.231 L180.64,418.035 L179.167,417.811 L173.394,412.928 L172.684,413.032 L171.86,413.501 L171.162,413.729 L170.845,413.082 L171.398,412.878 L170.824,412.586 L169.583,412.721 L168.668,412.573 L167.945,412.811 L167.568,412.861 L167.276,412.783 L165.074,411.294 L164.416,411.233 L164.214,411.329 L163.953,411.508 L163.677,411.653 L163.431,411.657 L163.229,411.334 L163.26,410.407 L163.085,410.098 L163.549,409.605 L163.933,409.345 L164.286,409.045 L164.667,408.433 L164.897,407.767 L164.845,407.391 L164.706,407.033 L164.676,406.442 L165.362,406.07 L166.958,404.291 L167.421,403.958 L168.491,404.503 L168.756,404.405 L168.969,403.773 L168.723,403.646 L167.953,403.744 L167.556,403.381 L167.264,402.715 L167.239,401.793 L167.647,400.661 L168.603,399.784 L168.946,399.379 L169.321,398.753 L169.92,397.568 L170.215,397.061 L170.431,396.597 L170.832,396.202 L171.301,395.945 L171.719,395.889 L171.994,396.2 L172.265,397.423 L172.778,398.45 L172.885,399.788 L172.739,401.442 L172.279,403.059 L171.933,402.88 L172.004,403.684 L172.492,404.171 L174.786,405.508 L175.446,405.664 L176.033,405.472 L175.986,404.922 L176.372,404.548 L176.916,404.319 L177.354,404.199 L177.805,404.201 L179.06,404.59 L180.32,404.341 L180.644,404.381 L180.88,404.658 L181.212,405.526 L181.465,405.817 L182.233,405.904 L183.973,405.451 L186.21,406.073 L187.111,405.947 L188.183,405.291 L188.215,405.716 L188.34,405.857 L188.547,405.805 L188.83,405.618 L189.013,405.274 L188.834,404.617 L188.474,403.903 L187.995,403.426 L187.459,403.47 L188.194,402.694 L189.028,402.557 L189.727,402.835 L190.053,403.324 L189.831,403.287 L189.182,403.419 L189.495,403.875 L189.95,404.185 L190.478,404.333 L190.998,404.335 L191.222,403.914 L190.728,403.665 L191.236,403.528 L191.426,403.531 L190.677,401.361 L190.1,400.508 L189.298,400.032 L190.333,399.827 L193.603,402.327 L193.661,402.642 L193.647,402.961 L193.687,403.218 L194.209,404.029 L194.613,404.186 L195.187,403.971 L194.884,403.883 L194.641,403.724 L194.472,403.463 L194.384,403.084 L194.902,402.795 L195.179,402.703 L195.467,402.657 L195.175,402.025 L195.928,402.1 L196.638,402.802 L196.967,402.956 L197.463,403.19 L198.567,402.317 L197.471,402.386 L196.989,402.219 L196.991,401.75 L197.512,401.165 L198.076,401.026 L199.044,401.02 L198.725,400.462 L198.041,400.007 L197.24,399.747 L196.557,399.774 L197.212,398.896 L197.168,398.322 L196.752,397.965 L196.063,397.654 L195.793,397.622 L195.493,397.679 L195.177,397.861 L194.375,397.214 L195.414,396.946 L195.892,396.606 L196.214,395.718 L198.252,395.401 L199.418,394.927 L200.454,394.237 L201.477,393.296 L201.881,393.08 L202.429,393.028 L203.017,393.164 L204.001,393.858 L204.449,394.082 L204.082,393.633 L203.983,393.321 L203.793,393.226 L205.207,392.88 L205.864,392.885 L206.484,393.073 L209.323,395.614 L209.954,395.765 L210.715,395.29 L211.21,394.98 L209.801,395.068 L209.37,394.508 L209.553,394.159 L209.777,394.284 L210.019,394.294 L210.561,394.177 L210.709,393.928 L210.711,393.843 L210.647,393.811 L210.613,393.717 L210.764,393.791 L206.547,391.34 L204.239,390.809 L201.54,391.569 L201.323,391.754 L201.083,392.024 L200.791,392.302 L200.419,392.504 L200.128,392.563 L199.574,392.496 L199.337,392.397 L198.88,392.354 L197.362,392.615 L196.79,392.879 L195.455,393.246 L194.402,392.513 L192.69,390.586 L192.913,390.17 L194.246,390.074 L195.696,389.637 L197.106,388.895 L197.794,388.827 L198.139,389.414 L198.838,389.157 L200.082,389.345 L200.714,389.259 L198.705,387.948 L197.92,387.849 L197.285,387.939 L196.844,387.875 L196.64,387.389 L196.701,386.214 L197.303,385.853 L197.717,385.21 L198.15,384.425 L198.797,383.656 L198.993,383.669 L199.099,383.931 L199.251,384.034 L199.894,383.127 L200.265,382.742 L200.985,382.123 L201.334,381.548 L202.241,381.283 L205.674,381.231 L207.167,381.8 L207.319,381.244 L207.242,380.85 L206.991,380.611 L206.617,380.51 L206.803,380.162 L207.46,379.961 L209.043,379.156 L210.034,379.194 L210.675,378.907 L211.007,378.836 L211.364,378.923 L212.323,379.495 L211.694,380.455 L212.088,380.86 L212.693,381.179 L212.694,381.9 L212.787,382.465 L214.705,383.584 L214.48,383.072 L213.601,382.744 L213.245,382.417 L213.104,381.862 L213.037,380.734 L212.75,380.239 L213.948,380.306 L214.13,379.962 L211.678,378.737 L211.901,378.315 L213.713,378.148 L214.401,377.897 L214.852,377.987 L216.567,379.238 L216.787,378.821 L215.149,377.495 L214.552,377.223 L212.267,377.906 L211.617,377.688 L211.82,377.305 L212.762,377.015 L213.209,376.716 L213.638,376.239 L212.131,376.817 L211.768,376.843 L211.396,377.023 L210.551,378.239 L210.107,378.385 L209.789,378.422 L208.271,378.916 L206.779,378.772 L206.236,378.906 L205.976,379.745 L205.404,379.137 L204.636,378.989 L202.599,379.498 L201.665,379.902 L201.166,379.944 L200.875,379.591 L200.91,378.92 L201.242,378.385 L202.284,377.351 L201.703,377.265 L201.133,377.771 L200.606,378.592 L200.137,379.46 L199.902,379.682 L198.755,380.46 L198.384,380.914 L198.172,381.308 L197.962,381.621 L197.591,381.855 L196.739,381.41 L196.495,381.541 L196.3,381.702 L196.006,381.77 L195.705,381.77 L195.489,381.715 L195.202,381.334 L195.041,380.618 L195.065,379.699 L195.327,378.714 L194.943,378.942 L194.643,379.061 L194.38,379.097 L194.103,379.06 L194.825,378.525 L195.588,377.804 L196.139,377.013 L196.237,376.263 L196.385,376.229 L196.768,376.048 L196.703,375.736 L196.588,375.498 L196.417,375.323 L196.183,375.218 L196.383,375.085 L196.934,374.628 L197.122,374.28 L195.581,374.553 L194.731,374.536 L194.241,374.223 L194.182,373.326 L194.506,372.323 L195.072,371.484 L195.719,371.085 L195.905,370.742 L195.539,370.45 L195.465,369.881 L195.511,369.312 L195.507,369.026 L195.29,368.711 L195.167,368.134 L195.117,367.461 L195.137,366.847 L195.581,367.415 L195.852,368.365 L196.265,369.073 L197.136,368.894 L196.636,368.636 L197.124,368.078 L197.578,367.774 L198.01,367.644 L198.434,367.61 L198.373,367.189 L198.317,367.062 L198.525,366.681 L202.014,366.687 L203.176,366.145 L203.382,365.763 L201.221,366.305 L200.213,366.328 L199.369,366.096 L198.498,366.186 L197.343,366.605 L196.414,366.746 L196.224,365.986 L196.654,365.92 L197.83,365.306 L198.585,365.047 L199.08,364.781 L199.621,364.325 L199.652,363.853 L199.918,363.862 L200.136,363.771 L200.709,363.424 L200.524,362.525 L199.926,361.96 L198.378,361.25 L199.13,360.662 L198.65,360.414 L199.122,359.935 L200.153,359.063 L200.714,358.513 L200.434,358.456 L200.1,358.566 L199.716,358.835 L199.292,359.238 L199.427,358.645 L199.349,358.338 L199.093,358.288 L198.698,358.444 L198.49,358.824 L197.404,357.775 L196.937,357.772 L195.809,358.205 L195.335,358.169 L195.107,358.583 L195.262,358.733 L195.54,359.296 L193.143,358.603 L192.817,357.883 L193.326,358.25 L193.633,358.351 L193.956,358.23 L194.06,357.883 L194.069,357.303 L194.005,356.737 L193.899,356.446 L191.42,355.154 L191.56,355.752 L192.372,357.162 L192.162,357.542 L191.547,357.19 L190.995,355.62 L190.391,355.15 L190.551,355.234 L187.769,353.424 L184.984,351.959 L181.905,350.018 L181.37,347.842 L180.134,347.059 L177.588,346.383 L177.484,346.533 L176.656,347.063 L176.369,347.203 L175.99,347.307 L175.632,347.342 L175.311,347.261 L173.093,344.953 L173.476,344.854 L173.829,344.862 L174.147,344.968 L173.851,344.592 L172.686,344.602 L171.462,345.15 L170.73,346.191 L169.561,347.255 L168.618,347.451 L166.87,346.572 L166.284,346.443 L164.935,346.47 L164.425,346.282 L163.318,344.373 L163.265,343.647 L163.011,343.332 L162.921,342.933 L163.36,341.95 L163.513,341.389 L163.451,339.24 L163.531,338.137 L163.708,337.017 L164.425,334.329 L164.818,334.005 L165.001,334.747 L165.555,334.779 L167,334.207 L167.604,334.166 L170.53,334.8 L170.986,335.124 L172.128,337.375 L174.408,339.215 L174.709,339.19 L175.532,338.717 L175.894,338.583 L176.82,339.152 L177.193,339.035 L178.808,338.131 L179.947,338.308 L181.833,339.804 L182.068,339.39 L182.025,339.159 L182.048,338.963 L182.107,338.749 L182.166,338.458 L183.174,338.909 L183.322,339.374 L182.625,340.226 L182.798,340.623 L183.164,340.435 L183.557,340.161 L183.806,340.317 L183.59,340.893 L183.157,341.669 L183.051,342.134 L183.799,341.789 L183.963,342.332 L184.502,342.341 L185.139,342.004 L185.59,341.533 L185.822,340.921 L186.051,340.11 L186.374,339.255 L186.88,338.514 L188.013,337.967 L189.075,338.302 L190.796,339.607 L190.267,340.149 L189.738,340.424 L188.804,340.561 L188.75,340.813 L188.554,341.411 L188.248,341.328 L187.935,341.434 L187.576,341.7 L187.13,342.083 L187.334,344.687 L187.666,343.773 L188.04,343.068 L188.477,342.528 L189.01,342.096 L189.227,342.053 L189.715,342.2 L189.978,342.164 L190.302,341.932 L190.847,341.299 L191.143,341.017 L192.262,340.495 L192.963,340.777 L193.285,341.659 L193.267,342.916 L193.956,342.534 L194.642,342.43 L196.264,342.82 L198.189,342.562 L197.954,342.127 L196.132,341.485 L196.363,341.07 L197.463,341.355 L200.207,340.441 L201.249,340.693 L200.992,340.847 L200.736,340.915 L201.086,341.243 L201.33,341.714 L201.058,342.122 L201.004,342.19 L201.013,342.2 L200.93,342.437 L200.338,342.752 L199.417,342.784 L198.981,342.976 L199.24,343.222 L199.644,343.394 L200.133,343.439 L200.639,343.304 L200.199,344.461 L200.147,345.016 L200.716,345.786 L201.218,347.303 L201.519,347.669 L201.709,347.323 L201.516,346.996 L201.447,346.512 L201.438,345.184 L202.54,344.656 L203.035,344.564 L203.509,344.795 L203.96,344.081 L204.513,344.091 L206.402,345.19 L208.56,345.71 L211.359,347.969 L212.851,350.134 L213.169,350.108 L213.798,349.789 L214.095,349.749 L215.848,350.641 L216.201,351.104 L216.49,351.754 L216.883,352.188 L216.95,352.167 L217.557,351.994 L217.943,352.725 L220.764,353.615 L221.671,353.393 L222.067,352.712 L221.947,351.947 L221.308,351.468 L221.941,350.846 L222.35,350.74 L222.762,350.774 L223.401,350.582 L224.046,350.087 L226.552,347.72 L227.664,345.823 L227.343,344.657 L226.28,344.251 L225.156,344.638 L224.951,345.02 L225.123,345.107 L224.9,345.523 L219.747,346.799 L218.318,346.701 L212.4,342.006 L211.493,339.766 L211.304,339.52 L211.173,339.496 L210.705,339.577 L208.41,338.822 L207.94,338.224 L207.801,337.454 L208.285,337.402 L209.081,337.536 L209.876,337.319 L209.893,337.7 L210.001,337.907 L210.423,338.134 L210.651,337.719 L210.35,337.07 L209.797,336.709 L208.439,336.575 L209.13,335.875 L209.988,335.296 L211.336,334.663 L212.696,334.697 L213.362,334.437 L213.799,333.497 L214.219,333.523 L214.971,334.102 L215.296,333.991 L216.712,333.004 L217.59,332.84 L220.569,333.417 L223.672,332.821 L225.082,332.184 L226.978,332.162 L227.829,331.836 L227.809,331.748 L227.745,331.732 L227.73,331.664 L227.873,331.416 L228.197,331.632 L228.416,332.562 L228.739,332.783 L228.999,331.816 L229.043,331.48 L230.928,331.359 L234.437,331.911 L235.594,331.821 L236.328,331.765 L233.718,331.39 L232.902,331.004 L231.463,329.797 L229.874,329.518 L229.444,329.254 L230.364,328.618 L231.209,328.211 L230.827,326.962 L230.844,326.198 L231.19,325.289 L230.766,325.365 L230.374,325.343 L230.019,325.231 L229.716,325.025 L229.421,325.874 L229.08,326.021 L228.796,325.776 L228.662,325.461 L228.723,324.852 L229.019,323.297 L228.964,322.696 L228.571,323.698 L228.088,324.459 L227.51,325.044 L226.837,325.5 L226.15,325.81 L225.808,325.89 L225.559,325.861 L225.344,326.058 L224.917,326.506 L224.446,326.932 L224.096,327.057 L223.184,326.697 L222.883,326.34 L222.804,325.04 L222.142,323.093 L221.453,322.096 L221.594,321.539 L222.543,320.864 L223.244,320.746 L226.084,321.486 L227.559,321.274 L228.2,321.382 L228.587,321.266 L228.966,320.73 L229.327,320.075 L229.651,319.603 L230.062,319.311 L230.478,319.142 L231.203,318.982 L230.423,318.827 L227.397,320.437 L226.257,320.699 L225.318,320.673 L223.638,319.965 L223.846,319.585 L225.201,319.376 L225.713,319.132 L225.942,318.716 L225.448,318.508 L224.928,318.661 L223.891,319.169 L223.317,319.112 L223.152,318.736 L223.058,318.29 L222.69,318.016 L222.461,318.432 L222.505,319 L222.112,319.44 L221.549,319.686 L220.002,319.552 L219.762,319.201 L220.051,318.112 L220.531,317.189 L220.995,316.552 L221.371,315.857 L221.591,314.763 L222.231,312.58 L222.335,311.759 L221.766,312.126 L221.617,312.432 L221.268,313.025 L220.974,313.634 L220.683,315.004 L220.435,315.459 L220.07,315.82 L219.091,316.57 L218.632,316.821 L218.325,316.711 L218.284,316 L218.434,314.725 L218.292,314.532 L217.347,315.902 L216.931,316.25 L216.567,316.334 L216.286,316.155 L216.241,315.826 L216.308,315.285 L216.755,313.273 L217.16,312.317 L217.841,311.473 L218.935,310.699 L218.898,310.286 L218.739,310.305 L218.521,310.436 L218.3,310.368 L217.924,309.638 L217.541,309.438 L217.359,309.464 L217.076,309.727 L216.912,310.053 L216.448,311.357 L216.284,312.157 L214.55,316.717 L214.038,316.896 L213.697,316.606 L213.396,316.169 L213.006,315.91 L213.199,315.565 L213.585,315.173 L213.682,314.74 L213.561,314.362 L213.307,314.109 L212.413,313.964 L212.201,313.765 L212.474,313.232 L212.264,313.177 L211.713,312.832 L211.945,312.417 L212.808,312.147 L214.611,311.269 L215.387,311.285 L215.5,310.557 L215.353,309.935 L214.841,308.999 L214.026,310.168 L212.971,310.985 L212.099,311.215 L211.829,310.62 L212.023,309.935 L212.299,309.205 L212.473,308.582 L212.357,308.229 L212.088,308.572 L210.995,310.26 L210.634,310.747 L209.796,311.126 L208.241,310.745 L207.394,310.776 L206.496,309.485 L206.217,308.442 L206.743,307.666 L206.939,307.32 L206.095,307.36 L205.092,308.248 L204.379,309.439 L204.399,310.389 L204.166,310.8 L203.697,310.955 L202.697,311.537 L202.218,311.686 L201.79,311.607 L200.381,310.619 L199.92,310.004 L199.59,309.749 L199.34,309.38 L199.304,308.827 L199.333,308.205 L199.199,308.15 L199.03,308.095 L198.878,308.113 L198.509,308.229 L198.569,308.358 L198.593,308.467 L198.618,308.78 L197.81,308.707 L196.043,309.072 L192.622,308.056 L192.161,307.273 L191.761,307.923 L191.541,308.083 L191.356,308.056 L191.05,308.152 L189.334,309.192 L184.497,308.116 L183.522,307.416 L182.979,306.271 L183.217,305.862 L183.597,305.495 L183.734,305.043 L183.715,304.561 L183.625,304.11 L184.019,304.365 L184.128,304.574 L184.178,304.908 L184.831,303.934 L184.712,303.196 L184.154,302.988 L183.502,303.589 L183.44,302.473 L180.834,299.11 L174.019,296.254 L173.293,295.638 L173.348,294.458 L174.688,294.406 L175.485,294.178 L176.03,293.741 L176.533,293.525 L178.344,293.404 L179.021,293.151 L179.614,292.35 L180.193,291.361 L180.802,290.614 L181.763,290.533 L182.47,290.215 L182.858,290.268 L183.364,290.689 L184.767,293.21 L185.334,293.664 L185.581,294.111 L185.873,296.393 L186.425,297.45 L187.745,298.74 L187.815,299.467 L187.783,300.263 L187.871,300.991 L188.348,301.711 L189.033,302.023 L190.707,302.042 L189.655,301.523 L189.207,301.085 L188.915,300.622 L188.833,299.926 L188.892,299.108 L188.873,298.355 L188.564,297.868 L188.179,296.025 L188.199,294.602 L188.054,293.232 L188.121,292.23 L191.086,294.796 L192.522,297.31 L193.095,297.914 L193.158,296.827 L192.974,295.782 L192.624,294.932 L191.861,294.035 L191.158,292.608 L190.347,291.654 L190.136,290.898 L190.013,289.309 L190.214,286.897 L190.548,286.16 L190.963,285.512 L191.315,284.854 L191.464,284.08 L192.908,283.742 L193.825,284.334 L200.143,294.832 L200.201,295.218 L200.128,296.253 L200.462,297.065 L200.463,297.421 L200.13,298.527 L199.672,299.096 L199.106,299.539 L198.443,300.27 L198.713,300.721 L199.471,300.581 L201.15,299.759 L201.082,300.084 L201.167,300.754 L201.708,300.367 L202.753,301.339 L203.396,301.458 L203.611,301.081 L203.476,300.865 L203.455,300.689 L203.579,300.082 L203.988,300.263 L205.408,300.077 L205.798,300.227 L206.128,300.474 L206.498,300.613 L207.006,300.439 L204.204,299.239 L203.214,298.186 L202.659,297.125 L202.558,296.59 L202.532,295.724 L202.668,294.988 L203.05,294.833 L204.825,294.925 L205.056,295.217 L205.224,295.72 L207.202,296.45 L208.093,296.224 L208.625,296.346 L209.389,297.255 L209.842,297.561 L210.55,297.374 L210.431,296.933 L210.343,296.763 L210.15,296.669 L210.343,296.33 L211.829,296.141 L212.387,295.897 L211.054,295.421 L210.157,295.373 L208.686,295.747 L202.727,292.348 L201.825,291.215 L201.481,290.041 L200.049,289.844 L199.918,289.464 L199.938,288.725 L199.495,286.543 L199.644,284.664 L200.208,282.614 L201.252,280.994 L202.833,280.417 L203.286,280.628 L204.096,282.09 L205.534,283.367 L205.792,283.868 L205.985,285.141 L206.201,285.699 L206.592,285.98 L208.245,286.309 L210.358,288.917 L211.274,289.407 L212.187,291.454 L212.846,292.276 L213.846,292.301 L212.341,290.704 L212.18,289.637 L210.433,288.47 L210.055,287.813 L209.559,286.288 L207.571,284.484 L207.285,283.818 L207.045,281.488 L206.638,280.806 L206.638,280.13 L206.278,279.664 L205.831,278.131 L205.669,276.357 L205.731,275.802 L206.713,274.107 L207.357,273.287 L208.068,272.809 L209.175,272.965 L210.165,273.853 L212.749,278.502 L213.747,279.45 L214.808,279.411 L214.415,280.331 L214.265,281.217 L214.358,281.936 L214.706,282.357 L214.904,282.012 L214.863,281.222 L215.165,279.139 L214.915,278.533 L214.476,278.141 L213.753,277.077 L213.291,276.672 L213.153,276.351 L213.02,275.047 L212.412,273.158 L212.283,271.695 L213.346,271.478 L215.381,272.327 L216.701,273.355 L217.718,276.06 L217.476,275.282 L217.442,273.361 L217.118,272.785 L214.968,271.692 L214.671,271.015 L214.908,270.606 L215.365,270.307 L215.973,269.984 L216.585,269.768 L216.855,269.782 L217.057,269.794 L217.997,270.24 L218.525,270.111 L218.976,269.4 L219.626,269.41 L222.51,272.283 L223.797,275.177 L224.373,275.686 L224.608,276.276 L224.69,277.416 L224.507,278.614 L223.952,279.389 L223.425,280.67 L225.512,279.501 L226.062,279.526 L226.219,279.797 L226.239,280.245 L226.162,281.092 L226.233,281.898 L226.109,282.221 L225.66,283.032 L225.275,283.988 L225.007,284.54 L224.629,284.869 L223.85,285.386 L223.7,285.45 L221.61,286.847 L221.085,287.273 L221.224,287.346 L222.149,286.814 L224.227,285.939 L225.094,285.43 L225.471,285.049 L225.856,284.598 L226.226,284.268 L226.555,284.24 L226.724,284.554 L226.66,285.057 L226.252,286.531 L226.009,286.721 L225.869,286.863 L226.37,286.713 L227.023,286.368 L227.552,286.253 L227.684,286.795 L226.581,287.88 L225.329,288.898 L224.096,289.605 L223.06,289.742 L223.992,290.157 L224.522,290.156 L225.166,289.871 L226.39,288.678 L227.189,288.004 L227.533,287.957 L227.007,289.614 L226.001,291.08 L225.066,292.274 L224.747,293.121 L227.013,290.841 L227.664,289.582 L228.555,288.205 L229.484,286.979 L230.238,286.177 L230.653,285.852 L230.824,285.828 L232.257,286.762 L232.311,287.017 L231.944,287.795 L231.832,288.324 L231.871,288.705 L232.212,290.074 L232.157,292.05 L232.177,292.316 L232.058,292.577 L231.288,293.727 L230.853,294.289 L230.416,294.756 L230.086,294.937 L230.031,295.17 L230.049,295.357 L227.072,298.025 L226.267,298.921 L226.168,299.477 L226.51,299.542 L228.228,297.679 L231.973,294.4 L232.728,293.382 L232.959,292.967 L233.488,293.128 L233.501,293.813 L233.324,294.735 L233.281,295.62 L233.421,295.745 L233.626,295.677 L233.764,295.751 L233.686,296.318 L233.425,296.758 L233.066,297.127 L232.839,297.492 L233,297.915 L233.195,297.653 L233.609,297.15 L233.993,296.805 L234.089,297.014 L233.881,297.395 L232.68,299.211 L232.148,299.737 L231.038,300.534 L230.48,301.038 L230.289,301.384 L231.426,301.336 L234.015,298.692 L235.053,298.485 L235.282,298.069 L235.102,297.32 L235.436,296.116 L235.951,294.806 L236.3,293.728 L236.41,292.769 L236.395,291.802 L236.229,291.001 L235.881,290.536 L236.512,289.936 L236.986,289.31 L237.459,288.754 L238.089,288.362 L238.639,288.362 L239.1,288.566 L239.604,288.565 L240.281,287.945 L239.812,287.726 L238.396,287.72 L237.746,287.917 L237.242,287.865 L235.894,285.952 L235.38,283.982 L231.394,278.886 L230.787,276.396 L231.25,273.453 L232.664,273.324 L236.456,274.148 L237.131,274.546 L239.059,276.533 L239.671,276.853 L239.856,277.072 L239.848,277.361 L239.776,277.661 L239.759,277.92 L240.006,278.86 L240.275,279.105 L240.762,278.931 L240.718,278.663 L240.749,278.357 L240.959,277.524 L242.051,278.101 L242.7,278.237 L244.115,277.951 L245.287,278.41 L246.036,278.248 L242.751,277.173 L242.102,276.208 L241.926,275.514 L242.06,275.488 L242.439,275.633 L243.01,275.436 L243.476,275.143 L243.985,274.98 L244.458,274.982 L244.815,275.182 L245.046,274.767 L244.714,274.225 L244.361,274.178 L242.963,274.661 L241.487,274.707 L240.107,274.149 L239.194,273.39 L239.078,273.168 L239.384,272.554 L239.8,272.253 L240.563,271.942 L240.865,271.591 L241.606,270.542 L241.889,270.239 L242.379,270.062 L243.687,270.157 L242.615,269.859 L241.598,270.003 L238.778,271.004 L238.238,271.018 L237.238,270.688 L237.15,270.222 L237.707,269.576 L238.476,268.995 L239.014,268.732 L239.246,268.321 L238.312,268.241 L237.755,268.37 L237.103,268.663 L235.322,269.798 L235.081,269.82 L234.188,270.18 L231.225,267.401 L230.615,266.395 L230.472,265.278 L230.318,264.728 L230.372,264.289 L230.959,263.935 L231.837,264.323 L232.661,265.306 L233.682,265.663 L235.14,264.198 L234.573,263.775 L234.185,262.871 L233.99,261.747 L234.002,260.651 L234.986,260.455 L235.49,261.144 L235.823,262.136 L236.284,262.842 L236.473,262.136 L236.514,261.81 L236.493,261.439 L239.127,262.302 L239.246,262.45 L239.254,262.743 L239.057,263.514 L239.057,263.818 L239.461,264.191 L239.897,263.875 L240.129,263.28 L239.928,262.81 L239.649,262.479 L239.126,260.915 L241.318,262.072 L241.607,262.452 L241.943,263.64 L242.716,264.547 L242.955,265.948 L243.224,266.497 L243.843,266.693 L245.28,266.111 L246.011,266.001 L246.076,265.773 L246.297,265.173 L247.34,265.306 L247.726,266.258 L247.957,267.448 L249.053,269 L249.425,269.102 L249.904,268.471 L250.049,267.84 L249.698,267.134 L249.825,266.52 L250.462,267.011 L250.757,268.058 L250.669,269.415 L250.145,270.831 L250.079,271.848 L251.317,273.79 L251.3,275.122 L250.679,275.513 L250.781,275.994 L251.602,276.695 L251.393,277.075 L251.062,276.73 L250.761,276.88 L250.466,277.34 L250.142,277.933 L250.094,278.277 L250.076,278.764 L249.887,279.2 L249.325,279.416 L249.098,279.827 L249.178,280.355 L249.667,280.172 L250.174,280.075 L250.635,280.095 L250.98,280.266 L251.048,280.619 L251.018,281.091 L251.141,281.368 L251.527,281.073 L252.241,280.541 L252.659,280.61 L252.651,281.104 L252.078,281.842 L253.757,282.482 L254.068,282.858 L254.453,282.608 L254.661,282.691 L254.756,282.954 L254.816,283.239 L254.828,283.511 L254.914,285.232 L254.488,285.519 L252.562,285.974 L252.748,286.329 L253.118,286.321 L253.576,286.183 L254.007,286.183 L254.29,286.434 L254.853,287.847 L254.902,288.418 L254.995,288.664 L255.208,288.79 L255.82,288.727 L256.094,288.785 L257.087,290.257 L256.913,290.423 L256.528,290.899 L256.777,291.348 L256.64,292.128 L256.27,292.801 L255.806,292.957 L255.702,293.347 L255.586,293.99 L255.475,294.289 L255.493,294.272 L255.036,295.205 L254.571,296.336 L254.518,296.786 L254.671,297.226 L255.017,297.337 L255.547,297.286 L255.985,297.377 L256.056,297.923 L255.685,298.717 L255.11,299.317 L254.496,299.822 L253.998,300.325 L254.33,300.388 L254.828,300.309 L255.305,300.136 L255.589,299.912 L256.101,298.984 L256.31,298.541 L256.448,297.194 L256.895,297.023 L257.916,297.014 L258.954,297.445 L259.021,297.72 L258.922,298.195 L258.794,298.661 L258.781,298.897 L259.09,299.305 L258.585,299.606 L257.892,299.761 L257.642,299.73 L257.207,300.19 L257.089,300.647 L257.118,301.621 L257.433,302.39 L260.251,302.92 L260.808,303.418 L261.174,303.998 L261.236,304.854 L260.879,306.179 L257.528,304.898 L256.65,305.105 L253.878,304.626 L253.544,304.45 L253.536,305.449 L254.218,305.678 L255.87,305.614 L258.201,306.284 L258.661,307.005 L258.135,307.342 L256.781,307.563 L256.577,307.945 L259.398,308.45 L259.721,309.266 L259.454,310.557 L259.014,311.556 L258.357,312.154 L257.447,312.243 L258.13,313.543 L257.91,313.96 L255.359,316.078 L254.562,316.601 L253.558,316.89 L250.759,316.643 L250.536,317.062 L251.011,317.345 L251.949,317.605 L252.266,317.924 L253.063,319.315 L253.234,319.849 L250.142,321.208 L249.191,321.397 L246.161,319.706 L246.079,319.336 L246.092,318.42 L246.069,318.213 L245.807,317.66 L245.792,316.715 L245.933,315.797 L246.148,315.301 L246.012,314.215 L245.522,313.472 L244.778,313.123 L243.898,313.202 L244.301,313.498 L244.99,314.751 L245.163,315.291 L245.129,315.467 L245.119,316.619 L245.172,316.75 L244.869,317.455 L244.558,317.864 L244.174,318.144 L243.65,318.447 L243.813,319.013 L243.852,319.391 L243.777,319.722 L243.499,320.524 L243.535,321.078 L243.677,321.438 L243.714,321.652 L243.919,322.231 L247.801,324.04 L248.274,324.716 L248.191,325.598 L247.618,326.12 L245.414,327.442 L244.68,327.721 L244.479,328.099 L245.925,328.03 L248.019,327.436 L249.373,327.756 L248.612,330.416 L247.043,332.223 L244.844,333.624 L242.721,334.383 L241.371,334.287 L241.168,334.67 L243.985,334.913 L245.31,334.546 L245.911,334.607 L246.372,334.974 L246.547,335.445 L246.61,336.885 L247.044,338.172 L246.993,338.836 L246.332,339.683 L246.55,340.052 L246.379,340.499 L246.017,341.054 L245.658,341.764 L245.146,343.072 L244.694,343.942 L243.997,345.015 L243.717,345.575 L243.077,348.972 L243.093,349.703 L242.946,349.758 L242.904,349.783 L242.872,349.831 L242.738,349.963 L242.941,350.697 L242.924,351.863 L242.598,353.081 L241.869,353.964 L241.686,354.314 L241.891,354.839 L242.105,354.786 L242.453,354.698 L243.04,354.018 L243.413,352.482 L243.82,351.542 L243.969,351.098 L244.063,350.424 L244.096,349.375 L244.171,348.789 L244.843,348.329 L245.069,348.222 L245.24,347.694 L245.576,347.411 L245.975,347.213 L246.331,346.954 L246.909,346.153 L247.304,345.367 L249.045,340.88 L249.884,339.055 L250.597,338.406 L251.097,339.165 L251.07,340.411 L250.532,343.149 L250.496,344.852 L250.394,345.29 L250.34,345.774 L250.999,346.423 L251.167,346.871 L251.561,346.533 L251.626,344.945 L251.976,342.951 L252.525,340.905 L253.196,339.156 L253.8,337.794 L254.269,336.878 L254.797,336.087 L255.061,335.829 L255.622,335.432 L255.866,335.212 L256.954,333.708 L257.218,333.378 L258.64,332.346 L262.325,331.535 L264.739,330.304 L265.289,330.272 L265.797,330.358 L266.242,330.282 L266.607,329.766 L266.937,329.862 L267.07,330.141 L267.169,330.998 L267.259,331.065 L267.591,331.139 L267.644,331.227 L267.461,331.745 L267.313,331.868 L267.184,331.846 L267.055,331.95 L265.731,334.125 L265.125,335.251 L264.877,336.173 L264.108,336.991 L263.588,337.965 L263.533,338.676 L264.161,338.71 L264.455,337.565 L264.964,336.675 L265.482,336.396 L265.798,337.097 L265.812,337.561 L265.719,337.796 L265.539,338.119 L265.291,338.819 L264.975,339.202 L264.73,339.556 L264.572,339.865 L264.663,339.884 L264.963,340.354 L265.023,340.563 L264.945,341.111 L264.834,341.532 L264.777,341.927 L264.853,342.401 L265.003,342.739 L265.21,343.01 L265.476,343.2 L265.797,343.291 L265.903,343.273 L266.502,343.172 L267.013,342.668 L267.578,341.854 L268.467,340.777 L268.321,340.414 L268.353,339.486 L268.171,339.202 L267.338,338.888 L267.009,338.594 L266.883,338.059 L267.81,338.127 L268.568,338.427 L268.937,338.456 L269.847,338.232 L270.241,338.275 L270.265,338.945 L268.891,341.408 L268.814,342.379 L269.193,341.43 L269.752,340.858 L270.232,340.279 L270.388,339.298 L271.034,338.676 L272.34,337.933 L273.041,337.431 L273.339,337.06 L273.834,336.155 L274.163,335.792 L274.49,335.68 L274.913,335.89 L275.462,335.978 L275.003,335.625 L275.041,334.779 L275.399,333.974 L275.904,333.747 L276.316,333.669 L276.835,333.198 L277.385,332.462 L277.904,331.568 L278.457,330.035 L278.684,328.498 L278.647,327.182 L278.406,326.328 L278.62,325.909 L279.527,325.623 L280.132,323.874 L280.474,321.761 L280.624,319.359 L280.732,318.757 L281.578,316.747 L281.914,315.746 L282.04,314.901 L282.006,314.163 L281.863,313.481 L282.495,313.098 L283.255,312.397 L283.83,311.534 L283.906,310.671 L284.576,310.478 L285.171,310.643 L286.052,311.221 L286.33,311.232 L286.987,310.947 L287.306,310.862 L287.654,310.908 L288.408,311.387 L288.975,311.343 L290.33,310.682 L290.811,310.62 L291.715,311.562 L291.949,311.707 L292.197,311.858 L292.922,311.621 L292.427,312.881 L292.372,313.727 L292.596,314.362 L292.956,314.976 L292.585,315.327 L292.709,315.664 L292.662,316.169 L292.447,317.126 L292.324,318.106 L292.348,320.421 L292.236,320.986 L292.092,321.464 L292.03,321.881 L292.183,322.249 L292.015,322.386 L291.66,322.746 L291.493,322.88 L291.56,323.844 L291.838,324.739 L291.904,325.654 L291.332,326.668 L291.546,327.263 L292.683,328.05 L293.294,329.103 L293.579,329.218 L293.805,329.389 L293.858,329.993 L293.914,330.209 L294.237,330.292 L294.313,330.443 L294.253,330.921 L294.115,331.35 L293.955,331.761 L293.839,332.167 L293.785,333.535 L293.714,334.016 L293.465,334.765 L292.831,336.355 L292.66,337.075 L292.817,337.999 L293.822,338.907 L293.94,339.829 L294.684,338.688 L295.06,338.335 L295.728,338.757 L295.769,338.897 L295.852,339.522 L295.942,339.759 L296.166,339.956 L296.841,340.224 L296.574,340.254 L296.308,340.37 L296.112,340.594 L295.967,341.094 L296.157,341.201 L296.722,341.113 L297.137,340.887 L297.739,340.196 L298.798,338.698 L299.775,337.12 L300.49,335.702 L300.997,334.296 L301.348,332.747 L301.662,330.479 L301.777,330.093 L301.186,327.92 L301.376,327.511 L301.741,327.026 L302.162,326.665 L302.542,326.602 L303.008,326.698 L303.61,326.619 L304.281,326.344 L304.946,325.852 L304.854,325.411 L304.857,324.851 L304.953,324.207 L305.329,322.902 L305.422,322.708 L305.597,322.601 L306.044,322.252 L306.715,321.921 L308.377,321.814 L311.697,322.69 L312.363,324.015 L312.885,324.38 L313.295,323.42 L313.585,323.275 L313.836,323.22 L314.06,323.262 L314.253,323.421 L314.081,323.611 L313.716,324.126 L314.087,324.52 L314.465,325.698 L314.878,326.032 L314.723,325.623 L314.673,325.202 L314.723,324.72 L314.863,324.134 L315.02,324.205 L315.194,323.849 L315.111,323.698 L315.009,323.364 L314.929,323.214 L315.1,322.863 L315.508,322.566 L316.407,321.344 L316.851,320.812 L317.245,320.567 L318.387,320.214 L320.808,320.376 L321.363,321.061 L320.311,322.811 L320.838,322.829 L321.387,322.467 L322.503,321.479 L322.672,320.873 L322.707,320.624 L323.291,320.475 L323.858,320.866 L324.731,321.96 L324.141,322.729 L323.156,323.791 L322.674,324.606 L322.938,324.702 L325.436,323.699 L326.034,324.157 L326.259,325.091 L326.373,326.049 L326.629,326.593 L326.652,327.096 L325.951,327.881 L324.603,329.036 L324.883,329.191 L325.257,329.248 L325.67,329.18 L326.06,328.957 L326.434,328.861 L326.619,329.237 L326.732,329.745 L326.896,330.057 L327.053,330.645 L326.801,331.742 L326.433,332.979 L326.259,333.975 L325.016,335.493 L324.354,336.414 L324.2,336.88 L325.66,338.045 L326.404,339.205 L328.007,340.222 L328.15,340.515 L328.188,341.071 L328.137,342.178 L328.471,342.846 L328.375,343.511 L328.183,344.227 L328.034,345.071 L327.96,345.341 L327.949,345.564 L328.101,345.72 L328.559,345.843 L328.748,345.981 L328.808,346.26 L328.706,347.074 L328.548,347.571 L327.75,349.138 L327.537,349.827 L327.487,351.039 L327.413,351.534 L326.282,353.931 L326.185,354.334 L326.312,354.583 L326.523,355.895 L327.086,357.548 L326.601,358.259 L326.241,359.329 L326.145,360.265 L326.466,360.57 L327.095,359.987 L327.564,358.93 L327.848,357.806 L327.93,357.021 L327.842,356.722 L327.53,356.133 L327.496,355.849 L327.697,355.157 L327.938,354.819 L328.187,354.611 L328.418,354.325 L328.655,353.781 L328.786,353.362 L328.956,352.942 L329.332,352.372 L329.56,352.202 L330.137,352.005 L330.339,351.869 L330.973,350.868 L331.329,349.924 L331.892,346.908 L331.959,346.721 L332.349,345.933 L332.391,345.69 L332.244,345.638 L332.048,345.631 L331.948,345.522 L331.966,344.305 L331.881,344.033 L331.884,343.964 L331.909,343.823 L331.896,343.68 L331.8,343.574 L331.626,343.646 L331.178,344.109 L331.044,344.182 L330.829,343.83 L330.545,342.587 L330.505,342.064 L331.105,341.351 L331.378,340.545 L331.353,339.844 L330.894,339.101 L330.984,337.715 L330.946,337.039 L331.131,336.648 L331.304,336.724 L331.507,336.295 L331.632,334.424 L332.603,331.637 L333.995,329.019 L335.395,327.645 L335.766,327.643 L337.77,328.689 L337.996,329.015 L338.319,329.751 L338.591,329.99 L339.624,330.321 L339.937,330.579 L339.792,330.767 L339.756,330.844 L339.723,330.875 L339.566,330.929 L340.192,331.819 L342.756,332.206 L343.534,333.083 L343.519,333.605 L343.233,334.836 L343.237,335.347 L343.433,335.589 L344.431,335.863 L344.724,336.281 L344.968,337.678 L345.29,338.107 L345.472,338.518 L345.718,340.348 L345.719,341.106 L345.332,342.918 L345.244,343.8 L345.733,344.685 L345.867,345.43 L345.927,346.862 L345.625,349.587 L345.62,350.478 L346.215,349.426 L346.411,348.487 L346.458,347.586 L346.601,346.637 L347.043,346.859 L347.646,346.94 L348.297,346.88 L348.887,346.679 L346.69,345.879 L346.342,345.164 L350.088,346.682 L350.235,347.073 L350.092,347.793 L349.943,348.996 L350.086,349.057 L350.398,348.704 L350.619,348.29 L350.712,347.858 L350.657,347.431 L350.822,347.071 L351.241,346.937 L351.703,346.619 L352.115,346.091 L352.388,345.309 L352.952,345.044 L353.23,344.824 L353.545,344.442 L353.71,344.119 L354.062,343.206 L354.44,342.479 L355.089,341.817 L356.026,341.308 L356.977,341.019 L357.66,341.039 L358.052,340.837 L358.491,340.22 L358.819,339.556 L358.864,339.207 L358.687,338.926 L358.987,338.566 L359.826,337.959 L360.761,337.561 L362.429,337.616 L363.128,336.713 L363.479,337.548 L365.707,338.766 L366.125,339.446 L366.126,339.5 L366.133,340.684 L365.928,343.524 L365.763,343.826 L365.708,343.947 L365.669,344.064 L365.559,344.347 L365.24,344.214 L365.064,344.607 L365.227,344.967 L365.463,345.215 L365.78,345.28 L366.195,345.077 L366.061,344.966 L365.876,344.478 L366.991,344.191 L370.298,344.585 L371.041,345.216 L370.671,345.047 L370.245,345.021 L369.774,345.135 L369.255,345.411 L369.463,345.739 L369.512,346.744 L369.71,346.74 L370.696,345.784 L371.18,345.434 L371.52,345.413 L371.894,346.327 L371.924,347.548 L372.055,348.634 L372.738,349.13 L372.898,348.768 L372.39,347.917 L372.2,347.351 L372.133,346.83 L372.285,345.999 L372.53,345.429 L372.637,344.94 L372.394,344.333 L373.799,343.998 L375.625,344.12 L376.856,343.901 L377.404,344.105 L374.8,344.481 L373.841,344.928 L376.251,344.95 L376.725,345.177 L377.063,345.93 L377.359,347.696 L377.932,348.037 L378.121,347.607 L377.674,346.411 L377.542,345.594 L377.645,344.879 L378.124,344.036 L378.693,343.39 L379.298,342.961 L379.878,342.749 L380.012,342.482 L380.006,342.397 L380.208,342.152 L380.611,341.784 L381.012,341.292 L381.31,340.773 L381.639,339.866 L381.697,339.319 L381.645,338.833 L381.659,337.809 L381.63,337.573 L381.553,337.395 L381.428,337.294 L381.4,337.031 L381.572,336.527 L381.783,335.998 L381.876,335.663 L381.463,334.879 L381.509,334.352 L382.127,333.397 L382.06,333.345 L381.811,333.268 L381.977,333.111 L382.32,332.734 L382.49,332.576 L382.399,332.345 L382.394,332.052 L382.507,331.231 L382.996,331.431 L383.17,331.036 L383.017,330.312 L383.387,328.294 L383.076,327.252 L383.636,326.542 L384.422,326.036 L384.957,326.149 L384.942,326.545 L384.766,326.991 L384.668,327.372 L384.894,327.556 L385.072,327.321 L385.411,326.806 L385.823,326.332 L386.234,326.206 L387.005,326.191 L387.37,326.364 L387.177,326.823 L386.779,327.411 L386.621,327.981 L386.644,328.46 L386.798,328.756 L386.892,328.423 L386.986,328.162 L387.071,327.895 L387.131,327.54 L387.558,327.545 L388.496,327.238 L388.754,327.232 L388.963,327.668 L389.021,328.215 L389.138,328.694 L389.51,328.927 L389.031,329.148 L388.848,329.165 L389.088,329.686 L389.555,329.775 L390.103,329.737 L390.268,329.74 L390.456,329.31 L390.89,329.005 L391.37,328.546 L391.798,328.294 L391.776,328.456 L391.648,329.184 L391.873,329.391 L392.267,329.157 L392.932,328.143 L393.247,327.958 L393.869,327.907 L394.509,328.165 L395.576,329.268 L395.903,329.227 L397.18,329.187"
         id="path1007" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['IS'],polycolors['IS'],labels['IS'], x1, polybary['IS'], x2, polybary['IS'] )
    if 'IS' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1726.57,1513.95 L1726.85,1513.18 L1727.5,1514.01 L1728.34,1514 L1729.93,1513.27 L1730.7,1513.46 L1731.89,1514.42 L1732.27,1514.04 L1734.38,1515.41 L1734.74,1515.81 L1735.13,1516.8 L1735.48,1516.5 L1735.79,1515.71 L1736.07,1515.27 L1736.68,1515.6 L1737.05,1516.26 L1737.5,1516.84 L1739.02,1517.05 L1739.14,1517.57 L1738.78,1518.14 L1738,1518.38 L1738.08,1518.76 L1739.35,1519.17 L1740.17,1519.25 L1740.58,1518.71 L1740.96,1518.79 L1741.4,1519.01 L1741.69,1519.27 L1741.69,1519.32 L1741.74,1519.63 L1741.44,1520.29 L1741.41,1520.66 L1742.06,1522.08 L1743.04,1523.09 L1744.21,1523.74 L1745.42,1524.11 L1745.52,1524.56 L1744.78,1525.14 L1744.88,1525.39 L1744.99,1525.61 L1745.12,1525.8 L1745.26,1525.95 L1746.2,1525.23 L1746.91,1525.89 L1747.47,1526.63 L1747.95,1526.19 L1748.54,1526.62 L1748.72,1527.17 L1748.56,1527.78 L1748.18,1528.34 L1749.19,1528.83 L1749.48,1528.91 L1749.51,1529.03 L1749.57,1529.29 L1749.08,1530.45 L1749.65,1530.9 L1750.69,1531.2 L1751.64,1531.91 L1751.67,1532.2 L1752.17,1534.07 L1752.22,1534.46 L1752.72,1534.68 L1753.25,1534.71 L1753.75,1534.82 L1754.18,1535.27 L1752.98,1536.48 L1753.56,1536.77 L1754.33,1537.52 L1755,1538.33 L1755.31,1538.8 L1755.53,1539.04 L1756.38,1539.41 L1756.64,1539.56 L1756.69,1539.66 L1756.93,1540.14 L1757.01,1540.81 L1757.34,1541.2 L1757.43,1541.58 L1756.53,1542 L1757.21,1543.25 L1758.53,1544.65 L1762.02,1547.56 L1763.43,1548.46 L1764.83,1548.65 L1764.91,1548.81 L1765.07,1549.13 L1765.43,1549.86 L1765.65,1550.21 L1765.76,1550.42 L1765.88,1550.97 L1766.02,1551.22 L1766.21,1551.29 L1766.67,1551.24 L1766.78,1551.26 L1767.46,1552.83 L1767.93,1553.65 L1768.46,1553.94 L1768.83,1553.9 L1769.07,1553.98 L1769.26,1554.09 L1769.53,1554.16 L1770.6,1553.93 L1770.65,1553.88 L1771.26,1553.98 L1771.96,1554.19 L1772.6,1554.54 L1773.03,1555.07 L1772.76,1555.42 L1772.64,1555.94 L1772.63,1556.61 L1772.68,1557.36 L1773.56,1557.82 L1773.67,1559.78 L1774.34,1560.04 L1774.46,1560.1 L1775.01,1560.39 L1776.57,1562.55 L1777.31,1563.32 L1777.67,1563.41 L1778.84,1563.39 L1779.34,1563.56 L1779.72,1563.75 L1780.09,1563.88 L1781.18,1563.88 L1781.72,1564.24 L1782.91,1565.51 L1783.04,1566.29 L1783.66,1566.95 L1785.02,1568.04 L1785.91,1569.58 L1785.99,1569.6 L1786.22,1570.06 L1786.42,1570.39 L1786.41,1570.67 L1785.98,1570.93 L1786.63,1571.5 L1788.65,1572.04 L1788.75,1572.45 L1788.61,1572.96 L1788.83,1573.1 L1789.26,1573.13 L1789.71,1573.32 L1791.31,1574.42 L1792.66,1574.9 L1793.29,1575.25 L1793.51,1576.12 L1794.57,1576.07 L1795.69,1576.45 L1796.68,1576.65 L1797.4,1576.04 L1798.02,1576.95 L1798.08,1577 L1799.75,1578.32 L1800.56,1579.22 L1801.91,1581.23 L1802.24,1582.12 L1802.29,1582.27 L1802.16,1583.27 L1802.96,1583.58 L1803.16,1584.26 L1803.29,1585.01 L1803.89,1585.49 L1804,1585.94 L1804.11,1586.69 L1804.18,1587.18 L1805.26,1588.73 L1807.64,1591.21 L1807.91,1592.35 L1811.38,1596.49 L1812.2,1598.03 L1812.29,1598.26 L1812.54,1598.88 L1812.73,1599.72 L1812.61,1600.44 L1812.38,1601.31 L1812.38,1602.24 L1812.98,1603.15 L1812.66,1603.52 L1812.61,1604.01 L1813.08,1606.32 L1813.28,1606.78 L1814.08,1607.73 L1814.29,1607.84 L1814.45,1607.73 L1814.61,1607.76 L1814.79,1608.25 L1815.25,1610.14 L1815.32,1610.66 L1815.19,1611.19 L1814.67,1611.63 L1815.17,1612.39 L1815.06,1612.78 L1814.99,1612.83 L1814.72,1613.05 L1814.51,1613.4 L1814.71,1615.32 L1814.58,1615.63 L1814.11,1615.93 L1814.12,1616.4 L1814.58,1617.59 L1814.8,1618.33 L1814.79,1618.56 L1814.52,1619.97 L1814.14,1620.59 L1813.79,1620.91 L1813.35,1621.17 L1813,1621.5 L1812.92,1621.96 L1813.31,1622.97 L1813.29,1623.32 L1812.85,1623.57 L1812.41,1623.92 L1812.62,1624.42 L1813.69,1625.5 L1814.04,1625.79 L1814.39,1626.15 L1814.65,1626.65 L1814.5,1628.19 L1814.61,1629.75 L1814.4,1631.99 L1814.36,1632.35 L1814.27,1633.28 L1813.99,1633.59 L1813.95,1633.92 L1814.04,1634.25 L1814.34,1634.36 L1814.43,1634.69 L1814.42,1636.96 L1814.48,1637.58 L1814.83,1638.4 L1816.59,1641.2 L1816.83,1641.87 L1817.02,1642.21 L1817.37,1642.57 L1817.66,1643.01 L1817.68,1643.49 L1817.62,1644 L1817.68,1644.5 L1817.9,1644.92 L1818.46,1645.6 L1818.6,1645.81 L1818.55,1646.14 L1818.09,1646.69 L1817.98,1647.07 L1818.52,1648.8 L1819.61,1650.49 L1820.7,1651.76 L1821.2,1652.24 L1823.02,1658.26 L1823.17,1659.26 L1823.62,1659.99 L1823.09,1660.44 L1821.69,1660.78 L1821.18,1661.12 L1820.6,1661.81 L1820.06,1662.66 L1819.69,1663.48 L1822.36,1664.16 L1823.52,1664.68 L1824.54,1665.8 L1824.79,1666.14 L1824.93,1666.24 L1825.02,1666.37 L1825.53,1668.41 L1825.98,1668.96 L1827.43,1670.06 L1828.17,1670.62 L1828.26,1670.59 L1829.2,1670.24 L1830.61,1670.63 L1831.71,1671.57 L1831.78,1672.84 L1833.83,1675.81 L1835.02,1677.12 L1836.15,1677.51 L1836.92,1677.52 L1838.62,1678.01 L1842.65,1678.02 L1846.64,1679 L1852.62,1678.82 L1854.16,1678.2 L1855.49,1677.81 L1855.97,1677.44 L1856.32,1676.75 L1854.97,1676.51 L1854.6,1676.32 L1854.29,1675.92 L1854.04,1675.4 L1854,1674.91 L1855.07,1673.97 L1854.78,1673.08 L1854.61,1672.19 L1855.67,1671.62 L1856.39,1671.67 L1857.71,1672.19 L1858.72,1672.11 L1859.36,1672.36 L1860.24,1672.93 L1861.21,1673.21 L1862.1,1672.59 L1862.19,1672.12 L1862.03,1670.91 L1862.16,1670.32 L1862.4,1670.01 L1862.75,1669.8 L1863.68,1669.52 L1864.19,1669.08 L1865.92,1666.68 L1867.52,1665.72 L1868.17,1665.2 L1868.62,1664.39 L1868.93,1664.13 L1869.94,1663.79 L1870.11,1663.56 L1870.16,1663.08 L1870.41,1662.55 L1870.91,1661.82 L1871.69,1661.07 L1872.38,1660.86 L1873.11,1660.8 L1873.99,1660.55 L1874.32,1660.31 L1874.63,1659.95 L1874.88,1659.55 L1875.04,1659.16 L1875.37,1658.79 L1875.86,1658.77 L1876.38,1658.89 L1876.83,1658.91 L1880.17,1658.37 L1886.61,1658.93 L1889.05,1660.19 L1889.62,1660.33 L1890.79,1660.97 L1892.04,1662.85 L1892.86,1665.16 L1892.76,1667.06 L1893.1,1667.37 L1893.28,1667.79 L1893.3,1668.27 L1893.19,1668.77 L1893.19,1668.78 L1893.19,1668.78 L1892.79,1668.8 L1891.69,1669.14 L1891.78,1669.52 L1892.65,1670.24 L1893.39,1671.2 L1894.33,1671.82 L1895.76,1671.59 L1895.04,1672.9 L1894.96,1674.72 L1895.98,1680.16 L1896.01,1681.25 L1895.88,1683.12 L1895.99,1684.01 L1896.19,1685.01 L1896.3,1685.94 L1896.15,1686.6 L1896.76,1687.83 L1896.11,1689.8 L1896.94,1690.78 L1896.84,1691.59 L1896.56,1691.92 L1896.17,1692.1 L1895.8,1692.46 L1895.43,1693.56 L1895.17,1693.93 L1894.21,1693.7 L1893.77,1693.92 L1893.37,1694.27 L1893,1694.5 L1884.97,1698.07 L1879.75,1699.84 L1878.33,1700.76 L1874.77,1704.89 L1873.81,1706.47 L1873.09,1708.33 L1872.69,1707.95 L1872.51,1707.04 L1871.65,1706.08 L1871.14,1705.21 L1872.01,1704.55 L1873,1704.09 L1874.37,1702.85 L1875.98,1701.97 L1876.78,1701.03 L1877.29,1699.77 L1877.2,1698.32 L1876.76,1697.63 L1876.01,1696.96 L1874.4,1695.92 L1874.88,1696.83 L1875.06,1697.29 L1875.24,1697.96 L1874.53,1697.99 L1874.02,1698.59 L1873.5,1698.93 L1872.77,1698.16 L1872.31,1697.09 L1871.3,1693.46 L1871.46,1692.19 L1873.27,1690.92 L1873.58,1689.99 L1873.04,1689.46 L1870.87,1688.7 L1870.25,1688.61 L1868.94,1688.95 L1867.49,1688.85 L1867.07,1689.05 L1866.99,1689.4 L1867,1690 L1866.88,1690.61 L1866.47,1690.97 L1865.53,1691.29 L1865.14,1691.58 L1863.94,1693.5 L1863.65,1694.32 L1863.71,1695.67 L1863.87,1696 L1864.34,1696.2 L1864.54,1696.58 L1864.54,1696.95 L1864.38,1697.17 L1864.21,1697.33 L1864.17,1697.54 L1864.43,1698.05 L1864.85,1698.43 L1865.27,1698.67 L1866.34,1698.84 L1868.95,1700.49 L1868.55,1701.23 L1868.35,1701.46 L1867.98,1701.63 L1868.47,1701.86 L1868.67,1701.89 L1868.78,1702.34 L1868.34,1702.84 L1868.44,1703.41 L1868.91,1703.94 L1869.6,1704.3 L1869.26,1704.73 L1868.76,1705 L1866.86,1705.57 L1866.45,1705.88 L1864.25,1708.17 L1863.92,1708.69 L1863.71,1709.65 L1863.83,1710.46 L1864.2,1710.71 L1864.7,1710.03 L1864.7,1710.82 L1863.96,1711.3 L1863.98,1712.23 L1864.35,1713.17 L1864.78,1713.35 L1866.97,1712.34 L1867.85,1711.45 L1868.56,1710.27 L1869.16,1708.87 L1868.24,1707.79 L1871.48,1707.97 L1872.29,1708.3 L1872.73,1709.31 L1872.67,1710.61 L1872.17,1711.78 L1871.24,1712.35 L1871.75,1711.13 L1872.16,1709.78 L1872.03,1708.74 L1870.9,1708.42 L1871.01,1708.87 L1871.44,1709.21 L1871.39,1709.43 L1870.63,1709.86 L1869.9,1709.61 L1869.63,1709.64 L1869.49,1709.85 L1869.37,1710.19 L1869.26,1710.66 L1868.59,1711.5 L1867.59,1713.3 L1866.93,1713.92 L1865.1,1715.54 L1864.64,1716.23 L1865.05,1714.63 L1865.03,1713.97 L1864.37,1714.04 L1863.84,1714.31 L1863.55,1714.83 L1863.57,1715.67 L1864.13,1715.53 L1864.01,1716.21 L1864.29,1716.64 L1864.81,1716.88 L1865.44,1716.95 L1863.86,1717.92 L1863.34,1718.7 L1863.57,1719.65 L1863.69,1719.31 L1863.91,1718.93 L1863.98,1718.65 L1864.68,1718.85 L1865.36,1719.18 L1865.19,1720.1 L1865.04,1721.78 L1865.1,1723.53 L1865.59,1724.66 L1865.97,1724.66 L1866.2,1724.12 L1866.28,1723.38 L1866.2,1721.87 L1866.51,1721.28 L1867.24,1720.46 L1868.09,1718.36 L1868.65,1717.33 L1869.33,1716.78 L1870.09,1716.35 L1870.53,1715.55 L1870.7,1714.62 L1870.85,1713.34 L1870.86,1714.6 L1870.73,1715.72 L1869.24,1721.97 L1867.58,1724.07 L1864.96,1730.36 L1864.85,1731.86 L1864.89,1732.64 L1864.22,1732.34 L1863.61,1732.82 L1863.16,1733.48 L1862.88,1734.29 L1862.81,1735.1 L1862.89,1735.98 L1863.39,1738.31 L1863.76,1739.01 L1864.76,1740.13 L1865.83,1741.89 L1866.17,1742.77 L1866.16,1743.34 L1866.56,1743.68 L1865.89,1743.48 L1865.38,1743.77 L1865.08,1744.34 L1864.99,1745 L1865.55,1746.06 L1867.45,1750.8 L1868.95,1753.75 L1869.06,1754.11 L1869.07,1754.64 L1868.81,1755.31 L1868.85,1756.49 L1868.66,1758.42 L1868.72,1759.28 L1868.18,1761.65 L1868.13,1762.29 L1868.23,1765.81 L1868.11,1766.44 L1868.11,1767.05 L1868.45,1767.58 L1868.39,1768.17 L1868.43,1768.81 L1868.56,1769.46 L1868.76,1770.1 L1868.7,1770.75 L1862.48,1772.72 L1852.75,1773.44 L1842.73,1771.54 L1840.98,1770.73 L1837.38,1764.2 L1835.94,1762.7 L1835.94,1762.69 L1833.57,1763.57 L1830.94,1766.03 L1828.23,1767.45 L1825.98,1765.51 L1825.61,1765.2 L1824.32,1763.55 L1823.24,1763.44 L1820.85,1764.88 L1812.51,1766.95 L1811.96,1766.72 L1810.78,1765.43 L1810.15,1765.07 L1807.42,1764.47 L1806.27,1763.63 L1806.17,1762.2 L1805.33,1761.81 L1805.25,1761.78 L1804.2,1762.13 L1803.16,1761.83 L1798.35,1761.98 L1794.73,1760.73 L1793.76,1761.73 L1788.73,1763.44 L1785.12,1766.88 L1783.65,1767.72 L1781.68,1768.17 L1781.2,1768.41 L1780.45,1769.16 L1780.06,1769.37 L1779.2,1769.68 L1777.84,1770.54 L1769.4,1773.79 L1765.85,1775.15 L1764.92,1775.51 L1761.76,1777.82 L1758.4,1779.53 L1757.06,1780.69 L1755.81,1782.87 L1755.27,1784.22 L1755.14,1785.24 L1750.64,1790.32 L1750.36,1791.13 L1750.18,1792.08 L1748.64,1795.19 L1747.47,1796.24 L1746.31,1798.08 L1746.28,1798.32 L1745.59,1800.26 L1743.91,1801.45 L1743.62,1801.52 L1740.91,1802.19 L1740.23,1802.93 L1739.67,1803.74 L1738.84,1804.53 L1737.92,1805.17 L1737.13,1805.55 L1736.39,1806.16 L1735.4,1806.47 L1733.38,1806.78 L1733.11,1806.89 L1732.47,1807.15 L1730.79,1808.36 L1729.83,1808.84 L1727.85,1808.98 L1726.04,1808.37 L1724.28,1807.51 L1724.12,1807.45 L1722.48,1806.88 L1720.58,1806.81 L1714.74,1807.57 L1709.14,1806.75 L1699.9,1809.31 L1697.75,1809.42 L1695.67,1809.08 L1687.99,1807.8 L1686.46,1807.92 L1684.99,1808.73 L1682.86,1811.08 L1682.13,1811.46 L1681.19,1811.7 L1673.37,1813.61 L1672.92,1813.58 L1656.67,1812.61 L1653.87,1811.59 L1652.85,1811.58 L1649.23,1813.12 L1648.54,1813.13 L1647.25,1813.15 L1641.99,1811.37 L1634.83,1812.29 L1630.89,1813.59 L1629.37,1814.79 L1627.87,1815.4 L1626.62,1816.21 L1623.14,1817.16 L1617.3,1818.73 L1615.88,1818.64 L1614.59,1817.86 L1613.85,1816.78 L1613.6,1815.39 L1613.92,1810.58 L1614.18,1809.11 L1614.98,1808.37 L1615.87,1808.07 L1617.54,1807.18 L1618.49,1806.88 L1619.9,1805.75 L1620.34,1803.73 L1619.63,1801.9 L1618.55,1801.62 L1617.58,1801.38 L1615.52,1801.46 L1613.8,1801.03 L1603.22,1796.17 L1603.17,1796.15 L1602.79,1795.28 L1602.49,1792.29 L1602,1791.51 L1601.1,1791.26 L1599.23,1791.31 L1597.41,1790.83 L1595.72,1789.82 L1594.29,1788.41 L1593.23,1786.75 L1593.1,1786.28 L1593.05,1785.29 L1592.93,1784.84 L1592.67,1784.44 L1591.61,1783.74 L1591.4,1783.26 L1591.28,1782.46 L1591.26,1781.61 L1591.36,1781 L1591.62,1780.67 L1592.01,1780.47 L1593.55,1780.14 L1594.07,1779.78 L1594.34,1779.12 L1594.31,1778.03 L1594.4,1776.96 L1594.95,1776.22 L1595.78,1775.81 L1596.71,1775.69 L1597.69,1775.91 L1599.47,1776.71 L1600.46,1776.78 L1601.28,1776.48 L1602.19,1775.85 L1602.83,1774.92 L1602.91,1773.76 L1600.29,1771.66 L1599.57,1771.3 L1596.03,1771.48 L1594.43,1771 L1592.73,1770.03 L1589.13,1767.17 L1588.42,1766.89 L1587.58,1766.86 L1586.5,1767 L1586.09,1767.44 L1584.83,1769.38 L1584.06,1770.04 L1582.3,1770.76 L1581.73,1771.36 L1581.62,1772.38 L1578.02,1782.26 L1577.54,1782.95 L1576.53,1783.4 L1575.56,1783.45 L1574.5,1783.18 L1573.62,1782.56 L1572.88,1780.85 L1572.41,1780.51 L1571.86,1780.29 L1571.33,1779.93 L1571.02,1779.49 L1570.74,1778.74 L1570.59,1778.34 L1570.43,1777.91 L1568.74,1775.14 L1568.24,1774.77 L1566.8,1774.77 L1562.5,1773.33 L1561.75,1773.3 L1561.02,1773.62 L1559.52,1774.58 L1557.6,1775.3 L1555.33,1775.67 L1553.08,1775.36 L1551.23,1773.99 L1550.65,1772.94 L1550.22,1771.87 L1549.69,1770.94 L1548.81,1770.34 L1547.91,1770.23 L1545.25,1770.87 L1541.45,1771.03 L1540.68,1770.81 L1539.68,1769.29 L1538.8,1768.81 L1538.77,1768.8 L1538.77,1768.8 L1537.97,1768.59 L1538.01,1767.71 L1538.31,1766.98 L1538.81,1766.38 L1539.92,1765.78 L1542.49,1765.51 L1543.7,1765.13 L1545.41,1764.35 L1545.95,1763.72 L1545.9,1762.52 L1545.4,1761.58 L1544.67,1761.1 L1543.05,1760.74 L1541.87,1760.37 L1539.71,1760.33 L1538.59,1759.79 L1538.52,1759.46 L1538.56,1759.04 L1538.38,1758.74 L1537.63,1758.82 L1537.02,1758.75 L1536.85,1758.25 L1536.97,1757.57 L1537.21,1757.03 L1537.61,1756.75 L1538.72,1756.49 L1539.15,1756.26 L1539.64,1755.72 L1539.74,1755.4 L1539.7,1754.96 L1539.74,1754.07 L1539.83,1753.79 L1540.14,1753.28 L1540.22,1753 L1540.2,1752.69 L1540.14,1752.24 L1540.05,1751.86 L1539.98,1751.78 L1540.24,1751.22 L1540.56,1750.78 L1540.98,1750.47 L1541.52,1750.36 L1542,1749.79 L1542.05,1748.99 L1541.73,1748.2 L1541.27,1747.79 L1539.98,1746.65 L1538.73,1745.91 L1537.4,1745.42 L1532.55,1744.68 L1530.7,1744.91 L1529.87,1744.7 L1528.27,1743.9 L1527.44,1743.16 L1525.63,1741.18 L1524.85,1740.76 L1524.42,1740.94 L1523.83,1741.83 L1523.43,1742.09 L1522.9,1742.04 L1522.47,1741.76 L1522.04,1741.39 L1521.54,1741.1 L1519.34,1740.47 L1517.71,1740 L1516.96,1739.55 L1514.97,1737.58 L1511.8,1735.44 L1510.08,1733.56 L1509.37,1733.01 L1508.6,1732.76 L1507.79,1732.63 L1507.1,1732.3 L1506.7,1731.47 L1507.57,1730.54 L1508.11,1729.81 L1508.1,1728.92 L1507.34,1727.57 L1505.76,1725.55 L1505.46,1724.57 L1505.58,1722.97 L1505.91,1721.77 L1506.01,1721.21 L1506,1720.31 L1505.52,1717.1 L1505.53,1716.45 L1505.62,1715.84 L1505.55,1715.27 L1505.11,1714.72 L1504.54,1714.52 L1504.09,1714.79 L1503.78,1715.37 L1503.63,1716.13 L1503.14,1716.65 L1502.59,1716.91 L1501.99,1716.87 L1501.4,1716.47 L1500.88,1715.69 L1499.71,1714.55 L1499.19,1713.91 L1499.03,1713.53 L1498.95,1713.33 L1498.68,1712.09 L1498.46,1711.59 L1498.11,1711.25 L1497.29,1710.85 L1496.94,1710.61 L1495.16,1708.24 L1494.4,1707.67 L1493.55,1707.46 L1491.88,1707.51 L1491.04,1707.25 L1488.48,1705.46 L1487.6,1705.03 L1485.8,1704.51 L1484.99,1704.06 L1484.19,1703.16 L1482.91,1700.45 L1482.27,1699.59 L1479.11,1696.59 L1480.56,1694.13 L1487.45,1692.97 L1488.27,1691.13 L1490.08,1691.28 L1491.88,1691.75 L1493.29,1692.73 L1493.76,1692.96 L1494.31,1693.08 L1494.63,1693.04 L1496.98,1692.18 L1497.76,1691.63 L1498.32,1690.8 L1498.47,1690.16 L1498.45,1690.02 L1498.39,1689.27 L1498.54,1688.77 L1498.85,1688.61 L1499.27,1688.69 L1499.68,1688.7 L1499.98,1688.3 L1499.92,1687.8 L1499.33,1686.49 L1499.22,1685.91 L1499.36,1685.53 L1500.85,1683.89 L1501.2,1683.63 L1501.64,1683.3 L1502.51,1682.91 L1503.37,1682.82 L1503.8,1682.97 L1504.66,1683.51 L1505.04,1683.63 L1505.46,1683.53 L1506.04,1683.09 L1506.32,1682.95 L1507.11,1683 L1508.81,1683.46 L1509.65,1683.3 L1510.01,1683.07 L1510.38,1683.01 L1510.74,1683.13 L1511.09,1683.45 L1512.03,1684.05 L1512.76,1683.64 L1514.45,1680.84 L1514.7,1680.65 L1515.96,1680.47 L1516.32,1680.09 L1516.68,1679.07 L1516.87,1677.82 L1516.65,1675.04 L1516.88,1673.63 L1517.48,1672.68 L1518.27,1672.25 L1520.07,1671.9 L1520.95,1670.99 L1520.48,1669.67 L1518.88,1667.48 L1518.8,1666.19 L1519.25,1665.12 L1519.85,1664.07 L1520.21,1662.86 L1520.03,1660.46 L1520.1,1659.25 L1520.64,1658.37 L1521.51,1657.99 L1523.09,1657.87 L1523.94,1657.24 L1524.65,1655.94 L1524.82,1655.09 L1524.86,1654.84 L1525.21,1654.04 L1526.34,1653.66 L1527.12,1653.04 L1527.72,1651.77 L1527.68,1650.52 L1526.42,1649.92 L1526.27,1649.82 L1526.14,1649.69 L1526.02,1649.53 L1526.01,1649.15 L1526.05,1648.77 L1526.14,1648.42 L1526.29,1648.1 L1526.82,1645.46 L1527.21,1644.37 L1527.97,1643.4 L1529.45,1642.74 L1529.83,1642.29 L1530.06,1641.44 L1529.95,1640.77 L1529.7,1640.14 L1529.55,1639.39 L1529.51,1638.65 L1529.53,1638.47 L1529.71,1638.37 L1531.4,1636.48 L1531.69,1635.97 L1532.02,1632.79 L1532.18,1632.7 L1532.14,1632.64 L1531.65,1631.98 L1531.32,1631.79 L1530.9,1631.77 L1530.51,1631.65 L1530.36,1631.34 L1530.27,1631.15 L1531.62,1628.9 L1532.42,1627.86 L1534.28,1626.14 L1535.01,1625.16 L1535.2,1624.57 L1535.24,1623.45 L1535.34,1622.85 L1535.65,1622.2 L1536.44,1621.16 L1536.72,1620.61 L1536.86,1619.8 L1536.76,1619.23 L1536.57,1618.69 L1536.47,1617.97 L1536.54,1617.26 L1536.88,1616.26 L1537.01,1615.66 L1537.15,1613.37 L1537.29,1612.61 L1538.54,1609.97 L1539.17,1608.99 L1539.85,1608.4 L1541.63,1607.55 L1542.2,1605.69 L1541.81,1603.64 L1541.11,1601.56 L1540.69,1599.65 L1541.25,1598.01 L1542.27,1596.45 L1544.55,1594.07 L1546.5,1593.21 L1546.71,1593 L1547,1592.69 L1547.14,1592.19 L1547.14,1592.12 L1547.14,1591.71 L1547.1,1591.25 L1547.16,1590.82 L1547.95,1589.16 L1548,1588.61 L1547.95,1587.45 L1548.03,1586.97 L1548.28,1586.53 L1548.9,1585.9 L1549.17,1585.52 L1549.85,1583.99 L1550.24,1583.42 L1550.94,1582.86 L1551.64,1582.47 L1552.16,1582.33 L1554.12,1582.29 L1554.71,1582.14 L1555.22,1581.82 L1555.63,1581.28 L1555.9,1578.73 L1557.13,1578.21 L1560.47,1579.32 L1561.94,1579.31 L1563.49,1578.8 L1564.89,1577.88 L1565.93,1576.61 L1566.72,1575.08 L1567.16,1574.59 L1567.94,1574.05 L1568.82,1573.8 L1569.04,1573.6 L1569.27,1572.84 L1569.11,1572.4 L1568.84,1572.04 L1568.73,1571.52 L1569.03,1570.6 L1569.62,1570.02 L1571.17,1569.11 L1571.74,1568.37 L1572.45,1566.25 L1572.99,1565.33 L1573.78,1564.92 L1574.44,1564.28 L1574.64,1563.38 L1574.07,1562.2 L1574.29,1561.4 L1574.91,1561.24 L1575.7,1561.45 L1576.42,1561.79 L1577.13,1562.07 L1577.84,1562.16 L1578.48,1561.94 L1580,1560.22 L1580.37,1559.07 L1580.79,1556.02 L1581.38,1554.62 L1582.15,1554.06 L1583.11,1554.03 L1586.15,1554.51 L1586.95,1554.9 L1589.01,1556.63 L1591.27,1557.96 L1592.39,1558.88 L1592.71,1558.95 L1593.03,1558.93 L1593.33,1558.84 L1593.63,1558.66 L1593.69,1558.62 L1593.75,1558.6 L1593.81,1558.6 L1593.88,1558.61 L1596.79,1559.46 L1597.63,1559.25 L1597.77,1559.09 L1597.88,1558.97 L1598.06,1558.64 L1598.21,1558.25 L1598.31,1557.8 L1598.69,1557.35 L1599.12,1557.13 L1600.62,1556.56 L1601.43,1556.67 L1604.14,1556.48 L1605.94,1556.74 L1606.92,1556.69 L1609.75,1556 L1610.49,1556.21 L1613,1557.8 L1613.45,1558.7 L1614.33,1558.52 L1618.16,1556.01 L1619.47,1555.82 L1620.24,1556.18 L1622.39,1556.35 L1623.28,1556.57 L1625.04,1557.72 L1625.8,1557.73 L1625.85,1557.67 L1628.46,1558.14 L1629.36,1558 L1631.84,1556.06 L1633.86,1555.59 L1635.2,1553.87 L1636.04,1553.16 L1636.83,1552.98 L1639.29,1553.08 L1641.61,1552.58 L1642.45,1552.62 L1642.82,1552.76 L1644.59,1553.38 L1645.79,1554.01 L1646.59,1554.69 L1647.06,1555.61 L1647.26,1556.41 L1647.59,1557.13 L1648.48,1557.83 L1650.04,1558.42 L1653.63,1559.06 L1654.33,1559.48 L1654.99,1560.09 L1656.88,1562.32 L1658.14,1563.62 L1659.02,1564 L1660.3,1563.49 L1660.84,1563.27 L1663.82,1562.08 L1666.17,1560.42 L1666.17,1560.41 L1666.18,1560.4 L1667.56,1558.37 L1670.21,1550.89 L1671.72,1549.29 L1691.38,1542.77 L1693.84,1541.08 L1695.94,1540.33 L1697.06,1539.51 L1697.72,1539.23 L1699.71,1539 L1702.14,1537.63 L1705.13,1536.94 L1706.07,1536.72 L1707.81,1535.38 L1708.07,1534.67 L1708.52,1532.38 L1708.88,1531.61 L1709.75,1530.46 L1710.11,1529.77 L1710.46,1528.33 L1710.7,1526.27 L1710.71,1524.24 L1710.35,1522.86 L1710.21,1520.77 L1710.58,1520.84 L1711.34,1520.74 L1711.99,1520.37 L1713.19,1519.38 L1713.84,1519.04 L1715.73,1518.51 L1716.37,1518.19 L1721.23,1515.86 L1722.34,1514.98 L1724.07,1513.6 L1724.99,1513.35 L1726.1,1513.96 L1726.57,1513.95"
         id="path1009" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['RO'],polycolors['RO'],labels['RO'], x1, polybary['RO'], x2, polybary['RO'] )
    if 'RO' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1314.2,2205.23 L1313.81,2205.34 L1313.31,2205.34 L1311.39,2206.8 L1310.75,2207.68 L1310.47,2208.49 L1310.38,2209.4 L1311.17,2209.32 L1309.01,2213.41 L1307.89,2216.72 L1302.98,2223.83 L1298.95,2231.05 L1298.12,2233.07 L1297.34,2234.43 L1296.16,2235.52 L1294.88,2237.73 L1294.61,2238.6 L1294.74,2239.26 L1295.32,2240.39 L1295.45,2241.09 L1295.28,2241.72 L1294.63,2242.47 L1294.51,2243.08 L1294.47,2244.36 L1293.98,2246.22 L1293.76,2247.57 L1293.16,2249.88 L1290.93,2252.85 L1290.07,2254.63 L1289.88,2255.7 L1289.84,2256.82 L1290.33,2262.93 L1290.63,2263.96 L1291.02,2264.79 L1291.41,2265.26 L1295,2266.95 L1296.9,2267.18 L1297.9,2267.75 L1298.73,2268.75 L1299.29,2270.19 L1298.42,2269.48 L1297.97,2269.78 L1297.68,2270.52 L1297.28,2271.18 L1297.29,2270.1 L1296.97,2269.67 L1296.45,2269.71 L1295.84,2270.04 L1296.19,2270.79 L1295.48,2272.09 L1296.16,2273.97 L1297.34,2275.2 L1298.14,2274.55 L1298.89,2274.88 L1298.91,2275.31 L1298,2275.5 L1298.18,2276.41 L1298.99,2277.39 L1299.93,2277.81 L1301.15,2277.85 L1301.9,2278.24 L1302.19,2279.13 L1302.02,2280.64 L1301.26,2280.29 L1300.99,2280.81 L1301.12,2281.57 L1301.55,2281.94 L1302.57,2281.75 L1303.03,2281.95 L1303.5,2282.69 L1303.71,2283.68 L1303.34,2283.9 L1302.79,2283.69 L1302.51,2283.38 L1302.23,2283.5 L1300.52,2284.59 L1300.89,2284.99 L1300.46,2285.28 L1300.35,2285.43 L1300.59,2285.82 L1300.62,2286.28 L1299.43,2286.43 L1298.57,2286.94 L1297.84,2287.5 L1297.07,2287.79 L1296.02,2288.4 L1295.28,2289.68 L1293.1,2296.28 L1293.09,2297 L1292.49,2297.6 L1292.81,2298.82 L1293.85,2300.76 L1294.96,2304.37 L1294.98,2305.15 L1294.03,2306.16 L1293.01,2306.53 L1291.96,2306.34 L1290.99,2305.64 L1290.2,2304.69 L1289.79,2304.33 L1289.16,2304.03 L1288.45,2303.87 L1287.98,2303.93 L1287.79,2304.33 L1287.25,2304.81 L1285.94,2304.34 L1283.91,2303.1 L1282.03,2302.55 L1280.26,2302.93 L1277.01,2304.81 L1276.75,2304.31 L1276.6,2304.17 L1275.98,2304.45 L1275.1,2303.7 L1272.06,2303.46 L1271.34,2302.41 L1270.67,2301.82 L1267.63,2300.87 L1266.7,2300.35 L1265.88,2300.59 L1261.86,2299.89 L1261.24,2299.21 L1256.7,2290.79 L1255.37,2289.12 L1250.56,2285 L1246.91,2282.9 L1242.68,2281.2 L1240.77,2280.96 L1235.14,2281.72 L1234.69,2281.87 L1233.77,2282.5 L1233.27,2282.7 L1231.04,2282.52 L1227.83,2280.28 L1225.88,2279.61 L1223.7,2279.27 L1221.86,2278.6 L1220.19,2277.56 L1215.76,2273.6 L1214.23,2272.63 L1212.46,2272.17 L1210.14,2271.89 L1208.31,2271.33 L1206.71,2270.44 L1204.16,2268.58 L1201.97,2267.43 L1201.02,2266.57 L1200.54,2266.28 L1199.89,2266.1 L1199.32,2265.84 L1197.99,2263.54 L1197.68,2263.29 L1196.56,2262.72 L1195.96,2261.99 L1195.55,2261.65 L1194.97,2261.51 L1194.69,2261.33 L1194.17,2260.47 L1193.9,2260.29 L1187.73,2259.69 L1187,2260.08 L1186.48,2260.2 L1186,2259.95 L1185.66,2259.39 L1185.3,2258.15 L1185.07,2257.64 L1183.73,2256.33 L1182.13,2255.7 L1178.09,2255.37 L1174.03,2255.65 L1169.89,2256.51 L1167.97,2256.43 L1166.78,2255.41 L1164.6,2252.04 L1163.72,2251.19 L1163,2250.86 L1161,2250.89 L1160.12,2250.47 L1159.23,2249.44 L1158.46,2248.21 L1157.97,2247.18 L1157.71,2246.27 L1157.62,2245.46 L1157.46,2244.72 L1157.03,2244.02 L1156.42,2243.3 L1155.93,2242.46 L1155.93,2241.7 L1156.78,2241.25 L1156.86,2241.51 L1157.56,2241.04 L1157.78,2240.8 L1157.68,2240.62 L1157.99,2238.89 L1158.09,2238.84 L1158.33,2237.7 L1158.36,2237.16 L1158.25,2236.87 L1157.79,2236.19 L1157.63,2235.65 L1157.58,2235.22 L1157.59,2235.14 L1157.72,2235.01 L1158.81,2232.63 L1159.07,2230.97 L1159.36,2229.94 L1159.39,2229.07 L1158.81,2228.45 L1160.31,2227.78 L1161.57,2226.29 L1162.87,2225.2 L1164.48,2225.72 L1165.95,2224.61 L1166.57,2223.84 L1166.8,2222.85 L1167.06,2222.29 L1167.73,2222.29 L1168.53,2222.5 L1169.16,2222.56 L1170.16,2221.58 L1170.04,2220.07 L1169.79,2218.52 L1170.34,2217.43 L1170.67,2218.15 L1171.24,2218.06 L1171.81,2218.11 L1172.11,2219.23 L1173.42,2222.01 L1174.87,2224.39 L1175.41,2224.87 L1176.83,2225.52 L1177.44,2225.94 L1177.49,2226.49 L1179.09,2227.32 L1181.22,2226.95 L1185.28,2224.95 L1186.93,2223.58 L1187.32,2223.09 L1187.39,2222.82 L1187.29,2222.45 L1187.2,2221.68 L1187.08,2221.12 L1186.59,2220.54 L1186.46,2219.98 L1188.13,2218.41 L1187.83,2217.89 L1188.1,2217.34 L1188.65,2216.9 L1189.21,2216.71 L1189.71,2216.8 L1191.45,2217.43 L1192.57,2217.6 L1193.95,2217.55 L1195.07,2216.97 L1195.45,2215.57 L1196.23,2215.81 L1196.79,2215.62 L1197.33,2215.27 L1198.29,2214.99 L1198.95,2214.64 L1199.34,2214.55 L1199.6,2214.71 L1199.7,2215.1 L1199.77,2215.52 L1199.9,2215.8 L1201.82,2216.99 L1202.12,2217.4 L1202.27,2219.39 L1202.46,2219.96 L1203.26,2221.67 L1209.3,2220.94 L1210.72,2221.9 L1210.67,2222.97 L1210.77,2223.54 L1211.12,2224.23 L1211.64,2224.64 L1213.14,2225.19 L1214.95,2226.59 L1216.42,2227.41 L1218.76,2227.75 L1219.47,2228.07 L1219.28,2228.25 L1219.25,2228.32 L1219.15,2228.55 L1220.23,2228.47 L1223.43,2228.76 L1224.51,2228.44 L1226.67,2227.17 L1228.69,2226.59 L1229.36,2226.04 L1229.96,2225.43 L1230.61,2224.97 L1234,2224.21 L1234.63,2223.49 L1236.77,2224.57 L1237.82,2224.91 L1238.97,2224.99 L1239.69,2224.88 L1241.19,2224.41 L1247.28,2224.93 L1250.15,2224.59 L1252.76,2223.73 L1255.04,2222.36 L1256.68,2223.02 L1259.02,2222.44 L1264.99,2219.78 L1266.58,2218.58 L1267.9,2217.14 L1269.65,2214.76 L1270.18,2214.5 L1271.81,2214.53 L1272.83,2214.4 L1276.53,2213.21 L1277.2,2212.74 L1277.65,2212.5 L1278.12,2212.36 L1279.53,2212.38 L1280.02,2212.6 L1280.98,2213.68 L1281.33,2213.92 L1281.93,2213.94 L1283.22,2214.26 L1284.78,2214.06 L1285.28,2214.12 L1285.49,2214.32 L1286.05,2215.07 L1286.33,2215.33 L1287.29,2215.45 L1288.14,2215.07 L1288.96,2214.55 L1291.62,2213.42 L1292.8,2211.66 L1294.43,2207.95 L1294.25,2207.49 L1294.17,2207.36 L1294.2,2207.2 L1294.35,2206.69 L1294.17,2206.5 L1294.46,2206.48 L1294.66,2207.12 L1294.47,2208.63 L1294.75,2209.42 L1295.38,2209.81 L1296.22,2209.92 L1297.09,2209.89 L1298.52,2209.55 L1302.32,2207.94 L1303.38,2207.12 L1303.93,2206.54 L1305.8,2205.5 L1307.66,2203.51 L1308.42,2203.15 L1309.94,2203.19 L1312.82,2204.22 L1314.6,2204.39 L1314.47,2204.93 L1314.2,2205.23"
         id="path1011" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M998.798,2029.91 L999.205,2029.98 L999.723,2029.87 L1000.2,2030.68 L1000.15,2032.84 L1000.66,2033.27 L1000.49,2032.99 L1000.45,2032.64 L1000.51,2032.19 L1000.68,2031.61 L1001.01,2031.61 L1001.19,2032.43 L1001.14,2032.61 L1001,2032.89 L1000.99,2033.28 L1001.67,2033.19 L1001.95,2033.1 L1002.29,2032.9 L1002.48,2033.2 L1003.25,2033.76 L1003.37,2032.26 L1004.16,2032.54 L1005.86,2034.21 L1005.86,2033.78 L1006.26,2034.01 L1006.61,2034.15 L1007,2034.22 L1007.48,2034.22 L1007.48,2034.61 L1007.22,2034.8 L1007.09,2035 L1006.99,2035.22 L1006.82,2035.49 L1007.37,2035.73 L1007.67,2036.18 L1007.96,2036.71 L1008.43,2037.2 L1008.15,2037.66 L1008.14,2038.29 L1008.42,2039.75 L1008.96,2038.51 L1009.26,2037.98 L1009.72,2037.63 L1009.57,2037.35 L1009.49,2037.11 L1009.36,2036.92 L1009.08,2036.74 L1009.68,2036.38 L1010.27,2036.43 L1010.88,2036.63 L1011.53,2036.76 L1011.81,2036.57 L1012.02,2036.25 L1012.28,2036.18 L1012.67,2036.77 L1012.31,2037.22 L1012.77,2037.53 L1013.07,2037.86 L1013.33,2037.98 L1013.64,2037.66 L1013.96,2037.66 L1014.04,2037.93 L1014.07,2037.97 L1014.12,2037.96 L1014.29,2038.08 L1014.22,2038.84 L1014.08,2039.46 L1013.8,2040.04 L1013.3,2040.67 L1013.2,2040.37 L1013.06,2040.07 L1012.98,2039.78 L1012.11,2043.52 L1011.66,2044.48 L1012.63,2044.48 L1012.28,2043.6 L1013.15,2043.2 L1013.68,2043.35 L1013.84,2043.94 L1013.6,2044.91 L1013.84,2044.92 L1013.88,2044.81 L1013.86,2044.66 L1013.93,2044.49 L1014.22,2044.82 L1014.35,2044.91 L1014.57,2044.92 L1015.29,2044.22 L1016.6,2044.3 L1017.95,2044.8 L1018.82,2045.36 L1018.27,2046.08 L1017.55,2046 L1016.78,2045.59 L1016.05,2045.35 L1015.25,2045.77 L1015.01,2046.76 L1014.83,2047.87 L1014.23,2048.7 L1014.45,2049.55 L1013.8,2049.78 L1011.63,2049.57 L1011.6,2049.88 L1011.66,2049.97 L1011.26,2050.03 L1011.63,2050.76 L1012.01,2050.77 L1012.44,2050.51 L1012.92,2050.43 L1013.41,2050.65 L1014.26,2051.2 L1014.54,2051.33 L1015.47,2051.11 L1016.18,2050.62 L1016.89,2050.26 L1017.82,2050.46 L1017.59,2051.02 L1017.31,2051.44 L1016.95,2051.69 L1016.48,2051.76 L1016.48,2052.15 L1018.19,2053.02 L1018.46,2053.23 L1018.55,2053.98 L1018.81,2054.34 L1019.22,2054.31 L1019.76,2053.9 L1020.08,2054.49 L1020.53,2054.86 L1021.08,2055.06 L1021.7,2055.18 L1020.13,2055.71 L1019.65,2056.11 L1020.07,2056.91 L1020.07,2057.3 L1019.34,2057.1 L1018.73,2057.4 L1018.54,2057.97 L1019.06,2058.56 L1019.11,2057.87 L1019.39,2058.67 L1020.73,2060.49 L1021.03,2061.35 L1021.21,2062.49 L1022.65,2065.41 L1022.83,2066.23 L1023.05,2070.08 L1023.21,2070.71 L1023.45,2071.2 L1023.76,2071.39 L1024.42,2071.53 L1024.6,2071.91 L1024.65,2072.46 L1024.91,2073.09 L1026.26,2074.37 L1026.54,2074.79 L1026.55,2075.42 L1026,2076.48 L1025.85,2077.11 L1025.69,2078.45 L1025.28,2079.4 L1024.21,2081.15 L1023.42,2083.42 L1022.82,2084.25 L1021.77,2084.57 L1021.05,2084.92 L1020.49,2085.75 L1019.63,2087.54 L1019.01,2088.25 L1017.82,2089.28 L1017.29,2090.11 L1016.88,2091.32 L1016.81,2092.27 L1017,2094.9 L1017.1,2095.48 L1018.22,2098.69 L1018.72,2099.5 L1019.42,2099.84 L1019.81,2100.18 L1020.88,2101.7 L1021.94,2102.47 L1021.85,2103.43 L1020.05,2108.12 L1019.75,2110.09 L1020.19,2111.79 L1020.48,2111.58 L1020.83,2111.39 L1021.11,2111.4 L1021.21,2111.8 L1021.05,2112.37 L1020.75,2112.51 L1020.42,2112.52 L1020.19,2112.68 L1019.84,2113.48 L1019.59,2114.25 L1019.56,2114.97 L1019.85,2115.65 L1019.52,2115.64 L1019.52,2116.03 L1019.75,2116.16 L1019.95,2116.37 L1020.17,2116.5 L1020.17,2116.88 L1019.57,2117.62 L1019.15,2118.92 L1018.88,2120.48 L1018.81,2122 L1018.86,2122.79 L1019.1,2124.31 L1019.16,2125.18 L1019.05,2125.81 L1018.59,2126.84 L1018.48,2127.74 L1018.37,2128.12 L1018.15,2128.61 L1017.92,2129.25 L1017.81,2130.09 L1018.13,2132.85 L1017.99,2134.57 L1016.78,2138.6 L1017.57,2139.71 L1017.28,2141.67 L1016.05,2144.96 L1015.45,2147.67 L1015.37,2148.58 L1015.51,2149.49 L1015.9,2150.1 L1017.02,2150.93 L1017.02,2151.32 L1015.69,2151.8 L1014.44,2153.21 L1013.73,2154.95 L1014,2156.46 L1013.62,2157 L1013.78,2157.36 L1014.13,2157.7 L1014.32,2158.12 L1014.22,2158.6 L1013.75,2159.84 L1013.65,2160.47 L1013.22,2161.06 L1012.33,2161.4 L1011.5,2161.97 L1011.29,2163.23 L1010.91,2163.01 L1010.73,2162.8 L1010.76,2162.49 L1010.97,2161.99 L1010.21,2161.42 L1009.45,2161.29 L1008.7,2161.51 L1007.96,2161.97 L1007.59,2161.97 L1007.36,2161.41 L1006.92,2161.08 L1005.93,2160.61 L1004.71,2159.16 L1004.43,2158.94 L1003.95,2158.67 L1002.84,2157.46 L1002.44,2157.19 L1001.92,2157.06 L1001.25,2156.74 L1000.63,2156.32 L1000.27,2155.9 L999.934,2156.36 L999.249,2155.93 L998.334,2155.64 L996.57,2155.44 L995.644,2155.86 L994.828,2156.72 L994.271,2157.57 L994.088,2157.96 L992.669,2157.49 L991.539,2156.41 L989.584,2154.09 L989.127,2154.42 L988.729,2154.14 L988.346,2153.61 L987.921,2153.22 L987.224,2153.05 L986.71,2153.1 L986.64,2153.31 L987.281,2153.6 L987.275,2154.06 L987.039,2154.56 L987.322,2154.86 L987.725,2155.08 L987.852,2155.4 L987.897,2155.34 L988.447,2155.5 L989.667,2155.25 L989.902,2155.58 L989.896,2156.08 L989.85,2156.34 L987.854,2158.38 L987.239,2159.23 L986.839,2160.41 L986.708,2161.79 L986.829,2163.03 L987.103,2164.14 L987.453,2165.16 L987.732,2165.54 L988.023,2165.87 L988.074,2166.32 L987.294,2167.61 L986.73,2169.77 L987.093,2169.77 L986.57,2170.12 L986.42,2170.05 L986.06,2169.76 L985.644,2170.53 L980.595,2175.46 L980.262,2175.6 L979.746,2175.64 L979.495,2175.83 L978.94,2176.69 L978.541,2176.93 L977.964,2176.77 L977.304,2176.24 L976.216,2175.16 L976.198,2176.44 L975.508,2175.25 L974.324,2174.24 L973.004,2173.58 L971.88,2173.44 L970.795,2173.88 L969.515,2174.77 L968.653,2175.93 L968.805,2177.2 L968.103,2177.61 L967.806,2177.18 L968.22,2176.02 L967.75,2175.67 L966.888,2175.63 L966.12,2175.42 L966.127,2174.99 L966.888,2173.95 L966.937,2172.42 L966.356,2171.34 L965.214,2171.59 L964.887,2168.29 L964.31,2165.6 L963.944,2165.6 L963.555,2165.93 L962.418,2165.31 L961.616,2165.17 L961.848,2164.86 L962.307,2163.91 L961.646,2163.43 L960.575,2164.06 L960.548,2163.83 L960.332,2163.43 L960.144,2162.38 L959.755,2161.62 L959.232,2161.27 L958.635,2161.68 L958.44,2161.03 L958.5,2159.5 L958.356,2158.71 L958.022,2158.25 L957.558,2157.98 L957.254,2158.08 L957.393,2158.69 L957.095,2158.47 L956.882,2158.15 L956.759,2157.72 L956.722,2157.17 L956.603,2156.48 L956.309,2156.3 L955.963,2156.27 L955.705,2156.07 L955.364,2155.16 L955.323,2154.63 L955.574,2154.18 L958.055,2151.42 L958.354,2150.27 L957.578,2148.9 L957.672,2147.95 L955.982,2146.63 L955.613,2145.43 L955.826,2144.87 L956.698,2143.88 L956.985,2143.38 L957.181,2142.46 L957.131,2141.92 L956.466,2140.27 L956.299,2140.07 L956.348,2139.8 L956.734,2139.09 L958.096,2137.43 L958.769,2136.41 L959.602,2134.37 L960.193,2133.23 L959.837,2132.76 L960.24,2131.93 L960.179,2130.73 L959.623,2128.13 L959.533,2125.32 L959.324,2124.66 L959.332,2124.23 L959.548,2123.65 L959.46,2122.03 L959.565,2121.69 L960.225,2121.83 L960.649,2122.18 L962.115,2124.62 L962.932,2125.51 L963.959,2125.91 L965.299,2125.62 L962.827,2124.79 L962.346,2124.29 L962.483,2123.24 L963.133,2122.2 L963.907,2121.26 L964.393,2120.52 L964.07,2120.51 L963.897,2120.93 L963.666,2121.28 L963.389,2121.56 L963.054,2121.76 L963.633,2121.15 L964.183,2120.12 L964.593,2119.01 L964.765,2118.17 L964.949,2116.19 L964.882,2115.49 L964.497,2114.58 L963.464,2113.3 L962.063,2112.43 L960.563,2112.29 L959.208,2113.2 L959.531,2113.6 L959.699,2113.27 L959.912,2113.21 L960.195,2113.25 L960.561,2113.23 L959.996,2113.6 L959.639,2114.17 L959.16,2115.75 L958.846,2115.27 L959.284,2114.59 L958.979,2114.06 L958.374,2113.6 L957.675,2112.93 L957.538,2112.89 L957.422,2112.77 L957.25,2112.31 L957.212,2111.83 L957.353,2111.43 L957.534,2111.05 L957.613,2110.62 L957.279,2108.33 L957.34,2107.61 L957.943,2106.29 L957.986,2105.52 L957.398,2104.64 L956.696,2105.01 L956.559,2104.78 L956.922,2104.28 L957.744,2103.8 L957.893,2103.83 L959.358,2103.76 L960.212,2103.46 L960.566,2103.4 L961,2103.1 L961.516,2102.37 L961.933,2101.42 L962.08,2100.5 L961.86,2099.64 L961.032,2097.97 L960.865,2096.83 L960.746,2093.54 L961.605,2090.69 L961.623,2088.83 L961.016,2087.42 L959.996,2086.35 L958.785,2085.46 L958.417,2085.92 L957.367,2085.03 L957.274,2083.79 L957.864,2080.56 L957.692,2079.41 L957.217,2077.76 L956.559,2076.29 L955.82,2075.64 L954.961,2075.1 L954.613,2073.84 L954.437,2072.3 L954.104,2070.89 L953.145,2069.8 L951.988,2069.85 L949.517,2071.25 L948.554,2070.77 L949.028,2069.55 L949.573,2068.67 L948.483,2068.31 L947.666,2068.97 L947.141,2070.21 L946.899,2071.61 L946.502,2070.73 L946.247,2069.77 L945.964,2068.16 L945.973,2067.77 L946.8,2067.35 L947.582,2066.79 L948.151,2065.88 L948.335,2064.39 L948.46,2064.4 L948.68,2064.1 L948.777,2063.74 L948.529,2063.55 L947.9,2063.49 L947.659,2063.35 L947.385,2063.06 L946.711,2061.91 L946.363,2061.51 L945.792,2061.32 L949.596,2052.92 L949.888,2051.94 L949.959,2050.81 L949.666,2049.88 L948.838,2049.47 L948.218,2048.99 L948.404,2047.93 L948.94,2046.79 L949.389,2046.09 L949.744,2046.09 L949.868,2046.83 L950.218,2046.91 L950.648,2046.91 L950.978,2047.4 L950.936,2047.91 L950.734,2048.65 L950.605,2049.45 L950.775,2050.17 L952.843,2053 L954.366,2053.95 L959.669,2054.8 L962.321,2055.64 L963.421,2055.77 L964.354,2055.63 L968.092,2054.09 L969.055,2053.28 L969.831,2052.34 L970.164,2051.42 L973.433,2049.97 L974.611,2049.95 L975.433,2050.1 L976.056,2050.44 L978.85,2048.4 L979.675,2047.53 L981.414,2044.66 L982.02,2044.16 L981.926,2043.92 L981.927,2043.85 L981.897,2043.82 L981.701,2043.7 L983.228,2043.16 L983.687,2042.91 L984.014,2042.49 L984.534,2041.41 L984.83,2041.2 L985.426,2040.87 L986.243,2040.09 L986.965,2039.19 L987.686,2037.67 L988.602,2037.25 L991.844,2036.9 L992.545,2036.64 L993.879,2035.36 L994.171,2035.83 L994.926,2034.84 L995.213,2033.89 L995.212,2031.33 L995.612,2030.24 L996.511,2029.74 L997.577,2029.46 L998.444,2029.05 L998.535,2029.61 L998.798,2029.91"
         id="path1013" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1054.72,1932.48 L1055,1933.21 L1054.7,1934.26 L1054.25,1934.96 L1053.64,1935.43 L1052.81,1935.77 L1053.69,1936.16 L1054.49,1937.18 L1054.75,1938.34 L1054.06,1939.17 L1053.84,1938.6 L1053.49,1938.55 L1053.02,1938.59 L1052.47,1938.33 L1052.28,1937.99 L1051.81,1936.87 L1051.67,1936.63 L1051.07,1936.27 L1050.45,1935.7 L1049.86,1935.7 L1049.34,1937.07 L1048.91,1936.03 L1047.11,1936.98 L1046.18,1936.19 L1045.94,1937.15 L1045.72,1937.52 L1045.46,1937.55 L1041.65,1937.45 L1040.62,1937.09 L1039.78,1936.2 L1039.53,1934.54 L1040,1933.47 L1040.99,1932.8 L1042.15,1932.46 L1043.13,1932.37 L1043.61,1932.49 L1044.64,1933.04 L1045.2,1933.26 L1046.03,1933.33 L1047.12,1933.25 L1047.07,1932.94 L1047.05,1932.88 L1046.97,1932.89 L1046.8,1932.78 L1047.41,1932.36 L1047.4,1931.89 L1047.4,1931.82 L1048.2,1931.93 L1049.93,1931.88 L1049.71,1932.79 L1050.21,1933.14 L1050.88,1933.15 L1051.22,1933 L1051.37,1932.42 L1051.73,1932.08 L1052.14,1931.82 L1052.44,1931.48 L1052.93,1929.89 L1053.28,1929.21 L1053.86,1928.93 L1054.3,1929.08 L1054.72,1929.48 L1055.07,1930.03 L1055.3,1930.66 L1054.96,1931.11 L1054.74,1931.75 L1054.72,1932.48"
         id="path1015" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1125.73,1672.02 L1126.03,1673.46 L1126.33,1674.14 L1126.64,1674.73 L1126.65,1675.3 L1126.04,1675.94 L1126.5,1676.11 L1127.51,1676.26 L1128,1676.49 L1128.5,1676.97 L1128.75,1677.35 L1129.02,1677.65 L1129.61,1677.85 L1131.41,1677.86 L1132.11,1678.3 L1132.54,1679.66 L1132.58,1680.42 L1132.47,1680.88 L1132.32,1681.32 L1132.23,1682.04 L1132.24,1682.85 L1132.36,1683.27 L1132.61,1683.51 L1133.03,1683.78 L1133.94,1684.1 L1134.84,1684.2 L1135.56,1684.63 L1135.98,1685.99 L1136.86,1687.96 L1138.39,1689.23 L1140.16,1689.89 L1141.2,1690.03 L1142.51,1690.22 L1143.87,1691.08 L1144.59,1691.39 L1145.23,1691.43 L1147.72,1691.02 L1149.85,1691.16 L1149.93,1691.19 L1150.27,1691.33 L1151.08,1691.89 L1151.48,1692.05 L1151.89,1692.02 L1152.57,1691.73 L1152.9,1691.67 L1153.28,1691.78 L1154.37,1692.11 L1156.88,1693.61 L1166.99,1693.93 L1170.56,1694.6 L1173.37,1696.16 L1174.29,1696.49 L1176.03,1696.5 L1180.39,1695.37 L1182.29,1695.61 L1184.91,1695.3 L1185.21,1695.4 L1185.85,1696.06 L1186.19,1696.28 L1188.03,1696.26 L1193.33,1697.7 L1193.98,1697.73 L1194.65,1697.57 L1194.68,1698.06 L1194.62,1698.5 L1194.47,1698.89 L1194.25,1699.24 L1194.26,1700.84 L1194.16,1701.07 L1193.84,1701.84 L1193.05,1702.31 L1191.98,1702.33 L1190.53,1702.59 L1189.51,1703.64 L1188.65,1704.97 L1187.65,1706.12 L1185.69,1707.29 L1184.69,1708.09 L1184.17,1708.38 L1183.6,1708.47 L1183.15,1709.05 L1182.63,1710.4 L1182.06,1711.49 L1181.89,1711.82 L1180.79,1712.57 L1181.15,1713.18 L1181.43,1713.91 L1181.76,1715.46 L1181.76,1715.47 L1181.77,1715.48 L1182.38,1716.24 L1182.57,1717.09 L1182.96,1717.62 L1184.14,1717.38 L1183.45,1716.31 L1184.14,1716.51 L1185.44,1716.54 L1186.06,1716.85 L1187.28,1717.04 L1188.13,1717.57 L1189.49,1718.8 L1190.6,1718.92 L1191.88,1718.72 L1192.46,1718.81 L1192.9,1718.87 L1193.09,1719.42 L1193.29,1720.02 L1192.15,1722.37 L1188.22,1725.67 L1187.48,1726.28 L1186.55,1727.65 L1186.92,1728 L1187.04,1728.4 L1186.94,1728.84 L1186.62,1729.3 L1186.4,1729.46 L1185.76,1730.09 L1186.36,1730.73 L1186.57,1730.88 L1186.61,1730.93 L1186.65,1730.99 L1186.68,1731.05 L1186.7,1731.12 L1187.95,1732.42 L1189.25,1732.26 L1190.63,1731.52 L1191.88,1731.16 L1192.1,1731.1 L1192.89,1732.25 L1192.63,1733.54 L1192.38,1734.77 L1192.03,1735.67 L1190.85,1738.71 L1190.81,1740.86 L1191.2,1741.54 L1191.54,1742.14 L1192.79,1742.74 L1194.3,1742.85 L1195.04,1743.04 L1196.84,1744.3 L1197.27,1744.6 L1200.37,1745.82 L1204.14,1751.5 L1204.64,1751.99 L1205.32,1752.33 L1205.71,1752.34 L1205.74,1752.39 L1205.79,1752.51 L1205.49,1753.34 L1204.71,1754.42 L1203.88,1755.56 L1201.81,1755.88 L1200.04,1755.03 L1197.88,1755.33 L1198.17,1754.85 L1198.63,1754.05 L1199.81,1753.95 L1201.12,1754.05 L1202.05,1753.8 L1201.49,1753.66 L1201.07,1753.22 L1200.62,1752.55 L1198.92,1751.78 L1199.71,1751.38 L1199.48,1750.31 L1198.79,1749.14 L1198.19,1748.4 L1194.56,1744.94 L1193.73,1744.32 L1191.28,1743.38 L1190.27,1743.25 L1189.57,1743.93 L1189.18,1745.32 L1189.42,1746.53 L1190.59,1746.69 L1189.03,1747.53 L1188.82,1747.8 L1188.59,1748.2 L1187.39,1748.5 L1186.92,1748.76 L1186.19,1749.61 L1185.19,1750.44 L1184.05,1750.85 L1182.94,1750.42 L1183.55,1750.23 L1184.34,1749.85 L1185,1749.36 L1185.27,1748.83 L1184.83,1747.78 L1183.03,1747.49 L1182.45,1746.6 L1181.69,1746.91 L1177.41,1746.11 L1177,1745.95 L1175.91,1744.93 L1175.62,1744.77 L1175.16,1744.85 L1173.63,1745.48 L1173.16,1746.39 L1172.81,1746.59 L1172.37,1746.4 L1171.97,1745.54 L1171.43,1745.37 L1171.07,1745.66 L1170.98,1746.25 L1171.16,1746.83 L1171.66,1747.07 L1171.68,1747.53 L1170.7,1747.64 L1169.85,1748.21 L1169.4,1749.1 L1169.68,1750.14 L1170.42,1750.59 L1171.3,1750.42 L1172.93,1749.57 L1172.46,1750.49 L1170.98,1751.82 L1170.69,1752.88 L1170.69,1752.89 L1170.25,1753.55 L1169.19,1753.92 L1165.74,1754.17 L1164.09,1754.63 L1157.1,1759.59 L1149.79,1763.43 L1145.49,1765.19 L1144.77,1765.84 L1144.42,1766.08 L1142.83,1766.53 L1141.64,1767.34 L1141.05,1767.47 L1141.37,1766.53 L1141.83,1765.71 L1142.1,1765.02 L1141.87,1764.46 L1142.29,1764.51 L1142.59,1764.42 L1142.74,1764.12 L1142.74,1763.54 L1143.04,1763.53 L1143.26,1764.06 L1143.45,1764.31 L1143.67,1764.28 L1143.95,1763.97 L1143.33,1763.14 L1143.83,1763.18 L1144.3,1763.13 L1144.74,1762.95 L1144.83,1763.09 L1144.86,1763 L1145.01,1763.18 L1145.15,1763.41 L1145.17,1763.46 L1145.02,1763.49 L1144.57,1763.48 L1144.58,1763.95 L1145.55,1763.79 L1145.77,1763.02 L1145.78,1762.15 L1146.16,1761.72 L1146.93,1761.61 L1147.56,1761.3 L1148.03,1760.76 L1148.36,1759.94 L1147.84,1760.18 L1146.74,1761.02 L1146.44,1761.07 L1145.58,1760.22 L1145.22,1759.66 L1145.02,1758.78 L1144.09,1759.33 L1142.95,1760.41 L1142.26,1761.56 L1142.7,1762.3 L1142.71,1762.69 L1141.99,1762.69 L1141.21,1762.01 L1140.89,1762.36 L1140.6,1762.37 L1140.26,1761.06 L1139.73,1762.09 L1140.03,1762.65 L1141.52,1763.2 L1141.53,1763.58 L1141,1763.91 L1140.2,1764.04 L1139.63,1763.86 L1139.72,1763.25 L1138.29,1763.27 L1136.52,1764.32 L1134.94,1765.95 L1134.11,1767.69 L1134.12,1768.08 L1134.4,1768.07 L1134.02,1768.66 L1133.86,1769.25 L1133.99,1769.8 L1134.46,1770.24 L1134.05,1770.86 L1133.98,1771.31 L1134.02,1771.76 L1133.92,1772.39 L1133.59,1772.61 L1133.17,1772.63 L1132.98,1772.86 L1133.38,1773.68 L1132.9,1774.54 L1132.72,1775.22 L1132.4,1775.68 L1131.46,1775.87 L1130.88,1775.63 L1130.36,1775.29 L1129.99,1775.48 L1129.79,1776.81 L1130.28,1777.36 L1130.72,1778.66 L1131.25,1778.9 L1131.72,1778.58 L1132.05,1777.84 L1132.13,1776.98 L1131.92,1776.28 L1132.8,1776.62 L1133,1777.27 L1132.99,1778.18 L1133.21,1779.27 L1132.73,1779.47 L1132.62,1779.8 L1132.81,1780.18 L1133.25,1780.55 L1133.12,1781.45 L1133.54,1782.19 L1134.3,1782.62 L1135.14,1782.62 L1135.01,1782.39 L1134.97,1782.36 L1134.81,1782.24 L1135.44,1781.72 L1136.4,1780.7 L1136.89,1780.44 L1136.51,1781.4 L1136.8,1782.31 L1137.32,1783.18 L1137.61,1784.06 L1137.71,1786.15 L1138.07,1788.49 L1137.62,1788.36 L1137.12,1788.1 L1137.16,1787.69 L1137.09,1787.08 L1137.08,1786.82 L1136.68,1788.21 L1136.69,1789.32 L1137.14,1789.93 L1138.11,1789.81 L1137.96,1789.58 L1137.88,1789.39 L1137.75,1788.97 L1138.65,1789.7 L1139.22,1790.58 L1139.7,1791.49 L1140.3,1792.3 L1140.32,1792.72 L1139.97,1792.8 L1139.78,1792.91 L1139.64,1793.06 L1139.4,1793.22 L1140.03,1794.05 L1140.65,1793.07 L1140.76,1792.3 L1140.48,1791.55 L1139.92,1790.65 L1140.58,1791.2 L1141.4,1791.58 L1141.95,1792 L1141.83,1792.67 L1142.56,1793.41 L1142.92,1793.68 L1143.39,1793.95 L1144.32,1795.1 L1145.3,1796.03 L1145.87,1795.57 L1145.72,1795.56 L1145.42,1795.61 L1145.29,1795.59 L1145.27,1795.12 L1146,1795.08 L1146.64,1795.3 L1147.14,1795.74 L1147.43,1796.37 L1147.41,1796.94 L1147,1797.63 L1147.18,1798.04 L1146.84,1798.65 L1146.7,1798.99 L1146.59,1799.37 L1146.32,1799.38 L1145.98,1799.45 L1145.82,1800.14 L1145.8,1802.16 L1145.58,1803.16 L1145.03,1804.2 L1143.76,1805.82 L1143.69,1805.54 L1143.47,1804.93 L1143.16,1804.25 L1142.78,1803.72 L1143.41,1802.27 L1143.56,1801.49 L1143.32,1800.81 L1142.81,1800.58 L1142.31,1800.92 L1141.95,1801.46 L1141.81,1801.88 L1141.33,1802.91 L1141.33,1803.51 L1142.45,1803.95 L1142.44,1804.43 L1142.11,1804.89 L1141.6,1805.08 L1141.62,1805.51 L1141.93,1805.58 L1142.5,1805.81 L1142.84,1805.85 L1142.53,1806.94 L1141.73,1807.56 L1140.74,1807.69 L1139.91,1806.97 L1139.42,1806.29 L1138.23,1805.19 L1137.41,1804.63 L1136.94,1804.62 L1135.81,1805.69 L1135.82,1806.07 L1136.43,1806.06 L1135.38,1810.41 L1135.14,1812.89 L1135.49,1815.58 L1136.13,1817.57 L1136.56,1818.53 L1136.96,1818.94 L1137.08,1819.63 L1136.95,1824.05 L1137.2,1826.81 L1138.75,1833.02 L1138.97,1834.64 L1139.12,1835.1 L1139.32,1835.46 L1139.97,1836.29 L1140.11,1836.58 L1140.24,1837.9 L1140.55,1839.15 L1141.45,1841.46 L1142.46,1843.17 L1143.73,1844.83 L1145.63,1846.58 L1146.62,1848.24 L1148.11,1849.73 L1149.8,1850.99 L1151.03,1851.44 L1153.72,1854.81 L1155.1,1856.12 L1156.52,1857.17 L1157.38,1857.62 L1159.37,1858.21 L1159.48,1858.23 L1161.65,1858.47 L1162.53,1858.95 L1164.35,1860.33 L1167.42,1861.43 L1167.88,1862.11 L1168.29,1862.53 L1182.63,1874.07 L1188.07,1877.12 L1188.98,1877.44 L1189.45,1877.98 L1191.62,1878.63 L1192.87,1879.3 L1193.3,1879.01 L1193.63,1878.54 L1193.87,1878.29 L1193.99,1878.22 L1194.08,1878.08 L1194.22,1877.94 L1194.49,1877.87 L1194.76,1877.98 L1194.9,1878.26 L1195,1878.53 L1195.15,1878.65 L1196.04,1879.08 L1196.75,1880.07 L1197.31,1881.07 L1197.77,1881.51 L1198.65,1881.68 L1199.48,1882.2 L1199.92,1883.01 L1199.62,1884.02 L1206.93,1900.35 L1207.71,1901.57 L1207.93,1902 L1208.32,1904.22 L1208.54,1904.74 L1209.13,1905.58 L1210.75,1908.67 L1211.49,1910.85 L1212.51,1916.07 L1214.67,1922.55 L1216.79,1928.89 L1218.65,1933.02 L1219.7,1934.96 L1222.79,1939.39 L1223.1,1940.32 L1223.44,1940.97 L1230.57,1948.6 L1233.65,1951.04 L1237.11,1953.05 L1238.62,1954.21 L1240.73,1957.28 L1247.16,1962.89 L1248.87,1963.85 L1251.61,1964.77 L1253.5,1965.01 L1254.26,1965.58 L1254.68,1966.52 L1255.1,1969.06 L1255.48,1970.27 L1256.06,1971.14 L1256.86,1971.45 L1257.14,1971.48 L1257.93,1971.6 L1258.88,1972.1 L1260.62,1973.37 L1262.33,1974.03 L1266.14,1974.45 L1268.13,1974.97 L1268.8,1975.51 L1270.17,1977.08 L1270.69,1977.4 L1271.55,1977.63 L1273.56,1978.9 L1274.48,1979.31 L1275.43,1979.44 L1276.46,1979.58 L1281.89,1979.19 L1287.52,1980.05 L1289.65,1979.95 L1297.28,1978.07 L1304.92,1977.92 L1316.6,1975.58 L1320.58,1976.26 L1320.64,1976.36 L1322,1976.65 L1323.08,1978.27 L1323.94,1978.16 L1323.86,1979.16 L1324.31,1980.3 L1324.88,1982.57 L1324.98,1983.37 L1325.06,1984.46 L1324.71,1985.19 L1323.96,1986.27 L1323.05,1987.25 L1322.24,1987.72 L1321.3,1988.1 L1320.54,1988.88 L1319.41,1990.51 L1317.84,1991.98 L1313.23,1994.83 L1312.16,1996.5 L1311.93,1998.88 L1312.34,2001.37 L1313.22,2003.4 L1314.35,2004.77 L1315.73,2005.91 L1318.72,2007.65 L1321.78,2008.35 L1322.54,2008.86 L1323.23,2009.46 L1332.36,2012.77 L1340.16,2016.19 L1343.06,2016.72 L1343.96,2017.31 L1344.77,2017.99 L1346.73,2019.09 L1358.21,2021.86 L1359.26,2022.6 L1359.58,2022.57 L1359.54,2022.11 L1359.84,2022.09 L1360.9,2023.01 L1369.24,2024.87 L1376.47,2028 L1380.15,2030.19 L1381.89,2031.61 L1383.92,2034.08 L1384.58,2034.37 L1385.38,2034.55 L1387.2,2035.7 L1390.45,2038.56 L1391.23,2038.84 L1392.21,2039 L1394.03,2039.97 L1398.21,2040.61 L1398.75,2040.88 L1403.83,2042.37 L1408.68,2045.22 L1412.44,2045.52 L1414.13,2046.13 L1414.23,2047.59 L1414.77,2047.54 L1415.95,2047.02 L1416.68,2047.09 L1416.91,2047.48 L1416.98,2048.03 L1417.25,2048.63 L1418.47,2049.56 L1418.69,2049.92 L1418.69,2052.74 L1422.93,2055.56 L1427.54,2057.56 L1428.51,2057.66 L1428.86,2057.86 L1430,2059.42 L1430.6,2059.95 L1432.31,2061.09 L1433.75,2063.01 L1434.54,2063.87 L1435.43,2064.17 L1435.9,2064.53 L1439.53,2068.55 L1443.26,2075.35 L1443.5,2075.71 L1443.78,2075.96 L1444.11,2076.09 L1443.95,2076.83 L1444.12,2077.26 L1444.34,2077.63 L1444.34,2078.24 L1444.01,2078.89 L1443.3,2079.59 L1443.09,2081.46 L1442.53,2082.26 L1440.93,2083.74 L1440.11,2086.57 L1440.01,2087.23 L1439.96,2088.55 L1439.65,2089.95 L1439.6,2090.55 L1440.11,2092.1 L1440.2,2092.83 L1439.88,2096.19 L1440.06,2096.69 L1439.83,2096.91 L1439.42,2097.84 L1439.18,2098.07 L1438.13,2098.09 L1437.72,2098.01 L1437.32,2097.85 L1434.81,2096.26 L1433.99,2096.08 L1432.01,2096.29 L1431.37,2096.25 L1430.1,2095.61 L1428.86,2095.2 L1427.22,2093.94 L1425.36,2092.94 L1424.7,2092.78 L1424.22,2092.51 L1420.69,2088.23 L1420.06,2087.72 L1420.01,2087.26 L1420.76,2087.14 L1421.16,2086.67 L1421.18,2085.95 L1420.79,2085.08 L1420.23,2084.6 L1418.9,2084.31 L1418.35,2084.02 L1419.95,2082.78 L1420.2,2081.4 L1419.4,2079.96 L1417.82,2078.53 L1416.44,2077.61 L1415.66,2076.87 L1415.26,2076 L1415.13,2074.82 L1414.92,2073.78 L1414.56,2072.98 L1413.96,2072.49 L1414.59,2072.43 L1414.55,2072.04 L1412,2070.41 L1411.24,2070.22 L1395.13,2071 L1394.24,2070.85 L1391.55,2069.65 L1391.07,2069.66 L1390.08,2069.85 L1389.59,2069.84 L1389.22,2069.62 L1388.31,2068.87 L1387.99,2068.72 L1387.16,2068.59 L1384.4,2067.32 L1381.63,2066.59 L1380.27,2066.04 L1379.59,2065.22 L1380.66,2064.62 L1381.52,2063.73 L1381.66,2062.72 L1380.6,2061.75 L1381.52,2061.3 L1383.86,2061.51 L1384.79,2060.88 L1385.03,2059.82 L1384.11,2059.85 L1382.15,2060.71 L1382.23,2060.56 L1382.44,2060.22 L1381.39,2059.9 L1380.55,2060.35 L1379.73,2061.05 L1378.77,2061.46 L1378.36,2061.28 L1378.06,2060.82 L1377.79,2060.28 L1377.49,2059.88 L1376.99,2059.72 L1375.91,2059.81 L1375.47,2059.68 L1371.88,2059.78 L1368.52,2061.75 L1365.71,2064.97 L1363.66,2068.54 L1360.27,2074.43 L1359.41,2076.5 L1358.57,2080.23 L1358.29,2080.9 L1357.62,2081.4 L1356.11,2084.5 L1354.35,2086.36 L1353.66,2087.09 L1352.83,2088.63 L1352.64,2091 L1352.82,2091.77 L1353.69,2094.03 L1354.7,2095.84 L1354.29,2096.67 L1353.56,2097.4 L1352.89,2098.9 L1350.8,2101.22 L1349.39,2104.33 L1348.75,2106.38 L1348.81,2108.27 L1349.54,2109.96 L1350.94,2111.43 L1350.97,2111.86 L1350.47,2113.46 L1352.07,2115.12 L1354.39,2116.42 L1356.06,2116.99 L1362.13,2116.78 L1363.75,2117.16 L1364.55,2117.6 L1365.48,2118.29 L1366.27,2119.15 L1366.66,2120.09 L1367.13,2120.53 L1370.14,2121.69 L1372.57,2123.37 L1373.46,2123.56 L1374.66,2123.59 L1375.47,2123.92 L1376.11,2124.53 L1376.79,2125.38 L1377.92,2126.47 L1379.54,2127.51 L1381.34,2128.19 L1383.03,2128.2 L1381.62,2132.57 L1381.56,2136.04 L1381.61,2136.87 L1381.83,2137.68 L1382.23,2138.36 L1383.19,2139.66 L1383.43,2140.36 L1383.34,2142.35 L1382.76,2146.01 L1383.03,2147.87 L1384.37,2150.03 L1384.6,2150.53 L1384.95,2150.83 L1386.66,2151.26 L1387.35,2151.29 L1386.45,2152.03 L1385.99,2152.5 L1385.82,2152.95 L1385.98,2155.38 L1386.05,2155.71 L1384.54,2157.34 L1384.38,2157.61 L1384.04,2157.88 L1383.66,2158.49 L1383.04,2159.83 L1382.49,2158.63 L1381.28,2158.51 L1379.93,2159.05 L1378.99,2159.77 L1377.62,2158.43 L1376.85,2158.02 L1375.66,2157.95 L1374.58,2158.14 L1369.97,2159.83 L1364.75,2162.75 L1362.91,2164.35 L1362.54,2164.9 L1362.15,2165.36 L1359.33,2167.11 L1358.19,2168.37 L1357.71,2169.21 L1357.32,2171.01 L1356.77,2172.01 L1356.26,2173.18 L1356.18,2174.65 L1356.8,2175.01 L1356.92,2175.2 L1356.98,2176.31 L1357.86,2180.97 L1359.07,2185.07 L1359.2,2186.66 L1359.18,2191.22 L1358.84,2192.08 L1357.04,2194.05 L1356.44,2195.07 L1355.93,2195.25 L1354.52,2196.65 L1352.98,2197.33 L1348.47,2200.14 L1346.92,2201.75 L1340.77,2210.58 L1340.05,2212.65 L1339.21,2219.31 L1338.63,2221.3 L1338.21,2222.15 L1337.7,2222.89 L1337.1,2223.52 L1336.44,2224.02 L1334.13,2224.7 L1333.42,2225.11 L1330.43,2224.47 L1321.5,2225.64 L1320.15,2225.39 L1318.79,2224.87 L1317.56,2224.18 L1316.58,2223.42 L1315.48,2222.17 L1314.88,2220.92 L1314.97,2219.76 L1315.9,2218.77 L1314.49,2216.9 L1313.99,2215.83 L1314.1,2214.87 L1314.72,2213.67 L1314.96,2212.58 L1314.82,2211.42 L1314.3,2209.97 L1313.9,2207.87 L1314.69,2206.52 L1316.31,2205.65 L1321.25,2203.72 L1321.67,2203.22 L1321.8,2202.89 L1322.46,2202.23 L1322.6,2201.88 L1323.08,2199.05 L1326.25,2191.15 L1326.71,2188.45 L1326.47,2186.38 L1325.57,2184.72 L1324.01,2183.22 L1323.64,2183 L1322.95,2182.74 L1322.64,2182.48 L1322.18,2181.87 L1322.06,2181.53 L1322.12,2181.19 L1322.14,2180.58 L1322.48,2179.84 L1323.36,2179.29 L1325.36,2178.44 L1328.32,2176.14 L1329.96,2175.22 L1332.22,2174.89 L1335.78,2175.03 L1336.51,2174.79 L1337.29,2174.29 L1338.52,2173.11 L1339.45,2171.28 L1339.97,2168.89 L1340.1,2166.24 L1339.83,2163.61 L1339.48,2162.4 L1338.91,2161.51 L1338.14,2160.98 L1337.11,2160.86 L1336.36,2160.41 L1335.56,2159.31 L1332.34,2153.25 L1329.28,2138.83 L1328.74,2136.72 L1326.45,2131.07 L1324.47,2128.81 L1323.1,2126.67 L1321.63,2125.4 L1320.17,2124.55 L1319.7,2123.81 L1319.21,2122.3 L1318.64,2120.72 L1317.81,2118.42 L1316.69,2116.87 L1316.17,2115.88 L1314.88,2109.68 L1314.91,2110.07 L1314.59,2108.87 L1314.3,2108.26 L1313.92,2108.01 L1313.82,2107.67 L1314.24,2105.24 L1313.78,2103.88 L1312.03,2101.61 L1311.61,2100.76 L1311.23,2099.52 L1310.44,2098.53 L1308.68,2096.92 L1307.22,2094.87 L1306.14,2094.07 L1305.83,2093.64 L1305.76,2092.85 L1305.1,2093.18 L1301.46,2093.16 L1300.46,2093.87 L1299.69,2095.23 L1298.05,2096.43 L1296.72,2098.36 L1295.8,2098.79 L1293.42,2098.66 L1292.54,2098.39 L1291.03,2096.65 L1290.47,2096.54 L1290.03,2096.81 L1289.54,2097 L1289.03,2097.1 L1288.5,2097.1 L1288.47,2096.68 L1289.29,2095.91 L1288.43,2094.34 L1286.9,2092.81 L1285.69,2092.16 L1284.72,2091.77 L1281.51,2088.84 L1280.8,2088.57 L1279.99,2088.73 L1278.56,2089.23 L1277.67,2089.15 L1277.05,2088.79 L1276.4,2087.57 L1275.4,2086.72 L1274.06,2086.25 L1273.58,2085.95 L1273.11,2085.55 L1272.21,2085.84 L1271.41,2085.25 L1270.86,2085.06 L1270.6,2084.02 L1271.7,2082.28 L1271.97,2080.92 L1271.58,2079.87 L1271.54,2079.28 L1272,2079 L1272.55,2078.83 L1273.03,2078.48 L1273.42,2078.02 L1274.43,2077.56 L1274.67,2076.89 L1274.46,2075.12 L1273.26,2073.08 L1268.58,2064.57 L1267.15,2062.54 L1265.81,2061.04 L1264.51,2059.85 L1263.13,2058.89 L1261.68,2058.55 L1260.18,2059.26 L1258.84,2060.45 L1258.26,2060.64 L1255.75,2060.64 L1255.17,2060.87 L1253.26,2062.74 L1251.99,2063.08 L1250.64,2062.77 L1249.17,2062.06 L1248.47,2062.6 L1246.15,2063.3 L1245.22,2064.19 L1243.09,2065.47 L1242.51,2066.31 L1241.85,2065.49 L1241.69,2064.19 L1242.1,2062.98 L1243.11,2062.41 L1244.07,2062.28 L1244.42,2062.07 L1244.92,2061.61 L1245.19,2060.84 L1246.97,2058.87 L1248.47,2058.21 L1248.58,2057.18 L1248.25,2056.07 L1247.56,2055.21 L1246.66,2054.58 L1245.66,2054.9 L1244.89,2054.57 L1244.35,2054.31 L1243.75,2053.68 L1242.96,2053.23 L1242.06,2052.25 L1240.22,2050.41 L1239.32,2049.79 L1238.44,2049.76 L1237.15,2049.98 L1235.99,2050.44 L1235.51,2051.13 L1235.13,2052.37 L1234.12,2052.44 L1233.28,2051.56 L1231.83,2051.47 L1230.66,2050.85 L1229.91,2050.62 L1229.28,2050.81 L1229.01,2051.39 L1229.1,2052.13 L1229.32,2052.94 L1229.43,2053.78 L1229.07,2053.63 L1227.75,2053.49 L1227.42,2053.08 L1227.82,2050.5 L1227.34,2048.25 L1226.09,2045.35 L1223.21,2040.86 L1221.49,2039.84 L1220.88,2039.28 L1220.69,2038.7 L1220.54,2037.38 L1220.41,2036.75 L1219.05,2034.32 L1217.38,2031.76 L1214.31,2028.27 L1210.93,2025.54 L1210.29,2025.32 L1208.9,2025.75 L1208.17,2025.78 L1207.83,2025.24 L1204.91,2025.78 L1203.49,2026.79 L1204.13,2028.63 L1202.65,2028.79 L1202.19,2028.72 L1201.74,2028.28 L1201.44,2027.69 L1201.19,2027.42 L1200.86,2027.93 L1200.51,2027.94 L1199.55,2026.9 L1192.04,2023.78 L1190.21,2023.8 L1186.41,2024.71 L1185.59,2025.1 L1183.94,2026.28 L1182.22,2026.93 L1181.14,2028.49 L1180.1,2028.83 L1178.97,2028.52 L1178.57,2027.73 L1178.38,2026.71 L1177.88,2025.69 L1177.29,2024.91 L1176.31,2023.14 L1175.66,2022.21 L1172.92,2019.28 L1171.45,2018.12 L1169.93,2017.29 L1168.95,2017.02 L1165.24,2016.98 L1164.55,2016.85 L1161.18,2014.84 L1160.01,2014.41 L1159.48,2014.89 L1159.16,2015.57 L1158.39,2014.87 L1154.75,2009.7 L1149.99,2004.47 L1144.92,2000.17 L1144.17,1999.63 L1142.91,1999.29 L1142.16,1998.71 L1141.76,1998.58 L1140.36,1998.36 L1139.99,1998.21 L1139.23,1997.33 L1138.81,1996.26 L1138.62,1995.06 L1138.56,1993.79 L1138.29,1992.59 L1137.71,1991.36 L1136.52,1989.39 L1135.41,1987.93 L1130.76,1984.53 L1130.23,1984.24 L1129.8,1983.77 L1129.6,1982.97 L1129.32,1982.52 L1127.99,1982.36 L1127.53,1982.15 L1126.9,1981.44 L1126.11,1980.79 L1125.26,1980.24 L1124.42,1979.86 L1120.82,1980.04 L1120.25,1980.19 L1119.98,1979.42 L1119.36,1978.21 L1118.6,1977.1 L1117.94,1976.62 L1117.56,1975.77 L1116.51,1974.45 L1116.29,1973.91 L1116.14,1972.94 L1115.8,1971.91 L1113.18,1966.79 L1112.03,1965.12 L1106.4,1960.91 L1104.07,1960.3 L1100.86,1958.67 L1099.71,1958.29 L1099.7,1957.83 L1099.27,1957.6 L1098.79,1957.44 L1098.28,1957.39 L1097.77,1957.44 L1098.12,1957.86 L1096.77,1957.87 L1093.32,1956.67 L1092.69,1956.59 L1091.84,1956.61 L1091.1,1956.83 L1090.79,1957.33 L1090.64,1958.29 L1089.98,1959.31 L1089.87,1960.12 L1089.44,1959.94 L1088.98,1959.95 L1088.64,1960.05 L1088.58,1960.14 L1088.19,1959.83 L1087.57,1959.04 L1086.26,1958.62 L1085.69,1957.92 L1085.32,1956.95 L1085.32,1956.26 L1085.87,1955.42 L1086.51,1955.19 L1087.86,1955.64 L1088.89,1955.56 L1089.6,1954.51 L1089.98,1953.02 L1089.99,1951.5 L1089.87,1950.5 L1088.88,1948.93 L1088.52,1947.83 L1088.06,1947.89 L1087.54,1948.16 L1087.14,1948.22 L1086.54,1947.37 L1086.12,1946.2 L1085.65,1945.2 L1084.88,1944.85 L1084.72,1943.64 L1083.59,1942.73 L1081.36,1941.46 L1081.1,1940.72 L1080.87,1939.72 L1080.6,1938.83 L1080.19,1938.47 L1079.78,1938.24 L1078.43,1936.83 L1075.98,1935.55 L1068.59,1933.11 L1069.7,1932.18 L1070.13,1931.17 L1070.18,1929.99 L1070.14,1928.57 L1070.27,1928.14 L1070.54,1927.87 L1070.68,1927.46 L1070.45,1926.63 L1070.01,1926 L1069.42,1925.57 L1066.22,1924.12 L1063.96,1923.55 L1061.69,1923.5 L1059.66,1924.17 L1059.87,1924.81 L1060.02,1925.06 L1058.73,1925.66 L1057.85,1924.72 L1056.82,1921.64 L1057.97,1920.67 L1058.62,1919.52 L1059.61,1916.05 L1059.45,1915.31 L1059.44,1913.99 L1059.6,1912.64 L1059.92,1911.83 L1059.54,1910.77 L1059.35,1909.46 L1059.26,1906.91 L1059.04,1905.63 L1058.52,1904.39 L1054.82,1896.82 L1051.89,1892.81 L1051.24,1892.06 L1049.74,1891.35 L1049.32,1890.37 L1048.28,1885.64 L1046.83,1870.63 L1046.26,1868.31 L1044.36,1863.73 L1043.03,1861.55 L1041.02,1860.09 L1039.63,1857.72 L1038.95,1857.19 L1038.11,1856.93 L1036.45,1855.79 L1035.52,1855.53 L1034.79,1855.15 L1034.33,1855.05 L1034.14,1855.31 L1033.98,1855.66 L1033.62,1855.92 L1033.17,1855.99 L1032.76,1855.74 L1032.04,1855.01 L1030.53,1853.79 L1029.79,1852.97 L1029.29,1853.29 L1027.85,1851.99 L1027.02,1852.08 L1027.33,1852.54 L1027.12,1852.65 L1026.91,1852.83 L1026.71,1852.96 L1026.93,1853.47 L1027.2,1853.85 L1027.54,1854.1 L1027.97,1854.25 L1027.97,1854.63 L1027.7,1854.81 L1027.58,1855 L1027.32,1855.52 L1027.91,1855.88 L1027.74,1856.2 L1027.28,1856.13 L1027.01,1855.3 L1026.72,1854.81 L1024.55,1853.81 L1022.26,1851.67 L1021.76,1851.41 L1020.54,1850.23 L1020.08,1849.96 L1018.76,1849.86 L1018.21,1849.72 L1017.77,1849.49 L1016.2,1847.64 L1015.92,1847.56 L1015.61,1846.77 L1014.87,1846.24 L1012.84,1845.28 L1012.29,1844.87 L1011.88,1844.3 L1011.64,1843.49 L1011.1,1843.74 L1010.26,1843.86 L1009.5,1843.75 L1009.16,1843.26 L1008.86,1842.66 L1006.71,1841.71 L1006.41,1840.3 L1001.72,1837.63 L1000.29,1836.54 L999.849,1837.14 L999.513,1837.8 L999.329,1838.58 L999.36,1839.55 L997.316,1838.72 L996.496,1838.07 L996.591,1837.35 L995.774,1835.89 L994.524,1835.21 L990.006,1834.33 L987.846,1833.52 L986.545,1832.76 L986.133,1832.92 L984.063,1832.84 L983.033,1832.59 L982.693,1832.36 L982.221,1831.65 L979.326,1831.37 L978.191,1831.56 L977.631,1831.85 L976.314,1832.9 L975.834,1833.43 L975.527,1833.64 L975.183,1833.63 L974.819,1833.57 L974.475,1833.62 L971.685,1835.56 L970.915,1835.68 L970.377,1835.89 L969.257,1836.93 L965.879,1839.1 L964.885,1840.35 L965.095,1841.57 L964.62,1842.26 L964.481,1842.6 L964.421,1843.05 L964.283,1843.19 L963.592,1843.64 L963.346,1843.86 L963.089,1844.6 L963.167,1845.29 L963.139,1845.95 L962.509,1846.63 L961.365,1847.15 L958.828,1847.91 L957.514,1848.64 L956.818,1849.22 L956.217,1849.85 L955.129,1851.37 L954.976,1851.82 L955.009,1852.82 L954.943,1853.31 L954.642,1854.26 L954.288,1854.99 L952.101,1857.35 L951.324,1858.79 L951.986,1860.09 L951.977,1860.47 L950.919,1860.69 L949.549,1862.21 L948.359,1862.67 L948.006,1862.95 L947.721,1863.29 L947.603,1863.6 L947.352,1864.09 L946.799,1864.36 L945.694,1864.61 L942.393,1866.69 L934.461,1868.39 L932.187,1869.8 L930.289,1870.18 L924.713,1870 L923.434,1870.38 L922.843,1870.36 L922.494,1870.07 L922.032,1869.86 L921.552,1869.84 L920.76,1866.82 L920.59,1865.23 L921.337,1863.97 L923.364,1861.88 L924.321,1860.46 L926.717,1858.57 L927.76,1857.48 L928.131,1856.78 L928.546,1855.39 L928.858,1854.64 L929.369,1854.1 L930.037,1853.62 L930.561,1852.97 L930.681,1851.89 L930.513,1851.5 L930.016,1850.35 L929.048,1849.33 L928.533,1848.15 L929.213,1846.14 L927.828,1845.85 L925.939,1847 L924.636,1847.06 L916.725,1849.04 L914.914,1848.93 L914.484,1848.82 L914.074,1848.39 L913.85,1847.86 L913.553,1847.4 L912.969,1847.15 L911.803,1846.91 L910.968,1846.52 L906.283,1843.4 L904.502,1842.68 L901.909,1841.17 L901.348,1840.98 L900.741,1841.01 L900.195,1841.17 L899.657,1841.23 L899.067,1840.96 L898.662,1840.47 L898.479,1839.91 L898.352,1839.33 L898.093,1838.7 L897.698,1838.2 L896.831,1837.53 L896.443,1837.09 L896.211,1836.61 L895.731,1835.32 L895.403,1834.69 L894.46,1833.5 L894.286,1833.2 L894.142,1832.62 L894.163,1832.31 L894.289,1831.94 L894.742,1830.01 L895.078,1829.49 L896.616,1828.72 L895.483,1827.7 L894.178,1826.21 L893.244,1824.42 L893.183,1822.48 L893.693,1821.66 L896.041,1820.01 L897.678,1817.42 L898.32,1816.98 L897.868,1815.64 L898.235,1814.39 L899.096,1813.41 L900.141,1812.89 L900.962,1812.92 L902.554,1813.46 L903.352,1813.48 L903.124,1812.65 L901.866,1810.04 L901.715,1809.51 L901.534,1807.96 L901.079,1806.59 L901.115,1806.51 L901.373,1805.53 L901.458,1805.49 L901.464,1805.19 L901.512,1805.03 L901.533,1804.84 L901.451,1804.47 L901.278,1804.15 L900.512,1803.24 L898.317,1802.23 L897.547,1802.14 L895.337,1802.49 L894.504,1802.24 L891.51,1800.33 L890.106,1799.01 L889.384,1797.67 L889.617,1794.18 L889.407,1792.38 L888.278,1791.49 L886.731,1791.27 L886.28,1790.85 L885.788,1789.91 L885.693,1789.34 L885.765,1788.84 L885.715,1788.4 L885.021,1787.75 L884.294,1786.53 L884.851,1785.85 L885.528,1785.33 L885.68,1785.21 L887.253,1784.4 L887.649,1784.35 L888.448,1784.41 L888.798,1784.37 L889.192,1784.17 L890.013,1783.59 L890.354,1783.43 L891.894,1783.76 L893.481,1784.71 L895.051,1785.33 L896.52,1784.7 L896.622,1784.32 L896.377,1784.02 L896.185,1783.67 L896.439,1783.16 L896.793,1783.02 L897.827,1783.05 L898.233,1782.97 L898.905,1782.41 L899.726,1781.1 L900.318,1780.58 L901.927,1779.92 L903.501,1779.55 L903.847,1779.72 L904.374,1780.41 L904.604,1780.51 L904.912,1780.21 L905.81,1778.72 L907.048,1777.76 L907.084,1776.78 L906.656,1775.51 L906.5,1773.69 L906.742,1773.33 L908.231,1771.69 L909.202,1770.1 L909.629,1769.16 L909.702,1768.38 L909.37,1768.11 L907.516,1767.22 L906.945,1766.75 L906.617,1766.03 L906.341,1765.25 L906.074,1764.79 L905.934,1764.55 L904.075,1763.03 L902.109,1761.83 L901.812,1760.91 L901.672,1758.32 L901.517,1757.85 L901.065,1757.09 L900.998,1756.59 L901.137,1756.07 L901.683,1755 L901.801,1754.65 L901.57,1753.71 L901.045,1753.31 L899.636,1752.89 L899.069,1752.42 L898.111,1751.39 L897.536,1751.15 L896.399,1750.87 L895.253,1749.94 L894.425,1748.57 L894.038,1745.97 L893.967,1744.85 L894.044,1743.74 L894.322,1742.72 L894.915,1741.83 L895.605,1741.45 L896.914,1741.32 L896.918,1741.21 L897.055,1740.95 L897.257,1740.67 L897.467,1740.51 L897.633,1740.54 L897.976,1740.78 L898.138,1740.82 L898.442,1740.78 L899.521,1740.84 L900.462,1740.6 L901.471,1740.14 L902.38,1739.43 L903.015,1738.46 L903.323,1737.63 L903.704,1736.89 L904.186,1736.33 L904.825,1736.06 L906.7,1738.33 L907.692,1738.97 L909.023,1739.3 L910.45,1739.33 L911.777,1739.13 L914.512,1738.13 L915.766,1737.42 L916.346,1737.24 L919.614,1737.71 L921.029,1737.27 L923.672,1735.48 L924.989,1734.96 L925.908,1734.87 L926.405,1734.27 L926.852,1733.57 L927.594,1733.22 L932.004,1734.47 L932.677,1734.89 L933.313,1735.51 L934.098,1736.73 L934.156,1736.89 L934.688,1737.09 L935.056,1737.02 L935.39,1736.87 L935.82,1736.84 L937.896,1737.66 L939.097,1737.68 L939.871,1737.94 L940.123,1737.96 L940.677,1737.68 L940.787,1737.47 L940.76,1737.14 L940.913,1736.51 L940.97,1736.41 L941.981,1734.61 L943.161,1733.83 L946.272,1733.22 L946.694,1733.1 L947.012,1732.85 L947.575,1732.16 L947.61,1732.03 L948.058,1731.11 L948.142,1730.99 L948.417,1729.21 L948.44,1728.52 L948.552,1727.8 L948.879,1727.17 L949.324,1726.84 L950.226,1726.77 L950.702,1726.63 L952.645,1725.03 L953.638,1723.03 L953.572,1720.74 L952.349,1718.23 L951.357,1717.33 L951.21,1717.08 L951.407,1716.56 L951.857,1715.96 L952.725,1715.13 L953.673,1714.72 L955.565,1714.34 L956.476,1713.74 L958.673,1710.99 L959.913,1710.4 L960.419,1710.03 L960.86,1709.53 L961.132,1708.96 L961.128,1708.33 L960.846,1708.1 L960.687,1707.83 L961.066,1707.05 L962.009,1706.1 L963.198,1705.49 L965.043,1705.13 L965.618,1705.02 L966.856,1705.26 L967.433,1706.14 L967.584,1707.55 L967.568,1709.43 L967.384,1711.23 L966.615,1714.45 L966.435,1716.05 L966.574,1717.58 L967.029,1718.59 L967.815,1719.27 L968.942,1719.74 L970.129,1720.37 L971.339,1721.66 L974.01,1725.76 L974.429,1725.98 L975.265,1726.29 L977.273,1727.51 L978.038,1727.56 L978.351,1727.39 L979.036,1726.81 L979.308,1726.68 L979.528,1726.77 L979.665,1726.82 L979.975,1727.41 L980.304,1727.64 L981,1727.75 L982.337,1727.74 L982.991,1727.98 L984.076,1729.46 L983.41,1730.91 L982.138,1732.4 L981.345,1734.15 L981.173,1734.44 L981.098,1734.61 L981.846,1734.67 L982.51,1734.92 L984.992,1736.29 L985.282,1736.53 L985.561,1736.93 L985.975,1737.93 L986.714,1739.29 L987.068,1740.11 L987.301,1740.93 L987.358,1742.01 L987.171,1742.78 L986.898,1743.52 L986.722,1744.48 L988.461,1743.98 L989.886,1744.64 L991.198,1744.9 L992.617,1743.21 L993.732,1741.11 L993.912,1740.05 L993.428,1739.01 L993.02,1738.74 L992.07,1738.54 L991.649,1738.29 L991.259,1737.7 L990.913,1736.56 L990.45,1736.08 L990.357,1735.92 L990.328,1735.76 L990.362,1735.6 L990.458,1735.45 L991.909,1734.14 L991.162,1731.96 L991.36,1731.25 L992.488,1730.41 L993.447,1730.12 L993.864,1729.88 L994.217,1729.3 L994.366,1728.54 L994.289,1727.13 L994.469,1726.32 L995.288,1725.13 L998.472,1723.04 L998.496,1722.99 L998.821,1722.4 L999.03,1721.68 L999.279,1721.06 L999.755,1720.72 L1000.28,1720.48 L1000.79,1720.02 L1001.19,1719.39 L1001.86,1717.16 L1003.15,1714.53 L1003.43,1713.16 L1003.38,1712.35 L1002.82,1710.13 L1002.82,1709.24 L1002.94,1708.45 L1002.86,1707.83 L1002.29,1707.42 L1001.88,1706.57 L1002.24,1705.05 L1003,1703.55 L1003.84,1702.79 L1005.94,1702.56 L1006.8,1702.79 L1006.82,1703.56 L1007.92,1704.62 L1008.24,1704.77 L1008.71,1704.58 L1008.93,1704.21 L1009.08,1703.77 L1009.38,1703.37 L1010.08,1702.86 L1010.42,1702.79 L1010.56,1703.18 L1010.77,1709.18 L1010.7,1710.13 L1010.79,1710.48 L1011.1,1710.79 L1012.05,1711.32 L1012.43,1711.64 L1013.28,1713.9 L1013.83,1714.66 L1014.75,1715.29 L1015.75,1715.67 L1020.72,1715.75 L1021.54,1715.42 L1022.21,1714.51 L1022.23,1714.02 L1022.16,1713.31 L1022.25,1712.6 L1022.72,1712.07 L1023.19,1712.08 L1024.24,1712.72 L1024.8,1712.84 L1025.7,1712.55 L1028.59,1711.08 L1030.48,1710.76 L1031.32,1710.82 L1032.21,1711.06 L1033.29,1711.77 L1033.58,1712.78 L1033.6,1714.02 L1033.88,1715.39 L1034.52,1716.25 L1036.23,1717.76 L1036.67,1718.82 L1036.71,1720.23 L1038.14,1720.26 L1039.97,1719.56 L1041.18,1718.8 L1041.75,1717.61 L1041.19,1716.49 L1039.4,1714.68 L1039.01,1713.98 L1038.83,1713.46 L1038.86,1712.89 L1039.07,1712.04 L1039.4,1711.43 L1040.31,1710.62 L1040.62,1710.2 L1040.95,1708.84 L1040.63,1708.14 L1039.88,1707.85 L1037.94,1707.47 L1036.67,1706.98 L1035.99,1706.14 L1036.76,1704.84 L1036.38,1704.59 L1036.17,1704.24 L1036.07,1703.78 L1036.01,1703.21 L1036.2,1702.54 L1036.21,1701.17 L1036.27,1700.72 L1036.64,1700.17 L1037.57,1699.24 L1037.92,1698.76 L1038.45,1696.72 L1038.64,1696.26 L1039.05,1696.03 L1043.13,1694.85 L1044.24,1694.84 L1044.93,1695.4 L1045.02,1696.1 L1044.79,1697.38 L1044.98,1698.06 L1046.76,1698.67 L1047.1,1698.96 L1047.33,1699.28 L1047.6,1699.56 L1048.09,1699.79 L1048.64,1699.87 L1050.15,1699.73 L1053.24,1700.52 L1054.02,1700.37 L1054.36,1699.8 L1054.62,1699.37 L1054.96,1697.82 L1054.98,1696.2 L1054.64,1695 L1053.74,1694.25 L1051.9,1694.07 L1051.12,1693.17 L1050.75,1691.97 L1050.95,1691.38 L1051.42,1690.93 L1051.83,1690.19 L1051.92,1689.31 L1051.88,1688.6 L1052.05,1688.02 L1052.77,1687.55 L1053.29,1686.76 L1053.18,1685.9 L1052.87,1684.99 L1052.78,1684.07 L1053.71,1682.93 L1053.96,1682.54 L1054.11,1681.97 L1054.33,1679.96 L1055.73,1681.08 L1057.51,1681.26 L1061.87,1680.03 L1062.62,1679.94 L1063.29,1680.11 L1066.59,1682.04 L1067.01,1682.67 L1066.81,1683.1 L1065.86,1683.88 L1065.66,1684.19 L1065.92,1684.74 L1066.39,1684.75 L1067.29,1684.46 L1067.79,1684.62 L1069.03,1685.3 L1069.45,1685.32 L1070.27,1685.08 L1070.71,1685.08 L1071.1,1685.27 L1071.84,1685.89 L1072.28,1686.07 L1072.7,1686.03 L1074.13,1685.53 L1074.87,1685.42 L1076.37,1685.55 L1077.1,1685.78 L1077.71,1685.69 L1078.31,1685.05 L1079.25,1683.39 L1079.42,1683.22 L1079.89,1683 L1080.08,1682.79 L1080.15,1682.48 L1080.09,1681.83 L1080.12,1681.59 L1080.9,1679.65 L1081.33,1677.45 L1081.67,1676.68 L1084.4,1673.88 L1085.16,1673.41 L1088.11,1672.4 L1091.1,1671.85 L1092.64,1672.15 L1093.98,1672.78 L1095.29,1672.82 L1096.72,1671.39 L1096.72,1671.38 L1097.04,1670.89 L1097.42,1670.59 L1097.84,1670.48 L1098.6,1670.65 L1098.88,1670.81 L1099.13,1671.04 L1099.37,1671.33 L1099.73,1671.55 L1100.12,1671.61 L1100.52,1671.53 L1100.92,1671.3 L1100.92,1671.3 L1100.93,1671.3 L1100.94,1671.3 L1102.17,1670.91 L1103.19,1670.79 L1105.4,1671.19 L1105.42,1671.19 L1105.47,1671.2 L1105.8,1671.23 L1106.12,1671.17 L1106.94,1671.24 L1108.34,1672.23 L1109.15,1672.52 L1109.64,1672.39 L1110.47,1671.69 L1110.94,1671.37 L1112.87,1671.01 L1114.32,1669.81 L1116.11,1668.78 L1117.98,1668.08 L1121.03,1667.85 L1125.3,1665.48 L1126.01,1665.32 L1128.03,1664.86 L1129.05,1665.17 L1129.06,1665.97 L1129.07,1666.82 L1128.19,1668.08 L1125.64,1669.6 L1125.22,1670.7 L1125.52,1671.31 L1125.73,1672.02 M1143.49,1863.07 L1144.87,1863.72 L1146.26,1863.47 L1147.03,1862.09 L1146.99,1861.9 L1147,1861.71 L1147.07,1861.54 L1147.18,1861.41 L1147.22,1861.32 L1147.53,1860.69 L1147.59,1859.62 L1147.41,1858.58 L1147.08,1858 L1145.78,1858.26 L1144.36,1859.05 L1143.91,1859.53 L1143.23,1860.26 L1142.82,1861.76 L1143.49,1863.07 M1149.7,1987.53 L1149.72,1987.54 L1149.76,1987.54 L1149.76,1987.47 L1149.72,1987.47 L1149.7,1987.5 L1149.7,1987.53"
         id="path1017" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['IT'],polycolors['IT'],labels['IT'], x1, polybary['IT'], x2, polybary['IT'] )
    if 'IT' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1149.72,1987.54 L1149.7,1987.53 L1149.7,1987.5 L1149.72,1987.47 L1149.76,1987.47 L1149.76,1987.54 L1149.72,1987.54"
         id="path1019" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['VA'],polycolors['VA'],labels['VA'], x1, polybary['VA'], x2, polybary['VA'] )
    if 'VA' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1489.33,1542.84 L1490.34,1542.6 L1491.58,1542.64 L1492.8,1542.99 L1493.75,1543.68 L1494.83,1543.76 L1497.78,1541.76 L1499.09,1541.24 L1499.88,1541.45 L1500.45,1541.8 L1500.95,1541.93 L1501.51,1541.5 L1501.88,1540.73 L1502.01,1540.03 L1502.27,1539.4 L1503,1538.8 L1503.66,1538.56 L1505.06,1538.31 L1507.08,1537.3 L1507.7,1537.39 L1509.21,1538 L1509.99,1538.17 L1510.39,1538.44 L1510.74,1538.93 L1511.3,1540.07 L1511.64,1540.46 L1512.34,1540.66 L1513.85,1540.4 L1514.58,1540.45 L1515.04,1541.09 L1515.97,1543.56 L1516.41,1544.17 L1519.26,1547.35 L1520.41,1548.33 L1521.62,1548.96 L1522.98,1549.2 L1524.21,1548.89 L1526.17,1547.45 L1527.89,1546.9 L1529.03,1545.98 L1529.59,1545.64 L1531.72,1545.18 L1532.43,1544.79 L1533.17,1544.62 L1535.66,1544.46 L1536.39,1544.11 L1537.01,1543.42 L1537.63,1542.29 L1538.62,1542.3 L1538.69,1542.28 L1538.7,1542.28 L1538.74,1542.26 L1539.07,1541.75 L1540.33,1540.99 L1541.75,1540.93 L1543.34,1541.42 L1543.04,1543.39 L1543.19,1544.38 L1544.31,1544.12 L1544.6,1544.12 L1544.99,1544.59 L1545.09,1545.2 L1545.12,1545.93 L1545.29,1546.77 L1545.94,1547.95 L1548.47,1550.74 L1548.78,1550.96 L1549.09,1551 L1549.39,1550.85 L1549.67,1550.54 L1549.83,1550.43 L1549.99,1550.38 L1550.17,1550.37 L1551.03,1550.57 L1551.7,1550.59 L1552.33,1550.43 L1553.06,1549.89 L1553.23,1549.84 L1553.4,1549.84 L1553.58,1549.9 L1557.31,1553.42 L1558.07,1554.6 L1558.98,1556.46 L1559.93,1557.8 L1560.19,1558.01 L1560.3,1558.01 L1560.79,1557.61 L1563.76,1557.09 L1564.47,1556.71 L1565.09,1556.14 L1565.77,1555.82 L1566.51,1556.13 L1566.71,1556.4 L1568.33,1556.98 L1569.74,1557.91 L1570.41,1558.52 L1570.98,1559.29 L1571.47,1560.41 L1571.43,1561.07 L1571.25,1561.69 L1571.34,1562.67 L1570.78,1563.67 L1571.26,1564.36 L1572.99,1565.33 L1572.45,1566.25 L1571.74,1568.37 L1571.17,1569.11 L1569.62,1570.02 L1569.03,1570.6 L1568.73,1571.52 L1568.84,1572.04 L1569.11,1572.4 L1569.27,1572.84 L1569.04,1573.6 L1568.82,1573.8 L1567.94,1574.05 L1567.16,1574.59 L1566.72,1575.08 L1565.93,1576.61 L1564.89,1577.88 L1563.49,1578.8 L1561.94,1579.31 L1560.47,1579.32 L1557.13,1578.21 L1555.9,1578.73 L1555.63,1581.28 L1555.22,1581.82 L1554.71,1582.14 L1554.12,1582.29 L1552.16,1582.33 L1551.64,1582.47 L1550.94,1582.86 L1550.24,1583.42 L1549.85,1583.99 L1549.17,1585.52 L1548.9,1585.9 L1548.28,1586.53 L1548.03,1586.97 L1547.95,1587.45 L1548,1588.61 L1547.95,1589.16 L1547.16,1590.82 L1547.1,1591.25 L1547.14,1591.71 L1547.14,1592.12 L1547.14,1592.19 L1547,1592.69 L1546.71,1593 L1546.5,1593.21 L1544.55,1594.07 L1542.27,1596.45 L1541.25,1598.01 L1540.69,1599.65 L1541.11,1601.56 L1541.81,1603.64 L1542.2,1605.69 L1541.63,1607.55 L1539.85,1608.4 L1539.17,1608.99 L1538.54,1609.97 L1537.29,1612.61 L1537.15,1613.37 L1537.01,1615.66 L1536.88,1616.26 L1536.54,1617.26 L1536.47,1617.97 L1536.57,1618.69 L1536.76,1619.23 L1536.86,1619.8 L1536.72,1620.61 L1536.44,1621.16 L1535.65,1622.2 L1535.34,1622.85 L1535.24,1623.45 L1535.2,1624.57 L1535.01,1625.16 L1534.28,1626.14 L1532.42,1627.86 L1531.62,1628.9 L1530.27,1631.15 L1530.36,1631.34 L1530.51,1631.65 L1530.9,1631.77 L1531.32,1631.79 L1531.65,1631.98 L1532.14,1632.64 L1532.18,1632.7 L1532.02,1632.79 L1531.69,1635.97 L1531.4,1636.48 L1529.71,1638.37 L1529.53,1638.47 L1529.51,1638.65 L1529.55,1639.39 L1529.7,1640.14 L1529.95,1640.77 L1530.06,1641.44 L1529.83,1642.29 L1529.45,1642.74 L1527.97,1643.4 L1527.21,1644.37 L1526.82,1645.46 L1526.29,1648.1 L1526.14,1648.42 L1526.05,1648.77 L1526.01,1649.15 L1526.02,1649.53 L1526.14,1649.69 L1526.27,1649.82 L1526.42,1649.92 L1527.68,1650.52 L1527.72,1651.77 L1527.12,1653.04 L1526.34,1653.66 L1525.21,1654.04 L1524.86,1654.84 L1524.82,1655.09 L1524.65,1655.94 L1523.94,1657.24 L1523.09,1657.87 L1521.51,1657.99 L1520.64,1658.37 L1520.1,1659.25 L1520.03,1660.46 L1520.21,1662.86 L1519.85,1664.07 L1519.25,1665.12 L1518.8,1666.19 L1518.88,1667.48 L1520.48,1669.67 L1520.95,1670.99 L1520.07,1671.9 L1518.27,1672.25 L1517.48,1672.68 L1516.88,1673.63 L1516.65,1675.04 L1516.87,1677.82 L1516.68,1679.07 L1516.32,1680.09 L1515.96,1680.47 L1514.7,1680.65 L1514.45,1680.84 L1512.76,1683.64 L1512.03,1684.05 L1511.09,1683.45 L1510.74,1683.13 L1510.38,1683.01 L1510.01,1683.07 L1509.65,1683.3 L1508.81,1683.46 L1507.11,1683 L1506.32,1682.95 L1506.04,1683.09 L1505.46,1683.53 L1505.04,1683.63 L1504.66,1683.51 L1503.8,1682.97 L1503.37,1682.82 L1502.51,1682.91 L1501.64,1683.3 L1501.2,1683.63 L1500.85,1683.89 L1499.36,1685.53 L1499.22,1685.91 L1499.33,1686.49 L1499.92,1687.8 L1499.98,1688.3 L1499.68,1688.7 L1499.27,1688.69 L1498.85,1688.61 L1498.54,1688.77 L1498.39,1689.27 L1498.45,1690.02 L1498.47,1690.16 L1498.32,1690.8 L1497.76,1691.63 L1496.98,1692.18 L1494.63,1693.04 L1494.31,1693.08 L1493.76,1692.96 L1493.29,1692.73 L1491.88,1691.75 L1490.08,1691.28 L1488.27,1691.13 L1487.45,1692.97 L1480.56,1694.13 L1479.11,1696.59 L1476.51,1694.91 L1475.69,1694.7 L1474.68,1695.37 L1474.37,1695.45 L1474.01,1695.32 L1473.49,1694.77 L1473.24,1694.61 L1472.51,1694.54 L1472.1,1694.65 L1471.08,1695.35 L1469.88,1695.66 L1467.94,1694.88 L1465.16,1695 L1463.48,1695.61 L1462.85,1695.97 L1459.46,1697.93 L1458.69,1697.88 L1455.84,1696.54 L1454.83,1696.06 L1453.9,1695.89 L1452.97,1695.96 L1450.52,1696.78 L1449.6,1696.87 L1448.82,1697.11 L1447.82,1697.73 L1446.9,1698.53 L1446.36,1699.32 L1446.5,1699.82 L1446.89,1700.34 L1447.07,1700.84 L1446.01,1701.6 L1445.76,1702.06 L1445.59,1702.55 L1445.34,1703.02 L1444.68,1703.71 L1444.33,1703.91 L1443.86,1704.04 L1443.36,1704.37 L1443.07,1704.95 L1442.82,1705.61 L1442.44,1706.16 L1441.75,1706.5 L1440.15,1706.71 L1439.39,1706.98 L1439.05,1707.31 L1438.6,1707.74 L1438.37,1708.55 L1438.26,1709.33 L1437.84,1710.03 L1436.67,1710.41 L1432.83,1710.48 L1431.8,1710.04 L1431.8,1710.03 L1431.02,1708.89 L1430,1708.64 L1429.05,1709.19 L1428.51,1710.45 L1428.68,1712.29 L1427.94,1712.6 L1426.82,1712.58 L1425.88,1713.43 L1425.77,1713.67 L1425.75,1713.91 L1425.83,1714.13 L1426.22,1714.62 L1426.35,1714.87 L1426.32,1715.08 L1426.09,1715.23 L1425.22,1714.95 L1422.53,1715.07 L1421.9,1715.19 L1421.53,1715.74 L1421.07,1716.07 L1420.23,1716.43 L1419.55,1717.03 L1419.3,1717.07 L1419.1,1717.1 L1418.48,1716.66 L1418.12,1717.37 L1417.99,1717.96 L1417.85,1718.42 L1417.43,1718.72 L1416.89,1718.72 L1415.08,1718.02 L1412.85,1717.53 L1412.05,1717.8 L1411.18,1718.89 L1411.06,1719.22 L1411.01,1719.98 L1410.94,1720.38 L1410.87,1720.4 L1410.31,1721.2 L1409.57,1723.16 L1409.12,1723.86 L1407.42,1725.68 L1406.33,1726.23 L1405.31,1725.89 L1404.65,1726.45 L1403.34,1728.49 L1402.59,1729.23 L1402.28,1729.35 L1401.98,1729.42 L1401.68,1729.42 L1401.39,1729.37 L1401.37,1729.37 L1400.55,1728.52 L1399.85,1728.58 L1399.19,1729.06 L1398.47,1729.44 L1397.69,1729.24 L1396.85,1728.51 L1395.86,1728.61 L1394.42,1727.76 L1393.59,1727.59 L1390.01,1727.98 L1389.64,1728.15 L1388.94,1728.78 L1388.49,1729.01 L1387.99,1729.06 L1383.28,1728.29 L1380.28,1728.61 L1379.56,1728.69 L1379.18,1728.96 L1379.02,1729.47 L1378.82,1729.94 L1378.31,1730.1 L1378.28,1729.85 L1377.27,1728.46 L1375.91,1727.68 L1370.45,1726.6 L1369.46,1726.64 L1369.08,1726.44 L1368.96,1726.03 L1368.87,1725.71 L1368.75,1725.01 L1368.43,1724.1 L1368.33,1723.6 L1367.95,1723.05 L1365.69,1721.09 L1364.02,1720.53 L1362.14,1720.34 L1358.42,1720.54 L1358.08,1720.8 L1357.98,1721.02 L1357.83,1721.23 L1357.61,1721.44 L1357.23,1720.45 L1356.76,1720.3 L1355.76,1720.8 L1354.9,1720.95 L1354.74,1720.78 L1354.76,1720.22 L1354.41,1719.18 L1354.01,1720.01 L1353.55,1719.95 L1353.08,1719.61 L1352.99,1719.55 L1352.33,1719.39 L1352.29,1718.96 L1352.58,1718.86 L1353.1,1718.56 L1353.42,1718.47 L1352.97,1717.96 L1352.26,1717.93 L1351.56,1717.75 L1351.16,1716.74 L1351,1716.12 L1350.27,1714.18 L1349.97,1713.62 L1348.03,1711.53 L1347.52,1711.29 L1343.99,1711.51 L1343.32,1711.25 L1341.08,1709.88 L1340.53,1709.39 L1340.25,1708.74 L1339.85,1708.3 L1337.38,1706.65 L1336.92,1706.13 L1335.68,1704.24 L1333.89,1702.5 L1333.55,1702.26 L1333.23,1701.92 L1332.69,1700.72 L1332.28,1699.81 L1332.34,1698.82 L1331.83,1697.85 L1331.54,1697.52 L1331.11,1697.04 L1330.51,1696.54 L1330.28,1696.56 L1330.21,1696.85 L1330.11,1697.07 L1329.81,1696.84 L1329.6,1696.62 L1329.4,1696.45 L1329.19,1696.32 L1328.99,1696.25 L1327.81,1696.1 L1327.48,1695.93 L1327.24,1696.84 L1326.95,1696.87 L1326.32,1695.81 L1324.94,1695.12 L1324.04,1694.14 L1323.24,1693.01 L1322.37,1692.12 L1321.39,1691.56 L1320.23,1691.44 L1319.82,1691.23 L1319.48,1691.2 L1319.23,1691.43 L1319.09,1692.01 L1318.82,1692.03 L1318.2,1691.5 L1316.24,1690.51 L1315.98,1690.33 L1315.13,1687.7 L1313.58,1686.59 L1311.78,1684.26 L1310.18,1683.48 L1309.34,1682.92 L1309.13,1682.49 L1308.93,1682.09 L1309.27,1681.44 L1310.06,1680.98 L1310.29,1680.71 L1310.6,1680.35 L1310.27,1679.19 L1309.59,1678.82 L1308.79,1678.84 L1308.56,1678.76 L1308.52,1678.75 L1308.18,1678.64 L1308.08,1677.59 L1307.49,1677.74 L1307.05,1677.31 L1306.63,1676.62 L1306.09,1676.04 L1305.35,1674.3 L1305.24,1674.07 L1305.27,1673.84 L1305.35,1673.36 L1305.68,1672.67 L1305.82,1672.38 L1306.06,1671.65 L1306.24,1670.87 L1306.29,1670.31 L1306.07,1670.01 L1305.44,1670.03 L1305.05,1669.86 L1304.84,1669.64 L1304.13,1668.89 L1303.67,1668.68 L1299.74,1669.35 L1297.87,1669.67 L1297.64,1669.64 L1296.07,1669.39 L1296.73,1669.01 L1297.23,1668.44 L1298.61,1666.2 L1299.06,1665.64 L1300.09,1664.76 L1300.95,1664.3 L1301.22,1664.05 L1301.47,1663.57 L1301.46,1663.19 L1301.4,1662.85 L1301.48,1662.46 L1301.88,1662.05 L1302.3,1661.92 L1302.62,1661.61 L1302.73,1660.65 L1303.08,1659.92 L1303.65,1659.8 L1305.27,1659.98 L1306.99,1659.62 L1307.9,1659.65 L1308.68,1660.14 L1308.92,1660.29 L1309.15,1660.34 L1309.35,1660.27 L1309.52,1660.08 L1309.52,1660.07 L1310.22,1659.88 L1311.33,1659.77 L1312.12,1659.51 L1311.87,1658.85 L1311.21,1658.34 L1310.59,1658.19 L1309.36,1658.15 L1309.85,1657.62 L1311.69,1656.68 L1312.14,1656.33 L1312.28,1655.97 L1312.11,1655.66 L1311.58,1655.45 L1310.7,1655.24 L1310.32,1654.45 L1310.52,1653.51 L1311.37,1652.88 L1312.23,1651.51 L1312.37,1650.76 L1311.79,1650.05 L1311.08,1650.05 L1309.7,1650.86 L1309.1,1650.53 L1309.09,1650.19 L1309.34,1649.1 L1309.28,1648.55 L1308.96,1648.17 L1308.61,1648.17 L1308.27,1648.23 L1308,1648.03 L1307.76,1647.02 L1308.3,1645.59 L1308.07,1644.52 L1308.07,1644.52 L1308.07,1644.51 L1309.33,1643.69 L1309.88,1643.08 L1310.09,1642.22 L1309.83,1641.22 L1308.32,1638.63 L1308.19,1638.52 L1308,1638.46 L1307.8,1638.37 L1307.66,1638.15 L1307.66,1637.77 L1307.83,1637.59 L1308.03,1637.46 L1308.1,1637.27 L1307.8,1634.89 L1308.16,1634.07 L1308.67,1633.88 L1309.27,1634.22 L1309.85,1635 L1314.22,1632.52 L1315.7,1631.14 L1316.24,1630.63 L1315.85,1628.13 L1316.29,1627.57 L1317.47,1626.94 L1317.85,1626.1 L1317.82,1625.14 L1317.42,1624.39 L1316.78,1623.85 L1316.03,1623.51 L1316.16,1622.28 L1315.7,1621.06 L1314.88,1620.14 L1313.92,1619.8 L1312.55,1620.16 L1309.68,1619.27 L1308.54,1619.63 L1306.1,1618.89 L1305.3,1618.51 L1305.63,1618.01 L1306.18,1616.92 L1306.47,1616.58 L1306.74,1616.87 L1306.96,1617.05 L1307.16,1617.07 L1307.42,1616.87 L1307.89,1616.38 L1309.49,1615.34 L1309.81,1614.97 L1310,1614.41 L1309.85,1613.6 L1310.05,1612.97 L1311.53,1612.14 L1313.32,1612.2 L1316.8,1613.21 L1317.36,1613.53 L1317.63,1614.07 L1317.85,1614.71 L1318.23,1615.33 L1318.74,1615.77 L1319.22,1616.01 L1321.62,1616.16 L1321.98,1616.04 L1322.4,1615.54 L1323.11,1614.15 L1323.62,1613.63 L1324.38,1615.2 L1326,1615.34 L1329.26,1614.2 L1332.28,1613.49 L1333.11,1613.01 L1332.6,1612.76 L1332.2,1612.32 L1331.94,1611.71 L1331.67,1609.68 L1331.27,1608.45 L1331.12,1607.38 L1331.68,1606.62 L1331.37,1606.25 L1330.86,1605.13 L1330.51,1604.91 L1329.59,1604.59 L1329.3,1604.33 L1329.22,1603.66 L1329.72,1603.33 L1331.14,1602.87 L1331.78,1602.26 L1332.14,1601.59 L1332.3,1600.75 L1332.37,1599.63 L1332.27,1599.51 L1332.05,1599.46 L1331.83,1599.36 L1331.76,1599.09 L1331.86,1598.9 L1332.22,1598.6 L1332.29,1598.49 L1332.36,1597.94 L1332.53,1597.53 L1332.51,1597.15 L1332.02,1596.68 L1334.43,1594.23 L1335.87,1593.15 L1337.4,1593.33 L1339.18,1593.64 L1339.62,1593.72 L1342.39,1593.86 L1343.81,1594.82 L1348.66,1600.14 L1349.11,1600.48 L1349.56,1600.62 L1350.63,1600.73 L1351.05,1600.95 L1352.67,1602.93 L1353.24,1603.4 L1353.67,1603.34 L1354.07,1603.09 L1354.53,1602.97 L1355.19,1603.22 L1356.12,1603.75 L1356.97,1604.41 L1357.36,1605.02 L1357.85,1605.46 L1359.73,1606.24 L1360.74,1606.65 L1364.35,1607.24 L1366.75,1606.82 L1376.29,1605.16 L1381.51,1605.12 L1383.05,1604.8 L1386.04,1603.17 L1394.5,1601.19 L1396.39,1601.1 L1397.99,1601.6 L1399.27,1601.69 L1400.47,1601.42 L1401.42,1600.67 L1402.62,1598.92 L1403.28,1598.29 L1404.2,1597.93 L1405.19,1597.44 L1405.25,1597.41 L1403.51,1596.43 L1402.15,1595.38 L1401.74,1594.25 L1401.69,1592.93 L1401.81,1590.31 L1401.59,1589.61 L1401.27,1589.4 L1401.19,1589.17 L1401.66,1588.44 L1402.01,1588.17 L1402.8,1587.94 L1403.16,1587.55 L1404,1585.09 L1404.66,1584.41 L1408.51,1583.04 L1410.47,1582.35 L1411.04,1581.99 L1411.98,1581.92 L1412.81,1581.85 L1415.25,1581.18 L1420.48,1581.17 L1420.92,1581.02 L1423.21,1579.11 L1428.83,1578.51 L1430.86,1576.64 L1430.88,1576.32 L1430.86,1575.99 L1430.8,1575.65 L1430.69,1575.32 L1430.67,1575.28 L1430.66,1575.24 L1430.66,1575.21 L1430.67,1575.18 L1431.04,1574.14 L1431.13,1571.68 L1431.47,1570.72 L1432.13,1570.21 L1435.81,1568.7 L1436.25,1568.66 L1436.68,1568.72 L1437.21,1569.06 L1438.25,1570.06 L1438.69,1570.22 L1440.55,1569.59 L1441.54,1569.63 L1442.4,1570.42 L1442.5,1571.03 L1442.33,1571.6 L1442.29,1572.13 L1442.76,1572.67 L1443.16,1572.67 L1444.58,1571.9 L1445.64,1572.09 L1447.41,1573.3 L1448.31,1573.52 L1449.24,1573.03 L1450.88,1571.02 L1451.76,1570.29 L1453.27,1569.58 L1453.4,1569.48 L1454.91,1568.24 L1455.65,1567.83 L1455.96,1567.51 L1456.18,1566.92 L1456.26,1565.77 L1456.36,1565.31 L1456.8,1564.63 L1457.14,1564.51 L1457.59,1564.6 L1458.32,1564.58 L1458.98,1564.18 L1460.07,1562.85 L1460.5,1562.58 L1461.38,1562.88 L1461.92,1563.33 L1462.45,1563.47 L1463.32,1562.85 L1464.36,1561.47 L1465.17,1559.74 L1465.79,1557.83 L1466.71,1552.68 L1467.05,1551.65 L1467.55,1550.74 L1468.62,1549.24 L1468.7,1549.12 L1469.14,1548.22 L1469.14,1547.99 L1469.13,1547.77 L1469.09,1547.55 L1469.04,1547.34 L1468.8,1546.27 L1468.74,1545.76 L1468.73,1545.28 L1469.84,1544.63 L1472.37,1544.09 L1480.76,1540.81 L1481.42,1540.7 L1482.11,1540.94 L1483.49,1541.88 L1484.1,1541.95 L1485.41,1541.89 L1487.84,1542.93 L1489.33,1542.84"
         id="path1021" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['HU'],polycolors['HU'],labels['HU'], x1, polybary['HU'], x2, polybary['HU'] )
    if 'HU' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1274.19,1012.72 L1274.49,1012.81 L1274.78,1012.59 L1274.93,1012.05 L1275.16,1012.03 L1275.2,1012.3 L1275.28,1012.46 L1275.38,1012.61 L1275.47,1012.81 L1275.67,1013.07 L1275.84,1013.36 L1275.97,1013.68 L1276.07,1014.04 L1275.15,1014.25 L1274.51,1015.09 L1273.59,1017.33 L1274.54,1019.06 L1274.69,1020 L1274.32,1021.65 L1274.39,1022.2 L1274.37,1022.45 L1274.25,1022.57 L1273.81,1022.83 L1273.69,1022.95 L1273.59,1023.54 L1273.55,1024.49 L1273.47,1025.08 L1272.33,1027.94 L1272.22,1028.87 L1272.15,1030.54 L1271.63,1031.31 L1270.91,1031.72 L1270.2,1032.31 L1270.47,1032.83 L1270.78,1033.17 L1271.14,1033.37 L1271.53,1033.47 L1270.92,1034.17 L1270.48,1034.9 L1269.7,1036.69 L1270.02,1037.47 L1270.36,1038.74 L1270.43,1039.95 L1269.96,1040.53 L1269.7,1041.24 L1269.47,1044.58 L1269.19,1045.76 L1269.1,1045.77 L1267.61,1046.86 L1267.42,1047.21 L1267.26,1047.77 L1267.17,1048.36 L1267.15,1048.98 L1267.19,1049.61 L1267.14,1050.88 L1265.77,1057.44 L1265.59,1058.7 L1265.61,1060.15 L1265.54,1060.71 L1265.31,1061.37 L1265.01,1061.9 L1264.75,1062.14 L1264.33,1062.4 L1264.17,1062.96 L1264.12,1064.34 L1264.14,1064.7 L1264.38,1064.86 L1264.43,1065.18 L1264.37,1065.49 L1264.13,1066.19 L1264.08,1066.5 L1264.12,1067.57 L1264.07,1068.28 L1263.93,1068.89 L1263.54,1070.08 L1263.2,1071.42 L1262.97,1072.75 L1262.87,1073.94 L1262.88,1074.44 L1263.05,1075.86 L1263,1076.5 L1262.59,1077.42 L1262.54,1078.05 L1262.36,1079.07 L1261,1083.38 L1260.44,1084.21 L1259.71,1084.81 L1258.96,1084.93 L1258.34,1084.27 L1258.22,1083.72 L1258.16,1082.28 L1258.1,1081.67 L1257.47,1080.01 L1257.63,1078.63 L1257.53,1077.09 L1257.2,1074.46 L1257.39,1073.45 L1257.08,1072.36 L1256.6,1071.2 L1256.29,1070.03 L1256.06,1064.65 L1256.11,1064.03 L1256.53,1062.03 L1257.19,1060.52 L1257.26,1060.02 L1257.37,1058.69 L1261.75,1043.54 L1262.27,1042.85 L1262.97,1042.53 L1264.01,1042.38 L1264.91,1042.01 L1265.33,1041.21 L1265.68,1038.79 L1266.84,1034.78 L1268.27,1031.17 L1269.35,1027.35 L1269.49,1022.69 L1269.56,1022.05 L1269.85,1021.7 L1270.53,1021.28 L1270.92,1020.66 L1271.05,1020.11 L1270.79,1016.57 L1270.83,1015.48 L1271.49,1014.6 L1272.35,1012.82 L1272.86,1012.25 L1273.47,1011.85 L1273.82,1011.85 L1274.19,1012.72"
         id="path1023" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1327.82,975.474 L1328.56,976.255 L1328.6,974.643 L1329.05,973.798 L1329.53,973.116 L1329.64,971.987 L1329.44,971.627 L1329.11,971.165 L1328.99,970.747 L1329.43,970.508 L1331.43,969.865 L1331.48,970.251 L1331.12,971.399 L1331.83,971.91 L1332.63,971.766 L1332.51,970.944 L1332.97,970.526 L1333.6,970.21 L1334.23,970.096 L1334.72,970.274 L1334.53,970.689 L1334.76,971.255 L1335.09,971.93 L1335.52,972.469 L1336.07,972.651 L1336.45,972.962 L1337.62,974.544 L1337.91,975.077 L1335.61,975.607 L1334.79,976.631 L1334.92,978.946 L1334.47,978.428 L1334.02,978.169 L1333.73,978.328 L1333.79,979.089 L1333.55,979.615 L1333.43,980.285 L1333.52,980.998 L1333.91,981.657 L1332.63,982.173 L1331.44,982.361 L1331.01,982.278 L1330.24,981.784 L1329.85,981.696 L1329.42,981.911 L1329.43,982.326 L1329.62,982.857 L1329.72,983.437 L1329.54,985.614 L1329.15,987.206 L1329.06,987.864 L1328.95,988.255 L1328.86,988.767 L1328.86,989.38 L1329.1,989.52 L1330.54,989.794 L1330.35,990.798 L1330.25,991.676 L1330.23,992.556 L1330.33,993.546 L1330.17,994.405 L1329.81,995.348 L1329.7,996.189 L1330.27,996.76 L1330.15,997.761 L1330.38,998.339 L1330.83,998.537 L1331.41,998.388 L1331.28,998.726 L1331.12,999.717 L1332.49,1000.11 L1333.88,1000.24 L1333.98,1000.01 L1333.99,999.601 L1334.13,999.327 L1334.6,999.494 L1335.18,999.829 L1335.38,1000.01 L1335.65,1000.44 L1335.93,1001.06 L1336.17,1001.89 L1336.14,1002.64 L1335.64,1003.03 L1334.6,1003.16 L1334.47,1003.11 L1334.05,1002.86 L1333.84,1002.82 L1333.7,1002.98 L1333.37,1003.61 L1333.16,1003.77 L1332.09,1004.19 L1331.13,1005.05 L1330.3,1006.2 L1329.66,1007.47 L1328.54,1008.77 L1328.19,1009.68 L1328.57,1010.84 L1328.85,1010.95 L1329.54,1010.72 L1329.85,1010.89 L1330.21,1012.83 L1329.09,1013.57 L1329.17,1013.78 L1326.26,1014.84 L1325.42,1015.53 L1325.48,1015.99 L1325.94,1015.93 L1325.99,1016.32 L1325.25,1016.62 L1321.34,1019.49 L1320.59,1020.29 L1320.01,1021.34 L1322.34,1021.07 L1321.63,1021.81 L1319.37,1023.39 L1318.96,1024.28 L1319.12,1027.51 L1319.58,1028.26 L1320.16,1028.64 L1321.43,1028.96 L1319.55,1029.18 L1319.61,1029.6 L1320.13,1030.01 L1319.34,1030.9 L1319.03,1031.44 L1318.85,1032.79 L1318.45,1033.24 L1315.43,1035.26 L1314.29,1035.59 L1313.16,1035.09 L1314.9,1030.84 L1314.93,1030.3 L1314.8,1029.69 L1314.57,1028.9 L1314.95,1028.31 L1316.14,1027.69 L1316.59,1027.16 L1316.77,1026.34 L1316.85,1024.64 L1317.04,1023.88 L1316.3,1024.49 L1315.67,1025.46 L1315.02,1026.16 L1314.22,1025.93 L1314.16,1025.47 L1314.55,1024.87 L1314.82,1024 L1314.8,1023.21 L1314.32,1022.86 L1314.42,1022.52 L1314.52,1021.91 L1314.63,1021.57 L1313.1,1021.05 L1311.61,1015.86 L1309.36,1014.28 L1309.28,1013.09 L1309.63,1011.85 L1310.77,1010.69 L1311.07,1009.51 L1311.09,1008.21 L1310.81,1007.3 L1311.16,1006.48 L1311.03,1006.11 L1310.69,1005.92 L1310.38,1005.63 L1310.24,1005.38 L1309.84,1004.83 L1309.74,1004.51 L1309.68,1003.71 L1309.53,1003.36 L1309.07,1002.57 L1308.57,1001.38 L1308.15,999.986 L1307.93,998.556 L1308.2,996.187 L1309.1,994.207 L1313.85,987.69 L1317.44,979.032 L1318.47,977.434 L1320.05,976.433 L1320.53,976.478 L1321.07,976.687 L1321.6,976.754 L1322.02,976.39 L1322.17,975.915 L1322.24,974.991 L1322.35,974.359 L1322.69,973.164 L1323.14,972.121 L1323.7,971.293 L1324.39,970.739 L1325.15,970.54 L1325.81,970.78 L1326.39,971.451 L1326.93,972.538 L1326.99,972.852 L1326.94,973.102 L1326.87,973.343 L1326.84,973.61 L1326.96,973.965 L1327.14,974.128 L1327.31,974.233 L1327.41,974.419 L1327.56,975.014 L1327.82,975.474"
         id="path1025" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1325.96,306.188 L1327.07,306.702 L1327.49,306.615 L1327.86,306.405 L1328.29,306.319 L1329.06,306.849 L1329.95,307.822 L1330.8,308.505 L1332.04,307.675 L1332.74,308.184 L1334.13,310.057 L1334.97,310.815 L1338.87,312.504 L1339.32,312.924 L1339.58,313.401 L1339.75,313.883 L1339.98,314.38 L1340.41,314.916 L1341.72,316.016 L1342.29,316.968 L1342.46,318.42 L1342.77,318.827 L1342.9,319.306 L1343.11,320.851 L1343.11,321.416 L1343.18,321.749 L1343.35,322.296 L1343.44,322.399 L1343.55,322.428 L1343.78,322.734 L1344.27,322.625 L1345.94,320.858 L1346.72,320.664 L1347.13,321.051 L1347.52,321.64 L1349.24,325.069 L1349.43,325.569 L1349.74,326.064 L1351.82,327.363 L1352.57,327.749 L1355.2,329.93 L1356.95,330.223 L1357.57,330.911 L1357.78,331.985 L1357.46,333.282 L1356.84,334.073 L1355.32,335.001 L1354.68,335.828 L1354.51,336.976 L1354.71,338.416 L1355.1,339.885 L1355.5,341.127 L1356.28,344.725 L1356.54,345.46 L1357.01,346.371 L1357.8,348.739 L1358.21,349.702 L1359.12,351.184 L1359.51,352.107 L1359.91,353.546 L1359.62,354.242 L1358.91,355.063 L1358.44,355.874 L1358.87,356.555 L1358.7,357.893 L1358.4,359.079 L1358.37,360.306 L1358.98,361.74 L1360.01,362.632 L1360.88,362.395 L1361.72,361.771 L1362.67,361.48 L1364.52,362.215 L1366.2,362.182 L1366.77,362.327 L1367.17,362.707 L1367.43,364.089 L1367.53,365.851 L1367.84,367.27 L1368.78,367.635 L1368.66,369.633 L1368.23,370.79 L1365.8,372.593 L1365.45,373.221 L1365.47,374.225 L1365.65,375.422 L1365.87,375.827 L1366.36,375.982 L1366.34,376.797 L1366.26,377.511 L1366.12,378.135 L1365.93,378.674 L1366.37,379.851 L1366.96,380.549 L1367.63,381.118 L1368.56,382.248 L1368.87,382.475 L1369.16,382.805 L1369.39,383.449 L1369.57,384.23 L1369.74,384.74 L1369.93,385.159 L1370.22,385.671 L1371.54,387.484 L1372.32,388.334 L1372.85,388.658 L1373.65,388.835 L1374.43,389.533 L1376.5,392.35 L1377.73,393.452 L1378.44,394.24 L1379.47,395.74 L1381.13,398.55 L1381.3,398.977 L1381.3,399.544 L1380.99,399.819 L1380.41,399.901 L1379.92,400.158 L1379.81,401.335 L1379.32,402.84 L1379.4,403.291 L1380.18,405.47 L1380.46,406.722 L1380.63,407.958 L1380.7,409.122 L1380.68,410.15 L1381.1,411.194 L1381.45,412.272 L1381.64,413.327 L1381.61,414.31 L1381.22,414.958 L1380.11,416.161 L1380.23,416.846 L1379.86,417.539 L1379.19,418.092 L1378.71,418.744 L1378.93,419.718 L1378.37,420.097 L1377.77,420.789 L1377.37,421.834 L1377.42,423.262 L1377.69,423.984 L1378.41,425.017 L1378.65,425.621 L1378.87,426.438 L1379.25,427.519 L1379.41,428.353 L1379.47,429.01 L1379.43,429.853 L1379.52,430.463 L1379.75,431.122 L1380.41,432.42 L1380.98,433.962 L1382.52,436.86 L1383.39,437.766 L1384.45,438.378 L1387.25,438.803 L1387.97,439.451 L1390.24,443.261 L1390.58,443.751 L1390.93,444.09 L1391.69,444.621 L1392.04,444.974 L1392.48,445.773 L1393.8,448.678 L1394.05,449.587 L1394.3,450.951 L1395.11,452.054 L1396.04,453.102 L1397.14,455.406 L1398.51,456.915 L1398.75,458.019 L1398.42,459.247 L1397.57,459.595 L1395.63,459.491 L1395.87,459.828 L1395.96,460.009 L1396.06,460.116 L1396.34,460.218 L1395.8,460.707 L1394.5,460.669 L1394,461.139 L1393.93,461.535 L1393.97,462.584 L1393.85,462.947 L1393.61,463.049 L1393.36,462.923 L1393.11,462.735 L1392.67,462.628 L1392.39,462.454 L1392.12,462.166 L1391.95,461.769 L1391.76,461.228 L1391.61,461.26 L1391.4,461.666 L1390.16,461.898 L1389.5,461.801 L1389.07,461.064 L1388.61,460.12 L1387.95,459.805 L1386.69,460.002 L1386.37,460.37 L1386.01,461.864 L1385.76,462.36 L1385.45,462.412 L1385.2,462.16 L1385.02,461.845 L1384.87,461.667 L1382.85,462.049 L1381.66,461.911 L1381.64,461.029 L1381.4,460.783 L1381.08,460.915 L1380.8,461.172 L1380.84,461.273 L1380.8,462.081 L1380.66,462.309 L1380.51,462.426 L1380.36,462.618 L1380.27,463.053 L1380.61,463.268 L1380.8,463.581 L1380.9,464.026 L1380.98,464.652 L1381.02,465.168 L1380.91,465.275 L1380.64,465.207 L1380.19,465.201 L1379.68,464.887 L1379.21,464.011 L1378.88,462.936 L1378.79,462.04 L1378.26,462.228 L1378,462.808 L1377.82,463.487 L1377.56,463.975 L1376.99,464.098 L1376.37,463.756 L1375.14,462.749 L1374.66,462.566 L1374.22,462.656 L1374,463.163 L1374.19,464.234 L1374.56,464.949 L1375.05,465.398 L1376.02,466.013 L1375.69,466.401 L1374.25,467.53 L1374.04,467.863 L1373.69,468.671 L1373.96,469.823 L1373.42,470.586 L1372.77,470.548 L1372.71,469.255 L1371.63,467.722 L1371.24,467.406 L1369.78,467.488 L1368.48,467.033 L1368.23,466.87 L1367.48,465.906 L1367.09,465.813 L1366.55,466.156 L1366.09,466.168 L1365.36,465.565 L1364.05,463.212 L1363.33,462.347 L1359.98,460.342 L1359.44,460.271 L1359,460.553 L1359.45,461.114 L1360.26,461.809 L1360.63,462.419 L1360.82,462.929 L1361.23,464.473 L1361.49,465.254 L1361.73,465.828 L1361.91,466.445 L1362.01,467.367 L1361.98,468.23 L1361.83,468.95 L1361.57,469.456 L1361.2,469.679 L1361.04,468.248 L1360.25,467.614 L1358.71,467.537 L1358.4,467.84 L1357.75,468.957 L1357.48,469.104 L1357.22,468.598 L1357.24,467.892 L1357.41,467.232 L1357.58,466.876 L1357.07,466.146 L1356.6,465.704 L1355.59,465.113 L1355.61,464.877 L1355.61,464.241 L1353.71,464.141 L1352.83,464.438 L1352.99,465.584 L1352.01,466.155 L1353.39,467.635 L1353.46,468.379 L1353.14,468.517 L1352.25,468.315 L1352.66,468.833 L1353.3,470.088 L1353.8,470.638 L1353.23,472.372 L1352.1,472.073 L1350.86,471.227 L1349.94,471.328 L1350.67,472.208 L1351.28,472.705 L1351.94,472.917 L1353.71,473.003 L1354.37,473.386 L1354.94,474.298 L1355.57,475.946 L1355.23,475.942 L1354.92,476.037 L1354.65,476.256 L1354.42,476.627 L1353.87,475.188 L1353.18,474.579 L1352.5,474.66 L1351.99,475.288 L1352.25,475.566 L1352.4,475.963 L1352.42,476.446 L1352.32,477.003 L1352.59,477.307 L1352.87,477.794 L1353.42,478.93 L1352.91,479.401 L1351.06,479.824 L1353.09,480.244 L1354.06,480.085 L1354.3,478.773 L1354.05,478.168 L1353.25,477.428 L1353.02,476.878 L1353.13,476.312 L1353.58,476.503 L1354.1,477.031 L1354.43,477.494 L1354.31,477.904 L1355.84,478.339 L1356.02,478.699 L1355.06,479.821 L1354.99,480.383 L1355.36,480.86 L1356.01,481.168 L1357.08,481.341 L1357.29,481.566 L1357.6,482.136 L1358.11,483.324 L1357.58,483.865 L1357.01,483.454 L1356.42,482.75 L1355.84,482.438 L1355.04,482.418 L1354.42,482.217 L1353.69,481.652 L1353.36,481.601 L1353.12,482.052 L1353.44,482.89 L1352.68,482.833 L1349.69,481.355 L1347.6,480.862 L1346.63,479.916 L1346.16,479.564 L1345.02,479.452 L1343.96,478.58 L1342.26,478.296 L1341.81,478.008 L1341.51,477.709 L1340.76,476.628 L1340.33,476.216 L1339.43,475.81 L1338.98,475.364 L1340.05,477.777 L1341.88,479.093 L1345.36,480.354 L1347.87,482.063 L1349.73,482.633 L1350.15,482.894 L1350.64,483.365 L1351.41,484.266 L1351.8,484.48 L1352.3,484.365 L1351.74,485.048 L1350.3,485.32 L1349.64,485.7 L1349.45,485.959 L1349.43,486.127 L1349.45,486.368 L1349.4,486.85 L1349.19,487.113 L1348.9,486.867 L1348.59,486.542 L1348.35,486.577 L1347.74,487.165 L1344,486.588 L1343.48,486.774 L1345.07,488.656 L1345.92,489.259 L1346.44,488.821 L1345.78,488.232 L1345.55,488.148 L1346.18,487.947 L1348.04,489.686 L1348.86,490.168 L1348.82,491.05 L1348.83,491.803 L1348.75,492.404 L1348.45,492.832 L1348.02,492.861 L1347.55,492.505 L1346.71,491.409 L1346.59,492.19 L1346.8,492.805 L1347.09,493.358 L1347.19,493.957 L1346.91,494.494 L1346.38,494.67 L1345.85,494.453 L1345.53,493.813 L1344.88,494.628 L1344.18,495.161 L1343.39,495.355 L1341.05,494.843 L1340.62,495.053 L1341.15,495.862 L1339.96,496.126 L1339.28,496.058 L1338.52,495.008 L1337.92,495.138 L1336.9,495.714 L1336.97,496.096 L1338.12,496.412 L1338.48,496.313 L1338.55,496.695 L1336.63,496.709 L1335.68,497.086 L1335.36,498.135 L1335.53,498.181 L1335.8,498.393 L1335.94,498.467 L1335.04,499.484 L1335.61,499.554 L1336.24,499.844 L1336.87,500.352 L1337.42,501.051 L1337.91,501.445 L1338.53,501.514 L1338.95,501.121 L1338.82,500.148 L1339.34,500.246 L1339.88,500.614 L1340.91,501.56 L1341.9,502.151 L1342.4,502.758 L1342.5,503.648 L1341.9,504.94 L1340.86,505.874 L1339.7,506.24 L1338.74,505.817 L1338.37,505.234 L1337.6,503.724 L1337.14,503.022 L1336.67,502.482 L1336.18,502.023 L1335.64,501.726 L1335.08,501.673 L1334.27,501.128 L1333.23,499.917 L1332.18,498.967 L1331.34,499.229 L1331.5,499.286 L1331.79,499.49 L1331.95,499.562 L1331.74,500.394 L1330.68,500.654 L1330.47,501.533 L1333.93,501.901 L1334.88,502.571 L1336.97,505.248 L1338.75,506.854 L1341.33,510.349 L1341.8,511.589 L1341.4,512.222 L1339.97,512.565 L1340.38,512.895 L1340.62,513.402 L1340.72,514.05 L1340.56,515.908 L1340.77,516.218 L1340.77,516.221 L1340.77,516.221 L1341.21,516.273 L1341.82,516.577 L1341.32,517.162 L1339.1,517.465 L1338.86,517.939 L1339.3,518.814 L1339.91,519.61 L1340.19,519.872 L1339.51,520.743 L1339.29,520.891 L1339,520.836 L1338.42,520.576 L1338.13,520.647 L1338.15,520.76 L1338.15,521.038 L1338.13,521.34 L1338.09,521.519 L1337.71,521.997 L1337.42,522.54 L1337.42,523.121 L1337.93,523.707 L1337.57,524.194 L1334.4,524.922 L1334.11,525.199 L1334.14,525.683 L1334.36,526.018 L1334.56,526.425 L1334.54,527.111 L1334.41,527.59 L1334.2,528.128 L1333.94,528.566 L1333.62,528.733 L1334.17,530.321 L1333.77,531.071 L1333.32,531.633 L1333.74,532.677 L1333.38,533.116 L1332.54,533.654 L1332.07,533.775 L1331.61,533.546 L1331.33,532.905 L1331.16,532.193 L1330.96,531.751 L1330.5,531.624 L1330.13,531.94 L1329.93,532.592 L1329.96,533.483 L1330.23,534.25 L1330.62,534.533 L1331.48,534.729 L1331.48,535.388 L1331.47,535.634 L1332.05,536.145 L1332.57,536.307 L1333.6,536.155 L1335.2,535.387 L1337.03,535.879 L1337.77,536.293 L1338.1,537.154 L1337.81,537.517 L1336.43,537.619 L1335.96,537.891 L1336.6,538.854 L1337.3,539.583 L1338.69,540.509 L1338.77,540.929 L1338.33,541.024 L1338.2,541.377 L1338.36,541.826 L1338.8,542.222 L1338.87,542.605 L1338.11,542.402 L1336.6,541.186 L1335.86,540.97 L1335.23,540.603 L1334.54,539.775 L1333.86,539.164 L1333.25,539.47 L1333.11,540.334 L1333.35,541.372 L1333.83,542.326 L1334.42,542.934 L1335.2,543.065 L1336.11,543 L1336.97,543.207 L1337.65,544.179 L1337.17,544.342 L1337.28,544.933 L1337.75,545.553 L1338.31,545.802 L1338.89,545.738 L1339.16,545.808 L1339.47,546.001 L1339.66,546.275 L1340.03,547.036 L1340.15,547.183 L1340.4,547.344 L1340.92,548.155 L1341.21,548.309 L1341.43,548.102 L1341.44,547.758 L1341.34,547.475 L1341.24,547.475 L1341.29,546.695 L1341.23,546.108 L1341.34,545.853 L1341.93,546.07 L1342.63,546.799 L1343.43,547.841 L1344.01,548.29 L1344.05,546.71 L1344.29,546.662 L1344.58,546.869 L1344.74,547.113 L1345,548.546 L1345.44,549.213 L1345.89,549.587 L1346.37,549.694 L1346.9,549.567 L1346.63,550.029 L1346.32,550.413 L1345.95,550.598 L1345.53,550.44 L1344.62,549.298 L1344.37,549.122 L1343.28,550.508 L1342.62,551.102 L1342.18,550.561 L1342.03,550.148 L1341.81,549.971 L1341.6,550.013 L1341.46,550.253 L1341.46,550.644 L1341.64,550.769 L1342.03,550.803 L1342.73,551.805 L1343.14,552.062 L1343.4,551.64 L1343.66,551.379 L1344.72,551.053 L1345.09,551.121 L1345.23,551.37 L1345.62,552.354 L1345.76,552.779 L1345.61,553.776 L1345.78,554.433 L1346.18,554.778 L1346.7,554.825 L1346.92,554.588 L1347.06,553.668 L1347.2,553.443 L1347.78,553.493 L1348.38,553.68 L1348.05,552.83 L1348.41,552.769 L1348.78,553.024 L1349.09,553.575 L1349.27,554.395 L1349.21,555.024 L1348.92,555.158 L1348.56,555.218 L1348.27,555.646 L1348.04,556.109 L1347.45,556.628 L1347.21,557.117 L1346.17,560.017 L1345.28,561.338 L1344.91,562.239 L1344.36,562.928 L1343.67,562.427 L1343.01,561.542 L1342.52,561.097 L1342.73,561.627 L1342.86,562.146 L1342.91,562.675 L1342.9,563.235 L1343.1,563.397 L1343.21,563.587 L1343.3,563.808 L1343.44,564.053 L1343.03,564.305 L1342.36,565.032 L1341.91,565.163 L1341.48,564.998 L1341.28,564.807 L1341.13,564.932 L1338.62,570.775 L1338.53,570.897 L1337.65,570.656 L1337.53,570.587 L1337.35,570.94 L1336.98,572.909 L1335.31,575.842 L1335.02,577.109 L1335.54,578.117 L1335.55,579.257 L1334.87,581.757 L1334.85,582.016 L1334.88,582.348 L1334.91,583.134 L1334.95,583.001 L1335.05,583.211 L1335.15,583.551 L1335.21,583.774 L1335.19,583.977 L1335.09,584.031 L1334.98,584.128 L1334.96,584.456 L1335.13,585.701 L1335.15,586.345 L1335.02,586.648 L1334.73,586.829 L1334.56,587.154 L1334.48,587.463 L1334.42,587.607 L1333.89,588.079 L1333.59,589.055 L1333.4,591.22 L1333.23,593.293 L1333.07,593.901 L1331.96,595.326 L1330.75,595.583 L1330.53,595.979 L1330.54,596.839 L1330.82,597.207 L1331.25,597.241 L1331.68,597.098 L1331.33,598.636 L1331.08,599.275 L1330.77,599.198 L1329.95,597.851 L1329.53,597.514 L1329.03,597.505 L1327.97,599.177 L1328.01,599.998 L1327.87,600.537 L1327.59,600.688 L1327.2,600.373 L1327.37,599.687 L1327.22,599.136 L1326.88,598.682 L1326.16,598.017 L1326.04,597.974 L1325.23,601.243 L1324.82,602.002 L1324.11,601.704 L1324.35,602.188 L1324.38,602.507 L1324.34,602.836 L1324.38,603.385 L1324.79,604.704 L1324.87,605.079 L1324.81,605.796 L1324.67,606.53 L1324.44,607.117 L1324.17,607.377 L1323.63,607.261 L1323.27,606.783 L1323.11,606.033 L1323.21,605.093 L1323.1,604.513 L1322.71,603.736 L1322.23,603.086 L1321.82,602.905 L1321.71,603.8 L1322.36,607.559 L1322.32,609.122 L1321.99,609.305 L1321.48,608.938 L1321.04,608.286 L1320.92,607.625 L1321.04,606.891 L1321.1,605.998 L1321.08,605.112 L1320.9,604.376 L1320.49,605.266 L1320.18,606.669 L1320.14,608.231 L1320.55,609.604 L1319.88,609.544 L1318.08,609.967 L1317.82,610.154 L1317.54,610.465 L1317.23,610.62 L1316.89,610.357 L1316.6,609.777 L1316.45,609.529 L1316.25,609.371 L1315.72,609.34 L1315.19,609.596 L1314.73,610.079 L1314.42,610.715 L1314.3,611.574 L1314.48,612.258 L1315.14,613.448 L1312.82,612.723 L1312.41,612.745 L1312.07,613.266 L1311.16,613.981 L1311,614.258 L1310.82,615.917 L1309.97,617.148 L1308.95,618.211 L1308.18,619.371 L1308.2,620.006 L1308.91,621.585 L1309.28,622.986 L1309.21,623.236 L1308.58,623.546 L1308.36,623.849 L1308.07,624.397 L1307.53,625.176 L1307.21,625.371 L1305.75,624.72 L1305.54,623.851 L1305.55,623.178 L1305.38,622.663 L1304.64,622.293 L1304.67,622.045 L1304.67,621.612 L1304.68,621.389 L1304.27,621.194 L1303.36,620.24 L1301.82,619.864 L1301.38,619.478 L1300.67,618.615 L1299.98,618.164 L1299.32,618.292 L1298.71,619.151 L1299.07,619.464 L1299.43,619.662 L1300.18,619.852 L1300.02,621.469 L1300.44,621.92 L1301.22,622.034 L1302.14,622.636 L1301.45,623.305 L1301.09,623.537 L1300.64,623.663 L1300.57,623.916 L1300.92,624.406 L1302.3,625.925 L1302.46,626.348 L1302.6,627.055 L1302.39,627.267 L1301.79,627.012 L1300.72,626.273 L1298,623.595 L1297.2,623.301 L1297.25,623.818 L1298.31,625.738 L1297.76,626.119 L1297.11,625.683 L1296.46,625.007 L1295.9,624.696 L1295.62,625.183 L1295.63,626.234 L1295.96,628.401 L1296.01,628.999 L1296.02,630.021 L1296.07,630.576 L1296.34,632.456 L1296.17,633.324 L1295.84,633.791 L1295.48,633.629 L1295.21,632.606 L1294.74,633.698 L1293.38,634.324 L1292.81,635.075 L1293.01,635.219 L1293.51,635.883 L1293.24,636.334 L1292.92,637.842 L1292.7,638.568 L1293.29,638.491 L1292.98,639.568 L1292.42,639.564 L1291.1,638.31 L1290.8,639.706 L1291.37,640.416 L1292.28,640.654 L1292.99,640.642 L1292.87,641.396 L1292.56,641.513 L1292.2,641.505 L1291.9,641.882 L1291.8,642.53 L1291.8,643.043 L1291.68,643.377 L1291.24,643.493 L1291.08,643.255 L1290.14,642.27 L1289.96,642.133 L1289.68,641.772 L1289.03,641.233 L1288.71,640.806 L1288.73,639.975 L1288.35,639.588 L1287.82,639.36 L1287.38,639.024 L1286.81,638.448 L1286.18,638.185 L1285.84,638.439 L1286.16,639.414 L1285.81,639.657 L1284.16,639.201 L1282.98,639.244 L1282.5,639.474 L1282.11,639.925 L1282.77,640.493 L1287.47,641.829 L1287.05,642.778 L1286.21,643.25 L1284.35,643.513 L1284.56,644.109 L1284.66,644.335 L1284.28,645.005 L1283.88,645.01 L1283.45,644.511 L1283.19,643.659 L1282.98,642.674 L1283.55,642.338 L1284.95,642.149 L1284.88,641.686 L1282.11,641.252 L1281.35,641.738 L1281.79,642.095 L1282.03,642.584 L1282.18,643.164 L1282.41,643.757 L1283.25,645.258 L1283.48,645.772 L1279.23,644.952 L1277.92,645.175 L1277.97,645.596 L1278.57,645.548 L1279.27,645.8 L1279.92,646.263 L1280.39,646.821 L1280.72,647.474 L1280.98,648.208 L1280.99,648.836 L1280.6,649.138 L1279.98,648.959 L1279.35,648.565 L1278.75,648.434 L1278.21,649.003 L1278.08,649.613 L1278.05,650.328 L1277.94,651.041 L1277.58,651.661 L1278.54,651.543 L1277.43,652.815 L1277.07,653.009 L1276.94,653.368 L1276.95,654.871 L1276.76,655.238 L1276.44,655.062 L1275.29,653.698 L1274.82,653.404 L1274.36,653.3 L1273.92,653.446 L1273.53,653.91 L1274.55,654.547 L1275.11,655.018 L1275.56,655.77 L1273.64,655.614 L1273.54,655.49 L1273.21,654.863 L1273.03,654.63 L1272.77,654.505 L1272.73,654.728 L1272.76,655.069 L1272.72,655.294 L1272.77,655.383 L1272.44,656.145 L1271.8,656.689 L1270.21,657.308 L1270.98,657.815 L1278.3,656.835 L1279.11,657.113 L1279.76,657.838 L1280.06,658.727 L1279.85,659.065 L1279.41,658.806 L1279.01,657.931 L1278.8,657.957 L1278.71,659.181 L1278.09,659.546 L1277.26,659.401 L1275.46,658.6 L1275.08,658.928 L1274.93,660.18 L1276.4,661.075 L1276.86,661.235 L1277.84,660.708 L1278.4,660.564 L1278.8,660.998 L1278.67,662.312 L1277.55,662.603 L1275.41,662.269 L1275.95,663.08 L1277.41,663.506 L1277.81,664.131 L1277.79,664.708 L1277.6,665.147 L1277.31,665.436 L1276.64,665.709 L1276.41,665.982 L1276.22,666.304 L1275.97,666.578 L1275.37,666.952 L1275.05,667.021 L1274.48,666.875 L1274.01,666.904 L1273.81,666.837 L1273.69,666.654 L1273.59,666.374 L1273.48,665.979 L1272.4,664.819 L1271.88,664.427 L1271.31,664.478 L1271.41,664.715 L1271.62,665.34 L1270.24,665.503 L1270.29,665.889 L1270.63,666.094 L1270.75,666.523 L1270.69,667.058 L1270.51,667.616 L1271.19,667.902 L1271.98,666.629 L1272.62,666.978 L1272.18,668.975 L1271.84,669.745 L1271.32,670.097 L1271.09,670.393 L1270.88,670.93 L1270.62,671.248 L1270.23,670.909 L1270.17,670.509 L1270.26,670.01 L1270.42,669.571 L1270.54,669.332 L1270.1,669.153 L1269.89,669.654 L1269.77,670.464 L1269.57,671.201 L1269.1,671.771 L1268.53,671.921 L1267.28,671.859 L1267.34,671.261 L1267.39,671.025 L1266.87,670.648 L1266.29,670.594 L1265.16,670.813 L1265.75,669.671 L1268.29,669.733 L1268.87,668.632 L1267.6,668.232 L1266.99,667.901 L1266.52,667.226 L1266.57,667.087 L1266.59,667.013 L1266.6,666.94 L1266.65,666.782 L1266.02,667.038 L1265.13,667.87 L1264.34,668.925 L1264.06,669.865 L1263.88,670.115 L1263.4,669.971 L1262.88,669.648 L1262.59,669.354 L1262.46,668.811 L1262.26,666.619 L1262.18,666.441 L1261.84,665.914 L1261.74,665.586 L1261.75,665.27 L1261.96,664.709 L1262,664.5 L1261.74,663.575 L1261.4,663.612 L1260.99,663.967 L1260.56,664.001 L1260.32,663.565 L1259.54,661.346 L1259.23,660.849 L1258.89,660.468 L1258.51,660.188 L1258.09,659.99 L1258.26,659.687 L1258.53,658.651 L1257.91,658.813 L1257.13,659.844 L1256.73,660.142 L1256.17,659.935 L1255.63,659.44 L1255.13,659.129 L1254.68,659.471 L1258.58,662.259 L1259.6,663.467 L1260.26,664.61 L1261.18,666.501 L1261.76,668.377 L1261.42,669.487 L1261.48,669.946 L1262.83,670.693 L1263.21,671.036 L1263.62,671.519 L1263.76,671.718 L1263.85,672.915 L1264.15,673.584 L1265.28,674.99 L1265.63,676.129 L1265.65,676.829 L1265.51,677.914 L1265.59,678.299 L1265.81,678.882 L1265.74,679.202 L1265.49,679.356 L1265.23,679.432 L1265.01,679.334 L1264.28,678.819 L1263.94,678.719 L1262.16,678.92 L1262.83,679.651 L1267.43,681.13 L1267.54,681.808 L1267.71,682.326 L1267.94,682.698 L1268.27,682.945 L1267.67,683.244 L1265.77,685.449 L1265.57,685.275 L1265.43,684.513 L1264.98,682.696 L1264.95,682.319 L1264.84,682.139 L1264.45,682.132 L1264.29,682.355 L1264.21,682.845 L1264.13,684.834 L1264.15,685.354 L1264.19,685.812 L1264.15,686.331 L1263.93,686.586 L1263.33,686.946 L1263.84,687.742 L1263.14,689.499 L1262.69,690.249 L1262.15,690.552 L1261.8,690.363 L1261.21,689.587 L1260.92,689.436 L1259.43,689.601 L1258.29,690.23 L1257.72,690.815 L1257.44,691.538 L1257.59,692.593 L1258.11,692.918 L1258.72,693.147 L1259.14,693.926 L1255.33,694.161 L1254.88,693.995 L1254.51,693.749 L1253.87,692.913 L1253.7,692.801 L1252.12,690.396 L1250.64,688.606 L1250.28,688.372 L1249.73,688.777 L1249.59,690.317 L1249.14,690.712 L1248.57,690.884 L1248.12,691.3 L1247.88,691.976 L1247.99,692.939 L1249.52,694.963 L1249.61,695.176 L1249.72,695.675 L1249.8,696.23 L1249.8,696.607 L1249.62,696.913 L1249.3,697.09 L1248.94,697.168 L1248.66,697.157 L1249.03,697.491 L1249.43,697.978 L1249.68,698.624 L1249.61,699.434 L1249.65,700.086 L1249.97,700.842 L1250.41,701.451 L1250.79,701.686 L1251.68,701.375 L1252.09,701.424 L1252.63,701.921 L1252.92,702.448 L1253.19,703.086 L1253.49,703.659 L1253.87,703.977 L1253.91,704.359 L1253.64,704.37 L1253.47,704.491 L1253.15,704.908 L1255.83,706.124 L1256.39,705.762 L1257.37,705.927 L1258.39,706.376 L1259.05,706.849 L1256.82,708.804 L1256.11,709.078 L1256,708.791 L1256.63,707.112 L1256.11,706.838 L1255.59,707.034 L1255.18,707.523 L1254.95,708.147 L1254.92,709.075 L1255.07,710.079 L1255.25,710.962 L1255.33,711.539 L1255.22,712.377 L1255.15,712.576 L1254.78,713.692 L1254.67,714.574 L1254.75,715.808 L1254.86,716.561 L1254.75,717.093 L1254.21,717.664 L1254.89,717.971 L1254.7,718.84 L1253.89,720.315 L1254.27,720.589 L1254.77,721.119 L1255.06,721.65 L1254.8,721.933 L1253.42,722.31 L1253.13,722.54 L1253.03,723.199 L1253.18,723.89 L1253.16,724.449 L1252.55,724.743 L1252.29,724.96 L1252.14,725.46 L1252.11,726.149 L1252.19,726.928 L1252.39,727.697 L1252.92,728.749 L1253.05,729.215 L1253.01,729.634 L1252.63,730.001 L1252.57,730.317 L1252.54,730.213 L1252.56,730.683 L1252.62,731.186 L1252.67,731.168 L1252.75,731.205 L1252.91,731.352 L1253.01,731.669 L1252.89,732.234 L1252.77,732.535 L1252.69,732.853 L1252.69,733.188 L1252.83,733.526 L1253.23,733.737 L1254.72,733.136 L1254.64,733.742 L1254.46,734.161 L1254.19,734.411 L1253.86,734.515 L1253.91,734.978 L1254.44,735.226 L1254.58,735.916 L1254.58,736.855 L1254.66,737.859 L1254.94,738.98 L1255.22,739.289 L1256.63,738.753 L1257.11,738.385 L1257.62,738.267 L1258.24,738.768 L1258.69,739.457 L1258.95,740.148 L1259.02,740.841 L1258.86,741.548 L1258.77,742.387 L1258.79,743.466 L1258.72,744.403 L1258.32,744.844 L1257.72,744.852 L1257.24,744.698 L1256.8,744.336 L1256.34,743.727 L1254.96,741.18 L1254.22,740.225 L1253.21,739.767 L1253.36,740.573 L1253.58,741.172 L1254.1,742.247 L1252.95,742.066 L1252.5,741.612 L1252.49,740.661 L1251.29,739.745 L1250,739.526 L1247.45,739.893 L1247.9,740.554 L1248.38,740.75 L1249.39,740.593 L1251.56,741.652 L1251.42,742.716 L1250.57,743.498 L1249.51,743.962 L1248.75,744.086 L1248.8,744.51 L1249.03,744.548 L1249.66,744.895 L1248.85,745.349 L1247.42,744.904 L1246.84,745.571 L1250.75,746.058 L1251.51,746.809 L1250.89,747.248 L1250.36,747.195 L1249.21,746.615 L1248.55,746.542 L1246.57,746.884 L1246.61,747.308 L1247.67,747.148 L1248.16,747.333 L1248.56,747.968 L1247.35,748.729 L1245.98,749.046 L1246.83,749.646 L1250.01,749.967 L1250.05,750.392 L1247.17,750.567 L1246.9,750.944 L1247.2,751.236 L1247.9,751.411 L1248.51,751.898 L1249.29,752.018 L1249.63,752.148 L1249.68,752.611 L1248.43,752.737 L1248.48,753.124 L1249.71,753.576 L1250,753.866 L1250.22,754.603 L1250.1,755.021 L1249.84,755.414 L1249.62,756.05 L1249.84,756.564 L1250.02,757.067 L1250.3,757.471 L1250.64,757.697 L1251.02,757.655 L1251.17,757.32 L1251.29,756.922 L1251.56,756.711 L1251.78,756.887 L1251.86,757.318 L1251.78,757.772 L1251.5,758 L1251.16,758.088 L1250.85,758.281 L1250.59,758.589 L1250.37,759.015 L1250.1,758.557 L1249.69,758.008 L1249.35,757.821 L1249.21,759.016 L1248.69,759.628 L1248.62,760.263 L1248.82,760.604 L1249.85,762.029 L1250.81,762.78 L1251.85,763.153 L1251.98,763.833 L1254.36,765.037 L1253.49,765.363 L1252.5,765.268 L1250.54,764.533 L1251.1,765.725 L1251.97,766.212 L1252.92,766.422 L1253.7,766.786 L1252.4,766.842 L1251.85,766.73 L1250.11,765.343 L1249.59,765.173 L1248.94,765.162 L1248.99,765.587 L1249.75,765.878 L1250.76,766.673 L1251.74,767.695 L1252.45,768.665 L1252.99,770.126 L1252.59,770.733 L1251.84,771.203 L1251.37,772.248 L1252.5,772.953 L1252.53,774.531 L1252.36,775.069 L1251.94,775.621 L1252.65,776.403 L1252.85,777.183 L1252.56,777.726 L1251.78,777.826 L1252.14,777.976 L1252.96,777.905 L1253.27,778.06 L1253.45,778.412 L1253.55,778.829 L1253.51,779.178 L1253.3,779.346 L1253.07,779.646 L1252.92,780.339 L1252.85,781.199 L1252.86,782.001 L1252.99,782.243 L1253.26,782.549 L1253.57,782.788 L1253,783.51 L1252.81,783.683 L1254.39,783.27 L1255.14,783.292 L1255.57,783.872 L1253.81,784.54 L1253.27,785.375 L1253.18,787.12 L1253.23,787.88 L1253.28,788.123 L1254.01,788.752 L1254.55,788.761 L1254.8,788.881 L1254.8,789.658 L1254.93,790.075 L1255.28,790.803 L1255.39,790.992 L1255.84,790.615 L1256.13,790.717 L1256.3,790.983 L1257.78,794.413 L1257.54,794.517 L1257.29,794.717 L1257.1,795.003 L1257.06,795.344 L1257.25,796.061 L1257.48,796.068 L1257.72,795.915 L1257.97,796.145 L1258.14,796.954 L1258.13,797.6 L1258.21,798.131 L1258.68,798.609 L1259.28,798.739 L1259.74,798.652 L1259.92,798.849 L1259.65,799.792 L1259.33,800.271 L1258.4,800.933 L1257.68,802.027 L1256.63,803.002 L1255.94,803.357 L1255.82,803.66 L1255.86,803.952 L1256.08,804.068 L1256.44,803.958 L1257.09,803.569 L1257.4,803.46 L1258.03,803.602 L1258.75,804.442 L1259.78,804.764 L1261.61,806.07 L1260.45,807.046 L1260.5,807.476 L1261.02,807.782 L1261.65,807.81 L1262.15,807.607 L1262.26,807.563 L1262.68,807.027 L1263.2,806.615 L1267.21,805.483 L1268.07,805.47 L1268.71,805.938 L1269.22,806.444 L1270.44,807.056 L1270.92,807.651 L1270.61,808.235 L1269.6,809.039 L1269.4,809.336 L1269.6,810.234 L1270.21,811.581 L1270.39,812.237 L1270.95,813.49 L1272.03,814.087 L1274.25,814.199 L1274.2,813.739 L1273.54,811.993 L1274.42,810.1 L1275.91,808.552 L1278.95,807.275 L1279.88,807.232 L1280.77,807.846 L1281.65,809.935 L1281.69,810.695 L1281.58,811.347 L1281.36,811.838 L1281.05,812.15 L1281.1,812.575 L1281.91,812.558 L1282.74,813.009 L1284.33,814.356 L1285.48,815.046 L1285.66,815.403 L1285.97,816.399 L1286.11,816.734 L1287.26,818.075 L1288.46,818.858 L1291.16,819.583 L1290.68,819.985 L1290,820.898 L1289.67,821.48 L1289.73,821.942 L1290.23,822.422 L1290.47,822.723 L1290.74,823.115 L1291.18,822.626 L1291.89,821.579 L1292.43,821.156 L1292.94,820.978 L1293.35,821.017 L1294.4,821.392 L1296.54,821.063 L1297.21,821.449 L1297.2,821.64 L1297.24,822.148 L1297.32,822.739 L1297.43,823.182 L1297.58,823.366 L1298.36,823.931 L1298.42,824.355 L1298.05,824.507 L1297.66,824.48 L1297.29,824.546 L1296.99,824.952 L1301.5,826.101 L1302.88,827.255 L1301.7,828.077 L1296.39,826.7 L1295.27,825.611 L1293.33,823.198 L1294.03,825.335 L1295.45,826.874 L1297.03,828.167 L1298.23,829.536 L1297.71,829.645 L1297.25,829.359 L1296.36,828.432 L1296.77,829.916 L1297.51,830.73 L1299.14,831.616 L1300.03,832.489 L1300.48,832.753 L1301,832.684 L1301.35,832.45 L1301.59,832.082 L1301.65,831.567 L1301.65,831.452 L1301.88,833.332 L1302.06,834.241 L1302.26,834.702 L1303.11,836.691 L1303.37,837.56 L1303.56,837.536 L1303.61,836.589 L1303.49,835.378 L1303.22,834.2 L1302.82,833.325 L1303.43,832.866 L1304.23,832.607 L1305.05,832.668 L1306.26,833.62 L1307.38,833.781 L1307.92,834.422 L1308.2,835.408 L1308.43,837.556 L1308.76,838.423 L1310.38,840.248 L1310.73,841 L1310.05,839.454 L1309.25,837.967 L1308.74,836.378 L1308.91,834.536 L1309.99,833.734 L1310.44,833.934 L1310.58,834.162 L1310.62,835.158 L1310.82,836.193 L1311.08,836.961 L1313.05,841.417 L1314.52,844.165 L1315.23,844.89 L1316.65,845.181 L1317.43,845.554 L1318.02,845.954 L1318.16,846.087 L1318.2,846.239 L1318.23,846.751 L1318.27,846.9 L1318.84,847.572 L1319.1,847.728 L1319.46,847.647 L1319.36,847.356 L1319.17,846.677 L1319.07,846.405 L1319.64,846.447 L1320.21,846.651 L1320.23,847.558 L1320.51,850.042 L1320.62,850.55 L1319.57,850.6 L1319.11,850.343 L1316.35,846.914 L1315.17,845.719 L1314.13,845.308 L1314.56,846.062 L1315.9,847.885 L1316.18,848.922 L1315.96,849.971 L1314.41,853.273 L1313.9,853.809 L1313.22,854.077 L1311.84,854.175 L1311.15,854.429 L1310.72,855.011 L1310.69,855.898 L1311.25,855.823 L1312.36,855.044 L1320.22,853.675 L1321.68,854.293 L1321.53,855.732 L1321.12,856.64 L1320.5,857.202 L1318.67,858.031 L1318.09,858.095 L1317.57,857.872 L1317.2,857.441 L1316.74,856.552 L1316.46,856.249 L1316.22,857.356 L1315.66,858.254 L1314.96,858.879 L1312.72,859.885 L1312.07,860.628 L1312.11,861.531 L1311,861.728 L1310.83,861.904 L1310.86,862.461 L1310.79,862.906 L1310.62,863.234 L1310.38,863.467 L1311.91,863.278 L1311.77,863.79 L1311.47,864.41 L1311.11,864.939 L1310.82,865.175 L1310.48,865.038 L1310.17,864.709 L1309.84,864.427 L1309.43,864.445 L1310.2,865.543 L1310.51,866.199 L1310.25,866.539 L1309.87,866.785 L1309.22,867.753 L1307.95,868.514 L1306.38,870.482 L1304.68,871.407 L1303.95,872.28 L1302.77,874.37 L1301.83,875.405 L1300.75,875.74 L1298.31,875.753 L1298.36,876.177 L1299.55,876.897 L1299.16,877.345 L1298.29,877.33 L1297.87,877.527 L1298.49,877.791 L1299.06,878.249 L1298.98,879.133 L1299.56,879.601 L1300.38,879.776 L1301.02,879.777 L1300.14,880.619 L1298.24,880.786 L1296.29,880.4 L1295.25,879.59 L1295.58,879.306 L1295.95,879.139 L1296.36,879.149 L1296.77,879.414 L1296.6,878.903 L1296.31,877.66 L1296.09,877.301 L1295.68,877.272 L1295.06,878.048 L1294.65,878.296 L1294.39,878.188 L1293.95,877.656 L1293.7,877.576 L1293.55,877.741 L1293.23,878.336 L1293.05,878.48 L1292.75,878.821 L1292.83,879.514 L1293.12,880.23 L1293.42,880.662 L1292.46,881.487 L1292.1,881.635 L1291.73,881.557 L1290.97,881.095 L1290.58,880.986 L1295.45,884.729 L1294.35,885.123 L1293.84,885.03 L1293.93,885.75 L1296.4,885.481 L1297.1,884.92 L1297.44,884.975 L1297.77,885.75 L1297.97,885.727 L1298.73,882.928 L1300.65,881.875 L1302.97,882.008 L1304.94,882.754 L1304.78,882.972 L1304.52,883.47 L1304.38,883.682 L1305.37,883.872 L1305.78,883.712 L1306.07,883.007 L1304.95,881.008 L1304.61,880.174 L1306.13,879.992 L1304.76,879.261 L1304.1,878.734 L1303.63,877.709 L1305.48,877.626 L1306.36,877.902 L1307.27,878.561 L1307.35,878.77 L1307.58,879.578 L1307.67,879.806 L1310,880.816 L1309.74,881.633 L1309.19,881.702 L1308.51,881.505 L1307.88,881.502 L1308.66,881.839 L1309.8,882.049 L1310.87,882.478 L1311.44,883.457 L1311.32,884.429 L1310.77,884.925 L1309.12,885.225 L1308.63,885.431 L1308.16,885.758 L1307.65,885.934 L1307.08,885.708 L1306.64,885.453 L1305.69,885.277 L1305.26,885.298 L1302.65,886.008 L1301.76,885.714 L1301.79,885.418 L1301.79,884.667 L1301.82,884.376 L1301.18,884.428 L1300.66,884.754 L1300.2,885.282 L1299.79,885.946 L1300.23,886.186 L1300.65,886.576 L1301.01,887.033 L1301.3,887.491 L1301.35,887.915 L1300.27,888.027 L1299.99,888.36 L1300.29,889.115 L1302.28,889.958 L1303.06,890.107 L1304,890.657 L1304.89,891.513 L1305.49,892.588 L1302.28,890.506 L1301.59,890.703 L1301.68,891.447 L1302.25,891.849 L1302.95,892.002 L1303.43,892.01 L1303.7,892.38 L1303.97,893.123 L1304.33,893.593 L1304.89,893.128 L1304.81,893.384 L1304.77,893.597 L1304.7,893.8 L1304.56,894.027 L1304.62,894.454 L1305.41,893.967 L1305.76,893.956 L1305.97,894.523 L1305.98,895.262 L1305.82,895.883 L1305.53,896.163 L1305.13,895.877 L1304.42,895.808 L1303.26,896.325 L1302.19,897.11 L1301.71,897.843 L1302.06,897.697 L1302.39,897.736 L1302.68,897.989 L1302.9,898.489 L1300.07,900.319 L1299.15,900.649 L1299.53,900.233 L1300.36,899.583 L1300.74,899.172 L1300.25,898.798 L1300.44,898.553 L1300.64,898.36 L1300.6,897.974 L1300.06,897.977 L1299.59,898.155 L1299.18,898.546 L1298.86,899.175 L1298.51,899.554 L1297.45,899.799 L1297.06,900.068 L1296.83,900.52 L1296.33,902.09 L1295.96,902.429 L1294.25,902.537 L1294.35,902.833 L1294.54,903.544 L1294.64,903.825 L1293.79,903.494 L1293.41,903.565 L1293.11,903.999 L1293.73,904.915 L1293.36,906.021 L1292.68,907.323 L1292.32,908.819 L1292.46,909.541 L1292.7,909.996 L1292.83,910.492 L1292.62,911.363 L1292.26,911.893 L1291.73,912.247 L1291.16,912.325 L1290.69,912.008 L1290.61,912.793 L1290.64,913.645 L1290.87,915.432 L1290.4,915.076 L1289.99,914.522 L1289.78,913.803 L1289.89,912.957 L1289.56,912.179 L1289.43,911.936 L1289.08,911.615 L1288.84,911.652 L1288.64,911.804 L1288.42,911.832 L1287.86,911.535 L1287.51,911.214 L1286.76,910.06 L1286.31,909.662 L1286.47,910.492 L1287,911.799 L1287.64,912.814 L1287.4,913.938 L1287.26,914.37 L1287.05,914.562 L1286.86,914.289 L1286.82,913.62 L1286.86,912.89 L1286.93,912.424 L1285.78,912.04 L1285.57,910.88 L1285.58,909.465 L1285.1,908.328 L1285.05,907.865 L1285.4,906.886 L1285.35,904.961 L1285.12,903.085 L1284.96,902.249 L1284.85,901.982 L1285.1,900.459 L1285.01,899.739 L1284.63,899.442 L1284.22,899.687 L1283.9,900.245 L1283.81,900.869 L1283.98,902.81 L1283.92,903.734 L1283.62,904.156 L1283.29,903.909 L1281.99,902.573 L1281.98,901.65 L1282.46,900.178 L1282.52,899.327 L1282.32,898.631 L1281.96,898.005 L1281.54,897.49 L1281.15,897.117 L1281.84,899.19 L1281.56,900.173 L1281.21,900.939 L1280.74,901.525 L1280.14,901.951 L1280.59,903.239 L1280.6,904.585 L1280.68,905.775 L1281.33,906.586 L1281.11,907.601 L1281.03,908.974 L1280.91,909.915 L1280.56,909.642 L1280.25,910.047 L1280.24,910.053 L1280.11,910.305 L1280,910.596 L1280.29,910.923 L1280.58,911.153 L1280.89,911.285 L1281.2,911.328 L1281.24,911.753 L1280.99,912.309 L1281.42,912.501 L1281.88,912.843 L1281.7,913.853 L1281.15,914.412 L1280.46,914.393 L1279.74,914.17 L1279.06,914.136 L1279.46,914.874 L1279.97,915.404 L1281.07,916.111 L1281.11,916.498 L1280.43,916.571 L1280.68,916.828 L1280.94,916.985 L1280.03,917.047 L1277.37,915.316 L1276.24,914.863 L1276.29,915.289 L1276.91,915.656 L1277.12,916.331 L1277.04,918.254 L1276.87,919.317 L1276.87,919.79 L1277.13,919.966 L1277.37,920.042 L1278.23,920.669 L1277.36,921.276 L1273.3,920.755 L1274.59,922.349 L1274.86,923.24 L1274.33,924.127 L1272.62,924.227 L1271.7,924.545 L1271.82,925.674 L1268.12,925.968 L1267.09,925.724 L1266.69,925.352 L1265.93,924.34 L1265.46,924.17 L1265.1,924.342 L1264.21,925.019 L1263.64,925.211 L1264.11,925.425 L1265.15,925.463 L1265.53,925.881 L1265.59,926.43 L1265.44,926.969 L1265.21,927.404 L1265.02,927.651 L1265.74,928.04 L1267.09,927.72 L1267.77,927.841 L1267.81,928.226 L1267.15,929.13 L1266.76,930.026 L1266.33,930.624 L1265.57,930.638 L1263.53,929.582 L1262.34,929.477 L1261.76,930.547 L1263.32,930.481 L1263.95,930.922 L1264.6,932.025 L1263.85,932.091 L1261.07,933.37 L1249.7,933.074 L1238.33,932.723 L1238.37,933.187 L1238.81,933.348 L1239.19,933.746 L1239.39,934.276 L1239.22,934.828 L1238.74,935.01 L1237.34,934.811 L1236.75,935.043 L1236.79,935.429 L1237.18,935.17 L1237.56,935.006 L1237.91,935.144 L1238.16,935.779 L1238.78,935.268 L1240.32,935.898 L1240.95,935.298 L1241.35,934.534 L1241.77,934.31 L1242.18,934.483 L1242.96,935.298 L1243.1,935.359 L1243.14,935.571 L1243.2,936.406 L1243.41,936.927 L1243.8,937.114 L1244.2,937.032 L1244.47,936.739 L1244.62,935.944 L1244.7,934.942 L1244.84,934.019 L1245.16,933.44 L1245.63,933.533 L1247.37,934.32 L1247.84,934.676 L1251.68,934.183 L1252.4,934.396 L1253.08,935.069 L1253.56,935.346 L1254.91,935.418 L1255.54,935.946 L1255.45,935.645 L1255.29,934.952 L1255.19,934.654 L1256.38,934.94 L1256.79,934.932 L1256.59,936.112 L1256.9,937.086 L1257.56,937.692 L1258.43,937.778 L1258.16,938.172 L1258.03,938.284 L1258.06,938.632 L1258.95,938.546 L1259,939.012 L1258.74,939.626 L1259.18,939.908 L1260.47,940.154 L1262.44,941.681 L1261.71,942.234 L1260.07,942.419 L1259.39,942.835 L1259.77,943.704 L1259.1,944.459 L1258.03,944.999 L1257.25,945.227 L1257.05,945.374 L1256.72,945.984 L1256.56,946.152 L1256.31,946.137 L1255.84,945.852 L1255.6,945.773 L1251.51,945.393 L1249.49,944.353 L1246.17,943.536 L1245.28,944.151 L1246.46,944.213 L1253.74,947.314 L1254.83,947.339 L1255.29,947.613 L1255.67,948.377 L1255.73,948.952 L1255.67,949.483 L1255.66,950.003 L1255.88,950.544 L1256.16,950.482 L1256.65,950.173 L1257.05,949.812 L1257.1,949.574 L1257.53,949.858 L1257.72,950.186 L1257.99,951.633 L1253.19,951.188 L1253.8,951.804 L1255.09,952.649 L1255.65,953.145 L1255.59,953.815 L1255.71,954.236 L1256.13,954.383 L1256.49,954.278 L1256.63,954.028 L1256.58,953.632 L1256.32,953.081 L1258.21,952.497 L1258.88,952.556 L1258.67,953.712 L1258.5,953.818 L1257.94,953.756 L1257.75,953.8 L1257.74,953.998 L1257.76,954.379 L1257.76,954.794 L1257.65,955.096 L1257.29,955.368 L1256.12,955.67 L1256.16,956.099 L1256.93,956.208 L1259.49,957.46 L1259.21,957.856 L1259.09,957.967 L1259.57,958.313 L1260,958.939 L1260.33,959.744 L1260.51,960.622 L1260.53,961.451 L1260.38,961.62 L1260.07,961.48 L1259.66,961.384 L1259.41,961.581 L1259.56,961.966 L1259.9,962.364 L1260.23,962.583 L1259.35,964.652 L1257.39,963.623 L1255.21,961.585 L1253.64,960.627 L1254.23,961.664 L1255.66,962.816 L1256.24,963.854 L1256.02,963.852 L1255.58,963.947 L1255.36,963.937 L1256.21,965.093 L1257.5,966.486 L1258.28,967.63 L1257.61,968.02 L1257.78,968.808 L1257.64,969.278 L1257.29,969.473 L1256.79,969.42 L1257.54,969.894 L1258.14,970.09 L1258.69,970.42 L1259.26,971.297 L1259.04,971.845 L1258.77,972.019 L1258.45,971.857 L1258.13,971.404 L1257.23,972.125 L1257.11,972.521 L1257.61,973.166 L1256.88,972.77 L1255.99,970.952 L1255.53,970.788 L1255.15,971.499 L1255.43,972.317 L1256.32,973.759 L1256.71,973.625 L1257.21,973.675 L1257.23,973.734 L1257.7,974.859 L1258.04,975.364 L1258.39,975.048 L1258.68,974.552 L1258.94,974.662 L1259.16,975.259 L1259.31,976.245 L1259.53,976.587 L1260.52,977.562 L1260.86,978.052 L1260.29,978.098 L1258.94,977.374 L1259.33,978.282 L1260.39,979.12 L1260.57,979.793 L1260.28,980.501 L1259.72,980.545 L1259.09,980.208 L1258.62,979.763 L1257.82,978.271 L1257.53,978.129 L1257.54,979.417 L1257.29,979.878 L1256.62,979.976 L1255.27,979.862 L1255.33,979.302 L1255.27,978.767 L1255.11,978.25 L1254.84,977.759 L1255.21,977.673 L1255.57,977.716 L1255.92,977.866 L1256.25,978.096 L1251.49,973.664 L1251.18,973.73 L1250.96,974.08 L1250.89,974.65 L1251.02,975.067 L1251.3,975.484 L1252.33,976.64 L1254.35,977.375 L1254.19,977.579 L1253.91,978.121 L1253.75,978.327 L1254.02,978.934 L1254.11,979.514 L1254.01,980.025 L1253.72,980.435 L1253.04,979.802 L1252.35,979.4 L1250.84,978.986 L1251.37,979.756 L1251.17,980.239 L1250.92,980.037 L1250.23,979.859 L1250.28,980.32 L1258.22,987.772 L1259.35,988.536 L1259.16,988.98 L1258.57,988.818 L1256.87,987.822 L1256.35,987.333 L1253.32,983.541 L1248.92,980.443 L1249.29,981.527 L1249.74,982.21 L1250.99,983.26 L1253.01,984.434 L1254.63,986.165 L1254.93,986.666 L1255.18,987.15 L1255.47,987.499 L1255.91,987.609 L1256.3,987.818 L1257.57,988.974 L1258.04,989.515 L1257.65,990.02 L1258.35,989.944 L1258.97,990.142 L1259.49,990.651 L1259.91,991.489 L1259.18,991.557 L1259.23,991.983 L1259.68,991.94 L1259.73,992.404 L1257.44,992.38 L1256.39,992.076 L1255.26,991.492 L1255.55,992.049 L1256.24,992.699 L1256.55,993.09 L1256.96,994.039 L1256.93,994.568 L1256.51,994.7 L1255.76,994.447 L1256.11,995.059 L1256.5,995.184 L1256.91,995.223 L1257.27,995.596 L1257.54,996.346 L1257.42,996.641 L1257.11,996.808 L1256.83,997.162 L1256.56,997.619 L1255.71,998.615 L1255.48,998.808 L1255.27,998.582 L1255.14,998.008 L1255.08,997.311 L1255.05,996.699 L1254.8,997.031 L1254.52,997.155 L1254.2,997.076 L1253.87,996.808 L1254.27,997.505 L1254.45,997.738 L1254.71,997.976 L1254.28,998.762 L1254.07,998.938 L1254.11,999.364 L1254.88,999.723 L1254.92,1000.15 L1254,1000.23 L1254.04,1000.62 L1256.65,1000.47 L1257.93,1000.85 L1259.13,1003.34 L1260.05,1004.88 L1260.45,1005.97 L1259.43,1005.78 L1259.96,1006.77 L1259.8,1007.06 L1259.32,1007.15 L1258.89,1007.54 L1258.79,1008.3 L1259.04,1008.93 L1259.56,1009.37 L1260.24,1009.52 L1259.72,1009.96 L1259.1,1010.21 L1258.76,1010.7 L1259.07,1011.82 L1257.04,1011.58 L1256.23,1011.97 L1255.93,1013.39 L1257.18,1014.14 L1256.92,1014.74 L1256.59,1015.04 L1256.54,1015.32 L1257.11,1015.86 L1254.64,1017.64 L1254.21,1018.89 L1255.26,1020.83 L1254.38,1021.85 L1254.26,1023.37 L1254.64,1025.01 L1255.31,1026.37 L1256.44,1027.99 L1256.86,1028.41 L1257.24,1028.57 L1257.44,1028.47 L1257.64,1028.47 L1258.01,1028.93 L1258.73,1030.35 L1258.25,1030.75 L1258.52,1031.45 L1259.07,1032.23 L1259.44,1032.9 L1259.01,1032.97 L1257.75,1032.62 L1257.69,1032.46 L1257.71,1032.16 L1257.68,1031.89 L1257.43,1031.79 L1257.22,1031.87 L1257.11,1032.03 L1257.07,1032.18 L1257.04,1032.26 L1256.91,1032.43 L1256.86,1032.75 L1256.84,1033.09 L1256.78,1033.35 L1256.65,1033.48 L1256.52,1033.38 L1256.36,1033.24 L1255.8,1033.17 L1254.95,1032.9 L1254.52,1032.91 L1254.83,1033.35 L1255.13,1033.7 L1255.45,1033.95 L1255.78,1034.09 L1255.83,1034.55 L1255.14,1034.63 L1254.74,1035.26 L1254.67,1036.29 L1254.96,1037.6 L1255.37,1038.32 L1255.81,1038.78 L1256.08,1039.21 L1255.98,1039.9 L1255.22,1041.69 L1255.25,1042.09 L1255.54,1043.75 L1255.59,1044.44 L1255.53,1045.07 L1254.88,1048.18 L1254.9,1048.76 L1255.24,1049.43 L1255.7,1049.7 L1256.09,1049.41 L1256.45,1048.93 L1256.8,1048.67 L1257.14,1048.86 L1257.25,1049.35 L1257.21,1049.92 L1257.07,1050.32 L1256.68,1050.54 L1255.71,1050.17 L1255.18,1050.1 L1254.66,1050.31 L1254,1050.94 L1253.5,1051.11 L1253.37,1051.45 L1253.66,1052.18 L1254.29,1053.42 L1254.57,1055.41 L1254.62,1056.55 L1254.44,1057.46 L1254.04,1057.93 L1253.47,1058.18 L1252.89,1058.17 L1252.01,1057.55 L1251.58,1057.71 L1251.17,1058.05 L1250.82,1058.25 L1250.51,1058.5 L1250.25,1059.02 L1249.62,1060.85 L1249.56,1061.2 L1249.6,1061.51 L1249.62,1062.18 L1249.58,1063.52 L1249.63,1064.98 L1249.48,1065.62 L1248.51,1065.79 L1248.34,1066.42 L1248.23,1067.64 L1246.85,1071.07 L1246.27,1073.38 L1245.72,1078.66 L1245.19,1083.79 L1244.99,1084.14 L1244.62,1084.38 L1244.46,1084.91 L1244.37,1085.55 L1244.21,1086.11 L1243.69,1086.89 L1242.45,1088.27 L1241.93,1089.11 L1240.3,1093.77 L1239.6,1094.85 L1239.01,1094.8 L1238.32,1094.24 L1237.29,1093.78 L1238.16,1091.6 L1238.34,1090.37 L1237.58,1089.9 L1235.92,1090.73 L1235.03,1090.77 L1234.32,1089.73 L1233.96,1090.04 L1233.75,1089.87 L1233.54,1089.55 L1233.2,1089.39 L1232.77,1089.3 L1232.22,1088.78 L1231.95,1088.67 L1231.41,1089.27 L1230.81,1090.35 L1230.16,1090.95 L1229.53,1090.11 L1229.74,1089.71 L1230.01,1088.96 L1230.09,1088.24 L1229.72,1087.95 L1229.26,1088.15 L1228.03,1089.37 L1226.63,1090.23 L1226.43,1090.56 L1226.45,1091.54 L1226.3,1091.79 L1225.55,1091.27 L1225.27,1090.99 L1224.94,1090.58 L1224.54,1090.23 L1224,1090.1 L1223.51,1090.25 L1223.17,1090.57 L1222.65,1091.49 L1222.48,1092.11 L1222.5,1092.52 L1222.36,1092.77 L1220.81,1092.97 L1220.33,1092.88 L1219.88,1092.55 L1220.53,1092.13 L1220.8,1092.1 L1220.55,1091.51 L1219.69,1090 L1219.02,1091.12 L1217.72,1092.1 L1216.27,1092.47 L1215.15,1091.83 L1214.87,1091.59 L1214.6,1091.71 L1214.35,1091.97 L1214.08,1092.12 L1212.77,1092.21 L1212.48,1092.01 L1212.28,1090.93 L1212.03,1090.55 L1211.8,1090.61 L1210.97,1091.01 L1210.64,1091.08 L1210.79,1091.45 L1211.11,1092.42 L1211.26,1092.79 L1209.08,1091.96 L1207.84,1091.81 L1207,1092.4 L1206.24,1092.66 L1205.08,1092.52 L1204.21,1092.62 L1204.34,1093.65 L1203.68,1094 L1203.06,1093.57 L1202.4,1092.91 L1201.63,1092.55 L1199.9,1092.68 L1199.11,1093.13 L1198.62,1094.03 L1198.56,1094.56 L1198.54,1095.29 L1198.63,1095.93 L1198.88,1096.2 L1199.39,1096.36 L1199.67,1096.82 L1199.86,1097.39 L1200.08,1097.87 L1201.17,1099.3 L1201.56,1100.05 L1201.98,1101.13 L1201.59,1101.37 L1201.18,1101.77 L1200.8,1102.25 L1200.53,1102.74 L1200.27,1103.02 L1200.08,1102.77 L1199.87,1102.13 L1199.17,1102.18 L1198.96,1102.68 L1198.77,1102.94 L1198.47,1102.88 L1198.21,1102.73 L1197.93,1102.68 L1196.96,1102.75 L1196.59,1102.71 L1196.51,1102.45 L1196.97,1101.85 L1196.64,1101.51 L1196.35,1101.1 L1196.15,1100.56 L1196.11,1099.8 L1195.43,1100.74 L1195.02,1100.32 L1194.84,1100.04 L1194.92,1099.88 L1194.49,1100.07 L1193.72,1100.72 L1193.41,1100.87 L1192.48,1101.13 L1191.86,1101.72 L1190.81,1103.6 L1189.59,1105.29 L1188.86,1106.01 L1188.2,1106.33 L1187.53,1106.9 L1187.15,1108.11 L1186.87,1109.3 L1186.49,1109.86 L1185.11,1110.74 L1183.96,1112.8 L1183.19,1115.44 L1182.99,1118.04 L1182.98,1120.43 L1183.1,1121.44 L1183.52,1122.49 L1184.15,1123.19 L1185.58,1124.15 L1186.07,1124.91 L1186.19,1125.58 L1186.2,1126.24 L1186.28,1126.94 L1186.63,1127.67 L1187.02,1128.14 L1187.84,1128.89 L1189.46,1131.22 L1189.86,1132.34 L1189.79,1133.71 L1189.34,1134.56 L1187.55,1136.86 L1186.95,1137.34 L1186.55,1137.82 L1185.09,1140.88 L1184.46,1141.84 L1183.83,1142.39 L1183.04,1142.67 L1180.75,1142.79 L1179.67,1142.63 L1178.62,1142.3 L1175.7,1140.6 L1174.75,1140.28 L1173.58,1140.21 L1173.05,1140.35 L1172.17,1140.94 L1171.67,1141.17 L1171.16,1141.19 L1169.44,1140.89 L1164.73,1141.53 L1164.11,1141.74 L1163.73,1142.22 L1163.42,1142.81 L1163,1143.37 L1162.47,1143.67 L1160.93,1143.47 L1159.72,1143.83 L1156.53,1146.25 L1155.5,1146.69 L1153.05,1147.01 L1151.9,1146.89 L1145.92,1144.98 L1144,1145 L1143.28,1144.76 L1142.89,1144.72 L1142.67,1144.61 L1142.07,1144.1 L1141.76,1143.91 L1141.01,1143.74 L1140.27,1143.97 L1139.89,1144.23 L1139.39,1144.77 L1138.97,1144.88 L1138.6,1144.83 L1138.11,1144.56 L1137.74,1144.5 L1137.06,1144.81 L1136.37,1145.26 L1135.84,1145.25 L1135.66,1144.16 L1136.66,1141.94 L1136.9,1141.92 L1137.1,1141.95 L1137.33,1142.13 L1137.66,1142.56 L1138.31,1142.93 L1139.2,1142.85 L1140.01,1142.42 L1140.45,1141.79 L1140.2,1140.98 L1140.3,1140.58 L1140.65,1140.57 L1141.12,1140.94 L1140.7,1139.26 L1138.9,1136.52 L1138.44,1134.83 L1138.77,1132.83 L1139.75,1131.59 L1142.12,1129.7 L1142.48,1129.15 L1142.85,1128.39 L1143.11,1127.53 L1143.19,1126.66 L1142.91,1125.45 L1142.37,1124.76 L1141.85,1124.35 L1141.28,1123.36 L1140.64,1123.56 L1139.98,1123.54 L1139.62,1122.32 L1138.76,1122.41 L1138.09,1122.18 L1137.65,1121.29 L1138.09,1121.19 L1138.37,1120.91 L1138.51,1120.41 L1138.55,1119.59 L1137.99,1116.88 L1136.72,1115.71 L1135.26,1115.01 L1134.17,1113.74 L1133.76,1112.78 L1133.65,1112.29 L1133.59,1111.39 L1133.49,1111.1 L1133.25,1110.76 L1132.95,1110.49 L1132.71,1110.37 L1132.24,1109.95 L1131.98,1108.93 L1131.77,1107.79 L1131.47,1106.96 L1130.83,1106.39 L1130.28,1106.03 L1129.88,1105.47 L1129.69,1104.27 L1129.43,1103.6 L1126.18,1099.65 L1125.91,1099.24 L1124.48,1096.26 L1124.08,1095.65 L1123.69,1093.69 L1123.53,1093.11 L1122.62,1091.28 L1122.15,1089.85 L1120.73,1088.87 L1120.24,1088.09 L1127.13,1090.8 L1127.4,1091 L1127.87,1091.89 L1128.14,1092.09 L1128.42,1092.21 L1128.96,1092.78 L1129.26,1092.9 L1129.55,1092.82 L1130.03,1092.51 L1130.31,1092.43 L1131.78,1092.55 L1132.45,1092.43 L1133.05,1091.9 L1133.46,1091.02 L1133.63,1090.03 L1133.46,1089.23 L1131.82,1088.36 L1131.05,1087.02 L1129.86,1084.29 L1126.72,1082.13 L1125.89,1080.7 L1127.03,1078.41 L1128.97,1077.58 L1133.18,1078.97 L1135.12,1078.09 L1135.58,1077.34 L1136.09,1076.17 L1136.48,1074.83 L1136.61,1073.52 L1136.41,1072.29 L1135.23,1069.46 L1134.5,1066.73 L1133.98,1065.98 L1132.94,1065.74 L1132.12,1065.86 L1130.76,1066.43 L1130.22,1066.51 L1129.68,1066.48 L1129.15,1066.32 L1127.28,1064.35 L1126.93,1063.58 L1126.66,1062.2 L1126.12,1061.54 L1125.48,1061.01 L1124.91,1060.03 L1124.1,1057.68 L1124.24,1056.31 L1124.14,1055.61 L1121.87,1054.11 L1121.71,1053.91 L1121.68,1053.66 L1121.67,1053.43 L1121.6,1053.27 L1121.38,1053.15 L1120.81,1052.99 L1120.63,1052.87 L1119.93,1051.67 L1119.56,1051.33 L1118.45,1051.11 L1117.55,1050.55 L1116.28,1050.19 L1115.62,1049.47 L1115.19,1048.39 L1115.04,1047.04 L1115.1,1046.73 L1115.39,1046.17 L1115.45,1045.98 L1115.41,1045.61 L1115.24,1045 L1115.19,1044.68 L1115.11,1044.33 L1114.89,1043.85 L1114.62,1043.43 L1113.52,1042.94 L1112.45,1042.12 L1111.6,1041.03 L1111.49,1039.89 L1111.24,1039.75 L1110.55,1039.06 L1111.17,1038.08 L1111,1037.25 L1109.96,1035.62 L1109.83,1035.14 L1109.68,1033.89 L1109.65,1033.32 L1109.44,1033.3 L1108.41,1033.4 L1108,1033.16 L1109.03,1032.13 L1109.38,1031.59 L1109.02,1031.38 L1108.23,1031.36 L1107.57,1031.21 L1106.95,1030.83 L1106.28,1030.18 L1107.23,1030.15 L1107.69,1030.01 L1108.14,1029.73 L1107.65,1029.43 L1107.14,1029.32 L1106.04,1029.33 L1106.03,1028.91 L1106.37,1028.53 L1106.59,1028.1 L1106.84,1027.74 L1107.6,1027.43 L1107.74,1027.08 L1107.73,1026.66 L1107.57,1026.29 L1107.59,1025.54 L1107.33,1024.96 L1106.94,1024.52 L1106.56,1024.18 L1106.47,1024.01 L1106.36,1023.71 L1106.2,1023.43 L1105.96,1023.31 L1105.68,1023.41 L1105.15,1023.76 L1104.92,1023.77 L1104.32,1023.07 L1104.76,1022.28 L1105.62,1021.36 L1106.23,1020.31 L1104.59,1020.36 L1105.33,1019.84 L1105.63,1019.27 L1105.54,1018.54 L1104.44,1016.33 L1104.23,1016.1 L1103.9,1016.16 L1103.67,1016.47 L1103.46,1016.81 L1103.22,1016.98 L1102.7,1017.92 L1102.36,1022.2 L1101.93,1023.43 L1101.66,1022.34 L1100.92,1022.56 L1100.36,1022.58 L1100.7,1020.89 L1100.32,1020.95 L1099.95,1020.93 L1099.6,1020.79 L1099.28,1020.51 L1099.67,1019.93 L1099.59,1019.1 L1099.34,1018.12 L1099.18,1017.09 L1099.28,1016.07 L1099.58,1015.37 L1100.01,1014.89 L1100.51,1014.49 L1099.92,1014.01 L1099.32,1013.67 L1099.54,1013.59 L1100.02,1013.18 L1099.29,1012.55 L1099.05,1012.39 L1099.51,1011.42 L1099.21,1007.94 L1099.34,1006.33 L1099.2,1006.2 L1098.83,1006 L1098.67,1005.96 L1098.52,1006.85 L1098.12,1007.28 L1097.7,1007.22 L1097.49,1006.61 L1097.35,1005.65 L1097.04,1004.99 L1096.73,1004.45 L1096.52,1003.84 L1096.81,1002.51 L1098.98,1002.21 L1099.43,1001.2 L1098.72,1001.63 L1097.79,1001.84 L1096.83,1001.86 L1096,1001.72 L1094.27,1000.75 L1093.51,1000.68 L1093.67,1001.78 L1093.21,1001.96 L1092.6,1001.96 L1092.11,1001.66 L1092.04,1000.96 L1092.1,1000.19 L1095.21,996.842 L1095.5,996.387 L1095.4,995.972 L1094.78,995.85 L1094.03,995.96 L1093.53,996.222 L1092.68,995.102 L1092,993.861 L1091.27,992.972 L1090.25,992.884 L1090.97,991.971 L1091.37,991.659 L1091.81,991.529 L1091.43,990.244 L1091.87,989.592 L1093.59,988.884 L1093.58,988.497 L1093.13,988.508 L1093.12,988.081 L1093.44,987.945 L1093.57,987.825 L1093.78,987.209 L1092.78,986.615 L1092.99,985.543 L1094.37,982.882 L1094.76,981.334 L1094.83,980.439 L1094.61,979.673 L1094.29,979.05 L1094.11,978.418 L1094.21,977.825 L1094.68,977.324 L1094.57,977.085 L1094.51,976.896 L1094.41,976.476 L1095.22,976.262 L1095.54,975.123 L1095.72,970.995 L1095.84,970.577 L1096.19,970.412 L1096.63,970.373 L1096.97,970.211 L1097.23,969.823 L1097.4,969.098 L1096.88,969.198 L1096.44,969.021 L1096.13,968.515 L1095.85,966.85 L1095.51,966.336 L1095.12,965.891 L1094.79,965.319 L1094.59,963.827 L1095.13,962.915 L1096.97,961.849 L1096.54,961.076 L1095.86,961.262 L1094.48,962.303 L1092.66,962.15 L1091.84,962.338 L1091.83,963.221 L1091.36,963.571 L1090.29,963.604 L1088.67,965.266 L1088.35,965.474 L1088.15,965.723 L1087.46,966.866 L1087.14,967.211 L1086.68,967.329 L1086.18,967.271 L1085.7,967.082 L1085.31,966.824 L1085.46,967.133 L1085.79,968.096 L1084.66,967.756 L1084.34,966.806 L1084.54,965.435 L1085.01,963.839 L1085.72,962.073 L1086.54,960.711 L1088.29,958.603 L1087.81,958.186 L1088.25,957.486 L1088.74,957.002 L1089.82,956.437 L1089.22,955.499 L1088.45,955.896 L1087.7,956.906 L1087.17,957.816 L1086.81,957.01 L1086.14,954.887 L1085.92,953.96 L1085.71,953.964 L1085.97,955.786 L1085.96,957.952 L1085.68,960.112 L1085.08,961.919 L1084.21,963.424 L1083.28,964.591 L1081.01,966.484 L1081.29,965.565 L1081.64,964.861 L1082.05,964.313 L1082.51,963.891 L1082.5,963.428 L1082,963.51 L1081.06,964.083 L1080.73,963.927 L1080.53,963.142 L1080.71,962.456 L1081.08,961.965 L1081.48,961.779 L1082.64,961.626 L1082.39,961.173 L1081.76,960.262 L1081.77,958.742 L1081.21,958.802 L1080.91,959.339 L1080.71,960.12 L1080.44,960.906 L1080.06,961.318 L1079.56,961.552 L1079.25,961.425 L1079.41,960.73 L1080.5,958.149 L1081.18,956.863 L1081.44,956.199 L1081.7,955.331 L1081.42,956.008 L1080.05,958.04 L1079.52,958.498 L1079.28,958.791 L1079.17,959.06 L1078.94,959.812 L1078.83,960.082 L1077.94,960.572 L1077.02,960.206 L1076.18,960.237 L1075.49,961.89 L1075.12,961.484 L1075.07,961.08 L1075.17,960.598 L1075.23,959.955 L1075.11,959.4 L1074.52,958.064 L1074.65,956.947 L1075.12,956.568 L1076.29,956.715 L1076.02,955.46 L1076.33,954.836 L1076.89,954.399 L1077.38,953.703 L1076.18,952.966 L1075.99,952.643 L1075.91,952.278 L1075.6,951.724 L1075.52,951.369 L1075.59,951.078 L1075.74,950.839 L1075.89,950.698 L1075.96,950.7 L1075.95,948.837 L1076.03,948.368 L1076.33,947.914 L1076.59,947.701 L1076.83,947.442 L1077,946.838 L1076.33,946.851 L1076.45,946.077 L1076.28,945.798 L1076.02,945.619 L1075.86,945.154 L1075.86,944.599 L1076.06,943.985 L1076.05,943.44 L1075.73,942.545 L1074.41,940.675 L1074.51,939.439 L1074.62,938.872 L1074.44,938.725 L1073.6,938.749 L1073.29,938.23 L1073.3,937.111 L1073.65,936.072 L1074.33,935.785 L1074.19,934.94 L1074.37,934.406 L1074.69,933.894 L1074.97,933.133 L1075.01,932.598 L1074.92,930.378 L1074.74,930.053 L1074.35,929.915 L1073.97,929.692 L1073.79,929.118 L1073.76,927.635 L1073.68,926.98 L1073.49,926.365 L1073.59,926.119 L1073.64,925.927 L1073.72,925.47 L1072.06,924.953 L1071.18,924.364 L1070.79,923.576 L1071.07,922.618 L1071.76,922.012 L1073.2,921.171 L1071.84,921.308 L1071.22,921.036 L1070.95,920.157 L1071.08,919.366 L1071.4,918.712 L1073.42,915.619 L1074.26,914.978 L1077.51,914.335 L1078.61,914.808 L1079.1,916.585 L1079.37,917.916 L1079.96,919.296 L1080.64,920.427 L1081.16,921.025 L1081.42,921.02 L1081.41,921 L1081.42,921.001 L1081.44,921.037 L1081.97,922.942 L1082.01,926.971 L1083.02,927.522 L1085.12,927.613 L1087.2,927.04 L1088.87,925.318 L1089.74,921.956 L1089.91,920.349 L1090.16,918.937 L1090.91,916.172 L1091.22,914.5 L1091.25,911.503 L1092.02,907.577 L1092.29,906.595 L1092.83,905.563 L1093.08,904.662 L1093.1,904.603 L1093.07,903.275 L1092.87,901.862 L1092.03,897.692 L1088.31,885.374 L1088.07,884.036 L1088.15,882.373 L1088.73,881.41 L1092.95,878.966 L1093.69,878.093 L1094.87,876.036 L1095.04,875.518 L1095,874.391 L1095.14,872.013 L1095.06,871.068 L1094.65,870.062 L1094.45,869.819 L1093.5,868.66 L1093.1,867.789 L1093.25,865.659 L1094.02,864.856 L1094.5,864.353 L1096.12,863.731 L1101.2,864.09 L1102.49,863.774 L1106.99,859.862 L1111.7,853.927 L1112.77,852.006 L1113.55,849.681 L1113.88,846.935 L1113.7,844.696 L1112.74,840.704 L1112.4,838.596 L1112.5,837.568 L1112.88,836.721 L1114.47,834.522 L1115.03,833.5 L1115.47,832.288 L1115.78,830.768 L1115.92,828.498 L1115.72,826.56 L1115.21,824.8 L1111.41,816.033 L1107.95,810.104 L1107.03,807.681 L1106.49,805.476 L1105.67,800.943 L1105.03,798.872 L1103.75,796.039 L1103.62,794.769 L1104.44,793.9 L1111.39,791.544 L1113.15,791.385 L1116.62,791.901 L1117.25,791.51 L1117.44,790.827 L1117.51,790.565 L1117.93,786.301 L1118.08,785.462 L1118.29,784.89 L1120.36,781.465 L1120.91,780.065 L1121.29,778.28 L1122.04,772.354 L1121.97,771.614 L1121.65,771.034 L1113.28,760.356 L1112.33,759.513 L1111.35,759.185 L1109.29,759.12 L1108.15,758.887 L1107.28,758.368 L1104.81,755.284 L1103.38,754.198 L1099.8,750.537 L1099.36,749.719 L1101.14,733.761 L1102.89,717.803 L1102.73,715.452 L1102.68,714.557 L1101.75,711.508 L1101.53,710.966 L1095.55,696.036 L1095.47,694.788 L1095.8,693.05 L1096.73,689.983 L1097.11,688.235 L1097.13,686.788 L1095.11,677.307 L1094.97,676.008 L1095.25,675.006 L1098.48,670.03 L1093.42,657.497 L1091.34,652.307 L1097.02,641.797 L1097.39,640.56 L1097.34,639.169 L1095.74,632.883 L1099.06,629.105 L1103.99,619.423 L1109.16,610.07 L1111.1,607.808 L1117.19,603.351 L1124.04,601.048 L1133.89,603.105 L1143.77,605.113 L1144.46,605.074 L1144.92,604.425 L1148.89,594.857 L1149.16,593.702 L1149.12,592.568 L1146.92,577.974 L1146.32,576.506 L1145.3,575.859 L1143.13,575.5 L1141.96,575.134 L1133.56,569.804 L1139.95,556.747 L1146.24,543.671 L1146.25,543.661 L1149.54,537.68 L1149.98,535.791 L1150,535.723 L1150.12,535.225 L1150.45,529.408 L1150.72,527.861 L1151.19,526.885 L1153.21,524.682 L1153.92,522.801 L1153.83,519.907 L1153.41,516.655 L1153.18,513.696 L1153.44,498.931 L1153.62,497.942 L1155.06,493.822 L1155.27,492.486 L1154.97,489.978 L1151.03,472.612 L1157.52,471.308 L1163.99,469.98 L1174.26,461.609 L1174.44,460.775 L1174.29,459.508 L1173.91,457.52 L1173.69,456.656 L1171.94,450.012 L1171.7,448.305 L1177.21,441.107 L1184.75,422.479 L1186.62,419.39 L1190.71,414.302 L1192.08,413.28 L1192.74,412.489 L1193.03,411.416 L1192.89,403.312 L1192.64,402.231 L1183.7,389.154 L1184.54,384.481 L1184.93,383.705 L1188.88,382.509 L1190.29,381.607 L1191.44,380.011 L1193.51,375.565 L1193.88,374.193 L1196.01,359.629 L1196.38,358.354 L1196.88,357.452 L1205.72,348.463 L1207.18,344.694 L1207.63,344.184 L1214.89,347.637 L1222.17,351.058 L1222.75,350.791 L1223.31,349.584 L1223.85,347.954 L1227.04,338.669 L1227.23,336.127 L1224.39,324.133 L1224.17,322.245 L1223.87,316.504 L1223.9,315.777 L1224.09,315.395 L1224.47,315.193 L1229.86,312.322 L1231.01,312.924 L1235.27,316.194 L1236.6,316.378 L1244.29,314.753 L1255.27,318.387 L1266.36,321.451 L1267.28,321.249 L1268.05,320.343 L1272.34,312.573 L1265.18,309.619 L1268.86,304.952 L1270.27,301.458 L1271.5,294.581 L1271.6,293.297 L1271.5,292.122 L1270.41,286.721 L1270.06,285.708 L1263.54,279.64 L1269.31,278.328 L1275.07,276.993 L1276.44,277.924 L1279.2,277.925 L1281.01,279.209 L1282.13,279.381 L1282.8,280.138 L1282.99,281.237 L1282.8,282.28 L1282.18,282.75 L1282.34,284.008 L1282.95,284.672 L1286.97,285.607 L1289.1,287.006 L1290.81,288.268 L1292.6,290.545 L1292.96,290.89 L1293.1,290.981 L1293.37,291.504 L1293.51,291.696 L1293.74,291.768 L1295.35,291.512 L1295.87,291.738 L1296.19,292.324 L1296.45,292.99 L1297.49,294.771 L1297.87,295.313 L1298.73,295.886 L1300.57,296.086 L1302.98,297.811 L1304.41,298.47 L1305.23,300.305 L1309.61,300.787 L1310.89,301.909 L1311.09,302.258 L1311.92,303.058 L1312.35,303.578 L1312.7,303.896 L1312.81,303.943 L1313.01,304.087 L1313.12,304.477 L1313.19,304.915 L1313.28,305.193 L1314.19,305.744 L1319.48,304.852 L1321.13,305.044 L1321.23,305.165 L1321.34,305.767 L1321.46,305.885 L1323.29,305.698 L1324.93,306.257 L1325.96,306.188"
         id="path1027" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['SE'],polycolors['SE'],labels['SE'], x1, polybary['SE'], x2, polybary['SE'] )
    if 'SE' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M789.745,1388.7 L790.547,1389.12 L792.122,1389.26 L793.572,1389.81 L794.526,1389.75 L798.473,1388.43 L799.048,1387.86 L799.36,1386.63 L799.777,1386.11 L800.614,1386.22 L801.496,1386.64 L802.028,1387.04 L802.253,1387.46 L802.612,1388.46 L802.927,1388.87 L803.35,1389.06 L804.283,1389.26 L804.648,1389.61 L804.89,1389.74 L806.231,1389.96 L806.982,1389 L807.248,1388.79 L808.003,1388.94 L808.819,1389.51 L808.778,1389.77 L808.372,1391.2 L808.105,1391.8 L806.4,1393.78 L804.338,1395.15 L799.293,1397.2 L797.394,1398.45 L796.493,1398.66 L791.856,1398.33 L791.389,1398.08 L790.865,1397.33 L790.856,1396.68 L790.979,1395.99 L790.847,1395.15 L790.149,1394.6 L785.508,1392.6 L784.526,1392.35 L783.556,1392.32 L781.971,1392.53 L781.342,1392.52 L780.756,1392.61 L780.289,1393.01 L780.152,1393.72 L780.183,1394.52 L780.018,1395.1 L778.141,1395.23 L776.906,1394.97 L776.039,1394.47 L775.772,1394.31 L774.917,1393.18 L774.506,1391.57 L774.667,1390.12 L774.968,1388.59 L774.849,1386.15 L774.851,1386.15 L781.325,1384.68 L781.944,1384.26 L782.559,1384.24 L784.885,1386.06 L788.591,1387.48 L789.745,1388.7"
         id="path1029" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M802.485,1368.04 L802.647,1368.38 L802.866,1368.74 L803.227,1369.09 L804.807,1369.68 L805.5,1370.26 L805.497,1371.27 L805.046,1371.87 L804.372,1372.26 L803.662,1372.47 L800.634,1372.81 L799.949,1372.72 L798.551,1372.06 L797.765,1371.4 L797.463,1370.52 L797.284,1369.7 L796.713,1369.17 L795.245,1368.43 L795.008,1368.16 L794.794,1367.8 L794.492,1367.51 L793.978,1367.42 L792.926,1368.48 L792.681,1368.68 L790.935,1368.63 L790.162,1368.32 L789.71,1367.52 L789.992,1365.52 L791.478,1364.72 L794.777,1364.53 L794.942,1364.48 L795.57,1364.6 L796.056,1364.92 L799.756,1365.33 L800.597,1365.6 L801.186,1366.14 L802.485,1368.04"
         id="path1031" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M917.78,1266.63 L918.815,1271.84 L919.007,1271.93 L919.886,1272.62 L919.95,1272.78 L920.123,1272.92 L920.297,1273.22 L920.543,1273.53 L921.709,1273.97 L923.166,1274.88 L923.84,1274.69 L924.142,1275.07 L924.588,1275.2 L925.035,1275.1 L925.37,1274.75 L925.626,1274.76 L925.609,1275.19 L925.299,1275.44 L925.142,1275.62 L925.14,1275.89 L925.302,1276.42 L925.34,1277.44 L926.452,1278.14 L929.134,1278.94 L929.555,1278.97 L929.594,1280.75 L929.594,1281.75 L929.439,1282.72 L929.141,1283.79 L928.555,1285.14 L928.442,1285.64 L928.44,1286.39 L928.568,1286.8 L928.738,1287.13 L928.856,1287.68 L929.032,1292.17 L928.886,1294.41 L928.461,1296.39 L927.559,1298.45 L924.26,1303.22 L923.936,1304.03 L923.506,1305.1 L923.092,1307.17 L922.463,1313.91 L922.094,1316.1 L921.347,1317.41 L920.008,1316.93 L919.444,1316.67 L917.557,1316.87 L915.567,1316.17 L911.957,1316.07 L910.666,1316.41 L909.96,1316.87 L909.281,1317.32 L909.57,1317.99 L909.469,1318.55 L909.273,1319.06 L909.261,1319.58 L909.51,1319.99 L910.71,1321.11 L908.547,1322.08 L907.914,1322.11 L908.338,1323.11 L908.42,1324.24 L908.425,1325.34 L908.615,1326.28 L909.329,1327.2 L910.313,1327.73 L913.317,1328.3 L915.258,1329.14 L916.329,1329.36 L917.335,1329.34 L918.302,1329.1 L918.672,1328.77 L918.926,1328.36 L919.172,1328.26 L921.342,1332.15 L921.814,1333.78 L921.396,1336.04 L920.555,1337.8 L920.239,1338.73 L920.168,1339.87 L920.454,1341.22 L920.662,1341.65 L920.751,1341.83 L920.631,1342.15 L918.881,1343.12 L918.448,1343.58 L917.486,1345.01 L916.69,1345.66 L915.011,1346.32 L914.23,1347.07 L913.304,1348.77 L912.8,1349.19 L910.514,1349.35 L910.293,1349.36 L909.661,1349.68 L908.978,1350.33 L908.76,1350.68 L908.409,1351.46 L908.081,1351.84 L907.609,1352.05 L907.201,1352.06 L906.847,1352.21 L906.551,1352.85 L906.587,1353.81 L907.181,1354.27 L908.014,1354.56 L910.47,1356.08 L911.42,1357.18 L911.606,1357.49 L911.684,1357.92 L911.614,1358.68 L911.41,1358.86 L911.113,1358.9 L910.782,1359.24 L909.829,1360.8 L908.882,1361.83 L907.81,1362.35 L904.382,1362.31 L900.472,1363.44 L898.498,1364.82 L897.844,1365.08 L897.056,1365.04 L894.613,1364.18 L894.859,1365.61 L894.668,1366.22 L893.195,1366.55 L893.128,1365.57 L892.519,1365.18 L891.046,1364.88 L890.881,1364.65 L890.493,1363.86 L890.231,1363.58 L889.913,1363.52 L888.692,1363.73 L887.413,1363.12 L886.223,1362.05 L885.008,1361.39 L883.655,1362.04 L884.115,1362.43 L885.477,1364.02 L885.954,1364.87 L885.338,1364.68 L884.439,1364.4 L883.201,1363.99 L881.603,1364.33 L881.031,1364.6 L880.199,1365.15 L879.433,1365.43 L877.838,1365.63 L877.168,1366.05 L877.005,1366.6 L877.848,1367.32 L878.297,1368.56 L878.159,1369.82 L877.492,1370.41 L877.25,1370.63 L877.18,1371.3 L877.557,1371.48 L878.684,1371.51 L879.118,1371.75 L879.948,1372.45 L880.311,1372.87 L879.693,1374.61 L880.522,1375.8 L883.09,1377.1 L882.561,1378.47 L882.304,1379.36 L882.432,1380.13 L884.833,1383.84 L886.309,1385.74 L886.48,1387.8 L886.611,1388.94 L886.405,1390.17 L886.038,1391.49 L885.925,1392.64 L886.478,1393.37 L886.224,1393.91 L886.031,1395.27 L885.781,1396.04 L885.364,1396.57 L884.332,1397.21 L883.903,1397.59 L881.698,1401.19 L880.818,1402.16 L880.38,1402.83 L879.977,1404.05 L879.788,1405.44 L880.006,1406.61 L880.9,1407.17 L882.927,1405.93 L883.812,1406.29 L883.49,1406.46 L882.554,1407.14 L883.337,1407.96 L882.712,1408.7 L880.409,1409.8 L876.617,1412.46 L876.055,1413.04 L875.211,1414.59 L874.697,1415.09 L873.968,1415.13 L873.47,1414.62 L873.036,1413.98 L872.508,1413.65 L871.373,1414.17 L871.514,1415.64 L872.02,1417.47 L871.972,1419.05 L872.516,1418.97 L873.567,1418.52 L874.109,1418.4 L874.651,1418.5 L875.668,1418.9 L877.091,1418.81 L877.155,1419.14 L876.886,1419.65 L876.707,1420.16 L876.715,1421.07 L876.789,1421.57 L877.073,1421.92 L877.972,1422.56 L879.237,1423.07 L879.146,1423.32 L878.92,1423.65 L878.824,1424.06 L879.09,1425.36 L879.051,1425.86 L878.776,1426.49 L878.306,1426.99 L877.419,1426.87 L876.859,1427.15 L876.478,1427.76 L876.485,1428.3 L876.54,1428.88 L876.291,1429.61 L875.693,1430.11 L875.1,1429.92 L875.224,1430.73 L875.498,1431.27 L875.876,1431.57 L876.114,1431.97 L875.983,1432.77 L875.177,1432.88 L872.307,1432.63 L871.884,1432.46 L871.882,1432.46 L871.439,1432.3 L867.999,1432.19 L867.61,1431.88 L866.851,1430.94 L866.462,1430.76 L866.095,1431.01 L865.631,1431.63 L865.419,1431.91 L865,1431.97 L864.575,1431.86 L863.879,1431.4 L864.372,1429.99 L864.485,1429.26 L863.878,1429.04 L863.047,1428.58 L862.827,1428.41 L862.193,1427.9 L861.608,1426.91 L861.605,1425.51 L864.643,1422.49 L865.111,1422.34 L865.577,1422.2 L865.774,1422.06 L866.328,1421.32 L867.093,1419.91 L867.59,1419.17 L867.22,1418.92 L866.841,1418.89 L865.947,1419.08 L867.643,1416.68 L868.354,1415.2 L867.863,1414.44 L868.004,1413.62 L868.238,1413.02 L868.571,1412.84 L869.008,1413.26 L869.328,1412.87 L869.569,1412.44 L869.911,1411.6 L869.815,1411.34 L869.765,1411.15 L869.693,1410.69 L870.122,1410.76 L870.548,1410.72 L870.955,1410.59 L871.331,1410.36 L870.491,1409.35 L870.658,1408.65 L871.241,1408.13 L871.161,1407.82 L870.857,1407.01 L870.19,1406.92 L868.967,1407.15 L868.469,1406.68 L867.715,1405.5 L867.097,1405.08 L866.597,1405 L864.934,1405.24 L864.261,1405.21 L863.853,1404.88 L863.501,1404.42 L862.993,1404.01 L860.811,1403.21 L860.181,1402.6 L859.962,1402.39 L859.923,1400.96 L859.717,1399.73 L859.051,1398.74 L858.168,1398.1 L857.329,1397.95 L856.533,1398.28 L854.817,1399.33 L853.981,1399.58 L849.359,1399.09 L848.115,1399.32 L847.857,1399.36 L847.181,1399.14 L847.306,1396.92 L846.676,1396 L845.957,1395.88 L844.436,1395.91 L843.786,1395.49 L843.623,1395.07 L843.315,1393.67 L843.146,1393.11 L841.901,1391.53 L841.692,1390.9 L841.92,1389.56 L842.423,1388.59 L842.63,1387.56 L841.963,1386.05 L840.848,1385.19 L840.749,1385.03 L840.139,1385.15 L839.879,1385.51 L839.707,1386.08 L839.384,1386.83 L838.706,1387.8 L837.723,1388.96 L836.673,1389.85 L835.814,1390.02 L834.354,1389.22 L833.649,1388.98 L830.926,1388.69 L830.125,1388.27 L830.4,1387.54 L830.844,1387.5 L832.489,1388.42 L832.284,1387.31 L832.524,1386.38 L832.829,1385.5 L832.811,1384.55 L832.446,1384 L831.825,1383.52 L831.136,1383.2 L830.576,1383.1 L829.245,1383.67 L825.963,1387.16 L825.538,1387.43 L824.999,1387.59 L824.392,1387.59 L823.17,1387.18 L822.03,1387.31 L821.518,1387.22 L820.812,1386.63 L820.93,1386.02 L821.319,1385.27 L821.449,1384.23 L821.178,1383.69 L820.694,1383.56 L819.529,1383.66 L817.381,1384.27 L815.252,1385.31 L815.684,1385.86 L815.569,1386.3 L815.278,1386.77 L815.201,1387.4 L815.46,1387.98 L816.255,1389.1 L816.446,1389.73 L816.17,1390.79 L815.398,1391.13 L813.587,1390.87 L812.852,1390.55 L811.5,1389.67 L811.151,1389.63 L810.37,1389.54 L810.23,1389.34 L809.973,1388.58 L809.929,1387.81 L809.998,1387.17 L809.96,1386.68 L809.586,1386.36 L809.082,1386.92 L808.205,1387.11 L804.707,1386.73 L803.92,1386.4 L803.173,1385.85 L801.182,1383.81 L800.566,1383.49 L798.954,1383.19 L798.385,1383.49 L797.046,1385.58 L796.551,1386.19 L795.925,1386.46 L794.51,1386.49 L793.982,1386.77 L793.557,1386.81 L793.154,1386.58 L792.184,1385.36 L789.934,1384 L788.54,1382.4 L787.934,1382.28 L786.27,1382.75 L785.389,1382.71 L783.228,1382.01 L782.562,1381.57 L782.049,1380.63 L781.618,1379.59 L781.214,1378.83 L780.037,1377.59 L779.674,1376.88 L779.61,1375.85 L780.068,1375.32 L784.28,1373.21 L794.793,1372.98 L795.644,1373.18 L796.197,1373.56 L796.773,1374.2 L797.265,1375.01 L797.549,1375.95 L797.128,1376.08 L796.43,1376.47 L796.15,1376.69 L796.118,1377.08 L799.804,1377.78 L801.629,1378.7 L803.119,1380.23 L803.552,1381.25 L804.144,1383.47 L804.658,1384.29 L805.568,1384.64 L807.62,1384.52 L808.352,1384.98 L809.142,1384.79 L810.406,1384.95 L810.846,1384.87 L811.617,1384.72 L812.254,1383.34 L812.043,1381.97 L811.311,1380.9 L810.272,1380.17 L809.917,1380.08 L808.544,1379.72 L806.994,1378.87 L806.4,1378.77 L805.157,1378.82 L804.583,1378.68 L803.692,1378.04 L800.856,1374.9 L802.996,1374.09 L804.049,1373.88 L807.51,1374.34 L808.499,1374.73 L809.35,1375.58 L809.365,1375.59 L809.278,1374.4 L808.823,1373.89 L808.675,1373.87 L807.149,1373.68 L806.748,1373.49 L806.381,1373.19 L806.102,1372.79 L805.95,1372.23 L805.888,1371.51 L805.936,1370.79 L806.096,1370.27 L806.645,1369.76 L807.336,1369.54 L808.663,1369.53 L808.287,1369.08 L807.366,1368.3 L804.738,1367.25 L804.247,1366.39 L804.171,1364.3 L803.626,1362.95 L802.762,1362.02 L801.73,1361.17 L800.955,1361.66 L799.948,1361.88 L798.952,1361.8 L798.179,1361.95 L798.016,1362.19 L797.738,1362.22 L797.371,1362.17 L797.159,1360.41 L798.345,1359.74 L800.83,1359.2 L801.197,1358.78 L801.955,1358.79 L803.222,1359.19 L804.249,1359.86 L804.407,1359.91 L804.695,1359.9 L805.112,1359.75 L805.616,1359.38 L805.371,1358.79 L804.499,1357.35 L804.256,1357.12 L804.054,1356.84 L803.998,1356.44 L804.123,1355.96 L804.33,1355.64 L804.529,1355.4 L804.63,1355.21 L804.583,1354.24 L804.05,1351.73 L803.708,1350.65 L803.963,1350.2 L804.26,1349.8 L806.018,1350.35 L807.526,1350.22 L808.91,1349.47 L814.933,1343.21 L820.158,1336.68 L824.57,1329.93 L827.39,1323.3 L827.855,1321.56 L827.889,1321.1 L828.756,1319.56 L831.269,1307.56 L832.171,1304.52 L832.345,1303.58 L832.853,1301.65 L834.755,1297.62 L835.271,1295.59 L835.396,1294.44 L835.655,1292.96 L836.032,1291.7 L836.507,1291.19 L838.798,1291.71 L839.199,1291.38 L839.447,1291.86 L838.745,1293.73 L839.544,1294.94 L841.059,1295.51 L842.549,1295.51 L843.44,1295.08 L845.14,1293.8 L845.942,1293.56 L847.46,1293.47 L848.984,1292.91 L857.021,1286.61 L857.618,1285.99 L858.354,1285.45 L860.115,1285.16 L860.912,1284.71 L861.781,1283.21 L863.08,1279.01 L863.964,1277.15 L865.379,1275.69 L866.234,1275.03 L867.649,1274.51 L868.324,1273.84 L868.962,1273.02 L869.68,1272.37 L870.455,1272.14 L872.264,1271.95 L873.638,1270.89 L879.767,1268.82 L881.169,1267.93 L882.031,1267.91 L884.608,1266.76 L888.221,1266.78 L892.288,1266.87 L895.575,1266.93 L895.957,1267.08 L896.583,1267.71 L896.935,1267.86 L897.26,1267.77 L898.25,1267.07 L903.888,1265.72 L912.754,1264.76 L916.416,1265.55 L917.171,1265.94 L917.78,1266.63"
         id="path1033" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['NL'],polycolors['NL'],labels['NL'], x1, polybary['NL'], x2, polybary['NL'] )
    if 'NL' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1208.29,1415.44 L1209.37,1414.88 L1210.8,1414.88 L1211.23,1414.62 L1211.52,1413.73 L1211.78,1414.25 L1211.89,1414.62 L1212.05,1414.99 L1212.45,1415.48 L1212.96,1415.79 L1214.04,1415.99 L1214.55,1416.34 L1215.06,1416.89 L1215.39,1417.36 L1215.55,1417.97 L1215.54,1418.91 L1215.28,1419.68 L1214.86,1420.61 L1214.69,1421.38 L1215.17,1421.71 L1217.29,1420.67 L1217.77,1420.65 L1218.03,1421.34 L1217.85,1422.18 L1217.28,1423.8 L1217.21,1424.92 L1217.43,1425.38 L1218.83,1425.68 L1219.41,1425.96 L1221,1427.01 L1222.49,1427.24 L1223.35,1427.22 L1223.97,1427 L1224.33,1426.54 L1224.66,1425.79 L1225.17,1424.07 L1226.02,1424.05 L1227.42,1423.55 L1231.95,1423.58 L1232,1423.04 L1231.8,1420.87 L1231.74,1420.7 L1231.7,1420.46 L1231.71,1420.11 L1231.81,1419.89 L1232.17,1419.42 L1232.27,1419.1 L1232.1,1417.29 L1231.44,1416.46 L1230.77,1415.99 L1230.55,1415.3 L1230.56,1415.3 L1230.72,1415.07 L1230.91,1414.89 L1231.13,1414.77 L1231.35,1414.71 L1231.89,1414.25 L1232.05,1413.95 L1232.13,1413.43 L1232.92,1414.07 L1234.95,1414.37 L1235.32,1414.96 L1235.82,1415.39 L1236.39,1415.62 L1236.83,1415.5 L1236.91,1414.86 L1236.91,1414.85 L1237.21,1414.01 L1237.69,1413.62 L1238.17,1413.82 L1238.42,1414.75 L1238.86,1415.64 L1241.1,1415.52 L1242.19,1415.96 L1242.89,1416.94 L1242.89,1417.55 L1242.6,1418.23 L1242.4,1419.4 L1242.59,1420.27 L1243.03,1421.35 L1243.57,1422.35 L1244.04,1422.99 L1244.87,1423.57 L1245.61,1423.83 L1246.25,1424.29 L1246.8,1425.42 L1246.86,1426.03 L1246.8,1426.66 L1246.81,1427.26 L1247.1,1427.75 L1247.6,1427.87 L1247.67,1427.89 L1248.07,1427.4 L1248.4,1426.71 L1248.74,1426.22 L1250.37,1425.96 L1258.49,1428.13 L1258.96,1428.4 L1259.55,1429.16 L1259.86,1429.42 L1260.29,1429.52 L1263.51,1428.47 L1264.5,1428.48 L1264.62,1428.9 L1264.78,1429.27 L1267.05,1432.52 L1267.31,1432.59 L1268.4,1432.47 L1270.68,1431.4 L1271.34,1431.44 L1271.91,1431.91 L1272.51,1432.85 L1272.92,1433.66 L1273.21,1434.51 L1273.18,1435.25 L1272.65,1435.76 L1272.59,1435.97 L1272.57,1436.18 L1272.61,1436.37 L1272.71,1436.55 L1274.37,1436.1 L1276.63,1433.52 L1278.32,1433.55 L1279.65,1434.4 L1280.28,1434.64 L1280.93,1434.43 L1281.36,1433.77 L1281.58,1432.95 L1281.82,1432.3 L1282.32,1432.15 L1283.76,1432.06 L1285.08,1432.28 L1286.36,1432.86 L1290.04,1436.15 L1290.51,1437.3 L1289.31,1437.93 L1289.5,1438.38 L1289.52,1438.59 L1289.42,1438.77 L1289.28,1439.15 L1289.25,1439.47 L1289.23,1439.96 L1289.27,1440.32 L1289.38,1440.28 L1289.08,1440.78 L1288.33,1441.66 L1288,1442.22 L1287.33,1442.43 L1286.03,1442.19 L1285.42,1442.45 L1285.18,1443.28 L1284.63,1443.84 L1283.97,1444.32 L1283.4,1444.88 L1283.19,1444.99 L1282.84,1445.25 L1282.6,1445.29 L1282.85,1445.97 L1282.94,1446.15 L1282.13,1446.04 L1281.91,1447.06 L1282.37,1448.14 L1283.68,1448.26 L1284.34,1449.82 L1284.55,1450.17 L1285.1,1450.54 L1285.39,1450.45 L1285.65,1450.17 L1286.1,1449.98 L1287.92,1449.83 L1288.31,1449.91 L1288.64,1450.47 L1288.72,1451.05 L1288.73,1451.62 L1288.88,1452.15 L1289.67,1453.03 L1291.51,1453.53 L1292.45,1454.07 L1293.09,1454.83 L1294.27,1456.59 L1294.91,1457.33 L1295.5,1458.2 L1296.82,1459.35 L1297.36,1460.49 L1298,1462.58 L1298.45,1463.33 L1299.23,1464.18 L1300.73,1465.33 L1302.48,1466.05 L1304.1,1465.78 L1305.18,1463.99 L1306.01,1463.26 L1306.44,1462.52 L1306.79,1461.64 L1307.41,1460.48 L1308.24,1459.63 L1309.02,1459.44 L1309.83,1459.42 L1310.3,1459.25 L1310.74,1459.09 L1311.08,1458.72 L1311.66,1457.74 L1312.11,1457.36 L1312.46,1457.31 L1313.22,1457.51 L1313.58,1457.45 L1313.66,1457.21 L1313.58,1456.46 L1313.67,1456.27 L1314.26,1456.23 L1314.51,1456.37 L1314.73,1456.73 L1315.22,1457.32 L1315.72,1457.46 L1315.93,1456.91 L1315.77,1456.09 L1315.18,1455.45 L1314.98,1453.95 L1314.46,1452.87 L1313.66,1452.14 L1312.63,1451.66 L1311.81,1450.96 L1311.31,1449.79 L1310.89,1448.49 L1310.33,1447.4 L1308.88,1445.63 L1308.79,1444.76 L1309.81,1444 L1310.73,1444.03 L1311.92,1444.6 L1312.54,1444.9 L1313.45,1444.7 L1317.64,1445.45 L1318.1,1445.68 L1319.1,1446.37 L1319.64,1446.59 L1320.19,1446.54 L1321.38,1446.17 L1321.85,1446.27 L1321.79,1446.45 L1321.99,1447.11 L1322.31,1447.89 L1322.6,1448.41 L1323.04,1448.8 L1324.53,1449.55 L1325.5,1449.73 L1327.36,1449.48 L1328.22,1449.75 L1327.52,1450.12 L1327.65,1450.35 L1327.81,1450.54 L1328,1450.71 L1328.19,1450.85 L1328.22,1451.66 L1328.38,1452.44 L1328.72,1453.05 L1329.29,1453.4 L1329.61,1452.22 L1330.49,1452.06 L1331.43,1452.68 L1331.91,1453.8 L1332.13,1453.92 L1332.35,1453.94 L1332.55,1453.88 L1332.75,1453.72 L1333.77,1452.17 L1335.65,1451.81 L1339.28,1452.07 L1340.18,1451.73 L1340.8,1451.1 L1341.82,1449.34 L1341.99,1448.8 L1342.04,1448.37 L1342.29,1448.17 L1342.99,1448.35 L1343.23,1448.54 L1344.07,1449.45 L1344.18,1449.58 L1344.22,1449.75 L1344.21,1449.94 L1344.14,1450.14 L1343.83,1451.36 L1343.81,1451.93 L1343.97,1452.55 L1344.43,1452.92 L1345.17,1454.01 L1344.9,1454.99 L1344.16,1455.78 L1342.57,1456.98 L1341.55,1457.51 L1339.63,1457.78 L1339.21,1458.05 L1338.98,1458.71 L1339.11,1459.16 L1341.31,1461.41 L1341.96,1461.63 L1342.36,1461.52 L1342.68,1461.28 L1343.09,1461.14 L1343.63,1461.23 L1344.78,1461.65 L1345.35,1461.71 L1345.37,1461.77 L1345.39,1461.83 L1345.38,1461.88 L1345.37,1461.92 L1345.03,1462.28 L1344.94,1462.64 L1345.1,1462.98 L1345.52,1463.32 L1346.02,1463.8 L1346.36,1464.46 L1346.66,1465.21 L1347.06,1465.92 L1347.57,1466.49 L1350.44,1468.8 L1351.93,1468.95 L1353.34,1468.35 L1354.41,1467.17 L1355.12,1466.9 L1356.72,1466.67 L1357.98,1466.14 L1358.16,1465.85 L1358.02,1465.23 L1357.69,1464.83 L1357.34,1464.86 L1357.04,1465 L1356.82,1464.93 L1356.49,1463.54 L1357.17,1463.63 L1357.75,1463.71 L1360.26,1464.99 L1360.12,1466.16 L1360.71,1466.71 L1363.32,1466.91 L1363.97,1467.18 L1365.35,1468.13 L1365.84,1468.29 L1366.81,1468.38 L1367.32,1468.68 L1367.53,1469 L1367.91,1469.93 L1368.19,1470.33 L1369.05,1470.89 L1369.62,1470.71 L1370.63,1469.6 L1370.58,1469.5 L1370.52,1469.43 L1370.44,1469.41 L1370.35,1469.42 L1371.95,1469.03 L1373.56,1469.41 L1376.73,1470.71 L1377.66,1470.6 L1379.12,1469.4 L1379.75,1469.69 L1380.13,1470.79 L1380.14,1471.76 L1380.41,1472.5 L1381.54,1472.95 L1380.58,1474.36 L1380.79,1476.03 L1381.71,1477.84 L1382.88,1479.68 L1383.5,1481.44 L1383.8,1481.88 L1384.38,1482.19 L1385.64,1482.15 L1386.23,1482.26 L1386.81,1482.62 L1387.71,1483.46 L1388.25,1483.71 L1388.83,1483.59 L1389.43,1483.24 L1390.04,1483.06 L1390.7,1483.43 L1390.92,1483.9 L1391.45,1486.01 L1391.71,1486.58 L1392.31,1487.61 L1392.56,1488.15 L1393.45,1490.73 L1393.72,1492 L1393.67,1493.06 L1392.02,1493.3 L1391.3,1493.68 L1389.82,1495.41 L1388.7,1495.61 L1387.49,1495.39 L1386.08,1494.88 L1385.8,1494.86 L1385.52,1494.95 L1384.44,1495.69 L1382.62,1495.64 L1381.85,1496.27 L1381.7,1496.8 L1381.73,1497.97 L1381.59,1498.54 L1381.3,1498.92 L1380.05,1499.79 L1378.56,1502.11 L1377.72,1502.81 L1376.51,1502.68 L1376.74,1505.64 L1375.87,1506.5 L1374.5,1507.85 L1369.27,1510.6 L1368.2,1511.87 L1367.39,1513.58 L1366.84,1515.56 L1366.54,1517.66 L1366.58,1519.34 L1366.62,1519.72 L1366.98,1522.49 L1366.85,1523.89 L1366.16,1525.42 L1365.08,1526.67 L1363.79,1527.44 L1361.58,1527.51 L1360.61,1527.77 L1359.81,1528.42 L1359.38,1529.56 L1359.23,1530.52 L1359.11,1532.48 L1358.87,1533.36 L1358.19,1534.18 L1357.41,1534.31 L1356.54,1534.25 L1355.55,1534.54 L1354.94,1535.16 L1353.78,1537.74 L1353.12,1538.42 L1345.55,1542.35 L1344.02,1542.27 L1342.63,1541.07 L1342.01,1540.84 L1341.06,1541.22 L1340.36,1541.79 L1339.61,1542.4 L1338.91,1542.61 L1333.99,1540.7 L1331.97,1540.38 L1330.17,1540.95 L1328.11,1542.77 L1327.8,1543.39 L1327.67,1544.59 L1327.15,1545.39 L1325.85,1546.76 L1325,1548.58 L1323.47,1554.79 L1323.08,1555.7 L1322.5,1556.56 L1322.51,1557.75 L1320.93,1556.23 L1319.98,1552.16 L1318.88,1550.88 L1318.2,1550.89 L1316.67,1551.6 L1315.88,1551.77 L1315.1,1551.56 L1313.52,1550.7 L1312.87,1550.55 L1311.31,1550.75 L1310.65,1550.7 L1310.07,1550.37 L1309.71,1549.66 L1309.48,1548.77 L1309.09,1548.05 L1308.27,1547.81 L1307.5,1547.78 L1304.98,1547.29 L1304.24,1546.9 L1303.85,1546.81 L1303.48,1546.89 L1302.76,1547.27 L1302.34,1547.33 L1301.13,1547.26 L1300.46,1547.8 L1298.68,1551.56 L1298.26,1552.02 L1297.63,1552.27 L1297.39,1552.22 L1296.81,1551.82 L1295.97,1552.02 L1295.92,1552.07 L1290.09,1551.7 L1286.3,1552.25 L1284.04,1551.48 L1279.66,1548.8 L1279.57,1548.5 L1279.54,1548.07 L1279.3,1547.82 L1278.62,1548.06 L1278.52,1547.4 L1278.2,1547.15 L1277.73,1547.16 L1277.22,1547.33 L1277.6,1546.74 L1277.26,1546.79 L1276.94,1546.67 L1276.46,1546.38 L1275.67,1546.38 L1275.38,1546.29 L1275.15,1545.98 L1274.93,1545.23 L1274.68,1545.07 L1273.42,1545.17 L1273.07,1545.29 L1271.66,1546.17 L1270.99,1546.45 L1270.04,1546.52 L1269.1,1546.37 L1265.77,1544.82 L1263.25,1544.05 L1262.29,1543.76 L1260.13,1542.15 L1259.21,1541.71 L1257.34,1541.23 L1255.29,1540.36 L1254.36,1540.17 L1252.43,1540.29 L1252.22,1540.16 L1252.03,1539.86 L1251.8,1539.62 L1251.49,1539.65 L1251.34,1539.91 L1251.3,1540.76 L1251.18,1541.1 L1250.66,1541.82 L1250.46,1541.97 L1247.38,1543.04 L1246.57,1543.08 L1246.78,1542.17 L1246.74,1541.32 L1246.5,1540.48 L1246.14,1539.62 L1242.93,1539.59 L1240.6,1538.96 L1239.7,1539.14 L1239.58,1539.25 L1239.53,1539.46 L1239.55,1539.75 L1239.61,1540.09 L1239.14,1541.44 L1239.24,1543.22 L1239.53,1545.12 L1239.64,1546.84 L1239.03,1550.81 L1238.78,1551.62 L1238.74,1552.06 L1238.85,1552.44 L1239.06,1552.93 L1239.27,1553.32 L1239.37,1553.42 L1238.97,1554.55 L1238.14,1554.68 L1235.95,1553.95 L1233.78,1553.79 L1233.17,1554.07 L1232.96,1554.68 L1232.85,1555.45 L1232.48,1556.21 L1232.35,1557.42 L1230.29,1559.47 L1229.59,1560.78 L1229.55,1562.48 L1229.69,1564.39 L1229.58,1566.02 L1229.4,1566.22 L1228.8,1566.9 L1228.12,1566.9 L1227.77,1566.52 L1227.49,1565.94 L1227,1565.38 L1226.73,1565.34 L1226.11,1565.6 L1225.79,1565.59 L1225.64,1565.33 L1225.61,1564.92 L1225.53,1564.52 L1225.25,1564.3 L1224.62,1564.39 L1223.42,1565.07 L1222.8,1565.24 L1222.31,1565.16 L1220.61,1564.39 L1219.55,1563.28 L1218.97,1563.74 L1218.18,1566.26 L1217.57,1566.97 L1215.47,1568.02 L1214.67,1568.74 L1214.35,1568.89 L1213.96,1568.96 L1209.76,1567.74 L1203.86,1567.42 L1202.42,1566.9 L1202.05,1566.61 L1201.29,1565.76 L1201.09,1565.43 L1200.86,1564.59 L1200.92,1564.4 L1201.15,1564.3 L1201.4,1563.7 L1201.52,1563.61 L1201.72,1563.64 L1201.91,1563.57 L1201.96,1563.17 L1201.84,1562.8 L1201.58,1562.64 L1201.29,1562.54 L1201.07,1562.35 L1200.77,1561.85 L1200.03,1560.83 L1199.65,1560.46 L1196.79,1559.08 L1195.84,1558.38 L1195.06,1557.83 L1194.25,1557.44 L1192.57,1557.08 L1191.72,1556.29 L1189.53,1552.42 L1188.5,1550.97 L1187.9,1550.68 L1186.35,1550.52 L1185.56,1550.19 L1184.82,1549.52 L1184.13,1548.58 L1183.64,1547.46 L1183.48,1546.29 L1182.74,1546.41 L1182.01,1546.05 L1181.29,1545.53 L1180.55,1545.12 L1179.86,1545.01 L1179.59,1544.97 L1179.04,1545.26 L1178.68,1545.96 L1178.3,1547.07 L1177.32,1546.65 L1175.98,1545.78 L1174.74,1544.73 L1174.11,1543.8 L1174.01,1542.05 L1173.5,1540.98 L1171.67,1539.28 L1170.35,1537.67 L1169.83,1537.26 L1169,1536.98 L1166.51,1536.97 L1165.32,1536.35 L1164.62,1535.33 L1164.03,1534.12 L1159.98,1529.07 L1159.46,1528.81 L1159.46,1528.8 L1159.45,1528.8 L1158.09,1527.3 L1157.88,1526.72 L1157.8,1526.12 L1157.55,1525.62 L1156.83,1525.36 L1155.63,1524.18 L1155.02,1523.73 L1154.31,1523.39 L1152.73,1523.02 L1152.18,1523.22 L1152.39,1524.02 L1151.63,1524.15 L1149.21,1523.89 L1148.39,1523.63 L1147.72,1522.98 L1146.27,1520.04 L1145.42,1519.22 L1143.51,1518.4 L1142.69,1517.77 L1142.19,1516.88 L1141.95,1515.88 L1141.76,1513.83 L1141.67,1512.78 L1141.36,1512.54 L1140.93,1512.62 L1140.47,1512.5 L1140.01,1512.05 L1139.68,1511.59 L1139.4,1511.04 L1139.11,1510.3 L1138.93,1509.58 L1138.69,1508.2 L1138.38,1507.53 L1137.94,1507.07 L1136.95,1506.57 L1136.52,1506.08 L1136.54,1505.93 L1136.72,1505.43 L1136.39,1505.28 L1136.28,1505.2 L1136.6,1504.43 L1136.54,1503.6 L1136.18,1502.95 L1135.61,1502.67 L1134.24,1502.07 L1133.61,1501.67 L1133.01,1501.16 L1132.13,1499.98 L1131.68,1499.2 L1131.46,1498.54 L1131.91,1497.59 L1133.56,1496.82 L1134.18,1496.15 L1134.28,1495.12 L1134.26,1493.81 L1134.48,1492.92 L1135.29,1493.14 L1135.56,1492.16 L1136.43,1490.41 L1136.73,1489.44 L1136.81,1488.22 L1136.54,1487.78 L1135.96,1487.57 L1134.28,1486.66 L1134.3,1485.98 L1134.55,1485.13 L1134.42,1484.22 L1133.72,1483.59 L1132.23,1483.49 L1131.59,1482.9 L1131.55,1482.9 L1125.35,1479.83 L1125.21,1479.44 L1125.19,1478.87 L1125.08,1478.23 L1124.71,1477.63 L1124.11,1477.14 L1123.45,1476.79 L1122.87,1476.29 L1122.49,1475.35 L1122.44,1474.41 L1122.68,1472.98 L1122.54,1472.05 L1122.21,1471.31 L1121.73,1470.63 L1120.72,1469.58 L1118.7,1468.4 L1118.27,1467.67 L1118.75,1466.9 L1119.15,1466.61 L1119.42,1466.31 L1119.47,1465.51 L1119.29,1464.62 L1118.95,1464.08 L1118,1463.14 L1119.02,1463.19 L1120.94,1463.27 L1121.73,1463.79 L1121.98,1464.37 L1122,1465.44 L1122.17,1465.92 L1122.63,1466.3 L1123.22,1466.47 L1124.37,1466.46 L1124.66,1466.61 L1124.8,1466.83 L1124.79,1467.12 L1124.62,1467.48 L1124.56,1467.49 L1124.51,1467.51 L1124.45,1467.55 L1124.4,1467.6 L1124.3,1467.72 L1124.27,1467.85 L1124.31,1467.98 L1124.42,1468.13 L1125.14,1468.81 L1125.32,1469.04 L1125.5,1469.44 L1125.58,1469.81 L1125.64,1470.13 L1126.17,1471.84 L1126.6,1472.46 L1127.28,1472.52 L1127.83,1472.1 L1127.81,1471.54 L1127.56,1470.88 L1127.42,1470.11 L1127.51,1468.99 L1127.7,1468.4 L1128.46,1467.36 L1128.54,1466.36 L1128.85,1465.87 L1129.29,1465.54 L1129.73,1465.04 L1130.9,1462.75 L1131.31,1462.31 L1132.47,1461.61 L1132.93,1461.17 L1133.07,1461.08 L1133.43,1461.08 L1133.59,1460.98 L1133.67,1460.75 L1133.64,1460.51 L1133.57,1460.3 L1133.56,1460.17 L1133.66,1459.89 L1133.73,1459.57 L1133.86,1459.25 L1134.15,1458.98 L1135.23,1458 L1136.82,1457.46 L1139.8,1457.14 L1141.95,1457.45 L1142.47,1457.37 L1142.9,1456.92 L1143.73,1455.52 L1144.23,1454.99 L1144.87,1454.73 L1145.51,1454.67 L1146.23,1454.74 L1146.47,1454.64 L1147.02,1454.24 L1147.37,1454.16 L1147.71,1454.24 L1151.5,1456.35 L1152.87,1456.37 L1153.18,1456.04 L1154.01,1455.15 L1154.55,1454.46 L1154.76,1453.31 L1154.92,1450.75 L1155.24,1450.5 L1155.57,1450.41 L1155.9,1450.47 L1156.22,1450.69 L1156.58,1450.78 L1156.95,1450.79 L1157.32,1450.75 L1157.68,1450.63 L1157.68,1450.63 L1157.69,1450.63 L1158.8,1450.19 L1160.91,1450.22 L1161.86,1449.45 L1162.38,1448.39 L1163.55,1444.78 L1164.33,1444.82 L1165,1445.24 L1165.67,1445.51 L1166.49,1445.04 L1166.85,1443.76 L1167.06,1443.35 L1168.5,1442.22 L1168.73,1441.96 L1168.74,1441.73 L1168.85,1441.65 L1169.34,1441.8 L1169.57,1442.03 L1170.17,1442.95 L1170.45,1443.26 L1171.36,1443.66 L1172.09,1443.44 L1172.73,1442.79 L1173.38,1441.9 L1173.77,1441.66 L1173.99,1441.35 L1174.01,1440.97 L1173.76,1440.48 L1174.39,1439.95 L1174.89,1439.24 L1175,1438.39 L1174.5,1437.45 L1176.08,1436.39 L1177.86,1435.97 L1181.42,1435.81 L1182.18,1435.62 L1183.66,1434.98 L1184.51,1434.85 L1186.34,1434.96 L1187.05,1434.78 L1188.16,1434 L1188.4,1433.98 L1188.62,1433.89 L1188.87,1433.49 L1188.88,1433.16 L1188.76,1432.72 L1188.68,1432.19 L1188.82,1431.58 L1189.16,1431.11 L1189.49,1431.08 L1189.89,1431.18 L1190.38,1431.07 L1190.77,1430.75 L1191.33,1429.94 L1191.74,1429.61 L1192.51,1429.4 L1193.96,1429.51 L1194.72,1429.42 L1200.73,1426.24 L1201.19,1425.76 L1201.85,1424.63 L1202.25,1424.18 L1203.65,1423.8 L1206.77,1423.86 L1207.87,1422.81 L1208.01,1421.24 L1206.92,1420.69 L1205.44,1420.35 L1204.38,1419.41 L1204.37,1418.75 L1204.7,1418.5 L1204.75,1418.37 L1203.94,1418.1 L1202.6,1418 L1202.12,1417.73 L1202.53,1417.06 L1202.98,1415.23 L1203.86,1414.22 L1205.11,1413.94 L1206.6,1414.34 L1206.91,1414.59 L1207.14,1414.92 L1207.42,1415.25 L1207.86,1415.46 L1208.29,1415.44"
         id="path1035" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['CZ'],polycolors['CZ'],labels['CZ'], x1, polybary['CZ'], x2, polybary['CZ'] )
    if 'CZ' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1157.93,1190.15 L1157.68,1191.05 L1157.84,1192.16 L1158.2,1193.03 L1158.71,1193.7 L1159.35,1194.18 L1160.89,1194.63 L1165.55,1193.49 L1167.19,1193.77 L1168.34,1194.93 L1168.67,1196.53 L1167.8,1198.09 L1166.24,1199.25 L1165.48,1200.07 L1165.18,1201 L1165.37,1202.04 L1165.74,1203.12 L1166.2,1204.03 L1166.61,1204.57 L1167.38,1204.92 L1169.13,1205.25 L1169.92,1205.76 L1170.98,1207.49 L1171.65,1208.28 L1172.56,1208.58 L1172.09,1208.96 L1171.65,1209.58 L1171.33,1210.41 L1171.24,1211.43 L1171.29,1212.42 L1171.2,1212.82 L1170.81,1212.8 L1170,1212.56 L1170.43,1212.54 L1170.76,1212.33 L1170.89,1211.9 L1170.7,1211.25 L1170.45,1211.26 L1170.1,1211.68 L1169.55,1211.93 L1168.96,1211.96 L1168.5,1211.78 L1168.48,1211.36 L1169.81,1210.71 L1170.34,1210.23 L1170.61,1209.5 L1168.72,1209.73 L1167.88,1210.04 L1167.19,1210.57 L1169.13,1209.05 L1169.8,1208.72 L1169.78,1208.3 L1168.79,1208.25 L1168.28,1208.33 L1167.94,1208.6 L1167.48,1209.21 L1167.16,1209.08 L1166.86,1208.68 L1166.43,1208.47 L1162.82,1209.07 L1162.18,1209.52 L1160.33,1211.32 L1159.96,1211.99 L1159.58,1212.97 L1158.69,1213.2 L1157.9,1213.66 L1157.84,1215.34 L1158.32,1214.88 L1158.31,1214.57 L1158.23,1214.49 L1158.13,1214.49 L1158.07,1214.43 L1159.8,1214.04 L1160.06,1214.15 L1160.4,1215.56 L1160.35,1216.22 L1159.79,1216.53 L1158.73,1216.41 L1157.55,1216.05 L1156.21,1215.26 L1156.49,1214.92 L1156.65,1214.17 L1156.99,1213.19 L1156.47,1213.54 L1156.24,1213.8 L1156.03,1214.14 L1156.18,1214.3 L1156.3,1214.51 L1155.56,1214.48 L1155.18,1214.57 L1154.92,1214.81 L1154.6,1215 L1154.23,1214.77 L1153.87,1214.42 L1153.63,1214.24 L1153.09,1214.17 L1152.43,1213.88 L1152.01,1213.36 L1152.18,1212.56 L1151.64,1212.42 L1150.54,1213.02 L1150.18,1213.08 L1149.91,1212.6 L1150.11,1212.17 L1150.59,1211.97 L1151.16,1212.18 L1151.14,1211.79 L1148.8,1210.14 L1149.24,1208.78 L1149.91,1207.64 L1150.82,1207.18 L1151.97,1207.86 L1152.2,1207.42 L1152.64,1207.82 L1153.14,1207.7 L1154.19,1206.91 L1153.61,1206.87 L1153.16,1206.47 L1152.96,1205.87 L1153.1,1205.21 L1153.08,1204.82 L1152.58,1204.79 L1152.29,1204.5 L1152.05,1204.17 L1151.67,1204.03 L1150.07,1204.53 L1150.64,1203.2 L1150.9,1202.85 L1151.87,1202.56 L1152.95,1201.93 L1153.69,1201.76 L1153.67,1201.33 L1153.71,1201.32 L1153.93,1201.18 L1154.14,1200.88 L1153.11,1200.54 L1152.66,1200.26 L1152.35,1199.72 L1152.46,1199.57 L1152.57,1199.45 L1152.63,1199.25 L1152.58,1198.81 L1151.52,1199.29 L1150.87,1199.39 L1150.57,1199.12 L1150.49,1198.92 L1150.31,1198.88 L1150.14,1198.74 L1150.04,1198.26 L1149.97,1197.72 L1149.82,1197.4 L1149.58,1197.27 L1149.25,1197.29 L1150.07,1196.85 L1152.83,1196.31 L1153.44,1196.42 L1153.65,1196.79 L1154.08,1196.94 L1154.53,1197.24 L1154.76,1198.05 L1154.93,1198.49 L1155.2,1198.24 L1155.3,1197.53 L1154.96,1196.57 L1155.45,1195.82 L1156.08,1195.09 L1156.74,1194.54 L1157.34,1194.28 L1157,1195.57 L1156.75,1196.13 L1156.45,1196.5 L1156.68,1196.48 L1156.88,1196.51 L1157.06,1196.65 L1157.24,1196.93 L1157.07,1197.24 L1156.78,1198.16 L1157.3,1198.17 L1157.72,1197.99 L1158.02,1197.58 L1158.21,1196.88 L1157.9,1196.81 L1157.54,1196.54 L1157.22,1196.46 L1157.2,1196.04 L1157.67,1195.55 L1158.11,1196.37 L1158.8,1197.15 L1159.27,1197.98 L1159.08,1198.94 L1159.85,1199.95 L1160.77,1200.24 L1162.64,1200.1 L1162.35,1199.64 L1162.66,1198.8 L1162.82,1198.53 L1163.05,1198.32 L1163.03,1197.86 L1162.7,1197.01 L1162.48,1196.57 L1162.18,1196.23 L1162.16,1195.8 L1162.37,1195.63 L1162.48,1195.49 L1162.62,1195.38 L1162.9,1195.3 L1162.88,1194.91 L1162.43,1194.96 L1162.15,1195.25 L1161.94,1195.59 L1161.68,1195.82 L1161.29,1195.91 L1160.04,1195.9 L1159.84,1195.71 L1159.62,1195.28 L1159.35,1194.85 L1159,1194.67 L1158.44,1194.58 L1158.2,1194.28 L1158.06,1193.87 L1157.8,1193.44 L1157.34,1192.84 L1157.25,1192.59 L1156.78,1193.02 L1156.52,1193.34 L1155.97,1194.17 L1155.14,1194.73 L1154.55,1195.43 L1153.92,1195.95 L1153.18,1195.75 L1153.02,1195.02 L1153.23,1193.86 L1153.65,1192.78 L1154.14,1192.29 L1154.24,1192.02 L1154.37,1191.42 L1154.42,1190.83 L1154.3,1190.57 L1152.7,1191.07 L1152.18,1191.46 L1151.95,1191.4 L1151.93,1190.68 L1152.12,1190.25 L1152.48,1189.81 L1152.91,1189.48 L1157.81,1187.8 L1158.82,1187.74 L1159.8,1188.21 L1159.82,1188.6 L1159.29,1189.07 L1158.56,1189.54 L1157.93,1190.15"
         id="path1037" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M995.301,1177.26 L998.119,1178.47 L1005.76,1180.37 L1006.35,1180.92 L1006.51,1181.55 L1006.61,1182.42 L1006.87,1183.17 L1007.5,1183.44 L1010.14,1183.47 L1010.69,1183.38 L1011.04,1183.01 L1011.56,1182.88 L1011.96,1182.52 L1012.61,1182.38 L1013.34,1183.08 L1013.95,1183.15 L1014.49,1182.98 L1014.51,1182.96 L1014.51,1182.96 L1015.01,1182.97 L1017.17,1181.89 L1017.86,1181.04 L1018.75,1180.69 L1019.21,1179.88 L1019.68,1179.5 L1019.78,1181.75 L1021.53,1182.41 L1023.75,1182.54 L1025.25,1183.21 L1025.88,1183.7 L1026.82,1184.11 L1027.68,1184.72 L1028.05,1185.8 L1028.47,1186.39 L1029.41,1186.23 L1031.06,1185.6 L1030.92,1184.3 L1031.78,1184.24 L1032.95,1184.97 L1033.78,1186.03 L1034.28,1188.16 L1034.51,1188.66 L1034.83,1189.14 L1035.51,1189.87 L1035.05,1190.23 L1034.57,1190.17 L1034.12,1189.96 L1033.78,1189.87 L1033.8,1190.01 L1032.9,1190.72 L1032.79,1190.72 L1032.56,1190.89 L1032.4,1190.87 L1032.31,1191 L1032.29,1191.58 L1032.82,1191.62 L1033.77,1191.33 L1034.72,1191.28 L1035.27,1192.01 L1035.66,1191.67 L1036.07,1191.64 L1036.27,1192.02 L1036.01,1192.9 L1036.19,1193.77 L1036.22,1195.01 L1036.01,1197.6 L1035.83,1198.71 L1035.58,1199.59 L1035.23,1200.32 L1034.77,1200.98 L1033.6,1202.08 L1032.13,1202.89 L1030.55,1203.41 L1029.02,1203.58 L1029.02,1204 L1030.6,1204.83 L1039.4,1202.73 L1040.08,1202.97 L1042.29,1204.89 L1041.56,1205.62 L1041.56,1206.31 L1041.85,1207.07 L1042.05,1208.08 L1041.84,1208.92 L1041.32,1209.7 L1040.66,1210.26 L1040.03,1210.45 L1040.13,1210.59 L1040.26,1210.7 L1040.27,1210.9 L1040.04,1211.31 L1040.43,1211.73 L1040.54,1212.26 L1040.4,1212.85 L1040.04,1213.45 L1040.84,1213.15 L1041.45,1212.4 L1042.3,1210.45 L1042.78,1208.37 L1043.05,1207.85 L1043.41,1207.6 L1044.79,1207.45 L1045.29,1207.15 L1045.53,1206.81 L1045.72,1206.46 L1046.05,1206.13 L1046.47,1205.99 L1048.02,1206.12 L1048.83,1206.51 L1051.05,1208.27 L1056.83,1210.79 L1060.12,1213.84 L1061.63,1214.21 L1063.3,1213.88 L1064.52,1213.1 L1066.59,1211.12 L1067.79,1210.33 L1069.04,1209.78 L1072.08,1209.34 L1072.09,1209.77 L1071.43,1209.81 L1070.52,1209.83 L1071.16,1210.11 L1071.79,1209.89 L1072.65,1210.48 L1075.2,1208.86 L1076.35,1209.28 L1075.01,1210.83 L1074.63,1211.44 L1074.41,1211.26 L1073.85,1211.03 L1074.27,1212.04 L1074.64,1214.53 L1074.95,1215.71 L1074.69,1216.5 L1074.73,1217.57 L1075,1219.36 L1074.86,1220.44 L1074.52,1221.01 L1074.17,1221.41 L1074.02,1221.96 L1069.79,1224.8 L1068.34,1226.71 L1067.7,1227.14 L1066.22,1227.67 L1065.93,1227.66 L1065.62,1227.94 L1065.37,1227.77 L1065.11,1227.49 L1064.78,1227.46 L1064.49,1227.65 L1064.35,1227.82 L1062.95,1230.05 L1062.58,1230.47 L1062.58,1230.86 L1063.08,1232.08 L1063.63,1232.98 L1064.35,1233.56 L1065.35,1233.87 L1066.78,1233.7 L1067.15,1233.85 L1067.42,1234.49 L1067.46,1235.24 L1067.61,1235.81 L1068.15,1235.96 L1068.3,1235.97 L1068.73,1236 L1068.91,1235.95 L1069.38,1235.58 L1070.44,1234.43 L1071.05,1234.18 L1071.75,1234.02 L1073.7,1232.94 L1078.17,1232.28 L1078.5,1232.39 L1078.76,1232.75 L1079.01,1233.62 L1079.28,1234.06 L1080.6,1234.4 L1080.81,1234.71 L1080.66,1236.14 L1080.75,1236.79 L1081.23,1237.05 L1082.33,1237.11 L1082.63,1237.03 L1084.1,1235.72 L1084.39,1235.9 L1084.78,1236.78 L1085.02,1236.98 L1086.56,1236.96 L1086.58,1237.15 L1088.03,1238.91 L1088.31,1239.09 L1088.73,1239.08 L1088.77,1238.42 L1089.12,1237.41 L1089.38,1236.16 L1089.2,1235.15 L1089.46,1234.69 L1089.81,1234.4 L1090.12,1234.32 L1090.04,1234.04 L1089.94,1233.33 L1089.86,1233.04 L1091.1,1231.74 L1091.09,1231.27 L1090.84,1231.28 L1090.83,1230.89 L1091.78,1231.26 L1092.1,1231.25 L1092.83,1230.68 L1093.45,1229 L1093.94,1228.64 L1094.23,1228.53 L1094.47,1228.25 L1094.65,1227.85 L1094.8,1227.38 L1094.81,1227.07 L1094.53,1226.86 L1094.53,1226.53 L1094.64,1226.21 L1094.78,1225.99 L1094.92,1225.82 L1095.79,1224.15 L1096.94,1223.32 L1098.37,1223.08 L1101.45,1223.55 L1110.15,1221.43 L1111.5,1220.5 L1111.74,1222.29 L1111.66,1224.03 L1111.78,1225.51 L1112.67,1226.49 L1112.29,1225.74 L1112.06,1224.12 L1111.85,1223.52 L1112.21,1222.78 L1113.25,1222.15 L1113.54,1221.29 L1113.12,1221.27 L1112.73,1221.35 L1112.36,1221.52 L1112.02,1221.77 L1112.28,1221.17 L1113.48,1219.5 L1114.2,1218.81 L1115.03,1217.31 L1115.54,1216.96 L1116.01,1216.81 L1118.72,1215.21 L1120.08,1214 L1120.6,1213.34 L1121.83,1210.74 L1123.92,1207.48 L1124.14,1206.87 L1125.55,1204.2 L1126.07,1202.54 L1126.47,1201.98 L1127.2,1201.58 L1127.21,1201.96 L1126.97,1201.97 L1127.79,1203.2 L1129.38,1203.79 L1137.68,1203.48 L1139.08,1204.09 L1139.9,1203.58 L1140.95,1203.79 L1141.45,1204.47 L1140.58,1205.35 L1139.65,1205.59 L1135.73,1205.4 L1134.15,1204.88 L1133.19,1204.78 L1132.9,1204.92 L1132.54,1205.5 L1132.34,1205.67 L1131.59,1205.69 L1130.64,1205.55 L1130.3,1205.76 L1130.1,1206.6 L1129.86,1206.15 L1129.67,1206.27 L1129.36,1206.56 L1129.13,1206.64 L1129.21,1207.17 L1129.38,1207.4 L1129.61,1207.47 L1129.91,1207.51 L1129.78,1207.54 L1129.68,1207.55 L1129.63,1207.63 L1129.65,1207.9 L1128.94,1207.55 L1128.16,1207.65 L1127.51,1208.11 L1127.21,1208.84 L1126.59,1208.11 L1125.85,1207.77 L1124.53,1207.7 L1123.94,1208 L1123.47,1208.74 L1123.27,1209.7 L1123.5,1210.68 L1122.98,1210.59 L1122.76,1210.51 L1122.49,1210.29 L1122.26,1210.72 L1122.52,1211.46 L1122.37,1212.21 L1121.93,1212.77 L1121.33,1212.93 L1121.62,1213.77 L1121.94,1214.43 L1122.04,1214.98 L1121.65,1215.49 L1123.28,1216.55 L1124.18,1216.75 L1124.95,1216.23 L1124.38,1216.27 L1123.94,1216.07 L1123.17,1215.44 L1123.38,1214.69 L1123.71,1214.14 L1124.6,1213.21 L1124.8,1213.06 L1125.2,1212.92 L1125.35,1212.79 L1125.46,1212.48 L1125.46,1211.69 L1125.53,1211.43 L1126.13,1210.88 L1128.47,1209.66 L1127.96,1209.3 L1127.73,1209.25 L1127.71,1208.83 L1128.38,1208.55 L1128.63,1208.56 L1128.93,1208.78 L1128.92,1208.36 L1129.62,1208.94 L1130.7,1208.61 L1132.65,1207.41 L1132.56,1207.2 L1132.45,1206.73 L1132.37,1206.52 L1132.74,1206.23 L1133.16,1206.04 L1133.6,1205.96 L1134.07,1205.99 L1134.09,1206.46 L1133.54,1206.62 L1133.19,1207.13 L1133.01,1207.85 L1132.95,1208.64 L1133.38,1207.83 L1133.53,1207.4 L1133.63,1206.9 L1133.98,1207.48 L1134.38,1208.02 L1134.79,1208.2 L1135.18,1207.7 L1135.62,1208.06 L1136.01,1208.03 L1136.36,1207.73 L1136.66,1207.26 L1136.78,1207.73 L1136.98,1208.18 L1137.22,1208.57 L1137.88,1209.34 L1138.02,1209.45 L1138.15,1209.38 L1138.51,1209.28 L1139.21,1209.3 L1139.58,1209.18 L1139.97,1208.79 L1140.27,1208.28 L1140.6,1207.25 L1140.9,1206.62 L1141.8,1205.66 L1143.21,1204.66 L1144.63,1204.06 L1145.57,1204.29 L1145.13,1204.76 L1145.14,1205.51 L1145.45,1206.29 L1145.91,1206.88 L1147.52,1207.63 L1147.98,1208.04 L1147.34,1209.3 L1147.53,1210.25 L1148.64,1211.89 L1148.78,1212.57 L1148.81,1213.25 L1148.94,1213.77 L1149.35,1213.97 L1150.98,1213.9 L1151.01,1214.36 L1150.66,1214.39 L1150.42,1214.5 L1150.02,1214.79 L1151.22,1214.62 L1153.18,1215.63 L1154.47,1215.88 L1155.43,1216.35 L1156.76,1218.78 L1157.79,1219.61 L1157.24,1220.04 L1156.83,1220.51 L1157.74,1220.46 L1159.08,1219.7 L1159.81,1219.95 L1159.56,1220.69 L1159.4,1220.99 L1159.12,1221.26 L1159.57,1221.51 L1159.83,1221.35 L1160.06,1221.13 L1160.4,1221.2 L1160.61,1221.51 L1161.21,1222.92 L1161.73,1223.82 L1162.3,1224.54 L1163,1224.96 L1163.87,1224.93 L1163.85,1224.5 L1163.53,1224.23 L1163.08,1223.53 L1162.76,1223.27 L1162.74,1222.84 L1167.23,1221.73 L1168.29,1221.17 L1170.1,1219.79 L1171.13,1219.39 L1171.31,1219.71 L1171.26,1219.99 L1171.03,1220.2 L1170.67,1220.31 L1170.69,1220.7 L1171.84,1221.22 L1174.89,1223.52 L1174.91,1223.94 L1174.43,1224.89 L1174.19,1226.11 L1173.8,1227.23 L1172.83,1227.9 L1173.29,1228.22 L1174.43,1229.53 L1174.85,1230.15 L1175.08,1230.3 L1176.14,1230.69 L1176.41,1230.86 L1176.59,1231.27 L1176.73,1231.72 L1176.86,1232.01 L1176.91,1232.23 L1176.88,1232.52 L1176.88,1232.76 L1177.03,1232.86 L1177.55,1232.78 L1177.68,1232.82 L1177.88,1232.98 L1178.12,1233.1 L1178.35,1233.18 L1178.56,1233.2 L1179.04,1233.43 L1179.13,1234.02 L1178.94,1234.73 L1178.55,1235.34 L1176.03,1237.67 L1175.69,1238.49 L1176.06,1239.24 L1176.77,1239.54 L1177.46,1239.66 L1177.79,1239.88 L1178.06,1240.55 L1178.64,1241.31 L1179.27,1241.81 L1179.66,1241.7 L1182.51,1243.84 L1183.75,1244.05 L1184.39,1244.77 L1184.55,1244.9 L1185.43,1244.78 L1185.71,1244.83 L1186.67,1245.34 L1187.13,1245.5 L1190.12,1245.55 L1191.15,1245.17 L1191.54,1244.28 L1191.8,1244.07 L1192.41,1244.15 L1192.97,1244.64 L1193.14,1245.65 L1192.87,1246.13 L1191.21,1247.48 L1191.69,1247.72 L1192.25,1247.88 L1192.85,1247.92 L1193.14,1247.87 L1193.14,1247.88 L1193.29,1249.63 L1193.19,1250.61 L1193.82,1251.8 L1194.75,1254.19 L1194.86,1254.8 L1194.89,1257.09 L1195,1258.47 L1195.35,1259.75 L1196.34,1261.09 L1196.81,1262.38 L1197.67,1263.65 L1198.26,1266.86 L1199.26,1270.46 L1199.59,1271.61 L1199.68,1272.14 L1199.77,1273.27 L1199.92,1273.88 L1200.18,1274.26 L1200.78,1274.71 L1200.95,1275.05 L1201.46,1275.47 L1200.61,1277.46 L1200.36,1277.89 L1199.97,1278.13 L1199.64,1278.71 L1199.41,1279.49 L1199.36,1280.32 L1199.56,1281.33 L1199.84,1282.2 L1199.97,1283.1 L1199.06,1286.71 L1198.99,1287.24 L1198.55,1288.4 L1197.47,1289.68 L1195.35,1291.58 L1193.13,1292.91 L1191.38,1294.38 L1191.41,1294.77 L1191.67,1295.38 L1192.26,1297.57 L1192.39,1298.35 L1192.3,1299.54 L1191.95,1300.32 L1191,1301.25 L1191.48,1302.1 L1192.17,1302.58 L1192.98,1302.79 L1193.82,1302.8 L1194.6,1303.09 L1195.92,1304.48 L1196.99,1304.97 L1201.19,1307.71 L1201.82,1308.36 L1203.28,1310.84 L1203.69,1311.2 L1204.44,1311.5 L1205.62,1312.92 L1207.95,1313.68 L1210.31,1316.03 L1210.55,1316.26 L1211.79,1317.16 L1210.87,1319.47 L1210.67,1320.94 L1211.42,1321.54 L1211.64,1322.2 L1211.1,1323.7 L1210.27,1325.23 L1208.41,1327.1 L1208.78,1329.57 L1209.86,1332.24 L1210.77,1334 L1210.62,1335.15 L1211.54,1336.07 L1212.88,1336.65 L1213.98,1336.83 L1215.08,1337.24 L1215.75,1338.29 L1215.82,1339.57 L1215.14,1340.61 L1215.16,1341 L1215.77,1341.83 L1215.68,1342.74 L1215.34,1343.9 L1215.22,1345.53 L1215.66,1346.41 L1218.28,1348.12 L1217.63,1349.05 L1217.37,1349.6 L1217.28,1350.17 L1217.25,1350.74 L1217.12,1351.34 L1216.39,1352.91 L1216.22,1353.53 L1216.33,1357.35 L1216.09,1358.6 L1215.56,1359.76 L1214.91,1360.76 L1212.84,1363.16 L1212.4,1364.28 L1212.6,1365.59 L1213.3,1366.52 L1214.03,1366.97 L1214.61,1367.65 L1214.87,1369.29 L1215.37,1370.85 L1216.31,1371.82 L1218.29,1373.27 L1218.87,1374.32 L1219.35,1375.88 L1219.52,1377.52 L1219.14,1378.8 L1218.47,1380.05 L1218.23,1381.28 L1218.34,1381.72 L1218.51,1382.38 L1219.43,1383.22 L1222.03,1383.91 L1222.89,1384.44 L1222.89,1384.45 L1223.82,1384.9 L1225.68,1385.32 L1226.59,1385.69 L1227.23,1386.12 L1228,1386.83 L1228.46,1387.66 L1228.19,1388.47 L1228.44,1390.1 L1228.86,1392.74 L1228.86,1392.79 L1229.11,1393.9 L1229.24,1394.33 L1230.05,1395.55 L1231.01,1396.53 L1231.66,1397.72 L1231.59,1399.54 L1231.74,1399.73 L1231.92,1399.89 L1231.33,1401.26 L1230.92,1402.96 L1230.72,1404.73 L1230.78,1406.31 L1230.75,1407.12 L1230.52,1407.55 L1230.24,1407.86 L1230.05,1408.31 L1230.03,1408.93 L1230.1,1410.13 L1230.03,1410.85 L1228.57,1415.43 L1226.9,1420.3 L1225.6,1421.95 L1225.1,1422.91 L1225.17,1424.07 L1224.66,1425.79 L1224.33,1426.54 L1223.97,1427 L1223.35,1427.22 L1222.49,1427.24 L1221,1427.01 L1219.41,1425.96 L1218.83,1425.68 L1217.43,1425.38 L1217.21,1424.92 L1217.28,1423.8 L1217.85,1422.18 L1218.03,1421.34 L1217.77,1420.65 L1217.29,1420.67 L1215.17,1421.71 L1214.69,1421.38 L1214.86,1420.61 L1215.28,1419.68 L1215.54,1418.91 L1215.55,1417.97 L1215.39,1417.36 L1215.06,1416.89 L1214.55,1416.34 L1214.04,1415.99 L1212.96,1415.79 L1212.45,1415.48 L1212.05,1414.99 L1211.89,1414.62 L1211.78,1414.25 L1211.52,1413.73 L1211.23,1414.62 L1210.8,1414.88 L1209.37,1414.88 L1208.29,1415.44 L1207.86,1415.46 L1207.42,1415.25 L1207.14,1414.92 L1206.91,1414.59 L1206.6,1414.34 L1205.11,1413.94 L1203.86,1414.22 L1202.98,1415.23 L1202.53,1417.06 L1202.12,1417.73 L1202.6,1418 L1203.94,1418.1 L1204.75,1418.37 L1204.7,1418.5 L1204.37,1418.75 L1204.38,1419.41 L1205.44,1420.35 L1206.92,1420.69 L1208.01,1421.24 L1207.87,1422.81 L1206.77,1423.86 L1203.65,1423.8 L1202.25,1424.18 L1201.85,1424.63 L1201.19,1425.76 L1200.73,1426.24 L1194.72,1429.42 L1193.96,1429.51 L1192.51,1429.4 L1191.74,1429.61 L1191.33,1429.94 L1190.77,1430.75 L1190.38,1431.07 L1189.89,1431.18 L1189.49,1431.08 L1189.16,1431.11 L1188.82,1431.58 L1188.68,1432.19 L1188.76,1432.72 L1188.88,1433.16 L1188.87,1433.49 L1188.62,1433.89 L1188.4,1433.98 L1188.16,1434 L1187.05,1434.78 L1186.34,1434.96 L1184.51,1434.85 L1183.66,1434.98 L1182.18,1435.62 L1181.42,1435.81 L1177.86,1435.97 L1176.08,1436.39 L1174.5,1437.45 L1175,1438.39 L1174.89,1439.24 L1174.39,1439.95 L1173.76,1440.48 L1174.01,1440.97 L1173.99,1441.35 L1173.77,1441.66 L1173.38,1441.9 L1172.73,1442.79 L1172.09,1443.44 L1171.36,1443.66 L1170.45,1443.26 L1170.17,1442.95 L1169.57,1442.03 L1169.34,1441.8 L1168.85,1441.65 L1168.74,1441.73 L1168.73,1441.96 L1168.5,1442.22 L1167.06,1443.35 L1166.85,1443.76 L1166.49,1445.04 L1165.67,1445.51 L1165,1445.24 L1164.33,1444.82 L1163.55,1444.78 L1162.38,1448.39 L1161.86,1449.45 L1160.91,1450.22 L1158.8,1450.19 L1157.69,1450.63 L1157.68,1450.63 L1157.68,1450.63 L1157.32,1450.75 L1156.95,1450.79 L1156.58,1450.78 L1156.22,1450.69 L1155.9,1450.47 L1155.57,1450.41 L1155.24,1450.5 L1154.92,1450.75 L1154.76,1453.31 L1154.55,1454.46 L1154.01,1455.15 L1153.18,1456.04 L1152.87,1456.37 L1151.5,1456.35 L1147.71,1454.24 L1147.37,1454.16 L1147.02,1454.24 L1146.47,1454.64 L1146.23,1454.74 L1145.51,1454.67 L1144.87,1454.73 L1144.23,1454.99 L1143.73,1455.52 L1142.9,1456.92 L1142.47,1457.37 L1141.95,1457.45 L1139.8,1457.14 L1136.82,1457.46 L1135.23,1458 L1134.15,1458.98 L1133.86,1459.25 L1133.73,1459.57 L1133.66,1459.89 L1133.56,1460.17 L1133.57,1460.3 L1133.64,1460.51 L1133.67,1460.75 L1133.59,1460.98 L1133.43,1461.08 L1133.07,1461.08 L1132.93,1461.17 L1132.47,1461.61 L1131.31,1462.31 L1130.9,1462.75 L1129.73,1465.04 L1129.29,1465.54 L1128.85,1465.87 L1128.54,1466.36 L1128.46,1467.36 L1127.7,1468.4 L1127.51,1468.99 L1127.42,1470.11 L1127.56,1470.88 L1127.81,1471.54 L1127.83,1472.1 L1127.28,1472.52 L1126.6,1472.46 L1126.17,1471.84 L1125.64,1470.13 L1125.58,1469.81 L1125.5,1469.44 L1125.32,1469.04 L1125.14,1468.81 L1124.42,1468.13 L1124.31,1467.98 L1124.27,1467.85 L1124.3,1467.72 L1124.4,1467.6 L1124.45,1467.55 L1124.51,1467.51 L1124.56,1467.49 L1124.62,1467.48 L1124.79,1467.12 L1124.8,1466.83 L1124.66,1466.61 L1124.37,1466.46 L1123.22,1466.47 L1122.63,1466.3 L1122.17,1465.92 L1122,1465.44 L1121.98,1464.37 L1121.73,1463.79 L1120.94,1463.27 L1119.02,1463.19 L1118,1463.14 L1118.95,1464.08 L1119.29,1464.62 L1119.47,1465.51 L1119.42,1466.31 L1119.15,1466.61 L1118.75,1466.9 L1118.27,1467.67 L1118.7,1468.4 L1120.72,1469.58 L1121.73,1470.63 L1122.21,1471.31 L1122.54,1472.05 L1122.68,1472.98 L1122.44,1474.41 L1122.49,1475.35 L1122.87,1476.29 L1123.45,1476.79 L1124.11,1477.14 L1124.71,1477.63 L1125.08,1478.23 L1125.19,1478.87 L1125.21,1479.44 L1125.35,1479.83 L1131.55,1482.9 L1131.59,1482.9 L1132.23,1483.49 L1133.72,1483.59 L1134.42,1484.22 L1134.55,1485.13 L1134.3,1485.98 L1134.28,1486.66 L1135.96,1487.57 L1136.54,1487.78 L1136.81,1488.22 L1136.73,1489.44 L1136.43,1490.41 L1135.56,1492.16 L1135.29,1493.14 L1134.48,1492.92 L1134.26,1493.81 L1134.28,1495.12 L1134.18,1496.15 L1133.56,1496.82 L1131.91,1497.59 L1131.46,1498.54 L1131.68,1499.2 L1132.13,1499.98 L1133.01,1501.16 L1133.61,1501.67 L1134.24,1502.07 L1135.61,1502.67 L1136.18,1502.95 L1136.54,1503.6 L1136.6,1504.43 L1136.28,1505.2 L1136.39,1505.28 L1136.72,1505.43 L1136.54,1505.93 L1136.52,1506.08 L1136.95,1506.57 L1137.94,1507.07 L1138.38,1507.53 L1138.69,1508.2 L1138.93,1509.58 L1139.11,1510.3 L1139.4,1511.04 L1139.68,1511.59 L1140.01,1512.05 L1140.47,1512.5 L1140.93,1512.62 L1141.36,1512.54 L1141.67,1512.78 L1141.76,1513.83 L1141.95,1515.88 L1142.19,1516.88 L1142.69,1517.77 L1143.51,1518.4 L1145.42,1519.22 L1146.27,1520.04 L1147.72,1522.98 L1148.39,1523.63 L1149.21,1523.89 L1151.63,1524.15 L1152.39,1524.02 L1152.18,1523.22 L1152.73,1523.02 L1154.31,1523.39 L1155.02,1523.73 L1155.63,1524.18 L1156.83,1525.36 L1157.55,1525.62 L1157.8,1526.12 L1157.88,1526.72 L1158.09,1527.3 L1159.45,1528.8 L1159.46,1528.8 L1159.46,1528.81 L1159.98,1529.07 L1164.03,1534.12 L1164.62,1535.33 L1165.32,1536.35 L1166.51,1536.97 L1169,1536.98 L1169.83,1537.26 L1170.35,1537.67 L1171.67,1539.28 L1173.5,1540.98 L1174.01,1542.05 L1174.11,1543.8 L1174.74,1544.73 L1175.98,1545.78 L1177.32,1546.65 L1178.3,1547.07 L1178.68,1545.96 L1179.04,1545.26 L1179.59,1544.97 L1179.86,1545.01 L1180.55,1545.12 L1181.29,1545.53 L1182.01,1546.05 L1182.74,1546.41 L1183.48,1546.29 L1183.64,1547.46 L1184.13,1548.58 L1184.82,1549.52 L1185.56,1550.19 L1186.35,1550.52 L1187.9,1550.68 L1188.5,1550.97 L1189.53,1552.42 L1191.72,1556.29 L1192.57,1557.08 L1191.47,1559.75 L1191.4,1560.35 L1191.61,1560.26 L1192.1,1561.1 L1192.31,1561.28 L1192.84,1561.5 L1192.32,1562.06 L1192.25,1562.81 L1192.57,1564.81 L1192.59,1565.87 L1192.51,1566.77 L1192.29,1567.6 L1191.66,1569.18 L1191.48,1569.49 L1191.15,1569.84 L1190.83,1570 L1189.97,1570.19 L1189.83,1570.16 L1189.51,1570.89 L1189.23,1572.58 L1189.13,1572.48 L1187.39,1571.8 L1186.71,1570.86 L1186.01,1570.39 L1185.27,1570.04 L1180.92,1569.05 L1179.54,1569.3 L1178.21,1569.87 L1177.85,1570.23 L1177.64,1570.68 L1177.68,1570.9 L1177.82,1571.18 L1178.03,1572.3 L1178.45,1572.88 L1178.56,1573.43 L1178.48,1574.04 L1177.97,1575.25 L1177.8,1575.83 L1177.75,1576.42 L1177.91,1578.18 L1177.85,1578.83 L1177.69,1579.27 L1177.51,1579.6 L1177.44,1579.93 L1177.27,1581.26 L1176.73,1582.26 L1172.82,1586 L1171.45,1586.85 L1165.75,1588.1 L1161.49,1590 L1160.84,1590.5 L1159.35,1592.06 L1158.03,1592.86 L1157.4,1593.41 L1157.46,1593.56 L1155.17,1594.1 L1154.55,1594.46 L1154.18,1594.89 L1153.46,1596.13 L1152.94,1596.77 L1151.32,1598 L1151.16,1598.21 L1150.6,1599.17 L1150.35,1599.3 L1150.07,1599.29 L1149.81,1599.4 L1149.57,1599.85 L1149.52,1600.7 L1149.76,1601.52 L1150.16,1602.25 L1150.57,1602.86 L1153.64,1605.83 L1154.32,1607.2 L1154.45,1607.87 L1154.47,1607.96 L1154.72,1608.6 L1155.08,1609.08 L1156.11,1609.65 L1156.99,1610.49 L1157.33,1610.66 L1158.07,1611.32 L1159.62,1614.57 L1160.58,1615.89 L1160.79,1616.08 L1158.24,1620.54 L1158.03,1621.04 L1157.96,1621.44 L1157.96,1621.84 L1157.9,1622.26 L1157.65,1622.71 L1157.65,1622.71 L1157.64,1622.72 L1156.9,1623.96 L1157.62,1624.63 L1159.99,1624.98 L1160.62,1624.84 L1161.67,1624.33 L1162.32,1624.41 L1162.86,1624.78 L1163.41,1625.39 L1164.29,1626.76 L1164.66,1627.66 L1164.81,1628.45 L1164.81,1629.21 L1164.75,1629.97 L1164.65,1630.11 L1164.2,1631.37 L1164.18,1631.54 L1163.54,1632.46 L1163.42,1632.43 L1163.51,1633.84 L1163.6,1633.8 L1163.56,1633.89 L1163.2,1634.92 L1163.1,1635.09 L1163.14,1636.18 L1163.19,1636.56 L1163.3,1636.9 L1163.45,1637.22 L1163.57,1637.59 L1163.61,1638.11 L1162.46,1639.59 L1162.17,1639.86 L1161.72,1639.9 L1161.45,1639.75 L1161.2,1639.51 L1160.79,1639.31 L1160.34,1639.34 L1160,1639.54 L1159.65,1639.68 L1159.18,1639.51 L1157.04,1638.03 L1155.03,1635.97 L1154.66,1635.7 L1154.25,1635.57 L1153.6,1635.48 L1152.52,1634.68 L1152.24,1633.15 L1152.7,1631.68 L1153.84,1631.07 L1152.97,1630.29 L1151.85,1629.61 L1151.15,1628.82 L1151.55,1627.71 L1150.83,1627.83 L1148.44,1627.31 L1146.93,1627.36 L1145.44,1627.78 L1144.65,1628.21 L1143.21,1629.58 L1142.82,1629.95 L1142.17,1630.27 L1141.29,1630.47 L1140.42,1630.48 L1139.81,1630.21 L1138.52,1629.24 L1138.17,1628.84 L1137.92,1628.34 L1137.44,1627.02 L1137.23,1626.67 L1136.53,1626.56 L1134.9,1627.25 L1134.18,1627.38 L1131.73,1626.94 L1130.81,1627.1 L1129.88,1627.62 L1129.43,1627.71 L1129,1627.35 L1128.75,1626.66 L1128.87,1626.22 L1129.13,1625.84 L1129.27,1625.34 L1129.35,1624.67 L1129.48,1624.38 L1129.37,1624.41 L1128.77,1624.7 L1128.41,1624.94 L1127.42,1625.81 L1126.76,1626.1 L1126.85,1626.61 L1127,1626.95 L1127.32,1627.35 L1127.67,1627.69 L1127.89,1627.82 L1128.02,1628.16 L1128.14,1629.83 L1128.11,1630.41 L1127.97,1630.82 L1126.81,1632.4 L1126.47,1632.48 L1116.75,1632.33 L1113.21,1633.14 L1112.74,1633.44 L1112.35,1634.51 L1111.91,1634.66 L1109.52,1634.23 L1106.07,1634.3 L1104.22,1633.97 L1103.43,1633.96 L1103.29,1634.15 L1102.13,1635.2 L1101.86,1635.86 L1101.55,1637.32 L1101.35,1637.9 L1100.6,1638.77 L1099.69,1639.16 L1097.7,1639.53 L1096.74,1639.24 L1095.69,1639.12 L1094.73,1639.36 L1092.78,1641.7 L1093.3,1641.83 L1093.77,1642.12 L1093.94,1642.59 L1093.59,1643.21 L1092.9,1643.52 L1091.06,1643.6 L1090.28,1643.86 L1089.47,1644.64 L1088.92,1645.4 L1088.32,1646.06 L1087.4,1646.5 L1086.38,1646.4 L1086.45,1644.7 L1085.54,1644.35 L1084.47,1644.66 L1081.72,1646.62 L1080.9,1646.88 L1076.47,1646.88 L1075.87,1646.54 L1075.42,1645.7 L1075.53,1645.69 L1075.66,1645.4 L1075.75,1644.97 L1075.73,1644.55 L1075.6,1644.27 L1073.45,1642.02 L1071.26,1641.03 L1070.96,1640.54 L1072.32,1639.59 L1072.67,1639.16 L1071.69,1638.44 L1071.24,1638.19 L1070.65,1638.16 L1070.05,1638.33 L1068.35,1639.13 L1067.1,1639.3 L1066.73,1639.19 L1066.52,1638.9 L1066.38,1638.58 L1066.2,1638.38 L1060.6,1636.32 L1059.57,1636.32 L1058.99,1636.73 L1058.14,1637.93 L1057.6,1638.37 L1057.1,1638.46 L1055.3,1638.2 L1054.64,1637.9 L1054.29,1637.65 L1054.06,1637.37 L1053.99,1636.83 L1054.14,1636.43 L1054.34,1636.13 L1054.4,1635.89 L1054.2,1635.31 L1054.05,1635.19 L1053.73,1635.32 L1053.02,1635.45 L1052.39,1635.7 L1052.45,1636.24 L1052.83,1636.93 L1053.11,1637.63 L1053.06,1638.21 L1052.64,1640.7 L1053.24,1641.02 L1053.61,1641.42 L1053.84,1641.97 L1054.02,1642.74 L1054.04,1644.08 L1053.66,1645.49 L1053.05,1646.76 L1052.35,1647.7 L1051.98,1647.95 L1051.09,1648.26 L1050.65,1648.56 L1050.4,1648.95 L1049.46,1650.86 L1048.7,1651.89 L1047.87,1652.65 L1045.97,1653.82 L1045.04,1654.19 L1042.3,1654.53 L1041.66,1654.61 L1041.47,1654.5 L1041.64,1654.11 L1042.03,1653.4 L1042.2,1653.28 L1042.77,1653.11 L1042.97,1652.91 L1043.08,1652.49 L1043.05,1652.12 L1042.96,1651.71 L1043.72,1649.4 L1043.75,1648.27 L1042.96,1647.85 L1041.1,1648.19 L1040.59,1648.75 L1040.41,1648.89 L1040.19,1648.89 L1039.75,1648.62 L1039.54,1648.6 L1038.99,1648.86 L1038.66,1649.11 L1038.38,1649.12 L1037.96,1648.72 L1037.73,1648.12 L1037.69,1647.45 L1037.58,1646.8 L1037.11,1646.26 L1038.08,1645.51 L1038.27,1644.85 L1037.9,1644.12 L1035.81,1641.07 L1035.33,1641.29 L1034.95,1641.62 L1034.58,1641.78 L1034.11,1641.5 L1034.05,1641.14 L1034.04,1640.95 L1034.04,1640.76 L1033.65,1639.92 L1033.12,1639.39 L1032.68,1638.8 L1032.56,1637.77 L1032.05,1638.18 L1031.41,1638.3 L1030.15,1638.2 L1028.87,1637.73 L1028.66,1637.79 L1028.14,1638.12 L1027.8,1638.14 L1026.93,1637.7 L1026.77,1637.04 L1026.86,1636.28 L1026.72,1635.54 L1026.23,1635 L1025.63,1634.78 L1025,1634.85 L1024.38,1635.17 L1023.45,1636.24 L1022.93,1637.37 L1022.31,1638.33 L1021.15,1638.87 L1018.43,1638.91 L1015.96,1638.21 L1015.78,1638.16 L1015.68,1638.1 L1004.13,1630.79 L1002.49,1630.39 L1000.91,1630.38 L1000.34,1629.48 L998.007,1629.45 L993.302,1628.86 L992.498,1629.17 L991.816,1629.89 L990.272,1630.36 L988.614,1630.49 L987.583,1630.2 L986.369,1629.27 L986.338,1629.24 L985.75,1628.63 L985.752,1628.2 L986.532,1628.03 L985.456,1626.98 L984.09,1626.15 L982.956,1626.08 L982.557,1627.3 L982.895,1627.7 L980.664,1627.69 L980.588,1627.28 L980.492,1626.8 L980.163,1626.38 L979.993,1625.87 L980.139,1625.46 L980.716,1624.63 L980.833,1624.39 L980.565,1623.76 L980.152,1623.67 L979.684,1623.76 L979.25,1623.65 L978.945,1623.15 L978.626,1622.42 L978.232,1621.8 L977.69,1621.61 L977.3,1622 L977.057,1623.36 L976.528,1623.69 L976.12,1623.38 L975.985,1622.61 L975.961,1621.81 L975.904,1621.35 L975.131,1620.98 L974.082,1620.91 L973.404,1621.28 L973.777,1622.27 L973.135,1622.58 L970.855,1622.99 L970.385,1622.96 L970.05,1623.15 L969.462,1623.98 L969.256,1624.43 L968.912,1625.67 L968.454,1626.08 L967.373,1626.65 L966.949,1627.11 L966.891,1627.57 L967.06,1628.03 L967.154,1628.57 L966.869,1629.24 L967.527,1629.5 L967.736,1629.53 L968.803,1630.38 L969.666,1630.9 L970.418,1630.88 L971.049,1630.57 L971.643,1630.16 L972.283,1629.86 L974.342,1629.55 L974.93,1629.66 L975.408,1629.87 L975.602,1629.95 L975.985,1630 L975.718,1631.47 L975.445,1631.33 L975.426,1630.94 L975.389,1630.84 L975.205,1630.85 L974.981,1630.93 L974.825,1631.05 L974.723,1631.4 L974.787,1631.68 L974.883,1631.93 L974.897,1632.14 L974.836,1632.57 L974.85,1633.01 L974.793,1633.48 L974.604,1633.8 L974.509,1633.96 L973.936,1634.14 L973.565,1633.66 L973.479,1633.55 L972.993,1632.7 L972.353,1632.08 L971.083,1632.18 L969.772,1633.01 L969.256,1634.07 L969.198,1634.36 L968.019,1634.54 L967.89,1634.55 L965.183,1634.48 L963.587,1634.02 L963.177,1633.74 L962.894,1633.13 L962.68,1632.52 L962.458,1632.24 L961.97,1632.18 L960.881,1631.82 L960.116,1631.8 L957.825,1632.13 L957.603,1632.27 L957.386,1632.57 L957.09,1632.87 L956.311,1633.08 L955.381,1633.55 L955.029,1633.81 L954.66,1634.21 L954.481,1634.52 L954.286,1634.79 L953.868,1635.06 L951.954,1635.44 L946.446,1635.29 L946.34,1635.02 L946.273,1634.43 L946.153,1633.84 L945.903,1633.56 L943.179,1633.32 L942.593,1633 L941.789,1634.18 L941.092,1634.96 L940.282,1635.39 L938.603,1635.69 L936.734,1636.02 L935.806,1635.86 L935.201,1635.52 L933.654,1634.64 L933.764,1634.64 L934.76,1634.68 L935.224,1634.26 L935.828,1632.72 L934.867,1632.79 L932.69,1633.36 L932.882,1632.73 L932.99,1632.51 L932.917,1631.91 L932.705,1631.58 L932.441,1631.33 L932.198,1631.01 L931.306,1629.08 L930.76,1628.26 L930.147,1627.91 L929.99,1627.62 L929.877,1626.94 L929.803,1625.6 L929.963,1625.23 L930.685,1624.51 L930.947,1624.1 L931.002,1623.08 L930.55,1620.88 L930.692,1620.02 L931.358,1618.01 L931.603,1617.72 L932.182,1617.43 L932.296,1616.73 L932.116,1615.56 L932.134,1615.28 L932.104,1615.12 L932.087,1614.97 L932.144,1614.71 L932.278,1614.44 L932.565,1614.18 L932.744,1613.88 L933.03,1613.26 L933.117,1612.98 L933.097,1611.96 L933.349,1611.14 L934.507,1609.99 L934.93,1609.25 L934.9,1607.3 L933.189,1604.05 L933.142,1601.47 L933.445,1600.26 L933.756,1599.31 L934.077,1598.76 L934.374,1598.4 L934.602,1597.91 L934.712,1597.01 L935.013,1596.26 L937.771,1592.64 L938.239,1591.75 L938.535,1590.7 L938.66,1589.43 L938.989,1588.13 L939.678,1587.6 L940.457,1587.21 L941.058,1586.31 L941.136,1585.44 L940.949,1584.6 L940.702,1583.79 L940.591,1583.03 L940.663,1582.22 L941.018,1580.57 L941.25,1579.87 L941.555,1579.29 L941.838,1578.89 L942.067,1578.29 L942.188,1577.1 L942.419,1576.54 L942.932,1576.05 L943.995,1575.26 L944.008,1574.83 L943.557,1573.49 L943.625,1571.31 L944.036,1569.28 L945.862,1566.61 L946.355,1566.28 L947.046,1566.07 L947.695,1565.54 L948.867,1564.22 L949.367,1563.91 L949.93,1563.4 L950.39,1562.81 L950.588,1562.32 L950.671,1561.13 L950.904,1560.58 L951.378,1560.44 L952.541,1560.45 L952.86,1560.35 L953.101,1560.15 L953.199,1559.81 L953.304,1559.22 L953.505,1558.9 L953.702,1558.72 L953.796,1558.57 L954.159,1558.56 L955.696,1557.78 L955.954,1557.57 L956.488,1556.73 L958.488,1551.62 L959.834,1549.22 L960.704,1548.27 L960.722,1548.25 L960.256,1547.79 L956.23,1546.85 L949.801,1543.19 L949.059,1543.09 L946.711,1543.26 L943.32,1542.17 L942.862,1542.28 L941.882,1542.78 L941.376,1542.89 L940.983,1542.82 L940.122,1542.48 L939.67,1542.39 L939.148,1542.48 L938.661,1542.68 L938.165,1542.79 L938.006,1542.74 L937.599,1542.64 L937.227,1542.3 L936.854,1541.48 L936.564,1541.09 L935.667,1540.59 L933.881,1540.22 L933.004,1539.62 L932.488,1538.92 L931.609,1537.38 L931.053,1536.76 L931.132,1536.55 L931.238,1536.41 L931.55,1536.23 L931.309,1535.31 L930.577,1535.02 L928.81,1534.84 L928.656,1534.13 L927.862,1534.04 L927.057,1534.25 L926.89,1534.29 L926.194,1534.66 L925.992,1534.96 L925.663,1535.86 L925.443,1536.19 L925.129,1536.33 L924.286,1536.4 L923.916,1536.52 L923.647,1536.82 L923.182,1537.69 L922.931,1537.93 L922.674,1537.88 L921.59,1537.32 L918.055,1537.15 L917.418,1536.91 L916.787,1536.55 L915.262,1535.34 L915.039,1535.31 L914.652,1535.66 L914.672,1536.01 L914.791,1536.34 L914.715,1536.65 L913.424,1537.4 L912.549,1536.65 L912.143,1534.92 L912.252,1532.71 L911.369,1532.63 L908.871,1531.08 L908.421,1530.99 L907.222,1531.06 L905.529,1530.59 L905.114,1530.68 L904.841,1531.53 L905.107,1532.56 L905.22,1533.6 L904.473,1534.51 L903.947,1534.61 L902.375,1533.95 L900.55,1533.85 L900.076,1533.62 L899.589,1532.72 L899.737,1531.96 L899.979,1531.18 L899.765,1530.18 L899.6,1530.17 L899.034,1530.51 L898.851,1530.52 L898.751,1530.29 L898.692,1529.8 L898.652,1529.66 L898.532,1529.39 L898.263,1528.22 L898.097,1528.03 L897.54,1527.8 L897.328,1527.62 L897.293,1527.37 L897.405,1526.74 L897.354,1526.47 L894.755,1523.72 L894.109,1522.86 L893.885,1522.12 L894.135,1521.56 L894.862,1521.56 L895.063,1521.06 L894.941,1520.68 L893.056,1518.48 L892.793,1518.05 L892.875,1518.03 L892.882,1517.68 L892.797,1517.16 L892.568,1516.63 L891.901,1515.92 L888.252,1513.87 L887.805,1513.77 L887.467,1513.89 L886.666,1514.38 L886.229,1514.44 L885.882,1514.39 L885.908,1513.93 L885.874,1512.03 L886.239,1510.03 L886.347,1509.43 L886.444,1507.45 L886.709,1506.77 L887.368,1505.84 L888.489,1504.59 L888.941,1503.76 L889.269,1503.33 L889.391,1503.05 L889.374,1502.64 L889.138,1502.38 L888.886,1502.19 L888.825,1501.98 L889.496,1501.13 L892.567,1499.34 L892.494,1499.21 L892.232,1498.87 L892.385,1498.58 L892.578,1498.49 L892.923,1498.65 L892.652,1498.25 L892.481,1497.79 L892.413,1497.27 L892.468,1496.72 L892.878,1496.39 L892.906,1496.15 L892.81,1495.86 L892.835,1495.42 L893.092,1494.07 L893.152,1494.03 L893.181,1493.51 L893.273,1493.42 L893.285,1493.44 L893.079,1493.19 L892.92,1493.17 L891.692,1492.75 L889.753,1492.65 L889.057,1492.42 L888.393,1491.95 L887.286,1490.79 L886.622,1490.32 L886.268,1490.3 L885.644,1490.63 L885.32,1490.59 L885.288,1490.38 L885.078,1489.72 L884.937,1489.28 L884.762,1488.92 L884.193,1488.51 L882.855,1487.83 L882.257,1487.16 L881.095,1483.9 L880.461,1482.74 L879.984,1483.58 L879.691,1483.01 L879.17,1481.23 L878.53,1480.13 L878.382,1479.61 L878.531,1478.93 L878.349,1477.61 L878.114,1477.08 L877.691,1476.81 L878.103,1476.1 L878.264,1475.9 L878.173,1475.88 L877.931,1475.89 L877.863,1475.84 L878.208,1474.1 L878.448,1473.25 L878.774,1472.36 L879.162,1471.95 L879.3,1471.54 L879.179,1471.15 L878.789,1470.8 L878.893,1470.5 L879.062,1469.8 L879.147,1469.6 L879.503,1469.41 L880.306,1469.38 L880.615,1469.2 L880.886,1468.63 L880.947,1467.92 L880.76,1467.2 L880.257,1466.58 L880.699,1466.06 L881.639,1465.46 L882.064,1464.88 L882.916,1464.53 L884.663,1464.13 L885.433,1463.5 L885.283,1462.93 L885.398,1462.31 L885.693,1461.72 L886.114,1461.26 L886.671,1460.97 L887.116,1461.03 L887.562,1461.23 L888.095,1461.36 L889.167,1461.22 L889.693,1460.71 L889.672,1460.23 L889.653,1459.83 L888.384,1457.35 L888.367,1456.6 L888.685,1455.78 L889.043,1454.32 L889.123,1453.28 L888.981,1453.06 L888.699,1452.99 L888.385,1452.41 L888.198,1451.77 L888.111,1451.2 L888.234,1450.7 L888.701,1450.29 L887.978,1450.21 L886.272,1449.73 L885.677,1449.41 L885.591,1449.31 L885.502,1449.28 L885.401,1449.3 L885.311,1449.39 L883.943,1449.73 L883.548,1449.69 L883.244,1449.42 L882.939,1448.69 L882.633,1448.38 L882.978,1447.85 L882.515,1447.79 L882.201,1447.64 L882.162,1447.36 L882.5,1446.89 L881.881,1446.02 L882.672,1444.91 L883.98,1443.93 L884.895,1443.45 L884.907,1443.22 L885.387,1442.7 L885.694,1442.15 L885.623,1441.78 L882.883,1441.57 L882.088,1441.09 L881.671,1440.17 L882.218,1439.86 L879.995,1436.04 L879.273,1435.24 L878.617,1435.14 L877.024,1435.49 L876.503,1435.4 L876.375,1434.87 L876.558,1434.21 L876.57,1433.61 L875.93,1433.23 L875.879,1433.11 L875.881,1433 L875.911,1432.89 L875.983,1432.77 L876.114,1431.97 L875.876,1431.57 L875.498,1431.27 L875.224,1430.73 L875.1,1429.92 L875.693,1430.11 L876.291,1429.61 L876.54,1428.88 L876.485,1428.3 L876.478,1427.76 L876.859,1427.15 L877.419,1426.87 L878.306,1426.99 L878.776,1426.49 L879.051,1425.86 L879.09,1425.36 L878.824,1424.06 L878.92,1423.65 L879.146,1423.32 L879.237,1423.07 L877.972,1422.56 L877.073,1421.92 L876.789,1421.57 L876.715,1421.07 L876.707,1420.16 L876.886,1419.65 L877.155,1419.14 L877.091,1418.81 L875.668,1418.9 L874.651,1418.5 L874.109,1418.4 L873.567,1418.52 L872.516,1418.97 L871.972,1419.05 L872.02,1417.47 L871.514,1415.64 L871.373,1414.17 L872.508,1413.65 L873.036,1413.98 L873.47,1414.62 L873.968,1415.13 L874.697,1415.09 L875.211,1414.59 L876.055,1413.04 L876.617,1412.46 L880.409,1409.8 L882.712,1408.7 L883.337,1407.96 L882.554,1407.14 L883.49,1406.46 L883.812,1406.29 L882.927,1405.93 L880.9,1407.17 L880.006,1406.61 L879.788,1405.44 L879.977,1404.05 L880.38,1402.83 L880.818,1402.16 L881.698,1401.19 L883.903,1397.59 L884.332,1397.21 L885.364,1396.57 L885.781,1396.04 L886.031,1395.27 L886.224,1393.91 L886.478,1393.37 L885.925,1392.64 L886.038,1391.49 L886.405,1390.17 L886.611,1388.94 L886.48,1387.8 L886.309,1385.74 L884.833,1383.84 L882.432,1380.13 L882.304,1379.36 L882.561,1378.47 L883.09,1377.1 L880.522,1375.8 L879.693,1374.61 L880.311,1372.87 L879.948,1372.45 L879.118,1371.75 L878.684,1371.51 L877.557,1371.48 L877.18,1371.3 L877.25,1370.63 L877.492,1370.41 L878.159,1369.82 L878.297,1368.56 L877.848,1367.32 L877.005,1366.6 L877.168,1366.05 L877.838,1365.63 L879.433,1365.43 L880.199,1365.15 L881.031,1364.6 L881.603,1364.33 L883.201,1363.99 L884.439,1364.4 L885.338,1364.68 L885.954,1364.87 L885.477,1364.02 L884.115,1362.43 L883.655,1362.04 L885.008,1361.39 L886.223,1362.05 L887.413,1363.12 L888.692,1363.73 L889.913,1363.52 L890.231,1363.58 L890.493,1363.86 L890.881,1364.65 L891.046,1364.88 L892.519,1365.18 L893.128,1365.57 L893.195,1366.55 L894.668,1366.22 L894.859,1365.61 L894.613,1364.18 L897.056,1365.04 L897.844,1365.08 L898.498,1364.82 L900.472,1363.44 L904.382,1362.31 L907.81,1362.35 L908.882,1361.83 L909.829,1360.8 L910.782,1359.24 L911.113,1358.9 L911.41,1358.86 L911.614,1358.68 L911.684,1357.92 L911.606,1357.49 L911.42,1357.18 L910.47,1356.08 L908.014,1354.56 L907.181,1354.27 L906.587,1353.81 L906.551,1352.85 L906.847,1352.21 L907.201,1352.06 L907.609,1352.05 L908.081,1351.84 L908.409,1351.46 L908.76,1350.68 L908.978,1350.33 L909.661,1349.68 L910.293,1349.36 L910.514,1349.35 L912.8,1349.19 L913.304,1348.77 L914.23,1347.07 L915.011,1346.32 L916.69,1345.66 L917.486,1345.01 L918.448,1343.58 L918.881,1343.12 L920.631,1342.15 L920.751,1341.83 L920.662,1341.65 L920.454,1341.22 L920.168,1339.87 L920.239,1338.73 L920.555,1337.8 L921.396,1336.04 L921.814,1333.78 L921.342,1332.15 L919.172,1328.26 L918.926,1328.36 L918.672,1328.77 L918.302,1329.1 L917.335,1329.34 L916.329,1329.36 L915.258,1329.14 L913.317,1328.3 L910.313,1327.73 L909.329,1327.2 L908.615,1326.28 L908.425,1325.34 L908.42,1324.24 L908.338,1323.11 L907.914,1322.11 L908.547,1322.08 L910.71,1321.11 L909.51,1319.99 L909.261,1319.58 L909.273,1319.06 L909.469,1318.55 L909.57,1317.99 L909.281,1317.32 L909.96,1316.87 L910.666,1316.41 L911.957,1316.07 L915.567,1316.17 L917.557,1316.87 L919.444,1316.67 L920.008,1316.93 L921.347,1317.41 L922.094,1316.1 L922.463,1313.91 L923.092,1307.17 L923.506,1305.1 L923.936,1304.03 L924.26,1303.22 L927.559,1298.45 L928.461,1296.39 L928.886,1294.41 L929.032,1292.17 L928.856,1287.68 L928.738,1287.13 L928.568,1286.8 L928.44,1286.39 L928.442,1285.64 L928.555,1285.14 L929.141,1283.79 L929.439,1282.72 L929.594,1281.75 L929.594,1280.75 L929.555,1278.97 L929.933,1278.78 L930.378,1278.34 L930.725,1277.89 L931.087,1277.23 L931.243,1276.49 L930.989,1275.79 L931.99,1274.39 L932.508,1274.15 L933.377,1274.17 L934.18,1274.41 L935.442,1275.28 L936.139,1275.59 L935.288,1274.23 L933.825,1273.63 L925.313,1273 L923.959,1272.26 L923.483,1270.49 L923.663,1265.86 L923.871,1264.81 L924.672,1262.94 L924.874,1262.02 L925.207,1261.46 L925.937,1261.5 L927.425,1261.89 L927.646,1261.39 L928.008,1261.22 L928.303,1260.62 L928.267,1259.83 L927.955,1259.37 L926.889,1258.77 L926.425,1258.25 L926.379,1257.43 L926.934,1256.76 L928.167,1255.97 L929.577,1254.51 L930.322,1254.04 L931.757,1252.65 L934.733,1251.62 L940.467,1251.18 L940.856,1251.47 L941.232,1252.03 L942.105,1252.27 L947.619,1251.3 L958.805,1250.08 L961.749,1251.03 L961.737,1251.5 L961.446,1253.25 L962.425,1255.54 L963.904,1257.56 L965.129,1258.46 L965.118,1258.85 L964.782,1259.63 L965.351,1259.93 L966.169,1260.19 L966.571,1260.85 L966.427,1261.97 L966.06,1262.54 L965.237,1263.13 L964.554,1263.46 L963.082,1263.55 L962.415,1263.95 L963.039,1266.05 L963.361,1266.85 L963.854,1267.41 L964.335,1267.19 L964.765,1267.02 L965.391,1267.06 L965.822,1267.31 L966.512,1268.15 L966.775,1268.34 L967.073,1268.61 L967.362,1269.2 L967.74,1269.8 L968.272,1270.08 L968.899,1270.08 L969.358,1269.99 L969.765,1269.75 L970.236,1269.31 L971.055,1268.02 L971.939,1265.88 L972.189,1263.74 L971.139,1262.46 L969.664,1262.81 L968.884,1262.66 L968.979,1261.75 L969.323,1260.73 L969.642,1258.61 L969.979,1257.73 L970.488,1257.21 L971.124,1256.98 L972.679,1256.94 L973.383,1257.28 L973.917,1258.06 L974.352,1258.93 L974.792,1259.55 L976.279,1260.36 L979.446,1261.05 L980.873,1261.78 L980.863,1262.24 L979.956,1262.99 L979.115,1263.97 L978.483,1265.31 L978.216,1267.12 L978.185,1270.64 L978.371,1272.19 L978.853,1273.35 L978.995,1271.8 L978.681,1267.89 L978.75,1266.04 L979.349,1264.7 L980.218,1264.11 L981.011,1263.37 L981.386,1261.59 L981.207,1261.28 L980.302,1258.81 L980.025,1258.44 L979.655,1257.59 L979.347,1256.64 L979.109,1255.44 L978.612,1254.01 L978.516,1252.97 L978.62,1251.91 L978.866,1251 L980.327,1247.55 L981.387,1243.8 L982.053,1242.14 L982.599,1241.35 L983.335,1240.67 L984.163,1240.2 L985.003,1240.02 L985.773,1240.09 L986.208,1240.29 L987.098,1241.34 L988.391,1242.48 L989.779,1243.23 L992.663,1244 L994.251,1243.98 L996.952,1243.35 L998.432,1243.49 L999.136,1242.68 L1002.09,1241.99 L1005.01,1242.06 L1005.65,1241.61 L1006.15,1241.99 L1006.76,1242.11 L1008.01,1242.06 L1008.33,1242.28 L1009.14,1243.36 L1012.69,1249.78 L1014.16,1251.55 L1015.2,1252.42 L1015.71,1253.08 L1016.2,1254.98 L1018.73,1258.76 L1019.3,1259.29 L1019.99,1259.6 L1020.88,1259.71 L1021.34,1259.94 L1023.54,1261.48 L1026.05,1262.05 L1028.1,1262.51 L1028.59,1262.31 L1028.13,1261.49 L1026.73,1260.67 L1023.39,1260.27 L1022.03,1259.72 L1019.35,1257.96 L1018.14,1256.57 L1017.72,1254.57 L1017.62,1252.86 L1017.48,1251.77 L1017.11,1251.43 L1015.37,1250.45 L1014.82,1250.24 L1013.92,1249.4 L1013.07,1247.51 L1012.5,1245.49 L1012.45,1244.24 L1011.11,1243.69 L1008.79,1241.64 L1007.4,1240.78 L1006.03,1240.43 L1001.33,1240.27 L999.933,1239.6 L998.995,1239.33 L998.561,1239.6 L998.164,1240.23 L997.311,1239.93 L996.025,1238.92 L994.813,1237.37 L991.86,1231.13 L993.289,1230.49 L993.982,1230.45 L994.86,1230.71 L996.294,1231.7 L997.082,1231.98 L997.894,1231.61 L998.186,1230.93 L998.503,1229.69 L998.725,1228.35 L998.729,1227.38 L998.31,1226.74 L997.636,1226.12 L997.142,1225.47 L997.257,1224.76 L997.263,1224.37 L996.37,1224.08 L995.083,1224.3 L993.878,1224.86 L993.205,1225.59 L992.957,1225.59 L991.98,1223.82 L991.354,1222.41 L991.255,1222.1 L991.574,1220.85 L992.154,1219.78 L992.552,1218.8 L992.308,1217.85 L992.315,1217.42 L992.771,1217.08 L994.329,1216.66 L994.962,1216.61 L995.354,1216.3 L995.932,1215.53 L996.519,1214.54 L996.899,1213.61 L996.06,1213.76 L994.662,1214.94 L993.87,1215.27 L993.945,1215.06 L994.056,1214.62 L994.134,1214.42 L993.02,1214.54 L991.101,1215.09 L989.972,1215.21 L988.667,1214.96 L988.097,1215.18 L986.856,1216.33 L986.424,1216.47 L985.456,1216.09 L984.431,1215.12 L983.701,1213.73 L983.629,1212.1 L984.16,1211.47 L985.168,1211.03 L986.294,1210.82 L987.153,1210.88 L987.161,1210.46 L986.507,1210.29 L986.013,1209.8 L985.534,1209.64 L984.939,1210.41 L984.64,1209.72 L984.761,1209.02 L985.112,1208.37 L985.488,1207.86 L986.26,1208.22 L989.234,1207.46 L993.246,1207.53 L994.177,1207.3 L994.957,1206.82 L996.519,1205.48 L997.936,1204.81 L998.552,1204.35 L998.812,1203.34 L998.836,1201.82 L997.929,1200.37 L995.006,1197.12 L994.593,1196.44 L994.426,1195.78 L994.096,1195.48 L993.348,1195.49 L992.578,1195.7 L992.165,1195.98 L991.924,1195.5 L993.183,1194.67 L991.972,1191.25 L991.517,1190.37 L990.486,1189.7 L988.558,1187.94 L987.339,1187.31 L987.347,1186.88 L987.572,1185.75 L987.124,1184.47 L986.444,1183.2 L985.979,1182.11 L985.804,1180.46 L985.994,1179.02 L986.519,1177.21 L987.777,1177.63 L989.108,1177.72 L991.582,1176.83 L992.427,1176.72 L995.301,1177.26"
         id="path1039" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['DE'],polycolors['DE'],labels['DE'], x1, polybary['DE'], x2, polybary['DE'] )
    if 'DE' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1009.39,1940.77 L1009.43,1942.57 L1009.37,1942.73 L1009.23,1942.88 L1009.1,1943.09 L1009.08,1943.38 L1009.19,1943.55 L1009.4,1943.68 L1009.62,1943.76 L1009.71,1943.77 L1009.83,1944.33 L1009.94,1944.68 L1009.92,1945.01 L1009.69,1945.51 L1010.06,1945.81 L1010.22,1946.23 L1010.34,1946.71 L1010.62,1947.22 L1010.99,1947.51 L1012.24,1948.08 L1012.12,1947.18 L1011.92,1946.76 L1011.23,1945.14 L1012.56,1947.12 L1012.99,1948.2 L1013.18,1949.4 L1013.05,1953.01 L1013.2,1954.89 L1013.77,1956.21 L1013.49,1957.02 L1013.27,1957.88 L1013.14,1958.85 L1013.11,1960.02 L1013.33,1961.3 L1014.17,1963.73 L1014.36,1964.96 L1013.97,1972.7 L1014.31,1974.17 L1014.18,1974.69 L1014.08,1975.77 L1013.98,1976.26 L1013.64,1976.82 L1012.37,1977.99 L1011.86,1978.79 L1011.14,1980.3 L1007.9,1984.9 L1007.19,1985.6 L1006.92,1986.17 L1006.89,1988.38 L1006.46,1992.7 L1006.84,1996.95 L1006.74,2001.51 L1006.59,2001.82 L1006.24,2001.97 L1005.77,2002.36 L1005.35,2002.9 L1005.1,2003.5 L1005.74,2003.51 L1005.6,2004.61 L1005.37,2005.31 L1004.92,2005.63 L1004.43,2005.58 L1004.05,2005.6 L1003.91,2005.95 L1004.11,2006.89 L1003.48,2006.59 L1002.78,2006.41 L1002.1,2006.36 L1001.55,2006.45 L1001.63,2006.99 L1001.18,2007.61 L1000.9,2008.18 L1001.53,2008.61 L1003.09,2007.98 L1004.03,2007.83 L1004.45,2008.4 L1004.28,2009.28 L1003.82,2010.02 L1003.14,2010.52 L1002.31,2010.7 L1001.86,2011 L1001.51,2011.64 L1001.02,2012.24 L1000.2,2012.38 L1000.39,2012.82 L1000.44,2013.26 L1000.38,2013.68 L1000.19,2014.12 L1000.73,2014.34 L1000.86,2014.82 L1000.65,2015.37 L1000.17,2015.78 L1000.46,2016.25 L999.721,2016.56 L998.431,2017.39 L997.612,2017.49 L997.607,2017.95 L998.045,2018.32 L998.194,2018.8 L998.032,2019.28 L997.591,2019.65 L997.587,2020.04 L998.123,2019.82 L998.883,2019.12 L999.47,2018.78 L999.285,2019.71 L998.906,2020.76 L998.49,2021.69 L998.154,2022.21 L997.374,2022.61 L996.768,2022.4 L995.61,2021.33 L994.868,2021 L993.403,2020.76 L992.19,2020.21 L992.719,2018.62 L992.951,2018.42 L993.387,2018.29 L992.63,2017.61 L991.491,2017.11 L990.822,2016.56 L991.457,2015.69 L991.462,2015.3 L990.911,2015.47 L989.509,2016.55 L989.195,2016.12 L989.301,2015.99 L989.359,2015.94 L989.417,2015.87 L989.519,2015.66 L989.015,2015.8 L988.826,2015.9 L988.551,2016.11 L988.234,2015.65 L987.744,2015.21 L986.551,2014.82 L985.395,2014.63 L984.672,2014.78 L984.162,2014.42 L983.788,2014 L983.328,2013.64 L982.235,2013.38 L982.119,2013.15 L982.031,2012.87 L981.777,2012.62 L981.334,2012.49 L980.958,2012.48 L980.587,2012.43 L980.177,2012.13 L980.827,2011.71 L980.833,2011.33 L980.238,2011.11 L979.9,2010.89 L979.454,2010.93 L978.798,2010.75 L978.204,2010.47 L977.951,2010.2 L978.187,2009 L978.044,2008.58 L977.335,2008.3 L977.692,2007.49 L978.021,2006.23 L978.493,2005.37 L979.299,2005.78 L981.833,2004.67 L982.24,2004.3 L982.47,2003.45 L983.004,2003.02 L983.652,2002.66 L984.227,2002 L983.122,2001.93 L980.186,2001.05 L979.055,2001.01 L977.953,2000.79 L977.366,2000.16 L977.797,1998.92 L976.632,1998.89 L974.694,1999.6 L973.589,1999.66 L973.711,1999.44 L973.949,1998.86 L973.28,1998.89 L972.778,1998.74 L972.312,1998.52 L971.71,1998.36 L972.331,1998.06 L973.652,1997.87 L974.261,1997.55 L974.662,1997.03 L974.883,1996.39 L974.808,1995.81 L974.294,1995.47 L974.301,1995 L975.37,1995.02 L976.329,1994.71 L977.873,1993.82 L977.653,1993.23 L977.297,1992.82 L976.835,1992.58 L976.295,1992.48 L977.843,1992.01 L978.226,1991.67 L978.113,1991.53 L977.956,1991.11 L977.77,1990.76 L977.598,1990.81 L978.172,1989.94 L978.252,1989.93 L978.39,1989.62 L978.809,1989.1 L978.91,1988.7 L978.8,1988.67 L978.276,1988.27 L978.014,1987.7 L977.806,1987.33 L977.513,1987.1 L977.016,1986.97 L976.077,1987.05 L974.467,1987.89 L973.453,1988.19 L971.527,1988.28 L970.693,1988.46 L969.953,1988.93 L969.639,1988.12 L970.19,1987.09 L970.006,1986.24 L968.716,1984.67 L971.022,1984.48 L971.942,1984.12 L972.603,1983.03 L972.017,1982.62 L972.424,1981.9 L973.36,1981.23 L974.38,1980.94 L974.626,1980.77 L975.888,1980.13 L976.139,1980.12 L976.223,1979.35 L975.956,1978.92 L975.235,1978.36 L974.457,1976.58 L974.139,1976.03 L973.655,1975.91 L972.346,1976.24 L972.081,1975.99 L971.748,1975.2 L970.91,1974.68 L969.847,1974.43 L968.894,1974.43 L969.128,1973.85 L969.257,1973.62 L968.914,1973.31 L968.537,1973.13 L968.117,1973.08 L967.672,1973.13 L968.144,1973.04 L968.531,1972.85 L969.28,1972.35 L969.288,1971.88 L967.983,1971.86 L967.99,1971.43 L968.548,1971.22 L968.752,1970.63 L968.577,1970.01 L968.02,1969.74 L968.419,1969.04 L968.039,1968.67 L967.421,1968.45 L967.092,1968.21 L967.299,1967.63 L967.787,1967.51 L969.013,1967.59 L969.365,1967.41 L969.722,1967.15 L970.17,1966.91 L972.051,1966.58 L972.689,1966.32 L973.176,1965.97 L973.447,1965.84 L973.65,1965.9 L973.769,1965.83 L973.822,1965.34 L973.146,1964.79 L973.056,1964.69 L971.337,1963.81 L971.227,1963.57 L971.109,1963.15 L970.905,1962.91 L970.241,1963.46 L969.972,1963.43 L969.795,1963.16 L969.731,1962.71 L969.962,1962.49 L971.372,1961.68 L970.365,1960.82 L969.377,1960.89 L967.528,1961.61 L967.885,1961.15 L967.611,1960.62 L967.445,1959.94 L967.431,1959.23 L967.582,1958.64 L967.981,1958.96 L968.431,1958.96 L968.859,1958.71 L969.176,1958.28 L969.745,1958.62 L969.924,1958.36 L969.986,1957.83 L970.167,1957.41 L970.395,1957.13 L970.49,1956.85 L970.709,1956.65 L972.209,1956.45 L972.617,1956.04 L972.725,1955.43 L972.726,1954.67 L973.249,1953.9 L973.461,1953.4 L973.246,1953.17 L972.673,1953.1 L972.555,1952.92 L972.665,1952.65 L973.069,1951.08 L973.68,1950.38 L974.482,1950.03 L975.365,1949.81 L975.106,1949.22 L975.384,1947.86 L975.062,1947.25 L975.07,1946.79 L975.429,1946.98 L975.557,1946.99 L975.607,1947.11 L975.72,1947.65 L976.491,1947.24 L976.982,1947.47 L977.39,1947.91 L977.925,1948.15 L978.656,1947.92 L979.216,1947.32 L979.628,1946.49 L979.89,1945.59 L979.553,1945.16 L981.295,1945.09 L984.254,1943.61 L989.441,1942.58 L990.914,1941.85 L991.067,1940.65 L993.6,1937.92 L994.1,1937.54 L995.192,1937.37 L996.29,1937.37 L996.789,1937.52 L997.206,1937.49 L999.591,1938.19 L1000.56,1938.74 L1000.87,1939.06 L1001.15,1939.47 L1001.39,1939.98 L1001.81,1941.19 L1002.12,1941.19 L1003.8,1938.39 L1004.49,1936.59 L1004.06,1935.22 L1004.44,1934.03 L1004.19,1932.94 L1003.57,1931.92 L1002.83,1930.95 L1004.11,1929.23 L1004.11,1928.69 L1003.81,1928.09 L1003.81,1927.57 L1004,1926.96 L1004.31,1926.54 L1005.39,1925.41 L1005.24,1925.1 L1005.19,1924.92 L1005.25,1924.66 L1005.4,1924.13 L1004.94,1923.63 L1004.64,1922.63 L1004.58,1921.54 L1004.81,1920.73 L1005.57,1920.25 L1007.46,1920.26 L1008.3,1919.9 L1008.9,1920.76 L1009.3,1921.12 L1010.2,1921.62 L1010.2,1922.09 L1009.8,1922.58 L1009.72,1923.25 L1009.94,1923.97 L1010.46,1924.6 L1010.24,1925.77 L1011.07,1931.45 L1010.99,1933.32 L1009.45,1939.13 L1009.49,1939.89 L1009.39,1940.77"
         id="path1041" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M741.236,1402.43 L741.349,1405.22 L741.616,1406.46 L742.862,1408.17 L743.384,1409.16 L743.385,1410.42 L743.116,1410.77 L742.119,1411.58 L741.834,1412.13 L742.334,1414.94 L742.248,1415.34 L741.873,1416 L741.804,1416.28 L741.908,1416.51 L742.36,1416.8 L742.519,1417.02 L742.777,1417.85 L742.949,1418.22 L743.225,1418.38 L743.791,1418.56 L744.112,1418.61 L744.855,1418.58 L745.226,1418.66 L745.716,1418.98 L745.88,1419.36 L745.96,1419.8 L746.174,1420.29 L747.503,1421.84 L747.686,1422.28 L748.092,1423.59 L748.263,1423.99 L748.933,1424.68 L751.951,1426.44 L752.123,1426.54 L752.727,1426.73 L753.289,1426.38 L754.675,1424.43 L755.468,1423.73 L756.433,1423.24 L758.562,1422.79 L761.792,1422.11 L762.786,1422.52 L763.396,1423.21 L763.942,1424.46 L763.943,1424.46 L764.399,1425.49 L764.775,1426.72 L765.055,1427.12 L765.604,1427.25 L766.119,1427.47 L766.38,1428.08 L766.693,1429.61 L766.046,1430.59 L766.199,1432.43 L766.926,1436.03 L766.806,1438.03 L766.946,1438.75 L767.497,1439.61 L767.98,1440.07 L769.634,1441.14 L770.31,1441.4 L771.285,1441.37 L773.083,1440.89 L773.657,1440.52 L774.086,1440.16 L774.516,1439.93 L775.087,1439.94 L775.642,1440.16 L775.872,1440.51 L775.766,1441.01 L775.319,1441.67 L775.779,1442.08 L778.344,1442.32 L779.227,1442.59 L780.06,1443.1 L780.783,1443.93 L781.327,1445.16 L781.457,1447.6 L781.211,1450.51 L781.342,1453.09 L782.632,1454.53 L782.823,1454.14 L782.883,1454.11 L782.944,1454.17 L783.137,1454.06 L783.26,1453.88 L783.477,1453.38 L784.298,1452.22 L784.65,1451.88 L785.207,1451.72 L786.957,1451.74 L788.596,1452.08 L789.386,1452.57 L790.03,1453.11 L790.715,1453.51 L791.618,1453.58 L794.241,1452.72 L795.093,1452.7 L795.838,1453.13 L797.043,1454.71 L797.854,1455.27 L798.62,1456.07 L799.526,1458.51 L800.234,1458.92 L800.55,1458.65 L800.535,1458.22 L800.587,1457.82 L801.103,1457.65 L801.452,1457.8 L801.963,1458.18 L802.424,1458.66 L802.636,1459.09 L802.49,1459.73 L802.062,1460.15 L801.062,1460.81 L800.588,1461.39 L800.366,1461.88 L800.061,1463 L799.093,1465.44 L798.908,1466.56 L799.453,1466.89 L800.281,1466.59 L800.717,1466.56 L801.099,1466.79 L801.563,1469.54 L801.757,1470.15 L801.996,1470.66 L801.952,1471.07 L800.952,1471.48 L799.971,1472.07 L799.182,1472.69 L798.863,1473.03 L798.623,1473.48 L798.399,1474.2 L798.583,1474.3 L798.728,1475.27 L798.706,1476.19 L798.416,1476.14 L798.388,1476.13 L800.544,1477.53 L801.073,1477.73 L801.676,1477.69 L804.183,1477.52 L810.094,1479.65 L810.387,1479.75 L811.597,1479.63 L817.411,1477.27 L819.105,1477.13 L819.562,1476.87 L819.983,1476.18 L820.337,1475.21 L820.787,1473.24 L820.777,1472.81 L820.65,1472.49 L820.581,1472.11 L820.733,1471.5 L821.001,1471.03 L825.586,1467 L826.702,1466.58 L826.819,1467.52 L827.258,1467.74 L828.531,1467.57 L828.866,1468.08 L828.737,1469.17 L828.432,1470.45 L828.265,1471.53 L827.866,1471.03 L827.617,1471.07 L826.746,1472.68 L826.661,1473.24 L826.687,1473.77 L826.603,1474.43 L824.788,1478.18 L824.543,1479.16 L825.019,1480.07 L826.87,1480.99 L827.418,1482.18 L827.433,1483 L827.292,1483.53 L827.051,1484.04 L826.782,1484.79 L826.626,1485.11 L826.238,1485.5 L826.137,1485.88 L826.198,1486.2 L826.549,1486.88 L826.623,1487.26 L826.453,1489.59 L826.784,1490.09 L827.829,1490.32 L828.745,1490.23 L830.54,1489.74 L831.284,1489.75 L831.391,1489.75 L832.094,1490.21 L833.497,1491.8 L834.273,1492.31 L835.686,1492.82 L836.137,1493.32 L837.695,1495.93 L838.259,1496.55 L838.96,1497.15 L839.875,1497.48 L841.737,1497.09 L842.613,1497.13 L843.405,1498.02 L844.127,1499.03 L844.348,1499.99 L844.15,1500.2 L843.81,1500.39 L843.628,1501.27 L843.688,1501.85 L843.87,1502.1 L844.536,1502.28 L844.831,1502.21 L845.065,1501.91 L845.258,1501.57 L845.42,1501.42 L845.668,1501.47 L846.112,1501.78 L847.341,1502.32 L847.635,1502.62 L848.053,1503.05 L849.791,1506.71 L849.924,1507.29 L849.876,1507.94 L849.71,1508.52 L849.634,1509.08 L849.852,1509.65 L850.716,1509.93 L851.808,1509.38 L852.877,1508.64 L853.662,1508.34 L854.9,1509.01 L855.243,1509.13 L855.848,1509.08 L855.998,1508.84 L856.116,1508.4 L856.638,1507.77 L857.681,1507.31 L858.584,1507.57 L859.432,1508.06 L860.301,1508.26 L860.818,1508.07 L861.382,1507.41 L861.832,1507.2 L862.265,1507.29 L863.124,1507.89 L863.592,1508.03 L864.774,1509.81 L865.447,1510.51 L866.442,1511.05 L868.989,1511.81 L869.713,1512.62 L870.175,1514.44 L870.895,1514.25 L874.378,1514.23 L874.988,1514.03 L875.415,1513.46 L875.917,1513.29 L876.027,1512.96 L876.043,1512.56 L876.278,1512.19 L876.574,1512.09 L877.728,1512.05 L877.492,1511.46 L879.843,1511.32 L880.993,1511.5 L882.085,1512.01 L882.154,1512.06 L884.021,1513.61 L885.073,1514.26 L885.882,1514.39 L886.229,1514.44 L886.666,1514.38 L887.467,1513.89 L887.805,1513.77 L888.252,1513.87 L891.901,1515.92 L892.568,1516.63 L892.797,1517.16 L892.882,1517.68 L892.875,1518.03 L892.793,1518.05 L893.056,1518.48 L894.941,1520.68 L895.063,1521.06 L894.862,1521.56 L894.135,1521.56 L893.885,1522.12 L894.109,1522.86 L894.755,1523.72 L897.354,1526.47 L897.405,1526.74 L897.293,1527.37 L897.328,1527.62 L897.54,1527.8 L898.097,1528.03 L898.263,1528.22 L898.532,1529.39 L898.652,1529.66 L898.692,1529.8 L898.751,1530.29 L898.851,1530.52 L899.034,1530.51 L899.6,1530.17 L899.765,1530.18 L899.979,1531.18 L899.737,1531.96 L899.589,1532.72 L900.076,1533.62 L900.55,1533.85 L902.375,1533.95 L903.947,1534.61 L904.473,1534.51 L905.22,1533.6 L905.107,1532.56 L904.841,1531.53 L905.114,1530.68 L905.529,1530.59 L907.222,1531.06 L908.421,1530.99 L908.871,1531.08 L911.369,1532.63 L912.252,1532.71 L912.143,1534.92 L912.549,1536.65 L913.424,1537.4 L914.715,1536.65 L914.791,1536.34 L914.672,1536.01 L914.652,1535.66 L915.039,1535.31 L915.262,1535.34 L916.787,1536.55 L917.418,1536.91 L918.055,1537.15 L921.59,1537.32 L922.674,1537.88 L922.931,1537.93 L923.182,1537.69 L923.647,1536.82 L923.916,1536.52 L924.286,1536.4 L925.129,1536.33 L925.443,1536.19 L925.663,1535.86 L925.992,1534.96 L926.194,1534.66 L926.89,1534.29 L927.057,1534.25 L927.862,1534.04 L928.656,1534.13 L928.81,1534.84 L930.577,1535.02 L931.309,1535.31 L931.55,1536.23 L931.238,1536.41 L931.132,1536.55 L931.053,1536.76 L931.609,1537.38 L932.488,1538.92 L933.004,1539.62 L933.881,1540.22 L935.667,1540.59 L936.564,1541.09 L936.854,1541.48 L937.227,1542.3 L937.599,1542.64 L938.006,1542.74 L938.165,1542.79 L938.661,1542.68 L939.148,1542.48 L939.67,1542.39 L940.122,1542.48 L940.983,1542.82 L941.376,1542.89 L941.882,1542.78 L942.862,1542.28 L943.32,1542.17 L946.711,1543.26 L949.059,1543.09 L949.801,1543.19 L956.23,1546.85 L960.256,1547.79 L960.722,1548.25 L960.704,1548.27 L959.834,1549.22 L958.488,1551.62 L956.488,1556.73 L955.954,1557.57 L955.696,1557.78 L954.159,1558.56 L953.796,1558.57 L953.702,1558.72 L953.505,1558.9 L953.304,1559.22 L953.199,1559.81 L953.101,1560.15 L952.86,1560.35 L952.541,1560.45 L951.378,1560.44 L950.904,1560.58 L950.671,1561.13 L950.588,1562.32 L950.39,1562.81 L949.93,1563.4 L949.367,1563.91 L948.867,1564.22 L947.695,1565.54 L947.046,1566.07 L946.355,1566.28 L945.862,1566.61 L944.036,1569.28 L943.625,1571.31 L943.557,1573.49 L944.008,1574.83 L943.995,1575.26 L942.932,1576.05 L942.419,1576.54 L942.188,1577.1 L942.067,1578.29 L941.838,1578.89 L941.555,1579.29 L941.25,1579.87 L941.018,1580.57 L940.663,1582.22 L940.591,1583.03 L940.702,1583.79 L940.949,1584.6 L941.136,1585.44 L941.058,1586.31 L940.457,1587.21 L939.678,1587.6 L938.989,1588.13 L938.66,1589.43 L938.535,1590.7 L938.239,1591.75 L937.771,1592.64 L935.013,1596.26 L934.712,1597.01 L934.602,1597.91 L934.374,1598.4 L934.077,1598.76 L933.756,1599.31 L933.445,1600.26 L933.142,1601.47 L933.189,1604.05 L934.9,1607.3 L934.93,1609.25 L934.507,1609.99 L933.349,1611.14 L933.097,1611.96 L933.117,1612.98 L933.03,1613.26 L932.744,1613.88 L932.565,1614.18 L932.278,1614.44 L932.144,1614.71 L932.087,1614.97 L932.104,1615.12 L932.134,1615.28 L932.116,1615.56 L932.296,1616.73 L932.182,1617.43 L931.603,1617.72 L931.358,1618.01 L930.692,1620.02 L930.55,1620.88 L931.002,1623.08 L930.947,1624.1 L930.685,1624.51 L929.963,1625.23 L929.803,1625.6 L929.877,1626.94 L929.99,1627.62 L930.147,1627.91 L930.76,1628.26 L931.306,1629.08 L932.198,1631.01 L932.441,1631.33 L932.705,1631.58 L932.917,1631.91 L932.99,1632.51 L932.882,1632.73 L932.69,1633.36 L932.666,1633.37 L931.161,1633.88 L930.128,1634.41 L929.89,1634.6 L928.232,1635.86 L928.332,1635.91 L929.018,1636.47 L929.163,1636.47 L929.156,1637.1 L928.96,1637.45 L928.615,1637.55 L928.18,1637.44 L927.931,1637.56 L927.867,1637.76 L927.951,1638.01 L928.239,1638.45 L928.272,1638.58 L928.278,1638.61 L928.268,1638.77 L928.205,1638.94 L928.127,1639.05 L927.46,1639.61 L926.915,1639.51 L926.561,1639.27 L926.375,1639.14 L925.725,1638.89 L925.225,1639.02 L925.217,1639.4 L925.418,1639.79 L925.553,1639.95 L925.75,1640.25 L925.808,1640.6 L925.652,1641.18 L925.408,1641.48 L924.778,1642.25 L923.998,1642.53 L923.582,1642.68 L921.819,1642.54 L920.64,1642.45 L919.509,1642.64 L917.879,1643.28 L917.592,1643.33 L917.268,1643.18 L917.106,1642.94 L916.995,1642.72 L916.825,1642.57 L915.594,1642.13 L914.694,1641.54 L914.491,1640.52 L915.328,1638.77 L914.171,1638.85 L913.688,1638.75 L913.61,1638.73 L913.07,1638.39 L912.08,1638.15 L909.955,1638.43 L908.862,1638.23 L908.488,1637.92 L908.107,1637.8 L907.722,1637.89 L907.334,1638.17 L906.536,1638.37 L906.614,1639.07 L907.018,1639.95 L907.194,1640.71 L906.854,1641.22 L906.201,1641.73 L905.497,1642.1 L904.374,1642.3 L904.26,1643.48 L903.142,1644.07 L902.482,1644.87 L901.915,1645.82 L901.672,1646.61 L906.744,1646.32 L907.54,1645.98 L908.152,1646.52 L908.766,1647.12 L909.185,1647.77 L908.829,1648.45 L908.42,1648.69 L907.98,1648.79 L907.526,1649.03 L906.874,1649.84 L906.642,1649.92 L906.253,1649.95 L905.421,1650.74 L905.097,1652.01 L905.207,1653.57 L902.231,1655.57 L900.941,1656.79 L900.076,1658.09 L899.978,1658.17 L897.174,1660.55 L895.877,1660.94 L895.884,1662.01 L895.258,1662.3 L895.094,1662.4 L894.919,1662.79 L893.83,1663.13 L893.411,1663.5 L892.795,1664.47 L893.257,1665.65 L892.22,1667.01 L889.272,1669.05 L884.605,1670.28 L882.475,1671.39 L881.737,1673.54 L881.887,1674.12 L882.408,1675.23 L882.514,1675.93 L882.403,1676.79 L882.228,1677.17 L882.136,1677.37 L881.814,1677.9 L881.046,1679.89 L880.969,1680.19 L881.073,1680.55 L881.285,1680.87 L881.588,1681.24 L881.557,1682.29 L881.368,1682.8 L880.856,1683.38 L880.388,1683.7 L878.927,1684.38 L877.734,1685.6 L877.291,1685.94 L874.13,1687.47 L868.082,1692.47 L867.465,1693.19 L867.557,1694.02 L868.541,1695.24 L866.921,1697.08 L865.288,1699.57 L864.77,1700.08 L864.581,1700.45 L864.565,1700.77 L864.643,1701.07 L864.733,1701.33 L864.754,1701.52 L864.73,1702 L864.798,1702.43 L864.696,1702.83 L864.177,1703.28 L864.377,1703.42 L866.436,1704.83 L867.036,1705.55 L867.516,1706.52 L867.539,1707.21 L866.683,1708.89 L865.982,1710.27 L865.814,1710.75 L865.416,1712.5 L865.36,1713.17 L865.371,1713.75 L865.162,1714.15 L864.192,1714.4 L863.94,1714.42 L863.682,1714.37 L863.347,1714.24 L863.267,1714.23 L863.195,1714.23 L863.122,1714.25 L860.457,1715.38 L859.364,1716 L859.164,1716.74 L859.572,1717.62 L860.289,1718.62 L860.127,1719.15 L859.762,1719.76 L859.112,1721.08 L860.191,1720.51 L862.185,1720.15 L864.169,1720.18 L865.612,1720.91 L867.058,1720.27 L869.403,1717.79 L872.268,1716.1 L873.446,1714.97 L873.291,1713.52 L872.993,1713.38 L872.251,1713.67 L871.81,1713.41 L871.639,1713.14 L871.273,1712.09 L871.263,1711.84 L870.916,1710.75 L870.768,1710.12 L871.042,1709.28 L872.015,1708.09 L873.328,1706.53 L874.79,1705.39 L876.142,1705.02 L877.566,1705.02 L878.986,1704.74 L882.787,1702.39 L885.579,1701.97 L888.455,1702.2 L894.821,1704.15 L895.458,1704.5 L895.46,1704.5 L895.836,1705.13 L895.858,1705.71 L895.881,1706.32 L895.529,1707.35 L894.333,1708.64 L894.07,1709.35 L894.838,1710.81 L896.306,1712.5 L897.222,1714.24 L895.561,1717.15 L894.714,1719.36 L894.226,1721.48 L894.559,1722.54 L897.987,1723.45 L898.626,1724.12 L898.562,1724.59 L898.237,1725.06 L897.869,1725.47 L897.658,1725.78 L897.782,1725.73 L897.833,1726.35 L897.82,1727 L897.755,1727.06 L897.753,1727.54 L897.653,1727.8 L897.672,1728 L898.042,1728.3 L898.454,1728.38 L898.797,1728.14 L899.12,1727.83 L899.491,1727.7 L900.463,1728.18 L903.281,1731.62 L903.487,1731.77 L903.619,1732.44 L904.069,1733.76 L904.332,1734.91 L904.542,1735.55 L904.825,1736.06 L904.186,1736.33 L903.704,1736.89 L903.323,1737.63 L903.015,1738.46 L902.38,1739.43 L901.471,1740.14 L900.462,1740.6 L899.521,1740.84 L898.442,1740.78 L898.138,1740.82 L897.976,1740.78 L897.633,1740.54 L897.467,1740.51 L897.257,1740.67 L897.055,1740.95 L896.918,1741.21 L896.914,1741.32 L895.605,1741.45 L894.915,1741.83 L894.322,1742.72 L894.044,1743.74 L893.967,1744.85 L894.038,1745.97 L894.425,1748.57 L895.253,1749.94 L896.399,1750.87 L897.536,1751.15 L898.111,1751.39 L899.069,1752.42 L899.636,1752.89 L901.045,1753.31 L901.57,1753.71 L901.801,1754.65 L901.683,1755 L901.137,1756.07 L900.998,1756.59 L901.065,1757.09 L901.517,1757.85 L901.672,1758.32 L901.812,1760.91 L902.109,1761.83 L904.075,1763.03 L905.934,1764.55 L906.074,1764.79 L906.341,1765.25 L906.617,1766.03 L906.945,1766.75 L907.516,1767.22 L909.37,1768.11 L909.702,1768.38 L909.629,1769.16 L909.202,1770.1 L908.231,1771.69 L906.742,1773.33 L906.5,1773.69 L906.656,1775.51 L907.084,1776.78 L907.048,1777.76 L905.81,1778.72 L904.912,1780.21 L904.604,1780.51 L904.374,1780.41 L903.847,1779.72 L903.501,1779.55 L901.927,1779.92 L900.318,1780.58 L899.726,1781.1 L898.905,1782.41 L898.233,1782.97 L897.827,1783.05 L896.793,1783.02 L896.439,1783.16 L896.185,1783.67 L896.377,1784.02 L896.622,1784.32 L896.52,1784.7 L895.051,1785.33 L893.481,1784.71 L891.894,1783.76 L890.354,1783.43 L890.013,1783.59 L889.192,1784.17 L888.798,1784.37 L888.448,1784.41 L887.649,1784.35 L887.253,1784.4 L885.68,1785.21 L885.528,1785.33 L884.851,1785.85 L884.294,1786.53 L885.021,1787.75 L885.715,1788.4 L885.765,1788.84 L885.693,1789.34 L885.788,1789.91 L886.28,1790.85 L886.731,1791.27 L888.278,1791.49 L889.407,1792.38 L889.617,1794.18 L889.384,1797.67 L890.106,1799.01 L891.51,1800.33 L894.504,1802.24 L895.337,1802.49 L897.547,1802.14 L898.317,1802.23 L900.512,1803.24 L901.278,1804.15 L901.451,1804.47 L901.533,1804.84 L901.512,1805.03 L901.464,1805.19 L901.458,1805.49 L901.373,1805.53 L901.115,1806.51 L901.079,1806.59 L901.534,1807.96 L901.715,1809.51 L901.866,1810.04 L903.124,1812.65 L903.352,1813.48 L902.554,1813.46 L900.962,1812.92 L900.141,1812.89 L899.096,1813.41 L898.235,1814.39 L897.868,1815.64 L898.32,1816.98 L897.678,1817.42 L896.041,1820.01 L893.693,1821.66 L893.183,1822.48 L893.244,1824.42 L894.178,1826.21 L895.483,1827.7 L896.616,1828.72 L895.078,1829.49 L894.742,1830.01 L894.289,1831.94 L894.163,1832.31 L894.142,1832.62 L894.286,1833.2 L894.46,1833.5 L895.403,1834.69 L895.731,1835.32 L896.211,1836.61 L896.443,1837.09 L896.831,1837.53 L897.698,1838.2 L898.093,1838.7 L898.352,1839.33 L898.479,1839.91 L898.662,1840.47 L899.067,1840.96 L899.657,1841.23 L900.195,1841.17 L900.741,1841.01 L901.348,1840.98 L901.909,1841.17 L904.502,1842.68 L906.283,1843.4 L910.968,1846.52 L911.803,1846.91 L912.969,1847.15 L913.553,1847.4 L913.85,1847.86 L914.074,1848.39 L914.484,1848.82 L914.914,1848.93 L916.725,1849.04 L924.636,1847.06 L925.939,1847 L927.828,1845.85 L929.213,1846.14 L928.533,1848.15 L929.048,1849.33 L930.016,1850.35 L930.513,1851.5 L930.681,1851.89 L930.561,1852.97 L930.037,1853.62 L929.369,1854.1 L928.858,1854.64 L928.546,1855.39 L928.131,1856.78 L927.76,1857.48 L926.717,1858.57 L924.321,1860.46 L923.364,1861.88 L921.337,1863.97 L920.59,1865.23 L920.76,1866.82 L921.552,1869.84 L920.941,1870.24 L920.749,1870.24 L920.562,1870.2 L920.363,1870.28 L920.327,1870.51 L920.372,1871.3 L920.324,1871.52 L919.957,1871.69 L919.695,1871.65 L919.308,1871.95 L918.914,1872.39 L918.514,1872.78 L918.03,1872.01 L917.172,1871.48 L916.279,1871.8 L915.579,1872.53 L915.31,1873.25 L915.218,1873.96 L914.633,1874.37 L914.122,1874.85 L913.91,1875.43 L913.543,1876.16 L913.017,1876.22 L912.2,1876.03 L911.274,1875.65 L910.227,1875.72 L909.504,1876.11 L908.883,1876.75 L909.01,1877.43 L908.159,1878.32 L907.695,1878.57 L907.399,1877.78 L906.541,1877.55 L905.939,1877.78 L905.347,1877.83 L904.989,1878.4 L904.632,1878.65 L904.457,1879.1 L904.08,1880.84 L904.289,1882.53 L904.299,1882.86 L904.441,1883.56 L904.544,1884.09 L903.92,1884.45 L903.385,1883.39 L903.002,1883.14 L902.279,1883.09 L901.47,1883.55 L900.856,1884.2 L900.245,1884.66 L899.873,1885.01 L899.601,1884.28 L898.863,1884.3 L897.927,1884.15 L897.093,1884.24 L896.462,1884.64 L895.663,1885.41 L895.431,1886.32 L895.815,1886.62 L895.788,1886.65 L896.001,1886.83 L896.012,1887.11 L894.616,1888.76 L894.344,1890.23 L893.308,1890.88 L893.091,1891.52 L892.04,1891.57 L891.991,1891.19 L891.857,1890.98 L891.702,1890.73 L891.468,1890.91 L891.364,1891.37 L891.516,1892.18 L890.894,1892.36 L890.361,1892.14 L889.633,1892.09 L887.982,1892.49 L887.547,1891.85 L886.833,1891.56 L886.338,1891.84 L885.612,1892.43 L885.622,1894.04 L884.897,1894.96 L884.858,1895.97 L883.76,1896.06 L882.871,1896.15 L882.447,1896.95 L882.203,1897.62 L881.759,1898.06 L880.457,1899.19 L879.174,1899.87 L878.844,1900.8 L879.782,1901.13 L882.588,1900.51 L883.541,1900.75 L883.734,1901.23 L883.111,1902.29 L882.31,1902.79 L882.127,1903.92 L882.408,1904.41 L882.735,1905.17 L881.818,1905.83 L880.88,1906.32 L880.36,1906.42 L879.46,1906.78 L878.987,1906.66 L878.817,1906.35 L878.737,1905.86 L878.179,1906.04 L877.375,1906 L876.279,1906.09 L876.094,1906.68 L875.616,1907.41 L874.627,1907.96 L873.426,1907.59 L872.142,1907.73 L870.798,1907.69 L869.851,1907.99 L868.76,1908.61 L868.14,1909.67 L868.277,1910.49 L868.487,1911.42 L866.969,1911.35 L866.381,1910.75 L866.247,1910.48 L865.369,1910.11 L864.212,1909.48 L862.985,1909.28 L861.425,1909.47 L860.187,1909.89 L859.503,1910.23 L858.751,1910.83 L858.242,1912.09 L858.152,1913.57 L858.987,1913.98 L859.643,1913.81 L859.369,1914.39 L858.187,1914.47 L855.574,1914.49 L855.164,1914.24 L855.848,1913.91 L856.606,1913.68 L857.216,1913.24 L857.136,1912.07 L856.284,1911.22 L855.467,1911.26 L853.278,1911.44 L852.475,1911.4 L852.058,1910.08 L850.933,1909.81 L849.416,1909.74 L848.327,1908.97 L848.457,1908.61 L848.566,1908.01 L848.704,1907.71 L847.952,1907.65 L847.212,1907.8 L845.842,1908.41 L846.598,1908.76 L847.186,1910.18 L847.931,1910.57 L849.013,1911.1 L848.476,1911.64 L847.423,1911.47 L846.08,1911.43 L845.49,1911.61 L844.554,1913.05 L842.856,1912.42 L841.592,1911.32 L841.239,1910.73 L842.234,1910.17 L842.964,1909.58 L842.685,1909.09 L842.202,1908.74 L841.425,1908.53 L841.336,1907.97 L841.145,1907.5 L840.436,1907.38 L839.541,1907.33 L837.542,1906.29 L837.221,1904.9 L836.628,1904.18 L835.792,1903.77 L834.836,1903.62 L834.283,1904.24 L833.748,1905.22 L832.668,1904.69 L831.96,1904.58 L831.374,1903.76 L830.88,1902.97 L830.497,1902.1 L829.926,1901.74 L829.423,1902.02 L828.951,1902.12 L826.933,1901.82 L826.012,1901.82 L824.642,1901.5 L823.996,1901.4 L822.838,1901.61 L821.799,1901.04 L822.029,1899.95 L822.211,1899.33 L822.571,1898.71 L823.01,1897.71 L822.506,1897.06 L822.486,1896.33 L822.317,1895.95 L821.859,1893.65 L819.897,1892.21 L817.245,1893.75 L813.826,1893.45 L812.076,1893.56 L809.33,1893.24 L807.955,1892.69 L807.126,1891.23 L806.837,1889.73 L807.882,1888.07 L809.343,1887.37 L813.399,1887.63 L814.682,1887.33 L815.592,1886.69 L816.602,1885.61 L817.267,1884.25 L817.196,1882.76 L815.815,1883.55 L814.509,1883.93 L813.359,1883.65 L812.447,1882.46 L812.191,1881.48 L812.194,1880.94 L811.98,1880.61 L811.057,1880.23 L810.715,1880.16 L809.636,1880.14 L809.46,1879.92 L808.868,1878.99 L808.598,1878.79 L808.263,1879.17 L807.918,1879.98 L807.643,1880.82 L807.536,1881.28 L807.707,1882.47 L808.086,1883.01 L808.618,1883.35 L809.243,1883.95 L809.691,1885.54 L809.097,1886.9 L807.978,1887.6 L806.642,1887.65 L805.951,1887.46 L803.555,1885.85 L802.369,1886.3 L801.862,1886.66 L800.735,1886.83 L800.265,1887.19 L799.839,1888.02 L800.033,1888.36 L800.411,1889.53 L800.48,1890.33 L802.013,1890.32 L800.863,1891.25 L799.851,1891.75 L799.693,1890.66 L798.986,1889.17 L797.164,1887.98 L795.713,1886.43 L795.121,1884.74 L795.477,1881.09 L795.432,1880.02 L795.209,1879.07 L794.811,1877.96 L794.239,1877 L793.483,1876.47 L793.458,1876.86 L794.668,1879.1 L794.825,1879.77 L794.53,1884.23 L795.017,1886.36 L796.171,1887.73 L797.489,1888.82 L798.459,1890.06 L798.25,1891.35 L796.922,1891.33 L791.216,1890.78 L787.915,1889.77 L786.776,1888.99 L786.813,1887.94 L787.8,1886.99 L787.96,1886.22 L787.631,1885.37 L786.577,1884.47 L784.712,1883.55 L779.963,1883.7 L770.759,1882.24 L769.674,1881.91 L768.74,1881.36 L767.561,1880.29 L766.958,1878.78 L767.77,1877.45 L767.712,1877.23 L766.609,1876.16 L764.002,1875.69 L761.41,1875.82 L759.637,1876.35 L758.981,1876.76 L749.344,1883.46 L743.799,1885.36 L742.222,1886.48 L738.374,1890.77 L737.383,1891.11 L735.635,1890.05 L734.524,1890.16 L732.558,1890.08 L729.365,1891.1 L725.79,1893.51 L724.945,1894.39 L724.597,1894.92 L723.928,1895.4 L723.355,1895.97 L722.511,1896.61 L721.859,1897.35 L721.244,1898.37 L720.325,1899.18 L719.242,1900 L718.693,1901.11 L718.24,1902.02 L717.683,1903.13 L716.822,1904.74 L716.395,1905.87 L715.983,1906.81 L715.374,1908.54 L714.931,1909.84 L715.008,1910.47 L715.566,1910.88 L715.522,1912.12 L714.808,1915.8 L714.175,1919.32 L713.893,1921.79 L713.953,1923.4 L713.469,1925.95 L713.372,1929.33 L713.349,1931.91 L713.435,1933.97 L714.023,1934.83 L714.972,1935.36 L716.2,1935.75 L717.221,1936.19 L717.562,1936.85 L717.502,1937.56 L717.31,1938.25 L717.588,1938.9 L718.013,1939.39 L718.637,1940.61 L718.787,1941.96 L718.741,1942.18 L716.717,1942.34 L715.504,1942.29 L714.336,1942.05 L713.411,1941.66 L712.888,1941.26 L712.48,1940.74 L712.167,1940.13 L711.93,1939.42 L711.632,1939.3 L710.285,1939.14 L709.802,1938.99 L708.97,1939.43 L707.526,1938.72 L705.905,1939.75 L705.121,1939.81 L704.331,1939.7 L703.579,1939.71 L702.801,1940.08 L701.286,1941.3 L700.504,1941.69 L699.526,1941.75 L697.741,1941.3 L696.79,1941.28 L695.935,1941.62 L693.847,1942.96 L693.431,1943.46 L693.743,1944.69 L694.258,1945.38 L694.252,1945.66 L693.008,1945.6 L692.007,1945.31 L689.947,1944.38 L689.052,1944.32 L688.663,1944.61 L688.183,1945.57 L687.844,1945.84 L687.36,1945.81 L686.86,1945.59 L685.972,1944.96 L685.138,1944.65 L684.31,1943.86 L683.675,1942.82 L683.43,1941.78 L682.698,1941.67 L681.29,1941.21 L678.723,1939.68 L678.213,1939.17 L676.994,1938.31 L675.529,1938.32 L672.59,1938.91 L670.437,1938.41 L669.668,1938.68 L669.417,1938.98 L668.979,1939.86 L668.724,1940.2 L667.539,1940.92 L666.68,1941.67 L666.111,1941.95 L665.333,1942.07 L664.116,1942.07 L663.501,1941.91 L662.969,1941.48 L662.14,1940.6 L661.797,1939.59 L661.423,1937.21 L660.898,1936.1 L660.214,1935.71 L659.356,1935.57 L658.3,1935.2 L655.892,1933.15 L654.966,1932.65 L652.144,1931.94 L651.333,1931.49 L651.376,1931.31 L651.359,1931.15 L651.282,1931.01 L651.153,1930.91 L651.479,1929.38 L652.121,1928.15 L653.046,1927.42 L654.213,1927.44 L654.057,1927.16 L653.727,1926.56 L652.691,1926.11 L651.998,1925.56 L652.527,1924.34 L647.354,1923.25 L646.875,1922.96 L644.561,1920.98 L643.853,1920.95 L642.451,1921.31 L641.006,1921.07 L640.039,1923.43 L638.978,1923.73 L638.405,1922.79 L637.859,1920.57 L637.107,1919.26 L636.452,1917.71 L636.09,1917.46 L635.661,1917.38 L635.456,1917.11 L635.75,1916.25 L634.706,1915.83 L629.058,1915.16 L627.473,1915.47 L626.973,1915.34 L626.25,1914.68 L625.875,1913.77 L625.636,1912.77 L625.285,1911.82 L624.556,1911 L623.633,1910.59 L619.758,1909.86 L619.431,1909.61 L618.744,1908.9 L618.373,1908.7 L617.887,1908.8 L617.417,1909.09 L616.956,1909.27 L616.488,1909 L615.841,1908.19 L615.479,1907.84 L614.182,1907.15 L612.34,1905.74 L611.571,1905.5 L610.672,1905.48 L609.894,1905.21 L608.388,1904.39 L607.594,1904.21 L606.331,1904.14 L605.229,1904.43 L604.908,1905.29 L604.853,1906.63 L604.007,1908.28 L604.432,1909.04 L604.06,1909.27 L603.859,1909.51 L603.507,1910.1 L604.128,1910.41 L604.597,1911.17 L604.8,1912.14 L604.636,1913.13 L604.527,1913.27 L604.162,1913.75 L603.535,1913.89 L601.936,1913.47 L596.205,1912.91 L595.483,1912.5 L593.681,1912.73 L592.782,1912.66 L591.887,1912.37 L591.689,1912.13 L591.686,1911.87 L591.733,1911.61 L591.656,1911.4 L590.52,1910.16 L589.653,1910.35 L588.744,1911.06 L587.24,1912.7 L586.547,1912.71 L585.936,1911.77 L585.165,1909.91 L584.478,1909.45 L582.12,1908.5 L581.364,1908.52 L580.434,1908.98 L577.576,1909.03 L573.883,1909.85 L572.28,1909.79 L570.563,1908.83 L570.333,1908.52 L570.159,1908.18 L569.879,1907.86 L569.318,1907.61 L568.863,1907.33 L568.786,1906.92 L568.782,1906.47 L568.545,1906.09 L567.87,1905.37 L567.434,1904.38 L567.303,1903.27 L567.541,1902.21 L566.676,1902.45 L565.307,1902.63 L564.102,1902.32 L563.706,1901.09 L561.261,1899.12 L561.165,1899.04 L560.548,1898.29 L560.221,1898.86 L559.844,1899.29 L559.469,1899.38 L559.125,1898.92 L556.707,1900.34 L555.518,1900.85 L554.109,1900.76 L551.798,1899.27 L550.64,1899.07 L550.365,1900.53 L549.947,1900.61 L549.113,1901.03 L548.747,1901.03 L548.345,1900.68 L548.4,1900.31 L548.517,1899.92 L548.303,1899.51 L548.11,1899.42 L547.37,1899.31 L547.087,1899.13 L547.127,1898.9 L547.25,1898.64 L547.219,1898.37 L545.764,1895.75 L545.044,1894.95 L543.611,1893.76 L543.159,1893.62 L542.706,1893.62 L542.266,1893.5 L541.863,1892.98 L541.685,1892.38 L541.779,1892.06 L541.97,1891.79 L542.052,1891.36 L541.929,1890.85 L541.494,1889.04 L540.333,1888.45 L536.74,1888.57 L533.157,1887.74 L531.496,1886.92 L530.076,1885.5 L529.611,1884.77 L528.887,1884.44 L527.342,1884.01 L527.041,1883.67 L526.357,1883.08 L525.984,1882.87 L524.258,1882.77 L522.947,1881.34 L521.509,1880.24 L518.944,1879.38 L518.164,1878.73 L517.523,1877.73 L517.437,1876.87 L517.919,1876.12 L518.984,1875.45 L517.999,1875.02 L516.928,1875.25 L516.022,1876.01 L515.532,1877.2 L515.191,1878.81 L514.685,1879.58 L513.878,1879.72 L512.65,1879.42 L511.582,1878.95 L510.662,1878.22 L509.973,1877.17 L509.597,1875.74 L510.045,1875.27 L512.2,1873.55 L512.884,1872.87 L513.854,1871.02 L514.698,1868.64 L514.849,1866.35 L513.743,1864.81 L513.203,1864.72 L512.677,1864.82 L512.162,1864.84 L510.421,1863.56 L509.077,1862.92 L508.186,1862.75 L507.452,1863.05 L506.941,1864.05 L506.605,1864.55 L506.071,1864.76 L505.491,1864.74 L505.014,1864.59 L504.5,1864.26 L504.317,1863.94 L504.143,1862.83 L504.148,1862.28 L504.329,1861.92 L504.379,1861.63 L503.98,1861.24 L503.184,1860.7 L502.779,1860.59 L501.805,1860.83 L500.816,1860.81 L500.248,1860.98 L499.748,1860.99 L499.676,1860.74 L499.594,1860.45 L499.62,1859.23 L499.172,1858.37 L497.972,1857.54 L498.072,1856.78 L498.298,1855.49 L498.512,1854.79 L498.621,1854.89 L498.645,1854.92 L499.259,1855.53 L499.666,1855.12 L500.544,1854.91 L504.632,1854.89 L505.895,1853.57 L506.981,1853.18 L507.914,1852.7 L508.384,1852.23 L509.638,1850.46 L509.933,1849.8 L510.399,1849.37 L512.503,1847.34 L512.95,1846.68 L514.543,1845.04 L515.132,1843.14 L516.722,1839.79 L520.024,1829.95 L521.598,1826.58 L524.319,1818.81 L529.706,1801.75 L532.167,1793.27 L533.021,1789.47 L533.044,1788.55 L532.917,1786.74 L533.04,1786.22 L534.453,1784.7 L534.681,1784.33 L534.907,1783.6 L536.104,1780.88 L537.132,1779.93 L538.244,1780.05 L540.851,1781.13 L542.437,1781.13 L543.421,1780.62 L543.569,1779.68 L542.64,1778.37 L543.243,1778.46 L542.935,1777.52 L540.051,1773.88 L539.602,1773.61 L539.413,1773.45 L539.254,1773.23 L539.066,1773.07 L538.759,1773.05 L538.544,1773.21 L538.42,1773.46 L538.327,1773.7 L538.208,1773.79 L537.841,1774.33 L535.421,1777.05 L534.531,1779.71 L533.925,1781.14 L533.536,1781.79 L540.605,1751.68 L543.829,1742.16 L546.484,1729.23 L546.888,1728.36 L548.802,1725.57 L549.561,1724.7 L550.298,1724.54 L550.725,1724.84 L550.787,1725.13 L550.698,1725.6 L550.572,1726.29 L550.421,1726.47 L550.28,1726.79 L550.233,1727.12 L550.32,1727.47 L550.672,1728.28 L550.984,1728.82 L553.573,1731.53 L555.263,1732.83 L555.908,1733.14 L560.36,1739.8 L561.329,1741.79 L561.58,1743.74 L561.521,1744.99 L561.782,1747.38 L561.805,1753.24 L561.886,1753.97 L562.872,1757.78 L564.097,1759.95 L565.555,1761.83 L566.85,1762.86 L567.178,1763.35 L567.436,1764.18 L567.424,1764.76 L567.163,1765.74 L567.042,1767.04 L567.059,1768.35 L567.29,1769.41 L567.823,1769.94 L567.763,1769.06 L567.836,1767.28 L567.673,1766.48 L568.315,1764.87 L567.997,1763.45 L567.156,1762.32 L566.207,1761.53 L567.554,1761.79 L568.656,1762.2 L569.597,1762.85 L570.481,1763.82 L569.637,1761.72 L568.008,1760.6 L566.187,1759.71 L564.791,1758.27 L564.213,1756.54 L563.925,1754.5 L563.896,1742.4 L563.656,1739.92 L562.743,1735.81 L562.11,1733.95 L561.352,1732.26 L559.661,1729.65 L554.136,1724.25 L554.027,1723.97 L554.019,1722.78 L553.781,1722.47 L552.424,1721.73 L550.434,1719.06 L549.205,1718.33 L548.412,1718.04 L547.76,1717.49 L547.239,1716.77 L546.813,1715.99 L546.097,1715.25 L544.526,1715.36 L543.796,1714.92 L545.166,1710.72 L545.274,1709.93 L548.275,1709.48 L549.114,1709.65 L551.4,1713.32 L551.846,1713.73 L552.292,1714.29 L555.123,1716.16 L554.308,1714.91 L550.872,1711.18 L549.946,1709.67 L549.796,1709.08 L549.836,1708.35 L549.778,1707.4 L549.621,1706.46 L549.347,1705.76 L549.863,1705.58 L550.217,1705.25 L550.491,1704.92 L550.736,1704.67 L551.283,1704.31 L552.011,1704 L552.42,1704.01 L552.755,1704.13 L553.051,1704.03 L553.335,1703.34 L553.373,1702.69 L553.211,1702.01 L552.979,1701.42 L552.757,1701.05 L553.024,1700.89 L553.523,1700.48 L553.765,1700.34 L553.126,1699.76 L552.771,1698.86 L552.455,1696.73 L553.268,1697.5 L553.984,1697.65 L554.629,1697.22 L555.253,1696.25 L555.308,1695.87 L555.167,1695.56 L554.963,1695.24 L554.712,1694.97 L553.937,1694.54 L553.857,1694.01 L553.905,1693.38 L553.713,1692.49 L553.811,1691.92 L553.787,1691.67 L553.642,1691.55 L553.116,1691.28 L552.984,1691.15 L552.771,1690.83 L552.54,1690.58 L552.583,1690.32 L553.167,1689.89 L552.274,1688.97 L551.958,1688.84 L551.801,1688.69 L551.852,1688.41 L551.951,1688.12 L551.927,1687.93 L551.703,1687.85 L551.046,1687.8 L550.621,1687.6 L550.229,1687.57 L549.873,1687.44 L549.528,1686.94 L549.556,1686.41 L549.918,1685.85 L550.42,1685.39 L550.86,1685.18 L550.802,1683.96 L551.39,1683.22 L554.608,1681.82 L555.069,1681.09 L555.104,1680.58 L554.987,1680.03 L554.997,1679.13 L554.259,1677.67 L552.361,1677.25 L550.634,1677.96 L550.397,1679.93 L549.292,1679.43 L548.622,1678.31 L548.015,1677.01 L547.361,1676.71 L546.94,1676.57 L545.742,1675.57 L544.898,1674.58 L544.361,1674.25 L541.983,1674.62 L540.907,1674.4 L540.639,1673 L540.62,1671.4 L540.108,1670.17 L539.228,1669.33 L538.092,1668.91 L535.256,1668.34 L534.167,1667.78 L534.644,1667.07 L534.051,1666.3 L531.84,1665.53 L530.604,1664.04 L529.447,1663.52 L528.965,1663.1 L528.733,1662.54 L528.463,1661.11 L528.16,1660.38 L528.054,1661.15 L528.146,1662.2 L528.12,1662.75 L527.896,1662.99 L527.53,1662.71 L527.216,1662.08 L526.969,1657.39 L527.322,1656.35 L526.685,1655.95 L526.457,1655.2 L526.389,1654.34 L526.237,1653.58 L525.716,1652.73 L525.151,1652.06 L524.736,1651.38 L524.552,1649.73 L524.06,1649.18 L523.384,1648.82 L522.698,1648.63 L522.395,1648.2 L521.543,1645.42 L519.576,1642.56 L518.413,1641.4 L517.378,1640.84 L516.951,1640.35 L516.795,1639.34 L516.816,1638.13 L516.933,1637.06 L517.251,1635.74 L517.638,1635.16 L518.846,1634.55 L518.74,1633.63 L519.464,1633.08 L521.237,1632.34 L521.899,1631.49 L522.962,1629.38 L523.636,1628.37 L524.464,1627.93 L524.551,1626.87 L524.252,1625.7 L523.911,1624.91 L523.363,1624.15 L521.059,1622.31 L516.347,1620.64 L514.982,1619.57 L515.6,1619.28 L516.123,1618.93 L516.669,1618.66 L518.068,1618.69 L518.424,1618.53 L518.634,1618.05 L518.908,1617.15 L519.06,1616.31 L518.769,1615.27 L518.801,1614.32 L518.962,1613.64 L519.253,1612.87 L519.642,1612.18 L520.098,1611.71 L521.175,1611.44 L525.097,1611.68 L525.423,1611.5 L525.729,1611.4 L526.047,1611.4 L526.315,1611.88 L526.581,1612.22 L528.641,1613.14 L529.29,1613.22 L529.067,1612.76 L529.551,1612.85 L529.98,1613.05 L530.347,1613.38 L530.658,1613.84 L529.801,1613.7 L529.729,1614.16 L531.252,1615.71 L532.944,1616.34 L533.326,1616.7 L534.099,1617.85 L534.996,1618.44 L537.251,1618.88 L536.491,1618.18 L534.822,1617.55 L534.184,1616.83 L533.148,1615.11 L531.862,1613.64 L530.554,1612.53 L529.757,1612.11 L528.784,1611.85 L528.434,1611.64 L527.942,1610.94 L526.425,1610.22 L520.58,1609.66 L519.132,1610.25 L518.763,1610.66 L518.532,1611.04 L518.244,1611.39 L517.746,1611.72 L516.438,1611.88 L515.301,1612.69 L514.454,1612.88 L513.63,1612.73 L512.926,1612.29 L512.394,1611.62 L511.566,1610.26 L510.957,1609.75 L510.273,1609.54 L509.877,1609.5 L509.554,1609.51 L509.174,1609.75 L508.949,1610.14 L508.822,1610.5 L508.749,1610.67 L507.53,1610.43 L503.952,1607.69 L504.022,1607.27 L504.959,1607.57 L506.616,1608.74 L507.593,1608.75 L508.452,1608.02 L508.202,1607.13 L507.499,1606.16 L507.007,1605.18 L506.675,1605.28 L506.52,1605.35 L506.374,1605.46 L506.504,1605.95 L506.481,1606.2 L506.159,1606.76 L505.847,1606.71 L506.074,1604.91 L505.751,1603.88 L504.2,1602.45 L507.856,1601.02 L508.297,1601.12 L508.698,1601.38 L509.376,1601.64 L509.237,1601.97 L509.187,1601.96 L508.993,1602.05 L509.238,1602.48 L509.716,1602.08 L511.287,1601.52 L511.487,1601.13 L510.869,1600.89 L509.845,1600.69 L509.205,1600.3 L509.729,1599.49 L509.125,1598.91 L508.27,1598.67 L507.793,1598.24 L508.334,1597.13 L507.877,1596.22 L508.409,1595.89 L511.036,1596.19 L512.63,1596.02 L513.486,1596.23 L512.841,1595.92 L511.685,1595.93 L510.923,1595.79 L507.949,1593.98 L507.308,1593.76 L506.754,1593.74 L505.314,1593.97 L503.752,1593.7 L503.349,1593.76 L503.073,1593.94 L502.811,1593.94 L502.448,1593.47 L503.095,1593.46 L503.904,1593.13 L504.641,1592.54 L505.098,1591.76 L504.348,1592.29 L503.742,1592.28 L503.136,1592.12 L502.398,1592.16 L502.725,1592.29 L502.962,1592.46 L503.09,1592.72 L503.11,1593.15 L502.055,1592.98 L501.586,1592.78 L501.192,1592.38 L500.879,1593.85 L500.255,1594.08 L498.952,1593.34 L498.249,1593.26 L495.952,1593.64 L495.131,1593.58 L494.518,1593.4 L494.02,1593.03 L493.551,1592.39 L493.506,1592.14 L493.072,1590.82 L492.926,1590.73 L492.81,1590.37 L492.444,1590.1 L491.986,1589.94 L491.87,1589.85 L491.944,1589.78 L491.971,1589.47 L491.304,1589.07 L492.084,1588.69 L493.219,1588.88 L493.621,1590.23 L493.909,1590.28 L494.195,1590 L494.455,1589.9 L494.73,1590.05 L495.044,1590.48 L495.157,1590.09 L495.549,1589.23 L496.065,1590.12 L497.063,1590.65 L498.215,1590.73 L499.197,1590.3 L498.958,1589.83 L500.964,1587.84 L501.35,1586.73 L500.617,1585.34 L500.553,1585.73 L500.882,1586.21 L500.87,1586.62 L500.571,1586.88 L500.029,1586.94 L500.55,1587.46 L500.315,1587.72 L500.111,1587.83 L499.623,1587.77 L499.468,1587.39 L497.614,1585.69 L498.461,1585.97 L499.242,1586.4 L498.871,1585.6 L498.431,1585.2 L497.184,1584.74 L497.691,1585.23 L496.762,1585.49 L494.576,1585.34 L494.075,1585.7 L493.584,1586.37 L492.704,1586.61 L491.272,1586.7 L491.338,1586.32 L491.619,1586.37 L491.697,1585.91 L491.14,1585.91 L490.992,1585.99 L490.77,1586.22 L490.485,1586.17 L490.455,1585.39 L490.519,1584.75 L490.732,1584.3 L491.137,1584.07 L490.543,1583.94 L490.162,1583.54 L489.987,1582.92 L490.01,1582.14 L489.699,1582.08 L489.673,1582.57 L489.737,1583.38 L489.721,1583.82 L489.467,1585.3 L489.51,1585.94 L489.763,1586.69 L490.122,1587.39 L490.481,1587.9 L490.303,1588.31 L490.205,1588.48 L490.052,1588.7 L489.979,1588.45 L489.655,1587.76 L489.267,1587.94 L488.961,1587.47 L488.627,1586.75 L488.151,1586.15 L488.255,1587.01 L487.956,1587.35 L487.601,1587.12 L487.543,1586.28 L487.556,1585.3 L487.272,1585.35 L487.009,1586.01 L487.118,1586.87 L486.533,1586.72 L486.304,1586.6 L486.081,1586.25 L484.649,1586.73 L483.977,1586.68 L483.457,1586.22 L483.754,1586.09 L484.071,1585.89 L483.732,1585.44 L483.304,1585.05 L482.833,1584.99 L482.368,1585.59 L482.54,1585.71 L482.68,1585.95 L482.863,1586.11 L482.05,1588.11 L481.978,1589.91 L482.417,1591.47 L483.154,1592.72 L482.594,1592.59 L481.761,1592.19 L481.449,1592.41 L481.19,1592.37 L480.996,1590.5 L481.027,1589.93 L481.234,1589.27 L481.805,1588.29 L482.039,1587.48 L482.186,1586.02 L482.081,1584.9 L481.766,1584.01 L481.29,1583.22 L480.939,1582.89 L480.507,1582.61 L480.157,1582.3 L480.061,1581.89 L480.292,1581.11 L481.094,1579.47 L484.465,1578.58 L484.361,1578.22 L484.333,1577.86 L484.374,1577.42 L484.481,1576.8 L484.26,1576.9 L483.741,1577.01 L483.526,1577.1 L483.728,1575.8 L483.826,1575.38 L483.568,1575.33 L483.131,1575.93 L482.706,1575.73 L482.199,1575.35 L481.521,1575.36 L482.061,1575.69 L482.377,1576.21 L482.525,1576.92 L482.552,1577.8 L482.338,1577.81 L482.29,1577.84 L482.273,1577.94 L482.172,1578.16 L481.742,1577.71 L481.412,1577.89 L481.017,1578.22 L480.402,1578.28 L480.53,1578.49 L480.844,1579.22 L480.216,1579.4 L479.918,1579.94 L479.686,1580.49 L479.263,1580.68 L478.998,1580.38 L477.987,1578.71 L476.983,1578.21 L475.249,1577.38 L474.147,1577.14 L474.95,1576.97 L475.673,1577.19 L477.216,1578.09 L477.283,1577.71 L476.697,1577.19 L476.005,1576.79 L474.297,1576.3 L474.378,1575.84 L474.754,1575.87 L474.953,1575.76 L475.055,1575.46 L475.156,1574.89 L475.341,1574.67 L476.026,1574.91 L476.299,1574.88 L477.073,1573.61 L477.576,1572.95 L478.181,1572.62 L478.261,1572.16 L477.174,1573.06 L475.847,1573.94 L474.456,1574.45 L473.55,1574.5 L473.835,1574.74 L474.047,1575.05 L474.188,1575.41 L474.12,1575.79 L470.573,1576.02 L470.016,1575.73 L469.897,1575.36 L468.946,1573.53 L468.579,1572.16 L468.218,1571.54 L468.09,1571.24 L468.015,1570.75 L467.492,1571.04 L466.818,1570.9 L465.424,1570.26 L464.344,1570.11 L464.01,1570 L463.718,1569.75 L463.346,1569.12 L463.175,1568.97 L461.595,1568.76 L460.852,1568.32 L460.657,1567.41 L460.332,1567.18 L457.954,1567.99 L457.462,1567.28 L456.58,1567.13 L455.619,1567.15 L454.884,1566.93 L454.865,1566.72 L454.347,1565.09 L454.118,1564.71 L453.863,1564.41 L453.787,1564.02 L454.089,1563.33 L453.734,1563.25 L453.496,1563.09 L453.045,1562.66 L453.04,1561.97 L452.431,1560.92 L451.469,1559.97 L450.419,1559.54 L450.556,1560.87 L450.085,1561.75 L448.97,1562.59 L448.176,1562.2 L448.584,1562.31 L448.928,1562.29 L449.218,1562.05 L449.463,1561.54 L448.929,1561.63 L448.514,1561.57 L448.178,1561.65 L447.867,1562.14 L446.884,1560.94 L446.377,1560.62 L445.724,1560.42 L445.276,1560.04 L444.887,1559.21 L444.664,1558.16 L444.726,1557.14 L444.155,1557.78 L444.223,1558.76 L444.619,1559.71 L445.031,1560.29 L444.959,1560.67 L444.433,1560.61 L444,1560.68 L443.576,1560.88 L443.082,1561.21 L442.559,1559.71 L441.952,1559.76 L441.692,1560.49 L442.21,1561.04 L442.052,1561.25 L441.877,1561.41 L441.805,1561.8 L442.931,1562.02 L442.844,1562.48 L441.26,1563.52 L440.555,1563.77 L439.484,1563.87 L433.491,1562.38 L433.604,1561.92 L433.677,1561.05 L433.812,1560.71 L434.105,1560.6 L434.491,1560.67 L434.853,1560.67 L435.089,1560.28 L435.156,1559.15 L434.844,1557.45 L434.763,1555.7 L434.362,1554.45 L432.893,1551.5 L429.812,1547.88 L429.444,1547.58 L428.827,1547.81 L428.203,1548.33 L427.611,1548.51 L426.617,1547.06 L425.934,1546.79 L424.533,1546.59 L421.623,1545.09 L421.938,1544.93 L422.218,1544.59 L422.467,1544.24 L422.803,1543.84 L422.873,1543.62 L422.972,1543.4 L423.156,1543.21 L423.296,1543.23 L423.547,1543.67 L423.767,1543.81 L437.531,1543.48 L439.738,1544.66 L440.396,1544.52 L440.665,1544.12 L441.393,1542.46 L441.342,1541.72 L440.905,1540.86 L440.864,1540.41 L440.921,1538.99 L440.477,1538.32 L438.818,1537.59 L438.534,1537.32 L438.333,1536.97 L438.222,1536.57 L438.2,1536.16 L437.099,1536.34 L435.193,1534.5 L433.98,1534.66 L433.175,1535.63 L432.331,1536.92 L431.382,1537.95 L430.254,1538.1 L430.866,1536.86 L431.148,1536.5 L430.825,1535.7 L430.675,1535.12 L430.663,1534.65 L431.008,1533.96 L431.376,1533.76 L431.577,1533.54 L431.418,1532.86 L430.945,1532.63 L428.512,1532.03 L428.747,1531.41 L428.759,1530.77 L429.254,1530.61 L429.587,1530.71 L429.9,1530.92 L430.329,1531.09 L430.882,1531.01 L431.057,1530.61 L431.035,1530.16 L431.008,1529.92 L432.107,1528.14 L432.725,1527.75 L433.338,1528.2 L432.363,1530.37 L433.387,1531.14 L437.071,1531.61 L437.444,1531.9 L437.667,1532.22 L437.937,1532.51 L438.449,1532.71 L438.953,1532.76 L440.988,1532.58 L443.764,1532.93 L443.456,1533.14 L443.178,1533.2 L442.576,1533.09 L443.352,1533.59 L446.833,1533.53 L446.92,1533.07 L445.623,1532.91 L445.189,1532.58 L445.157,1531.82 L444.688,1532.23 L444.096,1532.27 L443.627,1531.97 L443.521,1531.3 L443.251,1531.32 L441.484,1531.1 L441.933,1530.61 L442.133,1530.5 L442.505,1530.43 L442.575,1530.17 L442.814,1529.95 L443.269,1529.74 L444.223,1529.02 L443.532,1528.89 L442.476,1529.39 L441.799,1529.45 L441.88,1529.03 L442.182,1529.09 L442.27,1528.63 L441.968,1528.57 L441.784,1528.85 L441.468,1529.11 L441.268,1529.35 L440.711,1528.96 L439.25,1529.61 L438.303,1529.59 L438.462,1529.38 L438.507,1529.37 L438.68,1529.27 L438.458,1528.79 L438.048,1529.06 L437.532,1529.22 L436.955,1529.26 L436.38,1529.21 L436.455,1528.83 L439.332,1526.45 L439.852,1526.17 L444.197,1524.61 L443.43,1524.69 L442.014,1525.17 L441.143,1525.32 L438.303,1525.23 L436.934,1525.52 L434.045,1526.64 L431.954,1526.73 L430.618,1527.11 L429.802,1527.06 L429.123,1526.75 L427.819,1525.88 L427.054,1525.59 L426.87,1525.76 L426.782,1526.21 L426.63,1526.64 L426.23,1526.77 L424.176,1526.33 L423.672,1526.01 L423.438,1525.31 L423.44,1523.98 L424.212,1524.06 L424.303,1523.26 L423.938,1521.46 L424.105,1520.29 L424.538,1519.18 L425.185,1518.43 L425.997,1518.34 L426.073,1517.96 L425.193,1516.99 L425.955,1515.29 L427.327,1513.62 L428.266,1512.69 L428.633,1513.06 L429.035,1513.32 L428.824,1512.81 L429.56,1512.49 L430.373,1512.39 L431.657,1512.51 L432.402,1512.89 L433.599,1513.93 L434.139,1513.88 L433.451,1512.39 L433.322,1511.49 L433.927,1511.22 L435.69,1512.08 L436.454,1512.2 L435.433,1511.52 L435.607,1511.3 L435.72,1511.11 L435.669,1510.89 L435.356,1510.59 L435.431,1510.21 L438.6,1510.85 L439.548,1510.86 L441.014,1510.49 L441.96,1510.65 L441.468,1510.19 L441.235,1510.06 L441.324,1509.61 L442.517,1509.63 L443.774,1509.18 L444.951,1508.96 L445.994,1509.22 L446.559,1510.25 L445.999,1511.56 L446.969,1511.64 L448.35,1511.13 L448.802,1511.09 L449.28,1511.28 L450.129,1511.83 L450.674,1511.93 L450.761,1511.47 L450.024,1511.09 L450.356,1510.51 L451.189,1509.92 L451.96,1509.56 L452.8,1509.4 L453.775,1509.4 L454.685,1509.64 L455.334,1510.22 L455.643,1509.89 L455.865,1509.95 L456.011,1510.28 L456.091,1510.76 L456.342,1510.81 L456.667,1510.57 L456.901,1510.24 L457.015,1509.78 L456.962,1509.19 L457.544,1509.19 L458.075,1509.1 L459.034,1508.72 L460.548,1508.57 L460.763,1508.82 L460.551,1509.38 L460.322,1509.8 L460.309,1510.12 L460.75,1510.4 L460.679,1510.78 L460.405,1510.73 L460.319,1511.18 L460.676,1511.53 L460.862,1512.04 L460.807,1512.67 L460.46,1513.35 L461.817,1512.43 L462.47,1512.23 L463.243,1512.58 L462.769,1512.87 L462.576,1512.93 L462.504,1513.31 L462.872,1514.07 L463.928,1515.07 L464.323,1515.88 L464.702,1515.51 L464.479,1515.04 L464.858,1514.67 L464.741,1514.25 L464.693,1513.72 L464.705,1512.42 L464.956,1512.47 L465.019,1512.86 L465.109,1513.07 L465.37,1513.46 L465.464,1512.37 L465.803,1511.2 L466.251,1510.58 L466.678,1511.06 L466.905,1510.81 L467.049,1510.53 L467.137,1510.2 L467.196,1509.81 L467.586,1510.66 L468.3,1511.04 L470.205,1511.29 L470.621,1511.56 L471.399,1512.58 L471.812,1512.9 L472.328,1513.02 L473.849,1512.89 L473.643,1513.93 L474.581,1514.43 L475.769,1514.65 L476.406,1514.44 L476.262,1512.94 L476.269,1511.96 L476.511,1511.57 L478.698,1511.58 L478.178,1511.09 L477.097,1509.07 L477.113,1508.03 L477.618,1507.58 L478.884,1507.26 L480.054,1505.83 L480.372,1505.36 L480.619,1505.41 L480.943,1505.79 L482.212,1506.1 L483.462,1507.19 L484.014,1507.35 L483.751,1507.49 L483.612,1507.63 L483.502,1507.82 L483.311,1508.08 L484.247,1508.34 L484.754,1508.38 L485.157,1508.21 L485.656,1507.78 L485.968,1507.64 L489.617,1506.97 L490.156,1506.7 L490.433,1506.75 L490.797,1506.92 L492.273,1505.62 L492.899,1505.89 L493.003,1506.83 L492.765,1508.03 L492.276,1509.09 L491.641,1509.61 L491.636,1509.9 L491.655,1510.08 L491.79,1510.51 L492.077,1509.76 L492.512,1509.23 L493.994,1507.93 L495.143,1507.25 L495.582,1506.86 L495.968,1507.13 L496.882,1506.34 L497.424,1506.28 L497.864,1506.7 L498.054,1507.31 L497.965,1507.96 L497.575,1508.52 L497.507,1508.9 L497.981,1509.46 L497.586,1509.91 L496.781,1511.42 L496.429,1511.86 L495.356,1512.86 L495.275,1513.32 L495.578,1513.38 L497.406,1510.84 L498.403,1509.81 L499.038,1510.08 L499.499,1509.82 L499.948,1509.78 L500.354,1509.97 L500.702,1510.38 L500.323,1510.75 L500.471,1510.84 L500.569,1510.85 L500.597,1510.93 L500.555,1511.22 L500.143,1511.28 L499.799,1511.4 L499.019,1511.82 L498.951,1512.2 L499.403,1512.76 L500.373,1513.35 L502.415,1513.82 L503.169,1514.26 L503.026,1514.77 L502.963,1515.14 L502.943,1515.56 L502.494,1515.83 L502.318,1515.89 L502.251,1516.28 L502.779,1516.77 L503.402,1517.54 L503.899,1518.38 L504.061,1519.04 L506.315,1521.81 L506.403,1523.73 L506.484,1524.25 L506.493,1524.58 L506.324,1524.84 L506.188,1525.14 L506.26,1525.55 L506.431,1525.77 L506.915,1526.13 L507.765,1527.05 L509.898,1528.6 L509.684,1529.14 L509.443,1529.54 L509.153,1529.85 L508.782,1530.1 L509.258,1530.21 L509.613,1530.5 L509.895,1530.92 L510.161,1531.45 L510.649,1532.02 L510.857,1531.46 L510.984,1530.57 L511.23,1530.13 L512.84,1530.51 L513.557,1530.32 L514.492,1529.6 L515.716,1528.23 L516.093,1527.9 L516.581,1527.63 L517.346,1527.52 L520.386,1526.27 L521.069,1525.72 L521.139,1525.3 L520.801,1525.19 L520.071,1524.68 L521.982,1524.43 L522.849,1524.72 L523.067,1525.63 L524.199,1525.38 L525.563,1524.78 L526.801,1523.9 L527.554,1522.87 L527.581,1523.62 L527.799,1524.12 L528.179,1524.38 L528.735,1524.38 L528.659,1524.84 L528.315,1525.04 L528.084,1525.33 L527.876,1525.65 L527.618,1525.96 L526.27,1527.04 L526.895,1527.74 L527.741,1527.35 L528.717,1526.62 L529.716,1526.32 L529.855,1526.68 L529.843,1528.17 L529.928,1528.53 L530.355,1528.82 L530.541,1529.41 L530.573,1530.12 L530.547,1530.79 L531.033,1530.62 L531.405,1530.2 L531.649,1529.59 L531.752,1528.83 L531.777,1529.76 L531.872,1530.57 L532.177,1531 L532.847,1530.74 L532.915,1530.33 L532.758,1530.23 L532.446,1529.85 L533.012,1529.23 L533.549,1528.94 L534.042,1529.08 L534.471,1529.72 L534.251,1529.29 L534.1,1528.83 L533.908,1527.89 L535.835,1527.54 L537.82,1527.67 L537.879,1527.97 L538.482,1529.71 L538.442,1530.39 L538.549,1531.04 L538.835,1531.55 L539.312,1531.82 L539.041,1532.4 L538.873,1532.61 L539.67,1534.4 L539.788,1535.26 L539.41,1536.6 L539.883,1536.22 L541.287,1535.61 L540.274,1534.62 L540.571,1534.17 L540.145,1532.97 L540.153,1531.95 L540.431,1532 L540.554,1532.31 L540.698,1532.52 L541.136,1532.98 L540.948,1532.22 L540.639,1531.78 L540.237,1531.56 L539.778,1531.46 L539.453,1531.04 L538.991,1529.07 L538.717,1528.25 L538.926,1527.78 L539.319,1527.27 L539.862,1526.9 L540.523,1526.82 L540.694,1526.64 L540.764,1526.2 L540.907,1525.78 L541.304,1525.64 L542.838,1525.89 L542.656,1525.39 L546.945,1525.23 L546.783,1525.48 L546.73,1525.75 L546.814,1526.1 L547.041,1526.58 L545.763,1528.08 L545.543,1529.53 L546.305,1530.73 L547.953,1531.5 L550.939,1532.02 L554.241,1532.03 L556.382,1531.26 L556.636,1531.25 L557.058,1531.22 L560.228,1532.16 L564.771,1531.86 L566.18,1532.66 L565.699,1531.49 L564.86,1530.87 L563.319,1530.48 L563.291,1530.62 L563.062,1530.83 L562.82,1530.89 L562.754,1530.59 L562.768,1530.18 L562.682,1529.81 L562.503,1529.54 L561.161,1528.9 L560.703,1528.49 L560.545,1528.08 L560.423,1526.56 L559.117,1524.96 L558.999,1523.44 L559.244,1521.73 L559.3,1520.24 L558.958,1519.19 L558.017,1518.78 L558.088,1518.32 L558.692,1518.15 L559.209,1517.66 L559.592,1516.98 L560.383,1514.07 L560.752,1513.48 L561.021,1514.06 L561.359,1513.39 L561.476,1512.85 L561.325,1512.47 L560.859,1512.26 L560.75,1511.88 L561.744,1508.37 L562.15,1507.49 L562.74,1507.28 L563.656,1507.9 L563.714,1507.52 L562.991,1507.02 L562.08,1506.88 L561.253,1507.2 L560.785,1508.1 L560.618,1508.33 L560.387,1507.93 L560.198,1507.26 L560.15,1506.72 L560.413,1505.25 L560.605,1504.6 L560.881,1504 L560.328,1503.8 L560.185,1503.29 L560.335,1502.61 L560.654,1501.84 L561.061,1501.09 L561.4,1500.76 L562.276,1500.33 L561.84,1499.86 L561.727,1499.93 L561.473,1500.2 L561.285,1499.17 L561.195,1497.96 L561.21,1496.71 L561.319,1495.59 L561.725,1495.05 L562.468,1495.11 L563.857,1495.77 L563.916,1495.39 L563.102,1494.99 L562.143,1494.32 L561.275,1494.16 L560.777,1495.28 L560.254,1494.42 L559.943,1492.04 L559.461,1491.21 L559.766,1490.98 L559.893,1490.91 L560.085,1490.84 L559.072,1489.73 L558.689,1488.97 L558.537,1487.95 L559.361,1488.09 L559.422,1487.7 L559.1,1486.35 L558.797,1486.3 L558.302,1487.32 L557.632,1486.72 L556.886,1485.57 L556.165,1484.97 L556.339,1484.56 L556.438,1484.39 L556.602,1484.18 L556.02,1484.22 L555.451,1484.38 L554.972,1484.35 L554.65,1483.86 L554.654,1483.03 L555.142,1480.95 L555.256,1480.06 L555.206,1479.61 L555.101,1479.19 L554.685,1478.02 L554.681,1477.71 L554.702,1477.39 L554.639,1476.89 L554.349,1476.16 L553.516,1474.71 L553.366,1474.31 L553.756,1473.49 L555.055,1472.63 L555.413,1472.04 L555.904,1470.14 L556.057,1469.11 L555.9,1468.18 L555.685,1467.44 L555.56,1466.8 L555.272,1466.32 L554.029,1465.85 L552.39,1464.81 L552.944,1463.4 L553.017,1462.47 L553.267,1462.02 L554.362,1462.06 L554.715,1462.25 L555.414,1462.97 L555.866,1463.17 L557.007,1463.19 L557.378,1463.42 L557.53,1463.72 L557.556,1464.49 L557.689,1464.81 L557.865,1464.9 L558.316,1464.88 L558.507,1464.94 L559.339,1465.39 L561.924,1466.22 L564.75,1467.73 L565.393,1467.79 L565.39,1467.93 L565.355,1468.24 L565.406,1468.56 L565.675,1468.74 L566.071,1468.66 L566.71,1468.14 L567.166,1468.07 L567.621,1468.22 L568.403,1468.68 L568.879,1468.78 L569.607,1468.57 L570.322,1468.07 L571.574,1466.82 L572.219,1466.47 L574.158,1466.18 L574.898,1466.22 L579.569,1467.86 L579.866,1468.09 L579.956,1468.56 L579.999,1469.61 L580.591,1471.63 L580.607,1472.73 L579.471,1473.3 L579.213,1473.86 L579.027,1474.43 L578.826,1474.66 L577.933,1474.54 L577.56,1474.62 L577.255,1474.89 L576.837,1476.33 L577.159,1477.63 L577.679,1478.83 L578.019,1480.63 L580.5,1484.99 L580.746,1485.99 L580.682,1487.36 L580.499,1488.03 L580.246,1488.49 L580.014,1488.78 L579.909,1488.95 L579.941,1489.61 L580.279,1489.61 L580.754,1489.39 L581.222,1489.39 L581.598,1489.83 L581.837,1490.32 L582.226,1490.64 L583.043,1490.54 L583.107,1490.11 L582.782,1489.43 L583.207,1488.63 L583.995,1487.99 L584.742,1487.77 L590.014,1488.85 L594.516,1491.4 L601.37,1492.84 L603.024,1493.62 L604.005,1493.86 L605.59,1493.26 L606.544,1493.63 L608.173,1494.66 L610.935,1495.05 L611.601,1495.45 L612.708,1496.43 L614.677,1497.19 L618.146,1500.04 L619.302,1499.24 L625.995,1498.6 L628.006,1497.95 L629.246,1497.13 L630.202,1497 L630.6,1496.75 L631.773,1495.36 L634.715,1493.78 L639.895,1493.31 L645.069,1492.83 L647.048,1492.11 L647.916,1491.52 L648.705,1490.68 L645.605,1492.11 L642.211,1492.15 L638.93,1491.25 L636.173,1489.88 L635.346,1489.77 L634.648,1489.61 L633.762,1489.19 L633.023,1488.59 L632.761,1487.95 L632.683,1487.56 L632.164,1487.09 L632.104,1486.54 L632.244,1485.87 L632.428,1485.44 L634.996,1481.91 L637.332,1476.66 L638.029,1476.02 L640.249,1474.67 L640.687,1474.72 L641.145,1474.38 L646.704,1472.64 L647.187,1472.12 L652.593,1469.76 L654.033,1469.51 L655.746,1468.63 L659.022,1467.88 L662.465,1468.37 L679.601,1465.07 L680.668,1464.4 L681.678,1464.18 L683.48,1462.6 L689.181,1459.01 L691.593,1456.64 L692.106,1456.3 L692.515,1455.81 L693.784,1452.63 L695.358,1450.94 L696.748,1450.91 L700.471,1453.3 L700.864,1453.32 L701.239,1453.02 L701.287,1452.59 L700.863,1452.57 L700.608,1452.44 L700.43,1452.15 L700.263,1451.65 L701.08,1451.75 L701.132,1451.29 L700.363,1451.18 L699.589,1450.83 L699.029,1450.26 L698.849,1449.53 L698.558,1448.69 L696.949,1447.66 L696.626,1447.11 L696.919,1445.12 L697.531,1442.72 L698.535,1441.16 L699.425,1441.46 L700.021,1441.65 L700.065,1441.26 L698.073,1439.28 L698.122,1438.85 L699.089,1436.83 L699.452,1433.91 L699.942,1431.4 L701.279,1430.61 L700.399,1428.83 L700.283,1427.94 L700.447,1425 L700.341,1422.3 L700.469,1421.01 L700.792,1419.93 L701.905,1417.56 L702.458,1416.7 L702.811,1415.54 L702.724,1414.09 L702.354,1411.72 L702.36,1410.84 L702.387,1410.53 L703.427,1409.87 L703.783,1409.77 L704.609,1409.79 L705.389,1409.59 L706.129,1408.98 L707.443,1407.53 L708.779,1406.52 L716.659,1404.08 L717.345,1404.3 L718.771,1403.66 L725.151,1403.55 L732.105,1402.06 L732.543,1401.88 L732.877,1401.86 L735.198,1402.39 L738.482,1401.89 L740.15,1401.3 L740.772,1400.95 L740.785,1400.94 L741.236,1402.43 M662.47,1935.84 L663.504,1936.11 L663.633,1936.12 L664.1,1936.07 L664.264,1936.02 L663.842,1935.08 L663.479,1933.99 L663.069,1933.27 L662.52,1933.46 L661.904,1934.52 L661.912,1935.31 L662.47,1935.84"
         id="path1043" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['FR'],polycolors['FR'],labels['FR'], x1, polybary['FR'], x2, polybary['FR'] )
    if 'FR' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1446.57,1864.64 L1447.54,1864.49 L1448.38,1864.03 L1449.43,1862.46 L1450.54,1862.18 L1452.73,1862.25 L1451.99,1863.8 L1452.22,1865.57 L1453.15,1867.12 L1454.45,1867.98 L1460.99,1870 L1461.81,1870.46 L1464.04,1873.06 L1467.04,1875.51 L1467.69,1876.26 L1468.68,1877.91 L1469.22,1878.55 L1470.99,1879.69 L1471.09,1879.76 L1472.72,1881.42 L1473.59,1882.04 L1474.58,1882.41 L1476.67,1882.75 L1477.66,1882.74 L1478.22,1882.45 L1478.54,1882.03 L1478.91,1882.02 L1480.54,1884.26 L1481.51,1885.25 L1482.54,1885.89 L1483.65,1886.15 L1485.17,1886.06 L1486.69,1885.73 L1487.18,1885.33 L1488.11,1884.1 L1488.51,1883.81 L1489.1,1883.93 L1489.5,1884.38 L1489.84,1884.98 L1490.26,1885.53 L1490.77,1885.93 L1493.74,1887.49 L1495.51,1888.84 L1495.61,1888.91 L1498.71,1891.27 L1499.31,1891.38 L1499.85,1891.48 L1502.32,1891.4 L1503.7,1891.79 L1506.31,1893.18 L1509.29,1894.21 L1510.17,1895.09 L1510.44,1896.62 L1510.3,1899.07 L1506.69,1900.21 L1505.03,1901.1 L1504.67,1901.41 L1504.38,1902.14 L1504.41,1902.74 L1504.56,1903.29 L1504.56,1903.9 L1503.58,1905.35 L1501.98,1905.1 L1500.13,1904.3 L1498.45,1904.1 L1497.96,1904.41 L1497.58,1904.82 L1496.71,1905.71 L1496.4,1906.27 L1496.48,1907.51 L1497.12,1908.39 L1497.16,1908.43 L1500.09,1910.57 L1500.53,1911.16 L1500.65,1911.37 L1500.24,1913.05 L1499.84,1914.11 L1499.74,1914.64 L1499.86,1915.64 L1500.09,1916.49 L1500.14,1917.3 L1499.69,1918.19 L1498.42,1917.67 L1497.53,1918.51 L1496.15,1920.93 L1495.02,1921.42 L1492.78,1921.65 L1491.73,1922.59 L1491.73,1922.6 L1491.73,1922.61 L1491.35,1923.07 L1489.74,1924.33 L1489.46,1924.46 L1489.04,1924.65 L1488.19,1924.66 L1487.36,1924.37 L1485.71,1923.4 L1484.69,1921.58 L1484.46,1921.03 L1484.38,1920.62 L1484.35,1919.67 L1484.38,1919.22 L1484.63,1918.45 L1484.8,1918.08 L1484.89,1918.13 L1484.78,1917.51 L1484.66,1916.9 L1484.08,1915.37 L1483.21,1914.13 L1482.1,1913.72 L1481.1,1914.36 L1480.36,1915.22 L1479.95,1915.7 L1478.92,1917.26 L1478.3,1918.61 L1478.14,1919.5 L1478.07,1921.15 L1477.91,1921.93 L1477.41,1922.65 L1476.8,1923.11 L1476.31,1923.66 L1476.16,1924.62 L1476.16,1924.63 L1475.74,1925.85 L1475.23,1926.97 L1474.58,1927.93 L1473.74,1928.64 L1473.25,1929.73 L1471.21,1932.75 L1471.01,1933.14 L1470.7,1934.2 L1470.66,1934.66 L1470.79,1935.51 L1470.78,1935.83 L1467.16,1943.25 L1465.96,1944.89 L1465.9,1945.56 L1466.47,1946.49 L1466.48,1946.5 L1467.3,1947.21 L1470.17,1948.84 L1470.96,1949.59 L1471.09,1949.71 L1471.27,1950.29 L1470.64,1952.25 L1470.54,1953.34 L1470.63,1954.57 L1470.93,1955.73 L1471.47,1956.61 L1471.48,1956.62 L1471.87,1957.01 L1471.75,1958.08 L1471.5,1958.35 L1471.12,1958.44 L1470.91,1958.7 L1471.37,1960.09 L1471.31,1960.54 L1471.17,1960.89 L1471.15,1961.2 L1471.22,1961.61 L1471.22,1961.87 L1471.36,1962.11 L1471.84,1962.47 L1472.29,1963.08 L1472.44,1964.7 L1472.56,1965.35 L1472.56,1965.35 L1471.63,1964.73 L1469.71,1963.06 L1468.87,1962.77 L1465.36,1962.07 L1464.5,1961.66 L1463.31,1961.37 L1462.9,1961.17 L1462.46,1960.59 L1462.41,1960.26 L1462.43,1959.91 L1462.22,1959.3 L1461.99,1959.23 L1461.46,1958.85 L1461.07,1958.42 L1461.28,1958.17 L1461.28,1957.91 L1460.73,1954.88 L1457.83,1952.79 L1457.37,1951.94 L1457.69,1951 L1457.31,1950.68 L1456.7,1950.49 L1455.85,1949.33 L1454.1,1949.58 L1453.63,1948.97 L1453.45,1948.31 L1453.2,1948.18 L1452.89,1948.21 L1452.57,1948.04 L1451.79,1946.78 L1451.04,1946.17 L1449.03,1945 L1448.39,1944.12 L1447.84,1942.12 L1447.34,1941.3 L1446.64,1941 L1445.82,1941.1 L1445.16,1941.48 L1444.87,1942.03 L1444.61,1941.76 L1444.3,1941.52 L1444.14,1941.26 L1443.44,1941.76 L1443.12,1942.05 L1442.91,1942.35 L1442.81,1942.48 L1442.47,1942.69 L1442.13,1942.39 L1441.59,1941.55 L1439.93,1940.47 L1439.15,1939.7 L1439.04,1938.86 L1438.75,1938.67 L1438.3,1938.26 L1438,1938.08 L1437.96,1937.69 L1437.96,1936.8 L1437.08,1936.13 L1436.16,1936.2 L1436.04,1937.49 L1435.74,1937.52 L1435.34,1937.32 L1434.35,1937.57 L1434.13,1937.48 L1433.85,1937.1 L1433.24,1936.72 L1432.55,1936.41 L1432.06,1936.27 L1432.15,1936.04 L1432.14,1935.98 L1432.18,1935.93 L1432.36,1935.77 L1431.11,1935.08 L1430.68,1934.51 L1430.84,1933.77 L1431.4,1933.49 L1435.32,1934.3 L1435.87,1934.42 L1437.29,1934.98 L1438.05,1935.58 L1438.3,1934.58 L1438.14,1933.81 L1436.84,1931.59 L1436.75,1931.02 L1436.84,1930.08 L1435.52,1932.18 L1433.47,1932.82 L1428.5,1932.75 L1428.55,1933.21 L1429.49,1933.74 L1429.29,1935.58 L1430.13,1936.03 L1429.19,1935.83 L1428.48,1935.21 L1428.45,1934.29 L1428.15,1933.62 L1427.3,1933.25 L1426.92,1933.07 L1426.26,1932.43 L1425.68,1931.67 L1425.25,1930.75 L1425.1,1929.67 L1425.15,1927.65 L1424.83,1926.67 L1425.26,1926.18 L1425.76,1925.9 L1426.29,1925.88 L1426.85,1926.16 L1427.31,1926.04 L1427.45,1925.64 L1427.29,1925.09 L1426.9,1924.51 L1427.77,1923.87 L1428.22,1923.54 L1429.09,1922.48 L1429.24,1921.95 L1429.47,1921.14 L1429.29,1919.33 L1428.64,1917.69 L1427.74,1916.76 L1426.71,1915.9 L1424.84,1913.49 L1424.06,1912.08 L1423.48,1910.64 L1423.33,1909.57 L1423.7,1908.83 L1424.23,1908.32 L1424.5,1907.68 L1424.11,1906.52 L1422.8,1904.46 L1422.22,1903.27 L1422.08,1902.2 L1422.7,1899.67 L1423.94,1898.2 L1426.43,1897.33 L1429.14,1896.99 L1431.02,1897.03 L1429.7,1892.4 L1429.5,1890.79 L1429.51,1890.66 L1429.67,1888.74 L1430.24,1887.09 L1431.9,1884.11 L1430.75,1883.7 L1431.33,1882.61 L1436.66,1877.68 L1437.81,1877.16 L1437.64,1876.93 L1437.32,1876.39 L1437.14,1876.16 L1437.93,1875.66 L1440.6,1875.09 L1441.73,1875.27 L1443.05,1875.98 L1443.42,1876.38 L1443.49,1876.81 L1443.51,1877.26 L1443.67,1877.76 L1444.16,1878.41 L1445.22,1879.49 L1445.25,1879.55 L1445.42,1879.83 L1445.85,1879.35 L1446.15,1878.79 L1446.38,1878.21 L1446.58,1877.85 L1446.67,1877.7 L1447.16,1877.35 L1448.34,1877.14 L1448.63,1876.82 L1446.97,1874.39 L1445.21,1870.82 L1443.41,1868.95 L1443.06,1868.75 L1442.09,1869.13 L1442.06,1868.81 L1442.2,1868.27 L1442.18,1867.87 L1441.47,1867 L1441.12,1866.83 L1440.4,1866.73 L1439.93,1866.45 L1439.85,1865.93 L1440.06,1865.39 L1440.45,1865.03 L1441.24,1864.52 L1442.18,1863.19 L1442.79,1862.62 L1443.81,1862.39 L1444.51,1862.94 L1445.1,1863.79 L1445.78,1864.47 L1446.57,1864.64"
         id="path1045" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['ME'],polycolors['ME'],labels['ME'], x1, polybary['ME'], x2, polybary['ME'] )
    if 'ME' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M832.284,1387.31 L832.489,1388.42 L830.844,1387.5 L830.4,1387.54 L830.125,1388.27 L830.926,1388.69 L833.649,1388.98 L834.354,1389.22 L835.814,1390.02 L836.673,1389.85 L837.723,1388.96 L838.706,1387.8 L839.384,1386.83 L839.707,1386.08 L839.879,1385.51 L840.139,1385.15 L840.749,1385.03 L840.848,1385.19 L841.963,1386.05 L842.63,1387.56 L842.423,1388.59 L841.92,1389.56 L841.692,1390.9 L841.901,1391.53 L843.146,1393.11 L843.315,1393.67 L843.623,1395.07 L843.786,1395.49 L844.436,1395.91 L845.957,1395.88 L846.676,1396 L847.306,1396.92 L847.181,1399.14 L847.857,1399.36 L848.115,1399.32 L849.359,1399.09 L853.981,1399.58 L854.817,1399.33 L856.533,1398.28 L857.329,1397.95 L858.168,1398.1 L859.051,1398.74 L859.717,1399.73 L859.923,1400.96 L859.962,1402.39 L860.181,1402.6 L860.811,1403.21 L862.993,1404.01 L863.501,1404.42 L863.853,1404.88 L864.261,1405.21 L864.934,1405.24 L866.597,1405 L867.097,1405.08 L867.715,1405.5 L868.469,1406.68 L868.967,1407.15 L870.19,1406.92 L870.857,1407.01 L871.161,1407.82 L871.241,1408.13 L870.658,1408.65 L870.491,1409.35 L871.331,1410.36 L870.955,1410.59 L870.548,1410.72 L870.122,1410.76 L869.693,1410.69 L869.765,1411.15 L869.815,1411.34 L869.911,1411.6 L869.569,1412.44 L869.328,1412.87 L869.008,1413.26 L868.571,1412.84 L868.238,1413.02 L868.004,1413.62 L867.863,1414.44 L868.354,1415.2 L867.643,1416.68 L865.947,1419.08 L866.841,1418.89 L867.22,1418.92 L867.59,1419.17 L867.093,1419.91 L866.328,1421.32 L865.774,1422.06 L865.577,1422.2 L865.111,1422.34 L864.643,1422.49 L861.605,1425.51 L861.608,1426.91 L862.193,1427.9 L862.827,1428.41 L863.047,1428.58 L863.878,1429.04 L864.485,1429.26 L864.372,1429.99 L863.879,1431.4 L864.575,1431.86 L865,1431.97 L865.419,1431.91 L865.631,1431.63 L866.095,1431.01 L866.462,1430.76 L866.851,1430.94 L867.61,1431.88 L867.999,1432.19 L871.439,1432.3 L871.882,1432.46 L871.884,1432.46 L872.307,1432.63 L875.177,1432.88 L875.983,1432.77 L875.911,1432.89 L875.881,1433 L875.879,1433.11 L875.93,1433.23 L876.57,1433.61 L876.558,1434.21 L876.375,1434.87 L876.503,1435.4 L877.024,1435.49 L878.617,1435.14 L879.273,1435.24 L879.995,1436.04 L882.218,1439.86 L881.671,1440.17 L882.088,1441.09 L882.883,1441.57 L885.623,1441.78 L885.694,1442.15 L885.387,1442.7 L884.907,1443.22 L884.895,1443.45 L883.98,1443.93 L882.672,1444.91 L881.881,1446.02 L882.5,1446.89 L882.162,1447.36 L882.201,1447.64 L882.515,1447.79 L882.978,1447.85 L882.633,1448.38 L882.939,1448.69 L883.244,1449.42 L883.548,1449.69 L883.943,1449.73 L885.311,1449.39 L885.401,1449.3 L885.502,1449.28 L885.591,1449.31 L885.677,1449.41 L886.272,1449.73 L887.978,1450.21 L888.701,1450.29 L888.234,1450.7 L888.111,1451.2 L888.198,1451.77 L888.385,1452.41 L888.699,1452.99 L888.981,1453.06 L889.123,1453.28 L889.043,1454.32 L888.685,1455.78 L888.367,1456.6 L888.384,1457.35 L889.653,1459.83 L889.672,1460.23 L889.693,1460.71 L889.167,1461.22 L888.095,1461.36 L887.562,1461.23 L887.116,1461.03 L886.671,1460.97 L886.114,1461.26 L885.693,1461.72 L885.398,1462.31 L885.283,1462.93 L885.433,1463.5 L884.663,1464.13 L882.916,1464.53 L882.064,1464.88 L881.639,1465.46 L880.699,1466.06 L880.257,1466.58 L880.76,1467.2 L880.947,1467.92 L880.886,1468.63 L880.615,1469.2 L880.306,1469.38 L879.503,1469.41 L879.147,1469.6 L879.062,1469.8 L878.893,1470.5 L878.789,1470.8 L879.179,1471.15 L879.3,1471.54 L879.162,1471.95 L878.774,1472.36 L878.19,1472.06 L878.125,1471.57 L878.202,1470.98 L878.058,1470.39 L877.566,1469.87 L877.253,1469.89 L876.981,1470.16 L876.621,1470.35 L875.701,1470.45 L875.252,1470.36 L875.08,1470.26 L874.735,1470.07 L874.515,1469.6 L874.415,1469 L874.173,1468.7 L873.528,1469.16 L872.669,1469.21 L872.298,1469.76 L872.085,1470.61 L871.717,1471.49 L871.085,1472.08 L869.55,1472.73 L868.853,1473.3 L868.685,1473.69 L868.55,1474.7 L868.426,1475.12 L868.161,1475.46 L867.547,1475.94 L867.284,1476.31 L867.289,1477.03 L867.257,1477.62 L867.129,1478.11 L866.821,1478.4 L865.985,1478.72 L865.717,1479.04 L865.622,1479.9 L865.885,1480.26 L866.028,1480.62 L865.539,1481.45 L865.188,1481.74 L864.585,1481.85 L864.291,1482.02 L863.95,1482.42 L863.782,1482.74 L863.668,1483.05 L863.483,1483.44 L861.948,1485.78 L861.751,1486.36 L862.213,1486.62 L863.066,1487.35 L863.409,1487.94 L862.356,1487.77 L862.34,1488.14 L862.27,1488.32 L862.187,1488.46 L862.103,1488.68 L862.47,1489.07 L861.972,1489.36 L861.758,1489.95 L861.774,1490.7 L861.975,1491.49 L862.36,1492.23 L862.758,1492.45 L863.188,1492.55 L863.684,1492.9 L863.929,1493.31 L864.776,1495.25 L864.822,1495.55 L864.861,1496.42 L864.836,1496.57 L865.084,1496.79 L865.639,1497 L865.874,1497.15 L866.098,1497.2 L866.362,1497.13 L866.624,1497.16 L866.838,1497.55 L866.846,1497.69 L866.866,1498.03 L866.701,1498.28 L866.492,1498.49 L866.41,1498.83 L866.366,1499.24 L866.233,1499.43 L866.162,1499.65 L866.279,1500.11 L866.544,1500.34 L867.351,1500.48 L867.631,1500.74 L867.808,1501.64 L867.552,1502.17 L867.129,1502.53 L866.785,1502.89 L866.185,1504.3 L865.827,1504.91 L865.345,1505.31 L866.012,1505.73 L865.572,1506.68 L864.584,1507.63 L863.592,1508.03 L863.124,1507.89 L862.265,1507.29 L861.832,1507.2 L861.382,1507.41 L860.818,1508.07 L860.301,1508.26 L859.432,1508.06 L858.584,1507.57 L857.681,1507.31 L856.638,1507.77 L856.116,1508.4 L855.998,1508.84 L855.848,1509.08 L855.243,1509.13 L854.9,1509.01 L853.662,1508.34 L852.877,1508.64 L851.808,1509.38 L850.716,1509.93 L849.852,1509.65 L849.634,1509.08 L849.71,1508.52 L849.876,1507.94 L849.924,1507.29 L849.791,1506.71 L848.053,1503.05 L847.635,1502.62 L847.341,1502.32 L846.112,1501.78 L845.668,1501.47 L845.42,1501.42 L845.258,1501.57 L845.065,1501.91 L844.831,1502.21 L844.536,1502.28 L843.87,1502.1 L843.688,1501.85 L843.628,1501.27 L843.81,1500.39 L844.15,1500.2 L844.348,1499.99 L844.127,1499.03 L843.405,1498.02 L842.613,1497.13 L841.737,1497.09 L839.875,1497.48 L838.96,1497.15 L838.259,1496.55 L837.695,1495.93 L836.137,1493.32 L835.686,1492.82 L834.273,1492.31 L833.497,1491.8 L832.094,1490.21 L831.391,1489.75 L831.284,1489.75 L830.54,1489.74 L828.745,1490.23 L827.829,1490.32 L826.784,1490.09 L826.453,1489.59 L826.623,1487.26 L826.549,1486.88 L826.198,1486.2 L826.137,1485.88 L826.238,1485.5 L826.626,1485.11 L826.782,1484.79 L827.051,1484.04 L827.292,1483.53 L827.433,1483 L827.418,1482.18 L826.87,1480.99 L825.019,1480.07 L824.543,1479.16 L824.788,1478.18 L826.603,1474.43 L826.687,1473.77 L826.661,1473.24 L826.746,1472.68 L827.617,1471.07 L827.866,1471.03 L828.265,1471.53 L828.432,1470.45 L828.737,1469.17 L828.866,1468.08 L828.531,1467.57 L827.258,1467.74 L826.819,1467.52 L826.702,1466.58 L825.586,1467 L821.001,1471.03 L820.733,1471.5 L820.581,1472.11 L820.65,1472.49 L820.777,1472.81 L820.787,1473.24 L820.337,1475.21 L819.983,1476.18 L819.562,1476.87 L819.105,1477.13 L817.411,1477.27 L811.597,1479.63 L810.387,1479.75 L810.094,1479.65 L804.183,1477.52 L801.676,1477.69 L801.073,1477.73 L800.544,1477.53 L798.388,1476.13 L798.416,1476.14 L798.706,1476.19 L798.728,1475.27 L798.583,1474.3 L798.399,1474.2 L798.623,1473.48 L798.863,1473.03 L799.182,1472.69 L799.971,1472.07 L800.952,1471.48 L801.952,1471.07 L801.996,1470.66 L801.757,1470.15 L801.563,1469.54 L801.099,1466.79 L800.717,1466.56 L800.281,1466.59 L799.453,1466.89 L798.908,1466.56 L799.093,1465.44 L800.061,1463 L800.366,1461.88 L800.588,1461.39 L801.062,1460.81 L802.062,1460.15 L802.49,1459.73 L802.636,1459.09 L802.424,1458.66 L801.963,1458.18 L801.452,1457.8 L801.103,1457.65 L800.587,1457.82 L800.535,1458.22 L800.55,1458.65 L800.234,1458.92 L799.526,1458.51 L798.62,1456.07 L797.854,1455.27 L797.043,1454.71 L795.838,1453.13 L795.093,1452.7 L794.241,1452.72 L791.618,1453.58 L790.715,1453.51 L790.03,1453.11 L789.386,1452.57 L788.596,1452.08 L786.957,1451.74 L785.207,1451.72 L784.65,1451.88 L784.298,1452.22 L783.477,1453.38 L783.26,1453.88 L783.137,1454.06 L782.944,1454.17 L782.883,1454.11 L782.823,1454.14 L782.632,1454.53 L781.342,1453.09 L781.211,1450.51 L781.457,1447.6 L781.327,1445.16 L780.783,1443.93 L780.06,1443.1 L779.227,1442.59 L778.344,1442.32 L775.779,1442.08 L775.319,1441.67 L775.766,1441.01 L775.872,1440.51 L775.642,1440.16 L775.087,1439.94 L774.516,1439.93 L774.086,1440.16 L773.657,1440.52 L773.083,1440.89 L771.285,1441.37 L770.31,1441.4 L769.634,1441.14 L767.98,1440.07 L767.497,1439.61 L766.946,1438.75 L766.806,1438.03 L766.926,1436.03 L766.199,1432.43 L766.046,1430.59 L766.693,1429.61 L766.38,1428.08 L766.119,1427.47 L765.604,1427.25 L765.055,1427.12 L764.775,1426.72 L764.399,1425.49 L763.943,1424.46 L763.942,1424.46 L763.396,1423.21 L762.786,1422.52 L761.792,1422.11 L758.562,1422.79 L756.433,1423.24 L755.468,1423.73 L754.675,1424.43 L753.289,1426.38 L752.727,1426.73 L752.123,1426.54 L751.951,1426.44 L748.933,1424.68 L748.263,1423.99 L748.092,1423.59 L747.686,1422.28 L747.503,1421.84 L746.174,1420.29 L745.96,1419.8 L745.88,1419.36 L745.716,1418.98 L745.226,1418.66 L744.855,1418.58 L744.112,1418.61 L743.791,1418.56 L743.225,1418.38 L742.949,1418.22 L742.777,1417.85 L742.519,1417.02 L742.36,1416.8 L741.908,1416.51 L741.804,1416.28 L741.873,1416 L742.248,1415.34 L742.334,1414.94 L741.834,1412.13 L742.119,1411.58 L743.116,1410.77 L743.385,1410.42 L743.384,1409.16 L742.862,1408.17 L741.616,1406.46 L741.349,1405.22 L741.236,1402.43 L740.785,1400.94 L740.786,1400.94 L741.635,1400.44 L743.17,1400.1 L748.887,1396.61 L757.484,1392.61 L765.832,1388.18 L774.832,1386.16 L774.849,1386.15 L774.968,1388.59 L774.667,1390.12 L774.506,1391.57 L774.917,1393.18 L775.772,1394.31 L776.039,1394.47 L776.906,1394.97 L778.141,1395.23 L780.018,1395.1 L780.183,1394.52 L780.152,1393.72 L780.289,1393.01 L780.756,1392.61 L781.342,1392.52 L781.971,1392.53 L783.556,1392.32 L784.526,1392.35 L785.508,1392.6 L790.149,1394.6 L790.847,1395.15 L790.979,1395.99 L790.856,1396.68 L790.865,1397.33 L791.389,1398.08 L791.856,1398.33 L796.493,1398.66 L797.394,1398.45 L799.293,1397.2 L804.338,1395.15 L806.4,1393.78 L808.105,1391.8 L808.372,1391.2 L808.778,1389.77 L808.819,1389.51 L808.82,1389.51 L808.845,1389.53 L809.526,1390.28 L809.789,1390.92 L809.976,1392.08 L810.56,1392.92 L811.22,1393.6 L811.65,1394.25 L811.762,1394.65 L811.79,1395.04 L811.651,1395.47 L811.352,1395.89 L811.266,1396.01 L811.739,1395.81 L812.081,1395.49 L812.738,1394.33 L811.465,1393.66 L811.179,1393.39 L811.07,1392.82 L811.204,1391.47 L811.134,1390.81 L810.884,1390.29 L810.37,1389.54 L810.37,1389.54 L811.151,1389.63 L811.5,1389.67 L812.852,1390.55 L813.587,1390.87 L815.398,1391.13 L816.17,1390.79 L816.446,1389.73 L816.255,1389.1 L815.46,1387.98 L815.201,1387.4 L815.278,1386.77 L815.569,1386.3 L815.684,1385.86 L815.252,1385.31 L817.381,1384.27 L819.529,1383.66 L820.694,1383.56 L821.178,1383.69 L821.449,1384.23 L821.319,1385.27 L820.93,1386.02 L820.812,1386.63 L821.518,1387.22 L822.03,1387.31 L823.17,1387.18 L824.392,1387.59 L824.999,1387.59 L825.538,1387.43 L825.963,1387.16 L829.245,1383.67 L830.576,1383.1 L831.136,1383.2 L831.825,1383.52 L832.446,1384 L832.811,1384.55 L832.829,1385.5 L832.524,1386.38 L832.284,1387.31"
         id="path1047" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['BE'],polycolors['BE'],labels['BE'], x1, polybary['BE'], x2, polybary['BE'] )
    if 'BE' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1307.49,1677.74 L1308.08,1677.59 L1308.18,1678.64 L1308.52,1678.75 L1308.56,1678.76 L1308.79,1678.84 L1309.59,1678.82 L1310.27,1679.19 L1310.6,1680.35 L1310.29,1680.71 L1310.06,1680.98 L1309.27,1681.44 L1308.93,1682.09 L1309.13,1682.49 L1309.34,1682.92 L1310.18,1683.48 L1311.78,1684.26 L1313.58,1686.59 L1315.13,1687.7 L1315.98,1690.33 L1314.87,1689.59 L1314.42,1689.38 L1313.96,1689.35 L1313.04,1689.55 L1312.66,1689.54 L1312.28,1689.31 L1311.49,1688.6 L1311.09,1688.39 L1310.6,1688.39 L1310.06,1688.49 L1309.5,1688.46 L1308.97,1688.1 L1308.74,1688.59 L1308.72,1688.57 L1308.38,1688.15 L1308.23,1688.36 L1307.83,1688.92 L1307.02,1689.27 L1306.4,1689.74 L1305.09,1690.38 L1304.97,1690.54 L1303.95,1691.89 L1303.97,1692.42 L1303.98,1692.81 L1304.21,1693.64 L1304.86,1695.32 L1304.93,1695.82 L1304.93,1696.83 L1305.09,1697.34 L1305.38,1697.63 L1306.15,1698.06 L1306.38,1698.33 L1306.48,1698.87 L1306.5,1699.02 L1306.32,1699.23 L1305.33,1699.3 L1303.82,1699.8 L1303.46,1699.85 L1302.72,1699.74 L1302.07,1699.43 L1300.96,1698.53 L1300.53,1698.36 L1300,1698.51 L1299.67,1698.92 L1299.38,1699.43 L1299,1699.8 L1298.5,1699.94 L1298.25,1699.91 L1296.9,1699.74 L1296.97,1700.85 L1297.14,1701.74 L1297.2,1702.55 L1296.95,1703.43 L1296.85,1703.55 L1296.42,1704.11 L1295.62,1704.78 L1294.75,1705.3 L1294.55,1705.38 L1294.06,1705.56 L1292.64,1705.93 L1291.4,1706.75 L1289.95,1707.7 L1289.82,1707.72 L1289.76,1707.72 L1287.81,1707.94 L1287.13,1708.17 L1286.5,1708.53 L1286.36,1708.66 L1285.93,1709.04 L1285.17,1710.59 L1284.38,1711.19 L1281.25,1711.2 L1280.53,1711.2 L1279.63,1711.75 L1279.11,1712.56 L1278.97,1712.8 L1278.28,1714.39 L1277.78,1716.21 L1277.89,1717.77 L1278.6,1719.13 L1279.52,1719.97 L1279.89,1720.31 L1280.45,1720.59 L1281.7,1721 L1282.26,1721.35 L1282.69,1721.91 L1282.94,1722.23 L1282.8,1722.9 L1282.35,1723.7 L1282.1,1724.98 L1282.1,1724.99 L1282.25,1726.28 L1282.46,1728.18 L1282.47,1729.22 L1282.3,1730.02 L1282.05,1730.78 L1281.93,1731.54 L1282.18,1732.32 L1282.78,1733.53 L1282.87,1734.42 L1282.48,1735.08 L1281.62,1735.62 L1280.79,1735.93 L1279.1,1736.11 L1277.44,1735.96 L1276.29,1735.85 L1275.88,1736.09 L1275.12,1736.82 L1274.7,1737.01 L1274.19,1736.96 L1273.68,1736.83 L1273.19,1736.81 L1272.73,1737.07 L1272.56,1737.67 L1272.78,1738.31 L1272.88,1738.9 L1272.42,1739.34 L1272.39,1739.36 L1268.21,1741.09 L1267.05,1741.57 L1265.35,1742.68 L1265.01,1743.14 L1264.83,1743.76 L1264.79,1743.88 L1264.9,1744.14 L1265.23,1744.27 L1265.68,1744.66 L1266.17,1745.46 L1266.42,1745.75 L1266.83,1746.01 L1267.41,1746.18 L1267.58,1746.04 L1267.62,1745.79 L1267.81,1745.6 L1268.33,1745.42 L1268.49,1745.33 L1268.63,1745.44 L1269.39,1746.2 L1269.48,1746.29 L1270.28,1747.42 L1270.56,1747.95 L1269.65,1748.01 L1269.06,1748.27 L1268.53,1748.6 L1267.28,1749.1 L1267.05,1749.58 L1266.93,1750.08 L1266.64,1750.36 L1266.12,1750.68 L1266.2,1751.19 L1266.55,1751.84 L1266.84,1752.52 L1266.86,1752.57 L1267.24,1754.18 L1267.68,1755.5 L1268.38,1756.52 L1270.69,1757.84 L1270.3,1758.24 L1269.59,1759.4 L1269.23,1759.78 L1268.75,1759.96 L1267.32,1760.03 L1264.93,1761.11 L1263.15,1761.91 L1261.16,1761.77 L1257.76,1759.26 L1257.3,1758.91 L1255.16,1759 L1254.69,1758.63 L1254.19,1758.44 L1253.79,1758.41 L1253.14,1758.37 L1252.34,1757.71 L1251.3,1757.13 L1250.5,1757.21 L1250.43,1758.55 L1249.67,1760.06 L1247.84,1760.85 L1245.98,1760.59 L1245.18,1758.89 L1245.18,1758.89 L1245.17,1758.88 L1245.17,1758.88 L1240.99,1757.35 L1240.07,1756.66 L1240.04,1756.34 L1240.13,1755.23 L1240.01,1754.75 L1239.74,1754.42 L1239.72,1754.42 L1239.44,1754.27 L1239.12,1754.16 L1237.54,1753.06 L1236.94,1752.5 L1236.49,1751.74 L1236.4,1750.9 L1236.38,1749.75 L1236.23,1748.8 L1235.74,1748.56 L1235.23,1748.81 L1235.01,1748.92 L1234.69,1749.32 L1234.22,1750.61 L1233.79,1751.31 L1232.76,1752.61 L1232.39,1753.24 L1232.16,1754.05 L1232.14,1754.32 L1232.04,1755.74 L1231.86,1756.63 L1231.34,1757.72 L1229.69,1759.08 L1228.92,1759.89 L1227.29,1760.95 L1225.27,1761.25 L1223.21,1760.98 L1221.42,1760.32 L1220.45,1760.13 L1219.34,1760.52 L1217.3,1761.62 L1216.18,1761.89 L1216.07,1761.89 L1216.05,1761.89 L1215,1761.89 L1213.82,1761.56 L1212.68,1760.81 L1212.68,1760.81 L1212.06,1760.25 L1211.42,1759.96 L1210.24,1759.77 L1209.55,1759.66 L1209.25,1759.88 L1209.12,1760.33 L1209.16,1761 L1210.14,1762.07 L1209.62,1762.87 L1207.6,1763.85 L1207,1764.51 L1206.64,1765.13 L1206.19,1765.5 L1203.82,1765.29 L1203.12,1765.1 L1202.5,1764.54 L1200.36,1763.33 L1196,1763.75 L1194.64,1763.88 L1192.82,1762.1 L1192.9,1761.83 L1192.76,1761.31 L1193.04,1760.65 L1193.02,1760.26 L1192.58,1760.2 L1192.24,1759.97 L1191.96,1759.57 L1191.76,1759 L1192.69,1759.17 L1196.42,1758.43 L1199.81,1757.75 L1199.79,1757.37 L1199.3,1756.72 L1198.6,1756.04 L1197.88,1755.33 L1200.04,1755.03 L1201.81,1755.88 L1203.88,1755.56 L1204.71,1754.42 L1205.49,1753.34 L1205.79,1752.51 L1205.74,1752.39 L1205.71,1752.34 L1205.32,1752.33 L1204.64,1751.99 L1204.14,1751.5 L1200.37,1745.82 L1197.27,1744.6 L1196.84,1744.3 L1195.04,1743.04 L1194.3,1742.85 L1192.79,1742.74 L1191.54,1742.14 L1191.2,1741.54 L1190.81,1740.86 L1190.85,1738.71 L1192.03,1735.67 L1192.38,1734.77 L1192.63,1733.54 L1192.89,1732.25 L1192.1,1731.1 L1191.88,1731.16 L1190.63,1731.52 L1189.25,1732.26 L1187.95,1732.42 L1186.7,1731.12 L1186.68,1731.05 L1186.65,1730.99 L1186.61,1730.93 L1186.57,1730.88 L1186.36,1730.73 L1185.76,1730.09 L1186.4,1729.46 L1186.62,1729.3 L1186.94,1728.84 L1187.04,1728.4 L1186.92,1728 L1186.55,1727.65 L1187.48,1726.28 L1188.22,1725.67 L1192.15,1722.37 L1193.29,1720.02 L1193.09,1719.42 L1192.9,1718.87 L1192.46,1718.81 L1191.88,1718.72 L1190.6,1718.92 L1189.49,1718.8 L1188.13,1717.57 L1187.28,1717.04 L1186.06,1716.85 L1185.44,1716.54 L1184.14,1716.51 L1183.45,1716.31 L1184.14,1717.38 L1182.96,1717.62 L1182.57,1717.09 L1182.38,1716.24 L1181.77,1715.48 L1181.76,1715.47 L1181.76,1715.46 L1181.43,1713.91 L1181.15,1713.18 L1180.79,1712.57 L1181.89,1711.82 L1182.06,1711.49 L1182.63,1710.4 L1183.15,1709.05 L1183.6,1708.47 L1184.17,1708.38 L1184.69,1708.09 L1185.69,1707.29 L1187.65,1706.12 L1188.65,1704.97 L1189.51,1703.64 L1190.53,1702.59 L1191.98,1702.33 L1193.05,1702.31 L1193.84,1701.84 L1194.16,1701.07 L1194.26,1700.84 L1194.25,1699.24 L1194.47,1698.89 L1194.62,1698.5 L1194.68,1698.06 L1194.65,1697.57 L1195.3,1697.59 L1198.19,1698.14 L1198.77,1698.1 L1201.55,1697.49 L1202.86,1697.64 L1206.91,1699.3 L1207.62,1699.35 L1208.31,1699.18 L1209.06,1699.01 L1209.82,1698.98 L1210.53,1699.15 L1211.18,1699.44 L1213.72,1701.39 L1214.19,1701.49 L1214.28,1701.51 L1218.27,1701.4 L1223.46,1701.25 L1224.89,1700.84 L1225.36,1700.92 L1225.58,1701.2 L1225.74,1701.54 L1226,1701.82 L1226.76,1702.11 L1227.37,1702.35 L1228.1,1702.42 L1229.57,1701.98 L1230.2,1702.75 L1230.75,1703.79 L1231.36,1704.35 L1232.09,1703.98 L1232.25,1703.48 L1232.43,1702.93 L1232.72,1701.7 L1233.31,1700.74 L1233.72,1700.55 L1234.65,1700.4 L1235.08,1700.25 L1235.51,1699.93 L1236.32,1698.97 L1236.51,1698.91 L1236.77,1698.94 L1237.09,1698.98 L1237.35,1698.19 L1237.82,1697.55 L1238.01,1697.11 L1238.25,1696.8 L1238.95,1696.43 L1239.35,1696.68 L1240.44,1696.42 L1241.37,1695.93 L1241.59,1695.7 L1241.89,1694.88 L1242.28,1693.82 L1242.54,1692.84 L1242.81,1691.81 L1243.23,1690.73 L1243.89,1689.63 L1244.41,1689.37 L1245.02,1689.4 L1245.75,1689.26 L1245.91,1689.23 L1246.8,1688.57 L1247.39,1688.15 L1248.01,1688.23 L1248.91,1689.36 L1250.37,1686.97 L1252.79,1686.01 L1253.82,1686.05 L1254.68,1686.08 L1257.58,1686.19 L1258.98,1686.24 L1264.45,1685.56 L1266.82,1685.27 L1268.12,1685.65 L1268.93,1686.26 L1269.15,1686.42 L1270.16,1686.96 L1271.43,1686.64 L1272.19,1685.95 L1272.29,1685.68 L1272.42,1685.31 L1272.49,1684.72 L1272.51,1684.6 L1272.86,1683.68 L1273.48,1683.13 L1274.39,1682.82 L1275.3,1682.74 L1275.96,1682.87 L1276.5,1682.67 L1276.71,1682.49 L1276.92,1682.3 L1277.14,1681.74 L1277.08,1680.94 L1277.14,1679.99 L1277.86,1680.36 L1281.19,1680.59 L1282.32,1680.44 L1283.54,1679.83 L1285.13,1679.04 L1286.32,1678.84 L1287.53,1678.98 L1290.55,1680.22 L1292.32,1680.39 L1292.81,1680.68 L1293.69,1681.62 L1293.7,1681.63 L1293.7,1681.62 L1293.49,1680.2 L1292.93,1679.27 L1291.98,1678.76 L1291.37,1677.28 L1291.25,1676.19 L1291.23,1675.28 L1291.23,1675.09 L1291.37,1673.13 L1291.3,1672.69 L1291.1,1672.57 L1291.05,1672.44 L1291.43,1671.97 L1291.67,1671.8 L1293.39,1671.23 L1293.54,1671.18 L1294.39,1670.57 L1296.07,1669.39 L1297.64,1669.64 L1297.87,1669.67 L1299.74,1669.35 L1303.67,1668.68 L1304.13,1668.89 L1304.84,1669.64 L1305.05,1669.86 L1305.44,1670.03 L1306.07,1670.01 L1306.29,1670.31 L1306.24,1670.87 L1306.06,1671.65 L1305.82,1672.38 L1305.68,1672.67 L1305.35,1673.36 L1305.27,1673.84 L1305.24,1674.07 L1305.35,1674.3 L1306.09,1676.04 L1306.63,1676.62 L1307.05,1677.31 L1307.49,1677.74"
         id="path1049" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['SI'],polycolors['SI'],labels['SI'], x1, polybary['SI'], x2, polybary['SI'] )
    if 'SI' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M875.701,1470.45 L876.621,1470.35 L876.981,1470.16 L877.253,1469.89 L877.566,1469.87 L878.058,1470.39 L878.202,1470.98 L878.125,1471.57 L878.19,1472.06 L878.774,1472.36 L878.448,1473.25 L878.208,1474.1 L877.863,1475.84 L877.931,1475.89 L878.173,1475.88 L878.264,1475.9 L878.103,1476.1 L877.691,1476.81 L878.114,1477.08 L878.349,1477.61 L878.531,1478.93 L878.382,1479.61 L878.53,1480.13 L879.17,1481.23 L879.691,1483.01 L879.984,1483.58 L880.461,1482.74 L881.095,1483.9 L882.257,1487.16 L882.855,1487.83 L884.193,1488.51 L884.762,1488.92 L884.937,1489.28 L885.078,1489.72 L885.288,1490.38 L885.32,1490.59 L885.644,1490.63 L886.268,1490.3 L886.622,1490.32 L887.286,1490.79 L888.393,1491.95 L889.057,1492.42 L889.753,1492.65 L891.692,1492.75 L892.92,1493.17 L893.079,1493.19 L893.285,1493.44 L893.273,1493.42 L893.181,1493.51 L893.152,1494.03 L893.092,1494.07 L892.835,1495.42 L892.81,1495.86 L892.906,1496.15 L892.878,1496.39 L892.468,1496.72 L892.413,1497.27 L892.481,1497.79 L892.652,1498.25 L892.923,1498.65 L892.578,1498.49 L892.385,1498.58 L892.232,1498.87 L892.494,1499.21 L892.567,1499.34 L889.496,1501.13 L888.825,1501.98 L888.886,1502.19 L889.138,1502.38 L889.374,1502.64 L889.391,1503.05 L889.269,1503.33 L888.941,1503.76 L888.489,1504.59 L887.368,1505.84 L886.709,1506.77 L886.444,1507.45 L886.347,1509.43 L886.239,1510.03 L885.874,1512.03 L885.908,1513.93 L885.882,1514.39 L885.073,1514.26 L884.021,1513.61 L882.154,1512.06 L882.085,1512.01 L880.993,1511.5 L879.843,1511.32 L877.492,1511.46 L877.728,1512.05 L876.574,1512.09 L876.278,1512.19 L876.043,1512.56 L876.027,1512.96 L875.917,1513.29 L875.415,1513.46 L874.988,1514.03 L874.378,1514.23 L870.895,1514.25 L870.175,1514.44 L869.713,1512.62 L868.989,1511.81 L866.442,1511.05 L865.447,1510.51 L864.774,1509.81 L863.592,1508.03 L864.584,1507.63 L865.572,1506.68 L866.012,1505.73 L865.345,1505.31 L865.827,1504.91 L866.185,1504.3 L866.785,1502.89 L867.129,1502.53 L867.552,1502.17 L867.808,1501.64 L867.631,1500.74 L867.351,1500.48 L866.544,1500.34 L866.279,1500.11 L866.162,1499.65 L866.233,1499.43 L866.366,1499.24 L866.41,1498.83 L866.492,1498.49 L866.701,1498.28 L866.866,1498.03 L866.846,1497.69 L866.838,1497.55 L866.624,1497.16 L866.362,1497.13 L866.098,1497.2 L865.874,1497.15 L865.639,1497 L865.084,1496.79 L864.836,1496.57 L864.861,1496.42 L864.822,1495.55 L864.776,1495.25 L863.929,1493.31 L863.684,1492.9 L863.188,1492.55 L862.758,1492.45 L862.36,1492.23 L861.975,1491.49 L861.774,1490.7 L861.758,1489.95 L861.972,1489.36 L862.47,1489.07 L862.103,1488.68 L862.187,1488.46 L862.27,1488.32 L862.34,1488.14 L862.356,1487.77 L863.409,1487.94 L863.066,1487.35 L862.213,1486.62 L861.751,1486.36 L861.948,1485.78 L863.483,1483.44 L863.668,1483.05 L863.782,1482.74 L863.95,1482.42 L864.291,1482.02 L864.585,1481.85 L865.188,1481.74 L865.539,1481.45 L866.028,1480.62 L865.885,1480.26 L865.622,1479.9 L865.717,1479.04 L865.985,1478.72 L866.821,1478.4 L867.129,1478.11 L867.257,1477.62 L867.289,1477.03 L867.284,1476.31 L867.547,1475.94 L868.161,1475.46 L868.426,1475.12 L868.55,1474.7 L868.685,1473.69 L868.853,1473.3 L869.55,1472.73 L871.085,1472.08 L871.717,1471.49 L872.085,1470.61 L872.298,1469.76 L872.669,1469.21 L873.528,1469.16 L874.173,1468.7 L874.415,1469 L874.515,1469.6 L874.735,1470.07 L875.08,1470.26 L875.252,1470.36 L875.701,1470.45"
         id="path1051" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['LU'],polycolors['LU'],labels['LU'], x1, polybary['LU'], x2, polybary['LU'] )
    if 'LU' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M429.223,1080.79 L430.744,1082.01 L431.982,1082.56 L433.386,1083.62 L433.971,1084.34 L434.823,1084.18 L435.078,1085.23 L434.66,1086.46 L433.475,1086.83 L432.454,1086.79 L431.17,1087.02 L429.894,1087.48 L428.421,1088.38 L427.141,1088.71 L426.127,1088.77 L425.554,1089.1 L423.502,1090.87 L422.708,1091.75 L421.987,1092.76 L421.851,1092.99 L421.114,1093.08 L420.712,1093.39 L420.338,1093.75 L419.975,1094 L417.668,1093.8 L417.171,1094.03 L416.72,1094.35 L415.35,1095.65 L415.36,1096.22 L415.242,1096.3 L415.04,1096.32 L414.792,1096.71 L414.598,1098.32 L414.428,1098.82 L414.078,1099.17 L414.009,1099.23 L413.245,1099.35 L412.863,1099.66 L412.552,1100.35 L412.309,1101.93 L412.149,1102.57 L411.46,1103.79 L410.638,1104.73 L407.301,1107.28 L406.871,1108.05 L406.418,1109.55 L406.419,1110.03 L406.554,1110.29 L406.52,1110.42 L406.011,1110.51 L405.649,1110.39 L405.086,1109.86 L403.955,1109.93 L402.84,1109.29 L402.122,1109.5 L400.597,1110.41 L399.219,1110.8 L398.01,1110.59 L397.011,1109.68 L396.238,1108.55 L395.764,1108.4 L394.255,1109.31 L392.297,1109.74 L392.745,1110.38 L392.928,1110.55 L392.422,1111.4 L392.547,1112.1 L393.028,1112.63 L393.59,1112.96 L393.835,1113.22 L393.99,1114.09 L394.188,1114.46 L394.478,1114.54 L395.128,1114.26 L395.414,1114.25 L395.825,1114.66 L396.686,1115.96 L397.178,1116.32 L397.841,1116.17 L398.49,1115.91 L398.838,1116.1 L398.627,1117.31 L398.626,1117.32 L398.62,1117.35 L397.044,1117.47 L392.687,1119.95 L391.204,1120.14 L389.84,1119.73 L387.343,1118.42 L386.386,1119.07 L385.398,1119.95 L384.735,1120.73 L384.488,1120.94 L383.986,1121.22 L382.988,1121.51 L380.975,1121.59 L380.238,1121.78 L380.022,1121.84 L380.357,1122.38 L380.342,1122.87 L380.287,1123.38 L380.496,1123.98 L381.092,1125.14 L382.033,1127.66 L382.633,1128.74 L383.395,1129.45 L384.421,1129.72 L384.674,1131.82 L384.908,1132.81 L385.338,1133.65 L385.615,1133.85 L385.867,1134.04 L387.196,1134.6 L387.613,1135.13 L387.597,1136.16 L387.048,1138.89 L387.276,1139.95 L387.855,1140.56 L389.731,1141.32 L391.874,1141.71 L392.475,1141.98 L394.54,1144.57 L394.69,1145.02 L394.613,1145.57 L394.599,1146.11 L394.94,1146.52 L398.503,1148.37 L399.307,1148.4 L401.062,1147.59 L401.93,1147.89 L401.402,1148.32 L401.504,1148.44 L401.875,1148.47 L402.148,1148.63 L402.275,1149.23 L402.279,1149.51 L402.426,1149.59 L402.982,1149.59 L404.56,1150.61 L404.943,1150.66 L405.168,1150.69 L404.78,1148.78 L404.881,1148.37 L405.214,1148.12 L406.455,1147.74 L406.539,1147.92 L406.327,1148.34 L406.143,1148.87 L405.973,1149.57 L405.799,1150.12 L405.867,1150.39 L406.418,1150.26 L406.841,1149.96 L407.229,1149.49 L407.476,1148.9 L407.467,1148.25 L408.251,1148.06 L408.286,1147.73 L408.026,1147.36 L407.915,1147.08 L408.265,1146.52 L408.592,1146.18 L408.957,1145.95 L409.439,1145.73 L411.512,1145.68 L412.393,1145.37 L412.412,1144.31 L413.075,1143.73 L412.077,1142.35 L412.499,1141.55 L411.457,1140.66 L411.533,1139.99 L411.714,1139.98 L411.958,1140.14 L412.213,1139.93 L412.256,1139.96 L412.67,1139.46 L412.666,1139.42 L412.596,1139.3 L412.537,1138.96 L412.541,1138.56 L412.665,1138.29 L412.786,1138.25 L412.94,1138.21 L413.533,1138.48 L413.825,1138.54 L415.206,1137.93 L417.595,1135.6 L418.88,1134.83 L420.015,1135 L421.058,1135.9 L422.803,1138.28 L422.95,1138.76 L423.022,1139.87 L423.265,1140.11 L423.68,1140.25 L423.841,1140.54 L423.955,1141.58 L424.091,1142.82 L423.939,1143.72 L423.506,1144.27 L422.804,1144.45 L423.904,1145.41 L424.134,1145.85 L424.172,1146.41 L424.152,1147.03 L424.21,1147.69 L424.564,1148.81 L425.063,1149.79 L425.666,1150.62 L426.319,1151.3 L426.734,1151.44 L427.095,1151.2 L427.459,1150.87 L427.894,1150.73 L428.303,1150.88 L428.631,1151.19 L429.222,1152.02 L429.594,1152.76 L429.8,1153.17 L429.746,1154.31 L429.146,1155.35 L428.096,1156.21 L428.309,1157 L428.098,1157.68 L427.204,1158.99 L427.643,1159.56 L428.314,1160.97 L428.754,1161.37 L428.978,1161.29 L429.26,1161.02 L429.501,1160.79 L429.737,1160.62 L430.033,1160.63 L430.524,1160.99 L430.797,1161.09 L434.12,1160.66 L435.118,1160.85 L435.117,1160.96 L435.198,1161.25 L435.347,1161.57 L435.537,1161.79 L435.793,1161.84 L436.791,1161.72 L437.666,1161.77 L437.943,1161.61 L438.213,1161.13 L438.306,1160.62 L438.355,1160.13 L438.488,1159.76 L439.153,1159.06 L439.433,1159.17 L439.609,1159.66 L439.96,1160.07 L440.462,1160.07 L441.031,1159.92 L441.581,1159.97 L442.005,1160.54 L442.004,1160.54 L443.738,1163.27 L444.23,1163.69 L444.438,1163.94 L444.753,1164.93 L444.927,1165.17 L445.673,1165.42 L445.955,1165.6 L446.173,1165.86 L446.629,1167.03 L446.323,1167.75 L444.835,1168.89 L444.162,1168.99 L443.349,1168.47 L442.63,1167.73 L442.237,1167.2 L441.703,1167.18 L439.453,1165.5 L437.622,1164.81 L437.515,1165.27 L437.876,1165.67 L437.525,1166.19 L436.893,1166.9 L436.417,1167.86 L435.804,1169.62 L435.523,1170.71 L435.488,1171.84 L436.027,1173.12 L437.084,1174.03 L439.659,1175.01 L439.553,1175.46 L438.915,1176.46 L438.629,1177.54 L438.772,1178.56 L439.422,1179.39 L438.489,1180.06 L437.925,1181.66 L437.651,1183.28 L437.19,1186.01 L437.158,1186.96 L437.193,1187.88 L437.302,1188.59 L437.575,1189 L438.033,1189.5 L438.366,1189.86 L438.647,1191.5 L439.478,1192.41 L440.442,1193.19 L441.047,1193.88 L441.207,1194.51 L441.279,1195.38 L441.157,1196.32 L440.725,1197.11 L439.884,1197.96 L439.624,1198.32 L439.428,1198.79 L439.044,1199.25 L438.368,1200.27 L437.939,1200.65 L437.808,1199.97 L437.197,1199.72 L436.403,1199.81 L435.71,1200.14 L437.781,1201.81 L437.801,1202.52 L437.767,1203.27 L438.005,1203.79 L437.832,1203.95 L437.493,1204.35 L437.318,1204.51 L437.84,1204.95 L439.316,1205.38 L439.88,1205.97 L439.223,1206.98 L438.758,1206.91 L438.362,1206.36 L437.91,1205.97 L437.795,1205.79 L437.479,1205.09 L437.367,1204.92 L437.005,1204.92 L436.3,1205.08 L436.022,1205.09 L433.517,1206.28 L433.26,1206.93 L433.39,1207.5 L434.032,1208.6 L434.5,1209.84 L434.883,1210.37 L435.85,1210.85 L436.16,1211.25 L436.417,1211.38 L436.809,1211.7 L436.698,1212.19 L436.395,1212.65 L436.217,1212.88 L435.869,1214.22 L435.739,1215.07 L435.739,1216.03 L435.968,1216.76 L435.973,1216.78 L436.281,1217.4 L436.346,1218.12 L435.836,1219.17 L436.162,1220.01 L436.815,1222.94 L436.7,1223.16 L436.67,1223.19 L436.634,1223.18 L436.483,1223.27 L436.426,1224.31 L436.041,1225.79 L435.505,1227.2 L435.003,1228.05 L434.626,1229 L434.894,1230.06 L436.033,1232.18 L436.042,1232.66 L435.519,1232.76 L435.262,1232.99 L434.635,1234.05 L434.388,1234.27 L433.887,1234.61 L431.977,1237.24 L432.15,1237.76 L430.229,1238.92 L429.666,1239.4 L429.057,1240.19 L427.519,1242.91 L427.432,1244.32 L427.251,1244.57 L426.263,1245.95 L423.824,1248.66 L423.369,1249.76 L423.201,1250.69 L423.188,1252.52 L423.072,1253.48 L422.379,1256.07 L421.648,1257.03 L418.541,1260.27 L417.683,1260.94 L416.981,1261.15 L416.219,1261.83 L415.007,1263.2 L414.442,1264.07 L414.161,1265.19 L413.984,1266.27 L413.731,1267.07 L413.58,1266.55 L413.352,1266.19 L413.044,1265.99 L410.845,1265.67 L410.13,1265.24 L410.603,1264.19 L410.237,1264.23 L409.873,1264.34 L409.519,1264.51 L409.164,1264.73 L409.076,1265.11 L409.586,1265.46 L409.994,1266.05 L410.262,1266.82 L410.355,1267.69 L409.971,1268.28 L411.074,1268.81 L411.301,1269.39 L411.353,1270.08 L411.935,1269.65 L413.172,1268.34 L413.085,1268.73 L412.166,1269.86 L412.587,1271.44 L414.059,1273.83 L412.379,1276.25 L411.746,1276.82 L411.285,1277.11 L410.991,1277.16 L410.651,1277 L410.256,1276.52 L410.409,1276.22 L410.748,1276 L411.257,1275.32 L411.269,1275.04 L410.844,1274.89 L410.592,1274.95 L410.158,1275.35 L409.615,1275.64 L409.521,1275.75 L409.398,1275.78 L407.771,1275.42 L407.859,1275.04 L408.622,1275.22 L408.217,1274.65 L407.831,1274.57 L407.092,1274.86 L405.221,1274.83 L403.425,1275.4 L402.795,1275.17 L402.07,1273.89 L401.474,1273.36 L398.908,1271.87 L396.178,1271.65 L395.5,1271.31 L395.608,1270.85 L396.23,1270.73 L396.895,1270.32 L397.452,1269.64 L397.744,1268.71 L397.149,1269.24 L396.367,1269.73 L395.559,1269.87 L394.908,1269.36 L394.564,1269.68 L394.489,1270.18 L394.498,1270.39 L394.457,1270.58 L394.251,1271.01 L394.723,1270.82 L394.893,1270.68 L395.123,1270.73 L394.791,1271.6 L394.592,1271.92 L394.228,1272.25 L394.477,1272.33 L394.738,1272.37 L394.631,1272.82 L394.501,1272.82 L394.247,1272.73 L394.121,1272.7 L394.302,1273.19 L391.212,1274.21 L390.74,1274.52 L390.135,1275.19 L389.509,1275.58 L388.971,1275.04 L389.492,1274.38 L390.031,1274.08 L390.577,1273.88 L391.105,1273.53 L391.256,1272.92 L391.2,1270.29 L391.086,1269.81 L390.852,1269.15 L390.538,1268.55 L390.197,1268.24 L389.873,1267.74 L389.785,1265.7 L389.468,1264.94 L389.406,1265.68 L389.348,1266.36 L389.32,1266.68 L389.2,1267.3 L389.246,1267.62 L389.521,1268.76 L389.505,1269.39 L388.668,1270.79 L388.614,1271.41 L389.542,1271.61 L389.443,1272.03 L388.734,1272.38 L387.302,1273.63 L386.534,1273.98 L385.658,1273.98 L383.963,1273.62 L383.114,1273.64 L383.499,1273.01 L384.896,1271.82 L384.163,1271.25 L383.717,1271.07 L383.185,1271.01 L383.094,1271.39 L383.265,1271.62 L383.324,1271.83 L383.355,1272.06 L383.423,1272.35 L383.051,1272.19 L382.639,1271.79 L382.357,1271.65 L381.86,1271.7 L381.524,1271.99 L381.251,1272.37 L380.932,1272.63 L380.108,1272.8 L379.415,1272.63 L378.09,1271.94 L370.761,1270.97 L367.854,1271.75 L366.392,1271.73 L366.176,1271.83 L365.959,1272.11 L365.688,1272.36 L365.298,1272.34 L365.096,1272.04 L365.045,1271.57 L365.035,1271.14 L364.96,1270.93 L364.266,1270.97 L363.425,1271.33 L362.776,1272.02 L362.669,1273.05 L363.146,1272.68 L363.743,1272.74 L364.377,1273.09 L365.362,1273.88 L365.621,1274 L365.617,1274.28 L365.216,1274.97 L363.822,1275.55 L363.225,1277.37 L362.989,1277.76 L362.647,1277.81 L362.256,1277.79 L361.886,1277.96 L361.074,1278.22 L359.148,1277.51 L358.272,1277.71 L357.75,1278.48 L357.603,1279.11 L357.307,1279.37 L353.482,1277.83 L353.43,1277.55 L353.509,1276.98 L353.482,1276.38 L353.123,1276.01 L353.005,1276.18 L352.929,1276.28 L352.524,1277.72 L352.3,1278.22 L351.605,1278.91 L350.908,1279.41 L350.178,1279.64 L349.379,1279.49 L350.225,1280.81 L350.631,1281.1 L350.072,1281.61 L349.13,1281.77 L347.491,1281.72 L345.31,1282.1 L344.566,1282.7 L345.063,1283.71 L344.356,1283.92 L341.381,1283.8 L339.812,1284 L337.397,1283.63 L336.39,1283.27 L335.955,1283.19 L335.591,1283.05 L335.484,1282.79 L335.692,1282.38 L336.024,1282.14 L336.407,1282.01 L336.775,1282 L336.747,1281.75 L336.722,1281.14 L337.288,1281.17 L337.866,1281.09 L338.461,1280.85 L339.082,1280.42 L338.883,1280.23 L338.726,1280.03 L338.522,1279.85 L338.165,1279.74 L338.647,1279.02 L338.889,1278.81 L339.25,1278.69 L339.203,1278.11 L338.99,1277.91 L338.649,1278.03 L338.207,1278.42 L337.806,1277.95 L337.092,1277.59 L335.545,1277.24 L335.641,1276.87 L335.402,1276.47 L334.723,1276.46 L333.95,1276.61 L333.204,1276.64 L331.694,1276.15 L330.91,1276.06 L330.031,1276.3 L331.151,1277.09 L332.655,1277.73 L333.402,1278.68 L332.233,1280.42 L333.669,1280.65 L334.066,1280.89 L333.969,1281.27 L333.714,1281.21 L333.598,1281.66 L333.857,1282.03 L333.989,1282.17 L333.417,1282.16 L333.137,1282.23 L332.849,1282.36 L334.151,1282.69 L333.243,1283.54 L333.066,1283.74 L333.032,1284.09 L333.111,1284.43 L333.107,1284.79 L332.821,1285.23 L331.227,1286.31 L330.653,1286.84 L329.505,1287.14 L328.012,1287.13 L327.457,1287.35 L326.788,1287.92 L326.732,1287.17 L326.338,1286.53 L325.797,1286.13 L325.307,1286.16 L325.745,1286.74 L325.752,1287.5 L325.4,1288.12 L324.751,1288.31 L324.801,1287.01 L324.119,1286.35 L322.403,1285.88 L322.305,1286.26 L322.623,1286.59 L323.363,1286.87 L323.72,1287.12 L324.171,1286.81 L324.416,1286.8 L324.466,1287.31 L324.298,1287.64 L323.97,1287.85 L323.61,1287.94 L323.358,1287.95 L322.41,1289.24 L322.209,1289.6 L322.073,1290.28 L322.009,1291.24 L321.896,1291.96 L321.532,1291.4 L321.314,1290.04 L320.9,1289.48 L320.299,1289.26 L318.327,1289.29 L313.972,1287.65 L314.354,1288.39 L314.919,1288.79 L316.287,1289.15 L316.169,1289.61 L315.877,1289.72 L315.575,1289.9 L316.242,1290.48 L315.735,1290.74 L315.38,1291.13 L315.197,1291.67 L315.189,1292.42 L313.819,1291.85 L313.424,1292.02 L313.211,1291.72 L313.054,1291.24 L312.82,1290.9 L311.438,1290.54 L311.068,1290.33 L310.824,1290.09 L310.592,1290 L310.262,1290.23 L309.584,1289.68 L308.87,1290.23 L308.2,1291.18 L307.658,1291.79 L306.214,1291.89 L305.969,1291.96 L305.639,1292.57 L305.345,1292.33 L305.144,1291.8 L305.114,1291.51 L304.894,1291.32 L304.296,1290.56 L303.912,1290.3 L303.461,1290.24 L302.817,1290.41 L300.629,1290.31 L300.194,1290.01 L299.493,1289.25 L299.055,1288.99 L298.935,1289.44 L299.077,1289.58 L299.322,1289.95 L298.837,1290.57 L298.231,1290.97 L297.576,1291.14 L296.972,1291.1 L296.067,1290.58 L295.742,1290.6 L295.689,1291.23 L295.588,1291.61 L295.043,1291.49 L294.683,1291.8 L294.366,1292.25 L293.963,1292.54 L293.539,1292.48 L293.247,1292.22 L292.929,1292.01 L292.426,1292.13 L292.515,1291.71 L292.513,1291.4 L292.483,1291.12 L292.495,1290.77 L291.293,1291.15 L291.063,1291.66 L290.826,1291.69 L290.537,1291.59 L290.16,1291.73 L289.613,1292.03 L289.042,1292.07 L287.829,1291.76 L288.341,1291.58 L288.773,1291.35 L289.172,1291 L289.58,1290.46 L288.933,1290.7 L288.438,1290.78 L288.345,1290.53 L288.911,1289.79 L289.456,1289.37 L290.554,1288.98 L291.141,1288.66 L291.067,1288.49 L291.033,1288.44 L291.016,1288.36 L290.982,1288.13 L290.583,1288.37 L289.186,1288.78 L288.945,1289.08 L288.479,1289.41 L287.513,1289.89 L287.23,1289.15 L287.436,1288.58 L287.869,1287.98 L288.255,1287.16 L288.095,1286.71 L287.578,1286.22 L286.928,1285.91 L286.367,1286.01 L286.49,1286.82 L285.279,1287.05 L282.464,1286.91 L282.25,1287.12 L282.035,1287.69 L281.849,1287.86 L281.205,1287.63 L280.939,1287.59 L280.028,1288.09 L279.552,1288.16 L278.917,1287.96 L279.037,1287.35 L278.766,1286.54 L278.322,1286.09 L277.925,1286.55 L275.366,1287.87 L275.531,1288.2 L275.609,1288.32 L275.764,1288.42 L272.864,1288.95 L273.159,1288.3 L272.788,1288.33 L271.437,1289.03 L271.308,1288.51 L270.928,1288.89 L271.192,1288.13 L271.341,1287.87 L271.549,1287.64 L271.653,1287.27 L271.354,1287.07 L271.238,1286.88 L271.116,1286.22 L272.005,1286.2 L272.822,1285.9 L274.53,1284.91 L275.337,1284.63 L276.747,1284.64 L277.68,1284.45 L278.492,1284.09 L280.647,1282.65 L281.505,1282.29 L283.021,1282.04 L283.942,1281.74 L282.268,1281.08 L274.037,1282.79 L271.651,1282.32 L273.52,1281.71 L273.907,1281.65 L274.34,1281.5 L274.761,1281.01 L275.253,1280.53 L276.859,1280.41 L282.15,1279.18 L282.869,1278.52 L284.132,1278.7 L285.719,1278.35 L287.182,1278.4 L288.412,1278.2 L288.756,1277.24 L289.014,1276.99 L289.274,1276.68 L289.489,1276.55 L289.592,1276.17 L289.134,1275.75 L288.459,1275.94 L287.754,1276.02 L287.086,1275.1 L286.757,1275.03 L286.631,1274.91 L286.659,1274.66 L286.94,1273.91 L286.97,1273.67 L286.807,1273.35 L286.562,1272.99 L286.255,1272.67 L285.915,1272.44 L285.86,1273.27 L285.655,1273.82 L285.537,1274.34 L285.756,1275.11 L284.949,1275.1 L284.341,1275.33 L283.012,1276.13 L282.158,1276.4 L281.296,1276.43 L278.785,1276.02 L277.35,1276.2 L272.46,1275.56 L270.728,1275.83 L270.058,1276.11 L269.946,1276.26 L269.875,1276.48 L269.78,1276.7 L269.595,1276.87 L269.346,1276.96 L266.786,1277.07 L265.686,1277.55 L265.277,1277.36 L264.632,1276.84 L264.275,1276.66 L262.496,1276.39 L261.547,1276.54 L260.709,1277.02 L260.585,1276.5 L260.927,1276.37 L261.097,1276.06 L261.221,1275.69 L261.442,1275.4 L261.711,1275.3 L264.454,1275.31 L265.122,1275.11 L265.197,1274.85 L265.168,1274.77 L265.074,1274.76 L264.973,1274.67 L265.332,1273.96 L265.157,1273.44 L264.66,1273.11 L264.072,1272.98 L264.756,1272.37 L265.611,1272.43 L267.088,1272.99 L268.003,1272.87 L270.23,1271.65 L270.356,1271.2 L268.834,1270.77 L269.776,1269.86 L271.197,1269.27 L272.657,1268.99 L273.705,1269.02 L273.511,1269.27 L273.137,1269.84 L272.849,1270.12 L273.547,1270.17 L274.252,1270.03 L274.844,1269.54 L275.205,1268.55 L275.597,1269.03 L276.215,1269.33 L277.65,1269.68 L277.765,1269.27 L277.488,1269.14 L276.844,1268.57 L276.947,1268.19 L277.542,1268 L278.801,1267.15 L279.653,1266.94 L280.308,1266.25 L280.575,1266.05 L280.887,1266.05 L281.842,1266.4 L283.153,1266.36 L285.668,1265.72 L286.96,1265.64 L287.085,1265.19 L284.299,1264.8 L279.991,1265.21 L279.159,1265.45 L278.741,1265.75 L278.401,1266.08 L277.178,1265.74 L276.553,1265.77 L276.657,1266.29 L275.343,1266.7 L274.905,1266.69 L275.066,1266.42 L275.36,1265.76 L275.516,1265.48 L274.99,1265.33 L274.613,1265.49 L274.312,1265.76 L274.015,1265.95 L273.525,1265.98 L272.592,1265.87 L272.118,1265.9 L270.129,1266.75 L269.164,1266.88 L268.452,1266.19 L267.478,1266.79 L266.328,1267.3 L265.169,1267.62 L264.179,1267.65 L264.622,1266.92 L261.589,1264.23 L263.134,1264.18 L263.884,1262.82 L263.685,1261.1 L262.394,1259.95 L261.876,1260.15 L260.532,1261.33 L260.016,1261.54 L259.501,1261.64 L258.037,1262.42 L257.368,1262.61 L257.385,1261.77 L258.33,1260.14 L258.325,1259.26 L257.818,1258.98 L257.051,1258.84 L256.599,1258.59 L257.047,1258 L257.111,1257.71 L257.048,1257.6 L256.956,1257.56 L256.913,1257.51 L257.055,1257.31 L257.174,1257.19 L257.288,1257.04 L257.403,1256.76 L257.282,1256.7 L257.019,1256.66 L256.897,1256.61 L257.016,1256.2 L258.281,1256.39 L261.67,1256.17 L262.314,1256.35 L262.665,1256.25 L262.903,1255.87 L262.995,1255.45 L262.922,1255.22 L262.49,1254.75 L262.364,1253.98 L262.159,1253.28 L261.489,1252.98 L261.596,1252.61 L262.103,1252.45 L263.111,1252.38 L263.63,1252.26 L263.595,1251.29 L264.476,1251.12 L265.579,1251.18 L266.71,1250.79 L269.379,1250.33 L273.095,1250.57 L274.847,1250.1 L276.223,1248.27 L276.498,1248.34 L276.188,1249.43 L276.401,1250 L276.887,1250.26 L277.408,1250.38 L277.446,1249.79 L279.229,1246.44 L280.176,1247.45 L281.314,1247.83 L284.159,1248.23 L283.963,1247.93 L283.762,1247.42 L283.611,1247.19 L285.124,1246.72 L277.689,1245.07 L277.822,1246.45 L277.394,1247.07 L277.149,1247.29 L277.206,1246.25 L276.962,1245.75 L276.548,1245.35 L276.065,1245.05 L274.971,1244.66 L269.666,1244.51 L268.629,1244.82 L268.332,1244.34 L268.134,1243.65 L267.849,1243.19 L267.423,1243.23 L267.072,1243.52 L266.753,1243.6 L266.086,1242.36 L265.684,1242.21 L264.562,1242.24 L264.454,1242.62 L264.818,1242.72 L265.102,1242.9 L265.616,1243.4 L265.509,1243.77 L264.623,1243.34 L263.055,1242.21 L262.194,1241.96 L262.11,1242.38 L262.169,1242.6 L262.492,1242.9 L261.688,1242.7 L260.032,1242.97 L259.208,1242.88 L258.416,1242.29 L258.38,1241.48 L258.609,1240.44 L258.633,1239.13 L259.006,1239.04 L259.873,1238.6 L259.981,1238.22 L259.741,1237.62 L260.322,1237.21 L261.241,1236.99 L261.995,1236.98 L261.488,1237.73 L262.099,1238.65 L262.51,1238.96 L262.941,1238.84 L263.184,1238.39 L263.18,1237.93 L263.287,1237.49 L263.87,1237.08 L263.585,1236.67 L263.451,1236.55 L264.058,1235.72 L264.562,1235.6 L265.105,1235.78 L265.856,1235.82 L266.408,1235.5 L267.592,1234.42 L268.08,1234.27 L272.214,1234.24 L272.652,1234.9 L271.553,1237.07 L271.52,1237.46 L274.029,1237.77 L275.288,1237.44 L276.57,1236.63 L277.257,1235.45 L276.716,1234.03 L278.009,1234.37 L278.356,1234.71 L278.005,1235.29 L277.929,1236.22 L277.691,1237.17 L277.625,1237.97 L278.066,1238.43 L278.632,1238.71 L279.607,1239.57 L280.773,1240.19 L282.29,1240.65 L282.896,1240.49 L283.944,1239.7 L284.438,1239.58 L284.833,1239.9 L284.529,1240.02 L284.236,1240.18 L284.737,1240.72 L287.502,1241.1 L287.093,1240.33 L286.614,1240.02 L285.461,1239.67 L284.455,1238.65 L283.943,1238.35 L283.272,1238.61 L283.082,1238.23 L282.983,1237.8 L282.969,1237.3 L283.041,1236.72 L283.363,1237.28 L283.638,1237.63 L284.019,1237.73 L284.658,1237.62 L284.264,1237.2 L284.049,1237.04 L283.746,1236.92 L283.872,1236.47 L284.241,1236.56 L285.277,1236.46 L285.152,1235.33 L286.097,1232.4 L285.665,1231.61 L283.836,1230.7 L282.951,1229.9 L282.494,1228.97 L283.204,1228.56 L283.971,1228.51 L286.49,1229.04 L290.61,1228.35 L291.215,1228.07 L291.993,1227.39 L292.725,1227.19 L293.674,1227.48 L294.556,1228 L295.078,1228.5 L295.154,1227.63 L294.838,1227.11 L294.29,1226.82 L293.659,1226.73 L293.956,1226.48 L294.12,1226.19 L294.138,1225.85 L294.005,1225.48 L294.12,1225.07 L294.521,1224.78 L294.683,1224.45 L294.795,1223.7 L295.05,1223.3 L296.208,1222.49 L296.357,1222.73 L296.636,1222.62 L297.004,1222.42 L297.433,1222.38 L298.813,1223.21 L301.718,1224.02 L301.968,1224.23 L302.234,1224.6 L302.567,1224.82 L303.015,1224.63 L303.392,1224.3 L303.678,1224.18 L303.99,1224.21 L306.734,1225.08 L306.873,1225.12 L307.151,1225.3 L307.753,1225.19 L310.07,1225.88 L311.447,1225.97 L312.105,1225.89 L312.745,1225.49 L313.371,1225.26 L318.212,1225.03 L319.004,1224.75 L319.125,1225.5 L319.751,1225.83 L320.536,1225.88 L321.133,1225.77 L321.44,1225.48 L321.819,1224.98 L322.252,1224.54 L322.718,1224.42 L329.338,1225.3 L330.688,1225.2 L330.026,1225.02 L327.904,1223.41 L326.763,1222.83 L324.28,1222.57 L323.689,1222.29 L323.359,1221.89 L323.418,1221.5 L323.994,1221.2 L324.105,1220.79 L324.094,1220.22 L324.342,1219.51 L324.422,1218.86 L323.923,1218.48 L324.363,1218.07 L324.979,1217.39 L325.352,1217.12 L324.322,1217.13 L323.011,1217.45 L321.941,1218.03 L321.651,1218.79 L319.854,1220.75 L315.42,1224.3 L314.625,1224.49 L311.269,1223.98 L311.035,1224.1 L310.234,1224.75 L309.795,1224.92 L309.328,1224.86 L308.882,1224.68 L308.488,1224.38 L308.181,1224.02 L308.909,1223.65 L309.856,1223.33 L310.633,1222.87 L310.847,1222.04 L309.492,1222.45 L306.589,1222.58 L304.714,1222.13 L303.963,1221.77 L303.344,1221.26 L303.188,1220.64 L303.03,1220.23 L301.832,1219.7 L301.479,1219.05 L301.928,1218.17 L301.626,1217.71 L300.282,1217.38 L300.167,1217.8 L300.475,1218.01 L300.697,1218.34 L300.829,1218.77 L300.852,1219.28 L297.729,1219.83 L296.012,1219.74 L295.119,1219.88 L294.596,1220.47 L294.236,1221.29 L293.929,1221.31 L293.658,1221 L293.426,1220.81 L292.248,1221.27 L285.461,1220.35 L285.568,1219.98 L286.318,1219.85 L288.19,1219.16 L290.212,1219.18 L299.595,1214.07 L299.563,1213.82 L299.58,1213.17 L300.029,1213.09 L300.671,1212.8 L301.073,1212.7 L301.583,1212.69 L302.684,1213.15 L303.112,1213.19 L303.876,1213.07 L304.302,1213.11 L304.723,1212.96 L305.029,1212.43 L305.339,1211.77 L305.769,1211.28 L305.643,1210.98 L305.646,1210.95 L305.766,1210.89 L305.987,1210.49 L305.84,1209.97 L306.289,1209.85 L306.788,1209.57 L307.301,1209.18 L307.785,1208.72 L308.146,1208.24 L308.389,1207.8 L308.636,1207.46 L309.009,1207.28 L309.123,1206.87 L309.406,1206.08 L312.269,1204.5 L312.999,1203.44 L312.378,1203.03 L309.489,1202.39 L308.634,1201.78 L309.943,1201.03 L310.528,1200.58 L312.107,1199.04 L312.566,1198.71 L313.037,1198.54 L312.881,1198.06 L315.841,1194.86 L316.861,1194.21 L317.74,1193.36 L318.588,1192.2 L319.509,1191.32 L320.358,1191.21 L321.073,1191.95 L322.324,1194.04 L323.021,1194.6 L323.782,1194.52 L324.657,1194.24 L325.523,1194.24 L326.243,1194.99 L326.608,1194.05 L326.24,1193.36 L324.796,1192.42 L324.92,1191.96 L327.147,1192.13 L327.649,1192.51 L328.501,1193.73 L329.178,1194.01 L329.279,1193.63 L328.011,1192.36 L328.135,1191.9 L328.865,1192.51 L329.31,1192.72 L329.854,1192.36 L330.318,1192.28 L330.58,1192.43 L330.231,1192.96 L330.455,1193.56 L330.743,1194.04 L331.126,1194.4 L331.623,1194.67 L331.638,1193.34 L331.586,1192.84 L331.442,1192.4 L332.154,1191.6 L332.902,1191.1 L334.351,1190.51 L333.968,1190.42 L333.212,1190.53 L332.884,1190.52 L329.351,1189.44 L328.911,1189.04 L329.654,1188.81 L331.405,1188.83 L332.121,1188.98 L332.232,1188.57 L331.223,1188.29 L331.325,1187.91 L331.745,1187.7 L332.754,1187.67 L333.297,1187.56 L333.408,1187.15 L330.657,1185.96 L329.897,1185.89 L328.488,1186.12 L316.542,1183.89 L315.251,1184.14 L314.601,1184.4 L313.741,1184.4 L312.244,1183.98 L311.582,1183.4 L311.323,1182.73 L311.176,1182.05 L310.841,1181.4 L311.821,1180.54 L311.755,1179.77 L311.148,1179.61 L310.5,1180.64 L309.973,1181.9 L309.339,1182.56 L308.757,1182.5 L308.382,1181.6 L308.507,1181.15 L309.551,1180.09 L310.023,1177.27 L310.882,1176.95 L310.985,1177.04 L311.268,1177.4 L311.511,1177.53 L311.691,1177.44 L312.081,1176.94 L312.23,1176.84 L312.385,1176.64 L311.338,1174.8 L311.857,1174.46 L313.418,1174.05 L312.143,1173.62 L311.48,1173.55 L310.833,1173.78 L311.103,1174.63 L310.879,1175.69 L310.312,1176.49 L309.546,1176.58 L309.095,1175.72 L309.576,1174.64 L310.239,1173.67 L310.35,1173.19 L310.178,1173.01 L310.09,1172.75 L309.965,1172.6 L309.68,1172.76 L309.01,1173.26 L308.518,1173.34 L308.099,1173.72 L307.445,1174.57 L306.146,1175.89 L305.389,1176.44 L303.473,1176.73 L302.887,1176.57 L302.491,1175.69 L301.961,1174.88 L300.291,1174.81 L299.844,1174.24 L300.07,1174.05 L300.469,1173.33 L300.737,1172.67 L300.99,1172.26 L301.305,1172.09 L301.645,1172.1 L302.131,1172.21 L302.783,1172.2 L304.526,1171.61 L305.207,1171.26 L305.313,1170.88 L304.071,1170.53 L304.199,1170.08 L305.058,1169.92 L305.175,1169.5 L304.642,1169.35 L304.197,1169.41 L303.222,1169.8 L303.122,1170.24 L302.559,1170.41 L302.031,1170.2 L302.02,1169.46 L302.367,1168.97 L303.213,1168.67 L303.69,1168.15 L303.211,1168.09 L302.848,1168.32 L302.51,1168.58 L302.114,1168.63 L301.821,1168.42 L301.584,1168.11 L301.271,1167.97 L300.758,1168.25 L300.483,1168.81 L300.392,1169.52 L300.248,1170.08 L299.793,1170.21 L299.371,1170.16 L298.684,1170.27 L297.258,1169.89 L296.826,1169.63 L296.483,1169.26 L296.262,1168.65 L296.201,1168.05 L296.063,1167.54 L295.615,1167.23 L294.102,1167.07 L293.268,1166.74 L292.928,1166.01 L292.555,1166.2 L292.231,1166.24 L291.568,1166.06 L292.167,1165.18 L292.736,1164.63 L293.367,1164.42 L294.144,1164.53 L295.423,1165.21 L295.986,1165.29 L296.692,1164.86 L296.821,1164.4 L296.423,1164.23 L296.144,1163.9 L295.993,1163.43 L295.956,1162.82 L296.694,1163.18 L297.456,1163.83 L298.14,1164.17 L298.635,1163.58 L297.908,1163.46 L296.646,1162.8 L295.463,1161.83 L294.969,1160.74 L295.045,1160.35 L295.052,1159.57 L294.952,1158.79 L294.702,1158.39 L293.579,1158.39 L293.151,1158.31 L292.901,1157.87 L293.926,1157.39 L294.75,1157.64 L295.533,1158.16 L296.413,1158.48 L296.417,1158.18 L296.499,1157.46 L296.504,1157.16 L298.081,1157.63 L299.557,1158.52 L299.694,1158.81 L299.954,1159.9 L300.338,1159.84 L301.056,1159.22 L301.506,1159.08 L299.759,1156.99 L299.44,1156.1 L300.886,1156.18 L303.579,1157.25 L304.673,1157.26 L306.119,1156.82 L306.613,1156.92 L306.941,1157.18 L307.535,1157.94 L309.15,1158.73 L311.887,1159.71 L311.993,1159.33 L311.468,1158.83 L308.925,1158.35 L307.546,1157.31 L305.171,1154.68 L305.436,1151.83 L306.016,1150.6 L306.692,1149.72 L307.175,1148.85 L307.174,1147.65 L307.58,1147.51 L308.996,1147.64 L310.064,1147.54 L310.933,1147.68 L311.352,1147.65 L311.825,1147.37 L312.038,1147.4 L312.622,1148.08 L312.968,1148.36 L319.896,1149.01 L320.021,1148.56 L319.512,1148.31 L318.293,1147.18 L318.519,1147.11 L318.704,1147.02 L318.897,1146.96 L319.137,1146.97 L319.253,1146.56 L319.24,1146.02 L319.77,1145.58 L321.285,1144.9 L321.003,1144.79 L320.455,1144.22 L320.809,1144.05 L321.129,1143.74 L321.412,1143.3 L321.647,1142.76 L321.395,1142.75 L321.227,1142.76 L321.053,1142.83 L320.804,1142.98 L319.901,1142.31 L314.86,1140.86 L310.809,1141.87 L308.761,1141.78 L307.952,1140.2 L308.489,1139.97 L309.192,1138.35 L309.91,1138.08 L309.969,1137.49 L310.033,1137.22 L310.264,1136.83 L310.372,1136.46 L310.109,1136.38 L310.226,1135.97 L312.046,1136 L312.31,1136.25 L312.928,1137.05 L313.377,1137.31 L313.698,1137.75 L313.437,1139.21 L313.753,1139.65 L314.11,1139.55 L314.239,1139.05 L314.368,1138.37 L314.714,1137.69 L314.485,1137.23 L314.324,1136.7 L314.086,1136.26 L313.62,1136.04 L313.512,1136.42 L313.604,1136.93 L313.241,1136.67 L313.125,1136.16 L313.142,1134.54 L313.25,1134.25 L313.32,1134 L313.234,1133.66 L313.021,1133.54 L312.405,1133.45 L312.345,1133.21 L312.338,1132.79 L312.174,1132.39 L312.062,1131.82 L312.202,1130.89 L312.621,1130.03 L313.087,1129.86 L313.597,1129.84 L314.16,1129.41 L313.73,1129.05 L313.823,1128.7 L314.236,1128.42 L314.766,1128.29 L314.621,1127.8 L315.852,1127.22 L315.599,1126.82 L315.197,1126.69 L314.192,1126.74 L314.139,1127.26 L314.035,1127.48 L313.826,1127.5 L313.443,1127.46 L313.324,1127.88 L313.565,1127.94 L312.18,1128.95 L311.128,1128.87 L310.457,1127.88 L310.199,1126.09 L310.873,1126.31 L312.599,1126.29 L313.338,1126.09 L312.499,1124.98 L312.236,1124.28 L312.24,1123.5 L312.426,1123.71 L312.859,1124.02 L313.051,1124.22 L312.811,1123.38 L311.998,1121.64 L312.376,1122.11 L312.783,1122.25 L313.19,1122.09 L313.554,1121.64 L312.184,1119.56 L311.478,1119.02 L310.773,1119.51 L311.088,1120.29 L310.747,1120.75 L310.088,1120.82 L309.461,1120.47 L309.594,1121.18 L309.7,1121.43 L309.58,1121.84 L309.105,1121.63 L308.646,1121.6 L307.773,1121.82 L307.664,1122.19 L308.07,1122.5 L308.141,1122.91 L307.896,1124.09 L307.592,1124.02 L306.794,1124.22 L307.162,1124.68 L307.434,1125.36 L307.359,1125.92 L306.69,1125.98 L306.048,1125.74 L305.599,1125.35 L305.455,1124.64 L305.735,1123.46 L306.151,1122.81 L307.443,1121.32 L307.82,1120.67 L308.254,1119.62 L308.696,1119.2 L309.205,1118.93 L309.829,1118.3 L308.738,1117.96 L308.359,1117.6 L308.118,1116.95 L308.788,1117.14 L309.264,1116.68 L309.687,1115.94 L310.174,1115.31 L310.661,1115.07 L312.474,1114.59 L312.69,1114.37 L312.863,1114.09 L313.078,1113.89 L313.43,1113.97 L313.578,1114.22 L313.552,1114.61 L313.488,1115.01 L313.531,1115.33 L313.875,1115.84 L314.389,1116.34 L314.985,1116.59 L315.584,1116.37 L315.704,1116.92 L315.915,1117.26 L316.227,1117.35 L316.651,1117.16 L316.655,1117.37 L316.622,1117.46 L316.646,1117.52 L316.805,1117.61 L315.591,1118.91 L314.561,1119.35 L313.72,1118.99 L313.077,1117.89 L312.515,1119.01 L312.955,1119.52 L313.769,1119.82 L314.317,1120.3 L314.779,1120.87 L315.388,1120.57 L316.047,1120.02 L316.66,1119.85 L316.393,1118.98 L316.9,1118.07 L317.71,1117.37 L318.345,1117.16 L319.043,1117.6 L320.009,1118.87 L320.746,1119.22 L320.689,1118.64 L320.509,1118.23 L319.911,1117.61 L320.03,1117.19 L320.412,1117.14 L320.749,1117.21 L321.467,1117.59 L320.924,1117.02 L319.209,1116.08 L319.005,1115.35 L319.046,1114.43 L319.244,1114 L321.451,1114.02 L322.138,1114.42 L323.225,1115.65 L323.849,1116.04 L329.734,1118.17 L330.143,1118.21 L330.887,1118.09 L331.298,1118.13 L331.572,1118.32 L331.907,1118.99 L332.132,1119.25 L334.806,1120.44 L335.172,1120.44 L335.667,1120.31 L336.162,1120.09 L336.523,1119.82 L336.927,1119.64 L337.335,1119.84 L338.695,1121.03 L338.994,1121.15 L339.185,1121.48 L339.155,1122.7 L339.264,1123.01 L339.92,1123.07 L340.316,1123.04 L340.436,1122.88 L340.536,1123.43 L340.455,1123.93 L340.247,1124.34 L339.968,1124.58 L339.853,1124.99 L340.513,1125.58 L340.388,1126.03 L339.58,1126.33 L339.59,1126.94 L340.862,1128.97 L341.117,1129.78 L341.148,1130.79 L340.82,1131.98 L341.039,1132.04 L341.369,1130.95 L341.76,1130.06 L342.044,1129.22 L342.074,1128.27 L342.958,1128.15 L343.966,1127.56 L344.907,1126.5 L345.585,1124.98 L346.382,1124.12 L347.609,1124.3 L349.602,1125.38 L349.871,1125.37 L350.118,1125.21 L350.336,1125.16 L350.643,1125.62 L351.15,1126.17 L351.658,1126.53 L352.291,1127.31 L352.792,1127.58 L354.526,1128.05 L355.354,1128.02 L356.236,1127.62 L357.006,1128.64 L359.273,1128.77 L360.127,1129.54 L360.181,1130.17 L360.097,1130.9 L360.16,1131.55 L361.047,1132.22 L361.347,1132.73 L361.779,1134.06 L362.352,1133.8 L362.931,1133.82 L364.068,1134.18 L364.178,1133.77 L363.743,1133.37 L363.158,1133 L362.697,1132.52 L362.644,1131.78 L362.546,1131.26 L362.106,1130.76 L361.512,1130.39 L360.97,1130.25 L361.091,1129.8 L361.407,1129.57 L362.009,1128.96 L362.385,1128.81 L363.059,1129.4 L363.502,1129.64 L363.792,1129.4 L364.265,1129.25 L366.624,1130.38 L366.266,1129.5 L365.763,1128.81 L365.168,1128.33 L364.528,1128.05 L363.797,1127.91 L363.588,1127.68 L363.62,1127.16 L363.978,1126.58 L364.372,1126.86 L365.002,1127.76 L365.822,1127.51 L365.712,1126.88 L365.088,1126.19 L364.383,1125.79 L360.891,1124.86 L360.751,1124.62 L360.871,1123.96 L361.149,1123.55 L361.611,1123.24 L362.108,1123.08 L363.206,1123.04 L368.071,1121.2 L368.702,1121.15 L368.965,1121.02 L369.192,1120.56 L369.549,1119.36 L369.952,1118.35 L370.164,1118.61 L370.245,1119.34 L370.242,1119.74 L370.368,1119.83 L370.409,1119.87 L370.336,1120.25 L371.554,1119.78 L372.739,1119.52 L372.832,1119.52 L375.846,1119.7 L376.318,1119.73 L378.199,1119.03 L379.385,1119.06 L379.504,1118.61 L378.356,1118.26 L377.905,1117.88 L377.599,1117.19 L379.94,1116.06 L380.163,1114.86 L382.18,1113.78 L382.474,1112.24 L382.787,1112.58 L382.962,1112.92 L383.168,1113.21 L383.582,1113.42 L384.496,1111.56 L384.817,1111.08 L384.743,1110.92 L384.712,1110.9 L384.699,1110.85 L384.677,1110.64 L383.838,1111.11 L383.049,1111.29 L380.552,1111.09 L379.001,1111.81 L378.171,1112.01 L378.27,1111.63 L378.774,1111.2 L378.967,1110.66 L378.84,1110.16 L378.382,1109.84 L377.787,1109.97 L376.582,1110.91 L375.57,1111.13 L373.448,1112.55 L372.871,1112.83 L372.297,1113 L371.754,1112.99 L371.853,1112.62 L372.979,1111.81 L373.629,1111.5 L374.339,1111.49 L374.93,1111.16 L375.056,1110.27 L374.764,1109.63 L374.091,1110.05 L373.852,1109.99 L374.318,1108.68 L373.866,1108.56 L373.062,1109.24 L372.466,1110.31 L372.199,1110.62 L371.911,1110.22 L371.704,1109.64 L371.662,1109.18 L371.768,1108.91 L371.799,1108.67 L371.547,1108.49 L371.299,1108.49 L370.862,1108.65 L370.62,1108.69 L369.758,1108.52 L369.306,1108.57 L368.712,1109.31 L368.301,1109.07 L367.733,1108.37 L365.092,1107.66 L364.593,1107.09 L365.033,1106.58 L364.793,1106.16 L362.21,1104.26 L361.59,1104.11 L361.77,1103.91 L361.901,1103.71 L362.06,1103.53 L362.302,1103.37 L362.404,1102.99 L362.032,1102.58 L361.916,1102.09 L362.1,1101.72 L362.633,1101.68 L363.158,1101.75 L363.626,1101.65 L364.017,1101.28 L364.312,1100.57 L364.831,1100.22 L367.801,1099.06 L368.784,1098.94 L372.253,1099.96 L373.917,1101.06 L374.855,1101.34 L374.956,1100.96 L372.683,1099.7 L372.207,1098.86 L372.817,1098.74 L374.403,1099.07 L374.764,1099.33 L375.008,1100.18 L375.379,1101.04 L375.912,1101.56 L376.643,1101.4 L375.293,1099.04 L374.11,1098.22 L373.102,1096.96 L372.309,1096.66 L372.609,1096.11 L372.903,1096.04 L373.245,1096.14 L373.698,1096.14 L374.11,1095.96 L374.742,1095.44 L375.218,1095.21 L375.717,1096.09 L376.462,1096.36 L378.183,1096.48 L379.913,1097.77 L380.627,1097.97 L380.736,1097.55 L380.15,1097.3 L379.469,1096.76 L379.084,1095.97 L379.369,1094.97 L379.999,1094.58 L381.3,1095.18 L381.849,1095.18 L381.495,1094.66 L381.393,1093.48 L380.986,1093.13 L380.363,1092.91 L380.168,1092.92 L379.951,1093.16 L379.78,1093.57 L379.664,1093.96 L379.589,1094.14 L378.615,1093.83 L378.04,1092.73 L377.741,1091.32 L377.606,1090.06 L378.146,1089.85 L378.431,1090.11 L378.664,1090.55 L379.066,1090.86 L379.564,1090.91 L380.073,1090.85 L381.086,1090.5 L380.175,1089.8 L379.603,1088.82 L378.729,1086.81 L378.85,1086.36 L379.222,1086.14 L379.617,1086.11 L379.989,1086.28 L380.29,1086.74 L380.81,1086.22 L380.964,1086.02 L380.749,1085.97 L381.15,1085.15 L381.465,1085.08 L381.797,1085.21 L382.252,1085.07 L383.319,1083.78 L383.624,1083.62 L383.859,1084.06 L383.579,1085.51 L383.827,1085.89 L384.09,1085.69 L384.607,1084.63 L384.985,1084.38 L385.066,1082.49 L385.213,1081.39 L385.491,1080.96 L385.998,1080.67 L387.048,1078.93 L387.558,1078.39 L388.1,1078.5 L389.823,1079.87 L390.483,1079.9 L391.696,1079.61 L392.296,1079.62 L391.939,1080.1 L391.601,1080.43 L391.315,1080.78 L391.092,1081.33 L391.194,1081.68 L391.693,1081.44 L393.115,1080.21 L395.562,1079.4 L396.298,1079.33 L396.9,1078.98 L397.476,1078.07 L398.128,1077.29 L398.951,1077.35 L399.685,1078.3 L399.217,1078.89 L397.357,1079.6 L397.249,1080.02 L398.912,1079.6 L399.089,1079.8 L399.243,1080.16 L399.776,1080.24 L400.388,1080.15 L400.777,1079.99 L401.024,1080.98 L400.926,1081.36 L400.61,1082.03 L400.927,1081.99 L401.548,1081.71 L402.129,1081.67 L401.804,1082.22 L401.43,1082.55 L401.024,1082.68 L400.594,1082.64 L400.487,1083.06 L401.129,1083.19 L401.489,1083.42 L401.8,1083.46 L402.268,1083.07 L402.961,1082.07 L403.033,1081.38 L402.55,1080.04 L402.661,1078.76 L403.402,1078.15 L404.489,1077.68 L405.636,1076.84 L405.648,1077.17 L405.68,1077.19 L405.776,1077.36 L404.249,1080.01 L404.082,1080.43 L404.124,1080.56 L404.172,1080.84 L404.269,1081.16 L404.457,1081.41 L404.763,1081.5 L405.207,1081.18 L405.54,1081.21 L405.947,1081.55 L406.535,1082.22 L407.036,1083 L407.176,1083.65 L407.185,1083.89 L407.286,1084.76 L407.276,1085.23 L407.153,1085.66 L406.696,1086.71 L406.506,1087.25 L406.915,1086.77 L407.878,1085.3 L408.212,1085.02 L408.444,1084.61 L408.465,1083.59 L408.383,1082.57 L408.314,1082.15 L408.415,1081.54 L408.454,1080.71 L408.33,1079.96 L407.954,1079.65 L407.567,1080.02 L407.41,1080.91 L407.398,1081.91 L407.441,1082.61 L406.988,1082.4 L406.742,1081.96 L406.57,1081.45 L406.346,1081.01 L405.408,1080.44 L405.09,1079.95 L405.411,1079.22 L406.853,1077.52 L406.941,1077.17 L408.122,1077.69 L408.721,1077.62 L410.234,1076.85 L410.711,1076.8 L411.237,1077.17 L411.454,1077.86 L411.575,1078.53 L411.809,1078.88 L411.873,1079.37 L411.568,1080.31 L411.183,1081.19 L411.011,1081.52 L411.061,1082.11 L411.407,1082.63 L412.272,1083.39 L412.106,1084.11 L412.18,1085 L412.412,1085.77 L412.724,1086.15 L413.079,1086.73 L412.896,1087.74 L412.44,1088.7 L411.985,1089.07 L411.427,1089.09 L410.922,1089.35 L410.097,1089.96 L407.885,1090.73 L407.779,1091.15 L409.959,1091.27 L410.538,1091.8 L409.833,1092.99 L409.308,1093.33 L407.265,1094.16 L405.682,1095.52 L405.093,1095.83 L404.987,1096.24 L405.734,1096.2 L406.477,1095.91 L410.601,1093.5 L410.987,1093.5 L411.453,1092.92 L412.232,1092.77 L413.08,1092.77 L413.749,1092.64 L414.387,1092 L414.849,1091.07 L414.911,1090.18 L414.366,1089.66 L414.265,1089.37 L414.548,1088.85 L415.129,1088.08 L415.497,1086.62 L415.275,1086.46 L414.441,1083.93 L413.8,1083.03 L413.621,1082.62 L413.578,1082.17 L413.607,1081.49 L413.764,1080.9 L414.714,1080.38 L415.358,1079.42 L415.762,1078.21 L415.637,1077.14 L416.204,1077.36 L416.651,1077.72 L417.118,1077.93 L417.758,1077.67 L417.717,1077.45 L417.722,1077.3 L417.744,1077.13 L417.753,1076.79 L418.288,1077.56 L418.906,1078.2 L419.674,1078.37 L421.353,1077.25 L422.109,1077.2 L422.712,1077.72 L422.966,1078.96 L422.228,1078.82 L422.003,1078.73 L422.538,1079.38 L423.21,1079.93 L423.971,1080.15 L424.782,1079.82 L423.021,1077.49 L422.32,1075.9 L422.598,1074.45 L421.823,1073.67 L421.501,1073.2 L421.304,1072.72 L421.942,1072.49 L422.529,1072.79 L423.523,1073.76 L425.682,1075.21 L426.222,1075.44 L427.366,1076.5 L427.73,1077.01 L428.82,1080.18 L429.223,1080.79"
         id="path1053" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['IE'],polycolors['IE'],labels['IE'], x1, polybary['IE'], x2, polybary['IE'] )
    if 'IE' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1341.34,1767.09 L1341.62,1767.41 L1342.52,1767.22 L1342.37,1766.8 L1342.15,1766.4 L1342.82,1766.57 L1343.41,1767.19 L1343.91,1767.46 L1344.26,1766.63 L1344.46,1767.09 L1344.57,1767.45 L1344.55,1767.83 L1344.41,1768.37 L1344.66,1768.12 L1345.32,1767.61 L1345.57,1767.36 L1346.59,1769.15 L1348.6,1770.39 L1349.8,1770.81 L1352.67,1771.82 L1353.12,1771.5 L1355.2,1771.14 L1355.37,1770.85 L1356.04,1768.92 L1357.86,1769.88 L1358.71,1770.14 L1360.25,1770.94 L1360.69,1771.25 L1361.33,1771.53 L1362.13,1771.36 L1363.55,1770.75 L1363.62,1770.53 L1363.7,1770.13 L1363.88,1769.87 L1364.25,1770.06 L1364.54,1770.45 L1364.62,1770.79 L1364.52,1771.15 L1364.24,1771.57 L1364.81,1771.71 L1365.33,1771.56 L1366.33,1770.9 L1366.05,1772.24 L1365.86,1772.66 L1366.18,1772.51 L1368.36,1772 L1369.34,1772 L1369.81,1771.88 L1370.67,1771.22 L1372.13,1769.33 L1373.03,1768.73 L1374.5,1768.67 L1375.88,1769.27 L1377.18,1770.32 L1379.97,1773.25 L1380.84,1773.89 L1381.76,1774.16 L1382.2,1774.06 L1383.44,1773.38 L1384.63,1773.02 L1384.98,1772.77 L1385.64,1772.01 L1387.99,1768.51 L1388.62,1768.09 L1389.3,1768.01 L1390.04,1768.32 L1392.7,1770.25 L1393.73,1770.6 L1394.6,1770.68 L1395.12,1770.63 L1395.12,1770.63 L1395.56,1770.59 L1396.48,1770.29 L1397.2,1769.72 L1397.43,1769.04 L1397.38,1768.25 L1397.4,1767.47 L1397.82,1766.81 L1398.88,1766.46 L1399.98,1766.77 L1402.13,1767.98 L1402.82,1768.2 L1405.95,1768.11 L1406.84,1768.4 L1407.67,1768.95 L1408.77,1769.91 L1408.77,1769.91 L1409.47,1770.51 L1410.21,1770.88 L1410.64,1771.09 L1411.84,1771.38 L1412.91,1771.3 L1412.73,1770.43 L1412.43,1769.7 L1412.4,1769.27 L1412.39,1769.15 L1413.01,1768.8 L1413.37,1768.79 L1414.1,1768.88 L1414.45,1768.85 L1414.74,1768.7 L1415.31,1768.26 L1415.59,1768.17 L1415.94,1768.28 L1416.17,1768.54 L1416.56,1769.32 L1416.87,1769.79 L1417.22,1770.17 L1417.59,1770.29 L1417.98,1770.02 L1418.09,1769.7 L1418.14,1769.32 L1418.22,1768.94 L1418.42,1768.67 L1418.72,1768.61 L1419,1768.74 L1419.25,1768.99 L1421.26,1772.16 L1421.86,1772.73 L1422.41,1772.9 L1423.65,1772.87 L1423.97,1772.95 L1424.17,1773.1 L1424.32,1773.34 L1424.49,1773.71 L1424.53,1773.84 L1424.56,1774.17 L1424.61,1774.35 L1424.73,1774.51 L1424.98,1774.79 L1425.08,1775 L1425.01,1775.62 L1424.14,1776.47 L1423.89,1777.11 L1424.56,1778.84 L1426.26,1780.28 L1427.42,1780.86 L1427.42,1780.86 L1428.2,1781.25 L1429.62,1781.54 L1434.7,1780.79 L1435.16,1780.59 L1436.52,1779.98 L1437.42,1779.74 L1438.1,1779.39 L1441.72,1776.02 L1442.26,1775.82 L1442.58,1776.16 L1442.81,1776.66 L1443.06,1776.93 L1443.5,1776.88 L1444.24,1776.45 L1444.67,1776.31 L1446.68,1776.48 L1447.08,1776.37 L1447.42,1776.14 L1447.81,1776.09 L1448.79,1776.82 L1449.25,1776.92 L1449.69,1776.82 L1449.8,1776.74 L1449.8,1776.68 L1450,1776.85 L1450,1776.85 L1450.29,1777.11 L1450.58,1777.33 L1450.8,1777.66 L1450.99,1778.16 L1451.11,1778.78 L1450.97,1778.97 L1450.75,1779.06 L1450.6,1779.37 L1449.93,1787.05 L1449.61,1788.25 L1449.26,1788.95 L1448.46,1789.81 L1448.05,1790.38 L1447.78,1790.99 L1447.36,1792.93 L1445.7,1796.74 L1445.52,1796.96 L1445.07,1797.31 L1444.88,1797.62 L1444.84,1798.04 L1445,1799.03 L1444.97,1799.48 L1444.77,1800.01 L1444.53,1800.46 L1444.22,1800.8 L1442.71,1801.52 L1442.71,1802.51 L1443.74,1805.17 L1443.85,1806.88 L1443.43,1807.85 L1442.91,1808.71 L1442.67,1810.05 L1442.89,1811.22 L1443.38,1812.42 L1444.61,1814.42 L1445.53,1815.3 L1446.48,1815.6 L1448.44,1815.78 L1449.42,1816.13 L1449.83,1816.21 L1450.46,1816.16 L1451.86,1815.64 L1452.38,1815.67 L1453.21,1816.21 L1454.12,1817.24 L1454.83,1818.5 L1455.07,1819.75 L1455.45,1820.51 L1456.39,1821.28 L1458.51,1822.47 L1459.05,1822.89 L1459.38,1823.03 L1459.64,1822.92 L1460.08,1822.34 L1460.35,1822.3 L1460.78,1822.74 L1461.01,1823.84 L1461.34,1824.2 L1462.11,1824.76 L1463.06,1825.83 L1463.37,1826.18 L1464.93,1826.85 L1465.77,1827.64 L1466.22,1827.92 L1466.37,1827.73 L1466.59,1827.32 L1466.98,1827.13 L1467.6,1827.56 L1468.07,1828.68 L1467.82,1829.75 L1467.16,1830.69 L1465.48,1832.3 L1464.47,1832.82 L1463.38,1833.06 L1460.8,1833.13 L1458.44,1833.6 L1457.79,1833.88 L1457.15,1834 L1456.53,1833.71 L1455.24,1832.77 L1454.01,1832.12 L1453.39,1831.97 L1452.75,1832.09 L1451.86,1832.51 L1451.51,1832.9 L1451.33,1833.52 L1451.39,1833.96 L1451.7,1834.71 L1451.65,1835.16 L1451.2,1835.73 L1451.76,1835.99 L1453.44,1836.98 L1455.02,1838.58 L1457.88,1842.15 L1463.07,1846.53 L1464.21,1848.46 L1465.7,1851.76 L1465.98,1853.39 L1464.96,1854.67 L1464.85,1855.44 L1464.88,1856.34 L1465.03,1857.19 L1465.33,1857.82 L1465.69,1858.08 L1465.85,1858.35 L1465.79,1858.62 L1465.46,1858.89 L1463.67,1858.42 L1463.13,1858.54 L1462.73,1859.97 L1462.4,1860.54 L1461.98,1860.04 L1461.49,1859.04 L1460.71,1857.94 L1459.82,1857.04 L1459.01,1856.69 L1458.54,1856.89 L1457.59,1857.85 L1457.09,1858.22 L1456.57,1858.31 L1455.4,1858.26 L1454.92,1858.47 L1454.41,1859.57 L1454.15,1861.01 L1453.73,1862.13 L1452.73,1862.25 L1450.54,1862.18 L1449.43,1862.46 L1448.38,1864.03 L1447.54,1864.49 L1446.57,1864.64 L1445.78,1864.47 L1445.1,1863.79 L1444.51,1862.94 L1443.81,1862.39 L1442.79,1862.62 L1442.18,1863.19 L1441.24,1864.52 L1440.45,1865.03 L1440.06,1865.39 L1439.85,1865.93 L1439.93,1866.45 L1440.4,1866.73 L1441.12,1866.83 L1441.47,1867 L1442.18,1867.87 L1442.2,1868.27 L1442.06,1868.81 L1442.09,1869.13 L1443.06,1868.75 L1443.41,1868.95 L1445.21,1870.82 L1446.97,1874.39 L1448.63,1876.82 L1448.34,1877.14 L1447.16,1877.35 L1446.67,1877.7 L1446.58,1877.85 L1446.38,1878.21 L1446.15,1878.79 L1445.85,1879.35 L1445.42,1879.83 L1445.25,1879.55 L1445.22,1879.49 L1444.16,1878.41 L1443.67,1877.76 L1443.51,1877.26 L1443.49,1876.81 L1443.42,1876.38 L1443.05,1875.98 L1441.73,1875.27 L1440.6,1875.09 L1437.93,1875.66 L1437.14,1876.16 L1437.32,1876.39 L1437.64,1876.93 L1437.81,1877.16 L1436.66,1877.68 L1431.33,1882.61 L1430.75,1883.7 L1431.9,1884.11 L1430.24,1887.09 L1429.67,1888.74 L1429.51,1890.66 L1429.5,1890.79 L1429.7,1892.4 L1431.02,1897.03 L1429.14,1896.99 L1426.43,1897.33 L1423.94,1898.2 L1422.7,1899.67 L1422.08,1902.2 L1422.22,1903.27 L1422.8,1904.46 L1424.11,1906.52 L1424.5,1907.68 L1424.23,1908.32 L1423.7,1908.83 L1423.33,1909.57 L1423.48,1910.64 L1424.06,1912.08 L1424.84,1913.49 L1426.71,1915.9 L1427.74,1916.76 L1428.64,1917.69 L1429.29,1919.33 L1429.47,1921.14 L1429.24,1921.95 L1429.09,1922.48 L1428.22,1923.54 L1427.77,1923.87 L1426.9,1924.51 L1427.29,1925.09 L1427.45,1925.64 L1427.31,1926.04 L1426.85,1926.16 L1426.29,1925.88 L1425.76,1925.9 L1425.26,1926.18 L1424.83,1926.67 L1424.05,1926.48 L1422.39,1926.48 L1421.66,1926.12 L1421.63,1926.07 L1421.62,1926.07 L1420.56,1924.82 L1420.01,1924.49 L1418.78,1924.71 L1416.18,1924.14 L1414.58,1923.49 L1406.01,1918.98 L1403.37,1917.66 L1402.2,1916.87 L1400.01,1914.86 L1398.81,1914.12 L1397.16,1913.84 L1396.62,1913.58 L1394.97,1911.48 L1394.81,1910.78 L1394.71,1909.28 L1394.52,1908.69 L1393.92,1908.04 L1393.51,1908.09 L1393.17,1908.45 L1392.83,1908.68 L1391.84,1908.61 L1389.85,1908.06 L1388.82,1907.96 L1387.83,1908.3 L1387.27,1909.02 L1386.78,1909.96 L1386.78,1909.96 L1386.76,1909.95 L1382.09,1907.69 L1382.89,1907.64 L1384.18,1908.08 L1384.98,1908.21 L1384.23,1907.61 L1383.15,1907.13 L1385.57,1906.36 L1386.77,1905.28 L1386.45,1903.57 L1386.44,1903.57 L1386.43,1903.56 L1386.43,1903.56 L1384.72,1900.42 L1383.17,1898.93 L1382.63,1898.4 L1376.02,1894.93 L1375.59,1894.71 L1374.82,1893.95 L1374.18,1892.81 L1373.68,1891.58 L1373.32,1890.95 L1372.79,1890.56 L1371.82,1890.06 L1369.8,1888.52 L1367.79,1886.01 L1366.5,1883.01 L1366.87,1877.71 L1366,1876.18 L1364.51,1875.37 L1360.06,1875.11 L1357.29,1873.88 L1356.2,1873.08 L1354.64,1871.92 L1352.22,1869.57 L1344.42,1862.92 L1339.06,1859.39 L1338.32,1858.42 L1337.83,1857.13 L1337.57,1856.1 L1337.16,1855.23 L1335.15,1853.67 L1333.51,1851.77 L1331.12,1849.7 L1329.68,1847.93 L1329.13,1847.24 L1328.37,1846.48 L1327.53,1845.94 L1325.5,1845.4 L1325.07,1844.7 L1324.79,1843.8 L1324.13,1842.9 L1323.67,1842.76 L1322.5,1842.94 L1322.01,1842.94 L1321.55,1842.72 L1321.01,1842.37 L1320.08,1841.58 L1319.33,1840.54 L1318.14,1838.19 L1317.4,1837.12 L1315.29,1835.19 L1314.47,1834.18 L1313.42,1831.58 L1312.84,1829.67 L1312.73,1828.73 L1312.8,1828.04 L1313.18,1826.99 L1313.29,1826.46 L1313.33,1825.41 L1313.32,1825.36 L1313.15,1824.8 L1312.63,1824.36 L1311.65,1823.84 L1310.82,1823.7 L1310.32,1823.72 L1310.17,1823.93 L1310.16,1823.44 L1310.43,1823.16 L1310.79,1822.88 L1311.03,1822.41 L1310.94,1821.46 L1310.55,1820.3 L1309.21,1817.33 L1308.88,1816.13 L1308.5,1815.06 L1307.93,1814.42 L1306.97,1814.34 L1305.41,1815.2 L1304.48,1815.15 L1303.48,1814.25 L1303.76,1813.22 L1304.42,1812.22 L1304.56,1811.38 L1304.91,1811.16 L1304.98,1811.1 L1304.75,1810.27 L1304.47,1809.51 L1304.07,1808.96 L1303.51,1808.74 L1302.41,1808.67 L1301.84,1808.53 L1301.36,1808.27 L1300.7,1807.45 L1300.25,1805.92 L1299.72,1805.28 L1299.38,1805.28 L1299.03,1805.52 L1298.66,1805.65 L1298.29,1805.32 L1298.1,1804.8 L1297.93,1804.16 L1297.72,1803.59 L1297.39,1803.26 L1296.84,1803.27 L1296.55,1803.68 L1296.35,1804.24 L1296.05,1804.66 L1295.68,1804.82 L1294.89,1804.94 L1294.47,1805.12 L1293.83,1805.27 L1293.41,1804.89 L1293.07,1804.28 L1292.69,1803.73 L1291.59,1803.09 L1291.32,1802.79 L1290.06,1801.02 L1289.49,1799.98 L1289.35,1798.94 L1289.94,1798.04 L1290.83,1797.68 L1291.58,1797.26 L1291.71,1796.15 L1291.21,1794.99 L1290.41,1793.93 L1289.71,1792.73 L1289.45,1791.17 L1289.56,1790.71 L1289.72,1790.05 L1290.68,1788.05 L1290.77,1786.94 L1290.1,1785.09 L1290.03,1783.93 L1290.87,1781.16 L1290.99,1780.18 L1290.56,1776.5 L1290.56,1776.02 L1290.96,1774.62 L1291.38,1774.17 L1292.3,1773.19 L1294.65,1772.61 L1298.88,1772.39 L1299.87,1772.11 L1300.36,1772.1 L1300.91,1772.32 L1301.34,1772.84 L1301.36,1773.39 L1301.23,1773.86 L1301.22,1774.15 L1301.63,1774.4 L1302.61,1774.53 L1303.19,1774.91 L1303.96,1775.97 L1305.09,1778.33 L1305.94,1779.28 L1310.3,1782.98 L1311.61,1783.65 L1313.35,1784.24 L1315.01,1784.47 L1316.07,1784.09 L1316.24,1783.19 L1316.24,1783.19 L1316.58,1781.37 L1317.34,1779.62 L1317.5,1779.09 L1317.53,1778.36 L1317.41,1777.96 L1317.37,1777.61 L1317.65,1777 L1318.04,1776.68 L1319.17,1776.22 L1319.62,1775.83 L1320.14,1774.83 L1321.32,1771.51 L1323.27,1769.91 L1325.84,1769.77 L1334.84,1771.6 L1335.95,1771.35 L1336.46,1770.81 L1336.49,1770.35 L1336.41,1769.81 L1336.59,1769.04 L1336.99,1768.57 L1337.47,1768.31 L1337.87,1767.97 L1338.06,1767.25 L1338.39,1767 L1338.81,1766.44 L1339.12,1765.89 L1339.23,1765.59 L1340.34,1764.46 L1340.74,1764.79 L1341.21,1766.06 L1341.32,1766.55 L1341.34,1767.09"
         id="path1055" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['BA'],polycolors['BA'],labels['BA'], x1, polybary['BA'], x2, polybary['BA'] )
    if 'BA' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M918.514,1872.78 L918.298,1872.99 L917.604,1873.53 L916.959,1874.31 L915.89,1874.19 L915.218,1873.96 L915.31,1873.25 L915.579,1872.53 L916.279,1871.8 L917.172,1871.48 L918.03,1872.01 L918.514,1872.78"
         id="path1057" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['MC'],polycolors['MC'],labels['MC'], x1, polybary['MC'], x2, polybary['MC'] )
    if 'MC' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1617.3,1818.73 L1623.14,1817.16 L1626.62,1816.21 L1627.87,1815.4 L1629.37,1814.79 L1630.89,1813.59 L1634.83,1812.29 L1641.99,1811.37 L1647.25,1813.15 L1648.54,1813.13 L1649.23,1813.12 L1652.85,1811.58 L1653.87,1811.59 L1656.67,1812.61 L1672.92,1813.58 L1673.37,1813.61 L1681.19,1811.7 L1682.13,1811.46 L1682.86,1811.08 L1684.99,1808.73 L1686.46,1807.92 L1687.99,1807.8 L1695.67,1809.08 L1697.75,1809.42 L1699.9,1809.31 L1709.14,1806.75 L1714.74,1807.57 L1720.58,1806.81 L1722.48,1806.88 L1724.12,1807.45 L1724.28,1807.51 L1726.04,1808.37 L1727.85,1808.98 L1729.83,1808.84 L1730.79,1808.36 L1732.47,1807.15 L1733.11,1806.89 L1733.38,1806.78 L1735.4,1806.47 L1736.39,1806.16 L1737.13,1805.55 L1737.92,1805.17 L1738.84,1804.53 L1739.67,1803.74 L1740.23,1802.93 L1740.91,1802.19 L1743.62,1801.52 L1743.91,1801.45 L1745.59,1800.26 L1746.28,1798.32 L1746.31,1798.08 L1747.47,1796.24 L1748.64,1795.19 L1750.18,1792.08 L1750.36,1791.13 L1750.64,1790.32 L1755.14,1785.24 L1755.27,1784.22 L1755.81,1782.87 L1757.06,1780.69 L1758.4,1779.53 L1761.76,1777.82 L1764.92,1775.51 L1765.85,1775.15 L1769.4,1773.79 L1777.84,1770.54 L1779.2,1769.68 L1780.06,1769.37 L1780.45,1769.16 L1781.2,1768.41 L1781.68,1768.17 L1783.65,1767.72 L1785.12,1766.88 L1788.73,1763.44 L1793.76,1761.73 L1794.73,1760.73 L1798.35,1761.98 L1803.16,1761.83 L1804.2,1762.13 L1805.25,1761.78 L1805.33,1761.81 L1806.17,1762.2 L1806.27,1763.63 L1807.42,1764.47 L1810.15,1765.07 L1810.78,1765.43 L1811.96,1766.72 L1812.51,1766.95 L1820.85,1764.88 L1823.24,1763.44 L1824.32,1763.55 L1825.61,1765.2 L1825.98,1765.51 L1828.23,1767.45 L1830.94,1766.03 L1833.57,1763.57 L1835.94,1762.69 L1835.94,1762.7 L1837.38,1764.2 L1840.98,1770.73 L1842.73,1771.54 L1852.75,1773.44 L1862.48,1772.72 L1868.7,1770.75 L1868.8,1771.61 L1870.42,1779.03 L1870.72,1779.79 L1871.42,1780.74 L1872.04,1781.36 L1872.5,1781.96 L1872.74,1782.83 L1872.72,1784.49 L1872.47,1786.61 L1872.06,1788.45 L1871.56,1789.31 L1871.16,1789.66 L1869.58,1792.14 L1869.31,1792.83 L1869.25,1793.66 L1869.47,1794.72 L1868.67,1794.04 L1866.42,1793.43 L1865.92,1793.18 L1865.29,1792.72 L1864.07,1792.54 L1861.9,1792.67 L1860.96,1793.11 L1859.5,1794.39 L1858.73,1794.78 L1855.83,1795.37 L1854.94,1795.73 L1853.47,1797.14 L1852.66,1799.12 L1852.58,1799.7 L1851.59,1806.79 L1851.22,1807.93 L1850.58,1808.7 L1847.74,1810.26 L1847.32,1810.74 L1847.04,1810.91 L1846.61,1811.02 L1846.7,1811.44 L1848.51,1812.22 L1848.91,1812.61 L1849.02,1813.34 L1848.77,1816.2 L1848.83,1819.31 L1848.65,1819.95 L1848.35,1820.45 L1848.1,1820.97 L1848.07,1821.69 L1848.46,1823.04 L1849.76,1825.66 L1850.21,1826.88 L1850.89,1831.06 L1851.35,1831.92 L1850.61,1832.82 L1850.82,1833.83 L1851.48,1835.06 L1852.09,1836.61 L1852.74,1839.52 L1852.89,1840.69 L1852.88,1841.14 L1850.62,1841.83 L1848.12,1842 L1846.13,1842.42 L1845.69,1842.63 L1845.45,1843.08 L1845.35,1844 L1845.35,1844.69 L1845.53,1845.24 L1845.95,1845.62 L1846.72,1845.82 L1845.57,1846.32 L1843.81,1847.64 L1842.84,1848.07 L1842.18,1848.98 L1842.62,1850.56 L1843.5,1852.14 L1844.12,1853.07 L1843.32,1852.9 L1839.11,1853.79 L1837.96,1854.87 L1837.65,1856.26 L1837.67,1857.84 L1837.49,1859.48 L1836.6,1859.27 L1836.3,1859.94 L1836.53,1860.85 L1837.21,1861.3 L1837.14,1861.76 L1837.16,1862.09 L1837.57,1862.97 L1837.6,1862.48 L1837.65,1862.39 L1837.62,1862.39 L1837.4,1862.17 L1837.44,1860.85 L1839.17,1862.01 L1839.66,1862.03 L1840.1,1861.38 L1840.31,1860.59 L1840.69,1860.19 L1841.64,1860.72 L1841.91,1860 L1842.31,1859.98 L1842.75,1860.24 L1843.15,1860.36 L1843.74,1860.03 L1844.12,1859.63 L1844.53,1859.36 L1845.2,1859.39 L1845.25,1859.71 L1845.21,1859.78 L1845.12,1859.79 L1844.99,1859.92 L1845.49,1860.91 L1846.1,1861.56 L1846.83,1861.54 L1847.74,1860.54 L1847.95,1861.09 L1848.27,1861.33 L1848.68,1861.31 L1849.16,1861.07 L1849.26,1861.52 L1849.02,1861.66 L1848.41,1862.17 L1848.87,1862.68 L1849.1,1862.79 L1849.54,1862.78 L1849.24,1863.48 L1849.38,1864.33 L1849.84,1865.11 L1850.49,1865.61 L1851.21,1865.74 L1851.73,1865.54 L1852.24,1865.23 L1852.95,1865.02 L1853.04,1865.4 L1852.85,1865.8 L1852.92,1866.14 L1853.2,1866.41 L1853.67,1866.64 L1852.76,1867.85 L1852.48,1869.11 L1852.94,1870.19 L1854.26,1870.85 L1854.09,1871.06 L1854.09,1871.13 L1854.05,1871.38 L1855.92,1872.05 L1856.25,1872.38 L1856.57,1872.78 L1857.21,1873.24 L1857.94,1873.63 L1858.52,1873.8 L1858.47,1874.03 L1858.46,1874.54 L1858.41,1874.75 L1859.92,1875.2 L1860.36,1875.54 L1861.52,1877.03 L1864.3,1877.99 L1864.73,1878.5 L1865.23,1878.98 L1865.91,1879.02 L1865.87,1879.74 L1866.22,1880.25 L1866.76,1880.62 L1867.3,1880.88 L1867.39,1881.18 L1867.35,1881.28 L1867.23,1881.31 L1867.09,1881.41 L1867.73,1881.84 L1868.25,1882.54 L1868.5,1883.37 L1868.37,1884.17 L1868.64,1884.75 L1868.64,1884.76 L1866.94,1884.79 L1866.21,1884.73 L1864.05,1885.53 L1863.39,1885.49 L1862.06,1885.19 L1860.88,1885.18 L1860.46,1885.26 L1859.63,1885.61 L1859.42,1885.58 L1859.24,1885.6 L1859.06,1885.67 L1858.9,1885.78 L1858.9,1886.43 L1859.08,1887.26 L1859.11,1887.88 L1859.92,1888.15 L1859.87,1888.55 L1859.35,1888.91 L1858.77,1889.08 L1858.1,1889.02 L1855.44,1888.28 L1853.76,1888.62 L1850.43,1890.39 L1850.46,1891.03 L1850.36,1891.36 L1850.13,1891.42 L1849.73,1891.24 L1849.46,1891.82 L1849.07,1892.24 L1848.17,1892.81 L1848.26,1892.94 L1848.41,1893 L1848.57,1893.05 L1848.71,1893.1 L1848.59,1893.27 L1848.34,1893.48 L1848.27,1893.65 L1848.95,1893.77 L1848.28,1894.27 L1847.59,1893.99 L1846.18,1892.71 L1845.36,1892.29 L1841.62,1891.19 L1840.32,1890.5 L1839.06,1889.55 L1836.5,1887.03 L1835,1886.09 L1833.37,1885.57 L1831.69,1885.56 L1830.73,1885.93 L1830.26,1886.52 L1829.55,1888.12 L1829.48,1888.39 L1829.49,1888.61 L1829.31,1888.74 L1828.15,1888.7 L1827.13,1888.89 L1826.63,1889.06 L1825.81,1888.65 L1824.95,1888.39 L1824.06,1888.31 L1823.26,1888.49 L1823.2,1888.5 L1822.23,1889.3 L1821.63,1891.43 L1820.88,1892.24 L1820.31,1892.66 L1820.03,1893.38 L1819.82,1894.2 L1819.44,1894.92 L1819.07,1895.12 L1818.19,1895.18 L1817.76,1895.49 L1817.37,1896.09 L1816.95,1896.18 L1816.49,1896.12 L1816,1896.25 L1815.58,1896.67 L1815.12,1897.26 L1814.74,1897.76 L1814.41,1898.04 L1813.88,1898.01 L1812.88,1897.47 L1812.41,1897.35 L1811.9,1897.64 L1811.13,1898.86 L1810.73,1899.3 L1809.88,1899.6 L1805.45,1899.85 L1804.66,1900.13 L1804.03,1900.83 L1803.01,1902.53 L1802.73,1902.86 L1802.51,1903.21 L1802.4,1903.65 L1802.53,1903.87 L1803.15,1904.1 L1803.33,1904.27 L1803.37,1904.63 L1803.28,1905.21 L1803.3,1905.54 L1803.34,1905.92 L1803.39,1906.36 L1803.43,1907.52 L1803.28,1908.72 L1802.84,1909.7 L1800.8,1910.86 L1796.07,1911.7 L1794.52,1913.77 L1794.17,1915.39 L1794.27,1916.75 L1794.84,1917.91 L1795.43,1918.46 L1793.71,1919.01 L1792.69,1918.94 L1792,1918.58 L1790.5,1917.45 L1790.08,1917.3 L1789.41,1917.39 L1788.6,1918.58 L1786.11,1919.21 L1784.95,1919.81 L1783.93,1921.08 L1783.61,1921.3 L1783.3,1921.38 L1783.02,1921.55 L1782.77,1921.97 L1782.67,1922.78 L1782.82,1923.68 L1783.12,1924.51 L1783.48,1925.18 L1783.75,1925.44 L1784.19,1925.88 L1784.9,1926.04 L1785.57,1926.03 L1786.15,1926.17 L1786.64,1926.55 L1787,1927.01 L1787.74,1928.48 L1787.86,1928.93 L1787.99,1929.91 L1788.15,1930.3 L1788.45,1930.46 L1789.2,1930.44 L1789.51,1930.67 L1789.9,1931.42 L1790.11,1932.11 L1790.17,1932.81 L1790.16,1933.62 L1789.74,1934.29 L1789.74,1934.72 L1790.43,1935.7 L1790.68,1935.93 L1791.44,1936.4 L1791.65,1936.74 L1791.67,1937.25 L1791.47,1937.44 L1791.2,1937.61 L1791.03,1938.03 L1790.82,1939.67 L1790.42,1941.37 L1790.06,1942.33 L1789.81,1942.55 L1789.45,1942.54 L1785.7,1944.31 L1785.16,1944.73 L1784.1,1945.83 L1783.15,1946.56 L1782.65,1946.73 L1782.01,1946.74 L1781.37,1946.88 L1780.34,1947.72 L1779.71,1948 L1778.74,1947.82 L1777.09,1946.6 L1775.99,1946.43 L1775.55,1946.72 L1774.06,1948.25 L1772.48,1948.71 L1772,1949 L1771.51,1949.54 L1771.25,1949.93 L1770.97,1950.22 L1770.47,1950.4 L1770,1950.37 L1768.93,1950.01 L1768.45,1949.94 L1764.32,1950.52 L1763.74,1950.86 L1763.49,1951.52 L1763.32,1952.26 L1763.01,1952.84 L1762.59,1953.12 L1762.22,1953.15 L1761.84,1953.11 L1761.35,1953.17 L1760.21,1953.62 L1752.93,1957.74 L1751.86,1958.05 L1750.76,1958.09 L1749.76,1957.74 L1747.28,1955.44 L1746.25,1954.87 L1746.05,1954.82 L1744.17,1954.36 L1743.95,1954.19 L1743.48,1953.65 L1743.31,1953.52 L1743.08,1953.57 L1742.35,1953.88 L1734.21,1952.2 L1732.63,1951.61 L1732,1951.66 L1731.56,1951.84 L1730.65,1952.38 L1728.88,1952.86 L1728.89,1953.71 L1729.21,1954.75 L1728.98,1955.72 L1728.05,1955.87 L1726.88,1955.17 L1724.94,1953.45 L1723.89,1952.82 L1722.97,1952.59 L1722.04,1952.63 L1720.98,1952.84 L1721.15,1952.17 L1719.54,1952.52 L1718.89,1952.48 L1718.05,1951.96 L1717.98,1951.75 L1717.97,1951.05 L1717.87,1950.75 L1717.65,1950.58 L1717.16,1950.39 L1716.96,1950.24 L1716.65,1949.93 L1716.37,1949.74 L1716.13,1949.49 L1715.9,1949.01 L1715.45,1947.34 L1714.59,1945.84 L1713.47,1945.17 L1712.24,1945.94 L1711.26,1946.37 L1710.56,1947.89 L1709.9,1948.18 L1708.89,1948.2 L1708.2,1948.41 L1706.73,1949.18 L1706.33,1949.3 L1705.06,1949.39 L1704.67,1949.53 L1704.4,1949.76 L1704.1,1949.93 L1703.65,1949.87 L1703.51,1949.54 L1703.52,1949 L1703.38,1948.47 L1702.38,1948.07 L1701.49,1947.37 L1700.7,1947.63 L1699.89,1948.16 L1699.21,1948.81 L1699.15,1948.86 L1698.58,1949.58 L1698.45,1950.01 L1698.42,1951.01 L1698.33,1951.27 L1697.98,1951.3 L1697.73,1951.08 L1697.47,1950.76 L1697.06,1950.55 L1695.61,1950.4 L1693.74,1950.59 L1692.5,1951.48 L1692.96,1953.42 L1693.32,1954.76 L1693.38,1955.29 L1693.18,1955.81 L1692.71,1956.16 L1692.14,1956.15 L1691,1955.67 L1690.91,1956.14 L1690.72,1956.32 L1690.47,1956.43 L1690.24,1956.65 L1689.61,1957.56 L1688.93,1957.74 L1686.48,1956.55 L1686.1,1956.57 L1685.04,1957.96 L1684.36,1958.47 L1683.44,1958.89 L1682.48,1959.18 L1681.66,1959.29 L1681.02,1959.75 L1680.27,1961.69 L1679.55,1962.03 L1677.97,1961.96 L1676.43,1962.25 L1675.59,1962.74 L1674.63,1964.13 L1674.51,1964.24 L1673.99,1964.72 L1672.43,1964.95 L1669.11,1963.92 L1664.52,1964.61 L1663.68,1965.06 L1662.44,1966.34 L1661.69,1966.94 L1660.72,1967.23 L1660.16,1966.85 L1659.61,1966.22 L1658.65,1965.77 L1657.84,1965.98 L1656.83,1966.63 L1655.9,1967.47 L1655.33,1968.24 L1655.25,1968.73 L1655.36,1969.87 L1655.22,1970.51 L1654.87,1971.01 L1654.41,1971.39 L1653.41,1971.88 L1651.5,1972.45 L1642.06,1972.64 L1643.03,1971.58 L1643.01,1970.41 L1642.57,1969.16 L1642.27,1967.86 L1642.35,1967.45 L1642.79,1966.7 L1642.85,1966.39 L1642.74,1966.04 L1642.37,1965.46 L1642.26,1965.17 L1641.45,1960.89 L1641.27,1959.95 L1641.34,1958.89 L1641.32,1958.52 L1641.14,1958.12 L1640.59,1957.51 L1640.39,1957.19 L1640.18,1956.38 L1640.08,1955.49 L1640.09,1954.61 L1640.2,1953.86 L1640.39,1953.61 L1641.09,1953.28 L1641.31,1953.08 L1641.41,1952.74 L1641.57,1951.8 L1641.87,1951.07 L1642.32,1949.99 L1642.6,1948.47 L1642.33,1947.02 L1641.3,1945.87 L1640.78,1945.72 L1639.67,1945.86 L1639.13,1945.72 L1638.77,1945.31 L1637.41,1943.17 L1636.59,1941.14 L1636.18,1940.47 L1635.92,1940.26 L1635.34,1940.06 L1635.18,1939.93 L1635.11,1939.87 L1634.92,1939.43 L1634.79,1938.52 L1634.65,1938.1 L1633.93,1936.78 L1633.36,1935.42 L1633.04,1933.96 L1632.83,1933.31 L1632.36,1932.71 L1632.35,1932.7 L1632.18,1931.89 L1632.17,1931.86 L1632.01,1931.44 L1631.71,1931.17 L1631.13,1930.92 L1630.87,1930.95 L1630.21,1931.31 L1629.88,1931.38 L1629.48,1931.16 L1629.25,1930.77 L1629.08,1930.38 L1628.84,1930.17 L1628.36,1930.2 L1626.28,1930.65 L1625.95,1930.58 L1625.71,1930.36 L1625.5,1930.12 L1625.26,1929.97 L1623.84,1929.92 L1621.42,1929.16 L1620.96,1929.01 L1616.53,1926.83 L1615.42,1926.02 L1615.23,1925.79 L1614.52,1924.92 L1613.63,1923.25 L1611.7,1922.24 L1606.23,1916.89 L1607.04,1916.28 L1608.94,1915.93 L1609.73,1915.52 L1610.26,1914.54 L1610.64,1912.15 L1611.2,1911.12 L1611.84,1910.66 L1612.4,1910.44 L1612.86,1910 L1613.19,1908.93 L1613.46,1906.57 L1613.41,1905.26 L1613.09,1904.37 L1613.08,1904.36 L1612.6,1903.55 L1611.91,1902.93 L1610.35,1902.16 L1607.64,1900.35 L1607.43,1900.29 L1607.4,1899.03 L1607.58,1896.47 L1607.63,1895.8 L1607.61,1894.24 L1607.15,1893.45 L1607.98,1891 L1608.5,1890.32 L1608.52,1890.12 L1608.51,1889.93 L1608.46,1889.74 L1608.38,1889.57 L1607.62,1889.15 L1606.87,1888.32 L1605.38,1885.88 L1605.18,1885.67 L1605.22,1885.43 L1605.32,1885.19 L1605.54,1884.7 L1605.88,1884.27 L1606.94,1883.47 L1607.35,1882.98 L1607.89,1881.77 L1608.24,1881.35 L1608.86,1881.23 L1609.69,1881.22 L1609.97,1880.98 L1609.98,1880.95 L1610.16,1880.57 L1610.72,1880.03 L1611.93,1879.66 L1615.49,1880 L1616.81,1879.43 L1618.1,1878.61 L1618.53,1877.87 L1618.55,1877.78 L1618.71,1877.04 L1619.03,1873.91 L1619.19,1873.08 L1619.42,1872.51 L1619.9,1872.1 L1621.1,1871.58 L1621.66,1871.25 L1621.66,1871.25 L1621.66,1871.25 L1621.66,1871.24 L1622.12,1870.28 L1623.73,1868.15 L1623.89,1867.64 L1624.03,1866.45 L1624.19,1865.98 L1624.52,1865.57 L1625.22,1865.08 L1625.55,1864.74 L1626.24,1863.18 L1626.74,1861.01 L1626.88,1858.86 L1626.67,1858.09 L1626.49,1857.39 L1625.68,1857.18 L1623.39,1857.09 L1622.5,1856.73 L1621.77,1856.21 L1620.32,1854.79 L1619.05,1853.89 L1618.71,1853.52 L1618.49,1853.06 L1618.17,1851.96 L1617.92,1851.5 L1617.22,1850.76 L1613.48,1848.08 L1613.05,1847.87 L1612.8,1847.74 L1612.01,1847.54 L1611.56,1847.56 L1610.72,1847.75 L1610.26,1847.67 L1609.96,1847.43 L1609.83,1847.33 L1609.16,1846.36 L1608.76,1846.05 L1608.33,1846 L1607.37,1846.22 L1606.87,1846.19 L1606.39,1845.94 L1605.63,1845.2 L1605.26,1844.93 L1603.66,1844.48 L1602.95,1844 L1602.34,1842.89 L1601.04,1840.1 L1600.31,1839.07 L1600.23,1838.77 L1600.22,1838.45 L1600.3,1838.12 L1600.33,1837.96 L1600.33,1837.79 L1600.28,1837.65 L1600.2,1837.51 L1600.02,1837.32 L1599.94,1837.1 L1599.95,1836.87 L1600.06,1836.62 L1600.07,1836.59 L1600.08,1836.56 L1600.06,1836.53 L1600.04,1836.49 L1599.58,1835.78 L1599.32,1834.38 L1598.89,1833.61 L1598.36,1833.24 L1596.92,1832.73 L1596.28,1832.31 L1595.77,1831.73 L1595.3,1831.03 L1594.89,1830.28 L1594.56,1829.5 L1594.47,1828.99 L1594.48,1827.93 L1594.35,1827.46 L1592.98,1826.26 L1592.13,1824.69 L1592.15,1823.31 L1592.5,1821.79 L1592.63,1819.82 L1592.43,1817.96 L1592.51,1817.62 L1592.82,1816.73 L1592.89,1816.43 L1592.84,1815.45 L1592.59,1813.66 L1592.57,1812.88 L1592.98,1811.95 L1593.91,1811.35 L1595.28,1810.88 L1595.95,1810.66 L1596.94,1810.46 L1597.34,1809.73 L1597.54,1808.79 L1597.93,1807.93 L1598.79,1807.45 L1599.71,1807.36 L1600.47,1807.07 L1600.84,1806.03 L1600.75,1805.49 L1600.26,1804.23 L1600.11,1803.64 L1600.09,1802.93 L1600.25,1801.02 L1600.19,1801.03 L1600.07,1800.85 L1599.97,1800.53 L1599.97,1800.15 L1600.06,1800.05 L1600.64,1799.09 L1601.14,1797.87 L1601.46,1797.39 L1603.22,1796.17 L1613.8,1801.03 L1615.52,1801.46 L1617.58,1801.38 L1618.55,1801.62 L1619.63,1801.9 L1620.34,1803.73 L1619.9,1805.75 L1618.49,1806.88 L1617.54,1807.18 L1615.87,1808.07 L1614.98,1808.37 L1614.18,1809.11 L1613.92,1810.58 L1613.6,1815.39 L1613.85,1816.78 L1614.59,1817.86 L1615.88,1818.64 L1617.3,1818.73"
         id="path1059" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['BG'],polycolors['BG'],labels['BG'], x1, polybary['BG'], x2, polybary['BG'] )
    if 'BG' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1363.17,1191.69 L1376.49,1187.92 L1379.29,1186.07 L1380.67,1185.55 L1381.75,1184.87 L1383.65,1182.25 L1384.23,1181.89 L1385.08,1182.42 L1386.96,1182.8 L1389.79,1182.56 L1397.3,1181.94 L1411.39,1180.7 L1421.89,1179.73 L1432.39,1178.72 L1445.39,1177.4 L1461.14,1175.72 L1471.24,1174.59 L1481.33,1173.42 L1489.8,1172.42 L1496.62,1171.58 L1498.94,1170.32 L1499.52,1169.64 L1500.12,1167.79 L1500.15,1167.69 L1500.97,1167.11 L1501.75,1167.08 L1504.1,1167.8 L1504.5,1167.78 L1505.29,1167.46 L1505.67,1167.48 L1506.12,1167.8 L1506.42,1168.25 L1506.66,1168.7 L1506.96,1169.05 L1507.42,1169.26 L1508.68,1169.18 L1508.97,1169.48 L1508.95,1170.09 L1508.87,1170.82 L1508.98,1171.48 L1509.8,1172.27 L1510.62,1172.11 L1511.41,1171.59 L1512.11,1171.29 L1512.82,1171.49 L1515.4,1172.96 L1516.93,1173.55 L1520.03,1174.11 L1520.87,1174.59 L1521.64,1175.04 L1522.09,1175.51 L1522.98,1176.65 L1523.41,1177.06 L1523.83,1177.25 L1524.64,1177.32 L1525.01,1177.45 L1525.77,1178.23 L1526.51,1179.36 L1527.17,1180.67 L1527.7,1181.93 L1528.77,1184.71 L1529.05,1186.1 L1528.69,1187.14 L1528.33,1187.59 L1528.18,1187.87 L1528.06,1188.18 L1528.16,1188.84 L1528.42,1189.37 L1528.8,1189.77 L1529.24,1190.01 L1529.51,1190.25 L1529.67,1190.54 L1529.71,1190.88 L1529.63,1191.27 L1530.02,1193.07 L1530.69,1194.5 L1531.44,1195.83 L1532.11,1197.31 L1533.07,1200.53 L1533.69,1201.72 L1534.82,1202.85 L1534.83,1202.86 L1534.83,1202.86 L1535.64,1206.63 L1537.33,1210.76 L1542.28,1219.75 L1544.7,1223.03 L1545.81,1224.84 L1548.41,1230.38 L1549.4,1232.01 L1550.23,1232.77 L1550.76,1233.57 L1551.22,1234.42 L1551.85,1235.28 L1552.65,1235.88 L1553.37,1236.25 L1553.91,1236.92 L1554.12,1238.43 L1553.96,1239.38 L1553.66,1240.21 L1553.47,1241.13 L1553.65,1242.34 L1554.17,1243.32 L1555.57,1244.55 L1556.17,1245.38 L1556.33,1245.84 L1556.38,1246.6 L1556.46,1246.99 L1557.11,1248.64 L1557.22,1249.14 L1557.19,1249.54 L1556.94,1250.45 L1556.93,1250.95 L1557.1,1251.61 L1557.29,1251.95 L1557.5,1252.23 L1557.69,1252.69 L1559.46,1260.09 L1559.9,1261.93 L1559.9,1264.66 L1558.77,1266.78 L1554.48,1271.14 L1548.58,1274.11 L1545.64,1276.68 L1542.86,1280.06 L1539.61,1287.97 L1538.45,1290.1 L1537.95,1291.32 L1536.87,1295.22 L1536.99,1295.23 L1537.31,1295.41 L1537.63,1295.6 L1538.16,1296.13 L1538.43,1296.92 L1538.33,1298.07 L1539.28,1298.46 L1542.12,1298.7 L1542.66,1298.36 L1543.25,1298.77 L1545.76,1299.22 L1546.56,1299.16 L1546.63,1299.54 L1546.46,1300.16 L1546.59,1300.24 L1547.09,1299.93 L1547.6,1300.05 L1548.01,1300.34 L1548.38,1300.34 L1548.78,1299.61 L1548.98,1299.86 L1549.41,1300.11 L1549.72,1300.35 L1550.23,1299.62 L1550.46,1300.05 L1550.5,1300.81 L1550.39,1301.05 L1551.83,1302.96 L1552.61,1303.06 L1554.39,1302.67 L1555.18,1302.96 L1555.52,1303.4 L1555.81,1303.86 L1556.11,1304.21 L1556.49,1304.3 L1557.01,1304.54 L1557.3,1305.22 L1557.48,1305.92 L1557.93,1306.63 L1558.91,1308.86 L1559.54,1309.83 L1559.03,1310.38 L1558.69,1311.27 L1558.29,1313.11 L1558.26,1314.29 L1558.61,1316.3 L1558.5,1317.04 L1558.66,1317.07 L1558.72,1317.06 L1558.77,1317.13 L1558.84,1317.41 L1558.43,1317.75 L1558.53,1319.12 L1558.2,1319.71 L1559.01,1320.72 L1559.89,1321.53 L1559.73,1323.05 L1559.14,1323.92 L1558.43,1324.56 L1557.93,1325.43 L1557.86,1325.9 L1557.83,1326.59 L1557.88,1328.05 L1558.08,1328.35 L1558.42,1328.73 L1558.57,1329.1 L1558.26,1329.33 L1558.14,1329.53 L1558.33,1329.91 L1558.65,1330.29 L1558.89,1330.52 L1558.51,1331.22 L1558.31,1331.41 L1557.97,1331.52 L1558.06,1331.98 L1558.58,1332.38 L1558.74,1332.96 L1558.86,1334.67 L1559.16,1335.53 L1559.65,1336.32 L1560.76,1337.62 L1560.5,1337.69 L1560.24,1337.71 L1560.89,1338.59 L1561.54,1338.71 L1562.12,1338.92 L1562.43,1339.72 L1562.57,1340.09 L1562.99,1340.9 L1563.67,1341.21 L1564.41,1341.33 L1565.02,1341.6 L1564.66,1342.95 L1564.79,1343.32 L1566.79,1345.34 L1567.24,1346.02 L1566.84,1346.31 L1566.64,1346.79 L1566.72,1347.44 L1567.11,1348.22 L1566.46,1348.63 L1565.92,1349.49 L1565.56,1350.5 L1565.51,1351.36 L1566,1352.62 L1566.67,1352.82 L1567.41,1352.69 L1568.15,1353 L1571.17,1357.29 L1572.24,1358.18 L1574.43,1359.05 L1576.62,1360.58 L1577.19,1361.25 L1576.62,1361.7 L1576.58,1362.3 L1577.39,1363.41 L1578.7,1364.79 L1579.2,1365.55 L1580.14,1368.04 L1580.22,1368.42 L1580.15,1368.94 L1580.48,1369.14 L1581.01,1369.24 L1581.54,1369.47 L1582.09,1370.05 L1582.3,1370.53 L1582.45,1371.08 L1582.81,1371.85 L1583.58,1372.7 L1586.66,1374.62 L1590.24,1375.78 L1590.89,1376.46 L1590.62,1377.62 L1589.49,1378.1 L1588.2,1378.36 L1587.48,1378.87 L1586.34,1378.49 L1585.3,1378.84 L1584.54,1379.69 L1584.27,1380.84 L1584.57,1382.01 L1585.29,1382.68 L1586.29,1382.93 L1587.4,1382.85 L1587.47,1383.23 L1587.18,1384.41 L1587.9,1385.25 L1589.09,1385.71 L1590.19,1385.76 L1590.22,1387.21 L1590.78,1388.45 L1592.23,1390.65 L1592.24,1390.67 L1591.64,1392.42 L1592.6,1395.27 L1593.04,1396.03 L1593.27,1396.16 L1593.25,1396.31 L1592.9,1397.11 L1592.31,1398.05 L1590.05,1399.86 L1590.09,1400.62 L1590.38,1401.8 L1590.46,1402.61 L1590.41,1403.31 L1589.94,1405.51 L1588.05,1406.76 L1580.97,1408.22 L1579.7,1408.9 L1579.08,1409.38 L1578.65,1410.01 L1578.39,1410.99 L1578.2,1412.72 L1578.19,1412.81 L1577.8,1413.71 L1575.33,1417.68 L1574.36,1418.82 L1574.36,1418.82 L1574.35,1418.82 L1574.35,1418.82 L1572.51,1420.88 L1570.99,1422.6 L1563.75,1434.1 L1562.9,1436.17 L1561.65,1437.57 L1560.39,1439.61 L1557.21,1446.72 L1555.85,1448.74 L1555.73,1449.29 L1555.62,1450.59 L1555.51,1451.06 L1555.23,1451.47 L1554.57,1451.86 L1554.26,1452.15 L1553.96,1452.7 L1552.28,1457.6 L1551.73,1458.4 L1551.33,1458.67 L1550.5,1458.99 L1550.12,1459.46 L1549.92,1460.04 L1549.58,1462.14 L1547.25,1466.76 L1546.66,1469.32 L1547.81,1471.4 L1548.23,1472.25 L1548.93,1474.84 L1549.2,1475.51 L1549.37,1475.83 L1550.03,1476.57 L1551.06,1477.34 L1551.36,1477.77 L1551.71,1478.72 L1553.23,1484.29 L1553.23,1485.22 L1552.87,1487.21 L1552.67,1489.5 L1552.27,1490.95 L1552.18,1491.74 L1552.48,1492.02 L1552.65,1491.87 L1553.06,1491.1 L1553.77,1491.43 L1553.73,1491.47 L1553.58,1491.53 L1555.05,1492.23 L1556.5,1493.58 L1557.33,1494 L1559.32,1494.7 L1559.91,1495.24 L1560.01,1495.76 L1559.85,1497.04 L1559.95,1497.89 L1560.22,1498.44 L1561.07,1499.45 L1561.15,1499.58 L1561.26,1499.76 L1560.96,1500.83 L1560.08,1500.62 L1559.04,1499.97 L1558.26,1499.72 L1557.78,1499.65 L1556.86,1498.73 L1556.38,1498.41 L1555.93,1498.44 L1555.07,1498.88 L1553.59,1499.02 L1552.74,1499.26 L1551.85,1499.3 L1550.74,1498.79 L1548.94,1497.38 L1548.09,1497.26 L1547.37,1498.23 L1545.86,1497.78 L1542.67,1498.19 L1541.11,1497.98 L1538.72,1496.27 L1537.81,1496.07 L1535.59,1496.53 L1533.6,1496.27 L1533.25,1496.06 L1533.1,1495.69 L1533.14,1495.27 L1533.07,1494.84 L1532.55,1494.42 L1532.2,1494.41 L1531.25,1494.69 L1530.84,1494.73 L1530.34,1494.6 L1528.89,1493.97 L1525.94,1493.9 L1524.66,1493.25 L1524.08,1491.32 L1523.22,1489.22 L1521.76,1487.54 L1520.08,1486.4 L1517.74,1485.67 L1516.05,1484.54 L1515.24,1484.24 L1514.46,1484.52 L1513.86,1485.28 L1513.4,1486.05 L1513.02,1486.4 L1512.32,1485.99 L1510.64,1484.12 L1509.85,1483.53 L1509.44,1483.48 L1508.83,1483.72 L1508.52,1483.73 L1508.07,1483.51 L1507.21,1482.89 L1506.76,1482.68 L1505.98,1482.61 L1503.15,1483.39 L1502.57,1483.74 L1501.89,1484.15 L1501.27,1484.07 L1499.83,1484.64 L1499.16,1484.75 L1495.07,1484.25 L1492.63,1483.39 L1492.05,1483.34 L1491.4,1483.96 L1490.43,1486.02 L1489.84,1486.78 L1488.99,1487.06 L1488.31,1486.73 L1487.64,1486.17 L1486.84,1485.78 L1486.2,1485.82 L1484.62,1486.4 L1484.04,1486.79 L1483.36,1487.83 L1483.94,1488.32 L1484.95,1488.74 L1485.58,1489.6 L1485.34,1490.22 L1484.62,1490.48 L1483.77,1490.62 L1483.15,1490.85 L1482.44,1491.79 L1481.41,1493.92 L1480.65,1494.81 L1479.75,1495.3 L1478.99,1495.27 L1478.26,1494.89 L1477.5,1494.3 L1476.67,1493.85 L1476.02,1493.85 L1475.33,1493.99 L1474.41,1493.99 L1473.7,1493.61 L1469.5,1489.84 L1468.85,1489.82 L1467.36,1490.29 L1466.47,1490.29 L1466.11,1490.37 L1465.73,1490.66 L1465.2,1491.56 L1464.8,1492.06 L1463.89,1492.54 L1462.97,1492.45 L1459.32,1491.19 L1458.71,1491.43 L1458.86,1492.49 L1456.79,1492.87 L1455.05,1492.48 L1454.57,1492.56 L1454.2,1492.93 L1454.1,1493.33 L1454.1,1493.92 L1454.01,1494.83 L1453.84,1495.73 L1453.68,1496.03 L1450.6,1496.75 L1450.03,1497.18 L1449.29,1498.36 L1448.95,1498.79 L1448.04,1498.8 L1447.93,1498.72 L1447.74,1499.06 L1447.2,1500.91 L1447.07,1501.68 L1446.89,1502.41 L1446.47,1503.09 L1446.53,1505.29 L1446.32,1506.9 L1445.61,1507.62 L1444.18,1507.14 L1441.82,1505.46 L1440.61,1505.02 L1439.34,1505.34 L1438.67,1505.99 L1437.99,1506.92 L1437.5,1507.58 L1436.64,1508.04 L1434.75,1508.15 L1433.68,1507.91 L1433.06,1507.26 L1433.06,1507.25 L1433.12,1506.42 L1434.32,1503.7 L1434.48,1503.53 L1434.73,1503.42 L1434.95,1503.25 L1435,1502.89 L1434.89,1502.63 L1434.87,1502.62 L1434.52,1502.33 L1434.51,1502.31 L1434.42,1502.18 L1434.42,1502.18 L1433.81,1501.84 L1433.64,1501.65 L1433.64,1501.65 L1433.63,1501.39 L1433.88,1500.88 L1433.91,1500.62 L1433.28,1497.65 L1432.94,1496.64 L1432.41,1495.52 L1432.4,1495.51 L1432,1495.28 L1430.72,1496.01 L1429.91,1496.2 L1429.01,1496.22 L1426.59,1495.74 L1426.59,1495.72 L1426.58,1494.98 L1426.43,1494.01 L1426.45,1493.66 L1426.58,1493.25 L1424.97,1493.44 L1424.09,1493.33 L1423.33,1492.88 L1423.07,1492.47 L1422.16,1490.57 L1421.03,1487.54 L1420.38,1486.35 L1419.35,1485.85 L1419.01,1485.56 L1418.18,1484.45 L1417.82,1484.35 L1417.58,1484.29 L1417.36,1484.42 L1417.25,1484.75 L1414.24,1489.06 L1413.97,1489.37 L1413.43,1489.67 L1413.02,1489.79 L1411.78,1489.94 L1411.02,1490.2 L1410.4,1490.61 L1409.88,1491.25 L1409.44,1492.2 L1409.05,1494.3 L1409.13,1494.36 L1409.28,1494.53 L1409.39,1494.81 L1409.35,1495.24 L1409.19,1495.4 L1408.69,1495.47 L1408.58,1495.51 L1408.44,1496.76 L1408.43,1497.55 L1408.2,1498.08 L1407.02,1498.74 L1406.03,1499.07 L1405.6,1499.09 L1405.19,1498.94 L1404.32,1498.45 L1403.93,1498.37 L1403.07,1498.69 L1401.62,1499.79 L1400.6,1500 L1399.79,1499.95 L1399.48,1499.28 L1399.4,1497.21 L1399.19,1496.31 L1398.89,1495.49 L1398.74,1494.61 L1398.97,1493.52 L1397.72,1492.94 L1393.67,1493.06 L1393.72,1492 L1393.45,1490.73 L1392.56,1488.15 L1392.31,1487.61 L1391.71,1486.58 L1391.45,1486.01 L1390.92,1483.9 L1390.7,1483.43 L1390.04,1483.06 L1389.43,1483.24 L1388.83,1483.59 L1388.25,1483.71 L1387.71,1483.46 L1386.81,1482.62 L1386.23,1482.26 L1385.64,1482.15 L1384.38,1482.19 L1383.8,1481.88 L1383.5,1481.44 L1382.88,1479.68 L1381.71,1477.84 L1380.79,1476.03 L1380.58,1474.36 L1381.54,1472.95 L1380.41,1472.5 L1380.14,1471.76 L1380.13,1470.79 L1379.75,1469.69 L1379.12,1469.4 L1377.66,1470.6 L1376.73,1470.71 L1373.56,1469.41 L1371.95,1469.03 L1370.35,1469.42 L1370.44,1469.41 L1370.52,1469.43 L1370.58,1469.5 L1370.63,1469.6 L1369.62,1470.71 L1369.05,1470.89 L1368.19,1470.33 L1367.91,1469.93 L1367.53,1469 L1367.32,1468.68 L1366.81,1468.38 L1365.84,1468.29 L1365.35,1468.13 L1363.97,1467.18 L1363.32,1466.91 L1360.71,1466.71 L1360.12,1466.16 L1360.26,1464.99 L1357.75,1463.71 L1357.17,1463.63 L1356.49,1463.54 L1356.82,1464.93 L1357.04,1465 L1357.34,1464.86 L1357.69,1464.83 L1358.02,1465.23 L1358.16,1465.85 L1357.98,1466.14 L1356.72,1466.67 L1355.12,1466.9 L1354.41,1467.17 L1353.34,1468.35 L1351.93,1468.95 L1350.44,1468.8 L1347.57,1466.49 L1347.06,1465.92 L1346.66,1465.21 L1346.36,1464.46 L1346.02,1463.8 L1345.52,1463.32 L1345.1,1462.98 L1344.94,1462.64 L1345.03,1462.28 L1345.37,1461.92 L1345.38,1461.88 L1345.39,1461.83 L1345.37,1461.77 L1345.35,1461.71 L1344.78,1461.65 L1343.63,1461.23 L1343.09,1461.14 L1342.68,1461.28 L1342.36,1461.52 L1341.96,1461.63 L1341.31,1461.41 L1339.11,1459.16 L1338.98,1458.71 L1339.21,1458.05 L1339.63,1457.78 L1341.55,1457.51 L1342.57,1456.98 L1344.16,1455.78 L1344.9,1454.99 L1345.17,1454.01 L1344.43,1452.92 L1343.97,1452.55 L1343.81,1451.93 L1343.83,1451.36 L1344.14,1450.14 L1344.21,1449.94 L1344.22,1449.75 L1344.18,1449.58 L1344.07,1449.45 L1343.23,1448.54 L1342.99,1448.35 L1342.29,1448.17 L1342.04,1448.37 L1341.99,1448.8 L1341.82,1449.34 L1340.8,1451.1 L1340.18,1451.73 L1339.28,1452.07 L1335.65,1451.81 L1333.77,1452.17 L1332.75,1453.72 L1332.55,1453.88 L1332.35,1453.94 L1332.13,1453.92 L1331.91,1453.8 L1331.43,1452.68 L1330.49,1452.06 L1329.61,1452.22 L1329.29,1453.4 L1328.72,1453.05 L1328.38,1452.44 L1328.22,1451.66 L1328.19,1450.85 L1328,1450.71 L1327.81,1450.54 L1327.65,1450.35 L1327.52,1450.12 L1328.22,1449.75 L1327.36,1449.48 L1325.5,1449.73 L1324.53,1449.55 L1323.04,1448.8 L1322.6,1448.41 L1322.31,1447.89 L1321.99,1447.11 L1321.79,1446.45 L1321.85,1446.27 L1321.38,1446.17 L1320.19,1446.54 L1319.64,1446.59 L1319.1,1446.37 L1318.1,1445.68 L1317.64,1445.45 L1313.45,1444.7 L1312.54,1444.9 L1311.92,1444.6 L1310.73,1444.03 L1309.81,1444 L1308.79,1444.76 L1308.88,1445.63 L1310.33,1447.4 L1310.89,1448.49 L1311.31,1449.79 L1311.81,1450.96 L1312.63,1451.66 L1313.66,1452.14 L1314.46,1452.87 L1314.98,1453.95 L1315.18,1455.45 L1315.77,1456.09 L1315.93,1456.91 L1315.72,1457.46 L1315.22,1457.32 L1314.73,1456.73 L1314.51,1456.37 L1314.26,1456.23 L1313.67,1456.27 L1313.58,1456.46 L1313.66,1457.21 L1313.58,1457.45 L1313.22,1457.51 L1312.46,1457.31 L1312.11,1457.36 L1311.66,1457.74 L1311.08,1458.72 L1310.74,1459.09 L1310.3,1459.25 L1309.83,1459.42 L1309.02,1459.44 L1308.24,1459.63 L1307.41,1460.48 L1306.79,1461.64 L1306.44,1462.52 L1306.01,1463.26 L1305.18,1463.99 L1304.1,1465.78 L1302.48,1466.05 L1300.73,1465.33 L1299.23,1464.18 L1298.45,1463.33 L1298,1462.58 L1297.36,1460.49 L1296.82,1459.35 L1295.5,1458.2 L1294.91,1457.33 L1294.27,1456.59 L1293.09,1454.83 L1292.45,1454.07 L1291.51,1453.53 L1289.67,1453.03 L1288.88,1452.15 L1288.73,1451.62 L1288.72,1451.05 L1288.64,1450.47 L1288.31,1449.91 L1287.92,1449.83 L1286.1,1449.98 L1285.65,1450.17 L1285.39,1450.45 L1285.1,1450.54 L1284.55,1450.17 L1284.34,1449.82 L1283.68,1448.26 L1282.37,1448.14 L1281.91,1447.06 L1282.13,1446.04 L1282.94,1446.15 L1282.85,1445.97 L1282.6,1445.29 L1282.84,1445.25 L1283.19,1444.99 L1283.4,1444.88 L1283.97,1444.32 L1284.63,1443.84 L1285.18,1443.28 L1285.42,1442.45 L1286.03,1442.19 L1287.33,1442.43 L1288,1442.22 L1288.33,1441.66 L1289.08,1440.78 L1289.38,1440.28 L1289.27,1440.32 L1289.23,1439.96 L1289.25,1439.47 L1289.28,1439.15 L1289.42,1438.77 L1289.52,1438.59 L1289.5,1438.38 L1289.31,1437.93 L1290.51,1437.3 L1290.04,1436.15 L1286.36,1432.86 L1285.08,1432.28 L1283.76,1432.06 L1282.32,1432.15 L1281.82,1432.3 L1281.58,1432.95 L1281.36,1433.77 L1280.93,1434.43 L1280.28,1434.64 L1279.65,1434.4 L1278.32,1433.55 L1276.63,1433.52 L1274.37,1436.1 L1272.71,1436.55 L1272.61,1436.37 L1272.57,1436.18 L1272.59,1435.97 L1272.65,1435.76 L1273.18,1435.25 L1273.21,1434.51 L1272.92,1433.66 L1272.51,1432.85 L1271.91,1431.91 L1271.34,1431.44 L1270.68,1431.4 L1268.4,1432.47 L1267.31,1432.59 L1267.05,1432.52 L1264.78,1429.27 L1264.62,1428.9 L1264.5,1428.48 L1263.51,1428.47 L1260.29,1429.52 L1259.86,1429.42 L1259.55,1429.16 L1258.96,1428.4 L1258.49,1428.13 L1250.37,1425.96 L1248.74,1426.22 L1248.4,1426.71 L1248.07,1427.4 L1247.67,1427.89 L1247.6,1427.87 L1247.1,1427.75 L1246.81,1427.26 L1246.8,1426.66 L1246.86,1426.03 L1246.8,1425.42 L1246.25,1424.29 L1245.61,1423.83 L1244.87,1423.57 L1244.04,1422.99 L1243.57,1422.35 L1243.03,1421.35 L1242.59,1420.27 L1242.4,1419.4 L1242.6,1418.23 L1242.89,1417.55 L1242.89,1416.94 L1242.19,1415.96 L1241.1,1415.52 L1238.86,1415.64 L1238.42,1414.75 L1238.17,1413.82 L1237.69,1413.62 L1237.21,1414.01 L1236.91,1414.85 L1236.91,1414.86 L1236.83,1415.5 L1236.39,1415.62 L1235.82,1415.39 L1235.32,1414.96 L1234.95,1414.37 L1232.92,1414.07 L1232.13,1413.43 L1232.05,1413.95 L1231.89,1414.25 L1231.35,1414.71 L1231.13,1414.77 L1230.91,1414.89 L1230.72,1415.07 L1230.56,1415.3 L1230.55,1415.3 L1230.77,1415.99 L1231.44,1416.46 L1232.1,1417.29 L1232.27,1419.1 L1232.17,1419.42 L1231.81,1419.89 L1231.71,1420.11 L1231.7,1420.46 L1231.74,1420.7 L1231.8,1420.87 L1232,1423.04 L1231.95,1423.58 L1227.42,1423.55 L1226.02,1424.05 L1225.17,1424.07 L1225.1,1422.91 L1225.6,1421.95 L1226.9,1420.3 L1228.57,1415.43 L1230.03,1410.85 L1230.1,1410.13 L1230.03,1408.93 L1230.05,1408.31 L1230.24,1407.86 L1230.52,1407.55 L1230.75,1407.12 L1230.78,1406.31 L1230.72,1404.73 L1230.92,1402.96 L1231.33,1401.26 L1231.92,1399.89 L1231.74,1399.73 L1231.59,1399.54 L1231.66,1397.72 L1231.01,1396.53 L1230.05,1395.55 L1229.24,1394.33 L1229.11,1393.9 L1228.86,1392.79 L1228.86,1392.74 L1228.44,1390.1 L1228.19,1388.47 L1228.46,1387.66 L1228,1386.83 L1227.23,1386.12 L1226.59,1385.69 L1225.68,1385.32 L1223.82,1384.9 L1222.89,1384.45 L1222.89,1384.44 L1222.03,1383.91 L1219.43,1383.22 L1218.51,1382.38 L1218.34,1381.72 L1218.23,1381.28 L1218.47,1380.05 L1219.14,1378.8 L1219.52,1377.52 L1219.35,1375.88 L1218.87,1374.32 L1218.29,1373.27 L1216.31,1371.82 L1215.37,1370.85 L1214.87,1369.29 L1214.61,1367.65 L1214.03,1366.97 L1213.3,1366.52 L1212.6,1365.59 L1212.4,1364.28 L1212.84,1363.16 L1214.91,1360.76 L1215.56,1359.76 L1216.09,1358.6 L1216.33,1357.35 L1216.22,1353.53 L1216.39,1352.91 L1217.12,1351.34 L1217.25,1350.74 L1217.28,1350.17 L1217.37,1349.6 L1217.63,1349.05 L1218.28,1348.12 L1215.66,1346.41 L1215.22,1345.53 L1215.34,1343.9 L1215.68,1342.74 L1215.77,1341.83 L1215.16,1341 L1215.14,1340.61 L1215.82,1339.57 L1215.75,1338.29 L1215.08,1337.24 L1213.98,1336.83 L1212.88,1336.65 L1211.54,1336.07 L1210.62,1335.15 L1210.77,1334 L1209.86,1332.24 L1208.78,1329.57 L1208.41,1327.1 L1210.27,1325.23 L1211.1,1323.7 L1211.64,1322.2 L1211.42,1321.54 L1210.67,1320.94 L1210.87,1319.47 L1211.79,1317.16 L1210.55,1316.26 L1210.31,1316.03 L1207.95,1313.68 L1205.62,1312.92 L1204.44,1311.5 L1203.69,1311.2 L1203.28,1310.84 L1201.82,1308.36 L1201.19,1307.71 L1196.99,1304.97 L1195.92,1304.48 L1194.6,1303.09 L1193.82,1302.8 L1192.98,1302.79 L1192.17,1302.58 L1191.48,1302.1 L1191,1301.25 L1191.95,1300.32 L1192.3,1299.54 L1192.39,1298.35 L1192.26,1297.57 L1191.67,1295.38 L1191.41,1294.77 L1191.38,1294.38 L1193.13,1292.91 L1195.35,1291.58 L1197.47,1289.68 L1198.55,1288.4 L1198.99,1287.24 L1199.06,1286.71 L1199.97,1283.1 L1199.84,1282.2 L1199.56,1281.33 L1199.36,1280.32 L1199.41,1279.49 L1199.64,1278.71 L1199.97,1278.13 L1200.36,1277.89 L1200.61,1277.46 L1201.46,1275.47 L1200.95,1275.05 L1200.78,1274.71 L1200.18,1274.26 L1199.92,1273.88 L1199.77,1273.27 L1199.68,1272.14 L1199.59,1271.61 L1199.26,1270.46 L1198.26,1266.86 L1197.67,1263.65 L1196.81,1262.38 L1196.34,1261.09 L1195.35,1259.75 L1195,1258.47 L1194.89,1257.09 L1194.86,1254.8 L1194.75,1254.19 L1193.82,1251.8 L1193.19,1250.61 L1193.29,1249.63 L1193.14,1247.88 L1193.14,1247.87 L1193.14,1247.87 L1193.5,1247.81 L1194.15,1247.48 L1194.61,1247 L1194.63,1246.48 L1193.95,1246.07 L1194.3,1245.83 L1194.4,1245.61 L1194.27,1245.4 L1193.9,1245.21 L1194.08,1244.74 L1194.46,1245.04 L1194.99,1245.27 L1195.47,1245.59 L1195.98,1246.77 L1196.53,1247.14 L1197.57,1247.57 L1198.36,1248.32 L1198.79,1248.64 L1200.44,1248.82 L1202.48,1249.41 L1203.21,1249.91 L1203.82,1250.68 L1205.19,1252.95 L1205.65,1253.48 L1205.84,1253.28 L1205.82,1252.28 L1205.95,1251.69 L1206.24,1251.28 L1206.72,1250.85 L1206.68,1250.61 L1206.65,1250.37 L1206.67,1250 L1206.26,1249.91 L1205.3,1249.88 L1204.86,1249.68 L1203.54,1246.98 L1203.69,1245.59 L1204.42,1242.82 L1204.87,1243.17 L1205.23,1243.04 L1205.62,1242.78 L1206.2,1242.71 L1205.83,1241.15 L1205.89,1239.88 L1206.07,1238.75 L1206.12,1237.57 L1205.86,1237.59 L1205.82,1238.34 L1205.57,1238.85 L1204.74,1239.8 L1205,1240.21 L1204.66,1240.1 L1204.39,1239.95 L1204.17,1239.73 L1203.95,1239.42 L1204.13,1239.23 L1204.44,1238.76 L1204.66,1238.52 L1204.08,1237.49 L1203.17,1237.05 L1199.05,1236.88 L1198.43,1236.27 L1198.82,1235.03 L1198.69,1234.91 L1198.66,1234.9 L1198.65,1234.85 L1198.56,1234.62 L1198.24,1235.26 L1197.96,1235.37 L1197.63,1235.22 L1197.19,1235.13 L1196.87,1235.28 L1196.64,1235.6 L1196.42,1236.02 L1196.13,1236.44 L1197.24,1236.59 L1197.86,1236.83 L1198.2,1237.21 L1198.17,1237.97 L1197.82,1238.55 L1197.37,1238.79 L1197.02,1238.57 L1196.05,1239.26 L1195.55,1239.46 L1195.03,1239.54 L1195.05,1239.97 L1195.57,1240.14 L1195.56,1240.27 L1195.08,1240.4 L1194.73,1240.36 L1194.13,1240.09 L1193.69,1240.05 L1193.05,1239.67 L1191.12,1237.2 L1190.23,1236.86 L1190.14,1236.87 L1190.14,1236.87 L1189.79,1235.92 L1189.1,1235.16 L1189.44,1235.04 L1189.73,1234.81 L1190.26,1233.09 L1190.89,1233.33 L1191.31,1233.64 L1192.46,1234 L1196.13,1234.08 L1197.58,1233.69 L1198.96,1232.84 L1201.48,1230.73 L1210.93,1225.77 L1219.17,1223 L1229.06,1217.96 L1235.53,1215.71 L1242.59,1213.2 L1249.42,1209.41 L1255.24,1208.19 L1259.17,1206.49 L1258.75,1207 L1257.01,1207.97 L1257.85,1208.46 L1261.02,1207.63 L1260.98,1207.24 L1260.74,1207.26 L1260.7,1206.8 L1262.43,1206.65 L1262.39,1206.35 L1262.35,1206.29 L1262.28,1206.32 L1262.15,1206.24 L1262.13,1205.23 L1261.29,1205.26 L1259.36,1206.05 L1260.56,1205.32 L1261.98,1204.21 L1264,1201.36 L1263.63,1202.33 L1262.89,1203.25 L1262.61,1203.74 L1262.93,1203.59 L1263.58,1203.4 L1264.14,1203.12 L1264.59,1202.46 L1265.82,1201.75 L1266.24,1201.16 L1265.72,1200.3 L1265,1200.34 L1264.71,1200.75 L1264.23,1201.34 L1265.53,1198.54 L1269.25,1192.74 L1269.29,1193.13 L1269.15,1193.75 L1269.53,1193.88 L1270.09,1193.6 L1270.5,1193.02 L1270.66,1192.16 L1270.42,1191.92 L1269.97,1192.16 L1269.48,1192.72 L1270.09,1191.17 L1271.23,1189.79 L1273.56,1188.02 L1281.02,1186.2 L1283.84,1184.75 L1285.36,1184.34 L1286.67,1183.72 L1289.76,1179.58 L1300.11,1173.39 L1313.45,1169.38 L1319.31,1166.62 L1328.79,1164.67 L1334.7,1163.99 L1335.54,1164.2 L1336.98,1165.19 L1337.49,1165.38 L1338.26,1165.47 L1351.43,1171.35 L1354.03,1173.99 L1354.48,1174.63 L1354.97,1175.55 L1355.11,1176.39 L1354.51,1176.78 L1354.33,1176.58 L1354.1,1176.15 L1353.82,1175.73 L1353.11,1175.35 L1352.52,1174.24 L1351.84,1173.55 L1349.76,1170.85 L1349.41,1170.73 L1348.12,1170.62 L1347.78,1170.53 L1347.37,1169.98 L1346.31,1169.75 L1343.98,1168.53 L1341.74,1167.77 L1341.35,1167.32 L1340.07,1166.56 L1339.75,1166.45 L1339.33,1166.7 L1339.14,1167.04 L1338.85,1168.88 L1338.83,1169.3 L1338.94,1170.21 L1339.16,1170.41 L1340.06,1170.7 L1340.28,1170.91 L1340.81,1171.62 L1341.47,1172.71 L1341.51,1173.18 L1341.42,1173.61 L1341.48,1174.02 L1341.95,1174.41 L1341.88,1175.65 L1342.35,1176.12 L1343.02,1175.9 L1343.53,1175.05 L1343.44,1176.11 L1343.83,1176.9 L1344.36,1177.54 L1345.62,1180.26 L1345.97,1180.83 L1346,1182.27 L1346.33,1184.73 L1346.85,1187.03 L1347.48,1188.01 L1348.07,1188.13 L1349.28,1188.85 L1349.87,1188.98 L1350.37,1188.97 L1350.99,1189.08 L1351.55,1189.36 L1351.84,1189.87 L1352.56,1190.66 L1358.92,1191.89 L1360.28,1191.82 L1361.33,1191.06 L1361.68,1191.34 L1362.13,1191.52 L1363.17,1191.69"
         id="path1061" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['PL'],polycolors['PL'],labels['PL'], x1, polybary['PL'], x2, polybary['PL'] )
    if 'PL' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1016.94,1667.95 L1016.06,1668.22 L1013.45,1667.77 L1012.49,1667.48 L1012.45,1666.9 L1012.95,1666.23 L1013.61,1665.57 L1014.02,1664.74 L1014.01,1663.4 L1013.66,1662.4 L1013.2,1661.5 L1012.88,1660.46 L1012.99,1658.36 L1013.74,1656.26 L1014.46,1655.07 L1014.84,1655.64 L1015.55,1656.31 L1015.26,1657.18 L1015.45,1657.73 L1015.85,1658.28 L1016.21,1659.14 L1016.17,1659.59 L1015.74,1660.53 L1015.74,1661.07 L1017,1661.83 L1018.01,1663.25 L1018.44,1664.84 L1018.13,1666.46 L1016.94,1667.95"
         id="path1063" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['LI'],polycolors['LI'],labels['LI'], x1, polybary['LI'], x2, polybary['LI'] )
    if 'LI' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M588.802,1401.14 L594.359,1404.5 L595.332,1404.62 L595.727,1404.77 L596.145,1405.21 L596.146,1405.37 L596.019,1405.88 L596.017,1406.05 L596.121,1406.11 L596.474,1406.08 L596.577,1406.14 L597.186,1407.53 L597.115,1408 L596.933,1407.97 L596.811,1408.06 L596.7,1408.21 L596.543,1408.34 L595.01,1408.55 L594.337,1408.77 L593.102,1409.57 L592.827,1409.83 L592.682,1410.13 L592.501,1411.47 L592.36,1412.15 L592.181,1412.43 L587.631,1413.05 L586.268,1412.42 L581.717,1407.96 L581.508,1407.54 L580.354,1406.4 L579.933,1406.14 L578.426,1406.09 L577.744,1406.25 L576.969,1406.58 L578.191,1404.99 L579.944,1403.9 L583.613,1402.83 L583.495,1403.02 L583.329,1403.42 L583.219,1403.63 L584.197,1403.67 L585.227,1403.05 L586.818,1401.6 L587.903,1401.07 L588.802,1401.14"
         id="path1065" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M506.205,1225.66 L506.646,1225.87 L507.126,1225.9 L508.019,1225.8 L508.486,1225.83 L508.729,1225.74 L509.172,1225.2 L509.413,1225.1 L509.649,1225.23 L510.049,1225.63 L510.196,1225.74 L511.276,1225.9 L511.763,1226.12 L512.139,1226.56 L511.222,1227.24 L509.471,1229.4 L508.492,1229.73 L507.549,1229.73 L506.526,1230.02 L505.524,1230.53 L504.655,1231.19 L504.481,1231.36 L504.381,1231.5 L504.308,1231.75 L504.217,1232.2 L504.081,1232.68 L503.961,1232.72 L503.813,1232.62 L502.33,1233.3 L501.609,1233.93 L498.981,1234.86 L498.569,1234.92 L498.134,1234.74 L497.274,1234.12 L497.235,1234.26 L496.974,1234.42 L496.655,1234.49 L496.444,1234.35 L496.462,1233.99 L496.601,1233.42 L496.769,1232.91 L496.899,1232.69 L497.24,1232.49 L497.782,1232 L498.091,1231.46 L497.754,1231.12 L497.142,1231.28 L496.093,1232.31 L495.508,1232.41 L495.512,1232.21 L494.512,1230.9 L494.418,1230.83 L494.108,1230.85 L494.01,1230.8 L494.019,1230.63 L494.171,1230.13 L494.181,1229.96 L494.227,1229.78 L494.274,1229.32 L494.247,1228.84 L493.649,1227.97 L492.924,1225.32 L492.437,1224.35 L492.531,1223.89 L492.87,1223.31 L493.137,1222.39 L493.398,1221.05 L494,1219.04 L494.103,1218.13 L493.92,1217.14 L494.661,1216.79 L499.669,1216.14 L500.13,1216.21 L502.577,1217.15 L503.257,1217.32 L504.49,1218.4 L504.782,1218.81 L504.81,1219.58 L504.754,1220.34 L504.809,1220.7 L504.743,1220.85 L505.126,1221.3 L505.595,1221.7 L505.786,1221.76 L505.761,1222.08 L505.634,1222.49 L505.548,1222.86 L505.65,1223.04 L505.655,1223.19 L506.094,1224.01 L506.153,1224.76 L506.118,1225.29 L506.205,1225.66"
         id="path1067" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M452.356,1089.53 L453.132,1089.98 L454.65,1089.4 L455.475,1089.83 L458.389,1092.55 L458.91,1092.84 L459.71,1092.89 L461.167,1092.62 L461.878,1092.64 L463.182,1093.21 L463.795,1093.7 L463.987,1094.22 L464.233,1094.45 L465.185,1096.2 L465.387,1096.93 L464.473,1100.32 L464.004,1101.27 L463.188,1102.47 L463.331,1103.08 L463.972,1103.32 L464.634,1103.43 L465.078,1103.6 L465.544,1103.95 L465.905,1104.44 L466.037,1105.07 L465.715,1106.15 L464.561,1108.29 L464.534,1108.73 L465.513,1109.13 L466.338,1109.83 L466.988,1110.81 L467.458,1112.02 L467.515,1112.47 L467.527,1113.51 L467.631,1114.02 L467.796,1114.34 L468.239,1114.95 L468.443,1115.32 L468.657,1115.88 L468.833,1116.56 L468.908,1117.37 L468.804,1118.28 L468.795,1120.05 L469.555,1120.82 L470.503,1121.47 L471.05,1122.94 L471.268,1122.99 L470.928,1121.53 L469.229,1118.62 L470.025,1117.69 L471.015,1118.22 L471.885,1119.44 L472.315,1120.58 L472.305,1122.61 L471.823,1124.3 L471.013,1125.55 L470.023,1126.26 L468.861,1126.56 L466.62,1126.72 L465.554,1127.02 L464.439,1127.67 L463.551,1128.43 L462.363,1129.81 L461.713,1130.55 L462.074,1130.94 L462.099,1131.51 L462.009,1132.06 L462.033,1132.39 L462.463,1132.53 L462.801,1132.29 L463.077,1131.94 L463.55,1131.65 L464.626,1130.55 L465.061,1130.23 L466.441,1129.9 L467.86,1129.25 L468.7,1129.06 L469.202,1129.4 L469.66,1129.96 L470.418,1130.2 L474.811,1130.44 L474.969,1130.68 L475.21,1131.64 L475.505,1131.91 L475.847,1132.12 L475.886,1132.5 L475.788,1133.03 L475.743,1133.68 L475.82,1135.95 L476.043,1136.92 L476.561,1137.41 L476.69,1137.96 L476.368,1140.43 L476.332,1141.31 L477.066,1143.29 L477.253,1143.51 L476.883,1144.04 L476.847,1144.38 L476.902,1144.65 L476.818,1144.97 L476.557,1145.3 L475.635,1146.23 L475.252,1146.92 L475.272,1147.99 L474.913,1148.98 L474.463,1149.38 L473.904,1149.69 L473.613,1150.14 L473.972,1150.93 L473.382,1151.11 L472.748,1151.88 L472.237,1151.87 L471.912,1151.33 L471.77,1150.4 L471.68,1149.11 L471.305,1148.8 L471.053,1148.17 L470.952,1147.43 L471.039,1146.74 L471.44,1146.06 L472.491,1145.19 L472.874,1144.31 L473.17,1142.36 L473.103,1140.88 L472.723,1139.7 L470.79,1136.78 L469.96,1136.01 L468.907,1135.72 L468.812,1136.14 L468.937,1136.79 L468.703,1137.38 L468.28,1137.83 L467.858,1138.13 L468.353,1138.26 L468.687,1138.61 L468.931,1139 L469.151,1139.25 L470.444,1139.64 L470.767,1140.08 L470.139,1140.18 L469.325,1140.16 L469.037,1140.38 L469.977,1141.23 L469.641,1141.93 L469.488,1142.64 L469.587,1143.21 L470.01,1143.47 L469.912,1143.73 L469.826,1143.89 L469.555,1144.25 L469.737,1144.69 L468.03,1147.67 L467.62,1148.18 L466.086,1148.95 L465.552,1149.51 L467.116,1149.29 L468.978,1148.67 L470.583,1148.57 L471.372,1149.92 L471.204,1152.33 L470.371,1154 L469.117,1155.19 L467.661,1156.17 L467.488,1156.18 L467.107,1155.98 L466.93,1156.01 L466.405,1157.21 L466.121,1157.33 L465.769,1157.42 L465.512,1157.28 L465.505,1156.77 L465.171,1156.22 L463.141,1155.58 L461.588,1155.4 L459.918,1155.63 L459.189,1155.61 L458.421,1155.72 L458.041,1156.41 L457.636,1158.53 L457.592,1158.76 L457.231,1159.91 L456.724,1161.14 L456.143,1162.28 L455.542,1163.18 L454.705,1163.85 L453.032,1164.29 L451.657,1165.23 L449.868,1166.05 L449.34,1166.18 L447.915,1165.97 L447.372,1165.73 L446.853,1165.13 L446.941,1164.75 L447.734,1164.56 L448.037,1164.56 L445.534,1162.91 L445.086,1162.77 L444.932,1161.55 L444.06,1160.98 L443.009,1160.65 L442.323,1160.18 L442.005,1160.54 L441.581,1159.97 L441.031,1159.92 L440.462,1160.07 L439.96,1160.07 L439.609,1159.66 L439.433,1159.17 L439.153,1159.06 L438.488,1159.76 L438.355,1160.13 L438.306,1160.62 L438.213,1161.13 L437.943,1161.61 L437.666,1161.77 L436.791,1161.72 L435.793,1161.84 L435.537,1161.79 L435.347,1161.57 L435.198,1161.25 L435.117,1160.96 L435.118,1160.85 L434.12,1160.66 L430.797,1161.09 L430.524,1160.99 L430.033,1160.63 L429.737,1160.62 L429.501,1160.79 L429.26,1161.02 L428.978,1161.29 L428.754,1161.37 L428.314,1160.97 L427.643,1159.56 L427.204,1158.99 L428.098,1157.68 L428.309,1157 L428.096,1156.21 L429.146,1155.35 L429.746,1154.31 L429.8,1153.17 L429.594,1152.76 L429.222,1152.02 L428.631,1151.19 L428.303,1150.88 L427.894,1150.73 L427.459,1150.87 L427.095,1151.2 L426.734,1151.44 L426.319,1151.3 L425.666,1150.62 L425.063,1149.79 L424.564,1148.81 L424.21,1147.69 L424.152,1147.03 L424.172,1146.41 L424.134,1145.85 L423.904,1145.41 L422.804,1144.45 L423.506,1144.27 L423.939,1143.72 L424.091,1142.82 L423.955,1141.58 L423.841,1140.54 L423.68,1140.25 L423.265,1140.11 L423.022,1139.87 L422.95,1138.76 L422.803,1138.28 L421.058,1135.9 L420.015,1135 L418.88,1134.83 L417.595,1135.6 L415.206,1137.93 L413.825,1138.54 L413.533,1138.48 L412.94,1138.21 L412.786,1138.25 L412.665,1138.29 L412.541,1138.56 L412.537,1138.96 L412.596,1139.3 L412.666,1139.42 L412.67,1139.46 L412.256,1139.96 L412.213,1139.93 L411.958,1140.14 L411.714,1139.98 L411.533,1139.99 L411.457,1140.66 L412.499,1141.55 L412.077,1142.35 L413.075,1143.73 L412.412,1144.31 L412.393,1145.37 L411.512,1145.68 L409.439,1145.73 L408.957,1145.95 L408.592,1146.18 L408.265,1146.52 L407.915,1147.08 L408.026,1147.36 L408.286,1147.73 L408.251,1148.06 L407.467,1148.25 L407.476,1148.9 L407.229,1149.49 L406.841,1149.96 L406.418,1150.26 L405.867,1150.39 L405.799,1150.12 L405.973,1149.57 L406.143,1148.87 L406.327,1148.34 L406.539,1147.92 L406.455,1147.74 L405.214,1148.12 L404.881,1148.37 L404.78,1148.78 L405.168,1150.69 L404.943,1150.66 L404.56,1150.61 L402.982,1149.59 L402.426,1149.59 L402.279,1149.51 L402.275,1149.23 L402.148,1148.63 L401.875,1148.47 L401.504,1148.44 L401.402,1148.32 L401.93,1147.89 L401.062,1147.59 L399.307,1148.4 L398.503,1148.37 L394.94,1146.52 L394.599,1146.11 L394.613,1145.57 L394.69,1145.02 L394.54,1144.57 L392.475,1141.98 L391.874,1141.71 L389.731,1141.32 L387.855,1140.56 L387.276,1139.95 L387.048,1138.89 L387.597,1136.16 L387.613,1135.13 L387.196,1134.6 L385.867,1134.04 L385.615,1133.85 L385.338,1133.65 L384.908,1132.81 L384.674,1131.82 L384.421,1129.72 L383.395,1129.45 L382.633,1128.74 L382.033,1127.66 L381.092,1125.14 L380.496,1123.98 L380.287,1123.38 L380.342,1122.87 L380.357,1122.38 L380.022,1121.84 L380.238,1121.78 L380.975,1121.59 L382.988,1121.51 L383.986,1121.22 L384.488,1120.94 L384.735,1120.73 L385.398,1119.95 L386.386,1119.07 L387.343,1118.42 L389.84,1119.73 L391.204,1120.14 L392.687,1119.95 L397.044,1117.47 L398.62,1117.35 L398.626,1117.32 L398.627,1117.31 L398.838,1116.1 L398.49,1115.91 L397.841,1116.17 L397.178,1116.32 L396.686,1115.96 L395.825,1114.66 L395.414,1114.25 L395.128,1114.26 L394.478,1114.54 L394.188,1114.46 L393.99,1114.09 L393.835,1113.22 L393.59,1112.96 L393.028,1112.63 L392.547,1112.1 L392.422,1111.4 L392.928,1110.55 L392.745,1110.38 L392.297,1109.74 L394.255,1109.31 L395.764,1108.4 L396.238,1108.55 L397.011,1109.68 L398.01,1110.59 L399.219,1110.8 L400.597,1110.41 L402.122,1109.5 L402.84,1109.29 L403.955,1109.93 L405.086,1109.86 L405.649,1110.39 L406.011,1110.51 L406.52,1110.42 L406.554,1110.29 L406.419,1110.03 L406.418,1109.55 L406.871,1108.05 L407.301,1107.28 L410.638,1104.73 L411.46,1103.79 L412.149,1102.57 L412.309,1101.93 L412.552,1100.35 L412.863,1099.66 L413.245,1099.35 L414.009,1099.23 L414.078,1099.17 L414.428,1098.82 L414.598,1098.32 L414.792,1096.71 L415.04,1096.32 L415.242,1096.3 L415.36,1096.22 L415.35,1095.65 L416.72,1094.35 L417.171,1094.03 L417.668,1093.8 L419.975,1094 L420.338,1093.75 L420.712,1093.39 L421.114,1093.08 L421.851,1092.99 L421.851,1092.99 L421.233,1094.05 L421.775,1094.21 L423.476,1094.12 L424.083,1094.34 L425.401,1095.51 L426.555,1095.98 L427.796,1096.09 L428.264,1095.89 L429.266,1095.09 L429.869,1094.83 L430.041,1094.04 L430.589,1092.99 L431.251,1092.09 L431.755,1091.76 L432.302,1091.26 L433.196,1088.55 L433.617,1087.75 L434.178,1087.84 L435.425,1089.22 L436.044,1089.66 L440.585,1090.51 L444.994,1089.59 L447.438,1089.97 L448.127,1089.66 L448.836,1089.05 L449.653,1088.6 L450.501,1088.38 L451.293,1088.46 L452.356,1089.53"
         id="path1069" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M501.086,1087.32 L499.433,1087.28 L497.608,1086.69 L496.146,1085.78 L494.907,1084.55 L493.764,1082.99 L493.857,1082.57 L493.929,1082.16 L493.978,1080.45 L493.941,1079.95 L494.034,1079.53 L494.245,1079.42 L494.347,1079.28 L494.455,1079.1 L494.684,1078.84 L494.1,1077.43 L493.729,1076.07 L493.753,1074.53 L494.366,1072.58 L495.126,1071.36 L496.252,1070.23 L497.525,1069.51 L498.738,1069.55 L498.713,1069.32 L498.726,1068.85 L498.696,1068.62 L499.605,1068.76 L499.873,1068.88 L501.857,1070.5 L502.338,1071.21 L502.8,1072.51 L502.997,1073.91 L503.024,1077.09 L502.447,1077.84 L503.599,1079.72 L503.381,1081.11 L502.996,1081.25 L502.495,1081.16 L502.13,1081.31 L502.142,1082.16 L502.375,1082.45 L503.646,1083.4 L502.757,1083.79 L502.649,1084.61 L502.796,1085.64 L502.666,1086.66 L502.065,1087.13 L501.086,1087.32"
         id="path1071" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M472.57,1055.82 L472.245,1057.69 L472.323,1058.31 L472.656,1059.05 L473.06,1059.66 L473.334,1059.94 L473.445,1060.2 L473.24,1060.69 L472.953,1061.22 L472.819,1061.58 L472.914,1061.95 L473.087,1062.23 L473.581,1062.63 L473.336,1063.31 L473.191,1064.01 L473.151,1064.65 L473.235,1065.19 L473.146,1065.58 L472.689,1065.69 L472.274,1065.87 L471.919,1066.17 L471.642,1066.63 L471.536,1066.13 L470.696,1066.94 L469.734,1067.48 L468.705,1067.79 L467.663,1067.88 L466.257,1067.31 L465.761,1067.44 L465.201,1068.68 L465.003,1068.87 L464.584,1069.17 L463.543,1069.69 L462.588,1069.82 L461.837,1069.34 L461.677,1067.86 L462.21,1066.52 L463.164,1065.82 L465.283,1065.17 L465.052,1062.84 L464.753,1062.22 L463.551,1061.23 L463.267,1061.14 L463.674,1060.07 L464.812,1059.15 L466.733,1057.98 L466.075,1057.43 L465.134,1057.04 L464.127,1056.97 L463.291,1057.41 L461.981,1059.34 L461.261,1060.23 L459.98,1060.75 L458.497,1062.15 L457.944,1062.54 L456.945,1062.48 L456.652,1061.8 L456.929,1060.81 L457.627,1059.83 L457.48,1059.67 L457.404,1059.4 L457.247,1059.26 L457.853,1058.34 L459.929,1056.4 L458.981,1055.3 L459.524,1055.01 L459.898,1054.69 L460.178,1054.15 L460.448,1053.2 L460.723,1052.45 L461.097,1052.03 L462.547,1051.41 L464.614,1051 L465.127,1050.64 L465.586,1050.42 L465.983,1050.72 L465.633,1051.05 L464.801,1052.69 L464.711,1053.07 L464.707,1054.23 L464.857,1054.73 L465.248,1054.96 L465.347,1054.54 L465.375,1053.28 L466.43,1052.09 L467.793,1051.23 L469.185,1050.78 L470.311,1049.8 L470.812,1049.64 L471.33,1049.61 L471.917,1049.4 L472.463,1049.09 L472.857,1048.79 L473.17,1048.99 L473.334,1049.28 L473.365,1049.67 L472.9,1052.32 L472.57,1055.82"
         id="path1073" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M486.265,1046.72 L482.063,1051.83 L480.751,1054.11 L480.423,1053.89 L479.329,1054.78 L478.69,1054.96 L478.332,1055.37 L477.444,1057.65 L476.999,1058.54 L475.982,1058.96 L474.664,1058.56 L473.628,1057.46 L473.453,1055.79 L473.839,1052.78 L474.154,1051.73 L474.711,1050.85 L475.323,1050.54 L475.977,1050.37 L476.671,1049.88 L477.876,1049.27 L480.675,1049.34 L482.014,1048.63 L481.211,1048.43 L479.871,1048.5 L478.706,1048.28 L478.427,1047.18 L479.172,1045.88 L480.496,1044.52 L481.846,1043.45 L482.668,1043.07 L483.774,1042.98 L486.963,1040.96 L489.111,1039.57 L490.31,1039.1 L490.959,1039.62 L490.949,1040.23 L490.854,1041.09 L490.636,1041.84 L489.978,1042.25 L489.257,1043.05 L488.409,1043.61 L486.265,1046.72"
         id="path1075" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M486.185,1008.18 L486.726,1009.6 L487.315,1013.21 L487.318,1013.67 L487.225,1014.07 L487.242,1014.39 L487.565,1014.59 L490.927,1015.35 L491.148,1015.64 L491.307,1016.05 L491.504,1016.34 L492.106,1016.1 L492.453,1016.11 L492.769,1016.25 L492.962,1016.49 L493.433,1017.85 L493.663,1018.17 L493.968,1018.34 L494.616,1018.37 L494.837,1018.43 L495.15,1018.83 L495.479,1019.5 L495.695,1020.3 L495.681,1021.08 L495.902,1021.27 L496.423,1021.25 L496.719,1021.57 L496.269,1022.75 L496.037,1022.47 L495.785,1022.31 L495.192,1022.07 L495.392,1022.55 L495.688,1023.52 L495.742,1024.42 L495.225,1024.71 L494.114,1024.49 L493.683,1024.14 L493.894,1023.49 L493.75,1023.19 L493.568,1023.02 L493.345,1022.94 L493.08,1022.91 L493.005,1023.74 L492.434,1024.37 L491.099,1025.1 L490.994,1025.56 L491.629,1025.26 L492.299,1025.13 L493.618,1025.23 L494.101,1025.64 L493.721,1026.26 L492.972,1026.85 L489.493,1028.25 L488.081,1028.27 L487.171,1027.29 L487.431,1027.16 L487.896,1026.84 L488.328,1026.39 L488.501,1025.84 L488.301,1025.44 L487.878,1025.38 L486.98,1025.49 L486.761,1025.65 L486.477,1026.04 L486.139,1026.42 L485.777,1026.54 L484.576,1026.55 L481.559,1027.33 L480.827,1027.85 L480.364,1028.09 L479.967,1027.88 L479.614,1027.5 L479.263,1027.25 L478.89,1027.3 L477.953,1027.68 L477.549,1027.78 L474.408,1027.06 L474.161,1027.21 L473.904,1027.6 L473.595,1027.98 L473.198,1028.1 L472.786,1027.96 L472.14,1027.56 L471.346,1026.78 L470.895,1025.96 L470.626,1024.91 L470.696,1023.73 L471.21,1022.93 L472.058,1022.6 L472.948,1022.63 L473.591,1022.9 L473.974,1023.67 L474.212,1024.67 L474.56,1025.3 L475.281,1025.01 L475.38,1024.6 L475.128,1024.47 L475.007,1024.34 L474.92,1024.19 L474.766,1024.02 L475.448,1023.88 L476.89,1024.58 L477.487,1024.43 L478.191,1024.01 L480.636,1024.05 L482.418,1023.28 L483.22,1022.76 L483.084,1022.41 L482.366,1022.44 L480.131,1023.09 L478.391,1023.14 L477.564,1022.89 L477.042,1022.14 L477.039,1021.45 L477.332,1020.87 L477.765,1020.49 L478.182,1020.41 L478.657,1020.39 L478.982,1020.15 L479.77,1019.01 L479.976,1018.59 L480.231,1018.19 L480.608,1017.88 L480.96,1017.84 L482.124,1018.23 L484.36,1017.73 L484.842,1017.49 L485.183,1016.92 L485.347,1016.33 L485.323,1015.79 L485.1,1015.35 L482.664,1015.99 L481.923,1015.94 L481.63,1016.04 L481.28,1016.24 L480.951,1016.29 L480.709,1015.91 L480.459,1014.52 L480.288,1013.89 L479.985,1013.34 L479.388,1012.77 L476.375,1011.5 L475.801,1011.04 L475.345,1010.48 L474.993,1009.9 L476.888,1009.06 L476.745,1008.42 L476.243,1007.49 L476.219,1006.7 L477.478,1006.87 L478.087,1006.74 L478.737,1006.37 L478.921,1006.91 L479.157,1007.22 L479.439,1007.29 L479.752,1007.08 L480.116,1005.92 L481.29,1005.37 L482.712,1005.26 L483.821,1005.38 L484.203,1005.81 L485.662,1006.68 L485.541,1006.94 L485.294,1007.59 L485.169,1007.85 L486.185,1008.18"
         id="path1077" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M492.782,939.813 L492.91,940.518 L493.107,940.973 L493.716,941.792 L493.921,942.284 L494.388,943.841 L494.429,944.157 L495.587,945.749 L496.017,946.706 L496.217,947.382 L496.242,947.923 L495.529,952.241 L495.406,954.144 L495.263,955 L494.958,955.892 L494.331,957.211 L494.073,958.022 L493.781,958.443 L492.711,959.098 L492.278,959.59 L492.77,959.287 L493.411,959.013 L494.041,958.938 L494.495,959.221 L494.629,959.913 L494.306,960.453 L493.839,960.875 L493.552,961.202 L493.472,962.251 L493.799,962.99 L494.187,963.557 L494.293,964.054 L493.922,964.54 L492.653,964.958 L492.106,965.273 L495.018,965.542 L495.718,965.931 L495.741,966.463 L495.221,966.931 L494.295,967.138 L494.196,967.556 L495.118,967.666 L495.93,967.459 L496.624,967.526 L497.177,968.454 L497.435,968.77 L497.859,968.922 L498.714,969.025 L499.148,969.381 L499.33,970.004 L499.416,970.629 L499.551,971.019 L500.015,971.177 L500.377,970.955 L500.666,970.606 L500.903,970.408 L502.259,970.778 L503.423,970.035 L504.424,969.899 L505.392,970.125 L505.763,971.023 L508.222,971.574 L507.326,972.971 L507.248,973.372 L507.24,974.067 L506.975,974.636 L506.607,975.072 L505.723,975.848 L504.283,976.602 L503.723,976.756 L502.377,976.595 L502.053,976.957 L501.68,978.295 L501.167,979.06 L500.337,979.47 L499.542,979.707 L499.154,979.941 L498.784,980.347 L496.189,982.673 L495.528,983.096 L494.812,983.39 L492.961,983.337 L492.544,983.016 L492.414,982.147 L492.766,981.116 L493.975,979.99 L494.009,979.431 L494.107,979.013 L494.337,978.924 L494.472,978.774 L494.745,978.239 L494.909,977.993 L495.206,977.746 L496.084,976.72 L496.479,976.38 L496.936,976.101 L500.499,975.191 L500.597,974.773 L499.566,975.074 L498.579,975.138 L495.743,974.55 L495.6,974.352 L495.64,973.976 L495.733,973.546 L495.737,973.188 L495.27,971.673 L494.971,971.03 L494.631,970.732 L494.619,971.408 L494.65,971.924 L494.815,972.977 L494.254,973.792 L493.931,974.167 L493.518,974.439 L493.637,974.944 L493.114,975.348 L492.677,975.966 L492.228,976.506 L491.676,976.659 L491.364,976.087 L491.433,974.954 L491.593,973.685 L491.754,972.754 L491.568,972.357 L491.407,972.228 L491.192,972.309 L490.855,972.546 L490.703,972.11 L490.498,971.782 L490.241,971.563 L489.925,971.45 L489.834,971.831 L490.195,972.436 L489.529,972.697 L484.57,972.846 L484.678,972.394 L485.133,972.148 L486.453,971.046 L486.544,970.666 L485.937,970.498 L485.313,970.513 L484.684,970.681 L484.07,970.971 L483.678,969.831 L483.455,968.582 L483.854,968.502 L484.217,968.351 L484.553,968.083 L484.873,967.635 L483.328,967.426 L482.817,967.125 L481.768,964.664 L481.734,964.441 L481.71,963.826 L481.673,963.539 L481.567,963.238 L481.458,963.057 L481.39,962.826 L481.396,962.37 L483.173,961.633 L483.894,961.493 L484.597,961.8 L485.053,962.458 L485.463,963.326 L485.998,964.118 L486.828,964.527 L486.596,963.617 L484.952,961.403 L484.655,960.774 L484.482,960.534 L484.124,960.37 L483.737,960.361 L482.949,960.499 L482.543,960.398 L483.105,960.131 L483.216,959.674 L482.967,959.557 L482.421,959.045 L482.679,959.011 L482.797,958.945 L482.948,958.728 L482.685,958.498 L482.545,958.118 L482.602,957.604 L482.927,956.963 L482.406,956.982 L481.557,957.649 L481.053,957.839 L481.351,957.342 L481.525,956.782 L481.541,956.205 L481.346,955.668 L480.878,956.405 L480.438,956.58 L480.015,956.645 L479.59,957.051 L479.365,957.739 L479.454,958.94 L479.221,959.612 L478.565,959.958 L475.804,958.194 L475.16,957.494 L474.728,956.601 L474.692,955.667 L474.781,954.006 L473.68,952.637 L473.669,951.188 L475.68,951.671 L475.356,951.031 L475.226,950.306 L475.346,949.556 L475.778,948.845 L476.381,948.287 L476.762,948.197 L477.022,948.473 L478.116,951.463 L478.183,952.07 L478.391,952.524 L478.866,953.14 L479.275,953.34 L479.275,952.53 L479.368,952.153 L479.537,952.067 L479.719,951.905 L479.897,951.84 L479.506,951.006 L479.451,950.851 L479.407,950.627 L479.219,950.117 L479.219,949.868 L479.363,949.623 L479.695,949.568 L479.813,949.373 L480.146,949.137 L480.71,949.35 L481.565,950.029 L481.513,948.847 L480.921,947.684 L480.089,946.777 L479.329,946.367 L479.908,945.866 L480.476,945.042 L480.76,944.185 L480.513,943.566 L480.626,943.107 L481.298,943.278 L481.645,943.852 L481.879,944.491 L482.478,945.151 L482.48,945.682 L482.393,946.255 L482.39,946.659 L482.606,947.12 L482.785,947.352 L484.097,948.566 L484.372,948.973 L484.604,949.627 L484.941,950.252 L485.387,950.305 L485.903,950.174 L486.465,950.265 L485.28,951.908 L485.583,952.084 L486.504,951.34 L487.181,950.233 L487.562,950.562 L489.245,954.482 L489.345,954.067 L489.235,953.823 L489.162,953.386 L489.079,953.122 L489.299,952.289 L489.03,951.41 L488.23,949.797 L487.997,948.393 L488.268,947.8 L488.854,947.297 L489.565,946.148 L489.155,946.123 L488.797,945.971 L488.519,945.729 L488.348,945.421 L488.694,944.62 L488.642,944.088 L488.453,943.668 L488.388,943.225 L488.527,942.494 L488.65,942.302 L488.889,942.287 L490.4,941.744 L490.676,941.559 L491.136,940.893 L491.506,940.2 L491.847,939.831 L492.216,940.122 L492.492,940.076 L492.618,940.011 L492.782,939.813"
         id="path1079" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M506.133,900.782 L506.52,901.04 L506.996,901.132 L506.887,901.587 L506.12,901.776 L504.418,902.54 L503.752,903.019 L503.782,903.596 L503.639,903.988 L503.34,904.188 L502.901,904.18 L502.475,904.285 L501.959,905.076 L500.917,905.238 L500.741,905.672 L500.706,906.231 L500.551,906.675 L500.185,906.91 L499.8,906.956 L498.909,906.77 L499.26,907.892 L500.352,908.542 L501.656,908.752 L502.641,908.56 L504.605,907.22 L505.156,907.049 L505.804,906.979 L506.13,907.093 L505.721,907.479 L505.63,907.859 L505.827,907.905 L505.605,908.168 L505.361,908.513 L505.181,908.676 L505.211,908.919 L505.22,909.563 L504.271,909.787 L503.939,909.93 L503.65,910.154 L503.202,910.674 L503.042,910.821 L502.38,911 L501.952,910.824 L501.171,909.943 L500.675,909.628 L499.196,909.481 L498.706,909.167 L497.904,908.311 L497.408,908.179 L497.612,910.577 L497.792,911.357 L497.457,911.564 L497.283,911.783 L497.234,912.095 L497.26,912.589 L496.578,912.915 L494.861,912.907 L493.453,914.005 L492.896,914.204 L492.597,914.113 L492.19,913.646 L491.899,913.529 L490.816,914.152 L489.026,914.218 L488.517,914.046 L488.414,914.463 L489.631,914.79 L494.063,914.764 L495.199,914.389 L495.654,914.417 L495.998,914.826 L496.176,915.514 L496.158,916.332 L495.925,917.12 L496.031,917.255 L496.255,917.678 L495.928,917.963 L495.603,918.147 L495.282,918.234 L494.96,918.216 L495.245,918.559 L495.589,918.797 L495.981,918.944 L496.4,918.996 L495.634,919.551 L495.011,920.134 L494.373,920.541 L493.561,920.57 L490.667,919.321 L490.057,918.877 L489.88,918.819 L489.582,919.229 L489.789,919.562 L490.409,920.025 L490.788,920.651 L491.85,921.264 L492.153,922.003 L492.051,922.42 L491.683,922.596 L490.923,923.44 L490.549,923.785 L489.865,924.08 L489.467,923.947 L489.113,923.616 L488.564,923.315 L488.452,923.769 L488.567,923.912 L488.675,924.129 L488.79,924.291 L488.679,924.745 L487.865,924.946 L487.292,924.681 L486.909,924.096 L486.667,923.344 L486.592,922.451 L486.495,922.081 L486.225,921.875 L485.982,922.148 L485.985,922.882 L486.113,923.693 L486.252,924.168 L485.937,924.495 L485.04,923.718 L484.799,923.34 L484.709,922.987 L484.692,922.189 L484.262,920.146 L484.208,919.271 L484.372,918.385 L485.243,916.901 L486.391,916.235 L487.657,916.133 L488.884,916.336 L488.977,915.956 L485.878,915.216 L485.784,915.595 L485.47,916.254 L484.476,917.048 L483.34,917.711 L482.611,917.962 L482.869,918.728 L483.084,920.721 L483.348,921.186 L483.754,921.512 L484.003,922.098 L484.179,922.742 L484.379,923.239 L484.176,923.317 L484.004,923.444 L483.809,923.537 L483.566,923.526 L483.686,923.967 L483.921,924.21 L484.157,924.361 L484.269,924.538 L484.253,924.968 L484.069,926.296 L481.584,924.993 L480.625,924.063 L480.273,924.188 L480.022,924.456 L479.765,924.805 L479.392,925.172 L479.695,925.511 L480.312,925.98 L480.547,926.253 L480.955,927.828 L481,928.202 L480.71,928.645 L480.191,928.844 L479.583,928.819 L479.026,928.611 L478.922,929.027 L479.162,929.085 L479.067,929.465 L478.375,929.172 L477.15,927.676 L476.813,928.076 L476.954,928.26 L477.115,928.754 L477.278,929.033 L476.77,929.541 L476.314,929.857 L475.284,930.352 L475.41,930.509 L475.522,930.731 L475.641,930.881 L474.797,931.415 L472.465,932.317 L471.709,932.812 L471.499,932.65 L470.347,930.276 L470.079,930.01 L468.752,927.034 L468.468,926.679 L468.433,926.205 L468.315,925.678 L468.396,925.3 L468.94,925.271 L469.321,925.454 L469.521,925.803 L469.566,926.352 L469.501,927.139 L472.783,925.289 L473.36,924.706 L474.151,924.643 L475.691,925.079 L474.433,923.483 L474.538,923.067 L476.427,922.497 L477.45,922.506 L478.356,922.965 L478.86,923.637 L478.626,923.259 L478.379,922.918 L478.177,922.622 L478.113,922.425 L478.07,922.201 L478.778,921.947 L479.046,921.956 L479.15,921.54 L477.279,921.086 L476.887,920.778 L476.264,919.687 L475.829,919.368 L473.947,919.003 L473.174,918.467 L473.152,917.432 L472.655,917.679 L472.382,917.602 L471.167,916.06 L471.798,915.747 L472.31,915.325 L472.823,915.064 L473.454,915.253 L473.549,914.879 L473.487,914.718 L473.478,914.656 L473.481,914.575 L473.461,914.375 L474.476,914.458 L475.623,914.336 L477.857,913.681 L474.617,913.79 L473.672,913.542 L473.962,913.021 L474.289,912.788 L474.637,912.792 L474.982,912.979 L475.079,912.6 L474.183,912.38 L474.526,912.251 L474.846,912.037 L475.139,911.73 L475.403,911.318 L473.747,912.002 L473.48,911.968 L473.042,909.754 L472.871,909.37 L473.224,908.685 L473.111,908.373 L472.864,908.17 L472.803,907.807 L473.098,907.058 L473.595,906.612 L473.728,906.005 L474.031,905.969 L474.195,905.857 L474.332,905.656 L474.564,905.365 L474.359,905.204 L474.192,904.903 L474.046,904.756 L474.142,904.38 L476.587,904.812 L477.318,904.68 L476.578,904.078 L476.326,903.666 L476.375,902.901 L476.634,902.17 L476.927,901.807 L477.734,901.207 L478.112,901.695 L479.181,902.683 L479.421,902.781 L479.712,902.631 L481.257,903.873 L481.152,904.289 L480.216,903.993 L480.212,904.908 L480.615,906.111 L480.9,906.675 L480.624,907.602 L480.635,908.952 L480.832,910.228 L481.105,910.941 L481.183,910.379 L481.253,910.038 L481.532,909.238 L481.404,908.796 L481.191,907.616 L481.15,906.975 L481.529,906.822 L481.754,906.411 L481.901,905.997 L482.05,905.833 L483.565,905.816 L483.948,905.63 L484.525,905.595 L485.249,906.215 L485.829,907.167 L485.988,908.144 L487.183,907.072 L486.508,906.376 L486.316,905.98 L486.283,905.578 L486.337,904.583 L486.273,904.205 L486.105,904.083 L485.598,904.103 L485.42,904 L485.399,903.829 L485.414,903.249 L485.4,903.072 L485.171,902.433 L485.088,902.092 L485.045,901.7 L485.238,901.62 L485.404,901.503 L485.585,901.418 L485.807,901.445 L485.751,900.963 L485.672,900.639 L485.495,900.46 L485.142,900.398 L488.338,898.679 L489.265,897.866 L489.598,898.19 L489.873,898.091 L490.147,897.841 L490.461,897.711 L491.548,897.972 L492.463,897.917 L494.75,897.311 L495.101,897.122 L496.174,896.216 L502.924,893.135 L505.001,891.425 L506.14,890.739 L506.894,890.983 L507.363,891.851 L507.661,893.077 L507.781,894.344 L507.698,895.353 L508.034,896.47 L507.98,897.134 L507.334,897.288 L507.013,897.578 L505.667,899.942 L505.841,900.399 L506.133,900.782"
         id="path1081" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M601.28,901.92 L601.467,902.205 L602.45,902.391 L604.685,902.121 L605.998,902.233 L607.43,903.77 L608.492,903.979 L610.688,903.939 L611.679,904.621 L611.261,905.704 L610.405,906.853 L610.075,907.741 L609.614,908.185 L608.917,909.208 L608.524,909.672 L607.522,910.334 L607.209,910.696 L606.615,911.823 L606.419,912.822 L606.616,913.622 L607.216,914.178 L607.626,914.205 L608.025,914.061 L608.347,914.038 L608.533,914.423 L608.5,915.164 L608.252,915.687 L607.893,915.983 L607.542,916.022 L606.821,918.14 L604.881,920.644 L602.678,922.838 L601.182,924.016 L599.674,924.558 L596.64,925.037 L595.361,925.529 L593.292,926.958 L592.89,927.473 L592.436,928.447 L591.624,929.359 L589.96,930.637 L581.63,935.17 L578.964,935.942 L578.244,936.431 L577.826,937.027 L577.251,938.21 L576.977,938.621 L576.625,938.869 L575.715,939.171 L574.633,939.209 L572.241,940.261 L571.891,940.395 L571.596,940.718 L571.37,941.204 L571.218,941.846 L570.748,941.432 L570.624,941.251 L570.927,940.913 L570.522,940.582 L569.709,940.305 L568.895,939.553 L568.689,939.77 L568.674,940.295 L568.845,940.895 L569.255,941.367 L570.261,941.618 L570.71,941.963 L570.873,942.285 L571.316,943.615 L570.404,944.628 L570.115,945.226 L570.116,945.958 L569.602,945.845 L568.711,945.266 L568.207,945.175 L567.859,945.356 L567.12,946.033 L566.81,946.205 L565.903,945.994 L564.229,944.637 L563.376,944.202 L560.593,944.366 L559.981,943.712 L559.244,942.013 L558.741,941.315 L558.135,940.949 L559.303,943.915 L560.158,945.001 L561.351,945.539 L562.802,945.169 L563.507,945.187 L563.969,945.851 L564.384,946.656 L565.042,947.378 L565.578,947.465 L565.609,946.401 L567.398,948.276 L568.484,948.814 L570.636,947.697 L571.632,948.152 L572.054,949.046 L571.444,949.755 L572.446,950.274 L573.775,949.959 L575.175,949.072 L576.38,947.889 L576.991,947.408 L577.383,947.555 L577.477,948.198 L577.183,949.178 L576.819,949.663 L571.159,954.68 L570.285,955.139 L569.146,956.298 L568.625,956.66 L567.948,956.666 L567.487,956.316 L567.449,955.676 L568.056,954.797 L567.355,953.764 L566.119,953.964 L563.098,955.771 L562.624,955.876 L561.778,955.763 L559.51,955.934 L558.925,955.692 L558.363,955.607 L557.972,956.267 L557.575,957.067 L556.99,957.37 L556.484,957.47 L553.887,959.216 L553.506,959.558 L553.108,960.117 L552.639,961.009 L552.794,961.088 L553.313,960.832 L554.39,960.613 L554.979,960.227 L555.572,959.7 L556.036,959.166 L556.551,958.688 L558.245,958.063 L559.791,956.994 L561.345,956.543 L561.619,956.564 L561.962,956.808 L562.28,957.451 L563.854,958.156 L564.58,958.275 L565.196,957.959 L565.703,957.592 L567.32,956.962 L567.839,956.98 L567.839,957.784 L566.894,958.923 L565.695,959.931 L564.577,960.499 L563.033,962.134 L563.069,962.47 L563.126,962.718 L563.205,962.906 L563.306,963.064 L561.747,962.907 L561.034,963.154 L560.327,963.774 L560.517,964.407 L560.275,964.446 L559.565,964.053 L557.767,963.688 L557.67,964.148 L558.62,964.307 L559.049,964.503 L559.389,964.893 L558.491,966.18 L557.993,966.728 L557.514,967.134 L558.637,967.7 L559.223,967.855 L559.988,967.634 L560.666,967.065 L561,966.713 L561.06,966.543 L561.816,966.632 L562.202,966.522 L563.008,965.731 L563.585,965.352 L564.173,965.116 L564.666,965.079 L565.325,964.903 L565.087,964.153 L564.088,962.784 L564.992,963.027 L568.109,963.149 L571.209,964.088 L572.287,963.975 L578.078,961.178 L580.133,961.098 L579.754,961.727 L579.527,961.985 L579.206,962.229 L579.568,962.669 L579.998,962.716 L580.484,962.602 L581.005,962.578 L580.656,960.805 L581.78,961.469 L582.418,961.673 L583.074,961.666 L583.616,961.396 L584.276,960.829 L584.744,960.108 L584.695,959.366 L585.145,958.794 L585.453,958.796 L585.737,959.043 L586.127,959.203 L586.583,959.111 L587.566,958.73 L587.987,958.685 L592.042,959.639 L594.181,961.73 L598.215,964.051 L599.648,964.369 L601.061,964.189 L603.38,963.08 L603.964,962.982 L609.17,963.931 L609.37,964.107 L609.59,964.769 L609.894,964.972 L610.287,964.911 L610.601,964.752 L610.874,964.754 L611.151,965.199 L612.375,964.851 L615.013,966.342 L616.059,966.471 L616.426,967.412 L616.707,967.366 L617.028,966.934 L617.511,966.729 L618.089,966.904 L619.499,967.987 L619.827,967.561 L620.065,967.696 L620.311,968.035 L620.648,968.19 L621.065,968.043 L621.73,967.587 L622.047,967.529 L622.616,968.043 L622.995,968.223 L623.227,967.953 L623.391,967.624 L623.687,967.334 L624.04,967.136 L624.34,967.062 L625.02,967.317 L626.315,968.477 L626.942,968.816 L627.829,968.783 L630.07,967.618 L630.628,967.577 L634.252,968.333 L634.235,968.571 L634.342,969.14 L634.528,969.762 L634.753,970.155 L635.148,970.273 L635.78,969.881 L636.183,969.962 L636.362,970.259 L636.596,971.152 L638.85,974.79 L638.964,975.898 L638.884,977.039 L638.881,978.163 L639.221,979.231 L639.155,979.613 L638.888,980.365 L639.119,981.105 L639.877,982.335 L639.394,982.314 L639.089,982.742 L639.027,983.277 L639.299,983.586 L639.682,983.821 L639.398,984.164 L638.508,984.714 L636.821,986.356 L636.051,987.373 L635.821,988.211 L631.764,990.912 L630.667,992.116 L627.51,997.48 L626.148,1000.55 L625.737,1001.78 L625.626,1002.94 L626.176,1003.55 L626.185,1004.1 L625.451,1005.14 L625.424,1005.18 L621.526,1009.55 L620.097,1011.59 L619.258,1013.74 L619.265,1014.28 L619.397,1014.69 L619.483,1015.18 L619.345,1015.93 L619.118,1016.25 L618.354,1016.95 L618.163,1017.24 L617.756,1018.32 L617.135,1018.93 L615.81,1019.66 L614.67,1020.8 L613.597,1022.12 L612.928,1022.69 L610.825,1023.57 L610.163,1023.98 L609.433,1024.6 L608.989,1025.21 L608.765,1025.52 L607.905,1028.17 L607.359,1028.85 L606.783,1029.39 L606.276,1030.24 L606.186,1030.72 L606.245,1031.61 L606.201,1031.97 L605.934,1032.37 L605.208,1032.92 L604.657,1033.93 L604.011,1034.5 L600.403,1036.24 L599.581,1037.43 L598.794,1037.81 L597.367,1038.23 L596.975,1038.65 L596.03,1040.16 L595.672,1040.26 L595.46,1040.06 L595.236,1039.76 L594.835,1039.55 L592.928,1039.49 L584.662,1039.2 L584.488,1039.19 L583.057,1039.56 L581.703,1040.26 L577.451,1043.24 L575.902,1043.58 L574.522,1042.81 L574.818,1043.47 L575.37,1043.81 L576.819,1044.08 L577.57,1044.01 L579.178,1043.41 L580.564,1043.33 L588.014,1041.11 L588.444,1040.76 L588.87,1040.23 L589.693,1039.89 L590.539,1039.9 L591.05,1040.39 L591.544,1041.1 L592.729,1041.42 L592.994,1042.26 L592.841,1043.45 L592.385,1044.51 L591.759,1045.34 L591.097,1045.88 L591.935,1046.03 L592.03,1046.24 L592.024,1047.12 L592.048,1047.36 L592.401,1047.61 L592.694,1047.71 L592.92,1047.89 L593.066,1048.37 L596.148,1049.18 L596.825,1049.7 L597.79,1051 L598.036,1051.2 L599.078,1052.53 L599.009,1052.91 L596.854,1054.39 L596.005,1055.5 L595.122,1055.75 L593.401,1055.83 L590.95,1056.94 L590.136,1057.02 L589.321,1056.72 L587.426,1055.22 L585.788,1054.89 L584.912,1054.97 L584.473,1055.36 L584.101,1055.87 L582.033,1056.85 L579.713,1058.69 L579.085,1058.92 L578.351,1059.04 L577.857,1059.63 L576.983,1061.62 L576.422,1062.45 L575.845,1062.71 L573.412,1062.39 L572.667,1062.57 L571.317,1063.17 L569.831,1063.56 L568.44,1063.53 L567.112,1063.19 L561.883,1060.47 L558.82,1059.94 L557.585,1059.24 L556.544,1058.13 L556.515,1058.09 L555.528,1056.66 L554.995,1055.6 L554.582,1055.19 L553.886,1055.03 L553.81,1055.41 L555.568,1057.26 L555.766,1057.79 L556.018,1058.02 L556.777,1060.36 L556.86,1060.85 L557.14,1061.01 L557.825,1061.01 L558.054,1061.08 L558.233,1061.45 L558.315,1061.9 L558.466,1062.31 L558.849,1062.54 L560.876,1062.48 L561.642,1062.64 L567.226,1065.45 L568.849,1065.76 L569.503,1065.58 L569.838,1065.67 L570.108,1066.23 L570.592,1066.86 L576.751,1068.54 L576.958,1068.78 L577.409,1069.77 L577.576,1070.04 L577.991,1070.2 L580.19,1070.38 L583.02,1069.7 L583.952,1069.14 L584.703,1068.29 L585.437,1067.64 L586.314,1067.69 L586.258,1066.91 L586.217,1066.61 L586.102,1066.34 L586.832,1065.9 L588.081,1064.85 L588.839,1064.66 L594.533,1066.12 L595.138,1066.53 L595.866,1068.08 L596.419,1068.59 L595.926,1069.55 L595.684,1069.84 L596.138,1070.1 L597.049,1070.09 L597.501,1070.56 L597.689,1070.34 L598.101,1070.08 L598.379,1069.85 L598.787,1070.47 L600.013,1071.72 L601.252,1072.38 L603.11,1074.62 L604.458,1075.7 L609.83,1077.75 L610.107,1077.97 L610.207,1078.51 L610.158,1079.09 L610.158,1079.56 L610.395,1079.79 L611.086,1080.22 L611.621,1081.11 L612.923,1085.39 L612.923,1085.39 L613.224,1086.37 L616.789,1093.07 L617.054,1094.6 L617.276,1095.42 L617.937,1096.07 L618.052,1096.54 L618.13,1097.01 L618.274,1097.26 L618.618,1097.22 L618.889,1096.96 L619.098,1096.68 L619.251,1096.56 L619.632,1096.68 L620.038,1096.98 L620.374,1097.56 L620.532,1098.51 L619.823,1098.39 L619.758,1098.77 L620.134,1099.1 L621.42,1098.95 L622.162,1099.17 L624.258,1101.29 L624.187,1101.71 L623.927,1102.08 L623.91,1102.54 L624.079,1103.04 L624.376,1103.52 L623.834,1103.82 L623.763,1104.24 L624.032,1104.49 L624.599,1105.25 L624.529,1105.67 L624.145,1106.16 L624.269,1106.49 L624.586,1106.75 L624.787,1107.05 L624.818,1107.82 L624.254,1111.99 L624.093,1112.58 L623.743,1113.56 L623.659,1113.89 L623.562,1114.42 L623.549,1115.81 L624.155,1117.55 L624.072,1118.65 L623.981,1118.93 L623.865,1119.2 L623.718,1119.45 L623.541,1119.69 L623.313,1120.12 L623.31,1120.54 L623.384,1120.96 L623.394,1121.38 L623.472,1122.13 L623.8,1122.89 L624.461,1123.92 L624.039,1124.68 L624.155,1125.09 L624.447,1125.48 L624.566,1126.14 L624.365,1126.74 L624.039,1127.23 L623.789,1127.74 L623.832,1128.41 L624.284,1129.45 L624.379,1129.93 L624.28,1131.28 L624.375,1132.65 L624.44,1133.06 L624.559,1133.19 L624.984,1133.47 L625.097,1133.6 L625.122,1133.78 L625.159,1134.31 L625.484,1135.97 L625.715,1136.58 L625.807,1137.16 L625.841,1137.97 L625.96,1138.71 L626.315,1139.07 L626.405,1139.27 L627.225,1140.48 L627.386,1141.19 L627.358,1141.57 L627.25,1141.84 L627.172,1142.21 L626.989,1143.6 L626.99,1144.16 L627.21,1144.42 L627.022,1144.98 L627.883,1153.01 L628.33,1155.08 L629.315,1156.93 L629.799,1157.43 L631.553,1158.58 L631.479,1159.04 L630.75,1158.93 L630.711,1159.21 L630.741,1159.41 L630.881,1159.82 L630.89,1160.33 L630.957,1160.8 L631.106,1161.23 L631.345,1161.62 L630.966,1162.11 L630.575,1162.36 L629.661,1162.62 L629.681,1163.08 L629.815,1163.36 L630.05,1163.51 L630.392,1163.6 L630.507,1163.51 L630.571,1163.31 L630.649,1163.16 L630.813,1163.23 L630.883,1163.4 L630.984,1163.73 L631.026,1164.04 L630.942,1164.16 L631.388,1164.14 L631.652,1163.85 L632.238,1162.16 L632.731,1163.09 L633.257,1163.7 L633.911,1164.07 L635.39,1164.51 L636.948,1165.96 L643.969,1169.4 L651.275,1175.77 L652.143,1176.94 L652.44,1177.86 L652.394,1178.43 L652.257,1178.9 L652.278,1179.55 L652.575,1180.13 L653.64,1181.2 L654.071,1181.78 L654.469,1182.84 L654.87,1184.41 L655.077,1186.04 L654.907,1187.29 L654.973,1187.72 L655.114,1188.16 L655.35,1188.52 L655.709,1188.7 L655.464,1189.32 L655.331,1189.56 L655.584,1189.92 L656.116,1191.03 L656.348,1191.39 L657.767,1192.01 L658.092,1192.33 L658.457,1192.83 L659.23,1193.37 L659.41,1194.04 L659.427,1195.72 L659.469,1195.99 L660.283,1197.15 L664.474,1199.58 L665.608,1200.97 L662.136,1202.3 L660.829,1203.44 L659.573,1205.77 L660.127,1207.62 L660.591,1211.83 L660.967,1213.78 L663.978,1220.42 L669.159,1231.17 L669.456,1233.3 L668.722,1235.25 L668.48,1235.45 L667.594,1236 L668.66,1234.2 L668.977,1233.17 L668.46,1232.62 L667.732,1232.26 L666.303,1230.86 L665.548,1230.45 L664.799,1230.53 L663.248,1231.22 L662.442,1231.32 L661.558,1231.11 L660.752,1230.73 L660.061,1230.23 L659.496,1229.61 L656.651,1224.53 L655.504,1223.38 L654.241,1222.99 L653.899,1222.88 L649.048,1223.79 L648.368,1223.69 L646.138,1223.37 L644.848,1223.48 L644.604,1223.34 L643.831,1222.31 L643.745,1222.15 L641.661,1221.58 L640.842,1221.72 L638.089,1223.04 L638.405,1223.21 L638.7,1223.44 L638.971,1223.73 L639.209,1224.07 L639.906,1223.17 L640.993,1222.84 L642.115,1222.93 L642.937,1223.29 L644.268,1224.6 L644.598,1225.09 L645.088,1225.53 L645.705,1225.39 L646.843,1224.72 L653.594,1224.45 L654.553,1225.04 L655.071,1226.39 L657.114,1230.89 L658.632,1231.98 L659.574,1233.51 L660.143,1234.06 L661.404,1234.44 L661.901,1234.74 L663.316,1237.15 L666.447,1240.56 L667.109,1240.97 L667.787,1241.23 L668.404,1241.62 L668.896,1242.4 L668.946,1242.88 L668.94,1243.53 L668.962,1244.11 L669.139,1244.38 L669.46,1244.54 L670.423,1245.46 L670.085,1245.84 L670.662,1246.42 L671.12,1247.36 L671.401,1248.4 L671.445,1249.26 L673.338,1258.07 L673.545,1260.48 L673.202,1263.14 L671.987,1266.29 L671.707,1266.69 L671.268,1266.81 L670.506,1266.86 L669.843,1267.17 L666.041,1270.12 L665.309,1270.31 L664.743,1270.65 L661.2,1274.61 L660.298,1275.35 L659.367,1275.59 L658.731,1275.92 L658.102,1276.71 L658.06,1277.48 L659.187,1277.72 L661.193,1277.53 L661.657,1277.79 L662.578,1278.81 L663.063,1279.08 L663.869,1279.82 L665.39,1283.09 L666.151,1284.33 L666.885,1283.92 L667.695,1283.91 L669.171,1284.26 L669.734,1284.7 L670.585,1286.36 L671.097,1286.72 L671.154,1286.29 L671.085,1285.3 L671.998,1284.24 L673.127,1283.26 L673.729,1282.5 L674.132,1281.01 L674.83,1279.4 L675.618,1277.96 L676.258,1276.97 L677.732,1275.78 L679.692,1275.15 L681.757,1275.02 L683.562,1275.33 L683.342,1274.83 L684.139,1274.65 L684.939,1274.96 L685.73,1275.42 L686.503,1275.71 L689.692,1276.12 L691.133,1276.7 L692.733,1277.75 L694.451,1278.42 L696.253,1277.85 L695.896,1277.51 L695.472,1277.29 L695.01,1277.17 L694.52,1277.16 L694.573,1276.74 L705.788,1280.97 L710.058,1283.87 L718.543,1292.15 L719.495,1293.53 L720.21,1295.17 L720.58,1298.2 L721.063,1300.09 L721.348,1301.97 L720.95,1303.27 L721.071,1304.25 L721.001,1305.49 L720.702,1307.73 L721.263,1310.75 L721.045,1311.29 L720.495,1311.83 L719.172,1314.94 L719.137,1315.55 L718.982,1316.13 L716.785,1320.41 L715.323,1322.47 L714.861,1322.88 L714.377,1323.54 L714.105,1325.13 L713.88,1327.94 L713.535,1329.12 L712.597,1331.19 L712.317,1332.29 L711.575,1333.91 L711.412,1334.77 L711.17,1335.08 L708.013,1336.01 L707.281,1336.68 L706.767,1336.16 L706.508,1336.65 L706.301,1337.44 L705.944,1337.81 L705.656,1338 L704.561,1339.33 L701.525,1341.99 L701.008,1342.63 L700.557,1342.75 L698.36,1339.06 L697.274,1338.56 L695.56,1336.78 L694.539,1336.44 L694.487,1336.87 L695.694,1338.33 L696.421,1339.01 L698.13,1339.64 L698.507,1339.89 L698.642,1340.19 L698.493,1341.19 L698.329,1341.45 L697.576,1341.16 L697.116,1341.17 L696.266,1341.34 L695.804,1341.34 L695.339,1341.15 L694.357,1340.3 L693.498,1340.26 L691.559,1340.66 L690.567,1340.69 L691.518,1341.3 L697.856,1342.49 L698.676,1342.11 L698.402,1343.26 L694.995,1346.02 L695.437,1346.45 L696.005,1346.79 L697.384,1347.17 L697.828,1346.83 L698.192,1346.22 L698.369,1346.17 L698.274,1347.49 L697.7,1348.42 L695.256,1350.2 L693.115,1351.31 L691.804,1351.68 L689.062,1351.75 L688.352,1351.56 L688.031,1351.19 L687.574,1349.85 L686.537,1347.97 L686.296,1347.1 L686.043,1348.27 L685.383,1348.87 L683.876,1349.39 L682.55,1350.35 L682.249,1350.47 L682.185,1350.81 L681.979,1351.54 L681.71,1352.26 L681.467,1352.57 L681.002,1352.63 L680.316,1353.09 L679.898,1353.23 L679.429,1353.17 L678.626,1352.69 L678.265,1352.56 L677.302,1352.54 L675.434,1353.01 L674.511,1353.41 L675.075,1353.78 L675.765,1353.97 L676.371,1354.38 L676.688,1355.42 L677.917,1354.47 L680.449,1354.24 L681.687,1353.89 L682.91,1353.12 L683.606,1353.08 L684.128,1353.77 L684.134,1354.3 L683.949,1354.89 L683.731,1355.34 L683.63,1355.43 L683.647,1355.93 L683.734,1356.44 L683.941,1357.23 L683.51,1357.52 L683.3,1358.03 L683.126,1359.48 L682.952,1359.79 L682.606,1360.06 L681.961,1360.43 L683.012,1360.49 L683.356,1360.75 L683.466,1361.05 L682.188,1362.7 L679.985,1364.25 L678.108,1365.12 L677.606,1365.35 L675.796,1365.69 L672.798,1364.79 L671.719,1364.73 L671.698,1364.7 L671.19,1364.62 L671.035,1364.63 L670.932,1364.63 L670.87,1364.72 L670.029,1365.34 L669.435,1365.45 L668.913,1365.41 L668.396,1365.47 L667.811,1365.91 L667.266,1365.24 L667.05,1365.22 L666.2,1365.16 L663.428,1365.52 L663.167,1365.94 L662.975,1366.55 L662.638,1367.22 L662.047,1367.77 L661.447,1368.02 L660.842,1368.16 L660.232,1368.41 L662.76,1368.31 L663.11,1368.16 L663.347,1367.74 L663.537,1367.32 L663.745,1367.15 L666.361,1366.78 L672.434,1368.49 L672.95,1368.97 L673.319,1369.92 L673.29,1370.52 L672.859,1370.81 L672.042,1370.81 L670.692,1370.32 L670.087,1370.27 L668.711,1371.29 L667.248,1371.45 L666.482,1371.82 L666.737,1372.21 L666.946,1372.67 L667.185,1373.08 L667.53,1373.29 L670.862,1373.72 L672.398,1374.35 L672.182,1373.58 L672.115,1372.7 L672.384,1372.12 L673.197,1372.25 L673.172,1373.06 L673.45,1374.25 L673.87,1375.34 L674.24,1375.84 L682.383,1377.82 L687.37,1376.92 L700.048,1377.26 L700.929,1377.94 L700.336,1379.94 L700.15,1380.32 L699.923,1380.67 L699.666,1380.94 L699.389,1381.12 L698.004,1380.94 L697.769,1381.14 L697.728,1382.49 L697.887,1383.57 L698.332,1385.3 L698.384,1387.8 L697.781,1390.14 L696.754,1392.03 L695.538,1393.19 L692.822,1393.78 L691.652,1394.47 L691.205,1394.47 L690.368,1394.29 L689.884,1394.4 L689.452,1394.75 L689.065,1395.15 L688.729,1395.42 L687.914,1395.65 L685.346,1395.41 L684.557,1395.52 L683.7,1395.96 L681.991,1397.18 L679.627,1399.87 L679.149,1400.55 L679.053,1402.17 L679.177,1404.63 L677.75,1405.01 L674.918,1403.66 L672.382,1402.45 L670.758,1402.77 L666.437,1406.04 L664.768,1406.56 L656.031,1407.27 L654.486,1407.73 L654.29,1407.69 L653.782,1408.22 L653.309,1408.82 L653.194,1409.05 L651.736,1409.76 L650.003,1411.65 L648.542,1411.47 L647.781,1411.25 L646.072,1410.33 L645.654,1410.2 L644.265,1410.02 L643.938,1409.77 L643.343,1409 L642.649,1408.65 L640.997,1408.3 L640.364,1407.97 L633.264,1404.19 L631.99,1403.88 L631.874,1403.93 L630.853,1404.3 L629.503,1403.52 L624.236,1404.62 L617.522,1403.64 L610.792,1404.86 L610.464,1404.69 L610.181,1404.35 L609.883,1404.1 L609.495,1404.19 L609.199,1404.7 L609.295,1405.91 L609.016,1406.52 L608.211,1406.83 L607.357,1406.41 L604.121,1403.57 L603.887,1403.16 L604.201,1402.56 L604.786,1402.23 L605.984,1401.94 L606.041,1401.55 L605.694,1401.21 L605.044,1400.32 L604.629,1400.01 L604.409,1400.19 L603.892,1400.39 L603.62,1400.27 L604.132,1399.5 L603.746,1399.2 L603.408,1398.94 L602.208,1398.52 L601.268,1398.37 L600.994,1398.33 L600.224,1398.43 L599.557,1399.46 L598.978,1400.89 L598.269,1401.98 L597.211,1401.95 L596.555,1400.93 L597.087,1399.9 L597.963,1398.95 L598.331,1398.14 L597.68,1397.65 L597.647,1397.65 L595.208,1397.47 L594.597,1397.57 L595.79,1398.93 L595.963,1399.36 L595.88,1400.58 L595.635,1401.21 L595.047,1401.62 L594.54,1401.51 L593.869,1401.12 L593.295,1400.62 L593.08,1400.19 L592.907,1399.57 L591.751,1398.7 L590.95,1397.75 L589.731,1397.1 L589.255,1396.74 L588.712,1395.82 L588.069,1394.71 L584.144,1391.6 L583.535,1391.11 L583.474,1391.5 L583.832,1391.76 L584.481,1392.33 L584.951,1392.49 L585.724,1393.81 L588.238,1397.72 L588.451,1398.61 L588.219,1399.33 L587.289,1399.51 L584.747,1399.34 L584.141,1399.44 L584.393,1399.71 L584.606,1399.99 L584.546,1400.38 L582.712,1400.43 L581.648,1400.64 L581.151,1400.92 L580.974,1400.92 L579.728,1400.48 L578.089,1402.61 L577.963,1402.85 L577.891,1402.86 L577.162,1402.51 L576.414,1401.81 L573.815,1400.86 L573.726,1400.85 L572.938,1400.73 L572.034,1400.85 L570.402,1401.41 L569.663,1401.46 L568.688,1401.11 L567.26,1400.59 L566.379,1400.49 L565.558,1400.61 L564.749,1400.9 L564.726,1400.91 L563.974,1401.33 L563.262,1401.95 L562.785,1402.19 L562.685,1401.74 L562.718,1401.09 L562.649,1400.74 L561.216,1400.07 L560.179,1399.97 L559.876,1399.85 L559.777,1399.59 L559.761,1399.51 L559.585,1398.72 L559.413,1398.51 L559.004,1398.77 L557.286,1400.76 L557.811,1400.78 L558.301,1400.68 L558.781,1400.48 L559.272,1400.18 L559.863,1401.84 L560.243,1402.38 L560.934,1402.67 L561.588,1402.77 L562.141,1402.99 L562.333,1403.51 L561.884,1404.52 L562.401,1404.99 L562.611,1405.11 L560.972,1406.58 L560.903,1407.51 L560.121,1407.74 L556.916,1407.27 L556.273,1407.02 L553.561,1405.12 L546.443,1403.07 L545.24,1401.99 L544.387,1401.97 L542.833,1402.22 L542.273,1402.73 L541.561,1403.97 L541.118,1405.25 L541.376,1405.91 L542.31,1406.4 L542.275,1407.21 L541.695,1408.11 L540.995,1408.89 L540.752,1408.85 L541.118,1406.5 L540.473,1404.81 L533.373,1397.05 L527.109,1393.33 L526.237,1393.14 L523.874,1393.26 L523.175,1393.3 L521.627,1393.98 L520.686,1393.92 L519.284,1393.23 L518.805,1393.29 L517.783,1393.87 L517.564,1393.95 L516.815,1394.4 L516.41,1394.5 L516.01,1394.38 L515.32,1393.91 L514.893,1393.74 L514.125,1393.73 L512.706,1394.11 L511.049,1394.2 L510.486,1394.74 L509.984,1395.49 L509.24,1396.23 L506.177,1396.96 L505.303,1396.76 L504.687,1396.42 L504.255,1395.82 L503.942,1394.81 L503.705,1393.24 L503.592,1392.96 L503.492,1392.88 L503.433,1392.73 L503.333,1392.58 L503.117,1392.48 L502.963,1392.58 L502.865,1392.87 L502.853,1393.16 L503.238,1393.81 L503.365,1396.03 L503.555,1396.91 L502.736,1397.45 L501.95,1398.19 L501.19,1399.13 L500.446,1400.29 L499.701,1401.9 L499.514,1403.12 L499.726,1405.82 L498.843,1405.72 L497.924,1405.78 L497.153,1406.16 L496.721,1407 L496.805,1408.07 L497.423,1408.53 L499.183,1408.77 L499.105,1409.19 L498.515,1410.19 L498.442,1410.41 L498.077,1410.4 L497.745,1410.52 L497.484,1410.8 L496.932,1412.29 L496.401,1412.54 L495.825,1412.22 L495.296,1411.56 L494.756,1412.89 L493.825,1413.24 L492.899,1413.31 L492.38,1413.83 L491.975,1414.83 L491.335,1416.04 L490.839,1417.45 L490.854,1419.05 L490.206,1419.24 L488.536,1419.4 L487.85,1419.35 L487.01,1418.96 L486.361,1418.47 L485.761,1418.27 L485.059,1418.81 L483.844,1417.19 L482.11,1413.9 L479.981,1411.03 L479.585,1410.74 L479.207,1410.77 L478.095,1411.15 L477.683,1411.21 L477.431,1411.3 L476.789,1411.8 L476.441,1411.88 L476.024,1411.7 L475.666,1411.38 L475.149,1410.71 L475.28,1410.51 L475.545,1409.96 L474.731,1409.55 L473.982,1408.74 L473.753,1407.72 L474.515,1406.66 L473.056,1406.48 L472.645,1406.3 L472.213,1405.85 L472.057,1405.41 L471.984,1404.88 L471.791,1404.19 L471.689,1403.11 L472.154,1402.3 L472.845,1401.58 L473.437,1400.78 L472.584,1401.02 L472.195,1400.9 L471.834,1400.46 L471.616,1400.93 L471.542,1401.09 L471.157,1401.48 L470.719,1401.61 L470.246,1401.5 L470.172,1401.88 L470.664,1402.38 L470.702,1403.03 L470.435,1403.66 L469.999,1404.07 L469.46,1404.13 L467.597,1403.59 L468.072,1404.68 L470.366,1404.75 L471.114,1405.6 L469.679,1405.71 L469.597,1406.13 L469.943,1406.11 L470.692,1406.35 L470.311,1406.75 L471.406,1406.97 L471.331,1407.35 L470.859,1407.42 L470.534,1407.7 L470.407,1408.11 L470.535,1408.58 L470.461,1408.96 L469.639,1408.74 L469.367,1407.91 L469.168,1406.96 L468.574,1406.41 L465.405,1404.86 L463.568,1404.7 L460.375,1405.94 L458.723,1406.23 L452.211,1405.3 L452.2,1405.41 L452.043,1405.56 L451.813,1405.67 L451.593,1405.62 L451.394,1403.83 L451.123,1403.73 L450.862,1403.84 L450.611,1404 L450.364,1404.06 L449.558,1403.97 L448.7,1404.07 L448.161,1404.49 L448.308,1405.34 L448.215,1405.8 L447.899,1405.97 L447.455,1406.3 L447.044,1406.68 L446.827,1407.04 L446.808,1407.43 L446.935,1407.89 L446.882,1408.15 L446.183,1409.2 L446.047,1409.53 L445.568,1409.88 L443.688,1409.17 L442.88,1409.33 L441.97,1410.07 L441.204,1410.33 L439.426,1410.54 L438.788,1410.98 L436.567,1413.49 L436.249,1412.63 L436.467,1410.4 L436.058,1409.84 L435.238,1409.9 L435.1,1410.44 L435.025,1411.16 L434.389,1411.72 L434.31,1412.1 L434.511,1412.33 L434.958,1413.15 L434.634,1413.06 L434.352,1413.12 L433.799,1413.35 L433.368,1413.55 L433.224,1413.99 L433.146,1414.5 L432.895,1414.91 L432.491,1415.11 L431.115,1415.41 L431.563,1415.49 L431.881,1415.65 L432.368,1416.11 L432.557,1416.18 L433.029,1416.19 L433.195,1416.29 L433.231,1416.57 L433.202,1417.42 L433.327,1417.63 L433.391,1418.18 L432.649,1419.21 L431.174,1420.68 L430.793,1420.69 L430.032,1420.37 L429.539,1420.33 L429.221,1420.51 L427.767,1421.66 L427.013,1423 L426.81,1423.3 L426.486,1423.09 L426.151,1422.67 L425.245,1421.1 L425.115,1420.76 L425.036,1420.3 L425.058,1419.5 L425.191,1419.01 L425.266,1418.57 L425.114,1417.91 L424.373,1415.82 L423.908,1414.92 L423.257,1414.19 L421.133,1412.55 L419.605,1411.91 L418.466,1410.83 L417.87,1410.4 L417.043,1410.28 L415.941,1410.42 L415.06,1410.89 L414.892,1411.73 L414.9,1412.85 L414.297,1413.58 L413.474,1414 L412.823,1414.17 L411.102,1414.35 L409.178,1414.14 L407.821,1413.24 L407.81,1411.28 L408.199,1411.32 L408.505,1411.26 L408.728,1411.05 L408.814,1410.67 L408.771,1410.35 L408.681,1410.16 L408.615,1410.15 L408.651,1408.94 L408.719,1408.28 L408.93,1408.02 L409.047,1407.84 L410.062,1406.71 L410.416,1406.57 L411.658,1406.39 L414.465,1405.47 L415.901,1405.29 L417.613,1404.63 L418.472,1404.8 L419.437,1406.23 L420.142,1406.79 L421.053,1406.47 L421.811,1405.67 L422.216,1405.13 L422.453,1404.58 L422.709,1404.39 L424.103,1404.69 L425.724,1404.18 L431.189,1400.39 L432.835,1399.69 L433.406,1399.62 L433.604,1399.33 L433.784,1398.63 L433.889,1397.85 L433.887,1396.56 L434.39,1396.34 L434.99,1396.42 L435.277,1396.51 L438.29,1395.61 L438.444,1395.28 L438.983,1394.52 L439.824,1392.21 L440.125,1390.97 L440.474,1389.94 L441.024,1389.17 L441.931,1388.71 L442.726,1388.88 L443.054,1388.73 L443.682,1388.24 L443.992,1388.23 L444.147,1388.61 L444.17,1389.36 L444.102,1390.22 L443.992,1390.93 L444.287,1390.82 L444.407,1390.75 L444.594,1390.58 L445.21,1391.16 L446.807,1391.58 L447.37,1392.03 L447.109,1391.12 L446.513,1390.57 L445.33,1389.82 L444.884,1388.9 L445.064,1388.34 L445.365,1387.87 L445.289,1387.18 L445.368,1386.8 L446.293,1387.2 L449.229,1387.6 L450.422,1387.46 L451.221,1386.68 L451.896,1385.66 L452.715,1384.78 L452.66,1383.81 L452.929,1383.18 L453.378,1382.96 L453.881,1383.23 L457.113,1381.4 L457.721,1380.73 L458.24,1379.71 L459.138,1379.02 L461.002,1378.12 L462.482,1376.19 L463.382,1370.73 L464.323,1368.27 L464.694,1367.81 L465.55,1366.37 L465.49,1365.87 L465.522,1364.96 L465.659,1364.09 L465.925,1363.74 L469.362,1364.43 L469.755,1364.66 L470.433,1365.42 L470.792,1365.64 L472.983,1366.33 L473.82,1366.24 L474.481,1365.94 L477.667,1363.51 L477.952,1363.2 L478.197,1362.86 L478.5,1362.58 L478.952,1362.42 L478.826,1361.78 L478.69,1360.01 L478.693,1359.55 L478.637,1359.14 L478.174,1358.01 L478.019,1357.43 L479.284,1357.79 L479.594,1357.74 L479.973,1357.12 L480.044,1356.46 L479.861,1355.88 L479.487,1355.5 L479.562,1355.11 L482.722,1354.51 L489.882,1355.23 L491.43,1354.95 L492.177,1354.97 L494.958,1355.51 L495.833,1355.5 L497.184,1355.08 L497.885,1355.15 L502.615,1357.42 L505.732,1357.9 L510.342,1359.93 L511.289,1361.24 L511.969,1361.74 L516.411,1362.36 L519.342,1361.99 L520.771,1362.05 L523.292,1362.65 L524.845,1362.7 L526.07,1362.14 L526.282,1362.62 L525.595,1363.4 L526.048,1363.92 L526.935,1362.76 L527.148,1362.34 L527.358,1361.56 L527.354,1361.15 L527.28,1360.79 L527.262,1360.18 L527.697,1357.88 L527.793,1356.68 L527.491,1355.87 L528.197,1356.24 L528.623,1356.14 L528.628,1356.13 L528.87,1355.59 L529.066,1354.64 L529.471,1354.13 L530.106,1353.66 L530.455,1353.13 L529.984,1352.41 L530.977,1351.68 L532.16,1352.15 L533.822,1351.01 L537.045,1347.98 L537.827,1347.55 L538.577,1347.27 L539.372,1347.17 L540.291,1347.25 L541.164,1347.06 L542.083,1346.39 L546.491,1341.13 L546.849,1340.94 L547.276,1340.83 L547.66,1340.62 L547.894,1340.09 L548.072,1339.6 L548.382,1339.07 L548.729,1338.59 L549.016,1338.27 L551.605,1336.33 L553.222,1334.76 L553.944,1334.5 L555.679,1334.21 L556.403,1333.75 L556.821,1333.09 L556.842,1332.37 L556.375,1331.72 L556.012,1332.84 L555.214,1333.37 L554.285,1333.51 L553.515,1333.45 L552.75,1333.6 L552.109,1334.18 L551.502,1334.93 L550.83,1335.59 L548.289,1336.71 L547.86,1337.04 L547.408,1337.58 L544.396,1340.13 L544.395,1340.13 L543.079,1341.24 L542.234,1341.7 L540.346,1341.8 L538.729,1342.55 L537.941,1342.92 L536.216,1343.21 L532.934,1342.85 L532.474,1342.57 L532.253,1342.08 L532.126,1341.6 L531.96,1341.38 L531.553,1341.58 L530.322,1342.67 L530.042,1343.03 L527.56,1344.06 L527.448,1344.08 L526.941,1344.16 L526.214,1344.53 L525.026,1345.38 L524.85,1345.61 L524.497,1346.28 L524.273,1346.55 L523.972,1346.67 L523.412,1346.64 L523.162,1346.77 L523.147,1346.78 L523.394,1348.18 L522.883,1349.45 L522.061,1350.12 L521.368,1349.74 L520.811,1349.56 L517.892,1350.2 L508.47,1347.53 L507.845,1347 L507.337,1345.92 L506.502,1344.15 L504.617,1341.84 L504.098,1341.69 L503.328,1341.9 L502.549,1341.3 L501.94,1340.4 L501.659,1339.7 L501.658,1339.42 L501.744,1339.33 L501.871,1339.31 L501.982,1339.29 L501.927,1338.02 L501.888,1337.69 L501.779,1336.8 L501.546,1335.7 L501.241,1334.78 L500.996,1334.4 L500.414,1333.83 L500.148,1333.31 L499.773,1332.02 L499.502,1331.58 L499.024,1331.34 L497.779,1331.26 L494.827,1331.05 L494.24,1331.18 L493.667,1331.41 L493.257,1331.73 L493.102,1332.3 L493.277,1332.96 L493.562,1333.62 L493.734,1334.21 L492.147,1333.9 L492.119,1334.06 L491.885,1334.3 L491.646,1334.37 L491.606,1334.03 L491.547,1333.77 L491.332,1333.53 L491.057,1333.34 L490.812,1333.24 L490.665,1333.35 L490.197,1333.86 L489.996,1333.95 L489.761,1333.82 L489.422,1333.43 L488.518,1333.02 L488.159,1332.92 L487.72,1333.03 L487.246,1333.32 L486.58,1333.87 L486.168,1334.03 L485.804,1333.85 L485.43,1333.49 L485.093,1333.38 L484.82,1334 L484.335,1333.89 L482.27,1332.48 L481.615,1332.29 L481.528,1332.05 L481.492,1331.86 L481.497,1331.39 L482.332,1330.99 L482.365,1330.32 L482.182,1329.46 L482.363,1328.51 L483.746,1328.35 L483.928,1328.18 L484.486,1327.5 L484.804,1327.21 L484.95,1327.72 L484.713,1327.67 L484.638,1328.05 L484.861,1328.3 L485.07,1328.61 L486.326,1328.45 L488.295,1328.71 L490.205,1328.6 L491.277,1327.38 L491.372,1326.97 L491.415,1326.64 L491.428,1326.53 L491.265,1326.38 L490.706,1326.85 L490.35,1327.02 L489.906,1326.98 L488.436,1326.51 L486.601,1324.95 L485.978,1325.22 L482.892,1325.08 L482.134,1324.6 L481.269,1323.57 L480.569,1322.3 L480.308,1321.07 L481.83,1321.6 L482.535,1321.65 L483.005,1321.17 L482.302,1321.08 L481.636,1320.76 L481.139,1320.21 L480.933,1319.41 L481.076,1319.3 L481.383,1318.92 L481.682,1318.38 L481.784,1317.83 L481.608,1317.35 L481.328,1317.49 L480.902,1318.09 L480.646,1318.26 L480.24,1318.7 L479.979,1318.78 L479.612,1318.61 L479.346,1318.25 L479.11,1317.73 L478.353,1317.71 L478.133,1318.11 L478.316,1318.75 L478.765,1319.45 L477.322,1319.9 L475.863,1319.66 L473.083,1318.69 L471.95,1318.59 L470.516,1318.68 L470.269,1318.7 L468.839,1319.35 L468.451,1320.85 L468.089,1320.85 L467.985,1320.94 L467.936,1321.18 L467.753,1321.62 L466.7,1323.22 L466.378,1323.03 L465.955,1322.48 L465.653,1322.62 L464.854,1323.17 L461.485,1322.39 L460.879,1322.4 L460.207,1322.69 L459.776,1323.08 L459.362,1323.56 L458.927,1323.94 L458.141,1324.11 L457.703,1324.63 L457.328,1324.72 L457.151,1324.61 L456.821,1324.21 L454.515,1322.87 L453.736,1322.01 L454.218,1321.44 L453.915,1320.13 L452.144,1319.25 L451.59,1318.26 L451.918,1317.89 L454.744,1318.92 L454.707,1318.7 L454.706,1318.21 L454.668,1317.99 L455.231,1318.09 L456.151,1318.64 L456.676,1318.85 L457.219,1318.83 L458.648,1318.35 L460.563,1318.35 L461.619,1317.9 L462.026,1317.88 L462.471,1318.27 L462.55,1317.88 L461.691,1316.57 L461.787,1315.03 L462.669,1313.91 L464.191,1313.85 L464.285,1313.39 L463.78,1313.61 L462.588,1313.84 L460.414,1313.59 L459.888,1313.79 L460.508,1313.91 L460.912,1314.26 L460.972,1314.86 L460.568,1315.72 L461.004,1316.32 L460.726,1316.98 L460.034,1317.48 L456.945,1317.56 L456.115,1317.42 L450.984,1315.58 L450.259,1315.7 L450.425,1316.78 L449.872,1317.22 L449.337,1317.08 L449.561,1316.43 L449.641,1316.05 L448.074,1314.2 L447.388,1313.82 L448.2,1313.97 L448.761,1313.98 L449.246,1313.76 L449.823,1313.22 L450.477,1312.81 L451.803,1312.82 L452.516,1312.67 L453.24,1312.18 L453.483,1311.62 L453.61,1310.04 L453.77,1309.07 L453.807,1308.12 L453.65,1307.3 L453.233,1306.7 L452.887,1307.06 L452.335,1306.33 L450.892,1305.8 L450.316,1305.2 L449.692,1305.49 L447.502,1305.48 L446.764,1305.32 L447.064,1305.15 L447.358,1305.05 L447.456,1304.59 L447.387,1304.29 L447.359,1304.08 L447.49,1303.91 L447.901,1303.76 L447.982,1303.39 L447.854,1302.54 L448.287,1302.25 L449.67,1302.43 L450.431,1302.31 L451.813,1301.37 L452.36,1301.21 L453.557,1301.28 L455.22,1301.03 L456.627,1300.23 L457.074,1298.66 L457.611,1298.78 L457.69,1298.4 L457.323,1297.91 L457.468,1297.51 L457.948,1297.28 L458.598,1297.28 L459.604,1297.56 L460.012,1297.78 L461.791,1299.29 L462.514,1299.45 L463.695,1299.22 L463.612,1298.33 L464.299,1298.6 L465.625,1299.48 L466.418,1299.79 L466.816,1299.6 L466.88,1298.99 L466.857,1298.26 L466.981,1297.71 L467.317,1297.53 L468.62,1297.58 L469.458,1297.21 L470.139,1296.65 L471.376,1295.3 L472.226,1294.67 L472.745,1294.77 L473.787,1296.02 L474.074,1294.18 L475.655,1293.77 L480.272,1294.89 L480.74,1294.81 L481.049,1294.55 L481.315,1294.16 L481.64,1293.84 L482.824,1293.72 L483.675,1293.36 L484.543,1292.82 L485.934,1291.69 L486.545,1291.44 L487.2,1291.39 L487.975,1291.5 L488.713,1291.43 L493.942,1288.44 L496.74,1285.73 L498.27,1283.79 L499.044,1282.5 L499.888,1279.97 L500.48,1278.62 L501.142,1277.43 L501.74,1276.64 L501.654,1276.03 L501.718,1275.27 L501.916,1274.55 L502.243,1274.08 L502.556,1274.07 L503.288,1274.67 L503.66,1274.83 L504.359,1274.72 L506.56,1273.65 L506.649,1273.2 L505.634,1273.01 L503.581,1273.32 L502.528,1273.3 L502.048,1272.72 L500.812,1270.01 L500.575,1268.98 L500.998,1267.48 L501.906,1266.07 L502.925,1265.05 L503.675,1264.71 L503.843,1264.56 L504.183,1263.77 L504.317,1263.53 L504.478,1263.47 L504.907,1263.39 L506.663,1262.49 L507.262,1261.96 L506.255,1261.89 L505.512,1262.28 L504.858,1262.3 L502.711,1258.25 L502.297,1256.6 L502.237,1256 L502.441,1255.51 L503.125,1255.43 L503.64,1255.08 L503.739,1254.11 L503.632,1253.1 L503.548,1252.66 L503.728,1251.79 L503.889,1251.39 L504.17,1251.18 L506.691,1250.02 L506.442,1249.85 L506.345,1249.72 L506.279,1249.46 L505.76,1249.86 L505.28,1250.07 L504.926,1249.9 L504.765,1249.16 L503.664,1249.71 L502.641,1249.59 L500.606,1248.82 L497.769,1249.12 L497.077,1249.38 L496.246,1249.68 L493.689,1249.14 L492.844,1249.39 L492.017,1249.86 L490.392,1251.13 L490.498,1251.32 L490.536,1251.6 L489.874,1251.84 L489.376,1252.47 L489.201,1253.26 L489.519,1254.01 L488.942,1254.42 L488.629,1254.58 L488.335,1254.65 L487.899,1254.5 L487.511,1253.77 L486.773,1253.32 L486.182,1252.6 L485.782,1252.38 L485.32,1252.39 L485.011,1252.59 L484.742,1252.82 L484.436,1252.98 L483.567,1253.03 L481.541,1252.38 L480.921,1252.62 L480.278,1253.13 L479.7,1253.27 L479.206,1252.78 L479.707,1252.29 L479.803,1252.02 L479.733,1251.61 L480.025,1251.36 L480.761,1250.47 L487.136,1245.51 L487.757,1245.31 L488.467,1245.39 L489.196,1245.32 L489.996,1245.08 L493.446,1243.35 L494.429,1242.23 L497.166,1241.13 L497.632,1240.39 L498.013,1239.32 L498.596,1236.77 L498.851,1236.08 L499.142,1235.88 L499.363,1236.05 L499.387,1236.49 L499.456,1236.92 L499.864,1236.49 L500.283,1235.83 L500.379,1235.58 L502.215,1234.08 L502.686,1233.86 L503.399,1233.71 L504.021,1233.12 L505.103,1231.68 L506.137,1230.82 L507.238,1230.37 L508.38,1230.25 L509.559,1230.37 L509.845,1230.53 L510.105,1230.77 L510.359,1230.92 L510.893,1230.69 L511.771,1230.46 L512.676,1230.39 L514.146,1229.57 L518.49,1228.67 L519.738,1228.92 L519.586,1227.72 L518.729,1226.39 L518.592,1225.6 L519.253,1225.8 L519.923,1226.16 L520.492,1226.67 L520.85,1227.34 L521.646,1227.09 L522.389,1227.51 L523.012,1228.35 L523.432,1229.39 L523.935,1229.87 L528.15,1230.55 L529.68,1230.49 L537.329,1228.32 L538.872,1228.55 L540.134,1229.51 L540.794,1230.27 L541.017,1230.95 L541.34,1231.52 L545.495,1235.49 L545.918,1236.23 L546.232,1237.09 L546.451,1237.43 L546.614,1237.21 L546.84,1235.97 L546.841,1235.97 L547.032,1234.93 L546.746,1234.13 L546.362,1233.06 L545.825,1230.99 L544.823,1228.99 L544.493,1227.97 L544.628,1227.01 L545.14,1226.46 L545.877,1226.22 L547.487,1226.26 L548.29,1226.11 L549.807,1225.28 L550.628,1225.49 L551.292,1226.7 L552.422,1232.82 L552.915,1233.83 L553.673,1234.81 L554.653,1235.49 L555.8,1235.65 L555.859,1235.53 L555.953,1235.27 L556.13,1235.01 L556.432,1234.89 L556.758,1234.97 L557.266,1235.36 L557.519,1235.48 L558.131,1235.54 L558.684,1235.48 L559.801,1235.06 L559.739,1234.23 L559.942,1233.68 L560.303,1233.31 L560.687,1233 L560.967,1233.21 L561.296,1233.22 L561.658,1233.06 L562.046,1232.77 L561.525,1232.64 L559.595,1232.76 L559.145,1232.94 L558.725,1233.8 L558.025,1234.02 L557.645,1233.9 L556.443,1233.55 L554.976,1232.67 L553.745,1231.4 L552.8,1229.92 L552.173,1228.39 L549.649,1218.26 L549.61,1217.38 L549.859,1216.53 L550.471,1215.64 L553.613,1212.07 L556.02,1209.88 L556.363,1209.71 L556.796,1209.12 L558.455,1208.34 L559.024,1207.78 L555.905,1207.18 L554.682,1206.33 L553.943,1204.68 L554.063,1202.57 L555.339,1197.99 L555.229,1196.2 L556.006,1195.24 L556.939,1194.87 L560.505,1194.2 L561.113,1194.23 L561.521,1194.7 L561.776,1194.85 L561.942,1194.6 L562.132,1194.25 L562.473,1193.98 L562.843,1193.8 L563.126,1193.72 L562.844,1192.95 L564.34,1191.88 L564.681,1190.91 L563.963,1191.27 L563.29,1191.41 L562.159,1191.33 L562.014,1191.14 L561.687,1189.7 L561.698,1189.21 L562.012,1188.9 L562.447,1188.67 L562.821,1188.4 L563.765,1187.21 L564.369,1186.72 L565.617,1186.44 L566.148,1185.9 L567.138,1184.4 L566.694,1183.38 L566.154,1181.52 L565.822,1180.37 L565.809,1179.77 L566.635,1179.6 L567.593,1178.95 L568.333,1177.96 L568.522,1176.77 L568.276,1176.73 L567.97,1177.63 L567.323,1178.28 L566.095,1178.95 L564.676,1179.42 L563.92,1179.83 L563.51,1180.44 L563.063,1181.31 L562.306,1181.62 L560.661,1181.44 L560.247,1180.96 L559.921,1179.97 L559.888,1178.95 L560.334,1178.37 L560.207,1177.61 L559.989,1176.96 L559.633,1176.61 L559.105,1176.76 L559.111,1177.64 L558.837,1178.48 L558.482,1179.25 L558.244,1179.91 L557.991,1180.4 L556.663,1181.58 L555.813,1182.76 L555.331,1183.29 L554.243,1183.65 L554.033,1184.34 L553.988,1185.2 L553.822,1185.89 L552.887,1183.86 L552.612,1183.49 L552.044,1183.34 L551.57,1183.11 L551.204,1182.77 L550.93,1182.27 L551.212,1181.34 L551.611,1180.32 L551.756,1179.42 L551.295,1178.85 L551.375,1178.43 L552.642,1178.27 L553.282,1176.53 L553.591,1174.43 L553.886,1173.19 L553.258,1173.42 L552.841,1174 L552.204,1175.49 L552.294,1176.75 L551.702,1176.94 L550.888,1176.71 L550.295,1176.72 L549.484,1176.91 L548.852,1175.97 L548,1173.4 L547.189,1171.85 L546.873,1170.94 L546.862,1169.91 L547.043,1169.46 L547.297,1168.98 L547.469,1168.44 L547.373,1167.52 L547.422,1166.88 L547.341,1166.74 L546.567,1166.15 L545.839,1163.35 L545.145,1162.43 L542.698,1157.03 L542.179,1156.6 L542.039,1156.36 L541.837,1155.4 L541.544,1154.9 L541.897,1154.49 L542.361,1154.12 L542.555,1154.01 L543.727,1152.02 L545.603,1147.38 L547.34,1144.82 L548.137,1143.39 L548.485,1142.89 L548.836,1142.68 L549.671,1142.53 L550.104,1142.33 L550.67,1141.81 L551.428,1140.91 L551.957,1139.79 L551.832,1138.69 L553.681,1135.29 L554.699,1133.74 L555.722,1132.88 L556.445,1132.84 L556.793,1133.25 L557.082,1133.77 L557.614,1134.11 L558.003,1134.13 L559.608,1133.61 L559.226,1133.18 L557.632,1131.93 L558.736,1131.09 L559.345,1130.78 L561.696,1130.51 L562.115,1130.73 L562.333,1131 L562.487,1131.29 L562.72,1131.57 L563.956,1132.27 L564.655,1132.45 L567.766,1132.1 L568.475,1131.77 L566.04,1131.32 L566.119,1130.9 L566.819,1130.91 L567.426,1130.65 L568.008,1130.53 L568.632,1130.92 L568.718,1130.46 L567.93,1129.9 L567.782,1129.91 L567.781,1129.91 L564.619,1130.18 L553.534,1127.24 L552.971,1126.96 L552.483,1126.54 L551.996,1126.26 L551.455,1126.37 L551.533,1126.51 L551.562,1126.53 L551.571,1126.59 L551.581,1126.87 L550.932,1127.04 L550.415,1126.71 L549.58,1125.61 L548.792,1127.21 L548.943,1127.92 L548.884,1129.17 L548.692,1130.52 L548.435,1131.5 L547.847,1132.42 L547.19,1132.66 L543.951,1131.7 L543.008,1131.67 L541.989,1132.82 L539.561,1131.93 L539.579,1132.27 L539.614,1132.29 L539.712,1132.44 L539.347,1132.6 L539.093,1132.5 L538.92,1132.21 L538.813,1131.79 L539.363,1131.11 L539.52,1131.23 L539.637,1131.55 L539.676,1131.08 L539.657,1130.87 L539.562,1130.66 L539.006,1131.27 L538.353,1131.7 L538.388,1132.37 L538.421,1132.62 L538.174,1132.64 L537.607,1132.94 L538.57,1134.4 L538.168,1134.62 L536.157,1135.09 L532.956,1136.67 L531.227,1136.57 L530.139,1136.28 L530.231,1135.82 L530.106,1135.72 L529.992,1135.67 L529.899,1135.56 L529.844,1135.38 L529.98,1135.22 L530.123,1135.02 L530.151,1134.97 L530.203,1134.33 L530.507,1133.56 L530.553,1132.92 L530.116,1132.6 L530.158,1132.31 L529.552,1132.66 L528.968,1133.47 L528.635,1134.49 L528.781,1135.53 L527.348,1135.49 L525.784,1133.95 L524.964,1131.61 L525.768,1129.21 L525.178,1129.43 L524.092,1130.36 L523.467,1130.54 L522.952,1130.36 L520.582,1128.56 L520.239,1128.14 L520.02,1127.65 L519.866,1126.7 L519.839,1125.99 L519.682,1125.52 L519.146,1125.25 L518.877,1125.47 L518.472,1126.04 L518.075,1126.75 L517.833,1127.4 L517.862,1128.92 L517.951,1129.88 L518.098,1130.32 L518.663,1130.47 L519.138,1130.72 L519.525,1131.13 L519.83,1131.79 L519.781,1132.6 L519.329,1133.24 L519,1133.95 L519.321,1134.95 L518.69,1135.75 L518.54,1137.06 L518.483,1138.26 L518.146,1138.72 L517.919,1138.81 L517.685,1139.06 L517.392,1139.3 L516.992,1139.36 L516.207,1139.08 L515.524,1138.59 L514.77,1137.71 L514.469,1137.5 L513.202,1137.28 L512.939,1136.96 L511.547,1133.36 L510.622,1131.96 L507.01,1128.48 L505.515,1127.69 L504.858,1127.06 L504.571,1125.19 L503.887,1124.7 L502.54,1124.45 L501.616,1124.65 L500.677,1125.11 L499.813,1125.72 L499.108,1126.39 L498.689,1126.87 L498.468,1127.2 L498.359,1127.51 L498.267,1127.96 L498.22,1128.5 L498.242,1129.25 L498.377,1129.94 L498.676,1130.28 L498.778,1131 L498.868,1132.53 L499.122,1134.1 L499.949,1135.2 L499.848,1135.74 L499.426,1136.81 L499.295,1137.38 L499.245,1137.68 L499.491,1138.77 L498.772,1138.41 L497.43,1137.27 L496.741,1136.92 L496.492,1136.43 L496.481,1135.41 L496.729,1134.41 L497.265,1133.92 L497.079,1133.49 L497.022,1133.05 L497.107,1132.63 L497.36,1132.2 L496.716,1131.98 L496.453,1131.8 L496.267,1131.53 L496.227,1130.94 L496.257,1130.04 L496.208,1129.2 L495.939,1128.79 L495.3,1128.3 L492.832,1123.5 L492.217,1121.43 L492.045,1119.11 L492.397,1116.63 L492.988,1114.44 L493.469,1113.38 L494.067,1113 L494.799,1113.2 L495.198,1113.23 L495.41,1113.07 L495.726,1112.39 L496.002,1112.8 L496.194,1113.55 L496.251,1113.91 L496.588,1114.89 L496.721,1115.53 L496.755,1116.21 L496.637,1116.89 L496.263,1118.36 L496.296,1118.96 L496.621,1119.73 L497.083,1120.4 L497.69,1120.64 L498.447,1120.11 L498.615,1119.9 L498.805,1119.71 L498.896,1119.29 L498.188,1116.46 L498.109,1115.86 L498.122,1115.33 L498.124,1115.23 L497.895,1113.91 L497.937,1113.17 L498.252,1111.81 L498.415,1111.41 L498.674,1110.92 L499.616,1109.55 L500.035,1108.77 L500.402,1107.54 L500.941,1106.46 L501.776,1105.91 L503.412,1105.27 L506.345,1102.64 L506.545,1102.53 L506.925,1102.51 L507.035,1102.33 L507.165,1101.67 L507.259,1101.37 L508.499,1099.47 L508.754,1098.95 L508.91,1098.35 L509.107,1097 L509.292,1096.43 L509.564,1096.1 L511.032,1094.89 L511.779,1094.48 L512.133,1094.18 L512.466,1093.63 L513.061,1092.26 L513.542,1091.63 L514.881,1090.83 L517.427,1090.21 L518.525,1089.16 L519.174,1087.55 L519.174,1086 L518.567,1084.77 L517.398,1084.07 L518.358,1083.11 L518.364,1083.04 L518.478,1081.71 L518.001,1080.28 L517.171,1079.21 L515.464,1077.97 L514.36,1077.7 L514.084,1077.41 L514.465,1076.89 L514.563,1076.43 L512.847,1074.68 L511.949,1073.42 L511.49,1072.28 L513.366,1070.96 L514.176,1069.76 L514.476,1067.66 L514.412,1066.87 L514.317,1066.3 L514.285,1065.63 L514.652,1062.94 L514.656,1062.89 L514.89,1061.17 L515.457,1059.57 L516.181,1058.6 L517.108,1058.09 L518.277,1057.86 L518.949,1057.85 L519.402,1058.01 L521.357,1059.5 L521.916,1059.72 L522.347,1059.63 L522.984,1060.24 L528.511,1062.33 L529.464,1061.93 L525.896,1060.35 L524.605,1059.78 L523.318,1058.51 L522.046,1055.91 L520.772,1055.08 L520.282,1054.3 L519.917,1053.37 L519.696,1052.5 L519.643,1050.57 L519.686,1050.27 L519.206,1050.03 L518.93,1050.47 L518.817,1051.08 L518.865,1051.89 L519.954,1054.71 L520.1,1055.9 L519.775,1056.35 L519.122,1056.34 L518.289,1056.15 L517.559,1055.63 L517.378,1054.68 L517.511,1053.55 L517.966,1051.2 L518.127,1050.7 L520.222,1047.45 L523.486,1044.02 L523.829,1043.17 L523.722,1042.9 L523.297,1043.1 L522.691,1043.65 L519.851,1047.36 L519.203,1047.98 L518.463,1047.86 L518.311,1047.33 L518.774,1045.46 L518.81,1044.6 L518.592,1043.9 L518.168,1043.63 L517.588,1044.09 L517.739,1045.27 L517.743,1047.54 L517.596,1049.71 L517.29,1050.64 L516.814,1050.73 L516.63,1051.18 L516.509,1052.45 L516.25,1053.65 L516.142,1054.49 L516.174,1055.23 L515.65,1055.37 L515.173,1055.08 L514.693,1054.65 L514.14,1054.36 L514.049,1054.78 L514.438,1055.08 L514.82,1055.49 L515.073,1056.05 L515.083,1056.79 L514.765,1057.64 L514.298,1058.17 L513.798,1058.58 L513.388,1059.06 L513.064,1059.74 L512.837,1060.32 L512.569,1060.86 L512.097,1061.41 L511.493,1061.85 L511.079,1061.84 L510.045,1061.37 L509.64,1060.58 L509.492,1058.91 L509.529,1057.08 L509.659,1055.84 L508.695,1053.68 L508.688,1052.87 L508.54,1052.31 L507.895,1052.63 L508.217,1053.75 L508.789,1057.37 L508.764,1058.71 L508.349,1059.24 L507.745,1058.87 L505.897,1056.51 L505.759,1055.66 L506.041,1054.38 L505.59,1054.77 L505.111,1055.51 L504.712,1056.32 L504.503,1056.91 L504.24,1057.34 L502.903,1058.61 L503.342,1060.51 L503.476,1061.53 L503.479,1062.68 L502.849,1062.54 L501.54,1061.34 L500.483,1060.67 L500.231,1060.12 L500.357,1058.53 L500.203,1058.27 L499.965,1058.06 L499.775,1057.69 L499.753,1057.23 L499.851,1056.65 L500.129,1055.53 L500.311,1055.07 L500.802,1054.47 L500.992,1053.97 L501.047,1053.43 L501.019,1051.54 L502.077,1050.75 L507.258,1044.99 L509.42,1044.58 L511.095,1043.76 L511.503,1043.47 L513.327,1040.92 L514.049,1040.23 L515.226,1039.76 L516.656,1039.46 L518.034,1038.83 L519.051,1037.35 L518.363,1037.64 L516.798,1038.78 L516.159,1038.97 L514.748,1038.78 L514.12,1038.91 L513.505,1039.28 L512.97,1039.96 L511.681,1041.93 L511.262,1042.31 L506.584,1044.22 L502.792,1047.97 L502.442,1048.1 L501.813,1048.01 L501.512,1048.13 L501.457,1048.45 L501.095,1049.28 L500.719,1049.98 L500.629,1049.9 L500.157,1050.52 L499.366,1050.59 L498.538,1050.33 L497.941,1049.99 L497.981,1049.69 L498.095,1049.16 L498.101,1048.75 L497.818,1048.84 L497.626,1049.14 L497.529,1049.52 L497.459,1049.93 L496.468,1053.17 L496.56,1053.33 L497.011,1053.72 L497.194,1053.81 L497.262,1055.1 L497.244,1055.84 L497.14,1056.42 L497.153,1056.93 L497.476,1057.43 L497.392,1057.81 L497.157,1057.76 L497.064,1058.17 L497.195,1058.28 L497.471,1058.66 L497.277,1058.75 L497.068,1058.92 L496.878,1059.01 L498.316,1061.5 L498.851,1063.15 L498.812,1064.78 L498.39,1065.58 L497.821,1065.88 L496.57,1066 L495.966,1066.26 L493.986,1067.71 L493.475,1068.21 L493.001,1068.81 L492.582,1069.52 L492.384,1070.01 L492.053,1071.07 L491.857,1071.56 L490.987,1071.97 L490.992,1072.25 L491.066,1072.56 L491.096,1072.75 L490.727,1074.13 L490.958,1074.58 L491.351,1074.79 L491.476,1075.46 L491.085,1076.22 L490.485,1076.34 L489.996,1076.61 L489.962,1077.81 L488.902,1079.1 L488.557,1079.65 L488.45,1079.97 L488.224,1081.14 L488.066,1081.63 L487.944,1081.64 L487.806,1081.52 L487.616,1081.64 L486.385,1083.53 L485.643,1084.24 L484.731,1084.56 L484.645,1084.94 L485.093,1084.97 L485.408,1085.19 L485.646,1085.59 L485.867,1086.13 L486.12,1087.64 L486.091,1087.89 L486.407,1088.38 L485.85,1089.23 L484.339,1090.62 L483.634,1091.1 L482.826,1091.49 L481.965,1091.72 L481.105,1091.7 L480.79,1091.55 L480.189,1091.05 L479.785,1090.93 L479.382,1091.02 L478.465,1091.42 L478.035,1091.46 L477.032,1091.22 L476.47,1090.84 L476.274,1090 L476.463,1087.29 L476.709,1085.88 L477.165,1084.73 L478.323,1084.12 L479.994,1082.85 L480.476,1082.3 L480.783,1081.01 L480.928,1079.59 L481.178,1078.21 L482.077,1076.51 L482.031,1076.03 L481.888,1075.57 L481.887,1075.1 L482.17,1074.45 L483.4,1072.8 L484.617,1070.82 L485.031,1069.62 L484.827,1068.69 L487.062,1067.5 L487.463,1066.86 L487.869,1066.06 L488.62,1065.12 L489.41,1064.35 L489.954,1064.07 L491.038,1063.85 L492.479,1062.99 L493.909,1061.61 L494.958,1059.91 L490.781,1062.98 L489.326,1063.72 L488.875,1063.81 L488.417,1063.73 L488.831,1062.99 L487.972,1062.41 L487.574,1062.02 L487.241,1061.31 L487.172,1060.42 L487.509,1058.77 L487.446,1058.24 L489.389,1056.36 L489.72,1055.91 L490.117,1055.15 L490.777,1054.62 L491.53,1054.16 L492.207,1053.59 L491.348,1053.82 L489.867,1054.95 L488.862,1055.22 L488.1,1055.74 L487.783,1055.69 L487.633,1055.22 L487.583,1054.47 L487.66,1053.78 L492.742,1048.72 L493.2,1048.02 L493.118,1047.72 L493.373,1047.54 L493.634,1047.28 L493.239,1046.99 L492.758,1047.13 L492.205,1047.45 L491.591,1047.67 L491.125,1048.76 L489.257,1050.16 L488.977,1051.52 L488.155,1052.49 L487.984,1052.65 L487.673,1052.44 L487.652,1052.06 L487.806,1051.64 L488.023,1051.3 L488.583,1049.72 L490.403,1047.48 L493.824,1044.29 L493.98,1044.09 L494.094,1043.82 L494.239,1043.59 L494.487,1043.51 L494.693,1043.62 L494.937,1043.81 L495.103,1044.01 L495.091,1044.09 L495.152,1044.24 L495.394,1044.5 L495.654,1044.59 L495.787,1044.24 L495.716,1043.82 L495.544,1043.52 L495.369,1043.33 L495.288,1043.21 L495.288,1042.2 L495.347,1041.66 L495.576,1041.29 L497.062,1039.79 L497.417,1039.29 L497.814,1038.5 L496.716,1038.58 L495.733,1039.36 L494.806,1040.27 L493.872,1040.74 L494.378,1039.53 L495.103,1038.37 L495.947,1037.32 L496.811,1036.48 L496.692,1035.6 L497.254,1035.05 L498.085,1034.77 L498.781,1034.76 L499.214,1034.65 L499.38,1034.21 L499.262,1033.72 L498.852,1033.42 L497.338,1033.56 L496.993,1033.69 L496.582,1034.02 L496.181,1034.26 L495.863,1034.12 L495.785,1033.53 L496.027,1032.86 L496.688,1031.67 L497.615,1029.47 L498.128,1028.75 L498.937,1028.17 L499.476,1028.69 L500.414,1028.79 L502.267,1028.47 L502.352,1028.08 L501.525,1027.96 L499.951,1028.14 L499.241,1027.8 L500.382,1025.92 L500.751,1025.5 L501.17,1025.33 L501.651,1025.22 L501.99,1024.98 L501.959,1024.45 L503.086,1023.43 L504.342,1023.11 L507.812,1023.55 L510.368,1024.93 L511.558,1024.89 L512.639,1024.3 L515.431,1021.81 L517.093,1019.53 L518.026,1018.68 L517.196,1018.93 L516.554,1019.75 L515.944,1020.76 L515.216,1021.59 L513.558,1022.54 L512.481,1023.69 L512.024,1024.01 L510.542,1023.87 L507.136,1022.25 L506.001,1022.9 L505.544,1022.38 L505.521,1019.92 L505.013,1020.04 L504.492,1020.5 L503.551,1021.68 L503.176,1021.2 L505.028,1018.53 L505.416,1017.7 L506.037,1018.69 L506.79,1019.2 L507.558,1019.34 L508.24,1019.2 L510.131,1018.29 L511.079,1018.08 L511.57,1017.86 L512.044,1017.39 L510.755,1017.85 L510.358,1017.87 L509.694,1017.68 L509.544,1017.69 L508.878,1018.02 L508.475,1018.08 L507.607,1018.54 L507.174,1018.57 L506.814,1018.29 L506.411,1017.55 L505.972,1017.39 L506.177,1016.95 L506.343,1016.78 L506.527,1016.69 L506.79,1016.47 L506.988,1016.21 L507.218,1015.71 L507.362,1015.49 L508.612,1014 L509,1013.66 L510.664,1012.69 L510.967,1012.34 L511.257,1012 L510.71,1011.4 L510.812,1010.94 L511.145,1010.83 L512.041,1010.33 L512.868,1010.62 L513.324,1010.67 L514.507,1009.99 L515.262,1010.15 L516.61,1010.92 L517.349,1010.87 L518.865,1010.26 L521.71,1010.21 L522.378,1009.93 L516.565,1009.97 L514.239,1008.14 L514.312,1007.77 L514.84,1006.99 L515.122,1006.73 L517.938,1004.72 L518.774,1003.84 L519.598,1002.75 L517.214,1004.11 L516.822,1004.55 L516.59,1004.92 L516.163,1005.18 L515.693,1005.33 L515.328,1005.34 L515.002,1005.47 L514.781,1005.89 L514.597,1006.43 L514.383,1006.89 L513.737,1007.71 L513.177,1008.13 L512.627,1008.25 L512.023,1008.17 L511.699,1008.24 L511.262,1008.72 L510.496,1008.84 L510.14,1009.13 L509.603,1009.8 L508.367,1010.8 L507.753,1011.05 L507.039,1011.04 L506.64,1011.22 L505.905,1012.23 L505.53,1012.42 L505.078,1012.45 L504.61,1012.67 L504.154,1013.02 L502.339,1014.77 L499.501,1016.96 L496.79,1018.32 L496.543,1018.37 L496.293,1018.07 L496.085,1017.18 L495.764,1016.92 L495.238,1016.69 L494.664,1016.25 L494.173,1015.71 L493.879,1015.14 L494.613,1014.26 L494.865,1013.81 L495.07,1013.21 L494.817,1013.15 L493.161,1014.63 L491.228,1013.73 L487.193,1008.42 L487.09,1007.89 L487.371,1006.94 L487.82,1006.55 L488.349,1006.48 L490.05,1006.75 L491.153,1007.25 L492.101,1008.06 L492.816,1009.19 L493.173,1008.79 L492.523,1007.95 L492.283,1007.46 L492.416,1006.86 L496.522,1005.25 L497.365,1005.33 L499.218,1006.92 L500.023,1007.29 L501.838,1007.55 L502.882,1007.51 L503.7,1007.19 L500.49,1006.91 L499.781,1006.62 L498.346,1005.15 L497.052,1004.48 L492.926,1005.69 L491.574,1005.78 L491.217,1005.71 L490.826,1005.43 L490.2,1004.72 L489.798,1004.5 L487.433,1004.16 L485.127,1002.9 L484.263,1002.79 L483.449,1003.29 L482.743,1003.56 L481.953,1003.19 L481.289,1002.56 L480.951,1002.03 L480.851,1001.65 L480.785,1001.28 L480.776,1000.9 L480.838,1000.49 L481.04,999.919 L481.212,999.927 L481.412,1000.09 L481.696,1000 L482.7,999.017 L483.221,998.744 L488.484,999.366 L488.935,999.257 L490.313,998.636 L490.851,998.546 L491.385,998.817 L491.647,999.238 L491.863,999.735 L492.272,1000.23 L493.852,1001.42 L494.089,1001.72 L494.206,1001.04 L494.195,1000.36 L493.958,999.83 L493.38,999.6 L493.486,999.144 L493.91,999.186 L494.31,999.315 L494.664,999.566 L494.957,999.957 L495.588,999.444 L496.401,999.355 L497.199,999.781 L497.821,1000.12 L495.604,998.303 L495.145,997.384 L495.287,996.714 L495.854,996.347 L499.429,996.625 L500.945,996.313 L502.279,995.448 L501.888,995.133 L501.256,995.071 L500.639,995.239 L499.874,996.144 L499.206,995.973 L497.972,995.324 L498.399,994.775 L500.071,993.595 L498.633,992.973 L495.326,992.973 L493.883,992.198 L495.073,992.145 L496.269,992.338 L496.374,991.887 L496.159,991.655 L495.648,990.8 L496.4,990.216 L497.062,989.045 L497.688,987.693 L498.328,986.569 L499.088,985.945 L500.178,985.478 L501.286,985.342 L502.109,985.702 L502.334,986.156 L502.722,987.495 L502.991,988.097 L503.379,988.577 L503.837,988.944 L504.893,989.363 L506.802,989.324 L508.586,988.864 L508.681,988.445 L508.071,988.491 L507.465,988.683 L506.935,988.744 L506.544,988.411 L505.676,988.773 L505.001,988.405 L504.409,987.773 L503.79,987.357 L503.363,986.9 L503.671,986.144 L503.921,985.312 L503.315,984.616 L503.007,984.594 L502.256,984.758 L501.878,984.734 L501.574,984.503 L501.138,983.816 L500.815,983.615 L500.558,983.072 L501.289,982.17 L503.091,980.671 L503.639,980.085 L503.923,979.915 L504.342,979.965 L505.299,980.289 L505.703,980.078 L506.368,979.858 L507.122,980.372 L508.28,981.768 L508.866,982.283 L512.124,983.043 L512.329,982.884 L512.68,982.498 L513.24,982.585 L513.752,982.86 L513.962,983.024 L514.189,983.074 L514.292,982.617 L513.734,982.055 L510.841,981.93 L509.988,982.076 L509.534,982.046 L509.351,981.768 L508.751,980.491 L508.457,980.051 L505.876,978.577 L505.608,977.857 L506.084,976.864 L507.627,975.447 L508.26,974.695 L507.885,973.52 L508.946,972.539 L510.535,971.953 L511.734,971.952 L512.609,973 L513.363,974.493 L514.287,975.594 L515.675,975.458 L515.769,975.042 L514.467,974.41 L514.223,974.063 L513.938,973.063 L513.702,972.574 L513.425,972.327 L513.208,971.844 L513.844,971.035 L514.837,970.341 L515.707,970.198 L515.811,969.742 L515.087,969.219 L514.419,969.6 L513.834,970.261 L513.354,970.558 L512.743,970.616 L511.394,971.199 L510.681,971.284 L510.01,971.033 L508.893,970.155 L508.281,969.869 L506.75,970.024 L506.086,969.817 L505.991,968.916 L506.174,968.737 L507.376,967.027 L507.715,966.796 L508.535,966.509 L508.89,966.25 L509.25,966.056 L509.429,966.355 L509.524,966.808 L509.638,967.054 L510.439,967.091 L514.533,966.134 L516.318,965.079 L516.992,963.417 L516.464,963.159 L515.365,963.89 L513.381,965.691 L512.374,966.163 L511.637,965.874 L510.928,965.307 L510.007,964.939 L510.929,964.596 L511.303,964.252 L511.355,963.706 L511.248,962.889 L511.279,962.472 L511.13,962.457 L510.487,962.853 L507.593,965.31 L506.663,965.552 L504.885,965.421 L504.212,964.861 L504.212,962.322 L504.285,962.394 L504.47,962.325 L504.66,962.182 L504.759,962.005 L504.718,961.781 L504.526,961.235 L504.513,961.03 L504.691,960.206 L504.897,959.825 L505.344,959.497 L505.051,959.221 L504.103,958.689 L503.922,958.499 L504.005,957.826 L503.947,956.697 L504.062,955.889 L504.262,955.473 L504.951,954.54 L505.168,954.183 L505.41,953.318 L505.578,952.538 L505.834,952.093 L506.336,952.212 L506.331,951.917 L506.39,951.197 L506.384,950.905 L507.021,950.798 L507.577,951.041 L508.054,951.547 L509.389,953.776 L509.847,954.319 L510.185,954.143 L510.546,955.152 L510.783,955.448 L511.606,956.371 L511.941,956.548 L511.844,956.165 L511.82,955.825 L511.817,955.507 L511.785,955.193 L514.243,955.302 L515.541,955.842 L516.315,955.99 L516.544,955.614 L515.741,954.544 L514.337,953.868 L512.837,953.748 L511.745,954.347 L510.936,952.767 L511.153,952.617 L511.565,952.073 L510.522,951.619 L510.172,950.537 L510.041,949.24 L509.662,948.125 L509.21,947.8 L508.721,947.616 L508.378,947.277 L508.383,946.476 L508.744,945.78 L509.585,945.176 L509.527,944.581 L509.633,944.126 L511.01,943.698 L511.64,943.808 L512.349,945.115 L512.77,945.149 L513.222,944.758 L513.635,944.147 L513.298,943.958 L513.137,943.595 L513.017,943.144 L512.783,942.675 L512.457,942.355 L509.891,941.103 L509.943,940.421 L510.473,938.869 L510.583,938.145 L510.784,937.661 L510.791,936.363 L510.93,935.585 L511.208,934.956 L511.588,934.414 L512.045,933.983 L512.552,933.717 L513.454,933.57 L514.313,933.753 L515.098,934.215 L515.775,934.919 L515.457,935.723 L515.277,936.957 L515.213,938.239 L515.251,939.196 L515.424,939.905 L515.746,940.464 L516.185,940.821 L516.722,940.923 L516.544,940.579 L516.464,940.239 L516.383,939.449 L516.799,939.516 L517.095,939.681 L517.613,940.2 L517.676,938.952 L518.011,938.116 L518.25,937.342 L518.022,936.299 L517.044,934.826 L516.789,934.036 L517.27,933.495 L517.035,933.369 L516.961,933.234 L516.926,932.978 L517.372,932.465 L518.361,931.613 L518.737,931.189 L519.066,931.891 L519.365,932.209 L520.558,932.474 L519.987,934.106 L520.624,935.546 L521.788,936.558 L522.8,936.92 L522.694,936.46 L523.209,936.019 L523.996,934.359 L524.452,933.777 L524.864,933.875 L526.191,934.85 L526.555,935.333 L527.219,936.635 L528.158,937.562 L530.3,939.004 L530.012,938.048 L529.733,937.587 L528.881,936.941 L527.791,935.139 L525.921,933.724 L525.719,933.418 L525.75,933.058 L525.703,932.899 L525.725,932.731 L525.962,932.352 L526.18,932.347 L526.361,932.683 L526.585,932.782 L526.932,932.086 L527.462,932.611 L528.066,934.155 L528.684,934.706 L529.06,934.757 L529.751,934.571 L530.114,934.579 L530.494,934.803 L531.107,935.507 L531.396,935.695 L531.743,936.048 L532.666,937.487 L533.897,938.428 L534.458,940.326 L535.024,940.905 L534.911,940.538 L534.809,939.936 L534.748,939.301 L534.761,938.857 L534.68,938.408 L532.07,934.827 L531.327,934.444 L531.429,933.987 L532.068,934.134 L532.426,933.741 L532.536,933.015 L532.417,932.187 L532.1,931.749 L531.104,931.307 L530.74,930.768 L529.841,930.31 L528.891,928.308 L528.106,927.952 L528.214,927.804 L528.241,927.757 L528.432,927.629 L528.104,926.828 L527.519,926.463 L526.172,926.253 L526.34,926.01 L526.451,925.815 L526.584,925.645 L526.815,925.476 L526.911,925.058 L526.29,922.859 L527.092,922.163 L528.24,922.787 L528.645,924.561 L529.486,924.134 L530.127,924.517 L530.82,924.771 L531.822,923.944 L531.618,923.9 L531.916,923.332 L532.019,922.769 L531.978,922.24 L531.851,921.757 L532.295,921.854 L532.39,921.436 L531.829,920.834 L532.33,920.607 L533.303,920.581 L533.795,920.425 L533.898,919.969 L532.822,919.786 L532.337,919.565 L531.967,919.113 L532.295,918.779 L532.123,918.425 L531.912,918.123 L531.668,917.881 L531.385,917.704 L531.518,917.432 L531.784,916.737 L531.92,916.466 L531.507,916.37 L531.237,916.076 L530.281,913.433 L530.54,912.418 L531.101,912.358 L531.767,912.844 L532.905,914.093 L533.416,914.365 L533.994,914.332 L535.468,913.738 L536.204,913.656 L536.884,913.895 L537.439,914.598 L537.783,914.203 L538.305,913.893 L538.819,913.698 L539.131,913.651 L539.662,913.967 L540.442,915.045 L540.837,915.333 L542.478,915.538 L542.939,915.786 L544.779,917.969 L544.605,917.255 L544.322,916.795 L543.597,915.927 L544.101,915.658 L545.056,915.69 L545.564,915.47 L543.612,914.613 L542.977,914.48 L542.678,914.593 L542.192,915.061 L541.821,915.107 L541.526,914.905 L540.991,914.191 L540.668,913.983 L540.266,913.551 L539.638,911.91 L539.382,911.51 L539.394,911.275 L538.997,910.203 L538.903,910.055 L538.832,909.227 L538.831,908.751 L538.861,908.289 L539.5,908.428 L539.322,907.707 L539.475,907.103 L539.837,906.541 L540.285,905.964 L540.386,905.659 L540.438,905.247 L540.514,904.892 L540.667,904.774 L540.896,904.889 L541.264,905.238 L541.448,905.34 L543.018,907.048 L544.371,907.285 L543.644,906.36 L543.431,905.894 L543.642,905.417 L543.527,904.916 L543.165,905.163 L542.808,905.255 L542.458,905.219 L542.11,905.086 L542.212,904.633 L542.527,904.716 L542.629,904.723 L542.723,904.304 L542.517,904.213 L542.405,904.079 L542.169,903.704 L544.182,903.028 L544.886,903.571 L545.692,904.463 L542.869,900.589 L543.117,899.474 L543.299,898.957 L543.612,898.727 L545.536,898.076 L546.128,897.65 L546.714,896.794 L546.498,896.747 L547.028,895.534 L547.425,894.303 L547.872,893.297 L548.551,892.763 L552.45,894.565 L553.091,895.226 L553.496,896.09 L553.429,896.418 L553.382,897.41 L553.393,898.282 L553.499,898.242 L553.325,898.915 L553.123,899.43 L552.88,899.865 L552.573,900.275 L553.265,899.92 L553.644,899.41 L553.841,898.805 L553.978,898.145 L554.315,897.281 L554.677,896.956 L554.947,896.559 L555.016,895.498 L555.471,896.111 L556.166,897.646 L556.64,898.028 L557.382,898.925 L557.765,899.141 L558.05,899.428 L557.962,899.941 L557.713,900.476 L557.498,900.838 L557.014,901.258 L555.634,902.143 L554.722,903.563 L553.856,904.589 L553.37,905.361 L553.986,905.391 L556.361,903.644 L556.704,903.536 L557.123,902.628 L557.741,902.306 L558.45,902.133 L559.166,901.662 L559.75,900.676 L560.181,899.607 L560.695,898.807 L561.551,898.618 L562.87,899.173 L564.082,900.058 L565.396,901.563 L565.298,902.022 L565.207,902.233 L565.173,902.262 L565.111,902.896 L564.169,903.627 L563.455,904.645 L562.768,905.828 L561.913,907.048 L562.519,906.915 L563.18,906.481 L564.333,905.364 L564.424,905.127 L564.446,904.826 L564.49,904.585 L564.642,904.507 L564.793,904.468 L566.399,903.305 L566.524,903.187 L567.945,902.566 L568.894,902.323 L569.223,902.349 L569.813,902.691 L570.576,903.802 L571.256,904.155 L570.897,903.607 L571.299,903.408 L571.567,903.079 L571.787,902.757 L572.05,902.568 L572.425,902.582 L572.965,903.017 L573.388,903.196 L573.709,903.068 L574.124,902.309 L574.524,902.194 L575.745,902.44 L576.371,902.793 L576.649,902.865 L577.057,902.705 L577.429,902.318 L578.051,901.37 L578.383,901.224 L578.69,901.212 L578.755,901.155 L578.763,901.313 L579.109,902.624 L579.251,902.824 L579.997,902.817 L582.649,903.821 L585.488,903.906 L587.006,903.523 L589.548,902.269 L590.901,901.914 L592.976,901.914 L593.997,902.146 L594.659,902.643 L594.276,902.912 L594.136,902.979 L594.765,903.859 L595.679,903.957 L597.638,903.653 L599.256,904.815 L600.136,904.972 L600.806,903.822 L600.31,903.481 L599.81,902.936 L599.417,902.203 L599.241,901.305 L600.172,900.448 L600.722,900.216 L601.234,900.415 L601.294,900.766 L601.263,901.355 L601.28,901.92"
         id="path1083" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['GB'],polycolors['GB'],labels['GB'], x1, polybary['GB'], x2, polybary['GB'] )
    if 'GB' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1454.83,1696.06 L1455.84,1696.54 L1458.69,1697.88 L1459.46,1697.93 L1462.85,1695.97 L1463.48,1695.61 L1465.16,1695 L1467.94,1694.88 L1469.88,1695.66 L1471.08,1695.35 L1472.1,1694.65 L1472.51,1694.54 L1473.24,1694.61 L1473.49,1694.77 L1474.01,1695.32 L1474.37,1695.45 L1474.68,1695.37 L1475.69,1694.7 L1476.51,1694.91 L1479.11,1696.59 L1482.27,1699.59 L1482.91,1700.45 L1484.19,1703.16 L1484.99,1704.06 L1485.8,1704.51 L1487.6,1705.03 L1488.48,1705.46 L1491.04,1707.25 L1491.88,1707.51 L1493.55,1707.46 L1494.4,1707.67 L1495.16,1708.24 L1496.94,1710.61 L1497.29,1710.85 L1498.11,1711.25 L1498.46,1711.59 L1498.68,1712.09 L1498.95,1713.33 L1499.03,1713.53 L1499.19,1713.91 L1499.71,1714.55 L1500.88,1715.69 L1501.4,1716.47 L1501.99,1716.87 L1502.59,1716.91 L1503.14,1716.65 L1503.63,1716.13 L1503.78,1715.37 L1504.09,1714.79 L1504.54,1714.52 L1505.11,1714.72 L1505.55,1715.27 L1505.62,1715.84 L1505.53,1716.45 L1505.52,1717.1 L1506,1720.31 L1506.01,1721.21 L1505.91,1721.77 L1505.58,1722.97 L1505.46,1724.57 L1505.76,1725.55 L1507.34,1727.57 L1508.1,1728.92 L1508.11,1729.81 L1507.57,1730.54 L1506.7,1731.47 L1507.1,1732.3 L1507.79,1732.63 L1508.6,1732.76 L1509.37,1733.01 L1510.08,1733.56 L1511.8,1735.44 L1514.97,1737.58 L1516.96,1739.55 L1517.71,1740 L1519.34,1740.47 L1521.54,1741.1 L1522.04,1741.39 L1522.47,1741.76 L1522.9,1742.04 L1523.43,1742.09 L1523.83,1741.83 L1524.42,1740.94 L1524.85,1740.76 L1525.63,1741.18 L1527.44,1743.16 L1528.27,1743.9 L1529.87,1744.7 L1530.7,1744.91 L1532.55,1744.68 L1537.4,1745.42 L1538.73,1745.91 L1539.98,1746.65 L1541.27,1747.79 L1541.73,1748.2 L1542.05,1748.99 L1542,1749.79 L1541.52,1750.36 L1540.98,1750.47 L1540.56,1750.78 L1540.24,1751.22 L1539.98,1751.78 L1540.05,1751.86 L1540.14,1752.24 L1540.2,1752.69 L1540.22,1753 L1540.14,1753.28 L1539.83,1753.79 L1539.74,1754.07 L1539.7,1754.96 L1539.74,1755.4 L1539.64,1755.72 L1539.15,1756.26 L1538.72,1756.49 L1537.61,1756.75 L1537.21,1757.03 L1536.97,1757.57 L1536.85,1758.25 L1537.02,1758.75 L1537.63,1758.82 L1538.38,1758.74 L1538.56,1759.04 L1538.52,1759.46 L1538.59,1759.79 L1539.71,1760.33 L1541.87,1760.37 L1543.05,1760.74 L1544.67,1761.1 L1545.4,1761.58 L1545.9,1762.52 L1545.95,1763.72 L1545.41,1764.35 L1543.7,1765.13 L1542.49,1765.51 L1539.92,1765.78 L1538.81,1766.38 L1538.31,1766.98 L1538.01,1767.71 L1537.97,1768.59 L1538.77,1768.8 L1538.77,1768.8 L1538.8,1768.81 L1539.68,1769.29 L1540.68,1770.81 L1541.45,1771.03 L1545.25,1770.87 L1547.91,1770.23 L1548.81,1770.34 L1549.69,1770.94 L1550.22,1771.87 L1550.65,1772.94 L1551.23,1773.99 L1553.08,1775.36 L1555.33,1775.67 L1557.6,1775.3 L1559.52,1774.58 L1561.02,1773.62 L1561.75,1773.3 L1562.5,1773.33 L1566.8,1774.77 L1568.24,1774.77 L1568.74,1775.14 L1570.43,1777.91 L1570.59,1778.34 L1570.74,1778.74 L1571.02,1779.49 L1571.33,1779.93 L1571.86,1780.29 L1572.41,1780.51 L1572.88,1780.85 L1573.62,1782.56 L1574.5,1783.18 L1575.56,1783.45 L1576.53,1783.4 L1577.54,1782.95 L1578.02,1782.26 L1581.62,1772.38 L1581.73,1771.36 L1582.3,1770.76 L1584.06,1770.04 L1584.83,1769.38 L1586.09,1767.44 L1586.5,1767 L1587.58,1766.86 L1588.42,1766.89 L1589.13,1767.17 L1592.73,1770.03 L1594.43,1771 L1596.03,1771.48 L1599.57,1771.3 L1600.29,1771.66 L1602.91,1773.76 L1602.83,1774.92 L1602.19,1775.85 L1601.28,1776.48 L1600.46,1776.78 L1599.47,1776.71 L1597.69,1775.91 L1596.71,1775.69 L1595.78,1775.81 L1594.95,1776.22 L1594.4,1776.96 L1594.31,1778.03 L1594.34,1779.12 L1594.07,1779.78 L1593.55,1780.14 L1592.01,1780.47 L1591.62,1780.67 L1591.36,1781 L1591.26,1781.61 L1591.28,1782.46 L1591.4,1783.26 L1591.61,1783.74 L1592.67,1784.44 L1592.93,1784.84 L1593.05,1785.29 L1593.1,1786.28 L1593.23,1786.75 L1594.29,1788.41 L1595.72,1789.82 L1597.41,1790.83 L1599.23,1791.31 L1601.1,1791.26 L1602,1791.51 L1602.49,1792.29 L1602.79,1795.28 L1603.17,1796.15 L1603.22,1796.17 L1601.46,1797.39 L1601.14,1797.87 L1600.64,1799.09 L1600.06,1800.05 L1599.97,1800.15 L1599.97,1800.53 L1600.07,1800.85 L1600.19,1801.03 L1600.25,1801.02 L1600.09,1802.93 L1600.11,1803.64 L1600.26,1804.23 L1600.75,1805.49 L1600.84,1806.03 L1600.47,1807.07 L1599.71,1807.36 L1598.79,1807.45 L1597.93,1807.93 L1597.54,1808.79 L1597.34,1809.73 L1596.94,1810.46 L1595.95,1810.66 L1595.28,1810.88 L1593.91,1811.35 L1592.98,1811.95 L1592.57,1812.88 L1592.59,1813.66 L1592.84,1815.45 L1592.89,1816.43 L1592.82,1816.73 L1592.51,1817.62 L1592.43,1817.96 L1592.63,1819.82 L1592.5,1821.79 L1592.15,1823.31 L1592.13,1824.69 L1592.98,1826.26 L1594.35,1827.46 L1594.48,1827.93 L1594.47,1828.99 L1594.56,1829.5 L1594.89,1830.28 L1595.3,1831.03 L1595.77,1831.73 L1596.28,1832.31 L1596.92,1832.73 L1598.36,1833.24 L1598.89,1833.61 L1599.32,1834.38 L1599.58,1835.78 L1600.04,1836.49 L1600.06,1836.53 L1600.08,1836.56 L1600.07,1836.59 L1600.06,1836.62 L1599.95,1836.87 L1599.94,1837.1 L1600.02,1837.32 L1600.2,1837.51 L1600.28,1837.65 L1600.33,1837.79 L1600.33,1837.96 L1600.3,1838.12 L1600.22,1838.45 L1600.23,1838.77 L1600.31,1839.07 L1601.04,1840.1 L1602.34,1842.89 L1602.95,1844 L1603.66,1844.48 L1605.26,1844.93 L1605.63,1845.2 L1606.39,1845.94 L1606.87,1846.19 L1607.37,1846.22 L1608.33,1846 L1608.76,1846.05 L1609.16,1846.36 L1609.83,1847.33 L1609.96,1847.43 L1610.26,1847.67 L1610.72,1847.75 L1611.56,1847.56 L1612.01,1847.54 L1612.8,1847.74 L1613.05,1847.87 L1613.48,1848.08 L1617.22,1850.76 L1617.92,1851.5 L1618.17,1851.96 L1618.49,1853.06 L1618.71,1853.52 L1619.05,1853.89 L1620.32,1854.79 L1621.77,1856.21 L1622.5,1856.73 L1623.39,1857.09 L1625.68,1857.18 L1626.49,1857.39 L1626.67,1858.09 L1626.88,1858.86 L1626.74,1861.01 L1626.24,1863.18 L1625.55,1864.74 L1625.22,1865.08 L1624.52,1865.57 L1624.19,1865.98 L1624.03,1866.45 L1623.89,1867.64 L1623.73,1868.15 L1622.12,1870.28 L1621.66,1871.24 L1621.66,1871.25 L1621.66,1871.25 L1621.66,1871.25 L1621.1,1871.58 L1619.9,1872.1 L1619.42,1872.51 L1619.19,1873.08 L1619.03,1873.91 L1618.71,1877.04 L1618.55,1877.78 L1618.53,1877.87 L1618.1,1878.61 L1616.81,1879.43 L1615.49,1880 L1611.93,1879.66 L1610.72,1880.03 L1610.16,1880.57 L1609.98,1880.95 L1609.97,1880.98 L1609.69,1881.22 L1608.86,1881.23 L1608.24,1881.35 L1607.89,1881.77 L1607.35,1882.98 L1606.94,1883.47 L1605.88,1884.27 L1605.54,1884.7 L1605.32,1885.19 L1605.22,1885.43 L1605.18,1885.67 L1605.38,1885.88 L1606.87,1888.32 L1607.62,1889.15 L1608.38,1889.57 L1608.46,1889.74 L1608.51,1889.93 L1608.52,1890.12 L1608.5,1890.32 L1607.98,1891 L1607.15,1893.45 L1607.61,1894.24 L1607.63,1895.8 L1607.58,1896.47 L1607.4,1899.03 L1607.43,1900.29 L1607.64,1900.35 L1610.35,1902.16 L1611.91,1902.93 L1612.6,1903.55 L1613.08,1904.36 L1613.09,1904.37 L1613.41,1905.26 L1613.46,1906.57 L1613.19,1908.93 L1612.86,1910 L1612.4,1910.44 L1611.84,1910.66 L1611.2,1911.12 L1610.64,1912.15 L1610.26,1914.54 L1609.73,1915.52 L1608.94,1915.93 L1607.04,1916.28 L1606.23,1916.89 L1605.32,1916.99 L1604.46,1916.81 L1603.64,1916.37 L1602.85,1915.69 L1602.65,1915.27 L1602.46,1914.23 L1602.21,1913.96 L1601.8,1914.11 L1600.79,1915.55 L1594.87,1919.24 L1593.33,1919.79 L1592.6,1919.82 L1591.77,1919.86 L1590.2,1919.56 L1587.56,1918.69 L1586.97,1918.66 L1586.51,1918.97 L1585.15,1920.57 L1584.84,1920.7 L1583,1920.97 L1582.11,1921.33 L1578.03,1924.76 L1577.48,1925.22 L1576.94,1926.13 L1576.31,1926.67 L1576.3,1926.67 L1575.51,1926.76 L1573.83,1926.57 L1571.56,1926.97 L1571.02,1926.78 L1570.3,1925.15 L1568.12,1922.72 L1567.96,1921.22 L1568.79,1920.04 L1571.39,1918.79 L1572.22,1917.74 L1572.28,1916.75 L1571.77,1914.83 L1571.73,1913.86 L1572.03,1913.11 L1573.6,1910.99 L1575.33,1906.87 L1575.58,1905.38 L1575.5,1905.15 L1575.1,1904.66 L1574.99,1904.4 L1575.05,1904.21 L1575.34,1904.01 L1575.41,1903.88 L1575.44,1903.7 L1575.51,1902.41 L1575.44,1902.23 L1575.84,1901.83 L1576.35,1901.5 L1576.79,1901.11 L1576.97,1900.53 L1576.39,1899.22 L1575.38,1898.76 L1575.1,1898.63 L1573.7,1898.54 L1572.72,1898.71 L1570.88,1899.92 L1570.21,1900.03 L1569.37,1899.64 L1567.64,1898.03 L1566.85,1897.52 L1565.76,1897.34 L1561.08,1897.41 L1559.99,1897.61 L1559.44,1897.71 L1558.64,1897.6 L1558.13,1897.35 L1558.13,1897.17 L1558.32,1896.96 L1558.42,1896.63 L1558.44,1895.65 L1558.77,1893.48 L1558.81,1892.42 L1558.63,1891.12 L1558.58,1890.79 L1558.08,1890.39 L1557.16,1890.49 L1555.65,1890.37 L1555.14,1890.15 L1554.15,1889.54 L1553.63,1889.31 L1553.09,1889.27 L1552.03,1889.44 L1551.51,1889.38 L1550.02,1888.07 L1549.48,1886.15 L1549.15,1884.24 L1548.25,1882.97 L1547.47,1882.96 L1546.92,1883.51 L1546.32,1883.94 L1545.47,1883.59 L1544.96,1882.84 L1543.82,1879.7 L1542.91,1878.37 L1542.11,1877.91 L1539.06,1878.19 L1538.07,1877.98 L1537.5,1877.75 L1533.53,1876.14 L1529.92,1874.68 L1529.55,1874.22 L1529.73,1874.14 L1529.75,1873.31 L1529.7,1872.41 L1529.63,1872.09 L1530.08,1871.54 L1530.56,1871.61 L1530.7,1871.61 L1530.15,1870.8 L1529.79,1870.43 L1529.28,1870.02 L1528.32,1869.43 L1527.86,1869.35 L1527.14,1869.24 L1526.03,1869.54 L1525,1870.19 L1521.81,1873.38 L1520.87,1873.91 L1519.4,1874.19 L1519.06,1874.51 L1518.87,1875.35 L1519.08,1876.02 L1519.78,1877.11 L1520.34,1878.4 L1520.67,1878.95 L1520.99,1879.32 L1521.41,1879.38 L1521.89,1879.25 L1522.36,1879.2 L1522.74,1879.54 L1522.75,1881.07 L1522.07,1883.26 L1521.11,1885.29 L1520.28,1886.36 L1519.73,1886.45 L1519.19,1886.37 L1518.72,1886.43 L1518.36,1886.91 L1518.06,1887.92 L1517.86,1888.37 L1517.54,1888.76 L1516.71,1889.19 L1515.9,1889.3 L1515.16,1889.65 L1514.52,1890.77 L1514.36,1892.55 L1515.18,1893.23 L1516.27,1893.72 L1516.62,1894.41 L1516.9,1894.98 L1516.05,1896.5 L1513.97,1897.73 L1510.3,1899.07 L1510.44,1896.62 L1510.17,1895.09 L1509.29,1894.21 L1506.31,1893.18 L1503.7,1891.79 L1502.32,1891.4 L1499.85,1891.48 L1499.31,1891.38 L1498.71,1891.27 L1495.61,1888.91 L1495.51,1888.84 L1493.74,1887.49 L1490.77,1885.93 L1490.26,1885.53 L1489.84,1884.98 L1489.5,1884.38 L1489.1,1883.93 L1488.51,1883.81 L1488.11,1884.1 L1487.18,1885.33 L1486.69,1885.73 L1485.17,1886.06 L1483.65,1886.15 L1482.54,1885.89 L1481.51,1885.25 L1480.54,1884.26 L1478.91,1882.02 L1478.54,1882.03 L1478.22,1882.45 L1477.66,1882.74 L1476.67,1882.75 L1474.58,1882.41 L1473.59,1882.04 L1472.72,1881.42 L1471.09,1879.76 L1470.99,1879.69 L1469.22,1878.55 L1468.68,1877.91 L1467.69,1876.26 L1467.04,1875.51 L1464.04,1873.06 L1461.81,1870.46 L1460.99,1870 L1454.45,1867.98 L1453.15,1867.12 L1452.22,1865.57 L1451.99,1863.8 L1452.73,1862.25 L1453.73,1862.13 L1454.15,1861.01 L1454.41,1859.57 L1454.92,1858.47 L1455.4,1858.26 L1456.57,1858.31 L1457.09,1858.22 L1457.59,1857.85 L1458.54,1856.89 L1459.01,1856.69 L1459.82,1857.04 L1460.71,1857.94 L1461.49,1859.04 L1461.98,1860.04 L1462.4,1860.54 L1462.73,1859.97 L1463.13,1858.54 L1463.67,1858.42 L1465.46,1858.89 L1465.79,1858.62 L1465.85,1858.35 L1465.69,1858.08 L1465.33,1857.82 L1465.03,1857.19 L1464.88,1856.34 L1464.85,1855.44 L1464.96,1854.67 L1465.98,1853.39 L1465.7,1851.76 L1464.21,1848.46 L1463.07,1846.53 L1457.88,1842.15 L1455.02,1838.58 L1453.44,1836.98 L1451.76,1835.99 L1451.2,1835.73 L1451.65,1835.16 L1451.7,1834.71 L1451.39,1833.96 L1451.33,1833.52 L1451.51,1832.9 L1451.86,1832.51 L1452.75,1832.09 L1453.39,1831.97 L1454.01,1832.12 L1455.24,1832.77 L1456.53,1833.71 L1457.15,1834 L1457.79,1833.88 L1458.44,1833.6 L1460.8,1833.13 L1463.38,1833.06 L1464.47,1832.82 L1465.48,1832.3 L1467.16,1830.69 L1467.82,1829.75 L1468.07,1828.68 L1467.6,1827.56 L1466.98,1827.13 L1466.59,1827.32 L1466.37,1827.73 L1466.22,1827.92 L1465.77,1827.64 L1464.93,1826.85 L1463.37,1826.18 L1463.06,1825.83 L1462.11,1824.76 L1461.34,1824.2 L1461.01,1823.84 L1460.78,1822.74 L1460.35,1822.3 L1460.08,1822.34 L1459.64,1822.92 L1459.38,1823.03 L1459.05,1822.89 L1458.51,1822.47 L1456.39,1821.28 L1455.45,1820.51 L1455.07,1819.75 L1454.83,1818.5 L1454.12,1817.24 L1453.21,1816.21 L1452.38,1815.67 L1451.86,1815.64 L1450.46,1816.16 L1449.83,1816.21 L1449.42,1816.13 L1448.44,1815.78 L1446.48,1815.6 L1445.53,1815.3 L1444.61,1814.42 L1443.38,1812.42 L1442.89,1811.22 L1442.67,1810.05 L1442.91,1808.71 L1443.43,1807.85 L1443.85,1806.88 L1443.74,1805.17 L1442.71,1802.51 L1442.71,1801.52 L1444.22,1800.8 L1444.53,1800.46 L1444.77,1800.01 L1444.97,1799.48 L1445,1799.03 L1444.84,1798.04 L1444.88,1797.62 L1445.07,1797.31 L1445.52,1796.96 L1445.7,1796.74 L1447.36,1792.93 L1447.78,1790.99 L1448.05,1790.38 L1448.46,1789.81 L1449.26,1788.95 L1449.61,1788.25 L1449.93,1787.05 L1450.6,1779.37 L1450.75,1779.06 L1450.97,1778.97 L1451.11,1778.78 L1450.99,1778.16 L1450.8,1777.66 L1450.58,1777.33 L1450.29,1777.11 L1450,1776.85 L1450,1776.85 L1449.8,1776.68 L1449.8,1776.74 L1449.69,1776.82 L1449.25,1776.92 L1448.79,1776.82 L1447.81,1776.09 L1447.42,1776.14 L1447.08,1776.37 L1446.68,1776.48 L1444.67,1776.31 L1444.24,1776.45 L1443.5,1776.88 L1443.06,1776.93 L1442.81,1776.66 L1442.58,1776.16 L1442.26,1775.82 L1441.72,1776.02 L1438.1,1779.39 L1437.42,1779.74 L1436.52,1779.98 L1435.16,1780.59 L1434,1778.89 L1433.73,1777.66 L1434.85,1776.85 L1435.47,1776.97 L1436.47,1777.83 L1437.12,1777.83 L1437.61,1777.45 L1437.96,1776.87 L1438.51,1775.61 L1438.9,1775.3 L1439.38,1775.02 L1439.64,1774.54 L1439.35,1773.61 L1438.9,1773.22 L1438.04,1773.67 L1437.51,1773.29 L1437.35,1772.67 L1437.4,1771.9 L1437.53,1771.12 L1437.55,1770.53 L1437.43,1769.88 L1436.85,1768.1 L1436.51,1766.63 L1436.31,1766.04 L1435.88,1765.31 L1435.51,1764.95 L1435.02,1764.57 L1434.62,1764.13 L1434.49,1763.58 L1435.09,1762.91 L1437.58,1762.85 L1438.51,1762.54 L1438.57,1761.52 L1437.86,1760.41 L1437.45,1759.54 L1438.39,1759.19 L1438.94,1759.41 L1439.91,1760.49 L1440.44,1760.92 L1441.32,1761.13 L1442.27,1761.07 L1444.07,1760.6 L1444.27,1760.42 L1444.37,1760.12 L1444.5,1759.83 L1444.8,1759.65 L1445.02,1759.74 L1445.44,1760.23 L1445.67,1760.36 L1449.44,1759.71 L1449.57,1759.55 L1450.01,1758.98 L1449.94,1757.51 L1449.33,1756.32 L1448.44,1756.04 L1447.64,1754.96 L1438.56,1753.78 L1436.83,1753.17 L1435.54,1751.96 L1435.36,1751.4 L1435.24,1750.7 L1435.02,1750.13 L1434.56,1749.93 L1432.75,1750.16 L1431.86,1750.06 L1431.04,1749.6 L1430.29,1748.87 L1429.62,1747.96 L1431.8,1746.61 L1432.11,1745.94 L1431.77,1745.24 L1430.82,1744.45 L1430.37,1744.07 L1429.95,1742.96 L1430.35,1741.35 L1430.72,1741.36 L1433.35,1741.42 L1434.47,1740.03 L1433.88,1739.19 L1433.28,1738.94 L1432.64,1738.88 L1431.86,1738.63 L1431.38,1738.3 L1431.15,1738.05 L1430.98,1737.75 L1430.65,1737.29 L1430.22,1737.04 L1429.85,1737.2 L1429.5,1737.51 L1428.79,1738.01 L1428.41,1738.59 L1427.9,1739.1 L1427.1,1739.21 L1426.62,1738.85 L1425.59,1737.54 L1425.19,1737.29 L1425.28,1736.74 L1425.2,1735.57 L1425.24,1734.38 L1425.71,1733.79 L1426.15,1733.38 L1427.15,1731.57 L1427.33,1731.01 L1426.98,1730.15 L1426.28,1729.67 L1425.49,1729.33 L1424.83,1728.93 L1424.38,1728.18 L1423.98,1726.31 L1423.67,1725.4 L1422.23,1723.99 L1421.86,1723.5 L1421.12,1722.9 L1420.91,1722.53 L1420.96,1722.26 L1421.09,1721.98 L1421.14,1721.6 L1421.07,1719.89 L1422.43,1719.6 L1422.97,1719.41 L1423.23,1718.97 L1423.05,1718.5 L1422.53,1718.05 L1421.56,1717.45 L1421.9,1716.94 L1422.2,1716.58 L1422.89,1716 L1422.53,1715.07 L1425.22,1714.95 L1426.09,1715.23 L1426.32,1715.08 L1426.35,1714.87 L1426.22,1714.62 L1425.83,1714.13 L1425.75,1713.91 L1425.77,1713.67 L1425.88,1713.43 L1426.82,1712.58 L1427.94,1712.6 L1428.68,1712.29 L1428.51,1710.45 L1429.05,1709.19 L1430,1708.64 L1431.02,1708.89 L1431.8,1710.03 L1431.8,1710.04 L1432.83,1710.48 L1436.67,1710.41 L1437.84,1710.03 L1438.26,1709.33 L1438.37,1708.55 L1438.6,1707.74 L1439.05,1707.31 L1439.39,1706.98 L1440.15,1706.71 L1441.75,1706.5 L1442.44,1706.16 L1442.82,1705.61 L1443.07,1704.95 L1443.36,1704.37 L1443.86,1704.04 L1444.33,1703.91 L1444.68,1703.71 L1445.34,1703.02 L1445.59,1702.55 L1445.76,1702.06 L1446.01,1701.6 L1447.07,1700.84 L1446.89,1700.34 L1446.5,1699.82 L1446.36,1699.32 L1446.9,1698.53 L1447.82,1697.73 L1448.82,1697.11 L1449.6,1696.87 L1450.52,1696.78 L1452.97,1695.96 L1453.9,1695.89 L1454.83,1696.06"
         id="path1085" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['RS'],polycolors['RS'],labels['RS'], x1, polybary['RS'], x2, polybary['RS'] )
    if 'RS' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1144.87,1863.72 L1143.49,1863.07 L1142.82,1861.76 L1143.23,1860.26 L1143.91,1859.53 L1144.36,1859.05 L1145.78,1858.26 L1147.08,1858 L1147.41,1858.58 L1147.59,1859.62 L1147.53,1860.69 L1147.22,1861.32 L1147.18,1861.41 L1147.07,1861.54 L1147,1861.71 L1146.99,1861.9 L1147.03,1862.09 L1146.26,1863.47 L1144.87,1863.72"
         id="path1087" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['SM'],polycolors['SM'],labels['SM'], x1, polybary['SM'], x2, polybary['SM'] )
    if 'SM' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1080.12,1173.62 L1081.85,1174.98 L1082.49,1175.26 L1083.15,1175.79 L1084.28,1178.15 L1085.15,1178.67 L1086.02,1178.85 L1092.53,1182.36 L1092.16,1181.28 L1091.72,1180.61 L1091.68,1180.07 L1092.46,1179.37 L1093.45,1179.35 L1094.31,1178.92 L1094.67,1178.86 L1094.33,1178.18 L1094.22,1177.4 L1094.26,1176.61 L1094.35,1175.91 L1097.17,1178.35 L1097.41,1179.25 L1098.3,1180.34 L1098.85,1180.76 L1099.46,1180.92 L1099.1,1181.27 L1098.95,1181.36 L1099.51,1182.46 L1101.36,1183.92 L1101.77,1184.51 L1101.89,1185.53 L1102.06,1185.99 L1102.07,1186.36 L1101.71,1187.1 L1101.31,1187.57 L1100.94,1187.73 L1100.63,1187.95 L1100.37,1188.63 L1101.81,1189.23 L1101.71,1190.58 L1100.59,1191.93 L1098.98,1192.51 L1093.3,1191.4 L1091.48,1191.86 L1089.76,1192.72 L1088.42,1194.02 L1087.59,1193.9 L1084.17,1191.95 L1081.92,1190.05 L1077.89,1187.58 L1076.09,1187.21 L1074.14,1186.33 L1072.17,1185.89 L1071.39,1185.21 L1071.01,1184.27 L1071.23,1183.21 L1072.13,1182.66 L1073.24,1182.82 L1074.26,1182.73 L1074.91,1181.41 L1074.32,1181.2 L1073.83,1180.82 L1071.87,1178.45 L1071.63,1178.04 L1071.62,1176.96 L1072.09,1175.96 L1073.57,1174.13 L1074.08,1174.37 L1079.24,1173.22 L1080.12,1173.62"
         id="path1089" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1027.43,1166.11 L1029.57,1168.66 L1030.71,1169.28 L1032.3,1169.53 L1033.45,1170.23 L1034.37,1171.31 L1035.26,1172.71 L1035.78,1173.69 L1037.2,1178.01 L1037.18,1178.62 L1036.67,1178.75 L1035.65,1178.72 L1035.11,1179.01 L1034.4,1179.56 L1033.7,1179.96 L1033.18,1179.77 L1032.58,1179.25 L1030.92,1178.72 L1030.31,1178.29 L1030.31,1177.87 L1031.27,1177.39 L1032.41,1177.88 L1033.55,1178.69 L1034.55,1179.15 L1034.55,1178.72 L1032.82,1177.2 L1031.18,1176.62 L1030.81,1176.58 L1030.45,1176.72 L1029.93,1177.24 L1029.57,1177.44 L1028.16,1177.36 L1026.95,1176.37 L1026.24,1174.72 L1026.37,1172.69 L1026.14,1172.69 L1026.14,1172.3 L1026.94,1172.86 L1027.79,1173.84 L1028.63,1174.58 L1029.35,1174.41 L1029.2,1174.19 L1028.97,1173.76 L1028.83,1173.35 L1028.97,1173.16 L1029.16,1173.03 L1028.92,1172.72 L1027.42,1171.65 L1027.18,1171.15 L1027.62,1170.56 L1027.62,1170.13 L1024.79,1170.39 L1024.3,1170.33 L1023.95,1170.05 L1023.51,1169.99 L1023.13,1169.83 L1022.94,1169.26 L1023.07,1168.98 L1023.75,1168.47 L1023.94,1167.98 L1023.28,1168.01 L1022.68,1168.23 L1022.1,1168.33 L1021.49,1167.97 L1021.49,1167.51 L1024.67,1166.01 L1025.94,1165.81 L1027.43,1166.11"
         id="path1091" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1209.7,1150.65 L1211.12,1151.64 L1212.09,1151.36 L1212.79,1151.51 L1213.46,1152.72 L1213.88,1153.17 L1214.34,1153.43 L1217.03,1155.07 L1218.76,1155.52 L1219.52,1156.24 L1219.37,1156.56 L1219.31,1156.78 L1219.25,1156.95 L1219.09,1157.13 L1219.42,1158.02 L1219.56,1158.6 L1219.49,1159.13 L1219.17,1159.93 L1218.5,1161.24 L1218.32,1161.88 L1218.49,1162.39 L1218.52,1162.77 L1218.68,1163.51 L1218.15,1164.21 L1217.81,1164.77 L1217.34,1165.28 L1215.55,1164.92 L1214.07,1164.92 L1212.02,1164.26 L1209.45,1162.9 L1206.61,1162.28 L1203.53,1160.09 L1202.93,1159.37 L1203.31,1158.01 L1203.29,1155.37 L1203.19,1151.56 L1203.27,1150.91 L1203.52,1150.21 L1204.13,1148.92 L1204.36,1147.97 L1204.53,1146.88 L1204.8,1146.2 L1205.31,1146.51 L1206.9,1148.5 L1207.43,1149.01 L1208.29,1149.8 L1209.7,1150.65"
         id="path1093" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1046.85,1132.97 L1049.31,1134.65 L1049.87,1135.34 L1050.31,1136.1 L1050.62,1136.51 L1050.96,1136.69 L1051.61,1136.76 L1052.2,1136.98 L1053.28,1137.73 L1052.7,1137.5 L1052.15,1137.5 L1051.74,1137.83 L1051.58,1138.57 L1051.71,1138.84 L1052.24,1139.26 L1052.32,1139.63 L1052.14,1140.15 L1051.84,1140.3 L1051.49,1140.32 L1051.11,1140.49 L1049.79,1142.42 L1050.38,1143.46 L1051.64,1143.39 L1052.33,1142 L1052.73,1142.1 L1055.02,1140.89 L1056.36,1140.46 L1056.52,1140.26 L1056.44,1139.83 L1056.26,1139.4 L1056.1,1139.21 L1055.78,1139.13 L1055.44,1138.92 L1055.15,1138.63 L1055,1138.33 L1055.1,1137.78 L1055.54,1137.98 L1056.02,1138.46 L1056.22,1138.75 L1056.66,1138.05 L1056.54,1137.06 L1056.18,1135.97 L1055.94,1134.9 L1056.12,1135.19 L1056.67,1135.75 L1056.67,1134.7 L1056.43,1133.14 L1056.43,1132.72 L1057.1,1132.68 L1058.15,1133.84 L1059.86,1136.62 L1059.93,1137.02 L1059.97,1137.57 L1060.05,1138.07 L1060.24,1138.28 L1060.33,1138.42 L1061.09,1139.17 L1061.33,1140.03 L1061.26,1140.53 L1060.93,1140.89 L1059.9,1141.57 L1058.25,1142.3 L1055.79,1142.09 L1055.19,1142.23 L1054.77,1142.6 L1054.58,1143.4 L1054.91,1143.8 L1055.49,1143.82 L1056.01,1143.49 L1055.84,1143.16 L1055.81,1142.97 L1056.06,1142.32 L1056.77,1142.49 L1059.17,1143.03 L1059.71,1143.5 L1063.28,1148.58 L1063.51,1149.13 L1064.45,1151.82 L1064.67,1152.84 L1064.41,1152.84 L1063.98,1152.18 L1063.32,1151.76 L1062.73,1151.94 L1062.48,1153.06 L1062.69,1153.82 L1063.67,1155.34 L1063.96,1156.27 L1064.04,1157.61 L1063.97,1158.85 L1063.74,1159.91 L1063.21,1161.2 L1063.11,1161.67 L1063.08,1162.92 L1062.97,1163.25 L1062.1,1164.44 L1061.77,1165.21 L1061.54,1165.92 L1061.22,1166.54 L1060.65,1167.02 L1060.21,1167.13 L1058.8,1167.04 L1057.19,1167.06 L1056.59,1167.17 L1055.92,1167.49 L1055.31,1167.97 L1054.86,1168.59 L1054.22,1169.24 L1053.45,1169.15 L1052.03,1168.37 L1049.62,1168.19 L1048.83,1167.96 L1048.48,1167.73 L1048.2,1167.45 L1047.88,1167.22 L1047.47,1167.12 L1046.57,1167.21 L1046.09,1167.17 L1043.27,1165.39 L1042.22,1165.3 L1041.95,1165.78 L1041.93,1167.14 L1041.39,1166.73 L1040.54,1165.74 L1039.94,1165.4 L1039.21,1165.38 L1037.85,1165.73 L1037.22,1165.4 L1039.97,1163.3 L1040.45,1162.4 L1039.83,1161.51 L1039.37,1160.49 L1038.9,1159.67 L1038.22,1159.38 L1037.68,1159.54 L1037.27,1159.77 L1036.85,1159.92 L1036.26,1159.85 L1035.45,1159.02 L1034.98,1158.83 L1034.55,1159.38 L1034.65,1160.36 L1035.9,1162.26 L1035.77,1163.31 L1035.59,1163.39 L1034.33,1163.18 L1034.24,1163 L1034.29,1162.75 L1034.28,1162.41 L1034.39,1162.45 L1034.48,1162.2 L1034.45,1161.67 L1033.93,1160.28 L1033.85,1159.61 L1033.93,1158.95 L1034.16,1158.35 L1034.12,1157.85 L1033.57,1157.52 L1032.13,1157.16 L1031.68,1156.87 L1031.28,1156.52 L1031.11,1156.22 L1031.03,1155.87 L1030.7,1155.31 L1030.62,1154.93 L1030.7,1154.55 L1031.03,1153.98 L1031.11,1153.63 L1031.04,1152.95 L1030.71,1151.85 L1030.63,1151.49 L1030.74,1149.34 L1030.37,1149.06 L1029.82,1149.01 L1028.78,1149.12 L1028.31,1148.94 L1027.8,1148.49 L1026.94,1147.36 L1027.54,1147.14 L1028.21,1147.06 L1028.79,1146.83 L1029.15,1146.13 L1027.21,1145.27 L1027.5,1145.33 L1028.42,1145.27 L1028.06,1144.54 L1027.41,1144.14 L1025.99,1143.94 L1025.5,1143.74 L1024.3,1142.23 L1024.3,1141.84 L1025.25,1142.28 L1026.25,1143.16 L1027.24,1143.8 L1028.16,1143.56 L1024.66,1140.43 L1023.34,1140.13 L1023.61,1139.48 L1023.96,1139.22 L1024.36,1139.09 L1024.82,1138.81 L1026.15,1137.11 L1028.07,1136.73 L1028.72,1136.85 L1028.84,1137.18 L1028.86,1137.7 L1029.17,1138.4 L1030.88,1139.34 L1032.73,1138.68 L1036.25,1136.27 L1040.84,1134.73 L1042.39,1133.58 L1043.27,1133.27 L1044.22,1133.2 L1044.5,1133.26 L1044.53,1133.45 L1044.75,1134.34 L1044.77,1134.54 L1045.27,1134.35 L1045.39,1133.88 L1045.27,1132.6 L1045.76,1132.46 L1046.85,1132.97"
         id="path1095" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1116.33,1099.34 L1119.41,1100.98 L1121.9,1100.9 L1122.59,1101.24 L1123.85,1102.31 L1124.91,1102.72 L1126.78,1104.18 L1126.35,1105.19 L1124.98,1107.2 L1124.51,1108.28 L1123.64,1110.8 L1123.22,1111.54 L1123.23,1112.01 L1124.25,1113.39 L1124.79,1114.33 L1125.03,1115.13 L1125.24,1116.3 L1125.67,1117.66 L1126.2,1118.88 L1126.67,1119.62 L1126.54,1120.23 L1126.5,1121.07 L1126.51,1122.4 L1126.34,1123.49 L1126.32,1123.91 L1126.38,1124.21 L1126.52,1124.52 L1126.87,1125.17 L1126.17,1126.17 L1124.85,1128.98 L1123.54,1129.96 L1123.34,1130.73 L1123.03,1131.49 L1122.25,1131.79 L1121.66,1131.6 L1121.23,1131.26 L1120.76,1131 L1120.04,1131 L1117.53,1132.58 L1116.87,1133.31 L1114.69,1135.74 L1114.69,1135.75 L1113.02,1139.48 L1114.08,1142.74 L1115.86,1144.22 L1116.47,1144.41 L1118.67,1144.5 L1119.27,1144.75 L1120.86,1146.01 L1121.7,1146.95 L1122.09,1147.86 L1122.18,1148.93 L1122.38,1149.84 L1122.94,1151.5 L1122.44,1152.18 L1122.24,1152.39 L1120.42,1153.78 L1112.98,1155.77 L1112.44,1156.41 L1111.54,1157.02 L1110.84,1158.1 L1110.45,1159.38 L1110.7,1160.44 L1110.44,1160.99 L1110.14,1161.23 L1109.82,1161.19 L1109.48,1160.91 L1109.71,1159.55 L1109.27,1158.72 L1108.54,1158.5 L1107.85,1159.03 L1107.54,1159.35 L1107.26,1159.46 L1107.07,1159.69 L1107.01,1160.33 L1107.22,1160.59 L1108.27,1161.37 L1111.29,1161.54 L1112.67,1161.96 L1113.01,1163.41 L1112.83,1164.03 L1112.61,1164.29 L1112.35,1164.43 L1112.05,1164.68 L1111.32,1165.11 L1110.97,1165.45 L1111.24,1165.6 L1112.68,1168.54 L1112.68,1169.73 L1112.22,1170.39 L1111.49,1170.67 L1110.64,1170.74 L1110.09,1170.98 L1108.91,1171.95 L1108.09,1172.14 L1106.82,1171.65 L1104.81,1169.85 L1103.6,1169.66 L1103.58,1169.95 L1103.64,1170.02 L1103.74,1170.02 L1103.85,1170.08 L1103.65,1170.2 L1103.55,1170.35 L1103.42,1170.48 L1103.15,1170.56 L1103.34,1171.75 L1103.1,1172.47 L1102.78,1173.09 L1102.73,1173.99 L1103.01,1174.68 L1103.5,1175.09 L1104.04,1175.14 L1104.49,1174.77 L1104.39,1174.54 L1104.2,1173.95 L1104.51,1173.9 L1105.42,1173.46 L1105.72,1173.8 L1106.27,1174.93 L1106.34,1175.18 L1106.69,1175.42 L1107.89,1176.53 L1108.49,1176.83 L1109.16,1176.73 L1109.85,1176.49 L1110.51,1176.45 L1111.08,1176.97 L1112.14,1178.48 L1113.27,1179.72 L1113.28,1180.11 L1110.16,1182.83 L1109.65,1183.44 L1108.31,1186.19 L1108.14,1186.71 L1107.78,1186.98 L1106.84,1188.19 L1106.47,1188.47 L1105.96,1189.35 L1106,1191.34 L1106.51,1194.88 L1106.48,1197.12 L1105.76,1197.22 L1104.97,1196.27 L1104.78,1195.39 L1104.24,1194.69 L1102.99,1192.38 L1102.7,1191.6 L1102.72,1190.3 L1103.13,1189.28 L1104.08,1187.68 L1103.34,1187.54 L1102.83,1187.2 L1102.68,1186.57 L1103.03,1185.53 L1102.62,1185.29 L1102.38,1184.98 L1102.01,1184.27 L1100.29,1182.56 L1100.15,1182.28 L1099.97,1181.76 L1099.88,1181.26 L1099.92,1180.93 L1099.9,1180.59 L1099.66,1180.1 L1099.01,1179.39 L1098.44,1178.96 L1098.03,1178.35 L1097.86,1177.11 L1097.82,1175.96 L1097.68,1175.08 L1097.3,1174.47 L1096.55,1174.15 L1096.54,1173.68 L1097.89,1172.55 L1098.23,1172.4 L1098.75,1172.58 L1099.47,1173.42 L1099.87,1173.6 L1100.26,1173.35 L1101.06,1172.45 L1101.21,1172.54 L1101.45,1172.88 L1101.96,1172.58 L1102.67,1171.82 L1102.66,1171.43 L1102.27,1170.95 L1101.54,1169.06 L1101.1,1168.44 L1098.66,1167.35 L1097.97,1167.23 L1097.78,1167.11 L1097.28,1166.53 L1096.99,1166.41 L1094.71,1166.33 L1094.02,1165.97 L1093.43,1165.44 L1092.89,1164.75 L1094.83,1165.12 L1098.67,1166.79 L1100.55,1167.17 L1099.72,1165.74 L1097.46,1164.13 L1096.49,1162.57 L1099.57,1161.93 L1099.77,1161.67 L1099.66,1161.17 L1098.87,1160.54 L1096.84,1160.75 L1096.21,1159.94 L1096.72,1159.36 L1097.01,1158.62 L1096.93,1157.9 L1096.37,1157.4 L1095.56,1157.44 L1094.97,1158 L1094.47,1158.7 L1093.96,1159.17 L1093.49,1159.14 L1092.16,1158.57 L1089.45,1157.95 L1088.79,1157.57 L1088.31,1157.84 L1085.81,1156.78 L1085.35,1156.91 L1084.38,1157.48 L1083.88,1157.67 L1081.41,1157.72 L1081.42,1158.15 L1082.16,1158.56 L1080.92,1158.54 L1080.02,1158.23 L1079.45,1157.35 L1079.17,1155.62 L1079.7,1156 L1080.23,1156.01 L1080.66,1155.6 L1080.87,1154.74 L1080.5,1155.02 L1080.13,1155.17 L1079.29,1155.23 L1079.17,1154.89 L1079.24,1153.36 L1079,1153.02 L1078.36,1152.64 L1076.36,1150.87 L1076.02,1150.29 L1076.34,1150.13 L1076.86,1150.06 L1077.28,1149.84 L1077.33,1149.21 L1077.05,1148.9 L1076.53,1148.76 L1075.98,1148.74 L1075.62,1148.81 L1075.38,1149.15 L1075.22,1149.68 L1074.97,1150.09 L1074.42,1150.1 L1074.19,1149.83 L1073.66,1148.44 L1073.66,1147.98 L1074.62,1148.06 L1075.13,1148.01 L1075.48,1147.74 L1075.79,1147.32 L1076.19,1146.98 L1076.63,1146.74 L1077.04,1146.65 L1077.79,1146.3 L1078.13,1145.54 L1078.08,1144.79 L1077.64,1144.46 L1077.76,1143.82 L1077.44,1142.42 L1076.69,1140.2 L1076.23,1139.43 L1075.77,1138.91 L1075.21,1138.62 L1074.48,1138.53 L1073.85,1138.99 L1073.45,1139.12 L1073.27,1138.78 L1073.33,1137.93 L1073.52,1137.48 L1073.86,1137.29 L1074.34,1137.25 L1075.27,1136.88 L1075.63,1135.97 L1075.51,1134.8 L1075.01,1133.61 L1073.41,1131.65 L1071.67,1130.64 L1067.78,1129.65 L1067.78,1129.18 L1073.37,1129.57 L1072.77,1128.53 L1070.54,1127.6 L1069.69,1127.06 L1069.31,1126.46 L1069.09,1125.99 L1068.8,1125.62 L1068.21,1125.33 L1066.36,1125.22 L1065.8,1124.9 L1065.79,1124.51 L1071.37,1125.29 L1072.52,1125.07 L1074.72,1124.16 L1075.94,1123.94 L1075.94,1124.36 L1075.55,1124.31 L1075.18,1124.35 L1074.82,1124.49 L1074.5,1124.78 L1075.08,1125.86 L1076.01,1126.7 L1077.04,1127 L1077.93,1126.47 L1076.44,1125.21 L1076.92,1124.74 L1077.25,1125.07 L1077.85,1125.21 L1079.12,1125.17 L1079.69,1124.93 L1080.87,1124.04 L1082.08,1123.75 L1082.62,1123.43 L1083.03,1122.84 L1083.18,1121.91 L1083.21,1121.42 L1083.36,1120.55 L1083.39,1120.15 L1083.25,1119.66 L1082.62,1118.77 L1082.39,1118.27 L1083.3,1118.85 L1084.17,1118.98 L1084.98,1118.62 L1085.73,1117.74 L1086.94,1117.98 L1087.87,1116.26 L1088.01,1113.78 L1086.8,1111.73 L1085.77,1111.33 L1083.51,1111.21 L1080.69,1110.2 L1079.88,1109.51 L1079.54,1108.42 L1084.07,1110.5 L1084.99,1110.26 L1090.43,1111 L1091.09,1111.6 L1092.54,1111.48 L1095,1110.66 L1096.26,1109.84 L1096.98,1109.68 L1097.4,1110.21 L1097.26,1110.84 L1096.3,1112.06 L1096.02,1112.81 L1096.45,1113.46 L1096.23,1113.88 L1095.66,1113.98 L1095.08,1113.69 L1095.15,1113.46 L1095.27,1112.83 L1094.16,1112.92 L1093.71,1113.18 L1093.38,1113.73 L1095.2,1116.76 L1095.91,1118.46 L1095.44,1119.7 L1094.91,1119.81 L1093.8,1119.53 L1093.28,1119.75 L1092.66,1120.7 L1092.25,1121.05 L1091.6,1121.08 L1091.61,1121.54 L1092.2,1121.72 L1093.93,1121.49 L1094.4,1121.26 L1094.95,1120.81 L1095.56,1120.47 L1096.21,1120.54 L1096.46,1121.03 L1096.85,1122.05 L1097.15,1123.05 L1097.13,1123.51 L1094.3,1124.09 L1093.68,1124.37 L1093.43,1124.76 L1093.89,1125.26 L1094.4,1125.32 L1095.99,1125.16 L1096.32,1125 L1096.58,1124.68 L1097.19,1125.27 L1098.08,1126.52 L1098.35,1127.33 L1098.33,1127.64 L1098.14,1128.03 L1097.88,1129.09 L1098.49,1128.48 L1098.93,1127.87 L1100.33,1124.59 L1100.4,1124.51 L1100.43,1124.47 L1100.35,1124.03 L1100,1123.24 L1099.91,1123.01 L1100.01,1122.29 L1100.26,1121.38 L1100.8,1119.96 L1101.31,1119.39 L1102.55,1119 L1103.17,1118.61 L1103.69,1116.83 L1102.91,1114.81 L1102.03,1113 L1102.26,1111.8 L1102.89,1111.83 L1104.16,1113.12 L1104.83,1113.44 L1105.17,1113.7 L1105.18,1114.33 L1105.06,1115.04 L1105.01,1115.53 L1105.18,1116.32 L1105.33,1116.6 L1105.5,1116.78 L1105.78,1117.26 L1106.49,1119.06 L1107.19,1121.63 L1107.37,1124.04 L1106.5,1125.4 L1105.21,1124.97 L1104.6,1125.07 L1104.37,1126.11 L1104.13,1126.84 L1102.92,1128.08 L1102.49,1128.97 L1103.21,1129.11 L1104.22,1128.61 L1105.01,1127.72 L1105.09,1126.72 L1105.64,1126.47 L1106.23,1126.64 L1106.75,1127.16 L1107.07,1127.94 L1105.96,1127.6 L1105.82,1127.75 L1106.06,1128.31 L1106.61,1128.73 L1107.82,1129.2 L1108.09,1127.88 L1108.84,1125.87 L1108.91,1124.47 L1107.24,1117.68 L1107.17,1115.47 L1107.02,1114.58 L1106.09,1111.26 L1105.52,1110.41 L1104.87,1110.09 L1104.11,1110.16 L1101.45,1111.33 L1100.44,1111.54 L1099.84,1111.01 L1100.39,1109.46 L1102.17,1107.85 L1105.45,1105.69 L1110.67,1100.97 L1113.41,1099.32 L1116.33,1099.34"
         id="path1097" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1050.34,1006.54 L1049.35,1010.3 L1050.31,1013.11 L1052.04,1015.69 L1053.33,1018.73 L1052.37,1021.49 L1052.41,1024.75 L1053.18,1031.16 L1052.6,1032.1 L1050.88,1033.4 L1050.06,1034.42 L1048.88,1036.96 L1048.34,1038.66 L1047.96,1041.36 L1047.13,1044.37 L1046.73,1045.37 L1046.44,1045.8 L1046.16,1046.05 L1045.79,1046.17 L1044.17,1046.15 L1043.68,1046.28 L1043.06,1046.47 L1040.19,1044.54 L1038.06,1042.25 L1036.65,1041.67 L1035.51,1040.5 L1034.67,1040.23 L1033.98,1040.62 L1033.01,1042.21 L1032.36,1042.37 L1032.99,1042.45 L1033.71,1041.52 L1034.31,1041.09 L1034.92,1040.79 L1035.62,1041.09 L1039,1044.65 L1040.12,1045.2 L1041.43,1046.4 L1042.19,1046.67 L1044.95,1046.73 L1045.59,1047.08 L1044.63,1048.29 L1044.43,1048.8 L1044.27,1049.63 L1044.09,1051.33 L1043.97,1051.79 L1044.22,1052.81 L1044.61,1056 L1044.77,1058.52 L1045.02,1059.85 L1045.88,1062.49 L1046.17,1062.91 L1046.51,1063.28 L1046.59,1063.69 L1046.12,1064.24 L1045.84,1064.26 L1045.03,1063.89 L1044.59,1063.78 L1043.04,1063.79 L1042.5,1064.03 L1041.88,1064.65 L1041.3,1064.03 L1040.65,1063.54 L1039.9,1063.4 L1039.03,1063.8 L1038.26,1064.42 L1037.94,1064.55 L1037.39,1064.65 L1036.86,1064.91 L1035.8,1065.79 L1034.54,1066.24 L1032.97,1067.42 L1028.15,1068.11 L1028.15,1068.53 L1029.72,1068.82 L1034.9,1067.22 L1036.06,1066.5 L1036.56,1066.37 L1036.68,1066.23 L1036.7,1065.94 L1036.76,1065.64 L1037.02,1065.51 L1037.3,1065.58 L1037.76,1065.9 L1037.98,1065.97 L1038.35,1065.88 L1038.77,1065.63 L1039.12,1065.28 L1039.27,1064.87 L1039.5,1064.02 L1040.04,1063.89 L1041.17,1064.26 L1041.09,1064.49 L1040.91,1065.08 L1041.52,1065.19 L1043.2,1065.51 L1044.63,1065.42 L1045.55,1064.85 L1046.38,1064.53 L1047.02,1065.91 L1047.2,1067.98 L1046.61,1069.76 L1044.29,1071.56 L1042.58,1073.51 L1042.27,1073.66 L1042.03,1075.69 L1041.92,1076.26 L1042.18,1076.97 L1042.02,1077.87 L1041.63,1078.7 L1041.21,1079.21 L1042.28,1079.15 L1042.68,1078.82 L1042.63,1076.45 L1042.73,1074.73 L1043.08,1073.9 L1044.52,1072.79 L1045.66,1071.25 L1046.27,1070.83 L1046.99,1071.29 L1047.11,1071.68 L1047.3,1073.02 L1047.34,1073.42 L1047.49,1073.63 L1048.52,1074.48 L1051.13,1075.95 L1053.96,1076.27 L1062.18,1074.89 L1063.33,1075.01 L1064.47,1075.63 L1064.99,1076.21 L1066.13,1078.17 L1068.31,1080.32 L1068.32,1080.71 L1067.53,1081.51 L1067.11,1082.96 L1066.96,1084.62 L1066.95,1086.1 L1066.59,1087.5 L1063.44,1092.11 L1061.82,1093.25 L1061.22,1094.11 L1061.04,1094.94 L1061.15,1096.89 L1060.68,1098.8 L1059.58,1099.04 L1058.35,1098.15 L1057.53,1096.71 L1058.42,1096.55 L1058.79,1096.21 L1058.96,1095.41 L1058.86,1094.58 L1058.5,1094.33 L1058.05,1094.25 L1057.63,1093.93 L1056.79,1093.64 L1055.93,1094.68 L1055.16,1096.05 L1054.05,1097.37 L1054.3,1098.83 L1054.78,1100.42 L1054.94,1101.47 L1054.6,1101.91 L1054.04,1102.19 L1053.45,1102.26 L1053.01,1102.11 L1052.65,1101.59 L1052.3,1100.39 L1052.04,1099.78 L1052.31,1099.48 L1052.9,1099.05 L1053.23,1098.92 L1052.97,1098.22 L1052.66,1097.89 L1051.21,1097.47 L1050.9,1097.53 L1050.12,1098.05 L1049.14,1098.24 L1048.54,1097.76 L1048,1096.88 L1047.24,1095.93 L1048.02,1095.84 L1049.18,1094.89 L1049.84,1094.63 L1050.33,1095 L1050.94,1095.64 L1051.54,1095.88 L1052,1095.01 L1051.41,1094.01 L1051.44,1093.23 L1052.45,1091.16 L1051.98,1091.17 L1051.47,1090.97 L1051.01,1090.6 L1050.66,1090.12 L1050.27,1089.86 L1049.38,1090.29 L1048.89,1089.9 L1048.77,1090.74 L1048.58,1091.26 L1048.28,1091.53 L1047.79,1091.61 L1047.54,1091.95 L1046.61,1094.01 L1045.76,1095.11 L1043.88,1096.68 L1043.04,1097.86 L1042.56,1099.33 L1042.79,1100.25 L1043.37,1101.1 L1043.92,1102.39 L1044.16,1103.97 L1044.02,1105.18 L1043.59,1106.27 L1042.96,1107.52 L1043.8,1107.1 L1044.42,1107.03 L1044.59,1107.47 L1043.72,1109.44 L1043.64,1110.24 L1043.7,1111.83 L1043.67,1112.92 L1043.62,1113.4 L1043.49,1113.93 L1042.64,1115.04 L1042.51,1115.48 L1042.34,1115.48 L1041.96,1115.73 L1041.55,1116.12 L1041.31,1116.49 L1041.49,1116.65 L1041.74,1117.27 L1041.88,1117.93 L1041.57,1118.38 L1041.46,1118.69 L1041.27,1119 L1040.94,1119.14 L1040.51,1119.02 L1040.18,1118.72 L1039.63,1117.85 L1039.47,1117.45 L1039.37,1116.99 L1039.21,1116.52 L1038.88,1116.11 L1038.44,1115.95 L1035.02,1116.12 L1034.56,1116.27 L1034.28,1116.64 L1034.06,1117.07 L1033.81,1117.39 L1032.9,1117.76 L1031.04,1117.65 L1030.18,1117.86 L1031.06,1118.84 L1036.11,1119.96 L1036.47,1120.24 L1036.34,1120.9 L1035.76,1122.17 L1035.46,1123.27 L1035.55,1123.75 L1036.04,1123.86 L1036.97,1123.84 L1036.97,1124.27 L1035.52,1125.1 L1035.09,1125.8 L1035.5,1126.83 L1035.36,1126.94 L1035.04,1127.3 L1034.2,1126.78 L1030.32,1127.92 L1028.54,1128.87 L1027.5,1128.96 L1025.33,1128.09 L1024.09,1126.98 L1023.76,1126.81 L1020.1,1126.77 L1019.28,1126.36 L1019.16,1126.6 L1019.1,1126.77 L1019.06,1126.96 L1019.03,1127.25 L1019.76,1127.61 L1022.31,1127.66 L1022.77,1127.94 L1024.59,1129.84 L1024.97,1130.59 L1025.22,1131.26 L1025.58,1131.75 L1026.28,1131.95 L1029.67,1131.95 L1029.09,1132.67 L1027.24,1134.12 L1026.25,1135.39 L1025.77,1135.72 L1024.94,1135.82 L1024.64,1136.09 L1024.58,1136.68 L1024.56,1137.26 L1024.44,1137.53 L1022.2,1138.08 L1021.39,1138.66 L1020.89,1138.8 L1020.47,1138.69 L1019.67,1138.34 L1019.2,1138.36 L1019.29,1138.62 L1019.44,1138.79 L1018.92,1139.52 L1017.6,1139.94 L1017,1140.48 L1017.62,1140.88 L1018.29,1140.84 L1020.04,1140.4 L1020.37,1140.2 L1020.71,1140.11 L1022.76,1141.24 L1022.9,1141.94 L1022.75,1142.8 L1022.35,1143.54 L1021.86,1143.87 L1020.65,1144.11 L1020.13,1144.38 L1020.32,1145.08 L1020.75,1146.03 L1020.87,1146.53 L1020.77,1147.23 L1020.56,1147.77 L1020.61,1148.24 L1021.33,1148.7 L1021.77,1149.11 L1022.19,1149.7 L1022.29,1150.31 L1021.8,1150.8 L1021.8,1151.27 L1022.16,1151.68 L1023.48,1153.96 L1023.92,1154.15 L1024.34,1154.56 L1024.49,1155.59 L1024.35,1156.32 L1024.09,1156.77 L1023.86,1157.32 L1023.74,1158.33 L1023.55,1158.79 L1023.08,1159.1 L1022.52,1159.29 L1022.03,1159.35 L1019.92,1158.95 L1019.44,1159.08 L1019.14,1159.41 L1018.9,1159.82 L1018.57,1160.23 L1017.71,1160.76 L1017.18,1160.98 L1016.72,1161.06 L1016.43,1161.33 L1016.43,1161.88 L1016.72,1162.34 L1017.34,1162.35 L1017.34,1162.78 L1016.28,1162.74 L1015.79,1162.86 L1015.35,1163.23 L1015.71,1163.51 L1016.16,1163.62 L1017.19,1163.63 L1017.67,1163.87 L1018.18,1164.43 L1019.03,1165.78 L1018.09,1167 L1017.47,1167.58 L1016.8,1167.94 L1015.16,1168.26 L1014.51,1168.74 L1014.81,1169.63 L1015.25,1169.65 L1017.29,1168.84 L1018.21,1168.91 L1018.52,1168.85 L1018.78,1168.64 L1019.04,1168.35 L1019.34,1168.08 L1019.75,1167.96 L1020.52,1168.29 L1021.96,1169.91 L1022.71,1170.54 L1024.68,1171.08 L1025.17,1171.44 L1025.44,1172.35 L1026.22,1177.11 L1026.31,1177.39 L1026.25,1177.58 L1025.9,1177.85 L1025.36,1177.87 L1024.85,1177.56 L1024.44,1177.5 L1024.15,1178.27 L1024.6,1178.43 L1025.03,1178.7 L1025.38,1179.07 L1025.62,1179.53 L1025.76,1180.05 L1025.83,1180.78 L1025.67,1181.42 L1025.13,1181.7 L1024.72,1181.43 L1024.21,1180.25 L1023.89,1179.98 L1022.39,1180.09 L1021.92,1179.97 L1021.25,1179.57 L1021.3,1179.32 L1021.59,1179.08 L1021.66,1178.69 L1021.03,1177.76 L1020.79,1177.19 L1021.05,1176.94 L1022.02,1176.57 L1021.83,1175.85 L1020.97,1175.35 L1019.95,1175.65 L1019.75,1176.02 L1019.26,1177.3 L1018.85,1177.99 L1018.62,1178.46 L1018.46,1178.67 L1018.18,1178.76 L1017.6,1178.67 L1017.34,1178.88 L1016.72,1179.6 L1015.21,1180.87 L1014.71,1181.64 L1014.46,1182.96 L1014.51,1182.96 L1014.49,1182.98 L1013.95,1183.15 L1013.34,1183.08 L1012.61,1182.38 L1011.96,1182.52 L1011.56,1182.88 L1011.04,1183.01 L1010.69,1183.38 L1010.14,1183.47 L1007.5,1183.44 L1006.87,1183.17 L1006.61,1182.42 L1006.51,1181.55 L1006.35,1180.92 L1005.76,1180.37 L998.119,1178.47 L995.301,1177.26 L992.427,1176.72 L991.582,1176.83 L989.108,1177.72 L987.777,1177.63 L986.519,1177.21 L986.519,1177.21 L986.827,1176.14 L986.568,1175.71 L987.085,1173.99 L986.81,1172.4 L986.264,1170.85 L985.955,1169.25 L986.132,1167.23 L986.585,1165.56 L987.131,1164.03 L987.556,1162.41 L983.754,1161.65 L983.327,1161.69 L983.182,1162.21 L983.042,1164.62 L982.811,1164.93 L982.459,1164.99 L981.989,1165.5 L981.48,1166.19 L980.918,1166.51 L980.365,1166.37 L979.888,1165.67 L979.513,1164.11 L979.595,1162.61 L979.853,1161.1 L980.023,1159.48 L980.72,1158.27 L982.214,1158.07 L983.712,1158.55 L984.454,1159.36 L983.97,1159.48 L983.576,1159.79 L982.935,1160.61 L983.031,1161.31 L985.161,1161.66 L987.108,1162.17 L987.838,1161.89 L987.778,1160.7 L987.159,1158.55 L986.749,1155.86 L986.533,1153.18 L986.616,1152.47 L986.953,1151.59 L987.036,1151.05 L986.99,1150.25 L986.619,1148.68 L986.352,1145.85 L986.117,1144.51 L985.627,1143.3 L984.746,1142.57 L983.475,1142.23 L981.104,1142.12 L979.405,1141.54 L977.985,1140.15 L974.874,1135.26 L974.914,1134.82 L975.668,1134.69 L975.677,1134.3 L975.003,1134.01 L974.19,1133.92 L973.391,1134.11 L972.763,1134.63 L972.334,1135.5 L972.426,1136.09 L973.453,1137.24 L973.121,1137.31 L972.901,1137.46 L972.434,1138.07 L973.252,1138.24 L974.202,1138.9 L974.99,1139.85 L975.299,1140.94 L974.838,1141.1 L970.671,1137.46 L969.817,1136.93 L967.746,1136.49 L967.215,1135.92 L967.197,1134.99 L969.333,1129.7 L970.065,1127.3 L970.575,1124.74 L970.701,1122.21 L969.342,1109.88 L969.188,1109.24 L969.088,1109.11 L969.062,1108.81 L969.104,1108.52 L969.21,1108.39 L969.42,1108.46 L969.51,1108.63 L969.566,1108.79 L969.667,1108.86 L969.886,1109.2 L970.636,1114.02 L971.223,1115.69 L971.297,1116.39 L971.198,1116.98 L970.832,1117.83 L970.736,1118.53 L970.835,1119.39 L971.131,1119.57 L972.738,1118.72 L974.455,1117.42 L975.256,1117.16 L976.655,1116.11 L978.093,1114.66 L978.314,1113.65 L978.113,1112.56 L977.646,1111.49 L977.107,1110.52 L975.833,1108.78 L975.88,1106.84 L975.534,1104.59 L974.668,1103.13 L973.476,1102.23 L970.807,1100.93 L970.077,1100.81 L969.747,1101.33 L969.71,1104 L969.616,1105.27 L969.364,1106.68 L969.465,1107.73 L969.36,1108.13 L968.85,1107.95 L968.714,1107.62 L968.58,1106.96 L968.469,1106.2 L968.432,1101.66 L968.579,1100.25 L969.529,1096.36 L969.68,1094.9 L969.866,1073.23 L970.041,1072.19 L970.317,1071.12 L972.012,1065.75 L972.754,1064.09 L973.289,1063.3 L973.495,1063.47 L973.618,1064.02 L973.927,1064.34 L973.891,1064.78 L973.064,1066.79 L972.78,1067.34 L972.769,1067.76 L972.977,1067.98 L973.207,1068.1 L973.829,1068.18 L973.941,1068.45 L974.06,1069.06 L974.228,1069.66 L974.502,1069.94 L975.913,1070.23 L976.286,1070.5 L975.781,1070.79 L975.894,1071.37 L975.776,1072.38 L975.964,1072.97 L976.271,1073.22 L976.545,1073.07 L976.654,1072.65 L976.458,1072.13 L976.468,1071.7 L977.566,1071.36 L979.247,1071.36 L980.776,1071.71 L981.43,1072.45 L981.748,1072.56 L983.43,1073.99 L983.914,1073.78 L984.37,1073.25 L984.757,1072.55 L985.029,1071.88 L985.243,1071.89 L986.231,1074.85 L986.249,1075.33 L986.112,1075.59 L986.153,1076.18 L986.367,1076.77 L986.925,1077.24 L987.584,1078.13 L987.978,1078.39 L988.57,1078.39 L990.695,1077.97 L990.926,1077.86 L991.118,1077.55 L991.248,1077.07 L991.306,1076.49 L991.331,1075.58 L991.388,1074.96 L991.613,1073.68 L991.769,1073.32 L991.955,1073.18 L992.032,1072.98 L991.869,1072.44 L990.813,1071.57 L990.467,1071.53 L990.208,1071.37 L990.057,1071.05 L990.016,1070.5 L989.872,1069.95 L989.29,1069.62 L989.316,1068.98 L989.782,1068.57 L991.364,1068.49 L993.658,1064.74 L994.182,1064.44 L995.606,1064.02 L997.509,1063.97 L997.579,1063.83 L996.759,1063.51 L995.254,1063.43 L994.906,1063.08 L995.245,1061.97 L995.765,1061 L996.183,1060.48 L997.291,1059.67 L998.942,1058.1 L999.582,1057.96 L999.956,1058.04 L1000.59,1058.37 L1000.97,1058.45 L1001.38,1058.34 L1001.59,1058.08 L1001.76,1057.79 L1002.04,1057.56 L1002.64,1057.55 L1003.43,1057.82 L1004.11,1058.35 L1004.39,1059.11 L1004.58,1060.57 L1005.05,1061.45 L1005.59,1062.07 L1006.01,1062.78 L1006.45,1064.1 L1005.48,1064.33 L1004.45,1065.94 L1003.57,1066.17 L1003.57,1066.63 L1003.7,1067.21 L1003.32,1067.75 L1002.36,1068.72 L1002.03,1069.6 L1001.98,1070.39 L1002.09,1072.38 L1002.35,1072.99 L1002.92,1072.89 L1003.49,1072.23 L1003.98,1070.19 L1004.49,1070.06 L1005.12,1070.19 L1005.66,1070.08 L1005.46,1069.48 L1005.18,1069.25 L1004.85,1069.32 L1004.49,1069.6 L1004.76,1068.92 L1005.04,1068.43 L1005.29,1067.9 L1005.47,1067.05 L1005.7,1067.05 L1005.48,1068.45 L1006.38,1068.68 L1007.61,1068.48 L1008.38,1068.59 L1008.85,1068.92 L1009.38,1068.82 L1009.85,1068.87 L1010.16,1069.66 L1010.05,1070.27 L1009.24,1071.43 L1008.92,1072.25 L1009.3,1072.41 L1009.37,1072.68 L1009.38,1073.06 L1009.62,1073.54 L1009.95,1073.89 L1010.81,1074.4 L1010.95,1074.66 L1011.03,1075 L1011.18,1075.27 L1011.51,1075.31 L1012.02,1074.26 L1012.24,1073.95 L1012.72,1074.42 L1013.15,1073.92 L1013.15,1073.13 L1012.38,1072.71 L1011.3,1073.34 L1011.23,1073.53 L1011.07,1073.59 L1010.7,1073.55 L1010.41,1073.38 L1010.39,1073.01 L1010.5,1072.72 L1010.59,1072.69 L1010.68,1070.37 L1011.27,1068.37 L1011.59,1066.61 L1011.59,1066.59 L1010.91,1064.97 L1010.57,1064.58 L1010.4,1064.44 L1009.71,1064.53 L1008.32,1065.15 L1007.97,1065.64 L1007.22,1066.11 L1006.49,1066.15 L1006.19,1065.34 L1006.59,1064.59 L1008.19,1063.64 L1008.56,1062.59 L1008.77,1061.72 L1008.7,1061.29 L1007.84,1060.97 L1007.49,1060.66 L1007.2,1060.24 L1006.97,1059.8 L1006.79,1059.39 L1006.6,1058.75 L1006.48,1058 L1006.53,1057.23 L1006.71,1057.03 L1007.51,1055.79 L1007.6,1055.5 L1007.77,1055.17 L1007.75,1054.45 L1007.6,1053.73 L1007.38,1053.4 L1006.98,1053.31 L1006.68,1053.09 L1006.13,1052.49 L1006.79,1050.93 L1007.45,1049.7 L1008.15,1048.7 L1009.23,1047.62 L1009.47,1047.48 L1009.72,1047.39 L1010.02,1047.37 L1010.21,1047.17 L1010.4,1046.29 L1010.52,1046.09 L1015.6,1044.43 L1016.73,1044.48 L1020.71,1046.17 L1020.71,1046.64 L1020.27,1046.93 L1020.24,1047.31 L1020.5,1047.68 L1020.96,1047.92 L1021.43,1047.88 L1021.76,1047.56 L1023.02,1045.58 L1023.59,1044.45 L1024.22,1043.56 L1025.56,1043.07 L1026.7,1042.36 L1028.33,1042.79 L1031.98,1042.37 L1029.96,1040.48 L1028.94,1040.09 L1028.25,1039.5 L1027.88,1039.37 L1027.61,1039.52 L1026.4,1041.15 L1025.74,1041.84 L1025.01,1042.18 L1024.38,1041.68 L1023.76,1041.06 L1023.11,1041.23 L1021.91,1042.33 L1020.53,1042.91 L1017.56,1042.93 L1009.71,1045.69 L1009.04,1045.75 L1008.39,1045.45 L1004.61,1042.18 L1004.81,1043.01 L1004.53,1043.43 L1003.4,1043.88 L1002.56,1044.5 L1002.12,1044.71 L1001.54,1044.71 L1001.76,1044.32 L1001.15,1043.91 L1000.56,1044.04 L999.965,1044.5 L999.39,1045.14 L998.838,1045.07 L998.126,1045.75 L997.518,1046.49 L997.263,1046.6 L996.984,1045.76 L996.31,1045.43 L995.529,1045.5 L994.938,1045.89 L995.43,1046.44 L994.168,1047.58 L992.452,1048.45 L991.238,1048.07 L990.433,1048.23 L989.623,1048.59 L989.246,1049.03 L989.036,1050.24 L988.562,1050.97 L988.009,1051.52 L987.549,1052.2 L987.318,1053.98 L987.276,1054.12 L987.207,1054.66 L987.041,1054.97 L986.81,1055.23 L985.685,1056.9 L984.908,1056.96 L984.536,1057.12 L984.371,1057.48 L984.109,1057.73 L983.538,1057.94 L982.964,1058.36 L982.689,1059.17 L982.936,1061.3 L982.881,1062.29 L982.35,1062.86 L982.341,1063.24 L983.721,1063.13 L984.035,1063.28 L984.277,1064.05 L984.058,1064.57 L983.603,1064.88 L983.151,1064.97 L982.75,1064.83 L981.955,1064.23 L981.512,1064.08 L981.097,1064.19 L980.394,1064.74 L979.952,1064.9 L979.942,1065.37 L981.597,1064.94 L982.087,1065.11 L982.799,1065.69 L983.133,1065.83 L985.589,1064.81 L986.093,1065.03 L986.179,1065.49 L986.059,1066.68 L986.048,1067.21 L986.551,1068.05 L986.772,1068.65 L986.242,1069.56 L985.879,1069.99 L985.411,1070.18 L984.92,1070.25 L984.612,1070.44 L984.621,1070.71 L985.048,1071 L984.347,1071 L983.404,1070.09 L982.582,1068.92 L982.233,1068.17 L981.939,1067.24 L981.181,1066.47 L980.223,1065.95 L979.343,1065.74 L978.739,1065.4 L978.028,1064.59 L977.372,1063.61 L976.944,1062.74 L976.615,1060.47 L976.405,1060.12 L975.942,1059.95 L974.989,1059.28 L974.419,1059.22 L974.857,1061.54 L974.999,1062.89 L974.894,1063.74 L974.378,1063.81 L974.151,1062.49 L974.151,1059.6 L974.62,1057.03 L975.44,1055.12 L981.825,1045.21 L983.709,1043.33 L986.626,1038.9 L987.729,1037.89 L988.949,1037.94 L991.772,1039 L993.366,1039.12 L996.462,1038.47 L997.844,1037.78 L999.508,1035.98 L1000.12,1035.71 L1008.63,1037.07 L1011.73,1036.73 L1014.83,1035.6 L1017.76,1033.65 L1020.35,1030.79 L1027.7,1017.29 L1028.99,1015.38 L1031.32,1012.78 L1032.49,1010.54 L1032.94,1010.22 L1033.74,1009.04 L1035.37,1008.71 L1038.48,1008.94 L1041.5,1008.42 L1044.26,1007.17 L1046.64,1005.41 L1050.71,1001.13 L1052.93,999.436 L1053.83,999.078 L1054.84,998.963 L1055.81,999.183 L1056.62,999.833 L1054.8,1000.54 L1053.07,1002.24 L1050.34,1006.54"
         id="path1099" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['DK'],polycolors['DK'],labels['DK'], x1, polybary['DK'], x2, polybary['DK'] )
    if 'DK' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M500.871,1176 L500.722,1175.15 L500.358,1175.02 L499.876,1175.18 L499.366,1175.25 L498.883,1174.94 L498.397,1173.91 L497.908,1173.6 L497.061,1173.8 L496.206,1174.37 L495.36,1174.71 L494.559,1174.25 L494.655,1173.8 L494.87,1173.72 L495.018,1173.64 L495.189,1173.59 L495.471,1173.57 L495.678,1172.37 L497.25,1170.89 L497.802,1169.88 L498.206,1168.03 L498.51,1167.27 L499.036,1166.43 L499.28,1165.86 L499.324,1165.44 L499.447,1165.21 L500.28,1165.18 L501.271,1164.76 L502.223,1163.94 L503.524,1163.18 L504.17,1162.6 L506.534,1159.07 L507.721,1157.59 L509.245,1156.63 L511.383,1155.89 L512.79,1155.83 L513.534,1155.6 L514.216,1155.52 L514.738,1155.96 L514.351,1156.81 L513.147,1158.97 L513.046,1159.81 L513.144,1160.31 L513.167,1161.38 L513.281,1161.83 L513.521,1162.06 L514.16,1162.28 L514.387,1162.53 L514.506,1162.91 L514.614,1163.51 L514.555,1164.04 L514.186,1164.2 L513.939,1164.35 L513.218,1165.31 L512.907,1165.47 L512.588,1165.92 L512.278,1166.45 L511.977,1166.85 L511.708,1167.04 L510.612,1167.41 L510.568,1169.15 L509.601,1169.89 L508.232,1170.32 L507.003,1171.09 L506.879,1171.85 L504.639,1172.79 L503.984,1173.34 L503.668,1173.77 L503.115,1173.9 L502.519,1173.94 L502.068,1174.06 L501.688,1174.44 L500.871,1176"
         id="path1101" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['IM'],polycolors['IM'],labels['IM'], x1, polybary['IM'], x2, polybary['IM'] )
    if 'IM' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1446.48,910.012 L1446.74,910.1 L1447.04,909.911 L1447.49,909.228 L1447.67,909.065 L1448.37,909.144 L1448.97,909.411 L1449.56,909.505 L1450.2,909.08 L1450.99,907.54 L1451.56,906.918 L1452.8,907.299 L1454.43,906.564 L1455.18,906.608 L1455.97,906.924 L1457.54,907.171 L1459.25,907.975 L1461.16,908.463 L1462,909.278 L1462.71,910.461 L1463.03,910.79 L1463.59,911.16 L1465.03,911.602 L1466.45,911.691 L1467.29,911.835 L1468.51,912.363 L1469.66,913.147 L1470.34,914.048 L1470.24,914.965 L1467.91,913.9 L1467.26,914.631 L1467.68,914.667 L1468.09,914.777 L1468.47,915.064 L1468.8,915.607 L1467.37,915.662 L1466.67,915.41 L1465.92,914.883 L1465.74,914.613 L1465.52,914.202 L1465.24,913.842 L1464.88,913.73 L1464.39,913.82 L1464.36,913.956 L1464.23,914.327 L1464.21,914.564 L1464.22,915.49 L1464.17,915.865 L1464,916.317 L1463.83,916.594 L1463.43,917.094 L1463.2,917.1 L1462.88,917.036 L1462.63,917.204 L1462.65,917.909 L1462.73,918.282 L1462.85,918.671 L1462.98,919.03 L1463.13,919.323 L1462.5,920.047 L1461.98,921.034 L1461.41,921.666 L1459.76,920.797 L1459.03,920.97 L1458.89,921.532 L1459.75,922.168 L1458.16,925.543 L1457.7,925.912 L1457.21,926.173 L1456.73,926.639 L1456.02,928.015 L1455.64,928.452 L1454.93,928.71 L1453.42,928.829 L1452.76,929.172 L1452.55,930.012 L1454.24,930.694 L1453.34,931.929 L1451.64,932.486 L1450.91,931.14 L1450.35,930.887 L1449.72,930.839 L1449.12,930.989 L1448.64,931.35 L1448.24,931.952 L1448.09,932.251 L1447.92,932.15 L1447.43,931.565 L1446.9,931.26 L1446.69,931.723 L1446.61,932.819 L1446.41,933.058 L1446.33,933.217 L1446.25,933.32 L1446.03,933.39 L1445.98,933.179 L1445.63,932.284 L1445.58,932.092 L1445.12,932.148 L1444.59,932.507 L1444.14,933.022 L1443.88,933.535 L1443.59,933.913 L1443.18,933.832 L1442.77,933.67 L1442.44,933.79 L1442.23,934.036 L1441.68,934.498 L1441.44,934.6 L1440.56,935.401 L1439.8,936.41 L1439.48,937.12 L1439.28,937.364 L1439.24,937.69 L1439.6,939.998 L1439.7,941.378 L1439.66,942.773 L1439.22,945.462 L1439.05,946.079 L1438.77,946.64 L1438.55,947.243 L1438.61,947.849 L1439,949.204 L1438.69,949.903 L1434.8,954.943 L1433.78,955.157 L1432.67,954.003 L1432.68,953.261 L1432.44,952.603 L1432.03,952.148 L1431.53,952.028 L1431.43,951.939 L1431.34,951.706 L1431.27,951.42 L1431.27,951.162 L1432.14,951.096 L1432.39,950.972 L1432.54,950.506 L1432.66,949.144 L1432.76,948.543 L1433.65,946.414 L1433.98,945.017 L1434.19,944.396 L1434.49,944.101 L1434.99,944.165 L1435.46,944.372 L1435.88,944.478 L1436.25,944.233 L1436.35,943.791 L1436.31,942.47 L1436.4,941.833 L1437.3,939.706 L1436.41,939.035 L1435.06,938.717 L1432.73,938.715 L1431.98,938.16 L1430.29,935.442 L1429.42,934.902 L1427.3,935.236 L1426.19,935.067 L1425.1,934.368 L1424.38,933.39 L1424.01,932.708 L1423.9,932.183 L1424.25,931.824 L1426.05,932.038 L1425.51,931.165 L1424.78,930.294 L1424.45,929.582 L1425.08,929.164 L1425.74,929.484 L1426.4,930.194 L1427.01,930.602 L1427.53,930.014 L1427.07,928.983 L1427.38,928.263 L1428.07,927.936 L1428.79,928.062 L1428.37,927.404 L1427.74,926.142 L1425.82,923.587 L1424.96,922.798 L1423.97,921.602 L1423.44,921.116 L1422.37,920.381 L1421.29,919.349 L1423.13,919.852 L1423.66,919.775 L1424.11,919.498 L1424.14,919.267 L1424.1,918.94 L1424.31,918.368 L1425.55,918.209 L1426.54,918.362 L1427.5,919.042 L1429.37,921.428 L1429.67,922.027 L1429.89,922.859 L1430.01,923.214 L1430.18,923.384 L1430.37,923.431 L1430.57,923.418 L1430.82,923.126 L1430.73,922.59 L1430.39,921.71 L1430.28,920.058 L1430.31,919.326 L1430.52,918.605 L1430.87,918.123 L1431.81,917.389 L1432.13,916.827 L1432.35,915.905 L1432.41,915.423 L1432.6,915.202 L1433.18,915.067 L1433.29,915.187 L1433.46,915.487 L1433.63,915.838 L1433.72,916.117 L1433.9,916.847 L1434.1,917.186 L1434.63,917.459 L1435.62,917.464 L1435.71,916.175 L1435.44,914.472 L1435.36,913.228 L1436.26,912.085 L1440.71,910.782 L1443.1,908.023 L1444.1,907.531 L1444.51,907.799 L1445.38,909.137 L1445.8,909.403 L1446.03,909.497 L1446.48,910.012"
         id="path1103" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1444.12,883.505 L1444.71,883.815 L1447.86,883.015 L1451.23,883.963 L1451.88,884.681 L1452.1,885.297 L1452.12,886.278 L1452.29,886.818 L1452.68,887.349 L1453.46,887.784 L1453.78,888.087 L1454.06,888.425 L1454.61,888.861 L1454.9,889.164 L1455.05,889.43 L1455.14,889.742 L1455.26,890.444 L1455.44,890.831 L1455.68,890.911 L1455.91,890.877 L1456.03,890.916 L1456.32,892.155 L1456.22,892.84 L1455.78,893.001 L1454.28,892.316 L1453.92,892.235 L1453.57,892.242 L1453.38,892.497 L1453.38,892.905 L1453.26,893.139 L1452.78,892.86 L1452.3,893.228 L1451.76,893.333 L1451.28,893.6 L1451.04,894.452 L1451.34,894.406 L1451.57,894.529 L1451.76,894.842 L1451.91,895.376 L1452.01,896.159 L1451.89,896.336 L1451.64,896.39 L1451.37,896.803 L1450.85,897.26 L1449.94,897.463 L1449.03,897.308 L1448.52,896.687 L1448.75,895.979 L1449.36,895.345 L1449.76,894.742 L1449.39,894.14 L1448.84,894.194 L1448.12,894.709 L1447.45,895.419 L1447.03,896.049 L1446.62,897.678 L1446.2,901.448 L1445.61,902.91 L1444.35,903.919 L1442.61,904.454 L1440.79,904.284 L1439.31,903.167 L1439.56,902.902 L1439.78,902.551 L1440.16,901.673 L1439.6,901.095 L1438.8,900.065 L1438.07,898.907 L1437.66,897.945 L1437.39,896.455 L1437.14,895.599 L1436.7,894.972 L1434.83,893.403 L1433.85,892.952 L1432.84,892.838 L1431.8,893.112 L1429.83,894.013 L1428.79,894.094 L1424.14,892.49 L1423.46,891.95 L1423.62,890.872 L1424.82,890.454 L1427.24,890.395 L1436.4,887.484 L1435.7,887.173 L1435.58,886.752 L1435.95,886.526 L1436.73,886.812 L1437.09,886.708 L1436.66,884.539 L1437.08,883.88 L1437.79,883.705 L1438.31,883.374 L1438.68,882.779 L1438.95,881.813 L1438.96,881.244 L1438.86,880.725 L1438.83,880.203 L1439.01,879.626 L1439.31,879.411 L1440.7,879.317 L1440.72,879.319 L1441.56,879.3 L1442.57,879.482 L1443.24,879.973 L1442.85,881.852 L1443.35,882.793 L1444.12,883.505"
         id="path1105" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1533.45,828.842 L1534.1,829.028 L1534.42,828.877 L1534.64,828.772 L1535.03,828.116 L1535.28,827.109 L1534.95,826.744 L1535.04,825.203 L1535.81,825.499 L1536.78,826.482 L1537.48,827.006 L1538.07,826.699 L1538.04,826.287 L1537.78,825.786 L1537.7,825.233 L1537.94,824.543 L1538.25,824.272 L1538.69,824.282 L1539.89,824.609 L1541.35,825.491 L1541.53,825.709 L1541.67,826.038 L1541.82,826.477 L1542.14,826.513 L1547.95,825.049 L1548.59,825.098 L1550.05,825.637 L1552.07,825.642 L1554.06,826.279 L1555.38,826.344 L1556.07,826.212 L1556.54,825.896 L1557.37,824.786 L1557.76,824.487 L1559.07,824.296 L1560.36,824.38 L1561.67,824.745 L1563.05,825.421 L1564.7,826.525 L1566.51,827.738 L1568.16,827.934 L1569.67,828.443 L1570.35,828.539 L1575.99,827.672 L1580.82,826.153 L1584.88,824.817 L1589.95,824.818 L1594.77,823.906 L1599.91,823.85 L1601.07,823.099 L1602.04,821.962 L1602.65,820.584 L1603.18,818.186 L1604.09,819.167 L1605.67,819.539 L1606.52,820.075 L1607.74,821.689 L1608.18,822.164 L1608.66,822.504 L1609.64,822.981 L1610.1,823.31 L1610.27,824.491 L1609.39,824.206 L1608.34,825.69 L1605.9,826.556 L1605.05,827.31 L1607.54,827.621 L1608.26,827.401 L1607.87,828.424 L1604.24,829.295 L1602.78,830.718 L1602.35,831.784 L1602.9,832.786 L1602.72,834.232 L1602.98,839.05 L1602.55,844.616 L1602.36,846.043 L1601.69,848.905 L1601.71,850.362 L1601.41,851.395 L1601.38,851.483 L1599.26,854.698 L1596.22,859.312 L1596.13,862.999 L1596.02,867.173 L1597.32,869.792 L1598.77,872.723 L1602.31,879.865 L1606.31,887.886 L1605.58,892.978 L1606.03,896.164 L1607.18,898.932 L1608.38,900.376 L1609.39,901.592 L1611.39,904.064 L1613.62,905.936 L1614.28,907.655 L1614.49,909.3 L1615,911.042 L1615.68,911.134 L1616.4,911.477 L1616.98,912.896 L1617.03,913.715 L1616.98,914.444 L1617.04,915.174 L1617.44,915.983 L1617.79,916.255 L1621.41,916.767 L1622.1,917.23 L1622.69,918.363 L1622.5,919.286 L1621.89,920.021 L1620.09,921.378 L1615.62,923.275 L1615.04,923.52 L1614.68,923.886 L1614.97,923.956 L1615.83,924.409 L1615.05,925.091 L1614.7,925.566 L1614.52,926.18 L1614.63,927.244 L1615.44,928.871 L1615.6,929.652 L1615.2,930.411 L1612.77,932.035 L1612.38,932.701 L1612.25,933.11 L1612.21,933.633 L1612.28,934.248 L1612.39,934.545 L1612.52,934.769 L1612.81,935.555 L1612.94,935.734 L1613.06,935.936 L1613.31,937.211 L1613.29,937.495 L1613.13,937.848 L1612.78,938.149 L1612.35,938.29 L1611.98,938.531 L1611.84,939.123 L1612.05,939.884 L1612.54,940.842 L1613.09,941.728 L1613.52,942.302 L1612.62,943.268 L1607.12,942.652 L1604.46,943.129 L1603.59,942.939 L1602.73,942.321 L1601.07,940.757 L1600.15,940.33 L1598.93,940.216 L1598.36,940.341 L1597.82,940.609 L1597.29,940.637 L1596.82,940.292 L1596.34,940.078 L1595.81,940.506 L1595.49,941.402 L1595.39,942.387 L1595.21,943.326 L1594.68,944.089 L1594.15,944.401 L1591.58,944.871 L1590.58,945.491 L1589.67,946.382 L1588.92,947.473 L1588.81,947.917 L1588.8,948.394 L1588.74,948.804 L1588.44,949.038 L1588.1,949.006 L1587.39,948.708 L1587.06,948.699 L1586.68,949.066 L1586.33,949.575 L1585.85,949.784 L1585.07,949.236 L1583.6,947.92 L1582.92,947.511 L1582.91,947.51 L1580.05,946.993 L1579.59,946.909 L1578,946.249 L1576.57,945.184 L1572.27,940.817 L1570.88,939.802 L1566.56,937.6 L1566.1,936.911 L1565.71,935.674 L1565.67,935.091 L1565.77,934.685 L1565.8,934.293 L1565.55,933.735 L1565.24,933.492 L1564.81,933.387 L1557.84,934.329 L1556.84,933.721 L1556.14,932.594 L1555.55,931.421 L1554.86,930.678 L1554.36,930.774 L1554.03,930.982 L1552.43,931.978 L1551.63,932.229 L1550.84,932.283 L1550.02,931.999 L1549.13,931.247 L1548.84,930.801 L1548.72,930.4 L1548.7,930.339 L1548.57,930.015 L1548.19,929.619 L1547.82,929.467 L1546.62,929.467 L1545.15,928.926 L1542.64,926.965 L1541.18,926.542 L1540.86,926.633 L1540.67,926.793 L1540.48,926.884 L1540.15,926.769 L1539.86,926.454 L1539.22,925.43 L1538.91,925.057 L1538.22,924.706 L1537.61,924.728 L1537.29,925.225 L1537.49,926.294 L1538.77,927.754 L1539.21,928.683 L1538.76,929.678 L1537.85,930.444 L1537.34,930.555 L1536.77,930.255 L1536.08,929.021 L1535.44,927.576 L1534.71,926.413 L1533.74,926.027 L1532.24,926.397 L1531.53,926.788 L1530.92,927.394 L1530.57,927.959 L1529.07,930.382 L1528.37,931.135 L1525.96,932.033 L1525.29,932.709 L1524.75,933.482 L1524.47,933.874 L1523.6,933.911 L1522.63,933.543 L1521.56,933.493 L1521.22,934.229 L1520.99,936.05 L1520.57,936.562 L1518.64,937.197 L1516.31,937.46 L1515.69,937.74 L1514.32,939.675 L1513.18,940.63 L1512.87,941.189 L1512.64,942.759 L1512.41,943.454 L1511.74,944.041 L1510.82,944.281 L1509.28,943.985 L1509.28,943.985 L1509.28,943.982 L1509.5,943.358 L1509.63,942.722 L1509.6,942.001 L1509.52,941.251 L1509.49,940.538 L1510.21,936.386 L1510.26,933.851 L1511.42,931.611 L1511.75,930.409 L1511.12,927.441 L1511.07,926.046 L1510.95,925.284 L1510.64,924.674 L1510.27,924.159 L1509.98,923.531 L1509.82,922.846 L1509.95,921.448 L1509.82,920.77 L1509.66,920.099 L1509.63,919.395 L1510.09,918.352 L1510.92,917.512 L1511.62,916.523 L1511.7,915.03 L1511.11,913.78 L1510.17,912.89 L1506.98,911.277 L1505.54,910.985 L1504.19,911.138 L1503.19,911.929 L1502.94,912.504 L1502.73,913.308 L1502.61,914.17 L1502.66,914.923 L1502.81,915.886 L1502.8,916.464 L1502.18,918.449 L1501.9,919.103 L1501.51,919.573 L1500.96,919.859 L1499.71,919.766 L1499.12,919.882 L1498.68,920.538 L1498.39,922.055 L1498.15,922.37 L1497.53,922.259 L1497.11,921.948 L1495.78,920.462 L1494.01,919.393 L1492.91,918.467 L1490.58,917.978 L1489.26,918.119 L1488.59,917.966 L1488.39,917.809 L1487.92,917.349 L1487.73,917.223 L1487.43,917.243 L1487.27,917.454 L1487.13,917.702 L1486.93,917.858 L1486.6,917.772 L1486.24,917.572 L1485.9,917.519 L1485.67,917.866 L1485.54,918.264 L1485.39,918.573 L1485.18,918.677 L1484.88,918.478 L1484.58,918.006 L1484.26,917.336 L1483.98,916.621 L1483.8,916.028 L1483.46,915.145 L1483,914.737 L1482.5,914.466 L1482.03,913.978 L1481.68,913.213 L1481.55,912.579 L1481.44,911.037 L1481.1,909.21 L1480.62,907.737 L1479.98,907.092 L1479.15,907.753 L1478.74,907.786 L1478.04,907.485 L1477.32,907.055 L1476.89,906.687 L1476.48,905.874 L1476.3,905.175 L1476.11,904.844 L1475.67,905.138 L1475.35,905.814 L1475.23,906.522 L1475.02,906.876 L1474.45,906.485 L1474.22,905.908 L1474.29,905.209 L1474.45,904.484 L1474.48,903.845 L1474.06,902.734 L1473.39,901.284 L1472.81,899.715 L1472.67,898.261 L1472.85,897.667 L1473.09,897.535 L1473.82,897.646 L1474.02,897.47 L1474.08,897.159 L1474,896.874 L1473.38,896.448 L1473.13,895.647 L1473.17,894.916 L1473.59,894.818 L1474.93,895.184 L1476.08,894.74 L1477.17,894.046 L1478.36,893.671 L1479.04,893.804 L1480.44,894.388 L1481,894.461 L1481.98,894.268 L1482.04,894.055 L1481.99,893.612 L1481.84,893.191 L1481.62,893.032 L1481.06,892.843 L1480.8,892.217 L1480.84,891.47 L1481.2,890.929 L1481.74,890.87 L1483.13,891.322 L1483.76,891.298 L1482.69,890.45 L1481.53,889.93 L1480.51,890.065 L1479.87,891.192 L1479.13,890.528 L1478.46,890.59 L1477.19,891.278 L1474.83,891.57 L1474.13,891.875 L1473.57,892.389 L1473.3,892.754 L1473.14,893.14 L1472.95,893.233 L1471.88,893.143 L1471.29,893.372 L1469.65,894.484 L1470.39,892.05 L1470.89,891.384 L1471.91,890.996 L1472.06,890.513 L1469.52,888.167 L1469.01,887.061 L1467.66,885.681 L1467.4,884.452 L1467.9,883.377 L1470.04,883.121 L1470.54,882.06 L1471.16,882.314 L1471.74,882.247 L1472.25,881.929 L1473.1,880.863 L1473.3,880.51 L1473.3,880.09 L1473.16,879.334 L1472.66,877.904 L1471.93,876.938 L1471.09,876.33 L1470.24,875.98 L1470.79,876.707 L1471.92,877.463 L1472.48,878.196 L1472.25,878.42 L1471.68,878.596 L1471.48,878.788 L1471,879.755 L1471,879.976 L1471.08,880.34 L1471.04,881.057 L1470.74,881.179 L1470.49,880.942 L1470.25,880.593 L1470.01,880.422 L1467.86,880.298 L1466.88,879.756 L1465.78,878.578 L1466,878.574 L1466.44,878.432 L1466.66,878.408 L1466.21,877.343 L1466.24,876.54 L1467.34,874.57 L1467.61,874.222 L1467.98,874.333 L1468.67,874.974 L1468.06,871.791 L1466.4,868.971 L1465.45,866.728 L1466.95,865.272 L1467.31,865.249 L1468.03,865.394 L1468.41,865.386 L1468.66,865.202 L1469,864.54 L1469.64,864.026 L1469.92,863.441 L1470.27,862.904 L1470.9,862.723 L1472.91,864.031 L1473.01,864.027 L1473.4,864.016 L1473.65,863.391 L1473.54,862.726 L1473.2,862.189 L1472.75,861.928 L1472.94,861.081 L1473.55,860.583 L1476.32,859.971 L1477.76,860.063 L1479.21,859.658 L1480.71,858.697 L1482.28,858.231 L1483.88,859.315 L1484.19,857.768 L1483.71,856.71 L1481.12,854.346 L1480.72,853.812 L1480.67,853.127 L1480.93,852.401 L1481.27,851.922 L1481.7,851.689 L1482.27,851.696 L1483.29,852.17 L1485.45,853.701 L1486.44,853.906 L1487.22,853.271 L1486.73,852.277 L1485.67,851.312 L1484.77,850.754 L1485.3,850.232 L1488.02,849.464 L1489.02,848.811 L1489.22,848.008 L1489.16,845.918 L1489.2,845.475 L1491.15,844.548 L1491.88,844.485 L1492.59,844.823 L1494,845.906 L1494.69,846.056 L1495.21,845.666 L1495.58,845.041 L1495.99,844.559 L1496.62,844.616 L1498.7,845.533 L1499.29,845.569 L1499.79,845.195 L1499.54,844.652 L1498.91,844.197 L1498.28,844.069 L1498.18,843.61 L1499.02,843.523 L1499.3,843.28 L1499.31,842.697 L1499.28,842.003 L1499.49,841.704 L1499.86,841.69 L1500.26,841.863 L1501.03,842.672 L1501.66,843.65 L1502.32,844.239 L1503.16,843.876 L1503.45,843.513 L1503.58,843.22 L1503.96,841.529 L1503.97,841.188 L1503.48,840.305 L1503.2,839.99 L1502.86,839.795 L1502.55,839.532 L1502.32,838.996 L1502.18,838.419 L1502.06,837.992 L1501.9,837.599 L1501.66,837.147 L1502.48,836.298 L1503.88,837.03 L1506.57,839.208 L1507.4,839.254 L1509.68,838.101 L1510.45,838.137 L1511.93,838.726 L1512.62,838.781 L1513.13,838.334 L1512.5,836.808 L1512.8,836.07 L1513.53,835.934 L1515.27,836.614 L1522.34,837.088 L1522.4,837.077 L1521.85,835.702 L1522.08,834.829 L1522.77,834.353 L1525.65,833.715 L1525.94,833.453 L1525.82,833.023 L1525.5,832.64 L1525.13,832.518 L1523.98,831.592 L1522.94,829.235 L1522.26,826.791 L1522.19,825.628 L1522.84,825.97 L1524.34,827.777 L1525.01,828.112 L1525.74,828.24 L1526.42,828.786 L1527.72,830.137 L1528.53,830.51 L1529.61,830.638 L1530.47,830.238 L1530.59,829.031 L1530.11,828.049 L1529.48,827.462 L1528.96,826.759 L1528.84,825.453 L1528.63,825.445 L1528.3,825.269 L1528.13,825.217 L1528.65,824.11 L1529.67,824.204 L1531.81,825.659 L1531.69,826.166 L1532.26,826.86 L1532.87,828.282 L1533.45,828.842"
         id="path1107" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['EE'],polycolors['EE'],labels['EE'], x1, polybary['EE'], x2, polybary['EE'] )
    if 'EE' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1429.91,1496.2 L1430.72,1496.01 L1432,1495.28 L1432.4,1495.51 L1432.41,1495.52 L1432.94,1496.64 L1433.28,1497.65 L1433.91,1500.62 L1433.88,1500.88 L1433.63,1501.39 L1433.64,1501.65 L1433.64,1501.65 L1433.81,1501.84 L1434.42,1502.18 L1434.42,1502.18 L1434.51,1502.31 L1434.52,1502.33 L1434.87,1502.62 L1434.89,1502.63 L1435,1502.89 L1434.95,1503.25 L1434.73,1503.42 L1434.48,1503.53 L1434.32,1503.7 L1433.12,1506.42 L1433.06,1507.25 L1433.06,1507.26 L1433.68,1507.91 L1434.75,1508.15 L1436.64,1508.04 L1437.5,1507.58 L1437.99,1506.92 L1438.67,1505.99 L1439.34,1505.34 L1440.61,1505.02 L1441.82,1505.46 L1444.18,1507.14 L1445.61,1507.62 L1446.32,1506.9 L1446.53,1505.29 L1446.47,1503.09 L1446.89,1502.41 L1447.07,1501.68 L1447.2,1500.91 L1447.74,1499.06 L1447.93,1498.72 L1448.04,1498.8 L1448.95,1498.79 L1449.29,1498.36 L1450.03,1497.18 L1450.6,1496.75 L1453.68,1496.03 L1453.84,1495.73 L1454.01,1494.83 L1454.1,1493.92 L1454.1,1493.33 L1454.2,1492.93 L1454.57,1492.56 L1455.05,1492.48 L1456.79,1492.87 L1458.86,1492.49 L1458.71,1491.43 L1459.32,1491.19 L1462.97,1492.45 L1463.89,1492.54 L1464.8,1492.06 L1465.2,1491.56 L1465.73,1490.66 L1466.11,1490.37 L1466.47,1490.29 L1467.36,1490.29 L1468.85,1489.82 L1469.5,1489.84 L1473.7,1493.61 L1474.41,1493.99 L1475.33,1493.99 L1476.02,1493.85 L1476.67,1493.85 L1477.5,1494.3 L1478.26,1494.89 L1478.99,1495.27 L1479.75,1495.3 L1480.65,1494.81 L1481.41,1493.92 L1482.44,1491.79 L1483.15,1490.85 L1483.77,1490.62 L1484.62,1490.48 L1485.34,1490.22 L1485.58,1489.6 L1484.95,1488.74 L1483.94,1488.32 L1483.36,1487.83 L1484.04,1486.79 L1484.62,1486.4 L1486.2,1485.82 L1486.84,1485.78 L1487.64,1486.17 L1488.31,1486.73 L1488.99,1487.06 L1489.84,1486.78 L1490.43,1486.02 L1491.4,1483.96 L1492.05,1483.34 L1492.63,1483.39 L1495.07,1484.25 L1499.16,1484.75 L1499.83,1484.64 L1501.27,1484.07 L1501.89,1484.15 L1502.57,1483.74 L1503.15,1483.39 L1505.98,1482.61 L1506.76,1482.68 L1507.21,1482.89 L1508.07,1483.51 L1508.52,1483.73 L1508.83,1483.72 L1509.44,1483.48 L1509.85,1483.53 L1510.64,1484.12 L1512.32,1485.99 L1513.02,1486.4 L1513.4,1486.05 L1513.86,1485.28 L1514.46,1484.52 L1515.24,1484.24 L1516.05,1484.54 L1517.74,1485.67 L1520.08,1486.4 L1521.76,1487.54 L1523.22,1489.22 L1524.08,1491.32 L1524.66,1493.25 L1525.94,1493.9 L1528.89,1493.97 L1530.34,1494.6 L1530.84,1494.73 L1531.25,1494.69 L1532.2,1494.41 L1532.55,1494.42 L1533.07,1494.84 L1533.14,1495.27 L1533.1,1495.69 L1533.25,1496.06 L1533.6,1496.27 L1535.59,1496.53 L1537.81,1496.07 L1538.72,1496.27 L1541.11,1497.98 L1542.67,1498.19 L1545.86,1497.78 L1547.37,1498.23 L1547.24,1499.31 L1547.18,1500.77 L1547.23,1502.22 L1547.42,1503.27 L1546.89,1503.91 L1545.37,1504.41 L1544.75,1505.09 L1544.3,1507.83 L1543.98,1509.02 L1543.98,1509.33 L1544.12,1510.13 L1544.1,1510.52 L1543.82,1511.14 L1542.98,1512.13 L1542.74,1512.63 L1542.69,1512.75 L1542.54,1513.55 L1542.59,1514.04 L1542.76,1514.53 L1543.19,1516.61 L1543.17,1517.04 L1543,1517.8 L1542.73,1518.37 L1542.4,1518.74 L1542.16,1519.2 L1542.17,1520.05 L1542.44,1521.36 L1542.39,1522.72 L1542.09,1523.96 L1541.58,1524.94 L1541.14,1525.34 L1540.11,1525.87 L1539.66,1526.3 L1539.42,1526.79 L1539.16,1527.85 L1538.98,1528.34 L1536.67,1530.95 L1536.22,1532.06 L1536.34,1533.33 L1537.22,1535.75 L1537.23,1536.75 L1536.94,1537.83 L1537.63,1542.29 L1537.01,1543.42 L1536.39,1544.11 L1535.66,1544.46 L1533.17,1544.62 L1532.43,1544.79 L1531.72,1545.18 L1529.59,1545.64 L1529.03,1545.98 L1527.89,1546.9 L1526.17,1547.45 L1524.21,1548.89 L1522.98,1549.2 L1521.62,1548.96 L1520.41,1548.33 L1519.26,1547.35 L1516.41,1544.17 L1515.97,1543.56 L1515.04,1541.09 L1514.58,1540.45 L1513.85,1540.4 L1512.34,1540.66 L1511.64,1540.46 L1511.3,1540.07 L1510.74,1538.93 L1510.39,1538.44 L1509.99,1538.17 L1509.21,1538 L1507.7,1537.39 L1507.08,1537.3 L1505.06,1538.31 L1503.66,1538.56 L1503,1538.8 L1502.27,1539.4 L1502.01,1540.03 L1501.88,1540.73 L1501.51,1541.5 L1500.95,1541.93 L1500.45,1541.8 L1499.88,1541.45 L1499.09,1541.24 L1497.78,1541.76 L1494.83,1543.76 L1493.75,1543.68 L1492.8,1542.99 L1491.58,1542.64 L1490.34,1542.6 L1489.33,1542.84 L1487.84,1542.93 L1485.41,1541.89 L1484.1,1541.95 L1483.49,1541.88 L1482.11,1540.94 L1481.42,1540.7 L1480.76,1540.81 L1472.37,1544.09 L1469.84,1544.63 L1468.73,1545.28 L1468.74,1545.76 L1468.8,1546.27 L1469.04,1547.34 L1469.09,1547.55 L1469.13,1547.77 L1469.14,1547.99 L1469.14,1548.22 L1468.7,1549.12 L1468.62,1549.24 L1467.55,1550.74 L1467.05,1551.65 L1466.71,1552.68 L1465.79,1557.83 L1465.17,1559.74 L1464.36,1561.47 L1463.32,1562.85 L1462.45,1563.47 L1461.92,1563.33 L1461.38,1562.88 L1460.5,1562.58 L1460.07,1562.85 L1458.98,1564.18 L1458.32,1564.58 L1457.59,1564.6 L1457.14,1564.51 L1456.8,1564.63 L1456.36,1565.31 L1456.26,1565.77 L1456.18,1566.92 L1455.96,1567.51 L1455.65,1567.83 L1454.91,1568.24 L1453.4,1569.48 L1453.27,1569.58 L1451.76,1570.29 L1450.88,1571.02 L1449.24,1573.03 L1448.31,1573.52 L1447.41,1573.3 L1445.64,1572.09 L1444.58,1571.9 L1443.16,1572.67 L1442.76,1572.67 L1442.29,1572.13 L1442.33,1571.6 L1442.5,1571.03 L1442.4,1570.42 L1441.54,1569.63 L1440.55,1569.59 L1438.69,1570.22 L1438.25,1570.06 L1437.21,1569.06 L1436.68,1568.72 L1436.25,1568.66 L1435.81,1568.7 L1432.13,1570.21 L1431.47,1570.72 L1431.13,1571.68 L1431.04,1574.14 L1430.67,1575.18 L1430.66,1575.21 L1430.66,1575.24 L1430.67,1575.28 L1430.69,1575.32 L1430.8,1575.65 L1430.86,1575.99 L1430.88,1576.32 L1430.86,1576.64 L1428.83,1578.51 L1423.21,1579.11 L1420.92,1581.02 L1420.48,1581.17 L1415.25,1581.18 L1412.81,1581.85 L1411.98,1581.92 L1411.04,1581.99 L1410.47,1582.35 L1408.51,1583.04 L1404.66,1584.41 L1404,1585.09 L1403.16,1587.55 L1402.8,1587.94 L1402.01,1588.17 L1401.66,1588.44 L1401.19,1589.17 L1401.27,1589.4 L1401.59,1589.61 L1401.81,1590.31 L1401.69,1592.93 L1401.74,1594.25 L1402.15,1595.38 L1403.51,1596.43 L1405.25,1597.41 L1405.19,1597.44 L1404.2,1597.93 L1403.28,1598.29 L1402.62,1598.92 L1401.42,1600.67 L1400.47,1601.42 L1399.27,1601.69 L1397.99,1601.6 L1396.39,1601.1 L1394.5,1601.19 L1386.04,1603.17 L1383.05,1604.8 L1381.51,1605.12 L1376.29,1605.16 L1366.75,1606.82 L1364.35,1607.24 L1360.74,1606.65 L1359.73,1606.24 L1357.85,1605.46 L1357.36,1605.02 L1356.97,1604.41 L1356.12,1603.75 L1355.19,1603.22 L1354.53,1602.97 L1354.07,1603.09 L1353.67,1603.34 L1353.24,1603.4 L1352.67,1602.93 L1351.05,1600.95 L1350.63,1600.73 L1349.56,1600.62 L1349.11,1600.48 L1348.66,1600.14 L1343.81,1594.82 L1342.39,1593.86 L1339.62,1593.72 L1339.18,1593.64 L1337.4,1593.33 L1335.87,1593.15 L1334.43,1594.23 L1333.37,1593.44 L1331.98,1593.09 L1330.97,1592.66 L1331.1,1591.62 L1330.56,1591.25 L1330.39,1591.16 L1330.52,1589.98 L1330.67,1589.33 L1331.05,1588.76 L1330.45,1588.23 L1330.22,1587.89 L1329.49,1586.81 L1329.03,1586.57 L1328.35,1586.54 L1327.74,1586.24 L1326.59,1585.19 L1326.2,1584.24 L1326.07,1582.89 L1325.77,1581.79 L1324.91,1579.61 L1324.87,1579.32 L1324.86,1578.6 L1324.8,1578.33 L1324.62,1578.13 L1324.36,1578.07 L1323.86,1577.69 L1323.46,1577.55 L1323.13,1577.38 L1322.96,1577.02 L1322.88,1576.75 L1322.71,1576.4 L1322.54,1576.1 L1322.42,1575.98 L1322.23,1575.87 L1322.28,1575.55 L1322.41,1575.17 L1322.47,1574.88 L1322.24,1574.41 L1321.77,1573.97 L1321.3,1573.65 L1321.05,1573.54 L1320.21,1573.53 L1319.87,1573.36 L1319.71,1573 L1319.67,1572.31 L1319.72,1572.02 L1319.8,1571.83 L1319.88,1571.43 L1319.94,1568.07 L1320.03,1567.16 L1320.37,1566.28 L1321.31,1564.62 L1321.46,1563.77 L1321.71,1563.15 L1322.34,1562.54 L1322.96,1561.76 L1323.02,1561.43 L1323.16,1560.63 L1322.51,1557.75 L1322.5,1556.56 L1323.08,1555.7 L1323.47,1554.79 L1325,1548.58 L1325.85,1546.76 L1327.15,1545.39 L1327.67,1544.59 L1327.8,1543.39 L1328.11,1542.77 L1330.17,1540.95 L1331.97,1540.38 L1333.99,1540.7 L1338.91,1542.61 L1339.61,1542.4 L1340.36,1541.79 L1341.06,1541.22 L1342.01,1540.84 L1342.63,1541.07 L1344.02,1542.27 L1345.55,1542.35 L1353.12,1538.42 L1353.78,1537.74 L1354.94,1535.16 L1355.55,1534.54 L1356.54,1534.25 L1357.41,1534.31 L1358.19,1534.18 L1358.87,1533.36 L1359.11,1532.48 L1359.23,1530.52 L1359.38,1529.56 L1359.81,1528.42 L1360.61,1527.77 L1361.58,1527.51 L1363.79,1527.44 L1365.08,1526.67 L1366.16,1525.42 L1366.85,1523.89 L1366.98,1522.49 L1366.62,1519.72 L1366.58,1519.34 L1366.54,1517.66 L1366.84,1515.56 L1367.39,1513.58 L1368.2,1511.87 L1369.27,1510.6 L1374.5,1507.85 L1375.87,1506.5 L1376.74,1505.64 L1376.51,1502.68 L1377.72,1502.81 L1378.56,1502.11 L1380.05,1499.79 L1381.3,1498.92 L1381.59,1498.54 L1381.73,1497.97 L1381.7,1496.8 L1381.85,1496.27 L1382.62,1495.64 L1384.44,1495.69 L1385.52,1494.95 L1385.8,1494.86 L1386.08,1494.88 L1387.49,1495.39 L1388.7,1495.61 L1389.82,1495.41 L1391.3,1493.68 L1392.02,1493.3 L1393.67,1493.06 L1397.72,1492.94 L1398.97,1493.52 L1398.74,1494.61 L1398.89,1495.49 L1399.19,1496.31 L1399.4,1497.21 L1399.48,1499.28 L1399.79,1499.95 L1400.6,1500 L1401.62,1499.79 L1403.07,1498.69 L1403.93,1498.37 L1404.32,1498.45 L1405.19,1498.94 L1405.6,1499.09 L1406.03,1499.07 L1407.02,1498.74 L1408.2,1498.08 L1408.43,1497.55 L1408.44,1496.76 L1408.58,1495.51 L1408.69,1495.47 L1409.19,1495.4 L1409.35,1495.24 L1409.39,1494.81 L1409.28,1494.53 L1409.13,1494.36 L1409.05,1494.3 L1409.44,1492.2 L1409.88,1491.25 L1410.4,1490.61 L1411.02,1490.2 L1411.78,1489.94 L1413.02,1489.79 L1413.43,1489.67 L1413.97,1489.37 L1414.24,1489.06 L1417.25,1484.75 L1417.36,1484.42 L1417.58,1484.29 L1417.82,1484.35 L1418.18,1484.45 L1419.01,1485.56 L1419.35,1485.85 L1420.38,1486.35 L1421.03,1487.54 L1422.16,1490.57 L1423.07,1492.47 L1423.33,1492.88 L1424.09,1493.33 L1424.97,1493.44 L1426.58,1493.25 L1426.45,1493.66 L1426.43,1494.01 L1426.58,1494.98 L1426.59,1495.72 L1426.59,1495.74 L1429.01,1496.22 L1429.91,1496.2"
         id="path1109" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['SK'],polycolors['SK'],labels['SK'], x1, polybary['SK'], x2, polybary['SK'] )
    if 'SK' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g
       id="EL" class="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1746.73,2320.47 L1748.19,2320.6 L1748.92,2320.2 L1751.82,2319.99 L1753.07,2319.71 L1753.57,2319.45 L1753.98,2319.1 L1754.23,2318.64 L1754.24,2317.99 L1754.38,2317.44 L1754.9,2317.18 L1756.21,2317.02 L1755.24,2314.95 L1756.81,2313.94 L1759.14,2313.78 L1760.39,2314.27 L1761.85,2316.75 L1761.79,2317.23 L1761.58,2317.45 L1761.3,2318.35 L1761.11,2318.69 L1760.67,2319 L1760.19,2319.2 L1759.13,2319.45 L1757.09,2319.39 L1755.95,2319.54 L1755.3,2320.16 L1755.64,2320.64 L1756.6,2321.04 L1757.62,2321.3 L1759.18,2321.46 L1761.03,2322.3 L1761.91,2322.41 L1762.78,2322.08 L1763.63,2321.5 L1764.38,2320.8 L1764.96,2320.12 L1766.01,2322.81 L1766.27,2323.95 L1766.52,2326.16 L1766.71,2326.67 L1767.15,2326.9 L1768.19,2326.77 L1768.72,2326.95 L1770.71,2327.27 L1774.95,2325.12 L1776.85,2325.22 L1777.82,2324.42 L1781.82,2322.9 L1783.11,2322.21 L1785.99,2319.99 L1786.99,2319.43 L1788.05,2319.01 L1789.18,2318.77 L1794.34,2318.66 L1795.48,2319.01 L1795.9,2318.56 L1796.68,2317.98 L1798.03,2317.21 L1801.37,2317.85 L1801.6,2317.41 L1801.77,2317.23 L1801.86,2317.01 L1801.85,2316.46 L1804.35,2316.57 L1805.02,2317.17 L1805.9,2316.86 L1806.21,2316.89 L1806.34,2317.28 L1806.29,2317.64 L1806.16,2318 L1806.14,2318.39 L1806.83,2319.85 L1808.04,2320.29 L1809.56,2320.11 L1811.88,2319.51 L1813.27,2318.89 L1814.06,2318.78 L1814.25,2318.89 L1814.55,2319.13 L1814.94,2319.33 L1815.43,2319.37 L1816.96,2319.06 L1819.67,2318.04 L1820.99,2318.17 L1821.74,2318.07 L1822.43,2317.48 L1822.86,2317.87 L1823.16,2317.34 L1824.23,2318.02 L1825.11,2318.86 L1826.12,2319.49 L1827.61,2319.53 L1829,2319.03 L1836.4,2314.45 L1837.42,2314.02 L1838.03,2314.03 L1839.2,2314.39 L1839.99,2314.36 L1842.95,2313.74 L1842.73,2314.65 L1842.28,2315.47 L1841.3,2316.71 L1841.75,2317.01 L1841.54,2317.64 L1841.58,2318.13 L1841.86,2318.47 L1842.4,2318.64 L1842.15,2317.59 L1842.31,2317.06 L1842.74,2317.05 L1843.25,2317.56 L1843.06,2318.03 L1843.09,2318.24 L1843.41,2318.43 L1843.2,2318.94 L1842.91,2318.78 L1842.71,2318.76 L1842.51,2318.97 L1842.2,2319.55 L1842.45,2320.05 L1842.15,2320.83 L1841.95,2321.81 L1842.44,2322.99 L1841.92,2323.62 L1842.09,2324.14 L1842.53,2324.67 L1842.85,2325.28 L1843.26,2326.47 L1843.8,2326.5 L1844.51,2326.22 L1845.42,2326.45 L1846.18,2326.83 L1846.84,2326.88 L1847.96,2326.59 L1848.53,2326.06 L1849.1,2324.16 L1849.61,2323.65 L1850.02,2323.54 L1850.3,2323.4 L1850.73,2322.94 L1851.17,2321.74 L1851.19,2321.56 L1851.77,2321.28 L1853.4,2320.82 L1855.21,2320.04 L1855.53,2319.82 L1856.92,2318.47 L1857.31,2317.93 L1857.63,2317.61 L1858.09,2317.39 L1858.45,2317.38 L1858.72,2317.3 L1858.93,2316.86 L1859.28,2316.79 L1860.09,2317.23 L1862.12,2317.32 L1861.65,2318.05 L1862.84,2318.4 L1864.1,2318 L1865.23,2317.18 L1866.02,2316.25 L1865.74,2316.2 L1865.72,2316.13 L1865.6,2315.92 L1866.74,2314.32 L1867.47,2312.79 L1868.39,2311.6 L1870.09,2311.02 L1869.56,2310.31 L1870.11,2310.21 L1870.28,2310.15 L1870.13,2309.97 L1869.92,2309.55 L1869.78,2309.36 L1871.14,2309.06 L1870.91,2310.27 L1870.32,2311.16 L1869.59,2311.99 L1868.99,2313.02 L1870.48,2315.98 L1871.53,2317.19 L1872.48,2316.2 L1872.19,2317.25 L1871.54,2318.41 L1871.2,2319.47 L1871.81,2320.24 L1871.66,2321.43 L1871.39,2322.64 L1871.04,2323.73 L1870.63,2324.6 L1870.47,2325.17 L1870.24,2326.74 L1870.01,2327.11 L1869.44,2327.34 L1868.99,2327.7 L1868.6,2328.08 L1865.91,2330 L1865.22,2330.33 L1863.81,2330.63 L1863.7,2330.47 L1862.82,2329.81 L1862.39,2329.64 L1861.16,2329.7 L1860.17,2330.04 L1859.27,2330.22 L1858.25,2329.82 L1857.72,2329.53 L1857.28,2329.49 L1856.89,2329.71 L1856.52,2330.19 L1856.14,2330.38 L1854.25,2330.45 L1852.18,2331.33 L1848.69,2333.4 L1838.12,2335.1 L1835.1,2336.76 L1834.4,2337.39 L1833.61,2337.85 L1828.9,2338.23 L1825.49,2339.06 L1824.57,2339.52 L1823.35,2339.92 L1822.92,2340.25 L1822.25,2340.91 L1819.57,2342.61 L1818.81,2342.93 L1815.88,2343.25 L1814.81,2343.73 L1814.53,2343.99 L1814.2,2344.57 L1813.88,2344.77 L1813.35,2344.88 L1813,2344.79 L1812.68,2344.67 L1812.26,2344.66 L1811.38,2344.91 L1810.65,2345.23 L1809.43,2346.05 L1808.57,2345.88 L1807.56,2345.97 L1795.17,2348.84 L1795.1,2348.46 L1795.68,2347.8 L1795.74,2347.07 L1795.2,2344.78 L1795.16,2344.03 L1795.27,2342.79 L1795.08,2341.94 L1794.82,2341.36 L1794.39,2340.58 L1793.89,2339.91 L1793.44,2339.67 L1793.15,2339.59 L1792.85,2339.35 L1792.46,2339.13 L1791.9,2339.11 L1787.39,2339.9 L1786.12,2339.88 L1785.31,2339.55 L1784.99,2339.61 L1784.54,2339.89 L1783.89,2339.49 L1782.6,2338.35 L1781.84,2338 L1780.83,2337.69 L1779.8,2337.66 L1778.99,2338.18 L1775.68,2337.24 L1775.5,2336.66 L1775.43,2336.28 L1769.99,2337.86 L1766.08,2338.04 L1765.83,2337.91 L1765.53,2337.65 L1765.3,2337.4 L1765.23,2337.29 L1764.9,2337.37 L1764.65,2337.59 L1764.46,2337.82 L1764.24,2337.95 L1760.6,2338.62 L1759.66,2338.38 L1759.18,2338.47 L1759.07,2339.12 L1758.85,2339.1 L1757.55,2339.62 L1756.55,2339.39 L1754.83,2337.88 L1753.97,2337.62 L1749.64,2338.55 L1747.47,2338.62 L1745.61,2337.94 L1745.24,2338.34 L1744.96,2338.55 L1744.66,2338.61 L1744.29,2338.57 L1743.52,2339.44 L1739.92,2339.96 L1738.59,2340.89 L1737.75,2340.32 L1734.8,2340.99 L1733.57,2340.93 L1733.27,2340.62 L1733.29,2340.05 L1733.03,2339.74 L1732.62,2339.59 L1731.92,2339.64 L1731.57,2339.57 L1731.1,2339.21 L1730.7,2338.8 L1730.25,2338.44 L1729.55,2338.21 L1729.86,2337.72 L1729.65,2337.72 L1729.53,2337.73 L1729.47,2337.66 L1729.42,2337.38 L1730.1,2336.86 L1730.4,2336.74 L1730.77,2336.67 L1730.8,2336.48 L1730.77,2336.35 L1730.81,2336.27 L1731.05,2336.22 L1730.67,2335.84 L1730.35,2335.41 L1730.21,2334.89 L1730.38,2334.16 L1729.82,2333.14 L1730.15,2331.64 L1730.75,2330.14 L1731.02,2329.11 L1730.85,2328.74 L1730.55,2328.4 L1730.27,2328.18 L1730.16,2328.19 L1730.15,2327.78 L1730.42,2326.14 L1730.67,2325.52 L1730.64,2325.04 L1729.95,2324.76 L1729.9,2324.38 L1730.1,2324.04 L1730.16,2323.88 L1730.06,2323.76 L1729.76,2323.51 L1730.29,2321.83 L1730.34,2321.08 L1730.02,2320.42 L1730.78,2318.57 L1730.9,2317.95 L1731.24,2319.16 L1731.66,2321.65 L1732.14,2322.65 L1732.52,2323.23 L1732.73,2323.36 L1733.15,2323.33 L1733.56,2323.36 L1733.64,2323.59 L1733.64,2323.88 L1733.79,2324.07 L1734.26,2324.21 L1734.74,2324.27 L1735.96,2324.15 L1736.93,2323.77 L1737.61,2323.02 L1737.93,2321.9 L1737.82,2320.38 L1737.35,2319.25 L1736.73,2318.21 L1736.42,2317.32 L1736.87,2316.61 L1736.21,2315.6 L1735.94,2314.77 L1736.05,2313.95 L1736.49,2313.03 L1737.1,2312.16 L1737.53,2312.11 L1738.1,2312.46 L1739.14,2312.76 L1739.21,2313.22 L1738.75,2313.95 L1738.89,2314.73 L1739.74,2316.56 L1739.93,2318.15 L1740.07,2318.61 L1741.41,2320.34 L1742.4,2320.77 L1746.73,2320.47"
         id="path1111" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1953.15,2219.21 L1952.65,2220.26 L1953.75,2222.86 L1952.86,2223.68 L1952.71,2224.09 L1952.86,2226.29 L1952.62,2227.03 L1951.81,2228.6 L1951.75,2229.41 L1951.99,2231.15 L1951.9,2231.95 L1951.52,2232.36 L1951.31,2232.83 L1951.08,2234.86 L1950.94,2235.51 L1950.79,2235.48 L1949.91,2235.69 L1949.59,2235.84 L1949.53,2236.11 L1949.53,2236.55 L1949.42,2237 L1948.99,2237.29 L1948.42,2240.52 L1948.94,2240.74 L1949.38,2240.87 L1949.8,2240.91 L1950.3,2240.89 L1950.28,2241.43 L1950.17,2241.9 L1949.95,2242.36 L1949.64,2242.83 L1949.9,2243.04 L1950.58,2243.91 L1947.84,2243.73 L1946.8,2243.94 L1945.73,2244.64 L1943.79,2246.57 L1943.16,2247.59 L1941.85,2251.31 L1940.81,2253.53 L1939.36,2255.61 L1937.32,2257.6 L1934.98,2256.68 L1934.63,2256.64 L1934.43,2256.26 L1933.15,2254.6 L1933.62,2253.56 L1934.17,2248.97 L1934.15,2247.4 L1933.72,2246.07 L1933.07,2245.47 L1932.28,2245.12 L1930.8,2244.04 L1930.66,2243.76 L1930.87,2243.21 L1930.75,2242.94 L1930.21,2242.69 L1929.9,2242.82 L1929.63,2242.89 L1929.19,2242.44 L1929.94,2241.93 L1931.58,2240.14 L1932.68,2239.45 L1933.04,2239 L1932.71,2238.56 L1933.07,2238.12 L1933.1,2237.76 L1933.02,2237.36 L1933.02,2236.79 L1933.19,2235.95 L1933.42,2235.31 L1933.67,2234.24 L1933.85,2233.94 L1935.23,2233.18 L1935.4,2232.9 L1935.66,2232.28 L1936.96,2230.19 L1937.19,2229.43 L1937.55,2228.86 L1940.29,2227.15 L1945.17,2222.87 L1945.79,2222.12 L1947.23,2220.95 L1950.5,2220.36 L1951.58,2218.12 L1951.99,2217.99 L1952.5,2218.26 L1953.07,2218.83 L1953.15,2219.21"
         id="path1113" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1568.95,2203.21 L1570.26,2203.75 L1571.77,2203.14 L1573.91,2203.66 L1575.86,2204.79 L1576.79,2206 L1577.82,2207.56 L1581.5,2209.24 L1582.21,2211.06 L1581.46,2210.35 L1579.94,2209.85 L1578.23,2209.6 L1576.88,2209.68 L1576.2,2210.11 L1575.81,2210.74 L1575.55,2211.31 L1575.24,2211.59 L1574.65,2212 L1574.75,2212.72 L1575.12,2213.39 L1575.34,2213.72 L1575.36,2214.54 L1575.19,2214.95 L1574.66,2215.14 L1573.63,2215.28 L1573.24,2215.05 L1571.49,2213.41 L1568.8,2211.77 L1567.48,2210.71 L1567.25,2209.73 L1566.06,2208.94 L1563.86,2206.67 L1562.64,2206.1 L1562.9,2205.34 L1562.55,2204.33 L1562.78,2203.91 L1562.32,2203.54 L1563.96,2200.42 L1565.08,2199.07 L1566.26,2199.18 L1566.32,2199.37 L1566.35,2199.52 L1566.6,2199.76 L1566.58,2199.89 L1567.84,2201.67 L1568.38,2202.82 L1568.95,2203.21"
         id="path1115" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1555.27,2166.34 L1558.86,2173.49 L1559.02,2173.98 L1559.06,2174.93 L1558.83,2175.47 L1558.42,2175.96 L1557.94,2176.7 L1558.4,2177.21 L1559.82,2179.24 L1560.47,2179.43 L1561.08,2178.77 L1561.66,2177.89 L1562.14,2177.4 L1562.06,2177.62 L1561.98,2178.07 L1561.92,2178.32 L1562.86,2178.39 L1563.23,2178.98 L1563.46,2179.81 L1563.91,2180.6 L1566.22,2182.17 L1566.96,2183.72 L1569.01,2185.88 L1569.49,2186.78 L1569.76,2187.76 L1569.74,2188.8 L1569.38,2189.91 L1569.12,2190.02 L1567.82,2189.89 L1567.38,2190.03 L1566.88,2190.42 L1566.4,2190.56 L1565.63,2190.43 L1564.92,2189.94 L1563.58,2188.75 L1562.06,2188.12 L1560.11,2187.85 L1558.48,2188.35 L1557.9,2190.01 L1556.9,2189.41 L1555.03,2189.13 L1554.25,2188.38 L1554.03,2187.85 L1553.8,2186.72 L1553.65,2186.28 L1552.67,2184.99 L1552.39,2184.35 L1553.02,2184.43 L1553.94,2185.13 L1554.56,2185.34 L1554.5,2184.88 L1553.63,2184.25 L1553.02,2183.39 L1551.29,2180.22 L1550.6,2179.38 L1550.08,2179.28 L1549.86,2180.39 L1550,2181.38 L1550.63,2183.36 L1550.72,2184.58 L1550.39,2185.73 L1549.81,2186.25 L1549.16,2186.66 L1548.65,2187.44 L1548.34,2187.48 L1547.92,2186.46 L1546.41,2185.84 L1545.96,2184.77 L1546.1,2184.24 L1546.41,2183.57 L1546.63,2182.9 L1546.49,2182.36 L1546.23,2181.87 L1546.29,2181.4 L1546.52,2181.03 L1546.81,2180.81 L1547.4,2179.95 L1547.81,2178.26 L1547.86,2176.21 L1547.71,2174.27 L1548.09,2175.11 L1548.2,2175.48 L1548.53,2175.44 L1549.05,2174.13 L1549.49,2174.52 L1550,2175.6 L1550.72,2176.42 L1551.06,2175.85 L1551.29,2176.18 L1551.54,2176.81 L1552,2177.1 L1552.47,2176.83 L1553.21,2175.82 L1554.27,2175.03 L1554.58,2173.94 L1554.47,2172.7 L1553.91,2171.66 L1554.96,2171.94 L1554.11,2169.96 L1553.98,2169.51 L1553.97,2168.69 L1554.05,2168.01 L1554.01,2167.44 L1553.66,2166.95 L1553.61,2166.57 L1555.27,2166.34"
         id="path1117" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1690.93,2118.07 L1691.43,2118.79 L1691.57,2119.45 L1692.31,2120.5 L1693.37,2121.44 L1695.65,2121.97 L1698.03,2123.39 L1699.34,2123.45 L1699.41,2123.91 L1699.36,2124.39 L1699.53,2124.61 L1699.82,2124.54 L1700.13,2124.17 L1700.83,2124.67 L1702.25,2124.69 L1702.98,2124.94 L1703.43,2124.52 L1703.99,2124.76 L1704.83,2123.98 L1705.82,2123.99 L1706.71,2124.62 L1707.2,2125.68 L1707.75,2126.34 L1708.83,2126.84 L1710.88,2127.38 L1713.15,2127.67 L1715.52,2127.43 L1716.6,2127.47 L1717.64,2127.66 L1718.54,2127.45 L1719.23,2126.31 L1720.35,2126.61 L1721.47,2126.41 L1722.55,2126.07 L1723.59,2125.92 L1726.84,2126.43 L1727.86,2126.81 L1727.39,2127.07 L1726.95,2127.54 L1726.62,2128.15 L1726.5,2128.83 L1726.65,2129.86 L1727.01,2130.39 L1731.95,2133.19 L1732.88,2134.11 L1731.83,2135.29 L1731.28,2137.46 L1731.45,2139.61 L1732.62,2140.71 L1732.69,2141.09 L1731.82,2141.56 L1732.06,2142.07 L1733.38,2143.16 L1733.61,2143.87 L1733.78,2144.83 L1734,2145.65 L1735.38,2146.86 L1735.59,2147.45 L1735.88,2147.72 L1736.15,2147.86 L1736.29,2148 L1736.44,2149.1 L1736.57,2149.64 L1736.9,2150 L1736.54,2150.89 L1736.46,2151.21 L1736.96,2151.95 L1737.7,2152.55 L1739.18,2153.28 L1741.02,2153.84 L1741.24,2154.17 L1741.48,2154.69 L1741.77,2155.15 L1742.14,2155.31 L1743.75,2154.67 L1744.44,2154.57 L1745.98,2155.33 L1747.96,2155.16 L1748.85,2155.32 L1749.7,2154.68 L1753.97,2153.03 L1754.08,2156.17 L1754.39,2157.7 L1755.33,2158.9 L1755.08,2159.84 L1755.02,2162.62 L1754.77,2163.72 L1754.16,2164.45 L1753.26,2165.2 L1752.24,2165.81 L1751.23,2166.17 L1750.66,2166.12 L1750.4,2165.8 L1750.18,2165.08 L1748.73,2163.45 L1748.52,2163.43 L1747.74,2162.97 L1746.54,2163.79 L1745.89,2164.88 L1746.74,2165.26 L1746.82,2165.72 L1746.19,2166.48 L1745.43,2166.53 L1744.63,2166.14 L1743.56,2165.31 L1743.26,2164.98 L1742.8,2164.29 L1742.4,2163.86 L1742.01,2163.55 L1741.96,2163.19 L1742.51,2162.62 L1741.68,2161.36 L1741.17,2161.03 L1740.53,2161.26 L1740.08,2160.32 L1739.15,2159.91 L1738.13,2160.01 L1737.36,2160.57 L1736.63,2160.3 L1736.38,2160.41 L1736.02,2160.83 L1735.98,2160.55 L1735.99,2160.42 L1736.08,2160.37 L1736.28,2160.31 L1736,2158.67 L1735.55,2158.4 L1734.77,2158.72 L1734.54,2158.28 L1734.67,2157.81 L1735.43,2156.73 L1735.57,2156.13 L1735.34,2155.37 L1734.49,2154.68 L1733.8,2153.18 L1733.09,2152.77 L1732.3,2152.73 L1731.62,2153.02 L1732.07,2153.81 L1730.93,2153.75 L1730.98,2152.58 L1731.61,2151.15 L1732.2,2150.33 L1731.59,2149.59 L1731.21,2149.69 L1730.91,2150.19 L1730.52,2150.65 L1730.46,2150.83 L1730.53,2151.11 L1730.51,2151.39 L1730.16,2151.57 L1729.71,2151.52 L1729.64,2151.26 L1729.76,2150.94 L1729.85,2150.77 L1729.77,2150.31 L1729.1,2150.44 L1729.75,2148.15 L1728.66,2147.77 L1727.54,2147.56 L1726.31,2147.55 L1724.87,2147.74 L1725.98,2147.15 L1726.42,2146.77 L1726.67,2146.11 L1726.2,2145.62 L1726.01,2145.1 L1725.99,2144.68 L1726.03,2144.54 L1725.36,2143.6 L1724.96,2143.24 L1724.86,2143.64 L1722,2143.95 L1718.75,2144.99 L1717.76,2145.04 L1716,2144.77 L1715.28,2144.77 L1714.36,2145.03 L1712.94,2145.65 L1712.21,2145.76 L1708.72,2145.54 L1708.01,2145.75 L1706.54,2146.46 L1705.83,2146.49 L1705.22,2145.98 L1703.6,2143.41 L1703.47,2143.68 L1703.33,2143.91 L1703.09,2144.04 L1702.68,2144.05 L1703.02,2143.04 L1703.77,2141.87 L1704.23,2140.77 L1703.72,2139.98 L1703.65,2139.52 L1704.01,2139.45 L1703.95,2139.07 L1702.61,2137.46 L1701.8,2136.76 L1700.95,2136.56 L1698.78,2136.61 L1697.69,2136.41 L1696.94,2135.99 L1696.7,2135.55 L1696.48,2134.48 L1696.25,2133.99 L1694.98,2133.29 L1694.75,2133.13 L1694.57,2132.95 L1694.21,2132.92 L1693.83,2132.72 L1693.56,2132.05 L1693.3,2131.63 L1692.78,2131.32 L1691.18,2130.67 L1689.88,2130.32 L1688.28,2129.08 L1687.78,2128.97 L1687.44,2128.75 L1686.79,2127.65 L1686.51,2127.43 L1685.3,2127.03 L1680.02,2124.32 L1678.22,2123.71 L1676.98,2123.54 L1676.35,2123.6 L1675.55,2123.79 L1674.12,2124.36 L1673.42,2124.42 L1672.7,2124.05 L1671.57,2122.98 L1671.17,2122.87 L1670.51,2122.94 L1669.69,2122.91 L1668.91,2122.71 L1668.2,2122.64 L1667.6,2122.97 L1667.67,2123.43 L1668.36,2123.33 L1669.04,2123.48 L1669.52,2123.95 L1669.59,2124.82 L1669.21,2125.24 L1668.47,2125.63 L1663.24,2127.03 L1662.23,2127.85 L1662.09,2127.17 L1662.05,2126.45 L1662.15,2125.86 L1662.41,2125.59 L1662.76,2125.33 L1663.76,2124.3 L1664.2,2124.03 L1665.27,2123.57 L1667.27,2122.01 L1668.67,2121.39 L1669.95,2120.41 L1671.04,2119.8 L1671.19,2119.58 L1671.27,2118.71 L1671.34,2118.46 L1671.86,2118.17 L1672.85,2117.96 L1673.23,2117.65 L1673.49,2117.09 L1673.57,2116.48 L1673.49,2115.93 L1673.28,2115.53 L1673.95,2115.17 L1674.7,2114.15 L1675.18,2113.87 L1678.15,2113.35 L1679.11,2112.82 L1681.39,2110.9 L1682.61,2110.45 L1683.14,2110.62 L1683.98,2111.32 L1684.29,2111.44 L1684.94,2111.41 L1685.69,2111.53 L1686.42,2111.8 L1686.99,2112.26 L1686.83,2113.49 L1687.47,2114.32 L1688.52,2114.74 L1689.57,2114.77 L1689.67,2115.39 L1690.93,2118.07"
         id="path1119" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1514.18,2089.27 L1514.5,2089.8 L1514.83,2089.76 L1514.78,2089.3 L1515.1,2089.26 L1515.25,2091.19 L1514.78,2092.38 L1513.86,2093.18 L1511.06,2094.79 L1510.63,2095.25 L1510.44,2096.01 L1510.69,2096.76 L1511.17,2097.32 L1511.73,2097.48 L1511.78,2097.94 L1511.12,2098.02 L1511.17,2098.41 L1514.17,2099.49 L1515.34,2099.57 L1515.31,2100.94 L1515.04,2101.27 L1514.18,2101.05 L1514.9,2102.26 L1516.66,2108.59 L1516.9,2108.92 L1517.49,2109.21 L1518.17,2109.38 L1519.55,2109.48 L1520.23,2109.67 L1521.33,2110.31 L1521.78,2110.38 L1522.63,2110.21 L1523.31,2109.85 L1523.7,2109.52 L1524.09,2109.57 L1524.69,2110.36 L1525,2110.99 L1525.65,2112.8 L1526.34,2113.62 L1526.4,2114.05 L1525.86,2114.52 L1525.39,2114.21 L1524.15,2114.02 L1522.93,2113.35 L1522.1,2113.12 L1521.23,2113.03 L1520.47,2113.06 L1520.01,2112.88 L1518.65,2111.95 L1518.12,2111.69 L1517.29,2111.69 L1516.62,2111.79 L1516.02,2111.71 L1515.77,2111.45 L1516.14,2111.47 L1516.44,2111.18 L1516.65,2110.57 L1514.91,2110.24 L1514.25,2110.03 L1514.24,2110.18 L1514.25,2110.42 L1514.26,2110.5 L1513.51,2109.64 L1512.84,2108.45 L1512.34,2107.21 L1511.71,2104.65 L1510.98,2103.46 L1509.94,2102.55 L1508.6,2101.78 L1507.42,2101.38 L1506.77,2101.02 L1505.14,2098.34 L1504.96,2097.93 L1504.12,2098.05 L1503.55,2098.04 L1503.1,2097.73 L1502.67,2096.94 L1502.63,2095.49 L1502.33,2095.07 L1501.5,2095.77 L1501.41,2094.79 L1501.16,2094.11 L1500.71,2093.78 L1499.98,2093.86 L1500.4,2093.29 L1501.44,2091.4 L1501.62,2090.81 L1502.06,2090.65 L1506.49,2089.9 L1507.87,2089.42 L1509.05,2088.73 L1510.24,2087.75 L1510.92,2087.88 L1511.41,2088.18 L1511.85,2088.52 L1512.36,2088.8 L1513.65,2089.04 L1514.18,2089.27"
         id="path1121" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1792.69,1918.94 L1793.71,1919.01 L1795.43,1918.46 L1796.61,1918.08 L1797.47,1918.16 L1798.94,1919.04 L1800.79,1919.1 L1801.67,1919.26 L1802.43,1919.29 L1802.48,1920.21 L1802.23,1921.07 L1805.23,1922.46 L1807.53,1923.07 L1808.08,1923.44 L1808.68,1921.94 L1808.77,1922.4 L1808.91,1922.84 L1809.1,1923.22 L1809.34,1923.53 L1809.29,1924.31 L1809.79,1925.85 L1809.63,1926.52 L1810.3,1927.29 L1810.74,1928.3 L1811.44,1930.52 L1812.8,1933.34 L1813.24,1934.52 L1813.52,1935.81 L1813.56,1937.14 L1813.36,1938.29 L1812.89,1939.01 L1812.05,1939.29 L1811.01,1939.25 L1810.03,1938.94 L1809.42,1938.44 L1807.6,1941.46 L1807.13,1941.65 L1806.97,1941.89 L1806.91,1942.19 L1806.71,1942.53 L1805.6,1943.65 L1804.61,1945.01 L1803.9,1945.6 L1801.04,1946.54 L1800.63,1947.14 L1801.56,1948.04 L1801.34,1949.49 L1801.37,1950.72 L1801.74,1951.77 L1802.5,1952.71 L1801.91,1954.25 L1802.01,1954.75 L1802.74,1954.8 L1803.13,1955 L1802.66,1956.76 L1802.96,1957.62 L1803.88,1958.85 L1804.41,1959.41 L1805.76,1959.74 L1805.94,1960.45 L1805.77,1961.24 L1805.44,1961.65 L1804.76,1961.88 L1804.41,1962.21 L1804.36,1962.63 L1804.6,1963.14 L1805.21,1963.66 L1805.71,1963.78 L1805.88,1963.96 L1805.54,1964.68 L1805,1965.33 L1803.8,1966.25 L1803.35,1966.95 L1803.54,1967.17 L1803.9,1967.54 L1804.02,1967.92 L1803.43,1968.24 L1803.03,1968.26 L1802.66,1968.14 L1802.37,1967.92 L1802.14,1967.61 L1801.86,1967.67 L1801.72,1968.97 L1801.56,1969.59 L1801.3,1970.01 L1800.98,1970.08 L1800.87,1969.84 L1800.53,1969.31 L1800.66,1969.95 L1800.68,1970.28 L1800.5,1970.46 L1800.11,1970.67 L1800.13,1970.97 L1800.22,1971.05 L1800.34,1971.04 L1800.51,1971.06 L1800.1,1971.38 L1800,1971.62 L1800.05,1972.03 L1800.51,1972.63 L1798.44,1975.33 L1797.95,1976.61 L1797.88,1978.32 L1797.09,1979.78 L1795.81,1980.79 L1794.02,1980.99 L1794.11,1980.71 L1793.42,1979.89 L1793.06,1978.66 L1792.55,1977.63 L1791.46,1977.37 L1791.38,1976.95 L1792.02,1976.82 L1792.84,1976.52 L1793.53,1976.13 L1793.8,1975.79 L1793.37,1975.23 L1792.65,1975.29 L1791.93,1975.65 L1791.51,1976.05 L1791.24,1975.67 L1791.04,1975.24 L1790.69,1975.31 L1790.78,1975.77 L1791.05,1976.04 L1791.3,1976.56 L1790.95,1976.64 L1790.07,1975.95 L1788.44,1975.36 L1786.82,1975.24 L1785.89,1975.95 L1785.07,1975.91 L1782.54,1976.36 L1781.94,1976.14 L1781.41,1976.17 L1778.17,1977.16 L1776.19,1977.17 L1774,1976.88 L1771.95,1976.92 L1770.34,1977.94 L1769.93,1977.92 L1768.91,1977.92 L1767.84,1977.59 L1767.04,1977.09 L1766.23,1976.78 L1763.81,1976.75 L1763,1976.86 L1761.15,1976.44 L1758.93,1975.46 L1756.9,1975.23 L1755.63,1977.08 L1755.31,1977.14 L1755.23,1976.84 L1751.48,1976.92 L1750.4,1976.57 L1749.72,1976.25 L1749.29,1976.26 L1749.08,1976.13 L1749.02,1975.35 L1749.12,1974.8 L1749.43,1974.33 L1749.85,1973.97 L1750.37,1973.77 L1749.4,1973.36 L1748.74,1973.19 L1744.98,1974 L1744.17,1974.38 L1744.38,1975.38 L1743.58,1976.47 L1743.07,1976.91 L1742.4,1977.13 L1742.95,1978.19 L1742.6,1978.76 L1741.8,1979.01 L1740.95,1979.16 L1740.37,1979.48 L1739.01,1980.64 L1738.3,1980.99 L1737.15,1982.2 L1736.11,1984.35 L1734.86,1985.95 L1733.04,1985.55 L1731.43,1986.67 L1730.53,1986.91 L1729.59,1986.67 L1729.78,1986.46 L1729.79,1986.4 L1729.86,1986.18 L1728.08,1985.62 L1727.25,1986.75 L1727.04,1987.17 L1726.72,1987.23 L1726.12,1986.33 L1725.48,1985.57 L1724.74,1984.91 L1724.81,1984.1 L1724.08,1983.91 L1723.5,1983.54 L1723.07,1982.94 L1722.81,1982.08 L1720.46,1982.13 L1719.83,1982.04 L1719.45,1982.47 L1718.93,1982.66 L1718.44,1982.93 L1718.15,1983.66 L1717.09,1983.43 L1715.84,1983.83 L1714.83,1984.6 L1714.16,1986.15 L1712.38,1988.26 L1711.75,1988.8 L1712.4,1989.16 L1712.85,1989.66 L1712.89,1990.14 L1711.94,1990.61 L1711.79,1990.86 L1711.84,1991.19 L1712.09,1991.57 L1712.39,1991.69 L1713.15,1991.48 L1713.58,1991.52 L1708.16,1996.24 L1703.91,1999.92 L1702.25,2000.65 L1700.21,2000.91 L1698.08,2000.75 L1692.34,1998.94 L1691.39,1998.89 L1690.79,1998.87 L1689.28,1999.33 L1686.88,2000.88 L1685.95,2001.75 L1685.11,2002.81 L1684.64,2003.92 L1684.66,2005.57 L1685.28,2006.83 L1686.27,2007.6 L1687.4,2007.74 L1688.17,2008.07 L1689.15,2009.03 L1690.03,2010.23 L1690.5,2011.3 L1690.82,2011.32 L1692.67,2012.43 L1693.33,2012.53 L1694.08,2012.46 L1694.78,2012.26 L1695.27,2011.94 L1695.68,2012.55 L1696.75,2013.24 L1696.88,2013.81 L1696.54,2014.14 L1693.36,2015.15 L1692.77,2016.01 L1692.78,2017.22 L1693.41,2018.78 L1695.86,2021.35 L1696.4,2021.72 L1698.02,2022.17 L1698.34,2022.42 L1700.21,2022.5 L1700.72,2022.6 L1701.22,2022.26 L1702.47,2021.84 L1702.84,2021.38 L1702.73,2020.81 L1702.32,2020.08 L1701.44,2019.01 L1701.72,2018.53 L1701.29,2018.17 L1702.23,2017.91 L1703.23,2018.8 L1704.79,2021.02 L1705.48,2021.61 L1706.6,2022.35 L1707.78,2022.67 L1708.62,2022.02 L1709.16,2022.21 L1710.67,2022.42 L1711.18,2022.4 L1711.57,2022.64 L1712.33,2023.92 L1712.68,2024.28 L1714.07,2024.05 L1715.92,2025.46 L1719.13,2028.72 L1722.61,2031.15 L1723.39,2032.69 L1722.07,2034.22 L1720.14,2035.07 L1719.23,2035.03 L1718.71,2034.21 L1718.23,2033.17 L1717.44,2032.66 L1716.52,2032.37 L1715.69,2031.97 L1714.12,2030.34 L1712.88,2028.58 L1711.48,2027.18 L1709.41,2026.64 L1705.5,2026.67 L1704.05,2026.35 L1702.72,2025.75 L1700.7,2024.31 L1699.59,2023.72 L1698.89,2023.59 L1697.16,2024.17 L1695.35,2024.03 L1694.66,2024.2 L1692.45,2025.84 L1691.42,2026.25 L1689.94,2025.9 L1689.86,2026.26 L1689.63,2026.85 L1689.44,2026.86 L1688.83,2028.17 L1688.72,2029.27 L1688.98,2030.26 L1690.05,2032.07 L1690.34,2032.39 L1690.75,2032.71 L1692.54,2033.48 L1693.02,2034.42 L1693.82,2035.08 L1694.56,2035.37 L1694.86,2035.21 L1698.34,2036.14 L1698.89,2035.81 L1699.73,2035.97 L1700.61,2036.44 L1701.33,2037.04 L1702.11,2037.41 L1704.1,2037.73 L1704.74,2038.37 L1704.77,2038.91 L1704.28,2039.61 L1704.38,2040.2 L1704.72,2040.51 L1705.67,2040.97 L1705.92,2041.41 L1705.86,2041.95 L1705.54,2042.11 L1705.21,2042.16 L1705.09,2042.43 L1705.33,2043.1 L1705.69,2043.33 L1706.1,2043.19 L1706.49,2042.79 L1706.59,2043.42 L1706.7,2043.84 L1706.85,2044.14 L1707.13,2044.41 L1706.05,2045.71 L1705.93,2046.3 L1706.53,2046.72 L1706.6,2047.14 L1705.68,2047.34 L1705.3,2047.8 L1705.5,2048.16 L1706.38,2048.04 L1706.46,2048.5 L1704.04,2049.03 L1703.04,2048.97 L1703.07,2048.23 L1703,2047.81 L1702.66,2047.87 L1702.34,2048.25 L1701.99,2048.22 L1701.78,2047.84 L1701.9,2047.15 L1700.05,2046.5 L1699.74,2046.45 L1699.41,2045.82 L1698.8,2045.46 L1698.13,2045.44 L1697.84,2045.78 L1697.84,2046.07 L1698.09,2046.56 L1695.87,2043.99 L1695.53,2043.34 L1695.66,2042.27 L1695.64,2041.64 L1695.38,2041.4 L1694.85,2041.29 L1694.49,2040.85 L1694.22,2040.31 L1693.94,2039.9 L1688.8,2035.04 L1687.03,2034.25 L1686.08,2034.16 L1682.14,2034.55 L1679.23,2033.5 L1676.98,2033.3 L1675.1,2033.73 L1673.69,2035 L1672.85,2037.28 L1672.93,2038.5 L1673.46,2039.47 L1674.1,2040.4 L1674.53,2041.49 L1674.73,2041.85 L1675.13,2041.97 L1676.1,2041.88 L1676.55,2041.93 L1676.88,2042.19 L1677.45,2042.9 L1680.57,2046.07 L1682.22,2047.34 L1687.59,2049.14 L1690.92,2049.32 L1691.1,2049.67 L1691.19,2050.21 L1691.44,2050.81 L1692.01,2051.46 L1692.66,2051.88 L1693.46,2052.06 L1694.47,2051.98 L1695.16,2052.06 L1694.88,2052.57 L1694.07,2053.18 L1693.22,2053.54 L1689,2053.64 L1681.61,2051.92 L1680.42,2051.94 L1676.98,2053.16 L1676.48,2053.51 L1676.62,2052.78 L1677.04,2051.51 L1677.06,2050.77 L1676.83,2050.29 L1675.98,2049.45 L1675.77,2049.1 L1675.15,2047.68 L1674.14,2046.76 L1673.12,2046.03 L1672.46,2045.16 L1672.79,2043.06 L1672.64,2040.67 L1672.01,2038.53 L1670.83,2037.17 L1668.75,2036.42 L1667.64,2036.21 L1665.83,2036.1 L1664.93,2035.75 L1664.11,2035.55 L1663.33,2035.9 L1660.58,2034.51 L1660.09,2034.53 L1659.52,2034.17 L1658.37,2034.1 L1657.19,2033.86 L1656.08,2032.44 L1653.59,2031.09 L1652.62,2030.83 L1651.78,2030.83 L1649.95,2031.29 L1650.24,2030.24 L1649.71,2029.3 L1648.03,2027.74 L1648.17,2027.66 L1648.3,2027.64 L1648.36,2027.57 L1648.3,2027.3 L1646.28,2025.63 L1645.69,2024.7 L1646.65,2024.09 L1649.64,2023.4 L1651.2,2022.8 L1651.81,2022.12 L1652.03,2021.52 L1652.53,2020.98 L1652.81,2020.28 L1652.34,2019.23 L1651.07,2018.7 L1650.76,2018.64 L1650.52,2018.48 L1650.54,2018.05 L1650.71,2017.31 L1649.72,2015.87 L1648.5,2015.22 L1647.18,2015.27 L1645.89,2015.99 L1646.41,2017.1 L1646.44,2018.06 L1646.17,2018.85 L1645.78,2019.49 L1645.09,2020.01 L1644.7,2020.4 L1644.76,2020.53 L1643.81,2020.69 L1643.38,2020.62 L1642.92,2020.38 L1642.09,2021.17 L1641.61,2022.04 L1641.6,2022.99 L1642.13,2024 L1641.84,2024.05 L1641.42,2023.69 L1641.56,2024.57 L1641.27,2024.62 L1641.2,2024.16 L1640.85,2024.22 L1640.77,2024.98 L1640.38,2025.01 L1639.83,2024.71 L1639.24,2024.5 L1639.17,2024.21 L1638.47,2024.37 L1637.83,2025.12 L1637.87,2025.88 L1638.85,2026.24 L1638.42,2026.37 L1638.14,2026.57 L1638.02,2026.92 L1638.06,2027.52 L1637.91,2027.88 L1637.47,2027.83 L1636.7,2027.51 L1635.58,2027.73 L1635.03,2028.07 L1634.58,2028.73 L1635.35,2029.23 L1635.97,2030.02 L1636.35,2030.96 L1636.39,2031.87 L1637.03,2032.3 L1637.6,2032.93 L1638.25,2033.44 L1639.16,2033.56 L1639.39,2034.18 L1638.97,2035.71 L1638.03,2038.1 L1637.25,2042.58 L1637.38,2043.28 L1636.16,2047.5 L1636.24,2048.92 L1636.76,2049.88 L1637.41,2050.83 L1637.87,2052.19 L1637.89,2053.63 L1638.1,2054.26 L1638.71,2054.43 L1638.97,2054.7 L1639.84,2056.36 L1642.64,2058.05 L1643.11,2058.21 L1643.41,2058.41 L1644.6,2058.6 L1644.9,2058.78 L1645.03,2058.98 L1645.47,2058.88 L1645.62,2059.11 L1645.73,2060.24 L1645.81,2060.48 L1645.84,2060.56 L1646.79,2062.67 L1647.27,2063.37 L1647.86,2063.94 L1649.9,2065.17 L1651.06,2065.57 L1651.84,2066.15 L1653.28,2067.52 L1653.87,2068.23 L1654.44,2069.22 L1654.92,2070.41 L1655.24,2071.7 L1655.61,2072.65 L1657.59,2075.9 L1659.28,2079.36 L1660.14,2080.4 L1660.88,2080.94 L1663.63,2082.15 L1666.22,2082.72 L1668.29,2083.82 L1668.45,2083.94 L1668.71,2084.17 L1669.59,2085.94 L1670.27,2086.61 L1675.08,2090.51 L1675.9,2091.67 L1678.53,2093.46 L1680.06,2096.6 L1683.21,2100.86 L1683.41,2101.64 L1682.96,2102.06 L1681.65,2103.74 L1680.76,2104.23 L1678.97,2104.83 L1678.1,2105.27 L1677.44,2105.72 L1677.72,2105.82 L1677.94,2105.93 L1678.16,2106.01 L1678.5,2106 L1678.56,2106.38 L1678.16,2106.59 L1677.84,2106.86 L1677.59,2107.15 L1677.41,2107.44 L1676.79,2107.35 L1675.73,2107.61 L1674.68,2108.1 L1674.04,2108.68 L1673.33,2109.19 L1672.19,2109.32 L1670.18,2109.14 L1670.5,2108.2 L1670.94,2107.32 L1670.55,2106.53 L1671.1,2105.86 L1672.58,2104.85 L1672.69,2105.64 L1672.45,2108.31 L1675.58,2105.85 L1675.98,2105.07 L1675.57,2104.56 L1675.81,2104.31 L1676.28,2104.26 L1676.55,2104.36 L1676.74,2104.6 L1677.03,2104.2 L1677.27,2103.46 L1677.33,2102.72 L1676.99,2102.09 L1675.41,2100.23 L1674.8,2099.71 L1674.84,2098.66 L1674.16,2097.66 L1673.15,2096.81 L1672.22,2096.21 L1671.21,2095.73 L1670.33,2095.58 L1666.55,2095.9 L1666.48,2095.69 L1666.5,2095.4 L1666.33,2095.12 L1666.04,2094.8 L1665.4,2093.89 L1665.09,2093.65 L1664.73,2093.65 L1663.45,2093.93 L1662.32,2093.84 L1662.08,2093.95 L1662.51,2094.49 L1662.57,2094.91 L1662.17,2095.49 L1662.1,2096.25 L1662.36,2096.96 L1662.96,2097.42 L1661.87,2097.9 L1660.01,2098.41 L1658.29,2099.15 L1657.65,2100.29 L1657.93,2102.68 L1658.14,2103.42 L1660.06,2107.01 L1660.36,2106.96 L1660.64,2106.24 L1661.16,2105.53 L1660.87,2104.9 L1661.11,2104.82 L1661.56,2105.16 L1661.91,2105.83 L1662.17,2105.35 L1662.76,2106.28 L1663.12,2107.45 L1663.59,2108.55 L1664.47,2109.26 L1664.41,2108.97 L1664.47,2108.89 L1664.58,2108.88 L1664.73,2108.79 L1664.47,2108.41 L1664.27,2108.01 L1665.6,2108.58 L1666.35,2109.53 L1666.96,2110.54 L1667.85,2111.31 L1667.68,2112.12 L1667.77,2112.27 L1668.05,2112.56 L1666.67,2113.72 L1666.39,2114.37 L1666.78,2115.32 L1669.03,2113.41 L1670.28,2112.76 L1671.73,2112.36 L1671.79,2112.74 L1671.46,2113.27 L1670.02,2114.38 L1669.65,2114.86 L1669.52,2115.34 L1669.36,2116.64 L1669.21,2117.12 L1668.95,2117.63 L1668.8,2117.85 L1668.62,2118.12 L1668.3,2118.51 L1668.03,2118.62 L1667.31,2118.62 L1666.94,2118.74 L1666.67,2119.09 L1666.43,2119.27 L1666.03,2119.37 L1666.35,2119.97 L1666.5,2120.19 L1665.82,2120.37 L1664.94,2121.13 L1664.48,2121.35 L1663.88,2121.33 L1663.2,2121.18 L1662.53,2121.13 L1661.91,2121.37 L1661.75,2121.85 L1661.71,2123.29 L1661.42,2123.64 L1660.22,2123.88 L1659.86,2124.19 L1659.73,2124.79 L1659.03,2124.88 L1657.96,2125.34 L1656.91,2125.62 L1656.37,2125.12 L1656.05,2124.67 L1655.57,2124.9 L1654.75,2125.63 L1654.31,2125.55 L1653.41,2125.13 L1653.14,2125.04 L1653.02,2124.89 L1651.15,2123.94 L1650.76,2124.09 L1649.97,2124.87 L1649.72,2125.21 L1649.37,2125.55 L1648.86,2125.76 L1648.39,2125.87 L1648.03,2126.05 L1647.73,2126.36 L1647.49,2126.85 L1647.28,2126.71 L1646.76,2126.5 L1647.01,2128.18 L1649.26,2127.41 L1649.32,2127.79 L1648.49,2128.56 L1649.06,2129.22 L1650.04,2129.32 L1650.44,2128.47 L1650.77,2128.41 L1650.57,2129.34 L1651.28,2129.19 L1651.89,2128.89 L1652.42,2128.72 L1652.88,2128.96 L1655.07,2127.34 L1655.66,2127.2 L1657.12,2128.91 L1658.34,2129.99 L1658.92,2130.21 L1659.52,2130.7 L1659.97,2130.82 L1660.36,2130.71 L1661.24,2130.21 L1661.72,2130.04 L1662.4,2130.13 L1663.26,2130.49 L1664.06,2131.05 L1664.52,2131.72 L1665.36,2130.57 L1665.83,2130.22 L1666.64,2130.07 L1667.19,2130.21 L1668.28,2130.89 L1669.03,2130.95 L1671.75,2130.52 L1673.06,2130.78 L1674.09,2131.8 L1676.35,2136.36 L1677.13,2136.9 L1678.07,2136.83 L1678.51,2136.58 L1679,2135.91 L1679.5,2135.54 L1680.12,2135.51 L1680.76,2135.3 L1681.3,2134.41 L1680.26,2133.95 L1679.82,2133.85 L1680.66,2133.29 L1681.57,2133.34 L1683.31,2134.06 L1684.14,2134.2 L1685.78,2134.14 L1686.75,2134.32 L1687.57,2134.77 L1688.2,2135.47 L1688.55,2136.29 L1688.55,2137.06 L1688.06,2137.76 L1687.4,2138.23 L1686.86,2138.94 L1686.69,2140.4 L1687.07,2139.6 L1687.64,2139.47 L1688.3,2139.76 L1688.88,2140.25 L1689.48,2140.59 L1690.83,2140.67 L1691.57,2140.87 L1691.63,2141.28 L1691.23,2141.52 L1690.12,2141.74 L1689.68,2142.02 L1689.35,2142.55 L1689.29,2142.97 L1689.61,2143.22 L1690.39,2143.19 L1693.66,2142.45 L1698.69,2142.41 L1699.22,2141.85 L1699.63,2140.92 L1700.91,2141.35 L1702.83,2142.72 L1701.94,2143.54 L1701.61,2143.77 L1701.68,2144.23 L1702.31,2144.26 L1702.85,2144.44 L1703.1,2144.78 L1702.88,2145.3 L1702.95,2145.72 L1704.58,2146.19 L1707.48,2149.3 L1709.48,2150.18 L1710.21,2150.09 L1712.91,2149.12 L1714.14,2149.6 L1714.7,2149.62 L1719.71,2150.74 L1720.72,2150.7 L1721.66,2150.84 L1722.55,2151.41 L1723.98,2152.72 L1726.6,2154.15 L1727.05,2154.51 L1727.32,2154.85 L1728.27,2155.05 L1728.56,2155.52 L1728.54,2156.03 L1728.47,2156.52 L1728.62,2156.96 L1729.23,2157.32 L1728.75,2157.61 L1728.54,2158.14 L1728.51,2158.83 L1728.56,2159.64 L1728.11,2158.84 L1727.38,2159.07 L1725.22,2160.81 L1725.47,2162.06 L1726.64,2164.55 L1727.03,2165.65 L1727.63,2166.42 L1728.21,2166.98 L1728.52,2167.45 L1728.48,2168.05 L1728.05,2169.48 L1727.91,2170.15 L1728,2170.67 L1728.23,2171.25 L1728.41,2171.91 L1728.33,2172.7 L1728.87,2172.57 L1729.47,2172.63 L1729.97,2172.95 L1730.22,2173.56 L1730.07,2173.92 L1728.67,2174.75 L1729.09,2175.09 L1729.58,2175.3 L1730.14,2175.35 L1730.84,2175.21 L1730.58,2175.73 L1730.86,2175.73 L1731.65,2175.92 L1731.44,2176.51 L1731.44,2176.96 L1731.62,2177.31 L1731.93,2177.59 L1731.7,2177.79 L1731.52,2177.98 L1731.43,2178.23 L1731.4,2178.59 L1732.69,2179.09 L1733.16,2179.6 L1733.46,2180.58 L1733.62,2181.91 L1733.52,2182.64 L1732.62,2184.01 L1733.35,2184.27 L1733.43,2184.72 L1732.76,2184.85 L1732.82,2185.23 L1733.14,2185.22 L1733.22,2185.4 L1733.22,2185.69 L1733.33,2186.03 L1733.24,2185.89 L1733.2,2185.71 L1733.13,2185.71 L1732.96,2186.1 L1733.69,2186.59 L1733.39,2187.34 L1732.67,2188.27 L1732.14,2189.27 L1731.52,2188.99 L1729.51,2188.57 L1728.72,2188.58 L1727.81,2188.44 L1727.35,2187.74 L1726.99,2185.45 L1726.66,2185.51 L1726.04,2186.01 L1725.39,2184.96 L1724.44,2182.47 L1724.28,2182.66 L1723.84,2183.05 L1723.32,2182.09 L1722.37,2181.22 L1721.38,2180.74 L1720.72,2181 L1720.47,2180.83 L1720.26,2180.74 L1720.08,2180.6 L1719.88,2180.29 L1719.58,2180.35 L1719.51,2181.69 L1719.01,2181.78 L1718.46,2181.66 L1718.18,2182.33 L1718.04,2181.93 L1717.95,2181.61 L1717.77,2181.48 L1717.34,2181.62 L1717.53,2180.81 L1717.2,2180.19 L1715.96,2179.24 L1715.64,2178.1 L1713.12,2175.49 L1712.81,2174.65 L1711.92,2174.41 L1711.12,2174.74 L1710.34,2175.27 L1709.5,2175.68 L1709.43,2175.38 L1709.43,2175.26 L1709.87,2174.96 L1709.99,2174.76 L1709.97,2174.34 L1709.28,2174.78 L1708.81,2174.5 L1708.34,2174.03 L1707.68,2173.85 L1706.97,2173.96 L1706.36,2173.97 L1705.82,2173.81 L1705.36,2173.41 L1706.86,2171.51 L1707.15,2170.52 L1706.62,2169.5 L1706.09,2169.13 L1705.76,2169.15 L1705.47,2169.37 L1705.08,2169.54 L1703.22,2169.88 L1702.74,2170.1 L1702.46,2170.44 L1702.26,2170.78 L1702.03,2170.95 L1700.88,2171.23 L1700.39,2171.46 L1699.92,2171.8 L1698.32,2173.79 L1697.57,2174.33 L1699.32,2174.49 L1698.37,2175.05 L1694.45,2175.35 L1689.47,2176.66 L1687.77,2177.89 L1686.49,2179.27 L1685.75,2180.05 L1683.57,2181.63 L1681.37,2181.12 L1680.95,2181.4 L1680.56,2181.44 L1680.12,2181.4 L1679.57,2181.43 L1679.44,2181.78 L1679.56,2182.48 L1679.59,2183.21 L1679.21,2183.6 L1678.99,2184.24 L1679.39,2185.4 L1680.17,2186.22 L1681.11,2185.85 L1682.03,2186.15 L1683.22,2186.1 L1684.36,2185.75 L1685.12,2185.16 L1686.94,2186.42 L1687.9,2186.91 L1688.87,2187.13 L1688.75,2187.77 L1688.76,2188.36 L1688.92,2188.86 L1689.2,2189.26 L1688.55,2189.5 L1687.74,2190.21 L1686.67,2190.58 L1686.55,2190.8 L1686.56,2191.2 L1686.5,2191.84 L1687.87,2192.55 L1688.58,2192.75 L1689.39,2192.67 L1689.15,2194.15 L1689.17,2195.11 L1689.49,2196.01 L1690.1,2197.28 L1689.43,2197.39 L1689.77,2198.7 L1690.05,2199.21 L1691.17,2199.44 L1693.38,2200.61 L1693.68,2200.09 L1695.78,2201.54 L1696,2201.58 L1696.26,2201.58 L1696.69,2201.72 L1697.81,2202.36 L1698.38,2202.58 L1699.08,2202.58 L1700.12,2201.63 L1699.81,2200.01 L1698.64,2198.57 L1697.15,2198.19 L1697.08,2197.77 L1697.81,2196.96 L1698.67,2196.42 L1701.14,2195.76 L1701.76,2196.41 L1701.79,2198.08 L1701.39,2199.91 L1700.73,2201 L1701.3,2201.43 L1701.89,2201.71 L1702.54,2201.84 L1703.28,2201.84 L1703.35,2202.3 L1702.45,2202.8 L1702.09,2202.91 L1702.16,2203.33 L1702.5,2203.27 L1702.95,2203.38 L1704.25,2203.39 L1705.47,2203.93 L1706.53,2204.08 L1709.12,2205.57 L1708.91,2205.9 L1708.85,2206.04 L1708.96,2206.16 L1709.25,2206.41 L1709.32,2206.83 L1708.4,2207.54 L1705.66,2209.03 L1705.43,2209.09 L1699.62,2210.41 L1698.75,2210.42 L1698.85,2210.63 L1698.92,2210.81 L1699.01,2211 L1699.19,2211.21 L1697.37,2211.28 L1696.72,2211.47 L1696.9,2212.08 L1695.27,2212.75 L1695.35,2213.35 L1695.75,2213.57 L1696.27,2213.53 L1696.75,2213.36 L1696.52,2213.86 L1696.57,2214.24 L1696.96,2214.23 L1697.96,2214.47 L1697.25,2214.97 L1696.36,2215.22 L1694.5,2215.46 L1693.51,2215.83 L1693.56,2216.35 L1694.13,2217.07 L1694.74,2218.01 L1693.75,2218.39 L1693.04,2218.5 L1692.39,2218.39 L1691.6,2218.12 L1691.54,2217.7 L1692.55,2217.53 L1692.48,2217.11 L1691.88,2217.05 L1691.21,2217.3 L1690.56,2217.34 L1690,2216.72 L1691.08,2216.96 L1690.75,2216.43 L1690.32,2216.07 L1689.13,2215.54 L1688.81,2216.01 L1688.32,2216.11 L1687.79,2215.9 L1687.77,2215.77 L1687.33,2213.86 L1687.34,2213.09 L1687.95,2213.4 L1688.55,2213.61 L1688.99,2213.07 L1689.22,2212.15 L1689.15,2211.24 L1690.33,2211.67 L1690.63,2211.84 L1690.57,2211.46 L1690.25,2210.94 L1690.11,2210.57 L1690,2210.02 L1689.82,2209.67 L1689.47,2209.58 L1689.09,2209.54 L1688.72,2209.36 L1687.97,2209.14 L1686.31,2209.88 L1685.47,2209.73 L1686.07,2209.19 L1685.21,2208.05 L1684.09,2208.98 L1683.67,2209.45 L1683.44,2210.08 L1682.42,2207.98 L1681.79,2207.27 L1680.84,2206.61 L1680.6,2206.59 L1680.04,2206.78 L1679.81,2206.79 L1679.59,2206.63 L1679.46,2206.37 L1679.39,2206.12 L1679.34,2206.01 L1678.58,2205.83 L1678.14,2205.82 L1677.6,2205.92 L1678.24,2205.99 L1678.47,2206.15 L1678.74,2206.58 L1677.69,2206.33 L1676.85,2206.58 L1676.15,2207.17 L1675.49,2207.96 L1675.09,2207.65 L1673.59,2206.99 L1673.66,2206.75 L1673.8,2206.1 L1673.13,2205.94 L1672.5,2205.58 L1672.12,2205.12 L1672.2,2204.65 L1671.57,2203.98 L1671,2203.56 L1669.99,2204.19 L1669.18,2204.88 L1668.72,2205.87 L1668.79,2207.41 L1668.94,2207.7 L1669.51,2208.36 L1669.67,2208.73 L1669.67,2209.13 L1669.54,2209.89 L1669.58,2210.25 L1669.89,2210.89 L1670.2,2211.21 L1670.51,2211.43 L1670.82,2211.79 L1670.92,2212.09 L1671.04,2213.26 L1671.18,2213.42 L1671.85,2213.81 L1671.92,2214.44 L1671.7,2215.31 L1671.82,2215.92 L1674.87,2218.54 L1675.36,2218.75 L1675.15,2219.69 L1675.83,2219.58 L1675.89,2219.96 L1675.55,2220.02 L1675.61,2220.43 L1676.21,2220.84 L1677.71,2222.75 L1678.23,2223.81 L1680.37,2226.53 L1680.57,2226.68 L1681.06,2227.21 L1681.41,2227.81 L1681.14,2228.16 L1680.95,2228.55 L1680.92,2229.38 L1681.05,2230.99 L1681.57,2231.69 L1682.53,2231.73 L1683.54,2231.43 L1684.18,2231.08 L1684.26,2231.36 L1684.37,2231.51 L1684.51,2231.65 L1684.68,2231.85 L1684.36,2232.13 L1684.43,2232.18 L1684.74,2232.27 L1684.44,2232.8 L1685.34,2232.95 L1685.82,2233.39 L1685.74,2233.71 L1684.93,2233.53 L1685.55,2234.54 L1687.18,2236.27 L1687.2,2237.01 L1687.76,2237.97 L1688.2,2239.02 L1687.94,2239.04 L1687.42,2239.17 L1687.16,2239.2 L1687.33,2239.48 L1687.81,2240.07 L1688.05,2240.3 L1688.12,2240.08 L1688.15,2240.03 L1688.19,2240.01 L1688.33,2239.86 L1688.58,2240.39 L1688.73,2240.91 L1688.75,2241.42 L1688.65,2241.95 L1689.27,2242.67 L1690.64,2244.97 L1691.08,2245.43 L1691,2245.94 L1691.83,2246.9 L1693.41,2248.04 L1693.83,2248.65 L1694.05,2249.32 L1693.95,2249.88 L1693.42,2250.18 L1693.48,2250.57 L1694,2250.5 L1694.4,2250.52 L1695.22,2250.73 L1695.08,2251.24 L1694.57,2251.77 L1694.43,2252.16 L1694.41,2252.52 L1694.44,2252.71 L1694.51,2252.89 L1694.59,2253.21 L1694.4,2253.92 L1693.76,2253.97 L1692.99,2253.68 L1692.44,2253.36 L1692.12,2254.12 L1691.59,2254.86 L1691.42,2255.52 L1692.18,2255.98 L1692.1,2256.2 L1692.03,2256.66 L1691.96,2256.87 L1693.32,2256.64 L1692.24,2257.94 L1692.54,2259.51 L1693.53,2261.01 L1694.75,2262.41 L1695.43,2263.4 L1695.67,2263.56 L1696.73,2263.8 L1697.36,2264.14 L1698.36,2264.4 L1698.61,2264.66 L1698.78,2264.94 L1699.02,2265.13 L1699.51,2265.1 L1699.1,2265.51 L1698.89,2265.64 L1698.95,2266.02 L1699.31,2266.31 L1699.39,2266.62 L1699.22,2266.96 L1698.8,2267.33 L1699.85,2268.42 L1701.49,2269.08 L1702.74,2269.88 L1702.58,2271.39 L1700.74,2270.85 L1700.34,2271.59 L1699.73,2271.79 L1697.41,2271.62 L1696.86,2271.06 L1696.38,2270.41 L1695.7,2270.01 L1695.64,2269.63 L1695.81,2268.75 L1695.31,2267.84 L1694.49,2267.14 L1693.66,2266.93 L1691.59,2267.77 L1690.57,2267.95 L1690.05,2267.35 L1689.66,2266.39 L1688.18,2264.21 L1687.71,2263.65 L1687.06,2263.21 L1684.67,2262.02 L1685.28,2261.17 L1684.94,2260.48 L1682.57,2258.95 L1682.29,2258.9 L1682.16,2259.41 L1681.59,2260.36 L1681.33,2260.07 L1681.18,2259.54 L1681.25,2258.06 L1681,2257.15 L1680.34,2255.9 L1679.66,2253.74 L1678.8,2252.77 L1677.62,2252.3 L1676.16,2252.27 L1673.61,2252.64 L1670.52,2253.53 L1668.45,2255.16 L1668.92,2257.76 L1667.45,2259.19 L1666.94,2260.17 L1666.72,2261.59 L1667.48,2261.46 L1668.02,2261.81 L1668.32,2262.52 L1668.4,2263.46 L1668.05,2263.52 L1667.9,2263.1 L1667.64,2262.69 L1667.29,2262.41 L1666.84,2262.43 L1666.47,2262.82 L1666.29,2263.44 L1666.3,2263.99 L1666.56,2264.19 L1667.48,2264.75 L1667.42,2266.05 L1666.84,2266.74 L1666.24,2265.49 L1665.57,2266.06 L1665.37,2266.74 L1665.87,2270.35 L1666.22,2271.72 L1666.69,2272.82 L1667.99,2275.1 L1667.83,2275.48 L1668.12,2275.98 L1667.52,2276.52 L1667.39,2277.2 L1667.64,2277.91 L1668.14,2278.55 L1668.2,2278.97 L1667.73,2278.99 L1667.63,2279.19 L1667.67,2279.51 L1667.63,2279.91 L1666.97,2279.32 L1666.65,2278.3 L1666.42,2277.18 L1666.03,2276.32 L1665.21,2275.83 L1663.36,2275.43 L1662.64,2274.69 L1662.3,2274.75 L1661.86,2275.36 L1661.35,2275.32 L1660.8,2274.92 L1660.33,2274.45 L1659.72,2273.61 L1659.67,2272.93 L1659.88,2272.21 L1660.05,2271.3 L1660.89,2271.4 L1661.02,2270.63 L1660.67,2269.65 L1660.06,2269.11 L1660.27,2268.77 L1660.6,2267.78 L1659.57,2267.95 L1659.55,2267.14 L1659.35,2266.21 L1659.41,2265.45 L1660.23,2265.23 L1660.17,2264.81 L1659.15,2264.63 L1659.11,2263.56 L1659.68,2261.42 L1658.25,2261.91 L1657.08,2260.57 L1656.02,2258.77 L1654.96,2257.87 L1654.86,2256.66 L1653.87,2255.21 L1649.7,2251.03 L1648.56,2250.65 L1647.44,2251.79 L1645.98,2251.02 L1645.36,2250.49 L1645.01,2249.61 L1645.13,2249.62 L1645.32,2249.24 L1645.48,2248.65 L1645.51,2248.03 L1645.46,2247.32 L1645.53,2246.09 L1645.37,2244.4 L1645.21,2243.98 L1641.75,2243.77 L1639.87,2243.96 L1639,2244.36 L1637.28,2245.81 L1635.48,2246.63 L1635,2247.35 L1634.83,2248.22 L1635.14,2254.16 L1634.99,2254.95 L1635.05,2255.51 L1635.35,2256.18 L1635.74,2256.67 L1635.95,2257.19 L1635.74,2257.97 L1636.7,2258.65 L1637.28,2258.91 L1637.99,2258.9 L1638.05,2259.32 L1637.39,2259.47 L1636.84,2259.82 L1636.58,2260.29 L1636.82,2260.76 L1636.58,2261.36 L1634.77,2263.59 L1634.53,2263.95 L1634.25,2264.2 L1633.82,2264.26 L1632.72,2262.65 L1632.54,2262.54 L1630.57,2259.63 L1629.68,2259.83 L1628.95,2260.13 L1627.76,2260.92 L1627.08,2260.4 L1625.84,2260.11 L1625.45,2259.57 L1625.14,2259.62 L1624.73,2259.95 L1624.39,2259.52 L1624.02,2258.77 L1623.52,2258.15 L1623.64,2257.54 L1623.53,2257.04 L1623.33,2256.46 L1623.17,2255.6 L1623.25,2255.01 L1623.77,2253.88 L1623.93,2253.3 L1623.81,2252.27 L1623.33,2251.43 L1622.54,2251.1 L1621.58,2251.56 L1622.02,2251.26 L1622.14,2251.04 L1622.15,2250.62 L1621.7,2250.66 L1621.47,2250.86 L1621.27,2251.61 L1620.63,2249.5 L1619.14,2248.08 L1617.44,2246.79 L1616.15,2245.04 L1615.38,2242.68 L1614.94,2240.23 L1614.98,2237.97 L1615.63,2236.13 L1618.41,2233.16 L1619.57,2231.52 L1619.97,2229.43 L1619.41,2226.98 L1618.76,2225.68 L1618.26,2224.69 L1614.04,2218.85 L1611.77,2216.32 L1610.69,2215.47 L1606.02,2213.12 L1604.93,2212.76 L1604.3,2212.29 L1603.18,2211.18 L1602.61,2210.8 L1599.59,2210.38 L1598.66,2210.79 L1598.62,2212.49 L1598.31,2212.54 L1598.14,2211.97 L1598.06,2211.35 L1597.89,2210.83 L1597.36,2210.51 L1597.8,2209.15 L1597.85,2208.12 L1597.57,2207.11 L1596.38,2204.44 L1595.76,2203.69 L1595.01,2203.28 L1592.7,2202.46 L1587.31,2201.66 L1586.82,2201.36 L1586.45,2200.65 L1586.53,2199.7 L1586.2,2199.27 L1587,2194.87 L1587.33,2194.82 L1588.65,2195.68 L1590.65,2194.35 L1592.56,2192.18 L1593.63,2190.46 L1594.35,2190.59 L1594.89,2190.13 L1595.35,2189.43 L1595.84,2188.85 L1595.78,2188.39 L1595.58,2188.29 L1595.52,2188.26 L1595.49,2188.2 L1595.36,2188.06 L1594.92,2188.34 L1594.48,2188.74 L1594.11,2189.16 L1593.85,2189.53 L1595.61,2185.84 L1595.65,2185.43 L1597.03,2182.42 L1597.16,2181.75 L1597.17,2180.93 L1596.77,2177.24 L1596.57,2176.24 L1597.17,2176.75 L1597.97,2177.77 L1598.53,2178.12 L1598.39,2177.39 L1598.19,2176.82 L1597.92,2176.4 L1598.23,2176.76 L1598.99,2177.02 L1599.87,2177.01 L1600.92,2176.87 L1601.65,2177.02 L1604.35,2178.49 L1608.35,2178.77 L1609.27,2178.45 L1610.19,2177.85 L1611.63,2176.52 L1612.71,2174.75 L1614.12,2170.48 L1615.01,2168.7 L1618.41,2165.79 L1618.78,2165.25 L1622.1,2165.41 L1623.15,2165.24 L1623.6,2165.06 L1624.02,2164.72 L1624.55,2165.06 L1625.21,2165.35 L1625.96,2165.54 L1626.73,2165.53 L1627.37,2165.66 L1627.76,2166.16 L1628.06,2166.83 L1628.5,2167.44 L1629.1,2167.8 L1630.66,2168.27 L1631.2,2168.29 L1631.58,2168.16 L1632.09,2167.79 L1632.33,2167.73 L1632.8,2168.09 L1633.18,2168.31 L1633.6,2168.38 L1633.94,2168.64 L1634.7,2170.13 L1635.14,2170.71 L1635.58,2170.86 L1637.44,2171.16 L1637.62,2171.27 L1637.84,2171.54 L1638.16,2171.79 L1638.57,2171.87 L1640.09,2171.62 L1642.28,2171.8 L1642.91,2172.02 L1643.75,2171.47 L1644.59,2171.84 L1645.52,2172.48 L1647.11,2172.84 L1647.66,2173.16 L1648.21,2173.32 L1648.7,2172.99 L1649.04,2172.85 L1653.02,2173.07 L1658.78,2174.99 L1661.86,2175.54 L1662.34,2175.92 L1662.83,2176 L1667.26,2178.76 L1668.37,2180.1 L1668.68,2180.24 L1669.66,2180.36 L1671.38,2181.43 L1672.18,2181.8 L1672.96,2181.79 L1673.7,2181.52 L1674.82,2180.91 L1675.2,2180.83 L1675.61,2180.81 L1676,2180.72 L1676.29,2180.47 L1677.05,2178.81 L1677.11,2178.47 L1676.52,2177.83 L1675.69,2177.23 L1675.08,2177 L1674.28,2176.89 L1672.92,2176.11 L1672.02,2176.2 L1672.15,2175.96 L1672.3,2175.85 L1670.64,2176.04 L1671.56,2174.99 L1672,2174.6 L1672.62,2174.37 L1673.23,2174.02 L1674.14,2172.72 L1674.54,2172.33 L1675.87,2172.19 L1678.43,2172.8 L1679.37,2172.35 L1680.92,2172.78 L1682.46,2172.29 L1685.82,2170.63 L1686.5,2169.57 L1687.3,2169.21 L1687.73,2168.79 L1687.3,2168.11 L1686.4,2167.53 L1685.45,2167.43 L1685.39,2167.12 L1685.32,2166.59 L1685.93,2166.41 L1686.98,2165.94 L1687.59,2165.76 L1687.52,2165.3 L1687.02,2165.32 L1683.57,2165.48 L1682.76,2165.31 L1683.63,2164.3 L1682.14,2163.16 L1681.48,2163.15 L1680.14,2164.91 L1679.64,2165.17 L1677.93,2165.29 L1677.87,2164.87 L1678.83,2164.7 L1678.04,2163.86 L1677.29,2163.46 L1676.55,2163.38 L1673.64,2163.55 L1672.79,2163.9 L1672.37,2164.92 L1673,2164.58 L1673.72,2164.51 L1674.39,2164.76 L1674.81,2165.39 L1673.97,2165.52 L1672.47,2165.5 L1671.84,2165.9 L1670.72,2163.05 L1670.38,2163.1 L1670.16,2164.04 L1670.05,2163.76 L1669.86,2163.5 L1669.73,2163.22 L1668.88,2163.66 L1667.13,2164.17 L1666.49,2164.66 L1664.3,2163.71 L1664.61,2163.01 L1665.38,2162.11 L1665.64,2161.33 L1664.68,2161.09 L1662.75,2160.88 L1661.51,2160.26 L1660.67,2160.16 L1660.33,2160.04 L1660.03,2159.76 L1659.05,2158.57 L1659.33,2157.42 L1658.45,2156.28 L1657.22,2155.96 L1656.43,2157.25 L1657.33,2157.22 L1657.46,2157.69 L1657.05,2158.11 L1656.38,2157.92 L1655.79,2157.84 L1655.31,2158.44 L1654.97,2159.22 L1654.82,2159.71 L1654.87,2160.09 L1655.14,2160.2 L1655.31,2160.29 L1655.51,2160.36 L1655.88,2160.4 L1655.94,2160.78 L1655.62,2161.08 L1655.59,2161.22 L1655.69,2161.36 L1655.77,2161.7 L1655.29,2162.38 L1654.72,2162.66 L1654.01,2162.57 L1653.11,2162.14 L1653.06,2161.97 L1652.35,2161.01 L1652.25,2161 L1652.05,2160.67 L1651.89,2160.51 L1651.97,2160.36 L1652.46,2160.1 L1651.59,2158.86 L1648.83,2156.37 L1648.26,2155.49 L1647.95,2155.23 L1647.73,2155.15 L1647.19,2155.08 L1644.71,2154.04 L1644.87,2154.67 L1645.25,2155.73 L1645.38,2156.31 L1644.21,2157.56 L1643.91,2158.03 L1644.34,2158.47 L1644.61,2158.66 L1644.81,2158.61 L1645,2158.29 L1645.27,2158.83 L1645.22,2159.31 L1644.96,2159.74 L1644.58,2160.08 L1644.97,2160.44 L1644.77,2161 L1644.06,2161.17 L1643.08,2160.98 L1641.78,2160.96 L1639.98,2161.49 L1639.66,2161.18 L1638.98,2160.95 L1638.28,2160.91 L1637.86,2161.17 L1637.55,2161.22 L1636.94,2161.18 L1636.24,2161.74 L1635.71,2162.67 L1635.53,2163.74 L1635.06,2163.38 L1634.67,2162.6 L1634.28,2162.18 L1633.8,2162.07 L1633.39,2162.21 L1633.13,2162.18 L1633.14,2161.54 L1632.32,2161.13 L1630.7,2160.91 L1628.02,2160.2 L1627.06,2160.3 L1626.24,2160.89 L1624.92,2159.93 L1623.46,2160.09 L1622.13,2160.85 L1620.35,2162.59 L1619.92,2162.77 L1619.18,2162.9 L1618.76,2162.77 L1618.46,2162.37 L1618.15,2161.96 L1617.85,2161.78 L1617.41,2161.99 L1615.31,2164.39 L1615.03,2165.13 L1615.05,2166.32 L1614.65,2166.71 L1613.61,2166.28 L1612.14,2165.25 L1609.29,2165.69 L1608.52,2166.02 L1607.86,2166.52 L1607.23,2166.82 L1606.49,2166.55 L1603.67,2169.31 L1603.26,2170.51 L1602.96,2170.55 L1602.45,2170.23 L1601.74,2170.19 L1601.11,2170.25 L1600.84,2170.23 L1600.44,2169.86 L1598.3,2169.08 L1599.1,2168.74 L1599.93,2169.06 L1600.75,2169.6 L1601.48,2169.92 L1601.41,2169.47 L1601.15,2169.23 L1601,2169.04 L1601.02,2168.76 L1601.25,2168.24 L1600.74,2167.98 L1600.52,2167.93 L1600.46,2167.47 L1600.64,2167.35 L1601.07,2166.95 L1600.1,2167.09 L1600.29,2166.21 L1599.87,2166.5 L1599.47,2166.65 L1599.07,2166.65 L1598.65,2166.45 L1598.4,2166.73 L1598.21,2166.86 L1597.71,2167.06 L1597.86,2165.81 L1597.18,2165.26 L1596.15,2165.16 L1595.24,2165.25 L1594.79,2164.73 L1593.82,2161.14 L1592.63,2160.08 L1592.01,2159.73 L1591.75,2159.66 L1591.55,2159.75 L1591.09,2159.88 L1591.32,2160.33 L1593.78,2163.33 L1593.08,2163.44 L1593.22,2163.65 L1593.27,2163.68 L1593.33,2163.69 L1593.5,2163.8 L1593.41,2164.47 L1593.38,2165.17 L1593.27,2165.75 L1592.95,2166.02 L1592.46,2166.21 L1591.5,2166.91 L1591.2,2167.19 L1591.35,2167.41 L1591.41,2167.42 L1591.62,2167.51 L1591.31,2168.02 L1590.6,2167.7 L1590.22,2167.95 L1590.12,2168.37 L1590.21,2168.58 L1590.47,2168.75 L1590.17,2169.27 L1589.54,2170 L1589.05,2170.43 L1588.73,2170.58 L1587.75,2170.66 L1587.33,2170.49 L1586.91,2170.18 L1586.53,2170.16 L1586.22,2170.89 L1586.5,2171.09 L1586.82,2171.2 L1587.19,2171.21 L1586.96,2171.31 L1586.39,2171.57 L1586.05,2171.99 L1586.11,2172.65 L1585.18,2172.71 L1584.75,2172.14 L1584.51,2171.36 L1584.16,2170.8 L1583.55,2170.67 L1582.05,2170.86 L1581.41,2170.77 L1581.95,2170.42 L1582.34,2170.05 L1582.46,2169.56 L1582.21,2168.9 L1581.77,2169.26 L1581.6,2169.46 L1581.91,2168.43 L1582.1,2168.06 L1582.01,2168.52 L1582.55,2168.7 L1583.06,2168.25 L1583.41,2167.65 L1583.66,2166.97 L1582.69,2166.4 L1582.32,2166.5 L1581.99,2167.22 L1581.39,2167.05 L1580.94,2166.65 L1580.62,2166.05 L1580.38,2165.32 L1580.51,2165.17 L1580.56,2165.16 L1580.59,2165.1 L1580.63,2164.82 L1581.2,2165.37 L1581.8,2165.58 L1582.24,2165.35 L1582.32,2164.57 L1581.71,2165.12 L1581.83,2164.28 L1581.22,2163.2 L1581.38,2162.6 L1581.32,2162.14 L1580.85,2162.21 L1580.52,2162.15 L1579.94,2161.96 L1580.35,2161.62 L1580.56,2161.49 L1580.88,2161.39 L1580.67,2161.29 L1580.36,2161.09 L1580.13,2161.03 L1580.46,2159.98 L1580.53,2159.46 L1580.41,2159.1 L1579.92,2158.89 L1579.57,2159.19 L1579.24,2159.66 L1578.82,2159.93 L1578.67,2160.2 L1578.2,2160.72 L1577.68,2161.07 L1577.38,2160.83 L1577.2,2160.28 L1576.63,2159.46 L1576.44,2158.81 L1576.48,2156.24 L1576.31,2154.9 L1575.14,2153.94 L1574.08,2151.74 L1573.57,2151.27 L1571.67,2151.53 L1570.99,2151.39 L1570.32,2150.89 L1569.88,2150.18 L1568.05,2145.37 L1567.45,2144.46 L1566.73,2144.04 L1566.05,2144.39 L1565.43,2145.22 L1564.96,2146.18 L1564.69,2146.91 L1564.25,2146.29 L1563.98,2146.44 L1563.71,2146.8 L1563.32,2146.75 L1562.85,2147.27 L1562.63,2147.05 L1562.47,2146.5 L1562.22,2146.06 L1560.38,2144.95 L1560.33,2144.57 L1560.6,2143.72 L1560.68,2142.56 L1560.93,2141.5 L1561.75,2140.94 L1562.8,2140.83 L1563.35,2140.7 L1563.56,2140.47 L1563.35,2140.1 L1562.93,2139.8 L1562.5,2139.61 L1561.39,2139.27 L1560.86,2138.4 L1560.62,2137.3 L1560.67,2136.34 L1561.12,2135.09 L1561.43,2135.51 L1561.88,2136.49 L1562.73,2136.91 L1563.71,2137.08 L1564.56,2136.49 L1565.12,2135.48 L1565.47,2134.37 L1565.8,2134.33 L1566.02,2134.79 L1566.12,2135.11 L1566.11,2135.35 L1565.97,2135.58 L1566.07,2135.68 L1566.23,2135.78 L1566.29,2135.99 L1566.08,2136.43 L1567.06,2136.3 L1567.86,2136.07 L1568.38,2135.49 L1568.47,2134.33 L1568.8,2134.28 L1569.35,2134.85 L1569.85,2134.8 L1570.41,2134.53 L1571.18,2134.41 L1571.86,2134.56 L1572.36,2134.79 L1572.94,2134.94 L1573.82,2134.92 L1574,2135.29 L1573.95,2136.18 L1574.01,2137.06 L1574.47,2137.39 L1575.15,2137.46 L1575.85,2137.78 L1577.18,2138.71 L1577.17,2138.15 L1577.24,2137.81 L1577.42,2137.58 L1577.7,2137.31 L1577.64,2136.93 L1577.16,2136.19 L1577.63,2135.73 L1578.53,2135.63 L1579.38,2135.99 L1580.7,2137.17 L1580.74,2137.33 L1581.06,2137.46 L1581.35,2137.52 L1581.56,2137.33 L1581.67,2136.72 L1581.22,2136.43 L1581.02,2136.17 L1580.76,2135.26 L1580.58,2135.23 L1580.37,2135.32 L1580.14,2135.19 L1579.38,2134.21 L1579.22,2133.58 L1579.45,2132.73 L1579.13,2132.27 L1579.21,2131.69 L1579.53,2131.31 L1579.95,2131.41 L1579.71,2129.96 L1578.32,2128.56 L1575.56,2126.48 L1575.27,2126.92 L1575.14,2127.35 L1575.2,2127.76 L1575.47,2128.17 L1574.78,2128 L1574.55,2128.13 L1574.31,2128.26 L1573.58,2129.34 L1574.7,2129.99 L1571.98,2130 L1571.92,2129.58 L1571.31,2129.53 L1570.69,2129.21 L1570.19,2128.88 L1569.96,2128.78 L1569.43,2129.18 L1568.93,2128.98 L1568.53,2128.46 L1568.33,2127.92 L1568.43,2127.91 L1568.54,2127.62 L1568.94,2127.44 L1568.89,2127.02 L1568.53,2127.07 L1568.1,2127.45 L1567.56,2127.54 L1567.01,2127.36 L1566.55,2126.89 L1566.18,2126.95 L1565.99,2127.6 L1565.19,2128.21 L1565.06,2128.63 L1565.35,2129.21 L1565.9,2129.25 L1566.15,2129.51 L1564.14,2129.77 L1564.82,2129.32 L1564.68,2128.92 L1564.58,2128.33 L1564.4,2127.94 L1563.99,2128.16 L1563.97,2127.92 L1563.91,2127.78 L1563.86,2127.68 L1563.93,2127.59 L1563.88,2127.28 L1563.45,2127.27 L1563.21,2127.39 L1563.08,2127.61 L1562.94,2127.88 L1563.37,2126.44 L1563.04,2125.44 L1562.43,2125.15 L1561.97,2125.84 L1561.62,2125.46 L1561.49,2125.93 L1561.25,2126.31 L1560.89,2126.63 L1560.43,2126.91 L1560.49,2127.37 L1561.22,2127.43 L1561.09,2127.98 L1561.17,2128.68 L1561.09,2129.43 L1560.79,2129.93 L1560.44,2130.25 L1560.13,2130.63 L1560.02,2131.3 L1559.27,2130.93 L1559.38,2131.56 L1559.49,2131.83 L1559.69,2131.85 L1560.07,2131.72 L1562.56,2133.24 L1563.57,2133.56 L1563.92,2133.95 L1564.09,2134.37 L1563.96,2134.59 L1563.53,2134.55 L1562.59,2134.15 L1561.1,2133.89 L1560.8,2133.97 L1560.4,2134.24 L1560.41,2134.52 L1560.01,2135.31 L1559.5,2135.88 L1559.17,2135.47 L1558.95,2134.91 L1558.54,2134.54 L1558.16,2134.25 L1557.97,2133.94 L1557.86,2133.58 L1557.73,2133.4 L1557.6,2133.28 L1557.53,2133.13 L1557.58,2132.71 L1557.78,2132.43 L1557.99,2132.22 L1558.06,2132.01 L1557.3,2130.01 L1556.18,2128.67 L1554.76,2127.77 L1553.08,2127.1 L1551.39,2126.12 L1547.02,2121.91 L1545.47,2121.13 L1544.94,2120.67 L1545.05,2120.08 L1544.73,2119.55 L1544.53,2118.85 L1544.51,2118.13 L1544.73,2117.56 L1544.67,2117.1 L1544.12,2117.44 L1543.64,2117.58 L1543.26,2117.43 L1542.97,2116.91 L1542.3,2117.41 L1541.61,2117.48 L1540.92,2117.23 L1540.27,2116.81 L1538.99,2117.57 L1536.9,2116.84 L1535.13,2115.23 L1534.79,2113.31 L1534.06,2112.78 L1533.6,2112.05 L1533.01,2111.51 L1531.89,2111.53 L1531.83,2111.11 L1532.16,2111.07 L1532.12,2110.68 L1531.66,2110.25 L1531.01,2109.36 L1530.59,2109.14 L1531.39,2108.76 L1532.39,2108.65 L1533.03,2108.39 L1532.76,2107.56 L1532.24,2107.13 L1530.22,2106.15 L1530.18,2105.77 L1531.02,2105.7 L1531.64,2105.78 L1532.1,2105.67 L1532.45,2105.03 L1532.4,2104.61 L1531.86,2104.33 L1530.96,2104.04 L1530.29,2104.1 L1530.4,2104.88 L1529.93,2104.55 L1529.34,2104.29 L1528.78,2104.28 L1529.26,2103.71 L1528.32,2103.62 L1527.48,2103.64 L1526.71,2103.81 L1525.96,2104.16 L1526.28,2103.61 L1526.41,2103.01 L1526.41,2102.33 L1526.29,2101.54 L1526.69,2101.91 L1527.02,2101.84 L1527.31,2101.48 L1527.58,2100.98 L1527.58,2100.77 L1527.44,2099.98 L1527.42,2099.68 L1527.51,2099.63 L1527.93,2099.24 L1528.03,2099.21 L1527.57,2098 L1526.7,2097.54 L1525.86,2097.41 L1525.45,2097.16 L1525.25,2096.92 L1524.42,2096.39 L1524.16,2096.3 L1523.6,2096.24 L1523.16,2095.99 L1522.74,2095.6 L1520.55,2095.4 L1519.4,2095.13 L1518.8,2094.84 L1518.53,2094.92 L1518.31,2094.91 L1519.03,2094.32 L1520.7,2094.64 L1522.69,2094.99 L1525.03,2095.85 L1526.55,2096.89 L1527.59,2097.21 L1528.32,2096.92 L1528.45,2096.57 L1528.38,2096.12 L1528.37,2095.64 L1528.7,2095.17 L1529.07,2094.99 L1529.48,2094.94 L1529.91,2095.01 L1530.37,2095.21 L1530.34,2093.84 L1530.98,2093.04 L1531.81,2092.41 L1532.37,2091.52 L1532.43,2090.83 L1532.25,2090.52 L1531.96,2090.28 L1531.7,2089.77 L1531.6,2089.6 L1531.19,2089.21 L1531.05,2089.02 L1531.08,2088.82 L1531.29,2088.04 L1531.32,2087.71 L1531.1,2087.25 L1530.79,2087.03 L1530.64,2086.79 L1530.92,2086.28 L1531.3,2086.06 L1531.78,2086.07 L1531.85,2086.09 L1533.72,2086.53 L1534.63,2086.92 L1535.46,2086.9 L1536.18,2085.93 L1536.42,2084.64 L1536.09,2083.6 L1533.43,2080.27 L1532.15,2079.31 L1531.86,2078.97 L1530.67,2075.33 L1530.37,2074.88 L1530.93,2074.62 L1533.3,2074.22 L1534.07,2073.94 L1534.14,2072.81 L1534.4,2071.65 L1534.82,2070.6 L1535.36,2069.78 L1536.22,2069.25 L1537.82,2069.21 L1538.52,2068.68 L1539.36,2068.33 L1541.93,2068.35 L1543.11,2068.07 L1544.79,2066.97 L1545.9,2066.24 L1546.22,2065.95 L1546.37,2065.49 L1546.82,2063.38 L1546.86,2063.09 L1546.48,2061.98 L1546.43,2061.67 L1546.18,2061.01 L1546.16,2060.63 L1546.33,2060.26 L1547.01,2059.94 L1547.24,2059.7 L1547.58,2058.15 L1547.43,2056.88 L1547.17,2055.66 L1547.2,2054.23 L1547.5,2053.57 L1547.97,2053.14 L1548.45,2052.8 L1548.78,2052.41 L1548.88,2052.05 L1549.38,2050.13 L1549.77,2049.09 L1549.91,2048.55 L1549.98,2047.76 L1549.87,2047.12 L1549.44,2045.47 L1549.44,2044.88 L1549.81,2044.39 L1550.76,2044.01 L1551.15,2043.72 L1551.46,2043.03 L1551.57,2042.41 L1551.75,2041.86 L1552.33,2041.35 L1553.59,2041.14 L1554.74,2041.49 L1555.83,2041.6 L1556.91,2040.63 L1557.28,2039.92 L1557.39,2039.59 L1557.48,2039.21 L1557.35,2038.45 L1557.56,2037.86 L1558.01,2037.44 L1558.6,2037.19 L1558.84,2036.51 L1559.23,2036.1 L1559.63,2035.78 L1559.91,2035.39 L1560.07,2034.73 L1560.08,2034.24 L1560.03,2033.72 L1560.02,2033 L1560.23,2030.78 L1560.25,2029.62 L1560.02,2028.42 L1559.57,2027.39 L1558.97,2026.59 L1557.51,2025.26 L1556,2024.18 L1555.73,2023.75 L1555.57,2023.27 L1555.33,2022.83 L1554.82,2022.52 L1555.51,2021.44 L1555.62,2020.09 L1555.14,2017.18 L1562.02,2015.88 L1565.22,2014.39 L1566.45,2014.27 L1568.23,2014.37 L1568.96,2014.41 L1570.58,2014.17 L1572.13,2013.59 L1572.8,2013.06 L1574.28,2011.11 L1575.32,2010.44 L1576.47,2010.25 L1580.12,2010.2 L1580.3,2010.19 L1582.64,2011.73 L1583.99,2011.78 L1584.36,2011.45 L1585.3,2010.19 L1585.72,2009.74 L1585.99,2009.66 L1586.67,2009.59 L1586.89,2009.47 L1587.02,2009.1 L1587.01,2008.29 L1587.07,2008.03 L1588.31,2007.22 L1590.82,2007.39 L1591.73,2007.03 L1592.12,2006.88 L1592.62,2006.18 L1592.69,2005.46 L1592.6,2004.79 L1592.61,2004.27 L1592.99,2003.61 L1594.58,2002.11 L1594.58,2002.1 L1594.58,2002.1 L1594.59,2002.09 L1595.06,2000.84 L1596.49,1998.97 L1596.99,1997.92 L1597.01,1997.44 L1596.99,1997.24 L1596.88,1996.21 L1596.98,1995.59 L1597.29,1995.12 L1597.98,1994.43 L1598.36,1994.04 L1599.67,1993.05 L1602.53,1991.69 L1602.73,1991.49 L1603.13,1990.88 L1603.3,1990.73 L1603.66,1990.79 L1603.85,1991.07 L1603.98,1991.37 L1604.12,1991.52 L1605.79,1992.11 L1606.16,1992.18 L1607.09,1991.67 L1608.54,1989.88 L1609.54,1989.21 L1610.58,1988.95 L1610.95,1988.93 L1611.5,1988.91 L1613.35,1989.18 L1614.8,1989.13 L1615.25,1989.18 L1615.41,1989.4 L1615.6,1990.2 L1615.73,1990.4 L1616.02,1990.39 L1616.38,1990.07 L1617.26,1989.9 L1618.41,1989.49 L1619,1989.44 L1619.45,1989.57 L1620.54,1990.05 L1621.1,1990.16 L1622.5,1989.99 L1623.23,1989.79 L1623.86,1989.51 L1624.72,1989.09 L1625.46,1988.96 L1627.02,1988.93 L1628.04,1988.7 L1628.71,1988.3 L1628.87,1988.2 L1629.56,1987.4 L1630.21,1986.24 L1630.23,1985.72 L1630.11,1985.17 L1630.2,1984.7 L1630.88,1984.38 L1631.44,1984.43 L1631.5,1984.48 L1631.76,1984.74 L1632.05,1985.18 L1632.48,1985.6 L1633.43,1986.2 L1634.08,1986.4 L1634.53,1985.95 L1634.87,1984.62 L1634.94,1982.16 L1634.37,1977.26 L1634.39,1977.04 L1634.55,1975.23 L1634.99,1974.69 L1635.74,1973.78 L1636.46,1973.52 L1637.78,1973.04 L1642.06,1972.64 L1651.5,1972.45 L1653.41,1971.88 L1654.41,1971.39 L1654.87,1971.01 L1655.22,1970.51 L1655.36,1969.87 L1655.25,1968.73 L1655.33,1968.24 L1655.9,1967.47 L1656.83,1966.63 L1657.84,1965.98 L1658.65,1965.77 L1659.61,1966.22 L1660.16,1966.85 L1660.72,1967.23 L1661.69,1966.94 L1662.44,1966.34 L1663.68,1965.06 L1664.52,1964.61 L1669.11,1963.92 L1672.43,1964.95 L1673.99,1964.72 L1674.51,1964.24 L1674.63,1964.13 L1675.59,1962.74 L1676.43,1962.25 L1677.97,1961.96 L1679.55,1962.03 L1680.27,1961.69 L1681.02,1959.75 L1681.66,1959.29 L1682.48,1959.18 L1683.44,1958.89 L1684.36,1958.47 L1685.04,1957.96 L1686.1,1956.57 L1686.48,1956.55 L1688.93,1957.74 L1689.61,1957.56 L1690.24,1956.65 L1690.47,1956.43 L1690.72,1956.32 L1690.91,1956.14 L1691,1955.67 L1692.14,1956.15 L1692.71,1956.16 L1693.18,1955.81 L1693.38,1955.29 L1693.32,1954.76 L1692.96,1953.42 L1692.5,1951.48 L1693.74,1950.59 L1695.61,1950.4 L1697.06,1950.55 L1697.47,1950.76 L1697.73,1951.08 L1697.98,1951.3 L1698.33,1951.27 L1698.42,1951.01 L1698.45,1950.01 L1698.58,1949.58 L1699.15,1948.86 L1699.21,1948.81 L1699.89,1948.16 L1700.7,1947.63 L1701.49,1947.37 L1702.38,1948.07 L1703.38,1948.47 L1703.52,1949 L1703.51,1949.54 L1703.65,1949.87 L1704.1,1949.93 L1704.4,1949.76 L1704.67,1949.53 L1705.06,1949.39 L1706.33,1949.3 L1706.73,1949.18 L1708.2,1948.41 L1708.89,1948.2 L1709.9,1948.18 L1710.56,1947.89 L1711.26,1946.37 L1712.24,1945.94 L1713.47,1945.17 L1714.59,1945.84 L1715.45,1947.34 L1715.9,1949.01 L1716.13,1949.49 L1716.37,1949.74 L1716.65,1949.93 L1716.96,1950.24 L1717.16,1950.39 L1717.65,1950.58 L1717.87,1950.75 L1717.97,1951.05 L1717.98,1951.75 L1718.05,1951.96 L1718.89,1952.48 L1719.54,1952.52 L1721.15,1952.17 L1720.98,1952.84 L1722.04,1952.63 L1722.97,1952.59 L1723.89,1952.82 L1724.94,1953.45 L1726.88,1955.17 L1728.05,1955.87 L1728.98,1955.72 L1729.21,1954.75 L1728.89,1953.71 L1728.88,1952.86 L1730.65,1952.38 L1731.56,1951.84 L1732,1951.66 L1732.63,1951.61 L1734.21,1952.2 L1742.35,1953.88 L1743.08,1953.57 L1743.31,1953.52 L1743.48,1953.65 L1743.95,1954.19 L1744.17,1954.36 L1746.05,1954.82 L1746.25,1954.87 L1747.28,1955.44 L1749.76,1957.74 L1750.76,1958.09 L1751.86,1958.05 L1752.93,1957.74 L1760.21,1953.62 L1761.35,1953.17 L1761.84,1953.11 L1762.22,1953.15 L1762.59,1953.12 L1763.01,1952.84 L1763.32,1952.26 L1763.49,1951.52 L1763.74,1950.86 L1764.32,1950.52 L1768.45,1949.94 L1768.93,1950.01 L1770,1950.37 L1770.47,1950.4 L1770.97,1950.22 L1771.25,1949.93 L1771.51,1949.54 L1772,1949 L1772.48,1948.71 L1774.06,1948.25 L1775.55,1946.72 L1775.99,1946.43 L1777.09,1946.6 L1778.74,1947.82 L1779.71,1948 L1780.34,1947.72 L1781.37,1946.88 L1782.01,1946.74 L1782.65,1946.73 L1783.15,1946.56 L1784.1,1945.83 L1785.16,1944.73 L1785.7,1944.31 L1789.45,1942.54 L1789.81,1942.55 L1790.06,1942.33 L1790.42,1941.37 L1790.82,1939.67 L1791.03,1938.03 L1791.2,1937.61 L1791.47,1937.44 L1791.67,1937.25 L1791.65,1936.74 L1791.44,1936.4 L1790.68,1935.93 L1790.43,1935.7 L1789.74,1934.72 L1789.74,1934.29 L1790.16,1933.62 L1790.17,1932.81 L1790.11,1932.11 L1789.9,1931.42 L1789.51,1930.67 L1789.2,1930.44 L1788.45,1930.46 L1788.15,1930.3 L1787.99,1929.91 L1787.86,1928.93 L1787.74,1928.48 L1787,1927.01 L1786.64,1926.55 L1786.15,1926.17 L1785.57,1926.03 L1784.9,1926.04 L1784.19,1925.88 L1783.75,1925.44 L1783.48,1925.18 L1783.12,1924.51 L1782.82,1923.68 L1782.67,1922.78 L1782.77,1921.97 L1783.02,1921.55 L1783.3,1921.38 L1783.61,1921.3 L1783.93,1921.08 L1784.95,1919.81 L1786.11,1919.21 L1788.6,1918.58 L1789.41,1917.39 L1790.08,1917.3 L1790.5,1917.45 L1792,1918.58 L1792.69,1918.94"
         id="path1123" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['EL'],labels['EL'], x1, polybary['EL'], x2, polybary['EL'] )
    if 'EL' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1548.38,1028.58 L1549.83,1030.97 L1551.51,1034.5 L1552.14,1035.56 L1552.61,1036.12 L1554.02,1037.14 L1554.48,1037.66 L1555.52,1039.51 L1556.91,1041.15 L1558.34,1041.79 L1566.82,1041.45 L1570.56,1041.28 L1572.78,1040.35 L1574.67,1040.8 L1575.28,1040.79 L1577.27,1040.18 L1577.73,1040.27 L1578.1,1040.75 L1579.2,1042.67 L1579.65,1043.24 L1580.16,1043.57 L1582.43,1044.5 L1583.69,1045.41 L1586.95,1047.77 L1590.5,1049.14 L1592.72,1049.99 L1595.39,1051.91 L1599.48,1054.1 L1600.66,1055.25 L1601.89,1056.94 L1604.47,1059.8 L1607.01,1060.95 L1612.32,1062.16 L1614.39,1062.24 L1616.38,1061.94 L1617.05,1063.31 L1617.58,1064.92 L1617.88,1066.55 L1617.87,1067.97 L1617.46,1069.2 L1616.86,1070.38 L1616.45,1071.65 L1616.65,1073.16 L1617.6,1074.29 L1617.66,1075.02 L1616.74,1076.56 L1616.6,1077.31 L1616.69,1079.69 L1616.59,1080.33 L1616.13,1082.06 L1616.06,1083.28 L1616.37,1083.88 L1617.01,1084.13 L1619.26,1084.43 L1619.85,1084.32 L1621.81,1083.26 L1627.8,1082.92 L1628.74,1083.34 L1629.31,1084.29 L1629.16,1085.37 L1628.48,1086.21 L1627.65,1086.66 L1626.39,1087.35 L1625.94,1087.79 L1625.18,1089.02 L1624.81,1089.92 L1624.72,1090.63 L1624.9,1092.37 L1624.94,1094.24 L1624.62,1095.26 L1623.89,1095.58 L1619.82,1094.78 L1619.34,1094.95 L1619.11,1095.32 L1618.99,1095.73 L1618.84,1096.02 L1618.52,1096.21 L1618.23,1096.29 L1614.13,1096.17 L1612.63,1096.81 L1611.94,1098.81 L1612.01,1099.52 L1612.4,1101.03 L1612.51,1101.79 L1612.46,1102.35 L1611.84,1105.47 L1611.44,1106.55 L1611.07,1107.59 L1610.68,1108.29 L1609.59,1109.31 L1605.54,1111.16 L1604.89,1111.29 L1603.54,1111.28 L1602.9,1111.41 L1601.71,1112.27 L1601.38,1112.72 L1601.36,1113.21 L1601.42,1113.78 L1601.34,1114.45 L1600.8,1115.49 L1600.13,1116.31 L1599.6,1117.2 L1599.5,1118.43 L1599.67,1118.97 L1600.25,1120.12 L1600.39,1120.62 L1600.35,1121.17 L1600.2,1121.33 L1599.96,1121.39 L1599.65,1121.64 L1599.04,1122.51 L1598.79,1123.14 L1598.87,1123.99 L1599.69,1127.09 L1599.78,1127.73 L1599.79,1128.41 L1599.73,1129.02 L1599.71,1129.66 L1599.85,1130.44 L1600.36,1131.6 L1601.72,1133.58 L1602.19,1134.72 L1602.28,1135.98 L1602,1137.04 L1601.49,1137.89 L1599.61,1139.67 L1599.23,1140.53 L1599.12,1141.86 L1599.33,1143.5 L1599.64,1144.77 L1599.77,1145.98 L1599.43,1147.45 L1598.24,1149.79 L1597.86,1151.26 L1598.16,1152.82 L1598.82,1153.5 L1599.66,1153.63 L1601.18,1153.19 L1603.07,1151.59 L1603.57,1151.32 L1604.12,1151.5 L1604.44,1151.96 L1604.68,1152.53 L1605.3,1153.54 L1605.49,1153.73 L1605.64,1153.74 L1606.02,1153.51 L1606.19,1153.51 L1606.62,1153.82 L1607.03,1154.2 L1607.44,1154.69 L1607.83,1155.31 L1608.02,1155.48 L1608.23,1155.51 L1608.43,1155.6 L1608.61,1155.96 L1608.58,1156.25 L1608.29,1156.91 L1608.24,1157.22 L1608.84,1160.03 L1608.77,1161.05 L1608.09,1161.86 L1607.68,1162.02 L1606.3,1162.13 L1605.95,1162.4 L1605.56,1163.32 L1605.24,1163.62 L1604.46,1163.82 L1601.17,1163.99 L1600.1,1163.72 L1599.11,1163.12 L1598.6,1162.08 L1598.77,1161.59 L1599.71,1161.01 L1599.87,1160.5 L1599.56,1160.02 L1598.59,1159.74 L1598.54,1159.11 L1599.18,1158.53 L1600.03,1158.42 L1600.28,1158.12 L1597,1155.21 L1596.75,1154.9 L1596.46,1154.72 L1595.99,1154.71 L1595.18,1155.34 L1594.22,1157.78 L1593.48,1158.53 L1590.58,1159.33 L1588.14,1159 L1587.52,1159.25 L1587.24,1160.15 L1587.14,1163.2 L1586.82,1164.61 L1585.71,1166.57 L1584.37,1167.92 L1582.82,1168.53 L1579.75,1168.02 L1578.05,1168.13 L1576.43,1168.66 L1576.28,1168.81 L1575.36,1169.67 L1575.29,1170.09 L1575.39,1170.56 L1575.52,1170.96 L1575.55,1171.2 L1574.97,1171.76 L1574.74,1172.23 L1574.51,1172.3 L1574.49,1172.52 L1574.86,1173.43 L1575.56,1174.66 L1576.27,1175.61 L1576.83,1176.72 L1577.12,1178.41 L1577.11,1178.41 L1577.11,1178.42 L1577.11,1178.43 L1576.87,1179.57 L1576.31,1180.05 L1574.05,1180.92 L1573.39,1181.01 L1572.74,1180.59 L1572.03,1179.48 L1571.88,1179.43 L1571.74,1179.42 L1571.61,1179.47 L1570.89,1180.32 L1568.68,1181.24 L1567.08,1182.68 L1563.71,1187.25 L1562.54,1188.18 L1561.24,1188.43 L1560.02,1188.32 L1558.81,1188.62 L1558.08,1188.62 L1557.58,1188.1 L1556.6,1186.46 L1555.34,1185.43 L1554.12,1185.26 L1552.98,1185.71 L1550.99,1187.36 L1549.98,1187.87 L1547.67,1188.36 L1547.05,1188.33 L1545.85,1187.85 L1545.24,1187.72 L1544.71,1187.94 L1542.26,1190.07 L1541.99,1190.18 L1541.67,1190.14 L1540.94,1189.88 L1540.68,1189.88 L1540.17,1190.21 L1539.63,1190.8 L1537.43,1191.56 L1535.79,1192.67 L1535.24,1192.82 L1534.58,1192.72 L1530.95,1191.36 L1529.63,1191.27 L1529.71,1190.88 L1529.67,1190.54 L1529.51,1190.25 L1529.24,1190.01 L1528.8,1189.77 L1528.42,1189.37 L1528.16,1188.84 L1528.06,1188.18 L1528.18,1187.87 L1528.33,1187.59 L1528.69,1187.14 L1529.05,1186.1 L1528.77,1184.71 L1527.7,1181.93 L1527.17,1180.67 L1526.51,1179.36 L1525.77,1178.23 L1525.01,1177.45 L1524.64,1177.32 L1523.83,1177.25 L1523.41,1177.06 L1522.98,1176.65 L1522.09,1175.51 L1521.64,1175.04 L1520.87,1174.59 L1520.03,1174.11 L1516.93,1173.55 L1515.4,1172.96 L1512.82,1171.49 L1512.11,1171.29 L1511.41,1171.59 L1510.62,1172.11 L1509.8,1172.27 L1508.98,1171.48 L1508.87,1170.82 L1508.95,1170.09 L1508.97,1169.48 L1508.68,1169.18 L1507.42,1169.26 L1506.96,1169.05 L1506.66,1168.7 L1506.42,1168.25 L1506.12,1167.8 L1505.67,1167.48 L1505.29,1167.46 L1504.5,1167.78 L1504.1,1167.8 L1501.75,1167.08 L1500.97,1167.11 L1500.15,1167.69 L1500.12,1167.79 L1499.52,1169.64 L1498.94,1170.32 L1496.08,1166.85 L1494.75,1164.89 L1494.1,1162.51 L1493.49,1157.68 L1493.74,1154.6 L1493.7,1153.45 L1493.54,1152.79 L1493.06,1151.42 L1492.95,1150.79 L1493.02,1150.33 L1493.59,1149.03 L1493.57,1148.78 L1493.42,1148.52 L1493.29,1148.24 L1493.31,1147.93 L1493.42,1147.8 L1493.7,1147.67 L1494.02,1147.28 L1494.58,1146.82 L1494.81,1146.44 L1495.15,1145.52 L1496.1,1144.51 L1496.53,1143.83 L1496.79,1142.67 L1496.69,1141.56 L1496.33,1140.45 L1495.34,1138.04 L1494.91,1137.31 L1494.38,1136.88 L1492.09,1136.26 L1491.71,1135.84 L1491.95,1134.93 L1491.44,1134.83 L1491.03,1134.51 L1490.7,1134.13 L1490.43,1133.91 L1490.07,1133.87 L1488.09,1134.24 L1487.45,1134.03 L1486.9,1133.35 L1486.18,1132.06 L1485.08,1129.37 L1484.47,1128.2 L1484.07,1127.87 L1483.82,1127.67 L1482.87,1127.33 L1479.37,1129.09 L1473.36,1129.38 L1472.69,1129.42 L1469.03,1131.08 L1468.65,1131.63 L1468.35,1132.26 L1467.78,1132.86 L1466.98,1133.08 L1465.97,1132.94 L1465.07,1132.4 L1464.57,1131.43 L1464.4,1130.42 L1464.17,1129.75 L1463.69,1129.51 L1462.79,1129.76 L1461.99,1130.62 L1461.71,1130.81 L1461.34,1130.73 L1460.37,1130.3 L1459.07,1130.19 L1458.31,1129.86 L1456.93,1129 L1454.39,1128.87 L1453.7,1128.45 L1452.96,1127.74 L1450.14,1126.19 L1448.59,1125.72 L1445,1126.18 L1443.55,1125.35 L1440.74,1121.91 L1439.48,1121 L1438.62,1121.27 L1436.11,1124.15 L1436.11,1124.15 L1436.11,1123.84 L1435.72,1122.44 L1435.65,1121.98 L1435.83,1121.7 L1436.1,1121.75 L1436.35,1121.68 L1436.46,1121.03 L1436.31,1120.53 L1435.95,1119.85 L1435.56,1119.25 L1435.28,1119 L1434.95,1118.78 L1434.74,1118.14 L1434.54,1116.74 L1434.16,1116.75 L1432.44,1119.02 L1432.17,1118.58 L1432.15,1118.45 L1432.24,1118.38 L1433.08,1115.07 L1433.31,1114.55 L1433.74,1113.79 L1433.5,1112.27 L1432.69,1109.84 L1431.96,1108.07 L1431.83,1107.59 L1431.64,1107.41 L1431.25,1107.09 L1430.83,1106.64 L1430.59,1106.08 L1430.5,1105.32 L1430.48,1104.81 L1430.42,1104.35 L1430.21,1103.73 L1428.34,1100.43 L1427.9,1099.38 L1427.61,1098.23 L1427.29,1097.65 L1425.44,1095.34 L1425.04,1095.01 L1424.7,1094.85 L1424.41,1094.49 L1424.01,1092.98 L1423.49,1091.82 L1422.11,1086.68 L1421.68,1083.96 L1421.65,1083.27 L1421.83,1082.36 L1421.82,1081.77 L1420.97,1076.46 L1420.45,1074.32 L1421.71,1073.79 L1423.74,1073.45 L1425.07,1072.85 L1425.39,1071.59 L1425.37,1069.84 L1425.63,1067.76 L1426.28,1066.26 L1427.29,1064.72 L1428.39,1063.44 L1429.36,1062.72 L1430.9,1062.39 L1431.41,1062.13 L1435.59,1057.65 L1436.75,1056.78 L1439.77,1056.15 L1448.61,1050 L1449.17,1049.61 L1452.64,1047.17 L1454.2,1047.01 L1454.91,1047.23 L1456.35,1047.97 L1457.07,1048.13 L1467.14,1046.01 L1469.42,1046.13 L1469.51,1046.16 L1470.51,1046.49 L1472.32,1047.83 L1472.95,1047.97 L1473.45,1047.8 L1477.91,1044.92 L1479,1043.82 L1479.87,1042.93 L1481.04,1042.49 L1482.26,1042.91 L1483.43,1044.53 L1485.21,1047.35 L1487,1048.28 L1488.64,1047.55 L1490,1045.37 L1489.58,1044.79 L1490.13,1044.27 L1493.9,1042.6 L1494.69,1042.6 L1497.18,1043.15 L1501,1043.99 L1502.25,1043.85 L1504.05,1042.23 L1505.09,1041.7 L1507.48,1041.36 L1508.38,1041.02 L1508.68,1041.07 L1509.06,1041.44 L1509.19,1041.89 L1509.23,1042.28 L1509.31,1042.48 L1510.41,1042.44 L1512.65,1041.44 L1513.78,1041.25 L1514.25,1041.3 L1518.25,1041.77 L1521.96,1043.67 L1523.24,1044.09 L1524.31,1044.08 L1524.36,1044.06 L1525.23,1043.71 L1528.89,1040.72 L1529.91,1040.32 L1531.09,1040.53 L1533.59,1041.69 L1534.78,1041.76 L1535.83,1041.01 L1537.55,1039.46 L1538.05,1038.65 L1540.01,1034.31 L1541.12,1032.97 L1542.24,1032.16 L1545.01,1030.15 L1545.35,1030.06 L1546.17,1030.2 L1546.43,1030.08 L1546.53,1029.5 L1546.42,1028.81 L1546.41,1028.13 L1546.78,1027.59 L1547.55,1027.68 L1548.38,1028.58"
         id="path1125" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['LT'],polycolors['LT'],labels['LT'], x1, polybary['LT'], x2, polybary['LT'] )
    if 'LT' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1395.38,120.416 L1395.8,121.194 L1395.72,122.811 L1396.29,123.04 L1397.61,122.703 L1398.14,122.786 L1399.29,123.444 L1399.76,123.525 L1401.33,122.531 L1401.55,121.882 L1401.68,121.195 L1402.02,120.77 L1402.46,120.649 L1402.93,120.867 L1403.05,121.097 L1403.14,121.443 L1403.26,121.791 L1403.45,122.049 L1403.76,122.234 L1403.87,122.165 L1404.43,119.916 L1404.7,119.28 L1405.12,118.946 L1405.52,118.975 L1406.08,119.203 L1406.65,119.668 L1407.05,120.377 L1407.18,121.398 L1406.92,121.906 L1406.6,122.309 L1406.53,123.006 L1406.93,122.895 L1406.77,123.067 L1406.66,123.363 L1406.62,123.778 L1406.63,124.302 L1406.88,124.092 L1407.45,123.153 L1407.54,123.798 L1407.52,124.312 L1407.43,124.714 L1407.28,125.008 L1407.95,125.216 L1409.64,122.992 L1412.65,122.437 L1413.74,123.181 L1413.92,123.203 L1414.11,123.318 L1414.3,123.738 L1414.36,124.087 L1414.33,124.647 L1414.54,125.834 L1414.43,127.115 L1414.49,127.665 L1414.54,129.049 L1413.85,129.601 L1412.32,129.833 L1412.82,130.464 L1413.26,130.503 L1413.71,130.404 L1414.21,130.634 L1414,131.547 L1413.62,132.047 L1412.56,132.416 L1412.67,132.785 L1412.91,132.888 L1413.03,133.015 L1413.14,133.214 L1413.34,133.528 L1410.45,133.882 L1410.94,134.222 L1411.42,134.447 L1412.33,134.644 L1411.95,135.88 L1411.27,136.657 L1410.44,137.077 L1409.56,137.213 L1408.19,136.771 L1407.78,136.776 L1407.11,137.336 L1406.78,137.218 L1406.57,136.222 L1406.21,136.107 L1406,136.333 L1405.84,136.692 L1405.61,136.963 L1405.57,137.441 L1403.11,138.527 L1402.76,138.371 L1402.54,138.335 L1402.1,138.768 L1401.86,138.855 L1402.36,139.355 L1403.16,139.451 L1403.6,139.712 L1403.4,140.15 L1403.14,140.472 L1402.96,140.892 L1402.99,141.634 L1404.33,140.019 L1405.08,139.481 L1407.03,139.674 L1407.22,139.813 L1407.45,140.287 L1407.69,140.411 L1409.13,139.837 L1409.7,139.801 L1410.31,140.139 L1409.73,142.412 L1407.22,144.138 L1406.29,145.636 L1406.42,145.773 L1406.55,146.006 L1405.29,147.903 L1404.57,148.435 L1404.11,149.164 L1403.3,149.795 L1402.93,150.494 L1405.73,149.275 L1411.84,143.475 L1413.14,140.632 L1413.7,140.34 L1414.36,141.189 L1414.98,144.172 L1414.59,146.775 L1413.74,148.798 L1412.12,151.311 L1411.89,152.293 L1411.36,153.385 L1409.95,154.728 L1409.4,155.714 L1409.19,156.788 L1409.69,157.504 L1410.04,158.126 L1410.87,159.864 L1410.88,159.151 L1410.69,157.006 L1410.79,156.364 L1411.02,155.886 L1411.59,155.224 L1411.37,156.096 L1411.23,157.08 L1413.27,155.031 L1415.66,153.711 L1414,155.858 L1413.74,157.064 L1413.78,158.006 L1414.01,158.089 L1414.61,157.346 L1414.9,156.468 L1415.16,156.015 L1415.14,156.048 L1414.9,156.639 L1414.9,157.085 L1415.13,157.58 L1415.6,157.701 L1415.71,157.356 L1415.99,155.344 L1416.39,154.006 L1416.91,152.93 L1417.59,152.232 L1418.45,152.017 L1418.56,153.323 L1418.37,154.045 L1418.08,154.618 L1417.94,155.473 L1417.92,156.651 L1417.73,158.451 L1417.36,160.215 L1417.36,161.036 L1417.58,161.924 L1418.12,162.984 L1418.52,163.938 L1418.77,165.008 L1419.08,166.934 L1419.77,169.135 L1421.36,172.901 L1421.88,174.597 L1421.99,175.754 L1421.89,176.542 L1421.72,177.207 L1421.45,178.962 L1421.05,179.767 L1420.51,180.306 L1419.97,180.448 L1421.21,180.641 L1422.16,179.191 L1422.7,176.849 L1422.73,174.362 L1422.49,173.125 L1422.16,172.441 L1421.72,171.82 L1421.18,170.777 L1420.99,170.161 L1420.75,168.963 L1420.56,168.518 L1420.38,168.15 L1419.78,166.742 L1419.55,165.684 L1419.46,163.867 L1419.36,162.888 L1419.07,161.659 L1418.75,160.623 L1418.53,159.659 L1418.53,158.661 L1418.65,158.213 L1419.11,157.314 L1419.6,156.791 L1419.51,156.207 L1419.31,155.52 L1419.2,154.904 L1419.44,153.933 L1420.01,153.155 L1420.75,152.777 L1421.47,152.989 L1421.75,154.055 L1422.33,154.985 L1422.92,155.46 L1423.22,155.151 L1423.14,154.574 L1422.89,154.11 L1422.61,153.674 L1422.45,153.157 L1422.49,152.564 L1422.68,152.12 L1423.18,151.629 L1423.04,151.188 L1421.52,151.522 L1420.96,151.321 L1419.38,149.99 L1419.14,149.842 L1419.03,149.181 L1419.36,144.741 L1419.29,144.137 L1419.05,142.767 L1419.01,142.146 L1419.15,141.436 L1419.46,141.224 L1420.4,141.316 L1421.18,141.15 L1421.23,140.845 L1420.83,140.27 L1419.9,139.716 L1419.06,139.471 L1418.53,138.79 L1418.57,136.923 L1419.61,134.116 L1419.85,131.42 L1420.14,130.1 L1420.56,129 L1421.06,128.266 L1426.08,126.128 L1426.65,126.181 L1427.23,126.51 L1427.26,126.779 L1426.51,127.146 L1426.64,127.55 L1429.89,126.881 L1430.99,127.578 L1431.11,127.848 L1431.37,128.694 L1431.53,129.016 L1431.8,129.292 L1432.08,129.51 L1432.59,129.773 L1434.29,129.657 L1434.92,130.018 L1434.73,131.052 L1433.52,133.088 L1434.23,133.441 L1436.12,132.814 L1435.9,133.387 L1435.46,133.801 L1434.47,134.144 L1434.78,134.724 L1435.06,134.92 L1435.32,134.868 L1435.56,134.712 L1435.6,135.616 L1435.24,136.066 L1434.76,136.374 L1434.45,136.85 L1434.58,136.998 L1434.67,137.078 L1434.86,137.177 L1434.85,138.067 L1435.51,138.398 L1435.82,137.846 L1436.08,135.889 L1436.36,135.259 L1438.71,134.012 L1441.6,130.69 L1442.54,130.472 L1443.53,131.232 L1444.32,133.445 L1444.42,135.007 L1443.72,136.837 L1443.64,138.137 L1446.25,136.463 L1446.47,136.026 L1446.51,135.505 L1446.51,134.934 L1446.59,134.362 L1447.56,131.443 L1448.08,130.735 L1449.54,130.128 L1450.34,130.198 L1452.97,131.306 L1452.9,132.423 L1452.48,133.119 L1451.45,133.959 L1452.31,134.291 L1454.62,133.918 L1457.1,134.953 L1457.51,135.479 L1457.33,136.269 L1456.39,137.548 L1454.38,138.918 L1453.95,139.477 L1454.09,139.881 L1454.32,139.711 L1454.58,139.667 L1454.86,139.867 L1455.18,140.438 L1454.34,140.663 L1452.42,140.421 L1451.59,140.64 L1451.73,141.042 L1455.27,140.843 L1460.41,136.863 L1461.37,136.739 L1463.32,137.202 L1463.92,137.825 L1463.43,138.771 L1467.86,140.619 L1468.6,142.112 L1469.95,143.073 L1470.59,143.259 L1470.61,142.591 L1470.74,142.096 L1472.02,140.091 L1472.56,140.317 L1473.7,141.844 L1474.13,142.143 L1474.6,142.355 L1476.66,144.469 L1476.56,145.188 L1476.37,145.74 L1476.33,146.335 L1476.69,147.189 L1477.15,147.707 L1477.61,147.966 L1478.11,148.352 L1478.67,149.239 L1477.93,149.436 L1476.19,151.376 L1468.69,154.729 L1468.2,155.51 L1468.66,156.882 L1468.4,157.802 L1466.71,158.824 L1466.42,159.541 L1466.33,160.395 L1466.57,161.104 L1467.27,161.371 L1467.17,162.984 L1466.63,163.954 L1463.6,166.514 L1463.23,167.117 L1463.34,167.531 L1463.75,167.725 L1464.32,167.671 L1454.18,169.133 L1444.03,170.523 L1444.16,170.485 L1437.67,170.345 L1431.18,170.176 L1430.51,170.713 L1430.39,171.756 L1433.52,170.474 L1433.42,171.682 L1432.83,172.835 L1432.06,173.711 L1431.44,174.117 L1432.08,174.795 L1432.94,175.013 L1434.38,174.702 L1434.99,173.697 L1435.25,173.639 L1436.45,174.053 L1436.8,174.015 L1436.92,174.38 L1435.01,174.846 L1434.62,175.475 L1438.8,175.047 L1440.57,175.71 L1443.53,174.745 L1443.54,175.395 L1443.51,175.634 L1444.27,175.342 L1444.74,175.318 L1445.05,175.632 L1444.96,175.935 L1444.16,177.218 L1444.88,177.498 L1448.57,175.936 L1449.75,176.363 L1456.58,176.646 L1456.19,176.996 L1454.95,179.328 L1454.2,180.229 L1453.59,181.363 L1452.82,182.233 L1452.64,182.966 L1452.63,183.408 L1452.65,183.661 L1452.64,183.908 L1452.56,184.344 L1452.45,184.595 L1452.03,185.142 L1453.11,185.004 L1453.48,184.542 L1453.72,182.812 L1454.22,181.267 L1454.54,180.854 L1454.82,180.739 L1455.15,180.751 L1455.51,180.884 L1455.88,181.121 L1456.23,181.199 L1456.64,180.698 L1457.53,180.806 L1458.27,179.786 L1458.95,179.986 L1459.01,180.818 L1459.31,181.294 L1459.64,181.735 L1459.77,182.452 L1461.02,183.858 L1460.79,184.56 L1460.92,186.211 L1460.32,187.017 L1459.69,188.596 L1459.77,189.127 L1459.92,189.566 L1459.53,189.805 L1459.45,190.425 L1459.48,191.195 L1459.41,191.9 L1459.13,192.373 L1458.31,192.847 L1457.97,193.217 L1458.72,193.027 L1458.77,193.863 L1458.54,195.114 L1458.46,196.18 L1459.03,195.531 L1459.25,195.041 L1459.48,193.677 L1461.05,190.57 L1461.69,190.011 L1463.83,189.183 L1465.67,189.13 L1465.79,189.019 L1466.95,186.675 L1467.57,185.969 L1468.36,186.102 L1468.14,186.747 L1468.1,188.316 L1467.96,189.139 L1467.2,190.038 L1467.01,190.72 L1467.42,191.772 L1467.35,190.653 L1467.7,189.961 L1468.6,189.148 L1469.14,187.844 L1469.37,187.857 L1470.26,189.054 L1470.72,189.566 L1472.39,190.355 L1472.6,190.202 L1472.47,189.719 L1472.19,189.394 L1471.6,188.926 L1470.13,187.288 L1469.65,186.642 L1470.01,186.27 L1471.15,185.757 L1471.37,185.186 L1471.13,184.334 L1470.64,183.879 L1470.07,183.547 L1469.55,183.065 L1469.43,182.27 L1469.02,180.913 L1468.56,179.61 L1468.3,178.982 L1469.87,178.501 L1470.73,177.78 L1471.08,177.889 L1471.85,178.706 L1471.89,179.207 L1471.14,180.343 L1471.92,180.574 L1473.24,179.555 L1473.97,179.917 L1474.21,180.325 L1474.41,180.84 L1474.6,181.387 L1474.8,181.898 L1475.38,183.184 L1475.44,183.262 L1476.36,185.848 L1476.98,187.903 L1476.97,189.34 L1476.03,190.043 L1476.64,190.283 L1477.96,189.446 L1478.17,189.097 L1478.06,188.441 L1477.61,186.874 L1477.26,186.211 L1477.13,185.824 L1477.35,185.469 L1477.33,185.219 L1477.25,184.892 L1477.14,184.544 L1476.82,183.9 L1476.48,183.409 L1476.24,182.938 L1476.22,182.349 L1475.63,181.254 L1476.14,180.58 L1480.82,179.75 L1481.5,180.254 L1481.31,180.628 L1481.37,181.222 L1481.61,181.737 L1481.99,181.892 L1482.27,181.674 L1482.52,181.269 L1482.91,180.255 L1483.07,180.206 L1483.82,178.979 L1484.71,181.296 L1485.56,182.157 L1485.99,182.767 L1486.46,183.766 L1486.64,184.337 L1486.87,184.799 L1487.5,185.477 L1488.4,186.296 L1490.08,190.833 L1490.34,191.831 L1490.45,194.038 L1489.84,195.331 L1487.91,196.33 L1482.49,197.056 L1481.68,196.796 L1476.22,193.502 L1474.54,192.902 L1473.52,192.95 L1472.61,193.323 L1472.25,193.78 L1472.58,194.211 L1473.2,194.542 L1474.66,194.738 L1475.06,195.164 L1475.61,196.186 L1476.15,197.571 L1476.35,198.653 L1476.31,199.49 L1476.11,200.144 L1475.66,200.76 L1475.58,201.159 L1475.67,202.113 L1475.87,202.983 L1476.12,203.72 L1476.27,204.419 L1476.2,205.164 L1474.56,208.751 L1474.25,210.224 L1474,210.815 L1473.13,211.553 L1469.98,212.569 L1469.6,213.05 L1468.7,215.045 L1467.66,215.993 L1464.1,217.259 L1463.06,218.229 L1462.62,219.934 L1462.73,220.86 L1463.04,222.574 L1463.02,223.14 L1463.14,223.641 L1463.3,224.14 L1463.4,224.688 L1463.41,225.281 L1463.32,226.332 L1463.33,226.93 L1463.64,229.178 L1463.71,230.179 L1463.61,231.291 L1463.12,233.32 L1462.41,234.991 L1461.43,236.06 L1460.16,236.279 L1459.47,236.652 L1459.17,237.189 L1455.7,234.734 L1455.24,234.182 L1454.93,233.227 L1453.91,228.686 L1453.66,227.309 L1453.69,226.453 L1458.47,214.671 L1459.24,208.751 L1453.91,202.391 L1451.24,198.098 L1450.71,197.583 L1442.82,196.22 L1434.94,194.815 L1434.03,194.697 L1433.03,194.274 L1432.14,193.219 L1431.55,191.192 L1426.99,190.517 L1426.38,190.147 L1420.4,184.321 L1418.49,182.015 L1417.56,181.142 L1416.85,181.036 L1415.58,182.431 L1412.77,182.577 L1411.79,183.239 L1411.29,184.734 L1411.44,185.812 L1409.33,187.678 L1408.49,189.764 L1408.2,190.001 L1407.87,190.116 L1407.61,190.354 L1407.53,190.952 L1407.72,191.045 L1408.15,191.368 L1408.33,191.45 L1407.82,192.823 L1403.85,195.63 L1403.43,195.662 L1401.54,194.888 L1400.97,194.98 L1400.4,195.328 L1398.99,194.772 L1398.27,194.712 L1396.49,195.785 L1395.01,195.727 L1391.53,197.223 L1390.94,197.811 L1390.84,199.05 L1390.39,200.293 L1390.3,200.797 L1390.86,201.828 L1390.58,203.207 L1389.08,206.184 L1388.65,208.639 L1388.11,210.458 L1388.01,211.064 L1387.86,211.577 L1387.56,211.916 L1384.61,213.559 L1384.16,214.138 L1383.88,215.131 L1383.8,215.898 L1383.82,216.576 L1384.03,217.156 L1384.54,217.625 L1385.41,218.866 L1385.77,220.298 L1385.73,221.699 L1385.44,222.841 L1384.42,224.15 L1384.3,224.892 L1384.48,225.236 L1384.81,225.523 L1385.03,225.937 L1384.9,226.67 L1384.95,227.413 L1385.35,228.85 L1385.28,229.39 L1385.12,229.814 L1384.96,231.337 L1386.25,233.511 L1386.48,233.985 L1385.43,237.949 L1385.39,238.852 L1385.84,243.906 L1386.19,245.543 L1386.95,247.04 L1387.16,247.352 L1387.71,248.336 L1387.89,248.745 L1387.98,249.216 L1387.98,249.518 L1387.95,249.75 L1387.97,250.004 L1388.45,252.01 L1388.52,252.437 L1388.5,253.054 L1388.62,253.43 L1389.2,254.311 L1389.27,254.56 L1389.31,254.809 L1389.41,255.074 L1389.67,255.351 L1390.2,255.689 L1390.46,255.949 L1390.65,256.355 L1390.66,257.174 L1389.97,258.292 L1389.92,259.105 L1390.01,259.937 L1389.92,260.602 L1389.49,261.782 L1389.33,262.656 L1389.29,263.633 L1389.19,264.548 L1388.8,265.236 L1388.27,265.488 L1385.34,265.167 L1384.35,265.641 L1383.46,266.44 L1380.27,271.122 L1379.83,273.013 L1379.84,274.285 L1380.06,275.641 L1380.73,278.473 L1380.74,278.958 L1380.69,279.329 L1380.77,279.701 L1381.17,280.159 L1381.21,281.451 L1381.03,282.129 L1380.65,282.466 L1380.08,282.742 L1378.89,284.083 L1378.3,284.18 L1377.79,284.934 L1377.73,286.019 L1377.75,287.157 L1377.45,288.079 L1376.9,288.337 L1376.36,287.908 L1375.2,286.204 L1372.86,283.924 L1370.41,282.658 L1365.98,282.141 L1362.36,281.675 L1358.58,280.185 L1357.92,279.669 L1356.89,278.527 L1351.8,277.034 L1350.09,277.357 L1348.86,278.683 L1349.03,281.434 L1348.83,284.047 L1347.01,285.715 L1343.54,287.006 L1338.02,292.382 L1337.47,292.26 L1335.08,289.647 L1334.34,289.108 L1333.09,288.979 L1329.45,290.372 L1323.06,288.469 L1322.44,288.447 L1321.83,288.608 L1319.45,290.056 L1318.98,289.963 L1318.46,288.861 L1316.77,283.426 L1316.32,282.867 L1314.23,281.685 L1311.87,278.884 L1311.49,278.309 L1310.87,276.837 L1310.41,276.135 L1306.66,273.27 L1304.3,270.295 L1295.68,259.326 L1295.05,258.919 L1287.63,258.908 L1283.45,262.869 L1283.14,263.508 L1282.94,264.498 L1282.66,266.669 L1286.44,271.948 L1285.64,274.839 L1285.11,275.376 L1284.19,275.251 L1277.06,272.403 L1276.5,272.843 L1275.07,276.993 L1269.31,278.328 L1263.54,279.64 L1270.06,285.708 L1270.41,286.721 L1271.5,292.122 L1271.6,293.297 L1271.5,294.581 L1270.27,301.458 L1268.86,304.952 L1265.18,309.619 L1272.34,312.573 L1268.05,320.343 L1267.28,321.249 L1266.36,321.451 L1255.27,318.387 L1244.29,314.753 L1236.6,316.378 L1235.27,316.194 L1231.01,312.924 L1229.86,312.322 L1224.47,315.193 L1224.09,315.395 L1223.9,315.777 L1223.87,316.504 L1224.17,322.245 L1224.39,324.133 L1227.23,336.127 L1227.04,338.669 L1223.85,347.954 L1223.31,349.584 L1222.75,350.791 L1222.17,351.058 L1214.89,347.637 L1207.63,344.184 L1207.18,344.694 L1205.72,348.463 L1196.88,357.452 L1196.38,358.354 L1196.01,359.629 L1193.88,374.193 L1193.51,375.565 L1191.44,380.011 L1190.29,381.607 L1188.88,382.509 L1184.93,383.705 L1184.54,384.481 L1183.7,389.154 L1192.64,402.231 L1192.89,403.312 L1193.03,411.416 L1192.74,412.489 L1192.08,413.28 L1190.71,414.302 L1186.62,419.39 L1184.75,422.479 L1177.21,441.107 L1171.7,448.305 L1171.94,450.012 L1173.69,456.656 L1173.91,457.52 L1174.29,459.508 L1174.44,460.775 L1174.26,461.609 L1163.99,469.98 L1157.52,471.308 L1151.03,472.612 L1154.97,489.978 L1155.27,492.486 L1155.06,493.822 L1153.62,497.942 L1153.44,498.931 L1153.18,513.696 L1153.41,516.655 L1153.83,519.907 L1153.92,522.801 L1153.21,524.682 L1151.19,526.885 L1150.72,527.861 L1150.45,529.408 L1150.12,535.225 L1150,535.723 L1149.98,535.791 L1149.54,537.68 L1146.25,543.661 L1146.24,543.671 L1139.95,556.747 L1133.56,569.804 L1141.96,575.134 L1143.13,575.5 L1145.3,575.859 L1146.32,576.506 L1146.92,577.974 L1149.12,592.568 L1149.16,593.702 L1148.89,594.857 L1144.92,604.425 L1144.46,605.074 L1143.77,605.113 L1133.89,603.105 L1124.04,601.048 L1117.19,603.351 L1111.1,607.808 L1109.16,610.07 L1103.99,619.423 L1099.06,629.105 L1095.74,632.883 L1097.34,639.169 L1097.39,640.56 L1097.02,641.797 L1091.34,652.307 L1093.42,657.497 L1098.48,670.03 L1095.25,675.006 L1094.97,676.008 L1095.11,677.307 L1097.13,686.788 L1097.11,688.235 L1096.73,689.983 L1095.8,693.05 L1095.47,694.788 L1095.55,696.036 L1101.53,710.966 L1101.75,711.508 L1102.68,714.557 L1102.73,715.452 L1102.89,717.803 L1101.14,733.761 L1099.36,749.719 L1099.8,750.537 L1103.38,754.198 L1104.81,755.284 L1107.28,758.368 L1108.15,758.887 L1109.29,759.12 L1111.35,759.185 L1112.33,759.513 L1113.28,760.356 L1121.65,771.034 L1121.97,771.614 L1122.04,772.354 L1121.29,778.28 L1120.91,780.065 L1120.36,781.465 L1118.29,784.89 L1118.08,785.462 L1117.93,786.301 L1117.51,790.565 L1117.44,790.827 L1117.25,791.51 L1116.62,791.901 L1113.15,791.385 L1111.39,791.544 L1104.44,793.9 L1103.62,794.769 L1103.75,796.039 L1105.03,798.872 L1105.67,800.943 L1106.49,805.476 L1107.03,807.681 L1107.95,810.104 L1111.41,816.033 L1115.21,824.8 L1115.72,826.56 L1115.92,828.498 L1115.78,830.768 L1115.47,832.288 L1115.03,833.5 L1114.47,834.522 L1112.88,836.721 L1112.5,837.568 L1112.4,838.596 L1112.74,840.704 L1113.7,844.696 L1113.88,846.935 L1113.55,849.681 L1112.77,852.006 L1111.7,853.927 L1106.99,859.862 L1102.49,863.774 L1101.2,864.09 L1096.12,863.731 L1094.5,864.353 L1094.02,864.856 L1093.25,865.659 L1093.1,867.789 L1093.5,868.66 L1094.45,869.819 L1094.65,870.062 L1095.06,871.068 L1095.14,872.013 L1095,874.391 L1095.04,875.518 L1094.87,876.036 L1093.69,878.093 L1092.95,878.966 L1088.73,881.41 L1088.15,882.373 L1088.07,884.036 L1088.31,885.374 L1092.03,897.692 L1092.87,901.862 L1093.07,903.275 L1093.1,904.603 L1093.08,904.662 L1092.83,905.563 L1092.29,906.595 L1092.02,907.577 L1091.25,911.503 L1091.22,914.5 L1090.91,916.172 L1090.16,918.937 L1089.91,920.349 L1089.74,921.956 L1088.87,925.318 L1087.2,927.04 L1085.12,927.613 L1083.02,927.522 L1082.01,926.971 L1081.97,922.942 L1081.44,921.037 L1081.42,921.001 L1081.41,921 L1081.41,921 L1081.41,920.593 L1080.26,918.094 L1080.11,917.527 L1079.9,916.134 L1079.74,915.466 L1079.11,914.234 L1078.26,913.578 L1077.26,913.36 L1074.13,913.714 L1073.02,913.381 L1072.6,912.218 L1072.74,911.829 L1073.01,911.702 L1073.59,911.733 L1073.77,911.514 L1073.6,911.089 L1073.33,910.781 L1073.22,910.928 L1072.99,910.629 L1072.84,910.324 L1072.76,909.867 L1072.8,909.34 L1072.88,909.117 L1072.89,908.88 L1072.74,908.333 L1072.43,908.634 L1071.43,909.209 L1071.91,909.7 L1072.02,910.49 L1071.85,911.279 L1071.48,911.768 L1070.94,911.774 L1070.55,911.253 L1069.9,909.696 L1070.24,911.396 L1070.31,912.29 L1070.06,912.681 L1068.17,912.709 L1068.39,911.721 L1067.96,911.221 L1067.34,910.775 L1067.02,909.955 L1066.73,909.748 L1065.41,909.287 L1065.01,909.303 L1065.31,910.444 L1065.29,911.416 L1064.98,911.98 L1064.41,911.87 L1064.13,911.398 L1063.88,910.516 L1063.78,909.461 L1063.9,908.462 L1063.29,909.042 L1062.96,909.247 L1062.61,909.333 L1062.25,909.513 L1062.19,909.923 L1062.18,910.376 L1061.97,910.699 L1061.38,910.549 L1060.69,909.005 L1059.95,908.975 L1060.1,907.917 L1059.51,906.331 L1059.7,905.057 L1059.35,905.509 L1058.97,906.826 L1058.61,907.282 L1058.43,905.77 L1058.75,904.073 L1059.31,902.472 L1059.86,901.253 L1059.35,901.108 L1059,900.651 L1058.72,900.168 L1058.42,899.948 L1058.09,900.218 L1057.61,901.398 L1057.35,901.668 L1056.49,901.112 L1056.19,899.822 L1056,898.389 L1055.45,897.42 L1055.86,896.493 L1055.97,895.836 L1055.75,895.339 L1055.18,894.859 L1053.92,894.384 L1053.78,893.953 L1054.07,892.734 L1054.9,890.931 L1055.43,890.031 L1055.81,889.733 L1056.02,890.303 L1055.97,891.268 L1055.6,893.112 L1056.04,893.48 L1056.44,893.29 L1056.75,892.7 L1056.92,891.894 L1056.9,890.798 L1056.68,890.259 L1056.36,889.89 L1056.02,889.304 L1055.62,887.949 L1055.43,886.511 L1055.46,885.066 L1055.75,883.682 L1055.18,882.368 L1054.76,881.029 L1054.48,879.419 L1054.36,877.288 L1053.82,878.081 L1053.12,877.146 L1052.62,875.487 L1052.7,874.103 L1053.13,872.97 L1053.89,869.996 L1054.28,868.757 L1055.56,867.308 L1056.37,869.134 L1056.96,872.264 L1057.61,874.699 L1057.91,872.306 L1058.49,869.924 L1058.82,867.683 L1058.38,865.694 L1058.61,865.264 L1058.2,864.877 L1057.29,864.54 L1056.58,863.855 L1056.33,864.282 L1056.13,864.943 L1055.88,865.291 L1054.46,864.841 L1054.28,864.95 L1053.88,865.533 L1053.61,865.739 L1052.32,865.622 L1051.57,865.74 L1050.9,866.418 L1049.88,867.768 L1049.83,868.31 L1050.62,868.785 L1050.31,869.873 L1050.25,870.695 L1050.19,871.603 L1050.24,873.526 L1050.43,875.188 L1050.79,876.594 L1051.33,877.821 L1052,878.691 L1052.74,879.011 L1053.37,879.492 L1053.9,880.616 L1054.27,881.903 L1054.41,882.878 L1053.75,885.218 L1052.16,887.408 L1050.29,888.739 L1048.79,888.505 L1048.36,887.541 L1048.31,886.28 L1048.52,883.539 L1048.45,882.945 L1048.06,880.788 L1047.93,879.514 L1047.83,878.958 L1047.65,878.232 L1047.48,877.907 L1046.74,876.896 L1046.63,876.744 L1046.29,876.479 L1046.08,876.5 L1046.04,876.503 L1045.87,876.519 L1045.39,876.453 L1044.26,875.176 L1043.31,874.697 L1042.43,874.761 L1041.97,875.696 L1042.4,875.997 L1043.01,877.114 L1043.39,877.357 L1043.88,877.446 L1044.29,877.689 L1046.29,878.87 L1046.9,879.767 L1047.28,881.026 L1047.42,882.714 L1047.41,884.263 L1047.29,886.019 L1047,887.455 L1046.47,888.052 L1045.47,887.633 L1043.44,885.991 L1042.41,885.936 L1042.75,886.985 L1043.4,888.114 L1044.14,889.017 L1044.76,889.381 L1045.43,889.98 L1046.66,892.462 L1049.53,893.345 L1050.13,894.044 L1050.22,895.139 L1049.97,896.28 L1049.17,898.107 L1049.1,899.094 L1049.66,899.915 L1050.86,900.906 L1051.36,901.956 L1051.06,902.553 L1050.5,903.137 L1050.2,904.134 L1049.95,905.412 L1049.36,905.342 L1048.23,904.342 L1046.79,903.808 L1046.45,903.962 L1046.4,904.352 L1046.46,905.595 L1046.46,906.053 L1046.59,906.795 L1046.5,907.198 L1045.63,907.499 L1045.48,907.821 L1045.43,908.286 L1045.07,909.702 L1044.95,910.967 L1044.91,912.196 L1044.98,912.928 L1044.2,912.519 L1044.11,913.751 L1044.41,915.637 L1044.77,917.198 L1044.31,916.548 L1044.11,915.664 L1043.96,914.703 L1043.64,913.789 L1043.51,915.018 L1043.57,917.948 L1043.34,918.526 L1043.26,917.812 L1042.31,913.793 L1042.09,913.794 L1042.04,914.634 L1041.73,916.059 L1041.68,916.782 L1041.85,917.259 L1042.16,917.8 L1042.36,918.418 L1042.22,919.13 L1041.55,919.916 L1040.52,920.613 L1039.41,920.952 L1038.56,920.633 L1038.92,920.039 L1040.09,919.225 L1040.34,918.729 L1040.14,917.676 L1039.66,917.723 L1038.67,918.539 L1038.21,918.738 L1037.33,919.619 L1036.78,919.818 L1036.59,919.523 L1035.93,919.01 L1035.39,918.847 L1035.93,921.36 L1035.33,922.632 L1034.32,923.398 L1033.47,923.66 L1029.86,923.721 L1028.76,923.23 L1029.63,922.705 L1029.84,921.615 L1029.57,920.362 L1029.01,919.346 L1027.79,917.742 L1027.28,917.665 L1027.03,918.915 L1026.73,918.568 L1025.93,917.202 L1025.76,917.089 L1025.54,917.037 L1025.35,916.915 L1025.27,916.558 L1025.29,915.663 L1025.27,915.495 L1024.79,915.256 L1024.69,915.925 L1024.71,917.014 L1024.6,918.051 L1024.14,917.711 L1023.89,917.407 L1023.51,916.341 L1023.39,915.692 L1023.38,915.054 L1023.27,914.519 L1022.86,914.163 L1023.25,913.572 L1023.23,912.967 L1022.93,912.706 L1021.97,913.685 L1019.98,913.368 L1020.61,914.509 L1022.62,916.215 L1023.04,917.83 L1023.66,918.594 L1024.96,919.088 L1026.01,919.985 L1025.91,921.938 L1025.71,921.938 L1025.61,921.647 L1025.41,921.346 L1025.25,921.043 L1024.99,921.721 L1024.64,921.964 L1023.7,921.929 L1023.7,922.356 L1024.08,922.419 L1024.4,922.642 L1024.64,923.04 L1024.79,923.638 L1023.94,923.502 L1021.69,924.477 L1021.69,924.905 L1021.82,925.007 L1022.13,925.375 L1021.09,925.822 L1017.65,925.346 L1018.9,926.472 L1019.39,927.15 L1018.98,927.45 L1017.69,927.089 L1017.18,927.281 L1016.99,928.29 L1016.27,927.559 L1015.35,927.216 L1014.51,927.402 L1014.06,928.266 L1014.76,927.888 L1015.17,927.754 L1015.42,927.85 L1015.47,928.316 L1015.32,928.734 L1015.06,929.045 L1014.84,929.165 L1013.38,929.442 L1012.7,929.222 L1012.07,928.247 L1011.91,929.07 L1011.13,931.262 L1011.51,931.271 L1012.17,931.644 L1012.48,931.703 L1012.82,931.564 L1013.84,930.861 L1014.67,930.606 L1015.44,930.621 L1016.18,930.871 L1016.97,931.315 L1016.5,932.115 L1014.92,933.16 L1014.23,933.854 L1014.93,933.723 L1015.59,933.4 L1015.59,933.866 L1015.05,934.164 L1014.35,934.741 L1013.71,935.516 L1013.34,936.411 L1012.23,937.179 L1010.36,937.48 L1004.84,936.741 L1004.54,936.936 L1004.26,937.375 L1004.12,937.816 L1004.27,938.019 L1006.89,938.93 L1007.73,938.522 L1005.95,937.585 L1005.52,937.177 L1009.22,938.216 L1009.26,939.394 L1008.28,940.083 L1005.29,940.131 L1004.11,940.574 L1005.09,940.566 L1007.28,940.262 L1008.15,940.41 L1008.72,941.292 L1008.38,941.756 L1007.12,943.091 L1005.03,944.87 L1003.2,946.076 L1001.8,945.708 L1001.62,945.899 L1001.5,946.192 L1001.36,946.444 L1001.14,946.548 L1000.91,946.423 L1000.63,945.874 L1000.44,945.688 L999.66,945.259 L999.268,945.261 L999.433,945.904 L1000.14,946.539 L1000.41,946.98 L1000.31,947.625 L1000.41,949.011 L1000.19,950.333 L999.658,951.107 L999.033,951.885 L996.957,953.948 L996.16,954.846 L995.184,955.608 L994.625,955.992 L993.849,956.385 L993.377,957.124 L993.12,957.854 L992.504,958.599 L992.251,959.401 L991.434,959.894 L990.966,960.066 L990.441,960.15 L990.068,960.298 L989.81,960.731 L989.581,961.307 L989.304,961.861 L988.896,962.22 L988.489,962.298 L988.275,962.124 L988.263,961.663 L987.995,961.286 L986.824,963.421 L985.747,964.625 L985.499,965.261 L985.871,966.002 L985.294,966.876 L984.571,966.733 L983.803,966.416 L983.119,966.796 L982.744,966.009 L982.279,965.884 L981.979,966.267 L982.102,966.987 L982.136,967.622 L981.559,967.754 L980.393,967.552 L980.105,967.751 L979.163,968.704 L978.877,969.069 L978.416,969.506 L977.217,969.803 L976.693,970.065 L976.945,970.112 L977.389,970.344 L977.611,970.515 L977.115,971.605 L976.814,972.056 L976.084,972.55 L975.987,972.937 L975.909,973.263 L975.887,973.986 L975.903,974.317 L975.179,975.04 L974.677,974.408 L974.248,973.567 L973.757,973.659 L973.003,975.485 L972.521,976.282 L972.092,976.182 L971.722,975.864 L970.64,975.713 L970.204,975.523 L970.03,975.286 L969.67,974.449 L969.328,973.755 L970.299,973.336 L970.775,973.008 L971.198,972.489 L970.569,971.515 L970.142,969.839 L969.644,968.522 L968.798,968.61 L969.359,969.891 L969.504,971.125 L969.294,972.333 L968.768,973.525 L966.945,974.711 L966.268,975.646 L966.975,976.68 L965.608,977.353 L960.251,977.932 L958.901,978.477 L958.259,978.586 L957.212,978.005 L956.863,978.159 L957.065,979.365 L956.463,979.322 L956.197,979.191 L955.944,978.9 L955.982,980.22 L955.481,980.774 L954.803,980.486 L954.339,979.274 L954.176,980.109 L954.174,980.483 L954.277,981.021 L948.016,980.8 L943.969,979.495 L943.518,978.997 L943.436,978.061 L941.42,979.019 L940.878,979.673 L940.63,978.731 L940.281,978.455 L937.828,979.196 L937.685,979.453 L937.678,979.74 L937.659,979.933 L937.588,980.261 L937.588,980.702 L937.523,981.081 L937.261,981.236 L936.992,981.266 L935.732,981.741 L935.177,981.822 L934.877,981.56 L935.127,980.679 L935.039,979.465 L935.842,978.707 L936.911,978.17 L937.645,977.599 L938.083,977.092 L938.399,977.243 L938.674,977.651 L938.997,977.886 L939.286,977.595 L939.87,976.187 L940.126,975.759 L940.143,975.331 L939.478,975.463 L937.338,976.593 L936.874,977.142 L936.307,977.539 L933.729,978.508 L932.952,978.452 L933.557,977.688 L934.54,975.799 L935.121,975.125 L934.612,974.773 L932.62,977.199 L931.816,977.975 L932.077,977.245 L932.264,976.573 L932.321,975.862 L932.197,975 L931.783,975.245 L931.192,975.981 L930.759,976.222 L930.261,976.169 L929.638,975.877 L929.07,975.416 L928.755,974.849 L930.256,973.97 L931.018,973.752 L933.648,973.759 L934.488,973.428 L935.007,972.555 L934.001,972.574 L932.158,973.06 L931.162,972.78 L930.376,971.95 L928.898,969.892 L928.114,969.689 L929.123,973.102 L928.939,973.575 L927.248,973.762 L926.401,973.504 L925.919,972.544 L926.339,971.914 L926.376,971.143 L926.288,970.217 L926.31,969.143 L925.39,972.418 L924.97,973.395 L925.472,973.425 L925.697,973.747 L925.831,974.227 L926.042,974.726 L926.381,975.113 L927.036,975.617 L927.685,976.436 L927.97,976.709 L927.935,976.988 L927.296,977.384 L926.968,977.397 L926.308,976.98 L925.948,976.856 L922.497,976.55 L921.769,976.042 L920.082,973.97 L919.742,973.146 L920.405,973.033 L921.023,972.505 L921.492,971.618 L921.713,970.441 L922.12,969.757 L923.977,969.756 L924.704,969.532 L924.251,969.129 L924.048,969.038 L924.798,967.975 L926.5,967.136 L927.916,965.657 L930.634,964.207 L930.066,963.973 L929.212,964.202 L927.641,965.14 L925.624,966.913 L925.054,966.98 L924.66,966.432 L923.79,964.35 L923.635,963.67 L923.908,962.498 L924.467,961.959 L925.131,962.028 L925.714,962.7 L926.229,961.404 L925.564,960.474 L924.788,960.665 L924.001,961.089 L923.32,960.881 L923.146,964.287 L923.297,965.542 L921.182,965.541 L919.162,964.204 L915.935,963.458 L914.228,962.518 L912.466,960.998 L912.118,960.71 L911.634,960.603 L911.062,960.352 L910.52,959.989 L910.14,959.554 L909.645,959.168 L907.583,958.797 L905.833,957.417 L905.12,956.468 L904.773,956.15 L903.813,955.928 L903.459,955.735 L903.195,955.394 L902.677,954.345 L902.514,953.829 L902.396,953.226 L902.296,952.437 L902.275,951.692 L902.316,951.049 L902.263,950.452 L901.977,949.852 L901.143,949.422 L898.145,949.391 L896.841,948.818 L896.365,946.806 L895.195,946.354 L893.933,946.669 L892.147,944.813 L892.403,943.947 L891.29,941.967 L890.026,939.69 L889.413,937.23 L888.822,935.48 L888.636,934.219 L888.335,933.218 L887.494,931.607 L888.102,930.943 L889.202,929.875 L889.543,928.969 L889.752,927.944 L889.827,926.797 L889.817,925.56 L889.673,924.602 L889.981,924.229 L890.462,924.093 L890.816,923.871 L891.138,923.108 L891.102,922.728 L890.849,922.386 L890.515,921.71 L890.496,921.388 L890.356,920.507 L890.157,919.79 L890.246,919.161 L890.54,919.266 L890.852,919.75 L891.188,920.041 L892.041,921.813 L892.645,922.37 L893.006,921.245 L892.728,920.601 L892.048,919.661 L890.741,918.267 L891.088,917.845 L891.436,917.609 L891.774,917.62 L892.093,917.961 L891.731,917.189 L890.489,917.394 L889.941,916.931 L890.373,916.507 L890.639,915.835 L891.018,914.044 L896.103,918.206 L895.705,919.742 L895.403,921.797 L895.341,923.769 L895.67,925.063 L896.558,921.267 L897.179,920.978 L898.684,921.197 L899.262,920.959 L899.33,920.585 L899.329,919.484 L899.485,919.264 L900.518,919.225 L900.912,919.347 L901.434,919.638 L902.581,920.596 L903.017,921.176 L903.686,922.634 L904.058,923.275 L907.828,927.018 L909.183,927.571 L912.23,927.709 L911.526,927.106 L908.71,926.668 L908.198,926.253 L907.572,925.528 L907.066,924.585 L906.894,923.533 L907.133,922.82 L907.645,921.931 L908.613,920.6 L909.34,919.906 L911.629,918.661 L925.8,915.066 L925.821,914.634 L916.88,915.864 L915.094,916.892 L910.566,917.71 L909.427,918.558 L906.135,923.295 L904.648,920.296 L904.446,919.55 L904.812,918.279 L905.559,917.885 L906.392,917.814 L907.022,917.516 L906.607,917.204 L906.107,916.104 L905.775,915.733 L905.235,915.731 L903.959,916.204 L903.324,916.061 L902.709,915.404 L902.062,914.283 L901.613,912.95 L901.603,911.649 L902.244,910.626 L903.324,910.089 L906.554,909.48 L907.518,908.993 L909.402,908.61 L910.005,908.274 L908.826,907.929 L907.195,908.18 L906.22,908.006 L907.013,906.396 L908.216,904.493 L908.716,903.927 L910.268,902.94 L911.85,901.298 L914.495,900.448 L916.926,898.211 L918.403,897.889 L921.357,898.138 L921.379,897.713 L917.453,897.514 L916.63,897.699 L912.9,900.376 L911.319,900.872 L910.001,900.374 L909.24,898.32 L914.113,898.193 L914.137,897.766 L913.86,897.102 L913.43,896.475 L913.059,895.719 L913.08,896.184 L912.948,896.768 L912.633,897.261 L912.085,897.565 L911.864,897.163 L911.701,896.581 L911.347,896.335 L911.236,896.487 L910.739,897.372 L910.5,897.571 L910.153,897.636 L909.805,897.602 L909.527,897.518 L909.094,897.195 L908.825,896.727 L908.663,896.107 L908.536,895.325 L908.351,896.522 L907.793,896.94 L907.091,896.832 L906.473,896.453 L906.496,896.067 L906.906,895.72 L907.191,895.178 L907.445,894.551 L907.743,893.963 L908.129,893.437 L908.535,893.014 L909.388,892.342 L911.054,891.494 L911.875,890.906 L912.589,889.916 L912.676,890.16 L912.902,890.592 L912.996,890.827 L913.166,890.298 L913.353,889.845 L913.84,888.969 L914.009,888.473 L914.137,887.201 L914.321,886.625 L915.003,885.866 L921.025,883.648 L924.769,883.27 L923.565,882.503 L918.126,883.335 L916.002,884.819 L914.854,884.903 L915.318,883.988 L915.87,883.079 L916.358,882.091 L916.602,880.914 L916.694,879.712 L916.878,878.542 L917.196,877.568 L917.691,876.934 L916.875,877.283 L916.171,878.724 L915.66,880.582 L915.421,882.174 L915.282,882.664 L914.968,883.409 L914.313,884.678 L913.589,885.761 L912.31,888.357 L911.963,888.906 L911.525,889.466 L910.148,890.481 L909.429,891.277 L908.207,892.109 L907.734,892.25 L907.195,891.886 L906.913,890.568 L906.416,890.464 L906.557,891.328 L906.379,891.433 L906.08,891.686 L905.901,891.757 L905.877,892.146 L906.171,892.297 L906.402,892.576 L906.566,892.987 L906.647,893.512 L905.07,894.732 L904.272,894.967 L903.998,893.962 L903.918,893.303 L903.68,892.846 L903.416,892.451 L903.235,891.994 L903.165,891.18 L903.213,890.563 L903.32,889.968 L903.405,889.206 L904.61,889.329 L912.033,887.781 L911.452,886.96 L910.531,886.731 L909.542,886.924 L907.924,887.751 L904.026,887.231 L903.578,887.006 L903.131,886.621 L902.862,886.21 L902.466,885.283 L902.251,885.094 L902.018,884.826 L901.725,883.641 L901.59,883.304 L901.247,883.292 L900.618,883.927 L900.231,884.08 L900.95,886.068 L901.034,886.98 L900.638,888.032 L899.915,888.754 L899.002,889.02 L898.208,888.636 L897.82,887.394 L897.899,888.898 L897.28,889.941 L895.591,891.148 L896.165,891.311 L898.528,890.004 L899.417,889.886 L900.08,890.119 L900.629,890.725 L902.093,893.011 L902.232,893.671 L902.188,895.16 L901.749,895.376 L899.724,895.489 L899.098,895.169 L898.71,895.864 L898.248,895.928 L897.194,895.523 L896.964,895.647 L896.932,895.981 L896.936,896.407 L896.802,896.782 L896.366,897.183 L896.205,897.37 L895.825,897.668 L894.419,898.364 L894.031,898.287 L894.006,897.5 L894.341,896.812 L894.832,896.348 L895.261,896.262 L895.291,895.795 L894.645,894.47 L894.274,892.209 L893.721,890.533 L892.865,890.928 L893.156,891.066 L893.394,891.372 L893.593,891.843 L893.274,893.047 L893.052,895.723 L892.713,896.744 L892.05,897.745 L891.352,898.573 L890.551,899.028 L889.596,898.903 L889.016,898.254 L888.447,897.017 L888.16,895.57 L888.457,894.317 L888.863,893.15 L888.994,891.754 L888.772,890.934 L888.1,891.495 L887.741,892.504 L887.61,893.745 L887.546,896.416 L887.435,897.166 L887.236,897.627 L886.953,897.739 L886.6,897.425 L886.425,896.916 L886.369,896.214 L886.338,894.84 L886.027,893.876 L885.95,893.173 L886.305,891.895 L886.338,891.102 L886.264,890.396 L886.114,890.08 L885.723,890.108 L885.593,890.324 L885.29,893.652 L885.303,893.916 L884.17,894.42 L882.41,889.314 L881.491,888.916 L881.377,887.99 L881.328,886.811 L881.358,884.63 L881.313,884.313 L881.232,884.065 L881.236,883.785 L881.449,883.351 L881.85,882.988 L882.294,882.888 L883.222,883.004 L883.25,882.615 L881.978,882.063 L882.008,881.639 L882.695,881.346 L883.216,880.884 L885.067,878.422 L885.62,876.745 L886.211,876.157 L886.979,875.619 L887.738,875.355 L888.31,875.599 L888.502,876.322 L888.619,878.365 L888.945,879.066 L889.093,878.036 L889.541,877.206 L889.553,876.536 L889.319,875.664 L888.854,874.918 L888.831,874.35 L889.292,873.167 L890.182,872.033 L891.951,870.273 L891.75,870.91 L891.656,871.149 L891.991,871.514 L892.155,871.77 L892.246,872.043 L892.281,873.139 L892.163,874.439 L891.869,875.751 L891.388,876.885 L891.189,877.541 L891.214,878.248 L891.311,878.973 L891.32,879.68 L891.228,880.444 L891.133,880.767 L890.334,881.33 L890.199,881.526 L890.013,882.057 L889.933,882.161 L889.362,882.069 L889.069,882.101 L888.834,882.302 L888.641,882.807 L888.457,883.601 L888.438,883.949 L888.417,884.343 L888.672,884.686 L888.863,884.586 L889.276,884.067 L889.51,883.883 L891.621,883.08 L891.988,882.718 L892.382,881.843 L892.458,881.593 L892.81,880.43 L892.984,879.097 L892.597,878.481 L892.543,877.841 L892.968,876.448 L893.581,875.018 L894.071,874.256 L894.602,874.141 L895.46,874.633 L896.011,874.805 L898.178,874.937 L899.02,875.386 L899.244,876.391 L899.039,877.648 L898.597,878.852 L899.469,878.897 L899.838,878.794 L900.144,878.517 L900.226,878.155 L900.261,877.025 L900.374,876.588 L901.121,875.952 L902.116,875.861 L903.982,876.179 L904.063,874.859 L901.12,875.72 L900.331,875.536 L899.834,874.68 L900.186,873.901 L901.599,872.617 L901.812,872.217 L901.965,871.795 L902.2,871.467 L904.14,871.191 L904.514,871.037 L904.759,870.577 L904.793,870.029 L904.851,869.574 L905.149,869.402 L905.44,869.582 L905.674,869.973 L905.865,870.402 L906.054,870.698 L907.657,871.365 L909.345,870.869 L913.459,867.86 L913.752,867.511 L913.968,867.34 L914.422,867.341 L914.64,867.134 L915.093,866.221 L915.238,866.04 L917.699,865.365 L918.371,864.532 L916.213,865.083 L912.212,867.298 L909.387,869.643 L908.845,869.903 L908.262,869.967 L907.053,869.815 L906.488,869.408 L906.183,868.57 L906.252,867.598 L906.583,867.197 L906.972,866.906 L907.187,866.276 L907.157,864.73 L907.243,864.154 L907.59,863.517 L906.871,863.82 L906.423,864.673 L906.172,865.843 L906.056,867.082 L905.607,867.76 L904.64,868.338 L903.597,868.721 L902.921,868.809 L902.229,868.391 L901.681,867.668 L900.816,865.904 L900.104,865.097 L899.471,864.875 L899.258,864.462 L899.801,863.06 L899.212,863.644 L898.448,864.201 L897.761,864.253 L897.387,863.341 L898.159,863.281 L899.008,862.935 L900.548,861.787 L900.574,860.046 L901.662,858.806 L903.183,858.035 L904.49,857.705 L907.828,857.545 L908.041,857.374 L908.246,856.948 L908.399,856.442 L908.469,856.048 L908.34,855.809 L907.436,854.963 L909.614,851.334 L912.497,849.339 L918.646,847.805 L918.671,847.341 L917.897,847.392 L917.785,846.752 L917.872,845.87 L917.719,845.189 L917.419,845.258 L916.958,845.646 L916.531,846.16 L916.332,846.613 L915.827,847.419 L914.736,848.01 L913.556,848.343 L912.782,848.385 L912.955,846.721 L912.348,844.527 L912.164,842.653 L914.33,841.702 L914.789,841.024 L915.179,840.08 L915.695,839.061 L916.631,837.988 L916.796,837.403 L916.479,836.541 L918.277,836.117 L918.883,835.813 L919.233,835.409 L919.964,834.344 L920.356,834.143 L920.725,834.086 L920.811,833.897 L920.816,833.622 L920.934,833.318 L920.918,833.294 L921.018,833.077 L921.189,832.786 L921.386,832.524 L922.402,832.018 L923.479,831.737 L924.246,831.371 L925.499,829.478 L926.278,828.847 L927.382,828.692 L928.501,828.884 L930.504,829.948 L928.834,833.172 L928.102,835.197 L928.23,836.712 L927.854,837.491 L926.829,838.793 L926.597,839.408 L926.075,844.164 L925.686,846.207 L925.547,847.3 L925.465,848.875 L925.584,850.717 L925.799,850.727 L925.987,847.33 L926.235,845.547 L926.891,844.397 L927.968,841.827 L928.118,840.987 L928.128,840.346 L927.908,838.83 L928.752,837.473 L930.008,833.615 L930.822,832.096 L931.386,831.614 L932.346,831.104 L932.817,830.484 L933.944,828.24 L934.462,827.569 L935.093,827.01 L935.706,826.666 L940.086,825.643 L942.844,826.661 L944.282,826.624 L945.196,825.008 L938.837,824.739 L938.856,824.347 L939.094,824.327 L939.387,824.215 L939.633,824.02 L939.749,823.745 L939.903,823.075 L940.213,822.874 L940.561,822.854 L940.855,822.729 L942.678,819.776 L941.781,819.631 L940.571,820.495 L939.371,821.815 L938.511,823.014 L937.449,824.84 L936.924,825.265 L934.806,825.841 L933.641,826.676 L931.629,828.72 L931.043,828.94 L930.23,829.023 L929.481,828.82 L929.091,828.172 L929.477,826.849 L931.796,824.75 L932.518,823.633 L931.034,823.935 L928.217,826.931 L926.679,827.627 L925.72,827.775 L923.957,828.538 L923.007,828.683 L922.169,829.027 L920.484,830.702 L919.76,831.158 L918.942,831.049 L918.214,830.481 L917.61,829.56 L917.15,828.377 L916.865,826.266 L916.683,825.828 L916.337,825.817 L916.246,826.4 L916.313,827.176 L916.455,827.741 L916.654,828.241 L917.168,829.934 L917.767,830.641 L918.012,831.109 L917.932,831.663 L917.48,831.952 L916.48,831.305 L915.611,831.874 L914.609,831.84 L914.153,832.139 L914.605,832.553 L916.27,833.071 L915.938,834.001 L914.98,835.131 L914.594,836.011 L914.809,836.68 L914.95,837.011 L915.154,837.32 L914.809,838.463 L914.393,839.273 L913.894,839.722 L913.282,839.785 L912.632,839.405 L911.246,838.144 L910.618,837.925 L909.961,838.362 L909.57,839.242 L909.28,840.305 L908.925,841.291 L907.737,842.889 L907.05,844.213 L906.777,844.584 L906.242,844.731 L905.752,844.489 L905.334,844.475 L905.003,845.299 L904.993,846.08 L905.426,849.209 L905.306,851.495 L904.655,853.159 L903.607,854.096 L902.319,854.198 L902.433,853.769 L902.59,853.317 L902.009,853.017 L901.389,855.037 L900.878,854.769 L899.893,853.807 L899.774,853.382 L899.745,852.873 L899.627,852.425 L899.445,852.068 L899.243,851.834 L899.554,850.53 L899.917,850.19 L901.021,850.657 L901.679,850.689 L902.301,850.405 L902.809,849.74 L903.098,848.644 L901.816,848.567 L901.622,848.691 L901.496,848.976 L901.327,849.261 L901.005,849.372 L900.666,849.155 L900.488,848.701 L900.371,848.251 L900.089,847.867 L900.364,847.516 L900.94,847.021 L901.108,846.145 L900.482,844.359 L900.63,843.364 L901.187,842.755 L903.314,841.777 L903.341,841.351 L902.318,841.356 L901.062,841.754 L899.841,842.422 L898.922,843.26 L898.841,843.561 L898.752,844.511 L898.583,844.913 L898.303,845.018 L897.674,844.834 L896.877,845.498 L896.254,845.288 L895.708,844.676 L895.435,843.899 L895.963,843.665 L896.441,843.08 L896.614,842.388 L896.214,841.812 L896.245,841.345 L897.379,841.307 L899.645,840.132 L900.853,839.882 L900.878,839.496 L898.262,839.422 L897.428,839.671 L898.412,838.45 L898.798,837.756 L898.673,837.22 L898.101,837.208 L897.619,837.884 L897.151,838.731 L896.616,839.233 L896.795,838.144 L897.204,837.624 L897.624,837.347 L897.834,836.977 L898.04,836.437 L898.977,834.663 L899.277,834.226 L899.34,833.883 L899.554,833.244 L899.826,832.572 L900.07,832.139 L900.612,831.724 L901.739,831.278 L902.19,830.77 L902.371,829.963 L901.787,830.096 L901.06,830.599 L900.805,830.9 L900.327,831.031 L899.907,831.416 L899.532,831.953 L899.191,832.551 L899.171,832.191 L899.077,831.614 L899.062,831.259 L898.508,832.089 L897.858,833.355 L897.305,834.771 L897.025,836.053 L896.692,836.511 L895.287,839.69 L895.128,840.462 L894.795,840.688 L893.388,842.232 L892.188,843.101 L891.79,843.667 L891.584,844.319 L891.383,845.139 L891.12,845.586 L890.455,844.704 L889.412,843.94 L889.876,843.619 L890.084,843.556 L890.116,843.091 L888.537,842.246 L888.072,841.672 L888.39,841.13 L888.785,840.802 L889.651,840.494 L889.677,840.111 L888.73,839.766 L886.485,840.511 L885.628,840.226 L888.112,838.004 L888.595,837.004 L887.54,836.979 L884.306,835.899 L884.07,835.575 L884.098,834.865 L884.278,834.067 L884.479,833.499 L884.833,833.218 L885.339,833.206 L886.287,833.388 L885.827,832.602 L885.447,832.24 L884.208,831.946 L883.871,831.796 L883.711,831.305 L883.958,830.275 L885.02,829.065 L887.739,829.345 L888.778,828.854 L888.346,828.509 L887.454,826.584 L887.791,825.61 L887.944,825.375 L887.239,823.877 L887.345,822.919 L888.79,820.938 L889.605,820.623 L890.484,821.555 L891.296,822.812 L891.896,823.457 L892.23,823.724 L893.671,826.024 L894.476,826.779 L896.014,827.782 L897.398,828.223 L898.785,828.169 L899.953,827.388 L899.901,826.319 L900.457,825.818 L901.227,825.499 L901.809,824.976 L902.04,824.167 L902.102,823.249 L902.058,821.106 L902.144,819.257 L902.411,816.751 L902.892,814.508 L903.62,813.469 L903.055,812.749 L903.211,811.608 L903.68,810.439 L904.075,809.647 L903.365,810.338 L902.744,810.542 L901.377,810.375 L901.343,809.924 L901.703,808.943 L902.195,807.941 L902.535,807.414 L901.957,807.04 L901.465,807.82 L900.973,808.898 L900.384,809.422 L900.073,809.576 L899.436,810.362 L899.13,810.626 L898.699,810.713 L897.344,810.513 L896.983,811.166 L897.176,814.144 L896.845,814.801 L896.159,814.859 L894.214,815.512 L891.886,817.509 L889.569,818.584 L888.781,818.628 L888.051,818.317 L887.246,817.59 L886.29,816.286 L885.498,814.904 L883.729,812.654 L883.402,811.767 L884.566,812.016 L885.551,812.787 L887.429,815.035 L887.624,814.661 L885.796,811.488 L883.624,808.746 L884.13,808.774 L885.48,809.341 L884.938,808.019 L884.014,807.654 L882.99,807.478 L882.168,806.745 L880.879,804.263 L880.108,803.326 L879.187,802.872 L879.991,802.389 L880.903,802.452 L881.757,802.948 L883.06,804.47 L884.652,804.91 L885.767,806.434 L886.58,807.053 L889.664,808.503 L890.361,809.166 L890.293,810.056 L890.39,811.421 L890.789,812.734 L891.394,813.715 L892.125,814.063 L891.867,813.797 L891.692,813.421 L891.605,812.939 L891.61,812.322 L892.234,812.898 L893.342,814.301 L893.988,814.618 L893.566,813.688 L892.785,812.606 L892.245,811.645 L892.545,811.061 L892.981,811.296 L893.9,812.477 L894.134,812.487 L894.09,811.804 L893.773,811.363 L893.523,810.834 L893.678,809.889 L892.739,809.267 L892.482,808.919 L892.368,808.463 L892.339,807.905 L892.282,807.435 L892.063,807.222 L890.677,807.138 L889.903,806.896 L889.113,806.055 L888.813,806.004 L888.587,805.798 L888.547,805.023 L888.668,804.81 L889.458,804.052 L890.137,803.667 L891.547,803.353 L892.263,802.957 L891.961,801.265 L892.578,800.491 L894.923,799.612 L895.441,799.752 L895.939,800.036 L896.446,800.196 L896.917,800.14 L897.354,799.959 L898.193,799.416 L897.28,799.271 L896.481,798.756 L896.261,798.019 L897.083,797.205 L896.334,797.315 L894.025,798.758 L892.44,798.917 L891.734,799.477 L891.356,800.937 L890.511,802.239 L888.849,802.48 L887.307,801.713 L886.812,799.989 L886.604,801.466 L886.069,802.428 L885.498,802.567 L885.181,801.587 L885.656,801.151 L885.079,800.927 L883.63,801.031 L883.079,800.373 L882.917,799.627 L882.704,797.746 L882.49,796.908 L882.334,796.514 L882.264,796.219 L882.261,795.831 L882.306,795.164 L882.201,794.868 L881.691,794.052 L881.585,793.599 L881.903,792.895 L882.567,792.781 L883.321,793.003 L883.909,793.334 L884.49,793.88 L884.613,794.436 L884.312,796.589 L884.361,797.681 L884.591,798.383 L884.872,798.41 L885.064,797.496 L885.014,794.589 L885.143,793.849 L885.665,793.197 L886.207,793.005 L886.587,792.616 L886.6,791.38 L886.395,791.557 L886.304,791.681 L886.344,791.753 L885.973,791.844 L885.085,792.41 L884.605,792.527 L882.983,791.985 L882.668,791.731 L882.499,790.74 L882.56,789.584 L882.769,788.498 L883.074,787.712 L884.795,786.14 L886.829,786.083 L888.811,786.968 L891.103,788.768 L892.014,789.224 L893.923,789.699 L895.31,789.631 L895.688,789.814 L895.822,790.368 L895.862,791.206 L895.999,791.785 L896.419,791.568 L896.617,791.076 L896.549,790.537 L896.404,789.931 L896.352,789.233 L896.59,788.407 L897.024,788.066 L897.49,787.899 L898.731,786.952 L899.919,787.271 L901.135,787.892 L902.096,788.12 L902.126,787.656 L901.281,786.916 L900.915,786.48 L901.061,786.306 L902.195,786.04 L902.732,786.03 L903.246,786.441 L903.972,785.816 L905.22,785.194 L906.386,784.981 L906.858,785.608 L906.454,787.32 L906.401,788.117 L906.826,787.745 L907.181,787.236 L907.604,786.788 L908.049,786.47 L908.477,786.366 L908.214,785.655 L908.118,785.451 L909.122,784.929 L910.022,784.737 L912.562,784.945 L913.585,785.367 L914.175,785.415 L916.268,785.064 L917.268,785.343 L919.137,786.34 L920.171,786.595 L921.014,787.098 L921.341,788.161 L921.14,789.298 L920.374,790.022 L921.36,789.987 L921.764,789.688 L922.213,789.034 L922.566,788.762 L923.1,788.668 L923.64,788.722 L923.998,788.895 L924.296,789.616 L924.028,790.345 L923.539,791.196 L923.185,792.307 L923.705,792.423 L924.226,792.361 L924.185,791.468 L924.361,790.8 L924.663,790.276 L925.644,788.941 L926.43,788.164 L927.295,787.62 L929.038,787.34 L929.788,786.903 L930.466,786.01 L931.131,784.551 L931.457,783.07 L931.684,782.544 L932.049,782.656 L932.722,783.434 L933.079,783.723 L937.255,784.363 L938.736,785.116 L939.064,785.414 L939.296,785.844 L939.369,786.45 L939.338,787.219 L939.374,787.824 L939.548,788.221 L939.92,788.375 L941.704,788.247 L942.207,788.475 L942.465,788.974 L942.919,791.103 L943.375,792.219 L943.566,792.83 L943.626,793.484 L943.401,794.013 L942.908,794.626 L942.365,795.135 L941.375,795.621 L939.27,797.778 L942.138,796.135 L943.557,795.844 L944.774,796.308 L946.633,797.959 L947.074,798.768 L947.08,799.677 L946.159,802.804 L946.981,801.978 L947.832,800.628 L948.333,798.987 L948.091,797.298 L947.832,797.127 L946.26,796.687 L946.056,796.406 L945.736,795.706 L944.717,792.425 L944.444,791.864 L944.128,791.373 L943.87,790.818 L943.699,789.101 L943.534,788.59 L943.575,788.23 L944.112,787.739 L945.42,787.428 L948.666,787.37 L949.599,786.217 L950.048,785.626 L950.459,785.546 L952.918,786.272 L953.24,786.574 L953.903,787.451 L954.82,787.944 L956.83,788.239 L956.848,787.774 L956.296,787.7 L955.887,787.528 L955.551,787.246 L954.517,785.849 L954.217,785.543 L954.651,784.969 L956.071,784.068 L956.383,783.685 L956.775,782.942 L957.621,782.54 L962.238,782.356 L963.745,781.872 L965.004,780.337 L963.949,780.44 L963.126,780.743 L962.092,781.473 L957.477,782.25 L955.636,782.129 L954.759,781.631 L954.413,780.626 L954.407,779.917 L954.318,778.693 L954.309,778.059 L954.228,777.316 L953.988,777.017 L953.717,776.841 L953.544,776.481 L953.652,775.165 L954.965,774.101 L955.335,772.96 L955.503,771.568 L955.845,770.879 L957.109,769.824 L957.608,769.057 L958.09,768.067 L958.606,767.221 L959.2,766.876 L959.928,766.691 L960.698,766.166 L961.381,765.371 L961.858,764.406 L959.292,765.873 L954.832,770.167 L954.147,770.332 L953.011,769.759 L952.362,769.645 L952.87,770.451 L953.371,771.05 L953.642,771.701 L953.492,772.677 L951.571,775.108 L951.385,775.775 L951.949,776.672 L952.572,777.291 L953.058,778.133 L953.304,780.829 L953.655,781.605 L954.168,782.138 L954.747,782.539 L953.463,783.973 L953.023,784.216 L952.629,784.252 L952.276,784.154 L951.589,783.738 L951.382,783.54 L950.952,782.996 L950.78,782.852 L950.301,782.833 L949.11,783.274 L948.675,783.623 L948.587,783.838 L948.512,784.189 L948.397,784.574 L948.201,784.881 L947.879,785.11 L943.963,785.985 L942.823,786.769 L942.193,787.001 L941.66,786.785 L941.174,785.415 L941.961,784.447 L943.166,783.593 L943.929,782.572 L942.273,783.529 L941.399,783.817 L940.555,783.704 L938.055,782.442 L933.691,781.396 L930.548,779.403 L930.967,778.41 L932.134,777.423 L932.664,776.706 L933.952,773.546 L934.21,773.176 L934.922,772.363 L935.075,772.108 L935.135,771.648 L935.38,770.445 L935.48,770.203 L935.65,769.909 L935.836,769.268 L935.836,768.668 L935.462,768.495 L935.316,768.702 L933.209,774.218 L932.098,776.091 L930.888,776.854 L930.544,776.629 L930.123,775.674 L929.822,775.446 L929.575,775.64 L929.243,776.097 L928.974,776.602 L928.907,776.95 L929.278,777.91 L929.54,778.885 L929.538,779.218 L929.491,779.641 L929.55,780.12 L929.861,780.647 L927.873,780.105 L927.683,780.325 L927.986,780.714 L928.346,780.961 L928.639,781.327 L928.733,782.067 L928.748,782.92 L928.8,783.579 L928.729,784.169 L928.398,784.809 L928.024,785.147 L926.441,786.07 L924.555,786.015 L921.873,785.165 L919.564,783.589 L918.792,781.358 L917.364,782.802 L915.605,783.413 L913.806,783.22 L912.264,782.272 L913.688,781.426 L914.37,780.833 L914.71,779.85 L911.668,781.815 L909.61,782.424 L907.476,782.461 L906.876,781.847 L906.896,780.71 L906.986,779.875 L906.58,780.195 L906.329,780.758 L906.124,781.876 L905.818,782.362 L905.416,782.399 L904.871,782.254 L904.373,782.29 L904.123,782.86 L903.525,783.066 L899.903,783.205 L899.262,783.488 L897.241,785.675 L895.607,786.486 L894.181,786.343 L892.973,785.236 L892.003,783.154 L891.435,783.929 L890.767,784.276 L889.087,784.318 L888.909,784.451 L888.742,784.425 L888.482,784.009 L888.236,783.33 L888.096,783.073 L887.828,782.871 L888.173,782.215 L888.733,782.212 L889.372,782.363 L889.954,782.163 L889.278,781.852 L888.629,781.369 L888.003,781.186 L887.395,781.774 L886.84,781.802 L884.902,780.864 L884.468,780.5 L885.044,778.991 L886.815,779.239 L889.881,780.874 L889.832,780.592 L889.769,779.877 L889.726,779.584 L890.8,779.656 L889.322,777.852 L888.476,777.175 L887.642,776.878 L884.01,777.132 L883.081,776.559 L883.689,776.298 L884.399,776.285 L885.042,776.121 L885.467,775.402 L884.42,775.817 L883.868,775.86 L883.474,775.475 L883.233,774.831 L883.051,774.528 L882.761,774.517 L882.18,774.741 L882.53,773.945 L883.054,773.378 L884.201,772.711 L884.934,772.474 L885.673,772.424 L885.985,772.308 L886.517,771.759 L886.679,771.638 L888.007,771.786 L888.631,771.528 L889.313,769.802 L890.044,769.645 L890.858,769.907 L891.473,770.259 L891.933,770.652 L892.293,771.135 L892.562,771.723 L892.762,772.443 L892.985,774.59 L893.235,775.524 L893.67,775.318 L893.814,774.53 L893.797,773.365 L893.677,772.205 L893.527,771.462 L893.114,770.746 L892.144,770.04 L891.721,769.42 L897.353,768.584 L903.126,769.301 L902.177,768.358 L900.829,768.124 L888.985,768.735 L888.538,768.989 L887.815,769.676 L886.72,770.288 L885.646,770.547 L885.011,770.199 L884.735,769.476 L884.189,768.94 L883.933,768.025 L884.099,767.092 L884.55,766.206 L885.103,765.618 L885.584,765.576 L885.856,766.214 L885.894,766.981 L886.01,767.372 L886.514,766.882 L886.577,766.565 L886.586,765.526 L886.641,765.183 L886.876,764.993 L887.88,764.882 L887.874,764.589 L887.859,764.404 L887.818,764.226 L887.743,763.976 L887.117,764.59 L886.472,764.744 L885.197,764.695 L883.969,764.939 L882.728,764.906 L882.757,764.521 L886.168,763.015 L889.389,762.831 L890.19,762.009 L889.999,761.527 L891.107,761.883 L892.497,762.762 L893.807,763.219 L894.679,762.311 L898.93,764.661 L899.975,764.827 L900.749,764.449 L900.575,763.977 L901.192,763.674 L902.487,763.58 L903.11,763.28 L903.414,762.876 L903.75,762.329 L904.154,761.85 L904.659,761.664 L906.029,761.636 L906.719,761.836 L907.187,762.419 L907.632,763.734 L907.981,764.04 L908.641,764.044 L908.668,763.618 L908.016,763.101 L907.075,761.333 L906.369,760.916 L904.646,760.849 L903.814,761.047 L903.001,761.562 L902.072,762.624 L901.797,762.77 L900.262,762.683 L899.267,763.287 L898.331,763.301 L897.674,762.782 L897.852,761.622 L896.582,761.056 L893.326,761.294 L892.282,760.246 L890.101,759.19 L888.989,758.215 L889.288,757.242 L889.149,756.804 L889.134,756.803 L889.168,756.344 L889.657,756.357 L890.971,756.929 L891.345,756.901 L892.344,756.561 L895.688,756.784 L895.717,756.358 L894.917,756.29 L890.26,754.458 L889.541,753.802 L895.252,754.188 L894.647,754.484 L894.409,754.524 L894.379,754.945 L897.131,755.243 L897.898,754.753 L895.686,753.508 L893.324,752.778 L894.879,752.064 L895.555,751.207 L893.851,751.145 L892.55,751.786 L892.165,751.882 L885.111,751.285 L884.41,750.878 L884.261,750.516 L883.836,748.893 L883.756,748.187 L883.991,747.633 L884.369,747.247 L884.701,747.046 L884.512,746.467 L884.629,745.823 L884.968,745.366 L885.444,745.354 L885.361,745.096 L885.318,744.894 L885.249,744.711 L885.08,744.51 L885.526,744.022 L886.001,743.791 L886.993,743.795 L887.5,743.943 L887.976,744.249 L888.106,744.457 L888.125,744.72 L888.107,744.969 L888.135,745.116 L888.326,745.167 L888.758,745.083 L888.929,745.171 L889.382,745.959 L889.493,746.105 L889.827,746.199 L890.091,746.177 L890.767,745.726 L889.894,744.522 L889.65,743.981 L890.345,743.585 L891.635,743.068 L887.855,743.384 L888.351,742.54 L889.145,741.867 L890.011,741.361 L893.465,740.166 L894.271,739.553 L894.14,739.463 L894.067,739.363 L893.899,739.062 L895.092,738.798 L896.336,738.835 L895.856,738.421 L895.349,738.316 L895.014,738.128 L895.027,737.464 L895.248,737.089 L897.356,734.907 L898.335,734.62 L901.074,734.998 L902.057,735.624 L902.581,735.782 L903.791,735.675 L904.367,735.754 L905.397,736.46 L906.553,736.651 L906.987,737.126 L907.644,738.301 L908.388,739.171 L907.458,739.608 L906.346,739.586 L905.303,739.817 L904.585,741.035 L905.41,741.066 L908.115,739.972 L913.726,739.372 L914.249,739.731 L914.379,740.219 L914.247,740.869 L914.026,741.581 L913.597,743.61 L912.673,745.545 L912.6,746.294 L913.159,745.507 L913.596,745.048 L913.995,744.491 L914.433,743.371 L914.849,741.328 L915.145,740.653 L915.729,740.419 L916.254,740.711 L917.614,742.22 L918.685,744.12 L919.388,744.719 L921.095,745.023 L920.84,744.139 L920.388,743.781 L919.884,743.545 L919.47,743.038 L919.112,742.321 L918.65,741.68 L918.149,741.209 L917.648,740.993 L917.676,740.529 L924.595,740.905 L925.25,741.211 L926.549,742.48 L927.264,742.792 L929.93,743.102 L930.798,742.97 L931.374,742.576 L931.761,741.949 L932.352,740.482 L932.921,739.64 L933.586,739.143 L934.317,738.907 L935.103,738.868 L935.899,739.107 L937.446,740.062 L938.19,740.297 L938.455,740.523 L938.756,740.962 L939.076,741.251 L939.377,741.028 L939.854,740.129 L940.101,739.787 L940.376,739.54 L940.397,739.114 L939.906,739.114 L938.966,739.43 L938.554,739.243 L938.143,738.95 L936.754,738.521 L936.618,738.329 L936.455,737.772 L936.29,737.645 L936.1,737.71 L935.804,738.016 L935.651,738.083 L932.599,738.627 L932.053,739.354 L931.63,740.304 L931.111,741.238 L930.072,741.814 L923.715,739.553 L918.847,739.777 L918.454,739.582 L917.195,738.364 L916.646,738.242 L914.951,738.237 L913.383,737.398 L912.774,737.258 L912.339,737.367 L911.451,737.903 L911.013,738.007 L910.824,738.131 L910.425,738.682 L910.242,738.819 L910.009,738.711 L908.147,736.595 L908.178,736.129 L914.245,736.104 L915.105,735.726 L915.131,735.3 L905.808,734.237 L902.055,733.167 L900.036,733.847 L899.858,733.586 L899.695,733.213 L899.404,733.016 L897.05,733.519 L895.597,734.293 L894.84,734.464 L893.223,734.072 L892.39,734.143 L891.652,734.867 L891.193,735.08 L890.611,734.711 L889.598,733.678 L890.108,732.875 L890.533,732.031 L891.068,730.57 L891.082,730.36 L891.552,730.099 L891.707,730.4 L891.776,730.897 L892.012,731.239 L892.316,731.183 L892.523,730.892 L892.693,730.583 L892.894,730.484 L893.134,730.669 L893.491,731.241 L893.621,731.349 L893.936,731.225 L894.004,730.962 L894.06,730.584 L894.359,730.12 L895.565,729.325 L895.983,728.719 L896.347,727.261 L897.162,727.774 L898.259,728.045 L898.967,727.844 L898.611,726.945 L898.208,726.743 L897.297,726.68 L896.935,726.213 L896.571,725.364 L896.165,724.644 L893.689,722.343 L890.799,721.716 L890.353,720.831 L890.641,720.836 L890.887,720.762 L891.389,720.476 L890.998,719.809 L890.152,718.955 L889.922,718.234 L890.097,718.172 L890.411,717.932 L890.575,717.852 L890.482,717.654 L890.33,717.177 L890.238,716.975 L890.359,717.019 L890.417,716.952 L890.47,716.563 L890.811,716.432 L891.672,716.647 L891.818,716.621 L892.123,716.714 L892.274,716.688 L892.291,716.55 L892.311,715.983 L892.337,715.837 L893.061,715.276 L893.408,715.412 L894.157,716.619 L895.015,717.378 L895.327,717.949 L895.538,718.241 L895.767,718.225 L895.933,718.293 L895.96,718.845 L895.884,719.237 L895.662,720.052 L895.649,720.377 L896.925,722.296 L898.994,723.568 L900.659,725.344 L900.725,728.788 L900.922,728.656 L901.332,728.382 L901.825,727.556 L902.955,725.041 L903.057,724.673 L903.235,724.425 L903.661,724.314 L903.958,724.44 L904.493,725.039 L904.814,725.204 L904.113,723.816 L899.944,722.332 L899.392,721.873 L899.147,721.396 L899.213,720.797 L899.99,719.02 L900.246,718.171 L900.601,717.552 L901.3,717.292 L902.01,717.496 L903.276,718.486 L906.114,719.237 L906.607,719.726 L906.805,720.43 L906.905,721.282 L907.091,722.016 L907.988,722.598 L909.556,724.218 L909.045,723.164 L908.113,722.013 L907.383,720.652 L907.463,718.962 L908.011,718.468 L908.897,718.416 L910.518,718.682 L910.993,718.558 L912.614,717.356 L913.179,717.201 L913.887,717.497 L914.538,718.116 L914.93,718.943 L915.018,719.848 L914.946,720.704 L914.705,722.579 L914.206,724.183 L912.435,726.729 L912.1,728.679 L914.15,726.206 L915.054,724.622 L915.783,721.374 L916.26,721.462 L918.267,724.06 L918.539,724.899 L918.342,726.008 L919.492,725.484 L920.59,725.227 L923.032,725.449 L927.629,726.936 L927.652,726.509 L919.609,723.519 L919.262,723.166 L918.538,721.66 L918.314,721.312 L917.948,720.947 L916.774,718.662 L915.25,717.656 L915.047,717.066 L915.274,715.995 L915.689,715.775 L916.196,716.064 L917.287,717.009 L917.878,717.302 L919.254,717.52 L918.853,717.144 L918.275,716.844 L917.181,716.548 L916.924,716.218 L916.751,715.528 L916.492,714.903 L915.988,714.775 L917.014,712.704 L918.556,711.166 L920.245,710.217 L921.727,709.933 L925.176,708.07 L927.067,707.483 L927.814,708.748 L927.918,709.186 L928.682,711.158 L929.165,713.317 L929.701,714.214 L931.004,715.44 L931.602,716.47 L933.403,720.397 L933.647,721.566 L933.487,722.413 L932.616,724.201 L934.513,722.625 L934.264,721.08 L932.045,715.172 L929.979,711.438 L929.485,709.821 L929.353,708.062 L929.729,706.675 L930.755,706.147 L932.093,705.96 L932.687,706.066 L933.344,706.661 L933.618,707.102 L934.424,708.888 L934.399,707.88 L934.462,706.937 L934.396,706.209 L933.984,705.841 L934.007,705.415 L934.596,705.014 L937.142,704.061 L938.351,703.183 L938.932,703.093 L939.345,703.379 L940.03,704.242 L943.794,705.427 L943.805,706.396 L943.513,707.749 L943.453,709.31 L943.774,710.224 L946.321,713.927 L946.591,714.602 L946.766,715.319 L946.831,715.833 L946.943,718.624 L946.834,719.532 L946.533,720.259 L944.834,723.003 L944.671,723.967 L945.118,725.207 L946.028,726.057 L947.013,726.007 L949.37,725.244 L950.13,724.665 L950.582,724.541 L950.951,724.663 L951.953,725.491 L950.906,723.667 L947.297,725.417 L945.841,724.577 L945.714,723.539 L946.137,722.728 L947.401,721.249 L947.742,720.224 L947.82,719.027 L947.632,715.45 L947.708,715.087 L948.012,714.662 L948.42,714.492 L950.23,714.29 L952.976,716.248 L953.685,715.941 L954.32,715.04 L955.165,714.463 L956.03,714.451 L956.74,715.205 L957.181,715.997 L957.774,716.819 L958.409,717.453 L958.965,717.656 L958.983,717.192 L957.849,715.053 L956.657,714.246 L956.391,713.916 L955.567,713.411 L951.667,713.964 L947.617,712.724 L945.798,711.299 L944.671,708.471 L945.613,708.51 L946.06,708.407 L946.488,708.123 L943.1,703.111 L942.768,702.804 L942.163,702.632 L940.931,702.682 L940.364,702.343 L940.888,702.017 L941.4,701.539 L938.473,701.459 L932.308,704.084 L929.785,704.313 L928.685,703.869 L928.241,703.396 L927.597,702.363 L927.195,702.044 L926.603,702.087 L925.984,702.362 L925.369,702.451 L924.808,701.917 L925.625,701.412 L928.23,701.282 L930.515,700.316 L930.994,699.929 L931.675,699.574 L933.97,699.768 L935.058,700.368 L938.067,700.489 L938.088,700.063 L937.44,699.799 L936.387,698.696 L935,699.244 L932.364,698.935 L932.387,698.509 L934.364,698.089 L934.985,698.17 L934.095,697.541 L926.225,697.723 L926.38,697.248 L926.398,697.137 L926.343,697.108 L926.273,696.876 L926.54,696.025 L926.389,695.547 L926.11,695.209 L925.996,694.763 L926.193,694.13 L926.547,693.648 L927.303,693.085 L928.151,692.758 L934.482,692.638 L935.11,692.948 L935.758,693.72 L936.218,694.833 L936.285,696.14 L936.483,696.15 L936.853,694.429 L936.889,693.655 L936.639,693.129 L936.933,692.511 L937.25,692.32 L937.506,692.546 L937.627,693.177 L938.863,691.718 L940.165,690.703 L940.937,690.51 L941.531,690.805 L944.417,693.983 L944.92,694.831 L944.937,695.19 L944.823,695.452 L944.871,695.625 L945.381,695.705 L945.557,695.467 L945.501,694.905 L945.338,694.308 L945.177,693.95 L945.4,693.686 L945.831,692.703 L945.481,692.616 L945.181,692.359 L944.925,691.941 L944.704,691.372 L946.018,690.685 L950.653,691.165 L950.389,691.908 L949.96,692.417 L949.46,692.716 L948.997,692.839 L949.371,693.693 L950.668,693.585 L950.906,694.408 L950.971,695.394 L951.343,697.277 L951.34,698.517 L951.515,698.525 L951.664,697.938 L952.121,696.48 L952.227,695.994 L952.147,695.541 L951.777,694.756 L951.694,694.44 L952.481,692.596 L952.903,692.715 L953.936,693.504 L956.993,694.288 L958.4,695.099 L960.187,697.963 L961.885,697.956 L962.698,698.488 L962.96,699.366 L963.092,700.387 L963.338,700.85 L963.929,700.06 L964.097,699.383 L964.161,698.61 L964.33,697.988 L964.794,697.745 L970.344,696.647 L969.416,695.909 L962.403,696.899 L961.19,696.337 L960.601,695.729 L960.085,694.841 L959.839,693.785 L960.042,692.652 L960.831,691.949 L961.768,692.303 L962.386,693.552 L962.219,695.521 L962.881,695.773 L963.262,695.133 L963.319,693.997 L962.739,691.72 L962.79,690.815 L963.18,690.181 L964.584,689.742 L965.947,688.668 L972.345,686.879 L975.585,687.052 L976.962,686.802 L977.628,686.943 L978.28,687.319 L980.673,689.511 L981.27,689.641 L980.794,687.977 L978.975,686.63 L978.435,685.343 L978.707,685.213 L978.943,684.953 L979.139,684.577 L979.278,684.086 L978.575,684.12 L977.187,684.652 L974.512,685.001 L973.205,685.815 L966.853,686.918 L963.27,689.086 L962.228,689.37 L959.216,689.214 L958.511,688.525 L959.477,688.157 L959.812,687.717 L959.567,687.139 L958.795,686.368 L959.862,686.052 L961.908,684.338 L968.018,682.854 L967.314,682.915 L965.954,682.678 L961.311,684.055 L960.261,684.756 L959.683,684.925 L958.42,684.951 L957.264,685.614 L947.341,686.364 L946.744,685.88 L947.079,685.925 L947.402,685.741 L947.66,685.32 L947.787,684.647 L947.679,684.276 L947.154,683.89 L947.05,683.548 L947.266,682.701 L947.73,681.918 L948.732,680.847 L950.197,680.193 L954.663,681.091 L954,680.62 L952.983,679.255 L952.376,678.87 L949.005,679.192 L948.666,678.756 L947.388,676.292 L947.218,675.123 L946.691,674.686 L945.285,674.687 L945.634,674.039 L946.203,673.584 L948.846,672.334 L951.107,670.455 L952.051,670.007 L953.899,669.622 L955.817,668.657 L956.774,668.732 L957.505,669.373 L958.75,671.036 L959.496,671.397 L960.911,672.803 L961.045,673.21 L961.417,674.915 L963.105,675.415 L964.836,674.199 L966.513,672.44 L968.054,671.311 L968.678,671.217 L971.127,671.522 L971.251,671.833 L971.112,672.304 L970.846,672.91 L970.265,673.707 L967.918,675.142 L968.652,675.526 L972.681,673.326 L973.098,673.214 L974.097,673.173 L974.626,672.989 L975.811,671.69 L976.801,671.729 L979.664,673.438 L979.974,673.799 L980.141,674.302 L980.21,674.809 L980.345,675.247 L980.687,675.525 L980.672,676.695 L980.911,678.66 L981.289,680.671 L981.683,681.982 L982.674,683.064 L985.214,683.806 L986.369,684.461 L989.063,687.721 L990.031,688.146 L990.942,688.324 L991.867,688.927 L992.551,690.125 L992.725,692.082 L993.395,690.528 L993.354,689.199 L992.775,688.121 L990.264,685.846 L988.031,684.49 L986.751,682.992 L984.334,681.968 L982.598,679.953 L982.27,679.28 L982.185,678.122 L982.303,677.196 L982.552,676.555 L982.932,676.178 L983.441,676.021 L983.112,675.766 L982.775,675.616 L982.465,675.394 L982.186,674.925 L982.055,674.51 L981.899,673.841 L981.728,673.46 L981.215,672.931 L980.021,672.427 L977.627,669.899 L977.532,669.719 L977.295,668.874 L977.161,668.611 L976.857,668.486 L976.539,668.527 L976.245,668.709 L975.991,669.004 L976.315,668.983 L976.505,669.115 L976.621,669.408 L976.747,669.874 L976.111,670.069 L975.137,670.121 L974.31,669.97 L974.1,669.562 L974.585,668.812 L975.697,667.898 L976.224,667.307 L977.214,665.225 L977.693,664.558 L978.266,664.031 L978.941,663.634 L979.634,663.445 L980.274,663.544 L980.708,663.883 L981.228,664.544 L981.57,665.414 L981.358,667.283 L981.712,668.117 L982.23,668.791 L982.65,669.188 L984.669,670.006 L985.253,670.722 L985.747,671.648 L986.016,671.796 L986.509,671.839 L986.774,671.944 L986.998,672.221 L987.163,672.626 L987.255,673.136 L986.357,674.004 L986.607,674.911 L987.083,676.047 L986.839,677.616 L986.827,678.002 L987.267,677.905 L988.128,677.453 L988.57,677.468 L988.956,677.663 L991.35,679.833 L992.342,680.395 L992.803,679.902 L992.415,678.713 L990.426,677.909 L989.709,677.063 L990.339,677.148 L993.853,678.484 L994.413,678.868 L994.751,679.297 L995.26,680.226 L995.556,680.633 L996.013,680.995 L997.006,681.483 L997.495,681.873 L995.176,678.355 L994.509,677.888 L992.349,677.409 L991.726,677.015 L991.19,676.448 L989.622,673.892 L989.429,673.187 L995.991,672.791 L996.71,672.058 L995.897,671.454 L994.035,671.968 L993.171,671.988 L992.512,671.357 L992.863,670.671 L994.402,669.459 L993.474,669.29 L991.554,671.035 L988.816,672.299 L988.257,672.269 L988.131,672.098 L987.839,671.464 L987.689,671.212 L987.459,670.998 L986.738,670.565 L987.774,670.061 L990.07,670.017 L991.082,669.389 L990.1,668.67 L986.361,669.281 L985.134,669.165 L984.463,668.909 L984.036,668.406 L984.008,668.083 L984.038,667.598 L984.106,667.138 L984.25,666.558 L984.143,666.372 L983.985,666.211 L983.907,666.001 L983.559,664.692 L982.877,663.907 L982.533,663.253 L983.213,662.347 L984.111,662.098 L984.725,662.797 L985.304,663.819 L986.093,664.547 L986.035,664.835 L985.933,665.533 L985.865,665.821 L986.511,665.644 L986.845,665.413 L987.071,665.035 L987.047,664.702 L986.862,664.464 L986.739,664.182 L986.914,663.714 L986.507,662.739 L985.913,662.227 L984.594,661.565 L984.607,661.098 L993.338,662.44 L996.082,663.912 L995.638,662.753 L994.275,661.721 L994.008,660.657 L994.397,659.832 L995.258,659.441 L1002.15,658.172 L1004.8,657.676 L1002.7,657.742 L996.42,657.918 L995.711,658.178 L995.385,658.027 L995.158,656.745 L994.887,656.037 L994.347,654.932 L993.071,653.289 L992.374,652.677 L991.658,652.322 L992.759,651.604 L998.089,650.583 L999.109,649.934 L999.887,649.037 L998.862,649.024 L998.381,648.888 L997.948,648.54 L997.765,648.143 L997.587,647.584 L997.384,647.082 L996.392,646.302 L996.637,645.241 L997.374,644.504 L998.654,645.342 L999.3,644.981 L1000.01,644.346 L1000.6,644.024 L1000.76,643.94 L1001.65,644.07 L1002.18,644.642 L1002.62,645.398 L1003.24,646.075 L1003.56,644.603 L1004.54,644.459 L1005.7,644.811 L1006.54,644.846 L1006.55,644.458 L1005.87,644.448 L1005.72,643.883 L1005.94,643.064 L1006.69,641.856 L1006.85,641.79 L1007.18,641.838 L1008.15,642.737 L1009.85,644.891 L1010.08,645.571 L1010.23,646.312 L1010.42,647.002 L1010.78,647.536 L1010.14,648.381 L1009.46,649.56 L1009.04,651.002 L1009.15,652.625 L1010.07,650.977 L1010.59,650.343 L1011.97,649.888 L1014.05,648.383 L1018.87,647.336 L1020.08,645.884 L1019.31,646.061 L1017.72,646.939 L1013.94,647.24 L1013.08,647.753 L1012.74,647.803 L1012.36,647.517 L1012,647.071 L1011.34,645.917 L1011.4,644.719 L1011.8,643.894 L1020.25,641.562 L1020.31,641.384 L1020.33,640.973 L1020.23,640.699 L1019.98,640.791 L1019.54,641.16 L1016.93,641.968 L1016.05,641.942 L1015.15,641.233 L1014.72,641.117 L1014.57,640.941 L1014.24,640.115 L1013.96,639.83 L1014.06,640.76 L1014.23,641.493 L1014.49,642.036 L1014.89,642.393 L1012.58,642.35 L1011.38,641.872 L1010.87,640.683 L1011.26,639.358 L1012.13,638.965 L1013.59,638.974 L1013.4,637.678 L1013.93,637.633 L1014.7,637.991 L1015.23,637.925 L1015.71,637.153 L1016.38,636.391 L1017.1,635.918 L1017.66,635.992 L1017.36,636.509 L1017.27,637.229 L1017.36,637.968 L1017.64,638.549 L1018.06,638.723 L1018.44,638.325 L1019.68,636.041 L1020.16,635.415 L1020.68,635.1 L1021.33,635.211 L1021.83,635.706 L1022.48,637.048 L1022.85,637.352 L1022.51,635.8 L1022.1,634.881 L1021.49,634.435 L1020.55,634.317 L1021.29,633.432 L1022.12,633.026 L1023.86,632.634 L1025.54,631.333 L1026.42,630.918 L1027.12,631.408 L1027.01,632.4 L1027.55,633.534 L1028.83,635.248 L1029.23,636 L1029.53,636.909 L1029.62,637.94 L1029.39,639.081 L1032,641.355 L1033.37,643.092 L1033.74,644.452 L1033.2,646.525 L1032.83,647.603 L1030.13,650.615 L1030.79,650.828 L1031.86,650.469 L1032.91,649.811 L1034.26,648.459 L1035.23,648.558 L1036.18,649.156 L1037.66,650.863 L1038.79,651.021 L1040.84,650.612 L1041.71,650.764 L1042.06,651.504 L1042.2,654.444 L1042.47,654.086 L1042.57,653.545 L1042.57,652.098 L1042.6,651.915 L1042.67,651.811 L1042.74,651.641 L1042.77,651.263 L1042.71,650.918 L1042.58,650.728 L1042.44,650.599 L1042.38,650.411 L1041.78,649.657 L1037.23,648.595 L1036.79,648.329 L1036.45,647.872 L1036.24,647.309 L1036.05,646.507 L1036.01,645.782 L1036.25,645.474 L1036.55,645.246 L1037.21,644.149 L1037.5,643.768 L1039.25,642.912 L1043.32,642.014 L1043.79,642.189 L1044.86,643.356 L1045.38,643.551 L1045.77,643.466 L1046.57,642.883 L1047.46,642.6 L1048.44,642.726 L1049.41,643.191 L1050.26,643.946 L1051.13,644.198 L1054.32,642.821 L1054.48,643.132 L1056.26,643.654 L1056.97,644.621 L1057.38,644.969 L1057.73,644.726 L1058.26,643.512 L1058.58,643.295 L1059.56,643.164 L1059.56,643.161 L1060.04,642.932 L1060.46,642.508 L1060.68,641.863 L1058.03,641.312 L1056.76,640.603 L1056.2,638.739 L1056.81,637.391 L1058.21,636.666 L1060.78,636.36 L1060.77,635.896 L1059.5,635.775 L1058.92,635.475 L1059.04,634.895 L1059.53,634.595 L1060.03,634.523 L1060.46,634.258 L1060.74,633.384 L1058.74,633.684 L1057.8,634.152 L1056.92,634.92 L1056.43,635.21 L1055.36,635.364 L1054.92,635.794 L1054.58,636.265 L1054.17,636.611 L1053.73,636.813 L1053.3,636.833 L1053.3,636.444 L1053.75,636.227 L1053.95,635.636 L1054.12,635.021 L1054.43,634.736 L1054.65,634.627 L1054.74,634.368 L1054.8,634.025 L1054.9,633.666 L1055.04,633.388 L1055.31,633.053 L1055.46,632.769 L1056.04,632.032 L1056.93,631.414 L1058.51,630.817 L1059.06,630.348 L1060.18,628.279 L1063.73,626.066 L1064.39,626.251 L1065.13,626.955 L1066.15,626.944 L1067.18,626.514 L1067.93,625.961 L1068.53,625.083 L1069.51,623.11 L1070.16,622.517 L1072.14,621.872 L1075.08,621.694 L1075.48,621.442 L1075.64,620.708 L1075.51,620.041 L1075.21,619.643 L1074.85,619.468 L1074.5,619.452 L1074.6,619.26 L1074.77,618.795 L1074.87,618.593 L1074.22,617.72 L1072.54,617.875 L1071.77,617.588 L1071.05,616.98 L1070.36,616.716 L1069.65,616.742 L1065.77,617.906 L1065.5,617.765 L1065.35,617.38 L1065.28,616.833 L1065.27,616.211 L1065.31,615.414 L1065.46,615.171 L1065.75,615.128 L1067.52,614.155 L1068.16,613.49 L1068.43,612.521 L1068.77,611.876 L1070.87,610.585 L1074.64,609.415 L1075.35,608.771 L1075.99,607.921 L1076.27,607.032 L1075.08,607.27 L1074.71,606.76 L1074.28,606.576 L1072.63,606.703 L1069.73,606.207 L1069.52,606.099 L1069.16,605.253 L1069.11,604.666 L1069.43,604.221 L1070.18,603.705 L1071.94,601.879 L1072.39,601.109 L1071.47,601.294 L1070.69,601.76 L1068.68,603.991 L1068.12,606.778 L1067.97,607.192 L1067.36,608.062 L1061.54,611.834 L1060.82,612.567 L1060.13,613.489 L1059.62,614.379 L1059.05,615.03 L1058.14,615.66 L1054.87,616.785 L1054,617.387 L1052.84,619.085 L1051.43,619.907 L1050.84,620.604 L1051.39,620.623 L1051.93,620.451 L1052.44,620.144 L1054.03,618.643 L1054.35,618.48 L1056.27,617.988 L1059.86,615.175 L1060.82,614.949 L1062.66,616.215 L1063.49,616.543 L1064.22,617.199 L1064.74,618.774 L1064.18,619.057 L1063.53,619.569 L1062.91,620.219 L1062.48,620.939 L1062.37,621.397 L1062.25,622.558 L1062.13,623.072 L1061.96,623.348 L1061.49,623.806 L1061.29,624.167 L1060.27,625.375 L1057.52,626.087 L1055.57,627.965 L1053.64,628.205 L1052.84,628.786 L1052.59,629.289 L1051.99,631.121 L1051.73,631.397 L1050.74,632.075 L1049.7,632.441 L1046.53,634.959 L1046.13,635.151 L1045.15,635.355 L1044.28,635.922 L1043.85,636.046 L1041.63,636.055 L1041.27,636.267 L1040.61,637.172 L1040.3,637.381 L1039.53,637.637 L1037.11,639.475 L1036.01,639.842 L1034.81,639.941 L1034.61,640.053 L1034.45,640.255 L1034.29,640.359 L1034.12,640.154 L1033.99,639.933 L1033.8,639.712 L1033.61,639.54 L1032.55,639.026 L1032.27,637.922 L1032.15,636.539 L1031.71,635.255 L1032,634.557 L1032.1,634.367 L1031.64,634.016 L1030.64,633.87 L1030.17,633.549 L1029.75,632.825 L1029.55,632.01 L1029.65,631.318 L1030.18,630.954 L1029.99,630.367 L1029.73,630.034 L1029.04,629.677 L1029.13,629.429 L1029.18,629.226 L1029.26,629.044 L1029.42,628.861 L1029.35,628.615 L1029.23,627.975 L1030.47,627.705 L1032.73,626.508 L1034.85,625.987 L1035.87,625.317 L1036.83,624.36 L1037.68,623.26 L1036.97,623.37 L1035.7,624.482 L1035,624.574 L1035.12,623.881 L1035.34,623.364 L1035.63,623.024 L1035.95,622.836 L1035.68,622.268 L1035.19,622.481 L1034.63,622.983 L1034.12,623.262 L1033.62,623.341 L1032.24,623.902 L1030.87,625.498 L1028.69,626.018 L1027.27,627.33 L1025.03,627.495 L1024.94,627.677 L1024.89,627.971 L1024.86,628.247 L1024.84,628.382 L1024.7,628.427 L1024.38,628.328 L1024.25,628.379 L1023.28,629.225 L1022.09,629.578 L1022.3,628.489 L1022.77,627.015 L1022.35,626.201 L1022.36,625.813 L1027.02,623.538 L1028.66,623.253 L1028.66,622.826 L1023.17,623.755 L1022.19,622.795 L1025.46,620.046 L1030.16,617.392 L1031.36,617.298 L1031.36,616.913 L1030.15,616.834 L1029.6,616.574 L1029.08,616.014 L1031.86,614.846 L1032.42,614.956 L1032.48,615.534 L1032.29,615.851 L1032.26,615.996 L1032.79,616.022 L1033.1,615.86 L1034.03,614.744 L1036.74,613.904 L1038.07,612.441 L1039.67,612.877 L1040.32,612.611 L1039.63,611.434 L1039.25,611.031 L1038.78,610.915 L1038.35,611.134 L1037.51,611.97 L1036.63,612.288 L1035.72,612.87 L1034.68,613.174 L1033.8,613.831 L1033.27,613.859 L1034.09,612.398 L1037.27,610.86 L1038.4,609.597 L1037.98,609.58 L1037.68,609.885 L1037.4,610.263 L1037.08,610.451 L1036.78,610.249 L1036.64,609.843 L1036.47,609.595 L1034.66,610.983 L1033.83,611.287 L1033.09,611.302 L1033.32,610.541 L1034.09,609.78 L1034.41,609.179 L1034.55,608.18 L1034.61,607.171 L1034.8,606.511 L1035.36,606.585 L1035.24,606.014 L1035.25,605.705 L1035.56,604.88 L1035.26,604.378 L1034.94,603.663 L1034.74,602.824 L1034.79,601.944 L1035.23,601.368 L1035.84,601.576 L1037.18,602.46 L1037.57,603.045 L1037.83,603.176 L1040.15,602.441 L1040.61,601.922 L1040.85,601.039 L1040.13,601.45 L1037.82,601.472 L1038.09,600.64 L1038.54,600.137 L1039.05,599.994 L1039.53,600.234 L1039.28,599.565 L1039.15,599.345 L1039.15,598.918 L1040.27,598.916 L1039.95,598.739 L1039.75,598.377 L1039.71,597.862 L1039.89,597.212 L1039.73,596.985 L1039.65,596.764 L1039.59,596.554 L1039.52,596.366 L1040.01,596.431 L1040.5,596.671 L1040.96,596.758 L1041.37,596.361 L1041.29,596.229 L1041.27,596.209 L1041.2,595.936 L1041.85,595.951 L1043.01,595.67 L1043.64,595.926 L1044.89,597.213 L1045.15,597.192 L1044.89,595.834 L1043.74,594.901 L1041.59,594.19 L1041.95,593.453 L1042.63,593.019 L1044,592.519 L1043.92,592.291 L1043.81,591.668 L1044.49,591.611 L1045.75,592.264 L1045.59,591.43 L1045.57,590.806 L1046.02,590.71 L1046.81,590.974 L1047.16,590.632 L1047.52,590.399 L1047.9,590.29 L1048.31,590.325 L1048.3,589.94 L1047.39,589.971 L1047.16,589.237 L1047.22,588.104 L1047.16,586.932 L1047.73,586.891 L1047.91,586.928 L1047.72,586.503 L1048.35,586.511 L1048.99,586.905 L1050.16,588.186 L1049.99,587.308 L1049.59,586.7 L1048.64,586.032 L1048.64,585.644 L1049.81,585.724 L1050.36,586.029 L1051.21,587.153 L1051.66,587.109 L1052.59,586.464 L1052.58,586 L1051.82,586.078 L1051.44,585.919 L1051.27,585.395 L1051.19,584.833 L1051,584.523 L1050.72,584.383 L1050.42,584.355 L1050.33,584.217 L1050.25,583.913 L1050.14,583.61 L1049.94,583.467 L1047.51,583.484 L1047.7,582.821 L1048.02,582.32 L1048.41,581.982 L1048.8,581.817 L1049.69,582.155 L1051.2,583.644 L1052,583.913 L1051.85,583.123 L1051.57,582.474 L1051.27,582.122 L1051.05,582.225 L1050.99,581.899 L1050.87,581.735 L1050.81,581.639 L1051.02,581.196 L1051.18,580.733 L1051.32,580.519 L1051.61,580.648 L1052.05,581.228 L1052.26,581.363 L1052.47,581.559 L1052.82,582.433 L1053.02,582.628 L1054.52,582.613 L1054.81,582.757 L1054.95,583.109 L1055.04,583.526 L1055.19,583.882 L1055.43,584.166 L1057.12,585.567 L1057.66,585.631 L1058.2,585.087 L1056.66,583.67 L1056.31,583.02 L1056.81,582.421 L1056.57,581.706 L1055.96,581.117 L1055.36,580.868 L1053.38,581.322 L1052.74,580.894 L1052.73,580.506 L1053.11,580.373 L1053.69,579.518 L1054.02,579.216 L1054.32,579.228 L1055,579.581 L1055.34,579.628 L1055.34,579.486 L1055.39,579.181 L1055.46,578.876 L1055.59,578.735 L1056.43,578.726 L1056.78,578.95 L1057.37,579.519 L1057.75,579.601 L1057.62,578.542 L1057.36,577.985 L1057.01,577.596 L1056.61,577.023 L1057.27,576.772 L1057.92,577.17 L1058.5,577.964 L1058.95,578.927 L1059.14,579.036 L1059.35,578.676 L1059.56,578.178 L1059.78,577.876 L1060.07,577.933 L1060.32,578.224 L1060.58,578.434 L1060.88,578.244 L1060.04,577.32 L1059.93,576.982 L1060.07,576.612 L1060.37,576.636 L1061.06,576.967 L1061.05,576.5 L1060.66,576.161 L1060.62,575.649 L1060.75,574.993 L1060.82,574.183 L1060.68,573.775 L1060.35,573.323 L1059.98,572.935 L1059.7,572.73 L1059.7,572.304 L1059.98,572.339 L1060.2,572.237 L1060.61,571.828 L1060.69,571.657 L1060.75,571.378 L1060.78,571.124 L1060.77,571.017 L1061.16,571.037 L1061.83,571.652 L1063.37,572.425 L1063.61,572.675 L1063.7,573.272 L1063.62,574.44 L1063.73,575.016 L1064.09,575.916 L1064.31,576.047 L1064.78,576.06 L1066.6,578.023 L1066.91,578.672 L1066.87,579.494 L1066.53,580.715 L1067.14,580.978 L1068.77,580.288 L1069.44,580.253 L1070.1,580.454 L1070.76,580.881 L1071.42,581.517 L1071.18,582.744 L1070.88,583.587 L1069.51,585.608 L1068.37,588.382 L1068.98,588.176 L1069.51,587.3 L1070.36,585.364 L1072.45,583.65 L1072.76,583.15 L1072.22,582.81 L1072,582.779 L1072.2,582.222 L1072.92,580.613 L1073.25,580.203 L1073.44,580.298 L1073.94,580.894 L1074.18,581.036 L1074.47,580.944 L1075.09,580.165 L1076.24,579.501 L1078.75,579.004 L1079.92,578.314 L1073.99,579.336 L1074.25,578.638 L1074.33,577.434 L1074.55,576.502 L1074.94,576.116 L1076.19,576.028 L1076.68,575.835 L1076.67,575.409 L1076.36,575.33 L1076.08,575.1 L1075.86,574.716 L1075.73,574.156 L1077.56,573.652 L1077.05,574.371 L1076.85,574.558 L1077.49,574.813 L1079.77,572.749 L1082.3,571.68 L1083.08,570.963 L1079.82,571.203 L1079.2,571.948 L1078.88,571.332 L1078.95,570.706 L1079.09,570.056 L1078.95,569.362 L1078.46,569.097 L1077.87,569.463 L1077.31,570.125 L1076.93,570.727 L1077.19,571 L1077.47,571.128 L1077.76,571.145 L1078.06,571.084 L1077.62,572.456 L1076.83,572.9 L1075.99,572.622 L1075.37,571.801 L1075.11,571.248 L1074.5,570.265 L1074.23,569.7 L1073.92,569.288 L1073.54,569.019 L1073.29,568.725 L1073.37,568.249 L1073.36,567.786 L1073.05,567.508 L1072.75,567.068 L1072.57,566.468 L1072.56,565.672 L1072.72,565.501 L1072.97,565.633 L1073.21,565.628 L1073.3,565.018 L1073.21,564.799 L1072.98,564.583 L1072.73,564.421 L1072.53,564.357 L1073.2,563.902 L1074.77,563.512 L1076.14,561.923 L1076.91,562.045 L1077.68,562.398 L1078.41,562.143 L1078.25,562.066 L1077.83,561.69 L1077.83,561.018 L1077.52,560.804 L1077.08,560.733 L1076.68,560.481 L1083.04,559.325 L1083.91,558.671 L1084.4,558.589 L1084.39,558.126 L1081.62,558.197 L1081.79,557.766 L1081.01,557.657 L1079.76,557.101 L1079.02,556.987 L1079.91,556.383 L1082.86,556.076 L1082.84,555.772 L1082.8,555.581 L1082.64,555.194 L1082.83,554.723 L1081.02,555.785 L1080.08,556.069 L1079.15,555.708 L1080.71,554.73 L1081.15,553.579 L1081.66,553.094 L1082.21,552.766 L1083.49,552.368 L1086.18,550.42 L1088.18,550.05 L1088.62,549.677 L1088.89,549.223 L1089.7,548.317 L1090.04,548.144 L1094.39,549.289 L1093.32,548.07 L1092.16,547.49 L1089.74,547.342 L1089.21,547.548 L1088.29,548.429 L1087.87,548.633 L1085.99,548.916 L1081.47,552.671 L1078.24,554.084 L1078.02,554.566 L1078.41,555.297 L1078.42,555.726 L1076.45,557.238 L1075.78,557.486 L1075.04,557.566 L1074.6,557.841 L1074.25,558.434 L1073.81,559.46 L1073.27,560.384 L1072.66,561.053 L1072,561.52 L1068.85,562.705 L1068.24,562.351 L1068.57,562.026 L1070.15,561.332 L1070.89,560.735 L1071.25,560.594 L1071.49,560.359 L1073.08,558.017 L1073.29,557.539 L1071.06,558.263 L1069.86,557.228 L1068.59,557.586 L1067.96,557.215 L1068.5,556.264 L1069.28,555.832 L1075.75,554.582 L1079.48,552.582 L1080.32,551.383 L1079.66,551.391 L1077.79,552.759 L1068.28,555.039 L1068.86,554.266 L1072.38,550.931 L1072.96,550.736 L1073.69,550.901 L1075.39,552.348 L1075.96,552.498 L1076.51,552.288 L1077.04,551.787 L1077.57,551.063 L1077.56,550.637 L1077.11,550.812 L1076.37,551.739 L1075.96,551.948 L1075.63,551.742 L1074.4,550.245 L1074.87,549.72 L1076.58,548.92 L1077.12,548.926 L1078.23,549.425 L1078.62,549.34 L1078.48,549.138 L1078.25,548.674 L1078.07,548.46 L1078.41,548.082 L1078.7,548.268 L1078.99,548.654 L1079.34,548.856 L1080.6,548.567 L1082.08,547.681 L1082.33,547.312 L1083.49,547.003 L1083.88,546.653 L1083.41,546.369 L1083.04,545.93 L1082.95,545.37 L1083.27,544.715 L1084.78,543.187 L1085.7,542.71 L1086.62,541.856 L1087.55,541.432 L1088.48,542.237 L1088.42,542.824 L1088.46,543.454 L1088.61,543.845 L1088.89,543.731 L1089.03,543.413 L1089.19,542.46 L1089.3,541.995 L1089.58,541.576 L1089.75,541.916 L1089.84,542.496 L1089.87,542.812 L1090.21,543.185 L1090.58,543.05 L1090.93,542.722 L1091.24,542.54 L1091.82,542.685 L1092.27,543.123 L1092.63,543.796 L1092.95,544.651 L1092.94,544.861 L1092.94,544.953 L1092.86,545.338 L1092.89,545.7 L1093.19,545.921 L1093.45,545.865 L1093.64,545.59 L1093.64,545.585 L1093.78,545.157 L1093.87,544.622 L1093.6,544.323 L1093.36,543.93 L1093.17,543.404 L1093.08,542.714 L1093.31,542.403 L1095.62,541.728 L1097.12,540.286 L1097.9,539.773 L1099.75,540.062 L1100.63,539.739 L1100.96,538.219 L1101.51,537.717 L1104.13,537.327 L1104.9,536.965 L1104.47,536.634 L1104.24,536.557 L1103.99,536.617 L1104.36,535.83 L1104.87,535.582 L1105.43,535.5 L1105.92,535.189 L1106.87,533.565 L1108.36,531.827 L1108.78,531.671 L1108.91,531.344 L1109.01,530.62 L1109.15,529.894 L1109.41,529.559 L1109.93,529.145 L1112.21,525.385 L1112.41,524.416 L1112.18,524.091 L1111.3,525.449 L1110.3,526.505 L1110.02,526.906 L1109.53,527.852 L1105.59,532.91 L1103.02,534.829 L1102.63,534.928 L1102.3,535.132 L1101.64,536.039 L1101.33,536.248 L1099.38,536.672 L1098.34,536.502 L1097.75,535.947 L1098.04,534.35 L1100.38,533.933 L1101.1,532.815 L1100.79,532.834 L1100.09,532.684 L1099.82,532.473 L1099.45,532.021 L1099.43,531.877 L1099.44,531.345 L1099.42,531.208 L1098.85,531.021 L1097.84,531.606 L1097.24,531.715 L1097.22,531.284 L1098.1,530.27 L1098.58,529.99 L1099.18,529.944 L1100.49,530.334 L1101.23,530.285 L1101.53,529.648 L1105.68,529.587 L1106.59,529.244 L1105.74,528.745 L1103.3,528.944 L1101.39,528.59 L1101.03,528.753 L1100.32,529.402 L1099.88,529.494 L1099.98,528.869 L1100.03,528.639 L1099.24,528.319 L1096.87,528.748 L1096.23,529.241 L1095.71,530.224 L1095.12,531.019 L1094.3,530.96 L1094.59,530.236 L1095.68,528.788 L1095.97,528.246 L1096.41,527.06 L1096.7,526.586 L1097.64,526.128 L1098.64,526.132 L1099.56,525.803 L1100.22,524.34 L1099.73,524.347 L1098.46,525.286 L1098.64,524.489 L1099,523.891 L1100.5,522.307 L1100.68,522.194 L1100.88,522.313 L1100.99,522.603 L1101.07,522.897 L1101.17,523.028 L1103.4,523.101 L1104.07,523.539 L1104.47,524.27 L1104.94,524.845 L1105.42,525.025 L1105.85,524.557 L1105.03,523.039 L1104.01,522.898 L1102.93,522.882 L1101.96,522.573 L1101.7,522.053 L1101.54,520.849 L1101.33,520.005 L1100.98,519.31 L1100.26,518.378 L1099.96,517.504 L1101.25,517.323 L1101.79,517.407 L1102.3,517.84 L1102.38,518.118 L1102.37,518.486 L1102.39,518.851 L1102.54,519.147 L1102.78,519.184 L1102.97,518.431 L1103.24,518.269 L1103.98,518.934 L1105.44,521.449 L1106.29,521.984 L1104.29,519.221 L1103.95,518.242 L1104.44,517.106 L1107.84,516.39 L1106.79,515.925 L1103.33,516.564 L1103.31,516.104 L1103.63,515.927 L1104.21,515.38 L1104.55,515.205 L1103.92,514.775 L1102.95,514.774 L1102.44,514.205 L1102.05,513.245 L1102.26,512.888 L1102.78,512.704 L1103.35,512.272 L1102.77,511.951 L1102.34,511.001 L1102.11,509.74 L1102.11,508.488 L1101.58,509.876 L1101.27,511.956 L1100.72,513.388 L1099.42,512.841 L1099.94,511.804 L1100.48,511.092 L1100.91,510.286 L1101.07,508.952 L1100.47,509.561 L1099.42,511.895 L1098.7,512.44 L1097.8,512.069 L1097.55,511.023 L1097.71,509.634 L1097.99,508.215 L1097.47,508.263 L1097.07,508.515 L1096.73,508.511 L1096.35,507.807 L1096.4,507.585 L1096.43,506.97 L1096.36,506.243 L1096.11,505.686 L1096.48,504.794 L1097.32,503.599 L1097.7,502.867 L1098.09,501.753 L1098.28,501.399 L1098.52,501.114 L1099,500.705 L1099.21,500.244 L1099.7,499.646 L1100.47,499.555 L1101.24,499.835 L1101.78,500.384 L1101.98,500.825 L1102.11,501.223 L1102.17,501.672 L1102.21,502.279 L1102.39,502.475 L1103.48,502.444 L1103.24,502.77 L1102.98,503.266 L1102.81,503.904 L1102.84,504.638 L1103.01,504.56 L1104.14,505.397 L1105.07,504.994 L1105.57,504.985 L1106.35,505.851 L1106.79,505.804 L1107.23,505.553 L1107.53,505.264 L1107.24,504.61 L1106.73,504.244 L1106.18,504.086 L1105.12,504.067 L1104.78,503.936 L1104.52,503.57 L1104.21,502.842 L1104.05,502.179 L1103.95,501.493 L1103.78,500.955 L1103.07,500.624 L1102.08,499.909 L1101.75,499.533 L1101.27,498.319 L1101.48,497.513 L1102.68,496.061 L1103.13,495.198 L1103.4,494.878 L1103.78,494.737 L1104.13,494.965 L1104.46,495.454 L1104.83,495.888 L1105.32,495.96 L1104.95,495.158 L1105.06,493.603 L1104.58,493.1 L1103.87,493.208 L1102.86,493.672 L1102.58,493.647 L1102.46,493.234 L1102.52,492.235 L1102.62,491.943 L1102.91,491.653 L1103.02,491.365 L1103.07,490.937 L1103.07,490.035 L1103.11,489.664 L1103.52,488.388 L1104.05,487.443 L1106.69,484.952 L1107.41,484.689 L1108.22,484.783 L1110.03,485.56 L1110.37,485.966 L1110.98,486.909 L1111.36,487.204 L1111.86,487.211 L1112.81,486.891 L1113.28,487.121 L1113.51,487.492 L1113.83,488.39 L1114.08,488.79 L1114.32,488.964 L1114.76,489.051 L1114.99,489.173 L1115.13,489.182 L1115.3,489.113 L1115.45,489.117 L1115.52,489.362 L1115.54,489.898 L1115.55,489.999 L1116.52,491.654 L1116.99,491.313 L1116.71,490.337 L1116.23,489.061 L1116.14,487.843 L1115.69,487.106 L1115.46,486.55 L1115.35,485.947 L1115.08,485.726 L1113.43,485.799 L1111.58,485.066 L1110.62,484.237 L1110.09,483.002 L1110.62,482.62 L1111.04,481.956 L1111.13,481.164 L1110.67,480.39 L1111.11,479.299 L1112.43,478.34 L1112.81,477.317 L1111.83,477.408 L1109.97,478.808 L1104.51,480.216 L1104.58,479.931 L1104.62,479.732 L1104.67,479.558 L1104.82,479.352 L1104.48,479.031 L1104.13,478.925 L1103.38,478.985 L1103.78,478.239 L1105.35,476.277 L1105.89,475.871 L1106.71,475.569 L1108.28,474.34 L1112.79,473.03 L1113.52,473.386 L1113.88,473.364 L1114.01,472.781 L1114.19,472.39 L1115.53,471.843 L1115.94,471.452 L1117.23,470.744 L1119.18,468.649 L1124.74,466.992 L1125.64,467.316 L1126.07,468.244 L1125.63,469.126 L1124.79,469.757 L1124.02,469.946 L1124.05,470.413 L1126.33,470.297 L1126.11,471.14 L1125.49,472.445 L1125.27,473.29 L1125.23,474.436 L1125.41,475.032 L1125.72,475.176 L1126.07,474.951 L1126.34,474.445 L1126.96,472.912 L1127.34,472.332 L1127.9,471.991 L1129.74,471.817 L1130.6,471.289 L1130.95,471.288 L1131.23,471.442 L1131.74,471.912 L1132.58,472.378 L1132.8,472.259 L1132.88,471.646 L1132.76,471.029 L1132.51,470.837 L1131.32,470.631 L1129.51,470.942 L1127.03,470.261 L1127.71,468.055 L1129.05,466.856 L1130.6,466.542 L1131.91,466.983 L1131.72,467.642 L1131.11,468.518 L1130.82,469.208 L1131.53,469.083 L1132.68,467.696 L1136.09,466.515 L1136.42,466.304 L1136.64,465.748 L1136.75,465.037 L1136.9,464.412 L1137.54,463.966 L1137.9,463.337 L1138.77,462.933 L1139.39,462.31 L1139.89,461.403 L1140.05,460.308 L1139.68,459.752 L1138.93,460.184 L1137.7,461.507 L1135.61,462.378 L1134.99,462.981 L1134.5,464.485 L1134.31,464.719 L1131.63,464.282 L1131.02,464.014 L1130.58,463.363 L1130.3,463.229 L1129.96,463.646 L1130.03,464.385 L1129.65,465.242 L1129.07,465.96 L1127.09,466.822 L1126.37,466.677 L1125.89,465.561 L1127.06,464.687 L1126.78,464.344 L1126.45,464.054 L1126.12,463.874 L1125.79,463.865 L1125.5,464.088 L1124.06,465.85 L1123.17,466.118 L1121.84,466.949 L1119.63,466.891 L1118.63,467.236 L1116.64,469.595 L1113.28,471.652 L1112.52,471.338 L1112.24,470.193 L1112.81,469.38 L1113.65,468.732 L1114.21,468.078 L1114.72,467.042 L1115.41,466.26 L1119.3,463.513 L1123.7,463.315 L1124.68,462.685 L1123.92,462.146 L1121.33,462.386 L1121.5,461.721 L1121.78,461.388 L1122.49,461.056 L1121.99,460.706 L1121.55,460.768 L1120.67,461.146 L1120.28,461.096 L1120.09,460.892 L1119.93,460.518 L1119.64,459.956 L1119.13,459.218 L1118.79,458.915 L1118.71,459.253 L1118.97,460.414 L1119.42,461.214 L1119.63,461.782 L1119.49,462.049 L1119.13,462.178 L1116.9,463.809 L1116.54,463.932 L1115.07,463.498 L1114.6,463.555 L1114.84,463.176 L1115.09,462.59 L1115.16,462.059 L1114.86,461.845 L1114.45,461.962 L1113.52,462.752 L1112.72,462.939 L1112.03,462.62 L1111.43,461.764 L1110.97,460.353 L1114.45,458.392 L1115.12,456.729 L1114.99,456.482 L1114.86,456.136 L1114.76,455.712 L1114.7,455.238 L1114.5,454.814 L1113.27,454.223 L1113.6,453.879 L1113.99,453.755 L1114.8,453.767 L1114.78,453.342 L1114.39,453.147 L1113.59,453.122 L1113.21,452.949 L1112.94,452.51 L1112.79,451.97 L1112.62,451.509 L1112.27,451.329 L1111.56,451.304 L1110.96,451.128 L1110.4,450.747 L1109.78,450.125 L1113,448.707 L1113.29,448.834 L1113.82,449.421 L1115.17,449.926 L1115.77,450.362 L1116.06,450.927 L1116.19,451.795 L1116.38,452.554 L1116.49,453.195 L1116.36,453.695 L1116.38,454.082 L1119.82,453.819 L1120.67,453.063 L1117.93,453.624 L1117.38,453.221 L1117.42,452.171 L1117.72,451.244 L1117.77,450.555 L1117.04,450.224 L1117.02,449.8 L1117.65,449.651 L1118.75,448.671 L1119.38,448.41 L1121.54,448.477 L1122.13,448.042 L1122.71,447.784 L1123.44,448.09 L1124.78,449.027 L1126.09,449.548 L1127.41,449.706 L1126.8,448.791 L1126.12,448.317 L1122.46,447.493 L1121.9,446.974 L1123.04,445.639 L1125.33,444.093 L1128.47,444.078 L1127.46,443.448 L1122.86,444.265 L1122.13,444.872 L1121.01,447.079 L1120.35,447.453 L1118.81,447.589 L1117.21,448.189 L1116.42,448.196 L1115.54,447.743 L1115.6,446.661 L1115.78,446.067 L1116.43,445.109 L1116.26,444.935 L1116.08,444.7 L1116.49,444.119 L1117.44,443.914 L1118.51,443.248 L1119.77,443.635 L1120.3,443.226 L1119.52,442.763 L1116.44,442.92 L1114.38,441.845 L1114.85,440.912 L1115.43,441.136 L1116.04,441.639 L1116.61,441.529 L1117.19,441.167 L1117.78,441.375 L1118.35,441.788 L1118.86,442.019 L1119.52,441.841 L1120.73,441.092 L1121.4,441.084 L1123.2,442.124 L1123.86,442.237 L1123.17,440.851 L1122.05,440.023 L1116.63,438.645 L1115.55,437.925 L1116.29,437.265 L1122.25,437.601 L1122.43,437.789 L1122.89,438.645 L1123.18,438.828 L1127.32,438.751 L1127.93,438.154 L1127.4,437.757 L1127.37,437.335 L1129.9,437.355 L1131.18,436.934 L1132.08,435.764 L1127.64,436.008 L1126.78,436.322 L1125.18,437.44 L1124.3,437.496 L1123.56,436.893 L1123.41,435.973 L1123.75,435.066 L1124.47,434.512 L1127.73,435.04 L1128.56,434.295 L1127.74,433.858 L1123.79,433.59 L1123.49,433.904 L1123.3,433.969 L1122.36,433.343 L1123.15,432.001 L1124.88,431.501 L1126.76,431.574 L1128.83,432.36 L1134.44,432.225 L1133.28,431.641 L1129.59,431.264 L1129.18,431.031 L1128.44,429.908 L1127.94,429.61 L1127.06,429.99 L1126.68,429.835 L1126.37,428.848 L1126.52,428.779 L1126.73,428.502 L1126.85,428.397 L1126.51,428.209 L1125.8,428.095 L1125.46,427.847 L1125.28,427.438 L1125.18,426.387 L1125.02,425.901 L1124.37,425.482 L1122.96,426.008 L1122.27,425.661 L1122.78,424.631 L1123.27,423.953 L1123.86,423.56 L1127.7,423.432 L1128.56,422.816 L1128.36,422.53 L1128.22,422.211 L1127.98,421.494 L1129.94,421.62 L1130.52,421.354 L1130.66,421.118 L1130.93,420.358 L1131.11,420.044 L1131.38,419.793 L1131.66,419.663 L1132.28,419.59 L1133.13,419.542 L1132.84,420.838 L1132.34,421.366 L1131.77,421.649 L1131.25,422.203 L1132.02,423.695 L1132.47,424.088 L1133.05,423.758 L1132.88,423.707 L1132.35,423.415 L1133.47,422.077 L1133.51,421.876 L1133.52,421.572 L1133.56,421.298 L1133.74,421.172 L1134.42,421.223 L1134.6,421.122 L1134.94,420.286 L1134.86,419.411 L1134.51,418.549 L1134.05,417.748 L1134.24,417.27 L1134.4,416.1 L1134.55,415.635 L1135.02,415.304 L1135.33,415.749 L1135.56,416.42 L1135.75,416.758 L1135.77,417.155 L1136.22,418.832 L1136.52,419.587 L1137.64,420.706 L1137.93,420.225 L1138.24,420.176 L1138.59,420.342 L1138.84,420.262 L1139.07,420.088 L1139.36,419.986 L1139.45,419.851 L1139.4,419.294 L1139.48,419.165 L1139.98,418.556 L1140.33,417.28 L1140.29,416.089 L1139.66,415.714 L1140.25,415.042 L1143.83,414.604 L1144.32,414.77 L1145.29,415.588 L1145.78,415.754 L1146.64,415.353 L1147.02,415.523 L1147.37,416.502 L1147.34,417.175 L1147.73,417.173 L1148.17,416.606 L1148.3,415.588 L1147.87,414.653 L1147.12,414.348 L1145.77,414.479 L1142.75,413.783 L1140.62,414.032 L1140.03,413.565 L1140.5,413.168 L1140.85,412.457 L1141.17,411.631 L1141.54,410.918 L1141.15,410.226 L1141.26,409.557 L1141.64,409.023 L1142.06,408.721 L1144.21,408.198 L1144,408.622 L1143.72,409.523 L1143.48,410.516 L1143.42,411.184 L1143.75,411.742 L1144.16,411.358 L1144.56,410.549 L1144.84,409.815 L1144.7,409.62 L1144.61,409.43 L1144.55,409.226 L1144.45,408.994 L1149.76,408.222 L1150.59,408.711 L1151.31,410.277 L1151.58,410.67 L1151.73,410.31 L1151.68,409.613 L1151.55,408.983 L1151.24,408.355 L1150.88,408.044 L1149.05,407.203 L1145.67,407.214 L1146.18,406.56 L1146.95,405.89 L1147.76,405.572 L1148.37,405.971 L1149.02,406.672 L1149.67,406.469 L1150.3,405.919 L1151.16,405.434 L1151.73,404.812 L1152.06,404.654 L1154.59,404.982 L1156.9,404.312 L1156.94,404.774 L1155.93,404.991 L1155.64,405.255 L1155.93,406.49 L1155.65,407.042 L1155.22,407.459 L1155.03,408.279 L1155.29,408.96 L1156.41,409.996 L1157.29,411.172 L1157.83,411.429 L1158.16,411.21 L1158,410.428 L1157.5,409.634 L1156.52,408.852 L1156.04,408.207 L1156.32,407.885 L1157.63,405.313 L1157.94,404.874 L1159.22,404.202 L1159.73,404.165 L1160.68,405.19 L1161.21,405.318 L1162.87,405.244 L1163.4,405.478 L1164.51,406.304 L1165,406.534 L1166.39,406.624 L1167.09,407.139 L1167.45,408.027 L1167.49,409.265 L1167.24,410.815 L1167.98,411.579 L1168.8,412.055 L1169.65,412.273 L1170.46,412.268 L1170.15,411.652 L1169.2,411.131 L1168.77,410.695 L1168.8,410.04 L1168.65,409.387 L1169.3,407.835 L1169.49,407.194 L1169.31,407.208 L1168.78,406.067 L1167.35,405.174 L1166.7,404.431 L1167.45,403.592 L1168.54,403.479 L1175.16,405.457 L1175.74,405.408 L1175.49,404.539 L1175.14,404.209 L1174.76,403.982 L1174.39,403.426 L1174.68,402.937 L1172.3,403.848 L1170.84,402.923 L1167.27,402.196 L1166.68,402.308 L1166.06,402.775 L1165.06,404.042 L1164.48,404.289 L1163.8,403.801 L1164.3,403.369 L1164.89,403.243 L1165.4,402.93 L1165.64,401.957 L1164.16,401.734 L1163.81,401.905 L1163.52,402.201 L1163.19,402.424 L1162.51,402.623 L1158.77,402.091 L1159.08,401.277 L1159.56,401.042 L1160.07,400.958 L1160.5,400.607 L1160.69,400.08 L1160.96,398.018 L1160.29,398.034 L1158.35,399.101 L1157.69,399.271 L1157.36,399.457 L1157.23,399.801 L1157.21,400.275 L1157.08,400.609 L1156.88,400.811 L1156.65,400.889 L1154.9,400.689 L1151.63,401.242 L1152.02,401.581 L1152.8,401.682 L1153.19,402.022 L1152.65,402.719 L1149.42,404.685 L1148.04,404.928 L1149.32,403.632 L1149.78,402.683 L1149.62,401.38 L1149.77,400.985 L1149.33,400.855 L1148.87,401.107 L1147.99,401.952 L1146.86,402.664 L1142.85,403.183 L1142.37,403.426 L1141.9,403.851 L1141.4,404.065 L1140.86,403.691 L1142.33,402.463 L1142.77,401.832 L1142.62,401.309 L1142.86,400.893 L1143.64,400.113 L1144.5,398.394 L1146.92,394.843 L1147.11,394.174 L1146.93,393.819 L1146.23,393.952 L1146.2,393.566 L1146.46,393.548 L1146.71,393.441 L1147.16,393.038 L1147.24,392.125 L1147.93,391.724 L1150.14,391.964 L1150.84,391.715 L1151.53,391.759 L1152.15,392.697 L1151.97,392.915 L1151.84,393.221 L1151.58,394.014 L1152.34,394.413 L1153,393.702 L1153.63,392.674 L1154.28,392.121 L1155.07,391.874 L1155.96,391.321 L1157.45,389.767 L1154.81,390.341 L1151.96,390.16 L1152.07,389.959 L1152.38,389.238 L1151.87,388.982 L1150.96,389.266 L1150.53,388.983 L1150.81,388.495 L1150.49,388.157 L1150.15,387.902 L1149.8,387.764 L1149.44,387.737 L1149.71,386.954 L1150.26,386.081 L1150.9,385.361 L1155.41,382.211 L1157.03,381.675 L1157.64,381.858 L1158.46,382.404 L1159.22,383.179 L1159.65,384.07 L1159.36,384.305 L1159.06,384.444 L1158.43,384.546 L1158.19,384.432 L1157.73,383.878 L1157,383.427 L1156.42,382.815 L1155.88,382.611 L1155.43,383.526 L1155.94,384.029 L1156.78,385.357 L1157.25,385.485 L1156.31,387.357 L1153.28,387.612 L1151.99,388.413 L1152.61,388.672 L1155.85,388.701 L1156.56,388.478 L1157.03,387.865 L1157.4,387.171 L1157.93,386.66 L1158.5,386.337 L1158.98,386.21 L1159.39,385.891 L1159.76,385.31 L1160.14,384.937 L1160.58,385.238 L1160.5,385.696 L1160.62,386.126 L1160.86,386.43 L1161.17,386.511 L1161.2,386.891 L1160.86,387.206 L1160.51,387.369 L1159.75,387.469 L1159.78,387.85 L1159.93,387.92 L1160.33,388.278 L1160.2,389.064 L1159.98,389.613 L1159.78,390.236 L1159.73,391.26 L1160.14,390.978 L1160.86,389.773 L1161.34,389.478 L1161.51,389.308 L1161.61,388.915 L1161.64,388.423 L1161.63,387.944 L1161.71,387.639 L1161.93,387.427 L1165.04,386.655 L1166.46,386.841 L1167.64,388.098 L1167.84,388.614 L1168.27,389.91 L1169.75,393.04 L1169.84,393.399 L1169.96,394.328 L1170.07,394.755 L1170.15,394.823 L1170.85,395.969 L1171.75,396.889 L1172.68,397.565 L1172.72,397.988 L1171.77,398.395 L1171.8,399.053 L1172.4,399.487 L1173.17,399.231 L1174.03,397.454 L1173.78,396.875 L1172.4,396.27 L1171.94,395.713 L1171.84,395.356 L1171.85,394.944 L1171.71,394.237 L1171.16,393.083 L1171.06,392.739 L1170.72,391.951 L1170.04,391.131 L1168.83,390.175 L1169.14,389.379 L1169.95,388.536 L1170.3,387.885 L1169.61,387.976 L1169.27,387.896 L1168.95,387.61 L1169.53,385.949 L1170.39,384.847 L1174.71,381.952 L1177.36,381.336 L1176.55,380.385 L1171.85,382.106 L1169.34,384.556 L1167.99,385.004 L1162.33,384.58 L1161.63,384.33 L1161.28,383.717 L1161.21,382.52 L1161.29,382.019 L1161.91,381.31 L1161.77,381.237 L1161.38,380.922 L1162.17,380.278 L1162.94,380.183 L1164.65,380.67 L1165.48,380.653 L1165.86,380.89 L1166.22,382.069 L1166.46,381.931 L1166.7,381.463 L1166.84,380.926 L1165.56,379.323 L1165.68,379.239 L1166,378.824 L1165.58,378.127 L1165.12,377.698 L1164.67,377.665 L1164.28,378.144 L1164.58,378.745 L1164.68,378.927 L1164.12,379.402 L1163.41,379.08 L1162.66,378.531 L1161.99,378.321 L1160.81,379.24 L1160.13,379.489 L1159.71,378.92 L1160,377.262 L1163.15,374.366 L1162.39,374.463 L1162,374.378 L1161.63,374.057 L1161.92,372.734 L1162.12,372.389 L1162.46,372.251 L1162.82,372.325 L1163.79,372.878 L1164.95,374.119 L1168.14,374.823 L1167.57,373.866 L1167.37,373.608 L1170.9,374.172 L1169.06,372.83 L1168.08,372.478 L1166.68,373.359 L1165.92,372.863 L1164.52,371.277 L1164.48,370.854 L1169.04,370.056 L1170.59,370.789 L1174.29,373.976 L1174.83,374.007 L1175.09,372.972 L1174.72,373.032 L1174.37,372.927 L1174.03,372.664 L1173.7,372.237 L1173.75,371.822 L1173.71,371.353 L1173.75,370.954 L1170.7,369.113 L1169.77,369.154 L1169.94,368.331 L1170.3,367.857 L1170.72,367.473 L1171.06,366.922 L1170.64,366.941 L1170.29,367.152 L1169.23,368.328 L1168.86,368.51 L1164.35,367.462 L1164.5,366.41 L1164.99,365.185 L1164.91,364.397 L1164.51,363.442 L1164.25,363.706 L1164.1,364.651 L1164.04,365.744 L1163.37,363.871 L1162.98,363.285 L1162.75,364.178 L1162.87,364.648 L1163.48,366.379 L1163.66,367.052 L1162.01,368.917 L1161.02,370.474 L1160.15,371.096 L1159.55,372.602 L1159.29,372.956 L1158.84,373.12 L1156.09,375.498 L1154.04,376.514 L1153.67,377.004 L1153.28,377.324 L1151.35,378.02 L1150.66,377.706 L1149.32,377.819 L1148.64,377.159 L1152.05,374.374 L1152.02,373.912 L1149.68,373.65 L1149.9,372.745 L1150.41,372.705 L1150.99,372.943 L1151.44,372.871 L1151.87,372.463 L1157.6,371.378 L1156.86,371.022 L1155.98,371.308 L1155.14,371.262 L1154.51,369.901 L1155,369.531 L1155.42,368.985 L1151.7,369.851 L1150.61,370.607 L1150.84,370.375 L1150.91,370.158 L1150.81,369.954 L1150.55,369.76 L1150.52,368.779 L1150.21,368.451 L1149.81,368.691 L1149.53,369.405 L1149.29,369.295 L1149.05,369.01 L1148.82,368.608 L1148.62,368.158 L1149.01,367.378 L1149.37,366.927 L1149.78,366.742 L1151.37,366.78 L1153.35,365.749 L1155.82,365.473 L1156.71,364.637 L1156.01,364.042 L1150.7,365.458 L1150.67,365.073 L1150.9,365.092 L1151.09,364.997 L1151.45,364.556 L1151.75,363.635 L1152.78,362.703 L1154,362.074 L1154.9,362.07 L1156.1,362.431 L1158.65,362.316 L1159.7,363.099 L1159.32,363.376 L1159,363.925 L1158.88,364.71 L1159.1,365.698 L1161.24,361.85 L1162.02,361.217 L1162.09,361.693 L1162.61,361.663 L1165.4,359.823 L1166.63,359.588 L1166.84,359.751 L1166.37,361.513 L1166.23,361.895 L1166.12,362.107 L1166.11,362.4 L1166.27,363.011 L1166.49,363.559 L1166.72,363.824 L1166.93,363.772 L1167.12,363.371 L1166.66,362.896 L1166.6,362.201 L1166.78,361.45 L1167.06,360.819 L1167.43,360.378 L1167.84,360.097 L1168.13,359.664 L1168.19,358.794 L1171.6,357.172 L1172.46,357.145 L1173.3,357.929 L1173.63,358.541 L1173.84,359.602 L1173.99,360.152 L1174.24,360.301 L1174.46,360.202 L1174.54,359.991 L1174.55,358.498 L1174.83,357.723 L1175.82,356.455 L1176.08,355.951 L1176.22,355.416 L1176.41,354.08 L1176.11,354.106 L1176.39,353.239 L1177.12,352.947 L1178.62,352.994 L1178.58,352.576 L1174.7,352.451 L1173.43,351.743 L1173.39,351.358 L1174.19,350.353 L1174.38,349.781 L1173.87,349.574 L1173.59,349.787 L1172.73,350.805 L1172.56,351.197 L1172,352.011 L1167.56,352.131 L1167.31,351.85 L1167.13,351.344 L1166.8,350.626 L1166.34,350.032 L1166.01,349.987 L1165.76,350.286 L1165.49,350.731 L1162.48,353.487 L1162.75,354.197 L1163.1,354.276 L1163.48,354.147 L1163.85,354.227 L1162.97,355.199 L1161.98,355.215 L1160.97,354.386 L1159.99,352.828 L1160.23,352.156 L1160.57,351.871 L1160.95,351.715 L1161.32,351.448 L1161.48,350.016 L1162.51,348.404 L1163.81,347.077 L1164.81,346.495 L1164.75,347.39 L1164.87,348.179 L1165.13,348.751 L1165.53,348.991 L1165.63,348.796 L1165.58,347.912 L1165.64,347.704 L1166.7,347.619 L1166.91,347.368 L1167.67,346.119 L1167.81,345.595 L1167.65,344.193 L1167.8,343.967 L1168.33,344.041 L1168.28,344.293 L1168.27,344.508 L1168.26,344.716 L1168.24,344.939 L1168.71,345.141 L1169.65,345.271 L1170.09,345.599 L1169.76,346.137 L1169.37,346.512 L1169.72,346.78 L1170.11,347.257 L1170.38,347.879 L1170.37,348.595 L1169.84,349.055 L1168.22,348.44 L1167.62,348.822 L1167.88,349.216 L1167.97,349.722 L1167.93,350.303 L1167.81,350.932 L1168.09,351.167 L1168.39,351.295 L1168.69,351.327 L1168.99,351.261 L1168.7,350.265 L1168.95,349.894 L1170.49,349.767 L1171.09,349.387 L1171.14,349.18 L1171.11,348.902 L1171.14,348.64 L1171.42,348.508 L1172.49,348.356 L1172.94,348.069 L1173.36,347.49 L1173.89,346.17 L1174.2,345.704 L1174.63,345.679 L1173.66,344.395 L1173.17,343.473 L1173.03,342.409 L1173.42,342.031 L1174.11,342.312 L1174.71,342.326 L1174.86,341.15 L1174.76,340.614 L1174.61,340.063 L1174.51,339.524 L1174.57,339.029 L1174.86,338.492 L1175.21,338.284 L1176.03,338.242 L1176.62,338.602 L1176.88,339.513 L1177.08,340.51 L1177.46,341.137 L1177.35,341.685 L1177.19,342.12 L1176.99,342.391 L1176.76,342.473 L1177.21,343.582 L1177.66,343.439 L1178.07,342.897 L1178.43,342.79 L1178.67,343.583 L1178.68,344.724 L1178.56,345.748 L1178.15,346.606 L1178.35,347.456 L1178.86,348.932 L1178.81,349.987 L1178.53,350.578 L1178.2,351.056 L1177.99,351.774 L1180.92,353.396 L1181.86,354.447 L1182.42,355.813 L1182.64,357.301 L1182.77,358.727 L1183.05,359.88 L1183.5,360.407 L1184.62,360.707 L1185.08,361.188 L1185.39,361.63 L1188.75,364.671 L1189.94,365.169 L1190.62,364.723 L1189.98,364.479 L1189.23,363.993 L1188.75,363.175 L1188.86,361.941 L1185.33,360.843 L1184.22,359.987 L1183.82,359.335 L1183.84,358.797 L1184.03,358.264 L1184.15,357.652 L1184.07,356.786 L1183.81,355.306 L1183.85,354.655 L1183.42,353.97 L1182.97,353.547 L1182.06,353.113 L1182.91,352.61 L1183.63,352.778 L1184.34,353.467 L1187.29,357.82 L1187.53,358.62 L1188.53,358.984 L1188.86,358.918 L1188.71,358.434 L1188.56,358.026 L1188.2,357.278 L1188.3,357.08 L1188.48,356.57 L1188.59,356.387 L1188.21,356.05 L1187.06,355.639 L1186.81,355.403 L1186.12,354.446 L1185.13,353.341 L1185.05,352.851 L1185.57,352.366 L1186.12,352.346 L1186.57,352.807 L1187.42,354.326 L1188.09,355.042 L1188.85,355.322 L1189.6,355.264 L1190.25,354.95 L1189.78,354.594 L1188.75,354.559 L1188.28,354.055 L1188.22,353.476 L1188.41,352.812 L1188.91,351.632 L1185.62,351.087 L1185.72,350.049 L1186.01,349.215 L1186.39,348.647 L1186.81,348.417 L1187.16,348.057 L1187.31,347.387 L1187.54,346.886 L1188.11,347.02 L1189.75,348.429 L1192.51,349.134 L1193.02,348.678 L1192.88,348.636 L1192.48,348.342 L1192.73,348.005 L1192.99,347.817 L1190.26,347.201 L1189.7,346.781 L1188.78,345.667 L1188.25,345.302 L1186.96,345.614 L1184.85,347.892 L1183.44,347.453 L1183.52,347.128 L1183.64,346.156 L1182.83,345.925 L1182.42,345.652 L1182.06,345.022 L1183.33,345.32 L1183.96,345.253 L1184.78,344.034 L1185.82,344.38 L1186,343.807 L1185.82,343.649 L1185.64,343.414 L1186.01,343.006 L1186.45,343.049 L1186.93,343.24 L1187.39,343.25 L1187.09,342.471 L1186.66,342.026 L1185.78,341.656 L1185.74,341.274 L1188.99,342.334 L1190.03,342.108 L1189.58,341.74 L1188.62,341.549 L1187.16,340.38 L1186.92,340.275 L1186.73,340.023 L1186.63,338.866 L1186.43,338.618 L1186.17,338.84 L1185.71,339.764 L1185.45,339.986 L1180,340.754 L1179.72,340.314 L1180.16,339.651 L1180.84,339.27 L1183.95,338.848 L1184.13,338.62 L1184.08,338.179 L1183.88,337.743 L1183.65,337.558 L1180.1,337.328 L1178.35,336.526 L1177.9,335.984 L1178.28,335.524 L1178.8,334.039 L1179.3,333.695 L1179.78,333.776 L1180.65,334.253 L1181.14,334.381 L1181.44,334.308 L1181.67,334.168 L1181.92,334.107 L1182.27,334.278 L1182.46,334.545 L1182.73,335.287 L1182.89,335.534 L1183.43,335.689 L1184.56,335.319 L1184.85,335.741 L1187.54,336.829 L1188.9,338.112 L1189.22,340.484 L1193.73,343.507 L1194.53,343.82 L1195.26,343.77 L1193.83,342.303 L1193.57,341.572 L1193.28,341.191 L1191.06,339.845 L1190.21,338.845 L1189.62,337.627 L1189.77,336.593 L1188.79,335.408 L1187.66,334.73 L1185.01,334.363 L1183.67,333.673 L1183.2,333.145 L1182.62,332.672 L1181.41,333.103 L1180.82,332.705 L1181.68,331.814 L1181.3,331.697 L1180.6,331.877 L1180.21,331.522 L1181.16,330.76 L1183.25,330.025 L1184.69,328.742 L1190.9,327.144 L1191.56,327.33 L1193.59,328.512 L1194.66,328.456 L1195.19,328.774 L1195.52,329.829 L1195.38,330.146 L1194.68,331.331 L1194.46,331.876 L1195.01,331.859 L1198.29,330.21 L1199,329.283 L1199.38,328.971 L1201.17,328.145 L1201.61,328.358 L1202.51,329.952 L1202.72,330.18 L1203.17,331.999 L1203.31,332.444 L1204.05,333.38 L1206.75,334.872 L1206.46,336.1 L1206.44,336.377 L1206.58,336.934 L1206.78,337.256 L1206.97,337.503 L1207.1,337.823 L1207.14,338.851 L1207.06,339.668 L1207.09,340.193 L1207.48,340.341 L1207.93,339.992 L1208.06,339.248 L1207.93,338.354 L1207.63,337.57 L1207.47,336.687 L1207.93,336.177 L1208.98,335.876 L1208.06,334.043 L1207.04,333.032 L1204.91,331.658 L1204.45,331.193 L1204.02,330.615 L1203.63,329.967 L1203.34,329.261 L1203,327.946 L1203.21,327.619 L1203.65,327.448 L1204.01,326.59 L1203.92,326.373 L1203.73,325.807 L1205.57,325.462 L1209.79,327.715 L1211.72,327.892 L1210.86,326.865 L1206.39,325.057 L1207.03,324.391 L1210.34,323.697 L1214.39,324.282 L1217.09,325.669 L1218.3,325.014 L1216.97,324.899 L1211.84,322.125 L1208.52,322.696 L1209.76,318.125 L1209.73,317.406 L1209.23,317.526 L1208.29,318.174 L1207.41,319.134 L1205.73,319.542 L1204.96,320.356 L1203.66,322.739 L1198.87,324.555 L1197.73,323.833 L1198.2,323.485 L1199.5,322.986 L1199.79,322.302 L1199.59,321.569 L1199.06,321.105 L1198.45,320.87 L1198.03,320.814 L1197.78,320.702 L1197.25,320.147 L1196.98,320.032 L1196.75,320.147 L1195.21,322.434 L1194.99,323.023 L1194.18,324.12 L1191.61,324.145 L1190.08,325.169 L1186.97,325.472 L1186.4,325.339 L1186.1,324.763 L1185.99,323.879 L1185.89,323.396 L1185.54,322.47 L1185.24,321.914 L1184.99,321.6 L1184.88,321.197 L1185,320.353 L1185.23,319.981 L1186.45,318.941 L1187.41,319.429 L1187.99,319.723 L1188.34,319.4 L1188.51,318.586 L1188.39,318.313 L1188.07,318.223 L1187.63,317.976 L1187.28,317.651 L1187,317.299 L1186.79,316.784 L1186.62,315.945 L1186.69,314.654 L1187.15,313.522 L1187.82,312.702 L1189.31,312.003 L1191.7,309.869 L1194.26,308.846 L1195,308.068 L1195.71,307.782 L1196.43,308.59 L1197.14,309.648 L1198.04,310.287 L1198.11,310.668 L1198.15,311.096 L1198.22,311.381 L1198.39,311.538 L1199.37,312.037 L1199.82,312.081 L1200.24,311.93 L1200.64,311.594 L1200.42,311.063 L1200.17,310.828 L1199.59,310.812 L1199.28,310.706 L1198.89,310.165 L1198.63,310.058 L1198.45,309.902 L1198.32,309.498 L1198.27,309.005 L1198.3,308.598 L1198.44,308.19 L1198.57,308.044 L1200.55,307.067 L1201.49,306.184 L1201.99,306.367 L1202.98,307.042 L1203.45,306.913 L1203.88,306.646 L1204.33,306.681 L1205.29,308.238 L1205.76,308.738 L1206.25,309.044 L1206.72,309.199 L1207.41,309.184 L1209.33,308.447 L1211.27,309.284 L1211.79,309.028 L1210.28,307.891 L1207.33,307.655 L1206.06,306.247 L1205.55,305.377 L1204.84,304.544 L1204.12,304.256 L1203.61,305.018 L1203.11,305.533 L1202.31,305.308 L1200.89,304.241 L1201.46,303.422 L1202.92,302.086 L1204.27,301.357 L1204.92,300.55 L1205.63,300.028 L1208.19,301.41 L1211.85,301.645 L1213.48,303.328 L1213.25,301.507 L1212.15,300.807 L1209.96,300.694 L1205.66,299.036 L1207.32,297.858 L1208.58,296.198 L1209.01,295.812 L1209.76,295.44 L1210.35,294.733 L1210.64,294.607 L1210.9,294.73 L1211.52,295.4 L1212.2,295.771 L1212.85,295.926 L1213.47,295.845 L1214.06,295.5 L1213.32,294.272 L1210.48,293.343 L1210.54,293.024 L1210.63,292.047 L1206.24,293.629 L1204.92,294.83 L1204.21,294.928 L1204.34,294.045 L1205.06,291.816 L1205.36,291.351 L1206.01,291.262 L1206.19,290.879 L1206.13,289.991 L1205.92,289.282 L1205.63,289.089 L1204.95,289.266 L1204.9,288.845 L1206.23,286.446 L1207,284.556 L1208.08,283.343 L1208.47,282.439 L1208.18,282.213 L1207.61,281.98 L1207.3,281.717 L1207.82,280.949 L1212.51,280.14 L1214.31,278.233 L1215.72,277.621 L1218.6,277.386 L1216.89,276.641 L1216.59,276.149 L1216.48,274.921 L1216.54,274.313 L1216.72,273.806 L1216.94,272.888 L1216.24,272.466 L1214.95,273.114 L1214.28,272.814 L1214.51,271.856 L1214.2,269.669 L1214.33,268.501 L1214.89,267.605 L1216.25,266.441 L1216.67,265.2 L1216.36,264.187 L1216.55,263.37 L1216.93,262.593 L1217.15,261.689 L1217,261.432 L1216.85,261.069 L1216.72,260.923 L1216.73,260.121 L1216.55,259.817 L1216.32,259.651 L1216.19,259.279 L1216.18,258.691 L1216.24,258.557 L1216.39,258.506 L1216.93,257.861 L1217.94,257.325 L1218.08,257.081 L1218.18,256.787 L1218.32,256.526 L1218.58,256.395 L1219.16,256.508 L1219.7,256.931 L1220.21,257.573 L1221.42,259.698 L1221.61,260.247 L1221.59,260.718 L1221.49,261.171 L1221.43,262.078 L1221.37,262.196 L1221.44,262.136 L1223.18,261.845 L1223.65,261.989 L1224.16,262.444 L1224.61,263.136 L1224.87,263.987 L1224.98,265.107 L1225.04,267.032 L1225.13,268.052 L1225.5,268.476 L1225.71,268.878 L1225.71,269.476 L1225.59,269.98 L1225.52,270.409 L1225.56,270.863 L1225.75,271.43 L1226.03,269.72 L1226.07,268.721 L1226.01,267.705 L1225.71,265.569 L1225.59,264.397 L1225.61,263.681 L1226.03,263.298 L1226.46,263.977 L1226.83,265.163 L1227.06,266.295 L1227.41,267.155 L1227.99,267.525 L1228.62,267.754 L1229.11,268.192 L1229.38,268.672 L1229.73,269.162 L1229.99,269.19 L1229.97,268.298 L1229.77,267.643 L1229.5,267.373 L1229.2,267.165 L1228.97,266.713 L1229.21,265.825 L1230.16,265.309 L1232.13,264.987 L1233.06,265.239 L1234.74,266.492 L1236.52,266.901 L1236.67,266.787 L1236.68,266.354 L1236.57,266.032 L1236.39,265.831 L1236.21,265.777 L1235.95,265.625 L1235,264.422 L1234.49,264.13 L1228.21,262.73 L1227.35,261.986 L1226.98,261.84 L1226.74,261.595 L1226.52,260.991 L1226.33,260.287 L1226.15,259.767 L1225.75,259.161 L1225.36,258.857 L1224.6,258.678 L1223.72,258.765 L1223.17,258.614 L1222.83,258.087 L1222.78,257.4 L1222.92,256.856 L1223.13,256.47 L1223.34,256.276 L1222.71,254.991 L1222.73,254.214 L1223.2,253.81 L1224.64,253.343 L1225.96,252.272 L1230.75,250.951 L1230.99,251.27 L1231.22,252.242 L1231.27,252.785 L1231.25,253.255 L1231.15,253.657 L1231.01,254.02 L1231.35,253.935 L1231.74,253.976 L1232.14,254.194 L1232.49,254.644 L1232.67,255.113 L1232.85,256.109 L1233,256.558 L1233.52,257.229 L1235.32,258.119 L1235.36,259.39 L1235.71,260.024 L1236.76,260.573 L1238.45,262.056 L1239.03,262.371 L1243.57,262.278 L1244.49,263.553 L1244.51,264.912 L1244.14,265.567 L1242.82,266.683 L1241.51,266.351 L1241.22,266.523 L1240.73,267.179 L1240.28,267.56 L1240.38,267.96 L1240.64,268.361 L1240.9,268.575 L1241.08,268.439 L1241.6,267.69 L1241.73,267.395 L1242.08,266.866 L1242.55,267.308 L1242.76,268.143 L1242.33,268.811 L1242.71,269.221 L1243.66,269.799 L1244.05,269.863 L1244.45,269.635 L1244.96,268.746 L1245.23,268.415 L1246.01,268.464 L1247.62,269.458 L1248.25,268.849 L1247.66,268.167 L1247.1,267.761 L1246.61,267.118 L1246.21,265.708 L1246.19,265.165 L1246.26,264.762 L1246.31,264.289 L1246.19,263.532 L1246.01,263.153 L1245.41,262.178 L1243.3,260.127 L1242.7,259.901 L1242.09,259.871 L1239.03,260.829 L1238.13,260.39 L1237.86,260.052 L1237.12,258.774 L1237.04,258.517 L1236.99,258.195 L1236.92,257.85 L1236.77,257.536 L1236.57,257.371 L1235.95,257.216 L1235.39,256.733 L1234.81,255.928 L1234.45,254.798 L1234.53,253.345 L1234.97,252.349 L1235.68,251.496 L1237.89,250.107 L1238.28,250.069 L1239.03,251.015 L1239.38,251.312 L1239.74,251.491 L1240.03,251.538 L1239.72,250.825 L1239.4,250.338 L1239.1,249.79 L1238.88,248.895 L1237.11,249.323 L1236.22,250.001 L1235.16,250.382 L1234.83,251.15 L1234.51,251.582 L1233.98,251.323 L1233.41,250.717 L1232.98,250.109 L1232.2,246.818 L1234.56,241.614 L1234.46,238.951 L1234.82,238.452 L1235.25,238.134 L1236.21,237.861 L1236.55,237.596 L1237.07,236.444 L1237.36,235.997 L1238.32,235.501 L1241.61,235.414 L1242.41,235.033 L1243.87,233.65 L1247.61,232.313 L1248.72,232.698 L1248.69,233.507 L1248.6,233.979 L1248.04,235.779 L1248.13,235.967 L1248.32,236.821 L1248.31,237.868 L1248.08,238.998 L1247.78,239.985 L1247.53,240.572 L1248.12,241.606 L1248.69,242.076 L1249.08,242.823 L1249.1,244.673 L1248.95,245.881 L1248.22,249.314 L1248.15,249.891 L1248.12,250.592 L1248.17,251.351 L1248.29,252.108 L1248.52,252.718 L1248.9,253.537 L1249.16,254.261 L1249.06,254.607 L1248.46,254.982 L1247.86,255.834 L1247.37,256.99 L1247.08,258.271 L1250.15,255.055 L1250.15,253.719 L1250.29,252.058 L1250.25,250.565 L1249.75,249.718 L1250.14,245.343 L1249.98,245.17 L1249.89,245.011 L1249.82,244.825 L1249.71,244.585 L1250.17,244.59 L1251.01,244.301 L1253.84,244.456 L1255.44,245.233 L1256.98,245.632 L1258.06,245.078 L1257.25,245.265 L1256.26,245.158 L1255.25,244.771 L1254.37,244.125 L1253.66,243.821 L1252.83,243.753 L1252.07,243.326 L1251.56,241.945 L1251.42,241.384 L1250.99,240.579 L1250.81,240.103 L1250.7,239.67 L1250.51,238.242 L1250.65,237.466 L1251.08,236.843 L1251.35,236.085 L1251.04,234.934 L1251.3,234.176 L1252.03,233.489 L1252.3,232.82 L1252.66,230.488 L1252.82,229.958 L1253.14,229.642 L1253.29,230.122 L1253.29,230.985 L1253.11,232.543 L1253.13,234.629 L1253.91,234.195 L1254.27,232.77 L1254.68,229.255 L1255.13,227.116 L1255.3,224.342 L1255.41,223.529 L1255.62,223.038 L1255.8,223.732 L1255.83,223.962 L1255.84,224.369 L1256.1,225.852 L1256.32,226.812 L1256.55,227.227 L1256.93,227.517 L1257.22,228.025 L1257.33,228.009 L1257.15,226.743 L1256.54,224.029 L1256.59,223.545 L1257.05,222.601 L1257.56,221.137 L1258.01,220.87 L1258.56,221.238 L1261.53,225.398 L1261.84,226.178 L1261.84,226.81 L1261.71,227.437 L1261.63,228.208 L1261.68,228.913 L1261.89,230.193 L1261.92,230.74 L1261.84,231.205 L1261.67,231.767 L1261.54,232.421 L1261.56,233.154 L1261.61,233.76 L1261.6,234.515 L1261.55,235.243 L1261.49,235.757 L1261.6,238.22 L1261.57,238.51 L1261.8,239.253 L1262.02,239.59 L1263.22,240.734 L1263.47,241.167 L1263.66,241.879 L1263.77,242.606 L1263.83,243.366 L1263.8,243.975 L1263.61,244.244 L1262.86,244.555 L1262.09,245.215 L1261.85,246.064 L1262.68,246.961 L1262.42,247.406 L1262.26,247.992 L1262.2,248.734 L1262.23,249.643 L1262.73,250.671 L1262.65,251.757 L1262.18,252.349 L1261.51,251.892 L1261.61,253.686 L1261.55,255.367 L1261.25,256.639 L1260.2,257.476 L1259.15,259.108 L1257.99,259.97 L1257.68,260.607 L1257.82,260.778 L1258.12,261.217 L1258.27,261.379 L1257.19,263.852 L1256.93,265.091 L1257.68,265.367 L1258.14,264.858 L1258.61,263.812 L1259.03,262.562 L1259.29,261.443 L1259.66,260.499 L1262.59,257.026 L1263.48,256.686 L1263.65,256.368 L1264.57,251.466 L1264.84,250.535 L1265.67,248.858 L1265.91,247.825 L1265.8,246.268 L1265.8,245.762 L1265.91,245.16 L1266.07,244.616 L1266.23,244.312 L1266.56,244.277 L1266.93,244.562 L1267.3,244.986 L1268.51,246.036 L1270.43,247.223 L1273.73,247.433 L1274.17,247.797 L1274.58,248.257 L1274.94,248.459 L1275.22,248.055 L1274.17,247.309 L1271.92,246.536 L1270.98,245.464 L1270.1,243.788 L1269.76,243.293 L1269.21,242.808 L1267.15,242.129 L1265.93,241.254 L1265.61,240.7 L1265.62,239.975 L1266.15,238.043 L1266.18,236.305 L1265.77,234.793 L1265.1,233.746 L1264.35,233.406 L1264.28,232.952 L1264.85,232.482 L1267.66,231.847 L1270.42,229.581 L1270.72,228.892 L1270.93,227.811 L1270.85,227.01 L1270.48,226.288 L1270.04,225.579 L1269.71,224.801 L1270.49,224.457 L1274.59,219.915 L1275.43,219.548 L1275.5,219.926 L1273.56,222.644 L1272.6,225.138 L1272.14,225.663 L1272.21,226.081 L1272.7,226.263 L1273.16,226.173 L1273.61,226.281 L1274.06,227.073 L1273.85,227.138 L1273.71,227.291 L1273.64,227.572 L1273.61,228.007 L1275.04,229.365 L1275.66,229.595 L1276.21,228.911 L1275.8,228.497 L1275.04,228.073 L1274.73,227.433 L1275.19,227.259 L1275.79,226.814 L1276.09,226.093 L1275.68,225.089 L1276.19,224.602 L1277.92,223.719 L1278.38,224.025 L1278.37,224.971 L1277.85,226.917 L1277.7,227.81 L1278.18,227.646 L1278.74,227.119 L1279.2,226.331 L1279.39,225.38 L1279.28,224.643 L1278.39,221.637 L1280.9,221.221 L1280.24,220.518 L1279.46,220.057 L1278.69,219.869 L1278.09,219.969 L1277.98,217.474 L1278.26,215.646 L1278.84,214.411 L1279.65,213.692 L1280.94,213.56 L1284.94,215.377 L1285.06,216.253 L1285.46,216.8 L1286.37,217.284 L1286.09,219.055 L1286.93,219.92 L1287.88,220.578 L1288.8,220.972 L1290.46,221.142 L1294.31,224.095 L1295.08,225.032 L1295.58,226.275 L1296.01,226.798 L1298.39,228.145 L1298.09,228.623 L1297.68,228.663 L1296.78,228.43 L1297.39,228.917 L1298.11,229.173 L1298.79,229.11 L1299.26,228.636 L1299.39,227.965 L1299.17,227.512 L1298.3,226.871 L1297.96,225.457 L1296.73,224.488 L1294.39,223.252 L1294.8,222.925 L1296.24,223.14 L1296.51,222.445 L1296.21,221.607 L1295.61,221.179 L1294.56,221.031 L1294.21,220.71 L1294,219.846 L1293.82,218.815 L1293.57,217.992 L1293.17,217.204 L1293.07,216.753 L1293.18,215.632 L1293.11,215.344 L1292.96,215.093 L1292.85,214.803 L1292.93,214.386 L1293.15,214.184 L1294.11,214.178 L1294.64,213.895 L1294.94,213.881 L1295.25,214.21 L1295.32,214.552 L1295.32,215.722 L1296.12,216.644 L1296.09,215.496 L1295.64,213.496 L1295.15,211.872 L1295.47,211.728 L1296.04,211.226 L1296.39,211.187 L1296.31,210.808 L1293.06,212.526 L1291.25,212.867 L1289.51,211.578 L1290.16,210.333 L1293.22,209.638 L1293.62,209.33 L1293.88,208.877 L1294.32,207.676 L1294.85,206.893 L1294.99,206.343 L1294.91,205.378 L1293.43,206.72 L1293.18,206.82 L1292.88,207.713 L1292.35,208.093 L1289.72,208.057 L1289.56,208.295 L1289.46,209.225 L1289.24,209.469 L1287.55,210.441 L1286.83,210.279 L1285.27,208.71 L1284.84,208.514 L1282.79,209.297 L1282.05,208.99 L1281.67,208.559 L1280.68,206.791 L1280.49,206.04 L1280.78,205.404 L1281.22,204.84 L1281.47,204.277 L1278.66,205.438 L1277.64,205.387 L1277.23,205.215 L1276.9,204.963 L1276.6,204.593 L1276.27,204.071 L1275.89,203.673 L1275.14,203.356 L1274.99,203.251 L1275.34,202.201 L1275.36,202.12 L1275.95,201.198 L1276.63,200.555 L1277.28,200.253 L1278.51,198.961 L1279.16,198.714 L1279.3,199.913 L1279.65,200.623 L1280.94,202.256 L1280.9,201.713 L1280.85,201.476 L1280.77,201.255 L1280.65,200.761 L1280.63,200.311 L1280.79,199.622 L1280.81,199.227 L1280.68,198.424 L1280.53,197.691 L1280.42,196.995 L1280.41,196.286 L1280.61,195.486 L1280.96,195.162 L1281.38,195.224 L1281.76,195.584 L1281.84,195.847 L1282.01,196.752 L1282.12,197.089 L1282.83,198.061 L1283.41,198.495 L1284.52,198.668 L1285.1,198.923 L1287.46,201.271 L1287.72,201.75 L1287.93,202.316 L1288.31,203.099 L1288.61,203.475 L1289.81,204.598 L1289.76,203.751 L1289.59,203.204 L1289.37,202.726 L1289.18,202.083 L1289.11,201.972 L1289.01,201.945 L1288.92,201.789 L1288.85,201.28 L1288.92,200.946 L1289.1,200.781 L1289.32,200.741 L1289.5,200.776 L1288.96,200.394 L1287.87,199.919 L1287.31,199.161 L1287.04,198.133 L1287.25,197.379 L1290.07,193.427 L1290.54,193.268 L1291.07,193.57 L1290.91,194.185 L1290.5,195.098 L1290.48,195.825 L1290.98,195.592 L1292.09,194.251 L1292.54,193.973 L1292.78,194.16 L1292.95,194.774 L1293.56,197.733 L1293.91,198.418 L1294.37,198.145 L1294.5,197.488 L1294.35,196.706 L1293.78,195.236 L1296.31,194.093 L1297.22,194.422 L1297.6,195.275 L1297.5,196.402 L1297.19,197.464 L1296.92,198.122 L1297.37,198.774 L1297.68,199.695 L1297.97,200.731 L1298.41,201.767 L1298.23,202.39 L1297.71,202.965 L1297.59,203.642 L1298.44,203.785 L1301.06,202.578 L1303.89,203.541 L1304.59,203.217 L1303.94,202.525 L1303.23,202.198 L1299.13,201.635 L1299.11,200.845 L1299.26,198.558 L1299.18,196.651 L1299.11,196.427 L1299.58,195.823 L1300.25,195.533 L1300.98,195.462 L1302,195.664 L1303.06,196.44 L1303.31,196.44 L1303.49,196.213 L1303.71,196.082 L1304.06,196.372 L1304.11,197.273 L1304.36,197.452 L1305.81,196.888 L1312.16,196.672 L1312.48,197.154 L1312.47,197.758 L1310.96,202.043 L1310.54,202.494 L1309.89,202.834 L1308.65,204.014 L1304.96,205.74 L1303.27,205.718 L1302.41,206 L1301.78,207.45 L1299.79,209.317 L1300.26,209.54 L1300.83,209.353 L1301.34,208.903 L1301.92,207.849 L1307.1,205.968 L1308.26,204.511 L1311.37,203.242 L1313.22,200.397 L1314.04,200.135 L1314.13,202.183 L1314.23,203.013 L1314.46,204.16 L1314.73,205.14 L1315.1,206.042 L1315.5,206.752 L1315.86,207.171 L1316.26,206.965 L1316.86,206.391 L1317.45,206.059 L1317.86,206.568 L1318.05,207.852 L1318.13,208.717 L1318.38,209.369 L1319.06,210.009 L1318.9,210.609 L1318.71,211.017 L1318.47,211.293 L1318.17,211.478 L1318.98,212.01 L1319.61,211.464 L1321.69,208.375 L1321.98,208.137 L1322.3,208.056 L1322.38,208.223 L1322.63,208.642 L1322.94,208.901 L1323.08,209.093 L1323.1,209.415 L1322.77,210.653 L1322.83,210.915 L1323.28,210.914 L1324.81,208.727 L1325.7,208.062 L1325.82,207.592 L1325.79,206.741 L1326.1,206.358 L1326.48,206.056 L1326.71,205.429 L1325.5,205.035 L1321.35,205.502 L1320.75,205.19 L1318.08,202.808 L1318.36,202.428 L1319.12,202.275 L1319.41,201.919 L1319.65,201.486 L1319.94,201.226 L1321.63,200.661 L1321.98,200.089 L1322.23,198.964 L1321.62,198.487 L1320.32,198.136 L1319.69,197.734 L1320.19,197.141 L1321.71,196.494 L1321.91,195.526 L1319.97,196.062 L1318.89,195.932 L1317.88,195.027 L1318.43,194.34 L1320.73,193.599 L1321.29,193.183 L1322.74,192.818 L1323.05,192.631 L1323,192.241 L1323.56,192.128 L1322.82,191.673 L1322.11,191.649 L1318.53,193.13 L1318.78,192.247 L1320.31,190.573 L1323.74,189.123 L1324.46,188.09 L1322.41,188.936 L1322.44,188.422 L1322.75,187.136 L1322.31,187.299 L1321.56,187.988 L1321.12,187.895 L1321.34,187.094 L1321.51,184.824 L1321.87,184.279 L1322.37,183.928 L1322.84,183.216 L1323.58,181.517 L1325.32,178.778 L1326.13,176.183 L1326.5,175.469 L1327.94,174.388 L1333.71,174.162 L1336.3,174.829 L1336.63,175.04 L1337.26,175.597 L1337.65,175.723 L1337.71,175.295 L1337.53,174.749 L1337.24,174.516 L1336.54,174.254 L1334.93,172.648 L1334.22,172.158 L1331.6,171.804 L1331.31,171.655 L1331.4,171.16 L1332.91,168.53 L1333.27,167.549 L1333.42,167.352 L1333.58,167.235 L1333.71,167.026 L1333.78,166.573 L1333.71,165.978 L1333.37,164.503 L1333.34,163.847 L1333.45,163.259 L1334,161.948 L1334.3,161.518 L1334.68,161.507 L1335.59,161.845 L1336.98,161.431 L1337.48,161.657 L1340.63,164.27 L1341.41,164.624 L1342.31,164.661 L1342.94,164.095 L1342.9,162.627 L1342.31,161.42 L1340.67,160.318 L1340.06,159.534 L1340.14,159.438 L1340.24,159.063 L1338.67,158.608 L1338.11,158.222 L1337.41,157.331 L1337.11,157.134 L1336.75,157.101 L1335.72,157.436 L1333.14,157.126 L1332.01,157.358 L1331.39,157.121 L1330.75,156.33 L1330.99,155.279 L1330.69,154.043 L1330.47,152.963 L1332.03,151.078 L1332.63,150.694 L1333.25,151.458 L1333.2,151.904 L1334.13,152.009 L1334.53,152.348 L1334.73,153.349 L1335.97,153.49 L1337.25,154.027 L1338.39,154.991 L1338.78,154.902 L1338.79,153.691 L1338.46,153.139 L1338.12,152.768 L1337.4,152.299 L1337.38,151.329 L1337.23,150.68 L1337.16,150.194 L1337.36,149.7 L1337.87,149.628 L1338.58,149.847 L1339.13,149.783 L1339.21,148.86 L1337.14,147.998 L1336.54,148.103 L1336.64,147.396 L1336.75,147.187 L1335.91,146.727 L1335.57,146.396 L1335.24,145.782 L1335.48,145.274 L1335.85,144.872 L1336.28,144.619 L1336.68,144.555 L1336.27,144.299 L1335.53,144.393 L1335.11,144.072 L1335.42,143.501 L1335.91,143.067 L1336.47,142.816 L1338.8,142.908 L1339.01,143.004 L1339.2,143.272 L1339.49,143.743 L1339.78,144.028 L1340.42,144.19 L1340.72,144.361 L1341.13,144.928 L1341.1,145.352 L1340.85,145.771 L1340.63,146.333 L1343.86,145.218 L1343.29,144.354 L1343.25,143.347 L1343.26,142.556 L1342.86,142.364 L1343.16,141.692 L1343.98,141.555 L1344.78,142.251 L1345.04,144.086 L1344.78,145.385 L1344.29,146.583 L1343.69,147.512 L1343.09,148.003 L1343.52,148.265 L1344.17,147.576 L1344.61,147.882 L1345.06,148.301 L1346,148.323 L1346.47,148.551 L1346.16,147.33 L1346.16,146.607 L1346.5,144.939 L1346.81,144.211 L1346.94,144.051 L1347.19,144.134 L1347.39,144.385 L1347.54,144.653 L1347.63,144.766 L1347.97,144.962 L1349.49,146.562 L1349.86,147.171 L1350.92,149.286 L1351.03,151.436 L1351.28,151.41 L1351.41,151.161 L1351.49,150.747 L1351.53,150.211 L1351.69,149.876 L1352.01,149.753 L1352.65,149.718 L1352.72,149.305 L1352.56,145.146 L1352.4,144.145 L1352.06,143.756 L1351.77,143.495 L1351.28,142.813 L1350.79,141.982 L1350.48,141.267 L1350.59,139.748 L1351.84,140.2 L1354.86,142.157 L1355.07,142.851 L1355.08,143.72 L1354.95,144.435 L1354.7,145.067 L1353.94,146.219 L1353.57,146.497 L1354.23,147.501 L1354.67,147.234 L1355.22,145.464 L1355.4,145.467 L1355.99,146.268 L1356.31,146.298 L1356.39,146.011 L1356.59,144.451 L1357.03,143.844 L1357.85,143.221 L1358.72,143.002 L1359.29,143.621 L1359.46,143.989 L1359.76,144.434 L1360.07,144.799 L1360.31,144.926 L1360.5,144.672 L1360.58,144.19 L1360.62,143.717 L1360.67,143.494 L1361.24,143.038 L1361.71,142.393 L1362.26,142.156 L1363.09,142.92 L1363.38,143.35 L1363.67,143.869 L1363.78,144.339 L1363.29,144.862 L1362.97,145.841 L1362.22,146.53 L1361.9,147.373 L1361.44,149.459 L1361.03,150.132 L1359.25,151.284 L1359.5,151.798 L1359.72,152.164 L1360.25,152.837 L1360.14,153.701 L1359.88,154.438 L1359.53,154.977 L1358.86,155.523 L1358.33,157.424 L1357.13,158.293 L1357.05,159.558 L1356.08,162.031 L1355.24,162.724 L1354.93,163.214 L1354.58,164.6 L1354.38,165.123 L1352.18,168.28 L1351.91,169.601 L1355.09,167.315 L1356.2,168.157 L1355.65,169.819 L1351.83,170.489 L1352.89,171.42 L1355.79,170.311 L1357.04,171.45 L1356.63,171.55 L1356.43,172.102 L1356.52,172.939 L1357.34,174.621 L1357.54,175.362 L1357.48,175.8 L1356.05,175.232 L1355.29,175.813 L1354.25,178.256 L1354.59,178.472 L1354.9,178.514 L1355.48,178.376 L1355.2,178.828 L1355.11,179.534 L1355.26,180.196 L1355.73,180.537 L1356.04,180.181 L1356.37,179.452 L1356.75,178.966 L1357.22,179.329 L1357.55,180.227 L1357.55,180.844 L1355.36,184.693 L1354.79,185.2 L1354.6,185.494 L1354.44,185.879 L1354.36,186.286 L1354.3,186.461 L1354.11,186.569 L1354.12,186.796 L1354.23,187.239 L1354.26,187.425 L1354.25,188.013 L1354.28,188.476 L1354.36,188.94 L1354.7,189.791 L1354.68,190.232 L1354.49,191.061 L1354.59,191.615 L1355.25,194.415 L1356,195.648 L1357.11,196.444 L1357.8,196.27 L1357.29,194.592 L1356.95,194.128 L1356.6,193.714 L1356.32,193.14 L1356.22,192.182 L1356.41,191.577 L1357.65,190.593 L1357.6,191.004 L1357.62,192.338 L1357.98,192.227 L1358.27,191.934 L1358.56,191.767 L1358.93,192.045 L1358.34,192.517 L1358.13,192.864 L1358.09,193.546 L1358.43,193.835 L1358.76,194.355 L1359.14,194.814 L1359.66,194.929 L1360.12,194.693 L1360.94,193.908 L1361.4,193.667 L1361.77,193.016 L1361.89,191.623 L1361.91,188.523 L1362.03,187.779 L1362.27,187.247 L1362.62,186.895 L1363.04,186.715 L1363.23,186.401 L1363.46,185.697 L1363.61,184.875 L1363.59,184.192 L1363.62,183.13 L1364.57,181.098 L1364.74,179.784 L1364.05,179.081 L1361.88,178.701 L1361.99,177.585 L1364.9,170.623 L1367.32,162.154 L1367.76,161.574 L1368.32,161.135 L1368.79,160.22 L1369.12,159.016 L1369.25,157.726 L1369.02,153.527 L1369.05,152.254 L1369.22,151.961 L1369.95,151.723 L1370.21,151.296 L1370.25,150.754 L1370.19,149.565 L1370.2,149.075 L1370.5,147.979 L1371.41,146.243 L1371.89,143.827 L1372.64,141.391 L1376.11,135.363 L1377.55,134.574 L1377.52,134.779 L1377.51,135.462 L1378.71,136.075 L1378.78,136.808 L1378.94,137.103 L1379.14,137.326 L1379.36,137.842 L1379.48,138.448 L1379.46,138.906 L1379.34,139.641 L1379.19,140.027 L1378.92,140.193 L1378.73,140.451 L1378.79,141.095 L1379.02,141.76 L1379.26,142.176 L1379.5,142.372 L1379.74,142.39 L1379.96,142.106 L1380.18,141.641 L1380.43,141.372 L1380.73,141.703 L1380.85,142.237 L1380.81,142.769 L1380.65,143.472 L1380.61,144.443 L1380.64,145.534 L1380.82,146.666 L1381.25,147.748 L1380.71,149.77 L1375.61,154.793 L1376.46,155.199 L1378.61,153.221 L1379.37,152.975 L1381.28,153.138 L1382.16,153.649 L1382.41,158.848 L1382.73,159.675 L1382.61,160.252 L1382.34,160.794 L1382.25,161.511 L1382.43,162.333 L1382.66,162.745 L1382.97,163.128 L1383.35,163.868 L1382.78,164.976 L1382.59,165.868 L1382.7,166.919 L1382.88,167.384 L1383.25,167.949 L1383.38,168.284 L1383.44,168.725 L1383.49,169.603 L1383.56,170.029 L1383.54,170.164 L1383.57,170.619 L1383.69,171.082 L1383.96,171.242 L1384.85,171.423 L1385.08,171.167 L1385.14,170.781 L1385.1,170.389 L1385.1,169.997 L1385.25,169.61 L1385.52,169.478 L1386.13,169.591 L1386.37,169.33 L1385.64,168.145 L1385.49,167.759 L1385.51,167.412 L1385.75,166.599 L1385.86,166.466 L1386.58,166.681 L1387.2,166.484 L1387.71,165.945 L1388.08,165.137 L1388.2,164.066 L1388.04,163.149 L1388.03,162.484 L1390.71,161.62 L1390.71,162.673 L1391.12,162.825 L1393,161.472 L1393.55,161.441 L1394.14,161.581 L1393.83,161.359 L1392.94,161.049 L1392.66,160.807 L1392.03,159.966 L1390.58,158.788 L1390.19,158.202 L1390.6,157.731 L1390.71,157.365 L1390.68,156.762 L1390.06,156.528 L1389.87,156.53 L1389.95,155.387 L1390.28,154.267 L1390.75,153.425 L1391.25,153.104 L1391.44,153.201 L1391.97,153.61 L1392.2,153.738 L1392.49,153.739 L1393.32,153.451 L1395.01,153.921 L1395.51,153.801 L1394.58,152.983 L1392.02,152.069 L1391.15,151.334 L1391.01,150.428 L1391.61,148.621 L1391.12,148.25 L1390.18,148.007 L1389.71,147.751 L1389.86,146.033 L1390.25,145.168 L1393.17,144.223 L1393.85,144.342 L1394.03,144.015 L1394.04,143.081 L1393.53,143.113 L1392.95,142.971 L1392.49,142.393 L1392.35,141.122 L1392.35,139.634 L1392.52,139.24 L1392.99,139.002 L1398.99,138.737 L1395.63,138.311 L1395.51,137.902 L1395.82,137.853 L1396,137.531 L1396.15,137.12 L1396.35,136.8 L1396.01,136.529 L1395.58,135.981 L1395.3,135.757 L1395.1,135.756 L1394.72,135.94 L1394.49,135.752 L1394.31,135.446 L1394.22,135.289 L1394.14,135.229 L1393.94,135.195 L1393.32,135.632 L1392.95,135.734 L1392.66,135.333 L1392.59,134.842 L1392.69,133.883 L1392.67,133.369 L1392.42,133.725 L1392.22,134.84 L1392.05,135.29 L1391.69,135.57 L1388.61,135.667 L1387.68,135.215 L1386.77,134.421 L1386.51,132.668 L1386.57,131.764 L1386.97,131.468 L1390.95,132.024 L1390.63,131.428 L1390.27,131.002 L1389.6,130.416 L1389.12,130.117 L1388.12,129.74 L1387.67,129.362 L1387.07,128.092 L1387.57,127.488 L1392.36,127.501 L1393.06,128.155 L1393.67,129.608 L1394.32,130.281 L1394.82,130.162 L1395,129.24 L1394.6,128.522 L1394.33,127.768 L1394.22,127.019 L1394.28,126.309 L1393.47,126.175 L1393.1,125.884 L1392.77,125.153 L1392.46,124.229 L1392.24,123.682 L1392.12,123.202 L1392.1,122.452 L1392.59,121.607 L1393.49,121.647 L1394.08,121.327 L1393.67,119.396 L1395.38,120.416"
         id="path1127" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['NO'],polycolors['NO'],labels['NO'], x1, polybary['NO'], x2, polybary['NO'] )
    if 'NO' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M193.939,1876.76 L194.409,1877.35 L195.138,1877.33 L196.026,1877.15 L196.985,1877.29 L197.768,1877.88 L198.303,1878.72 L198.464,1879.72 L198.121,1880.75 L197.464,1881.4 L196.607,1881.83 L195.099,1882.37 L193.482,1883.34 L191.143,1885.51 L190.462,1886.5 L190.136,1887.62 L190.4,1888.84 L191.878,1889.88 L192.111,1890.44 L192.003,1891.27 L191.495,1892.68 L191.455,1892.8 L195.306,1894 L197.386,1893.92 L197.477,1893.92 L199.318,1892.73 L199.929,1892.1 L200.618,1891.65 L201.33,1891.5 L202.032,1891.8 L202.881,1891.99 L203.485,1891.3 L203.934,1890.34 L204.304,1889.69 L205.274,1889.56 L205.029,1891.24 L204.943,1893.07 L206.384,1893.43 L206.778,1893.14 L207.024,1892.79 L207.312,1892.5 L207.824,1892.46 L209.035,1892.83 L212.002,1892.83 L213.424,1892.47 L214.094,1892.83 L215.15,1894.04 L215.859,1894.62 L218.279,1895.31 L217.155,1897.06 L217.057,1897.64 L217.698,1898.23 L218.484,1898.5 L219.483,1898.59 L220.484,1898.48 L221.281,1898.15 L221.614,1897.76 L221.754,1897.36 L221.924,1897.02 L222.326,1896.8 L222.675,1896.89 L223.256,1897.54 L223.688,1897.69 L224.271,1898.07 L224.677,1899.02 L224.883,1900.18 L224.864,1901.2 L225.577,1900.82 L226.949,1899.81 L227.654,1899.54 L228.366,1899.58 L230.007,1900.02 L230.856,1900.08 L234.327,1899.72 L236.021,1899.13 L236.578,1898.61 L237.257,1897.98 L237.569,1897.04 L237.7,1894.99 L237.982,1894.16 L238.393,1893.88 L239.202,1893.51 L240.096,1893.24 L240.772,1893.29 L242.037,1894.55 L243.163,1896.07 L244.453,1897.06 L246.218,1896.72 L247.447,1896.19 L247.808,1896.49 L248.372,1896.96 L249.296,1898.09 L250.514,1898.64 L253.391,1899.05 L254.611,1898.63 L255.281,1896.87 L256.617,1897.44 L257.737,1900.39 L259.013,1901.11 L262.282,1901.71 L263.928,1901.57 L264.805,1900.4 L265.816,1901.12 L266.277,1902 L266.252,1903.05 L265.334,1905.61 L265.606,1906.12 L266.393,1906.42 L267.489,1907.08 L266.727,1908.11 L264.583,1912.86 L264.272,1913.39 L263.841,1913.82 L263.453,1914.34 L263.482,1914.9 L263.678,1915.5 L263.77,1916.12 L263.654,1917.28 L263.646,1918.53 L264.074,1919.65 L265.259,1920.43 L265.762,1920.4 L266.519,1920.2 L267.235,1919.91 L267.614,1919.6 L267.987,1919.4 L268.612,1919.56 L272.167,1921.15 L274.235,1922.47 L276.018,1924.26 L277.539,1926.73 L277.996,1927.77 L278.342,1928.48 L275.503,1931.22 L274.061,1932.22 L273.644,1932.77 L273.58,1933.26 L273.621,1933.7 L273.517,1934.1 L271.198,1936.53 L271.087,1937.05 L269.63,1936.85 L268.941,1936.95 L268.249,1937.3 L268.381,1937.36 L268.335,1937.74 L268.183,1938.2 L268.003,1938.52 L267.806,1938.53 L267.264,1938.29 L267.064,1938.31 L265.141,1940.34 L263.352,1941.53 L261.607,1942.37 L258.668,1942.95 L258.19,1943.15 L255.169,1942.53 L254.324,1942.48 L253.668,1942.96 L251.521,1945.06 L250.987,1945.9 L250.623,1946.34 L248.707,1947.37 L247.95,1948 L247.686,1948.39 L247.488,1949.93 L246.876,1950.95 L244.487,1953.25 L243.727,1953.74 L242.439,1953.59 L239.652,1952.68 L238.599,1952.96 L238.577,1952.96 L238.156,1954.07 L237.494,1954.76 L237.429,1955.55 L237.738,1956.34 L238.864,1958.22 L239.286,1959.46 L239.766,1962.17 L239.886,1963.54 L240.161,1964.14 L240.816,1964.48 L241.247,1964.95 L241.309,1965.81 L240.958,1966.54 L240.151,1966.61 L239.766,1968.14 L239.739,1968.56 L239.76,1969.23 L239.641,1969.56 L239.445,1969.87 L239.234,1970.45 L238.988,1971.63 L238.909,1972.52 L239.084,1974.65 L239.034,1976.78 L238.903,1977.89 L238.747,1978.63 L238.206,1979.6 L236.72,1981.33 L236.245,1982.22 L236.12,1983.31 L236.393,1984.02 L236.801,1984.69 L237.117,1985.63 L237.157,1986.75 L236.874,1987.45 L235.467,1988.75 L234.165,1990.7 L234.026,1992.75 L234.743,1994.77 L236.009,1996.6 L234.888,1997.88 L234.388,1998.22 L232.065,1999.25 L231.705,1999.51 L231.436,1999.95 L231.297,2000.46 L231.169,2000.94 L230.956,2001.28 L229.973,2001.68 L229.054,2001.53 L228.157,2001.24 L227.833,2001.22 L227.224,2001.19 L224.936,2002.05 L224.016,2002.23 L223.478,2002.7 L222.724,2004.01 L222.177,2004.63 L222.378,2005.61 L222.281,2007.77 L222.664,2008.73 L223.196,2009.02 L224.401,2009.49 L224.865,2009.79 L225.272,2010.36 L225.7,2011.51 L226.47,2013.06 L227.654,2016.89 L226.569,2018.05 L226.215,2018.65 L226.165,2019.33 L225.869,2019.84 L225.284,2021.28 L225.092,2021.49 L224.564,2021.73 L224.369,2021.97 L224.302,2022.32 L224.343,2022.97 L224.27,2023.28 L224.078,2023.8 L223.831,2024.81 L223.549,2025.36 L222,2026.87 L221.498,2027.03 L220.328,2027.23 L219.897,2027.46 L219.386,2028.21 L219.315,2028.88 L219.401,2029.49 L219.369,2030.01 L219.071,2030.55 L218.302,2031.42 L217.967,2031.95 L217.835,2032.51 L217.887,2032.96 L217.846,2033.42 L217.032,2034.6 L216.283,2034.73 L209.169,2034 L207.715,2033.59 L206.825,2032.73 L205.965,2032.8 L204.099,2032.3 L203.182,2032.28 L202.337,2032.43 L201.659,2032.46 L191.6,2029.59 L191.633,2029.76 L191.81,2030.7 L192.434,2035.29 L193.117,2036.27 L194.121,2038.47 L194.751,2039.42 L196.17,2040.62 L196.466,2041.24 L196.633,2042.68 L197.228,2043.44 L198.13,2044.16 L200.003,2045.15 L200.225,2045.84 L199.947,2047.14 L199.408,2048.48 L198.852,2049.27 L198.424,2050.07 L198.166,2051.42 L198.118,2052.77 L198.317,2053.57 L199.06,2054.49 L200.808,2057.92 L199.936,2058.88 L199.6,2060.15 L199.71,2061.45 L200.173,2062.56 L201.213,2063.49 L202.459,2064.15 L203.468,2064.92 L203.808,2066.22 L203.579,2066.82 L202.812,2067.89 L202.772,2068.41 L203.159,2069.02 L203.783,2069.54 L204.482,2069.83 L205.102,2069.74 L206.383,2069.31 L207.665,2069.46 L208.836,2070.08 L209.777,2071.09 L210.39,2072 L210.833,2073.25 L210.888,2074.51 L210.345,2075.48 L210.801,2076.03 L210.764,2076.6 L210.374,2077.16 L209.781,2077.66 L207.548,2080.4 L206.178,2081.64 L204.975,2082.26 L205.39,2083.6 L204.953,2084.5 L204.831,2085.38 L204.389,2085.54 L203.705,2085.88 L203.002,2086.35 L202.152,2086.55 L199.849,2086.53 L199.309,2086.81 L198.924,2087.33 L198.119,2087.94 L197.184,2088.42 L196.405,2088.54 L195.603,2088.94 L193.112,2090.19 L192.98,2090.4 L192.404,2091.12 L192.153,2091.52 L191.392,2093.53 L191.007,2094.89 L190.901,2095.52 L190.874,2096.02 L191.03,2096.46 L191.309,2096.71 L191.547,2096.99 L191.598,2097.49 L191.163,2098.28 L188.556,2100.71 L187.052,2103.58 L186.432,2104.19 L186.819,2105.11 L186.62,2105.81 L186.016,2106.38 L185.175,2106.94 L185.764,2107.43 L186.919,2108.74 L190.656,2116.44 L190.786,2116.84 L190.93,2117.72 L191.061,2118.15 L191.321,2118.57 L192.017,2119.37 L192.213,2119.7 L193.142,2125.09 L193.524,2125.64 L194.053,2126.4 L194.783,2126.58 L196.054,2126.61 L197.335,2126.35 L198.095,2125.66 L198.336,2125.81 L198.473,2125.85 L199.807,2125.76 L201.175,2125.95 L201.737,2126.57 L201.863,2126.71 L201.024,2128.34 L200.462,2129.15 L199.399,2131.04 L198.123,2132.68 L197.328,2134.43 L196.719,2135.09 L196.022,2136.48 L194.766,2136.34 L192.259,2134.6 L191.44,2134.32 L190.923,2134.69 L190.185,2135.85 L189.549,2136.4 L189.139,2136.55 L188.467,2136.58 L187.804,2136.76 L187.222,2136.55 L186.646,2136.22 L186.029,2136.02 L185.437,2136.25 L183.837,2136.21 L183.431,2136.5 L183.357,2136.99 L183.385,2137.61 L183.28,2138.23 L182.804,2139.25 L181.159,2141.85 L180.196,2143.8 L178.375,2145.86 L173.149,2148.55 L172.035,2149.6 L170.999,2151.28 L169.966,2154.11 L169.52,2154.96 L169.085,2155.37 L167.519,2156.44 L166.996,2156.64 L167.045,2157.04 L166.945,2157.33 L166.785,2157.58 L166.665,2157.87 L166.025,2158.34 L165.818,2158.55 L165.714,2159.19 L165.773,2159.28 L165.965,2159.31 L166.047,2160.21 L166.004,2160.63 L166.034,2161.01 L166.061,2161.08 L166.334,2161.8 L166.627,2162.28 L167.32,2163.16 L167.512,2163.59 L167.621,2164.81 L167.603,2166.37 L167.468,2167.74 L167.218,2168.4 L167.668,2170.49 L167.04,2175.41 L167.111,2177.52 L166.595,2178.36 L166.55,2179.79 L166.732,2182.64 L166.814,2182.97 L166.863,2183.25 L166.97,2183.48 L167.248,2183.67 L167.171,2184.04 L166.715,2184.41 L166.026,2183.61 L165.038,2183.08 L163.924,2182.78 L162.852,2182.64 L161.709,2182.81 L159.448,2183.71 L158.568,2183.81 L155.331,2185 L154.841,2185.43 L146.383,2188.7 L144.464,2189.78 L143.758,2189.6 L143.067,2189.08 L142.078,2188.62 L139.066,2187.86 L138.302,2187.32 L137.734,2187.92 L137.225,2187.73 L136.681,2187.19 L136.263,2186.83 L136.585,2187.7 L136.663,2188.13 L136.616,2188.67 L135.858,2186.79 L134.41,2184.89 L131.338,2182.12 L129.433,2181.2 L128.037,2180.12 L126.374,2179.79 L122.333,2179.39 L120.792,2178.39 L119.077,2177.29 L117.479,2177.6 L115.293,2177.34 L113.347,2176.03 L113.229,2176.1 L110.948,2174.02 L108.95,2173.13 L106.64,2172.73 L105.873,2172.76 L105.137,2173.05 L103.231,2174.33 L100.239,2173.58 L99.6179,2173.59 L98.1814,2174.07 L97.4834,2174.19 L96.7337,2174.06 L95.5209,2174.08 L94.2386,2174.76 L93.3621,2174.13 L92.2879,2174.78 L91.823,2174.99 L91.269,2175.05 L90.1087,2175.63 L89.5591,2175.69 L88.8825,2175.23 L88.4703,2174.56 L88.0164,2174.1 L87.2171,2174.22 L87.9311,2172.61 L88.3852,2171.83 L88.6992,2171.53 L91.2682,2169.02 L92.3646,2167.54 L93.336,2166.26 L93.2942,2164.35 L94.3562,2163.75 L95.7492,2162.86 L97.042,2160.36 L96.8975,2159.2 L96.7308,2158.15 L97.8537,2156.82 L99.554,2155.63 L100.139,2154.23 L101.433,2152.42 L102.618,2149.84 L103.281,2148.29 L104.015,2145.54 L104.055,2144.03 L103.794,2142.95 L103.326,2142.02 L103.398,2141.2 L104.658,2138.69 L105.251,2137.09 L106.03,2135.65 L106.731,2135.5 L108.166,2135.6 L108.754,2134.21 L108.806,2133.98 L108.248,2134.27 L107.113,2134.28 L105.92,2134 L105.756,2132.84 L106.806,2129.84 L108.133,2126.73 L108.499,2124.7 L108.541,2123.18 L108.124,2121.45 L106.595,2120.74 L104.902,2119.01 L105.247,2118.56 L105.627,2118.43 L106.039,2118.35 L106.504,2118.06 L110.203,2113.32 L112.071,2109.23 L113.887,2106.25 L114.45,2102.94 L114.954,2099.01 L115.251,2095.54 L114.558,2092.82 L113.615,2091.21 L112.511,2089.07 L111.102,2087.19 L109.887,2086.33 L110.788,2085.76 L111.772,2086.49 L113.187,2088.49 L115.197,2089.79 L116.102,2090.62 L116,2091.43 L118.003,2091.86 L122.494,2094.36 L125.061,2094.19 L123.646,2093.74 L122.466,2093.17 L119.333,2090.62 L119.024,2090 L118.936,2089.1 L119.046,2088.23 L119.297,2087.48 L119.671,2086.97 L120.146,2086.78 L120.248,2086.33 L119.623,2085.91 L119.316,2085.08 L119.364,2084.2 L119.833,2083.59 L119.112,2083.28 L117.742,2084.97 L116.724,2085.04 L116.639,2085.41 L117.665,2085.83 L117.961,2086.84 L117.749,2087.83 L117.243,2088.2 L116.122,2087.82 L115.228,2087.32 L113.751,2086 L112.069,2084.89 L109.63,2084.74 L106.924,2086.67 L107.235,2087.06 L104.268,2087.5 L99.5093,2086.88 L98.4557,2087.01 L97.5493,2087.38 L96.6366,2087.35 L95.4877,2086.67 L95.2803,2086.6 L95.1968,2086.22 L95.3099,2085.78 L95.5252,2085.44 L95.7471,2085.32 L96.2138,2084.89 L97.9693,2083.83 L98.7049,2081.99 L99.3721,2079.12 L98.8602,2076.03 L97.9489,2073.35 L97.6968,2071.35 L98.6758,2071.1 L100.96,2071.4 L103.029,2071.73 L103.881,2072.6 L104.197,2073.46 L104.491,2073.54 L104.484,2072.22 L105.286,2072.02 L106.88,2072 L107.606,2072.1 L107.693,2071.73 L108.6,2071.06 L109.548,2070.49 L111.14,2069.79 L113.18,2068.75 L113.395,2068.67 L113.726,2068.49 L113.959,2068.47 L114.044,2068.1 L113.752,2067.17 L113.477,2065.82 L113.477,2064.5 L114.029,2063.66 L112.912,2063.45 L112.638,2062.35 L112.703,2060.8 L112.607,2059.26 L115.6,2055.5 L117.259,2053.91 L119.198,2053.05 L120.198,2053.06 L122.265,2053.41 L123.252,2053.25 L125.258,2052.39 L126.178,2051.8 L126.825,2051.04 L125.355,2051.65 L122.887,2052.68 L118.703,2052.09 L117.963,2052.45 L116.657,2053.95 L113.606,2056.67 L109.777,2061.24 L107.93,2062.55 L107.242,2063.23 L106.713,2064.15 L106.558,2065.33 L106.187,2067.37 L106.045,2068.86 L105.037,2070.12 L102.451,2070.26 L99.6401,2070.03 L96.6207,2068.93 L95.5971,2069.38 L94.4832,2070.21 L92.9124,2068.56 L91.0219,2067.13 L89.9088,2066.63 L87.8254,2066.39 L86.949,2065.83 L87.2555,2064.62 L87.808,2063.67 L87.8256,2060.87 L88.133,2059.56 L88.5303,2059.06 L89.2649,2058.51 L90.7006,2057.62 L93.4043,2053.12 L94.1383,2049.43 L94.146,2047 L94.9348,2045.77 L95.5121,2044.19 L98.1346,2041.31 L99.0965,2039.83 L100.583,2037.88 L101.316,2035.63 L102.057,2033.79 L102.128,2030.62 L101.857,2029.28 L100.587,2027.06 L100.376,2026.37 L100.955,2026.34 L102.92,2027.5 L106.92,2026.11 L107.965,2025.57 L108.949,2025.72 L109.368,2026.58 L109.82,2026.84 L110.5,2026.42 L110.329,2025.53 L109.734,2024.59 L110.784,2023.56 L112.133,2022.59 L112.675,2022.02 L113.675,2021.67 L115.222,2021.16 L116.165,2020.19 L116.871,2019.35 L118.236,2017.88 L118.58,2016.85 L118.754,2015.67 L119.714,2014.19 L120.543,2011.96 L126.517,2003.03 L136.301,1987.41 L137.196,1986.71 L137.551,1987.65 L138.238,1988.21 L140.084,1988.77 L140.188,1988.31 L139.603,1987.41 L139.024,1986.71 L138.285,1986.19 L136.221,1985.37 L135.546,1984.65 L135.282,1983.63 L135.497,1982.32 L136.482,1980.76 L138.549,1978.29 L144.446,1966.98 L145.958,1963.35 L147.482,1960.8 L148.892,1957.41 L149.058,1957.25 L149.364,1957.13 L149.097,1958.42 L148.972,1958.83 L150.251,1957.51 L151.041,1957.14 L151.6,1957.7 L151.884,1957.77 L151.971,1957.39 L151.898,1957.21 L151.791,1956.82 L151.774,1956.39 L151.975,1956.08 L152.486,1955.93 L152.704,1956.17 L152.809,1956.54 L152.995,1956.77 L153.494,1956.7 L153.987,1956.36 L154.78,1955.47 L154.149,1954.84 L153.905,1954.02 L154.182,1953.39 L155.112,1953.31 L156.202,1953.48 L157.101,1953.37 L157.933,1953.04 L158.815,1952.53 L158.027,1952.39 L156.728,1952.69 L155.702,1952.65 L155.501,1952.7 L155.243,1952.65 L154.288,1951.91 L154.068,1951.33 L154.086,1949.57 L156.41,1947.54 L156.262,1946.14 L155.954,1947.11 L155.476,1947.87 L155.087,1948.01 L155.035,1947.14 L154.099,1948.22 L153.238,1949.68 L152.534,1951.28 L152.068,1952.78 L151.638,1953.71 L150.814,1954.99 L149.897,1956.14 L149.167,1956.68 L158.36,1937.08 L158.869,1934.85 L159.27,1930.63 L159.81,1928.5 L160.82,1927.87 L160.906,1927.49 L159.288,1926.65 L158.787,1924.76 L158.673,1922.5 L158.212,1920.57 L158.623,1919.82 L158.675,1919.11 L158.581,1918.39 L158.549,1917.58 L158.991,1915.71 L158.915,1915.07 L158.635,1914.82 L158.591,1914.14 L158.57,1913.46 L158.182,1912.2 L158.131,1911.25 L158.306,1910.51 L158.634,1908.74 L159.598,1906.83 L159.393,1905.31 L159.607,1904.03 L159.603,1903.12 L159.72,1902.08 L159.566,1900.08 L159.681,1898.66 L159.616,1897.58 L159.832,1896.46 L159.72,1895.27 L159.969,1893.82 L160.286,1893.18 L161.394,1893.04 L164.288,1893.29 L165.162,1893.12 L166.008,1892.77 L166.781,1892.31 L161.404,1892.55 L160.323,1892.42 L159.754,1892.2 L159.051,1891.47 L158.647,1889.27 L159.059,1887.53 L160.08,1884.88 L160.179,1883.07 L160.518,1882.48 L160.983,1881.98 L164.82,1880.12 L166.76,1878.77 L167.83,1876.66 L168.424,1876.41 L169.567,1876.15 L170.812,1876.09 L171.881,1875.7 L172.367,1875.18 L173.154,1873.99 L173.689,1873.48 L174.592,1873.09 L175.417,1873.13 L177.079,1873.5 L177.988,1873.37 L179.791,1872.69 L180.715,1872.57 L187.945,1873.19 L188.797,1872.97 L190.199,1872.2 L190.659,1872.04 L191.019,1872.05 L191.8,1872.2 L192.706,1872.07 L194.281,1871.43 L194.552,1872.11 L194.749,1873.14 L194.013,1875.73 L193.939,1876.76"
         id="path1129" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['PT'],polycolors['PT'],labels['PT'], x1, polybary['PT'], x2, polybary['PT'] )
    if 'PT' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M651.153,1930.91 L650.662,1931.37 L650.123,1931.55 L649.565,1931.5 L649.038,1931.24 L648.939,1931.03 L648.816,1930.97 L648.671,1931.02 L648.484,1931.2 L647.811,1932.81 L646.27,1933.27 L643.325,1933.32 L643.037,1933.58 L642.8,1933.91 L642.472,1934.23 L641.907,1934.47 L641.526,1934.47 L638.773,1933.79 L638.286,1933.34 L638.369,1932.56 L637.261,1930.34 L638.079,1930.08 L638.348,1930.04 L639.326,1929.49 L639.273,1928.5 L638.541,1927.69 L637.49,1927.71 L637.717,1927.02 L638.618,1925.79 L638.648,1925.56 L638.291,1925.25 L638.41,1924.67 L638.734,1924.08 L638.978,1923.73 L640.039,1923.43 L641.006,1921.07 L642.451,1921.31 L643.853,1920.95 L644.561,1920.98 L646.875,1922.96 L647.354,1923.25 L652.527,1924.34 L651.998,1925.56 L652.691,1926.11 L653.727,1926.56 L654.057,1927.16 L654.213,1927.44 L653.046,1927.42 L652.121,1928.15 L651.479,1929.38 L651.153,1930.91"
         id="path1131" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['AD'],polycolors['AD'],labels['AD'], x1, polybary['AD'], x2, polybary['AD'] )
    if 'AD' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1606.23,1916.89 L1611.7,1922.24 L1613.63,1923.25 L1614.52,1924.92 L1615.23,1925.79 L1615.42,1926.02 L1616.53,1926.83 L1620.96,1929.01 L1621.42,1929.16 L1623.84,1929.92 L1625.26,1929.97 L1625.5,1930.12 L1625.71,1930.36 L1625.95,1930.58 L1626.28,1930.65 L1628.36,1930.2 L1628.84,1930.17 L1629.08,1930.38 L1629.25,1930.77 L1629.48,1931.16 L1629.88,1931.38 L1630.21,1931.31 L1630.87,1930.95 L1631.13,1930.92 L1631.71,1931.17 L1632.01,1931.44 L1632.17,1931.86 L1632.18,1931.89 L1632.35,1932.7 L1632.36,1932.71 L1632.83,1933.31 L1633.04,1933.96 L1633.36,1935.42 L1633.93,1936.78 L1634.65,1938.1 L1634.79,1938.52 L1634.92,1939.43 L1635.11,1939.87 L1635.18,1939.93 L1635.34,1940.06 L1635.92,1940.26 L1636.18,1940.47 L1636.59,1941.14 L1637.41,1943.17 L1638.77,1945.31 L1639.13,1945.72 L1639.67,1945.86 L1640.78,1945.72 L1641.3,1945.87 L1642.33,1947.02 L1642.6,1948.47 L1642.32,1949.99 L1641.87,1951.07 L1641.57,1951.8 L1641.41,1952.74 L1641.31,1953.08 L1641.09,1953.28 L1640.39,1953.61 L1640.2,1953.86 L1640.09,1954.61 L1640.08,1955.49 L1640.18,1956.38 L1640.39,1957.19 L1640.59,1957.51 L1641.14,1958.12 L1641.32,1958.52 L1641.34,1958.89 L1641.27,1959.95 L1641.45,1960.89 L1642.26,1965.17 L1642.37,1965.46 L1642.74,1966.04 L1642.85,1966.39 L1642.79,1966.7 L1642.35,1967.45 L1642.27,1967.86 L1642.57,1969.16 L1643.01,1970.41 L1643.03,1971.58 L1642.06,1972.64 L1637.78,1973.04 L1636.46,1973.52 L1635.74,1973.78 L1634.99,1974.69 L1634.55,1975.23 L1634.39,1977.04 L1634.37,1977.26 L1634.94,1982.16 L1634.87,1984.62 L1634.53,1985.95 L1634.08,1986.4 L1633.43,1986.2 L1632.48,1985.6 L1632.05,1985.18 L1631.76,1984.74 L1631.5,1984.48 L1631.44,1984.43 L1630.88,1984.38 L1630.2,1984.7 L1630.11,1985.17 L1630.23,1985.72 L1630.21,1986.24 L1629.56,1987.4 L1628.87,1988.2 L1628.71,1988.3 L1628.04,1988.7 L1627.02,1988.93 L1625.46,1988.96 L1624.72,1989.09 L1623.86,1989.51 L1623.23,1989.79 L1622.5,1989.99 L1621.1,1990.16 L1620.54,1990.05 L1619.45,1989.57 L1619,1989.44 L1618.41,1989.49 L1617.26,1989.9 L1616.38,1990.07 L1616.02,1990.39 L1615.73,1990.4 L1615.6,1990.2 L1615.41,1989.4 L1615.25,1989.18 L1614.8,1989.13 L1613.35,1989.18 L1611.5,1988.91 L1610.95,1988.93 L1610.58,1988.95 L1609.54,1989.21 L1608.54,1989.88 L1607.09,1991.67 L1606.16,1992.18 L1605.79,1992.11 L1604.12,1991.52 L1603.98,1991.37 L1603.85,1991.07 L1603.66,1990.79 L1603.3,1990.73 L1603.13,1990.88 L1602.73,1991.49 L1602.53,1991.69 L1599.67,1993.05 L1598.36,1994.04 L1597.98,1994.43 L1597.29,1995.12 L1596.98,1995.59 L1596.88,1996.21 L1596.99,1997.24 L1597.01,1997.44 L1596.99,1997.92 L1596.49,1998.97 L1595.06,2000.84 L1594.59,2002.09 L1594.58,2002.1 L1594.58,2002.1 L1594.58,2002.11 L1592.99,2003.61 L1592.61,2004.27 L1592.6,2004.79 L1592.69,2005.46 L1592.62,2006.18 L1592.12,2006.88 L1591.73,2007.03 L1590.82,2007.39 L1588.31,2007.22 L1587.07,2008.03 L1587.01,2008.29 L1587.02,2009.1 L1586.89,2009.47 L1586.67,2009.59 L1585.99,2009.66 L1585.72,2009.74 L1585.3,2010.19 L1584.36,2011.45 L1583.99,2011.78 L1582.64,2011.73 L1580.3,2010.19 L1580.12,2010.2 L1576.47,2010.25 L1575.32,2010.44 L1574.28,2011.11 L1572.8,2013.06 L1572.13,2013.59 L1570.58,2014.17 L1568.96,2014.41 L1568.23,2014.37 L1566.45,2014.27 L1565.22,2014.39 L1562.02,2015.88 L1555.14,2017.18 L1554.9,2015.54 L1554.36,2014.44 L1553.47,2013.8 L1551.04,2013.44 L1548.51,2013.45 L1548.47,2013.46 L1547.58,2013.82 L1546.19,2015.35 L1545.41,2015.8 L1544.18,2015.71 L1543.64,2015.37 L1542.95,2014.93 L1542.08,2013.6 L1540.71,2010.18 L1539.27,2006.27 L1538.66,2005.35 L1538.1,2005.03 L1537.68,2005.03 L1537.55,2005.03 L1536.96,2005.14 L1536.31,2005.16 L1535.92,2005.04 L1534.91,2004.73 L1534.47,2003.94 L1534.35,2002.84 L1533.94,2001.48 L1533.01,2000.15 L1530.96,1997.95 L1530.17,1996.47 L1528.94,1993.25 L1528.45,1991.42 L1528.45,1991.3 L1528.46,1990.06 L1529.16,1990.18 L1529.78,1989.67 L1530.29,1988.82 L1530.63,1987.92 L1530.82,1986.84 L1530.72,1986 L1530.65,1985.87 L1530.34,1985.28 L1529.68,1984.58 L1529.3,1984.39 L1528.51,1984.38 L1528.14,1984.14 L1528,1983.82 L1527.78,1982.83 L1527.56,1982.37 L1527.45,1982.21 L1526.82,1981.32 L1526.42,1981 L1525.87,1980.84 L1525.42,1979.97 L1525.34,1978.28 L1525.05,1977.43 L1527.26,1976.62 L1527.9,1976.24 L1528.48,1975.77 L1528.82,1975.31 L1528.99,1974.64 L1528.97,1974.51 L1528.91,1974.1 L1527.55,1971.37 L1527.15,1970.08 L1526.2,1965.67 L1526.25,1965 L1526.52,1964.12 L1526.89,1963.46 L1527.84,1962.25 L1528.07,1961.66 L1527.98,1961.27 L1527.79,1960.4 L1527.21,1958.9 L1527.18,1958.56 L1527.09,1957.52 L1528.18,1956.64 L1529.4,1957.63 L1530,1957.85 L1530.75,1957.7 L1531.06,1957.36 L1531.44,1956.35 L1531.71,1956.11 L1532.18,1956.32 L1533.22,1957.42 L1533.71,1957.69 L1534.66,1957.2 L1535.11,1956.53 L1535.47,1956.01 L1536.06,1954.58 L1536.4,1953.35 L1536.39,1953.14 L1536.38,1951.83 L1536.16,1951.27 L1535.47,1949.46 L1535.34,1948.04 L1535.5,1944.92 L1535.79,1943.52 L1536.54,1942.3 L1537.63,1941.45 L1541.76,1939.36 L1544.88,1937.77 L1546.14,1937.12 L1547.23,1936.37 L1547.7,1935.76 L1549.03,1934.02 L1550.04,1933.14 L1550.4,1933.1 L1550.72,1933.14 L1551.39,1933.38 L1553.32,1934.47 L1555.18,1935.83 L1556.1,1936.95 L1556.65,1937.77 L1556.86,1937.93 L1557.3,1938.27 L1557.68,1938.29 L1559.72,1938.39 L1560.19,1938.21 L1560.2,1937.77 L1560.2,1937.63 L1559.98,1936.38 L1559.84,1935.87 L1559.65,1935.56 L1559.51,1935.24 L1559.49,1934.69 L1561.62,1930.14 L1561.89,1929.84 L1562.15,1929.56 L1562.94,1929.37 L1564.63,1929.74 L1565.12,1929.61 L1565,1929.21 L1564.59,1928.71 L1564.45,1928.19 L1565.14,1927.65 L1565.7,1927.74 L1566.19,1928.11 L1566.68,1928.16 L1566.85,1927.88 L1567.21,1927.28 L1568.11,1927.71 L1569.05,1927.55 L1570.91,1926.74 L1571.02,1926.78 L1571.56,1926.97 L1573.83,1926.57 L1575.51,1926.76 L1576.3,1926.67 L1576.31,1926.67 L1576.94,1926.13 L1577.48,1925.22 L1578.03,1924.76 L1582.11,1921.33 L1583,1920.97 L1584.84,1920.7 L1585.15,1920.57 L1586.51,1918.97 L1586.97,1918.66 L1587.56,1918.69 L1590.2,1919.56 L1591.77,1919.86 L1592.6,1919.82 L1593.33,1919.79 L1594.87,1919.24 L1600.79,1915.55 L1601.8,1914.11 L1602.21,1913.96 L1602.46,1914.23 L1602.65,1915.27 L1602.85,1915.69 L1603.64,1916.37 L1604.46,1916.81 L1605.32,1916.99 L1606.23,1916.89"
         id="path1133" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['MK'],polycolors['MK'],labels['MK'], x1, polybary['MK'], x2, polybary['MK'] )
    if 'MK' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1268.97,2358.19 L1268.75,2359.19 L1268.03,2358.85 L1267.28,2359.76 L1267.05,2360.29 L1267.09,2361.05 L1266.25,2360.78 L1266.02,2360.66 L1264.25,2360.59 L1261.74,2359.97 L1259.48,2358.91 L1258.46,2357.5 L1258.29,2357.32 L1257.54,2356.98 L1257.37,2356.69 L1257.28,2354.78 L1256.94,2352.61 L1256.6,2351.65 L1256.03,2351.04 L1256.66,2350.44 L1257.29,2350.01 L1257.99,2349.74 L1258.8,2349.62 L1258.82,2350.08 L1258.36,2350.19 L1258.07,2350.36 L1257.43,2350.97 L1261.5,2351.19 L1262.36,2351.48 L1264.16,2352.82 L1265.67,2353.25 L1265.96,2353.72 L1266.05,2354.33 L1266.07,2354.95 L1266.35,2354.98 L1267.85,2355.51 L1268.69,2356.7 L1268.97,2358.19"
         id="path1135" />
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1253.63,2348.48 L1252.74,2348.73 L1251.79,2348.71 L1250.73,2348.49 L1249.7,2348.12 L1248.84,2347.64 L1248.97,2346.96 L1248.75,2346.3 L1248.69,2345.78 L1249.28,2345.54 L1252.38,2344.9 L1253.73,2345.31 L1255.43,2346.25 L1256.47,2347.23 L1255.87,2347.71 L1254.97,2347.79 L1253.63,2348.48"
         id="path1137" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['MT'],polycolors['MT'],labels['MT'], x1, polybary['MT'], x2, polybary['MT'] )
    if 'MT' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M256.058,2268.84 L257.038,2269.04 L256.858,2269.72 L256.836,2270.09 L256.543,2270.88 L256.228,2270.26 L256.058,2268.84"
         id="path1139" />
        <text class="label" x="10" y="100" font-size="70">%s</text>
        <line class="bar" x1="%d" y1="%d" x2="%d" y2="%d"/>
    </g>
''' % (polyclass['GI'],polycolors['GI'],labels['GI'], x1, polybary['GI'], x2, polybary['GI'] )
    if 'GI' in codes_selected: svg_selected += svg_elem.replace('<path','<path %s' % stroke_replace)
    else:                      svg += svg_elem

    svg_elem = '''
    <g class="%s" fill="%s">
      <path
         vector-effect="none"
         fill-rule="evenodd"
         d="M1417.56,181.142 L1418.49,182.015 L1420.4,184.321 L1426.38,190.147 L1426.99,190.517 L1431.55,191.192 L1432.14,193.219 L1433.03,194.274 L1434.03,194.697 L1434.94,194.815 L1442.82,196.22 L1450.71,197.583 L1451.24,198.098 L1453.91,202.391 L1459.24,208.751 L1458.47,214.671 L1453.69,226.453 L1453.66,227.309 L1453.91,228.686 L1454.93,233.227 L1455.24,234.182 L1455.7,234.734 L1459.17,237.189 L1457.35,240.485 L1455.04,242.203 L1450.68,247.822 L1449.71,248.196 L1450.11,249.281 L1450.58,249.767 L1451.05,249.945 L1455.35,248.369 L1457.28,248.257 L1459.43,249.569 L1459.64,254.4 L1459.56,256.513 L1457.3,269.176 L1457.26,270.01 L1457.53,270.9 L1458.26,272.241 L1459.23,273.82 L1468.06,288.137 L1469.04,289.214 L1470.07,289.608 L1477.36,290.5 L1484.66,291.359 L1485.56,291.69 L1486.53,292.45 L1489.65,295.815 L1489.65,295.818 L1491.1,297.391 L1497.6,304.41 L1499.63,305.837 L1508.46,310.171 L1508.23,311.115 L1508.19,311.794 L1508.35,312.592 L1509.41,316.495 L1509.56,317.309 L1509.83,320.182 L1509.79,321.387 L1504.66,336.613 L1504.58,337.291 L1504.66,338.433 L1504.65,338.91 L1504.54,339.347 L1503.79,341.13 L1500.6,357.784 L1500.6,357.786 L1500.59,357.789 L1500.59,357.793 L1501,363.102 L1503.2,367.049 L1511.7,374.87 L1513.67,377.79 L1513.75,377.897 L1514.79,378.853 L1515.03,379.36 L1515.24,379.939 L1515.55,380.528 L1515.89,380.934 L1520.12,384.746 L1520.68,385.704 L1521.77,388.132 L1522.26,389.239 L1523.36,391.179 L1524.54,392.939 L1527.04,395.996 L1532.52,401.374 L1535.45,403.597 L1536.55,404.941 L1537.8,407.146 L1539.99,411.556 L1540.56,412.524 L1542.34,414.762 L1542.94,415.621 L1544.64,418.405 L1546.35,420.681 L1546.9,421.704 L1549.4,427.02 L1549.99,428.466 L1550.35,429.778 L1550.15,430.437 L1549.21,430.372 L1547.36,429.846 L1542.02,433.397 L1540.85,434.963 L1541.97,435.983 L1544.28,436.863 L1545.45,437.822 L1545.38,438.688 L1544.15,441.703 L1544.08,442.671 L1544.05,443.032 L1544.06,443.048 L1544.56,445.324 L1546.56,450.542 L1546.98,452.788 L1545.47,457.132 L1545.23,459.424 L1546.78,460.288 L1550.28,459.123 L1552.19,459.368 L1553.07,461.601 L1552.98,462.168 L1552.81,462.638 L1552.72,463.122 L1552.88,463.722 L1553.36,464.157 L1553.84,464.126 L1554.21,464.233 L1554.35,465.074 L1553.99,466.121 L1550.42,468.197 L1549.68,469.438 L1549.47,471.319 L1550.02,473.965 L1552.77,478.672 L1555.7,482.259 L1558.62,484.25 L1561.33,484.156 L1566.07,482.791 L1567.25,483.009 L1567.83,483.33 L1567.99,483.698 L1567.99,484.172 L1568.08,484.794 L1568.27,485.573 L1568.33,485.951 L1568.5,486.264 L1569.01,486.843 L1569.43,487.22 L1570.82,488.086 L1571.63,489.107 L1571.83,490.331 L1571.63,491.486 L1571.22,492.31 L1567.52,495.817 L1567.41,496.097 L1567.42,496.479 L1567.58,496.628 L1567.78,496.672 L1567.89,496.743 L1568.27,497.772 L1568.53,498.295 L1568.9,498.805 L1568.9,498.983 L1568.94,499.162 L1569.01,499.323 L1569.11,499.476 L1569.44,499.533 L1569.53,499.667 L1569.58,500.081 L1571,500.493 L1572.27,501.397 L1572.52,502.859 L1572.44,504.55 L1572.72,506.171 L1573.61,507.14 L1574.98,507.836 L1577.2,508.437 L1581.75,508.66 L1583.36,509.278 L1584.45,510.11 L1585.39,511.035 L1586.29,511.672 L1587.93,511.633 L1588.96,511.915 L1589.83,512.632 L1590.03,513.941 L1589.95,515.258 L1590.28,516.249 L1590.91,517.119 L1591.74,518.112 L1593.21,520.469 L1593.71,522.522 L1593.54,524.5 L1593.01,526.65 L1593.01,526.653 L1593.01,526.656 L1590.1,533.799 L1589.69,538.206 L1588.73,540.99 L1585.29,543.908 L1584.21,545.322 L1583.84,545.794 L1584.88,547.202 L1593.86,553.276 L1598.5,555.118 L1600.92,557.068 L1601.81,558.027 L1602.75,558.446 L1613.34,560.882 L1619.18,564.762 L1625.91,566.867 L1626.86,567.395 L1627.78,568.202 L1628.39,569.35 L1628.79,570.443 L1629.4,571.669 L1630.07,572.791 L1630.69,573.576 L1631.71,574.302 L1633.86,575.105 L1634.93,575.708 L1637.59,577.896 L1638.44,578.312 L1638.88,578.514 L1639.92,579.379 L1640.8,580.749 L1641.73,582.015 L1642.92,582.531 L1642.12,588.06 L1641.81,590.629 L1641.82,593.409 L1642.43,599.479 L1642.43,600.948 L1641.89,603.318 L1641.83,604.855 L1642.13,607.935 L1642.11,609.359 L1641.75,610.24 L1641.74,610.242 L1641.74,610.253 L1640.59,611.641 L1640.33,612.293 L1640.23,613.038 L1640.15,614.491 L1637.71,620.263 L1637.54,621.014 L1637.5,623.238 L1637.29,624.123 L1636.97,624.83 L1634.59,628.191 L1632.91,631.509 L1632.04,632.468 L1631.68,633.041 L1631.5,633.674 L1631.35,635.87 L1629.42,641.95 L1626.91,647.689 L1626.91,647.697 L1626.9,647.698 L1624.1,657.798 L1621.63,666.546 L1619.04,671.76 L1618.74,672.865 L1618.71,675.162 L1618.47,676.313 L1615.86,682.438 L1615.19,683.409 L1613.6,684.798 L1613.13,685.58 L1612.9,686.68 L1612.82,688.302 L1612.7,689.707 L1609.88,695.435 L1609.43,697.012 L1609.29,697.811 L1609.11,698.779 L1608.2,701.95 L1606.66,704.503 L1603.13,708.445 L1601.9,710.1 L1600.45,712.798 L1599.04,714.219 L1598.74,714.945 L1598.28,716.741 L1597.08,719.281 L1596.8,720.328 L1596.67,721.706 L1596.67,721.708 L1596.67,721.715 L1596.67,721.72 L1596.03,723.828 L1593.65,725.237 L1592.6,726.611 L1589.61,732.235 L1588.05,737.059 L1585.79,739.86 L1583.08,746.646 L1582.65,747.716 L1580.78,751.138 L1579.1,754.899 L1578.45,754.473 L1577.86,753.882 L1577.29,753.609 L1576.71,754.145 L1576.88,754.312 L1577.19,754.737 L1577.36,754.906 L1576.92,755.298 L1577.02,756.005 L1577.46,756.779 L1578.03,757.388 L1577.67,757.844 L1577.53,757.96 L1578.03,758.195 L1578.24,758.182 L1577.49,759.503 L1577.15,759.788 L1574.98,759.902 L1575.07,759.145 L1575.87,758.447 L1576.19,757.861 L1574.94,757.942 L1574.37,758.151 L1574.05,758.855 L1574.01,759.652 L1574.22,760.106 L1574.6,760.53 L1575.12,761.236 L1573.91,761.142 L1570.81,759.568 L1569.87,760.326 L1570.18,760.347 L1570.34,760.459 L1570.7,760.995 L1570.56,761.212 L1570.43,761.75 L1570.32,762.017 L1570.7,762.149 L1571.05,762.335 L1571.37,762.607 L1571.67,763.004 L1570.07,763.005 L1566.91,761.591 L1565.46,761.439 L1563.14,761.889 L1562.58,761.754 L1562.31,760.904 L1563.03,760.521 L1564.63,760.357 L1563.76,759.879 L1561.21,759.006 L1561.37,757.692 L1560.89,757.464 L1560.29,757.95 L1560.07,758.808 L1560.49,759.682 L1561.13,760.504 L1561.46,761.314 L1560.95,762.163 L1560.68,761.386 L1560.44,761.219 L1559.9,761.539 L1559.48,761.432 L1558.84,760.654 L1558.45,760.539 L1558.29,760.785 L1558.22,761.734 L1558.06,761.996 L1557.73,761.893 L1557.02,761.145 L1556.71,760.97 L1556.45,761.1 L1556.16,761.429 L1555.98,761.844 L1556.07,762.248 L1556.32,762.806 L1556.22,763.192 L1556.02,763.555 L1555.96,764.958 L1555.39,765.388 L1553.99,765.609 L1554.97,766.411 L1555.43,766.997 L1555.52,767.703 L1555.26,767.627 L1554.07,766.782 L1553.67,766.392 L1551.18,768.086 L1550.3,768.318 L1550.43,767.886 L1550.48,767.43 L1550.5,766.464 L1549.96,766.563 L1549.39,766.899 L1548.9,767.421 L1548.6,768.05 L1548.61,768.509 L1548.93,769.541 L1548.98,769.964 L1548.83,770.261 L1548.3,770.847 L1548.11,771.275 L1547.72,771.484 L1547.02,770.991 L1546.25,770.292 L1545.67,769.881 L1545.07,769.876 L1544.4,770.079 L1543.96,770.506 L1544.03,771.157 L1543.46,771.549 L1542.57,771.318 L1541.61,770.697 L1540.8,769.926 L1540.27,768.811 L1540.61,768.03 L1541.26,767.377 L1541.69,766.646 L1542.08,764.758 L1542.09,764.328 L1541.65,762.974 L1541.61,762.443 L1541.82,761.719 L1542.89,760.285 L1544.19,760.178 L1547.27,761.514 L1546.03,759.316 L1544.64,757.888 L1543.26,757.471 L1542.09,758.318 L1541.61,759.37 L1541.02,761.885 L1539.68,767.604 L1539.91,769.717 L1541.26,771.379 L1541.02,772.489 L1540.86,772.953 L1540.65,773.286 L1539.81,773.498 L1539.66,773.816 L1540.14,774.731 L1539.77,774.688 L1539.39,774.479 L1538.6,773.774 L1538.81,774.474 L1538.8,774.87 L1538.62,775.062 L1538.29,775.169 L1538,775.444 L1538.03,775.888 L1538.12,776.323 L1537.98,776.561 L1537.32,776.236 L1536.66,775.335 L1536.07,774.407 L1535.69,774.02 L1535.26,773.747 L1534.59,773.14 L1533.93,772.657 L1533.52,772.77 L1533.63,773.531 L1534.25,774.297 L1535.51,775.428 L1532.53,775.58 L1532.29,775.699 L1531.86,776.518 L1531.65,776.73 L1531.31,776.369 L1529.8,775.399 L1528.69,775.08 L1528.22,774.853 L1526.78,773.57 L1525.6,772.994 L1524.43,772.721 L1523.54,772.848 L1523.65,773.302 L1525.49,773.72 L1525.91,774.062 L1526.81,774.987 L1527.68,775.247 L1528.02,775.721 L1528.27,776.324 L1528.42,776.84 L1528.27,777.218 L1527.8,777.258 L1527.5,777.402 L1527.84,778.094 L1528.28,778.498 L1529.6,779.004 L1530.44,779.949 L1530.8,780.28 L1531.23,780.347 L1531.51,780.632 L1531.64,781.368 L1531.62,782.144 L1531.44,782.537 L1529.99,783.313 L1529.62,781.927 L1527.88,780.251 L1527.95,779.386 L1527.58,779.49 L1527.39,779.838 L1527.24,780.218 L1527,780.447 L1526.64,780.467 L1526.33,780.354 L1524.09,778.92 L1523.33,778.585 L1522.5,778.446 L1523.93,779.679 L1524.74,780.181 L1525.37,780.42 L1525.37,781.728 L1524.98,783.166 L1525.14,783.991 L1525.65,784.845 L1526.91,786.248 L1527.47,787.412 L1526.83,787.497 L1526.34,787.389 L1525.88,787.087 L1525.31,786.587 L1525.44,786.119 L1524.81,785.524 L1524.16,785.431 L1523.5,785.508 L1522.85,785.396 L1521.62,784.311 L1521.02,784.023 L1520.68,784.569 L1521.13,785.603 L1522.4,786.16 L1524.71,786.726 L1522.85,788.07 L1522.94,788.45 L1522.45,788.871 L1521.54,788.634 L1518.46,786.387 L1518.18,786.016 L1518.02,785.173 L1518.11,784.554 L1518.41,784.184 L1517.68,783.932 L1517.58,783.477 L1519.23,783.103 L1518.49,781.889 L1517.81,781.927 L1517.25,782.588 L1516.86,783.237 L1516.2,783.969 L1514.04,785.19 L1514.13,785.566 L1514.66,785.744 L1514.97,786.349 L1515.12,787.186 L1515.17,788.048 L1515.08,788.263 L1514.88,788.385 L1514.73,788.619 L1514.8,789.684 L1514.7,790.05 L1514.5,790.292 L1510.23,791.305 L1510.07,790.383 L1509.59,790.029 L1508.39,789.955 L1508.52,790.223 L1508.79,790.909 L1508.91,791.196 L1508.06,791.737 L1505.24,792.403 L1505.1,792.521 L1505.1,792.764 L1505.27,793.469 L1505.26,793.516 L1505.19,793.829 L1504.81,793.767 L1504.17,793.473 L1504.25,793.994 L1505.51,794.452 L1506.01,794.863 L1505.78,795.134 L1505.38,795.242 L1505.99,795.742 L1505.49,796.323 L1503.77,797.185 L1503.08,797.295 L1503.06,796.844 L1503.06,796.645 L1503.09,796.376 L1501.96,797.474 L1501.75,798.061 L1502.44,798.312 L1502.33,798.728 L1501.84,798.817 L1501.48,799.47 L1500.87,799.378 L1500.32,798.771 L1500.61,798.246 L1500.74,797.311 L1500.66,796.336 L1500.29,795.669 L1499.65,795.696 L1499.4,796.608 L1499.35,799.692 L1499.15,800.255 L1496.6,801.017 L1495.92,800.929 L1495.42,800.261 L1496.46,799.663 L1496.15,798.571 L1495.14,797.831 L1494.11,798.31 L1495.11,799.887 L1494.72,800.253 L1494.08,801.163 L1493.73,801.498 L1492.2,802.062 L1491.27,802.638 L1491.05,802.837 L1490.92,803.164 L1490.95,803.619 L1491.24,804.373 L1491.16,804.667 L1490.74,804.771 L1490.41,804.289 L1490.15,803.616 L1489.92,803.14 L1489.47,802.774 L1487.35,802.084 L1486.86,802.061 L1486.75,802.288 L1487.24,802.865 L1487.84,803.185 L1488.31,803.278 L1488.56,803.544 L1488.45,804.366 L1488.15,804.882 L1487.76,805.167 L1487.44,805.547 L1487.37,806.338 L1489.26,805.066 L1489.22,805.679 L1489.05,806.153 L1488.58,806.958 L1489.31,807.282 L1489.39,807.664 L1487.66,809.622 L1486.85,811.684 L1486.37,812.384 L1485.74,812.782 L1484.85,813.024 L1484.92,812.132 L1485.36,809.947 L1485.58,809.335 L1485.02,809.393 L1484.41,810.291 L1483.4,812.454 L1482.95,811.452 L1482.49,811.899 L1482.04,812.836 L1481.59,813.303 L1481.56,812.243 L1481.82,811.709 L1482.17,811.308 L1482.46,810.657 L1482.45,810.206 L1482.31,809.666 L1482.21,809.021 L1482.27,808.27 L1480.77,808.458 L1480.09,808.829 L1479.71,809.672 L1480.23,810.042 L1479.42,810.927 L1479.18,811.093 L1478.77,811.2 L1477.6,810.978 L1476.74,811.06 L1476.25,811.479 L1476.35,812.006 L1477.24,812.404 L1476.74,812.891 L1476.2,812.839 L1475.63,812.569 L1475.05,812.41 L1474.79,812.488 L1473.77,813.419 L1473.48,813.489 L1473.04,813.254 L1472.69,814.355 L1472.11,814.395 L1471.76,813.799 L1472.12,813.003 L1472.02,812.546 L1471.48,812.749 L1471.24,813.226 L1471.11,813.867 L1470.9,814.56 L1470.53,815.159 L1470.13,815.509 L1468.13,816.309 L1466.34,818.996 L1465.32,818.471 L1460.42,820.157 L1459.96,819.959 L1459.48,819.273 L1459.11,819.067 L1458.75,819.142 L1456.82,820.124 L1455.65,821.042 L1454.93,821.221 L1455.09,820.22 L1455.92,817.54 L1456.27,816.962 L1457.1,816.51 L1457.46,816.103 L1457.43,815.667 L1457.11,815.157 L1456.95,814.633 L1456.92,814.04 L1457.06,813.32 L1456.02,813.935 L1455.72,814.257 L1455.57,814.785 L1455.56,815.447 L1455.46,816.043 L1455.05,816.367 L1454.79,816.692 L1454.79,817.358 L1454.84,818.143 L1454.78,818.835 L1454.56,819.247 L1453.97,819.886 L1453.76,820.318 L1453.08,822.293 L1452.17,823.893 L1449.72,825.956 L1449.4,826.769 L1449.62,827.9 L1450.16,828.809 L1450.77,828.95 L1450.85,829.37 L1445.73,831.245 L1445.42,831.69 L1443.46,831.714 L1442.78,831.997 L1442.25,832.439 L1440.48,833.218 L1439.93,833.194 L1440.41,832.694 L1440.92,831.122 L1441.31,830.722 L1442.52,830.244 L1446.47,827.312 L1447.69,826.92 L1447.22,825.65 L1447.27,825.164 L1447.62,825.114 L1448.02,825.156 L1449.04,824.932 L1449.72,824.397 L1450.23,823.421 L1450.72,821.895 L1450.94,820.529 L1451.1,818.589 L1451.05,817.291 L1450.62,817.848 L1449.49,822.781 L1448.9,824.078 L1448.29,824.194 L1448.3,823.982 L1448.28,823.564 L1448.31,823.317 L1447.95,823.091 L1447.59,823.032 L1447.27,823.172 L1447.03,823.56 L1447.21,823.98 L1445.91,823.808 L1445.69,824.447 L1445.71,825.068 L1445.5,825