"""Database utilities and repository layer for FraiseQL using psycopg and connection pooling."""

import logging
from collections.abc import Awaitable, Callable, Mapping
from dataclasses import dataclass
from typing import Any, Optional, TypeVar, Union, get_args, get_origin

from psycopg.rows import dict_row
from psycopg.sql import SQL, Composed
from psycopg_pool import AsyncConnectionPool

from fraiseql.audit import get_security_logger
from fraiseql.core.rust_pipeline import (
    RustResponseBytes,
    execute_via_rust_pipeline,
)
from fraiseql.utils.casing import to_snake_case

logger = logging.getLogger(__name__)

T = TypeVar("T")

# Type registry for development mode
_type_registry: dict[str, type] = {}

# Table metadata registry - stores column information at registration time
# This avoids expensive runtime introspection
_table_metadata: dict[str, dict[str, Any]] = {}


@dataclass
class DatabaseQuery:
    """Encapsulates a SQL query, parameters, and fetch flag."""

    statement: Composed | SQL
    params: Mapping[str, object]
    fetch_result: bool = True


def register_type_for_view(
    view_name: str,
    type_class: type,
    table_columns: set[str] | None = None,
    has_jsonb_data: bool | None = None,
    jsonb_column: str | None = None,
) -> None:
    """Register a type class for a specific view name with optional metadata.

    This is used in development mode to instantiate proper types from view data.
    Storing metadata at registration time avoids expensive runtime introspection.

    Args:
        view_name: The database view name
        type_class: The Python type class decorated with @fraise_type
        table_columns: Optional set of actual database columns (for hybrid tables)
        has_jsonb_data: Optional flag indicating if table has a JSONB 'data' column
        jsonb_column: Optional name of the JSONB column (defaults to "data")
    """
    _type_registry[view_name] = type_class
    logger.debug(f"Registered type {type_class.__name__} for view {view_name}")

    # Store metadata if provided
    if table_columns is not None or has_jsonb_data is not None or jsonb_column is not None:
        metadata = {
            "columns": table_columns or set(),
            "has_jsonb_data": has_jsonb_data or False,
            "jsonb_column": jsonb_column,  # Always store the jsonb_column value
        }
        _table_metadata[view_name] = metadata
        logger.debug(
            f"Registered metadata for {view_name}: {len(table_columns or set())} columns, "
            f"jsonb={has_jsonb_data}, jsonb_column={jsonb_column}"
        )


class FraiseQLRepository:
    """Asynchronous repository for executing SQL queries via a pooled psycopg connection.

    Rust-first architecture (v1+): Always uses Rust transformer for optimal performance.
    No mode detection or branching - single execution path.
    """

    def __init__(self, pool: AsyncConnectionPool, context: Optional[dict[str, Any]] = None) -> None:
        """Initialize with an async connection pool and optional context."""
        self._pool = pool
        self.context = context or {}
        # Get query timeout from context or use default (30 seconds)
        self.query_timeout = self.context.get("query_timeout", 30)
        # Cache for type names to avoid repeated registry lookups
        self._type_name_cache: dict[str, Optional[str]] = {}

    def _get_cached_type_name(self, view_name: str) -> Optional[str]:
        """Get cached type name for a view, or lookup and cache it if not found.

        This avoids repeated registry lookups for the same view across multiple queries.
        """
        # Check cache first
        if view_name in self._type_name_cache:
            return self._type_name_cache[view_name]

        # Lookup and cache the type name
        type_name = None
        try:
            type_class = self._get_type_for_view(view_name)
            if hasattr(type_class, "__name__"):
                type_name = type_class.__name__
        except Exception:
            # If we can't get the type, continue without type name
            pass

        # Cache the result (including None for failed lookups)
        self._type_name_cache[view_name] = type_name
        return type_name

    async def _set_session_variables(self, cursor_or_conn) -> None:
        """Set PostgreSQL session variables from context.

        Sets app.tenant_id, app.contact_id, app.user_id, and app.is_super_admin
        session variables if present in context.
        Uses SET LOCAL to scope variables to the current transaction.

        Args:
            cursor_or_conn: Either a psycopg cursor or an asyncpg connection
        """
        from psycopg.sql import SQL, Literal

        # Check if this is a cursor (psycopg) or connection (asyncpg)
        is_cursor = hasattr(cursor_or_conn, "execute") and hasattr(cursor_or_conn, "fetchone")

        if "tenant_id" in self.context:
            if is_cursor:
                await cursor_or_conn.execute(
                    SQL("SET LOCAL app.tenant_id = {}").format(
                        Literal(str(self.context["tenant_id"]))
                    )
                )
            else:
                # asyncpg connection
                await cursor_or_conn.execute(
                    "SET LOCAL app.tenant_id = $1", str(self.context["tenant_id"])
                )

        if "contact_id" in self.context:
            if is_cursor:
                await cursor_or_conn.execute(
                    SQL("SET LOCAL app.contact_id = {}").format(
                        Literal(str(self.context["contact_id"]))
                    )
                )
            else:
                # asyncpg connection
                await cursor_or_conn.execute(
                    "SET LOCAL app.contact_id = $1", str(self.context["contact_id"])
                )
        elif "user" in self.context:
            # Fallback to 'user' if 'contact_id' not set
            if is_cursor:
                await cursor_or_conn.execute(
                    SQL("SET LOCAL app.contact_id = {}").format(Literal(str(self.context["user"])))
                )
            else:
                # asyncpg connection
                await cursor_or_conn.execute(
                    "SET LOCAL app.contact_id = $1", str(self.context["user"])
                )

        # RBAC-specific session variables for Row-Level Security
        if "user_id" in self.context:
            if is_cursor:
                await cursor_or_conn.execute(
                    SQL("SET LOCAL app.user_id = {}").format(Literal(str(self.context["user_id"])))
                )
            else:
                # asyncpg connection
                await cursor_or_conn.execute(
                    "SET LOCAL app.user_id = $1", str(self.context["user_id"])
                )

        # Set super_admin flag based on user roles
        if "roles" in self.context:
            is_super_admin = (
                any(r.get("name") == "super_admin" for r in self.context["roles"])
                if isinstance(self.context["roles"], list)
                else False
            )
            if is_cursor:
                await cursor_or_conn.execute(
                    SQL("SET LOCAL app.is_super_admin = {}").format(Literal(is_super_admin))
                )
            else:
                # asyncpg connection
                await cursor_or_conn.execute("SET LOCAL app.is_super_admin = $1", is_super_admin)
        elif "user_id" in self.context:
            # If roles not provided in context, check database for super_admin role
            # This is a fallback that may be slower but ensures correctness
            try:
                user_id = self.context["user_id"]
                if is_cursor:
                    # For psycopg, we need to use the existing connection
                    # Simplified check - production needs more robust role checking
                    await cursor_or_conn.execute(
                        SQL(
                            "SET LOCAL app.is_super_admin = EXISTS (SELECT 1 FROM "
                            "user_roles ur INNER JOIN roles r ON ur.role_id = r.id "
                            "WHERE ur.user_id = {} AND r.name = 'super_admin')"
                        ).format(Literal(str(user_id)))
                    )
                else:
                    # asyncpg connection
                    result = await cursor_or_conn.fetchval(
                        "SELECT EXISTS (SELECT 1 FROM user_roles ur INNER JOIN "
                        "roles r ON ur.role_id = r.id WHERE ur.user_id = $1 AND "
                        "r.name = 'super_admin')",
                        str(user_id),
                    )
                    await cursor_or_conn.execute("SET LOCAL app.is_super_admin = $1", result)
            except Exception:
                # If role checking fails, default to False for security
                if is_cursor:
                    await cursor_or_conn.execute(
                        SQL("SET LOCAL app.is_super_admin = {}").format(Literal(False))
                    )
                else:
                    await cursor_or_conn.execute("SET LOCAL app.is_super_admin = $1", False)

    async def run(self, query: DatabaseQuery) -> list[dict[str, object]]:
        """Execute a SQL query using a connection from the pool.

        Args:
            query: SQL statement, parameters, and fetch flag.

        Returns:
            List of rows as dictionaries if `fetch_result` is True, else an empty list.
        """
        try:
            async with (
                self._pool.connection() as conn,
                conn.cursor(row_factory=dict_row) as cursor,
            ):
                # Set statement timeout for this query
                if self.query_timeout:
                    # Use literal value, not prepared statement parameters
                    # PostgreSQL doesn't support parameters in SET LOCAL
                    timeout_ms = int(self.query_timeout * 1000)
                    await cursor.execute(
                        f"SET LOCAL statement_timeout = '{timeout_ms}ms'",
                    )

                # Set session variables from context
                await self._set_session_variables(cursor)

                # Handle statement execution based on type and parameter presence
                if isinstance(query.statement, Composed) and not query.params:
                    # Composed objects without params have only embedded literals
                    # This fixes the "%r" placeholder bug from WHERE clause generation
                    await cursor.execute(query.statement)
                elif isinstance(query.statement, (Composed, SQL)) and query.params:
                    # Composed/SQL objects with params - pass parameters normally
                    # This handles legitimate cases like SQL.format() with remaining placeholders
                    await cursor.execute(query.statement, query.params)
                elif isinstance(query.statement, SQL):
                    # SQL objects without params execute directly
                    await cursor.execute(query.statement)
                else:
                    # String statements use parameters normally
                    await cursor.execute(query.statement, query.params)
                if query.fetch_result:
                    return await cursor.fetchall()
                return []
        except Exception as e:
            logger.exception("❌ Database error executing query")

            # Log query timeout specifically
            error_msg = str(e)
            if "statement timeout" in error_msg or "canceling statement" in error_msg:
                security_logger = get_security_logger()
                security_logger.log_query_timeout(
                    user_id=self.context.get("user_id"),
                    execution_time=self.query_timeout,
                    metadata={
                        "error": str(e),
                        "query_type": "database_query",
                    },
                )

            raise

    async def run_in_transaction(
        self,
        func: Callable[..., Awaitable[T]],
        *args: object,
        **kwargs: object,
    ) -> T:
        """Run a user function inside a transaction with a connection from the pool.

        The given `func` must accept the connection as its first argument.
        On exception, the transaction is rolled back.

        Example:
            async def do_stuff(conn):
                await conn.execute("...")
                return ...

            await repo.run_in_transaction(do_stuff)

        Returns:
            Result of the function, if successful.
        """
        async with self._pool.connection() as conn, conn.transaction():
            return await func(conn, *args, **kwargs)

    def get_pool(self) -> AsyncConnectionPool:
        """Expose the underlying connection pool."""
        return self._pool

    async def execute_function(
        self,
        function_name: str,
        input_data: dict[str, object],
    ) -> dict[str, object]:
        """Execute a PostgreSQL function and return the result.

        Args:
            function_name: Fully qualified function name (e.g., 'graphql.create_user')
            input_data: Dictionary to pass as JSONB to the function

        Returns:
            Dictionary result from the function (mutation_result type)
        """
        import json

        # Check if this is psycopg pool or asyncpg pool
        if hasattr(self._pool, "connection"):
            # psycopg pool
            async with (
                self._pool.connection() as conn,
                conn.cursor(row_factory=dict_row) as cursor,
            ):
                # Set statement timeout for this query
                if self.query_timeout:
                    # Use literal value, not prepared statement parameters
                    # PostgreSQL doesn't support parameters in SET LOCAL
                    timeout_ms = int(self.query_timeout * 1000)
                    await cursor.execute(
                        f"SET LOCAL statement_timeout = '{timeout_ms}ms'",
                    )

                # Set session variables from context
                await self._set_session_variables(cursor)

                # Validate function name to prevent SQL injection
                if not function_name.replace("_", "").replace(".", "").isalnum():
                    msg = f"Invalid function name: {function_name}"
                    raise ValueError(msg)

                await cursor.execute(
                    f"SELECT * FROM {function_name}(%s::jsonb)",
                    (json.dumps(input_data),),
                )
                result = await cursor.fetchone()
                return result if result else {}
        else:
            # asyncpg pool
            async with self._pool.acquire() as conn:
                # Set up JSON codec for asyncpg
                await conn.set_type_codec(
                    "jsonb",
                    encoder=json.dumps,
                    decoder=json.loads,
                    schema="pg_catalog",
                )
                # Validate function name to prevent SQL injection
                if not function_name.replace("_", "").replace(".", "").isalnum():
                    msg = f"Invalid function name: {function_name}"
                    raise ValueError(msg)

                result = await conn.fetchrow(
                    f"SELECT * FROM {function_name}($1::jsonb)",
                    input_data,  # Pass the dict directly, asyncpg will encode it
                )
                return dict(result) if result else {}

    async def execute_function_with_context(
        self,
        function_name: str,
        context_args: list[object],
        input_data: dict[str, object],
    ) -> dict[str, object]:
        """Execute a PostgreSQL function with context parameters.

        Args:
            function_name: Fully qualified function name (e.g., 'app.create_location')
            context_args: List of context arguments (e.g., [tenant_id, user_id])
            input_data: Dictionary to pass as JSONB to the function

        Returns:
            Dictionary result from the function (mutation_result type)
        """
        import json

        # Validate function name to prevent SQL injection
        if not function_name.replace("_", "").replace(".", "").isalnum():
            msg = f"Invalid function name: {function_name}"
            raise ValueError(msg)

        # Build parameter placeholders
        param_count = len(context_args) + 1  # +1 for the JSONB parameter

        # Check if this is psycopg pool or asyncpg pool
        if hasattr(self._pool, "connection"):
            # psycopg pool
            if context_args:
                placeholders = ", ".join(["%s"] * len(context_args)) + ", %s::jsonb"
            else:
                placeholders = "%s::jsonb"
            params = [*list(context_args), json.dumps(input_data)]

            async with (
                self._pool.connection() as conn,
                conn.cursor(row_factory=dict_row) as cursor,
            ):
                # Set statement timeout for this query
                if self.query_timeout:
                    # Use literal value, not prepared statement parameters
                    # PostgreSQL doesn't support parameters in SET LOCAL
                    timeout_ms = int(self.query_timeout * 1000)
                    await cursor.execute(
                        f"SET LOCAL statement_timeout = '{timeout_ms}ms'",
                    )

                # Set session variables from context
                await self._set_session_variables(cursor)

                await cursor.execute(
                    f"SELECT * FROM {function_name}({placeholders})",
                    tuple(params),
                )
                result = await cursor.fetchone()
                return result if result else {}
        else:
            # asyncpg pool
            if context_args:
                placeholders = (
                    ", ".join([f"${i + 1}" for i in range(len(context_args))])
                    + f", ${param_count}::jsonb"
                )
            else:
                placeholders = "$1::jsonb"
            params = [*list(context_args), input_data]

            async with self._pool.acquire() as conn:
                # Set up JSON codec for asyncpg
                await conn.set_type_codec(
                    "jsonb",
                    encoder=json.dumps,
                    decoder=json.loads,
                    schema="pg_catalog",
                )

                # Set session variables from context
                await self._set_session_variables(conn)

                result = await conn.fetchrow(
                    f"SELECT * FROM {function_name}({placeholders})",
                    *params,
                )
                return dict(result) if result else {}

    async def _ensure_table_columns_cached(self, view_name: str) -> None:
        """Ensure table columns are cached for hybrid table detection.

        PERFORMANCE OPTIMIZATION:
        - Only introspect once per table per repository instance
        - Cache both successes and failures to avoid repeated queries
        - Use connection pool efficiently
        """
        if not hasattr(self, "_introspected_columns"):
            self._introspected_columns = {}
            self._introspection_in_progress = set()

        # Skip if already cached or being introspected (avoid race conditions)
        if view_name in self._introspected_columns or view_name in self._introspection_in_progress:
            return

        # Mark as in progress to prevent concurrent introspections
        self._introspection_in_progress.add(view_name)

        try:
            await self._introspect_table_columns(view_name)
        except Exception:
            # Cache failure to avoid repeated attempts
            self._introspected_columns[view_name] = set()
        finally:
            self._introspection_in_progress.discard(view_name)

    async def find(
        self, view_name: str, field_name: str | None = None, info: Any = None, **kwargs
    ) -> RustResponseBytes:
        """Find records using unified Rust-first pipeline.

        PostgreSQL → Rust → HTTP (zero Python string operations).

        Args:
            view_name: Database table/view name
            field_name: GraphQL field name for response wrapping
            info: Optional GraphQL resolve info for field selection
            **kwargs: Query parameters (where, limit, offset, order_by)

        Returns:
            RustResponseBytes ready for HTTP response
        """
        # 1. Extract field paths from GraphQL info
        field_paths = None
        if info:
            from fraiseql.core.ast_parser import extract_field_paths_from_info
            from fraiseql.utils.casing import to_snake_case

            field_paths = extract_field_paths_from_info(info, transform_path=to_snake_case)

        # 2. Get JSONB column from cached metadata (NO sample query!)
        jsonb_column = None  # default to None (use row_to_json)
        if view_name in _table_metadata:
            metadata = _table_metadata[view_name]
            # For hybrid tables with JSONB data, always use the data column
            if metadata.get("has_jsonb_data", False):
                jsonb_column = metadata.get("jsonb_column") or "data"
            elif "jsonb_column" in metadata:
                jsonb_column = metadata["jsonb_column"]

        # 3. Build SQL query
        query = self._build_find_query(
            view_name,
            field_paths=field_paths,
            info=info,
            jsonb_column=jsonb_column,
            **kwargs,
        )

        # 4. Get type name for Rust transformation
        type_name = self._get_cached_type_name(view_name)

        # 5. Execute via Rust pipeline (ALWAYS)
        async with self._pool.connection() as conn:
            return await execute_via_rust_pipeline(
                conn,
                query.statement,
                query.params,
                field_name or view_name,  # Use view_name as default field_name
                type_name,
                is_list=True,
                field_paths=field_paths,  # NEW: Pass field paths for Rust-side projection!
            )

    async def find_one(
        self, view_name: str, field_name: str | None = None, info: Any = None, **kwargs
    ) -> RustResponseBytes:
        """Find single record using unified Rust-first pipeline.

        Args:
            view_name: Database table/view name
            field_name: GraphQL field name for response wrapping
            info: Optional GraphQL resolve info
            **kwargs: Query parameters (id, where, etc.)

        Returns:
            RustResponseBytes ready for HTTP response
        """
        # 1. Extract field paths
        field_paths = None
        if info:
            from fraiseql.core.ast_parser import extract_field_paths_from_info
            from fraiseql.utils.casing import to_snake_case

            field_paths = extract_field_paths_from_info(info, transform_path=to_snake_case)

        # 2. Get JSONB column from cached metadata
        jsonb_column = None  # default to None (use row_to_json)
        if view_name in _table_metadata:
            metadata = _table_metadata[view_name]
            # For hybrid tables with JSONB data, always use the data column
            if metadata.get("has_jsonb_data", False):
                jsonb_column = metadata.get("jsonb_column") or "data"
            elif "jsonb_column" in metadata:
                jsonb_column = metadata["jsonb_column"]

        # 3. Build query (automatically adds LIMIT 1)
        query = self._build_find_one_query(
            view_name,
            field_paths=field_paths,
            info=info,
            jsonb_column=jsonb_column,
            **kwargs,
        )

        # 4. Get type name
        type_name = self._get_cached_type_name(view_name)

        # 5. Execute via Rust pipeline (ALWAYS)
        async with self._pool.connection() as conn:
            return await execute_via_rust_pipeline(
                conn,
                query.statement,
                query.params,
                field_name or view_name,  # Use view_name as default field_name
                type_name,
                is_list=False,
                field_paths=field_paths,  # NEW: Pass field paths for Rust-side projection!
            )

    def _extract_type(self, field_type: type) -> Optional[type]:
        """Extract the actual type from Optional, Union, etc."""
        origin = get_origin(field_type)
        if origin is Union:
            args = get_args(field_type)
            # Filter out None type
            non_none_args = [arg for arg in args if arg is not type(None)]
            if non_none_args:
                return non_none_args[0]
        return field_type if origin is None else None

    def _get_type_for_view(self, view_name: str) -> type:
        """Get the type class for a given view name."""
        # Check the global type registry
        if view_name in _type_registry:
            return _type_registry[view_name]

        # Try to find type by convention (remove _view suffix and check)
        type_name = view_name.replace("_view", "")
        for registered_view, type_class in _type_registry.items():
            if registered_view.lower().replace("_", "") == type_name.lower().replace("_", ""):
                return type_class

        available_views = list(_type_registry.keys())
        logger.error(f"Type registry state: {_type_registry}")
        raise NotImplementedError(
            f"Type registry lookup for {view_name} not implemented. "
            f"Available views: {available_views}. Registry size: {len(_type_registry)}",
        )

    def _build_find_query(
        self,
        view_name: str,
        field_paths: list[Any] | None = None,
        info: Any = None,
        jsonb_column: str | None = None,
        **kwargs,
    ) -> DatabaseQuery:
        """Build a SELECT query for finding multiple records.

        Unified Rust-first: always SELECT jsonb_column::text
        Rust handles field projection, not PostgreSQL!

        Args:
            view_name: Name of the view to query
            field_paths: Optional field paths for projection (passed to Rust)
            info: Optional GraphQL resolve info
            jsonb_column: JSONB column name to use
            **kwargs: Query parameters (where, limit, offset, order_by)
        """
        from psycopg.sql import SQL, Composed, Identifier, Literal

        where_parts = []

        # Extract special parameters
        where_obj = kwargs.pop("where", None)
        limit = kwargs.pop("limit", None)
        offset = kwargs.pop("offset", None)
        order_by = kwargs.pop("order_by", None)

        # Process where object
        if where_obj:
            if hasattr(where_obj, "to_sql"):
                where_composed = where_obj.to_sql()
                if where_composed:
                    where_parts.append(where_composed)
            elif hasattr(where_obj, "_to_sql_where"):
                # Convert GraphQL WhereInput to SQL where type, then get SQL
                sql_where_obj = where_obj._to_sql_where()
                if hasattr(sql_where_obj, "to_sql"):
                    where_composed = sql_where_obj.to_sql()
                    if where_composed:
                        where_parts.append(where_composed)
            elif isinstance(where_obj, dict):
                # Use sophisticated dict processing for complex filters
                # For now, pass None for table_columns to avoid async issues
                # The method will fall back to heuristics
                dict_where_sql = self._convert_dict_where_to_sql(where_obj, view_name, None)
                if dict_where_sql:
                    where_parts.append(dict_where_sql)

        # Process remaining kwargs as simple equality filters
        for key, value in kwargs.items():
            where_condition = Composed([Identifier(key), SQL(" = "), Literal(value)])
            where_parts.append(where_condition)

        # Handle schema-qualified table names
        if "." in view_name:
            schema_name, table_name = view_name.split(".", 1)
            table_identifier = Identifier(schema_name, table_name)
        else:
            table_identifier = Identifier(view_name)

        if jsonb_column is None:
            # For tables with jsonb_column=None, select all columns as JSON
            # This allows the Rust pipeline to extract individual fields
            query_parts = [
                SQL("SELECT row_to_json(t)::text FROM "),
                table_identifier,
                SQL(" AS t"),
            ]
        else:
            # For JSONB tables, select the JSONB column as text
            target_jsonb_column = jsonb_column or "data"
            query_parts = [
                SQL("SELECT "),
                Identifier(target_jsonb_column),
                SQL("::text FROM "),
                table_identifier,
            ]

        # Add WHERE clause
        if where_parts:
            where_sql_parts = []
            for part in where_parts:
                if isinstance(part, (SQL, Composed)):
                    where_sql_parts.append(part)
                else:
                    where_sql_parts.append(SQL(part))
            if where_sql_parts:
                query_parts.extend([SQL(" WHERE "), SQL(" AND ").join(where_sql_parts)])

        # Add ORDER BY
        if order_by:
            if hasattr(order_by, "to_sql"):
                order_sql = order_by.to_sql()
                if order_sql:
                    query_parts.extend([SQL(" ORDER BY "), order_sql])
            elif isinstance(order_by, str):
                query_parts.extend([SQL(" ORDER BY "), SQL(order_by)])

        # Add LIMIT
        if limit is not None:
            query_parts.extend([SQL(" LIMIT "), Literal(limit)])

        # Add OFFSET
        if offset is not None:
            query_parts.extend([SQL(" OFFSET "), Literal(offset)])

        statement = SQL("").join(query_parts)
        return DatabaseQuery(statement=statement, params={}, fetch_result=True)

    def _build_find_one_query(
        self,
        view_name: str,
        field_paths: list[Any] | None = None,
        info: Any = None,
        jsonb_column: str | None = None,
        **kwargs,
    ) -> DatabaseQuery:
        """Build a SELECT query for finding a single record."""
        # Force limit=1 for find_one
        kwargs["limit"] = 1
        return self._build_find_query(
            view_name,
            field_paths=field_paths,
            info=info,
            jsonb_column=jsonb_column,
            **kwargs,
        )

    async def _get_table_columns_cached(self, view_name: str) -> set[str] | None:
        """Get table columns with caching.

        Returns set of column names or None if unable to retrieve.
        """
        if not hasattr(self, "_introspected_columns"):
            self._introspected_columns = {}

        if view_name in self._introspected_columns:
            return self._introspected_columns[view_name]

        try:
            columns = await self._introspect_table_columns(view_name)
            self._introspected_columns[view_name] = columns
            return columns
        except Exception:
            return None

    def _convert_dict_where_to_sql(
        self,
        where_dict: dict[str, Any],
        view_name: str | None = None,
        table_columns: set[str] | None = None,
    ) -> Composed | None:
        """Convert a dictionary WHERE clause to SQL conditions.

        This method handles dynamically constructed where clauses used in GraphQL resolvers.
        Unlike WhereInput types (which use JSONB paths), dictionary filters use direct
        column names for regular tables.

        Args:
            where_dict: Dictionary with field names as keys and operator dictionaries as values
                       e.g., {'name': {'contains': 'router'}, 'port': {'gt': 20}}
            view_name: Optional view/table name for hybrid table detection
            table_columns: Optional set of actual table columns for accurate detection

        Returns:
            A Composed SQL object with parameterized conditions, or None if no valid conditions
        """
        from psycopg.sql import SQL, Composed

        conditions = []

        for field_name, field_filter in where_dict.items():
            if field_filter is None:
                continue

            # Convert GraphQL field names to database field names
            db_field_name = self._convert_field_name_to_database(field_name)

            if isinstance(field_filter, dict):
                # Check if this might be a nested object filter
                # (e.g., {machine: {id: {eq: value}}})
                # IMPORTANT: is_nested_object must be reset for each field
                # to prevent state carry-over between iterations
                is_nested_object = False
                if "id" in field_filter and isinstance(field_filter["id"], dict):
                    # This looks like a nested object filter
                    # Check if we have a corresponding SQL column for this relationship
                    potential_fk_column = f"{db_field_name}_id"

                    # Validate that this is likely a nested object, not a field literally named "id"
                    # True nested objects have:
                    # 1. A single "id" key (or very few keys like "id" + metadata)
                    # 2. The "id" value is a dict with operator keys
                    # 3. The field name suggests a relationship (not a scalar field)
                    looks_like_nested = (
                        len(field_filter) == 1  # Only contains "id" key
                        or (
                            len(field_filter) <= 2
                            and all(k in ("id", "__typename") for k in field_filter)
                        )
                    )

                    if table_columns and potential_fk_column in table_columns:
                        # BEST CASE: We have actual column metadata
                        # We know for sure this FK column exists, so treat as nested object
                        is_nested_object = True
                        logger.debug(
                            f"Dict WHERE: Detected nested object filter for {field_name} "
                            f"(FK column {potential_fk_column} exists)"
                        )
                    elif table_columns is None and looks_like_nested:
                        # FALLBACK CASE: No column metadata available (development/testing)
                        # Use heuristics to determine if this is a nested object
                        # RISK: If a field is literally named "id" with operator filters like
                        # {"id": {"eq": value}}, it will be treated as nested object.
                        # However, this is an unlikely naming pattern in practice.
                        is_nested_object = True
                        logger.debug(
                            f"Dict WHERE: Assuming nested object filter for {field_name} "
                            f"(table_columns=None, using heuristics). "
                            f"If incorrect, register table metadata with "
                            f"register_type_for_view()."
                        )
                    elif not looks_like_nested:
                        # Safety check: Even if table_columns is None, if the structure doesn't
                        # look like a nested object (e.g., has multiple keys beyond "id"),
                        # treat it as regular field operators
                        is_nested_object = False
                        logger.debug(
                            f"Dict WHERE: Treating {field_name} as regular field filter "
                            f"(structure doesn't match nested object pattern)"
                        )

                    if is_nested_object:
                        # Extract the filter value from the nested structure
                        id_filter = field_filter["id"]

                        # Validate that id_filter contains operator keys
                        if not isinstance(id_filter, dict) or not id_filter:
                            logger.warning(
                                f"Dict WHERE: Nested object filter for {field_name} has invalid "
                                f"id_filter structure: {id_filter}. Skipping."
                            )
                        else:
                            for operator, value in id_filter.items():
                                if value is None:
                                    continue
                                # Build condition using the FK column directly
                                condition_sql = self._build_dict_where_condition(
                                    potential_fk_column, operator, value, view_name, table_columns
                                )
                                if condition_sql:
                                    conditions.append(condition_sql)

                if not is_nested_object:
                    # Handle regular operator-based filtering: {'contains': 'router', 'gt': 10}
                    field_conditions = []

                    for operator, value in field_filter.items():
                        if value is None:
                            continue

                        # Build SQL condition using converted database field name
                        condition_sql = self._build_dict_where_condition(
                            db_field_name, operator, value, view_name, table_columns
                        )
                        if condition_sql:
                            field_conditions.append(condition_sql)

                    # Combine multiple conditions for the same field with AND
                    if field_conditions:
                        if len(field_conditions) == 1:
                            conditions.append(field_conditions[0])
                        else:
                            # Multiple conditions for same field: (cond1 AND cond2 AND ...)
                            combined_parts = []
                            for i, cond in enumerate(field_conditions):
                                if i > 0:
                                    combined_parts.append(SQL(" AND "))
                                combined_parts.append(cond)
                            conditions.append(Composed([SQL("("), *combined_parts, SQL(")")]))

            else:
                # Handle simple equality: {'status': 'active'}
                condition_sql = self._build_dict_where_condition(db_field_name, "eq", field_filter)
                if condition_sql:
                    conditions.append(condition_sql)

        # Combine all field conditions with AND
        if not conditions:
            return None

        if len(conditions) == 1:
            return conditions[0]

        # Multiple field conditions: (field1_cond AND field2_cond AND ...)
        result_parts = []
        for i, condition in enumerate(conditions):
            if i > 0:
                result_parts.append(SQL(" AND "))
            result_parts.append(condition)

        return Composed(result_parts)

    def _build_dict_where_condition(
        self,
        field_name: str,
        operator: str,
        value: Any,
        view_name: str | None = None,
        table_columns: set[str] | None = None,
    ) -> Composed | None:
        """Build a single WHERE condition using FraiseQL's operator strategy system.

        This method now uses the sophisticated operator strategy system instead of
        primitive SQL templates, enabling features like IP address type casting,
        MAC address handling, and other advanced field type detection.

        For hybrid tables (with both regular columns and JSONB data), it determines
        whether to use direct column access or JSONB path based on the actual table structure.

        Args:
            field_name: Database field name (e.g., 'ip_address', 'port', 'status')
            operator: Filter operator (eq, contains, gt, in, etc.)
            value: Filter value
            view_name: Optional view/table name for hybrid table detection
            table_columns: Optional set of actual table columns (for accurate detection)

        Returns:
            Composed SQL condition with intelligent type casting, or None if operator not supported
        """
        from psycopg.sql import SQL, Composed, Identifier, Literal

        from fraiseql.sql.operator_strategies import get_operator_registry

        try:
            # Get the operator strategy registry (contains the v0.7.1 IP filtering fixes)
            registry = get_operator_registry()

            # Determine if this field is a regular column or needs JSONB path
            use_jsonb_path = False

            if table_columns is not None:
                # We have actual column info - use it!
                # Field is JSONB if: table has 'data' column AND field is NOT a regular column
                has_data_column = "data" in table_columns
                is_regular_column = field_name in table_columns
                use_jsonb_path = has_data_column and not is_regular_column
            elif view_name:
                # Fall back to heuristic-based detection
                use_jsonb_path = self._should_use_jsonb_path_sync(view_name, field_name)

            if use_jsonb_path:
                # Field is in JSONB data column, use JSONB path
                path_sql = Composed([SQL("data"), SQL(" ->> "), Literal(field_name)])
            else:
                # Field is a regular column, use direct column name
                path_sql = Identifier(field_name)

            # Get the appropriate strategy for this operator
            # field_type=None triggers fallback detection (IP addresses, MAC addresses, etc.)
            strategy = registry.get_strategy(operator, field_type=None)

            if strategy is None:
                # Operator not supported by strategy system, fall back to basic handling
                return self._build_basic_dict_condition(
                    field_name, operator, value, use_jsonb_path=use_jsonb_path
                )

            # Use the strategy to build intelligent SQL with type detection
            # This is where the IP filtering fixes from v0.7.1 are applied
            sql_condition = strategy.build_sql(path_sql, operator, value, field_type=None)

            return sql_condition

        except Exception as e:
            # If strategy system fails, fall back to basic condition building
            logger.warning(f"Operator strategy failed for {field_name} {operator} {value}: {e}")
            return self._build_basic_dict_condition(field_name, operator, value)

    def _build_basic_dict_condition(
        self, field_name: str, operator: str, value: Any, use_jsonb_path: bool = False
    ) -> Composed | None:
        """Fallback method for basic WHERE condition building.

        This provides basic SQL generation when the operator strategy system
        is not available or fails. Used as a safety fallback.
        """
        from psycopg.sql import SQL, Composed, Identifier, Literal

        # Basic operator templates for fallback scenarios
        basic_operators = {
            "eq": lambda path, val: Composed([path, SQL(" = "), Literal(val)]),
            "neq": lambda path, val: Composed([path, SQL(" != "), Literal(val)]),
            "gt": lambda path, val: Composed([path, SQL(" > "), Literal(val)]),
            "gte": lambda path, val: Composed([path, SQL(" >= "), Literal(val)]),
            "lt": lambda path, val: Composed([path, SQL(" < "), Literal(val)]),
            "lte": lambda path, val: Composed([path, SQL(" <= "), Literal(val)]),
            "ilike": lambda path, val: Composed([path, SQL(" ILIKE "), Literal(val)]),
            "like": lambda path, val: Composed([path, SQL(" LIKE "), Literal(val)]),
            "isnull": lambda path, val: Composed(
                [path, SQL(" IS NULL" if val else " IS NOT NULL")]
            ),
        }

        if operator not in basic_operators:
            return None

        # Build path based on whether this is a JSONB field or regular column
        if use_jsonb_path:
            # Use JSONB path for fields in data column
            path_sql = Composed([SQL("data"), SQL(" ->> "), Literal(field_name)])
        else:
            # Use direct column name for regular columns
            path_sql = Identifier(field_name)

        # Generate basic condition
        return basic_operators[operator](path_sql, value)

    async def _introspect_table_columns(self, view_name: str) -> set[str]:
        """Introspect actual table columns from database information_schema.

        This provides accurate column information for hybrid tables.
        Results are cached for performance.
        """
        if not hasattr(self, "_introspected_columns"):
            self._introspected_columns = {}

        if view_name in self._introspected_columns:
            return self._introspected_columns[view_name]

        try:
            # Query information_schema to get actual columns
            # PERFORMANCE: Use a single query to get all we need
            query = """
                SELECT
                    column_name,
                    data_type,
                    udt_name
                FROM information_schema.columns
                WHERE table_name = %s
                AND table_schema = 'public'
                ORDER BY ordinal_position
            """

            async with self._pool.connection() as conn, conn.cursor() as cursor:
                await cursor.execute(query, (view_name,))
                rows = await cursor.fetchall()

                # Extract column names and identify if JSONB exists
                columns = set()
                has_jsonb_data = False

                for row in rows:
                    # Handle both dict and tuple cursor results
                    if isinstance(row, dict):
                        col_name = row.get("column_name")
                        udt_name = row.get("udt_name", "")
                    else:
                        # Tuple-based result (column_name, data_type, udt_name)
                        col_name = row[0] if row else None
                        udt_name = row[2] if len(row) > 2 else ""

                    if col_name:
                        columns.add(col_name)

                        # Check if this is a JSONB data column
                        if col_name == "data" and udt_name == "jsonb":
                            has_jsonb_data = True

                # Cache the result
                self._introspected_columns[view_name] = columns

                # Also cache whether this table has JSONB data column
                if not hasattr(self, "_table_has_jsonb"):
                    self._table_has_jsonb = {}
                self._table_has_jsonb[view_name] = has_jsonb_data

                return columns

        except Exception as e:
            logger.warning(f"Failed to introspect table {view_name}: {e}")
            # Cache empty set to avoid repeated failures
            self._introspected_columns[view_name] = set()
            return set()

    def _should_use_jsonb_path_sync(self, view_name: str, field_name: str) -> bool:
        """Check if a field should use JSONB path or direct column access.

        PERFORMANCE OPTIMIZED:
        - Uses metadata from registration time (no DB queries)
        - Single cache lookup per field
        - Fast path for registered tables
        """
        # Fast path: use cached decision if available
        if not hasattr(self, "_field_path_cache"):
            self._field_path_cache = {}

        cache_key = f"{view_name}:{field_name}"
        cached_result = self._field_path_cache.get(cache_key)
        if cached_result is not None:
            return cached_result

        # BEST CASE: Check registration-time metadata first (no DB query needed)
        if view_name in _table_metadata:
            metadata = _table_metadata[view_name]
            columns = metadata.get("columns", set())
            has_jsonb = metadata.get("has_jsonb_data", False)

            # Use JSONB path only if: has data column AND field is not a regular column
            use_jsonb = has_jsonb and field_name not in columns
            self._field_path_cache[cache_key] = use_jsonb
            return use_jsonb

        # SECOND BEST: Check if we have runtime introspected columns
        if hasattr(self, "_introspected_columns") and view_name in self._introspected_columns:
            columns = self._introspected_columns[view_name]
            has_data_column = "data" in columns
            is_regular_column = field_name in columns

            # Use JSONB path only if: has data column AND field is not a regular column
            use_jsonb = has_data_column and not is_regular_column
            self._field_path_cache[cache_key] = use_jsonb
            return use_jsonb

        # Fallback: Use fast heuristic for known patterns
        # PERFORMANCE: This avoids DB queries for common cases
        if not hasattr(self, "_table_has_jsonb"):
            self._table_has_jsonb = {}

        if view_name not in self._table_has_jsonb:
            # Quick pattern matching for known table types
            known_hybrid_patterns = ("jsonb", "hybrid")
            known_regular_patterns = ("test_product", "test_item", "users", "companies", "orders")

            view_lower = view_name.lower()
            if any(p in view_lower for p in known_regular_patterns):
                self._table_has_jsonb[view_name] = False
            elif any(p in view_lower for p in known_hybrid_patterns):
                self._table_has_jsonb[view_name] = True
            else:
                # Conservative default: assume regular table
                self._table_has_jsonb[view_name] = False

        # If no JSONB data column, always use direct access
        if not self._table_has_jsonb[view_name]:
            self._field_path_cache[cache_key] = False
            return False

        # For hybrid tables, use a small set of known regular columns
        # PERFORMANCE: Using frozenset for O(1) lookup
        REGULAR_COLUMNS = frozenset(
            {
                "id",
                "tenant_id",
                "created_at",
                "updated_at",
                "name",
                "status",
                "type",
                "category_id",
                "identifier",
                "is_active",
                "is_featured",
                "is_available",
                "is_deleted",
                "start_date",
                "end_date",
                "created_date",
                "modified_date",
            }
        )

        use_jsonb = field_name not in REGULAR_COLUMNS
        self._field_path_cache[cache_key] = use_jsonb
        return use_jsonb

    def _where_obj_to_dict(self, where_obj: Any, table_columns: set[str]) -> dict[str, Any] | None:
        """Convert a WHERE object to a dictionary for hybrid table processing.

        This method examines a WHERE object and converts it to a dictionary format
        that can be processed by our dict-based WHERE handler, which knows how to
        handle nested objects in hybrid tables correctly.

        Args:
            where_obj: The WHERE object with to_sql() method
            table_columns: Set of actual table column names

        Returns:
            Dictionary representation of the WHERE clause, or None if conversion fails
        """
        result = {}

        # Iterate through attributes of the where object
        if hasattr(where_obj, "__dict__"):
            for field_name, field_value in where_obj.__dict__.items():
                if field_value is None:
                    continue

                # Skip special fields
                if field_name.startswith("_"):
                    continue

                # Check if this is a nested object filter
                if hasattr(field_value, "__dict__"):
                    # Check if it has an 'id' field with filter operators
                    id_value = getattr(field_value, "id", None)
                    if hasattr(field_value, "id") and isinstance(id_value, dict):
                        # This is a nested object filter, convert to dict format
                        result[field_name] = {"id": id_value}
                    else:
                        # Try to convert recursively
                        nested_dict = {
                            nested_field: nested_value
                            for nested_field, nested_value in field_value.__dict__.items()
                            if nested_value is not None and not nested_field.startswith("_")
                        }
                        if nested_dict:
                            result[field_name] = nested_dict
                elif isinstance(field_value, dict):
                    # Direct dict value, use as-is
                    result[field_name] = field_value
                elif isinstance(field_value, (str, int, float, bool)):
                    # Scalar value, wrap in eq operator
                    result[field_name] = {"eq": field_value}

        return result if result else None

    def _convert_field_name_to_database(self, field_name: str) -> str:
        """Convert GraphQL field name to database field name.

        Automatically converts camelCase to snake_case while preserving
        existing snake_case names for backward compatibility.

        Args:
            field_name: GraphQL field name (camelCase or snake_case)

        Returns:
            Database field name in snake_case

        Examples:
            'ipAddress' -> 'ip_address'
            'status' -> 'status' (unchanged)
        """
        if not field_name or not isinstance(field_name, str):
            return field_name or ""

        # Preserve existing snake_case for backward compatibility
        if "_" in field_name:
            return field_name

        # Convert camelCase to snake_case
        return to_snake_case(field_name)
