"""
Tests for the CosmosPromptManager tool.

This module tests the enhanced CosmosPromptManager functionality including
initialization, caching, CRUD operations, batch operations, consistency levels,
health checks, async operations, and error handling with retry logic.
"""

import pytest
import asyncio
from unittest.mock import Mock, patch, MagicMock, AsyncMock
from typing import Dict, Any
import time
from datetime import datetime

from azpaddypy.tools.cosmos_prompt_manager import (
    CosmosPromptManager, 
    create_cosmos_prompt_manager,
    retry_with_exponential_backoff
)
from azpaddypy.resources.cosmosdb import AzureCosmosDB
from azpaddypy.mgmt.logging import AzureLogger


class TestCosmosPromptManager:
    """Test the enhanced CosmosPromptManager class."""

    def setup_method(self):
        """Set up test fixtures."""
        self.mock_cosmos_client = Mock(spec=AzureCosmosDB)
        self.mock_logger = Mock(spec=AzureLogger)
        
        # Configure mock logger
        self.mock_logger.create_span.return_value.__enter__ = Mock()
        self.mock_logger.create_span.return_value.__exit__ = Mock()
        
        self.prompt_manager = CosmosPromptManager(
            cosmos_client=self.mock_cosmos_client,
            database_name="test_db",
            container_name="test_container",
            service_name="test_prompt_manager",
            service_version="1.0.0",
            logger=self.mock_logger,
            max_retries=2,
            base_retry_delay=0.1
        )

    def test_initialization(self):
        """Test CosmosPromptManager initialization with enhanced features."""
        assert self.prompt_manager.cosmos_client == self.mock_cosmos_client
        assert self.prompt_manager.database_name == "test_db"
        assert self.prompt_manager.container_name == "test_container"
        assert self.prompt_manager.service_name == "test_prompt_manager"
        assert self.prompt_manager.service_version == "1.0.0"
        assert self.prompt_manager.max_retries == 2
        assert self.prompt_manager.base_retry_delay == 0.1

    def test_get_cache_staleness_ms(self):
        """Test cache staleness configuration for different consistency levels."""
        # Test eventual consistency
        assert self.prompt_manager._get_cache_staleness_ms("eventual") == 30000
        
        # Test bounded consistency
        assert self.prompt_manager._get_cache_staleness_ms("bounded") == 5000
        
        # Test strong consistency
        assert self.prompt_manager._get_cache_staleness_ms("strong") == 0
        
        # Test default fallback
        assert self.prompt_manager._get_cache_staleness_ms("invalid") == 5000

    def test_get_prompt_with_consistency_levels(self):
        """Test getting prompt with different consistency levels."""
        # Mock Cosmos DB response
        mock_doc = {
            "id": "test_prompt",
            "name": "test_prompt",
            "prompt_template": "test template"
        }
        
        self.mock_cosmos_client.read_item.return_value = mock_doc
        
        # Test with bounded consistency (default)
        result = self.prompt_manager.get_prompt("test_prompt")
        assert result == "test template"
        
        # Test with eventual consistency
        result = self.prompt_manager.get_prompt("test_prompt", consistency_level="eventual")
        assert result == "test template"
        
        # Test with strong consistency
        result = self.prompt_manager.get_prompt("test_prompt", consistency_level="strong")
        assert result == "test template"
        
        # Verify the read_item was called with appropriate staleness
        assert self.mock_cosmos_client.read_item.call_count == 3

    def test_get_prompt_with_custom_staleness(self):
        """Test getting prompt with custom cache staleness override."""
        mock_doc = {
            "id": "test_prompt",
            "name": "test_prompt",
            "prompt_template": "test template"
        }
        
        self.mock_cosmos_client.read_item.return_value = mock_doc
        
        # Test with custom staleness override
        result = self.prompt_manager.get_prompt(
            "test_prompt", 
            max_integrated_cache_staleness_in_ms=10000
        )
        assert result == "test template"
        
        # Verify the custom staleness was used
        self.mock_cosmos_client.read_item.assert_called_with(
            database_name="test_db",
            container_name="test_container",
            item_id="test_prompt",
            partition_key="test_prompt",
            max_integrated_cache_staleness_in_ms=10000
        )

    def test_get_prompts_batch(self):
        """Test batch retrieval of multiple prompts stored as individual JSON documents."""
        # Mock query response - each prompt is a separate JSON document in Cosmos DB
        mock_docs = [
            {"id": "prompt1", "prompt_template": "template1"},
            {"id": "prompt2", "prompt_template": "template2"}
        ]
        self.mock_cosmos_client.query_items.return_value = mock_docs
        
        # Test batch retrieval using SQL query with IN clause (efficient for individual documents)
        prompt_names = ["prompt1", "prompt2", "prompt3"]  # prompt3 doesn't exist
        result = self.prompt_manager.get_prompts_batch(prompt_names)
        
        # Verify results
        assert len(result) == 3
        assert result["prompt1"] == "template1"
        assert result["prompt2"] == "template2"
        assert result["prompt3"] is None  # Not found
        
        # Verify query was called
        self.mock_cosmos_client.query_items.assert_called_once()

    def test_get_prompts_batch_empty(self):
        """Test batch retrieval with empty list."""
        result = self.prompt_manager.get_prompts_batch([])
        assert result == {}
        
        # Verify no query was made
        self.mock_cosmos_client.query_items.assert_not_called()

    def test_get_prompts_batch_error(self):
        """Test batch retrieval with error."""
        self.mock_cosmos_client.query_items.side_effect = Exception("Query failed")
        
        prompt_names = ["prompt1", "prompt2"]
        result = self.prompt_manager.get_prompts_batch(prompt_names)
        
        # Should return all None values on error
        assert result == {"prompt1": None, "prompt2": None}

    def test_save_prompts_batch(self):
        """Test batch saving of multiple prompts as individual JSON documents."""
        prompts = [
            {"name": "prompt1", "data": "template1"},
            {"name": "prompt2", "data": {"prompt_template": "template2", "category": "test"}}
        ]
        
        # Mock successful upserts - each prompt saved as separate JSON document
        self.mock_cosmos_client.upsert_item.return_value = {"id": "test"}
        
        result = self.prompt_manager.save_prompts_batch(prompts)
        
        # Verify all prompts were saved successfully (individual upsert operations)
        assert result == {"prompt1": True, "prompt2": True}
        assert self.mock_cosmos_client.upsert_item.call_count == 2

    def test_save_prompts_batch_partial_failure(self):
        """Test batch saving with partial failures."""
        prompts = [
            {"name": "prompt1", "data": "template1"},
            {"name": "prompt2", "data": "template2"}
        ]
        
        # Mock partial failure
        def mock_upsert(database_name, container_name, item):
            if item["id"] == "prompt1":
                return {"id": "prompt1"}
            else:
                raise Exception("Save failed")
        
        self.mock_cosmos_client.upsert_item.side_effect = mock_upsert
        
        result = self.prompt_manager.save_prompts_batch(prompts)
        
        # Verify partial success
        assert result == {"prompt1": True, "prompt2": False}

    def test_save_prompts_batch_missing_name(self):
        """Test batch saving with missing name field."""
        prompts = [
            {"name": "prompt1", "data": "template1"},
            {"data": "template2"}  # Missing name
        ]
        
        # Mock successful upsert for valid prompt
        self.mock_cosmos_client.upsert_item.return_value = {"id": "test"}
        
        result = self.prompt_manager.save_prompts_batch(prompts)
        
        # Verify only valid prompt was saved
        assert result["prompt1"] is True
        assert "unnamed_prompt_1" in result
        assert result["unnamed_prompt_1"] is False
        assert self.mock_cosmos_client.upsert_item.call_count == 1

    def test_save_prompts_batch_empty(self):
        """Test batch saving with empty list."""
        result = self.prompt_manager.save_prompts_batch([])
        assert result == {}
        
        # Verify no upsert was made
        self.mock_cosmos_client.upsert_item.assert_not_called()

    def test_health_check_healthy(self):
        """Test health check when all systems are healthy."""
        # Mock successful operations
        self.mock_cosmos_client.get_database.return_value = Mock()
        self.mock_cosmos_client.get_container.return_value = Mock()
        
        with patch.object(self.prompt_manager, 'list_prompts', return_value=["prompt1", "prompt2"]):
            result = self.prompt_manager.health_check()
        
        assert result["status"] == "healthy"
        assert "timestamp" in result
        assert result["service"]["name"] == "test_prompt_manager"
        assert result["service"]["version"] == "1.0.0"
        assert "database_connection" in result["checks"]
        assert "container_access" in result["checks"]
        assert "basic_operations" in result["checks"]
        assert result["checks"]["basic_operations"]["prompt_count"] == 2

    def test_health_check_unhealthy(self):
        """Test health check when systems are unhealthy."""
        # Mock failure
        self.mock_cosmos_client.get_database.side_effect = Exception("Connection failed")
        
        result = self.prompt_manager.health_check()
        
        assert result["status"] == "unhealthy"
        assert "error" in result
        assert result["checks"]["error"]["status"] == "unhealthy"

    def test_retry_decorator(self):
        """Test retry decorator functionality."""
        @retry_with_exponential_backoff(max_retries=2, base_delay=0.01)
        def failing_function():
            failing_function.call_count += 1
            if failing_function.call_count < 3:
                raise Exception("Temporary failure")
            return "success"
        
        failing_function.call_count = 0
        
        result = failing_function()
        assert result == "success"
        assert failing_function.call_count == 3

    def test_retry_decorator_max_retries_exceeded(self):
        """Test retry decorator when max retries are exceeded."""
        @retry_with_exponential_backoff(max_retries=2, base_delay=0.01)
        def always_failing_function():
            raise Exception("Always fails")
        
        with pytest.raises(Exception, match="Always fails"):
            always_failing_function()

    # Update existing tests to account for retry behavior
    def test_get_prompt_from_cosmos_db(self):
        """Test getting prompt from Cosmos DB with retry logic."""
        # Mock Cosmos DB response
        mock_doc = {
            "id": "test_prompt",
            "name": "test_prompt",
            "prompt_template": "cosmos template"
        }
        
        self.mock_cosmos_client.read_item.return_value = mock_doc
        
        result = self.prompt_manager.get_prompt("test_prompt")
        
        assert result == "cosmos template"
        self.mock_cosmos_client.read_item.assert_called_once()

    def test_get_prompt_not_found(self):
        """Test getting prompt that doesn't exist."""
        self.mock_cosmos_client.read_item.return_value = None
        
        result = self.prompt_manager.get_prompt("nonexistent_prompt")
        
        assert result is None
        self.mock_logger.warning.assert_called_with("Prompt not found in Cosmos DB: nonexistent_prompt")

    def test_save_prompt_string_data(self):
        """Test saving prompt with string data."""
        # Mock successful upsert
        self.mock_cosmos_client.upsert_item.return_value = {"id": "test_prompt"}
        
        result = self.prompt_manager.save_prompt("test_prompt", "new template")
        
        assert result is True
        self.mock_cosmos_client.upsert_item.assert_called_once()

    def test_save_prompt_dict_data(self):
        """Test saving prompt with dictionary data."""
        # Mock successful upsert
        self.mock_cosmos_client.upsert_item.return_value = {"id": "test_prompt"}
        
        prompt_data = {"prompt_template": "new template", "category": "test"}
        result = self.prompt_manager.save_prompt("test_prompt", prompt_data)
        
        assert result is True
        self.mock_cosmos_client.upsert_item.assert_called_once()

    def test_save_prompt_with_retry(self):
        """Test saving prompt with retry logic."""
        # Mock successful upsert (retry logic is tested separately)
        self.mock_cosmos_client.upsert_item.return_value = {"id": "test_prompt"}
        
        result = self.prompt_manager.save_prompt("test_prompt", "template")
        
        assert result is True
        self.mock_cosmos_client.upsert_item.assert_called_once()

    def test_list_prompts_optimized(self):
        """Test listing prompts with optimized query."""
        # Mock query response
        mock_docs = [
            {"name": "prompt1"},
            {"name": "prompt2"}
        ]
        self.mock_cosmos_client.query_items.return_value = mock_docs
        
        result = self.prompt_manager.list_prompts()
        
        assert result == ["prompt1", "prompt2"]
        
        # Verify optimized query was used
        args, kwargs = self.mock_cosmos_client.query_items.call_args
        assert "ORDER BY c.name" in kwargs["query"]
        assert "max_integrated_cache_staleness_in_ms" in kwargs

    def test_delete_prompt_with_retry(self):
        """Test deleting prompt with retry logic."""
        # Mock successful delete (retry logic is tested separately)
        self.mock_cosmos_client.delete_item.return_value = True
        
        result = self.prompt_manager.delete_prompt("test_prompt")
        
        assert result is True
        self.mock_cosmos_client.delete_item.assert_called_once()

    def test_delete_prompt_not_found(self):
        """Test deleting prompt that doesn't exist."""
        self.mock_cosmos_client.delete_item.return_value = False
        
        result = self.prompt_manager.delete_prompt("nonexistent_prompt")
        
        assert result is False
        self.mock_logger.warning.assert_called_with("Prompt not found for deletion: nonexistent_prompt")

    def test_get_prompt_details(self):
        """Test getting prompt details."""
        mock_doc = {
            "id": "test_prompt",
            "name": "test_prompt",
            "prompt_template": "cosmos template",
            "timestamp": "2023-01-01T00:00:00.000000Z"
        }
        self.mock_cosmos_client.read_item.return_value = mock_doc
        
        result = self.prompt_manager.get_prompt_details("test_prompt")
        
        assert result['id'] == mock_doc['id']
        assert result['name'] == mock_doc['name']
        assert result['prompt_template'] == mock_doc['prompt_template']
        assert 'timestamp' in result

    def test_get_prompt_details_not_found(self):
        """Test getting details for a non-existent prompt."""
        self.mock_cosmos_client.read_item.return_value = None
        
        result = self.prompt_manager.get_prompt_details("nonexistent_prompt")
        
        assert result is None
        self.mock_logger.warning.assert_called_with(
            "Prompt not found in Cosmos DB: nonexistent_prompt"
        )

    def test_get_prompt_details_not_found_exception(self):
        """Test getting details for a non-existent prompt that raises an exception."""
        from azure.core.exceptions import ResourceNotFoundError
        self.mock_cosmos_client.read_item.side_effect = ResourceNotFoundError("Not found")
        
        result = self.prompt_manager.get_prompt_details("nonexistent_prompt")
        
        assert result is None
        self.mock_logger.warning.assert_called_with(
            "Prompt not found in Cosmos DB: nonexistent_prompt"
        )

    def test_get_all_prompt_details_optimized(self):
        """Test getting all prompt details with optimized query."""
        mock_docs = [
            {
                "id": "prompt1",
                "name": "prompt1",
                "prompt_template": "template1",
                "timestamp": "2023-01-01T00:00:00.000000Z"
            },
            {
                "id": "prompt2",
                "name": "prompt2",
                "prompt_template": "template2",
                "timestamp": "2023-01-01T00:00:00.000000Z"
            }
        ]
        self.mock_cosmos_client.query_items.return_value = mock_docs
        
        result = self.prompt_manager.get_all_prompt_details()
        
        assert len(result) == 2
        assert result[0]["name"] == "prompt1"
        
        # Verify optimized query was used
        args, kwargs = self.mock_cosmos_client.query_items.call_args
        assert "SELECT c.id, c.name, c.prompt_template, c.timestamp FROM c" in kwargs["query"]
        assert "ORDER BY c.name" in kwargs["query"]
        assert "max_integrated_cache_staleness_in_ms" in kwargs

    @pytest.mark.asyncio
    async def test_get_prompt_async(self):
        """Test async prompt retrieval."""
        # Test the async context manager - actual implementation would require
        # a fully mocked async cosmos client which is complex to set up
        async with self.prompt_manager.async_context() as manager:
            assert manager is self.prompt_manager

    @pytest.mark.asyncio
    async def test_get_prompt_async_not_found(self):
        """Test async prompt retrieval when not found."""
        # Mock async client and context
        mock_async_client = AsyncMock()
        mock_container = AsyncMock()
        mock_database = AsyncMock()
        
        mock_async_client.get_database_client.return_value = mock_database
        mock_database.get_container_client.return_value = mock_container
        mock_container.read_item.return_value = None
        
        # Mock async context manager properly
        async def mock_async_context(self):
            return mock_async_client
        
        self.mock_cosmos_client.async_client_context.return_value.__aenter__ = mock_async_context
        self.mock_cosmos_client.async_client_context.return_value.__aexit__ = AsyncMock()
        
        result = await self.prompt_manager.get_prompt_async("nonexistent_prompt")
        
        assert result is None

    @pytest.mark.asyncio
    async def test_async_context(self):
        """Test async context manager."""
        async with self.prompt_manager.async_context() as manager:
            assert manager is self.prompt_manager


class TestCreateCosmosPromptManager:
    """Test the enhanced factory function for CosmosPromptManager."""

    def test_create_cosmos_prompt_manager_with_enhanced_features(self):
        """Test factory function creation with enhanced features."""
        mock_cosmos_client = Mock(spec=AzureCosmosDB)
        mock_logger = Mock(spec=AzureLogger)
        
        prompt_manager = create_cosmos_prompt_manager(
            cosmos_client=mock_cosmos_client,
            database_name="test_db",
            container_name="test_container",
            service_name="test_service",
            service_version="2.0.0",
            logger=mock_logger,
            max_retries=5,
            base_retry_delay=2.0
        )
        
        assert isinstance(prompt_manager, CosmosPromptManager)
        assert prompt_manager.cosmos_client == mock_cosmos_client
        assert prompt_manager.database_name == "test_db"
        assert prompt_manager.service_name == "test_service"
        assert prompt_manager.service_version == "2.0.0"
        assert prompt_manager.max_retries == 5
        assert prompt_manager.base_retry_delay == 2.0

    def test_create_cosmos_prompt_manager_with_defaults(self):
        """Test factory function with default values."""
        mock_cosmos_client = Mock(spec=AzureCosmosDB)

        prompt_manager = create_cosmos_prompt_manager(
            cosmos_client=mock_cosmos_client
        )

        assert prompt_manager.database_name == "prompts"
        assert prompt_manager.container_name == "prompts"
        assert prompt_manager.service_name == "azure_cosmos_prompt_manager"
        assert prompt_manager.service_version == "1.0.0"
        assert prompt_manager.max_retries == 3
        assert prompt_manager.base_retry_delay == 1.0

