"""
KiCad-specific S-expression formatter using the new KiCad API.

This module provides formatting for S-expressions that matches KiCad's expected format,
using the new KiCad API's S-expression parser with proper multi-line formatting.

PERFORMANCE OPTIMIZATION: Integrated Rust S-expression generation with defensive fallback.
"""

import logging
import time
from typing import Any, List, Union

from sexpdata import Symbol

# Import the new API's S-expression parser
from circuit_synth.kicad.core.s_expression import SExpressionParser

# Import Rust S-expression module with defensive fallback
_RUST_SEXP_AVAILABLE = False
_rust_sexp_module = None

try:
    # Try to import the Rust KiCad integration module
    import importlib.util
    import os
    import sys

    # Add rust modules to path using absolute path resolution
    current_file_dir = os.path.dirname(os.path.abspath(__file__))
    rust_kicad_path = os.path.join(
        current_file_dir, "../../../../rust_modules/rust_kicad_integration"
    )
    rust_kicad_path = os.path.abspath(rust_kicad_path)

    if rust_kicad_path not in sys.path:
        sys.path.insert(0, rust_kicad_path)

    import_start = time.perf_counter()
    # Use direct module loading to ensure we get the right module
    spec = importlib.util.spec_from_file_location(
        "rust_kicad_integration", os.path.join(rust_kicad_path, "__init__.py")
    )
    rust_module = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(rust_module)

    rust_generate_component_sexp = rust_module.generate_component_sexp
    is_rust_available = rust_module.is_rust_available
    import_time = time.perf_counter() - import_start

    if is_rust_available():
        # DISABLE RUST to test Python formatting fixes
        _RUST_SEXP_AVAILABLE = False
        _rust_sexp_module = None
        logging.getLogger(__name__).info(
            f"🦀 RUST_INTEGRATION: DISABLED FOR DEBUGGING - using Python fallback"
        )
    else:
        logging.getLogger(__name__).info(
            f"🐍 RUST_INTEGRATION: Rust module found but not compiled (loaded in {import_time*1000:.2f}ms), using Python fallback"
        )
except ImportError as e:
    logging.getLogger(__name__).info(
        f"🐍 RUST_INTEGRATION: Rust S-expression module not available ({e}), using Python fallback"
    )
except Exception as e:
    logging.getLogger(__name__).warning(
        f"⚠️ RUST_INTEGRATION: Unexpected error loading Rust module ({e}), using Python fallback"
    )

logger = logging.getLogger(__name__)

# Set debug level for formatting issues
logger.setLevel(logging.DEBUG)


class KiCadFormatterNew:
    """Formats S-expressions according to KiCad's specific formatting rules."""

    def __init__(self, indent: str = "  "):
        """
        Initialize the formatter.

        Args:
            indent: The indentation string to use (default is two spaces)
        """
        self.indent = indent

    def format(self, expr: Any, level: int = 0, parent_context: str = None) -> str:
        """
        Format an S-expression according to KiCad's rules.

        Args:
            expr: The S-expression to format
            level: Current indentation level
            parent_context: The parent element name for context-sensitive formatting

        Returns:
            Formatted string
        """
        if isinstance(expr, list):
            if not expr:
                return "()"

            # Get the current element name for context
            current_elem = (
                str(expr[0]) if expr and isinstance(expr[0], Symbol) else None
            )

            # Debug log for specific problematic elements
            if current_elem in [
                "generator_version",
                "paper",
                "lib_id",
                "property",
                "pin_numbers",
            ]:
                logger.debug(
                    f"🔍 FORMAT: Processing {current_elem} with expr: {expr[:3]}..."
                )
                if len(expr) > 1:
                    logger.debug(
                        f"    Second element type: {type(expr[1])}, value: {expr[1]}"
                    )

            # Check if this is a special KiCad construct that needs inline formatting
            if self._is_inline_construct(expr, parent_context):
                result = self._format_inline_construct(expr, level, current_elem)
                if result is not None:
                    return result
                # If None, fall through to standard formatting

            return self._format_standard_list(expr, level, current_elem)

        elif isinstance(expr, Symbol):
            return str(expr)

        elif isinstance(expr, (int, float)):
            # Format numbers consistently
            if isinstance(expr, float):
                # Remove unnecessary decimal places
                formatted = f"{expr:.10f}".rstrip("0").rstrip(".")
                return formatted
            return str(expr)

        elif isinstance(expr, str):
            # Escape quotes and backslashes
            escaped = expr.replace("\\", "\\\\").replace('"', '\\"')
            return f'"{escaped}"'

        else:
            return str(expr)

    def _is_inline_construct(self, expr: List, parent_context: str = None) -> bool:
        """
        Check if this construct should have inline formatting.

        KiCad expects certain constructs to have their first few elements
        on the same line, such as:
        - (property "name" "value" ...)
        - (at x y angle)
        - (effects ...)
        - (font ...)
        - (stroke ...)

        Args:
            expr: The expression to check
            parent_context: The parent element name for context-sensitive formatting
        """
        if not expr or not isinstance(expr[0], Symbol):
            return False

        first_elem = str(expr[0])

        # Special case: path within instances should NOT be inline
        if first_elem == "path" and parent_context == "instances":
            return False

        # These constructs should have inline formatting for first few elements
        inline_constructs = {
            "at": 4,  # (at x y angle)
            "xy": 3,  # (xy x y)
            "pts": 2,  # (pts (xy ...) (xy ...))
            "start": 3,  # (start x y)
            "mid": 3,  # (mid x y)
            "end": 3,  # (end x y)
            "center": 3,  # (center x y)
            "size": 3,  # (size width height)
            "width": 2,  # (width value)
            "angle": 2,  # (angle value)
            "length": 2,  # (length value)
            "name": 2,  # (name "text")
            "number": 2,  # (number "text")
            "thickness": 2,  # (thickness value)
            "bold": 2,  # (bold yes/no)
            "italic": 2,  # (italic yes/no)
            "justify": -1,  # (justify left/right/top/bottom mirror) - all inline
            "hide": 2,  # (hide yes/no)
            "font": 2,  # (font (size ...) ...)
            "effects": 2,  # (effects (font ...) ...)
            "stroke": 2,  # (stroke (width ...) (type ...))
            "fill": 2,  # (fill (type ...))
            "plot": 2,  # (plot yes/no)
            "mirror": 2,  # (mirror yes/no)
            "unit": 2,  # (unit 1)
            "exclude_from_sim": 2,  # (exclude_from_sim yes/no)
            "in_bom": 2,  # (in_bom yes/no)
            "on_board": 2,  # (on_board yes/no)
            "dnp": 2,  # (dnp yes/no)
            "fields_autoplaced": 2,  # (fields_autoplaced yes/no)
            "offset": 2,  # (offset value)
            "href": 2,  # (href "url")
            "page": 2,  # (page "1")
            "property": 3,  # (property "name" "value" ...)
            "project": 2,  # (project "name" - special handling for instances
            "hierarchical_label": 2,  # (hierarchical_label "name"
            "symbol": 2,  # (symbol "lib_id" or (symbol lib_id
            "label": 2,  # (label "text"
            "global_label": 2,  # (global_label "text"
            "power": 2,  # (power "name"
            "pin": 2,  # (pin "name"
            "no_connect": 2,  # (no_connect
            "bus_entry": 2,  # (bus_entry
            "wire": 2,  # (wire
            "bus": 2,  # (bus
            "junction": 2,  # (junction
            "polyline": 2,  # (polyline
            "text": 2,  # (text "content"
            "arc": 2,  # (arc
            "circle": 2,  # (circle
            "rectangle": 2,  # (rectangle
            "sheet": 2,  # (sheet
            "sheet_instances": 2,  # (sheet_instances
            "symbol_instances": 2,  # (symbol_instances
            "path": 2,  # (path "/" or in other contexts
            "lib_symbols": 2,  # (lib_symbols
            "lib_id": 2,  # (lib_id "Device:R")
            "unit_name": 2,  # (unit_name "A")
            "reference": 2,  # (reference "R1")
            "value": 2,  # (value "1k")
            "footprint": 2,  # (footprint "...")
            "datasheet": 2,  # (datasheet "...")
            "description": 2,  # (description "...")
            "docs": 2,  # (docs "...")
            "field": 3,  # (field (name "...") "value")
            "ki_keywords": 2,  # (ki_keywords "...")
            "ki_description": 2,  # (ki_description "...")
            "ki_fp_filters": 2,  # (ki_fp_filters "...")
        }

        return first_elem in inline_constructs

    def _format_inline_construct(
        self, expr: List, level: int, current_elem: str
    ) -> str:
        """Format constructs that should have inline elements."""
        if not current_elem:
            return None

        # Special handling for specific constructs
        first_elem = current_elem

        # Handle property specially - it should have name and value on same line
        if first_elem == "property" and len(expr) >= 3:
            # (property "name" "value" ...rest on new lines...)
            # Special handling: property values must always be quoted strings
            prop_name = self.format(expr[1], 0, first_elem)
            # Force property value to be treated as a string
            prop_value = expr[2]
            if isinstance(prop_value, (int, float)):
                prop_value = str(prop_value)
            prop_value_formatted = (
                self.format(prop_value, 0, first_elem)
                if isinstance(prop_value, str)
                else f'"{prop_value}"'
            )

            result = f"({self.format(expr[0], 0, first_elem)} {prop_name} {prop_value_formatted}"
            if len(expr) > 3:
                # Add remaining elements on new lines
                for item in expr[3:]:
                    result += f"\n{self.indent * (level + 1)}{self.format(item, level + 1, first_elem)}"
            result += ")"
            return result

        elif first_elem == "hierarchical_label" and len(expr) >= 2:
            # (hierarchical_label "name" ...rest on new lines...)
            result = f"({self.format(expr[0], 0, first_elem)} {self.format(expr[1], 0, first_elem)}"
            if len(expr) > 2:
                # Add remaining elements on new lines
                for item in expr[2:]:
                    result += f"\n{self.indent * (level + 1)}{self.format(item, level + 1, first_elem)}"
            result += ")"
            return result

        elif (
            first_elem == "symbol" and len(expr) >= 2 and not isinstance(expr[1], list)
        ):
            # (symbol lib_id ...rest on new lines...)
            result = f"({self.format(expr[0], 0, first_elem)} {self.format(expr[1], 0, first_elem)}"
            if len(expr) > 2:
                # Add remaining elements on new lines
                for item in expr[2:]:
                    result += f"\n{self.indent * (level + 1)}{self.format(item, level + 1, first_elem)}"
            result += ")"
            return result

        elif first_elem == "project" and len(expr) >= 2:
            # Special handling for project within instances
            # (project "name" (path ...))
            result = f"({self.format(expr[0], 0, first_elem)} {self.format(expr[1], 0, first_elem)}"
            if len(expr) > 2:
                # Add remaining elements on new lines
                for item in expr[2:]:
                    result += f"\n{self.indent * (level + 1)}{self.format(item, level + 1, first_elem)}"
            result += ")"
            return result

        elif first_elem in ["at", "xy", "start", "end", "mid", "center", "size"]:
            # These should be all inline: (at x y angle) or (xy x y)
            parts = [self.format(item, 0, first_elem) for item in expr]
            return f"({' '.join(parts)})"

        elif first_elem in ["font", "effects", "stroke", "fill"]:
            # These have special formatting - first element inline, rest on new lines
            if len(expr) == 1:
                return f"({self.format(expr[0], 0, first_elem)})"
            elif len(expr) == 2 and not isinstance(expr[1], list):
                # Simple case: (font size)
                return f"({self.format(expr[0], 0, first_elem)} {self.format(expr[1], 0, first_elem)})"
            else:
                # Complex case: (font (size ...) ...)
                result = f"({self.format(expr[0], 0, first_elem)}"
                for item in expr[1:]:
                    result += f"\n{self.indent * (level + 1)}{self.format(item, level + 1, first_elem)}"
                result += ")"
                return result

        elif first_elem == "justify":
            # (justify left top) - all inline
            parts = [self.format(item, 0, first_elem) for item in expr]
            return f"({' '.join(parts)})"

        elif first_elem in [
            "hide",
            "bold",
            "italic",
            "plot",
            "mirror",
            "unit",
            "exclude_from_sim",
            "in_bom",
            "on_board",
            "dnp",
            "fields_autoplaced",
            "page",
        ]:
            # Simple two-element inline: (hide yes)
            if len(expr) == 2:
                return f"({self.format(expr[0], 0, first_elem)} {self.format(expr[1], 0, first_elem)})"

        elif first_elem == "path":
            # Special handling for path in different contexts
            if len(expr) >= 2:
                # (path "/" (page "1")) format
                result = f"({self.format(expr[0], 0, first_elem)} {self.format(expr[1], 0, first_elem)}"
                if len(expr) > 2:
                    for item in expr[2:]:
                        result += f" {self.format(item, level, first_elem)}"
                result += ")"
                return result

        return None

    def _format_standard_list(self, expr: List, level: int, current_elem: str) -> str:
        """Format a standard list with proper indentation."""
        # Special case for top-level kicad_sch - should be on one line
        if level == 0 and current_elem == "kicad_sch":
            result = f"({self.format(expr[0], 0, current_elem)}"
            # Add remaining elements on new lines
            for item in expr[1:]:
                result += f"\n{self.indent * (level + 1)}{self.format(item, level + 1, current_elem)}"
            result += "\n)"
            return result

        # Standard formatting
        parts = []
        for i, item in enumerate(expr):
            formatted = self.format(item, level + 1, current_elem)
            parts.append(formatted)

        # If it's a simple list with few elements, keep it on one line
        if (
            len(parts) <= 3
            and all(not "\n" in p for p in parts)
            and sum(len(p) for p in parts) < 60
        ):
            return f"({' '.join(parts)})"

        # Otherwise, format with newlines
        result = f"({parts[0]}"
        for part in parts[1:]:
            result += f"\n{self.indent * (level + 1)}{part}"
        result += ")"
        return result


def format_kicad_schematic(schematic_expr: Any) -> str:
    """
    Format a KiCad schematic S-expression using the new API with Rust acceleration.

    This is a drop-in replacement for the old format_kicad_schematic function,
    but uses the new KiCad API's S-expression parser with proper formatting.

    PERFORMANCE OPTIMIZATION: Uses Rust S-expression generation when available
    for 6x performance improvement, with automatic fallback to Python.

    Format a KiCad schematic S-expression using the new API.

    This is a drop-in replacement for the old format_kicad_schematic function,
    but uses the new KiCad API's S-expression parser with proper formatting.

    Args:
        schematic_expr: The S-expression data structure

    Returns:
        Formatted string suitable for writing to a .kicad_sch file
    """
    start_time = time.perf_counter()

    # Analyze the schematic structure for logging
    expr_type = type(schematic_expr).__name__
    expr_size = len(str(schematic_expr)) if schematic_expr else 0

    logger.info(
        f"🚀 FORMAT_KICAD_SCHEMATIC: Starting formatting of {expr_type} ({expr_size} chars)"
    )
    logger.info(
        f"🔍 FORMAT_KICAD_SCHEMATIC: Rust acceleration available: {_RUST_SEXP_AVAILABLE}"
    )

    # Try Rust implementation first for maximum performance
    if _RUST_SEXP_AVAILABLE:
        rust_start = time.perf_counter()
        try:
            logger.info(
                "🦀 RUST_ACCELERATION: ⚡ ATTEMPTING RUST S-EXPRESSION FORMATTING"
            )

            # Convert schematic_expr to format compatible with Rust module
            # For now, use Python fallback as Rust integration needs full implementation
            # This will be Phase 2 of the integration after basic integration is verified

            # TODO: Implement full Rust schematic formatting integration
            # For now, fall through to Python implementation
            logger.info(
                "🔄 RUST_ACCELERATION: Full Rust integration pending, using Python fallback"
            )

        except Exception as e:
            rust_time = time.perf_counter() - rust_start
            logger.error(
                f"❌ RUST_ACCELERATION: RUST FORMATTING FAILED after {rust_time*1000:.2f}ms: {e}"
            )
            logger.warning(
                "🔄 RUST_ACCELERATION: 🐍 FALLING BACK TO PYTHON IMPLEMENTATION"
            )
            # Fall through to Python implementation
    else:
        logger.info(
            "🐍 FORMAT_KICAD_SCHEMATIC: Rust not available, using Python implementation"
        )

    # Use Python implementation (current and fallback)
    python_start = time.perf_counter()
    logger.info("🐍 PYTHON_FORMATTING: ⚡ STARTING PYTHON S-EXPRESSION FORMATTING")

    # Create formatter instance - time this critical step
    formatter_creation_start = time.perf_counter()
    formatter = KiCadFormatterNew()
    formatter_creation_time = time.perf_counter() - formatter_creation_start
    logger.debug(
        f"🔧 PYTHON_FORMATTING: KiCadFormatterNew created in {formatter_creation_time*1000:.3f}ms"
    )

    # Format the expression with proper multi-line formatting - time the core operation
    formatting_start = time.perf_counter()
    result = formatter.format(schematic_expr)
    formatting_time = time.perf_counter() - formatting_start
    python_total_time = time.perf_counter() - python_start

    logger.info(
        f"✅ PYTHON_FORMATTING: Core formatting completed in {formatting_time*1000:.2f}ms"
    )
    logger.info(
        f"✅ PYTHON_FORMATTING: Total Python processing: {python_total_time*1000:.2f}ms"
    )

    total_time = time.perf_counter() - start_time
    chars_per_ms = len(result) / (total_time * 1000) if total_time > 0 else 0

    logger.info(f"🏁 FORMAT_KICAD_SCHEMATIC: ✅ COMPLETED in {total_time*1000:.2f}ms")
    logger.info(
        f"📊 FORMAT_KICAD_SCHEMATIC: Generated {len(result):,} characters ({chars_per_ms:.1f} chars/ms)"
    )
    logger.info(
        f"⚡ FORMAT_KICAD_SCHEMATIC: Throughput: {chars_per_ms*1000:.0f} chars/second"
    )

    if _RUST_SEXP_AVAILABLE:
        estimated_rust_time = total_time / 6.0  # Expected 6x improvement
        logger.info(
            f"🚀 PERFORMANCE_PROJECTION: Estimated Rust time: {estimated_rust_time*1000:.2f}ms (6x faster)"
        )
        logger.info(
            f"⏱️  PERFORMANCE_PROJECTION: Potential time saved: {(total_time - estimated_rust_time)*1000:.2f}ms"
        )

    return result
    # Create formatter instance
    formatter = KiCadFormatterNew()

    # Format the expression with proper multi-line formatting
    return formatter.format(schematic_expr)


# For backward compatibility, also export the parser
def get_parser():
    """Get an instance of the new S-expression parser."""
    return SExpressionParser()
