"""Scaffolding tools and scenarios for `dipdup new` command.

Ask user some question with Click; render Jinja2 templates with answers.
"""

import dataclasses
import logging
import re
from itertools import chain
from pathlib import Path
from typing import Any

from pydantic import ConfigDict
from pydantic import TypeAdapter
from pydantic.dataclasses import dataclass
from pydantic.fields import FieldInfo
from typing_extensions import TypedDict

from dipdup import __version__
from dipdup import env
from dipdup.cli import big_yellow_echo
from dipdup.cli import echo
from dipdup.config import DipDupConfig
from dipdup.config import ToStr
from dipdup.config._mixin import TerminalOptions
from dipdup.utils import load_template
from dipdup.utils import write
from dipdup.yaml import DipDupYAMLConfig

_logger = logging.getLogger(__name__)

CODEGEN_HEADER = f'# generated by DipDup {__version__.split("+")[0]}'

SINGULAR_FORMS = {
    'datasources': 'datasource',
    'contracts': 'contract',
    'indexes': 'index',
    'hooks': 'hook',
    'jobs': 'job',
    'runtimes': 'runtime',
}

# NOTE: All templates are stored in src/dipdup/projects
TEMPLATES: dict[str, tuple[str, ...]] = {
    'evm': (
        'demo_evm_events',
        'demo_evm_transactions',
        'demo_evm_uniswap',
    ),
    'starknet': ('demo_starknet_events',),
    'substrate': ('demo_substrate_events',),
    'tezos': (
        'demo_tezos_auction',
        'demo_tezos_dao',
        'demo_tezos_dex',
        'demo_tezos_domains',
        'demo_tezos_events',
        'demo_tezos_etherlink',
        'demo_tezos_factories',
        'demo_tezos_head',
        'demo_tezos_nft_marketplace',
        'demo_tezos_raw',
        'demo_tezos_token',
        'demo_tezos_token_balances',
        'demo_tezos_token_transfers',
    ),
    'other': ('demo_blank',),
}

# TODO: demo_jobs
# TODO: demo_backup
# TODO: demo_sql
# TODO: demo_timescale
# TODO: demo_cli


class Answers(TypedDict):
    """Answers for survey/replay in order of appearance"""

    dipdup_version: ToStr
    template: str
    package: str
    version: str
    description: str
    license: str
    name: str
    email: str
    postgres_image: str
    postgres_data_path: str
    hasura_image: str
    line_length: ToStr
    package_manager: str


def get_default_answers(package: str | None = None) -> Answers:
    return Answers(
        dipdup_version=__version__.split('.')[0],
        template='demo_blank',
        package=package or 'dipdup_indexer',
        version='0.0.1',
        description='A blockchain indexer built with DipDup',
        license='MIT',
        name='John Doe',
        email='john_doe@example.com',
        postgres_image='postgres:16',
        postgres_data_path='/var/lib/postgresql/data',
        hasura_image='hasura/graphql-engine:latest',
        line_length='120',
        package_manager='uv',
    )


@dataclass(config=ConfigDict(extra='forbid', defer_build=True), kw_only=True)
class ReplayConfig:
    spec_version: ToStr
    replay: Answers


def get_replay_path(name: str) -> Path:
    return Path(__file__).parent / 'projects' / name / 'replay.yaml'


def namespace_from_terminal() -> str | None:
    _, res = prompt_anyof(
        question='What blockchain are you going to index?',
        options=(
            'EVM',
            'Starknet',
            'Substrate',
            'Tezos',
            '[multiple]',
        ),
        comments=(
            'EVM-compatible',
            'Starknet',
            'Substrate',
            'Tezos',
            '(disable filtering)',
        ),
        default=0,
    )
    if res == '[multiple]':
        return None
    return res.lower()


def template_from_terminal(package: str) -> tuple[str | None, dict[str, Any] | None]:
    _, mode = prompt_anyof(
        question='How would you like to set up your new DipDup project?',
        options=(
            'From template',
            'Interactively',
            'Blank',
        ),
        comments=(
            'Use one of demo projects',
            'Guided setup with prompts',
            'Begin with an empty project',
        ),
        default=0,
    )

    if mode == 'Blank':
        return ('demo_blank', None)

    namespace = namespace_from_terminal()

    if mode == 'Interactively':
        replay_path = get_replay_path('demo_blank')
        opts = TerminalOptions(
            package=package,
            namespace=namespace,
        )
        config = DipDupConfig.from_terminal(opts)
        return ('demo_blank', config._json)

    if mode == 'From template':
        options, comments = [], []
        templates = TEMPLATES[namespace] if namespace else chain(v for v in TEMPLATES.values())
        for name in templates:
            replay_path = get_replay_path(name)  # type: ignore[arg-type]
            _answers = answers_from_replay(replay_path)
            options.append(_answers['template'])
            comments.append(_answers['description'])

        _, template = prompt_anyof(
            'Choose a project template:',
            options=tuple(options),
            comments=tuple(comments),
            default=0,
        )
        return (template, None)

    raise NotImplementedError


def answers_from_terminal(package: str | None = None) -> Answers:
    """Script running on dipdup new command and will create a new project base from interactive survey"""
    import survey  # type: ignore[import-untyped]

    big_yellow_echo(
        'Welcome to DipDup! This command will help you to create a new project.\n'
        'You can abort at any time by pressing Ctrl+C twice. Press Enter to use default value.'
    )

    answers = get_default_answers(package)

    while package is None:
        package = survey.routines.input(
            'Enter project name (the name will be used for folder name and package name): ',
            value=answers['package'],
        )
        if package and re.match(r'^[a-zA-Z_][a-zA-Z0-9_]*$', package):
            break
        package = None

        echo(
            f'"{package}" is not valid Python package name. Please use only letters, numbers and underscores.',
            fg='red',
        )

    answers['package'] = package

    # NOTE: Used in pyproject.toml, README.md and some other places
    answers['description'] = survey.routines.input(
        'Enter project description: ',
        value=answers['description'],
    )

    answers['license'] = survey.routines.input(
        'Enter project license (DipDup itself is MIT-licensed): ',
        value=answers['license'],
    )

    answers['name'] = survey.routines.input(
        "Enter author's name: ",
        value=answers['name'],
    )
    answers['email'] = survey.routines.input(
        "Enter author's email: ",
        value=answers['email'],
    )

    big_yellow_echo('Now choose versions of software you want to use.')

    _, answers['postgres_image'] = prompt_anyof(
        question='Choose PostgreSQL version. Try TimescaleDB when working with time series.',
        options=(
            'postgres:16',
            'timescale/timescaledb:latest-pg16',
            'timescale/timescaledb-ha:pg16',
        ),
        comments=(
            'PostgreSQL',
            'TimescaleDB',
            'TimescaleDB HA',
        ),
        default=0,
    )
    if 'timescaledb-ha' in answers['postgres_image']:
        answers['postgres_data_path'] = '/home/postgres/pgdata/data'
        echo(
            '`timescaledb-ha` Docker image uses `/home/postgres/pgdata/data` as a data path; generated files were updated accordingly.',
            fg='yellow',
        )

    return answers


def answers_from_replay(path: Path) -> Answers:
    yaml_config, _ = DipDupYAMLConfig.load([path])
    yaml_config['replay'] = {
        **get_default_answers(),
        **yaml_config['replay'],
    }
    return TypeAdapter(ReplayConfig).validate_python(yaml_config).replay


def render_project(
    answers: Answers,
    force: bool = False,
) -> None:
    """Render project from template"""
    # NOTE: Common base
    _render_templates(answers, Path('base'), force)

    # NOTE: Config and handlers
    _render_templates(answers, Path(answers['template']), force)

    # NOTE: Replay to use with `init --base` and `new --replay`
    Path(answers['package']).joinpath('configs').mkdir(parents=True, exist_ok=True)
    _render(
        answers,
        template_path=Path(__file__).parent / 'templates' / 'replay.yaml.j2',
        output_path=env.get_package_path(answers['package']) / 'configs' / 'replay.yaml',
        force=force,
    )


def render_base(
    answers: Answers,
    force: bool = False,
    include: set[str] | None = None,
) -> None:
    """Render base from template"""
    # NOTE: Render common base
    _render_templates(
        answers=answers,
        path=Path('base'),
        force=force,
        include=include,
        exists=True,
    )
    # NOTE: Don't forget to update replay.yaml with new values
    _render(
        answers=answers,
        template_path=Path(__file__).parent / 'templates' / 'replay.yaml.j2',
        output_path=env.get_package_path(answers['package']) / Path('configs') / 'replay.yaml',
        force=force,
    )


def _render_templates(
    answers: Answers,
    path: Path,
    force: bool = False,
    include: set[str] | None = None,
    exists: bool = False,
) -> None:
    from jinja2 import Template

    project_path = Path(__file__).parent / 'projects' / path
    project_templates = set(project_path.glob('**/*.j2'))
    project_files = set(project_path.glob('**/*')) - project_templates

    for path in project_templates:
        template_path = path.relative_to(Path(__file__).parent)
        relative_path = str(Path(*template_path.parts[2:]))[:-3]

        if include and not any(relative_path.startswith(i) for i in include):
            continue

        output_base = env.get_package_path(answers['package']) if exists else Path(answers['package'])
        output_path = Path(
            output_base,
            *path.relative_to(project_path).parts,
            # NOTE: Remove ".j2" from extension
        ).with_suffix(path.suffix[:-3])
        output_path = Path(Template(str(output_path)).render(project=answers))

        _render(answers, template_path, output_path, force)

    # NOTE: If there are files without .j2 extension, just copy them
    for path in project_files:
        if path.is_dir() or path.name == 'replay.yaml':
            continue
        output_path = Path(
            env.get_package_path(answers['package']),
            *path.relative_to(project_path).parts,
        )
        write(output_path, path.read_bytes(), overwrite=force)


def _render(answers: Answers, template_path: Path, output_path: Path, force: bool) -> None:
    if output_path.exists() and not force:
        _logger.info('File `%s` already exists, skipping', output_path)

    _logger.info('Generating `%s`', output_path)
    template = load_template(str(template_path))
    content = ''

    project = {k: str(v) for k, v in answers.items()} if 'dipdup.yaml.j2' in str(template_path) else answers
    env_file = '\n'.join(f'# {k}=' for k in env.extract_docstrings()) if 'dipdup.env.j2' in str(template_path) else ''
    content = template.render(
        project=project,
        header=CODEGEN_HEADER,
        env_file=env_file,
    )

    write(output_path, content, overwrite=force)


def prompt_bool(
    question: str,
    default: bool = False,
) -> bool:
    choices = ('yes', 'no') if default else ('no', 'yes')
    _, value = prompt_anyof(
        question,
        choices,
        ('', ''),
        default=0,
    )
    return value == 'yes'


def prompt_kind(
    entity: str,
    types: tuple[type, ...],
    filter: str | None,
) -> type:
    matched = {}
    for entity_type in types:
        try:
            kind = entity_type.__dataclass_fields__['kind'].default  # type: ignore[attr-defined]
        except KeyError:
            kind = entity_type.__name__

        assert kind

        if filter and '.' in kind and not kind.startswith(filter):
            continue

        matched[kind] = entity_type

    if len(matched) == 1:
        return next(iter(matched))  # type: ignore[no-any-return]

    kinds = tuple(sorted(matched.keys()))
    comments = tuple('' for _ in kinds)
    _, kind = prompt_anyof(
        f'Choose {entity} kind: ',
        kinds,
        comments,
        default=0,
    )
    return matched[kind]


def fill_type_from_input(
    type_: type,
) -> Any:
    import survey

    # Gather input for the fields of the chosen type
    entity_data = {}
    for field_name, field in type_.__dataclass_fields__.items():  # type: ignore[attr-defined]
        # print(field)

        if field_name in {'kind', 'http'}:
            continue
        if field_name.startswith('_'):
            continue
        if field_name == 'handlers':
            entity_data[field_name] = ()
            continue

        default = field.default

        if default == dataclasses.MISSING:
            default = None
        elif isinstance(default, FieldInfo):
            default = default.default_factory() if default.default_factory else default.default  # type: ignore[call-arg]

        field_value = survey.routines.input(
            f'Enter value for `{field_name}` [{field.type}]: ',
            value=str(default) if default is not None else '',
        )

        if field.default_factory == dataclasses.MISSING:
            default = None

        # NOTE: Basic transformations: string lists
        if 'tuple' in field.type or 'list' in field.type:
            if not field_value:
                field_value = default
            elif ',' in field_value:
                field_value = field_value.split(',')
            else:
                field_value = [field_value]

        entity_data[field_name] = field_value or default

    # Validate and add the entity
    try:
        obj = type_(**entity_data)
        return {k: v for k, v in obj.__dict__.items() if not k.startswith('_')}
    except Exception as e:
        print(f'Error: {e}. Please try again.')


def prompt_anyof(
    question: str,
    options: tuple[str, ...],
    comments: tuple[str, ...],
    default: int,
) -> tuple[int, str]:
    """Ask user to choose one of the options; returns index and value"""
    import survey
    from tabulate import tabulate

    table = tabulate(
        zip(options, comments, strict=False),
        tablefmt='plain',
    )
    index = survey.routines.select(
        question + '\n',
        options=table.split('\n'),
        index=default,
    )
    return index, options[index]
