"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Arbitrary (valid) array of intervals
 *
 * There are many invalid combinations of interval arrays, so we have
 * to be very specific about generating arrays that are valid. We do this
 * by taking a full, valid interval schedule and progressively stripping parts
 * away from it.
 *
 * Some of the changes may change its meaning, but we take care to never leave
 * a schedule with insufficient information so that the parser will error out.
 */
function generateArbitraryIntervals(mrng) {
    const ret = new Array();
    const absolute = mrng.nextBoolean();
    // Ascending or descending scaling
    const factor = (mrng.nextBoolean() ? 1 : -1) * (absolute ? 10 : 1);
    const bias = absolute ? 50 : 0;
    // Begin with a full schedule
    ret.push({ lower: 0, upper: 10, change: -2 * factor + bias });
    ret.push({ lower: 10, upper: 20, change: -1 * factor + bias });
    ret.push({ lower: 20, upper: 60, change: 0 + bias });
    ret.push({ lower: 60, upper: 80, change: 0 + bias });
    ret.push({ lower: 80, upper: 90, change: 1 * factor + bias });
    ret.push({ lower: 90, upper: Infinity, change: 2 * factor + bias });
    // Take away parts from this. First we see if we do something to the 0-change alarms.
    // The actions can be: remove it OR turn it into a regular change value.
    const noChanges = ret.filter(x => x.change === bias);
    if (!absolute) {
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[0]), 1);
            }
            else {
                noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            }
        }
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[1]), 1);
            }
            else {
                noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
            }
        }
    }
    else {
        // In absolute mode both have to get the same treatment at the same time
        // otherwise we'll end up with a timeline with two gaps
        if (mrng.nextBoolean()) {
            ret.splice(ret.indexOf(noChanges[0]), 1);
            ret.splice(ret.indexOf(noChanges[1]), 1);
        }
        else {
            noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
        }
    }
    // We might also take away either the bottom or the upper half
    if (mrng.nextInt(0, 2) === 0) {
        const signToStrip = mrng.nextBoolean() ? -1 : 1;
        let ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        while (ix >= 0) {
            ret.splice(ix, 1);
            ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        }
    }
    // Then we're going to arbitrarily get rid of bounds in the most naive way possible
    const iterations = mrng.nextInt(0, 10);
    for (let iter = 0; iter < iterations; iter++) {
        const i = mrng.nextInt(0, ret.length - 1);
        if (mrng.nextBoolean()) {
            // scrap lower bound
            // okay if current interval has an upper bound AND the preceding interval has an upper bound
            if (ret[i].upper !== undefined && (i === 0 || ret[i - 1].upper !== undefined)) {
                ret[i] = { ...ret[i], lower: undefined };
            }
        }
        else {
            // scrap upper bound
            // okay if current interval has a lower bound AND the succeeding interval has a lower bound
            if (ret[i].lower !== undefined && (i === ret.length - 1 || ret[i + 1].lower !== undefined)) {
                ret[i] = { ...ret[i], upper: undefined };
            }
        }
    }
    return { absolute, intervals: ret };
}
exports.generateArbitraryIntervals = generateArbitraryIntervals;
//# sourceMappingURL=data:application/json;base64,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