"use strict";
const fc = require("fast-check");
const interval_utils_1 = require("../lib/interval-utils");
const util_1 = require("./util");
function realisticRelativeIntervals() {
    // Function so we don't have to worry about cloning
    return [
        { upper: 10, change: -2 },
        { upper: 20, change: -1 },
        { lower: 80, change: +1 },
        { lower: 90, change: +2 },
    ];
}
module.exports = {
    'test bounds propagation'(test) {
        const intervals = interval_utils_1.normalizeIntervals(realisticRelativeIntervals(), false);
        test.deepEqual(intervals, [
            { lower: 0, upper: 10, change: -2 },
            { lower: 10, upper: 20, change: -1 },
            { lower: 20, upper: 80, change: undefined },
            { lower: 80, upper: 90, change: +1 },
            { lower: 90, upper: Infinity, change: +2 },
        ]);
        test.done();
    },
    'bounds propagation fails if middle boundary missing'(test) {
        test.throws(() => {
            interval_utils_1.normalizeIntervals([
                { lower: 0, change: -2 },
                { upper: 20, change: -1 },
            ], false);
        });
        test.done();
    },
    'lower alarm index is lower than higher alarm index'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            return (alarms.lowerAlarmIntervalIndex === undefined
                || alarms.upperAlarmIntervalIndex === undefined
                || alarms.lowerAlarmIntervalIndex < alarms.upperAlarmIntervalIndex);
        }));
        test.done();
    },
    'never pick undefined intervals for relative alarms'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            return (alarms.lowerAlarmIntervalIndex === undefined || intervals[alarms.lowerAlarmIntervalIndex].change !== undefined)
                && (alarms.upperAlarmIntervalIndex === undefined || intervals[alarms.upperAlarmIntervalIndex].change !== undefined);
        }));
        test.done();
    },
    'pick intervals on either side of the undefined interval, if present'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            // There must be an undefined interval and it must not be at the edges
            const i = intervals.findIndex(x => x.change === undefined);
            fc.pre(i > 0 && i < intervals.length - 1);
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            return (alarms.lowerAlarmIntervalIndex === i - 1 && alarms.upperAlarmIntervalIndex === i + 1);
        }));
        test.done();
    },
    'no picking upper bound infinity for lower alarm'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            fc.pre(alarms.lowerAlarmIntervalIndex !== undefined);
            return intervals[alarms.lowerAlarmIntervalIndex].upper !== Infinity;
        }));
        test.done();
    },
    'no picking lower bound 0 for upper alarm'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            fc.pre(alarms.upperAlarmIntervalIndex !== undefined);
            return intervals[alarms.upperAlarmIntervalIndex].lower !== 0;
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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