// ComponentInspector for React Grab functionality
class ComponentInspector {
    constructor(rootElement, model) {
        this.root = rootElement || document.body;
        this.model = model;
        this.overlay = null;
        this.tooltip = null;
        this.selectedElement = null;
        this.inspecting = false;
        this.componentMap = new WeakMap();
        
        // Bind methods
        this.handleMouseMove = this.handleMouseMove.bind(this);
        this.handleClick = this.handleClick.bind(this);
        this.handleKeyDown = this.handleKeyDown.bind(this);
    }
    
    startInspecting() {
        if (this.inspecting) return;
        
        this.inspecting = true;
        this.createOverlay();
        this.createTooltip();
        this.attachEventListeners();
        
        // Visual feedback
        document.body.style.cursor = 'crosshair';
    }
    
    stopInspecting() {
        this.inspecting = false;
        this.removeEventListeners();
        
        if (this.overlay) {
            this.overlay.remove();
            this.overlay = null;
        }
        
        if (this.tooltip) {
            this.tooltip.remove();
            this.tooltip = null;
        }
        
        document.body.style.cursor = '';
    }
    
    createOverlay() {
        this.overlay = document.createElement('div');
        this.overlay.style.cssText = `
            position: fixed;
            pointer-events: none;
            border: 2px solid #F37726;
            background: rgba(243, 119, 38, 0.1);
            z-index: 10000;
            transition: all 0.1s ease;
            box-shadow: 0 0 0 1px rgba(243, 119, 38, 0.3);
        `;
        document.body.appendChild(this.overlay);
    }
    
    createTooltip() {
        this.tooltip = document.createElement('div');
        this.tooltip.style.cssText = `
            position: fixed;
            pointer-events: none;
            background: #1a1a1a;
            color: #F37726;
            padding: 6px 10px;
            border-radius: 4px;
            font-family: 'JetBrains Mono', monospace;
            font-size: 11px;
            z-index: 10001;
            box-shadow: 0 2px 8px rgba(0,0,0,0.3);
            border: 1px solid rgba(243, 119, 38, 0.3);
            display: none;
        `;
        document.body.appendChild(this.tooltip);
    }
    
    attachEventListeners() {
        document.addEventListener('mousemove', this.handleMouseMove, true);
        document.addEventListener('click', this.handleClick, true);
        document.addEventListener('keydown', this.handleKeyDown, true);
    }
    
    removeEventListeners() {
        document.removeEventListener('mousemove', this.handleMouseMove, true);
        document.removeEventListener('click', this.handleClick, true);
        document.removeEventListener('keydown', this.handleKeyDown, true);
    }
    
    handleMouseMove(e) {
        if (!this.inspecting) return;
        
        const element = e.target;
        
        // Skip our own overlay and tooltip
        if (element === this.overlay || element === this.tooltip) return;
        
        const rect = element.getBoundingClientRect();
        
        // Update overlay position
        this.overlay.style.left = rect.left + 'px';
        this.overlay.style.top = rect.top + 'px';
        this.overlay.style.width = rect.width + 'px';
        this.overlay.style.height = rect.height + 'px';
        
        // Show component info tooltip
        this.showTooltip(element, e.clientX, e.clientY);
    }
    
    showTooltip(element, x, y) {
        const info = this.getElementInfo(element);
        
        this.tooltip.innerHTML = `
            <div style="margin-bottom: 4px; color: #fff;">${info.type}</div>
            <div style="color: rgba(243, 119, 38, 0.8); font-size: 10px;">${info.path}</div>
        `;
        
        // Position tooltip
        this.tooltip.style.display = 'block';
        this.tooltip.style.left = (x + 15) + 'px';
        this.tooltip.style.top = (y + 15) + 'px';
        
        // Keep tooltip in viewport
        const tooltipRect = this.tooltip.getBoundingClientRect();
        if (tooltipRect.right > window.innerWidth) {
            this.tooltip.style.left = (x - tooltipRect.width - 15) + 'px';
        }
        if (tooltipRect.bottom > window.innerHeight) {
            this.tooltip.style.top = (y - tooltipRect.height - 15) + 'px';
        }
    }
    
    handleClick(e) {
        if (!this.inspecting) return;
        
        e.preventDefault();
        e.stopPropagation();
        
        const element = e.target;
        
        // Skip our own elements
        if (element === this.overlay || element === this.tooltip) return;
        
        this.selectedElement = element;
        const componentData = this.extractComponentData(element);
        
        // Generate widget reference
        const widgetId = this.getWidgetId();
        const reference = `vw.component("${widgetId}:${componentData.domPath}")`;
        
        // Copy to clipboard
        this.copyToClipboard(reference);
        
        // Send to Python via model
        this.sendToPython(componentData);
        
        // Show success notification
        this.showNotification(`Copied: ${reference}`);
        
        // Stop inspecting
        this.stopInspecting();
    }
    
    handleKeyDown(e) {
        if (!this.inspecting) return;
        
        // ESC to cancel
        if (e.key === 'Escape') {
            e.preventDefault();
            this.stopInspecting();
        }
    }
    
    getElementInfo(element) {
        const reactInfo = this.findReactInfo(element);
        const domPath = this.getDOMPath(element);
        
        if (reactInfo) {
            return {
                type: `React: ${reactInfo.name}`,
                path: domPath
            };
        }
        
        return {
            type: `DOM: <${element.tagName.toLowerCase()}>`,
            path: domPath
        };
    }
    
    extractComponentData(element) {
        const reactInfo = this.findReactInfo(element);
        const domPath = this.getDOMPath(element);
        
        if (reactInfo) {
            return {
                type: 'react_component',
                name: reactInfo.name,
                props: reactInfo.props,
                state: reactInfo.state,
                hooks: reactInfo.hooks,
                domPath: domPath,
                elementInfo: {
                    tagName: element.tagName,
                    id: element.id,
                    className: element.className
                }
            };
        }
        
        // Fallback to DOM inspection
        return {
            type: 'dom_element',
            tagName: element.tagName,
            id: element.id,
            className: element.className,
            attributes: this.getAttributes(element),
            domPath: domPath,
            innerText: element.innerText?.substring(0, 100)
        };
    }
    
    findReactInfo(element) {
        // Look for React Fiber
        const fiberKey = Object.keys(element).find(key => 
            key.startsWith('__reactFiber') || 
            key.startsWith('__reactInternalInstance')
        );
        
        if (!fiberKey) return null;
        
        const fiber = element[fiberKey];
        if (!fiber) return null;
        
        // Walk up to find the component fiber
        let current = fiber;
        while (current && !current.elementType?.name && current.return) {
            current = current.return;
        }
        
        if (!current) return null;
        
        return {
            name: current.elementType?.name || current.elementType?.displayName || 'Component',
            props: this.sanitizeProps(current.memoizedProps),
            state: current.memoizedState,
            hooks: this.extractHooks(current)
        };
    }
    
    sanitizeProps(props) {
        if (!props) return {};
        
        const safe = {};
        for (const key in props) {
            if (key === 'children' || key.startsWith('__')) continue;
            
            const value = props[key];
            if (typeof value === 'function') {
                safe[key] = '[Function]';
            } else if (typeof value === 'object' && value !== null) {
                safe[key] = '[Object]';
            } else {
                safe[key] = value;
            }
        }
        return safe;
    }
    
    extractHooks(fiber) {
        const hooks = [];
        let hook = fiber.memoizedState;
        let hookIndex = 0;
        
        while (hook && hookIndex < 10) { // Limit to prevent infinite loops
            hooks.push({
                index: hookIndex,
                value: hook.memoizedState,
                deps: hook.deps
            });
            hook = hook.next;
            hookIndex++;
        }
        
        return hooks;
    }
    
    getDOMPath(element) {
        const path = [];
        let current = element;
        
        while (current && current !== document.body) {
            const selector = current.tagName.toLowerCase();
            
            if (current.id) {
                path.unshift(`#${current.id}`);
                break;
            } else if (current.className && typeof current.className === 'string') {
                const classes = current.className.split(' ').filter(c => c);
                if (classes.length > 0) {
                    path.unshift(`${selector}.${classes[0]}`);
                } else {
                    path.unshift(selector);
                }
            } else {
                // Add index if there are siblings
                const parent = current.parentElement;
                if (parent) {
                    const siblings = Array.from(parent.children).filter(
                        child => child.tagName === current.tagName
                    );
                    if (siblings.length > 1) {
                        const index = siblings.indexOf(current);
                        path.unshift(`${selector}[${index}]`);
                    } else {
                        path.unshift(selector);
                    }
                } else {
                    path.unshift(selector);
                }
            }
            current = current.parentElement;
        }
        
        return path.join(' > ');
    }
    
    getAttributes(element) {
        const attrs = {};
        for (const attr of element.attributes) {
            attrs[attr.name] = attr.value;
        }
        return attrs;
    }
    
    getWidgetId() {
        // Try to get widget ID from model or generate one
        if (this.model && this.model.get) {
            return this.model.get('widget_id') || this.model.model_id || 'widget';
        }
        return 'widget';
    }
    
    copyToClipboard(text) {
        navigator.clipboard.writeText(text).catch(err => {
            console.error('Failed to copy to clipboard:', err);
            // Fallback method
            const textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'absolute';
            textarea.style.left = '-9999px';
            document.body.appendChild(textarea);
            textarea.select();
            document.execCommand('copy');
            document.body.removeChild(textarea);
        });
    }
    
    sendToPython(data) {
        if (this.model && this.model.set) {
            this.model.set('grabbed_component', data);
            this.model.save_changes();
        }
    }
    
    showNotification(message) {
        const notification = document.createElement('div');
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: linear-gradient(135deg, #1a1a1a 0%, #2a2a2a 100%);
            color: #F37726;
            padding: 12px 20px;
            border-radius: 6px;
            font-family: 'Inter', sans-serif;
            font-size: 14px;
            z-index: 10002;
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
            border: 1px solid rgba(243, 119, 38, 0.3);
            animation: slideIn 0.3s ease;
        `;
        
        notification.innerHTML = `
            <style>
                @keyframes slideIn {
                    from { transform: translateX(100%); opacity: 0; }
                    to { transform: translateX(0); opacity: 1; }
                }
                @keyframes slideOut {
                    from { transform: translateX(0); opacity: 1; }
                    to { transform: translateX(100%); opacity: 0; }
                }
            </style>
            ${message}
        `;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
}

export default ComponentInspector;