# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ptera']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'ptera',
    'version': '0.1.3',
    'description': 'Call graph addressing library.',
    'long_description': '\n# Ptera\n\n**Note**: This is super alpha. A lot of the features are implemented very inefficiently and the error reporting is not very good. That will be fixed in due time, and then this note will disappear into the mists of git history.\n\n\n## What is Ptera?\n\nPtera is a bit like a tracer, a bit like aspect-oriented programming, a bit like a new way to program. It chiefly aims to facilitate algorithm research, especially machine learning.\n\n**Features!**\n\n* **"Selfless objects"** are objects that are defined as functions. The fields of these objects are simply the variables used in the function.\n* **Execution trace queries** lets you address any variable anywhere in functions decorated with `@ptera`, at arbitrary depths in the call tree, and either collect the values of these variables, or set these values.\n* **Automatic CLI** will create a command-line interface from any properly annotated variable anywhere in the code.\n\nAlthough they can be used independently, selfless objects and trace queries interact together in Ptera in order to form a new programming paradigm.\n\n* Define parameterized functions with basically no boilerplate.\n* Interfere with carefully selected parts of your program to test edge cases or variants.\n* Plot the values of some variable deep in the execution of a program.\n\nPtera can also be combined with other libraries to make certain tasks very easy:\n\n* [Using Ptera with PyTorch](https://github.com/mila-iqia/ptera/blob/master/ML-README.md), it is easy to get the gradient of any quantity with respect to any other quantity. These quantities can be *anywhere* in the program! They don\'t need to be in the same scope!\n\n\n## Selfless objects\n\nA selfless object is defined as a simple function. Any function can be a selfless object, but of particular interest are those that contain uninitialized variables. For example:\n\n```python\n@ptera.defaults(debug=False)\ndef bagel(x, y):\n    z: int\n    if debug:\n        print(x, y, z)\n    return x + y + z\n```\n\nNote that in this function, `z` is declared, but not initialized. We can instantiate a function with `new`:\n\n```python\nbagel3 = bagel.new(z=3)\nassert bagel3(1, 2) == 6\n```\n\nBut this is not all we can do. We can also set the `debug` variable to True:\n\n```python\nbagel3_debug = bagel.new(z=3, debug=True)\nassert bagel3(1, 2) == 6   # This prints "1 2 3"\n```\n\nAnd that\'s not all! We can set *any* variable, if we so wish. We can set `print`:\n\n```python\nbagel3_newprint = bagel.new(\n    z=3, debug=True, print=lambda *args: print(sum(args))\n)\nassert bagel3(1, 2) == 6   # This prints "6"\n```\n\nWe can set default arguments:\n\n```python\nbagel3_defaults = bagel.new(z=3, y=2)\nassert bagel3(1) == 6\n```\n\nWe can *force* an argument to have a certain value:\n\n```python\nbagel3_forcex = bagel.new(z=3, x=ptera.Override(5))\nassert bagel3(1, 2) == 10\n```\n\n\n## Trace queries\n\nTrace queries provide the same power, but over a whole call tree. They also allow you to extract any variables you want.\n\nSuppose you have this program:\n\n```python\n@ptera\ndef square(x):\n    rval = x * x\n    return rval\n\n@ptera\ndef sumsquares(x, y):\n    xx = square(x)\n    yy = square(y)\n    rval = xx + yy\n    return rval\n```\n\nWhat can you do?\n\n\n**Q**: What values can `x` take?\n\n**A**: The `using` method lets you extract these values. We give our query the name `q` in what follows so that we can access it, but you can give any name you want to the query:\n\n```python\nresults = sumsquares.using(q="x")(3, 4)\nassert results.q.map("x") == [3, 4, 3]\n```\n\n\n**Q**: Hold on, why is `3` listed twice?\n\n**A**: Because `x == 3` in the call to `sumsquares`, and `x == 3` in the first  call to `square`. These are two distinct `x`.\n\n\n**Q**: What if I just want the value of `x` in `square`?\n\n**A**: The expression `square > x` represents the value of the variable `x` inside a call to `square`.\n\n```python\nresults = sumsquares.using(q="square > x")(3, 4)\nassert results.q.map("x") == [3, 4]\n```\n\n\n**Q**: Can I also see the output of `square`?\n\n**A**: Yes. Variables inside `{}` will also be captured.\n\n```python\nresults = sumsquares.using(q="square{rval} > x")(3, 4)\nassert results.q.map("x", "rval") == [(3, 9), (4, 16)]\n```\n\n\n**Q**: Can I also see the inputs of `sumsquare` along with these?\n\n**A**: Yes, but there is a name conflict, so you need to rename them, which you can do in the query, like this:\n\n```python\nresults = sumsquares.using(\n    q="sumsquares{x as ssx, y as ssy} > square{rval} > x"\n)(3, 4)\nassert results.q.map("ssx", "ssy", "x", "rval") == [(3, 4, 3, 9), (3, 4, 4, 16)]\n```\n\n\n**Q**: I would like to have one entry for each call to `sumsquares`, not one entry for each call to `square`.\n\n**A**: Each query has one variable which is the *focus*. There will be one result for each value the focus takes. You can set the focus with the `!` operator. So here\'s something you can do:\n\n```python\nresults = sumsquares.using(\n    q="sumsquares{!x as ssx, y as ssy} > square{rval, x}"\n)(3, 4)\nassert (results.q.map_all("ssx", "ssy", "x", "rval")\n        == [([3], [4], [3, 4], [9, 16])])\n```\n\nNotice that you need to call `map_all` here, because some variables have multiple values with respect to the focus: we focus on the `x` argument to `sumsquares`, which calls `square` twice, so for each `sumsquares{x}` we get two `square{x, rval}`. The `map` method assumes there is only one value for each variable, so it will raise an exception.\n\nNote that this view on the data does not necessarily preserve the correspondance between `square{x}` and `square{rval}`: you can\'t assume that the first `x` is in the same scope as the first `rval`, and so on.\n\nAlso notice that the expression does not end with `> x`. That\'s because `square > x` is the same as `square{!x}`: it sets the focus on `x`. However, we can only have one focus, therefore if we ended the query with `> x` it would be invalid.\n\n\n**Q**: I want to do something crazy. I want `square` to always return 0.\n\n**A**: Uhh... okay? Are you sure? You can use the `tweak` method to do this:\n\n```python\nresult = sumsquares.tweak({"square > rval": 0})(3, 4)\nassert result == 0\n```\n\nThis will apply to all calls to `square` within the execution of `sumsquares`. And yes, `sumsquares.new(square=lambda x: 0)` would also work in this case, but there is a difference: using the `tweak` method will apply to all calls at all call depths, recursively. The `new` method will only change `square` directly in the body of `sumsquares`.\n\n\n**Q**: I want to do something else crazy. I want `square` to return x + 1.\n\n**A**: Use the `rewrite` method.\n\n```python\nresult = sumsquares.rewrite({"square{x} > rval": lambda x: x + 1})(3, 4)\nassert result == 9\n```\n\n## Automatic CLI\n\nPtera can automatically create a command-line interface from variables annotated with a certain category. The main advantage of `ptera.auto_cli` relative to other solutions is that the arguments are declared wherever you actually use them. Consider the following program, for example:\n\n```python\ndef main():\n    for i in range(1000):\n        do_something()\n\nif __name__ == "__main__":\n    main()\n```\n\nIt would be nice to be able to configure the number of iterations instead of using the hard-coded number 1000. Enter `ptera.auto_cli`:\n\n```python\nfrom ptera import auto_cli, cat, default, ptera\n\n@ptera\ndef main():\n    # Number of iterations\n    n: cat.CliArgument = default(1000)\n    for i in range(n):\n        do_something()\n\nif __name__ == "__main__":\n    auto_cli(main, category=cat.CliArgument)\n```\n\nThen you can run it like this, for example:\n\n```bash\n$ python script.py --n 15\n```\n\n* Ptera will look for any variable annotated with the specified category within `@ptera` functions that are accessible from `main`.\n  * There is no need to pass an options object around. If you need to add an argument to any function in any file, you can just plop it in there and ptera should find it and allow you to set it on the command line.\n  * You can declare multiple CLI arguments in multiple places with the same name. They will all be set to the same value.\n* The comment right above the declaration of the variable, if there is one, is used as documentation.\n* The `default` function provides a default value for the argument.\n  * You don\'t have to provide one.\n  * Ptera uses a priority system to determine which value to choose and `default` sets a low priority. If you set a value but don\'t wrap it with `default`, you will get a `ConflictError` when trying to set the variable on the command line. This is the intended behavior.\n\nIn the future, `auto_cli` will also support environment variables, configuration files, and extra options to catalogue all the variables that can be queried. For example, a planned feature is to be able to display where in the code each variable with a given category is declared and used.\n\nAnother future feature: since it is within ptera\'s ability to set different values for the parameter `param` depending on the call context (e.g. with the `tweak` method), the ability to do this in a configuration file will be added at some point (just need to figure out the format).\n\n\n## Query language\n\nHere is some code annotated with queries that will match various variables. The queries are not exhaustive, just examples.\n\n* The semicolon ";" is used to separate queries and it is not part of any query.\n* The hash character "#" *is* part of the query if there is no space after it, otherwise it starts a comment.\n\n```python\nfrom ptera import cat, ptera\n\n\n@ptera\ndef art(a, b):               # art > a ; art > b ; art{!a, b} ; art{a, !b}\n\n    a1: cat.Animal = bee(a)  # a1 ; art > a1 ; art{!a1} ; art > $x\n                             # a1:Animal ; $x:Animal\n                             # art{!a1} > bee > d  # Focus on a1, also includes d\n                             # art > bee  # This refers to the bee function\n                             # * > a1 ; *{!a1}\n\n    a2: cat.Thing = cap(b)   # a2 ; art > a2 ; art{!a2} ; art > $x\n                             # a2:Thing ; $x:Thing\n\n    return a1 + a2           # art > #value ; art{#value as art_result}\n                             # art{} as art_result\n                             # art > $x\n\n\n@ptera\ndef bee(c):\n    c1 = c + 1               # bee > c1 ; art >> c1 ; art{a2} > bee > c1\n                             # bee > c1 as xyz\n\n    return c1                # bee > #value ; bee{c} as bee_value\n\n\n@ptera\ndef cap(d: cat.Thing & int): # cap > d ; $x:Thing ; $x:int ; cap > $x\n                             # art{bee{c}} > cap > d\n    return d * d\n```\n\n* The `!` operator marks the **focus** of the query. There will be one result for each time the focus is triggered, and when using `tweak` or `rewrite` the focus is what is being tweaked or rewritten.\n  * Other variables are supplemental information, available along with the focus in query results. They can also be used to compute a value for the focus *if* they are available by the time the focus is reached.\n  * The nesting operators `>` and `>>` automatically set the focus to the right hand side if the rhs is a single variable and the operator is not inside `{...}`.\n* The wildcard `*` stands in for any function.\n* The `>>` operator represents **deep nesting**. For example, `art >> c1` encompasses the pattern `art > bee > c1`.\n  * In general, `a >> z` encompasses `a > z`, `a > b > z`, `a > b > c > z`, `a > * > z`, and so on.\n* A function\'s return value corresponds to a special variable named `#value`.\n* `$x` will match any variable name. Getting the variable name for the capture is possible but requires the `map_full` method. For example:\n  * Query: `art > $x`\n  * Getting the names: `results.map_full(lambda x: x.name) == ["a1", "a2", "#value"]`\n  * Other fields accessible from `map_full` are `value`, `names` and `values`, the latter two being needed if multiple results are captured together.\n* Variable annotations are preserved and can be filtered on, using the `:` operator. They may be types or "categories" (created using `ptera.Category("XYZ")` or `ptera.cat.XYZ`).\n* `art{bee{c}} > cap > d` triggers on the variable `d` in calls to `cap`, but it will *also* include the value of `c` for all calls to `bee` inside `art`.\n  * If there are multiple calls to `bee`, all values of `c` will be pooled together, and it will be necessary to use `map_all` to retrieve the values (or `map_full`).\n\n\n### Equivalencies\n\nPtera\'s query language syntax includes a lot of expressions, but it reduces to a relatively simple core:\n\n\n```\n# A lone symbol becomes a match for a variable in a wildcard function\na               <=>  *{!a}\n\n# Nesting operator > is sugar for {...}\na > b           <=>  a{!b}\na > b > c       <=>  a{b{!c}}\n\n# Infix >> is sugar for prefix >>\na >> b > c      <=>  a{>> b{!c}}\na >> b          <=>  a{>> !b}   <~>  a{>> *{!b}}  (see note)\n\n# $x is shorthand for as\nf{$x}           <=>  f{* as x}\n\n# Indexing is sugar for #key\na[0]            <=>  a{#key=0}\na[0] as a0      <=>  a{#key=0, #value as a0}\na[$i] as a      <=>  a{#key as i, #value as a}\n\n# Shorthand for #value\na{x, y} as b    <=>  a{x, y, #value as b}\na{} as b        <=>  a{#value as b}\n```\n\nNote: `a{>> b}` and `a{>> *{!b}}` are not 100% equivalent, because the former encompasses `a{> b}` and the latter does not, but internally the former is basically encoded as the latter plus a special "collapse" flag that there is no syntax for.\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/breuleux/ptera',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
