# LangSpend Python SDK Examples

This directory contains comprehensive examples showing how to use the LangSpend Python SDK with different LLM providers and frameworks.

## Prerequisites

1. Install the LangSpend Python SDK:
   ```bash
   pip install langspend
   ```

2. Install provider-specific dependencies:
   ```bash
   pip install langspend[all]  # For all providers
   # OR
   pip install langspend[openai]      # OpenAI only
   pip install langspend[anthropic]   # Anthropic only
   pip install langspend[bedrock]     # AWS Bedrock only
   ```

3. Set up your environment variables:
   ```bash
   export LANGSPEND_API_KEY="lsp_your_api_key_here"
   export OPENAI_API_KEY="your_openai_api_key"
   export ANTHROPIC_API_KEY="your_anthropic_api_key"
   export AWS_ACCESS_KEY_ID="your_aws_access_key"
   export AWS_SECRET_ACCESS_KEY="your_aws_secret_key"
   export AWS_REGION="us-east-1"
   ```

## Examples

### 1. Basic Usage (`basic_usage.py`)
The simplest way to get started with LangSpend. Shows how to wrap an OpenAI client and make a basic request.

```bash
python examples/basic_usage.py
```

### 2. Customer Attribution (`customer_attribution.py`)
Demonstrates how to use customer and feature objects for detailed cost attribution.

```bash
python examples/customer_attribution.py
```

### 3. Anthropic Integration (`anthropic_usage.py`)
Shows how to use LangSpend with Anthropic's Claude models.

```bash
python examples/anthropic_usage.py
```

### 4. AWS Bedrock Integration (`bedrock_usage.py`)
Demonstrates integration with AWS Bedrock for enterprise deployments.

```bash
python examples/bedrock_usage.py
```

### 5. Flask Integration (`flask_integration.py`)
Shows how to integrate LangSpend with Flask applications for automatic feature detection based on API endpoints.

```bash
python examples/flask_integration.py
```

Then test the endpoints:
```bash
curl -X POST http://localhost:5000/api/chat \
  -H "Content-Type: application/json" \
  -d '{"message": "Hello!", "customer_id": "user123", "customer_name": "John Doe"}'
```

### 6. FastAPI Integration (`fastapi_integration.py`)
Demonstrates integration with FastAPI applications with automatic feature detection.

```bash
pip install fastapi uvicorn
python examples/fastapi_integration.py
```

Then test the endpoints:
```bash
curl -X POST http://localhost:8000/api/chat \
  -H "Content-Type: application/json" \
  -d '{"message": "Hello!", "customer_id": "user123", "customer_name": "John Doe"}'
```

### 7. Streaming Responses (`streaming_example.py`)
Shows how to use LangSpend with streaming LLM responses.

```bash
python examples/streaming_example.py
```

### 8. Multi-Provider Usage (`multi_provider.py`)
Demonstrates using multiple LLM providers in the same application.

```bash
python examples/multi_provider.py
```

## Key Features Demonstrated

### Customer and Feature Objects
All examples use the new customer and feature object structure:

```python
from langspend import Customer, Feature

customer = Customer(
    id="user_123",
    name="John Doe",
    email="john@example.com",
    meta={"plan": "premium", "region": "us-west"}
)

feature = Feature(
    name="chat-api",
    meta={"version": "1.0.0", "category": "api"}
)

# Use in langspend_tags
langspend_tags={
    "customer": customer,
    "feature": feature,
    "session_id": "session_123",
    "environment": "production"
}
```

### Automatic Feature Detection
When using Flask or FastAPI, features are automatically detected from endpoint paths:

- `/api/chat` → feature name: `api-chat`
- `/api/summarize` → feature name: `api-summarize`
- `/api/translate` → feature name: `api-translate`

### Error Handling
All examples include proper error handling to ensure LLM calls succeed even if tracking fails.

### Async Support
Examples show both synchronous and asynchronous usage patterns.

## Customization

You can customize the examples by:

1. **Changing models**: Update the `model` parameter in API calls
2. **Adding metadata**: Include custom metadata in customer and feature objects
3. **Custom feature names**: Override automatic detection with manual feature names
4. **Different providers**: Mix and match OpenAI, Anthropic, and Bedrock as needed

## Troubleshooting

### Common Issues

1. **Missing API keys**: Ensure all required environment variables are set
2. **Import errors**: Make sure you've installed the required dependencies
3. **Tracking failures**: Check your LangSpend API key and network connectivity
4. **Provider errors**: Verify your provider API keys and quotas

### Debug Mode

Enable debug mode to see detailed tracking information:

```python
langspend = LangSpend(
    api_key="your_api_key",
    debug=True
)
```

## Support

For questions or issues:
- Check the [main documentation](https://docs.langspend.com/python)
- Open an issue on [GitHub](https://github.com/langspend/langspend-python)
- Contact support at support@langspend.com
