"""
Anthropic Usage Example

This example shows how to use LangSpend with Anthropic's Claude models.
"""

import os
from langspend import LangSpend, wrap_anthropic, Customer, Feature
from anthropic import Anthropic

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here')
)

# Wrap your Anthropic client
anthropic = wrap_anthropic(
    Anthropic(api_key=os.getenv('ANTHROPIC_API_KEY')),
    langspend
)

async def chat_with_claude(customer: Customer, feature: Feature, message: str):
    """Send a message to Claude with customer and feature attribution"""
    print(f'\nClaude chat for customer: {customer.name}')
    print(f'Feature: {feature.name}')
    
    response = anthropic.messages.create(
        model='claude-3-5-sonnet-20241022',
        max_tokens=1024,
        messages=[
            {'role': 'user', 'content': message}
        ],
        # Customer and feature objects
        langspend_tags={
            'customer': customer,
            'feature': feature,
            'session_id': 'session_456',
            'environment': 'production'
        }
    )
    
    print('Response:', response.content[0].text)
    print(f'✅ Cost attributed to customer: {customer.id}')
    
    return response

async def main():
    # Example: Using Claude for different use cases
    await chat_with_claude(
        Customer(
            id='user_101',
            name='Sarah Wilson',
            email='sarah@example.com',
            meta={'plan': 'premium', 'region': 'us-west'}
        ),
        Feature(
            name='code-review',
            meta={'version': '1.0.0', 'category': 'development'}
        ),
        'Review this Python code for best practices and suggest improvements'
    )
    
    await chat_with_claude(
        Customer(
            id='user_202',
            name='Mike Chen',
            email='mike@example.com',
            meta={'plan': 'basic', 'region': 'eu-central'}
        ),
        Feature(
            name='documentation',
            meta={'version': '2.1.0', 'category': 'content'}
        ),
        'Generate comprehensive documentation for this API endpoint'
    )
    
    print('\n✅ All Claude requests tracked with customer attribution!')

if __name__ == '__main__':
    import asyncio
    asyncio.run(main())
