"""
AWS Bedrock Usage Example

This example shows how to use LangSpend with AWS Bedrock.
"""

import os
import json
import boto3
from langspend import LangSpend, wrap_bedrock, Customer, Feature
from botocore.exceptions import ClientError

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here')
)

# Initialize Bedrock client
bedrock_client = boto3.client(
    'bedrock-runtime',
    region_name=os.getenv('AWS_REGION', 'us-east-1'),
    aws_access_key_id=os.getenv('AWS_ACCESS_KEY_ID'),
    aws_secret_access_key=os.getenv('AWS_SECRET_ACCESS_KEY')
)

# Wrap your Bedrock client
bedrock = wrap_bedrock(bedrock_client, langspend)

async def chat_with_bedrock(customer: Customer, feature: Feature, message: str):
    """Send a message to Bedrock Claude with customer and feature attribution"""
    print(f'\nBedrock chat for customer: {customer.name}')
    print(f'Feature: {feature.name}')
    
    try:
        # Create the invoke model command
        from boto3.session import Session
        from botocore.client import BaseClient
        
        # Prepare the request body
        body = json.dumps({
            'anthropic_version': 'bedrock-2023-05-31',
            'max_tokens': 1024,
            'messages': [{'role': 'user', 'content': message}]
        })
        
        # Send the request with LangSpend tracking
        response = bedrock.invoke_model(
            modelId='anthropic.claude-3-5-sonnet-20241022-v2:0',
            body=body,
            # LangSpend tracking tags
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': 'session_789',
                'environment': 'production'
            }
        )
        
        # Parse the response
        response_body = json.loads(response['body'].read())
        message_content = response_body['content'][0]['text']
        
        print('Response:', message_content)
        print(f'✅ Cost attributed to customer: {customer.id}')
        
        return response_body
        
    except ClientError as e:
        print(f'Error calling Bedrock: {e}')
        return None

async def main():
    # Example: Using Bedrock for different use cases
    await chat_with_bedrock(
        Customer(
            id='user_301',
            name='Alex Rodriguez',
            email='alex@example.com',
            meta={'plan': 'enterprise', 'region': 'us-east-1'}
        ),
        Feature(
            name='data-analysis',
            meta={'version': '1.5.0', 'category': 'analytics'}
        ),
        'Analyze this sales data and provide insights on trends'
    )
    
    await chat_with_bedrock(
        Customer(
            id='user_402',
            name='Lisa Park',
            email='lisa@example.com',
            meta={'plan': 'premium', 'region': 'us-west-2'}
        ),
        Feature(
            name='content-moderation',
            meta={'version': '2.0.0', 'category': 'safety'}
        ),
        'Review this user-generated content for policy violations'
    )
    
    print('\n✅ All Bedrock requests tracked with customer attribution!')

if __name__ == '__main__':
    import asyncio
    asyncio.run(main())
