"""
Customer Attribution Example

This example shows how to attribute LLM costs to specific customers
using customer and feature objects in langspend_tags.
"""

import os
from langspend import LangSpend, wrap_openai, Customer, Feature
from openai import OpenAI

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here')
)

# Wrap your OpenAI client
openai = wrap_openai(
    OpenAI(api_key=os.getenv('OPENAI_API_KEY')),
    langspend
)

async def chat_with_customer(customer: Customer, feature: Feature, message: str):
    """Send a chat message with customer and feature attribution"""
    print(f'\nChat for customer: {customer.name} ({customer.email})')
    print(f'Feature: {feature.name}')
    
    response = openai.chat.completions.create(
        model='gpt-4',
        messages=[
            {'role': 'user', 'content': message}
        ],
        # Customer and feature objects with rich metadata
        langspend_tags={
            'customer': customer,
            'feature': feature,
            'session_id': 'session_123',
            'environment': 'production'
        }
    )
    
    print('Response:', response.choices[0].message.content)
    print(f'✅ Cost attributed to customer object: {customer.id}')
    
    return response

async def main():
    # Example: Different customers with rich metadata
    await chat_with_customer(
        Customer(
            id='user_123',
            name='Alice Johnson',
            email='alice@example.com',
            meta={
                'plan': 'premium',
                'signup_date': '2024-01-15'
            }
        ),
        Feature(
            name='chat-assistant',
            meta={
                'version': '2.1.0',
                'category': 'ai-features'
            }
        ),
        'Explain quantum computing in simple terms'
    )
    
    await chat_with_customer(
        Customer(
            id='user_456',
            name='Bob Smith',
            email='bob@example.com',
            meta={
                'plan': 'basic',
                'signup_date': '2024-02-20'
            }
        ),
        Feature(
            name='content-generation',
            meta={
                'version': '1.5.2',
                'category': 'productivity'
            }
        ),
        'Write a haiku about programming'
    )
    
    # Example: Customer without email (optional field)
    await chat_with_customer(
        Customer(
            id='user_789',
            name='Charlie Brown',
            meta={
                'plan': 'enterprise',
                'signup_date': '2024-03-01'
            }
        ),
        Feature(
            name='data-analysis',
            meta={
                'version': '3.0.0',
                'category': 'analytics'
            }
        ),
        'Analyze this dataset and provide insights'
    )
    
    print('\n✅ All requests tracked with customer and feature objects!')

if __name__ == '__main__':
    import asyncio
    asyncio.run(main())
