"""
FastAPI Integration Example

This example shows how to integrate LangSpend with a FastAPI application
for automatic feature detection based on API endpoints.
"""

import os
from fastapi import FastAPI, HTTPException
from pydantic import BaseModel
from langspend import LangSpend, wrap_openai, Customer, Feature
from openai import OpenAI

# Initialize FastAPI app
app = FastAPI(title="LangSpend FastAPI Example", version="1.0.0")

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here')
)

# Wrap your OpenAI client
openai = wrap_openai(
    OpenAI(api_key=os.getenv('OPENAI_API_KEY')),
    langspend
)

# Pydantic models
class ChatRequest(BaseModel):
    message: str
    customer_id: str = "anonymous"
    customer_name: str = "User"
    customer_email: str = None

class ChatResponse(BaseModel):
    response: str
    customer_id: str

class SummarizeRequest(BaseModel):
    text: str
    customer_id: str = "anonymous"

class SummarizeResponse(BaseModel):
    summary: str
    customer_id: str

class TranslateRequest(BaseModel):
    text: str
    target_language: str = "Spanish"
    customer_id: str = "anonymous"

class TranslateResponse(BaseModel):
    translation: str
    customer_id: str

@app.post("/api/chat", response_model=ChatResponse)
async def chat_endpoint(request_data: ChatRequest, request):
    """Chat endpoint with automatic feature detection"""
    try:
        # Create customer and feature objects
        customer = Customer(
            id=request_data.customer_id,
            name=request_data.customer_name,
            email=request_data.customer_email,
            meta={'source': 'fastapi'}
        )
        
        feature = Feature(
            name='chat-api',  # This will be overridden by automatic detection
            meta={'version': '1.0.0', 'category': 'api'}
        )
        
        # Make LLM call - feature will be auto-detected as 'api-chat'
        response = openai.chat.completions.create(
            model='gpt-4',
            messages=[
                {'role': 'user', 'content': request_data.message}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': request.headers.get('x-session-id'),
                'environment': 'production',
                'request': request  # Pass request for endpoint detection
            }
        )
        
        return ChatResponse(
            response=response.choices[0].message.content,
            customer_id=request_data.customer_id
        )
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/api/summarize", response_model=SummarizeResponse)
async def summarize_endpoint(request_data: SummarizeRequest, request):
    """Summarization endpoint with automatic feature detection"""
    try:
        # Create customer and feature objects
        customer = Customer(
            id=request_data.customer_id,
            name='API User',
            meta={'source': 'fastapi'}
        )
        
        feature = Feature(
            name='summarization',  # This will be overridden by automatic detection
            meta={'version': '1.0.0', 'category': 'api'}
        )
        
        # Make LLM call - feature will be auto-detected as 'api-summarize'
        response = openai.chat.completions.create(
            model='gpt-3.5-turbo',
            messages=[
                {'role': 'user', 'content': f'Summarize this text: {request_data.text}'}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': request.headers.get('x-session-id'),
                'environment': 'production',
                'request': request  # Pass request for endpoint detection
            }
        )
        
        return SummarizeResponse(
            summary=response.choices[0].message.content,
            customer_id=request_data.customer_id
        )
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/api/translate", response_model=TranslateResponse)
async def translate_endpoint(request_data: TranslateRequest, request):
    """Translation endpoint with automatic feature detection"""
    try:
        # Create customer and feature objects
        customer = Customer(
            id=request_data.customer_id,
            name='API User',
            meta={'source': 'fastapi'}
        )
        
        feature = Feature(
            name='translation',  # This will be overridden by automatic detection
            meta={'version': '1.0.0', 'category': 'api'}
        )
        
        # Make LLM call - feature will be auto-detected as 'api-translate'
        response = openai.chat.completions.create(
            model='gpt-3.5-turbo',
            messages=[
                {'role': 'user', 'content': f'Translate this text to {request_data.target_language}: {request_data.text}'}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': request.headers.get('x-session-id'),
                'environment': 'production',
                'request': request  # Pass request for endpoint detection
            }
        )
        
        return TranslateResponse(
            translation=response.choices[0].message.content,
            customer_id=request_data.customer_id
        )
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@app.get("/health")
async def health_check():
    """Health check endpoint"""
    return {"status": "healthy", "service": "langspend-fastapi-example"}

if __name__ == '__main__':
    import uvicorn
    print('Starting FastAPI app with LangSpend integration...')
    print('Endpoints:')
    print('  POST /api/chat - Chat with automatic feature detection')
    print('  POST /api/summarize - Summarize text with automatic feature detection')
    print('  POST /api/translate - Translate text with automatic feature detection')
    print('  GET /health - Health check')
    print('\nFeatures will be automatically detected from endpoint paths!')
    
    uvicorn.run(app, host="0.0.0.0", port=8000)
