"""
Flask Integration Example

This example shows how to integrate LangSpend with a Flask application
for automatic feature detection based on API endpoints.
"""

import os
from flask import Flask, request, jsonify
from langspend import LangSpend, wrap_openai, Customer, Feature
from openai import OpenAI

# Initialize Flask app
app = Flask(__name__)

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here')
)

# Wrap your OpenAI client
openai = wrap_openai(
    OpenAI(api_key=os.getenv('OPENAI_API_KEY')),
    langspend
)

@app.route('/api/chat', methods=['POST'])
def chat_endpoint():
    """Chat endpoint with automatic feature detection"""
    try:
        data = request.get_json()
        message = data.get('message', '')
        customer_id = data.get('customer_id', 'anonymous')
        customer_name = data.get('customer_name', 'User')
        customer_email = data.get('customer_email')
        
        # Create customer and feature objects
        customer = Customer(
            id=customer_id,
            name=customer_name,
            email=customer_email,
            meta={'source': 'flask_api'}
        )
        
        feature = Feature(
            name='chat-api',  # This will be overridden by automatic detection
            meta={'version': '1.0.0', 'category': 'api'}
        )
        
        # Make LLM call - feature will be auto-detected as 'api-chat'
        response = openai.chat.completions.create(
            model='gpt-4',
            messages=[
                {'role': 'user', 'content': message}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': request.headers.get('X-Session-ID'),
                'environment': 'production',
                'request': request  # Pass request for endpoint detection
            }
        )
        
        return jsonify({
            'response': response.choices[0].message.content,
            'customer_id': customer_id
        })
        
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/summarize', methods=['POST'])
def summarize_endpoint():
    """Summarization endpoint with automatic feature detection"""
    try:
        data = request.get_json()
        text = data.get('text', '')
        customer_id = data.get('customer_id', 'anonymous')
        
        # Create customer and feature objects
        customer = Customer(
            id=customer_id,
            name='API User',
            meta={'source': 'flask_api'}
        )
        
        feature = Feature(
            name='summarization',  # This will be overridden by automatic detection
            meta={'version': '1.0.0', 'category': 'api'}
        )
        
        # Make LLM call - feature will be auto-detected as 'api-summarize'
        response = openai.chat.completions.create(
            model='gpt-3.5-turbo',
            messages=[
                {'role': 'user', 'content': f'Summarize this text: {text}'}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': request.headers.get('X-Session-ID'),
                'environment': 'production',
                'request': request  # Pass request for endpoint detection
            }
        )
        
        return jsonify({
            'summary': response.choices[0].message.content,
            'customer_id': customer_id
        })
        
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/translate', methods=['POST'])
def translate_endpoint():
    """Translation endpoint with automatic feature detection"""
    try:
        data = request.get_json()
        text = data.get('text', '')
        target_language = data.get('target_language', 'Spanish')
        customer_id = data.get('customer_id', 'anonymous')
        
        # Create customer and feature objects
        customer = Customer(
            id=customer_id,
            name='API User',
            meta={'source': 'flask_api'}
        )
        
        feature = Feature(
            name='translation',  # This will be overridden by automatic detection
            meta={'version': '1.0.0', 'category': 'api'}
        )
        
        # Make LLM call - feature will be auto-detected as 'api-translate'
        response = openai.chat.completions.create(
            model='gpt-3.5-turbo',
            messages=[
                {'role': 'user', 'content': f'Translate this text to {target_language}: {text}'}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': request.headers.get('X-Session-ID'),
                'environment': 'production',
                'request': request  # Pass request for endpoint detection
            }
        )
        
        return jsonify({
            'translation': response.choices[0].message.content,
            'customer_id': customer_id
        })
        
    except Exception as e:
        return jsonify({'error': str(e)}), 500

if __name__ == '__main__':
    print('Starting Flask app with LangSpend integration...')
    print('Endpoints:')
    print('  POST /api/chat - Chat with automatic feature detection')
    print('  POST /api/summarize - Summarize text with automatic feature detection')
    print('  POST /api/translate - Translate text with automatic feature detection')
    print('\nFeatures will be automatically detected from endpoint paths!')
    
    app.run(debug=True, port=5000)
