"""
Multi-Provider Example

This example shows how to use LangSpend with multiple LLM providers
in the same application.
"""

import os
import asyncio
from langspend import LangSpend, wrap_openai, wrap_anthropic, wrap_bedrock, Customer, Feature
from openai import OpenAI
from anthropic import Anthropic
import boto3

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here')
)

# Initialize and wrap multiple providers
openai_client = wrap_openai(
    OpenAI(api_key=os.getenv('OPENAI_API_KEY')),
    langspend
)

anthropic_client = wrap_anthropic(
    Anthropic(api_key=os.getenv('ANTHROPIC_API_KEY')),
    langspend
)

# Bedrock client (optional - requires AWS credentials)
bedrock_client = None
if os.getenv('AWS_ACCESS_KEY_ID') and os.getenv('AWS_SECRET_ACCESS_KEY'):
    bedrock_raw = boto3.client(
        'bedrock-runtime',
        region_name=os.getenv('AWS_REGION', 'us-east-1'),
        aws_access_key_id=os.getenv('AWS_ACCESS_KEY_ID'),
        aws_secret_access_key=os.getenv('AWS_SECRET_ACCESS_KEY')
    )
    bedrock_client = wrap_bedrock(bedrock_raw, langspend)

async def compare_providers(customer: Customer, feature: Feature, prompt: str):
    """Compare responses from multiple providers for the same prompt"""
    print(f'\nComparing providers for customer: {customer.name}')
    print(f'Feature: {feature.name}')
    print(f'Prompt: {prompt}')
    print('=' * 60)
    
    # Common tags for all providers
    common_tags = {
        'customer': customer,
        'feature': feature,
        'session_id': 'multi_provider_session',
        'environment': 'production'
    }
    
    # OpenAI response
    print('\n🤖 OpenAI GPT-4:')
    try:
        openai_response = openai_client.chat.completions.create(
            model='gpt-4',
            messages=[{'role': 'user', 'content': prompt}],
            langspend_tags=common_tags
        )
        print(openai_response.choices[0].message.content)
        print('✅ OpenAI response tracked')
    except Exception as e:
        print(f'❌ OpenAI error: {e}')
    
    # Anthropic response
    print('\n🧠 Anthropic Claude:')
    try:
        anthropic_response = anthropic_client.messages.create(
            model='claude-3-5-sonnet-20241022',
            max_tokens=1000,
            messages=[{'role': 'user', 'content': prompt}],
            langspend_tags=common_tags
        )
        print(anthropic_response.content[0].text)
        print('✅ Anthropic response tracked')
    except Exception as e:
        print(f'❌ Anthropic error: {e}')
    
    # Bedrock response (if available)
    if bedrock_client:
        print('\n☁️ AWS Bedrock Claude:')
        try:
            import json
            response = bedrock_client.invoke_model(
                modelId='anthropic.claude-3-5-sonnet-20241022-v2:0',
                body=json.dumps({
                    'anthropic_version': 'bedrock-2023-05-31',
                    'max_tokens': 1000,
                    'messages': [{'role': 'user', 'content': prompt}]
                }),
                langspend_tags=common_tags
            )
            response_body = json.loads(response['body'].read())
            print(response_body['content'][0]['text'])
            print('✅ Bedrock response tracked')
        except Exception as e:
            print(f'❌ Bedrock error: {e}')
    else:
        print('\n☁️ AWS Bedrock: Not configured (missing AWS credentials)')
    
    print('=' * 60)

async def use_different_providers_for_different_tasks(customer: Customer):
    """Use different providers for different types of tasks"""
    print(f'\nUsing different providers for different tasks - Customer: {customer.name}')
    
    # Use OpenAI for creative writing
    print('\n📝 Creative Writing (OpenAI):')
    creative_response = openai_client.chat.completions.create(
        model='gpt-4',
        messages=[{'role': 'user', 'content': 'Write a creative short story about a time-traveling detective'}],
        langspend_tags={
            'customer': customer,
            'feature': Feature(name='creative-writing', meta={'provider': 'openai', 'category': 'content'}),
            'session_id': 'multi_task_session',
            'environment': 'production'
        }
    )
    print(creative_response.choices[0].message.content[:200] + '...')
    
    # Use Anthropic for code analysis
    print('\n💻 Code Analysis (Anthropic):')
    code_response = anthropic_client.messages.create(
        model='claude-3-5-sonnet-20241022',
        max_tokens=1000,
        messages=[{'role': 'user', 'content': 'Analyze this Python code for potential bugs and suggest improvements:\n\ndef process_data(data):\n    result = []\n    for item in data:\n        if item > 0:\n            result.append(item * 2)\n    return result'}],
        langspend_tags={
            'customer': customer,
            'feature': Feature(name='code-analysis', meta={'provider': 'anthropic', 'category': 'development'}),
            'session_id': 'multi_task_session',
            'environment': 'production'
        }
    )
    print(code_response.content[0].text[:200] + '...')
    
    # Use Bedrock for data processing (if available)
    if bedrock_client:
        print('\n📊 Data Processing (Bedrock):')
        import json
        bedrock_response = bedrock_client.invoke_model(
            modelId='anthropic.claude-3-5-sonnet-20241022-v2:0',
            body=json.dumps({
                'anthropic_version': 'bedrock-2023-05-31',
                'max_tokens': 1000,
                'messages': [{'role': 'user', 'content': 'Explain the concept of data normalization in database design'}]
            }),
            langspend_tags={
                'customer': customer,
                'feature': Feature(name='data-processing', meta={'provider': 'aws-bedrock', 'category': 'analytics'}),
                'session_id': 'multi_task_session',
                'environment': 'production'
            }
        )
        response_body = json.loads(bedrock_response['body'].read())
        print(response_body['content'][0]['text'][:200] + '...')
    
    print('\n✅ All tasks completed with different providers!')

async def main():
    # Example 1: Compare providers for the same task
    await compare_providers(
        Customer(
            id='user_multi_001',
            name='Provider Comparison User',
            email='compare@example.com',
            meta={'plan': 'premium', 'use_case': 'provider_comparison'}
        ),
        Feature(
            name='provider-comparison',
            meta={'version': '1.0.0', 'category': 'evaluation'}
        ),
        'Explain the benefits of renewable energy in simple terms'
    )
    
    # Example 2: Use different providers for different tasks
    await use_different_providers_for_different_tasks(
        Customer(
            id='user_multi_002',
            name='Multi-Task User',
            email='multitask@example.com',
            meta={'plan': 'enterprise', 'use_case': 'multi_provider_workflow'}
        )
    )
    
    print('\n✅ All multi-provider requests tracked with customer attribution!')

if __name__ == '__main__':
    asyncio.run(main())
