# LangSpend Python SDK - Next.js Demo

A complete demo application showing how to use the LangSpend Python SDK with a Next.js frontend and FastAPI backend.

## Quick Start

### 1. Setup Backend

```bash
cd backend

# Create virtual environment
python -m venv venv
source venv/bin/activate  # On Windows: venv\Scripts\activate

# Install dependencies
pip install -r requirements.txt

# Set up environment variables
cp env.example .env
# Edit .env with your API keys

# Start the backend server
python main.py
```

The backend will start on `http://localhost:8000`

### 2. Setup Frontend

```bash
# Install dependencies
npm install

# Start the development server
npm run dev
```

The frontend will start on `http://localhost:3000`

## LangSpend Integration

The backend demonstrates several LangSpend Python SDK features:

### Customer level spend tracking


```python
customer = Customer(
    id=request.customer_id,
    name=request.customer_name,
    email=request.customer_email,
    meta={
        'source': 'nextjs_demo',
        'platform': 'web'
    }
)
```

### Track feature level spending

```python
feature = Feature(
    name='chat-demo',
    meta={
        'version': '1.0.0',
        'category': 'demo',
        'endpoint': '/api/chat'
    }
)
```

### Tracking

```python
response = openai_client.chat.completions.create(
    model='gpt-3.5-turbo',
    messages=[{'role': 'user', 'content': request.message}],
    langspend_tags={
        'customer': customer,
        'feature': feature,
        'session_id': f'demo_session_{request.customer_id}',
        'environment': 'demo'
    }
)
```

## Development

### Backend Development

```bash
cd backend
source venv/bin/activate
python main.py
```

### Frontend Development

```bash
npm run dev
```

## Troubleshooting

### Common Issues

1. **CORS Errors**: Make sure the backend is running on port 8000
2. **API Key Errors**: Check your `.env` file in the backend directory
3. **Import Errors**: Make sure you've installed all dependencies

### Debug Mode

The backend runs with debug mode enabled by default. Check the console for detailed tracking information.

### Health Check

Visit `http://localhost:8000/health` to verify the backend is running correctly.

## Customization
### Changing Models

Update the `model` parameter in the OpenAI API calls:

```python
response = openai_client.chat.completions.create(
    model='gpt-4',  # Change this
    messages=[...],
    langspend_tags={...}
)
```

### Custom Features

Add new features by creating additional `Feature` objects with custom metadata:

```python
feature = Feature(
    name='custom-feature',
    meta={
        'version': '1.0.0',
        'category': 'custom',
        'custom_field': 'value'
    }
)
```

## Support

For questions or issues:
- Check the [LangSpend Python SDK documentation](../../README.md)
- Open an issue on [GitHub](https://github.com/langspend/langspend-python)
- Contact support at support@langspend.com
