"""
FastAPI backend for LangSpend Python SDK demo
"""

import os
import asyncio
from fastapi import FastAPI, HTTPException
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
from langspend import LangSpend, wrap_openai, Customer, Feature
from openai import OpenAI
from dotenv import load_dotenv

# Load environment variables from .env file
load_dotenv()

# Initialize FastAPI app
app = FastAPI(
    title="LangSpend Python SDK Demo API",
    description="Backend API demonstrating LangSpend Python SDK integration",
    version="1.0.0"
)

# Add CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["http://localhost:3000"],  # Next.js dev server
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here'),
    base_url=os.getenv('LANGSPEND_BASE_URL', 'https://platform.langspend.com'),
    debug=True  # Enable debug mode for demo
)

print(f"LangSpend initialized with API key: {os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here')[:10]}...")

# Initialize and wrap OpenAI client
openai_client = wrap_openai(
    OpenAI(api_key=os.getenv('OPENAI_API_KEY')),
    langspend
)

# Pydantic models
class ChatRequest(BaseModel):
    message: str
    customer: str = "alice"
    model: str = "gpt-3.5-turbo"

class ChatResponse(BaseModel):
    message: str
    cost: float = 0.0
    tokens: int = 0

class HealthResponse(BaseModel):
    status: str
    service: str
    langspend_configured: bool

def calculate_openai_cost(model: str, input_tokens: int, output_tokens: int) -> float:
    """Calculate OpenAI cost based on model and tokens"""
    pricing = {
        'gpt-4': {'input': 0.03, 'output': 0.06},
        'gpt-3.5-turbo': {'input': 0.0005, 'output': 0.0015},
    }
    price = pricing.get(model, pricing['gpt-3.5-turbo'])
    return (input_tokens / 1000) * price['input'] + (output_tokens / 1000) * price['output']

@app.get("/health", response_model=HealthResponse)
async def health_check():
    """Health check endpoint"""
    return HealthResponse(
        status="healthy",
        service="langspend-python-demo-api",
        langspend_configured=bool(os.getenv('LANGSPEND_API_KEY'))
    )

@app.post("/api/chat", response_model=ChatResponse)
async def chat_endpoint(request: ChatRequest):
    """Chat endpoint with LangSpend tracking"""
    try:
        # Customer email mapping
        customer_emails = {
            'alice': 'alice@example.com',
            'bob': 'bob@example.com',
            'company': 'admin@company.com',
        }
        
        # Create customer and feature objects
        customer = Customer(
            id=request.customer,
            name=request.customer.title(),
            email=customer_emails.get(request.customer, 'unknown@example.com'),
            meta={
                'source': 'nextjs_demo',
                'platform': 'web'
            }
        )
        
        feature = Feature(
            name='chat',
            meta={
                'version': '1.0.0',
                'category': 'demo',
                'endpoint': '/api/chat'
            }
        )
        
        # Make LLM call with LangSpend tracking
        print(f"Making LLM call with customer: {customer.id}, feature: {feature.name}")
        response = openai_client.chat.completions.create(
            model=request.model,
            messages=[
                {'role': 'user', 'content': request.message}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': f'demo_session_{request.customer}',
                'environment': 'demo'
            }
        )
        print(f"LLM call completed, response received")
        
        # Calculate cost
        usage = response.usage
        cost = calculate_openai_cost(request.model, usage.prompt_tokens, usage.completion_tokens)
        
        return ChatResponse(
            message=response.choices[0].message.content,
            cost=cost,
            tokens=usage.prompt_tokens + usage.completion_tokens
        )
        
    except Exception as e:
        print(f"Error in chat endpoint: {e}")
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/api/summarize", response_model=ChatResponse)
async def summarize_endpoint(request: ChatRequest):
    """Summarization endpoint with LangSpend tracking"""
    try:
        # Customer email mapping
        customer_emails = {
            'alice': 'alice@example.com',
            'bob': 'bob@example.com',
            'company': 'admin@company.com',
        }
        
        # Create customer and feature objects
        customer = Customer(
            id=request.customer,
            name=request.customer.title(),
            email=customer_emails.get(request.customer, 'unknown@example.com'),
            meta={
                'source': 'nextjs_demo',
                'platform': 'web'
            }
        )
        
        feature = Feature(
            name='summarize',
            meta={
                'version': '1.0.0',
                'category': 'demo',
                'endpoint': '/api/summarize'
            }
        )
        
        # Make LLM call with LangSpend tracking
        response = openai_client.chat.completions.create(
            model=request.model,
            messages=[
                {'role': 'system', 'content': 'You are a helpful assistant that creates concise, accurate summaries.'},
                {'role': 'user', 'content': f'Summarize the following:\n\n{request.message}'}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': f'demo_session_{request.customer}',
                'environment': 'demo'
            }
        )
        
        # Calculate cost
        usage = response.usage
        cost = calculate_openai_cost(request.model, usage.prompt_tokens, usage.completion_tokens)
        
        return ChatResponse(
            message=response.choices[0].message.content,
            cost=cost,
            tokens=usage.prompt_tokens + usage.completion_tokens
        )
        
    except Exception as e:
        print(f"Error in summarize endpoint: {e}")
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/api/code", response_model=ChatResponse)
async def code_endpoint(request: ChatRequest):
    """Code generation endpoint with LangSpend tracking"""
    try:
        # Customer email mapping
        customer_emails = {
            'alice': 'alice@example.com',
            'bob': 'bob@example.com',
            'company': 'admin@company.com',
        }
        
        # Create customer and feature objects
        customer = Customer(
            id=request.customer,
            name=request.customer.title(),
            email=customer_emails.get(request.customer, 'unknown@example.com'),
            meta={
                'source': 'nextjs_demo',
                'platform': 'web'
            }
        )
        
        feature = Feature(
            name='code',
            meta={
                'version': '1.0.0',
                'category': 'demo',
                'endpoint': '/api/code'
            }
        )
        
        # Make LLM call with LangSpend tracking
        response = openai_client.chat.completions.create(
            model=request.model,
            messages=[
                {'role': 'system', 'content': 'You are a helpful coding assistant. Provide clear, concise code examples with explanations.'},
                {'role': 'user', 'content': request.message}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': f'demo_session_{request.customer}',
                'environment': 'demo'
            }
        )
        
        # Calculate cost
        usage = response.usage
        cost = calculate_openai_cost(request.model, usage.prompt_tokens, usage.completion_tokens)
        
        return ChatResponse(
            message=response.choices[0].message.content,
            cost=cost,
            tokens=usage.prompt_tokens + usage.completion_tokens
        )
        
    except Exception as e:
        print(f"Error in code endpoint: {e}")
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/api/translate", response_model=ChatResponse)
async def translate_endpoint(request: ChatRequest):
    """Translation endpoint with LangSpend tracking"""
    try:
        # Customer email mapping
        customer_emails = {
            'alice': 'alice@example.com',
            'bob': 'bob@example.com',
            'company': 'admin@company.com',
        }
        
        # Create customer and feature objects
        customer = Customer(
            id=request.customer,
            name=request.customer.title(),
            email=customer_emails.get(request.customer, 'unknown@example.com'),
            meta={
                'source': 'nextjs_demo',
                'platform': 'web'
            }
        )
        
        feature = Feature(
            name='translate',
            meta={
                'version': '1.0.0',
                'category': 'demo',
                'endpoint': '/api/translate'
            }
        )
        
        # Make LLM call with LangSpend tracking
        response = openai_client.chat.completions.create(
            model=request.model,
            messages=[
                {'role': 'user', 'content': f'Translate this text to Spanish: {request.message}'}
            ],
            langspend_tags={
                'customer': customer,
                'feature': feature,
                'session_id': f'demo_session_{request.customer}',
                'environment': 'demo'
            }
        )
        
        # Calculate cost
        usage = response.usage
        cost = calculate_openai_cost(request.model, usage.prompt_tokens, usage.completion_tokens)
        
        return ChatResponse(
            message=response.choices[0].message.content,
            cost=cost,
            tokens=usage.prompt_tokens + usage.completion_tokens
        )
        
    except Exception as e:
        print(f"Error in translate endpoint: {e}")
        raise HTTPException(status_code=500, detail=str(e))

if __name__ == "__main__":
    import uvicorn
    print("Starting LangSpend Python SDK Demo API...")
    print("Endpoints:")
    print("  GET  /health - Health check")
    print("  POST /api/chat - Chat with AI")
    print("  POST /api/code - Code generation")
    print("  POST /api/summarize - Summarize text")
    print("  POST /api/translate - Translate text")
    print("\nAll requests are tracked with LangSpend Python SDK!")
    
    uvicorn.run(app, host="0.0.0.0", port=8000)
