"""
Streaming Example

This example shows how to use LangSpend with streaming LLM responses.
"""

import os
import asyncio
from langspend import LangSpend, wrap_openai, Customer, Feature
from openai import OpenAI

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key_here')
)

# Wrap your OpenAI client
openai = wrap_openai(
    OpenAI(api_key=os.getenv('OPENAI_API_KEY')),
    langspend
)

async def stream_chat_response(customer: Customer, feature: Feature, message: str):
    """Stream a chat response with LangSpend tracking"""
    print(f'\nStreaming chat for customer: {customer.name}')
    print(f'Feature: {feature.name}')
    print('Response: ', end='', flush=True)
    
    # Create streaming response
    stream = openai.chat.completions.create(
        model='gpt-4',
        messages=[
            {'role': 'user', 'content': message}
        ],
        stream=True,
        langspend_tags={
            'customer': customer,
            'feature': feature,
            'session_id': 'streaming_session_123',
            'environment': 'production'
        }
    )
    
    # Process the stream
    full_response = ""
    for chunk in stream:
        if chunk.choices[0].delta.content is not None:
            content = chunk.choices[0].delta.content
            print(content, end='', flush=True)
            full_response += content
    
    print('\n')  # New line after streaming
    print(f'✅ Streamed response tracked for customer: {customer.id}')
    
    return full_response

async def stream_long_form_content(customer: Customer, feature: Feature, prompt: str):
    """Stream long-form content generation"""
    print(f'\nGenerating long-form content for customer: {customer.name}')
    print(f'Feature: {feature.name}')
    print('Content: ', end='', flush=True)
    
    # Create streaming response for long-form content
    stream = openai.chat.completions.create(
        model='gpt-4',
        messages=[
            {'role': 'user', 'content': prompt}
        ],
        stream=True,
        max_tokens=2000,  # Longer response
        langspend_tags={
            'customer': customer,
            'feature': feature,
            'session_id': 'streaming_session_456',
            'environment': 'production'
        }
    )
    
    # Process the stream
    full_content = ""
    chunk_count = 0
    for chunk in stream:
        if chunk.choices[0].delta.content is not None:
            content = chunk.choices[0].delta.content
            print(content, end='', flush=True)
            full_content += content
            chunk_count += 1
    
    print('\n')  # New line after streaming
    print(f'✅ Generated {chunk_count} chunks for customer: {customer.id}')
    
    return full_content

async def main():
    # Example 1: Streaming chat response
    await stream_chat_response(
        Customer(
            id='user_stream_001',
            name='Streaming User',
            email='stream@example.com',
            meta={'plan': 'premium', 'preference': 'streaming'}
        ),
        Feature(
            name='streaming-chat',
            meta={'version': '1.0.0', 'category': 'interactive'}
        ),
        'Tell me a short story about a robot learning to paint'
    )
    
    # Example 2: Streaming long-form content
    await stream_long_form_content(
        Customer(
            id='user_stream_002',
            name='Content Creator',
            email='creator@example.com',
            meta={'plan': 'enterprise', 'role': 'content_creator'}
        ),
        Feature(
            name='content-generation',
            meta={'version': '2.0.0', 'category': 'content'}
        ),
        'Write a comprehensive guide on machine learning for beginners. Include sections on supervised learning, unsupervised learning, and neural networks.'
    )
    
    # Example 3: Streaming with different customer
    await stream_chat_response(
        Customer(
            id='user_stream_003',
            name='Quick User',
            email='quick@example.com',
            meta={'plan': 'basic', 'preference': 'fast_responses'}
        ),
        Feature(
            name='quick-chat',
            meta={'version': '1.0.0', 'category': 'interactive'}
        ),
        'Explain quantum computing in one paragraph'
    )
    
    print('\n✅ All streaming requests tracked with customer attribution!')

if __name__ == '__main__':
    asyncio.run(main())
