"""
Example: Using LangSpend with Google Vertex AI

This example shows how to wrap the Google Vertex AI client to automatically
track costs, tokens, and customer attribution for Claude models on Vertex AI.
"""

import os
import asyncio
from langspend import LangSpend, wrap_vertex, Customer, Feature
from google.cloud import vertexai

# Initialize Vertex AI client
vertex = vertexai.VertexAI(
    project=os.getenv('GOOGLE_CLOUD_PROJECT', 'your-project-id'),
    location=os.getenv('GOOGLE_CLOUD_LOCATION', 'us-central1')
)

# Initialize LangSpend
langspend = LangSpend(
    api_key=os.getenv('LANGSPEND_API_KEY', 'lsp_your_api_key'),
    debug=True  # Enable debug logging
)

# Wrap the Vertex AI client
wrapped = wrap_vertex(vertex, langspend)

# Example 1: Basic usage with Claude 3.5 Sonnet
async def basic_example():
    print('\n=== Example 1: Basic Usage (Claude 3.5 Sonnet) ===')
    
    model = wrapped.get_generative_model(model='claude-3-5-sonnet-20241022@latest')
    
    result = model.generate_content(
        contents=[{
            'role': 'user',
            'parts': [{
                'text': 'Explain Google Cloud Platform in one sentence.'
            }]
        }],
        langspend_tags={
            'customer': Customer(
                id='user_123',
                name='John Doe',
                email='john@example.com'
            ),
            'feature': Feature(name='gcp-explanation')
        }
    )
    
    print('Response:', result.text)
    print('Usage:', result.response_metadata.usage_metadata)
    print('✅ Request tracked automatically!')

# Example 2: With customer attribution
async def customer_attribution_example():
    print('\n=== Example 2: Customer Attribution ===')
    
    model = wrapped.get_generative_model(model='claude-3-5-haiku-20241022@latest')  # Fastest & cheapest
    
    result = model.generate_content(
        contents=[{
            'role': 'user',
            'parts': [{
                'text': 'What is Google Cloud Storage?'
            }]
        }],
        langspend_tags={
            'customer': Customer(
                id='enterprise_customer_001',
                name='Acme Corp',
                email='admin@acme.com'
            ),
            'feature': Feature(name='cloud-storage-query'),
            'session_id': 'session_vertex_001',
            'environment': 'production'
        }
    )
    
    print('Response:', result.text)
    print('✅ Cost attributed to customer: enterprise_customer_001')

# Example 3: With feature attribution
async def feature_attribution_example():
    print('\n=== Example 3: Feature Attribution ===')
    
    model = wrapped.get_generative_model(model='claude-3-5-sonnet-20241022@latest')
    
    result = model.generate_content(
        contents=[{
            'role': 'user',
            'parts': [{
                'text': 'Generate a Terraform configuration for a Google Cloud Run service.'
            }]
        }],
        langspend_tags={
            'customer': Customer(
                id='dev_team_456',
                name='Development Team',
                email='dev@company.com'
            ),
            'feature': Feature(
                name='terraform-generator',
                meta={
                    'cloud_provider': 'gcp',
                    'service_type': 'cloud-run'
                }
            ),
            'session_id': 'session_terraform_001',
            'environment': 'development'
        }
    )
    
    print('Response length:', len(result.text), 'characters')
    print('✅ Cost attributed to feature: terraform-generator')

# Example 4: Streaming response
async def streaming_example():
    print('\n=== Example 4: Streaming Response ===')
    
    model = wrapped.get_generative_model(model='claude-3-5-sonnet-20241022@latest')
    
    result = model.generate_content_stream(
        contents=[{
            'role': 'user',
            'parts': [{
                'text': 'Write a short poem about cloud computing.'
            }]
        }],
        langspend_tags={
            'customer': Customer(
                id='poet_789',
                name='Cloud Poet'
            ),
            'feature': Feature(name='poetry-generator')
        }
    )
    
    print('Streaming response:')
    for chunk in result:
        if chunk.text:
            print(chunk.text, end='')
    print('\n✅ Streaming response tracked!')

# Example 5: Multi-turn conversation
async def conversation_example():
    print('\n=== Example 5: Multi-Turn Conversation ===')
    
    model = wrapped.get_generative_model(model='claude-3-5-sonnet-20241022@latest')
    
    conversation_history = []
    
    # Turn 1
    result1 = model.generate_content(
        contents=[{
            'role': 'user',
            'parts': [{
                'text': 'I want to deploy a containerized app on Google Cloud. What service should I use?'
            }]
        }],
        langspend_tags={
            'customer': Customer(
                id='architect_001',
                name='Cloud Architect'
            ),
            'session_id': 'conv_vertex_001'
        }
    )
    
    response1 = result1.text
    conversation_history.append({
        'role': 'user',
        'parts': [{'text': 'I want to deploy a containerized app on Google Cloud. What service should I use?'}]
    })
    conversation_history.append({
        'role': 'model',
        'parts': [{'text': response1}]
    })
    
    print('Turn 1:', response1[:100] + '...')
    
    # Turn 2
    result2 = model.generate_content(
        contents=[
            *conversation_history,
            {
                'role': 'user',
                'parts': [{
                    'text': 'What about serverless containers?'
                }]
            }
        ],
        langspend_tags={
            'customer': Customer(
                id='architect_001',
                name='Cloud Architect'
            ),
            'session_id': 'conv_vertex_001'  # Same session
        }
    )
    
    response2 = result2.text
    print('Turn 2:', response2[:100] + '...')
    print('✅ Both turns tracked with same session ID!')

# Example 6: Comparing models (Haiku vs Sonnet)
async def model_comparison_example():
    print('\n=== Example 6: Model Comparison (Haiku vs Sonnet) ===')
    
    prompt = 'Explain Google Cloud Functions in simple terms.'
    
    # Haiku: Fastest & cheapest
    import time
    haiku_start = time.time()
    haiku_model = wrapped.get_generative_model(model='claude-3-5-haiku-20241022@latest')
    haiku_result = haiku_model.generate_content(
        contents=[{
            'role': 'user',
            'parts': [{'text': prompt}]
        }],
        langspend_tags={
            'feature': Feature(name='model-comparison-vertex')
        }
    )
    haiku_time = (time.time() - haiku_start) * 1000
    haiku_usage = haiku_result.response_metadata.usage_metadata
    
    # Sonnet: Balanced
    sonnet_start = time.time()
    sonnet_model = wrapped.get_generative_model(model='claude-3-5-sonnet-20241022@latest')
    sonnet_result = sonnet_model.generate_content(
        contents=[{
            'role': 'user',
            'parts': [{'text': prompt}]
        }],
        langspend_tags={
            'feature': Feature(name='model-comparison-vertex')
        }
    )
    sonnet_time = (time.time() - sonnet_start) * 1000
    sonnet_usage = sonnet_result.response_metadata.usage_metadata
    
    print(f'Haiku: {haiku_time:.0f}ms, {(haiku_usage.prompt_token_count or 0) + (haiku_usage.candidates_token_count or 0)} tokens')
    print(f'Sonnet: {sonnet_time:.0f}ms, {(sonnet_usage.prompt_token_count or 0) + (sonnet_usage.candidates_token_count or 0)} tokens')
    print('✅ Check dashboard to compare costs!')

# Run all examples
async def main():
    try:
        await basic_example()
        await customer_attribution_example()
        await feature_attribution_example()
        await streaming_example()
        await conversation_example()
        await model_comparison_example()
        
        print('\n✅ All examples completed! Check your dashboard at https://app.langspend.com')
    except Exception as error:
        print(f'Error: {error}')
        import traceback
        traceback.print_exc()

# Run if executed directly
if __name__ == '__main__':
    asyncio.run(main())
