#!/usr/bin/env python3
"""
LangSpend CLI - Command line interface for LangSpend Python SDK
"""

import argparse
import sys
from typing import Optional

from . import __version__


def main() -> None:
    """Main CLI entry point"""
    parser = argparse.ArgumentParser(
        prog="langspend",
        description="LangSpend Python SDK CLI - Track LLM costs and usage",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  langspend --version                    # Show version
  langspend --help                      # Show this help
  langspend test                        # Test SDK functionality
        """,
    )
    
    parser.add_argument(
        "--version",
        action="version",
        version=f"%(prog)s {__version__}",
        help="Show version and exit",
    )
    
    subparsers = parser.add_subparsers(dest="command", help="Available commands")
    
    # Test command
    test_parser = subparsers.add_parser(
        "test",
        help="Test LangSpend SDK functionality",
    )
    test_parser.add_argument(
        "--api-key",
        type=str,
        help="LangSpend API key for testing (or set LANGSPEND_API_KEY env var)",
    )
    
    args = parser.parse_args()
    
    if args.command == "test":
        test_sdk(args.api_key)
    else:
        parser.print_help()


def test_sdk(api_key: Optional[str] = None) -> None:
    """Test the LangSpend SDK functionality"""
    import os
    from .client import LangSpend
    from .types import Customer, Feature, TrackingData
    
    print("🧪 Testing LangSpend Python SDK...")
    
    # Use provided API key or environment variable
    key = api_key or os.getenv("LANGSPEND_API_KEY", "test_key")
    
    try:
        # Initialize LangSpend client
        langspend = LangSpend(api_key=key, debug=True)
        print("✅ LangSpend client initialized successfully")
        
        # Create test objects
        customer = Customer(
            id="cli_test_user",
            name="CLI Test User",
            email="test@example.com",
            meta={"source": "cli_test"}
        )
        
        feature = Feature(
            name="cli-test",
            meta={"version": "1.0.0", "category": "test"}
        )
        
        print("✅ Customer and feature objects created successfully")
        
        # Create tracking data
        tracking_data = TrackingData(
            provider="openai",
            model="gpt-3.5-turbo",
            input_tokens=10,
            output_tokens=5,
            cached_input_tokens=0,
            metadata={"test": True, "source": "cli"},
            tags={
                "customer": customer,
                "feature": feature,
                "session_id": "cli_test_session",
                "environment": "test"
            },
            latency=0.1
        )
        
        print("✅ Tracking data created successfully")
        
        # Test sync tracking
        result = langspend.track_sync(tracking_data)
        if result.success:
            print("✅ Sync tracking successful")
        else:
            print(f"⚠️  Sync tracking failed (expected with test key): {result.error}")
        
        print("\n🎉 SDK test completed successfully!")
        print("\nTo test with real tracking:")
        print("1. Set LANGSPEND_API_KEY environment variable")
        print("2. Run: langspend test --api-key your_api_key")
        print("3. Check your LangSpend dashboard for tracked requests")
        
    except Exception as e:
        print(f"❌ Test failed: {e}")
        sys.exit(1)


if __name__ == "__main__":
    main()
