"""
LangSpend client for tracking LLM usage
"""

import time
import json
import logging
from datetime import datetime
from typing import Optional
import requests
from .types import LangSpendConfig, TrackingData, TrackResponse


class LangSpend:
    """
    LangSpend client for tracking LLM usage and costs.
    
    Example:
        >>> langspend = LangSpend(api_key="lsp_your_api_key")
        >>> response = await langspend.track(tracking_data)
    """
    
    def __init__(self, api_key: str, **kwargs):
        """
        Initialize LangSpend client.
        
        Args:
            api_key: Your LangSpend API key
            **kwargs: Additional configuration options
        """
        if not api_key:
            raise ValueError("LangSpend API key is required")
            
        self.config = LangSpendConfig(
            api_key=api_key,
            **kwargs
        )
        
        self.logger = logging.getLogger(__name__)
        if self.config.debug:
            self.logger.setLevel(logging.DEBUG)
    
    async def track(self, data: TrackingData) -> TrackResponse:
        """
        Track an LLM request asynchronously.
        
        Args:
            data: Tracking data from the LLM call
            
        Returns:
            TrackResponse indicating success/failure
        """
        return await self._track_with_retry(data)
    
    def track_sync(self, data: TrackingData) -> TrackResponse:
        """
        Track an LLM request synchronously.
        
        Args:
            data: Tracking data from the LLM call
            
        Returns:
            TrackResponse indicating success/failure
        """
        return self._track_with_retry_sync(data)
    
    async def _track_with_retry(self, data: TrackingData) -> TrackResponse:
        """Track with retry logic (async version)"""
        import asyncio
        
        for attempt in range(self.config.max_retries):
            try:
                # Convert tracking data to JSON
                payload = self._prepare_payload(data)
                
                # Make async request using httpx (more reliable than aiohttp)
                try:
                    import httpx
                except ImportError:
                    # Fallback to sync requests in async context
                    return self._track_with_retry_sync(data)
                
                async with httpx.AsyncClient(timeout=self.config.timeout) as client:
                    response = await client.post(
                        f"{self.config.base_url}/api/track",
                        headers={
                            "Authorization": f"Bearer {self.config.api_key}",
                            "Content-Type": "application/json",
                        },
                        json=payload
                    )
                    
                    if response.status_code == 200:
                        result = response.json()
                        return TrackResponse(
                            success=True,
                            cost=result.get("cost")
                        )
                    else:
                        try:
                            error_data = response.json()
                            error_msg = error_data.get("error", f"HTTP {response.status_code}")
                        except:
                            error_msg = f"HTTP {response.status_code}"
                        
                        if self.config.debug:
                            self.logger.error(f"Tracking failed (attempt {attempt + 1}/{self.config.max_retries}): {error_msg}")
                        
                        # Don't retry on client errors (400-499)
                        if 400 <= response.status_code < 500:
                            return TrackResponse(success=False, error=error_msg)
                        
                        # Server error, retry
                        if attempt < self.config.max_retries - 1:
                            backoff_ms = (2 ** attempt) * 1000  # 1s, 2s, 4s
                            await asyncio.sleep(backoff_ms / 1000)
                            continue
                        
                        return TrackResponse(success=False, error=error_msg)
                        
            except Exception as e:
                if self.config.debug:
                    self.logger.error(f"Tracking failed (attempt {attempt + 1}/{self.config.max_retries}): {e}")
                
                # Retry on network errors
                if attempt < self.config.max_retries - 1:
                    backoff_ms = (2 ** attempt) * 1000
                    await asyncio.sleep(backoff_ms / 1000)
                    continue
                
                return TrackResponse(
                    success=False,
                    error=str(e) if isinstance(e, Exception) else "Unknown error"
                )
        
        return TrackResponse(success=False, error="Max retries exceeded")
    
    def _track_with_retry_sync(self, data: TrackingData) -> TrackResponse:
        """Track with retry logic (sync version)"""
        for attempt in range(self.config.max_retries):
            try:
                # Convert tracking data to JSON
                payload = self._prepare_payload(data)
                
                if self.config.debug:
                    print(f"LangSpend payload: {payload}")
                
                # Make sync request
                response = requests.post(
                    f"{self.config.base_url}/api/track",
                    headers={
                        "Authorization": f"Bearer {self.config.api_key}",
                        "Content-Type": "application/json",
                    },
                    json=payload,
                    timeout=self.config.timeout
                )
                
                if response.status_code == 200:
                    result = response.json()
                    return TrackResponse(
                        success=True,
                        cost=result.get("cost")
                    )
                else:
                    try:
                        error_data = response.json()
                        error_msg = error_data.get("error", f"HTTP {response.status_code}")
                    except:
                        error_msg = f"HTTP {response.status_code}"
                    
                    if self.config.debug:
                        self.logger.error(f"Tracking failed (attempt {attempt + 1}/{self.config.max_retries}): {error_msg}")
                    
                    # Don't retry on client errors (400-499)
                    if 400 <= response.status_code < 500:
                        return TrackResponse(success=False, error=error_msg)
                    
                    # Server error, retry
                    if attempt < self.config.max_retries - 1:
                        backoff_ms = (2 ** attempt) * 1000  # 1s, 2s, 4s
                        time.sleep(backoff_ms / 1000)
                        continue
                    
                    return TrackResponse(success=False, error=error_msg)
                    
            except Exception as e:
                if self.config.debug:
                    self.logger.error(f"Tracking failed (attempt {attempt + 1}/{self.config.max_retries}): {e}")
                
                # Retry on network errors
                if attempt < self.config.max_retries - 1:
                    backoff_ms = (2 ** attempt) * 1000
                    time.sleep(backoff_ms / 1000)
                    continue
                
                return TrackResponse(
                    success=False,
                    error=str(e) if isinstance(e, Exception) else "Unknown error"
                )
        
        return TrackResponse(success=False, error="Max retries exceeded")
    
    def _prepare_payload(self, data: TrackingData) -> dict:
        """Prepare tracking data for API payload - matches Node.js structure"""
        # Use the same structure as Node.js - send data directly
        payload = {
            "provider": data.provider,
            "model": data.model,
            "input_tokens": data.input_tokens,
            "output_tokens": data.output_tokens,
            "cached_input_tokens": data.cached_input_tokens,
            "metadata": data.metadata or {},
            "tags": data.tags.__dict__ if data.tags else {},
            "timestamp": data.timestamp or datetime.utcnow().isoformat() + 'Z',
            "latency": data.latency,
        }
        
        return payload
