"""
Call site detection for automatic feature naming
"""

import inspect
import sys
from typing import Optional
from ..types import CallSite


def detect_callsite() -> Optional[CallSite]:
    """
    Detect the calling function and file from the call stack.
    
    Returns:
        CallSite object with function name, file, and line number
    """
    try:
        # Get the current frame and walk up the stack
        frame = inspect.currentframe()
        if not frame:
            return None
        
        # Walk up the stack to find the first non-LangSpend function
        while frame:
            frame = frame.f_back
            if not frame:
                break
            
            # Get frame info
            frame_info = inspect.getframeinfo(frame)
            filename = frame_info.filename
            function_name = frame_info.function
            line_number = frame_info.lineno
            
            # Skip LangSpend internal functions
            if 'langspend' not in filename.lower():
                return CallSite(
                    function_name=function_name,
                    file=filename,
                    line=line_number
                )
        
    except Exception:
        # Graceful failure - never break the application
        pass
    
    return None
