"""
Endpoint detection for Flask and FastAPI applications
"""

import re
from typing import Optional, Any


def detect_endpoint(request: Optional[Any] = None) -> Optional[str]:
    """
    Detect the current HTTP endpoint from request context.
    
    Supports Flask and FastAPI request objects.
    
    Args:
        request: Flask or FastAPI request object
        
    Returns:
        Endpoint path if detected, None otherwise
    """
    if not request:
        return None
    
    try:
        # Try Flask request object
        if hasattr(request, 'endpoint') and hasattr(request, 'url_rule'):
            if request.endpoint and request.url_rule:
                return request.url_rule.rule
            elif hasattr(request, 'path'):
                return request.path
        
        # Try FastAPI request object
        if hasattr(request, 'url') and hasattr(request, 'scope'):
            if hasattr(request.url, 'path'):
                return request.url.path
        
        # Try generic request object with path attribute
        if hasattr(request, 'path'):
            return request.path
        
        # Try to extract from URL
        if hasattr(request, 'url'):
            url = str(request.url)
            # Extract path from URL
            match = re.search(r'https?://[^/]+(/[^?]*)', url)
            if match:
                return match.group(1)
        
    except Exception:
        # Graceful failure - never break the application
        pass
    
    return None
