"""
Type definitions for LangSpend Python SDK
"""

from typing import Dict, Any, Optional, Union
from dataclasses import dataclass


@dataclass
class Customer:
    """Customer information for cost attribution"""
    id: str
    name: str
    email: Optional[str] = None
    meta: Optional[Dict[str, Any]] = None


@dataclass
class Feature:
    """Feature information for cost attribution"""
    name: str
    meta: Optional[Dict[str, Any]] = None


@dataclass
class LangSpendTags:
    """Tags for customer and feature attribution"""
    customer: Optional[Customer] = None
    feature: Optional[Feature] = None
    session_id: Optional[str] = None
    environment: Optional[str] = None
    # Allow additional custom tags
    custom_tags: Optional[Dict[str, Any]] = None


@dataclass
class TrackResponse:
    """Response from tracking API"""
    success: bool
    cost: Optional[float] = None
    error: Optional[str] = None


@dataclass
class LangSpendConfig:
    """Configuration for LangSpend client"""
    api_key: str
    base_url: str = "https://platform.langspend.com"
    debug: bool = False
    auto_resolve_customers: bool = False
    max_retries: int = 3
    timeout: float = 30.0


@dataclass
class TrackingData:
    """Data structure for tracking LLM usage"""
    provider: str  # 'openai', 'anthropic', 'aws-bedrock', 'google-vertex'
    model: str
    input_tokens: int
    output_tokens: int
    cached_input_tokens: int = 0
    metadata: Optional[Dict[str, Any]] = None
    tags: Optional[LangSpendTags] = None
    timestamp: str = ""
    latency: Optional[float] = None


@dataclass
class CallSite:
    """Information about where a function was called from"""
    function_name: str
    file: str
    line: int


@dataclass
class FeatureContext:
    """Context information about the feature being used"""
    feature_name: str
    feature_source: str  # 'endpoint', 'function', 'manual', 'uncategorized'
    endpoint: Optional[str] = None
    call_site: Optional[str] = None
