"""
Anthropic wrapper for LangSpend tracking
"""

import time
import asyncio
from typing import Any, Optional
from ..client import LangSpend
from ..types import LangSpendTags, FeatureContext, TrackingData
from ..detection import detect_endpoint, detect_callsite, normalize_feature_name


def detect_feature_context(langspend_tags: Optional[LangSpendTags] = None, request: Optional[Any] = None) -> FeatureContext:
    """
    Detect feature context from the current execution environment.
    Priority: Manual → Endpoint → Function → Uncategorized
    """
    try:
        # Priority 1: Manual override via langspend_tags.feature
        if langspend_tags and langspend_tags.feature:
            return FeatureContext(
                feature_name=langspend_tags.feature.name,
                feature_source="manual",
                endpoint=None,
                call_site=None,
            )
        
        # Priority 2: Endpoint detection (HTTP contexts)
        endpoint = detect_endpoint(request)
        if endpoint:
            feature_name = normalize_feature_name(endpoint)
            return FeatureContext(
                feature_name=feature_name,
                feature_source="endpoint",
                endpoint=endpoint,
                call_site=None,
            )
        
        # Priority 3: Call site detection (fallback for non-HTTP contexts)
        call_site = detect_callsite()
        if call_site:
            call_site_str = f"{call_site.file}:{call_site.function_name}"
            feature_name = normalize_feature_name(call_site.function_name)
            return FeatureContext(
                feature_name=feature_name,
                feature_source="function",
                endpoint=None,
                call_site=call_site_str,
            )
        
        # Priority 4: Uncategorized (detection failed)
        return FeatureContext(
            feature_name="uncategorized",
            feature_source="uncategorized",
            endpoint=None,
            call_site=None,
        )
    except Exception:
        # Graceful failure - never block LLM calls
        return FeatureContext(
            feature_name="uncategorized",
            feature_source="uncategorized",
            endpoint=None,
            call_site=None,
        )




def wrap_anthropic(client: Any, langspend: LangSpend) -> Any:
    """
    Wrap an Anthropic client to automatically track requests.
    
    Args:
        client: Anthropic client instance
        langspend: LangSpend client instance
        
    Returns:
        Wrapped Anthropic client
    """
    # Store original methods
    original_create = client.messages.create
    original_acreate = getattr(client.messages, 'acreate', None)
    
    def wrapped_create(*args, **kwargs):
        """Wrapped synchronous create method"""
        start_time = time.time()
        
        # Extract LangSpend tags and request
        langspend_tags = kwargs.pop('langspend_tags', None)
        request = kwargs.pop('request', None)
        
        # Detect feature context
        feature_context = detect_feature_context(langspend_tags, request)
        
        # Make original LLM call
        response = original_create(*args, **kwargs)
        
        # Extract usage data
        usage = getattr(response, 'usage', None)
        if not usage:
            return response
        
        # Prepare tracking data
        tracking_data = TrackingData(
            provider="anthropic",
            model=kwargs.get('model', 'unknown'),
            input_tokens=getattr(usage, 'input_tokens', 0),
            output_tokens=getattr(usage, 'output_tokens', 0),
            cached_input_tokens=getattr(usage, 'cache_read_input_tokens', 0),
            metadata={
                "stop_reason": getattr(response, 'stop_reason', None),
                "stop_sequence": getattr(response, 'stop_sequence', None),
                "feature_name": feature_context.feature_name,
                "feature_source": feature_context.feature_source,
                "endpoint": feature_context.endpoint,
                "call_site": feature_context.call_site,
            },
            tags=langspend_tags,
            timestamp=time.time(),
            latency=time.time() - start_time,
        )
        
        # Track asynchronously (non-blocking)
        try:
            if asyncio.iscoroutinefunction(langspend.track):
                # If we're in an async context, schedule the tracking
                loop = asyncio.get_event_loop()
                if loop.is_running():
                    asyncio.create_task(langspend.track(tracking_data))
                else:
                    loop.run_until_complete(langspend.track(tracking_data))
            else:
                # Sync tracking
                langspend.track_sync(tracking_data)
        except Exception as e:
            if langspend.config.debug:
                print(f"LangSpend tracking failed: {e}")
        
        return response
    
    async def wrapped_acreate(*args, **kwargs):
        """Wrapped asynchronous create method"""
        start_time = time.time()
        
        # Extract LangSpend tags and request
        langspend_tags = kwargs.pop('langspend_tags', None)
        request = kwargs.pop('request', None)
        
        # Detect feature context
        feature_context = detect_feature_context(langspend_tags, request)
        
        # Make original LLM call
        response = await original_acreate(*args, **kwargs)
        
        # Extract usage data
        usage = getattr(response, 'usage', None)
        if not usage:
            return response
        
        # Prepare tracking data
        tracking_data = TrackingData(
            provider="anthropic",
            model=kwargs.get('model', 'unknown'),
            input_tokens=getattr(usage, 'input_tokens', 0),
            output_tokens=getattr(usage, 'output_tokens', 0),
            cached_input_tokens=getattr(usage, 'cache_read_input_tokens', 0),
            metadata={
                "stop_reason": getattr(response, 'stop_reason', None),
                "stop_sequence": getattr(response, 'stop_sequence', None),
                "feature_name": feature_context.feature_name,
                "feature_source": feature_context.feature_source,
                "endpoint": feature_context.endpoint,
                "call_site": feature_context.call_site,
            },
            tags=langspend_tags,
            timestamp=time.time(),
            latency=time.time() - start_time,
        )
        
        # Track asynchronously (non-blocking)
        try:
            await langspend.track(tracking_data)
        except Exception as e:
            if langspend.config.debug:
                print(f"LangSpend tracking failed: {e}")
        
        return response
    
    # Replace the methods
    client.messages.create = wrapped_create
    if original_acreate:
        client.messages.acreate = wrapped_acreate
    
    return client
