"""
OpenAI wrapper for LangSpend tracking
"""

import time
import asyncio
from datetime import datetime
from typing import Any, Optional, Dict, Union
from ..client import LangSpend
from ..types import LangSpendTags, FeatureContext, TrackingData
from ..detection import detect_endpoint, detect_callsite, normalize_feature_name


def detect_feature_context(langspend_tags: Optional[LangSpendTags] = None, request: Optional[Any] = None) -> FeatureContext:
    """
    Detect feature context from the current execution environment.
    Priority: Manual → Endpoint → Function → Uncategorized
    """
    try:
        # Priority 1: Manual override via langspend_tags.feature
        if langspend_tags and langspend_tags.feature:
            return FeatureContext(
                feature_name=langspend_tags.feature.name,
                feature_source="manual",
                endpoint=None,
                call_site=None,
            )
        
        # Priority 2: Endpoint detection (HTTP contexts)
        endpoint = detect_endpoint(request)
        if endpoint:
            feature_name = normalize_feature_name(endpoint)
            return FeatureContext(
                feature_name=feature_name,
                feature_source="endpoint",
                endpoint=endpoint,
                call_site=None,
            )
        
        # Priority 3: Call site detection (fallback for non-HTTP contexts)
        call_site = detect_callsite()
        if call_site:
            call_site_str = f"{call_site.file}:{call_site.function_name}"
            feature_name = normalize_feature_name(call_site.function_name)
            return FeatureContext(
                feature_name=feature_name,
                feature_source="function",
                endpoint=None,
                call_site=call_site_str,
            )
        
        # Priority 4: Uncategorized (detection failed)
        return FeatureContext(
            feature_name="uncategorized",
            feature_source="uncategorized",
            endpoint=None,
            call_site=None,
        )
    except Exception:
        # Graceful failure - never block LLM calls
        return FeatureContext(
            feature_name="uncategorized",
            feature_source="uncategorized",
            endpoint=None,
            call_site=None,
        )


def wrap_openai(client: Any, langspend: LangSpend) -> Any:
    """
    Wrap an OpenAI client to automatically track requests.
    
    Args:
        client: OpenAI client instance
        langspend: LangSpend client instance
        
    Returns:
        Wrapped OpenAI client
    """
    # Store original methods
    original_create = client.chat.completions.create
    original_acreate = getattr(client.chat.completions, 'acreate', None)
    
    def wrapped_create(*args, **kwargs):
        """Wrapped synchronous create method"""
        try:
            start_time = time.time()
            
            # Extract LangSpend tags and request
            langspend_tags_raw = kwargs.pop('langspend_tags', None)
            request = kwargs.pop('request', None)
            
            # Convert dict to LangSpendTags object if needed
            if langspend_tags_raw and isinstance(langspend_tags_raw, dict):
                if langspend.config.debug:
                    print(f"Converting dict to LangSpendTags: {langspend_tags_raw}")
                langspend_tags = LangSpendTags(
                    customer=langspend_tags_raw.get('customer'),
                    feature=langspend_tags_raw.get('feature'),
                    session_id=langspend_tags_raw.get('session_id'),
                    environment=langspend_tags_raw.get('environment'),
                    custom_tags=langspend_tags_raw.get('custom_tags')
                )
                if langspend.config.debug:
                    print(f"Converted to LangSpendTags object: {langspend_tags}")
            else:
                langspend_tags = langspend_tags_raw
            
            # Detect feature context
            feature_context = detect_feature_context(langspend_tags, request)
            
            # Make original LLM call
            response = original_create(*args, **kwargs)
            
            # Extract usage data
            usage = getattr(response, 'usage', None)
            if not usage:
                return response
            
            # Safely extract usage data
            try:
                input_tokens = getattr(usage, 'prompt_tokens', 0)
                output_tokens = getattr(usage, 'completion_tokens', 0)
            except Exception as e:
                if langspend.config.debug:
                    print(f"Error extracting usage data: {e}")
                return response
            
            # Prepare tracking data - match Node.js structure
            tracking_data = TrackingData(
                provider="openai",
                model=kwargs.get('model', 'unknown'),
                input_tokens=input_tokens,
                output_tokens=output_tokens,
                cached_input_tokens=0,  # Simplified for now
                metadata={
                    # Include customer and feature objects in metadata (like Node.js)
                    "customer": langspend_tags.customer.__dict__ if langspend_tags and hasattr(langspend_tags, 'customer') and langspend_tags.customer else None,
                    "feature": langspend_tags.feature.__dict__ if langspend_tags and hasattr(langspend_tags, 'feature') and langspend_tags.feature else None,
                    "sessionId": langspend_tags.session_id if langspend_tags and hasattr(langspend_tags, 'session_id') else None,
                    "environment": langspend_tags.environment if langspend_tags and hasattr(langspend_tags, 'environment') else None,
                    # Response metadata
                    "finish_reason": getattr(response.choices[0], 'finish_reason', None) if response.choices else None,
                    "system_fingerprint": getattr(response, 'system_fingerprint', None),
                    "feature_name": feature_context.feature_name,
                    "feature_source": feature_context.feature_source,
                    "endpoint": feature_context.endpoint,
                    "call_site": feature_context.call_site,
                },
                tags=langspend_tags,
                timestamp=datetime.utcnow().isoformat() + 'Z',
                latency=time.time() - start_time,
            )
            
            # Track asynchronously (non-blocking)
            try:
                # Use sync tracking for sync wrapper
                if langspend.config.debug:
                    print(f"LangSpend tracking data: {tracking_data}")
                result = langspend.track_sync(tracking_data)
                if langspend.config.debug:
                    print(f"LangSpend tracking result: {result}")
            except Exception as e:
                if langspend.config.debug:
                    print(f"LangSpend tracking failed: {e}")
                # Continue execution even if tracking fails
        
            return response
        except Exception as e:
            import traceback
            print(f"Error in wrapped_create: {e}")
            print(traceback.format_exc())
            # Re-raise to let FastAPI handle it
            raise
    
    async def wrapped_acreate(*args, **kwargs):
        """Wrapped asynchronous create method"""
        start_time = time.time()
        
        # Extract LangSpend tags and request
        langspend_tags = kwargs.pop('langspend_tags', None)
        request = kwargs.pop('request', None)
        
        # Detect feature context
        feature_context = detect_feature_context(langspend_tags, request)
        
        # Make original LLM call
        response = await original_acreate(*args, **kwargs)
        
        # Extract usage data
        usage = getattr(response, 'usage', None)
        if not usage:
            return response
        
        # Safely extract usage data
        try:
            input_tokens = getattr(usage, 'prompt_tokens', 0)
            output_tokens = getattr(usage, 'completion_tokens', 0)
        except Exception as e:
            if langspend.config.debug:
                print(f"Error extracting usage data: {e}")
            return response
        
        # Prepare tracking data - match Node.js structure
        tracking_data = TrackingData(
            provider="openai",
            model=kwargs.get('model', 'unknown'),
            input_tokens=input_tokens,
            output_tokens=output_tokens,
            cached_input_tokens=0,  # Simplified for now
            metadata={
                # Include customer and feature objects in metadata (like Node.js)
                "customer": langspend_tags.customer.__dict__ if langspend_tags and langspend_tags.customer else None,
                "feature": langspend_tags.feature.__dict__ if langspend_tags and langspend_tags.feature else None,
                "sessionId": langspend_tags.session_id if langspend_tags else None,
                "environment": langspend_tags.environment if langspend_tags else None,
                # Response metadata
                "finish_reason": getattr(response.choices[0], 'finish_reason', None) if response.choices else None,
                "system_fingerprint": getattr(response, 'system_fingerprint', None),
                "feature_name": feature_context.feature_name,
                "feature_source": feature_context.feature_source,
                "endpoint": feature_context.endpoint,
                "call_site": feature_context.call_site,
            },
            tags=langspend_tags,
            timestamp=datetime.utcnow().isoformat() + 'Z',
            latency=time.time() - start_time,
        )
        
        # Track asynchronously (non-blocking)
        try:
            await langspend.track(tracking_data)
        except Exception as e:
            if langspend.config.debug:
                print(f"LangSpend tracking failed: {e}")
            # Continue execution even if tracking fails
        
        return response
    
    # Replace the methods
    client.chat.completions.create = wrapped_create
    if original_acreate:
        client.chat.completions.acreate = wrapped_acreate
    
    return client
