# spec-driver

Specification-driven development toolkit with multi-language spec sync and documentation generation.

**Why?**
- Maintain verifiably accurate, evergreen specs covering your _entire system_
- Use cheap, fast, deterministically generated docs to complement and audit the work of messy, stochastic agents
- The combination of markdown and YAML is a surprisingly powerful platform for structured, legible data
- Tooling joins related entities through a registry for fast lookup, validation, and relational data 
- Stop banging rocks together

**What for?**
- greenfield spec-driven development with Claude Code & similar (alternatives exist for this, but ...)
- large codebases - 100kLoC+
- legacy conversions - with tooling support
- a low friction, conceptually coherent, unified CLI for spec-driven development you can adapt as your needs change
- (maybe already? if not soon): also ready to eat right out of the box!

![workflow diagram](https://raw.githubusercontent.com/davidlee/spec-driver/refs/heads/main/assets/simple.svg)

## Status

**Alpha** - Under active development. API and CLI interface may change.

## Features

A smorgasbord for you to build your own workflow around


|Feature|Blurb|
|---|---|
|**Multi-Language Spec Sync**|Automatically sync specifications with Go, Python & TypeScript/JS codebases|
|**Architecture Decision Records (ADRs)**|Record, find, manage and track architectural decisions|
|**Product Specs**|Product requirement docs, with customer-focused value drivers, use cases and personas|
|**Tech Specs**| Start with architectural vision - or fill it in after the fact, scaffolded by deterministically generated documentation|
|**Requirements**| Create and track requirements with categories, tags, and rich lifecycle support - functional, non-functional, technical and product.|
|**Policies & Standards**| Enforce project-wide policies & standards; encourage conventions and sensible defaults|
|**Delta/Change Tracking**|Create a delta to implement a set of requirements or backlog items, deltas, implementation plans and revisions. Record structured data (entry / exit criteria; files and commits) through planning and implementation. Record verification evidence (automated tests, agent or human review), driving requirement status updates.|
|**Backlog Management**|Tooling to create, search & filter Issues, Risks, Problem Statements, Improvements and related artefacts. Prioritise your backlog - just by moving lines up & down in your $EDITOR.|
|**Documentation Generation**|Compact, legible, deterministic markdown documentation from code. Autogenerated spec stubs for new code; change tracking.|
|**Metadata Schema Validation**|Ensure consistency across specification artifacts. CLI tooling support for agents: schema documentation and validation.|
|**Customisable Templates & Agent Commands**|Optionally install templates & commands locally to customise your workflow; fallback to defaults.|
|**Orphaned Spec Detection**|Identify & safely remove tech specs for deleted source files.|
|**Audits**|Record agent (or human) research & inspection to verify implementation, feeding back into requirements & spec status.|
|**Spec Revisions**|Optionally contextualize changes to product/architectural specs, with data as structured as makes sense. Start your changes with a revision to capture the design changes - or end with one to represent your findings and what actually happened.|
|**Markdown + Git \| jujutsu**|Why use a shitty Saas when version controlled text is this powerful, and agents can work with it so fluently? I'm sure you can pipe it into other, lesser tools if you need it.|
|**Zero Lock-In, Zero Cost**|Things change fast, but if text in open formats goes out of fashion, all bets are off.|

![ADR list](https://raw.githubusercontent.com/davidlee/spec-driver/refs/heads/main/assets/adr-list.png)

## Installation

### PyPi package

```bash
# try before you buy (no install)
uvx spec-driver --help

# Or settle in with it 
uv init
uv add speec-driver
uv run spec-driver --help
```

### From GitHub (Development)

```bash
# Install from latest commit
uv init 
uv add git+https://github.com/davidlee/spec-driver
uv run spec-driver --help

# Or use it right off the tubes
uvx --from git+https://github.com/davidlee/spec-driver spec-driver --help
```

## Quick Start

```bash
# Initialize workspace in your project
spec-driver install

# Sync specs with your codebase
spec-driver sync

# All commands have help
spec-driver --help

# List all your specs
spec-driver list specs

# Get the file path where a requirement is defined (every command supports --json)
spec-driver show requirement PROD-010.NF-001 --json | jq -r '.path'

# Create a new spec
spec-driver create spec --kind tech

# Better yet, have Claude do the legwork
/supekku.specify prod: make users punch a banner with a monkey to win prizes

# Create a new delta
spec-driver create delta 

# Create the next phase's runsheet for detailed tracking (under an implementation plan)
spec-driver create phase --plan IP-012
```

## Usage

All commands are accessed through the unified `spec-driver` CLI. You probably run it with `uv run spec-driver`, but that's a bit long so maybe set an alias? You'll be using it a bit.

### Installation & Setup

```bash
# Initialize spec-driver workspace structure
spec-driver install

# This creates:
# - specify/ directory (for specs, ADRs)
# - change/ directory (for deltas, revisions)
# - .spec-driver/registry/ (for YAML registries)
# - .spec-driver/templates (user-editable, markdown w/ Jinja2) 
```

### Synchronization

```bash
# Sync all specs with source code (auto-discovery)
spec-driver sync

# Sync only existing registered specs
spec-driver sync --existing

# Sync only specific language
spec-driver sync --language python

# Sync specific targets
spec-driver sync go:internal/foo python:module.py

# Dry run to preview changes
spec-driver sync --dry-run

# Check if docs are up-to-date (CI-friendly)
spec-driver sync --check

# Remove specs for deleted source files
spec-driver sync --prune

# Sync just the markdown registries (not generated tech specs / docs), log level INFO
spec-driver validate --sync --verbose
```

### Creating Artifacts

```bash
# Create a new technical spec
spec-driver create spec --kind tech

# Create a new product spec
spec-driver create spec --kind product

# Create a new delta (change proposal)
spec-driver create delta

# Create a requirement breakout
spec-driver create requirement

# Create a spec revision
spec-driver create revision

# Create an ADR
spec-driver create adr
```

Delta creation scaffolds a full bundle by default:
- `DE-XXX.md` — delta summary and scope
- `DR-XXX.md` — design revision capturing architecture intent, code impacts, and verification alignment
- `IP-XXX.md` — implementation plan
- `phases/phase-01.md` — first execution phase sheet
- `notes.md` — scratchpad for research and findings

### Listing Artefacts

```bash
# List all specs
spec-driver list specs

# List specs for specific package
spec-driver list specs --package internal/foo

# List specs with package details
spec-driver list specs --packages

# List deltas
spec-driver list deltas

# List deltas by status
spec-driver list deltas --status active

# List all changes (deltas, revisions, audits)
spec-driver list changes

# List changes by kind
spec-driver list changes --kind delta
```

### Architecture Decision Records

```bash
# Create new ADR
spec-driver create adr

# List ADRs
spec-driver list adrs

# List ADRs by status
spec-driver list adrs --status accepted

# Show ADR details
spec-driver show adr ADR-001

# Sync ADR registry
spec-driver sync --adr
```

### Policies and Standards

```bash
# Create a policy (hard rule)
spec-driver create policy "All code must have tests"

# Create a standard (convention/default)
spec-driver create standard "Use Google Python Style Guide"

# List policies
spec-driver list policies

# List policies by status
spec-driver list policies --status required

# List standards with "default" status
spec-driver list standards --status default

# Show policy details
spec-driver show policy POL-001

# Show standard details
spec-driver show standard STD-001

# List ADRs that reference a policy
spec-driver list adrs --policy POL-001

# List ADRs that reference a standard
spec-driver list adrs --standard STD-001
```

### Validation

```bash
# Validate workspace consistency
spec-driver validate
```

### Completing Work

```bash
# Mark delta as completed
spec-driver complete delta DE-001
```

## Project Integration

spec-driver integrates into your project using:

1. **Directory structure**: `specify/` and `change/` directories
2. **Registry files**: `.spec-driver/registry/*.yaml` for cross-references
3. **Templates**: `.spec-driver/templates/` for consistent artifact creation
4. **Agent instructions**: `.claude/commands` for AI-assisted development

## Common Workflows

### Daily Development

```bash
# 1. Create a delta for your change
spec-driver create delta

# 2. Write code, update specs as needed

# 3. Sync specs with code
spec-driver sync

# 4. Validate consistency
spec-driver validate

# 5. Mark delta complete
spec-driver complete delta DE-XXX
```

### Onboarding New Code

```bash
# Auto-discover and create specs for existing code
spec-driver sync --language python

# Review what would be created first
spec-driver sync --dry-run
```

### Cleaning Up

```bash
# Find orphaned specs (source files deleted)
spec-driver sync --existing --prune --dry-run

# Remove them
spec-driver sync --existing --prune
```

### Continuous Integration

```bash
# Verify specs are up-to-date
spec-driver sync --check

# Validate workspace
spec-driver validate
```

## Development

This package is under active development, and API stability is not even hinted at. 

I'll aim not to make breaking changes to data formats, though.

## Related
- [PyPi project](https://pypi.org/project/spec-driver/)
- [npm dependency for TS doc gen](https://www.npmjs.com/package/ts-doc-extract)
- [A Socratic dialogue wherein I talk myself into building this thing](https://supekku.dev)
- [Spec-Kit, which I stole / borrowed agent prompts and ideas from](https://github.com/github/spec-kit)
- [me](https://www.linkedin.com/in/davidlee-au/)

## License

MIT, go nuts.
